/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/***************************************************************************
 *            config.c
 *
 *  Sat Aug 21 23:59:26 2004
 *  Copyright  2004  Gergely Polonkai
 *  polesz@techinfo.hu
 ****************************************************************************/
/***************************************************************************
 * Configuration backends. Currently only GConf2 and samba is supported.
 * XML support is planned.
 ****************************************************************************/

#include "botcommander.h"
#if HAVE_CONFIG_H
# include "config.h"
#endif

#if (CONFIG_BACKEND == CB_GCONF2)
# include <gconf/gconf-client.h>
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
# include <stdio.h>
# include <stdlib.h>
# include <errno.h>
# include <sys/stat.h>
#endif
#include <gtk/gtk.h>
#include <string.h>

#include "typedefs.h"
#include "functions.h"
#include "variables.h"

#if (CONFIG_BACKEND == CB_GCONF2)
GConfClient *gconf_client;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
FILE *configfile;
#endif
global_config_data_t config_data;

#if (CONFIG_BACKEND == CB_GCONF2)
/*
 * gconf_sanity_check_string()
 * Copied from gnome-mud source code
 * Checks if gconf is available and operational
 */
gboolean
gconf_sanity_check_string(GConfClient *client, const gchar *key)
{
	gchar *string;
	GError *error = NULL;
	
	string = gconf_client_get_string(client, key, &error);
	
	if (error)
	{
		int err = 0;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "allow_bold", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_float(client, GCONF_CONFIG_ROOT "background_saturation", 1.0, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "change_to_message", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "debug", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "default_mode", "B", &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "sanity", "Needed for sanity check, never delete!", &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "scroll_on_output", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "tearoff_menus", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "transparent_background", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "vte_font", "Courier 10 Pitch 12", &error);
		if (error)
			err = 1;

		if (err == 1)
		{
			GtkWidget *dialog;
		
			dialog = gtk_message_dialog_new(NULL,
																			0,
																			GTK_MESSAGE_ERROR,
																			GTK_BUTTONS_OK,
																			"There was an error accessing GConf: %s",
																			error->message);
			gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
			gtk_dialog_run(GTK_DIALOG(dialog));
		
			return FALSE;
		}
	}
	
	if (!string)
	{
		int err = 0;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "allow_bold", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_float(client, GCONF_CONFIG_ROOT "background_saturation", 1.0, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "change_to_message", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "debug", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "default_mode", "B", &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "sanity", "Needed for sanity check, never delete!", &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "scroll_on_output", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "tearoff_menus", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "transparent_background", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "vte_font", "Courier 10 Pitch 12", &error);
		if (error)
			err = 1;
		
		if (err == 1)
		{
			GtkWidget *dialog;
		
			dialog = gtk_message_dialog_new(NULL,
																			0,
																			GTK_MESSAGE_ERROR,
																			GTK_BUTTONS_OK,
																			"<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
																			"The default configuration values could not be retrieved correctly.",
																			"Please check your GConf configuration, specifically the schemam have been installed correctly."
																			);
			gtk_label_set_use_markup(GTK_LABEL(GTK_MESSAGE_DIALOG(dialog)->label), TRUE);
			gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
			gtk_dialog_run(GTK_DIALOG(dialog));
			gtk_widget_destroy(dialog);
		
			return FALSE;
		}
	}
	
	g_free(string);
	
	return TRUE;
}
#endif

/*
 * get_string_value()
 * Get a string value from the configuration backend.
 */
gboolean
get_string_value(gchar *key, gchar **retval)
{
#if (CONFIG_BACKEND == CB_GCONF2)
	GError *error = NULL;
	GConfValue *value = NULL;
	gchar *full_key;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	gchar *content, **line;
	gboolean global_started = FALSE;
	signed int lnum;
	gint i;
#endif
#if (CONFIG_BACKEND == CB_XML)
# error "XML config backend is not supported yet"
#endif
	
#if (CONFIG_BACKEND == CB_GCONF2)
	if ((full_key = g_malloc(strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1)) == NULL)
		return FALSE;

	memset(full_key, 0, strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1);
	
	strcpy(full_key, GCONF_CONFIG_ROOT);
	strcat(full_key, key);
	
	value = gconf_client_get(gconf_client, full_key, &error);
	
	if (error || !value) {
		g_free(full_key);
		return FALSE;
	}

	if (value) {
		if (value->type == GCONF_VALUE_STRING) {
			if (retval)
				*retval = g_strdup(gconf_client_get_string(gconf_client, full_key, &error));
			g_free(full_key);
			return TRUE;
		}
	}
	g_free(full_key);
#endif /* (CONFIG_BACKEND == CB_GCONF2) */
	
#if (CONFIG_BACKEND == CB_SAMBA)
	if (configfile == NULL)
		return FALSE;
	if (!read_file(DEFAULT_CONFIG, &content))
		return FALSE;
	if ((lnum = wrap_string(content, "\n", &line)) < 1)
	{
		g_free(content);
		return FALSE;
	}
	for (i = 0; i < lnum; i++)
	{
		gchar *ret, *tkey, *tval, *eqsign;
		
		if ((ret = g_malloc(strlen(line[i]) + 1)) == NULL)
			continue;
		memset(ret, 0, strlen(line[i]) + 1);
		if (!trim(line[i], ret, NULL))
		{
			g_free(ret);
			continue;
		}
		
		if (str_eq(ret, "[global]", FALSE))
		{
			global_started = TRUE;
			continue;
		}
		if ((ret[0] == '[') && (ret[strlen(ret) - 1] == ']'))
		{
			global_started = FALSE;
			continue;
		}
		
		if ((eqsign = strchr(ret, '=')) == NULL)
		{
			g_free(ret);
			continue;
		}
		
		if ((tval = g_malloc(strlen(eqsign))) == NULL)
		{
			g_free(ret);
			continue;
		}
		memset(tval, 0, strlen(eqsign));
		
		if (!trim(eqsign + 1, tval, NULL))
		{
			g_free(tval);
			g_free(ret);
			continue;
		}
		
		*eqsign = 0;
		
		if ((tkey = g_malloc(strlen(ret))) == NULL)
		{
			g_free(tval);
			g_free(ret);
			continue;
		}
		memset(tkey, 0, strlen(ret));
		
		if (!trim(ret, tkey, NULL))
		{
			g_free(tkey);
			g_free(tval);
			g_free(ret);
			continue;
		}
		
		if (str_eq(tkey, key, FALSE))
		{
			g_free(tkey);
			g_free(ret);
			wl_free(line, lnum);
			if (retval)
				*retval = tval;
			return TRUE;
		}
		g_free(ret);
	}
	wl_free(line, lnum);
#endif
	return FALSE;
}

/*
 * get_boolean_value()
 * Get a boolean value from the configuration backend.
 */
gboolean
get_boolean_value(gchar *key, gboolean *retval)
{
#if (CONFIG_BACKEND == CB_GCONF2)
	GError *error = NULL;
	GConfValue *value = NULL;
	gchar *full_key;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	gchar *content, **line;
	gboolean global_started = FALSE;
	signed int lnum;
	gint i;
#endif
#if (CONFIG_BACKEND == CB_XML)
# error "XML config backend is not supported yet"
#endif
	
#if (CONFIG_BACKEND == CB_GCONF2)
	if ((full_key = g_malloc(strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1)) == NULL)
		return FALSE;
	
	memset(full_key, 0, strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1);
	
	strcpy(full_key, GCONF_CONFIG_ROOT);
	strcat(full_key, key);
	
	value = gconf_client_get(gconf_client, full_key, &error);
	
	if ((error != NULL) || (value == NULL)) {
		g_free(full_key);
		return FALSE;
	}

	if (value) {
		if (value->type == GCONF_VALUE_BOOL) {
			if (retval)
				*retval = gconf_client_get_bool(gconf_client, full_key, &error);
			g_free(full_key);
			return TRUE;
		}
	}
	g_free(full_key);
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	if (configfile == NULL)
		return FALSE;
	if (!read_file(DEFAULT_CONFIG, &content))
		return FALSE;
	if ((lnum = wrap_string(content, "\n", &line)) < 1)
	{
		g_free(content);
		return FALSE;
	}
	for (i = 0; i < lnum; i++)
	{
		gchar *ret, *tkey, *tval, *eqsign;
		
		if ((ret = g_malloc(strlen(line[i]) + 1)) == NULL)
			continue;
		memset(ret, 0, strlen(line[i]) + 1);
		if (!trim(line[i], ret, NULL))
		{
			g_free(ret);
			continue;
		}
		
		if (str_eq(ret, "[global]", FALSE))
		{
			global_started = TRUE;
			continue;
		}
		if ((ret[0] == '[') && (ret[strlen(ret) - 1] == ']'))
		{
			global_started = FALSE;
			continue;
		}
		
		if ((eqsign = strchr(ret, '=')) == NULL)
		{
			g_free(ret);
			continue;
		}
		
		if ((tval = g_malloc(strlen(eqsign))) == NULL)
		{
			g_free(ret);
			continue;
		}
		memset(tval, 0, strlen(eqsign));
		
		if (!trim(eqsign + 1, tval, NULL))
		{
			g_free(tval);
			g_free(ret);
			continue;
		}
		
		*eqsign = 0;
		
		if ((tkey = g_malloc(strlen(ret))) == NULL)
		{
			g_free(tval);
			g_free(ret);
			continue;
		}
		memset(tkey, 0, strlen(ret));
		
		if (!trim(ret, tkey, NULL))
		{
			g_free(tkey);
			g_free(tval);
			g_free(ret);
			continue;
		}
		
		if (str_eq(tkey, key, FALSE))
		{
			g_free(tkey);
			g_free(ret);
			wl_free(line, lnum);
			if (retval)
			{
				if (str_eq(tval, "yes", FALSE) || str_eq(tval, "true", FALSE) || str_eq(tval, "on", FALSE))
					*retval = TRUE;
				else
					*retval = FALSE;
			}
			return TRUE;
		}
		g_free(ret);
	}
	wl_free(line, lnum);
#endif
	return FALSE;
}

/*
 * get_float_value()
 * Get a float value from the configuration backend.
 */
gboolean
get_float_value(gchar *key, gfloat *retval)
{
#if (CONFIG_BACKEND == CB_GCONF2)
	GError *error = NULL;
	GConfValue *value = NULL;
	gchar *full_key;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	gchar *content, **line;
	gboolean global_started = FALSE;
	signed int lnum;
	gint i;
#endif
#if (CONFIG_BACKEND == CB_XML)
# error "XML config backend is not supported yet"
#endif
	
#if (CONFIG_BACKEND == CB_GCONF2)
	if ((full_key = g_malloc(strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1)) == NULL)
		return FALSE;
	
	memset(full_key, 0, strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1);
	
	strcpy(full_key, GCONF_CONFIG_ROOT);
	strcat(full_key, key);
	
	value = gconf_client_get(gconf_client, full_key, &error);
	
	if ((error != NULL) || (value == NULL)) {
		g_free(full_key);
		return FALSE;
	}

	if (value) {
		if (value->type == GCONF_VALUE_FLOAT) {
			if (retval)
				*retval = gconf_client_get_float(gconf_client, full_key, &error);
			g_free(full_key);
			return TRUE;
		}
	}
	g_free(full_key);
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	if (configfile == NULL)
		return FALSE;
	if (!read_file(DEFAULT_CONFIG, &content))
		return FALSE;
	if ((lnum = wrap_string(content, "\n", &line)) < 1)
	{
		g_free(content);
		return FALSE;
	}
	for (i = 0; i < lnum; i++)
	{
		gchar *ret, *tkey, *tval, *eqsign;
		
		if ((ret = g_malloc(strlen(line[i]) + 1)) == NULL)
			continue;
		memset(ret, 0, strlen(line[i]) + 1);
		if (!trim(line[i], ret, NULL))
		{
			g_free(ret);
			continue;
		}
		
		if (str_eq(ret, "[global]", FALSE))
		{
			global_started = TRUE;
			continue;
		}
		if ((ret[0] == '[') && (ret[strlen(ret) - 1] == ']'))
		{
			global_started = FALSE;
			continue;
		}
		
		if ((eqsign = strchr(ret, '=')) == NULL)
		{
			g_free(ret);
			continue;
		}
		
		if ((tval = g_malloc(strlen(eqsign))) == NULL)
		{
			g_free(ret);
			continue;
		}
		memset(tval, 0, strlen(eqsign));
		
		if (!trim(eqsign + 1, tval, NULL))
		{
			g_free(tval);
			g_free(ret);
			continue;
		}
		
		*eqsign = 0;
		
		if ((tkey = g_malloc(strlen(ret))) == NULL)
		{
			g_free(tval);
			g_free(ret);
			continue;
		}
		memset(tkey, 0, strlen(ret));
		
		if (!trim(ret, tkey, NULL))
		{
			g_free(tkey);
			g_free(tval);
			g_free(ret);
			continue;
		}
		
		if (str_eq(tkey, key, FALSE))
		{
			char *ep;
			float rv;
			
			g_free(tkey);
			g_free(ret);
			wl_free(line, lnum);
			
			rv = strtod(tval, &ep);
			if ((rv == 0) && (ep == tval))
				return FALSE;
			if (retval)
				*retval = rv;
			return TRUE;
		}
		g_free(ret);
	}
	wl_free(line, lnum);
#endif
	return FALSE;
}

/* 
 * get_int_value()
 * Get an integer value from the configuration backend.
 */
gboolean
get_int_value(gchar *key, gint *retval)
{
#if (CONFIG_BACKEND == CB_GCONF2)
	GError *error = NULL;
	GConfValue *value = NULL;
	gchar *full_key;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	gchar *content, **line;
	gboolean global_started = FALSE;
	signed int lnum;
	gint i;
#endif
#if (CONFIG_BACKEND == CB_XML)
# error "XML config backend is not supported yet"
#endif
	
#if (CONFIG_BACKEND == CB_GCONF2)
	if ((full_key = g_malloc(strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1)) == NULL)
		return FALSE;
	
	memset(full_key, 0, strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1);
	
	strcpy(full_key, GCONF_CONFIG_ROOT);
	strcat(full_key, key);
	
	value = gconf_client_get(gconf_client, full_key, &error);
	
	if ((error != NULL) || (value == NULL)) {
		g_free(full_key);
		return FALSE;
	}

	if (value) {
		if (value->type == GCONF_VALUE_INT) {
			if (retval)
				*retval = gconf_client_get_int(gconf_client, full_key, &error);
			g_free(full_key);
			return TRUE;
		}
	}
	g_free(full_key);
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	if (configfile == NULL)
		return FALSE;
	if (!read_file(DEFAULT_CONFIG, &content))
		return FALSE;
	if ((lnum = wrap_string(content, "\n", &line)) < 1)
	{
		g_free(content);
		return FALSE;
	}
	for (i = 0; i < lnum; i++)
	{
		gchar *ret, *tkey, *tval, *eqsign;
		
		if ((ret = g_malloc(strlen(line[i]) + 1)) == NULL)
			continue;
		memset(ret, 0, strlen(line[i]) + 1);
		if (!trim(line[i], ret, NULL))
		{
			g_free(ret);
			continue;
		}
		
		if (str_eq(ret, "[global]", FALSE))
		{
			global_started = TRUE;
			continue;
		}
		if ((ret[0] == '[') && (ret[strlen(ret) - 1] == ']'))
		{
			global_started = FALSE;
			continue;
		}
		
		if ((eqsign = strchr(ret, '=')) == NULL)
		{
			g_free(ret);
			continue;
		}
		
		if ((tval = g_malloc(strlen(eqsign))) == NULL)
		{
			g_free(ret);
			continue;
		}
		memset(tval, 0, strlen(eqsign));
		
		if (!trim(eqsign + 1, tval, NULL))
		{
			g_free(tval);
			g_free(ret);
			continue;
		}
		
		*eqsign = 0;
		
		if ((tkey = g_malloc(strlen(ret))) == NULL)
		{
			g_free(tval);
			g_free(ret);
			continue;
		}
		memset(tkey, 0, strlen(ret));
		
		if (!trim(ret, tkey, NULL))
		{
			g_free(tkey);
			g_free(tval);
			g_free(ret);
			continue;
		}
		
		if (str_eq(tkey, key, FALSE))
		{
			char *ep;
			long rv;
			
			g_free(tkey);
			g_free(ret);
			wl_free(line, lnum);
			
			rv = strtol(tval, &ep, 0);
			if ((rv == 0) && (ep == tval))
				return FALSE;
			if (retval)
				*retval = rv;
			return TRUE;
		}
		g_free(ret);
	}
	wl_free(line, lnum);
#endif
	return FALSE;
}

/*
 * set_string_value()
 * Sets a string value in the config backend
 */
gboolean
set_string_value(gchar *key_to_write, const gchar *value_to_write)
{
#if (CONFIG_BACKEND == CB_GCONF2)
	GError *error = NULL;
	gchar *full_key;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	gchar *content, **line, *filename, *myfilename;
	signed int line_num;
	int i;
	gboolean global_started = FALSE, found = FALSE;
	FILE *fh;
#endif
#if (CONFIG_BACKEND == CB_XML)
# warning "Saving to backend xml is not supported yet"
#endif
#if (CONFIG_BACKEND == CB_GCONF2)
	if ((full_key = g_malloc(strlen(GCONF_CONFIG_ROOT) + strlen(key_to_write) + 1)) == NULL)
		return FALSE;
	
	memset(full_key, 0, strlen(GCONF_CONFIG_ROOT) + strlen(key_to_write) + 1);
	
	strcpy(full_key, GCONF_CONFIG_ROOT);
	strcat(full_key, key_to_write);
	
	gconf_client_set_string(gconf_client, (const gchar *)full_key, (const char *)value_to_write, &error);
	
	if (error)
		return FALSE;
	
	return TRUE;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	if (!read_file(DEFAULT_CONFIG, &content))
	{
		if (config_data.debug_mode)
		{
			/* TODO */
			fprintf(stderr, "Unable to read config fie %s\n", DEFAULT_CONFIG);
		}
		return FALSE;
	}
	if ((line_num = wrap_string(content, "\r\n", &line)) < 0)
	{
		g_free(content);
		return FALSE;
	}
	
	if ((filename = g_malloc(strlen(DEFAULT_CONFIG) + 1)) == NULL)
	{
		g_free(content);
		return FALSE;
	}
	
	memset(filename, 0, strlen(DEFAULT_CONFIG) + 1);
	strcpy(filename, DEFAULT_CONFIG);
	
	if ((filename[0] == '~') && (filename[1] == '/'))
	{
		gchar *home = getenv("HOME");
		if ((myfilename = g_malloc(strlen(home) + strlen(DEFAULT_CONFIG))) == NULL)
		{
			g_free(content);
			g_free(filename);
			return FALSE;
		}
		strcpy(myfilename, home);
		if (myfilename[strlen(myfilename) - 1] != '/')
			strcat(myfilename, "/");
		strcat(myfilename, (DEFAULT_CONFIG) + 2);
		g_free(filename);
	}
	else
		myfilename = filename;
	
	fclose(configfile);

	if ((fh = fopen(myfilename, "w")) == NULL)
	{
		g_free(myfilename);
		g_free(content);
		configfile = fopen(myfilename, "r");
		return FALSE;
	}
	
	for (i = 0; i < line_num; i++)
	{
		gchar *ret, *eqsign;
		
		if ((ret = g_malloc(strlen(line[i]) + 1)) == NULL)
		{
			wl_free(line, line_num);
			fclose(fh);
			fclose(configfile);
			configfile = fopen(myfilename, "r");
			return FALSE;
		}
		memset(ret, 0, strlen(line[i]) + 1);
		
		if (!trim(line[i], ret, NULL))
		{
			g_free(ret);
			fprintf(fh, "%s\n", line[i]);
			continue;
		}
		
		if ((ret[0] == ';') || (ret[0] == '#'))
		{
			fprintf(fh, "%s\n", line[i]);
			g_free(ret);
			continue;
		}
		
		if (((ret[0] == '[') && (ret[strlen(ret) - 1] == ']')) && (!str_eq("[global]", ret, TRUE)))
		{
			if (global_started && !found)
				fprintf(fh, "%s = %s\n", key_to_write, value_to_write);
			global_started = FALSE;
			fprintf(fh, "\n%s\n", line[i]);
			g_free(ret);
			continue;
		}
		if (str_eq("[global]", ret, FALSE))
		{
			global_started = TRUE;
			fprintf(fh, "\n%s\n", line[i]);
			g_free(ret);
			continue;
		}
		if (!global_started)
		{
			fprintf(fh, "%s\n", line[i]);
			g_free(ret);
			continue;
		}
		
		if ((eqsign = strchr(ret, '=')) != NULL)
		{
			gchar *tval, *tkey;
			
			if ((tval = g_malloc(strlen(eqsign))) == NULL)
			{
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				continue;
			}
			memset(tval, 0, strlen(eqsign));
			if (!trim(eqsign + 1, tval, NULL))
			{
				g_free(tval);
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				continue;
			}
			*eqsign = 0;
			if ((tkey = g_malloc(strlen(ret))) == NULL)
			{
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				g_free(tval);
				continue;
			}
			memset(tkey, 0, strlen(ret));
			if (!trim(ret, tkey, NULL))
			{
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				g_free(tval);
				g_free(tkey);
				continue;
			}
			
			if (str_eq(tkey, key_to_write, FALSE))
			{
				found = TRUE;
				if (!str_eq(tval, (gchar *)value_to_write, TRUE))
				{
					fprintf(fh, "%s = %s\n", key_to_write, value_to_write);
					g_free(tkey);
					g_free(tval);
					g_free(ret);
					found = TRUE;
					continue;
				}
				else
				{
					fprintf(fh, "%s\n", line[i]);
					g_free(tkey);
					g_free(tval);
					g_free(ret);
					found = TRUE;
					continue;
				}
			}
		}
		
		fprintf(fh, "%s\n", line[i]);
		g_free(ret);
	}
	fclose(fh);
	g_free(content);
	configfile = fopen(myfilename, "r");
#endif
	return FALSE;
}

/*
 * set_boolean_value()
 * Sets a boolean config value in the config backend
 */
gboolean
set_boolean_value(gchar *key_to_write, gboolean value_to_write)
{
#if (CONFIG_BACKEND == CB_GCONF2)
	GError *error = NULL;
	gchar *full_key;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	gchar *content, **line, *filename, *myfilename;
	signed int line_num;
	int i;
	gboolean global_started = FALSE, found = FALSE;
	FILE *fh;
#endif
#if (CONFIG_BACKEND == CB_XML)
# warning "Saving to backend xml is not supported yet"
#endif
#if (CONFIG_BACKEND == CB_GCONF2)
	if ((full_key = g_malloc(strlen(GCONF_CONFIG_ROOT) + strlen(key_to_write) + 1)) == NULL)
		return FALSE;
	
	memset(full_key, 0, strlen(GCONF_CONFIG_ROOT) + strlen(key_to_write) + 1);
	
	strcpy(full_key, GCONF_CONFIG_ROOT);
	strcat(full_key, key_to_write);
	
	gconf_client_set_bool(gconf_client, (const gchar *)full_key, value_to_write, &error);
	if (error)
		return FALSE;
	
	return TRUE;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	if (!read_file(DEFAULT_CONFIG, &content))
	{
		if (config_data.debug_mode)
		{
			/* TODO */
			fprintf(stderr, "Unable to read config fie %s\n", DEFAULT_CONFIG);
		}
		return FALSE;
	}
	if ((line_num = wrap_string(content, "\r\n", &line)) < 0)
	{
		g_free(content);
		return FALSE;
	}
	
	if ((filename = g_malloc(strlen(DEFAULT_CONFIG) + 1)) == NULL)
	{
		g_free(content);
		return FALSE;
	}
	
	memset(filename, 0, strlen(DEFAULT_CONFIG) + 1);
	strcpy(filename, DEFAULT_CONFIG);
	
	if ((filename[0] == '~') && (filename[1] == '/'))
	{
		gchar *home = getenv("HOME");
		if ((myfilename = g_malloc(strlen(home) + strlen(DEFAULT_CONFIG))) == NULL)
		{
			g_free(content);
			g_free(filename);
			return FALSE;
		}
		strcpy(myfilename, home);
		if (myfilename[strlen(myfilename) - 1] != '/')
			strcat(myfilename, "/");
		strcat(myfilename, (DEFAULT_CONFIG) + 2);
		g_free(filename);
	}
	else
		myfilename = filename;
	
	fclose(configfile);

	if ((fh = fopen(myfilename, "w")) == NULL)
	{
		g_free(myfilename);
		g_free(content);
		configfile = fopen(myfilename, "r");
		return FALSE;
	}
	
	for (i = 0; i < line_num; i++)
	{
		gchar *ret, *eqsign;
		
		if ((ret = g_malloc(strlen(line[i]) + 1)) == NULL)
		{
			wl_free(line, line_num);
			fclose(fh);
			configfile = fopen(myfilename, "r");
			return FALSE;
		}
		memset(ret, 0, strlen(line[i]) + 1);
		
		if (!trim(line[i], ret, NULL))
		{
			g_free(ret);
			fprintf(fh, "%s\n", line[i]);
			continue;
		}
		
		if ((ret[0] == ';') || (ret[0] == '#'))
		{
			fprintf(fh, "%s\n", line[i]);
			g_free(ret);
			continue;
		}
		
		if (((ret[0] == '[') && (ret[strlen(ret) - 1] == ']')) && (!str_eq("[global]", ret, TRUE)))
		{
			if (global_started && !found)
				fprintf(fh, "%s = %s\n", key_to_write, (value_to_write) ? "yes" : "no");
			global_started = FALSE;
			fprintf(fh, "\n%s\n", line[i]);
			g_free(ret);
			continue;
		}
		if (str_eq("[global]", ret, FALSE))
		{
			global_started = TRUE;
			fprintf(fh, "\n%s\n", line[i]);
			g_free(ret);
			continue;
		}
		if (!global_started)
		{
			fprintf(fh, "%s\n", line[i]);
			g_free(ret);
			continue;
		}
		
		if ((eqsign = strchr(ret, '=')) != NULL)
		{
			gchar *tval, *tkey;
			
			if ((tval = g_malloc(strlen(eqsign))) == NULL)
			{
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				continue;
			}
			memset(tval, 0, strlen(eqsign));
			if (!trim(eqsign + 1, tval, NULL))
			{
				g_free(tval);
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				continue;
			}
			*eqsign = 0;
			if ((tkey = g_malloc(strlen(ret))) == NULL)
			{
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				g_free(tval);
				continue;
			}
			memset(tkey, 0, strlen(ret));
			if (!trim(ret, tkey, NULL))
			{
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				g_free(tval);
				g_free(tkey);
				continue;
			}
			
			if (str_eq(tkey, key_to_write, FALSE))
			{
				found = TRUE;
				if (!str_eq(tval, (gchar *)value_to_write, TRUE))
				{
					fprintf(fh, "%s = %s\n", key_to_write, (value_to_write) ? "yes" : "no");
					g_free(tkey);
					g_free(tval);
					g_free(ret);
					found = TRUE;
					continue;
				}
				else
				{
					fprintf(fh, "%s\n", line[i]);
					g_free(tkey);
					g_free(tval);
					g_free(ret);
					found = TRUE;
					continue;
				}
			}
		}
		
		fprintf(fh, "%s\n", line[i]);
		g_free(ret);
	}
	fclose(fh);
	g_free(content);
	configfile = fopen(myfilename, "r");
#endif

	return FALSE;
}

/*
 * set_float_value()
 * Sets a float value in the config backend
 */
gboolean
set_float_value(gchar *key_to_write, gfloat value_to_write)
{
#if (CONFIG_BACKEND == CB_GCONF2)
	GError *error = NULL;
	gchar *full_key;
#endif
#if (CONFIG_BACKEND == CB_XML)
# warning "Saving float values to xml config backend is not supported yet"
#endif
#if (CONFIG_BACKEND == CB_GCONF2)
	if ((full_key = g_malloc(strlen(GCONF_CONFIG_ROOT) + strlen(key_to_write) + 1)) == NULL)
		return FALSE;
	
	memset(full_key, 0, strlen(GCONF_CONFIG_ROOT) + strlen(key_to_write) + 1);
	
	strcpy(full_key, GCONF_CONFIG_ROOT);
	strcat(full_key, key_to_write);
	
	gconf_client_set_float(gconf_client, (const gchar *)full_key, value_to_write, &error);
	if (error)
		return FALSE;
	
	return TRUE;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	if (!read_file(DEFAULT_CONFIG, &content))
	{
		if (config_data.debug_mode)
		{
			/* TODO */
			fprintf(stderr, "Unable to read config fie %s\n", DEFAULT_CONFIG);
		}
		return FALSE;
	}
	if ((line_num = wrap_string(content, "\r\n", &line)) < 0)
	{
		g_free(content);
		return FALSE;
	}
	
	if ((filename = g_malloc(strlen(DEFAULT_CONFIG) + 1)) == NULL)
	{
		g_free(content);
		return FALSE;
	}
	
	memset(filename, 0, strlen(DEFAULT_CONFIG) + 1);
	strcpy(filename, DEFAULT_CONFIG);
	
	if ((filename[0] == '~') && (filename[1] == '/'))
	{
		gchar *home = getenv("HOME");
		if ((myfilename = g_malloc(strlen(home) + strlen(DEFAULT_CONFIG))) == NULL)
		{
			g_free(content);
			g_free(filename);
			return FALSE;
		}
		strcpy(myfilename, home);
		if (myfilename[strlen(myfilename) - 1] != '/')
			strcat(myfilename, "/");
		strcat(myfilename, (DEFAULT_CONFIG) + 2);
		g_free(filename);
	}
	else
		myfilename = filename;
	
	fclose(configfile);

	if ((fh = fopen(myfilename, "w")) == NULL)
	{
		g_free(myfilename);
		g_free(content);
		configfile = fopen(myfilename, "r");
		return FALSE;
	}
	
	for (i = 0; i < line_num; i++)
	{
		gchar *ret, *eqsign;
		
		if ((ret = g_malloc(strlen(line[i]) + 1)) == NULL)
		{
			wl_free(line, line_num);
			fclose(fh);
			configfile = fopen(myfilename, "r");
			return FALSE;
		}
		memset(ret, 0, strlen(line[i]) + 1);
		
		if (!trim(line[i], ret, NULL))
		{
			g_free(ret);
			fprintf(fh, "%s\n", line[i]);
			continue;
		}
		
		if ((ret[0] == ';') || (ret[0] == '#'))
		{
			fprintf(fh, "%s\n", line[i]);
			g_free(ret);
			continue;
		}
		
		if (((ret[0] == '[') && (ret[strlen(ret) - 1] == ']')) && (!str_eq("[global]", ret, TRUE)))
		{
			if (global_started && !found)
				fprintf(fh, "%s = %s\n", key_to_write, (value_to_write) ? "yes" : "no");
			global_started = FALSE;
			fprintf(fh, "\n%s\n", line[i]);
			g_free(ret);
			continue;
		}
		if (str_eq("[global]", ret, FALSE))
		{
			global_started = TRUE;
			fprintf(fh, "\n%s\n", line[i]);
			g_free(ret);
			continue;
		}
		if (!global_started)
		{
			fprintf(fh, "%s\n", line[i]);
			g_free(ret);
			continue;
		}
		
		if ((eqsign = strchr(ret, '=')) != NULL)
		{
			gchar *tval, *tkey;
			
			if ((tval = g_malloc(strlen(eqsign))) == NULL)
			{
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				continue;
			}
			memset(tval, 0, strlen(eqsign));
			if (!trim(eqsign + 1, tval, NULL))
			{
				g_free(tval);
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				continue;
			}
			*eqsign = 0;
			if ((tkey = g_malloc(strlen(ret))) == NULL)
			{
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				g_free(tval);
				continue;
			}
			memset(tkey, 0, strlen(ret));
			if (!trim(ret, tkey, NULL))
			{
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				g_free(tval);
				g_free(tkey);
				continue;
			}
			
			if (str_eq(tkey, key_to_write, FALSE))
			{
				found = TRUE;
				if (!str_eq(tval, (gchar *)value_to_write, TRUE))
				{
					fprintf(fh, "%s = %f\n", key_to_write, value_to_write);
					g_free(tkey);
					g_free(tval);
					g_free(ret);
					found = TRUE;
					continue;
				}
				else
				{
					fprintf(fh, "%s\n", line[i]);
					g_free(tkey);
					g_free(tval);
					g_free(ret);
					found = TRUE;
					continue;
				}
			}
		}
		
		fprintf(fh, "%s\n", line[i]);
		g_free(ret);
	}
	fclose(fh);
	g_free(content);
	configfile = fopen(myfilename, "r");
#endif
	return FALSE;
}

/*
 * set_int_value()
 * Sets an integer value in the config backend
 */
gboolean
set_int_value(gchar *key_to_write, gint value_to_write)
{
#if (CONFIG_BACKEND == CB_GCONF2)
	GError *error = NULL;
	gchar *full_key;
#endif
#if (CONFIG_BACKEND == CB_XML)
# warning "Saving float values to xml config backend is not supported yet"
#endif
#if (CONFIG_BACKEND == CB_GCONF2)
	if ((full_key = g_malloc(strlen(GCONF_CONFIG_ROOT) + strlen(key_to_write) + 1)) == NULL)
		return FALSE;
	
	memset(full_key, 0, strlen(GCONF_CONFIG_ROOT) + strlen(key_to_write) + 1);
	
	strcpy(full_key, GCONF_CONFIG_ROOT);
	strcat(full_key, key_to_write);
	
	gconf_client_set_int(gconf_client, (const gchar *)full_key, value_to_write, &error);
	if (error)
		return FALSE;
	
	return TRUE;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	if (!read_file(DEFAULT_CONFIG, &content))
	{
		if (config_data.debug_mode)
		{
			/* TODO */
			fprintf(stderr, "Unable to read config fie %s\n", DEFAULT_CONFIG);
		}
		return FALSE;
	}
	if ((line_num = wrap_string(content, "\r\n", &line)) < 0)
	{
		g_free(content);
		return FALSE;
	}
	
	if ((filename = g_malloc(strlen(DEFAULT_CONFIG) + 1)) == NULL)
	{
		g_free(content);
		return FALSE;
	}
	
	memset(filename, 0, strlen(DEFAULT_CONFIG) + 1);
	strcpy(filename, DEFAULT_CONFIG);
	
	if ((filename[0] == '~') && (filename[1] == '/'))
	{
		gchar *home = getenv("HOME");
		if ((myfilename = g_malloc(strlen(home) + strlen(DEFAULT_CONFIG))) == NULL)
		{
			g_free(content);
			g_free(filename);
			return FALSE;
		}
		strcpy(myfilename, home);
		if (myfilename[strlen(myfilename) - 1] != '/')
			strcat(myfilename, "/");
		strcat(myfilename, (DEFAULT_CONFIG) + 2);
		g_free(filename);
	}
	else
		myfilename = filename;
	
	fclose(configfile);

	if ((fh = fopen(myfilename, "w")) == NULL)
	{
		g_free(myfilename);
		g_free(content);
		configfile = fopen(myfilename, "r");
		return FALSE;
	}
	
	for (i = 0; i < line_num; i++)
	{
		gchar *ret, *eqsign;
		
		if ((ret = g_malloc(strlen(line[i]) + 1)) == NULL)
		{
			wl_free(line, line_num);
			fclose(fh);
			configfile = fopen(myfilename, "r");
			return FALSE;
		}
		memset(ret, 0, strlen(line[i]) + 1);
		
		if (!trim(line[i], ret, NULL))
		{
			g_free(ret);
			fprintf(fh, "%s\n", line[i]);
			continue;
		}
		
		if ((ret[0] == ';') || (ret[0] == '#'))
		{
			fprintf(fh, "%s\n", line[i]);
			g_free(ret);
			continue;
		}
		
		if (((ret[0] == '[') && (ret[strlen(ret) - 1] == ']')) && (!str_eq("[global]", ret, TRUE)))
		{
			if (global_started && !found)
				fprintf(fh, "%s = %s\n", key_to_write, (value_to_write) ? "yes" : "no");
			global_started = FALSE;
			fprintf(fh, "\n%s\n", line[i]);
			g_free(ret);
			continue;
		}
		if (str_eq("[global]", ret, FALSE))
		{
			global_started = TRUE;
			fprintf(fh, "\n%s\n", line[i]);
			g_free(ret);
			continue;
		}
		if (!global_started)
		{
			fprintf(fh, "%s\n", line[i]);
			g_free(ret);
			continue;
		}
		
		if ((eqsign = strchr(ret, '=')) != NULL)
		{
			gchar *tval, *tkey;
			
			if ((tval = g_malloc(strlen(eqsign))) == NULL)
			{
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				continue;
			}
			memset(tval, 0, strlen(eqsign));
			if (!trim(eqsign + 1, tval, NULL))
			{
				g_free(tval);
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				continue;
			}
			*eqsign = 0;
			if ((tkey = g_malloc(strlen(ret))) == NULL)
			{
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				g_free(tval);
				continue;
			}
			memset(tkey, 0, strlen(ret));
			if (!trim(ret, tkey, NULL))
			{
				fprintf(fh, "%s\n", line[i]);
				g_free(ret);
				g_free(tval);
				g_free(tkey);
				continue;
			}
			
			if (str_eq(tkey, key_to_write, FALSE))
			{
				found = TRUE;
				if (!str_eq(tval, (gchar *)value_to_write, TRUE))
				{
					fprintf(fh, "%s = %d\n", key_to_write, value_to_write);
					g_free(tkey);
					g_free(tval);
					g_free(ret);
					found = TRUE;
					continue;
				}
				else
				{
					fprintf(fh, "%s\n", line[i]);
					g_free(tkey);
					g_free(tval);
					g_free(ret);
					found = TRUE;
					continue;
				}
			}
		}
		
		fprintf(fh, "%s\n", line[i]);
		g_free(ret);
	}
	fclose(fh);
	g_free(content);
	configfile = fopen(myfilename, "r");
#endif
	return FALSE;
}

/*
 * config_init()
 * Initiates a connection to the gconf engine.
 */
gboolean
config_init(void)
{
	gboolean setting_b;
	gfloat setting_f;
	gchar *setting_s;
	gint setting_i;
#if (CONFIG_BACKEND == CB_SAMBA)
	char *filename;
#endif

#if (CONFIG_BACKEND == CB_GCONF2)
	gconf_client = gconf_client_get_default();
	if (!gconf_sanity_check_string(gconf_client, GCONF_CONFIG_ROOT "sanity"))
	{
		return FALSE;
	}
#endif
	
#if (CONFIG_BACKEND == CB_SAMBA)
	if ((filename=g_malloc(strlen(DEFAULT_CONFIG))) == NULL)
		return FALSE;

	strcpy(filename, DEFAULT_CONFIG);
	
	if ((filename[0] == '~') && (filename[1] == '/'))
	{
		gchar *home;
		
		home = getenv("HOME");
		g_free(filename);
		filename = g_malloc(strlen(home) + strlen(DEFAULT_CONFIG));
		memset(filename, 0, strlen(home) + strlen(DEFAULT_CONFIG));
		strcpy(filename, home);
		if (filename[strlen(filename) - 1] != '/')
			strcat(filename, "/");
		strcat(filename, (DEFAULT_CONFIG) + 2);
	}
	
	if ((configfile = fopen(filename, "r")) == NULL)
	{
		fprintf(stderr, "Unable to open configuration file '%s': %s. Exiting.\n", filename, strerror(errno));
		return FALSE;
	}
#endif
	
	/* Get the configuration data. */
	if (get_boolean_value("scroll_on_output", &setting_b))
		config_data.scroll_on_output = setting_b;
	else
		config_data.scroll_on_output = DEFAULT_SCROLL_ON_OUTPUT;
	
	if (get_boolean_value("transparent_background", &setting_b))
		config_data.transparent_background = setting_b;
	else
		config_data.transparent_background = DEFAULT_TRANSPARENT_BACKGROUND;
	
	if (get_float_value("background_saturation", &setting_f))
		config_data.background_saturation = setting_f;
	else
		config_data.background_saturation = DEFAULT_BACKGROUND_SATURATION;
	
	if (get_boolean_value("debug", &setting_b))
		config_data.debug_mode = setting_b;
	else
		config_data.debug_mode = DEFAULT_DEBUG_MODE;
	
	if (get_string_value("vte_font", &setting_s))
		config_data.vte_font = setting_s;
	else
		config_data.vte_font = g_strdup(DEFAULT_VTE_FONT);
	
	if (get_boolean_value("change_to_message", &setting_b))
		config_data.change_to_message = setting_b;
	else
		config_data.change_to_message = DEFAULT_CHANGE_TO_MESSAGE;
	
	if (get_boolean_value("autosave_on_exit", &setting_b))
		config_data.autosave_on_exit = setting_b;
	else
		config_data.autosave_on_exit = DEFAULT_AUTOSAVE_ON_EXIT;
	
	if (get_int_value("commandline_history_length", &setting_i))
		config_data.commandline_history_length = setting_i;
	else
		config_data.commandline_history_length = DEFAULT_COMMANDLINE_HISTORY_LENGTH;
	
	return TRUE;
}

/*
 * create_version_string()
 * Creates a version string. If need_feed is true, it will append a
 * linefeed to the end of the string, which is accepted by the vte.
 */
gchar *
create_version_string(gboolean need_feed)
{
	int len;
	gchar *ret;
	
	len = strlen(PACKAGE) + 1 + strlen(VERSION) + 1;
	if (need_feed)
		len += 2;
	
	if ((ret = g_malloc(len)) == NULL)
	{
		GtkWidget *dialog;
		
		dialog = gtk_message_dialog_new(NULL,
		                                0,
		                                GTK_MESSAGE_ERROR,
		                                GTK_BUTTONS_OK,
		                                "Cannot allocate enough memory. This is not good. Exiting.");
		gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
		gtk_dialog_run(GTK_DIALOG(dialog));
		
		gtk_main_quit();
	}
	
	strcpy(ret, PACKAGE);
	strcat(ret, " ");
	strcat(ret, VERSION);
	if (need_feed)
		strcat(ret, "\r\n");
	
	return ret;
}
