(* Copyright (C) 2014, Daniel Wyckoff, except for the portions so labeleled
which I got from Daniel Schepler*)
(*This file is part of BooleanAlgebrasIntro2.

BooleanAlgebrasIntro2 is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

BooleanAlgebrasIntro2 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with BooleanAlgebrasIntro2.  If not, see <http://www.gnu.org/licenses/>.*)


Require Export Ensembles.
Require Export EnsemblesImplicit.
Require Import FunctionProperties.
Require Export Finite_sets.
Require Import Image.
Require Import ImageImplicit.
Require Import DecidableDec.
Require Import SetUtilities.
Require Import ClassicalChoice.
Require Import Arith.
Require Import CSB.
Require Import LogicUtilities.
Require Import Basics.
Require Import ArithUtilities.
Require Import Omega. 
Require Import Description.
Require Import ProofIrrelevance.
Require Import EqdepFacts.


(*Many of these functions are copied and pasted from Daniel Schepler's
  "Zorn's Lemma" user contribution.  Those which are are commented 
  "Schepler."  Those which aren't are commented "Wyckoff"*)
Section FiniteTypes.

(*Schepler*)
Inductive FiniteT : Type -> Prop :=
  | empty_finite: FiniteT False
  | add_finite: forall T:Type, FiniteT T -> FiniteT (option T)
  | bij_finite: forall (X Y:Type) (f:X->Y), FiniteT X ->
    invertible f -> FiniteT Y.

(*Schepler*)
Lemma True_finite: FiniteT True.
Proof.
apply bij_finite with (option False)
  (fun _ => I).
constructor; constructor.
exists (True_rect None).
destruct x as [[]|].
remember (True_rect (@None False) I) as LHS.
destruct LHS as [[]|].
reflexivity.

exact (fun y:True => match y with
  | I => refl_equal I
  end).
Qed.


(*Schepler*)
Lemma finite_dec_exists: forall (X:Type) (P:X->Prop),
  FiniteT X -> (forall x:X, {P x} + {~ P x}) ->
  { exists x:X, P x } + { forall x:X, ~ P x }.
Proof.
intros.
apply exclusive_dec.
red; intro.
destruct H0.
destruct H0.
contradiction (H1 x).

revert P X0.
induction H.
right.
destruct x.

intros.
case (IHFiniteT (fun x:T => P (Some x))
  (fun x:T => X0 (Some x))).
left.
destruct H0.
exists (Some x).
assumption.
intro.
case (X0 None).
left.
exists None.
assumption.
right.
destruct x.
apply H0.
assumption.
destruct H0.
intros.
case (IHFiniteT (fun x:X => P (f x))
  (fun x:X => X0 (f x))).
left.
destruct H2.
exists (f x).
assumption.
right.
intro.
rewrite <- H1 with x.
apply H2.
Qed.

(*Schepler*)
Lemma finite_dec_forall: forall (X:Type) (P:X->Prop),
  FiniteT X -> (forall x:X, { P x } + { ~ P x }) ->
  { forall x:X, P x } + { exists x:X, ~ P x }.
Proof.
intros.
apply exclusive_dec.
intuition.
destruct H2.
contradiction (H1 x).

revert P X0.
induction H.
left.
destruct x.
intros.
case (IHFiniteT (fun x:T => P (Some x))
  (fun x:T => X0 (Some x))).
intro.
case (X0 None).
left.
destruct x.
apply H0.
assumption.
right.
exists None.
assumption.
right.
destruct H0.
exists (Some x).
assumption.

intros.
destruct H0.
case (IHFiniteT (fun x:X => P (f x))
  (fun x:X => X0 (f x))).
left.
intro y.
rewrite <- H1.
apply H2.
right.
destruct H2.
exists (f x).
assumption.
Qed.


(*Schepler*)
Lemma FiniteT_img: forall (X Y:Type) (f:X->Y),
  FiniteT X -> (forall y1 y2:Y, y1=y2 \/ y1<>y2) ->
  Finite (Im (Full_set _) f).
Proof.
intros.
induction H.
assert (Im (Full_set _) f = Empty_set _).
apply Extensionality_Ensembles.
red; split.
red; intros.
destruct H.
destruct x.
auto with sets.
rewrite H.
constructor.

assert ({exists x:T, f (Some x) = f None} +
        {forall x:T, f (Some x) <> f None}).
apply finite_dec_exists.
assumption.
intro.
apply decidable_dec.
apply H0.
case H1.
intro.
pose (g := fun (x:T) => f (Some x)).
assert (Im (Full_set _) f =
        Im (Full_set _) g).
apply Extensionality_Ensembles.
red; split.
red; intros.
destruct H2.
destruct x.
exists t.
constructor.
assumption.
destruct e.
exists x.
constructor.
transitivity (f None).
assumption.
symmetry; assumption.
red; intros.

destruct H2.
exists (Some x).
constructor.
assumption.
rewrite H2.
apply IHFiniteT.

intros.
pose (g := fun x:T => f (Some x)).
assert (Im (Full_set _) f =
  Add (Im (Full_set _) g) (f None)).
apply Extensionality_Ensembles.
red; split.
red; intros.
destruct H2.
destruct x.
left.
exists t.
constructor.
assumption.
right.
auto with sets.
red; intros.
destruct H2.
destruct H2.
exists (Some x).
constructor.
assumption.
destruct H2.
exists None.
constructor.
reflexivity.
rewrite H2.
constructor.
apply IHFiniteT.
red; intro.
destruct H3.
contradiction (n x).
symmetry; assumption.

pose (g := fun (x:X) => f (f0 x)).
assert (Im (Full_set _) f = Im (Full_set _) g).
apply Extensionality_Ensembles.
red; split.
red; intros.
destruct H2.
destruct H1.
rewrite H3.
rewrite <- H4 with x.
exists (g0 x).
constructor.
unfold g.
reflexivity.
red; intros.
destruct H2.
exists (f0 x).
constructor.
assumption.

rewrite H2.
apply IHFiniteT.
Qed.

(*Schepler*)
Lemma finite_subtype: forall (X:Type) (P:X->Prop),
  FiniteT X -> (forall x:X, P x \/ ~ P x) ->
  FiniteT {x:X | P x}.
Proof.
intros.
induction H.
apply bij_finite with False (False_rect _).
constructor.
exists (@proj1_sig _ _).
destruct x.
intro s; destruct s; destruct x.

destruct (H0 None).
pose (g := fun (x:option {x:T | P (Some x)}) =>
  match x return {x:option T | P x} with
  | Some (exist x0 i) => exist (fun x:option T => P x) (Some x0) i
  | None => exist (fun x:option T => P x) None H1
  end).
apply bij_finite with _ g.
apply add_finite.
apply IHFiniteT.
intro; apply H0.
pose (ginv := fun (s:{x0:option T | P x0}) =>
                       match s return option {x:T | P (Some x)} with
  | exist (Some x0) i => Some (exist (fun y:T => P (Some y)) x0 i)
  | exist None _ => None
  end).
exists ginv.
destruct x as [[x0]|].
simpl.
reflexivity.
simpl.
reflexivity.
destruct y as [[x0|]].
simpl.
reflexivity.
simpl.
destruct (proof_irrelevance _ H1 p).
reflexivity.

pose (g := fun (x:{x:T | P (Some x)}) =>
  match x return {x:option T | P x} with
    | exist x0 i => exist (fun x:option T => P x) (Some x0) i
  end).
apply bij_finite with _ g.
apply IHFiniteT.
intro; apply H0.
pose (ginv := fun s:{x0:option T | P x0} =>
  match s return {x:T | P (Some x)} with
    | exist (Some x0) i => exist (fun x:T => P (Some x)) x0 i
    | exist None i => False_rect _ (H1 i)
  end).
exists ginv.
destruct x; simpl.
reflexivity.
destruct y as [[x0|]].
simpl.
reflexivity.
contradiction H1.

pose (g := fun (x:{x:X | P (f x)}) =>
  match x with
  | exist x0 i => exist (fun x:Y => P x) (f x0) i
        end).
apply bij_finite with _ g.
apply IHFiniteT.
intro; apply H0.
destruct H1.
assert (forall y:Y, P y -> P (f (g0 y))).
intros; rewrite H2; assumption.
pose (ginv := fun (y:{y:Y | P y}) =>
  match y with
  | exist y0 i => exist (fun x:X => P (f x)) (g0 y0) (H3 y0 i)
  end).
exists ginv.
destruct x; simpl.
generalize (H3 (f x) p).
rewrite H1.
intro; destruct (proof_irrelevance _ p p0).
reflexivity.

destruct y; simpl.
generalize (H3 x p).
rewrite H2.
intro; destruct (proof_irrelevance _ p p0).
reflexivity.
Qed.

(*Wyckoff*)                   
Lemma FiniteT_Finite : forall (T:Type), FiniteT T -> 
  (forall (S:Ensemble T), Finite S).
intros T h1 S.
assert (h2: forall x:T, In S x \/ ~In S x).
  intro. tauto.
pose proof (finite_subtype T _ h1 h2) as h3.
assert (h4: forall y1 y2 : T, y1 = y2 \/ y1 <> y2).
  intros. tauto.
pose proof 
  (FiniteT_img (sig_set S) T (@proj1_sig T _) h3 h4) as h5.
assert (h6: (Im (full_sig S) (proj1_sig (P:=fun x : T => In S x)))
  = S).
  apply Extensionality_Ensembles. red. unfold Included.
  split.
  (*left*)
  intros x h7.
  inversion h7 as [y h8 z h9].
  pose proof (proj2_sig y) as h10.
  rewrite <- h9 in h10.
  assumption.
  (*right*)
  intros x h11.
  apply Im_intro with (exist _ x h11).
  apply Full_intro.
  simpl. reflexivity.
unfold full_sig in h6. rewrite h6 in h5.
assumption.
Qed.

(*Wyckoff*)
Lemma FiniteT_sig_Finite : forall {T:Type} (S:Ensemble T), 
  FiniteT (sig_set S) -> Finite S.
intros T S h1.
pose proof (FiniteT_img _ _ (@proj1_sig _ _) h1) as h2.
assert (h3: forall y1 y2 : T, y1 = y2 \/ y1 <> y2).
  intros. apply classic.
apply h2 in h3.
assert (h4:(Im (full_sig S)
  (proj1_sig (P:=fun x : T => In S x))) = 
    S).
  apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red. intros x h4.
  inversion h4 as [? ? ? h5].
  rewrite h5.
  apply proj2_sig.
  (* >= *)
  red. intros x h4.
  eapply Im_intro.
  apply (Full_intro _ (exist _ x h4)).
  reflexivity.
unfold full_sig in h4. unfold sig_set in h4.
rewrite h4 in h3; assumption.
Qed.

(*Schepler*)
Lemma Finite_ens_type: forall {X:Type} (S:Ensemble X),
  Finite S -> FiniteT (sig_set S).
Proof.
intros.
induction H.
apply bij_finite with False (False_rect _).
constructor.
assert (g:{x:X | In (Empty_set _) x}->False).
intro.
destruct X0.
destruct i.
exists g.
destruct x.
destruct y.
destruct g.

assert (Included A (Add A x)).
auto with sets.
assert (In (Add A x) x).
auto with sets.
pose (g := fun (y: option {x:X | In A x}) =>
  match y return {x0:X | In (Add A x) x0} with
                                          | Some (exist y0 i) => exist (fun x2:X => In (Add A x) x2) y0 (H1 y0 i)
  | None => exist (fun x2:X => In (Add A x) x2) x H2
  end).
apply bij_finite with _ g.
apply add_finite.
assumption.

assert (h:forall x0:X, In (Add A x) x0 ->
  { In A x0 } + { x0 = x }).
intros; apply exclusive_dec.
intuition.
destruct H6; auto.
destruct H3.
left; assumption.
right; destruct H3; reflexivity.

pose (ginv := fun s:{x0:X | In (Add A x) x0} =>
  match s return option {x:X | In A x} with
  | exist x0 i => match (h x0 i) with
                  | left iA => Some (exist _ x0 iA)
                  | right _ => None
   end
  end).
exists ginv.
intro; destruct x0.
destruct s.
simpl.
remember (h x0 (H1 x0 i)) as sum; destruct sum.
destruct (proof_irrelevance _ i i0).
reflexivity.
contradiction H0.
rewrite <- e; assumption.
simpl.
remember (h x H2) as sum; destruct sum.
contradiction H0.
reflexivity.
                   
intro.
unfold ginv.
destruct y.
destruct (h x0 i).
simpl.
generalize (H1 x0 i0); intro.
destruct (proof_irrelevance _ i i1).
reflexivity.
simpl.
destruct e.
destruct (proof_irrelevance _ H2 i).
reflexivity.
Qed.

(*Wyckoff*)
Lemma FiniteT_Full : forall T:Type, 
  FiniteT T <->  FiniteT {x:T | In (Full_set T) x}. 
intros T.
pose (fun (a:T) => exist _ a (Full_intro _ a)) as f.
assert (h2:invertible f).
  apply bijective_impl_invertible.
  red. split.
  (*inj*)
  red.
  intros x1 x2 h3.
  unfold f in h3.
  assert (h4: x1 = proj1_sig 
    (exist (In (Full_set T)) x1 (Full_intro T x1))).
    simpl. reflexivity.
  assert (h5: x2 = proj1_sig
   (exist (In (Full_set T)) x2 (Full_intro T x2))).
    simpl. reflexivity.
  congruence.
  (*surj*)
  red.
  intro y.
  exists (proj1_sig y).
  destruct y as [z h1].
  unfold f.
  simpl.
  pose proof 
    (proof_irrelevance _ h1 (Full_intro T z)) as h2.
  rewrite h2. 
  reflexivity.

pose (fun (a:{x:T | In (Full_set T) x}) => proj1_sig a) as f'.
assert (h3:invertible f').
  apply bijective_impl_invertible.
  red. split.
  (*inj*)
  red.
  intros x1 x2 h3.
  unfold f' in h3.
  apply proj1_sig_injective; trivial.
  (*surj*)
  red.
  intro y.
  exists (exist _ y (Full_intro _ y)).
  unfold f'.
  simpl.
  reflexivity.

split.
intro h1.  apply (bij_finite _ _ f);  trivial.
intro h1.  apply (bij_finite _ _ f'); trivial.
Qed.

(*Wyckoff*)
Lemma Finite_FiniteT : forall (T:Type), 
  Finite (Full_set T) -> FiniteT T.
intros T h1.
rewrite FiniteT_Full.
apply Finite_ens_type; trivial.
Qed.

(* Wyckoff *)
Lemma Finite_FiniteT_iff : forall (T:Type), 
  Finite (Full_set T) <-> FiniteT T.
intro T. split. apply Finite_FiniteT.
intro h1.
apply FiniteT_Finite.
assumption.
Qed.


(* Wyckoff *)
Lemma finite_full_sig_iff : 
  forall {T:Type} (A:Ensemble T),
    Finite A <-> Finite (full_sig A).
intros T A.
split.
intro h1.
pose proof (Finite_ens_type _ h1) as h2.
rewrite <- Finite_FiniteT_iff in h2.
assumption.
intro h1.
rewrite Finite_FiniteT_iff in h1.
apply FiniteT_sig_Finite. assumption.
Qed.


(*Wyckoff*)
Lemma finite_subset_sig_iff : forall {T:Type} (A B:Ensemble T) 
                                 (pf:Included A B),
         Finite A <-> Finite (subset_sig _ _ pf).
intros T A B h1.
split.
intro h2.
red in h1.
unfold subset_sig.
apply finite_image.
rewrite <- finite_full_sig_iff.
assumption.
intro h2.  unfold subset_sig in h2. 
assert (h3:FunctionProperties.injective  (fun x : {x : T | In A x} =>
             exist (In B) (proj1_sig x) (h1 (proj1_sig x) (proj2_sig x)))).
  red. intros x1 x2 h3.
  pose proof (exist_injective _ _ _ _ _ h3) as h4.
  apply proj1_sig_injective. assumption.
pose proof (finite_image_rev_inj _ _ h3 h2) as h4.
rewrite finite_full_sig_iff.
assumption.
Qed.

(*Wyckoff*)
Lemma finite_incl_sig_iff :
  forall {T:Type} (A B:Ensemble T) (pf:Included A B),
         Finite (incl_sig A B pf) <->
         Finite A.
intros T A B h1.
rewrite incl_sig_eq_subset_sig.
symmetry.
apply finite_subset_sig_iff.
Qed.


(*Wyckoff*)
Lemma card_fun_full_sig_eq :
  forall {T:Type} (A:Ensemble T) (pf:Finite A),
    card_fun A pf = card_fun (full_sig A) (iff1 (finite_full_sig_iff A) pf).
intros T A h1.
pose proof (card_fun_compat _ h1) as h2.
pose proof (card_fun_compat _ (iff1 (finite_full_sig_iff A) h1)) as h3.
pose proof (proj1_sig_injective (fun x=>In A x)) as h4.
eapply injective_preserves_cardinal. red. apply h4.
apply h3.
pose proof (im_full_sig_proj1_sig A) as h5. 
rewrite <- h5 at 1.
assumption.
Qed.


(*Schepler*)
Lemma finite_eq_dec: forall X:Type, FiniteT X ->
  forall x y:X, {x=y} + {x<>y}.
Proof.
intros.
apply decidable_dec.
induction H.
destruct x.
decide equality.

destruct H0.
case (IHFiniteT (g x) (g y)).
left.
rewrite <- H1.
rewrite <- H1 with x.
rewrite H2.
reflexivity.
right.
contradict H2.
rewrite H2.
reflexivity.
Qed.

(*Schepler*)
Lemma finite_inj_surj: forall (X:Type) (f:X->X),
  FiniteT X -> FunctionProperties.injective f -> surjective f.
Proof.
intros.
induction H.
red.
destruct y.

remember (f None) as f0; destruct f0 as [a|].
assert (forall x:T, f (Some x) <> Some a).
unfold not; intros.
assert (Some x = None).
apply H0.
congruence.
discriminate H2.
pose (g := fun x:T => match f (Some x) with
  | Some y => y
  | None => a
end).
assert (surjective g).
apply IHFiniteT.
red; intros.
remember (f (Some x1)) as fx1; destruct fx1;
remember (f (Some x2)) as fx2; destruct fx2.
unfold g in H2.
rewrite <- Heqfx1 in H2; rewrite <- Heqfx2 in H2.
destruct H2; assert (f (Some x1) = f (Some x2)).
congruence.
apply H0 in H2.
injection H2; trivial.

unfold g in H2; rewrite <- Heqfx1 in H2; rewrite <- Heqfx2 in H2.
destruct H2.
contradiction (H1 x1).
symmetry; assumption.

unfold g in H2; rewrite <- Heqfx1 in H2; rewrite <- Heqfx2 in H2.
destruct H2.
contradiction (H1 x2).
symmetry; assumption.
assert (Some x1 = Some x2).
apply H0.
congruence.
injection H3; trivial.

red; intro.
destruct y.
case (finite_eq_dec _ H t a).
exists None.
congruence.
destruct (H2 t).
exists (Some x).
unfold g in H3.
destruct (f (Some x)).
congruence.
contradiction n.
symmetry; assumption.
destruct (H2 a).
exists (Some x).
unfold g in H3.
remember (f (Some x)) as fx; destruct fx.
destruct H3.
contradiction (H1 x).
symmetry; assumption.
reflexivity.

assert (forall x:T, { y:T | f (Some x) = Some y }).
intros.
remember (f (Some x)) as fx; destruct fx.
exists t; reflexivity.
assert (Some x = None).
apply H0.
congruence.
discriminate H1.
pose (g := fun x:T => proj1_sig (X x)).
assert (surjective g).
apply IHFiniteT.
red; intros.
unfold g in H1.
repeat destruct X in H1.
simpl in H1.
assert (Some x1 = Some x2).
apply H0.
congruence.
injection H2; trivial.

red; intro.
destruct y.
destruct (H1 t).
unfold g in H2; destruct X in H2.
simpl in H2.
exists (Some x).
congruence.
exists None.
symmetry; assumption.

destruct H1.

pose (f' := fun (x:X) => g (f (f0 x))).
assert (surjective f').
apply IHFiniteT.
red; intros.
unfold f' in H3.
assert (f (f0 x1) = f (f0 x2)).
congruence.
apply H0 in H4.
congruence.

red; intro.
destruct (H3 (g y)).
unfold f' in H4.
exists (f0 x).
congruence.
Qed.

(*Schepler *)
Lemma surj_finite: forall (X Y:Type) (f:X->Y),
  FiniteT X -> surjective f ->
  (forall y1 y2:Y, y1=y2 \/ y1<>y2) ->
  FiniteT Y.
Proof.
intros.
apply bij_finite with {y:Y | In (Im (Full_set _) f) y}
  (@proj1_sig _ (fun y:Y => In (Im (Full_set _) f) y)).
apply Finite_ens_type.
apply FiniteT_img.
assumption.
assumption.
assert (forall y:Y, In (Im (Full_set _) f) y).
intro.
destruct (H0 y).
exists x; auto with sets.
constructor.

pose (proj1_sig_inv := fun y:Y =>
  exist (fun y0:Y => In (Im (Full_set _) f) y0) y (H2 y)).
exists proj1_sig_inv.
destruct x.
simpl.
unfold proj1_sig_inv.
destruct (proof_irrelevance _ (H2 x) i); trivial.
intros; simpl; reflexivity.
Qed.

(*Schepler*)
Lemma inj_finite: forall (X Y:Type) (f:X->Y),
  FiniteT Y -> FunctionProperties.injective f ->
  (forall y:Y, (exists x:X, f x = y) \/
               (~ exists x:X, f x = y)) ->
  FiniteT X.
Proof.
intros.
assert (forall y:{y:Y | exists x:X, f x = y}, {x:X | f x = proj1_sig y}).
intro.
destruct y.
simpl.

apply constructive_definite_description.
destruct e.
exists x0.
red; split.
assumption.
intros.
apply H0.
transitivity x.
assumption.
symmetry; assumption.

pose (g := fun y:{y:Y | exists x:X, f x = y} =>
  proj1_sig (X0 y)).
apply bij_finite with _ g.
apply finite_subtype.
assumption.
assumption.

pose (ginv := fun (x:X) => exist (fun y:Y => exists x:X, f x = y)
  (f x) (ex_intro _ x (refl_equal _))).
exists ginv.
destruct x as [y [x e]].
unfold g; simpl.
match goal with |- context [X0 ?arg] => destruct (X0 arg) end.
simpl.
unfold ginv; simpl.
simpl in e0.
repeat match goal with |- context [ex_intro ?f ?x ?e] =>
  generalize (ex_intro f x e) end.
rewrite <- e0.
intros; destruct (proof_irrelevance _ e1 e2).
reflexivity.

intro; unfold ginv.
unfold g; simpl.
match goal with |- context [X0 ?arg] => destruct (X0 arg) end.
simpl.
simpl in e.
auto.
Qed. 

(*Wyckoff*)
Lemma finite_sig_sat_compat :
  forall {T:Type} (P:T->Prop),
    FiniteT {x:T | P x} <-> Finite [x:T | P x].
intros T P. split.
intro h1.
rewrite finite_full_sig_iff. unfold full_sig. unfold sig_set.
rewrite Finite_FiniteT_iff.
pose (fun a:{x:T| In [x0:T | P x0] x} => 
        (exist _ (proj1_sig a) 
               (match (proj2_sig a) with 
                  |intro_characteristic_sat pf => pf
                end))) as f.
assert (h2:FunctionProperties.injective f).
  red. intros a b h2. destruct a; destruct b.
  destruct i; destruct i0. unfold f in h2. simpl in h2.
  apply proj1_sig_injective. simpl.
  apply exist_injective in h2. assumption.
eapply inj_finite; auto. apply h1. apply h2. intros; apply classic.  
intro h2.
rewrite finite_full_sig_iff in h2. unfold full_sig in h2. unfold sig_set in h2.
rewrite Finite_FiniteT_iff in h2.
pose (fun a:{x:T|P x} => 
        exist _ (proj1_sig a)
              (intro_characteristic_sat _ _ (proj2_sig a))) as f.
assert (h1:FunctionProperties.injective f).
  red. intros a b h1. destruct a; destruct b. unfold f in h1.
  apply exist_injective in h1. simpl in h1.
  apply proj1_sig_injective. simpl. assumption.
eapply inj_finite; auto. apply h2. apply h1. intros; apply classic.
Qed.


(* Wyckoff *)
Lemma cart_prod_fin_comm : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U),
          Finite (cart_prod A B) ->
          Finite (cart_prod B A).
intros T U A B h1.
apply Finite_ens_type in h1.
pose (fun pr:{p:U*T | In (cart_prod B A) p} => 
        exist _ (snd (proj1_sig pr), fst (proj1_sig pr)) 
              (in_cart_prod_comm B A (proj1_sig pr)
               (proj2_sig pr))) as f.
assert (h2:FunctionProperties.injective f).
  red. unfold f.
  intros pr1 pr2 h2.
  destruct pr1 as [p1 h3].
  destruct pr2 as [p2 h4].
  simpl in h2.
  apply existTexist.
  apply subsetT_eq_compat.
  inversion h2 as [h5].
  rewrite surjective_pairing.
  rewrite <- h5.
  rewrite <- H.
  apply surjective_pairing.
apply FiniteT_sig_Finite.
apply (inj_finite _ _ f h1 h2).
intro pr.
destruct pr as [p h4].
unfold f.
left.
exists (exist (fun p:U*T => In (cart_prod B A) p) (snd p, fst p) (in_cart_prod_comm A B p h4)).
simpl.
apply existTexist.
apply subsetT_eq_compat.
rewrite surjective_pairing.
reflexivity.
Qed.



(*Wyckoff*)
Lemma FiniteT_Finite_Union : forall {It T:Type} 
  (F:IndexedFamily It T),   (forall i:It, Finite (F i)) 
  -> FiniteT It -> Finite (IndexedUnion F).
Proof.
intros It T F h1 h2.
induction h2.
(*Empty*)
assert (h3:(IndexedUnion F) = Empty_set _ ).
  apply Extensionality_Ensembles.
  red. split. 
  (* <= *)
  red. intros x h4.
  inversion h4.
  contradiction.
  (* >= *)
  red. intros; contradiction.
rewrite h3. constructor.
(*Add*)
rename T0 into It.
pose (fun (i:It) => F (Some i)) as f.
assert (h4:(IndexedUnion F) = Union (F None) (IndexedUnion f)).
  apply Extensionality_Ensembles.
  red. split.
  (*<=*)
  red. intros x h5.   
  inversion h5 as [i ? h6].
  destruct i.
    (*Some*)
    right.
    unfold f.
    apply indexed_union_intro with i. assumption.
    (*None*)
    constructor. assumption.
  (*>=*)
  red. intros x h5.
  pose proof (Union_inv _ _ _ x h5) as h6.
  case h6 as [h7 | h8].
    apply indexed_union_intro with None; assumption.
  inversion h8.
    apply indexed_union_intro with (Some i); assumption.
assert (h5: forall i:It, Finite (f i)).
  intro i.
  apply (h1 (Some i)).
pose proof (IHh2 f h5) as h9.
assert (h10: Finite (F None)).
  apply (h1 None).
rewrite h4.
apply Union_preserves_Finite; assumption.
(*Bij*)
rename Y into It.
rename H into h3.
pose (fun (x:X) => F (f x)) as Ff.
assert (h4: forall x:X, Finite (Ff x)).
  intro x.
  apply (h1 (f x)).
pose proof (IHh2 Ff h4) as h5.
assert (h6:IndexedUnion Ff = IndexedUnion F).
  apply Extensionality_Ensembles.
  red. split.
  (*<=*)
  red.
  intros x h7.
  inversion h7 as [? ? h8].
  unfold Ff in h8.
  apply indexed_union_intro with (f i); assumption.
  (*>=*)  
  pose proof (invertible_impl_bijective _ h3) as h6.
  red. intros t h7.
  inversion h7 as [i y h8].
  unfold bijective in h6.
  destruct h6 as [h9 h10].
  red in h10.
  pose proof (h10 i) as h11.
  elim h11. intros x h12.
  rewrite <- h12 in h8.
  unfold Ff.
  apply indexed_union_intro with x; assumption.
rewrite <- h6.
assumption.
Qed.

(*Wyckoff*)
Lemma bool_finite : FiniteT bool.
Proof.
pose (fun x:(option (option False)) => 
        match x with
          | None => false
          | Some y => match y with 
                          | None => true
                          | Some z => (False_rec bool z)
                      end
        end) as f.
apply (bij_finite _ _ f); repeat constructor.
apply bijective_impl_invertible.
red. split.
(*injective*)
red.
intros a b h1.
unfold f in h1.
destruct a as [a2 | a3]; destruct b as [b2 | b3].
destruct a2 as [a2a | a2b]; destruct b2 as [b2a | b2b]; 
try contradiction; try reflexivity.
destruct a2 as [a2a | a2b]; try contradiction; try discriminate.
destruct b2 as [b2a | b2b]; try contradiction; try discriminate.
reflexivity.
(*surjective*)
red.
intro x.
unfold f.
destruct x. exists (Some None). reflexivity.
exists None. reflexivity.
Qed.






(*Schepler*)  
Lemma finite_dep_choice: forall (A:Type) (B:forall x:A, Type)
  (R:forall x:A, B x->Prop),
  FiniteT A -> (forall x:A, exists y:B x, R x y) ->
  exists f:(forall x:A, B x), forall x:A, R x (f x).
Proof.
intros.
revert B R H0.
induction H.
intros.
exists (fun x:False => False_rect (B x) x).
destruct x.
intros.
pose proof (IHFiniteT (fun x:T => B (Some x))
  (fun x:T => R (Some x))
  (fun x:T => H0 (Some x))).
destruct H1.
pose proof (H0 None).
destruct H2.
exists (fun y:option T =>
  match y return (B y) with
  | Some y0 => x y0
  | None => x0
  end).
destruct x1.
apply H1.
assumption.

intros.
destruct H0.
pose proof (IHFiniteT (fun x:X => B (f x))
  (fun x:X => R (f x))
  (fun x:X => H1 (f x))).
destruct H3.
pose (f0 := fun y:Y => x (g y)).
pose (conv := fun (y:Y) (a:B (f (g y))) =>
  eq_rect (f (g y)) B a y (H2 y)).

exists (fun y:Y => conv y (x (g y))).
intro.
unfold conv; simpl.
generalize (H2 x0).
pattern x0 at 2 3 6.
rewrite <- H2.
intro.
rewrite <- eq_rect_eq.
apply H3.
Qed.

(*Schepler*)
Lemma finite_choice : forall (A B:Type) (R:A->B->Prop),
  FiniteT A -> (forall x:A, exists y:B, R x y) ->
  exists f:A->B, forall x:A, R x (f x).
Proof.
intros.
apply finite_dep_choice.
assumption.
assumption.
Qed.

(*Schepler*)
Lemma finite_sum: forall X Y:Type, FiniteT X -> FiniteT Y ->
  FiniteT (X+Y).
Proof.
intros.
induction H0.
apply bij_finite with _ inl.
assumption.
pose (g := fun (x:X+False) => match x with
  | inl x => x
  | inr f => False_rect X f
end).
exists g.
intro; simpl.
reflexivity.
destruct y.
simpl.
reflexivity.
destruct f.

pose (g := fun (x:option (X+T)) => match x with
  | Some (inl x) => inl _ x
  | Some (inr t) => inr _ (Some t)
  | None => inr _ None
  end).
apply bij_finite with _ g.
apply add_finite.
assumption.
pose (ginv := fun (x:X + option T) => match x with
  | inl x => Some (inl _ x)
  | inr (Some t) => Some (inr _ t)
  | inr None => None
  end).
exists ginv.
destruct x as [[x|t]|]; trivial.
destruct y as [x|[t|]]; trivial.

pose (g := fun (x:X+X0) => match x with
  | inl x0 => inl _ x0
  | inr x0 => inr _ (f x0)
  end).
destruct H1.
pose (ginv := fun (x:X+Y) => match x with
  | inl x0 => inl _ x0
  | inr y0 => inr _ (g0 y0)
  end).
apply bij_finite with _ g.
assumption.
exists ginv.
destruct x as [x0|x0]; trivial.
simpl.
rewrite H1; reflexivity.
destruct y as [x|y0]; trivial.
simpl.
rewrite H2; reflexivity.
Qed.


(*Schepler*)
Lemma finite_prod: forall (X Y:Type), FiniteT X -> FiniteT Y ->
  FiniteT (X*Y).
Proof.
intros.
induction H0.
apply bij_finite with _ (False_rect _).
constructor.
exists (@snd X False).
destruct x.
destruct y.
destruct f.

pose (g := fun (x:X*T + X) => match x with
  | inl (pair x0 t) => pair x0 (Some t)
  | inr x0 => pair x0 None
  end).
pose (ginv := fun (x:X * option T) => match x with
  | (x0, Some t) => inl _ (x0, t)
  | (x0, None) => inr _ x0
  end).
apply bij_finite with _ g.
apply finite_sum.
assumption.
assumption.
exists ginv.
destruct x as [[x0 t]|x0]; trivial.
destruct y as [x0 [t|]]; trivial.

pose (g := fun (y:X*X0) => match y with
  | pair x x0 => pair x (f x0)
  end).
destruct H1.
pose (ginv := fun (y:X*Y) => let (x,y0) := y in
  (x, g0 y0)).
apply bij_finite with _ g.
assumption.
exists ginv.
destruct x as [x x0]; unfold ginv, g; try rewrite H1; trivial.
destruct y as [x y]; unfold ginv, g; try rewrite H2; trivial.
Qed.



(*Schepler*)
Require Import FunctionalExtensionality.
Lemma finite_exp: forall X Y:Type, FiniteT X -> FiniteT Y ->
  FiniteT (X->Y).
Proof.
intros.
induction H.
pose (g := fun (x:True) (f:False) => False_rect Y f).
pose (ginv := fun (_:False->Y) => I).
apply bij_finite with _ g.
apply True_finite.
exists ginv.
destruct x as [].
trivial.
intro y.
apply functional_extensionality.
(*intro; extensionality f.*)
intro; contradiction.
(*destruct f.*)

pose (g := fun (p:(T->Y)*Y) (x:option T) =>
  let (f,y0) := p in
  match x with
  | Some x0 => f x0
  | None => y0
  end).
pose (ginv := fun (f:option T->Y) =>
  (fun x:T => f (Some x), f None)).
apply bij_finite with _ g.
apply finite_prod.
assumption.
assumption.
exists ginv.
destruct x as [f y0]; try extensionality t;
try destruct t as [t0|]; trivial.
intro.
extensionality t; destruct t as [t0|]; trivial.

destruct H1.
pose (g0 := fun (h:X->Y) (y:Y0) => h (g y)).
apply bij_finite with _ g0.
assumption.
pose (g0inv := fun (h:Y0->Y) (x:X) => h (f x)).
exists g0inv.
intro.
extensionality x0; unfold g0; unfold g0inv; simpl.
rewrite H1; reflexivity.
intro.
extensionality y0; unfold g0; unfold g0inv; simpl.
rewrite H2; reflexivity.
Qed.


(*Wyckoff*)
(*Uses excluded middle*)

Lemma power_set_finitet : forall {T:Type} (A:Ensemble T),
                           Finite A -> 
                           FiniteT {S:Ensemble T | Included S A}.
intros T A h1.
assert (h2: (forall (S:Ensemble T) (x:T), 
                              In A x -> {In S x} + {~In S x})).
  intros; apply classic_dec.
pose (fun S:{S':(Ensemble T) | Included S' A} => 
        (fun x:{t:T | In A t} => 
           if (h2 (proj1_sig S) (proj1_sig x) (proj2_sig x)) 
           then true 
           else false)) as f.
pose proof (Finite_ens_type _ h1) as h3.
pose proof bool_finite as h4.
pose proof (finite_exp _ _ h3 h4) as h5.
apply (inj_finite _ _ f). assumption.
red.
intros S1 S2 h6.
apply proj1_sig_injective.
apply Extensionality_Ensembles.
red. split.
  (* -> *)
  red.
  intros x h7.
  pose proof (proj2_sig S1) as h11.
  simpl in h11.
  assert (h12:In A x). auto with sets.
  pose (exist _ _ h12) as x'.
  assert (h13: f S1 x' = f S2 x').  rewrite h6. reflexivity.
  unfold f in h13.
  pose proof (conditional_correspondence _ _ _ _ h13) as h14.
  simpl in h14.
  tauto.
  (* <- *)
  red.
  intros x h7.
  pose proof (proj2_sig S2) as h11.
  simpl in h11.
  assert (h12:In A x). auto with sets.
  pose (exist _ _ h12) as x'.
  assert (h13: f S1 x' = f S2 x').  rewrite h6. reflexivity.
  unfold f in h13.
  pose proof (conditional_correspondence _ _ _ _ h13) as h14.
  simpl in h14.
  tauto.

intros; tauto.
Qed.

(*Wyckoff*)
Lemma power_set_finite : 
  forall {T:Type} (A:Ensemble T),
    Finite A -> Finite (power_set A).
intros T A h1.
pose proof (power_set_finitet A h1) as h2.
unfold power_set.
rewrite <- finite_sig_sat_compat.
assumption.
Qed.


(*move to SetUtilities2*)
Require Import Infinite_sets.
(*Wyckoff*)
Lemma Finite_Finite_Union_rev : 
  forall {T:Type} (F:Family T), 
    Finite (FamilyUnion F) -> Finite F.
intros T F h2.
apply NNPP.
intro h3.
assert (h4:Approximant _ F [S:Ensemble T | Ensembles.In F S /\ Included S (FamilyUnion F)]).
constructor.
eapply Finite_downward_closed.
eapply power_set_finite. apply h2.
red. intros A h4.
destruct h4 as [h4].
destruct h4 as [h4l h4r].
constructor. assumption.
red. intros S h4.
destruct h4 as [h4]. destruct h4; auto.
pose proof (make_new_approximant _ _ _ h3 h4) as h5.
assert (h6: (Setminus F [S : Ensemble T | In F S /\ Included S (FamilyUnion F)]) = Empty_set _).
  apply Extensionality_Ensembles; red; split; auto with sets.
  red. intros A h6.
  destruct h6 as [h6 h7].
  contradict h7.
  constructor. split; auto.
  red. intros x h7. apply family_union_intro with A; auto.
rewrite h6 in h5.
destruct h5. contradiction.
Qed.


End FiniteTypes.

(*Whole section copied and pasted from file of same name in Schepler's Zorn's Lemma.*)
Section InfiniteTypes.

(*Schepler*)
Lemma finite_nat_initial_segment: forall n:nat,
  FiniteT { m:nat | m < n }.
Proof.
intros.  
apply Finite_ens_type.
rewrite <- characteristic_function_to_ensemble_is_identity.
induction n.
assert ([x:nat | x < 0] = (Empty_set _)).
apply Extensionality_Ensembles; split; auto with sets.
red; intros.
destruct H.
contradict H.
auto with arith. 
rewrite H; constructor.

assert ([x:nat | S x <= S n] = Add [x:nat | x < n] n).
apply Extensionality_Ensembles; split.
red; intros.
destruct H.
assert (x <= n); auto with arith. 
apply le_lt_or_eq in H0.
case H0. 
left; constructor; trivial.
right; auto with sets.
red; intros.
case H.
intros.
destruct H0; constructor.
auto with arith.
intros.
destruct H0.
constructor.
auto with arith.
assert (h1:forall x, S x <= S n <-> x < S n). intros; tauto; trivial with arith.
rewrite (sat_iff _ _ h1) in H.
rewrite H; constructor; trivial.
red; intro.
destruct H0.
contradict H0.
auto with arith.
Qed.

(*Schepler*)
Lemma nat_infinite: ~ FiniteT nat.
Proof.
red; intro.
assert (surjective S).
apply finite_inj_surj; trivial.
red; intros.
injection H0; trivial.

destruct (H0 0).
discriminate H1.
Qed.

(*Schepler*)
Lemma infinite_nat_inj: forall X:Type, ~ FiniteT X ->
  exists f:nat->X, FunctionProperties.injective f.
Proof.
intros.
assert (inhabited (forall S:Ensemble X, Finite S ->
  { x:X | ~ In S x})).
pose proof (choice (fun (x:{S:Ensemble X | Finite S}) (y:X) =>
  ~ In (proj1_sig x) y)).
simpl in H0.
match type of H0 with | ?A -> ?B => assert B end.
apply H0.
intros.
apply NNPP.
red; intro.
pose proof (not_ex_not_all _ _ H1); clear H1.
destruct x.
assert (x = (Full_set _)).
apply Extensionality_Ensembles; red; split; auto with sets.
intro x0; constructor.
symmetry in H1; destruct H1.
contradiction H.
clear H2.
apply bij_finite with (f:=@proj1_sig _ (fun x:X => In (Full_set _) x)).
apply Finite_ens_type; assumption.
exists (fun x:X => exist _ x (Full_intro _ x)).
destruct x; simpl.
generalize (Full_intro X x).
intro i0; destruct (proof_irrelevance _ i i0); trivial.
trivial.
clear H0.
destruct H1.
exists.
intros.
exists (x (exist _ S H1)).
exact (H0 (exist _ S H1)).
destruct H0.
 
assert (forall (n:nat) (g:forall m:nat, m<n -> X),
  { x:X | forall (m:nat) (Hlt:m<n), g m Hlt <> x }).
intros.
assert (Finite (fun x:X => exists m:nat, exists Hlt:m<n,
           g m Hlt = x)).
pose (h := fun x:{m:nat | m<n} =>
  g (proj1_sig x) (proj2_sig x)).

match goal with |- Finite ?S => assert (S =
  Im (Full_set _) h) end.
apply Extensionality_Ensembles; red; split; red; intros.
destruct H0.
destruct H0.
exists (exist (fun m:nat => m < n) x0 x1).
constructor.
unfold h; simpl.
symmetry; assumption.
destruct H0.
destruct x.
unfold h in H1; simpl in H1.
exists x; exists l; symmetry; assumption.

rewrite H0; apply FiniteT_img.
apply finite_nat_initial_segment.
intros; apply classic.

destruct (X0 _ H0).
unfold In in n0.
exists x.
intros; red; intro.
contradiction n0; exists m; exists Hlt; exact H1.

pose (f := Fix lt_wf (fun n:nat => X)
  (fun (n:nat) (g:forall m:nat, m<n->X) => proj1_sig (X1 n g))).
simpl in f.
assert (forall n m:nat, m<n -> f m <> f n).
pose proof (Fix_eq lt_wf (fun n:nat => X)
  (fun (n:nat) (g:forall m:nat, m<n->X) => proj1_sig (X1 n g))).
fold f in H0.
simpl in H0.
match type of H0 with | ?A -> ?B => assert (B) end.
apply H0.
intros.
assert (f0 = g).
Require Import FunctionalExtensionality.
extensionality y; extensionality p; apply H1.
destruct H2; trivial.
intros.
pose proof (H1 n).
destruct X1 in H3.
simpl in H3.
destruct H3.
auto.

exists f.
red; intros m n ?.
destruct (lt_eq_lt_dec m n) as [[Hlt|Heq]|Hlt]; trivial.
contradiction (H0 n m).
contradiction (H0 m n).
symmetry; assumption.
Qed.


Lemma inf_comp_fin : forall {T:Type}, ~FiniteT T -> 
  forall U:Ensemble T, Finite (Ensembles.Complement U) -> 
    ~ Finite U.
intros T h0 U h1.
case (classic (Finite U)); trivial.
intro h2.
pose proof (Union_preserves_Finite T U (Ensembles.Complement U) h2 h1) as h3.
rewrite excl_middle_full in h3.
pose proof (Finite_ens_type (Full_set T) h3) as h4.
pose (fun (a:{x:T | In (Full_set T) x}) => proj1_sig a) as f.
assert (h5:invertible f).
  apply bijective_impl_invertible.
  red. split.
  (*left*)
  red.
  intros x1 x2 h6.
  unfold f in h6.
  apply proj1_sig_injective; trivial.
  (*right*)
  red.
  intro y.
  exists (exist _ y (Full_intro _ y)).
  unfold f.
  simpl.
  reflexivity.
pose proof (bij_finite _ _ f h4 h5). 
contradiction.
Qed.

End InfiniteTypes.


Section CountableT.

(*Schepler*)
Inductive CountableT (X:Type) : Prop :=
  | intro_nat_injection: forall f:X->nat, FunctionProperties.injective f -> CountableT X.

(*Schepler*)
Lemma CountableT_is_FiniteT_or_countably_infinite:
  forall X:Type, CountableT X ->
      {FiniteT X} + {exists f:X->nat, bijective f}.
Proof.
intros.
apply exclusive_dec.
red; intro.
destruct H0 as [? [f ?]].
contradiction nat_infinite.
apply bij_finite with _ f; trivial.
apply bijective_impl_invertible; trivial.

case (classic (FiniteT X)).
left; trivial.
right.
apply infinite_nat_inj in H0.
destruct H.
destruct H0 as [g].
apply CSB with f g; trivial.
Qed.

(*Schepler*)
Lemma countable_nat_product: CountableT (nat*nat).
Proof.
pose (sum_1_to_n := fix sum_1_to_n n:nat := match n with
  | O => O
  | S m => (sum_1_to_n m) + n
end).
exists (fun p:nat*nat => let (m,n):=p in
  (sum_1_to_n (m+n)) + n).
assert (forall m n:nat, m<n ->
  sum_1_to_n m + m < sum_1_to_n n).
intros.
induction H.
simpl.
auto with arith.
apply lt_trans with (sum_1_to_n m0).
assumption.
simpl.
assert (0 < S m0); auto with arith.
assert (sum_1_to_n m0 + 0 < sum_1_to_n m0 + S m0); auto with arith.
assert (sum_1_to_n m0 + 0 = sum_1_to_n m0); auto with arith.
rewrite H2 in H1; assumption.

red; intros.
destruct x1 as [x1 y1].
destruct x2 as [x2 y2].
Require Import Compare_dec.
case (lt_eq_lt_dec (x1+y1) (x2+y2)); intro.
case s; intro.
assert (sum_1_to_n (x1+y1) + y1 < sum_1_to_n (x2+y2) + y2).
apply le_lt_trans with (sum_1_to_n (x1+y1) + (x1+y1)).
assert (sum_1_to_n (x1+y1) + (x1+y1) =
  (sum_1_to_n (x1+y1) + y1) + x1).
Require Import ArithRing.
ring.
auto with arith.
apply lt_le_trans with (sum_1_to_n (x2+y2)).
apply H; trivial.
auto with arith.
rewrite H0 in H1.
contradict H1.
auto with arith.

assert (y1=y2).
rewrite e in H0.
Require Import Arith.
apply plus_reg_l in H0.
assumption.
f_equal; trivial.
rewrite H1 in e.
rewrite plus_comm in e.
rewrite (plus_comm x2 y2) in e.
apply plus_reg_l in e.
assumption.

assert (sum_1_to_n (x2+y2) + y2 < sum_1_to_n (x1+y1) + y1).
apply le_lt_trans with (sum_1_to_n (x2+y2) + (x2+y2)).
auto with arith.
apply lt_le_trans with (sum_1_to_n (x1+y1)); auto with arith.
rewrite H0 in H1.
contradict H1.
auto with arith.
Qed.

(*Schepler*)
Lemma countable_sum: forall X Y:Type,
  CountableT X -> CountableT Y -> CountableT (X+Y).
Proof.
intros.
destruct H as [f].
destruct H0 as [g].
destruct countable_nat_product as [h].
exists (fun s:X+Y => match s with
  | inl x => h (0, f x)
  | inr y => h (1, g y)
end).
red; intros s1 s2 ?.
destruct s1 as [x1|y1]; destruct s2 as [x2|y2];
  apply H1 in H2; try discriminate H2;
  intros; f_equal; (apply H || apply H0); injection H2; trivial.
Qed.

(*Schepler*)
Lemma countable_product: forall X Y:Type,
  CountableT X -> CountableT Y -> CountableT (X*Y).
Proof.
intros.
destruct H as [f].
destruct H0 as [g].
pose (fg := fun (p:X*Y) => let (x,y):=p in (f x, g y)).
destruct countable_nat_product as [h].
exists (fun p:X*Y => h (fg p)).
red; intros.
apply H1 in H2.
destruct x1 as [x1 y1].
destruct x2 as [x2 y2].
unfold fg in H2.
injection H2; intros.
apply H0 in H3.
apply H in H4.
f_equal; trivial.
Qed.

Require Import FunctionalExtensionality.

(*Schepler*)
Lemma countable_exp: forall X Y:Type,
  FiniteT X -> CountableT Y -> CountableT (X->Y).
Proof.
intros.
induction H.
exists (fun _ => 0).
red; intros.
extensionality f.
destruct f.

destruct (countable_product (T->Y) Y); trivial.

exists (fun (g:option T->Y) =>
  f (fun x:T => g (Some x), g None)).
red; intros g1 g2 ?.
apply H1 in H2.
extensionality o.
destruct o.
injection H2; intros.
pose proof (equal_f H4).
simpl in H5.
apply H5.
injection H2; trivial.

destruct H1.
destruct IHFiniteT.
exists (fun (h:Y0->Y) => f0 (fun x:X => h (f x))).
red; intros h1 h2 ?.
apply H3 in H4.
pose proof (equal_f H4).
simpl in H5.
extensionality y.
rewrite <- (H2 y).
apply H5.
Qed.

(*Schepler*)
Definition Countable {X:Type} (S:Ensemble X) : Prop :=
  CountableT {x:X | In S x}.

(*Schepler*)
Lemma inj_countable: forall {X Y:Type} (f:X->Y),
  CountableT Y -> FunctionProperties.injective f -> CountableT X.
Proof.
intros.
destruct H as [g].
exists (fun x:X => g (f x)).
red; intros; auto.
Qed.

(*Schepler*)
Lemma surj_countable: forall {X Y:Type} (f:X->Y),
  CountableT X -> surjective f -> CountableT Y.
Proof.
intros.
Require Import ClassicalChoice.

pose proof (choice (fun (y:Y) (x:X) => f x = y)).
destruct H1 as [finv].
exact H0.

apply inj_countable with finv.
assumption.
red; intros.
congruence.
Qed.

(*Schepler*)
Lemma countable_downward_closed: forall {X:Type} (S T:Ensemble X),
  Countable T -> Included S T -> Countable S.
Proof.
intros.
destruct H.
exists (fun x:{x:X | In S x} => match x with
  | exist x0 i => f (exist _ x0 (H0 _ i))
  end).
red; intros.
destruct x1 as [x1].
destruct x2 as [x2].
apply H in H1.
injection H1; intros.
destruct H2.
destruct (proof_irrelevance _ i i0).
trivial.
Qed.

(*Schepler*)
Lemma countable_img: forall {X Y:Type} (f:X->Y) (S:Ensemble X),
  Countable S -> Countable (Im S f).
Proof.
intros.
assert (forall x:X, In S x -> In (Im S f) (f x)).
auto with sets.
pose (fS := fun x:{x:X | In S x} =>
  match x return {y:Y | In (Im S f) y} with
  | exist x0 i => exist _ (f x0) (H0 x0 i)
  end).
apply surj_countable with fS; trivial.
red; intros.
destruct y.
destruct i.
exists (exist _ x i).
simpl.
generalize (H0 x i); intro.
generalize (Im_intro X Y S f x i y e); intro.
destruct e.
destruct (proof_irrelevance _ i0 i1).
trivial.
Qed.

(*slightly modified from Schepler*)
Lemma countable_type_ensemble: forall {X:Type} (S:Ensemble X),
  CountableT X -> Countable S.
Proof.
intros.
red.
apply inj_countable with (@proj1_sig _ (fun x:X => In S x)).
assumption.
red.
intros a1 a2 h1.
destruct a1. destruct a2.
apply existTexist.
apply subsetT_eq_compat.
simpl in h1. assumption.
Qed.

(*Schepler*)
Lemma FiniteT_impl_CountableT: forall X:Type,
  FiniteT X -> CountableT X.
Proof.
intros.
induction H.
exists (False_rect nat).
red; intros.
destruct x1.
destruct IHFiniteT.
exists (fun x:option T => match x with
  | Some x0 => S (f x0)
  | None => 0
end).
red; intros.
destruct x1; destruct x2; try (injection H1 || discriminate H1); trivial.
intro.
apply H0 in H2.
destruct H2; trivial.

destruct IHFiniteT as [g].
destruct H0 as [finv].
exists (fun y:Y => g (finv y)).
red; intros y1 y2 ?.
apply H1 in H3.
congruence.
Qed.

(*Schepler*)
Lemma Finite_impl_Countable: forall {X:Type} (S:Ensemble X),
  Finite S -> Countable S.
Proof.
intros.
apply FiniteT_impl_CountableT.
apply Finite_ens_type; trivial.
Qed.

(* Schepler *)
Lemma countable_union: forall {X A:Type}
  (F:IndexedFamily A X), CountableT A ->
    (forall a:A, Countable (F a)) ->
    Countable (IndexedUnion F).
Proof.
intros.
destruct (choice_on_dependent_type (fun (a:A)
                               (f:{x:X | In (F a) x} -> nat) =>
  FunctionProperties.injective f)) as [choice_fun_inj].
intro.
destruct (H0 a).
exists f; trivial.

destruct (choice (fun (x:{x:X | In (IndexedUnion F) x}) (a:A) =>
  In (F a) (proj1_sig x))) as [choice_fun_a].
destruct x as [x [a]].
exists a.
assumption.

destruct countable_nat_product as [g].
destruct H as [h].
exists (fun x:{x:X | In (IndexedUnion F) x} =>
  g (h (choice_fun_a x), choice_fun_inj (choice_fun_a x)
                                   (exist _ (proj1_sig x) (H2 x)))).
red; intros.
apply H3 in H4.
injection H4; intros.
apply H in H6.
revert H5.
generalize (H2 x1).
generalize (H2 x2).
rewrite H6.
intros.
apply H1 in H5.
injection H5.
apply proj1_sig_injective.
Qed.


(*Wyckoff*)
Lemma countable_union2 : forall (T:Type) (S1 S2:Ensemble T), 
  Countable S1 -> Countable S2 -> Countable (Union S1 S2).
intros T S1 S2 h1 h2.
pose (option (option False)) as twoset.
pose (fun (t:twoset) => 
  match t with | None => S1 | Some _ => S2 end) as F.
assert (h3:FiniteT twoset).
  apply add_finite. apply add_finite. apply empty_finite.
pose proof (FiniteT_impl_CountableT _ h3) as h4.
assert (h5: forall t:twoset, Countable (F t)).
  destruct t; simpl; assumption.
pose proof (countable_union F h4 h5) as h6.
assert (h7:IndexedUnion F =  Union S1 S2).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h7. 
  (* <= *) 
  inversion h7 as [i t h8].
  destruct i. simpl in h8. apply Union_intror; assumption.
  simpl in h8. apply Union_introl; assumption.
  (* >= *)
  red. intros x h7.
  pose proof (Union_inv _ _ _ _ h7) as h8.
  case h8. 
    unfold F. intro h9. 
      apply indexed_union_intro with None; assumption.
    unfold F. intro h9. 
      apply indexed_union_intro with (Some None); assumption.
rewrite h7 in h6. assumption.
Qed.
   


(*Wyckoff*)
(*A leftover from before I wrote previous lemma, included for
  conservation's sake.*)
Lemma countable_union2': forall (T:Type) (S1 S2:Ensemble T), 
  Countable S1 -> Countable S2 -> Countable (Union S1 S2).
intros T S1 S2 h1 h2.
red in h1. red in h2.
pose proof (countable_sum _ _ h1 h2) as h3.
red.
pose (fun s:{x:T | In S1 x} + {x:T | In S2 x} => match s with
  | inl x => (exist _ (proj1_sig x) (Union_introl _ S1 S2 (proj1_sig x) (proj2_sig x)))
  | inr y => (exist _ (proj1_sig y) (Union_intror _ S1 S2 (proj1_sig y) (proj2_sig y)))
end) as f.
apply (surj_countable f). assumption.
red.
intros y.
destruct y as [z h4].
pose proof (Union_inv _ _ _ z h4) as h5.
case h5.
  (*left*)
  intro h6.
  exists (inl (exist _ z h6)).
  simpl.
  pose proof (proof_irrelevance _ h4 (Union_introl T S1 S2 z h6)) as h7.
  rewrite h7. reflexivity.
  (*right*)
  intro h8.
  exists (inr (exist _ z h8)).
  simpl.
  pose proof (proof_irrelevance _ h4 (Union_intror T S1 S2 z h8)) as h9.
  rewrite h9. reflexivity.
Qed.

(*Wyckoff*)
Lemma countable_intersection: forall (T:Type) (S1 S2:Ensemble T), 
  Countable S1 -> Countable S2 -> Countable (Intersection S1 S2).
intros T S1 S2 h1 h2.
apply (countable_downward_closed (Intersection S1 S2) S1).
assumption.
auto with sets.
Qed.

(*Schepler*)
Lemma Countable_Full : forall {T:Type}, 
  CountableT T->Countable (Full_set T).
intros T h1.
red.
pose (fun (x:T) => (exist _ x (Full_intro _ x))) as f.
apply surj_countable with f. assumption.
red. intro y.
exists (proj1_sig y).
unfold f.
destruct y.
simpl.
pose proof (proof_irrelevance _ i (Full_intro T x)) as h2.
rewrite h2. reflexivity.
Qed.

(*Schepler*)
Lemma Countable_Sig : forall {T:Type} (P:T->Prop), CountableT T -> 
  CountableT {t:T|P t}.
intros T P h1.
pose [t:T|P t] as S.
pose proof (countable_downward_closed S (Full_set _)) as h2.
pose proof (Countable_Full h1) as h3.
assert (h4:Included S (Full_set _)).
  red. intros. constructor.
pose proof (h2 h3 h4) as h5.
red in h5.
assert (h6:forall x:T, In S x <-> P x).
  intro x.
  split.
  (*->*)
  intro h7.
  inversion h7. assumption.
  (*<-*)
  intro h8.
  constructor. assumption.
assert (h7:forall x:T, In S x -> P x).
  intros x h8.
  rewrite h6 in h8. assumption.

pose (fun a:{x:T | In S x} => 
  exist _ (proj1_sig a) (h7 (proj1_sig a) (proj2_sig a))) as f.
apply surj_countable with f.
assumption.

red.
intros y.
destruct y as [t h8].
assert (h9:In S t).
  rewrite h6. assumption.
exists (exist _ t (h9)).
unfold f. simpl.
pose proof (proof_irrelevance _ (h7 t h9) h8) as h10.
rewrite h10. reflexivity.
Qed.

(*Wyckoff*)
Corollary CountableT_Countable : forall (T:Type), CountableT T -> 
  (forall (S:Ensemble T), Countable S).
intros T h1 S.
red.
apply Countable_Sig; assumption.
Qed.

End CountableT.

Section FiniteT_nat_cardinal.

(*Schepler*)
Lemma FiniteT_has_nat_cardinal: forall X:Type, FiniteT X ->
  exists! n:nat, cardinal _ (@Full_set X) n.
Proof.
intros.
apply -> unique_existence; split.
apply finite_cardinal.
pose (idX := fun x:X => x).
assert (Im (Full_set _) idX = Full_set _).
apply Extensionality_Ensembles.
red; split.
red; intros; constructor.
red; intros.
exists x.
constructor.
trivial.

rewrite <- H0.
apply FiniteT_img with (f:=fun x:X => x).
assumption.
intros.
case (finite_eq_dec X H y1 y2); tauto.

red; intros.
apply cardinal_unicity with X (Full_set _); trivial.
Qed.


(*Schepler*)
Definition FiniteT_nat_cardinal (X:Type) (H:FiniteT X) : nat :=
  proj1_sig (constructive_definite_description _
              (FiniteT_has_nat_cardinal X H)).

(*Schepler*)
Lemma FiniteT_nat_cardinal_def: forall (X:Type) (H:FiniteT X),
  cardinal _ (@Full_set X) (FiniteT_nat_cardinal X H).
Proof.
intros; unfold FiniteT_nat_cardinal.
destruct constructive_definite_description.
assumption.
Qed.

(*Schepler*)
Lemma FiniteT_nat_cardinal_cond: forall (X:Type) (H:FiniteT X)
  (n:nat),
  cardinal _ (@Full_set X) n ->
  FiniteT_nat_cardinal X H = n.
Proof.
intros.
pose proof (FiniteT_has_nat_cardinal X H).
destruct H1.
red in H1.
destruct H1.
transitivity x.
symmetry; apply H2.
apply FiniteT_nat_cardinal_def.
apply H2; trivial.
Qed.

(*Schepler*)
Lemma FiniteT_nat_cardinal_False:
  FiniteT_nat_cardinal False empty_finite = 0.
Proof.
apply FiniteT_nat_cardinal_cond.
assert (@Full_set False = @Empty_set False).
apply Extensionality_Ensembles; red; split; auto with sets.
red; intros.
destruct x.
rewrite H.
constructor.
Qed.

(*Schepler*)
Lemma injection_preserves_cardinal: forall (X Y:Type)
  (f:X->Y) (n:nat) (S:Ensemble X), cardinal _ S n ->
  FunctionProperties.injective f -> cardinal _ (Im S f) n.
Proof.
intros.
induction H.
assert (Im (Empty_set _) f = (Empty_set _)).
apply Extensionality_Ensembles; split; auto with sets.
red; intros.
destruct H.
destruct H.
rewrite H; constructor.
assert (Im (Add A x) f = Add (Im A f) (f x)).
apply Extensionality_Ensembles; split.
red; intros.
destruct H2.
symmetry in H3; destruct H3.
destruct H2.
left; exists x0; auto with sets.
destruct H2; right; auto with sets.
red; intros.
destruct H2.
destruct H2.
exists x0.
left; auto with sets.
assumption.
destruct H2.
exists x; trivial; right; auto with sets.
rewrite H2.
constructor; trivial.
red; intro H3; inversion H3.
apply H0 in H5; destruct H5.
contradiction H1.
Qed.

(*Schepler*)
Lemma FiniteT_nat_cardinal_option:
  forall (X:Type) (H:FiniteT X),
  FiniteT_nat_cardinal (option X) (add_finite X H) =
  S (FiniteT_nat_cardinal X H).
Proof.
intros.
apply FiniteT_nat_cardinal_cond.
assert ((Full_set _) =
        Add (Im (Full_set _) (@Some X)) None).
apply Extensionality_Ensembles; split.
red; intros.
destruct x.
left; exists x; constructor.
right; constructor.
red; intros; constructor.
rewrite H0.
constructor.
apply injection_preserves_cardinal.
apply FiniteT_nat_cardinal_def.
red; intros x1 x2 Heq; injection Heq; trivial.
red; intro.
inversion H1.
discriminate H3.
Qed.


(*Schepler*)
Lemma FiniteT_nat_cardinal_bijection:
  forall (X Y:Type) (H:FiniteT X) (g:X->Y) (Hinv:invertible g),
    FiniteT_nat_cardinal Y (bij_finite X Y g H Hinv) =
    FiniteT_nat_cardinal X H.
Proof.
intros.
apply FiniteT_nat_cardinal_cond.
apply invertible_impl_bijective in Hinv.
destruct Hinv as [g_inj g_surj].
assert ((Full_set _)= Im (Full_set _) g).
apply Extensionality_Ensembles; split; red; intros;
  try constructor.
destruct (g_surj x).
exists x0; try constructor; auto.
rewrite H0; apply injection_preserves_cardinal; trivial.
apply FiniteT_nat_cardinal_def.
Qed.

(*Wyckoff*)
Lemma unit_finite : FiniteT unit.
pose (fun x:(option False) => tt) as f.
assert (h1:invertible f).
  apply bijective_impl_invertible.
  red; split. 
  red. intros x1 x2. destruct x1; destruct x2; try contradiction; auto.
  red. intros x1. exists None. unfold f. destruct x1. reflexivity.
eapply bij_finite; auto.
eapply add_finite.
apply empty_finite.
apply h1.
Qed.


(*Wyckoff*)
Lemma FiniteT_nat_cardinal_unit : 
  FiniteT_nat_cardinal _ unit_finite = 1.
assert (h2:FiniteT_nat_cardinal _ (add_finite _ empty_finite) = 1).
  pose proof (FiniteT_nat_cardinal_False) as h2.
  rewrite <- h2.
  apply FiniteT_nat_cardinal_option.
rewrite <- h2.
pose (fun x:(option False) => tt) as f.
assert (h3:invertible f).
  apply bijective_impl_invertible.
  red; split. 
  red. intros x1 x2. destruct x1; destruct x2; try contradiction; auto.
  red. intros x1. exists None. unfold f. destruct x1. reflexivity.
generalize unit_finite.
intro h4.
assert (h5:h4 = (bij_finite _ _ f (add_finite _ empty_finite) h3)). apply proof_irrelevance. subst.
eapply FiniteT_nat_cardinal_bijection.
Qed.

(*Wyckoff*)
Lemma FiniteT_nat_cardinal_card_fun1_compat : 
  forall {T:Type} (pf:FiniteT T),
    FiniteT_nat_cardinal _ pf = card_fun1 (Full_set T).
intros T h1.
pose proof (card_fun1_compat (Full_set T)) as h2.
destruct h2 as [h2l h2r].
pose proof (Finite_FiniteT_iff T) as h3.
pose proof h1 as h1'.
rewrite <- h3 in h1'.
specialize (h2l h1').
pose proof (FiniteT_nat_cardinal_cond T h1 _ h2l) as h4.
assumption.
Qed.



End FiniteT_nat_cardinal.
  

Section Transfer.
(* Wyckoff *)
Definition transfer {T U:Type} (pf:T=U) (x:T) : U.
revert x. rewrite pf. intro u. refine u.
Defined.

(*Wyckoff*)
Definition transfer_r {T U:Type} (pf:T = U) (y:U) : T.
revert y. rewrite pf. intro u. refine u.
Defined.

(*Wyckoff*)
Lemma transfer_eq : 
  forall {T U:Type} (pf:T=U) (x:T),
    transfer pf x = eq_rect _ id x _ pf.
intros; subst; simpl; auto.
Qed.

(*Wyckoff*)
Lemma transfer_eq_refl : 
  forall {T:Type} (x:T),
    transfer eq_refl x = x.
auto.
Qed.

(*Wyckoff*)
Lemma transfer_r_eq_refl : 
  forall {T:Type} (x:T),
    transfer_r eq_refl x = x.
auto.
Qed.


(*Wyckoff*)
Lemma transfer_r_eq : 
  forall {T U:Type} (pf:T=U) (x:U),
    transfer_r pf x = eq_rect_r id x pf.
intros; subst; simpl; auto.
Qed.


(*Wyckoff*)
Lemma transfer_transfer_r_compat : 
  forall {T U:Type} (pf:T=U) (pf':U=T) (x:T),
    transfer pf x = transfer_r pf' x.
intros T U ? ? x.
unfold transfer. unfold transfer_r.
destruct pf. 
assert (h1:pf' = eq_refl). apply proof_irrelevance.
subst.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_undoes_transfer_r : 
  forall {T U:Type} (pf:T=U) (y:U),
    transfer pf (transfer_r pf y) = y.
intros T U h1 y. subst.
unfold transfer, transfer_r. simpl. unfold eq_rect_r.  
simpl.
reflexivity.
Qed.         

(*Wyckoff*)
Lemma transfer_r_undoes_transfer : 
  forall {T U:Type} (pf:T=U) (x:T),
    transfer_r pf (transfer pf x) = x.
intros T U h1 y. subst.
unfold transfer, transfer_r. simpl. unfold eq_rect_r.  
simpl.
reflexivity.
Qed.         

(*Wyckoff*)
Lemma transfer_inj : 
  forall {T U:Type} (pf:T=U) (x y:T),
    transfer pf x = transfer pf y ->
    x = y.
intros T U pf x y h1.
subst. 
unfold transfer in h1. unfold eq_rect_r in h1.
rewrite <- eq_rect_eq in h1.
assumption.
Qed.

(*Wyckoff*)
Lemma transfer_hetero_inj : 
  forall {T U V:Type} (pft:T=V) (pfu:U=V) (x:T) (y:U),
    transfer pft x = transfer pfu y ->
    existT id _ x = existT id _ y.
intros T U V h1 h2 x y. subst.
intro h1.
apply transfer_inj in h1. subst.
reflexivity.
Qed.



                    
(*Wyckoff*)
Lemma transfer_r_inj : 
  forall {T U:Type} (pf:T=U) (x y:U),
    transfer_r pf x = transfer_r pf y ->
    x = y.
intros T U pf x y h1.
subst. 
unfold transfer_r in h1. unfold eq_rect_r in h1.
rewrite <- eq_rect_eq in h1.
assumption.
Qed.


(*Wyckoff*)
Lemma transfer_r_hetero_inj : 
  forall {T U V:Type} (pft:V=T) (pfu:V=U) (x:T) (y:U),
    transfer_r pft x = transfer_r pfu y ->
    existT id _ x = existT id _ y.
intros T U V h1 h2 x y.  subst. subst.
intro h1.
apply transfer_r_inj in h1. subst.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_transfer_hetero : 
  forall {T U V:Type} (pf:T = U) 
         (pf':U = V) (a:T),
    transfer pf' (transfer pf a) =
         transfer (eq_trans pf pf') a.
intros T U V h1 h2 a.
subst. subst.
unfold transfer. unfold transfer_r. unfold eq_rect_r. simpl.
reflexivity.
Qed.

 
(*Wyckoff*)
Lemma transfer_transfer_r_hetero : 
  forall {T U V:Type} (pf:T = U) 
         (pf':T = V) (a:U),
    transfer pf' (transfer_r pf a) =
         transfer (eq2 pf pf') a.
intros T U V h1 h2 a.
subst. subst.
unfold transfer. unfold transfer_r. unfold eq_rect_r. simpl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_r_transfer_hetero : 
  forall {T U V:Type} (pf:T = U) 
         (pf':V = U) (a:T),
    transfer_r pf' (transfer pf a) =
         transfer (eq2' pf pf') a.
intros T U V h1 h2 a.
subst. subst.
unfold transfer. unfold transfer_r. unfold eq_rect_r. simpl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_r_transfer_r_hetero : 
  forall {T U V:Type} (pf:T = U) 
         (pf':V = T) (a:U),
    transfer_r pf' (transfer_r pf a) =
         transfer_r (eq_trans' pf pf') a.
intros T U V h1 h2 a.
subst. subst.
unfold transfer. unfold transfer_r. unfold eq_rect_r. simpl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_pred : forall {T T':Type} (pf:T = T')
                             (P:{U:Type & U} -> Prop),
                        forall x:T,
                          P (existT id T x) <-> P (existT id T' (transfer pf x)).
intros T T' h1 P x.
subst.
unfold transfer.
unfold eq_rect_r.
simpl.
tauto.
Qed.


(*Wyckoff*)
Lemma transfer_fun_pred : 
  forall {T T' U U':Type} (pft:T = T') (pfu:U = U') (f:T->U)
    (P:{pT:Type*Type & ((fst pT)->(snd pT))}->Prop),
    P (existT _ (T,U) f)  <-> 
    P (existT _ (T', U') (fun x:(fst (T', U')) => transfer (eq_refl (snd (T', U'))) (transfer pfu (f (transfer_r pft x))))).
simpl.
intros T T' U U' h1 h2 f P.
subst.
unfold transfer, transfer_r, eq_rect_r. simpl.
tauto.
Qed.

(*Wyckoff*)
Lemma transfer_fun_pred' : 
  forall {T T' U U':Type} (pft:T' = T) (pfu:U = U') (f:T->U)
    (P:{pT:Type*Type & ((fst pT)->(snd pT))}->Prop),
    P (existT _ (T,U) f)  <-> 
    P (existT _ (T', U') (fun x:(fst (T', U')) => transfer (eq_refl (snd (T', U'))) (transfer pfu (f (transfer pft x))))).
simpl.
intros T T' U U' h1 h2 f P.
subst.
unfold transfer, eq_rect_r. simpl.
tauto.
Qed.


(*Wyckoff*)
Definition transfer_dep {T:Type} {U:T->Type} {x y:T} (pf:x = y)
           (a:U x):U y.
subst. refine a.
Defined.

(*Wyckoff*)
Definition transfer_dep_r {T:Type} {U:T->Type} {x y:T} (pf:y = x)
           (a:U x):U y.
subst. refine a.
Defined.

(*Wyckoff*)
Lemma transfer_dep_transfer_dep_r_compat : 
  forall {T:Type} {U:T->Type} {x y:T} (pf:x = y) (pf':y = x) (a:U x),
    transfer_dep pf a = transfer_dep_r pf' a.
intros T U x y h1 h2 a.
unfold transfer_dep, transfer_dep_r.
destruct h2. unfold eq_rect_r. simpl.
assert (h2:eq_sym h1 = eq_refl). apply proof_irrelevance.
rewrite h2.
simpl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_dep_r_undoes_transfer_dep : 
  forall {T:Type} {U:T->Type} {x y:T} (pf:x = y) (a:U x),
    transfer_dep_r pf (transfer_dep pf a) = a.
intros T U x y h1 a.
subst.
unfold transfer_dep_r, transfer_dep, eq_rect_r. simpl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_dep_undoes_transfer_dep_r : 
  forall {T:Type} {U:T->Type} {x y:T} (pf:x = y) (a:U y),
    transfer_dep pf (transfer_dep_r pf a) = a.
intros T U x y h1 a.
subst.
unfold transfer_dep_r, transfer_dep, eq_rect_r. simpl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_dep_inj : 
  forall {T:Type} {U:T->Type} {x y:T} (pf:x = y) (a b:U x),
    transfer_dep pf a = transfer_dep pf b ->
    a = b.
intros T U x y h1 a b h2.
subst.
unfold transfer_dep in h2.
unfold eq_rect_r in h2. simpl in h2.
assumption.
Qed.

(*Wyckoff*)
Lemma transfer_dep_r_inj : 
  forall {T:Type} {U:T->Type} {x y:T} (pf:x = y) (a b:U y),
    transfer_dep_r pf a = transfer_dep_r pf b ->
    a = b.
intros T U x y h1 a b h2.
subst.
unfold transfer_dep_r, eq_rect_r in h2.
simpl in h2.
assumption.
Qed.


(*Wyckoff*)
Lemma transfer_dep_existT : forall {T:Type} {U:T->Type} (x y:T) (pf:x = y)
                                   (a:U x), 
                              existT _ _ a = existT _ _ (transfer_dep pf a).
intros T U x y h1 a.
subst.
f_equal.
Qed.

(*Wyckoff*)
Lemma projT_injective : forall {T:Type} (A:T->Type) 
                               (x y:{a:T & (A a)})
                          (pf:projT1 x = projT1 y),
                               (transfer_dep pf (projT2 x)) =
                               projT2 y->
                          x = y.
intros T A x y h1 h2.
destruct x; destruct y.
simpl in h1. simpl in h2.
subst.
apply transfer_dep_existT.
Qed.

(*Wyckoff*)
Lemma transfer_dep_eq_refl : forall {T:Type} (U:T->Type) (x:T)
                             (p:U x),
                               transfer_dep eq_refl p = p.
auto.
Defined.

(*Wyckoff*)
Lemma transfer_dep_r_eq_refl : forall {T:Type} (U:T->Type) (x:T)
                                      (p:U x),
                                 transfer_dep_r eq_refl p = p.
auto.
Defined.


(*Wyckoff*)
Lemma existT_injective1 : forall {T:Type} (U:T->Type) (x y:T) (p:U x) (q:U y),
  existT _ x p = existT _ y q -> x = y.
intros T U x y p q h1.
pose proof (f_equal (@projT1 _ _) h1) as h2. 
simpl in h2.
assumption.
Qed.


(*Wyckoff*)
Lemma transfer_dep_prop : 
  forall {T:Type} {U:T->Type} (x y:T) (pf:x = y)
         (a:U x) (P:{t:T & (U t)}->Prop),
    P (existT _ _ a) <-> P (existT _ _ (transfer_dep pf a)).
intros T U x y h1 a P.
rewrite (transfer_dep_existT _ _ h1 a).
tauto.
Qed.

(*Wyckoff*)
Lemma transfer_dep_eq_iff : 
  forall {T:Type} {U:T->Type} (x y:T) (pf:x = y)
         (a:U x) (b:U y), 
    existT _ _ a = existT _ _ b <->
        transfer_dep pf a = b.
intros T U x y h1 a b.
subst.
split.
intro h1.
apply inj_pair2 in h1. 
unfold transfer_dep.
unfold eq_rect_r.
rewrite <- eq_rect_eq.
assumption.
intro h1.
unfold transfer_dep in h1. unfold eq_rect_r in h1.
rewrite <- eq_rect_eq in h1.
subst.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_dep_r_eq_iff : 
  forall {T:Type} {U:T->Type} (x y:T) (pf:y = x)
         (a:U x) (b:U y), 
    existT _ _ a = existT _ _ b <->
        transfer_dep_r pf a = b.
intros T U x y h1 a b.
subst.
split.
intro h1.
apply inj_pair2 in h1. 
unfold transfer_dep_r.
unfold eq_rect_r.
rewrite <- eq_rect_eq.
assumption.
intro h1.
unfold transfer_dep_r in h1. unfold eq_rect_r in h1.
rewrite <- eq_rect_eq in h1.
subst.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_dep_fun1_eq : 
  forall {T U:Type} (pf:T = U)
         (f:T->T) (x:U),
    (transfer_dep (U:=fun x=>x->x) pf f) x =
         (transfer pf (f (transfer_r pf x))).
intros T U h1 f x.
subst.
rewrite transfer_dep_eq_refl.
unfold transfer, transfer_r, eq_rect_r. simpl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_dep_fun2_eq : 
  forall {T U:Type} (pf:T = U)
         (f:T->T->T) (x y:U),
    (transfer_dep (U:=fun x=>x->x->x) pf f) x y =
         (transfer pf (f (transfer_r pf x) (transfer_r pf y))).
intros T U h1 f x y.
subst.
rewrite transfer_dep_eq_refl.
unfold transfer, transfer_r, eq_rect_r. simpl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_dep_r_fun1_eq : 
  forall {T U:Type} (pf:T = U)
         (f:U->U) (x:T),
    (transfer_dep_r (U:=fun x=>x->x) pf f) x =
         (transfer_r pf (f (transfer pf x))).
intros T U h1 f x.
subst.
rewrite transfer_dep_r_eq_refl.
unfold transfer, transfer_r, eq_rect_r. simpl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_dep_r_fun2_eq : 
  forall {T U:Type} (pf:T = U)
         (f:U->U->U) (x y:T),
    (transfer_dep_r (U:=fun x=>x->x->x) pf f) x y =
         (transfer_r pf (f (transfer pf x) (transfer pf y))).
intros T U h1 f x y.
subst.
rewrite transfer_dep_r_eq_refl.
unfold transfer, transfer_r, eq_rect_r. simpl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_dep_r_hetero : 
  forall {T:Type} {U:T->Type} (x y y':T) (pf:x = y) 
         (pf':x = y') (a:U y) (a':U y'),
    transfer_dep_r pf a = transfer_dep_r pf' a' <->
         a = transfer_dep_r (eq2 pf pf') a'.
intros T U x y y' h1 h2 a a'.
subst. subst.
do 2 rewrite transfer_dep_r_eq_refl. 
unfold eq2. unfold eq_ind_r. unfold eq_ind. simpl.
rewrite transfer_dep_r_eq_refl.
tauto.
Qed.


(*Wyckoff*)
Definition transfer_fun {T U V:Type} (pf:T=U) (f:T->V) : U->V.
revert f. rewrite pf. intro f. refine f.
Defined.

(*Wyckoff*)
Definition transfer_fun' {T U:Type} (pf:T=U) (f:T->T) : U->U.
revert f. rewrite pf. intro f. refine f.
Defined.


(*Wyckoff*)
Definition transfer_fun2 {T U V:Type} (pf:T=U) (f:T->T->V) : U->U->V.
revert f. rewrite pf. intro f. refine f.
Defined.


(*Wyckoff*)
Definition transfer_fun2' {T U:Type} (pf:T=U) (f:T->T->T) : U->U->U.
revert f. rewrite pf. intro f. refine f.
Defined.


(*Wyckoff*)
Definition transfer_fun_r {T U V:Type} (pf:T=U) (f:U->V) : T->V.
revert f. rewrite pf. intro f. refine f.
Defined.


(*Wyckoff*)
Definition transfer_fun_r' {T U:Type} (pf:T=U) (f:U->U) : T->T.
revert f. rewrite pf. intro f. refine f.
Defined.


(*Wyckoff*)
Definition transfer_fun2_r {T U V:Type} (pf:T=U) (f:U->U->V) : T->T->V.
revert f. rewrite pf. intro f. refine f.
Defined.

(*Wyckoff*)
Definition transfer_fun2_r' {T U:Type} (pf:T=U) (f:U->U->U) : T->T->T.
revert f. rewrite pf. intro f. refine f.
Defined.


(*Wyckoff*)
Definition transfer_fun_eq_refl : 
  forall {T U:Type} (f:T->U),
    transfer_fun eq_refl f = f.
intros. apply functional_extensionality. intro.
unfold transfer_fun. unfold eq_rect_r. simpl.
reflexivity.
Qed.

(*Wyckoff*)
Definition transfer_fun2_eq_refl : 
  forall {T U:Type} (f:T->T->U),
    transfer_fun2 eq_refl f = f.
intros. 
apply functional_extensionality. intro.
apply functional_extensionality. intro.
unfold transfer_fun2. unfold eq_rect_r. simpl.
reflexivity.
Qed.

(*Wyckoff*)
Definition transfer_fun_eq_refl' : 
  forall {T:Type} (f:T->T),
    transfer_fun' eq_refl f = f.
intros. apply functional_extensionality. intro.
unfold transfer_fun'. unfold eq_rect_r. simpl.
reflexivity.
Qed.



(*Wyckoff*)
Definition transfer_fun2_eq_refl' : 
  forall {T:Type} (f:T->T->T),
    transfer_fun2' eq_refl f = f.
intros. 
apply functional_extensionality. intro.
apply functional_extensionality. intro.
unfold transfer_fun2'. unfold eq_rect_r. simpl.
reflexivity.
Qed.


(*Wyckoff*)
Definition transfer_fun_r_eq_refl : 
  forall {T U:Type} (f:T->U),
    transfer_fun_r eq_refl f = f.
intros. apply functional_extensionality. intro.
unfold transfer_fun. unfold eq_rect_r. simpl.
reflexivity.
Qed.

(*Wyckoff*)
Definition transfer_fun2_r_eq_refl : 
  forall {T U:Type} (f:T->T->U),
    transfer_fun2_r eq_refl f = f.
intros. 
apply functional_extensionality. intro.
apply functional_extensionality. intro.
unfold transfer_fun2. unfold eq_rect_r. simpl.
reflexivity.
Qed.

(*Wyckoff*)
Definition transfer_fun_r_eq_refl' : 
  forall {T:Type} (f:T->T),
    transfer_fun_r' eq_refl f = f.
intros. apply functional_extensionality. intro.
unfold transfer_fun_r'. unfold eq_rect_r. simpl.
reflexivity.
Qed.

(*Wyckoff*)
Definition transfer_fun2_r_eq_refl' : 
  forall {T:Type} (f:T->T->T),
    transfer_fun2_r' eq_refl f = f.
intros. 
apply functional_extensionality. intro.
apply functional_extensionality. intro.
unfold transfer_fun2_r'. unfold eq_rect_r. simpl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun_eq : 
  forall {T U V:Type} (pf:T = U) (f:T->V),
    transfer_fun pf f =
    (fun x:U => f (transfer_r pf x)).
intros T U V h1 f. subst.
rewrite transfer_fun_eq_refl.
apply functional_extensionality. intro x.
rewrite transfer_r_eq_refl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_fun_eq' : 
  forall {T U:Type} (pf:T=U) (f:T->T),
    transfer_fun' pf f =
    (fun x:U => (transfer pf (f (transfer_r pf x)))).
intros T U h1 f. subst.
rewrite transfer_fun_eq_refl'.
apply functional_extensionality. intro x.
rewrite transfer_eq_refl.
rewrite transfer_r_eq_refl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun2_eq : 
  forall {T U V:Type} (pf:T = U) (f:T->T->V),
    transfer_fun2 pf f =
    (fun (x:U) (y:U) => f (transfer_r pf x) (transfer_r pf y)).
intros T U V h1 f. subst.
rewrite transfer_fun2_eq_refl.
apply functional_extensionality. intro x.
rewrite transfer_r_eq_refl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun2_eq' : 
  forall {T U:Type} (pf:T=U) (f:T->T->T),
    transfer_fun2' pf f =
    (fun (x:U) (y:U) => (transfer pf (f (transfer_r pf x) (transfer_r pf y)))).
intros T U h1 f. subst.
rewrite transfer_fun2_eq_refl'. 
apply functional_extensionality. intro x. apply functional_extensionality. intro y.
rewrite transfer_eq_refl.
do 2 rewrite transfer_r_eq_refl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun_r_eq : 
  forall {T U V:Type} (pf:T = U) (f:U->V),
    transfer_fun_r pf f =
    (fun x:T => f (transfer pf x)).
intros T U V h1 f. subst.
rewrite transfer_fun_r_eq_refl.
apply functional_extensionality. intro x.
rewrite transfer_eq_refl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_fun_r_eq' : 
  forall {T U:Type} (pf:T=U) (f:U->U),
    transfer_fun_r' pf f =
    (fun x:T => (transfer_r pf (f (transfer pf x)))).
intros T U h1 f. subst.
rewrite transfer_fun_r_eq_refl'.
apply functional_extensionality. intro x.
rewrite transfer_eq_refl.
rewrite transfer_r_eq_refl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun2_r_eq : 
  forall {T U V:Type} (pf:T = U) (f:U->U->V),
    transfer_fun2_r pf f =
    (fun (x:T) (y:T) => f (transfer pf x) (transfer pf y)).
intros T U V h1 f. subst.
rewrite transfer_fun2_r_eq_refl.
apply functional_extensionality. intro x.
rewrite transfer_eq_refl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun2_r_eq' : 
  forall {T U:Type} (pf:T=U) (f:U->U->U),
    transfer_fun2_r' pf f =
    (fun (x:T) (y:T) => (transfer_r pf (f (transfer pf x) (transfer pf y)))).
intros T U h1 f. subst.
rewrite transfer_fun2_r_eq_refl'. 
apply functional_extensionality. intro x. apply functional_extensionality. intro y.
do 2 rewrite transfer_eq_refl.
rewrite transfer_r_eq_refl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun_compat : 
  forall {T U V:Type} (pf:T = U) (f:T->V) (x:T),
    (transfer_fun pf f) (transfer pf x) = f x.
intros T U V h1 f x.
unfold transfer_fun. unfold transfer.
destruct h1.   compute.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun2_compat : 
  forall {T U V:Type} (pf:T = U) (f:T->T->V) (x y:T),
    (transfer_fun2 pf f) (transfer pf x) (transfer pf y) = f x y.
intros T U V h1 f x y.
unfold transfer_fun2. unfold transfer, eq_rect_r. subst.
simpl. reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun_compat' : 
  forall {T U:Type} (pf:T = U) (f:T->T) (x:T),
    (transfer_fun' pf f) (transfer pf x) =
    transfer pf (f x). 
intros T U h1 f x. subst.
unfold transfer_fun'. unfold transfer. unfold eq_rect_r.
simpl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun2_compat' : 
  forall {T U:Type} (pf:T = U) (f:T->T->T) (x y:T),
    (transfer_fun2' pf f) (transfer pf x) (transfer pf y) =
    transfer pf (f x y).
intros T U h1 f x y.
unfold transfer_fun2'. unfold transfer, eq_rect_r. subst.
simpl. reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun_r_compat : 
  forall {T U V:Type} (pf:T = U) (f:U->V) (x:U),
    f x = (transfer_fun_r pf f) (transfer_r pf x).
intros T U V h1 f x.
unfold transfer_fun_r. unfold transfer_r.
destruct h1.   compute.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun2_r_compat : 
  forall {T U V:Type} (pf:T = U) (f:U->U->V) (x y:U),
    f x y = (transfer_fun2_r pf f) (transfer_r pf x) (transfer_r pf y).
intros T U V h1 f x y.
unfold transfer_fun2_r. unfold transfer, eq_rect_r. subst.
simpl. reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun_r_compat' : 
  forall {T U:Type} (pf:T = U) (f:U->U) (x:U),
    (transfer_fun_r' pf f) (transfer_r pf x) = transfer_r pf (f x).
intros T U h1 f x.
unfold transfer_fun_r'. unfold transfer_r.
destruct h1.   compute.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun2_r_compat' : 
  forall {T U:Type} (pf:T = U) (f:U->U->U) (x y:U),
    (transfer_fun2_r' pf f) (transfer_r pf x) (transfer_r pf y) =
    transfer_r pf (f x y).
intros T U h1 f x y.
unfold transfer_fun2_r'. unfold transfer_r, eq_rect_r. subst.
simpl. reflexivity.
Qed.



(*Wyckoff*)
Lemma transfer_fun_transfer_dep_compat : 
  forall {T U V:Type} (pf:T = U) (f:T->V),
    transfer_fun pf f = 
    transfer_dep pf (U:=fun A:Type=>A->V) f.
intros T U V h1 f. subst.
simpl.
rewrite transfer_fun_eq_refl, transfer_dep_eq_refl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_fun2_transfer_dep_compat : 
  forall {T U V:Type} (pf:T = U) (f:T->T->V),
    transfer_fun2 pf f = 
    transfer_dep pf (U:=fun A:Type=>A->A->V) f.
intros T U V h1 f. subst.
simpl.
rewrite transfer_fun2_eq_refl, transfer_dep_eq_refl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_fun_r_transfer_dep_r_compat : 
  forall {T U V:Type} (pf:T = U) (f:U->V),
    transfer_fun_r pf f = 
    transfer_dep_r pf (U:=fun A:Type=>A->V) f.
intros T U V h1 f. subst.
simpl.
rewrite transfer_fun_r_eq_refl, transfer_dep_r_eq_refl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun2_r_transfer_dep_r_compat : 
  forall {T U V:Type} (pf:T = U) (f:U->U->V),
    transfer_fun2_r pf f = 
    transfer_dep_r pf (U:=fun A:Type=>A->A->V) f.
intros T U V h1 f. subst.
simpl.
rewrite transfer_fun2_r_eq_refl, transfer_dep_r_eq_refl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun_transfer_dep_compat' : 
  forall {T U:Type} (pf:T = U) (f:T->T),
    transfer_fun' pf f = 
    transfer_dep pf (U:=fun A:Type=>A->A) f.
intros T U h1 f. subst.
simpl.
rewrite transfer_fun_eq_refl', transfer_dep_eq_refl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_fun2_transfer_dep_compat' : 
  forall {T U:Type} (pf:T = U) (f:T->T->T),
    transfer_fun2' pf f = 
    transfer_dep pf (U:=fun A:Type=>A->A->A) f.
intros T U h1 f. subst.
simpl.
rewrite transfer_fun2_eq_refl', transfer_dep_eq_refl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun_r_transfer_dep_r_compat' : 
  forall {T U:Type} (pf:T = U) (f:U->U),
    transfer_fun_r' pf f = 
    transfer_dep_r pf (U:=fun A:Type=>A->A) f.
intros T U h1 f. subst.
simpl.
rewrite transfer_fun_r_eq_refl', transfer_dep_r_eq_refl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun2_r_transfer_dep_r_compat' : 
  forall {T U:Type} (pf:T = U) (f:U->U->U),
    transfer_fun2_r' pf f = 
    transfer_dep_r pf (U:=fun A:Type=>A->A->A) f.
intros T U h1 f. subst.
simpl.
rewrite transfer_fun2_r_eq_refl', transfer_dep_r_eq_refl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_fun_transfer_compat : 
  forall {T U V:Type} (pf:T = U) (f:T->V),
    transfer_fun pf f = transfer (f_equal (fun A=>A->V) pf) f.
intros T U V h1 f.
subst.
rewrite transfer_fun_eq_refl. simpl.
rewrite transfer_eq_refl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun_transfer_compat' : 
  forall {T U:Type} (pf:T = U) (f:T->T),
    transfer_fun' pf f = transfer (f_equal (fun A=>A->A) pf) f.
intros T U h1 f.
subst.
rewrite transfer_fun_eq_refl'. simpl.
rewrite transfer_eq_refl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_fun_r_transfer_r_compat : 
  forall {T U V:Type} (pf:T = U) (f:U->V),
    transfer_fun_r pf f = transfer_r (f_equal (fun A=>A->V) pf) f.
intros T U V h1 f.
subst.
rewrite transfer_fun_r_eq_refl. simpl.
rewrite transfer_r_eq_refl.
reflexivity.
Qed.



(*Wyckoff*)
Lemma transfer_fun_r_transfer_r_compat' : 
  forall {T U:Type} (pf:T = U) (f:U->U),
    transfer_fun_r' pf f = transfer_r (f_equal (fun A=>A->A) pf) f.
intros T U h1 f.
subst.
rewrite transfer_fun_r_eq_refl'. simpl.
rewrite transfer_r_eq_refl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_fun2_transfer_compat : 
  forall {T U V:Type} (pf:T = U) (f:T->T->V),
    transfer_fun2 pf f = transfer (f_equal (fun A=>A->A->V) pf) f.
intros T U V h1 f.
subst.
rewrite transfer_fun2_eq_refl. simpl.
rewrite transfer_eq_refl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun2_transfer_compat' : 
  forall {T U:Type} (pf:T = U) (f:T->T->T),
    transfer_fun2' pf f = transfer (f_equal (fun A=>A->A->A) pf) f.
intros T U h1 f.
subst.
rewrite transfer_fun2_eq_refl'. simpl.
rewrite transfer_eq_refl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_fun2_r_transfer_r_compat : 
  forall {T U V:Type} (pf:T = U) (f:U->U->V),
    transfer_fun2_r pf f = transfer_r (f_equal (fun A=>A->A->V) pf) f.
intros T U V h1 f.
subst.
rewrite transfer_fun2_r_eq_refl. simpl.
rewrite transfer_r_eq_refl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma transfer_fun2_r_transfer_r_compat' : 
  forall {T U:Type} (pf:T = U) (f:U->U->U),
    transfer_fun2_r' pf f = transfer_r (f_equal (fun A=>A->A->A) pf) f.
intros T U h1 f.
subst.
rewrite transfer_fun2_r_eq_refl'. simpl.
rewrite transfer_r_eq_refl.
reflexivity.
Qed.






(*Wyckoff*)
Lemma transfer_prop : forall {T U:Type} (pf:T=U) (f:Type->Prop), 
  f T <-> f U.
  intros T U h1. subst.
tauto.
Qed.

(*Wyckoff*)
Lemma transfer_prop_dependent : 
  forall {T:Type} {PT:T->Type} (x y:T) (P:PT x->Prop)
         (pf:PT x = PT y) (a:PT x),
    P a <-> (transfer_fun pf P) (transfer pf a).
intros T PT x y P h1 a. split.
intro h2.
rewrite <- (transfer_fun_compat h1 P a) in h2.
assumption.
intro h2.
rewrite <- (transfer_fun_compat h1 P a).
assumption.
Qed.

(*Wyckoff*)
Lemma transfer_in : 
  forall {T T':Type} (pf:T = T') (A:Ensemble T) (x:T),
    In A x <-> In (transfer_dep pf A) (transfer pf x).
intros T T' h1 A x. subst.
unfold transfer_dep. unfold eq_rect_r. simpl.
unfold transfer. unfold eq_rect_r. simpl.
tauto.
Qed. 

(*Wyckoff*)
Lemma transfer_in_r : 
  forall {T T':Type} (pf:T' = T) (A:Ensemble T) (x:T),
    In A x <-> In (transfer_dep_r pf A) (transfer_r pf x).
intros T T' h1 A x. subst.
tauto.
Qed.

(*Wyckoff*)
Lemma transfer_sig_set_eq : 
  forall {T:Type} (A A':Ensemble T) 
         (pf:A = A') (pfs:sig_set A = sig_set A')
         (x:sig_set A),
    transfer pfs x = exist (fun x=>In A' x) (proj1_sig x) (@transfer_dep _ (fun S=>In S (proj1_sig x)) _ _ pf (proj2_sig x)).
intros T A A' h1 h2 x.
subst.
assert (h3:h2 = eq_refl _). apply proof_irrelevance.
subst. unfold transfer. unfold eq_rect_r.
simpl.
destruct x as [x h1].
apply proj1_sig_injective.
simpl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_r_sig_set_eq : 
  forall {T:Type} (A A':Ensemble T) 
         (pf:A = A') (pfs:sig_set A = sig_set A')
         (x:sig_set A'),
    transfer_r pfs x = exist (fun x=>In A x) (proj1_sig x) (@transfer_dep_r _ (fun S=>In S (proj1_sig x)) _ _ pf (proj2_sig x)).
intros T A A' h1 h2 x.
subst.
assert (h3:h2 = eq_refl _). apply proof_irrelevance.
subst. unfold transfer_r. unfold eq_rect_r.
simpl.
destruct x as [x h1].
apply proj1_sig_injective.
simpl.
reflexivity.
Qed.



(*Wyckoff*)
Lemma transfer_dep_transfer_dep_sig_set_eq_compat : 
  forall {T:Type} (A B:Ensemble T) (pf:A = B)
         (U: Type->Type)  (f:U (sig_set A)),
    transfer_dep pf f (U:=fun C:Ensemble T=> U (sig_set C))  = transfer_dep (sig_set_eq _ _ pf) f.
intros T A B h1 U f.
subst.
rewrite transfer_dep_eq_refl.
assert (h1:sig_set_eq B B eq_refl = eq_refl). apply proof_irrelevance.
rewrite h1.
rewrite transfer_dep_eq_refl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_dep_r_transfer_dep_r_sig_set_eq_compat : 
  forall {T:Type} (A B:Ensemble T) (pf:A = B)
         (U: Type->Type)  (f:U (sig_set B)),
    transfer_dep_r pf f (U:=fun C:Ensemble T=> U (sig_set C))  = transfer_dep_r (sig_set_eq _ _ pf) f.
intros T A B h1 U f.
subst.
rewrite transfer_dep_r_eq_refl.
assert (h1:sig_set_eq B B eq_refl = eq_refl). apply proof_irrelevance.
rewrite h1.
rewrite transfer_dep_r_eq_refl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_dep_transfer_compat : 
  forall {T:Type} (A B:T) (pf:A = B)
         (U: T->Type)  (f:U A),
    transfer_dep pf f (U:=fun C:T=> U C)  = 
    transfer (f_equal U pf) f.
intros T A B h1 U f.
subst.
simpl.
rewrite transfer_dep_eq_refl.
rewrite transfer_eq_refl.
reflexivity.
Qed.                                

(*Wyckoff*)
Lemma transfer_dep_r_transfer_r_compat : 
  forall {T:Type} (A B:T) (pf:A = B)
         (U: T->Type)  (f:U B),
    transfer_dep_r pf f (U:=fun C:T=> U C)  = 
    transfer_r (f_equal U pf) f.
intros T A B h1 U f.
subst.
simpl.
rewrite transfer_dep_r_eq_refl.
rewrite transfer_r_eq_refl.
reflexivity.
Qed.                                





(*Include transfer_dep_r analogue for above.*)
(*Wyckoff*)
Lemma transfer_dep_id_transfer_compat : 
  forall (T U:Type) (pf:T = U)
         (a:T),
         transfer_dep pf a (U:=@id Type) = transfer pf a.
intros T U h1 a. subst.
rewrite transfer_dep_eq_refl, transfer_eq_refl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma transfer_dep_r_id_transfer_r_compat : 
  forall (T U:Type) (pf:T = U)
         (a:U),
         transfer_dep_r pf a (U:=@id Type) = transfer_r pf a.
intros T U h1 a. subst.
rewrite transfer_dep_r_eq_refl, transfer_r_eq_refl.
reflexivity.
Qed.



(*Wyckoff*)
Lemma extends_prim_impl_extends_sig : 
  forall {T T' U:Type}
         (f:T->U) (g:T'->U),
    extends_prim f g ->
    exists (A:Ensemble T) (pf:T' = sig_set A),
      extends_sig1 f (transfer_dep (U:=fun V=>V->U) pf g).
intros T T' U f g h1.
red in h1. destruct h1 as [A h1]. destruct h1 as [h1 h2].
subst.
exists A. exists eq_refl. 
red.
intro x.
rewrite transfer_dep_eq_refl.
specialize (h2 (proj1_sig x) x (proj2_sig x)).
rewrite <- unfold_sig in h2.
specialize (h2 eq_refl).
rewrite h2.
reflexivity.
Qed.

End Transfer.

Lemma proj2_sig_iff_sig_eq : 
  forall {T:Type} (P Q:T->Prop),
    (forall x:T, P x <-> Q x) -> 
    {x|P x}={x|Q x}.
intros T P Q h1.
f_equal.
apply functional_extensionality.
intro x. specialize (h1 x).
apply prop_ext.
assumption.
Qed.