(* Copyright (C) 2014-15, Daniel Wyckoff *)

(*This file is part of BooleanAlgebrasIntro2.

BooleanAlgebrasIntro2 is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

BooleanAlgebrasIntro2 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with BooleanAlgebrasIntro2.  If not, see <http://www.gnu.org/licenses/>.*)
  
Require Import SetUtilities.
Require Import LogicUtilities.
Require Export BoolAlgBasics.
Require Import FieldsOfSets.
Require Import InfiniteOperations.
Require Import TypeUtilities.
Require Import Arith.
Require Import FunctionProperties.
Require Import Equality.
Require Import FunctionalExtensionality.
Require Import Description.
Require Import FiniteMaps.
Require Import DecidableDec.
Require Import Equality.
Require Import FiniteOperations.
Require Import FiniteMaps2.
Require Import NPeano.
Require Import SetUtilities2.
Require Import SetUtilities3.
Require Import ArithUtilities.
Require Import ClassicalChoice.


(*See Section ParametricAnalogues for Bool_Alg_p subalgebras (i.e. 
  with an underlying type, which makes a posteriori extensions possible,
  and makes families of subalgebras better suited for certain applications.*)
Section AlgClosed.
Variable B:Bool_Alg.
Let Bt := bt B.
Variable A:(Ensemble Bt). 

Definition SubBtype := {a:Bt |In A a}.

Definition Bplus_sub (x y:SubBtype) := (proj1_sig x) + (proj1_sig y).
Definition Btimes_sub (x y:SubBtype) := (proj1_sig x) * (proj1_sig y).
Definition Bcomp_sub (x:SubBtype) := -(proj1_sig x).

Notation  "x +|' y" := (Bplus_sub x y) (at level 50, left associativity).
Notation  "x *|' y" := (Btimes_sub x y) (at level 40, left associativity).
Notation "-|' x" := (Bcomp_sub x) (at level 30).

Definition Plus_closed_sub : Prop := (forall (x y:SubBtype), 
               In A (x +|' y)).
Definition Times_closed_sub : Prop := (forall (x y:SubBtype), 
               In A (x *|' y)).
Definition One_closed_sub  : Prop := In A 1.
Definition Zero_closed_sub : Prop := In A 0.
Definition Comp_closed_sub : Prop := (forall x:SubBtype, 
               In A (-|' x)).

Lemma Plus_closed'' : forall (x y:Bt),
  Plus_closed_sub -> In A x -> In A y -> In A (x+y).
intros x y h1 h2 h3.
unfold Plus_closed_sub in h1.
apply h1 with (x := exist (In A) x h2) (y := exist (In A) y h3).
Qed.

Lemma Times_closed'' : forall (x y:Bt),
  Times_closed_sub -> In A x -> In A y -> In A (x*y).
intros x y h1 h2 h3.
unfold Times_closed_sub in h1.
apply h1 with (x := exist (In A) x h2) (y := exist (In A) y h3).
Qed.

Lemma Comp_closed'' : forall (b:Bt), Comp_closed_sub -> (In A b) ->
  In A (-b).
intros b h1 h2.
unfold Comp_closed_sub in h1.
apply h1 with (x := exist (In A) b h2).
Qed.

Record alg_closed : Prop := {
  P_c : Plus_closed_sub;
  T_c : Times_closed_sub;
  O_c : One_closed_sub;
  Z_c : Zero_closed_sub;
  C_c : Comp_closed_sub}.




Variable Ac : alg_closed.

Definition Bc' := 
  Build_Bconst SubBtype
    (Full_set SubBtype)
    (fun (x y:SubBtype) => exist (In A) (x +|' y)
      (P_c Ac x y))
    (fun (x y:SubBtype) => exist (In A) (x *|' y)
      (T_c Ac x y))
    (exist (In A ) 1 (O_c Ac))
    (exist (In A ) 0 (Z_c Ac))
    (fun (x:SubBtype) => (exist (In A) (-|' x)
      (C_c Ac x))).

Section Bc''.


Infix "+|" := (Bplus Bc')  (at level 50, left associativity). 
Infix "*|" := (Btimes Bc') (at level 40, left associativity).
Notation "0|" := (Bzero Bc').
Notation "1|" := (Bone Bc').
Notation "-| x" := ((Bcomp Bc') x) (at level 30).


(*These are just all the basic axioms applied to the subset type SubBtype, derived from the main boolean algebra B *)
Lemma assoc_sum' : forall n m p:SubBtype, n +| (m +| p) = n +| m +| p.
intros n m p.  
red in n.
red in m.
red in p.
inversion n as [x]. 
inversion m as [y]. 
inversion p as [z].
simpl.
unfold Bplus_sub.
simpl.
apply proj1_sig_injective.
simpl. apply assoc_sum.
Qed.

Lemma assoc_prod' : forall n m p:SubBtype, n *| (m *| p) = n *| m *| p.
intros n m p.
red in n.
red in m.
red in p.
inversion n as [x]. 
inversion m as [y]. 
inversion p as [z].
simpl.
unfold Btimes_sub.
simpl.
apply proj1_sig_injective.
simpl. apply assoc_prod.
Qed.

Lemma comm_sum' : forall n m:SubBtype, n +| m = m +| n.
intros n m.
red in n. red in m.
inversion n. inversion m.
simpl.
unfold Bplus_sub.
simpl.
apply proj1_sig_injective.
simpl. apply comm_sum.
Qed.

Lemma comm_prod' : forall n m:SubBtype, n *| m = m *| n.
intros n m.
red in n. red in m.
inversion n. inversion m.
simpl.
unfold Btimes_sub.
simpl.
apply proj1_sig_injective.
simpl. apply comm_prod.
Qed.

Lemma abs_sum'  : forall n m:SubBtype, n +| (n *| m) = n.
intros n m.
red in n. red in m.
inversion n. inversion m.
simpl.
unfold Btimes_sub. unfold Bplus_sub.
simpl.
apply proj1_sig_injective.
simpl. apply abs_sum.
Qed.

Lemma abs_prod':  forall n m:SubBtype, n *| (n +| m) = n.
intros n m.
red in n. red in m.
inversion n. inversion m.
simpl.
unfold Btimes_sub. unfold Bplus_sub.
simpl.
apply proj1_sig_injective.
simpl. apply abs_prod.
Qed.

Lemma dist_sum' : forall n m p:SubBtype, p *| (n +| m) = p *| n + p *| m.
intros n m p.
red in n. red in m. red in p.
inversion n. inversion m. inversion p.
simpl.
unfold Btimes_sub. unfold Bplus_sub.
simpl.
apply proj1_sig_injective.
simpl. apply dist_sum.
Qed.

Lemma dist_prod': forall n m p:SubBtype, p +| (n *| m) = (p +| n) *| (p +| m).
intros n m p.
red in n. red in m. red in p.
inversion n. inversion m. inversion p.
simpl.
unfold Btimes_sub. unfold Bplus_sub.
simpl.
apply proj1_sig_injective.
simpl. apply dist_prod.
Qed.

Lemma comp_sum':  forall n:SubBtype, n +| (-| n) = 1.
intros n.
red in n. inversion n.
simpl.
unfold Bplus_sub. unfold Bcomp_sub.
simpl.
apply proj1_sig_injective.
simpl. apply comp_sum.
Qed.

Lemma comp_prod': forall n:SubBtype, n *| (-| n) = 0.
intros n.
red in n. inversion n.
simpl.
unfold Bplus_sub. unfold Bcomp_sub.
simpl.
apply proj1_sig_injective.
simpl. apply comp_prod.
Qed.

Lemma und_set' : (BS Bc') = Full_set (Btype Bc').
unfold Bc'.
simpl. reflexivity.
Qed.

Definition Subalg := Build_Bool_Alg Bc' und_set' assoc_sum' 
  assoc_prod' comm_sum' comm_prod' abs_sum' abs_prod' dist_sum'
  dist_prod' comp_sum' comp_prod'.


(*Being closed under 2 operations is sufficient to ensure that a subset is a subalgebra *)
Lemma two_ops_imp_times : Comp_closed_sub -> Plus_closed_sub -> Times_closed_sub.
intros h1 h2. unfold Times_closed_sub.
intros x y.
unfold SubBtype in x.   unfold SubBtype in y.
assert (h3: proj1_sig x * proj1_sig y = -(-(proj1_sig x) + -(proj1_sig y))).
apply times_eq_plus. unfold Btimes_sub. rewrite h3.
apply Comp_closed'' with (b:= (-proj1_sig x + -proj1_sig y)).
apply h1.
apply Plus_closed'' with (x := -proj1_sig x) (y := -proj1_sig y).
apply h2.
apply Comp_closed''.  apply h1. 
apply (proj2_sig x).
apply Comp_closed''.  apply h1.
apply (proj2_sig y).
Qed.



Lemma two_ops_imp_plus : Comp_closed_sub -> Times_closed_sub -> Plus_closed_sub.
intros h1 h2. unfold Plus_closed_sub.
intros x y.
unfold SubBtype in x.   unfold SubBtype in y.
assert (h3: proj1_sig x + proj1_sig y = -(-(proj1_sig x) * -(proj1_sig y))).
apply plus_eq_times. unfold Bplus_sub. rewrite h3.
apply Comp_closed'' with (b:= (-proj1_sig x * -proj1_sig y)).
apply h1.
apply Times_closed'' with (x := -proj1_sig x) (y := -proj1_sig y).
apply h2.
apply Comp_closed''.  apply h1. 
apply (proj2_sig x).
apply Comp_closed''.  apply h1.
apply (proj2_sig y).
Qed.

End Bc''.
End AlgClosed.

Arguments alg_closed [B] _.
Arguments Subalg [B] _ _.
Arguments Plus_closed_sub [B] _.
Arguments Times_closed_sub [B] _.
Arguments Comp_closed_sub [B] _.
Arguments Zero_closed_sub [B] _.
Arguments One_closed_sub [B] _.



Lemma alg_closed_ba_ens : forall B:Bool_Alg,
                            alg_closed (ba_ens B).
intro B. 
constructor; red; intros; constructor.
Qed.



Lemma alg_closed_improper_subalg : 
  forall {B:Bool_Alg} (S:Ensemble (bt B))
         (S':Ensemble (bt (Subalg (ba_ens B) (alg_closed_ba_ens B)))),
    S = im_proj1_sig S' ->
    (alg_closed S <-> alg_closed S').
intros B S S' h1. subst.
split.
intro h1.
destruct h1 as [h1 h2 h3 h4 h5].
red in h1, h2, h3, h4, h5.
constructor; red.
intros x y.
unfold SubBtype in h1. 
destruct x as [x h6], y as [y h7].
destruct x as [x h8], y as [y h9].
assert (h10:In (im_proj1_sig S') x).
  apply Im_intro with (exist (fun a : Btype (Bc B) => In (ba_ens B) a) x h8). 
  assumption. simpl. 
  reflexivity.
assert (h11:In (im_proj1_sig S') y).
  apply Im_intro with (exist (fun a : Btype (Bc B) => In (ba_ens B) a) y h9). 
  assumption. simpl. 
  reflexivity.
specialize (h1 (exist _ _ h10) (exist _ _ h11)).
inversion h1 as [z h12 ? h13]. subst. clear h1.
assert (h14:z = (Bplus_sub (Subalg (ba_ens B) (alg_closed_ba_ens B)) S'
        (exist
           (fun a : Btype (Bc (Subalg (ba_ens B) (alg_closed_ba_ens B))) =>
            In S' a) (exist (fun a : Btype (Bc B) => In (ba_ens B) a) x h8)
           h6)
        (exist
           (fun a : Btype (Bc (Subalg (ba_ens B) (alg_closed_ba_ens B))) =>
            In S' a) (exist (fun a : Btype (Bc B) => In (ba_ens B) a) y h9)
           h7))
).
  destruct z as [z h14]. apply proj1_sig_injective.
  simpl. simpl in h13. subst.
  f_equal.
  unfold Bplus_sub. simpl. reflexivity.
rewrite <- h14 at 1; auto.

intros x y.
unfold SubBtype in h1. 
destruct x as [x h6], y as [y h7].
destruct x as [x h8], y as [y h9].
assert (h10:In (im_proj1_sig S') x).
  apply Im_intro with (exist (fun a : Btype (Bc B) => In (ba_ens B) a) x h8). 
  assumption. simpl. 
  reflexivity.
assert (h11:In (im_proj1_sig S') y).
  apply Im_intro with (exist (fun a : Btype (Bc B) => In (ba_ens B) a) y h9). 
  assumption. simpl. 
  reflexivity.
specialize (h2 (exist _ _ h10) (exist _ _ h11)).
inversion h2 as [z h12 ? h13]. subst. clear h2.
assert (h14:z = (Btimes_sub (Subalg (ba_ens B) (alg_closed_ba_ens B)) S'
        (exist
           (fun a : Btype (Bc (Subalg (ba_ens B) (alg_closed_ba_ens B))) =>
            In S' a) (exist (fun a : Btype (Bc B) => In (ba_ens B) a) x h8)
           h6)
        (exist
           (fun a : Btype (Bc (Subalg (ba_ens B) (alg_closed_ba_ens B))) =>
            In S' a) (exist (fun a : Btype (Bc B) => In (ba_ens B) a) y h9)
           h7))
).
  destruct z as [z h14]. apply proj1_sig_injective.
  simpl. simpl in h13. subst.
  f_equal.
  unfold Bplus_sub. simpl. reflexivity.
rewrite <- h14 at 1; auto.


inversion h3 as [a h6 ? h7]. subst.
assert (h8:a = (Bone (Bc (@Subalg B (ba_ens B) (alg_closed_ba_ens B))))
).
  apply proj1_sig_injective. rewrite h7 at 1.
  reflexivity.
subst.
assumption.

inversion h4 as [a h6 ? h7]. subst.
assert (h8:a = (Bzero (Bc (@Subalg B (ba_ens B) (alg_closed_ba_ens B))))
).
  apply proj1_sig_injective. rewrite h7 at 1.
  reflexivity.
subst.
assumption.

intro x.
unfold SubBtype in h1.
destruct x as [x h6]. destruct x as [x h8].
assert (h10:In (im_proj1_sig S') x).
  apply Im_intro with (exist (fun a : Btype (Bc B) => In (ba_ens B) a) x h8).
  assumption. simpl.
  reflexivity.
specialize (h5 (exist _ _ h10)).
inversion h5 as [z h12 ? h13]. subst. clear h5.
assert (h14:z = (Bcomp_sub (Subalg (ba_ens B) (alg_closed_ba_ens B)) S'
        (exist
           (fun a : Btype (Bc (Subalg (ba_ens B) (alg_closed_ba_ens B))) =>
            In S' a) (exist (fun a : Btype (Bc B) => In (ba_ens B) a) x h8)
           h6))).
  destruct z as [z h14]. apply proj1_sig_injective.
  simpl. simpl in h13. subst.
  f_equal.
  unfold Bplus_sub. simpl. reflexivity.
rewrite <- h14 at 1; auto.

intro h1.
destruct h1 as [h1 h2 h3 h4 h5].
red in h1, h2, h3, h4, h5.
constructor; red.


intros x y.
destruct x as [x h6], y as [y h7].
destruct h6 as [x h6], h7 as [y h7]. subst.
destruct x as [x h8], y as [y h9]. subst.
simpl.
unfold SubBtype in h1. simpl in h1.
unfold SubBtype in h1. simpl in h1.
specialize (h1 (exist _ _ h6) (exist _ _ h7)).
apply Im_intro with (Bplus_sub (Subalg (ba_ens B) (alg_closed_ba_ens B)) S'
            (exist (In S')
               (exist (fun x0 : Btype (Bc B) => In (ba_ens B) x0) x h8) h6)
            (exist (In S')
               (exist (fun x0 : Btype (Bc B) => In (ba_ens B) x0) y h9) h7)); auto.


intros x y.
destruct x as [x h6], y as [y h7].
destruct h6 as [x h6], h7 as [y h7]. subst.
destruct x as [x h8], y as [y h9]. subst.
simpl.
unfold SubBtype in h1. simpl in h1.
unfold SubBtype in h1. simpl in h1.
specialize (h2 (exist _ _ h6) (exist _ _ h7)).
apply Im_intro with (Btimes_sub (Subalg (ba_ens B) (alg_closed_ba_ens B)) S'
            (exist (In S')
               (exist (fun x0 : Btype (Bc B) => In (ba_ens B) x0) x h8) h6)
            (exist (In S')
               (exist (fun x0 : Btype (Bc B) => In (ba_ens B) x0) y h9) h7)); auto.

eapply Im_intro. apply h3. simpl. reflexivity.
eapply Im_intro. apply h4. simpl. reflexivity.

intros x.
destruct x as [x h6]. 
destruct h6 as [x h6]. subst.
destruct x as [x h8]. subst.
simpl.
unfold SubBtype in h5. simpl in h5.
unfold SubBtype in h5. simpl in h5.
specialize (h5 (exist _ _ h6)).
apply Im_intro with (Bcomp_sub (Subalg (ba_ens B) (alg_closed_ba_ens B)) S'
            (exist (In S')
               (exist (fun x0 : Btype (Bc B) => In (ba_ens B) x0) x h8) h6)); auto.
Qed.



Lemma two_ops_plus_closed : 
  forall {B:Bool_Alg} (A:Ensemble (bt B)),
         Inhabited A ->
         Comp_closed_sub A -> Plus_closed_sub A -> alg_closed A.
intros B A h0 h1 h2.
assert (h3:One_closed_sub A).
  destruct h0 as [x h0].
  red in h1. specialize (h1 (exist _ _ h0)).
  red in h2.
  specialize (h2 (exist _ _ h0) (exist _ _ h1)).
  unfold Bplus_sub in h2.
  simpl in h2. unfold Bcomp_sub in h2. simpl in h2.
  rewrite comp_sum in h2.
  red. assumption.
constructor; auto.
apply two_ops_imp_times; auto.
red in h3. red in h1. specialize (h1 (exist _ _ h3)).
unfold Bcomp_sub in h1. simpl in h1.
rewrite one_comp in h1.
red. assumption.
Qed.


Lemma two_ops_times_closed : 
  forall {B:Bool_Alg} (A:Ensemble (bt B)),
         Inhabited A ->
         Comp_closed_sub A -> Times_closed_sub A -> alg_closed A.
intros B A h0 h1 h2.
assert (h3:Zero_closed_sub A).
  destruct h0 as [x h0].
  red in h1. specialize (h1 (exist _ _ h0)).
  red in h2.
  specialize (h2 (exist _ _ h0) (exist _ _ h1)).
  unfold Btimes_sub in h2.
  simpl in h2. unfold Bcomp_sub in h2. simpl in h2.
  rewrite comp_prod in h2.
  red. assumption.
constructor; auto.
apply two_ops_imp_plus; auto.
red in h3. red in h1. specialize (h1 (exist _ _ h3)).
unfold Bcomp_sub in h1. simpl in h1.
rewrite zero_comp in h1.
red. assumption.
Qed.



Lemma subalg_functional : 
  forall {B:Bool_Alg} (A A':Ensemble (Btype (Bc B))),
    A = A' ->
    forall (pfa:alg_closed A) (pfa':alg_closed A'),
      Subalg A pfa = Subalg A' pfa'.
intros B A A' h1.
rewrite h1.
intros h2 h3. assert (h4:h2 = h3). apply proof_irrelevance.
subst. reflexivity.
Qed.     
                                                                 

Section Closed.
Variable B:Bool_Alg.
Let Bt := Btype (Bc B).

Lemma plus_closed : 
  forall (S:Ensemble Bt), 
    alg_closed S ->
    forall (x y:Bt), In S x -> In S y -> In S (x + y).
intros S h1 x y h2 h3.
pose proof (P_c _ _ h1) as h4.
red in h4.
specialize (h4 (exist _ _ h2) (exist _ _ h3)).
unfold Bplus_sub in h4. simpl in h4. assumption.
Qed.

Lemma times_closed : 
  forall (S:Ensemble Bt), 
    alg_closed S ->
    forall (x y:Bt), In S x -> In S y -> In S (x * y).
intros S h1 x y h2 h3.
pose proof (T_c _ _ h1) as h4.
red in h4.
specialize (h4 (exist _ _ h2) (exist _ _ h3)).
unfold Btimes_sub in h4. simpl in h4. assumption.
Qed.

Lemma comp_closed : 
  forall (S:Ensemble Bt), 
    alg_closed S ->
    forall x:Bt, In S x -> In S (- x).
intros S h1 x h2.
pose proof (C_c _ _ h1) as h4.
red in h4.
specialize (h4 (exist _ _ h2)).
unfold Bcomp_sub in h4. simpl in h4. assumption.
Qed.

Lemma zero_closed : 
  forall (S:Ensemble Bt), 
    alg_closed S -> In S 0.
intros S h1.
pose proof (Z_c _ _ h1) as h4.
red in h4.
assumption.
Qed.

Lemma one_closed : 
  forall (S:Ensemble Bt), 
    alg_closed S -> In S 1.
intros S h1.
pose proof (O_c _ _ h1) as h4.
red in h4.
assumption.
Qed.

End Closed.


Lemma subalg_preserves_atom : 
  forall {B:Bool_Alg} (A:Ensemble (bt B)) 
         (pfc:alg_closed A) (x:bt B) (pfi:In A x),
   atom x ->
   atom (exist _ _ pfi) (B:=Subalg _ pfc).
intros B A h1 x h2.
intro h3. red in h3. red.
destruct h3 as [h3 h4]. split.
intro h5. subst. contradict h3. simpl in h5. 
pose proof (f_equal (@proj1_sig _ _) h5) as h5'. clear h5.
simpl in h5'. assumption.
intros b h5.
destruct b as [b h6].
assert (h7:le b x).
red in h5. red. simpl in h5. apply exist_injective in h5. 
unfold Bplus_sub in h5. simpl in h5. assumption.
specialize (h4 _ h7).
destruct h4 as [h4a | h4b]; subst. left.
apply proj1_sig_injective. simpl. reflexivity. 
right. apply proj1_sig_injective. simpl.
reflexivity.
Qed.




Section Families_Ind_Families_Subalgebras.
Variable B:Bool_Alg.
Let Bt := Btype (Bc B).
Variable It : Type.
Variable A:Ensemble Bt.

(*Subalgebra -- Indexed representation*)
Record SubalgI := {  
  sai_A:It->(Ensemble Bt);
  sai_closed:forall (i:It), alg_closed (sai_A i);
  sai_B := fun (i:It) =>  Subalg 
    (sai_A i) (sai_closed i)
  }.

(*Subalgebra -- Family representation*)
Record SubalgF  := {
  saf_A:Family Bt;
  saf_closed : forall (S:Ensemble Bt), In saf_A S -> alg_closed S;
  saf_B := fun (S:{S':Ensemble Bt | In saf_A S'}) =>
    Subalg (proj1_sig S) (saf_closed (proj1_sig S) (proj2_sig S))}.

Variable SI:SubalgI.
Variable SF:SubalgF.

Definition int_sai_A := IndexedIntersection (sai_A SI).
Definition int_saf_A := FamilyIntersection (saf_A SF).
 
Lemma Plus_closed_i : Plus_closed_sub int_sai_A.
red.
intros x y.
assert (h1: In int_sai_A (proj1_sig x)).
apply proj2_sig.
assert (h2: In int_sai_A (proj1_sig y)).
apply proj2_sig.
inversion h1. inversion h2.
assert (h3: forall (i:It), Plus_closed_sub (sai_A SI i)).
  intro i.
  apply (P_c _ _ (sai_closed SI i)).
apply indexed_intersection_intro.
intro i.
apply Plus_closed''. 
apply (h3 i). 
apply (H i). 
apply (H1 i).
Qed.

Lemma Plus_closed_F : Plus_closed_sub int_saf_A.
red.
intros x y.
constructor.
intros S h1.
apply Plus_closed''.
apply (P_c _ S (saf_closed SF S h1)).
destruct x as [? h2].
simpl.
inversion h2 as [? h3].
apply (h3 S h1).
destruct y as [? h4].
simpl.
inversion h4 as [? h5].
apply (h5 S h1).
Qed.


(*Dual*)
Lemma Times_closed_i : Times_closed_sub int_sai_A.
red.
intros x y.
assert (h1: In int_sai_A (proj1_sig x)).
apply proj2_sig.
assert (h2: In int_sai_A (proj1_sig y)).
apply proj2_sig.
inversion h1. inversion h2.
assert (h3: forall (i:It), Times_closed_sub (sai_A SI i)).
  intro i.
  apply (T_c _ _ (sai_closed SI i)).
apply indexed_intersection_intro.
intro i.
apply Times_closed''. 
apply (h3 i). 
apply (H i). 
apply (H1 i).
Qed.

Lemma Times_closed_F : Times_closed_sub int_saf_A.
red.
intros x y.
constructor.
intros S h1.
apply Times_closed''.
apply (T_c _ S (saf_closed SF S h1)).
destruct x as [? h2].
simpl.
inversion h2 as [? h3].
apply (h3 S h1).
destruct y as [? h4].
simpl.
inversion h4 as [? h5].
apply (h5 S h1).
Qed.

Lemma Comp_closed_i : Comp_closed_sub int_sai_A.
red.  intro x.
assert (h1: In int_sai_A (proj1_sig x)).
apply proj2_sig.
assert (h2: forall (i:It), Comp_closed_sub (sai_A SI i)).
  intro i.
  apply (C_c _ _ (sai_closed SI i)).
apply indexed_intersection_intro.
intro i.
apply Comp_closed''.
apply (h2 i).
inversion h1.
apply (H i).
Qed.

Lemma Comp_closed_F : Comp_closed_sub int_saf_A.
red.
intros x.
constructor.
intros S h1.
apply Comp_closed''.
apply (C_c _ S (saf_closed SF S h1)).
destruct x as [? h2]. simpl.
inversion h2 as [? h3]. 
apply (h3 S h1).
Qed.

Lemma One_closed_i : One_closed_sub int_sai_A.
red.
apply indexed_intersection_intro.
intro i.
pose proof (O_c _ _ (sai_closed SI i)) as h1.
unfold One_closed_sub in h1.
assumption.
Qed.

Lemma One_closed_F : One_closed_sub int_saf_A.
red.
constructor.
intros S h1.
apply (O_c _ S (saf_closed SF S h1)).
Qed.

Lemma Zero_closed_i : Zero_closed_sub int_sai_A.
red.
apply indexed_intersection_intro.
intro i.
pose proof (Z_c _ _ (sai_closed SI i)) as h1.
unfold Zero_closed_sub in h1.
assumption.
Qed.

Lemma Zero_closed_F : Zero_closed_sub int_saf_A.
red.
constructor.
intros S h1.
apply (Z_c _ S (saf_closed SF S h1)).
Qed.


Lemma int_sai_alg_closed : alg_closed int_sai_A.
constructor.
apply Plus_closed_i.
apply Times_closed_i.
apply One_closed_i.
apply Zero_closed_i.
apply Comp_closed_i.
Qed.

Lemma int_saf_alg_closed : alg_closed int_saf_A.
constructor.
apply Plus_closed_F.
apply Times_closed_F.
apply One_closed_F.
apply Zero_closed_F.
apply Comp_closed_F.
Qed.


Definition int_sub_algi := Subalg _ int_sai_alg_closed.
Definition int_sub_algF := Subalg _ int_saf_alg_closed.


(*This defines the family of all subalgebra sets that contain A, as a set*)

Definition S_clo_cont_A := [S:Ensemble Bt | Included A S /\ alg_closed S].

(* Ditto as a subalgebra family *)
Definition SA_cont_A : SubalgF. 
pose S_clo_cont_A as F.
red in F.
assert (h1: forall (S:Ensemble Bt), In F S -> alg_closed S).
  intros S h2.
  inversion h2 as [h3].  destruct h3 as [h3l h3r]. assumption.
refine (Build_SubalgF F h1).
Defined.

Lemma saf_A_eq : saf_A SA_cont_A = S_clo_cont_A.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros S h1.
inversion h1.
constructor; assumption.
(* >= *)
red.
intros S h1.
unfold S_clo_cont_A in h1.
inversion h1.
simpl. constructor; assumption.
Qed.

End Families_Ind_Families_Subalgebras.
Arguments SA_cont_A [B] _.
Arguments int_saf_A [B] _ _.
Arguments int_saf_alg_closed [B] _.
Arguments S_clo_cont_A [B] _ _.
Arguments saf_A [B] _ _.
Arguments saf_A_eq [B] _.
Arguments saf_closed [B] _ _ _.

Section GenAlg.
Variable B:Bool_Alg.
Let Bt := bt B.


Definition Gen_Ens (A:Ensemble Bt) : Ensemble Bt := 
  FamilyIntersection (S_clo_cont_A A).

Lemma closed_gen_ens : forall (E:Ensemble Bt), alg_closed (Gen_Ens E).
intro E. unfold Gen_Ens.
constructor.
(* + *)
red.
intros x y.
destruct x as [x h1].
destruct y as [y h2].
destruct h1 as [x h1]. destruct h2 as [y h2].
constructor.
intros S h3. 
pose proof (h1 _ h3) as h4.
pose proof (h2 _ h3) as h5.
unfold Bplus_sub. simpl.
destruct h3 as [h3]. destruct h3 as [h3l h3r].
apply plus_closed; auto.
(* * *)
red.
intros x y.
destruct x as [x h1].
destruct y as [y h2].
destruct h1 as [x h1]. destruct h2 as [y h2].
constructor.
intros S h3. 
pose proof (h1 _ h3) as h4.
pose proof (h2 _ h3) as h5.
unfold Bplus_sub. simpl.
destruct h3 as [h3]. destruct h3 as [h3l h3r].
apply times_closed; auto.
(* 1 *)
red. constructor.
intros S h3.
destruct h3 as [h3]. destruct h3 as [h3l h3r].
apply one_closed; auto.
(* 0 *)
red. constructor.
intros S h3.
destruct h3 as [h3]. destruct h3 as [h3l h3r].
apply zero_closed; auto.
(* - *)
red. intro x.
destruct x as [x h1].
destruct h1 as [x h1].
constructor.
intros S h3.
pose proof (h1 _ h3) as h4.
unfold Bcomp_sub. simpl.
destruct h3 as [h3]. destruct h3 as [h3l h3r].
apply comp_closed; auto.
Qed.

Definition Gen (E:Ensemble Bt) := Subalg _ (closed_gen_ens E).


Lemma gen_ens_minimal : forall (A S:Ensemble Bt), 
  let F := (S_clo_cont_A A) in In F S -> Included (Gen_Ens A) S.
intros A S F h0.
unfold Gen_Ens.
red.
intros x h2.
inversion h2 as [? h3]. subst.
apply h3.  assumption.
Qed.  

Lemma gen_ens_includes : forall (S:Ensemble Bt), Included S (Gen_Ens S).
intro S.
red.
intros x h1.
constructor.
intros S0 h2.
inversion h2 as [h3].
destruct h3. auto with sets.
Qed.

Lemma gen_minimal'': forall (A S:Ensemble Bt), alg_closed S ->
  Included A S -> 
  Included (int_saf_A (SA_cont_A A)) S.
intros A S h1 h2.
unfold int_saf_A.
pose proof (saf_A_eq A) as h3.
rewrite h3.
red.
intros x h4.
inversion h4 as [? h5].
apply h5. constructor. split; assumption.
Qed.
End GenAlg.

Arguments alg_closed [B] _.
Arguments Gen_Ens [B] _ _.
Arguments Gen [B] _.


Section Gen_examples.
Variable B:Bool_Alg.
Let Bt := Btype (Bc B).

Example alg_closed_two : alg_closed (@Couple Bt 0 1).
  constructor.
  (* + *)
  red. intros x y.
  destruct x as [x h2].  destruct y as [y h3].
  unfold Bplus_sub. simpl.
  pose proof (Couple_inv _ _ _ _ h2) as h4.
  pose proof (Couple_inv _ _ _ _ h3) as h5.
  destruct h4 as [h4l | h4r].
    (*h4l*)
    destruct h5 as [h5l | h5r].
      (*h5l*)
      rewrite h4l. rewrite h5l.
      pose proof (@zero_sum B 0) as h6.
      rewrite h6.
      apply Couple_l.
      (*h5r*)
      rewrite h4l. rewrite h5r.
      pose proof (@one_sum B 0) as h6.
      rewrite h6.
      apply Couple_r.
    (*h4r*)
    destruct h5 as [h5l | h5r].
      (*h5l*)
      rewrite h4r. rewrite h5l.
      pose proof (@zero_sum B 1) as h6.
      rewrite h6.
      apply Couple_r.
      (*h5r*)
      rewrite h4r. rewrite h5r.
      pose proof (@one_sum B 1) as h6.
      rewrite h6.
      apply Couple_r.
  (* * *)
  red.  
  intros x y.
  destruct x as [x h1]. destruct y as [y h2].
  unfold Btimes_sub. simpl.
  pose proof (Couple_inv _ _ _ _ h1) as h3.
  pose proof (Couple_inv _ _ _ _ h2) as h4.
  destruct h3 as [h3l | h3r].
  (*h3l*)
    destruct h4 as [h4l | h4r].
    (*h4l*)
    rewrite h3l. rewrite h4l.
    pose proof (@zero_prod B 0) as h5.
    rewrite h5.
    apply Couple_l.
    (*h4r*)
    rewrite h3l. rewrite h4r.
    pose proof (@one_prod B 0) as h5.
    rewrite h5.
    apply Couple_l.
  (*h3r*)
    destruct h4 as [h4l | h4r].
    (*h4l*)
    rewrite h3r. rewrite h4l.
    pose proof (@zero_prod B 1) as h5.
    rewrite h5.
    apply Couple_l.
    (*h4r*)
    rewrite h3r. rewrite h4r.
    pose proof (@one_prod B 1) as h5.
    rewrite h5.
    apply Couple_r.
  (* 1 *)
  red. apply Couple_r.
  (* 0 *)
  red. apply Couple_l.
  (* - *)
  red. intro x.
  destruct x as [x h1].
  unfold Bcomp_sub. simpl.
  pose proof (Couple_inv _ _ _ _ h1) as h2.
  destruct h2 as [h2l | h2r].
  (*h2l*)
  rewrite h2l.
  pose proof (@zero_comp B) as h3.
  rewrite h3.
  apply Couple_r.
  (*h2r*)
  rewrite h2r.
  pose proof (@one_comp B) as h3.
  rewrite h3.
  apply Couple_l.
Qed.


Definition two : Bool_Alg.
pose (@Couple Bt 0 1) as A.
pose proof (alg_closed_two) as h1.
refine (Subalg A h1).
Defined.

Example two_smallest_subalgebra : forall (A:Ensemble Bt), alg_closed A -> 
  let C := (Couple 0 1) in Included C A.
intros A h1 C.
red.
intros x h2.
pose proof (Couple_inv _ _ _ _ h2) as h3.
destruct h3 as [h3l | h3r].
rewrite h3l.
apply (Z_c _ _ h1).
rewrite h3r.
apply (O_c _ _ h1).
Qed.

Lemma gen_ens_closed_eq : forall (S:Ensemble Bt), alg_closed S -> Gen_Ens S = S.
intros S h1.
apply Extensionality_Ensembles.
red. split.
(* <= *)
apply gen_ens_minimal. 
constructor. split. auto with sets. assumption.
(* >= *)
apply gen_ens_includes.
Qed.

(*"if E is empty and A is not degenerate, then the
subalgebra generated by E is the smallest possible subalgebra of A, namely 2."*)

Example gen_ens_empty : Gen_Ens (Empty_set Bt) = (Couple 0 1).
unfold Gen_Ens. 
apply Extensionality_Ensembles.
red. split.
(* <= *)
apply gen_ens_minimal.
constructor. split; auto with sets.
apply alg_closed_two.

(* >= *)
red.
intros x h1.
constructor.
intros S h2.
inversion h2 as [h3].
destruct h3 as [? h4].
pose proof (Couple_inv _ _ _ _ h1) as [h5 | h6].
rewrite h5.
apply (Z_c _ _ h4).
rewrite h6.
apply (O_c _ _ h4).
Qed.

Example gen_empty : Gen (Empty_set Bt) = two.
unfold Gen. unfold two.
generalize (closed_gen_ens B (Empty_set Bt)).
rewrite gen_ens_empty.
intros h1.
assert (h1 = alg_closed_two). apply proof_irrelevance.
subst.
reflexivity.
Qed.



End Gen_examples.


Section More_Gen_Ens.


(*"A simple but useful remark for subsets E and F of a Boolean algebra A is
that if F is included in E, then the subalgebra generated by F is included (as
a subalgebra) in the subalgebra generated by E."*)

Lemma gen_ens_preserves_inclusion : forall {B:Bool_Alg} (S U: Ensemble (Btype (Bc B))), Included S U ->
  Included (Gen_Ens S) (Gen_Ens U).
intros B S U h1.
unfold Gen_Ens.
red.
intros x h2.
inversion h2 as [? h3].
assert (forall S0, In (S_clo_cont_A U) S0 -> In S0 x).
  intros S0 h5.
  assert (h6:In (S_clo_cont_A S) S0).
  inversion h5 as [h6].
  destruct h6.
  assert (h7:Included S S0).
    auto with sets.
  constructor. split; assumption.
  apply h3; assumption.
constructor; assumption.
Qed.


Lemma two_incl_gen_ens : forall {B:Bool_Alg} (E:Ensemble (bt B)),
                           Included (Couple 0 1) (Gen_Ens E).
intros B E.
assert (h1:Included (Empty_set _) E). auto with sets.
pose proof (gen_ens_preserves_inclusion _ _ h1) as h2.
rewrite gen_ens_empty in h2.
assumption.
Qed.


Lemma inhabited_gen_ens : forall {B:Bool_Alg} (E:Ensemble (bt B)),
                           Inhabited (Gen_Ens E).
intros B E.
pose proof (two_incl_gen_ens E) as h1.
assert (h2:In (Couple 0 1) (Bzero (Bc B))).
constructor.
apply Inhabited_intro with 0.
auto with sets.
Qed.



(*"The relation of one Boolean algebra being a subalgebra of another is a
partial order on the class of all Boolean algebras."  Technically I don't define
[subalg_of] exactly like that, since I presume an underlying algebra B, but 
my definition is equivalent to the informal definition given in the quotes, 
so I know this definition and lemma are pretty vacuous.*)

Definition subalg_of {B:Bool_Alg} (A1 A2:{A:Ensemble (Btype (Bc B)) | alg_closed A}) : Prop := 
  (Included (proj1_sig A1) (proj1_sig A2)).

Lemma order_subalg_of : forall B:Bool_Alg, Order _ (@subalg_of B).
constructor.
(*refl*)
red.
intro.
red.
auto with sets.
(*trans*)
red. unfold subalg_of. intros.
auto with sets.
(*antisymm*)
red. unfold subalg_of. intros A1 A2 h1 h2.
cut ((proj1_sig A1) = (proj1_sig A2)).
apply proj1_sig_injective.
auto with sets.
Qed.


(*"The family is said to be directed if any two members B_i
and B_j of the family are always subalgebras of some third member Bk."*)

Definition directed_sa {A:Bool_Alg} (F:(SubalgF A)) : Prop := forall (S1 S2:Ensemble (Btype (Bc A))), 
  In (saf_A F) S1 -> In (saf_A F) S2 -> (exists (S3:Ensemble (Btype (Bc A))),
    In (saf_A F) S3 /\
    Included S1 S3 /\ Included S2 S3).


(*"Lemma 1. The union of a non-empty, directed family of subalgebras is again
a subalgebra."*)

Lemma union_directed_Subalgf_closed : forall {B:Bool_Alg} (F:(SubalgF B)), directed_sa F -> 
  Inhabited (saf_A F) -> alg_closed (FamilyUnion (saf_A F)).
intros B F h1 h2.
constructor.
(* + *)
red.
intros p q.
destruct p as [p h3]. destruct q as [q h4].
unfold Bplus_sub. simpl.
inversion h3 as [B_i ? h5 h6].
inversion h4 as [B_j ? h7 h8].
unfold directed_sa in h1.
pose proof (h1 _ _ h5 h7) as h9.
elim h9. intros B_k h10.
destruct h10 as [h10 [h11l h11r]].
assert (h12:In B_k p).  auto with sets.
assert (h13:In B_k q).  auto with sets.
pose proof (saf_closed F B_k h10) as h14.
pose proof (P_c _ _ h14) as h15.
red in h15.
pose proof (h15 (exist _ p h12) (exist _ q h13)) as h16.
unfold Bplus_sub in h16.
simpl in h16.
apply family_union_intro with B_k; assumption.
(* * *)
red.
intros p q.
destruct p as [p h3]. destruct q as [q h4].
unfold Btimes_sub. simpl.
inversion h3 as [B_i ? h5 h6].
inversion h4 as [B_j ? h7 h8].
unfold directed_sa in h1.
pose proof (h1 _ _ h5 h7) as h9.
elim h9. intros B_k h10.
destruct h10 as [h10 [h11l h11r]].
assert (h12:In B_k p).  auto with sets.
assert (h13:In B_k q).  auto with sets.
pose proof (saf_closed F B_k h10) as h14.
pose proof (T_c _ _ h14) as h15.
red in h15.
pose proof (h15 (exist _ p h12) (exist _ q h13)) as h16.
unfold Btimes_sub in h16.
simpl in h16.
apply family_union_intro with B_k; assumption.
(* 1 *)
red.
inversion h2 as [S h3].
pose proof (saf_closed F S h3) as h4.
pose proof (O_c _ _ h4) as h5.
red in h5.
apply family_union_intro with S; assumption.
(* 0 *)
red.
inversion h2 as [S h3].
pose proof (saf_closed F S h3) as h4.
pose proof (Z_c _ _ h4) as h5.
red in h5.
apply family_union_intro with S; assumption.
(* - *)
red. intro p.
destruct p as [p h3].
unfold Bcomp_sub. simpl.
inversion h3 as [B_i ? h4 h5].
pose proof (saf_closed F B_i h4) as h6.
pose proof (C_c _ _ h6) as h7.
red in h7.
pose proof (h7 (exist _ p h5)) as h8.
red in h8. unfold Bcomp_sub in h8.
simpl in h8.
apply family_union_intro with B_i; assumption.
Qed.

(* "More precisely, a family {Bi } of subalgebras is a chain if for any two members Bi
and Bj of the family, either Bi is a subalgebra of Bj , or vice versa." *)

Definition chain_subalg {B:Bool_Alg} (F:SubalgF B) : Prop := chain (saf_A F).

Lemma chain_impl_directed : forall {B:Bool_Alg} (F:SubalgF B), chain_subalg F -> directed_sa F.
intros B F h1.
red in h1.
red.
intros S1 S2 h2 h3.
red in h1.
pose proof (h1 _ _ h2 h3) as h4.
case h4 as [h4l | h4r].
  (* left *)
  exists S2. split. assumption.
  split. assumption. auto with sets.
  (* right *)
  exists S1. split. assumption.
  split. auto with sets. assumption. 
Qed. 

  
Lemma union_chain_Subalgf_closed : forall {B:Bool_Alg} (F:SubalgF B), chain_subalg F -> 
  Inhabited (saf_A F) -> alg_closed (FamilyUnion (saf_A F)).
intros B F h1 h2.
apply union_directed_Subalgf_closed.
apply chain_impl_directed; assumption.
assumption.
Qed.

(** "Corollary 1. Let $A$ be a Boolean algebra generated by a set $E$, and for each finite subset $F$ of $E$, let $B_F$ be the subalgebra of $A$ generated by $F$. The family $\{B_F:F \subseteq E \text{ and }  F \text{ is finite}\}$ is directed, and its union is $A$."
[or in my terminology]: 
Assume B is generated by a set E, and for each
finite subset S of E, let A be the subalgebra of B generated by S. The family
{A : A is generated by S and S <= E and S is finite} is directed, and its union is B. *)

Lemma family_included_subs_directed_eq_B : 
  forall {B:Bool_Alg},
    let Bt := (Btype (Bc B)) in 
    forall (E:Ensemble Bt) (F:SubalgF B), 
      (Full_set Bt) = Gen_Ens E -> (saf_A F) = 
                                   [A:Ensemble Bt | exists (S:Ensemble Bt), 
                                                    A = Gen_Ens S /\ Included S E /\ Finite S] -> 
      directed_sa F /\ FamilyUnion (saf_A F) = (Full_set Bt).
intros B Bt E F h1 h2.
assert (h3:directed_sa F).
  red.
  intros S1' S2' h3 h4.
  rewrite h2.
  rewrite h2 in h3. rewrite h2 in h4.
  inversion h3 as [h5].
  inversion h4 as [h6].
  elim h5. intros S1 h7. destruct h7 as [h7a [h7b h7c]].
  elim h6. intros S2 h8. destruct h8 as [h8a [h8b h8c]].
  assert (h9 :Included S1 (Union S1 S2)). auto with sets.
  assert (h10:Included S2 (Union S1 S2)). auto with sets.
  pose proof (gen_ens_preserves_inclusion S1 (Union S1 S2) h9)  as h11.
  pose proof (gen_ens_preserves_inclusion S2 (Union S1 S2) h10) as h12.
  exists (Gen_Ens (Union S1 S2)).
  constructor. constructor.
  exists (Union S1 S2). split. reflexivity.
  split. auto with sets. apply Union_preserves_Finite;assumption.
  rewrite h7a. rewrite h8a.
  split; assumption.

split. assumption.
(* F's union is Full_set *)
pose proof (@union_directed_Subalgf_closed B) as h4.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red. intros x ?. constructor.
(* >= *)
assert (h5:Included E (FamilyUnion (saf_A F))).
  red. intros x h6.
  pose proof (In_singleton _ x) as h7.
  apply family_union_intro with (Gen_Ens (Singleton x)).
  rewrite h2.
  constructor. exists (Singleton x). split. reflexivity.
  split. red. intros x' h8.
  pose proof (Singleton_inv _ _ _ h8) as h9.
  rewrite <- h9. assumption.
  apply Singleton_is_finite.
  constructor. intros S h8.
  inversion h8 as [h9]. destruct h9 as [h9l h9r].
  auto with sets. 
rewrite h1.
apply gen_ens_minimal.
constructor.
split. assumption.
pose proof (Inhabited_intro _ (saf_A F) (Couple 0 1)) as h6.
assert (h7:In (saf_A F) (Couple 0 1)).
  rewrite h2.
  constructor. exists (Empty_set Bt).
  split.
  (*left*) 
  rewrite <- gen_ens_empty. reflexivity.
  (*right*)
  split. auto with sets. constructor.
pose proof (h6 h7) as h8.
pose proof (h4 _ h3 h8). assumption. 
Qed.


Lemma family_included_subs_closed : 
  forall {A:Bool_Alg},
    let At:=Btype (Bc A) in
    forall (E S:Ensemble At),
      In [A:Ensemble At | exists (S:Ensemble At), 
                          A = Gen_Ens S /\ Included S E /\ Finite S] S -> alg_closed S.
intros A At E S h1.
destruct h1 as [h1]. destruct h1 as [F h1]. destruct h1 as [h1a [h1b h1c]].
subst.
apply closed_gen_ens.
Qed.


Lemma family_included_subs_directed_eq_B' :
  forall {B:Bool_Alg},
    let Bt := (Btype (Bc B)) in
    forall (E:Ensemble Bt),
      let saf :=[A:Ensemble Bt | exists S:Ensemble Bt,
    A = Gen_Ens S /\ Included S E /\ Finite S] in
    let F := Build_SubalgF B saf (family_included_subs_closed E)  in
  directed_sa F /\ FamilyUnion saf = Gen_Ens E.
intros B Bt E saf F.  
assert (h1:directed_sa F).
intros S1 S2 h1 h2. simpl in h1. simpl in h2.
destruct h1 as [h1]. destruct h1 as [S h1]. destruct h1 as [h1a [h1b h1c]].
destruct h2 as [h2]. destruct h2 as [S' h2]. destruct h2 as [h2a [h2b h2c]].
subst.
assert (h9 :Included S (Union S S')). auto with sets.
assert (h10:Included S' (Union S S')). auto with sets. 
pose proof (gen_ens_preserves_inclusion S (Union S S') h9)  as h11.
pose proof (gen_ens_preserves_inclusion S' (Union S S') h10) as h12.
exists (Gen_Ens (Union S S')). 
split; try tauto.
simpl.
constructor.  
exists (Union S S'). split; auto.
split; try apply Union_preserves_Finite; auto with sets.
pose proof (@union_directed_Subalgf_closed B) as h4.
split. apply h1.
apply Extensionality_Ensembles. 
red. split. 
red. intros x h5. 
destruct h5 as [S x h5 h6].
destruct h5 as [h5].
destruct h5 as [S' h5]. destruct h5 as [h5a [h5b h5c]].
subst.
pose proof (gen_ens_preserves_inclusion _ _ h5b x h6) as h7.
assumption.
red. intros S' h5.
assert (h6:Included E (FamilyUnion saf)).
  red. intros x h6.
  pose proof (In_singleton _ x) as h7.
  apply family_union_intro with (Gen_Ens (Singleton x)).
  constructor. exists (Singleton x). split. reflexivity.
  split. red. intros x' h8.
  pose proof (Singleton_inv _ _ _ h8) as h9.
  rewrite <- h9. assumption.
  apply Singleton_is_finite.
  constructor. intros S h8.
  inversion h8 as [h9]. destruct h9 as [h9l h9r].
auto with sets. 

assert (h7:In (S_clo_cont_A E) (FamilyUnion saf)).
  constructor. split; auto.
specialize (h4 _ h1).
apply h4.
apply (Inhabited_intro _ (saf_A F) (Couple 0 1)).
constructor.
exists (Empty_set _). split.
rewrite gen_ens_empty. reflexivity.
split; auto with sets.
pose proof (gen_ens_minimal _ E (FamilyUnion saf) h7) as h8.
auto with sets.
Qed.



(*"For any Boolean algebra A [B in my terminology], 
apply the preceding corollary to the set E = A [B]
to conclude that every Boolean algebra is the (directed) union of its finitely
generated subalgebras."*)

Corollary family_included_subs_directed_eq_B_full_set : 
  forall {B:Bool_Alg}, 
    let Bt:=(Btype (Bc B)) in
    forall   (F:SubalgF B),
      (saf_A F) = [A:Ensemble Bt | exists (S:Ensemble Bt), 
                                   A = Gen_Ens S /\ Finite S] -> 
      directed_sa F /\ FamilyUnion (saf_A F) = Full_set Bt.
Proof.
intros B Bt F h1.
pose proof (family_included_subs_directed_eq_B (Full_set Bt) F) as h2.
assert (h3:Full_set Bt = Gen_Ens (Full_set Bt)).
  apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red. intros x ?.
  unfold Gen_Ens.
  constructor.
  intros S h4.
  inversion h4 as [h5].
  destruct h5; auto with sets.
  (* >= *)
  red. intros.  constructor. 
assert (h4:saf_A F = [A : Ensemble Bt
     | exists S : Ensemble Bt,
         A = Gen_Ens S /\ Included S (Full_set Bt) /\ Finite S]).
  rewrite h1.
  apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red.
  intros S h4.
  inversion h4 as [h5].
  destruct h5 as [E h6].
  destruct h6.
  constructor.
  exists E.
  repeat split; auto with sets.
  (* >= *)
  red.  intros S h4.
  inversion h4 as [h5].
  destruct h5 as [E h6].
  destruct h6 as [? []].
  constructor.
  exists E.
  split; assumption.
apply h2; assumption.
Qed.


Lemma gen_ba_eq_ba : 
  forall {B:Bool_Alg} (E:Ensemble (bt B)),
    ba_ens B = Gen_Ens E ->
    Subalg _ (alg_closed_ba_ens B) = Gen E.
intros B E h1.
apply bc_inj.
unfold Subalg, Gen. unfold Subalg.
simpl.
pose proof (subsetT_eq_compat _ _ _ _ (alg_closed_ba_ens B) (closed_gen_ens B E) h1) as h2.
dependent rewrite -> h2.
reflexivity.
Qed.



Lemma gen_ba_eq_type : 
  forall {B:Bool_Alg} (E:Ensemble (bt B)),
    ba_ens B = Gen_Ens E ->
    bt (Gen E) = sig_set (ba_ens B).
intros B E h1. 
destruct B.
unfold bt, Gen.
simpl. unfold SubBtype. simpl.
destruct Bc. simpl. simpl in h1.
rewrite <- h1.
unfold ba_ens. simpl.
reflexivity.
Qed.





End More_Gen_Ens.
 
Require Import ListUtilities.

Section Closed_Finite.
Variable B:Bool_Alg.
Let Bt:= bt B.

Lemma plus_list_closed : 
  forall (S:Ensemble Bt) (l:list Bt),
    alg_closed S -> Included (list_to_set l) S -> 
    Ensembles.In S (plus_list l).
intros S l.
induction l as [|a l h1].
intros h1 h2. simpl. apply zero_closed. assumption.
intros h2 h3.
simpl. simpl in h3.
assert (h4:Included (list_to_set l) (Add (list_to_set l) a)).
  auto with sets.
assert (h5:Included (list_to_set l) S). auto with sets.
specialize (h1 h2 h5).
apply plus_closed; auto.
pose proof (Add_intro2 _ (list_to_set l) a).
auto with sets.
Qed.

Lemma times_list_closed : 
  forall (S:Ensemble Bt) (l:list Bt),
    alg_closed S -> Included (list_to_set l) S -> 
    Ensembles.In S (times_list l).
intros S l.
induction l as [|a l h1].
intros h1 h2. simpl. apply one_closed. assumption.
intros h2 h3.
simpl. simpl in h3.
assert (h4:Included (list_to_set l) (Add (list_to_set l) a)).
  auto with sets.
assert (h5:Included (list_to_set l) S). auto with sets.
specialize (h1 h2 h5).
apply times_closed; auto.
pose proof (Add_intro2 _ (list_to_set l) a).
auto with sets.
Qed.

Lemma plus_set_closed : 
  forall (R S:Ensemble Bt) (pf:Finite R),
    alg_closed S -> Included R S ->
    Ensembles.In S (plus_set R pf).
intros R S h1 h2 h3.
pose proof (finite_set_list _  h1) as h4.
destruct h4 as [l h4]. subst. 
pose proof (plus_set_compat' _ _ h1 (eq_refl _)) as h4.
rewrite h4.
apply plus_list_closed; auto.
Qed.

Lemma times_set_closed : 
  forall (R S:Ensemble Bt) (pf:Finite R),
    alg_closed S -> Included R S ->
    Ensembles.In S (times_set R pf).
intros R S h1 h2 h3.
pose proof (finite_set_list _  h1) as h4.
destruct h4 as [l h4]. subst.
pose proof (times_set_compat' _ _ h1 (eq_refl _)) as h4.
rewrite h4.
apply times_list_closed; auto.
Qed.

End Closed_Finite.

Section SubCompat.
Variable B:Bool_Alg.
Let Bt:= Btype (Bc B).


Lemma plus_list_sub_compat : 
  forall (A:Ensemble Bt) (pf:alg_closed A)
         (l:list (bt (Subalg _ pf))),
    plus_list (map (@proj1_sig _ _) l) = proj1_sig (plus_list l).
intros S h1 l.
induction l as [|a l h2].
simpl. reflexivity.
simpl. rewrite h2.
unfold Bplus_sub.
reflexivity.
Qed.

Lemma times_list_sub_compat : 
  forall (A:Ensemble Bt) (pf:alg_closed A)
         (l:list (bt (Subalg _ pf))),
    times_list (map (@proj1_sig _ _) l) = proj1_sig (times_list l).
intros S h1 l.
induction l as [|a l h2].
simpl. reflexivity.
simpl. rewrite h2.
unfold Btimes_sub.
reflexivity.
Qed.

Lemma plus_set_sub_compat : forall (A:Ensemble Bt) (pf:alg_closed A)
         (E:Ensemble (bt (Subalg _ pf))) (pfe:Finite E)
         (pfi:Finite (Im E (@proj1_sig _ _))),
    plus_set (Im E (@proj1_sig _ _)) pfi = proj1_sig (plus_set E pfe). 
intros S h1 E h2 h3.
pose proof (finite_set_list _ h2) as h4.
pose proof (finite_set_list _  h3) as h5.
destruct h4 as [l h4].
destruct h5 as [l' h5]. subst.
generalize dependent h3.
rewrite <- map_im_compat.
intro h3.
pose proof plus_set_compat'.
rewrite (plus_set_compat' _ _ h3 (eq_refl _)).
rewrite (plus_set_compat' _ _ h2 (eq_refl _)).
unfold Bt.
rewrite (plus_list_sub_compat _ h1). 
reflexivity.
Qed.

Lemma times_set_sub_compat : forall (A:Ensemble Bt) (pf:alg_closed A)
         (E:Ensemble (bt (Subalg _ pf))) (pfe:Finite E)
         (pfi:Finite (Im E (@proj1_sig _ _))),
    times_set (Im E (@proj1_sig _ _)) pfi = proj1_sig (times_set E pfe). 
intros S h1 E h2 h3.
pose proof (finite_set_list _ h2) as h4.
pose proof (finite_set_list _  h3) as h5.
destruct h4 as [l h4].
destruct h5 as [l' h5]. subst.
generalize dependent h3.
rewrite <- map_im_compat.
intro h3.
pose proof plus_set_compat'.
rewrite (times_set_compat' _ _ h3 (eq_refl _)).
rewrite (times_set_compat' _ _ h2 (eq_refl _)).
unfold Bt.
rewrite (times_list_sub_compat _ h1). 
reflexivity.
Qed.


(*This section is for the unfortunate syntactic difference 
  between the subalgebra of a full algebra (improper subalgebra), 
  and the full algebra itself.  
  These are formally different, and if that bothers you,
  then you can always use [Bool_Alg_p]s, in "Section ParametricAnalogues".  With those, you can easily express the reflexivity
  of the subalgebra relation.*)

Section SubalgBaEnsCompat.


(*This definition is used to assert that a predicate on 
  elements of an algebra behave the same, whether or not 
  those elements are considered to be elements of a the improper
  subalgebra or elements of the algebra itself.*)
Definition full_sig_equivalent (P:{C:Bool_Alg & (bt C)}->Prop) :=
  forall (x:Bt),
    P (existT _ B x) <-> P (existT _ (Subalg _ (alg_closed_ba_ens B)) (exist _ _ (Full_intro _ x))).

Lemma full_sig_equivalent_atom : 
  full_sig_equivalent (fun x:{C:Bool_Alg & (bt C)} => atom (projT2 x)).
red.
simpl. intro x.
split.
intro h1.
red in h1. red.
destruct h1 as [h1l h1r].
split.
intro h2.
pose proof (f_equal (@proj1_sig _ _) h2) as h3.
simpl in h3. contradiction.
intros b h2.
destruct b as [b].
red in h2. simpl in h2. unfold Bplus_sub in h2. simpl in h2.
pose proof (f_equal (@proj1_sig _ _) h2) as h4. simpl in h4.
assert (h5:le b x). red. assumption.
specialize (h1r _ h5). 
destruct h1r as [h1a | h1b]. subst. left. apply proj1_sig_injective. simpl. reflexivity.
clear h4. subst. right. apply proj1_sig_injective. simpl.
reflexivity.
intro h1.
red in h1. red.
destruct h1 as [h1l h1r].
split. 
intro h2. subst.
contradict h1l. apply proj1_sig_injective. simpl. reflexivity.
intros b h2.
specialize (h1r (exist _ _ (Full_intro _ b))).
assert (h3: le (B:=(Subalg _ (alg_closed_ba_ens B))) (exist (Ensembles.In (Full_set Bt)) b (Full_intro Bt b))
          (exist (Ensembles.In (Full_set Bt)) x (Full_intro Bt x))).
  red. simpl. unfold Bplus_sub. apply proj1_sig_injective. simpl.
  red in h2. assumption.
specialize (h1r h3).
simpl in h1r.
destruct h1r as [h1a | h1b]. 
pose proof (f_equal (@proj1_sig _ _) h1a) as h1a'. clear h1a.
simpl in h1a'. left. assumption.
pose proof (f_equal (@proj1_sig _ _) h1b) as h1b'. clear h1b.
simpl in h1b'. right. assumption.
Qed.


End SubalgBaEnsCompat.
End SubCompat.




Inductive sign:Set := pls|mns.
Definition signl:list sign := pls::mns::nil.
Definition signe:Ensemble sign := Couple pls mns.

Lemma sign_dec : forall s s':sign, {s = s'} + {s <> s'}.
intros s s'. destruct s, s'; auto; try right.
intro h1. discriminate h1. intro h1. discriminate h1.
Qed.

(*Whereas Givant/Halmos uses "p", I use the Handbook's notation epsilon, but with
  arguments in the same order as G/H's "p" *)
Definition eps {B:Bool_Alg} (b:(bt B)) (sgn:sign) := if sgn then b else -b.
Definition eps' {B:Bool_Alg} (pr:(bt B)*sign) := eps (fst pr) (snd pr).


Lemma card_signe : card_fun1 signe = 2.
apply card_fun1_couple.
intro; discriminate.
Qed.


Lemma list_to_set_sign : list_to_set signl = signe.
unfold signl. simpl.
rewrite <- couple_add_add.
unfold signe.
rewrite couple_comm.
reflexivity.
Qed.

Lemma signe_finite : Finite signe.
rewrite <- list_to_set_sign.
apply list_to_set_finite.
Qed.

Lemma eps_eps'_compat : forall {B:Bool_Alg} (x:(bt B)), eps x = (fun y:sign => eps' (x, y)).
intro x.
unfold eps. unfold eps'. simpl.  unfold eps.
reflexivity.
Qed.


Lemma eps_covers : 
  forall {B:Bool_Alg} (E:Ensemble (bt B)) (pf:Finite E) (x:bt B),
    plus_set (Im signe (eps x)) (finite_image _ _ signe (eps x) (signe_finite)) = 1.
intros B E h1 x.
assert (h2:list_to_set ((eps x pls) :: (- (eps x pls)) :: nil) =
           (Im signe (eps x))).
  apply Extensionality_Ensembles.
  red. split.
  red.
  intros y h3.
  rewrite <- list_to_set_in_iff in h3.
  simpl in h3.
  destruct h3 as [h3a | [h3b | h3c]].
  subst.
  apply Im_intro with pls. constructor. simpl. reflexivity.
  apply Im_intro with mns. constructor. simpl. subst. reflexivity.
  contradiction.
  red.
  intros y h3.
  destruct h3 as [y h3 z]. subst.
  unfold signe in h3.
  destruct h3.
  simpl. right. constructor. simpl. left. right. constructor.
rewrite (plus_set_compat' _ (eps x pls :: - eps x pls :: nil)).
simpl.
rewrite zero_sum.
rewrite comp_sum. reflexivity.
unfold bt. unfold bt in h2.
rewrite h2.
reflexivity.
Qed.


Section NormalForm.


Variable A:Bool_Alg.
Let At := Btype (Bc A).


Definition eps_map {E:Ensemble At} (f:Fin_map E signe mns) : 
  At->At :=
  (fun i:At => eps  i (f |-> i)).

Definition eps_map' {E:Ensemble At} {R:Ensemble sign}
           (f:Fin_map E R mns) : 
  At->At :=
  (fun i:At => eps  i (f |-> i)).

Lemma eps_maps_dec : 
  forall {E:Ensemble At} (f g:Fin_map E signe mns)
         (x:At), Ensembles.In E x ->
    {eps_map f x = eps_map g x} + {eps_map f x = -(eps_map g x)}.
intros E f g x h1.
unfold eps_map. 
destruct (f |-> x); destruct (g |-> x); simpl; auto.
right.
rewrite <- doub_neg.
reflexivity.
Qed.

Definition eps_map_compose {B:Bool_Alg} {E:Ensemble (bt B)} 
           (f:Fin_map E signe mns) (g:(bt B)->At) : (bt B) -> At :=
  (fun i:(bt B) => eps  (g i) (f |-> i)).

Definition eps_map_compose' {B:Bool_Alg} {E:Ensemble (bt B)} 
           {R:Ensemble sign}
           (f:Fin_map E R mns) (g:(bt B)->At) : (bt B) -> At :=
  (fun i:(bt B) => eps  (g i) (f |-> i)).



Lemma eps_maps_compose_dec : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)} 
         (f f':Fin_map E signe mns) (g:(bt B)->At)
         (x:(bt B)), Ensembles.In E x ->
         {eps_map_compose f g x = eps_map_compose f' g x} + 
         {eps_map_compose f g x = -(eps_map_compose f' g x)}.
intros B E f f' g x h1.
unfold eps_map_compose.
destruct (f |-> x); destruct (f' |-> x); simpl; auto.
right.
rewrite <- doub_neg.
reflexivity.
Qed.


Definition im_eps {E:Ensemble At} 
           (f:Fin_map E signe mns) : Ensemble At :=
Im E (eps_map f).

Definition im_eps' {E:Ensemble At} {R:Ensemble sign}
      (f:Fin_map E R mns) : Ensemble At :=
Im E (eps_map' f).


Definition im_eps_compose 
           {B:Bool_Alg} {E:Ensemble (bt B)} 
           (f:Fin_map E signe mns) (g:(bt B)->At) : Ensemble At :=
Im E (eps_map_compose f g).

Definition im_eps_compose' 
           {B:Bool_Alg} {E:Ensemble (bt B)} {R:Ensemble sign}
           (f:Fin_map E R mns) (g:(bt B)->At) : Ensemble At :=
Im E (eps_map_compose' f g).


Lemma finite_im_eps : 
  forall {E:Ensemble At} 
         (f:Fin_map E signe mns),
    Finite (im_eps f).
intros E f. apply finite_image.
apply (fin_map_fin_dom f).
Qed.


Definition finite_im_eps_compose :
  forall {B:Bool_Alg} {E:Ensemble (bt B)} 
         (f:Fin_map E signe mns) (g:(bt B)->At),
    Finite (im_eps_compose f g).
intros B E f g. apply finite_image. apply (fin_map_fin_dom f).
Qed.


Lemma im_eps_im_eps_compose_compat : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)}
         (g:sig_set E->At)
         (f : Fin_map (Im (full_sig E) g) signe mns)
         (pf:Finite E),
   im_eps f =
   im_eps_compose (fin_map_im_full_sig_eq f pf 0) (sig_fun_app g 0).
intros B E g f h1.
apply Extensionality_Ensembles.
red. split.
red. intros x h2.
destruct h2 as [x h2]. subst.
unfold im_eps_compose.
destruct h2 as [x h2]. subst. clear h2.
destruct x as [x h2].
apply Im_intro with x; auto.
unfold eps_map, eps_map_compose.
f_equal.
unfold sig_fun_app.
destruct (classic_dec (Ensembles.In E x)) as [h3 | h4].
f_equal. apply proj1_sig_injective. simpl. reflexivity.
contradiction.
pose proof (fin_map_im_full_sig_eq_compat f h1 0) as h3.
simpl in h3.
rewrite h3.
f_equal.
unfold sig_fun_app.
destruct (classic_dec (Ensembles.In E x)) as [h3' | h4'].
f_equal. apply proj1_sig_injective. simpl. reflexivity.
contradiction.
assumption.
red. intros x h2.
destruct h2 as [x h2]. subst.
unfold im_eps. rewrite im_im.
apply Im_intro with (exist _ _ h2). constructor.
unfold eps_map_compose, eps_map.
f_equal.
unfold sig_fun_app.
destruct (classic_dec (Ensembles.In E x)) as [h3' | h4'].
f_equal. apply proj1_sig_injective. simpl. reflexivity.
contradiction.
pose proof (fin_map_im_full_sig_eq_compat f h1 0) as h3.
simpl in h3.
rewrite h3.
f_equal.
unfold sig_fun_app.
destruct (classic_dec (Ensembles.In E x)) as [h3' | h4'].
f_equal. apply proj1_sig_injective. simpl. reflexivity.
contradiction.
assumption.
Qed.



Definition el_prod {E:Ensemble At} 
           (f:Fin_map E signe mns) : At.
pose (fun i:At => eps  i (f |-> i)) as p.
pose proof (fin_map_fin_dom f) as h1.
pose proof (finite_image _ _ E p h1) as h2.
refine (times_set _ h2).
Defined.

Definition el_sum {E:Ensemble At} 
           (f:Fin_map E signe mns) : At.
pose (fun i:At => eps  i (f |-> i)) as p.
pose proof (fin_map_fin_dom f) as h1.
pose proof (finite_image _ _ E p h1) as h2.
refine (plus_set _ h2).
Defined.


Definition el_prod_compose {B:Bool_Alg} {E:Ensemble (bt B)} 
           (g:(bt B)->At) (f:Fin_map E signe mns)  : At.
pose (fun i:(bt B) => eps (g i) (f |-> i)) as p.
pose proof (fin_map_fin_dom f) as h1.
pose proof (finite_image _ _ E p h1) as h2.
refine (times_set _ h2).
Defined.

(*This is to interface with some specific cases where we need
  "injectors" of g, i.e. injective restrictions of g, 
  whose image agrees with g on E *)
Definition el_prod_compose' {B:Bool_Alg} {E:Ensemble (bt B)} 
           (g:(bt B)->At) (f:Fin_map E signe mns) : At.
pose (fun i:sig_set E => g (proj1_sig i)) as g'.
pose (fun i:sig_set E => eps (g' i) (f |-> (proj1_sig i))) as p.
pose proof (fin_map_fin_dom f) as h1.
rewrite finite_full_sig_iff in h1.
pose proof (finite_image _ _ (full_sig E) p h1) as h2.
refine (times_set _ h2).
Defined.

Lemma el_prod_compose_el_prod_compose'_compat :
  forall {B:Bool_Alg} {E:Ensemble (bt B)}
         (g:(bt B)->At),
    el_prod_compose g = el_prod_compose' (E:=E) g.
intros B E g.
unfold el_prod_compose, el_prod_compose'.
apply functional_extensionality.
intro f.
apply times_set_functional.
apply Extensionality_Ensembles.
red. split.
red.
intros x h1.
destruct h1 as [x h1]. subst.
apply Im_intro with (exist _ _ h1). constructor. simpl.
reflexivity.
red. intros x h1. destruct h1 as [x h1]. subst.
destruct x as [x h2]. simpl.
apply Im_intro with x; auto.
Qed.


Definition el_sum_compose {B:Bool_Alg} {E:Ensemble (bt B)} 
           (g:(bt B)->At) (f:Fin_map E signe mns)  : At.
pose (fun i:(bt B) => eps (g i) (f |-> i)) as p.
pose proof (fin_map_fin_dom f) as h1.
pose proof (finite_image _ _ E p h1) as h2.
refine (plus_set _ h2).
Defined.


Lemma el_prod_el_prod_compose_compat : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)} {g:sig_set E->At}
         (f:Fin_map (Im (full_sig E) g) signe mns)
         (pf:Finite E),
    el_prod f = el_prod_compose (sig_fun_app g 0)
                                (fin_map_im_full_sig_eq f pf 0).
                                
intros B E g f h1.
unfold el_prod, el_prod_compose.
pose proof (im_im (full_sig E) g (fun i : At => eps i (f |-> i))) as h2.
simpl in h2.
pose proof (fin_map_im_full_sig_eq_compat f h1 0) as h5. 
simpl in h5.
assert (h3: Im (full_sig E) (fun x : sig_set E => eps (g x) (f |-> g x)) =  (Im E
        (fun i : bt B =>
         eps (sig_fun_app g 0 i) (fin_map_im_full_sig_eq f h1 0 |-> i)))).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h3. destruct h3 as [x h3]. subst.
  destruct x as [x h4].
  apply Im_intro with x. assumption.
  specialize (h5 _ h4).
  rewrite h5.
  assert (h6:sig_fun_app g 0 x = g (exist _ _ h4)).
    unfold sig_fun_app.
    destruct (classic_dec (Ensembles.In E x)) as [h7 | h8].
    assert (h7 = h4). apply proof_irrelevance. subst.
    reflexivity. contradiction.
  rewrite h6.
  f_equal.
  red. intros x h6.
  destruct h6 as [x h6]. subst.
  apply Im_intro with (exist _ _ h6). constructor.
  specialize (h5 _ h6).
  assert (h0:sig_fun_app g 0 x = g (exist _ _ h6)).
    unfold sig_fun_app.
    destruct (classic_dec (Ensembles.In E x)) as [h7 | h8].
    assert (h7 = h6). apply proof_irrelevance. subst.
    reflexivity. contradiction.
  rewrite h5.
  rewrite h0.
  reflexivity.
rewrite h3 in h2.
pose proof (subsetT_eq_compat _ _ _ _  (finite_image At At (Im (full_sig E) g) (fun i : At => eps i (f |-> i))
        (fin_map_fin_dom f))
                              (finite_image (bt B) At E
                                            (fun i : bt B =>
                                               eps (sig_fun_app g 0 i) (fin_map_im_full_sig_eq f h1 0 |-> i))
                                            (fin_map_fin_dom (fin_map_im_full_sig_eq f h1 0))) h2) as h6. 
unfold At, bt in h6. unfold At, bt.
dependent rewrite -> h6.
reflexivity.
Qed. 


Lemma el_sum_el_sum_compose_compat : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)} {g:sig_set E->At}
         (f:Fin_map (Im (full_sig E) g) signe mns)
         (pf:Finite E),
    el_sum f = el_sum_compose (sig_fun_app g 0)
                              (fin_map_im_full_sig_eq f pf 0).
                                
intros B E g f h1.
unfold el_sum, el_sum_compose.
pose proof (im_im (full_sig E) g (fun i : At => eps i (f |-> i))) as h2.
simpl in h2.
pose proof (fin_map_im_full_sig_eq_compat f h1 0) as h5. 
simpl in h5.
assert (h3: Im (full_sig E) (fun x : sig_set E => eps (g x) (f |-> g x)) =  (Im E
        (fun i : bt B =>
         eps (sig_fun_app g 0 i) (fin_map_im_full_sig_eq f h1 0 |-> i)))).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h3. destruct h3 as [x h3]. subst.
  destruct x as [x h4].
  apply Im_intro with x. assumption.
  specialize (h5 _ h4).
  rewrite h5.
  assert (h6:sig_fun_app g 0 x = g (exist _ _ h4)).
    unfold sig_fun_app.
    destruct (classic_dec (Ensembles.In E x)) as [h7 | h8].
    assert (h7 = h4). apply proof_irrelevance. subst.
    reflexivity. contradiction.
  rewrite h6.
  f_equal.
  red. intros x h6.
  destruct h6 as [x h6]. subst.
  apply Im_intro with (exist _ _ h6). constructor.
  specialize (h5 _ h6).
  assert (h0:sig_fun_app g 0 x = g (exist _ _ h6)).
    unfold sig_fun_app.
    destruct (classic_dec (Ensembles.In E x)) as [h7 | h8].
    assert (h7 = h6). apply proof_irrelevance. subst.
    reflexivity. contradiction.
  rewrite h5.
  rewrite h0.
  reflexivity.
rewrite h3 in h2.
pose proof (subsetT_eq_compat _ _ _ _  (finite_image At At (Im (full_sig E) g) (fun i : At => eps i (f |-> i))
        (fin_map_fin_dom f))
                              (finite_image (bt B) At E
                                            (fun i : bt B =>
                                               eps (sig_fun_app g 0 i) (fin_map_im_full_sig_eq f h1 0 |-> i))
                                            (fin_map_fin_dom (fin_map_im_full_sig_eq f h1 0))) h2) as h6.
unfold At, bt in h6. unfold At, bt.
dependent rewrite -> h6.
reflexivity.
Qed.

Lemma el_prod_el_prod_compose_compat' : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)} 
         (f : Fin_map E signe mns) (g:sig_set E->At)
         (f': Fin_map (Im (full_sig E) g) signe mns),
     (forall b:(bt B),
      Ensembles.In E b -> f' |-> ((g, (Bzero (Bc A))) ||-> b) =  
                          f |-> b) ->
  el_prod_compose (sig_fun_app g 0) f = el_prod f'.
intros B E f g f' h1.
pose proof (fin_map_fin_dom f) as h0.
unfold el_prod_compose, el_prod.
pose proof (im_im (full_sig E) g (fun i : At => eps i (f' |-> i))) as h2.
simpl in h2.
assert (h3: Im E (fun i : bt B => eps (sig_fun_app g 0 i) (f |-> i)) =  Im (full_sig E) (fun x : sig_set E => eps (g x) (f' |-> g x))).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h3. destruct h3 as [x h3]. subst.
  specialize (h1 _ h3).
  rewrite <- h1.
  apply Im_intro with (exist _ _ h3). constructor. 
  f_equal.
  unfold sig_fun_app. destruct (classic_dec (Ensembles.In E x)) as [h4 | h5].
  assert (h3 = h4). apply proof_irrelevance. subst. reflexivity.
  contradiction.
  f_equal. simpl.
  unfold sig_fun_app. destruct (classic_dec (Ensembles.In E x)) as [h4 | h5].
  assert (h3 = h4). apply proof_irrelevance. subst. reflexivity.
 contradiction.
 red. intros x h3.
 destruct h3 as [x h3]. subst.
 destruct x as [x h4]. clear h3.
 apply Im_intro with x. assumption.
 unfold sig_fun_app. destruct (classic_dec (Ensembles.In E x)) as [h4' | h5'].
 assert (h4 = h4'). apply proof_irrelevance. subst. 
 f_equal.
 specialize (h1 _ h4'). rewrite <- h1.
 f_equal.
 unfold sig_fun_app. destruct (classic_dec (Ensembles.In E x)) as [h4 | h5'].
 assert (h4 = h4'). apply proof_irrelevance. subst. 
 simpl. f_equal. contradiction. contradiction.
rewrite <- h3 in h2.
symmetry in h2.
pose proof (subsetT_eq_compat _ _ _ _ (finite_image (bt B) At E
        (fun i : bt B => eps (sig_fun_app g 0 i) (f |-> i))
        (fin_map_fin_dom f))  (finite_image At At (Im (full_sig E) g) (fun i : At => eps i (f' |-> i))
        (fin_map_fin_dom f')) h2) as h4.
unfold At, bt in h4. unfold At, bt.
dependent rewrite -> h4.
reflexivity.
Qed.


Lemma times_set_def_im_rel_class_im_rel_set_fun_value :
  forall {B:Bool_Alg} (E S:Ensemble (bt B)),
    Finite E ->
    forall (g:bt B->At),
    Ensembles.In (rel_classes_im_rel_set E g) S ->
    forall x:(bt B),
      Ensembles.In S x ->
      times_set_def _ 0 (Im S g) = g x.
intros B E S h0 g h1 x ha.
unfold times_set_def.
destruct classic_dec as [h2 | h3].
rewrite <- times_set_sing.
apply times_set_functional.
apply Extensionality_Ensembles.
red. split.
red. intros x'' h4. destruct h4 as [x'' h4]. subst.
destruct h1 as [x' h1]. subst.

destruct h4 as [h4]. destruct h4 as [h4 h5].
destruct ha as [ha]. destruct ha as [ha hb].
red in hb, h5.
rewrite <- hb, h5. constructor.
red. intros x'' h4. destruct h4. 
apply Im_intro with x; auto.
contradict h3.
apply finite_image.
eapply Finite_downward_closed.
apply h0.
destruct h1 as [S h1]. subst.
red.
intros x' h2.
destruct h2 as [h2]. destruct h2; auto.
Qed.


Lemma im_rel_classes_im_rel_set :  
  forall  {B:Bool_Alg} {E:Ensemble (bt B)} 
          (pf:Finite E) (g:bt B->At),
          Im (rel_classes_im_rel_set E g) 
             (fun S => (times_set_def _ 0 (Im S g))) =
          Im E g.
intros B E h1 g.
assert (h4: forall x, Ensembles.In E x ->
                      Im [y : bt B | Ensembles.In E y /\ im_rel g x y] g = Singleton (g x)).
  intros x' hin. 
  apply Extensionality_Ensembles.
  red. split.
  red. intros x'' h4. destruct h4 as [x'' h4]. subst.
  destruct h4 as [h4]. destruct h4 as [h4 h5].
  red in h5. rewrite h5. constructor.
  red. 
  intros x'' h4. destruct h4. 
  apply Im_intro with x'. constructor. split; auto.
  red. auto. auto. 
apply Extensionality_Ensembles.
red. split.
red.
intros x h2. 
destruct h2 as [x h2]. subst.
destruct h2 as [x h2 q h3]. subst.
apply Im_intro with x; auto. 
rewrite <- times_set_sing.
unfold times_set_def.
destruct classic_dec as [h3 | h4'].
apply times_set_functional.
rewrite h4. reflexivity. assumption.
contradict h4'.
apply finite_image. 
eapply Finite_downward_closed. apply h1.
red. 
intros x' h3. destruct h3 as [h3]. destruct h3; auto.
red. intros y h2.
destruct h2 as [y h2]. subst.  
apply Im_intro with [x:(bt B) | Ensembles.In E x /\ im_rel g y x].
unfold rel_classes_im_rel_set. unfold rel_classes.
apply Im_intro with y. assumption. 
reflexivity.
unfold times_set_def.
destruct classic_dec as [h3 | h4'].
rewrite <- (times_set_sing (g y)).
apply times_set_functional. 
rewrite h4; auto.
contradict h4'.
apply finite_image. 
eapply Finite_downward_closed. apply h1.
red. 
intros x' h3. destruct h3 as [h3]. destruct h3; auto.
Qed.





Lemma el_prod_le_ai : forall {E:Ensemble At} 
                             (a:Fin_map E signe mns) (i:At),
                        Ensembles.In E i ->
                        le (el_prod a) (eps i (a |-> i)).
intros E a i h1.
unfold el_prod.
apply le_times_set.
apply Im_intro with i; auto.
Qed.

Lemma ai_le_el_sum : forall {E:Ensemble At} 
                             (a:Fin_map E signe mns) (i:At),
                        Ensembles.In E i ->
                        le (eps i (a |-> i))  (el_sum a).
intros E a i h1.
unfold el_sum.
apply le_plus_set.
apply Im_intro with i; auto.
Qed.

Lemma non_zero_el_prod_inj : 
  forall {E:Ensemble At} (f g:Fin_map E signe mns),
    el_prod f <> 0 ->
    el_prod f = el_prod g ->
    im_eps f = im_eps g.
intros E f g h1 h2.
apply Extensionality_Ensembles.
red. split.
red. intros x h3.
destruct h3 as [x h3].
subst.
pose proof (el_prod_le_ai f _ h3) as h4.
destruct (eps_maps_dec f g x h3) as [h5 | h6].
rewrite h5.
apply Im_intro with x; auto.
pose proof (el_prod_le_ai g _ h3) as h7.
rewrite <- h2 in h7.
unfold eps_map in h6. rewrite h6 in h4.
pose proof (mono_prod _ _ _ _ h7 h4) as h8.
rewrite idem_prod in h8.
rewrite comp_prod in h8.
apply le_x_0 in h8.
contradiction.
red. intros x h3.
destruct h3 as [x h3].
subst.
pose proof (el_prod_le_ai g _ h3) as h4. 
destruct (eps_maps_dec f g x h3) as [h5 | h6].
rewrite <- h5.
apply Im_intro with x; auto.
pose proof (el_prod_le_ai f _ h3) as h7.
rewrite h2 in h7.
unfold eps_map in h6. rewrite h6 in h7.
pose proof (mono_prod _ _ _ _ h4 h7) as h8.
rewrite idem_prod in h8.
rewrite comp_prod in h8.
apply le_x_0 in h8.
rewrite h2 in h1.
contradiction.
Qed.


Lemma el_prod_disjoint : 
  forall (E:Ensemble At) 
         (a b:Fin_map E signe mns), a <> b ->
             (el_prod a) * (el_prod b) = 0. 
intros E a b h3.
pose proof (distinct_fin_maps_differ_at_point _ _ h3) as h4.
destruct h4 as [i h4].
destruct h4 as [h4l h4r].
pose proof (el_prod_le_ai a i h4l) as h5.
pose proof (el_prod_le_ai b i h4l) as h6.
destruct (a |->i) ; destruct (b |-> i).
contradict h4r. reflexivity.
simpl in h5. simpl in h6.
pose proof (mono_prod _ _ _ _ h5 h6) as h7.
rewrite comp_prod in h7.
red in h7.
rewrite zero_sum in h7. assumption.
simpl in h5. simpl in h6.
pose proof (mono_prod _ _ _ _ h5 h6) as h7.
rewrite (comm_prod _ (- i) i) in h7.
rewrite comp_prod in h7.
red in h7.
rewrite zero_sum in h7. assumption.
contradict h4r. reflexivity.
Qed.



Lemma el_prod_compose_le_ai : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)} 
         (a:Fin_map E signe mns) 
         (g:(bt B)->At) (i:(bt B)),
    Ensembles.In E i ->
    le (el_prod_compose g a) (eps (g i) (a |-> i)).
intros B E a g i h1.
unfold el_prod_compose.
apply le_times_set.
apply Im_intro with i; auto.
Qed.


Lemma ai_le_el_sum_compose : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)} 
         (a:Fin_map E signe mns) 
         (g:(bt B)->At) (i:(bt B)),
    Ensembles.In E i ->
    le (eps (g i) (a |-> i)) (el_sum_compose g a).
intros B E a g i h1.
unfold el_sum_compose.
apply le_plus_set.
apply Im_intro with i; auto.
Qed.


Lemma el_prod_compose_disjoint : 
  forall {B:Bool_Alg} (E:Ensemble (bt B))
         (g:(bt B)->At)
         (a b:Fin_map E signe mns), a <> b ->
             (el_prod_compose g a) * (el_prod_compose g b) = 0. 
intros B E g a b h3.
pose proof (distinct_fin_maps_differ_at_point _ _ h3) as h4.
destruct h4 as [i h4].
destruct h4 as [h4l h4r].
pose proof (el_prod_compose_le_ai a g i h4l) as h5.
pose proof (el_prod_compose_le_ai b g i h4l) as h6.
destruct (a |->i) ; destruct (b |-> i).
contradict h4r. reflexivity.
simpl in h5. simpl in h6.
pose proof (mono_prod _ _ _ _ h5 h6) as h7.
rewrite comp_prod in h7.
red in h7.
rewrite zero_sum in h7. assumption.
simpl in h5. simpl in h6.
pose proof (mono_prod _ _ _ _ h5 h6) as h7.
rewrite (comm_prod _ (- (g i)) (g i)) in h7.
rewrite comp_prod in h7.
red in h7.
rewrite zero_sum in h7. assumption.
contradict h4r. reflexivity.
Qed.


Lemma non_zero_el_prod_compose_inj : 
  forall 
    {B:Bool_Alg} {E:Ensemble (bt B)} 
    (f f':Fin_map E signe mns) 
    (g:(bt B)->At),
    el_prod_compose g f <> 0 ->
    el_prod_compose g f = el_prod_compose g f' ->
    im_eps_compose f g = im_eps_compose f' g.
intros B E f f' g h1 h2.
apply Extensionality_Ensembles.
red. split.
red. intros x h3.
destruct h3 as [x h3].
subst.
pose proof (el_prod_compose_le_ai f g _ h3) as h4.
destruct (eps_maps_compose_dec f f' g x h3) as [h5 | h6].
rewrite h5.
apply Im_intro with x; auto.
pose proof (el_prod_compose_le_ai f' g _ h3) as h7.
rewrite <- h2 in h7.
unfold eps_map_compose in h6. rewrite h6 in h4.
pose proof (mono_prod _ _ _ _ h7 h4) as h8.
rewrite idem_prod in h8.
rewrite comp_prod in h8.
apply le_x_0 in h8.
contradiction.
red. intros x h3.
destruct h3 as [x h3].
subst.
pose proof (el_prod_compose_le_ai f' g _ h3) as h4. 
destruct (eps_maps_compose_dec f f' g x h3) as [h5 | h6].
rewrite <- h5.
apply Im_intro with x; auto.
pose proof (el_prod_compose_le_ai f g _ h3) as h7.
rewrite h2 in h7.
unfold eps_map_compose in h6. rewrite h6 in h7.
pose proof (mono_prod _ _ _ _ h4 h7) as h8.
rewrite idem_prod in h8.
rewrite comp_prod in h8.
apply le_x_0 in h8.
rewrite h2 in h1.
contradiction.
Qed.

Lemma non_zero_el_prod_compose_constant : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)} 
         (g:bt B->At) (f:Fin_map E signe mns),
    el_prod_compose g f <> 0 ->
    forall (x x':bt B),
      Ensembles.In E x -> Ensembles.In E x' ->
      g x = g x' ->
      f |-> x = f |-> x'.
intros B E g f h1 x x' hx hx' h2.
pose proof (sign_dec (f |-> x) (f |-> x')) as h3.
destruct h3 as [|h3]; auto.
unfold el_prod_compose in h1.
assert (h4: (eps (g x) (f |-> x)) * (eps (g x') (f |-> x')) = 0).
  unfold eps. simpl.
  destruct (f |-> x); destruct (f |-> x').
  contradict h3. reflexivity.
  rewrite h2. apply comp_prod. rewrite h2. rewrite comm_prod.
  apply comp_prod. contradict h3. reflexivity.
assert (h5:Included (Couple x x') E). red; intros e h5.
  destruct h5; auto.
pose proof (decompose_setminus_inc _ _ h5) as h6.
pose proof (fin_map_fin_dom f) as h0.
assert (h7:Im E (fun i : bt B => eps (g i) (f |-> i)) =
           Im (Union (Couple x x') (Setminus E (Couple x x')))
              (fun i : bt B => eps (g i) (f |-> i))).
  f_equal; auto.
assert (h8:Finite  (Im (Union (Couple x x') (Setminus E (Couple x x')))
         (fun i : bt B => eps (g i) (f |-> i)))).
  rewrite <- h7. apply finite_image; auto.
pose proof (subsetT_eq_compat _ _ _ _ (finite_image (bt B) (bt A) E (fun i : bt B => eps (g i) (f |-> i))
            (fin_map_fin_dom f))  h8 h7) as h9.      
dependent rewrite -> h9 in h1.
pose proof (im_union (Couple x x') (Setminus E (Couple x x'))
                     (fun i : bt B => eps (g i) (f |-> i))) as h10.
assert (h11:Finite (Union (Im (Couple x x') (fun i : bt B => eps (g i) (f |-> i)))
          (Im (Setminus E (Couple x x'))
             (fun i : bt B => eps (g i) (f |-> i))))).
  rewrite <- h10. auto.
pose proof (subsetT_eq_compat _ _ _ _ h8 h11 h10) as h12.
dependent rewrite -> h12 in h1.
pose proof (finite_couple x x') as h13.
assert (h14:Finite (Setminus E (Couple x x'))).
  eapply Finite_downward_closed. apply h0.
  red. intros e h15. destruct h15; auto.
pose proof (finite_image _ _ _ (fun i:bt B => eps (g i) (f |-> i))
                         h13) as h16.
pose proof (finite_image _ _ _ (fun i:bt B => eps (g i) (f |-> i))
                         h14) as h17.
pose proof (times_set_union' _ _ h16 h17 h11) as h18.
unfold bt, At in h1, h18.
rewrite h18 in h1.
assert (h19: times_set
         (Im (Couple x x') (fun i : Btype (Bc B) => eps (g i) (f |-> i))) h16 = 0). 
  assert (h20: Im (Couple x x') (fun i : Btype (Bc B) => eps (g i) (f |-> i)) = Couple (eps (g x) (f |-> x)) 
                                                                                            (eps (g x') (f |-> x'))).
    apply Extensionality_Ensembles.
    red. split.
    red. intros a h19. destruct h19 as [a h19 q h20]. rewrite h20.
    clear h20 q.
    destruct h19. left. right.
    red. intros a h19.
    destruct h19.
    apply Im_intro with x. left. reflexivity.
    apply Im_intro with x'. right. reflexivity.
 assert (h21:Finite 
               (Couple (eps (g x) (f |-> x)) (eps (g x') (f |-> x')))).
   rewrite <- h20. assumption.
  pose proof (subsetT_eq_compat _ _ _ _ h16 h21 h20) as h22.
  dependent rewrite -> h22.
  rewrite times_set_couple'.
  assumption.
rewrite h19 in h1 at 1.
rewrite comm_prod in h1.
rewrite zero_prod in h1.
contradict h1.
reflexivity.
Qed.



Lemma el_prod_disjoint' : 
  forall (E:Ensemble At) (X Y:Ensemble (Fin_map E signe mns)),
    Inhabited [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x] ->
         (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod (fst x) * el_prod (snd x))) = Singleton 0.
intros E X Y h1.
apply Extensionality_Ensembles.
red. split.
red.
intros x h2.
inversion h2 as [x' h3 y h4 h5]. subst. clear h2.
destruct h3 as [h3].
destruct h3 as [h3l h3r].
destruct h3l as [h3l].
destruct h3l as [h3a h3b].
pose proof (el_prod_disjoint _ (fst x') (snd x') h3r) as h4.
rewrite h4. constructor.
destruct h1 as [pr h1].
destruct h1 as [h1].
destruct h1 as [h1l h1r].
pose proof (el_prod_disjoint _ (fst pr) (snd pr) h1r) as h2.  
red.
intros x h1. destruct h1; subst.
rewrite <- h2.
apply Im_intro with pr.
constructor; auto.
reflexivity.
Qed.

Lemma el_prod_compose_disjoint' : 
  forall {B:Bool_Alg} (E:Ensemble (bt B))
         (g:(bt B)->At)
         (X Y:Ensemble (Fin_map E signe mns)),
    Inhabited [x : Fin_map E signe mns * Fin_map E signe mns
              | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x] ->
    (Im
       [x : Fin_map E signe mns * Fin_map E signe mns
       | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
       (fun x : Fin_map E signe mns * Fin_map E signe mns =>
          el_prod_compose g (fst x) * el_prod_compose g (snd x))) = Singleton 0.
intros B E g X Y h1.
apply Extensionality_Ensembles.
red. split.
red.
intros x h2.
inversion h2 as [x' h3 y h4 h5]. subst. clear h2.
destruct h3 as [h3].
destruct h3 as [h3l h3r].
destruct h3l as [h3l].
destruct h3l as [h3a h3b].
pose proof (el_prod_compose_disjoint _ g (fst x') (snd x') h3r) as h4.
rewrite h4. constructor.
destruct h1 as [pr h1].
destruct h1 as [h1].
destruct h1 as [h1l h1r].
pose proof (el_prod_compose_disjoint _ g (fst pr) (snd pr) h1r) as h2.  
red.
intros x h1. destruct h1; subst.
rewrite <- h2.
apply Im_intro with pr.
constructor; auto.
reflexivity.
Qed.


Lemma plus_set_el_prod_disjoint : 
  forall (E:Ensemble At) (X Y:Ensemble (Fin_map E signe mns))
         (pf:Finite [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]),
         plus_set (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod (fst x) * el_prod (snd x))) (finite_image _ _ _ _ pf) = 0.
intros E X Y h1.
destruct (classic (Inhabited ([x : Fin_map E signe mns * Fin_map E signe mns
         | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]))) as [h2 | h3].
pose proof (el_prod_disjoint' _ X Y h2) as h3.
generalize  (finite_image (Fin_map E signe mns * Fin_map E signe mns) 
        (bt A)
        [x : Fin_map E signe mns * Fin_map E signe mns
        | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
        (fun x : Fin_map E signe mns * Fin_map E signe mns =>
         el_prod (fst x) * el_prod (snd x)) h1). 
unfold bt in h3. unfold bt.
rewrite h3.
intro. rewrite plus_set_sing' at 1. reflexivity.
pose proof (not_inhabited_empty _ h3) as h4.
pose proof (image_empty _ _ (fun x : Fin_map E signe mns * Fin_map E signe mns =>
         el_prod (fst x) * el_prod (snd x))) as h5.
rewrite <- h4 in h5.
pose proof (plus_set_functional _ _ (finite_image (Fin_map E signe mns * Fin_map E signe mns) 
        (Btype (Bc A))
        [x : Fin_map E signe mns * Fin_map E signe mns
        | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
        (fun x : Fin_map E signe mns * Fin_map E signe mns =>
         el_prod (fst x) * el_prod (snd x)) h1) (Empty_is_finite _) h5) as h6.
rewrite plus_set_empty in h6.
assumption.
Qed.

Lemma plus_set_el_prod_disjoint' : 
  forall (E:Ensemble At) (X Y:Ensemble (Fin_map E signe mns))
         (pf:Finite [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x])
         (pfi:Finite (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod (fst x) * el_prod (snd x)))),
         plus_set (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod (fst x) * el_prod (snd x))) pfi = 0.
intros E X Y h1 h2.
pose proof (plus_set_el_prod_disjoint _ X Y h1) as h3.
rewrite <- h3.
apply plus_set_functional. reflexivity.
Qed.


Lemma plus_set_el_prod_compose_disjoint : 
  forall {B:Bool_Alg} (E:Ensemble (bt B)) (g:(bt B)->At) (X Y:Ensemble (Fin_map E signe mns))
         (pf:Finite [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]),
         plus_set (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod_compose g (fst x) * el_prod_compose g (snd x))) (finite_image _ _ _ _ pf) = 0.
intros B E g X Y h1.
destruct (classic (Inhabited ([x : Fin_map E signe mns * Fin_map E signe mns
         | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]))) as [h2 | h3].
pose proof (el_prod_compose_disjoint' _ g X Y h2) as h3.
generalize  (finite_image (Fin_map E signe mns * Fin_map E signe mns) 
        (bt A)
        [x : Fin_map E signe mns * Fin_map E signe mns
        | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
        (fun x : Fin_map E signe mns * Fin_map E signe mns =>
         el_prod_compose g (fst x) * el_prod_compose g (snd x)) h1).
unfold bt in h3. unfold bt.
rewrite h3.
intro. apply plus_set_sing'.
pose proof (not_inhabited_empty _ h3) as h4.
pose proof image_empty.
pose proof (image_empty _ _ (fun x : Fin_map E signe mns * Fin_map E signe mns =>
         el_prod_compose g (fst x) * el_prod_compose g (snd x))) as h5.
rewrite <- h4 in h5.
pose proof (plus_set_functional _ _ (finite_image (Fin_map E signe mns * Fin_map E signe mns) 
        (Btype (Bc A))
        [x : Fin_map E signe mns * Fin_map E signe mns
        | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
        (fun x : Fin_map E signe mns * Fin_map E signe mns =>
         el_prod_compose g (fst x) * el_prod_compose g (snd x)) h1) (Empty_is_finite _) h5) as h6.
rewrite plus_set_empty in h6.
assumption.
Qed.

Lemma plus_set_el_prod_compose_disjoint' : 
  forall {B:Bool_Alg} (E:Ensemble (bt B)) (g:(bt B)->At) (X Y:Ensemble (Fin_map E signe mns))
         (pf:Finite [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x])
         (pfi:Finite (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod_compose g (fst x) * el_prod_compose g (snd x)))),
         plus_set (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod_compose g (fst x) * el_prod_compose g (snd x))) pfi = 0.
intros B E g X Y h1 h2.
pose proof (plus_set_el_prod_compose_disjoint _ g X Y h1) as h3.
rewrite <- h3.
apply plus_set_functional. reflexivity.
Qed.



Lemma el_prod_covers :
  forall (E:Ensemble At) (pf:Finite E),
    plus_set (Im (Full_set (Fin_map E signe mns)) el_prod)
    (finite_image _ _ _ _ (finite_fin_maps _ _ mns pf signe_finite)) = 1.
intros E h1.
assert (h2:times_plus_fin_pair_map1
      h1 (fun_to_fin_map _ 0 (cart_prod_fin _ _ h1 signe_finite) eps') = 1).
    unfold times_plus_fin_pair_map1.
  unfold fin_map_times.
generalize ((finite_image At (Btype (Bc A)) E
        (fin_map_app
           (plus_fin_pair_map1 h1
              (fun_to_fin_map (cart_prod E signe) 0
                 (cart_prod_fin E signe h1 signe_finite) eps')))
        (fin_map_fin_dom
           (plus_fin_pair_map1 h1
              (fun_to_fin_map (cart_prod E signe) 0
                 (cart_prod_fin E signe h1 signe_finite) eps'))))).
unfold plus_fin_pair_map1.
rewrite im_fin_map_app_undoes_fun_to_fin_map. 
intros h2.
assert (h3: Im E (fun x : At =>
         plus_set
           (im1
              (fun_to_fin_map (cart_prod E signe) 0
                 (cart_prod_fin E signe h1 signe_finite) eps') x)
           (im1_fin
              (fun_to_fin_map (cart_prod E signe) 0
                 (cart_prod_fin E signe h1 signe_finite) eps') x)) =
           Im E (fun x:At => plus_set (Im signe (fun y => (eps' (x, y)))) (finite_image _ _ _ _ signe_finite))).
  apply im_ext_in. intros x h.
  apply plus_set_functional.
apply im1_fun_to_fin_map; auto.
revert h2.
unfold At, bt in h3. unfold At, bt.
rewrite h3 at 1. clear h3.
intro h2. clear h2.
unfold At, bt in h1. unfold At, bt.
rewrite <- times_set_sing.
destruct (finite_inh_or_empty _ h1) as [hinh | hninh].
apply times_set_functional.
apply Extensionality_Ensembles.
red. split.
red.
intros x h2.
destruct h2 as [x h2]. subst.
rewrite fun_to_fin_map_compat. 
pose proof (eps_eps'_compat x) as h0.
unfold bt, At in h0. 
pose proof (eps_covers _ h1  x) as h8.
unfold At, bt in h8. 
assert (he: plus_set (Im signe (eps x))
         (finite_image sign (Btype (Bc A)) signe (eps x) signe_finite) = (plus_set
        (im1
           (fun_to_fin_map (cart_prod E signe) 0
              (cart_prod_fin E signe h1 signe_finite) eps') x)
        (im1_fin
           (fun_to_fin_map (cart_prod E signe) 0
              (cart_prod_fin E signe h1 signe_finite) eps') x))).
  apply plus_set_functional.
  unfold im1.
  assert (hife: (if eq_dec E (Empty_set At)
    then Empty_set (Btype (Bc A))
    else
     Im signe
       (fun y : sign =>
        fun_to_fin_map (cart_prod E signe) 0
          (cart_prod_fin E signe h1 signe_finite) eps' |-> 
        (x, y))) =    Im signe
       (fun y : sign =>
        fun_to_fin_map (cart_prod E signe) 0
          (cart_prod_fin E signe h1 signe_finite) eps' |-> 
        (x, y))).
    destruct (eq_dec E (Empty_set At)) as [hed | hned].
    subst. contradiction.
    reflexivity.
  unfold At, bt in hife. unfold At, bt. simpl in hife. simpl. 
  rewrite hife at 1.
apply im_ext_in. 
intros s hin.
rewrite fun_to_fin_map_compat.
rewrite eps_eps'_compat.
reflexivity. constructor. simpl.
split; auto.
unfold At, bt in he. unfold At, bt. simpl in he. simpl.
rewrite <- he at 1.
rewrite <- h8 at 1. constructor. assumption.  
destruct hinh as [e hinh].
pose proof (eps_covers _ h1 e) as hc.
red. intros x h2. destruct h2.
apply Im_intro with e. assumption.
rewrite fun_to_fin_map_compat. rewrite <- hc. 
apply plus_set_functional.
unfold im1.
assert (hif:(if eq_dec E (Empty_set (Btype (Bc A)))
    then Empty_set (Btype (Bc A))
    else
     Im signe
       (fun y : sign =>
        fun_to_fin_map (cart_prod E signe) 0
          (cart_prod_fin E signe h1 signe_finite) eps' |-> 
        (e, y))) = 
            Im signe
               (fun y : sign =>
                  fun_to_fin_map (cart_prod E signe) 0
                                 (cart_prod_fin E signe h1 signe_finite) eps' |-> 
                                 (e, y))).
  destruct (eq_dec E (Empty_set (Btype (Bc A)))) as [hed | hned].
  subst. contradiction.
  reflexivity.
  unfold At, bt in hif. unfold At, bt. simpl in hif. simpl.
  rewrite hif at 1.
apply im_ext_in.
intros x h3.
rewrite fun_to_fin_map_compat.
rewrite eps_eps'_compat. reflexivity.
constructor; auto. assumption.
subst.
assert (h2:
     Im (Empty_set At)
        (fin_map_app
           (fun_to_fin_map (Empty_set At) 0 h1
              (fun x : Btype (Bc A) =>
               plus_set
                 (im1
                    (fun_to_fin_map (cart_prod (Empty_set At) signe) 0
                       (cart_prod_fin (Empty_set At) signe h1 signe_finite)
                       eps') x)
                 (im1_fin
                    (fun_to_fin_map (cart_prod (Empty_set At) signe) 0
                       (cart_prod_fin (Empty_set At) signe h1 signe_finite)
                       eps') x)))) = Empty_set _).
  apply image_empty.
pose proof (subsetT_eq_compat _ _ _ _ (finite_image (Btype (Bc A)) (Btype (Bc A)) (Empty_set At)
        (fin_map_app
           (fun_to_fin_map (Empty_set At) 0 h1
              (fun x : Btype (Bc A) =>
               plus_set
                 (im1
                    (fun_to_fin_map (cart_prod (Empty_set At) signe) 0
                       (cart_prod_fin (Empty_set At) signe h1 signe_finite)
                       eps') x)
                 (im1_fin
                    (fun_to_fin_map (cart_prod (Empty_set At) signe) 0
                       (cart_prod_fin (Empty_set At) signe h1 signe_finite)
                       eps') x))))
        (fin_map_fin_dom
           (fun_to_fin_map (Empty_set At) 0 h1
              (fun x : Btype (Bc A) =>
               plus_set
                 (im1
                    (fun_to_fin_map (cart_prod (Empty_set At) signe) 0
                       (cart_prod_fin (Empty_set At) signe h1 signe_finite)
                       eps') x)
                 (im1_fin
                    (fun_to_fin_map (cart_prod (Empty_set At) signe) 0
                       (cart_prod_fin (Empty_set At) signe h1 signe_finite)
                       eps') x))))) (Empty_is_finite _) h2) as h3.
  unfold At, bt in h3. unfold At, bt. simpl in h3. simpl. 
  dependent rewrite -> h3. clear h3.
  rewrite times_set_empty'.
  rewrite times_set_sing.
  reflexivity.

rewrite <- h2.
rewrite (complete_dist_times_plus1 _ _ mns).
unfold plus_times_fun_all_maps1.
apply plus_set_functional.
f_equal.
apply functional_extensionality.
intro F.
unfold el_prod. unfold times_fun_fin_map1. unfold fin_map_times.
apply times_set_functional.
apply Extensionality_Ensembles.
red. split.
intro  y.
intros h3.
destruct h3 as [e h4 y h5].
apply Im_intro with e. assumption.
rewrite fun_to_fin_map_compat. rewrite fun_to_fin_map_compat.
unfold eps'. simpl. assumption.
split. simpl.
split; auto.
apply fin_map_app_in. assumption. assumption.
red.
intros y h3.
destruct h3 as [e h4 y h5].
rewrite fun_to_fin_map_compat in h5. rewrite fun_to_fin_map_compat in h5.
apply Im_intro with e. assumption. unfold eps' in h5. simpl in h5.
assumption.
split. simpl. split; auto. apply fin_map_app_in. assumption.
assumption.
Qed.



Lemma el_prod_covers' : 
  forall (E:Ensemble At) (pf:Finite E)
         (pf:Finite (Im (Full_set (Fin_map E signe mns)) el_prod)),
    plus_set (Im (Full_set (Fin_map E signe mns)) el_prod) pf = 1.
intros E h1 h2.
pose proof (el_prod_covers _ h1) as h3.
rewrite <- h3.
apply plus_set_functional. reflexivity.
Qed.


Lemma el_prod_ex_non_zero_el_prod_compose : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)} 
         (g:bt B->At) (a:Fin_map E signe mns),
    el_prod_compose g a <> 0 ->
    exists a':Fin_map (Im E g) signe mns,
      el_prod a' = el_prod_compose g a.
intros B E g a h0.
pose proof (non_zero_el_prod_compose_constant _ _ h0) as h0'.
pose proof (partition_rel_classes_im_rel_set E g) as h1.
red in h1. destruct h1 as [h1 h2]. symmetry in h2.  
pose proof (im_rel_classes_im_rel_set  (fin_map_fin_dom a) g) as h3.
assert (h4:forall y, Ensembles.In (Im E g) y -> 
                     exists! s:sign,
                       forall x:(bt B),
                         Ensembles.In E x ->
                         y = g x -> s = a |-> x).
  intros y h4.
  destruct h4 as [x h4]. clear h2. subst.
  exists (a |-> x). red. split.
  intros y h5.
  apply h0'; auto.
  intros s h5.
  specialize (h5 _ h4 eq_refl). rewrite h5. reflexivity.
pose (fun i:sig_set (Im E g) => (proj1_sig 
                                   (constructive_definite_description 
                                      _ (h4 _ (proj2_sig i))))) as f.
pose (sig_fun_to_fin_map f (finite_image _ _ _ g (fin_map_fin_dom a)) mns) as a'.
assert (h5:Included (Im (full_sig (Im E g)) f) signe).
  red. intros x h5. destruct x; constructor.
pose (fin_map_new_ran a' signe_finite h5) as a''.
exists a''.
unfold el_prod, el_prod_compose.
apply times_set_functional. rewrite im_im.
apply im_ext_in.
intros x h6.
f_equal. unfold a''. rewrite <- fin_map_new_ran_compat.
unfold a'. 
assert (h7:Ensembles.In (Im E g) (g x)).
  apply Im_intro with x. assumption. reflexivity.
rewrite (sig_fun_to_fin_map_compat f (finite_image (bt B) At E g (fin_map_fin_dom a)) mns _ h7).
unfold f. simpl.
destruct constructive_definite_description as [s h8].
simpl.
rewrite (h8 _ h6) at 1. reflexivity. reflexivity.
Qed.


Lemma non_zero_el_prod_compose_ex_el_prod : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)},
    Finite E ->
    forall (g:bt B->At) (a:Fin_map (Im E g) signe mns),
    el_prod a <> 0 ->
    exists a':Fin_map E signe mns,
      el_prod_compose g a' = el_prod a.
intros B E h0 g a h1.
pose (fun e:bt B => a |-> (g e)) as f.
pose (fun_to_fin_map E mns h0 f) as a'.
assert (h2:Included (Im E f) signe).
  red. intros x h2. destruct h2 as [x h2]. subst.
  destruct (f x); constructor.
pose (fin_map_new_ran a' signe_finite h2) as a''.
exists a''.
unfold el_prod_compose, el_prod.
apply times_set_functional. rewrite im_im.
apply im_ext_in.
intros x h3.
f_equal.
unfold a''. rewrite <- fin_map_new_ran_compat.
unfold a'. rewrite fun_to_fin_map_compat; auto.
Qed.




 
Lemma el_prod_compose_covers : 
  forall {B:Bool_Alg} (E:Ensemble (bt B)) (pf:Finite E)
         (g:(bt B)->At),
    plus_set (Im (Full_set (Fin_map E signe mns)) (el_prod_compose g)) 
    (finite_image _ _ _ _ (finite_fin_maps _ _ mns pf signe_finite)) = 1.
intros B E h1 g.
pose proof (excl_middle_sat (fun f:(Fin_map E signe mns) => el_prod_compose g f <> 0)) as h2.
assert (h3:Im (Full_set (Fin_map E signe mns)) (el_prod_compose g) =
           Im (Union
         [x : Fin_map E signe mns
         | (fun f : Fin_map E signe mns => el_prod_compose g f <> 0) x]
         [x : Fin_map E signe mns
         | ~ (fun f : Fin_map E signe mns => el_prod_compose g f <> 0) x]) (el_prod_compose g)).
  f_equal; auto. 
assert (h4:Finite ( Im (Union
         [x : Fin_map E signe mns
         | (fun f : Fin_map E signe mns => el_prod_compose g f <> 0) x]
         [x : Fin_map E signe mns
         | ~ (fun f : Fin_map E signe mns => el_prod_compose g f <> 0) x]) (el_prod_compose g))).
  rewrite <- h2. apply finite_image. apply finite_fin_maps. assumption. apply signe_finite.
pose proof (subsetT_eq_compat _ _ _ _ (finite_image (Fin_map E signe mns) (Btype (Bc A))
        (Full_set (Fin_map E signe mns)) (el_prod_compose g)
        (finite_fin_maps E signe mns h1 signe_finite)) h4 h3) as h5.
unfold At, bt in h5. unfold At, bt. simpl in h5. simpl.
dependent rewrite -> h5.
pose proof (im_union [x : Fin_map E signe mns | el_prod_compose g x <> 0]
           [x : Fin_map E signe mns | ~ el_prod_compose g x <> 0]
     (el_prod_compose g)) as h6.
assert (h7:Finite ( Union
         (Im [x : Fin_map E signe mns | el_prod_compose g x <> 0]
            (el_prod_compose g))
         (Im [x : Fin_map E signe mns | ~ el_prod_compose g x <> 0]
            (el_prod_compose g)))).
  rewrite <- h6. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h4 h7 h6) as h8.
unfold At, bt in h8. unfold At, bt. simpl in h8. simpl.
dependent rewrite -> h8.
assert (h9:Finite (Im [x : Fin_map E signe mns | el_prod_compose g x <> 0]
               (el_prod_compose g))).
  eapply Finite_downward_closed.
  apply h7. auto with sets.
assert (h10:Finite (Im [x : Fin_map E signe mns | ~ el_prod_compose g x <> 0]
               (el_prod_compose g))).
  eapply Finite_downward_closed. apply h7. auto with sets.
pose proof (plus_set_union' _ _ h9 h10 h7) as h11.
unfold At, bt in h11. unfold At, bt. rewrite h11 at 1.
assert (h12:plus_set
     (Im [x : Fin_map E signe mns | ~ el_prod_compose g x <> 0]
         (el_prod_compose g)) h10 = 0).
  destruct (classic (Inhabited [x : Fin_map E signe mns | ~ el_prod_compose g x <> 0])) as [h13 | h14].
    destruct h13 as [x h13]. destruct h13 as [h13]. 
    rewrite <- (plus_set_sing 0).
    apply plus_set_functional.
    apply Extensionality_Ensembles. red. split.
    red. intros y h14. destruct h14 as [y h14]. subst.
    destruct h14 as [h14]. apply NNPP in h14. rewrite h14.
    constructor.
    red.
    intros y h14. destruct h14.
    apply Im_intro with x. constructor. assumption.
    apply NNPP in h13. rewrite h13. reflexivity.
    apply not_inhabited_empty in h14.
    pose proof (image_empty (Fin_map E signe mns) _ (el_prod_compose g)) as h15.
    rewrite <- h14 in h15.
    pose proof (Empty_is_finite At) as h16.
    pose proof (subsetT_eq_compat _ _ _ _ h10 h16 h15) as h17.
    dependent rewrite -> h17.
    apply plus_set_empty'.
unfold At, bt in h12. unfold At, bt. rewrite h12 at 1.
rewrite zero_sum.
pose proof (finite_image _ _ _ g h1) as h0.
assert (h2':times_plus_fin_pair_map1 h0
       (fun_to_fin_map _ 0 (cart_prod_fin (Im E g) _ h0 signe_finite) eps') = 1).
    unfold times_plus_fin_pair_map1.
  unfold fin_map_times.
  generalize (finite_image At (Btype (Bc A)) (Im E g)
        (fin_map_app
           (plus_fin_pair_map1 h0
              (fun_to_fin_map (cart_prod (Im E g) signe) 0
                 (cart_prod_fin (Im E g) signe h0 signe_finite) eps')))
        (fin_map_fin_dom
           (plus_fin_pair_map1 h0
              (fun_to_fin_map (cart_prod (Im E g) signe) 0
                 (cart_prod_fin (Im E g) signe h0 signe_finite) eps')))).
  unfold plus_fin_pair_map1.
  rewrite im_fin_map_app_undoes_fun_to_fin_map.
  intros h2'. 
  assert (h3':
            (Im (Im E g)
                (fun x : At =>
                   plus_set
                     (im1
                        (fun_to_fin_map (cart_prod (Im E g) signe) 0
                                        (cart_prod_fin (Im E g) signe h0 signe_finite) eps') x)
                     (im1_fin
                        (fun_to_fin_map (cart_prod (Im E g) signe) 0
                                        (cart_prod_fin (Im E g) signe h0 signe_finite) eps') x))) =
            Im (Im E g) (fun x:At => plus_set (Im signe (fun y => (eps' (x, y)))) (finite_image _ _ _ _ signe_finite))).
  do 2 rewrite im_im.
  apply Extensionality_Ensembles.
  red. split.
  red. intros b h3'.
  destruct h3' as [a h4' b h5'].
  apply Im_intro with a. assumption.
  rewrite h5'.
  apply plus_set_functional.
  apply im1_fun_to_fin_map; auto. 
  apply Im_intro with a; auto.
  red. intros b h3'.
  destruct h3' as [a h4' b h5'].
  apply Im_intro with a. assumption. 
  rewrite h5'.
  apply plus_set_functional.
  symmetry. apply im1_fun_to_fin_map; auto.
  apply Im_intro with a; auto. 
  revert h2'.
  unfold At, bt in h3'. unfold At, bt. simpl in h3'. simpl. 
  rewrite h3' at 1. clear h3'.  
  clear h12 h11 h10 h9 h8 h7 h6 h5 h4 h3 h2. 
  pose proof (finite_inh_or_empty _ h1) as h1'.
  destruct h1' as [hinh | hninh].
  destruct hinh as [e hin]. 
(*  induction h1 as [|E h2' h3' e h4'].*)
  intro h2'.  
  rewrite <- times_set_sing.
  apply times_set_functional.
  rewrite im_im.
(*
 pose proof (image_empty _ _  (fun x : At =>
         plus_set (Im signe (fun y : sign => eps' (x, y)))
           (finite_image sign (Btype (Bc A)) signe
              (fun y : sign => eps' (x, y)) signe_finite))) as h3'.
  pose proof (image_empty _ _ g) as he.
  rewrite <- he in h3'. 
  
  (*unfold At, bt in h3'. unfold At, bt. simpl in h3'. simpl.  
  rewrite h3'.*)
 

  rewrite <- h3' at 1.
  do 2 rewrite im_im. do 2 rewrite image_empty.
  reflexivity.  

  rewrite <- h3' at 1.
  intro ha.
  apply times_set_functional. 
  do 2 rewrite im_im. 
  reflexivity.
  *) 
  apply Extensionality_Ensembles.
red. split.
red.
intros a h6'.
destruct h6' as [x h6' a h7'].
assert (heps: (fun y : sign => eps' (g x, y))  = eps (g x)).
  apply functional_extensionality. intro s. rewrite eps_eps'_compat.
  reflexivity.
rewrite fun_to_fin_map_compat in h7'.

(*rewrite heps in h7'.*)
pose proof (eps_covers _ h0 (g x)) as h8'.
assert (heq: plus_set
          (im1
             (fun_to_fin_map (cart_prod (Im E g) signe) 0
                (cart_prod_fin (Im E g) signe h0 signe_finite) eps') 
             (g x))
          (im1_fin
             (fun_to_fin_map (cart_prod (Im E g) signe) 0
                (cart_prod_fin (Im E g) signe h0 signe_finite) eps') 
             (g x)) = 
             plus_set (Im signe (eps (g x)))
          (finite_image sign (bt A) signe (eps (g x)) signe_finite)).
  apply plus_set_functional.
  unfold im1.
  assert (hedeq: (if eq_dec (Im E g) (Empty_set At)
    then Empty_set (Btype (Bc A))
    else
     Im signe
       (fun y : sign =>
        fun_to_fin_map (cart_prod (Im E g) signe) 0
          (cart_prod_fin (Im E g) signe h0 signe_finite) eps' |-> 
        (g x, y))) = 
                  Im signe
       (fun y : sign =>
        fun_to_fin_map (cart_prod (Im E g) signe) 0
          (cart_prod_fin (Im E g) signe h0 signe_finite) eps' |-> 
        (g x, y))).
    destruct (eq_dec (Im E g) (Empty_set At)) as [hed | hned].
    assert (h9:Ensembles.In (Im E g) (g e)).
      apply Im_intro with e; auto.
      rewrite hed in h9. contradiction.
    reflexivity.
  unfold At, bt in hedeq. unfold At, bt. rewrite hedeq at 1.
  apply im_ext_in.
  intros s hins.
  rewrite fun_to_fin_map_compat.
  rewrite eps_eps'_compat.
  reflexivity.
  constructor; simpl.
  split.
  apply Im_intro with x; auto. assumption.
  rewrite <- h8'. rewrite <- heq. subst.
  constructor. apply Im_intro with x; auto.
  red. intros x hone. destruct hone.
  apply Im_intro with e; auto.
  rewrite fun_to_fin_map_compat.
  pose proof (eps_covers _ h0 (g e)) as hec.
  rewrite <- hec.
  apply plus_set_functional.
  unfold im1.
  assert (h3: (if eq_dec (Im E g) (Empty_set (Btype (Bc A)))
    then Empty_set (Btype (Bc A))
    else
     Im signe
       (fun y : sign =>
        fun_to_fin_map (cart_prod (Im E g) signe) 0
          (cart_prod_fin (Im E g) signe h0 signe_finite) eps' |-> 
        (g e, y))) = Im signe
       (fun y : sign =>
        fun_to_fin_map (cart_prod (Im E g) signe) 0
          (cart_prod_fin (Im E g) signe h0 signe_finite) eps' |-> 
        (g e, y))).
    destruct (eq_dec (Im E g) (Empty_set (Btype (Bc A)))) as [heq | hneq].
    assert (h3:Ensembles.In (Im E g) (g  e)).
      apply Im_intro with e; auto.
      rewrite heq in h3. contradiction.
    reflexivity.
  unfold At, bt in h3. unfold At, bt.
  rewrite h3 at 1.
  apply im_ext_in.
  intros s h4.
  rewrite fun_to_fin_map_compat.
  rewrite eps_eps'_compat. reflexivity.
  constructor; simpl. split; auto.
  apply Im_intro with e; auto.
  apply Im_intro with e; auto.
  subst.
  intro h2.
  rewrite <- times_set_empty.
  apply times_set_functional.
  rewrite im_im. rewrite image_empty.
  reflexivity.
unfold bt in h2'. unfold bt. 
rewrite <- h2'.  
rewrite (complete_dist_times_plus1 _ _ mns).
unfold plus_times_fun_all_maps1. 

pose proof (excl_middle_sat (fun f:(Fin_map (Im E g) signe mns) => el_prod f <> 0)) as h2''.
assert (h3'':Im (Full_set (Fin_map (Im E g) signe mns)) (times_fun_fin_map1 eps') =
           Im (Union
         [x : Fin_map (Im E g) signe mns
         | (fun f : Fin_map (Im E g) signe mns => el_prod f <> 0) x]
         [x : Fin_map (Im E g) signe mns
         | ~ (fun f : Fin_map (Im E g) signe mns => el_prod f <> 0) x]) (times_fun_fin_map1 eps')).
  f_equal; auto. 
assert (h4'':Finite ( Im (Union
         [x : Fin_map (Im E g) signe mns
         | (fun f : Fin_map (Im E g) signe mns => el_prod f <> 0) x]
         [x : Fin_map (Im E g) signe mns
         | ~ (fun f : Fin_map (Im E g) signe mns => el_prod f <> 0) x]) (times_fun_fin_map1 eps'))).
  rewrite <- h2''. apply finite_image. apply finite_fin_maps. assumption. apply signe_finite.
pose proof (subsetT_eq_compat _ _ _ _ (finite_image (Fin_map (Im E g) signe mns) _
        (Full_set (Fin_map (Im E g) signe mns)) (times_fun_fin_map1 eps')
        (finite_fin_maps (Im E g) signe mns (finite_image _ _ _  g h1) signe_finite)) h4'' h3'') as h5''.
unfold At, bt in h5''. unfold At, bt. simpl in h5''. simpl. 
assert (ho:h0 = (finite_image (Btype (Bc B)) (Btype (Bc A)) E g h1)). apply proof_irrelevance. subst.
dependent rewrite -> h5''.
pose proof (im_union [x : Fin_map (Im E g) signe mns | el_prod x <> 0]
           [x : Fin_map (Im E g) signe mns | ~ el_prod x <> 0]
     (times_fun_fin_map1 eps')) as h6''.
unfold At, bt in h6''. unfold At, bt. 

assert (h7'':Finite ( Union
         (Im [x : Fin_map (Im E g) signe mns | el_prod x <> 0]
            (times_fun_fin_map1 eps'))
         (Im [x : Fin_map (Im E g) signe mns | ~ el_prod  x <> 0]
            (times_fun_fin_map1 eps')))).
  unfold At, bt. rewrite <- h6''. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h4'' h7'' h6'') as h8''.
dependent rewrite -> h8''.
assert (h9'':Finite (Im [x : Fin_map (Im E g) signe mns | el_prod x <> 0]
               (times_fun_fin_map1 eps'))).
  eapply Finite_downward_closed.
  apply h7''. auto with sets.
assert (h10'':Finite (Im [x : Fin_map (Im E g) signe mns | ~ el_prod x <> 0]
               (times_fun_fin_map1 eps'))).
  eapply Finite_downward_closed. apply h7''. auto with sets.
pose proof (plus_set_union' _ _ h9'' h10'' h7'') as h11''.
unfold At, bt in h11''. unfold At, bt. rewrite h11'' at 1.
assert (h12'':plus_set
     (Im [x : Fin_map (Im E g) signe mns | ~ el_prod x <> 0]
         (times_fun_fin_map1 eps')) h10'' = 0).
  destruct (classic (Inhabited [x : Fin_map (Im E g) signe mns | ~ el_prod x <> 0])) as [h13 | h14].
    destruct h13 as [x h13]. destruct h13 as [h13]. 
    rewrite <- (plus_set_sing 0).
    apply plus_set_functional.
    apply Extensionality_Ensembles. red. split.
    red. intros y h14. destruct h14 as [y h14]. subst.
    destruct h14 as [h14]. apply NNPP in h14. 
    assert (h15:times_fun_fin_map1 eps' y = 0).  
      unfold times_fun_fin_map1.  unfold fin_map_times. 
      unfold el_prod in h14.  rewrite <- h14.
      apply times_set_functional. 
      apply im_ext_in.
      intros d h16.
      rewrite fun_to_fin_map_compat. rewrite fun_to_fin_map_compat.
      rewrite eps_eps'_compat.
      reflexivity.
      constructor. simpl. split; auto. 
      apply fin_map_app_in. 
      simpl in h16. simpl. apply h16. 
      destruct h16 as [c h16]. subst.
      apply Im_intro with c; auto.
   rewrite h15. constructor.
  red. intros d h14. destruct h14. apply Im_intro with x.
  constructor. assumption.
        unfold times_fun_fin_map1.  unfold fin_map_times. 
      unfold el_prod in h13. apply NNPP in h13.  rewrite <- h13.
      apply times_set_functional. 
      apply im_ext_in.
      intros d h16.
      rewrite fun_to_fin_map_compat. rewrite fun_to_fin_map_compat.
      rewrite eps_eps'_compat.
      reflexivity.
      constructor. simpl. split; auto. 
      apply fin_map_app_in. 
      simpl in h16. simpl. apply h16. 
      destruct h16 as [c h16]. subst.
      apply Im_intro with c; auto.
  apply not_inhabited_empty in h14.
  pose proof (Empty_is_finite (Fin_map (Im E g) signe mns)) as h15.
  rewrite <- h14 in h15.
  apply finite_image with (f:=times_fun_fin_map1 eps') in h15.
  pose proof (feq_im _ _ (times_fun_fin_map1 eps') h14) as h16.
  rewrite image_empty in h16.
  pose proof (subsetT_eq_compat _ _ _ _ h10'' (Empty_is_finite  _) h16) as h17.
  unfold bt, At in h17. unfold bt, At. simpl in h17. simpl. dependent rewrite -> h17.
  rewrite plus_set_empty'.
  reflexivity. 
unfold At, bt in h12''. unfold At, bt. 
rewrite h12'' at 1.
rewrite zero_sum.
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split. red. intros x h13.
destruct h13 as [f h13]. subst. destruct h13 as [h13].
pose proof (el_prod_ex_non_zero_el_prod_compose g f h13) as h14.
destruct h14 as [f' h14]. 
apply Im_intro with f'.
constructor.
unfold bt, At in h14. unfold bt, At. simpl in h14. simpl. simpl. rewrite h14 at 1.
assumption. 
rewrite <- h14. 
unfold el_prod, times_fun_fin_map1.
unfold fin_map_times.
apply times_set_functional.
apply im_ext_in.
intros x h15.
rewrite fun_to_fin_map_compat. rewrite fun_to_fin_map_compat.
rewrite eps_eps'_compat. reflexivity. 
constructor. simpl. split; auto. 
apply fin_map_app_in. destruct h15 as [x h15]. subst.
apply Im_intro with x. assumption. reflexivity.
assumption.
red.
intros x h13.
destruct h13 as [f h13]. subst. destruct h13 as [h13].
pose proof (non_zero_el_prod_compose_ex_el_prod h1 g f h13) as h14.
destruct h14 as [f' h14].
apply Im_intro with f'.
constructor.
rewrite h14. assumption.
rewrite h14.
unfold times_fun_fin_map1, el_prod.
unfold fin_map_times.
apply times_set_functional.
apply im_ext_in.
intros x h15.
rewrite fun_to_fin_map_compat. rewrite fun_to_fin_map_compat.
rewrite eps_eps'_compat.
reflexivity.
constructor. simpl. split; auto. apply fin_map_app_in.
destruct h15 as [x h15]. subst. apply Im_intro with x.
assumption. reflexivity. assumption.
Qed.



Definition non_zero_el_prod_maps (E:Ensemble At) :=
  [a:Fin_map E signe mns | el_prod a <> 0].

Lemma non_zero_el_prod_maps_fin : 
  forall (E:Ensemble At), Finite E ->
                          Finite (non_zero_el_prod_maps E).
intros E h1.
assert (h2:Included (non_zero_el_prod_maps E) (Full_set (Fin_map E signe mns))). red. intros; constructor.
pose proof (finite_fin_maps _ _ mns h1 signe_finite) as h3.
apply (Finite_downward_closed _ _  h3 _ h2).
Qed.


Definition non_zero_el_prod_compose_maps 
           {B:Bool_Alg} (g:(bt B)->At) (E:Ensemble (bt B)) :=
  [a:Fin_map E signe mns | el_prod_compose g a <> 0].

Lemma non_zero_el_prod_compose_maps_fin : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)} 
         (g:bt B->At), 
    Finite E -> Finite (non_zero_el_prod_compose_maps g E).
intros B E g h1.
assert (h2:Included (non_zero_el_prod_compose_maps g E) (Full_set (Fin_map E signe mns))). red. intros; constructor.
pose proof (finite_fin_maps _ _ mns h1 signe_finite) as h3.
apply (Finite_downward_closed _ _  h3 _ h2).
Qed.



Definition plus_subset_non_zero_el_prod_maps (E:Ensemble At) (pfe:Finite E) (S:Ensemble (Fin_map E signe mns))
           (pfi:Included S (non_zero_el_prod_maps E)) : At :=
  plus_set (Im S el_prod) (finite_image _ _ _ el_prod (Finite_downward_closed _ _ (non_zero_el_prod_maps_fin _ pfe) _ pfi)).


Definition plus_subset_non_zero_el_prod_compose_maps 
           {B:Bool_Alg} (E:Ensemble (bt B)) (pfe:Finite E)
           (g:(bt B)->At) (S:Ensemble (Fin_map E signe mns))
           (pfi:Included S (non_zero_el_prod_compose_maps g E)) : At :=  
  plus_set (Im S (el_prod_compose g)) (finite_image _ _ _ (el_prod_compose g) (finite_fin_map_ens S pfe signe_finite)).



Lemma inclusion_plus_subset_non_zero_el_prod_maps_preserves_int : 
  forall (E:Ensemble At) (pfe:Finite E) (X Y:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_maps E) ->
    Included Y (non_zero_el_prod_maps E) ->
    Included (Intersection X Y) (non_zero_el_prod_maps E).
intros E h1 X Y h2 h3.
red.
intros a h4.
destruct h4 as [a h4l h4r].
constructor.
red in h3.
specialize (h3 _ h4r).
destruct h3.
assumption.
Qed.

Lemma inclusion_plus_subset_non_zero_el_prod_maps_preserves_union : 
  forall (E:Ensemble At) (pfe:Finite E) (X Y:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_maps E) ->
    Included Y (non_zero_el_prod_maps E) ->
    Included (Union X Y) (non_zero_el_prod_maps E).
intros E h1 X Y h2 h3. red.
intros a h4.
destruct h4 as [a h4l | a h4r].
red in h2. specialize (h2 _ h4l). assumption.
red in h3. specialize (h3 _ h4r). assumption.
Qed.


Lemma inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp : 
  forall (E:Ensemble At) (pfe:Finite E) (X:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_maps E) ->
    Included (Setminus (non_zero_el_prod_maps E) X) (non_zero_el_prod_maps E).
intros E h1 X h2. red.
intros a h4.
destruct h4 as [h4l].
assumption.
Qed.

Lemma inclusion_plus_subset_non_zero_el_prod_maps_preserves_setminus : 
  forall (E:Ensemble At) (pfe:Finite E) (X Y:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_maps E) ->
    Included (Setminus X Y) (non_zero_el_prod_maps E).
intros E h1 X Y h2. red.
intros a h3.
destruct h3 as [h3a h3b].
auto with sets.
Qed.

Lemma inclusion_plus_subset_non_zero_el_prod_maps_preserves_symdiff_full : 
  forall (E:Ensemble At) (pfe:Finite E) (X Y:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_maps E) ->
    Included Y (non_zero_el_prod_maps E) ->
    Included (Symdiff_full (non_zero_el_prod_maps E) X Y) (non_zero_el_prod_maps E).
intros E h1 X Y h2 h3. red.
intros a h4.
destruct h4 as [a h4l | a h4r].
destruct h4l as [a h4a h4b].  auto with sets.
destruct h4r as [a h4a h4b]. auto with sets.
Qed.


Lemma inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_union : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)}
         (pfe:Finite E) (g:(bt B)->At) 
         (X Y:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_compose_maps g E) ->
    Included Y (non_zero_el_prod_compose_maps g E) ->
    Included (Union X Y) (non_zero_el_prod_compose_maps g E).
intros B E h1 g X Y h2 h3. red.
intros a h4.
destruct h4 as [a h4l | a h4r].
red in h2. specialize (h2 _ h4l). assumption.
red in h3. specialize (h3 _ h4r). assumption.
Qed.

Lemma inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_int : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)}
         (pfe:Finite E) (g:(bt B)->At) 
         (X Y:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_compose_maps g E) ->
    Included Y (non_zero_el_prod_compose_maps g E) ->
    Included (Intersection X Y) (non_zero_el_prod_compose_maps g E).
intros B E h1 g X Y h2 h3.
red.
intros a h4.
destruct h4 as [a h4l h4r].
constructor.
red in h3.
specialize (h3 _ h4r).
destruct h3.
assumption.
Qed.


Lemma inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_comp : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)}
         (pfe:Finite E) (g:(bt B)->At) 
         (X:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_compose_maps g E) ->
    Included (Setminus (non_zero_el_prod_compose_maps g E) X) (non_zero_el_prod_compose_maps g E).
intros B E h1 g X h2. red.
intros a h4.
destruct h4 as [h4l].
assumption.
Qed.

(*maybe do other analogues as well.*)



Lemma plus_subset_non_zero_el_prod_maps_empty :
  forall (E:Ensemble At) (pfe:Finite E),
    plus_subset_non_zero_el_prod_maps E pfe _ (empty_inclusion _) = 0.
intros. unfold plus_subset_non_zero_el_prod_maps.
generalize ((finite_image (Fin_map E signe mns) At (Empty_set (Fin_map E signe mns))
        el_prod
        (Finite_downward_closed (Fin_map E signe mns)
           (non_zero_el_prod_maps E) (non_zero_el_prod_maps_fin E pfe)
           (Empty_set (Fin_map E signe mns))
           (empty_inclusion (non_zero_el_prod_maps E))))).
pose proof (image_empty (Fin_map E signe mns) _ el_prod) as h1.
rewrite h1.
intro. apply plus_set_empty'.
Qed.

Lemma le_plus_subset_non_zero_el_prod_maps : 
  forall (E:Ensemble At) (pfe:Finite E) (X:Ensemble (Fin_map E signe mns)) 
         (pf:Included X (non_zero_el_prod_maps E)) (a:Fin_map E signe mns), 
    Ensembles.In X a -> le (el_prod a) (plus_subset_non_zero_el_prod_maps E pfe _ pf).
intros E h1 X h2 a h3.
unfold plus_subset_non_zero_el_prod_maps.
apply le_plus_set.
apply Im_intro with a; auto.
Qed.

Lemma plus_subset_non_zero_el_prod_maps_empty_rev :
  forall (E:Ensemble At) (pfe:Finite E) (X:Ensemble (Fin_map E signe mns))
         (pf:Included X (non_zero_el_prod_maps E)),
         plus_subset_non_zero_el_prod_maps E pfe _ pf = 0 -> X = Empty_set _.
intros E h1 X h2 h3.
apply NNPP.
intros h4.
pose proof (not_empty_Inhabited _ _ h4) as h5.
destruct h5 as [a h5].
unfold non_zero_el_prod_maps in h2.
assert (h6:Ensembles.In [a:Fin_map E signe mns | el_prod a <> 0] a). auto with sets.
destruct h6 as [h6].
pose proof (le_plus_subset_non_zero_el_prod_maps _ h1 _ h2 _ h5) as h7.
contradict h6.
rewrite h3 in h7.
apply le_x_0.
assumption.
Qed.
                                                                                                  
Lemma plus_set_el_prod_zero : forall (E:Ensemble At) (pf:Finite _), plus_set (Im [x : Fin_map E signe mns | el_prod x = 0] el_prod) pf = 0.
intros E h1.
destruct (classic (Im [x : Fin_map E signe mns | el_prod x = 0] el_prod = Empty_set _)) as [h2 | h3].
generalize h1.
rewrite h2. intro. apply plus_set_empty'.
pose proof (not_empty_Inhabited _ _ h3) as h4.
rewrite <- plus_set_sing.
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split. red. intros x h15.
inversion h15 as [a h16 y]. subst.
inversion h15 as [b h17 z]. subst.
destruct h17 as [h17]. rewrite <- H in h17.
rewrite h17. constructor.
red. intros x h15.
destruct h15. subst.
destruct h4 as [a h4].
inversion h4. subst.
inversion h4. subst.
inversion H0. 
rewrite H2 in H1.
rewrite <- H1.
apply Im_intro with x. constructor. reflexivity. reflexivity.
Qed.

Lemma plus_set_el_prod_compose_zero :
  forall {B:Bool_Alg}  (g:bt B->At) (E:Ensemble (bt B)) 
         (pf:Finite _),
         plus_set (Im [x : Fin_map E signe mns | el_prod_compose g x = 0] (el_prod_compose g)) pf = 0.
intros B g E h1.
destruct (classic (Im [x : Fin_map E signe mns | el_prod_compose g x = 0] (el_prod_compose g) = Empty_set _)) as [h2 | h3].
generalize h1.
rewrite h2. intro. apply plus_set_empty'.
pose proof (not_empty_Inhabited _ _ h3) as h4.
rewrite <- plus_set_sing.
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split. red. intros x h15.
inversion h15 as [a h16 y]. subst.
inversion h15 as [b h17 z]. subst.
destruct h17 as [h17]. rewrite <- H in h17.
rewrite h17. constructor.
red. intros x h15.
destruct h15. subst.
destruct h4 as [a h4].
inversion h4. subst.
inversion h4. subst.
inversion H0. 
rewrite H2 in H1.
rewrite <- H1.
apply Im_intro with x. constructor. reflexivity. reflexivity.
Qed.



Lemma plus_subset_non_zero_el_prod_maps_full : 
  forall (E:Ensemble At) (pfe:Finite E),
    plus_subset_non_zero_el_prod_maps E pfe _ (inclusion_reflexive _) = 1.
intros E h1.
unfold plus_subset_non_zero_el_prod_maps.
unfold non_zero_el_prod_maps.
pose proof (excl_middle_sat (fun a:(Fin_map E signe mns) => el_prod a = 0)) as h2.
pose proof f_equal.
pose proof (feq_im _ _ el_prod h2) as h3.
simpl in h3. 
rewrite im_union in h3.
pose proof (finite_fin_maps _ _ mns h1 signe_finite) as h4. 
pose proof (finite_image _ _ _ el_prod h4) as h5.
pose proof h5 as h6.
rewrite h3 in h6.
pose proof f_equal.
pose proof (plus_set_functional _ _ h5 h6 h3) as h7.
pose proof (el_prod_covers' _ h1 h5) as h8.
rewrite h8 in h7. symmetry in h7.
pose proof (plus_set_union).
assert (h9:Included (Im [x : Fin_map E signe mns | el_prod x = 0] el_prod)  (Im (Full_set (Fin_map E signe mns)) el_prod)).
rewrite h3. auto with sets.
assert (h10:Included (Im [x : Fin_map E signe mns | el_prod x <> 0] el_prod)  (Im (Full_set (Fin_map E signe mns)) el_prod)).
rewrite h3. auto with sets.
pose proof (Finite_downward_closed).
pose proof (Finite_downward_closed _ _ h5 _ h9) as h11.
pose proof (Finite_downward_closed _ _ h5 _ h10) as h12.
pose proof (plus_set_union _ _ h11 h12) as h13.
assert (h14:(Union_preserves_Finite (Btype (Bc A))
             (Im [x : Fin_map E signe mns | el_prod x = 0] el_prod)
             (Im [x : Fin_map E signe mns | el_prod x <> 0] el_prod) h11 h12) = h6). apply proof_irrelevance.
rewrite h14 in h13 at 1.
unfold At, bt in h7. unfold At, bt in h13.
rewrite h7 in h13. 
pose proof plus_subset_non_zero_el_prod_maps_empty.
pose proof (plus_set_el_prod_zero _ h11) as h15.
unfold At in h15. unfold At in h13. rewrite h15 in h13.
rewrite comm_sum in h13.
rewrite zero_sum in h13.
rewrite h13.
apply plus_set_functional. reflexivity.
Qed.



Lemma plus_subset_non_zero_el_prod_maps_int :
  forall (E:Ensemble At) (pfe:Finite E) 
         (X Y:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_maps E))
         (pfy:Included Y (non_zero_el_prod_maps E)),
         (plus_subset_non_zero_el_prod_maps E pfe _ pfx) *
         (plus_subset_non_zero_el_prod_maps E pfe _ pfy) =
         plus_subset_non_zero_el_prod_maps E pfe (Intersection X Y) (inclusion_plus_subset_non_zero_el_prod_maps_preserves_int _ pfe X Y pfx pfy).
intros E h1 X Y h2 h3.
unfold plus_subset_non_zero_el_prod_maps.
rewrite dist_set_plus2.
pose proof (cart_prod_im X Y el_prod el_prod) as h4.
pose proof (excl_middle_sat' (cart_prod X Y) (fun pr => fst pr = snd pr)) as h5.
rewrite h5 in h4. clear h5.
rewrite im_union in h4.
pose proof (feq_im _ _ (fun p : Btype (Bc A) * Btype (Bc A) => fst p * snd p) h4) as h5.
rewrite im_union in h5.
do 2 rewrite im_im in h5. simpl in h5.
pose proof (plus_set_functional).
pose proof (finite_fin_map_ens X h1 signe_finite) as h6.
pose proof (finite_fin_map_ens Y h1 signe_finite) as h7.
pose proof (finite_image _ _ _ el_prod h6) as h8.
pose proof (finite_image _ _ _ el_prod h7) as h9.
pose proof (cart_prod_fin _ _ h8 h9) as h10.
pose proof (finite_image).
pose proof (finite_image _ _ _  (fun p : Btype (Bc A) * Btype (Bc A) => fst p * snd p) h10) as h11.
rewrite h5 in h11.
pose proof (plus_set_functional _ _ (finite_image (Btype (Bc A) * Btype (Bc A)) (Btype (Bc A))
        (cart_prod (Im X el_prod) (Im Y el_prod))
        (fun p : Btype (Bc A) * Btype (Bc A) => fst p * snd p)
        (cart_prod_fin (Im X el_prod) (Im Y el_prod)
           (finite_image (Fin_map E signe mns) At X el_prod
              (Finite_downward_closed (Fin_map E signe mns)
                 (non_zero_el_prod_maps E) (non_zero_el_prod_maps_fin E h1) X
                 h2))
           (finite_image (Fin_map E signe mns) At Y el_prod
              (Finite_downward_closed (Fin_map E signe mns)
                 (non_zero_el_prod_maps E) (non_zero_el_prod_maps_fin E h1) Y
                 h3)))) h11 h5) as h12.
assert (h13:Included (Im
                [x : Fin_map E signe mns * Fin_map E signe mns
                | Ensembles.In (cart_prod X Y) x /\ fst x = snd x]
                (fun x : Fin_map E signe mns * Fin_map E signe mns =>
                 el_prod (fst x) * el_prod (snd x)))
  (Union
             (Im
                [x : Fin_map E signe mns * Fin_map E signe mns
                | Ensembles.In (cart_prod X Y) x /\ fst x = snd x]
                (fun x : Fin_map E signe mns * Fin_map E signe mns =>
                 el_prod (fst x) * el_prod (snd x)))
             (Im
                [x : Fin_map E signe mns * Fin_map E signe mns
                | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
                (fun x : Fin_map E signe mns * Fin_map E signe mns =>
                 el_prod (fst x) * el_prod (snd x))))). auto with sets.

assert (h14:Included (Im
                [x : Fin_map E signe mns * Fin_map E signe mns
                | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
                (fun x : Fin_map E signe mns * Fin_map E signe mns =>
                 el_prod (fst x) * el_prod (snd x)))
  (Union
             (Im
                [x : Fin_map E signe mns * Fin_map E signe mns
                | Ensembles.In (cart_prod X Y) x /\ fst x = snd x]
                (fun x : Fin_map E signe mns * Fin_map E signe mns =>
                 el_prod (fst x) * el_prod (snd x)))
             (Im
                [x : Fin_map E signe mns * Fin_map E signe mns
                | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
                (fun x : Fin_map E signe mns * Fin_map E signe mns =>
                 el_prod (fst x) * el_prod (snd x))))). auto with sets.
pose proof (Finite_downward_closed _ _ h11 _ h13) as h15. clear h13.
pose proof (Finite_downward_closed _ _ h11 _ h14) as h16. clear h14.
rewrite (plus_set_union' _ _ h15 h16 h11) in h12 at 1.
pose proof (finite_fin_map_squared_ens [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x] h1 signe_finite) as h13.
rewrite (plus_set_el_prod_disjoint' _ X Y h13 h16) in h12.
rewrite zero_sum in h12.
unfold At. unfold At in h12.
rewrite h12 at 1.
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split.
red.
intros x h17.
inversion h17 as [a h18 b h19 h20]. subst.
destruct h18 as [h18].
destruct h18 as [h18l h18r].
destruct h18l as [h18l].
destruct h18l as [h18a h18b].
rewrite h18r.
rewrite idem_prod.
rewrite h18r in h18a.
apply Im_intro with (snd a).
split; auto. reflexivity.
red.
intros x h17.
inversion h17 as [a h18 b h19 h20]. subst.
destruct h18 as [a h18l h18r].
apply Im_intro with (a, a).
constructor.
simpl.
split; auto. split; auto.
simpl. rewrite idem_prod. reflexivity.
Qed.


Lemma plus_subset_non_zero_el_prod_maps_union :
  forall (E:Ensemble At) (pfe:Finite E) 
         (X Y:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_maps E))
         (pfy:Included Y (non_zero_el_prod_maps E)),
         (plus_subset_non_zero_el_prod_maps E pfe _ pfx) +
         (plus_subset_non_zero_el_prod_maps E pfe _ pfy) =
         plus_subset_non_zero_el_prod_maps E pfe (Union X Y) (inclusion_plus_subset_non_zero_el_prod_maps_preserves_union _ pfe X Y pfx pfy).
intros E h1 X Y h2 h3. 
unfold plus_subset_non_zero_el_prod_maps.
generalize (finite_image (Fin_map E signe mns) At (Union X Y) el_prod
        (Finite_downward_closed (Fin_map E signe mns)
           (non_zero_el_prod_maps E) (non_zero_el_prod_maps_fin E h1)
           (Union X Y)
           (inclusion_plus_subset_non_zero_el_prod_maps_preserves_union E h1
              X Y h2 h3))).
rewrite <- plus_set_union.
intro h4.
apply plus_set_functional.
rewrite im_union.
reflexivity.
Qed.

Lemma plus_subset_non_zero_el_prod_maps_comp :
  forall (E:Ensemble At) (pfe:Finite E) 
         (X:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_maps E)),
         - (plus_subset_non_zero_el_prod_maps E pfe _ pfx) =
         plus_subset_non_zero_el_prod_maps E pfe _ (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp _ pfe X pfx).
intros E h1 X h2.
assert (h3:(plus_subset_non_zero_el_prod_maps E h1 X h2) *
   plus_subset_non_zero_el_prod_maps E h1
     (Setminus (non_zero_el_prod_maps E) X)
     (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp E h1 X h2) = 0).
  rewrite plus_subset_non_zero_el_prod_maps_int.
  rewrite <- (plus_subset_non_zero_el_prod_maps_empty E h1).
  unfold plus_subset_non_zero_el_prod_maps.
  apply plus_set_functional.
  apply feq_im.
  apply int_setminus.
assert (h4:(plus_subset_non_zero_el_prod_maps E h1 X h2) +
   plus_subset_non_zero_el_prod_maps E h1
     (Setminus (non_zero_el_prod_maps E) X)
     (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp E h1 X h2) = 1).
  rewrite plus_subset_non_zero_el_prod_maps_union.
  rewrite <- (plus_subset_non_zero_el_prod_maps_full E h1).
  unfold plus_subset_non_zero_el_prod_maps.
  apply plus_set_functional. apply feq_im.
  apply union_setminus_incl. assumption.
pose proof (comp_char _ _ h3 h4) as h5.
symmetry.
assumption.
Qed.

Lemma plus_subset_non_zero_el_prod_maps_setminus :
  forall (E:Ensemble At) (pfe:Finite E) 
         (X Y:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_maps E))
         (pfy:Included Y (non_zero_el_prod_maps E)),
         (plus_subset_non_zero_el_prod_maps E pfe _ pfx) *
         - (plus_subset_non_zero_el_prod_maps E pfe _ pfy) =
         plus_subset_non_zero_el_prod_maps E pfe 
                                           (Intersection X (Setminus (non_zero_el_prod_maps E) Y))
                                                         (inclusion_plus_subset_non_zero_el_prod_maps_preserves_int _ pfe X _ pfx
                                                         (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp _ pfe Y pfy)).
intros E h1 X Y h2 h3. 
rewrite <- plus_subset_non_zero_el_prod_maps_int.
rewrite <- plus_subset_non_zero_el_prod_maps_comp.
reflexivity.
Qed.



Lemma plus_subset_non_zero_el_prod_maps_symdiff_full :
  forall (E:Ensemble At) (pfe:Finite E) 
         (X Y:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_maps E))
         (pfy:Included Y (non_zero_el_prod_maps E)),
         (plus_subset_non_zero_el_prod_maps E pfe _ pfx) /_\
         (plus_subset_non_zero_el_prod_maps E pfe _ pfy) =
         plus_subset_non_zero_el_prod_maps E pfe (Symdiff_full (non_zero_el_prod_maps E) X Y) (inclusion_plus_subset_non_zero_el_prod_maps_preserves_symdiff_full _ pfe X Y pfx pfy).
intros E h1 X Y h2 h3. 
unfold sym_diff.
do 2 rewrite plus_subset_non_zero_el_prod_maps_comp.
do 2 rewrite plus_subset_non_zero_el_prod_maps_int.
rewrite plus_subset_non_zero_el_prod_maps_union.
unfold Symdiff_full. unfold Setminus_full.
unfold plus_subset_non_zero_el_prod_maps.
apply plus_set_functional. reflexivity.
Qed.


Definition non_zero_el_prod_maps_of_set
           {E:Ensemble At} (pfe:Finite E)
           (S:Ensemble (Fin_map E signe mns)) :=
  Intersection S (non_zero_el_prod_maps E).

Lemma non_zero_el_prod_maps_of_set_eq : 
  forall {E:Ensemble At} (pfe:Finite E)
         (S:Ensemble (Fin_map E signe mns)),
    non_zero_el_prod_maps_of_set pfe S =
  [f:Fin_map E signe mns | Ensembles.In S f /\ el_prod f <> 0].
intros E h1 S.
apply Extensionality_Ensembles.
red. split.
red. intros x h2. destruct h2 as [f h2 h3].
destruct h3 as [h3].
constructor. split; auto.
red. intros x h2. destruct h2 as [h2]. destruct h2 as [h2a h2b].
constructor; auto. constructor. assumption.
Qed.

Lemma non_zero_el_prod_maps_of_set_decompose : 
  forall {E:Ensemble At} (pfe:Finite E)
         (S:Ensemble (Fin_map E signe mns)),
    S = Union (non_zero_el_prod_maps_of_set pfe S)
              [f:Fin_map E signe mns | Ensembles.In S f /\ el_prod f = 0].
intros E h1 S.
apply Extensionality_Ensembles.
red. split.
red. intros f h2.
destruct (classic (el_prod f = 0)) as [h3 | h4].
right. constructor. split; auto.
left. constructor; auto. constructor; auto.
red. intros f h2. 
destruct h2 as [f h2 | f h3].
destruct h2; auto.
destruct h3 as [h3]. destruct h3; auto.
Qed.


Lemma incl_non_zero_el_prod_maps_of_set : 
  forall {E:Ensemble At} (pfe:Finite E)
         (S:Ensemble (Fin_map E signe mns)),
    Included (non_zero_el_prod_maps_of_set pfe S) (non_zero_el_prod_maps E). 
intros E h1 S.
unfold non_zero_el_prod_maps_of_set. 
auto with sets.
Qed.

Lemma finite_non_zero_el_prod_maps_of_set : 
    forall {E:Ensemble At} (pfe:Finite E)
           (S:Ensemble (Fin_map E signe mns)),
      Finite (non_zero_el_prod_maps_of_set pfe S).
intros E h1 S.
eapply Finite_downward_closed.
apply (non_zero_el_prod_maps_fin _ h1). 
apply incl_non_zero_el_prod_maps_of_set.
Qed.




Definition non_zero_el_prod_compose_maps_of_set
           {B:Bool_Alg} {E:Ensemble (bt B)} 
           (pfe:Finite E) (g:bt B->At) 
           (S:Ensemble (Fin_map E signe mns)) :=
  Intersection S (non_zero_el_prod_compose_maps g E).


Lemma non_zero_el_prod_compose_maps_of_set_eq : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)} 
         (pfe:Finite E) (g:bt B->At) 
         (S:Ensemble (Fin_map E signe mns)),
    non_zero_el_prod_compose_maps_of_set pfe g S =
  [f:Fin_map E signe mns | Ensembles.In S f /\ el_prod_compose g f <> 0].
intros B E h1 g S.
apply Extensionality_Ensembles.
red. split.
red. intros x h2. destruct h2 as [f h2 h3].
destruct h3 as [h3].
constructor. split; auto.
red. intros x h2. destruct h2 as [h2]. destruct h2 as [h2a h2b].
constructor; auto. constructor. assumption.
Qed.


Lemma non_zero_el_prod_compose_maps_of_set_decompose : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)} 
         (pfe:Finite E) (g:bt B->At) 
         (S:Ensemble (Fin_map E signe mns)),
    S = Union (non_zero_el_prod_compose_maps_of_set pfe g S)
              [f:Fin_map E signe mns | Ensembles.In S f /\ el_prod_compose g f = 0].
intros B E h1 g S.
apply Extensionality_Ensembles.
red. split.
red. intros f h2.
destruct (classic (el_prod_compose g f = 0)) as [h3 | h4].
right. constructor. split; auto.
left. constructor; auto. constructor; auto.
red. intros f h2. 
destruct h2 as [f h2 | f h3].
destruct h2; auto.
destruct h3 as [h3]. destruct h3; auto.
Qed.



Lemma incl_non_zero_el_prod_compose_maps_of_set : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)} 
         (pfe:Finite E) (g:bt B->At) 
         (S:Ensemble (Fin_map E signe mns)),
    Included (non_zero_el_prod_compose_maps_of_set pfe g S) (non_zero_el_prod_compose_maps g E). 
intros B E h1 g S.
unfold non_zero_el_prod_compose_maps_of_set.  
auto with sets.
Qed.

Lemma finite_non_zero_el_prod_compose_maps_of_set : 
    forall {B:Bool_Alg} {E:Ensemble (bt B)} 
           (pfe:Finite E) (g:bt B->At)
           (S:Ensemble (Fin_map E signe mns)),
      Finite (non_zero_el_prod_compose_maps_of_set pfe g S).
intros B E h1 g S.
eapply Finite_downward_closed.
apply (non_zero_el_prod_compose_maps_fin g h1). 
apply incl_non_zero_el_prod_compose_maps_of_set.
Qed.


Lemma plus_set_zero_el_prod_maps_of_set :
  forall {E:Ensemble At},
    Finite E ->
    forall (S:Ensemble (Fin_map E signe mns))
         (pf:Finite (Im [f : Fin_map E signe mns | Ensembles.In S f /\ el_prod f = 0]
             el_prod)),
    plus_set _ pf = 0.
intros E h1 S h2.
destruct (classic (Inhabited  (Im [f : Fin_map E signe mns | Ensembles.In S f /\ el_prod f = 0]
        el_prod))) as [h3 | h4].
destruct h3 as [f h3]. destruct h3 as [f h3]. subst.
destruct h3 as [h3].
rewrite <- (plus_set_sing 0).
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split.
red. intros g h4. destruct h4 as [g h4]. subst.
destruct h4 as [h4]. destruct h4 as [h4 h5]. rewrite h5.
constructor.
red. intros x h4. destruct h4.
apply Im_intro with f. constructor. assumption.
destruct h3 as [h3 h4]. rewrite h4. reflexivity.
apply not_inhabited_empty in h4.
pose proof (Empty_is_finite At) as h5.
pose proof (subsetT_eq_compat _ _ _ _ h2 h5 h4) as h6.
dependent rewrite -> h6.
apply plus_set_empty'.
Qed.



Lemma plus_subset_el_prod_maps_eq_same_non_zero : 
  forall {E:Ensemble At} (pfe:Finite E)
         (S:Ensemble (Fin_map E signe mns)),
    plus_set (Im S el_prod) (finite_image _ _ _ el_prod (finite_fin_map_ens S pfe signe_finite)) =
    plus_set (Im (non_zero_el_prod_maps_of_set pfe S) el_prod)
             (finite_image _ _ _ el_prod 
                           (finite_non_zero_el_prod_maps_of_set pfe S)).
intros E h1 S.
pose proof (non_zero_el_prod_maps_of_set_decompose h1 S) as h2.
pose proof (f_equal (fun X=>Im X el_prod) h2) as h3. simpl in h3.
assert (h4:Finite (Im
         (Union (non_zero_el_prod_maps_of_set h1 S)
            [f : Fin_map E signe mns | Ensembles.In S f /\ el_prod f = 0])
         el_prod)).
  rewrite <- h3. apply finite_image. apply finite_fin_map_ens.
  assumption. apply signe_finite.
pose proof (subsetT_eq_compat _ _ _ _ (finite_image (Fin_map E signe mns) At S el_prod
        (finite_fin_map_ens S h1 signe_finite)) h4 h3) as h5.
dependent rewrite -> h5.
pose proof (im_union (non_zero_el_prod_maps_of_set h1 S)
           [f : Fin_map E signe mns | Ensembles.In S f /\ el_prod f = 0] el_prod) as h6.
assert (h7:Finite ( Union (Im (non_zero_el_prod_maps_of_set h1 S) el_prod)
         (Im [f : Fin_map E signe mns | Ensembles.In S f /\ el_prod f = 0]
            el_prod))).
  rewrite <- h6. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h4 h7 h6) as h8.
dependent rewrite -> h8.
assert (h9:Finite (Im (non_zero_el_prod_maps_of_set h1 S) el_prod)).
  eapply Finite_downward_closed. apply h7. auto with sets.
assert (h10:Finite (Im [f : Fin_map E signe mns | Ensembles.In S f /\ el_prod f = 0]
           el_prod)).
  eapply Finite_downward_closed. apply h7. auto with sets.
pose proof (plus_set_union'  _ _ h9 h10 h7) as h11.
unfold At, bt in h11. unfold At, bt. rewrite h11 at 1.
rewrite plus_set_zero_el_prod_maps_of_set.
rewrite zero_sum.
apply plus_set_functional.
reflexivity. assumption.
Qed.


Lemma plus_set_zero_el_prod_compose_maps_of_set :
  forall {B:Bool_Alg} {E:Ensemble (bt B)},
    Finite E ->
    forall (g:bt B->At) (S:Ensemble (Fin_map E signe mns))
         (pf:Finite (Im [f : Fin_map E signe mns | Ensembles.In S f /\ el_prod_compose g f = 0]
             (el_prod_compose g))),
    plus_set _ pf = 0.
intros B E h1 g S h2.
destruct (classic (Inhabited  (Im [f : Fin_map E signe mns | Ensembles.In S f /\ el_prod_compose g f = 0]
        (el_prod_compose g)))) as [h3 | h4].
destruct h3 as [f h3]. destruct h3 as [f h3]. subst.
destruct h3 as [h3].
rewrite <- (plus_set_sing 0).
apply plus_set_functional.
apply Extensionality_Ensembles.
red. split.
red. intros k h4. destruct h4 as [k h4]. subst.
destruct h4 as [h4]. destruct h4 as [h4 h5]. rewrite h5.
constructor.
red. intros x h4. destruct h4.
apply Im_intro with f. constructor. assumption.
destruct h3 as [h3 h4]. rewrite h4. reflexivity.
apply not_inhabited_empty in h4.
pose proof (Empty_is_finite At) as h5.
pose proof (subsetT_eq_compat _ _ _ _ h2 h5 h4) as h6.
dependent rewrite -> h6.
apply plus_set_empty'.
Qed.


Lemma plus_subset_el_prod_compose_maps_eq_same_non_zero : 
  forall {B:Bool_Alg} {E:Ensemble (bt B)} (pfe:Finite E)
         (g:bt B->At) (S:Ensemble (Fin_map E signe mns)),
    plus_set (Im S (el_prod_compose g)) (finite_image _ _ _ (el_prod_compose g) (finite_fin_map_ens S pfe signe_finite)) =
    plus_set (Im (non_zero_el_prod_compose_maps_of_set pfe g S)
                 (el_prod_compose g))
            (finite_image _ _ _ (el_prod_compose g) 
                           (finite_non_zero_el_prod_compose_maps_of_set pfe g S)).
intros B E h1 g S.
pose proof (non_zero_el_prod_compose_maps_of_set_decompose h1 g S) as h2.
pose proof (f_equal (fun X=>Im X (el_prod_compose g)) h2) as h3. simpl in h3.
assert (h4:Finite (Im
         (Union (non_zero_el_prod_compose_maps_of_set h1 g S)
            [f : Fin_map E signe mns | Ensembles.In S f /\ el_prod_compose g f = 0])
         (el_prod_compose g))).
  rewrite <- h3. apply finite_image. apply finite_fin_map_ens.
  assumption. apply signe_finite.
pose proof (subsetT_eq_compat _ _ _ _ (finite_image (Fin_map E signe mns) At S (el_prod_compose g)
        (finite_fin_map_ens S h1 signe_finite)) h4 h3) as h5.
dependent rewrite -> h5.
pose proof (im_union (non_zero_el_prod_compose_maps_of_set h1 g S)
           [f : Fin_map E signe mns | Ensembles.In S f /\ el_prod_compose g f = 0] (el_prod_compose g)) as h6.

assert (h7:Finite ( Union (Im (non_zero_el_prod_compose_maps_of_set h1 g S) (el_prod_compose g))
         (Im [f : Fin_map E signe mns | Ensembles.In S f /\ el_prod_compose g f = 0]
            (el_prod_compose g)))).
  rewrite <- h6. assumption.
assert (h9:Finite (Im (non_zero_el_prod_compose_maps_of_set h1 g S) (el_prod_compose g))).
  eapply Finite_downward_closed. apply h7. auto with sets.
assert (h10:Finite (Im [f : Fin_map E signe mns | Ensembles.In S f /\ el_prod_compose g f = 0]
           (el_prod_compose g))).
  eapply Finite_downward_closed. apply h7. auto with sets.
pose proof (subsetT_eq_compat _ _ _ _ h4 h7 h6) as h8.
dependent rewrite -> h8.
pose proof (plus_set_union'  _ _ h9 h10 h7) as h11.
unfold At, bt in h11. unfold At, bt. rewrite h11 at 1.
pose proof (plus_set_zero_el_prod_compose_maps_of_set h1 g _ h10) as h12.
unfold bt, At in h12. unfold bt, At. rewrite h12 at 1.
rewrite zero_sum.
apply plus_set_functional.
reflexivity.
Qed.



Lemma plus_subset_non_zero_el_prod_compose_maps_empty :
  forall {B:Bool_Alg} (E:Ensemble (bt B)) (pfe:Finite E)
         (g:(bt B)->At),
    plus_subset_non_zero_el_prod_compose_maps E pfe g _ (empty_inclusion _) = 0.
intros. unfold plus_subset_non_zero_el_prod_compose_maps.
generalize ((finite_image (Fin_map E signe mns) At (Empty_set (Fin_map E signe mns))
        (el_prod_compose g)
        (finite_fin_map_ens (Empty_set (Fin_map E signe mns)) pfe
                            signe_finite))).
pose proof (image_empty (Fin_map E signe mns) _ (el_prod_compose g)) as h1.
rewrite h1.
intro. apply plus_set_empty'.
Qed.

Lemma plus_subset_non_zero_el_prod_compose_maps_full :
  forall {B:Bool_Alg} (E:Ensemble (bt B)) (pfe:Finite E)
         (g:(bt B)->At),
    plus_subset_non_zero_el_prod_compose_maps E pfe g _ (inclusion_reflexive _) = 1.
intros B E h1 g.
unfold plus_subset_non_zero_el_prod_compose_maps.
unfold non_zero_el_prod_compose_maps.
pose proof (excl_middle_sat (fun a:(Fin_map E signe mns) => el_prod_compose g a = 0)) as h2.
pose proof (feq_im _ _ (el_prod_compose g) h2) as h3.
simpl in h3.
rewrite im_union in h3.
pose proof (finite_fin_maps _ _ mns h1 signe_finite) as h4.
pose proof (finite_image _ _ _ (el_prod_compose g) h4) as h5.
pose proof h5 as h6.
rewrite h3 in h6.
pose proof f_equal.
pose proof (plus_set_functional _ _ h5 h6 h3) as h7.
pose proof (el_prod_compose_covers _ h1 g) as h8.
assert (h9:h5 = (finite_image (Fin_map E signe mns) (Btype (Bc A))
            (Full_set (Fin_map E signe mns)) (el_prod_compose g)
            (finite_fin_maps E signe mns h1 signe_finite))).
  apply proof_irrelevance. subst.
rewrite h8 in h7 at 1. symmetry in h7.
assert (h9:Included (Im [x : Fin_map E signe mns | el_prod_compose g x = 0] (el_prod_compose g))  (Im (Full_set (Fin_map E signe mns)) (el_prod_compose g))).
rewrite h3. auto with sets.
assert (h10:Included (Im [x : Fin_map E signe mns | el_prod_compose g x <> 0] (el_prod_compose g))  (Im (Full_set (Fin_map E signe mns)) (el_prod_compose g))).
rewrite h3. auto with sets.
pose proof (Finite_downward_closed).
pose proof (Finite_downward_closed _ _ (finite_image (Fin_map E signe mns) (Btype (Bc A))
            (Full_set (Fin_map E signe mns)) (el_prod_compose g)
            (finite_fin_maps E signe mns h1 signe_finite)) _ h9) as h11.
pose proof (Finite_downward_closed _ _ (finite_image (Fin_map E signe mns) (Btype (Bc A))
            (Full_set (Fin_map E signe mns)) (el_prod_compose g)
            (finite_fin_maps E signe mns h1 signe_finite)) _ h10) as h12.
pose proof (plus_set_union _ _ h11 h12) as h13.
assert (h14:(Union_preserves_Finite (Btype (Bc A))
             (Im [x : Fin_map E signe mns | el_prod_compose g x = 0] (el_prod_compose g))
             (Im [x : Fin_map E signe mns | el_prod_compose g x <> 0] (el_prod_compose g)) h11 h12) = h6). apply proof_irrelevance.
rewrite h14 in h13 at 1.
unfold At in h7. unfold At in h13.
rewrite h7 in h13 at 1.
pose proof (plus_set_el_prod_compose_zero g _ h11) as h15.
unfold At in h15. unfold At in h13. rewrite h15 in h13.
rewrite comm_sum in h13.
rewrite zero_sum in h13.
rewrite h13.
apply plus_set_functional. reflexivity.
Qed.



Lemma plus_subset_non_zero_el_prod_compose_maps_int : 
  forall {B:Bool_Alg} (E:Ensemble (bt B)) (pfe:Finite E)
         (g:(bt B)->At) (X Y:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_compose_maps g E))
         (pfy:Included Y (non_zero_el_prod_compose_maps g E)),
    plus_subset_non_zero_el_prod_compose_maps E pfe g (Intersection X Y)
    (inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_int pfe g X Y pfx pfy) =
    plus_subset_non_zero_el_prod_compose_maps E pfe g X pfx *
    plus_subset_non_zero_el_prod_compose_maps E pfe g Y pfy.
intros B E h1 g X Y h2 h3.
unfold plus_subset_non_zero_el_prod_compose_maps.
rewrite dist_set_plus2.
pose proof (cart_prod_im X Y (el_prod_compose g) (el_prod_compose g)) as h4.
pose proof (excl_middle_sat' (cart_prod X Y) (fun pr => fst pr = snd pr)) as h5.
rewrite h5 in h4. clear h5.
rewrite im_union in h4.
pose proof (feq_im _ _ (fun p : Btype (Bc A) * Btype (Bc A) => fst p * snd p) h4) as h5.
rewrite im_union in h5.
do 2 rewrite im_im in h5. simpl in h5.
pose proof (plus_set_functional).
pose proof (finite_fin_map_ens X h1 signe_finite) as h6.
pose proof (finite_fin_map_ens Y h1 signe_finite) as h7.
pose proof (finite_image _ _ _ (el_prod_compose g) h6) as h8.
pose proof (finite_image _ _ _ (el_prod_compose g) h7) as h9.
pose proof (cart_prod_fin _ _ h8 h9) as h10.
pose proof (finite_image).
pose proof (finite_image _ _ _  (fun p : Btype (Bc A) * Btype (Bc A) => fst p * snd p) h10) as h11.
rewrite h5 in h11.
pose proof (plus_set_functional _ _ (finite_image (Btype (Bc A) * Btype (Bc A)) (Btype (Bc A))
        (cart_prod (Im X (el_prod_compose g)) (Im Y (el_prod_compose g)))
        (fun p : Btype (Bc A) * Btype (Bc A) => fst p * snd p)
        (cart_prod_fin (Im X (el_prod_compose g)) (Im Y (el_prod_compose g))
           (finite_image (Fin_map E signe mns) At X (el_prod_compose g)
                         (finite_fin_map_ens X h1 signe_finite))
                
           (finite_image (Fin_map E signe mns) At Y (el_prod_compose g)
                         (finite_fin_map_ens Y h1 signe_finite)))) h11 h5) as h12.
assert (h13:Included (Im
                [x : Fin_map E signe mns * Fin_map E signe mns
                | Ensembles.In (cart_prod X Y) x /\ fst x = snd x]
                (fun x : Fin_map E signe mns * Fin_map E signe mns =>
                 (el_prod_compose g) (fst x) * (el_prod_compose g) (snd x)))
  (Union
             (Im
                [x : Fin_map E signe mns * Fin_map E signe mns
                | Ensembles.In (cart_prod X Y) x /\ fst x = snd x]
                (fun x : Fin_map E signe mns * Fin_map E signe mns =>
                 (el_prod_compose g) (fst x) * (el_prod_compose g) (snd x)))
             (Im
                [x : Fin_map E signe mns * Fin_map E signe mns
                | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
                (fun x : Fin_map E signe mns * Fin_map E signe mns =>
                 (el_prod_compose g) (fst x) * (el_prod_compose g) (snd x))))). auto with sets.
assert (h14:Included (Im
                [x : Fin_map E signe mns * Fin_map E signe mns
                | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
                (fun x : Fin_map E signe mns * Fin_map E signe mns =>
                 (el_prod_compose g) (fst x) * (el_prod_compose g) (snd x)))
  (Union
             (Im
                [x : Fin_map E signe mns * Fin_map E signe mns
                | Ensembles.In (cart_prod X Y) x /\ fst x = snd x]
                (fun x : Fin_map E signe mns * Fin_map E signe mns =>
                 (el_prod_compose g) (fst x) * (el_prod_compose g) (snd x)))
             (Im
                [x : Fin_map E signe mns * Fin_map E signe mns
                | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
                (fun x : Fin_map E signe mns * Fin_map E signe mns =>
                 (el_prod_compose g) (fst x) * (el_prod_compose g) (snd x))))). auto with sets.
pose proof (Finite_downward_closed _ _ h11 _ h13) as h15. clear h13.
pose proof (Finite_downward_closed _ _ h11 _ h14) as h16. clear h14.
rewrite (plus_set_union' _ _ h15 h16 h11) in h12 at 1.
pose proof (finite_fin_map_squared_ens [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x] h1 signe_finite) as h13.
rewrite (plus_set_el_prod_compose_disjoint' _ g X Y h13 h16) in h12.
rewrite zero_sum in h12.
unfold At. unfold At in h12.
rewrite h12 at 1.
apply plus_set_functional.   
symmetry.
apply Extensionality_Ensembles.
red. split.
red.
intros x h17.
inversion h17 as [a h18 b h19 h20]. subst.
destruct h18 as [h18].
destruct h18 as [h18l h18r].
destruct h18l as [h18l].
destruct h18l as [h18a h18b].
rewrite h18r.
rewrite idem_prod.
rewrite h18r in h18a.
apply Im_intro with (snd a).
split; auto. reflexivity.
red.
intros x h17.
inversion h17 as [a h18 b h19 h20]. subst.
destruct h18 as [a h18l h18r].
apply Im_intro with (a, a).
constructor.
simpl.
split; auto. split; auto.
simpl. rewrite idem_prod. reflexivity.
Qed.


Lemma plus_subset_non_zero_el_prod_compose_maps_union : 
  forall {B:Bool_Alg} (E:Ensemble (bt B)) (pfe:Finite E)
         (g:(bt B)->At) (X Y:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_compose_maps g E))
         (pfy:Included Y (non_zero_el_prod_compose_maps g E)),
    plus_subset_non_zero_el_prod_compose_maps E pfe g (Union X Y)
    (inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_union pfe g X Y pfx pfy) =
    plus_subset_non_zero_el_prod_compose_maps E pfe g X pfx +
    plus_subset_non_zero_el_prod_compose_maps E pfe g Y pfy.
intros B E h1 g X Y hx hy.
unfold plus_subset_non_zero_el_prod_compose_maps.
pose proof (im_union X Y (el_prod_compose g)) as h2.
assert (h3:Finite (Im X (el_prod_compose g))).
  apply finite_image. apply finite_fin_map_ens; auto. apply signe_finite.
assert (h4:Finite (Im Y (el_prod_compose g))).
  apply finite_image. apply finite_fin_map_ens; auto. apply signe_finite.
pose proof (subsetT_eq_compat _ _ _ _ (finite_image (Fin_map E signe mns) At (Union X Y) 
        (el_prod_compose g) (finite_fin_map_ens (Union X Y) h1 signe_finite)) (Union_preserves_Finite _ _ _ h3 h4) h2) as h5.
dependent rewrite -> h5.
rewrite plus_set_union.
f_equal. apply plus_set_functional; auto. apply plus_set_functional; auto.
Qed.




Lemma plus_subset_non_zero_el_prod_compose_maps_comp : 
  forall {B:Bool_Alg} (E:Ensemble (bt B)) (pfe:Finite E)
         (g:(bt B)->At) (X:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_compose_maps g E)),
         plus_subset_non_zero_el_prod_compose_maps E pfe g (Setminus (non_zero_el_prod_compose_maps g E) X)
         (inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_comp pfe g X pfx) =
    - plus_subset_non_zero_el_prod_compose_maps E pfe g X pfx.
intros B E h1 g X h2.
assert (h3:(plus_subset_non_zero_el_prod_compose_maps E h1 g X h2) *
   plus_subset_non_zero_el_prod_compose_maps E h1 g
     (Setminus (non_zero_el_prod_compose_maps g E) X)
     (inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_comp h1 g X h2) = 0).
  rewrite <- plus_subset_non_zero_el_prod_compose_maps_int.
  rewrite <- (plus_subset_non_zero_el_prod_compose_maps_empty E h1 g).
  unfold plus_subset_non_zero_el_prod_compose_maps.
  apply plus_set_functional.
  apply feq_im.
  apply int_setminus. 
assert (h4:(plus_subset_non_zero_el_prod_compose_maps E h1 g X h2) +
   plus_subset_non_zero_el_prod_compose_maps E h1 g
     (Setminus (non_zero_el_prod_compose_maps g E) X)
     (inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_comp h1 g X h2) = 1).
  rewrite <- plus_subset_non_zero_el_prod_compose_maps_union.
  rewrite <- (plus_subset_non_zero_el_prod_compose_maps_full E h1 g).
  unfold plus_subset_non_zero_el_prod_maps.
  apply plus_set_functional. apply feq_im.
  apply union_setminus_incl. assumption.
pose proof (comp_char _ _ h3 h4) as h5.
assumption.
Qed.


Lemma closed_all_plus_subsets_non_zero_el_prod_maps : 
  forall (E:Ensemble At) (pfe:Finite E),
    alg_closed [a:At | exists (S:Ensemble (Fin_map E signe mns)) 
                            (pfi:Included S (non_zero_el_prod_maps E)),
                     a = plus_subset_non_zero_el_prod_maps E pfe S pfi].
intros E h1. 
constructor.
(* + *)
red. 
intros x y.
constructor. 
destruct x as [x h2].
destruct y as [y h3]. 
destruct h2 as [h2]. destruct h3 as [h3]. simpl.
destruct h2 as [X h2]. destruct h3 as [Y h3].
destruct h2 as [h2a h2b]. destruct h3 as [h3a h3b].
exists (Union X Y).
exists (inclusion_plus_subset_non_zero_el_prod_maps_preserves_union _ h1 _ _ h2a h3a).
unfold Bplus_sub. simpl.
rewrite h2b. rewrite h3b.
apply plus_subset_non_zero_el_prod_maps_union.
(* * *)
red. intros x y. constructor.
destruct x as [x h2].
destruct y as [y h3]. 
destruct h2 as [h2]. destruct h3 as [h3]. simpl.
destruct h2 as [X h2]. destruct h3 as [Y h3].
destruct h2 as [h2a h2b]. destruct h3 as [h3a h3b].
exists (Intersection X Y).
exists (inclusion_plus_subset_non_zero_el_prod_maps_preserves_int _ h1 _ _ h2a h3a).
unfold Btimes_sub. simpl.
rewrite h2b. rewrite h3b.
apply plus_subset_non_zero_el_prod_maps_int.
(* 1 *)
red. constructor.
exists (non_zero_el_prod_maps E). exists (inclusion_reflexive _).
symmetry.
apply plus_subset_non_zero_el_prod_maps_full.
(* 0 *)
red. constructor.
exists (Empty_set _). exists (empty_inclusion _).
symmetry.
apply plus_subset_non_zero_el_prod_maps_empty.
(* - *)
red. intro x. constructor.
destruct x as [x h2].
destruct h2 as [h2]. simpl.
destruct h2 as [X h2]. 
destruct h2 as [h2a h2b]. 
exists (Setminus (non_zero_el_prod_maps E) X).
exists (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp _ h1 X h2a).
unfold Bcomp_sub. simpl. 
rewrite h2b.
apply plus_subset_non_zero_el_prod_maps_comp.
Qed.

Lemma el_prod_sing_inc : 
  forall (E:Ensemble At) (pf:Finite E) (a:Fin_map E signe mns),
    el_prod a <> 0 -> Included (Singleton a) (non_zero_el_prod_maps E). 
intros E h1 a h2.
red.
intros a' h3.
destruct h3; subst.
unfold non_zero_el_prod_maps. constructor.
assumption.
Qed.

Lemma el_prod_sing : forall
  (E:Ensemble At) (pf:Finite E) (a:Fin_map E signe mns)
  (pfel:el_prod a <> 0), el_prod a = plus_subset_non_zero_el_prod_maps _ pf (Singleton a) (el_prod_sing_inc _ pf _ pfel).
intros E h1 a h2.
unfold plus_subset_non_zero_el_prod_maps.
assert (h3:Im (Singleton a) el_prod = (Singleton (el_prod a))).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x' h13.
  inversion h13. subst. destruct H. subst. constructor.
  red. intros x' h13. destruct h13.
  apply Im_intro with a. constructor.
  reflexivity.
generalize dependent (
     (finite_image (Fin_map E signe mns) At (Singleton a) el_prod
        (Finite_downward_closed (Fin_map E signe mns)
           (non_zero_el_prod_maps E) (non_zero_el_prod_maps_fin E h1)
           (Singleton a) (el_prod_sing_inc E h1 a h2)))).
rewrite h3.
intro h4.
rewrite plus_set_sing'.
reflexivity.
Qed.


Lemma atom_non_zero_el_prod : 
  forall (E:Ensemble At) (pf:Finite E), 
    let C := (Subalg _
(closed_all_plus_subsets_non_zero_el_prod_maps E pf))
  in forall (atm:Btype (Bc C)), atom atm <-> 
                              exists a:Fin_map E signe mns,
                                Ensembles.In (non_zero_el_prod_maps E) a /\ 
                                proj1_sig atm = el_prod a.
intros E h1 C atm.
split.
intros h2.
pose proof h2 as h3. 
rewrite atom_iff in h3.
pose proof (proj2_sig atm) as h4.
destruct h4 as [h4].
destruct h4 as [X h4]. destruct h4 as [h4 h5].
assert (h6: forall Y:Ensemble (Fin_map E signe mns), 
              Included Y (non_zero_el_prod_maps E) ->
                          Intersection Y X = X \/
                          Intersection Y X = Empty_set _).
  intros Y h7.
  assert (h8:Ensembles.In 
                [a : At
         | exists
             (S : Ensemble (Fin_map E signe mns)) (pfi : 
                                                  Included S
                                                  (non_zero_el_prod_maps E)),
             a = plus_subset_non_zero_el_prod_maps E h1 S pfi]
(plus_subset_non_zero_el_prod_maps E h1 _ h7)).
  constructor. exists Y, h7. reflexivity.
  pose (exist _ _ h8) as pY.
  red in h2.
  destruct h2 as [h2l h2r].
  specialize (h3 pY).
  destruct h3 as [h3l h3r].  
   destruct h3l as [h3a | h3b].
 (* h3a *)

  red in h3a. rewrite eq_ord in h3a. 
  simpl in h3a. unfold Btimes_sub in h3a. rewrite unfold_sig in h3a.
  pose proof (exist_injective _ _ _ _ _ h3a) as h6.
  clear h3a. unfold pY in h6. rewrite h5 in h6.
  simpl in h6.
  rewrite plus_subset_non_zero_el_prod_maps_int in h6.
  left.
  assert (h9:Included (Intersection X Y) X). auto with sets.
  rewrite comm_prod_psa.
  apply NNPP.
  intro h10.
  assert (h11:Strict_Included (Intersection X Y) X).  auto with sets.
  pose proof (Strict_Included_inv _ _ _ h11) as h12.
  destruct h12 as [h12l h12r]. clear h12l.
  rewrite setminus_same_int in h12r.  
  pose proof (Inhabited_not_empty _ _ h12r) as h13.
  pose proof (plus_subset_non_zero_el_prod_maps_empty_rev).

  assert (h14:Included (Setminus X Y) (non_zero_el_prod_maps E)).
    red. intros x h13'. destruct h13'. auto with sets.
 
  pose proof (plus_subset_non_zero_el_prod_maps_empty_rev _ h1 _ h14) as h15.
  assert (h16:plus_subset_non_zero_el_prod_maps E h1 (Setminus X Y) h14 <> 0). tauto.
  assert (h17:Setminus X Y = Intersection X (Setminus (non_zero_el_prod_maps E) Y)).
    apply Extensionality_Ensembles.
    red. split.
    red. 
    intros b h17.
    destruct h17 as [h17 h18].
    constructor; auto with sets.
    red. intros b h17.
    destruct h17 as [b h17 h18].
    constructor; auto with sets.
    destruct h18. assumption. 
  pose proof (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp _ h1 _ h7) as h18.
  assert (h19:plus_subset_non_zero_el_prod_maps E h1 (Setminus X Y) h14 = plus_subset_non_zero_el_prod_maps E h1 _ (inclusion_plus_subset_non_zero_el_prod_maps_preserves_int _ h1 _ _ h4 (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp _ h1 _ h7))).
  unfold plus_subset_non_zero_el_prod_maps.
  apply plus_set_functional. apply feq_im. assumption.
  rewrite h19 in h16.
  rewrite <- plus_subset_non_zero_el_prod_maps_int in h16.
  
  rewrite <- plus_subset_non_zero_el_prod_maps_comp in h16.
  unfold At in h16. rewrite <- le_iff in h16.
  unfold le in h16. rewrite eq_ord in h16.
  rewrite <- plus_subset_non_zero_el_prod_maps_int in h6.
  contradiction.
  (* h3b *)
  right.
  do 2 rewrite unfold_sig in h3b. unfold pY in h3b. simpl in h3b.
  pose proof (exist_injective _ _ _  _ _ h3b) as h6.
  unfold Btimes_sub in h6.
  simpl in h6.
  rewrite h5 in h6.
  rewrite plus_subset_non_zero_el_prod_maps_int in h6.
  rewrite comm_prod_psa.
  apply (plus_subset_non_zero_el_prod_maps_empty_rev _ h1 _ 
         (inclusion_plus_subset_non_zero_el_prod_maps_preserves_int _ h1 _ _ h4 h7)). assumption.
(* new goal *)
assert (h9:X <> Empty_set _).
  intro h10.
  generalize dependent h4.
  rewrite h10. intro h4.
  assert (h11:h4 = (empty_inclusion (non_zero_el_prod_maps E))).
    apply proof_irrelevance.
  rewrite h11.
  rewrite plus_subset_non_zero_el_prod_maps_empty.
  intro h12.
  red in h2. destruct h2 as [h2l h2r].  
  contradict h2l.
  rewrite (unfold_sig _ atm).
  rewrite (unfold_sig _ (Bzero (Bc C))).
  apply proj1_sig_injective.
  simpl. assumption.
pose proof (non_zero_el_prod_maps_fin _ h1) as h10.

pose proof (atom_sing _ _ h10 h4 h9) as h11.
rewrite h11 in h6.
destruct h6 as [a h6].
exists a.
destruct h6 as [h6l h6r].
generalize dependent h4.
rewrite h6r.
intros h4 h5. 
unfold plus_subset_non_zero_el_prod_maps in h5.
assert (h12:Im (Singleton a) (el_prod) = Singleton (el_prod a)).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x' h13.
  inversion h13. subst. destruct H. subst. constructor.
  red. intros x' h13. destruct h13.
  apply Im_intro with a. constructor.
  reflexivity.
generalize dependent  (finite_image (Fin_map E signe mns) At (Singleton a) el_prod
            (Finite_downward_closed (Fin_map E signe mns)
               (non_zero_el_prod_maps E) (non_zero_el_prod_maps_fin E h1)
               (Singleton a) h4)).
rewrite h12. intros h13 h14.
rewrite plus_set_sing' in h14.
split; auto.
intro h2.
destruct h2 as [a h2]. 
pose proof h2 as h2'.
rewrite atom_iff.
intro b. split.
pose proof (proj2_sig b) as h3. simpl in h3.
destruct h3 as [h3].
destruct h3 as [S h3].
destruct h3 as [h3 h4].
pose proof (proj2_sig atm) as h5. simpl in h5.
destruct h5 as [h5]. destruct h5 as [A' h5]. destruct h5 as [h5 h6].
destruct (classic (A' = Empty_set _)) as [h7 | h8].
generalize dependent h5.
rewrite h7. intros h5 h6.
assert (h8:h5 = (empty_inclusion (non_zero_el_prod_maps E))).
  apply proof_irrelevance.
rewrite h8 in h6.
rewrite plus_subset_non_zero_el_prod_maps_empty in h6.
right.
rewrite (unfold_sig _ atm). rewrite (unfold_sig _ b).
rewrite (unfold_sig _ (Bzero (Bc C))).
simpl.
apply proj1_sig_injective. simpl. unfold Btimes_sub. simpl.
rewrite h6. rewrite comm_prod. apply zero_prod.
assert (h9:plus_subset_non_zero_el_prod_maps E h1 A' h5 <> 0).
  intro h10.
  pose proof plus_subset_non_zero_el_prod_maps_empty_rev.
  pose proof (plus_subset_non_zero_el_prod_maps_empty_rev _ h1 _ h5 h10) as h11.
contradiction.
rewrite <- h6 in h9.
destruct h2 as [h2'' h2].
rewrite h2 in h9.
pose proof el_prod_sing.
pose proof (el_prod_sing _ h1 a h9) as h10.
rewrite h10 in h2.
pose proof (@plus_subset_non_zero_el_prod_maps_int).
pose proof 
     (plus_subset_non_zero_el_prod_maps_int 
        _ h1 _ _ 
        (el_prod_sing_inc E h1 a h9) h3) as h11.
rewrite <- h2 in h11. rewrite <- h4 in h11.
pose proof (singleton_inc_int a S) as h12.
destruct h12 as [h12l | h12r].
generalize dependent (inclusion_plus_subset_non_zero_el_prod_maps_preserves_int E h1
             (Singleton a) S (el_prod_sing_inc E h1 a h9) h3).
rewrite comm_prod_psa in h12l.
rewrite h12l.
intros h13 h14.
assert (h15:h13 = (empty_inclusion (non_zero_el_prod_maps E))).
  apply proof_irrelevance.
rewrite h15 in h14.
rewrite plus_subset_non_zero_el_prod_maps_empty in h14.
right.
rewrite (unfold_sig _ atm). rewrite (unfold_sig _ b).  rewrite (unfold_sig _ (Bzero (Bc C))).
simpl.
apply proj1_sig_injective. simpl. unfold Btimes_sub. simpl.
assumption. 
left.
red. rewrite eq_ord.
generalize dependent (inclusion_plus_subset_non_zero_el_prod_maps_preserves_int E h1
             (Singleton a) S (el_prod_sing_inc E h1 a h9) h3).
rewrite comm_prod_psa in h12r.
rewrite h12r. intros h13 h14.
assert (h15:h13 =  (el_prod_sing_inc E h1 a h9)). apply proof_irrelevance.
rewrite h15 in h14.
rewrite <- h2 in h14.
rewrite (unfold_sig _ atm). rewrite (unfold_sig _ b). simpl.
unfold Btimes_sub. simpl. apply proj1_sig_injective. simpl.
assumption.
(* new goal*)  
unfold le. rewrite eq_ord.
intro h3.
pose proof (proj2_sig b) as h4.
destruct h3 as [h3l h3r]. pose proof h3l as h3l'.
rewrite h3r in h3l.
destruct h2 as [h2l h2r].
rewrite (unfold_sig _ atm) in h3l.
rewrite (unfold_sig _ (Bzero (Bc C))) in h3l.
pose proof (exist_injective _ _ _ _ _ h3l) as h8.
simpl in h8.
rewrite <- h8 in h2r.
unfold non_zero_el_prod_maps in h2l.
destruct h2l as [h2l].
symmetry in h2r.
contradiction.
Qed.
  
Lemma plus_subset_non_zero_el_prod_maps_inj :
  forall (E:Ensemble At) (pfe:Finite E) 
         (S T:Ensemble (Fin_map E signe mns))
    (pfs:Included S (non_zero_el_prod_maps E))
    (pft:Included T (non_zero_el_prod_maps E)),
    plus_subset_non_zero_el_prod_maps E pfe S pfs =
    plus_subset_non_zero_el_prod_maps E pfe T pft ->
    S = T.
intros E h1 S T h2 h3 h4.
pose proof (plus_subset_non_zero_el_prod_maps_empty _ h1) as h5.
pose proof (symdiff_full_ref (non_zero_el_prod_maps E) S) as h6.
generalize dependent  (empty_inclusion (non_zero_el_prod_maps E)).
rewrite <- h6.
pose proof (plus_subset_non_zero_el_prod_maps_symdiff_full _ h1 _ _ h2 h2) as h7.
rewrite h4 in h7 at 2.
generalize dependent (inclusion_plus_subset_non_zero_el_prod_maps_preserves_symdiff_full
            E h1 S S h2 h2).
intros h7 h8 h9 h10.
assert (h11: h7 = h9).  apply proof_irrelevance. subst.
rewrite h10 in h8.
rewrite plus_subset_non_zero_el_prod_maps_symdiff_full in h8.
pose proof (inclusion_plus_subset_non_zero_el_prod_maps_preserves_symdiff_full E h1 S T h2 h3) as h11.
pose proof (plus_subset_non_zero_el_prod_maps_empty_rev _ h1 (Symdiff_full (non_zero_el_prod_maps E) S T) h11) as h12.
generalize dependent  (inclusion_plus_subset_non_zero_el_prod_maps_preserves_symdiff_full
            E h1 S T h2 h3). intro h13.
assert (h14:h11 = h13). apply proof_irrelevance. subst.
intro h14. specialize (h12 h14).
rewrite symdiff_full_empty_iff_eq in h12; auto.
Qed.


Lemma el_prod_eq_plus_subset_non_zero_el_prod_maps_extends : 
  forall (E F:Ensemble At) (pfe:Finite E),
    Included F E -> 
    forall (b:Fin_map F signe mns),
      let L := [a : Fin_map E signe mns
               | Ensembles.In (non_zero_el_prod_maps E) a /\ extends a b] in
      exists pf:Included L (non_zero_el_prod_maps E),
      el_prod b = plus_subset_non_zero_el_prod_maps E pfe L pf.
intros E F h1 h2 b L.
assert (h3: Included L (non_zero_el_prod_maps E)).
  red. intros x h3. destruct h3 as [h3]. destruct h3; assumption.
exists h3. 
assert (h4:forall a:Fin_map E signe mns, 
             Ensembles.In L a -> 
             Included (Im F (fun i : At => eps i (b |-> i)))
                      (Im E (fun i : At => eps i (a |-> i)))).
  intros a h5. red. intros p h6.
  destruct h5 as [h5]. destruct h5 as [h5l h5r].
  red in h5r. destruct h5r as [h5a h5b].
  destruct h6 as [f h6 p]. subst.
  rewrite h5b; auto.
  apply Im_intro with f. auto with sets.
  reflexivity.
assert (h5:forall a:Fin_map E signe mns, 
             Ensembles.In L a -> le (el_prod a) (el_prod b)). 
  intros a h6.
 unfold el_prod.
  apply times_set_inc_le. apply h4; auto.
assert (h6:forall a:Fin_map E signe mns,
             Ensembles.In (Setminus (non_zero_el_prod_maps E) L) a ->
             (el_prod a) * (el_prod b) = 0).
  intros a h7.
  destruct h7 as [h7 h8].
  unfold L in h8.
  assert (h9:~extends a b).
    intro h10.
    assert (h11:Ensembles.In  [a0 : Fin_map E signe mns
         | Ensembles.In (non_zero_el_prod_maps E) a0 /\ extends a0 b] a).
      constructor. split; assumption.
      contradiction.
  pose proof (not_extends_differs_at_point _ _ h2 h9) as h10.
  destruct h10 as [i h10].
  destruct h10 as [h10l h10r].
  pose proof (el_prod_le_ai b _ h10l) as h11.
  assert (h12:Ensembles.In E i). auto with sets.
  pose proof (el_prod_le_ai a _ h12) as h13.
  pose proof (mono_prod _ _ _ _ h11 h13) as h14.
  destruct (b |-> i); destruct (a |-> i). 
  contradict h10r. reflexivity.
  simpl in h14. rewrite comp_prod in h14. apply le_x_0. 
  rewrite comm_prod. assumption.
  simpl in h14. rewrite (comm_prod _ (-i) i) in h14. 
  rewrite comp_prod in h14. apply le_x_0.  
  rewrite comm_prod. assumption. contradict h10r. reflexivity.
assert (h7:el_prod b = (el_prod b) * 1). rewrite one_prod. reflexivity.
rewrite <- (plus_subset_non_zero_el_prod_maps_full _ h1) in h7.
pose proof (decompose_setminus_inc _ _ h3) as h8.
assert (h9: plus_subset_non_zero_el_prod_maps E h1 (non_zero_el_prod_maps E)
         (inclusion_reflexive (non_zero_el_prod_maps E)) = 
            plus_subset_non_zero_el_prod_maps E h1 (Union L (Setminus (non_zero_el_prod_maps E) L)) (inclusion_plus_subset_non_zero_el_prod_maps_preserves_union _ h1 _ _ h3 (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp _ h1 _ h3))).
  unfold plus_subset_non_zero_el_prod_maps.
  apply plus_set_functional.
  apply feq_im. assumption.
rewrite h9 in h7.
unfold plus_subset_non_zero_el_prod_maps in h7. 
rewrite dist_set_plus1 in h7.
generalize  dependent (finite_image (Btype (Bc A)) (Btype (Bc A))
            (Im (Union L (Setminus (non_zero_el_prod_maps E) L)) el_prod)
            (fun y : Btype (Bc A) => el_prod b * y)
            (finite_image (Fin_map E signe mns) At
               (Union L (Setminus (non_zero_el_prod_maps E) L)) el_prod
               (Finite_downward_closed (Fin_map E signe mns)
                  (non_zero_el_prod_maps E) (non_zero_el_prod_maps_fin E h1)
                  (Union L (Setminus (non_zero_el_prod_maps E) L))
                  (inclusion_plus_subset_non_zero_el_prod_maps_preserves_union
                     E h1 L (Setminus (non_zero_el_prod_maps E) L) h3
                     (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp
                        E h1 L h3))))).
  
rewrite im_im.  
rewrite im_union.   
intros h10. 
assert (h11:Included (Im L (fun x:Fin_map E signe mns => el_prod b * el_prod x))  (Union (Im L  (fun x:Fin_map E signe mns => el_prod b * el_prod x)) (Im (Setminus (non_zero_el_prod_maps E) L)  (fun x:Fin_map E signe mns => el_prod b * el_prod x)))).
  auto with sets.
assert (h12:Included  (Im (Setminus (non_zero_el_prod_maps E) L)  (fun x:Fin_map E signe mns => el_prod b * el_prod x))  (Union (Im L  (fun x:Fin_map E signe mns => el_prod b * el_prod x))
             (Im (Setminus (non_zero_el_prod_maps E) L)  (fun x:Fin_map E signe mns => el_prod b * el_prod x)))).
  auto with sets.
assert (h13:Finite (Im L  (fun x:Fin_map E signe mns => el_prod b * el_prod x))).  eapply Finite_downward_closed. apply h10. apply h11.
assert (h14:Finite (Im (Setminus (non_zero_el_prod_maps E) L)  (fun x:Fin_map E signe mns => el_prod b * el_prod x))). eapply Finite_downward_closed. apply h10. apply h12.  
assert (h16:plus_set (Im (Setminus (non_zero_el_prod_maps E) L)  (fun x:Fin_map E signe mns => el_prod b * el_prod x)) h14 = 0).
  destruct (classic (Inhabited (Setminus (non_zero_el_prod_maps E) L))) as [h17 | h18].
    rewrite <- (plus_set_sing 0).
    apply plus_set_functional.
    apply Extensionality_Ensembles.
    red. split.
    red.
    intros x h18.
    destruct h18 as [x h18]. subst.
    rewrite comm_prod.
    rewrite h6; auto. constructor.
    red.
    intros z h18.
    destruct h18; subst.
    destruct h17 as [a h17].
    apply Im_intro with a; auto.
    rewrite comm_prod. rewrite h6; auto.
    pose proof (not_inhabited_empty _ h18) as h19.
    pose proof (image_empty _ _ (fun x : Fin_map E signe mns => el_prod b * el_prod x)) as h20.
    rewrite <- h19 in h20.
    rewrite <- plus_set_empty.
    apply plus_set_functional.
    assumption.
unfold plus_subset_non_zero_el_prod_maps.
rewrite h7.
pose proof (im_im (Union L (Setminus (non_zero_el_prod_maps E) L)) el_prod (fun y:bt A => el_prod b * y)) as h17.
simpl in h17.
pose proof (@im_union).
pose proof (im_union L (Setminus (non_zero_el_prod_maps E) L)   (fun x : Fin_map E signe mns => el_prod b * el_prod x)) as hiu.
rewrite hiu in h17. 
assert (heq:  plus_set
   (Im (Im (Union L (Setminus (non_zero_el_prod_maps E) L)) el_prod)
      (fun y : bt A => el_prod b * y))
   (finite_image (bt A) (Btype (Bc A))
      (Im (Union L (Setminus (non_zero_el_prod_maps E) L)) el_prod)
      (fun y : bt A => el_prod b * y)
      (finite_image (Fin_map E signe mns) At
         (Union L (Setminus (non_zero_el_prod_maps E) L)) el_prod
         (Finite_downward_closed (Fin_map E signe mns)
            (non_zero_el_prod_maps E) (non_zero_el_prod_maps_fin E h1)
            (Union L (Setminus (non_zero_el_prod_maps E) L))
            (inclusion_plus_subset_non_zero_el_prod_maps_preserves_union E h1
               L (Setminus (non_zero_el_prod_maps E) L) h3
               (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp E
                  h1 L h3))))) = 
              plus_set  (Union (Im L (fun x : Fin_map E signe mns => el_prod b * el_prod x))
          (Im (Setminus (non_zero_el_prod_maps E) L)
             (fun x : Fin_map E signe mns => el_prod b * el_prod x))) h10).
  apply plus_set_functional. assumption.
unfold At, bt in heq. unfold At, bt. simpl in heq. simpl. rewrite heq at 1.
pose proof (plus_set_union (Im L (fun x : Fin_map E signe mns => el_prod b * el_prod x))
        (Im (Setminus (non_zero_el_prod_maps E) L)
           (fun x : Fin_map E signe mns => el_prod b * el_prod x)) h13 h14) as hpsu.
assert (hpi :  (Union_preserves_Finite (bt A)
              (Im L (fun x : Fin_map E signe mns => el_prod b * el_prod x))
              (Im (Setminus (non_zero_el_prod_maps E) L)
                 (fun x : Fin_map E signe mns => el_prod b * el_prod x)) h13
              h14) = h10). apply proof_irrelevance.
subst.
unfold bt, At in hpsu. unfold bt, At. simpl. simpl in hpsu. rewrite hpsu at 1.
unfold bt, At in h16. rewrite h16 at 1.
rewrite zero_sum.
apply plus_set_functional.
apply im_ext_in. 

intros x h17'.
specialize (h5 x h17'). unfold le in h5. rewrite eq_ord in h5. 
rewrite <- h5 at 2.
rewrite comm_prod.
reflexivity.
Qed.





Lemma el_prod_compose_eq_plus_subset_non_zero_el_prod_compose_maps_extends : 
  forall {B:Bool_Alg} (E F:Ensemble (bt B)) (pfe:Finite E),
    Included F E -> 
    forall (g:bt B->At) (b:Fin_map F signe mns),
      let L := [a : Fin_map E signe mns
               | Ensembles.In (non_zero_el_prod_compose_maps g E) a /\ extends a b] in
      exists pf:Included L (non_zero_el_prod_compose_maps g E),
      el_prod_compose g b = plus_subset_non_zero_el_prod_compose_maps E pfe g L pf.
intros B E F h1 h2 g b L.
assert (h3: Included L (non_zero_el_prod_compose_maps g E)).
  red. intros x h3. destruct h3 as [h3]. destruct h3; assumption.
exists h3. 
assert (h4:forall a:Fin_map E signe mns, 
             Ensembles.In L a -> 
             Included (Im F (fun i : bt B => eps (g i) (b |-> i)))
                      (Im E (fun i : bt B => eps (g i) (a |-> i)))).
  intros a h5. red. intros p h6.
  destruct h5 as [h5]. destruct h5 as [h5l h5r].
  red in h5r. destruct h5r as [h5a h5b].
  destruct h6 as [f h6 p]. subst.
  rewrite h5b; auto.
  apply Im_intro with f. auto with sets.
  reflexivity.
assert (h5:forall a:Fin_map E signe mns, 
             Ensembles.In L a -> le (el_prod_compose g a) (el_prod_compose g b)). 
  intros a h6.
 unfold el_prod_compose.
  apply times_set_inc_le. apply h4; auto.
assert (h6:forall a:Fin_map E signe mns,
             Ensembles.In (Setminus (non_zero_el_prod_compose_maps g E) L) a ->
             (el_prod_compose g a) * (el_prod_compose g b) = 0).
  intros a h7.
  destruct h7 as [h7 h8].
  unfold L in h8.
  assert (h9:~extends a b).
    intro h10.
    assert (h11:Ensembles.In  [a0 : Fin_map E signe mns
         | Ensembles.In (non_zero_el_prod_compose_maps g E) a0 /\ extends a0 b] a).
      constructor. split; assumption.
      contradiction.
  pose proof (not_extends_differs_at_point _ _ h2 h9) as h10.
  destruct h10 as [i h10].
  destruct h10 as [h10l h10r].
  pose proof (el_prod_compose_le_ai b g _ h10l) as h11.
  assert (h12:Ensembles.In E i). auto with sets.
  pose proof (el_prod_compose_le_ai a g _ h12) as h13.
  pose proof (mono_prod _ _ _ _ h11 h13) as h14.
  destruct (b |-> i); destruct (a |-> i). 
  contradict h10r. reflexivity. 
  simpl in h14. rewrite comp_prod in h14. apply le_x_0. 
  rewrite comm_prod. assumption.
  simpl in h14. rewrite (comm_prod _ (-g i) (g i)) in h14. 
  rewrite comp_prod in h14. apply le_x_0.  
  rewrite comm_prod. assumption. contradict h10r. reflexivity.
assert (h7:el_prod_compose g b = (el_prod_compose g b) * 1). rewrite one_prod. reflexivity.
rewrite <- (plus_subset_non_zero_el_prod_compose_maps_full _ h1 g) in h7.
pose proof (decompose_setminus_inc _ _ h3) as h8.
assert (h9: plus_subset_non_zero_el_prod_compose_maps E h1 g (non_zero_el_prod_compose_maps g E)
         (inclusion_reflexive (non_zero_el_prod_compose_maps g E)) = 
            plus_subset_non_zero_el_prod_compose_maps E h1 g (Union L (Setminus (non_zero_el_prod_compose_maps g E) L)) (inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_union h1 g _ _ h3 (inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_comp h1 g _ h3))).
  unfold plus_subset_non_zero_el_prod_compose_maps.
  apply plus_set_functional.
  apply feq_im. assumption.
rewrite h9 in h7.
unfold plus_subset_non_zero_el_prod_compose_maps in h7. 
rewrite dist_set_plus1 in h7.
generalize dependent  (finite_image (Btype (Bc A)) (Btype (Bc A))
            (Im (Union L (Setminus (non_zero_el_prod_compose_maps g E) L))
               (el_prod_compose g))
            (fun y : Btype (Bc A) => el_prod_compose g b * y)
            (finite_image (Fin_map E signe mns) At
               (Union L (Setminus (non_zero_el_prod_compose_maps g E) L))
               (el_prod_compose g)
               (finite_fin_map_ens
                  (Union L (Setminus (non_zero_el_prod_compose_maps g E) L))
                  h1 signe_finite))).
rewrite im_im.  
rewrite im_union. 
intros h10.
assert (h11:Included (Im L (fun x:Fin_map E signe mns => el_prod_compose g b * el_prod_compose g x))  (Union (Im L  (fun x:Fin_map E signe mns => el_prod_compose g b * el_prod_compose g x)) (Im (Setminus (non_zero_el_prod_compose_maps g E) L)  (fun x:Fin_map E signe mns => el_prod_compose g b * el_prod_compose g x)))).
  auto with sets.
assert (h12:Included  (Im (Setminus (non_zero_el_prod_compose_maps g E) L)  (fun x:Fin_map E signe mns => el_prod_compose g b * el_prod_compose g x))  (Union (Im L  (fun x:Fin_map E signe mns => el_prod_compose g b * el_prod_compose g x))
             (Im (Setminus (non_zero_el_prod_compose_maps g E) L)  (fun x:Fin_map E signe mns => el_prod_compose g b * el_prod_compose g x)))).
  auto with sets.
assert (h13:Finite (Im L  (fun x:Fin_map E signe mns => el_prod_compose g b * el_prod_compose g x))).  eapply Finite_downward_closed. apply h10. apply h11.
assert (h14:Finite (Im (Setminus (non_zero_el_prod_compose_maps g E) L)  (fun x:Fin_map E signe mns => el_prod_compose g b * el_prod_compose g x))). eapply Finite_downward_closed. apply h10. apply h12.  
(*rewrite (pleus_set_union' _ _ h13 h14) in h10'.*)
assert (h16:plus_set (Im (Setminus (non_zero_el_prod_compose_maps g E) L)  (fun x:Fin_map E signe mns => el_prod_compose g b * el_prod_compose g x)) h14 = 0).
  destruct (classic (Inhabited (Setminus (non_zero_el_prod_compose_maps g E) L))) as [h17 | h18].
    rewrite <- (plus_set_sing 0).
    apply plus_set_functional.
    apply Extensionality_Ensembles.
    red. split.
    red.
    intros x h18.
    destruct h18 as [x h18]. subst.
    rewrite comm_prod.
    rewrite h6; auto. constructor.
    red.
    intros z h18.
    destruct h18; subst.
    destruct h17 as [a h17].
    apply Im_intro with a; auto.
    rewrite comm_prod. rewrite h6; auto.
    pose proof (not_inhabited_empty _ h18) as h19.
    pose proof (image_empty _ _ (fun x : Fin_map E signe mns => el_prod_compose g b * el_prod_compose g x)) as h20.
    rewrite <- h19 in h20.
    rewrite <- plus_set_empty.
    apply plus_set_functional.
    assumption. 
rewrite  h7 at 1.
pose proof (im_im (Union L (Setminus (non_zero_el_prod_compose_maps g E) L))  (el_prod_compose g)) (fun y : bt A => el_prod_compose g b * y) as hii.
simpl in hii.
pose proof (im_union L (Setminus (non_zero_el_prod_compose_maps g E) L))  (fun x : Fin_map E signe mns =>
           el_prod_compose g b * el_prod_compose g x) as hiu.
rewrite hiu in hii.
assert (heq: plus_set
     (Im
        (Im (Union L (Setminus (non_zero_el_prod_compose_maps g E) L))
           (el_prod_compose g)) (fun y : bt A => el_prod_compose g b * y))
     (finite_image (bt A) (Btype (Bc A))
        (Im (Union L (Setminus (non_zero_el_prod_compose_maps g E) L))
           (el_prod_compose g)) (fun y : bt A => el_prod_compose g b * y)
        (finite_image (Fin_map E signe mns) At
           (Union L (Setminus (non_zero_el_prod_compose_maps g E) L))
           (el_prod_compose g)
           (finite_fin_map_ens
              (Union L (Setminus (non_zero_el_prod_compose_maps g E) L)) h1
              signe_finite))) = 
             plus_set  (Union
          (Im L
             (fun x : Fin_map E signe mns =>
              el_prod_compose g b * el_prod_compose g x))
          (Im (Setminus (non_zero_el_prod_compose_maps g E) L)
             (fun x : Fin_map E signe mns =>
              el_prod_compose g b * el_prod_compose g x))) h10).
  apply plus_set_functional; auto.
unfold At, bt in heq. unfold At, bt. simpl in heq. simpl. rewrite heq at 1.
unfold plus_subset_non_zero_el_prod_compose_maps.
pose proof (plus_set_union (Im L
           (fun x : Fin_map E signe mns =>
            el_prod_compose g b * el_prod_compose g x))
        (Im (Setminus (non_zero_el_prod_compose_maps g E) L)
           (fun x : Fin_map E signe mns =>
            el_prod_compose g b * el_prod_compose g x)) h13 h14) as h15.
assert (h10 =  Union_preserves_Finite (bt A)
             (Im L
                (fun x : Fin_map E signe mns =>
                 el_prod_compose g b * el_prod_compose g x))
             (Im (Setminus (non_zero_el_prod_compose_maps g E) L)
                (fun x : Fin_map E signe mns =>
                 el_prod_compose g b * el_prod_compose g x)) h13 h14).
  apply proof_irrelevance.
subst.
rewrite h15 at 1.
rewrite h16.
rewrite zero_sum.

apply plus_set_functional.
pose proof h5 as h5'. clear h5.
apply Extensionality_Ensembles. red.
split.
red.
intros x h17.
destruct h17 as [x h17]. subst.

unfold le in h5'. specialize (h5' x h17). rewrite eq_ord in h5'.
rewrite comm_prod.
rewrite h5'. apply Im_intro with x. assumption. reflexivity.
red.
intros x h17. destruct h17 as [x h17]. subst.
specialize (h5' x h17). unfold le in h5'. rewrite eq_ord in h5'.
rewrite <- h5'.
rewrite comm_prod.
apply Im_intro with x. assumption. reflexivity. 
Qed.





Lemma el_prod_inc_c : 
  forall (E F:Ensemble At) (pfe:Finite E)
         (b:Fin_map F signe mns), Included F E -> 
    let C := 
        [a:At | exists (S:Ensemble (Fin_map E signe mns)) 
                                           (pfi:Included S (non_zero_el_prod_maps E)),
                     a = plus_subset_non_zero_el_prod_maps E pfe S pfi] in
    Ensembles.In C (el_prod b).
intros E F h1 b h2 C.
pose proof (el_prod_eq_plus_subset_non_zero_el_prod_maps_extends E F h1 h2 b) as h3. simpl in h3.
destruct h3 as [h3 h4].
rewrite h4. constructor.
exists  [a : Fin_map E signe mns
         | Ensembles.In (non_zero_el_prod_maps E) a /\ extends a b].
exists h3.
reflexivity.
Qed.


Lemma elt_eq_plus_subset_non_zero_el_prod_maps_at_one : 
  forall (E:Ensemble At) (pfe:Finite E) (r:At),
    Ensembles.In E r -> 
    let L := [a : Fin_map E signe mns
               | Ensembles.In (non_zero_el_prod_maps E) a /\ a |-> r = pls] in
      exists pf:Included L (non_zero_el_prod_maps E),
        r = plus_subset_non_zero_el_prod_maps E pfe L pf.
intros E h1 x h2.
pose proof (In_singleton _ x) as h3.
assert (h4:Included (Singleton x) E). red. intros x' h4.
destruct h4; subst. assumption.
assert (h6:functionally_paired (Singleton x) signe (Singleton (x, pls))).
constructor.
intros x' h5.
exists pls. red. split. split.
unfold signe. left.
destruct h5; subst. constructor.
intros s h6.
destruct h6 as [h6l h6r].
inversion h6r. reflexivity.
intros pr h5. 
inversion h5 as [h6]. rewrite (surjective_pairing pr) in h6.
inversion h6. subst.
simpl. split. constructor. left.
pose proof (Singleton_is_finite _ x) as h7.
pose (fin_map_intro _ _ mns h7 signe_finite _ h6) as F.
assert (h8: x = el_prod F).
unfold F. unfold el_prod.
generalize (finite_image At (bt A) (Singleton x)
                         (fun i : At =>
                            eps i
                                (fin_map_intro (Singleton x) signe mns h7 signe_finite
                                               (Singleton (x, pls)) h6 |-> i))
                         (fin_map_fin_dom
                            (fin_map_intro (Singleton x) signe mns h7 signe_finite
                                           (Singleton (x, pls)) h6))).
rewrite im_singleton. intro h8.
rewrite times_set_sing'.
simpl.
pose proof (fps_to_f_s_compat h6 mns _ h3) as h9.
inversion h9. simpl. reflexivity.  simpl.
assert (h9: Included
            [a : Fin_map E signe mns
            | Ensembles.In (non_zero_el_prod_maps E) a /\ a |-> x = pls]
            (non_zero_el_prod_maps E)).
  red. intros f h9. destruct h9 as [h9]. destruct h9; auto.
exists h9.
pose proof (el_prod_eq_plus_subset_non_zero_el_prod_maps_extends _ _ h1 h4 F) as h10. simpl in h10.
destruct h10 as [h10 h11].
rewrite <- h8 in h11 at 1.
rewrite  h11 at 1.
assert (h12:[a : Fin_map E signe mns
         | Ensembles.In (non_zero_el_prod_maps E) a /\ a |-> x = pls] = 
            [a : Fin_map E signe mns
          | Ensembles.In (non_zero_el_prod_maps E) a /\ extends a F]).
  apply Extensionality_Ensembles.
  red. split. red.
  intros f h12. destruct h12 as [h12]. destruct h12 as [h12 h13].
  constructor. split; auto. red. split; auto.
  intros x' h14. destruct h14. simpl.
  rewrite h13.
  pose proof (fps_to_f_s_compat h6 mns x h3) as h14.
  inversion h14. reflexivity.
  red. intros f h12.
  destruct h12 as [h12]. destruct h12 as [h12 h13].
  constructor. split; auto. red in h13. destruct h13 as [h13 h14].
  specialize (h14 _ h3). rewrite <- h14. simpl.
  pose proof (fps_to_f_s_compat h6 mns x h3) as h15.
  inversion h15. reflexivity.
pose proof (subsetT_eq_compat _ (fun S => Included S (non_zero_el_prod_maps E)) _ _ h9 h10 h12) as h13.
dependent rewrite -> h13.
reflexivity.
Qed.


Lemma elt_eq_plus_subset_non_zero_el_prod_compose_maps_at_one : 
  forall {B:Bool_Alg} (E:Ensemble (bt B)) (pfe:Finite E)
         (g:bt B->At) (r:bt B),
    Ensembles.In E r -> 
    let L := [a : Fin_map E signe mns
               | Ensembles.In (non_zero_el_prod_compose_maps g E) a /\ a |-> r = pls] in
      exists pf:Included L (non_zero_el_prod_compose_maps g E),
        g r = plus_subset_non_zero_el_prod_compose_maps E pfe g L pf.
intros B E h1 g x h2.
pose proof (In_singleton _ x) as h3.
assert (h4:Included (Singleton x) E).  red. intros x' h4. 
  destruct h4. assumption.
assert (h6:functionally_paired (Singleton x) signe (Singleton (x, pls))).
constructor.
intros x' h5.
exists pls. red. split. split.
unfold signe. left.
destruct h5; subst. constructor.
intros s h6.
destruct h6 as [h6l h6r].
inversion h6r. reflexivity.
intros pr h5. 
inversion h5 as [h6]. rewrite (surjective_pairing pr) in h6.
inversion h6. subst.
simpl. split. constructor. left.
pose proof (Singleton_is_finite _ x) as h7.
pose (fin_map_intro _ _ mns h7 signe_finite _ h6) as F.
assert (h8: g x = el_prod_compose g F).
unfold F. unfold el_prod_compose.
generalize (finite_image (bt B) (bt B) (Singleton x)
                         (fun i : bt B =>
                            eps i
                                (fin_map_intro (Singleton x) signe mns h7 signe_finite
                                               (Singleton (x, pls)) h6 |-> i))
                         (fin_map_fin_dom
                            (fin_map_intro (Singleton x) signe mns h7 signe_finite
                                           (Singleton (x, pls)) h6))).
rewrite im_singleton. intro h8.
pose proof (im_singleton x  (fun i : bt B =>
         eps (g i)
           (fin_map_intro (Singleton x) signe mns h7 signe_finite
              (Singleton (x, pls)) h6 |-> i))) as hs.
pose proof (Singleton_is_finite _ ( (fun i : bt B =>
         eps (g i)
           (fin_map_intro (Singleton x) signe mns h7 signe_finite
              (Singleton (x, pls)) h6 |-> i)) x)) as hs'.
pose proof (subsetT_eq_compat _ _ _ _  (finite_image (bt B) (bt A) (Singleton x)
        (fun i : bt B =>
         eps (g i)
           (fin_map_intro (Singleton x) signe mns h7 signe_finite
              (Singleton (x, pls)) h6 |-> i))
        (fin_map_fin_dom
           (fin_map_intro (Singleton x) signe mns h7 signe_finite
              (Singleton (x, pls)) h6))) hs' hs) as hs''.
dependent rewrite -> hs''.
rewrite times_set_sing'.
simpl.
pose proof (fps_to_f_s_compat h6 mns _ h3) as h9.
inversion h9. simpl. reflexivity.  simpl.
assert (h9: Included
            [a : Fin_map E signe mns
            | Ensembles.In (non_zero_el_prod_compose_maps g E) a /\ a |-> x = pls]
            (non_zero_el_prod_compose_maps g E)).
  red. intros f h9. destruct h9 as [h9]. destruct h9; auto.
exists h9.
pose proof (el_prod_compose_eq_plus_subset_non_zero_el_prod_compose_maps_extends _ _ h1 h4 g F) as h10. simpl in h10.
destruct h10 as [h10 h11].
rewrite <- h8 in h11 at 1.
rewrite  h11 at 1.
assert (h12:[a : Fin_map E signe mns
         | Ensembles.In (non_zero_el_prod_compose_maps g E) a /\ a |-> x = pls] = 
            [a : Fin_map E signe mns
          | Ensembles.In (non_zero_el_prod_compose_maps g E) a /\ extends a F]).
  apply Extensionality_Ensembles.
  red. split. red.
  intros f h12. destruct h12 as [h12]. destruct h12 as [h12 h13].
  constructor. split; auto. red. split; auto.
  intros x' h14. destruct h14. simpl.
  rewrite h13.
  pose proof (fps_to_f_s_compat h6 mns x h3) as h14.
  inversion h14. reflexivity.
  red. intros f h12.
  destruct h12 as [h12]. destruct h12 as [h12 h13].
  constructor. split; auto. red in h13. destruct h13 as [h13 h14].
  specialize (h14 _ h3). rewrite <- h14. simpl.
  pose proof (fps_to_f_s_compat h6 mns x h3) as h15.
  inversion h15. reflexivity.
pose proof (subsetT_eq_compat _ (fun S => Included S (non_zero_el_prod_compose_maps g E)) _ _ h9 h10 h12) as h13.
dependent rewrite -> h13.
reflexivity.
Qed.



Lemma el_prod_empty : forall (F:Fin_map (Empty_set At) signe mns),
                        el_prod F = 1.
intros F. unfold el_prod. 
generalize dependent (finite_image At (bt A) (Empty_set At) (fun i : At => eps i (F |-> i))
        (fin_map_fin_dom F)).
rewrite image_empty.
intro h1. 
apply times_set_empty'.
Qed.


Lemma el_prod_decompose_restriction : 
  forall (E F:Ensemble At) (pf:Included F E)
         (a:Fin_map E signe mns),
  el_prod a = el_prod (restriction a pf) * 
              el_prod (restriction a (setminus_inc E F)).
intros E F h1 a.
unfold el_prod.
pose proof (decompose_setminus_inc _ _ h1) as h2.
generalize  (finite_image At (bt A) E (fun i : At => eps i (a |-> i))
        (fin_map_fin_dom a)).
pose proof (feq_im _ _ (fun i : At => eps i (a |-> i)) h2) as h3.
rewrite im_union in h3.
rewrite h3.
intro h4. 
rewrite <- times_set_union.
apply times_set_functional.
apply Extensionality_Ensembles.
red. split.
red.
intros y h5.
destruct h5 as [y h5l | y h5r].
left. 
destruct h5l as [x h5l y h6].  rewrite h6.
apply Im_intro with x. assumption.
f_equal.
rewrite restriction_compat; auto.
right.
destruct h5r as [x h5l y h6].  rewrite h6.
apply Im_intro with x. assumption.
f_equal. rewrite restriction_compat; auto.
red.
intros y h5.
destruct h5 as [y h5l | y h5r].
left.
destruct h5l as [x h5l y h6]. rewrite h6.
apply Im_intro with x; auto. f_equal. rewrite restriction_compat; auto.
destruct h5r as [x h5r y h6]. rewrite h6. right.
apply Im_intro with x; auto. f_equal. rewrite restriction_compat; auto.
Qed.

Lemma eps_in_closed :
  forall (S:Ensemble At),
    alg_closed S ->
    forall x:At,
      Ensembles.In S x ->
      forall s:sign, Ensembles.In S (eps x s).
intros S h1 x h2 s.
destruct s. simpl. assumption. simpl.
pose proof (C_c _ _ h1) as h3.
red in h3.
specialize (h3 (exist _ _ h2)).
unfold Bcomp_sub in h3.
simpl in h3.
assumption.
Qed.


Lemma in_gen_ens_im_g_plus_subset_el_prod_compose_maps : 
  forall {B:Bool_Alg} (E:Ensemble (bt B)) (pfe:Finite E)
         (g:(bt B)->At) (S:Ensemble (Fin_map E signe mns))
         (pfi:Included S (non_zero_el_prod_compose_maps g E)),
    Ensembles.In (Gen_Ens (Im E g)) (plus_subset_non_zero_el_prod_compose_maps _ pfe g S pfi).
intros B E h1 g S h0. 
pose proof (closed_gen_ens _ (Im E g)) as h2.
unfold plus_subset_non_zero_el_prod_compose_maps.
apply plus_set_closed; auto.
red.
intros x h3.
destruct h3 as [f h3]. subst.
unfold el_prod_compose.
apply times_set_closed; auto.
red. intros x h4.
destruct h4 as [x h4]. subst.
apply eps_in_closed; auto.
apply gen_ens_includes.
apply Im_intro with x; auto.
Qed.


Lemma el_prod_in_gen : 
  forall (E:Ensemble At)
         (F:Fin_map E signe mns),
    Ensembles.In (Gen_Ens E) (el_prod F).
intros E F.
pose proof (fin_map_fin_dom F) as h1.
revert F.
induction h1 as [|E h1 h2 x h3].
intros F.
rewrite gen_ens_empty.
rewrite el_prod_empty. right.
intro F.
assert (h4:Included E (Add E x)). auto with sets.
specialize (h2 (restriction F h4)).
pose proof (el_prod_decompose_restriction).
rewrite (el_prod_decompose_restriction _ _ h4).
assert (h5: el_prod (restriction F (setminus_inc (Add E x) E)) =
            eps x (F |-> x)).
  unfold el_prod.
  generalize  (finite_image At (bt A) (Setminus (Add E x) E)
        (fun i : At => eps i (restriction F (setminus_inc (Add E x) E) |-> i))
        (fin_map_fin_dom (restriction F (setminus_inc (Add E x) E)))).

  pose proof (setminus_add1 _ _ h3) as h5.
pose proof (feq_im _ _ (fun i : At => eps i (restriction F (setminus_inc (Add E x) E) |-> i)) h5) as h6.
rewrite h6.
rewrite im_singleton.
intro h7.
rewrite times_set_sing'.
f_equal.
rewrite restriction_compat. reflexivity. constructor. right.
constructor. assumption.
rewrite h5.
pose proof (gen_ens_preserves_inclusion  _ _ h4) as h6.
assert (h7:Ensembles.In (Gen_Ens (Add E x)) (el_prod (restriction F h4))). auto with sets.
pose proof (closed_gen_ens _ (Add E x)) as h8.
assert (h9:Ensembles.In (Gen_Ens (Add E x)) x).
  constructor. intros S h9.
  destruct h9 as [h9]. destruct h9 as [h9l h9r].
  pose proof (Add_intro2 _ E x). auto with sets.
pose proof (eps_in_closed _ h8 _ h9 (F |-> x)) as h10.
apply times_closed; auto.
Qed.

Lemma plus_subset_non_zero_el_prod_maps_in_gen : 
  forall (E:Ensemble At) (pf:Finite E) 
         (S:Ensemble (Fin_map E signe mns))
         (pfi:Included S (non_zero_el_prod_maps E)),
    Ensembles.In (Gen_Ens E) (plus_subset_non_zero_el_prod_maps E pf S pfi).
intros E h1 S.
pose proof (finite_fin_map_ens S h1 signe_finite) as h2.
induction h2 as [|S h3 h4 a h5].
intro h2.
unfold plus_subset_non_zero_el_prod_maps.
generalize  (finite_image (Fin_map E signe mns) At
           (Empty_set (Fin_map E signe mns)) el_prod
           (Finite_downward_closed (Fin_map E signe mns)
              (non_zero_el_prod_maps E) (non_zero_el_prod_maps_fin E h1)
              (Empty_set (Fin_map E signe mns)) h2)).
rewrite image_empty.
intro h3. rewrite plus_set_empty'.
pose proof (closed_gen_ens _ E) as h4.
apply zero_closed; auto.
intro h6.
assert (h7:Included S (Add S a)). auto with sets.
assert (h8:Included S (non_zero_el_prod_maps E)). auto with sets.
specialize (h4 h8).
unfold plus_subset_non_zero_el_prod_maps.
unfold Add.
generalize (finite_image (Fin_map E signe mns) At (Union S (Singleton a))
           el_prod
           (Finite_downward_closed (Fin_map E signe mns)
              (non_zero_el_prod_maps E) (non_zero_el_prod_maps_fin E h1)
              (Union S (Singleton a)) h6)).
rewrite im_union.
intro h9. 
unfold plus_subset_non_zero_el_prod_maps in h4.
pose proof (finite_image _ _ _ el_prod h3) as h10.
pose proof (Singleton_is_finite _ a) as h11.
pose proof (finite_image _ _ _ el_prod h11) as h12.
pose proof (plus_set_union' _ _ h10 h12 h9) as h13.
unfold At in h13. unfold At.
rewrite h13 at 1.
assert (h14:(finite_image (Fin_map E signe mns) At S el_prod
               (Finite_downward_closed (Fin_map E signe mns)
                  (non_zero_el_prod_maps E) (non_zero_el_prod_maps_fin E h1)
                  S h8)) = h10). apply proof_irrelevance.
rewrite h14 in h4.
pose proof (closed_gen_ens _ E) as h15.
apply plus_closed; auto. 
assert (h16: (plus_set (Im (Singleton a) el_prod) h12) = el_prod a).
  rewrite <- plus_set_sing.
  apply plus_set_functional.
  rewrite im_singleton. reflexivity.
unfold At in h16.
rewrite h16.
apply el_prod_in_gen.
Qed.


Lemma gen_e_eq_c : 
  forall (E:Ensemble At) (pf:Finite E), 
    let C :=
        [a:At | exists (S:Ensemble (Fin_map E signe mns))
                       (pfi:Included S (non_zero_el_prod_maps E)),
                a = plus_subset_non_zero_el_prod_maps E pf S pfi] in
    Gen_Ens E = C.
intros E h1 C.
apply Extensionality_Ensembles.
red. split.
(* <= *)
assert (h2:Included E C).
  red.
  intros x h2.
  pose proof (In_singleton _ x) as h3.
  assert (h4:Included (Singleton x) E). red. intros x' h4.
    destruct h4; subst. assumption.
  assert (h6:functionally_paired (Singleton x) signe (Singleton (x, pls))).
    constructor.
    intros x' h5.
    exists pls. red. split. split.
    unfold signe. left.
    destruct h5; subst. constructor.
    intros s h6.
    destruct h6 as [h6l h6r].
    inversion h6r. reflexivity.
    intros pr h5. 
    inversion h5 as [h6]. rewrite (surjective_pairing pr) in h6.
    inversion h6. subst.
    simpl. split. constructor. left.
  pose proof (Singleton_is_finite _ x) as h7.
  pose (fin_map_intro _ _ mns h7 signe_finite _ h6) as F.
  assert (h8: x = el_prod F).
    unfold F. unfold el_prod.
    generalize (finite_image At (bt A) (Singleton x)
        (fun i : At =>
         eps i
           (fin_map_intro (Singleton x) signe mns h7 signe_finite
              (Singleton (x, pls)) h6 |-> i))
        (fin_map_fin_dom
           (fin_map_intro (Singleton x) signe mns h7 signe_finite
              (Singleton (x, pls)) h6))).
   rewrite im_singleton. intro h8.
   rewrite times_set_sing'.
   simpl.
   pose proof (fps_to_f_s_compat h6 mns _ h3) as h9.
   inversion h9. simpl. reflexivity.
  rewrite h8.
  apply el_prod_inc_c. assumption.
pose proof (gen_ens_preserves_inclusion _ _ h2) as h3.
rewrite <- gen_ens_closed_eq.
assumption.
apply closed_all_plus_subsets_non_zero_el_prod_maps.
(* >=*)
red.
intros x h2.
destruct h2 as [h2].
destruct h2 as [S h2].
destruct h2 as [h2 h3].
subst.
apply plus_subset_non_zero_el_prod_maps_in_gen.
Qed.



(* The main theorem of this section, along with unprimed eponym!*)
Theorem normal_form' :
  forall (E:Ensemble At) (pfe:Finite E),
    let Bt := (Btype (Bc (Gen E))) in
    (forall p:Bt, atom p <-> 
                      ((p <> 0) /\ 
                       exists f:Fin_map E signe mns,
                         proj1_sig p = el_prod f))
                         /\
    (forall b:Bt, exists !(S:Ensemble (Fin_map E signe mns)),
       exists (pf:Included S (non_zero_el_prod_maps E)),
       plus_subset_non_zero_el_prod_maps E pfe S pf =
       proj1_sig b).
intros E h1 Bt.
pose proof (gen_e_eq_c _ h1) as h2.
pose proof (closed_gen_ens _ E) as h3.
pose proof (closed_all_plus_subsets_non_zero_el_prod_maps _ h1) as h4.
unfold Bt. unfold Gen.  
pose (fun (D:Ensemble At) (pf:alg_closed D) =>
 (forall p : Btype (Bc (Subalg D pf)),
    atom p <->
    p <> 0 /\ (exists f : Fin_map E signe mns, proj1_sig p = el_prod f)) /\
   (forall b : Btype (Bc (Subalg D pf)),
    exists ! S : Ensemble (Fin_map E signe mns),
      exists pf : Included S (non_zero_el_prod_maps E),
        plus_subset_non_zero_el_prod_maps E h1 S pf = proj1_sig b)) as f.
assert (h5:f _ (closed_gen_ens _ E) <-> f _ h4).  
  generalize (closed_gen_ens _ E). rewrite h2.  intro h6. 
  assert (h7:h4 = h6). apply proof_irrelevance. subst.
  tauto.
unfold f in h5. simpl in h5. simpl. 
rewrite h5.
split.
intro p. split.
intro h8. 
pose proof (atom_non_zero_el_prod _ h1 p) as h9.
unfold atom in h8. unfold atom in h9.
assert (h4 =  (closed_all_plus_subsets_non_zero_el_prod_maps E h1)). apply proof_irrelevance. subst. 
pose proof h8 as h8'. destruct h8' as [h8l h8r].
simpl in h8l. split. unfold At in h8l. unfold At.
assumption.
rewrite h9 in h8. 
destruct h8 as [a h8].
exists a. destruct h8; assumption.
intro h6.
pose proof (atom_non_zero_el_prod _ h1 p) as h9.
destruct h6 as [h6l h6r].
assert (h10: (exists a : Fin_map E signe mns,
          Ensembles.In (non_zero_el_prod_maps E) a /\ proj1_sig p = el_prod a)).
  destruct h6r as [a h6r].
  exists a. split.
  constructor.
  rewrite <- h6r.
  intro h10. 
  rewrite (unfold_sig _ p) in h6l.
  contradict h6l. apply proj1_sig_injective. simpl. assumption.
  assumption.
rewrite <- h9 in h10.
unfold atom. unfold atom in h10.
assert (h4 = (closed_all_plus_subsets_non_zero_el_prod_maps E h1)). apply proof_irrelevance. subst.
assumption. 
intro b.
pose proof (proj2_sig b) as h7.
simpl in h7.
destruct h7 as [h7].
destruct h7 as [S [h7 h8]].
exists S. red.
split.
exists h7. symmetry. assumption.
intros S' h10.
destruct h10 as [h10l h10r].
rewrite h8 in h10r.
symmetry in h10r.
apply (plus_subset_non_zero_el_prod_maps_inj _ h1 _ _ h7 h10l h10r).
Qed.


Lemma gen_e_eq_c_unq : 
  forall (E:Ensemble At) (pf:Finite E), 
    let C :=
        [a:At | exists! (S:Ensemble (Fin_map E signe mns)),
                exists (pfi:Included S (non_zero_el_prod_maps E)),
                plus_subset_non_zero_el_prod_maps E pf S pfi = a] in
    Gen_Ens E = C.
intros E h1 C. unfold C.
pose proof (normal_form' _ h1) as h2. simpl in h2.
destruct h2 as [h2l h2r].
pose proof (gen_e_eq_c _ h1) as h3. simpl in h3.
apply Extensionality_Ensembles.
red. split.
red. intros x h4.
specialize (h2r (exist _ _ h4)).
constructor. simpl in h2r.
assumption.
red.
intros x h4.
destruct h4 as [h4].
rewrite h3.
constructor.
destruct h4 as [S h4]. red in h4. destruct h4 as [h4a h4b].
destruct h4a as [h4a h4c].
exists S. exists h4a. rewrite h4c.
reflexivity.
Qed.


Lemma plus_set_incl_sig : 
  forall (E F:Ensemble At) (pfi:Included E F) (pfe:Finite E) 
         (pfss:Finite (incl_sig E F pfi)) (pff:alg_closed F),
         proj1_sig (@plus_set (Subalg _ pff) (incl_sig E _ pfi) pfss) =
         plus_set E pfe. 
intros E F h1 h2 h3 h4. 
assert (h5:E = Im (incl_sig E F h1) (@proj1_sig _ _)).
  apply Extensionality_Ensembles.
  red. split.
  red.
  intros x h5.
  apply Im_intro with (exist _ _ (h1 _ h5)).
  constructor. simpl. reflexivity.
  red. intros x h5.
  destruct h5 as [x h5]. subst.
  rewrite <- (incl_sig_iff _ _ h1). assumption.  
pose proof h2 as h2'.
rewrite h5 in h2'. 
assert (h6:plus_set E h2 = plus_set _ h2').
  apply plus_set_functional. assumption.
rewrite h6.
unfold At.
pose proof h2 as h2''.
rewrite <- (finite_incl_sig_iff _ _ h1) in h2''. 
pose proof (plus_set_sub_compat _ _ h4 _ h2'' h2') as h9.
assert (h10:h2'' = h3). apply proof_irrelevance. 
rewrite h10 in h9.
unfold At. unfold At in h9.
rewrite <- h9.
apply plus_set_functional.
reflexivity.
Qed.

Lemma times_set_incl_sig : 
  forall (E F:Ensemble At) (pfi:Included E F) (pfe:Finite E) 
         (pfss:Finite (incl_sig E F pfi)) (pff:alg_closed F),
         proj1_sig (@times_set (Subalg _ pff) (incl_sig E _ pfi) pfss) =
         times_set E pfe. 
intros E F h1 h2 h3 h4. 
assert (h5:E = Im (incl_sig E F h1) (@proj1_sig _ _)).
  apply Extensionality_Ensembles.
  red. split.
  red.
  intros x h5.
  apply Im_intro with (exist _ _ (h1 _ h5)).
  constructor. simpl. reflexivity.
  red. intros x h5.
  destruct h5 as [x h5]. subst.
  rewrite <- (incl_sig_iff _ _ h1). assumption.  
pose proof h2 as h2'.
rewrite h5 in h2'. 
assert (h6:times_set E h2 = times_set _ h2').
  apply times_set_functional. assumption.
rewrite h6.
unfold At.
pose proof h2 as h2''.
rewrite <- (finite_incl_sig_iff _ _ h1) in h2''. 
pose proof (times_set_sub_compat _ _ h4 _ h2'' h2') as h9.
assert (h10:h2'' = h3). apply proof_irrelevance. 
rewrite h10 in h9.
unfold At. unfold At in h9.
rewrite <- h9.
apply times_set_functional.
reflexivity.
Qed.



(* More faithful to Givant/Halmos' form of the normal_form *)
Theorem normal_form :
  forall (E:Ensemble At),
    let Bt := (Btype (Bc (Gen E))) in
    Finite E ->
    (forall p:Bt, atom p <-> 
                      ((p <> 0) /\ 
                       exists f:Fin_map E signe mns,
                         proj1_sig p = el_prod f))
                         /\
    (forall b:Bt, exists !(S:Ensemble Bt),
       exists pf:Finite S, 
         (forall x:Bt, Ensembles.In S x -> atom x) /\
         plus_set S pf = b).
intros E Bt h1.
pose proof (normal_form' E h1) as h2.
destruct h2 as [h2l h2r].
split. assumption. 
intros b.
specialize (h2r b).
destruct h2r as [S h2r].
pose (Im S el_prod) as imS.
assert (h3':Included (Im S el_prod) (Gen_Ens E)).
  red.
  intros y h4.
  destruct h4 as [y h4 x]. subst.
  apply el_prod_in_gen.
exists (incl_sig _ _ h3').
red in h2r. red.
destruct h2r as [h2a h2b].
destruct h2a as [h6 h7].
pose proof (finite_fin_map_ens S h1 signe_finite) as h3.
pose proof (finite_image _ _ _ el_prod h3) as h4.  
rewrite <- (finite_incl_sig_iff _ _ h3') in h4.
split. 
exists h4.
split. intros x h5.
inversion h5 as [c h8].  subst.
destruct h8 as [a h8 y]. subst.
assert (h9: (exist (Ensembles.In (Gen_Ens E)) (el_prod a)
        (h3' (el_prod a)
           (Im_intro (Fin_map E signe mns) At S el_prod a h8 
              (el_prod a) eq_refl))) <> (Bzero (Bc (Gen E))) /\
            (exists f:Fin_map E signe mns, proj1_sig  (exist (Ensembles.In (Gen_Ens E)) (el_prod a)
        (h3' (el_prod a)
           (Im_intro (Fin_map E signe mns) At S el_prod a h8 
              (el_prod a) eq_refl))) = el_prod f)).
  split.
  assert (h10:Ensembles.In (non_zero_el_prod_maps E) a).  auto with sets.
  destruct h10 as [h10].
  intro h11.
  pose proof (exist_injective _ _ _ _ _ h11).
  contradiction.
  exists a. simpl. reflexivity.
rewrite <- h2l in h9.
assumption.
apply proj1_sig_injective.
rewrite <- h7.
unfold plus_subset_non_zero_el_prod_maps.
pose proof (finite_image _ _ _ el_prod h3) as h8.
pose proof (closed_gen_ens _ E) as h9.
pose proof (plus_set_incl_sig _ _ h3' h8 h4 h9) as h10.
assert (h11:h8 = (finite_image (Fin_map E signe mns) At S el_prod
        (Finite_downward_closed (Fin_map E signe mns)
           (non_zero_el_prod_maps E) (non_zero_el_prod_maps_fin E h1) S h6))). apply proof_irrelevance.
subst. 
rewrite <- h10. 
f_equal.
unfold Gen.
assert (h11:h9 = (closed_gen_ens A E)). apply proof_irrelevance.
subst.
apply plus_set_functional.
reflexivity.
intros D h6'.
destruct h6' as [h6l h6r].
destruct h6r as [h6a h6b].
specialize (h2b [a:(Fin_map E signe mns) | Ensembles.In D (exist _ (el_prod a) (el_prod_in_gen E a)) /\ el_prod a <> 0]). 
assert (h7': Included
                  [a : Fin_map E signe mns | Ensembles.In D (exist _ (el_prod a) (el_prod_in_gen E a)) /\ el_prod a <> 0]
                  (non_zero_el_prod_maps E)).
  red. intros a h7'.
  destruct h7' as [h7']. destruct h7'.
  constructor. assumption.
assert (h8:plus_subset_non_zero_el_prod_maps E h1
             [a : Fin_map E signe mns
             | Ensembles.In D (exist _ (el_prod a) (el_prod_in_gen E a)) /\ el_prod a <> 0] h7' = 
           proj1_sig b).
unfold plus_subset_non_zero_el_prod_maps.
assert (h8:Finite [a:Fin_map E signe mns | Ensembles.In D (exist _ (el_prod a) (el_prod_in_gen E a)) /\ el_prod a = 0]).
  apply finite_fin_map_ens. assumption. apply signe_finite.
pose proof (finite_image _ _ _ el_prod h8) as h9.
assert (h10:plus_set (Im [a:Fin_map E signe mns | Ensembles.In D (exist _ (el_prod a) (el_prod_in_gen E a)) /\ el_prod a = 0] el_prod) h9 = 0).
  destruct (classic (Inhabited [a:Fin_map E signe mns | Ensembles.In D (exist _ (el_prod a) (el_prod_in_gen E a)) /\ el_prod a = 0])) as [h10 | h11].
  rewrite <- plus_set_sing.
  apply plus_set_functional.
  apply Extensionality_Ensembles.
  red. split.
  red.
  intros x h11. destruct h11 as [x h11 x' h12]. subst.
  destruct h11 as [h11]. destruct h11 as [h11a h11b].
  rewrite h11b. constructor.
  red. intros x h11. destruct h11; subst.
  destruct h10 as [x h10].
  destruct h10 as [h10].
  apply Im_intro with x. constructor. assumption. symmetry. destruct h10; auto.
  pose proof (not_inhabited_empty  _ h11) as h12.
  rewrite <- plus_set_empty.
  apply plus_set_functional.
  pose proof (feq_im _ _ el_prod h12) as h13.
  rewrite image_empty  in h13.
  assumption.
pose proof (f_equal (Bplus _  (plus_set
     (Im
        [a : Fin_map E signe mns
        | Ensembles.In D
            (exist (Ensembles.In (Gen_Ens E)) (el_prod a)
               (el_prod_in_gen E a)) /\ el_prod a <> 0] el_prod)
     (finite_image (Fin_map E signe mns) At
        [a : Fin_map E signe mns
        | Ensembles.In D
            (exist (Ensembles.In (Gen_Ens E)) (el_prod a)
               (el_prod_in_gen E a)) /\ el_prod a <> 0] el_prod
        (Finite_downward_closed (Fin_map E signe mns)
           (non_zero_el_prod_maps E) (non_zero_el_prod_maps_fin E h1)
           [a : Fin_map E signe mns
           | Ensembles.In D
               (exist (Ensembles.In (Gen_Ens E)) (el_prod a)
                  (el_prod_in_gen E a)) /\ el_prod a <> 0] h7')))) h10) as h11.
rewrite zero_sum in h11. 
unfold At in h11. unfold At.
rewrite <- h11 at 1.
rewrite <- plus_set_union.
rewrite <- h6b.
pose proof (finite_image _ _ _ (@proj1_sig _ _) h6l) as h12.
pose proof (plus_set_sub_compat _ _ (closed_gen_ens _ E) _ h6l h12) as h13.
unfold At in h13. unfold At.
unfold Gen.
rewrite <- h13. unfold Gen in h11.
apply plus_set_functional.

assert (h12':(Union
        [a : Fin_map E signe mns
        | Ensembles.In D (exist _ (el_prod a) (el_prod_in_gen E a)) /\ el_prod a <> 0]
        [a : Fin_map E signe mns
        | Ensembles.In D (exist _ (el_prod a) (el_prod_in_gen E a)) /\ el_prod a = 0]) =
            [a : Fin_map E signe mns | Ensembles.In D (exist _ (el_prod a) (el_prod_in_gen E a))]).
  apply Extensionality_Ensembles.
  red. split. red. 
  intros x h12'.
  destruct h12' as [x h12l | x h12r].
  destruct h12l as [h12l]. destruct h12l; constructor; auto.
  destruct h12r as [h12r]. destruct h12r; constructor; auto.
  red. intros x h12'.
  destruct h12' as [h12'].
  destruct (classic_dec (el_prod x = 0)) as [h13' | h14'].
  right. constructor; auto. left. constructor; auto.
rewrite <- im_union.
unfold At in h12'. unfold At.  

rewrite h12' at 1. 
  apply Extensionality_Ensembles.
  red. split. red.
  intros x h13'.
  destruct h13' as [x h13']. subst.
  destruct h13' as [h13']. 
  apply Im_intro with (exist (Ensembles.In (Gen_Ens E)) (el_prod x) (el_prod_in_gen E x)).
assumption.  simpl. reflexivity.
red.
intros x h14.
destruct h14 as [x h14]. subst.
pose proof h6a as h15. clear h6a. 
specialize (h15 _ h14).
rewrite h2l in h15.
destruct h15 as [h15l h15r].
destruct h15r as [a h15r].
apply Im_intro with a.
constructor. 
assert (h16:(exist (Ensembles.In (Gen_Ens E)) (el_prod a) (el_prod_in_gen E a)) = x).
apply proj1_sig_injective. simpl.
rewrite <- h15r. reflexivity.
rewrite h16 at 1.
assumption.
assumption.
assert (h9:(exists
           pf : Included
                  [a : Fin_map E signe mns
                  | Ensembles.In D
                      (exist (Ensembles.In (Gen_Ens E)) 
                         (el_prod a) (el_prod_in_gen E a)) /\ 
                    el_prod a <> 0] (non_zero_el_prod_maps E),
           plus_subset_non_zero_el_prod_maps E h1
             [a : Fin_map E signe mns
             | Ensembles.In D
                 (exist (Ensembles.In (Gen_Ens E)) 
                    (el_prod a) (el_prod_in_gen E a)) /\ 
               el_prod a <> 0] pf = proj1_sig b)).
  exists h7'. assumption.
specialize (h2b h9). 
subst.
apply Extensionality_Ensembles.
red. split. 
red.
intros x h10.
destruct h10 as [y h10]. 
destruct h10 as [x h10 y h11]. subst.
destruct h10 as [h10].
destruct h10 as [h10l h10r].
assert (h11: (exist (Ensembles.In (Gen_Ens E)) (el_prod x) (el_prod_in_gen E x)) =  (exist (Ensembles.In (Gen_Ens E)) (el_prod x)
        (h3' (el_prod x)
           (Im_intro (Fin_map E signe mns) At
              [a : Fin_map E signe mns
              | Ensembles.In D
                  (exist (Ensembles.In (Gen_Ens E)) 
                     (el_prod a) (el_prod_in_gen E a)) /\ 
                el_prod a <> 0] el_prod x
              (intro_characteristic_sat
                 (fun a : Fin_map E signe mns =>
                  Ensembles.In D
                    (exist (Ensembles.In (Gen_Ens E)) 
                       (el_prod a) (el_prod_in_gen E a)) /\ 
                  el_prod a <> 0) x (conj h10l h10r)) 
              (el_prod x) eq_refl)))).
  apply proj1_sig_injective.
simpl. reflexivity.
unfold At in h11. unfold At.
rewrite <- h11 at 1.
assumption.
red.
intros x h11.
specialize (h6a _ h11). pose proof h6a as h12. clear h6a.
rewrite h2l in h12.
destruct h12 as [h12l h12r].
destruct h12r as [a h13].
assert (h14:Ensembles.In  (Im
             [a : Fin_map E signe mns
             | Ensembles.In D
                 (exist (Ensembles.In (Gen_Ens E)) 
                    (el_prod a) (el_prod_in_gen E a)) /\ 
               el_prod a <> 0] el_prod) (proj1_sig x)).
  rewrite h13.
  apply Im_intro with a.
  constructor. split.
  assert (h14: (exist (Ensembles.In (Gen_Ens E)) (el_prod a) (el_prod_in_gen E a)) = x).
  apply proj1_sig_injective. simpl. rewrite h13. reflexivity.
rewrite h14. assumption.
intro h14. rewrite h14 in h13. contradict h12l. apply proj1_sig_injective. simpl. assumption.
reflexivity.
pose proof (incl_sig_intro _ _ h3' _ h14) as h16.
assert (h15: (exist (Ensembles.In (Gen_Ens E)) (proj1_sig x)
            (h3' (proj1_sig x) h14)) = x).
apply proj1_sig_injective. simpl. reflexivity.
unfold At in h15. unfold At in h16.
rewrite h15 in h16 at 1.
assumption.
Qed.

Corollary normal_form'' : 
  forall (E:Ensemble At),
    let Bt := (Btype (Bc (Gen E))) in
    Finite E ->
    (forall p:Bt, atom p <->
                      ((p <> 0) /\
                       (exists! (S:Ensemble At),
                          exists (pf:Finite S),
                            proj1_sig p = times_set _ pf /\
                            exists f:Fin_map E signe mns,
                              S = im_eps f)))
    /\
    (forall b:Bt, exists !(S:Ensemble Bt),
       exists pf:Finite S,
         (forall x:Bt, Ensembles.In S x -> atom x) /\
         plus_set S pf = b).
intros E Bt h1.
pose proof (normal_form _ h1) as h2.
destruct h2 as [h2l h2r]. split; auto. clear h2r.
intro p. split.
intro h3.
rewrite h2l in h3.
destruct h3 as [h3l h3r]. split; auto.
destruct h3r as [f h4].
exists (im_eps f).
red. split.
exists (finite_im_eps f).
split.
rewrite h4.
unfold el_prod.
apply times_set_functional. unfold im_eps.
reflexivity.
exists f. reflexivity. 
intros S h5.
destruct h5 as [h5 [h6 h7]].
destruct h7 as [f' h8]. subst.
rewrite h4 in h6.
assert (h7:el_prod f <> 0).
  intro h7.
  contradict h3l.
  rewrite <- h4 in h7.
  apply proj1_sig_injective.
  simpl. assumption.
assert (h8:el_prod f = el_prod f').
  rewrite h6. unfold el_prod. apply times_set_functional.
  reflexivity.
apply (non_zero_el_prod_inj f f' h7 h8).
intro h3.
destruct h3 as [h3l h3r].
destruct h3r as [S [h4 h5]].
destruct h4 as [h4a [h4b h4c]].
destruct h4c as [f h4c].
assert (h6:exists f : Fin_map E signe mns, proj1_sig p = el_prod f).
  exists f.
  rewrite h4b.
  unfold el_prod.
  apply times_set_functional.
  rewrite h4c. reflexivity.
specialize (h2l p).
pose proof (conj h3l h6) as h7.
simpl in h7. simpl in h2l. 
pose proof (iff2 h2l h7) as h8. 
assumption. 
Qed.


Corollary normal_form_gen_ba : 
  forall (E:Ensemble (bt A)),
    Finite E ->
    ba_ens A = Gen_Ens E ->
    (forall p : At,
       atom p <->
       p <> 0 /\ (exists f : Fin_map E signe mns, p = el_prod f)) /\
       (forall b : At,
        exists ! S : Ensemble At,
          exists pf : Finite S,
            (forall x : At, Ensembles.In S x -> atom x) /\
            plus_set S pf = b).
intros E h1 h2.
pose proof (normal_form _ h1) as h3.
pose proof (gen_ba_eq_ba _ h2) as h4.
unfold Gen in h3.
pose (fun (C:Ensemble At) (pfc:alg_closed C) =>
         (forall p : Btype (Bc (Subalg C pfc)),
        atom p <->
        p <> 0 /\ (exists f : Fin_map E signe mns, proj1_sig p = el_prod f)) /\
       (forall b : Btype (Bc (Subalg C pfc)),
        exists !
          S : Ensemble (Btype (Bc (Subalg C pfc))),
          exists pf : Finite S,
            (forall x : Btype (Bc (Subalg C pfc)),
             Ensembles.In S x -> atom x) /\ plus_set S pf = b)) as P.
assert (h5:P _ (closed_gen_ens A E) <-> P _ (alg_closed_ba_ens A)).
  pose proof (subsetT_eq_compat _ _ _ _ (alg_closed_ba_ens A) (closed_gen_ens A E) h2) as h6.
  dependent rewrite <- h6.
  tauto. 
pose proof (full_sig_equivalent_atom A) as h9.
red in h9. simpl in h9.
unfold P in h5. simpl in h5, h3.
pose proof (iff1 h5 h3) as h6. clear h5 h3.
destruct h6 as [h6 h7].  split.
intro p. split.  intro h8. 
specialize (h6 (exist _ _ (Full_intro _ p))).
rewrite h9 in h8.
unfold At in h6.
simpl in h6, h8.
rewrite h6 in h8.
destruct h8 as [h8l h8r].
split. intro h10. subst. contradict h8l. apply proj1_sig_injective. simpl.
reflexivity.
assumption. 
intro h8. destruct h8 as [h8l h8r].
assert (h10:  (exist _ _ (Full_intro _ p)) <>
       exist (Ensembles.In (Full_set (Btype (Bc A)))) 0
         (Z_c A (Full_set (Btype (Bc A))) (alg_closed_ba_ens A))).
  intro h10. apply exist_injective in h10. contradiction.
assert (h11:p = proj1_sig  (exist (Ensembles.In (Full_set At)) p (Full_intro At p))).
  reflexivity.
pose proof (conj h10 h8r) as h12.
pose proof (h6  (exist (Ensembles.In (Full_set At)) p (Full_intro At p))) as h13. unfold ba_ens in h13.
simpl in h13, h12. 
pose proof (iff2 h13 h12) as h14. clear h13 h12.
rewrite h9. assumption.
intros b.
specialize (h7 (exist _ _ (Full_intro _ b))).
destruct h7 as [S h7]. destruct h7 as [h7l h7r]. 
destruct h7l as [h7a [h7b h7c]].
exists (Im S (@proj1_sig _ _)).
red. split.
exists (finite_image _ _ _ _ h7a).
split. intros x h8.
destruct h8 as [x h8]. subst.
specialize (h7b _ h8).
destruct x as [x h10].
assert (h11:h10 = (Full_intro (Btype (Bc A)) x)). apply proof_irrelevance. subst.
pose proof (iff2 (h9 x) h7b) as h11.
simpl. assumption.
pose proof (f_equal (@proj1_sig _ _) h7c) as h10. simpl in h10.
rewrite <- h10.
rewrite (plus_set_sub_compat _ _ (alg_closed_ba_ens A) S h7a).
reflexivity.
intros S' h8.
specialize (h7r (Im S' (fun x => (exist _ _ (Full_intro _ x))))).
destruct h8 as [h8a [h8b h8c]].
simpl in h7r.
assert (h10:exists
           pf : Finite
                  (Im S'
                     (fun x : At =>
                      exist (Ensembles.In (Full_set At)) x (Full_intro At x))),
           (forall x : SubBtype A (ba_ens A),
            Ensembles.In
              (Im S'
                 (fun x0 : At =>
                  exist (Ensembles.In (Full_set At)) x0 (Full_intro At x0)))
              x -> atom (B:=Subalg (ba_ens A) (alg_closed_ba_ens A))  x) /\
           plus_set (B:=Subalg (ba_ens A) (alg_closed_ba_ens A))
             (Im S'
                (fun x : At =>
                 exist (Ensembles.In (Full_set At)) x (Full_intro At x))) pf =
           exist (Ensembles.In (Full_set At)) b (Full_intro At b)).
  exists (finite_image _ _ _ _ h8a).
  split.
  intros x h10.
  destruct h10 as [x h10]. subst.
  specialize (h8b _ h10).
  rewrite h9 in h8b. assumption.
  apply proj1_sig_injective. simpl.
  rewrite <- h8c. 

  assert (h11:S' = Im  (Im S'
           (fun x : At =>
            exist (Ensembles.In (Full_set At)) x (Full_intro At x))) (@proj1_sig _ _)).
    rewrite im_im. simpl.  rewrite im_id at 1. f_equal. 
assert (h12:Finite  (Im
          (Im S'
             (fun x : At =>
              exist (Ensembles.In (Full_set At)) x (Full_intro At x)))
          (proj1_sig (P:=Ensembles.In (Full_set At))))).
  apply finite_image; apply finite_image; auto.
pose proof (subsetT_eq_compat _ _ _ _ h8a h12 h11) as h13.
assert (h14:plus_set S' h8a = plus_set _ h12).
  dependent rewrite <- h13. reflexivity.
rewrite h14. symmetry.
pose proof (plus_set_sub_compat _ _ (alg_closed_ba_ens A) _ 
                              (finite_image At {x | Ensembles.In (Full_set At) x} S'
           (fun x : At =>
            exist (Ensembles.In (Full_set At)) x (Full_intro At x)) h8a) h12) as h15. unfold ba_ens, At in h15. unfold At. simpl in h15. simpl.
rewrite h15 at 1.
reflexivity.
specialize (h7r h10). clear h10.
rewrite h7r.
rewrite im_im. simpl. rewrite im_id. f_equal.
Qed.


Corollary normal_form_gen_ba' : 
  forall (E:Ensemble (bt A)),
    Finite E ->
    ba_ens A = Gen_Ens E ->
    (forall p:At, atom p <->
                  ((p <> 0) /\
                   (exists! (S:Ensemble At),
                      exists (pf:Finite S),
                        p = times_set _ pf /\
                        exists f:Fin_map E signe mns,
                          S = im_eps f)))
    /\
    (forall b:At, exists !(S:Ensemble At),
       exists pf:Finite S,
         (forall x:At, Ensembles.In S x -> atom x) /\
         plus_set S pf = b).
intros E h1 h2.
pose proof (normal_form'' _ h1) as h3.
pose proof (gen_ba_eq_ba _ h2) as h4.
unfold Gen in h3.
pose (fun (C:Ensemble At) (pfc:alg_closed C) =>
         (forall p : Btype (Bc (Subalg C pfc)),
        atom p <->
        p <> 0 /\
        (exists ! S : Ensemble At,
           exists pf : Finite S,
             proj1_sig p = times_set S pf /\
             (exists f : Fin_map E signe mns, S = im_eps f))) /\
       (forall b : Btype (Bc (Subalg C pfc)),
        exists !
          S : Ensemble (Btype (Bc (Subalg C pfc))),
          exists pf : Finite S,
            (forall x : Btype (Bc (Subalg C pfc)),
             Ensembles.In S x -> atom x) /\ plus_set S pf = b)) as P.
assert (h5:P _ (closed_gen_ens A E) <-> P _ (alg_closed_ba_ens A)).
  pose proof (subsetT_eq_compat _ _ _ _ (alg_closed_ba_ens A) (closed_gen_ens A E) h2) as h6.
  dependent rewrite <- h6.
  tauto.
pose proof (full_sig_equivalent_atom A) as h9.
red in h9. simpl in h9.
unfold P in h5. simpl in h5, h3.
pose proof (iff1 h5 h3) as h6. clear h5 h3.
destruct h6 as [h6 h7].  split.
intro p. split.  intro h8.
specialize (h6 (exist _ _ (Full_intro _ p))).
rewrite h9 in h8.
unfold At in h6.
simpl in h6, h8.
rewrite h6 in h8.
destruct h8 as [h8l h8r].
split. intro h10. subst. contradict h8l. apply proj1_sig_injective. simpl.
reflexivity.
assumption.
intro h8. destruct h8 as [h8l h8r].
assert (h10:  (exist _ _ (Full_intro _ p)) <>
       exist (Ensembles.In (Full_set (Btype (Bc A)))) 0
         (Z_c A (Full_set (Btype (Bc A))) (alg_closed_ba_ens A))).
  intro h10. apply exist_injective in h10. contradiction.
assert (h11:p = proj1_sig  (exist (Ensembles.In (Full_set At)) p (Full_intro At p))).
  reflexivity.
pose proof (conj h10 h8r) as h12.
pose proof (h6  (exist (Ensembles.In (Full_set At)) p (Full_intro At p))) as h13. unfold ba_ens in h13.
simpl in h13, h12.
pose proof (iff2 h13 h12) as h14. clear h13 h12.
rewrite h9. assumption.
intros b.
specialize (h7 (exist _ _ (Full_intro _ b))).
destruct h7 as [S h7]. destruct h7 as [h7l h7r].
destruct h7l as [h7a [h7b h7c]].
exists (Im S (@proj1_sig _ _)).
red. split.
exists (finite_image _ _ _ _ h7a).
split. intros x h8.
destruct h8 as [x h8]. subst.
specialize (h7b _ h8).
destruct x as [x h10].
assert (h11:h10 = (Full_intro (Btype (Bc A)) x)). apply proof_irrelevance. subst.
pose proof (iff2 (h9 x) h7b) as h11.
simpl. assumption.
pose proof (f_equal (@proj1_sig _ _) h7c) as h10. simpl in h10.
rewrite <- h10.
rewrite (plus_set_sub_compat _ _ (alg_closed_ba_ens A) S h7a).
reflexivity.
intros S' h8.
specialize (h7r (Im S' (fun x => (exist _ _ (Full_intro _ x))))).
destruct h8 as [h8a [h8b h8c]].
simpl in h7r.
assert (h10:exists
               pf : Finite
                      (Im S'
                          (fun x : At =>
                             exist (Ensembles.In (Full_set At)) x (Full_intro At x))),
           (forall x : SubBtype A (ba_ens A),
              Ensembles.In
                (Im S'
                    (fun x0 : At =>
                       exist (Ensembles.In (Full_set At)) x0 (Full_intro At x0)))
              x -> atom (B:=Subalg (ba_ens A) (alg_closed_ba_ens A)) x) /\
           plus_set (B:=Subalg (ba_ens A) (alg_closed_ba_ens A))
             (Im S'
                 (fun x : At =>
                    exist (Ensembles.In (Full_set At)) x (Full_intro At x))) pf =
           exist (Ensembles.In (Full_set At)) b (Full_intro At b)). 
  exists (finite_image _ _ _ _ h8a).
  split.
  intros x h10.
  destruct h10 as [x h10]. subst.
  specialize (h8b _ h10).
  rewrite h9 in h8b. assumption.
  apply proj1_sig_injective. simpl.
  rewrite <- h8c.
assert (h11:S' = Im  (Im S'
           (fun x : At =>
            exist (Ensembles.In (Full_set At)) x (Full_intro At x))) (@proj1_sig _ _)).
    rewrite im_im. simpl.  rewrite im_id at 1. f_equal.
assert (h12:Finite  (Im
          (Im S'
             (fun x : At =>
              exist (Ensembles.In (Full_set At)) x (Full_intro At x)))
          (proj1_sig (P:=Ensembles.In (Full_set At))))).
  apply finite_image; apply finite_image; auto.
pose proof (subsetT_eq_compat _ _ _ _ h8a h12 h11) as h13.
assert (h14:plus_set S' h8a = plus_set _ h12).
  dependent rewrite <- h13. reflexivity.
rewrite h14. symmetry.
pose proof (plus_set_sub_compat _ _ (alg_closed_ba_ens A) _
                              (finite_image At {x | Ensembles.In (Full_set At) x} S'
           (fun x : At =>
            exist (Ensembles.In (Full_set At)) x (Full_intro At x)) h8a) h12) as h15. unfold ba_ens, At in h15. unfold At. simpl in h15. simpl.
rewrite h15 at 1.
reflexivity.
specialize (h7r h10). clear h10.
rewrite h7r.
rewrite im_im. simpl. rewrite im_id. f_equal.
Qed.



Corollary normal_form_gen_im : 
  forall {B:Bool_Alg} (E:Ensemble (bt B)),
    Finite E ->
    forall (g:sig_set E->At),
      let C := Gen (Im (full_sig E) g) in
      (forall p : (bt C),
       atom p <->
       p <> 0 /\ (exists f : Fin_map E signe mns, 
                    proj1_sig p = el_prod_compose (sig_fun_app g 0) f)) /\
       (forall b : (bt C),
        exists ! S : Ensemble (bt C),
          exists pf : Finite S,
            (forall x : (bt C), Ensembles.In S x -> atom x) /\
            plus_set S pf = b).
intros B E h1 g C.
pose proof h1 as h1'.
rewrite finite_full_sig_iff in h1.
pose proof (finite_image _ _ (full_sig E) g h1) as h5.
pose proof (normal_form _ h5) as h6.
destruct h6 as [h6a h6b].
split; auto.
intro p.
specialize (h6a p).
unfold C. rewrite h6a.
split.
intro h7.
destruct h7 as [h7l h7r]. split; auto.
destruct h7r as [f h7r].
rewrite  h7r.
exists (fin_map_im_full_sig_eq f h1' 0).
rewrite <- (el_prod_el_prod_compose_compat _ h1').
reflexivity.
intro h7.
destruct h7 as [h7l h7r]. split; auto.
destruct h7r as [f h7r].     
rewrite h7r.  
pose proof (ex_same_im_subset_sig_inj h1' g) as h8. 
destruct h8 as [D [h9 [h10 h11]]]. 
destruct h10 as [h10]. destruct h10 as [h10 h12].
rewrite h12.
assert (h9 = h10). apply proof_irrelevance. subst.
pose proof (fin_map_to_fin_map_im_full_sig_comp (restriction f h10) (restriction_sig g D h10) 0 h11) as h8.  
destruct h8 as [F' h8].
exists F'.   
unfold el_prod_compose, el_prod.   
apply times_set_functional. 
rewrite im_im.
apply Extensionality_Ensembles.
red. split.
red. intros x h13.
destruct h13 as [x h13]. subst.
assert (h14:Ensembles.In (Im (full_sig E) g) (g (exist _ _ h13))).
  apply Im_intro with (exist _ _ h13). constructor. reflexivity. 
rewrite h12 in h14.
inversion h14 as [d h15 a h16]. subst.
apply Im_intro with d. constructor.
f_equal.
unfold sig_fun_app. destruct (classic_dec (Ensembles.In E x)) as [h17 | h18].
assert (h18:h17 = h13). apply proof_irrelevance. subst.
assumption. contradiction.
specialize (h8 _ (proj2_sig d)).
simpl in h8. unfold sig_fun_app in h8.
destruct (classic_dec (Ensembles.In D (proj1_sig d))) as [h17 | h18].
destruct d as [d h19]. simpl in h8. simpl in h17. 
assert (h20:h19 = h17). apply proof_irrelevance. subst. 
rewrite h8 at 1.
rewrite restriction_compat.
pose proof (el_prod_compose_le_ai f (sig_fun_app g 0) x h13) as h18.
pose proof (el_prod_compose_le_ai f (sig_fun_app g 0) d (h10 _ h17)) as h19. 
rewrite <- h7r in h18, h19.
unfold eps in h18, h19.
unfold sig_fun_app in h18, h19.
destruct (classic_dec (Ensembles.In E x)) as [h20 | h21].
destruct (classic_dec (Ensembles.In E d)) as [h22 | h23].
assert (h24:h13 = h20). apply proof_irrelevance. subst.
rewrite h16 in h18.
unfold restriction_sig in h18. simpl in h18.
assert (h23:h10 d h17 = h22). apply proof_irrelevance. subst.
destruct (f |-> x), (f |-> d); auto.
pose proof (mono_prod _ _ _ _ h18 h19) as h22.
rewrite comp_prod in h22. rewrite idem_prod in h22.
apply le_x_0 in h22.
contradict h7l.
apply proj1_sig_injective. assumption.
pose proof (mono_prod _ _ _ _ h19 h18) as h22.
rewrite comp_prod in h22. rewrite idem_prod in h22.
apply le_x_0 in h22.
contradict h7l.
apply proj1_sig_injective. assumption.
contradict h23.
apply h10. assumption. contradiction. assumption.
contradict h18. apply proj2_sig.   
 red. intros x h13.  
inversion h13 as [d h14 q h15]. subst. clear h13 h14.
pose proof (h8 _ (proj2_sig d)) as h13.
simpl in h13.
unfold sig_fun_app in h13.
destruct (classic_dec (Ensembles.In D (proj1_sig d))) as [h14 | h15].
destruct d as [d h15]. simpl in h14, h13. 
assert (h14 = h15). apply proof_irrelevance. subst. 
rewrite h13 at 1. unfold restriction_sig.  simpl. 
apply Im_intro with d. apply h10; auto.
f_equal.
unfold sig_fun_app. destruct (classic_dec (Ensembles.In E d)) as [h17 | h18].
f_equal. apply proj1_sig_injective.
reflexivity. contradict h18. apply h10; auto.
rewrite restriction_compat. reflexivity.
assumption.
contradict h15.
apply proj2_sig.
Qed.




(*Move upward or elsewhere*)
Definition atom_set (B:Bool_Alg) := [x:(Btype (Bc B)) | atom x].

Lemma finite_atom_set_fin_gen : 
  forall E:Ensemble At, Finite E -> Finite (atom_set (Gen E)).
intros E h1.
pose proof (normal_form E h1) as h2.
unfold atom_set.
destruct h2 as [h2l h2r].
rewrite (sat_iff _ _ h2l).
assert (h3:Finite [x:Btype (Bc (Gen E)) | 
  (exists f : Fin_map E signe mns, proj1_sig x = el_prod f)]).
  assert (h4: [x : Btype (Bc (Gen E))
     | exists f : Fin_map E signe mns, proj1_sig x = el_prod f] = 
              Im (Full_set (Fin_map E signe mns)) (fun f => (exist _ _ (el_prod_in_gen E f)))).
    apply Extensionality_Ensembles.
    red. split.
    red. intros x h3.
    destruct h3 as [h3]. destruct h3 as [F h3].
    apply Im_intro with F. constructor.
    apply proj1_sig_injective.
    simpl. apply h3.
    red. intros x h3.
    destruct h3 as [F h3]. subst.
    constructor. simpl.
    exists F. reflexivity.
  rewrite h4.
apply finite_image.
apply finite_fin_maps. assumption.
apply signe_finite.
eapply Finite_downward_closed.
apply h3.
red. intros x h4.
destruct h4 as [h4]. destruct h4 as [h4 h5].
constructor.
assumption.
Qed.

Corollary card_atom_set_fin_gen : 
  forall E:Ensemble At, 
    Finite E -> card_fun1 (atom_set (Gen E)) <= 2^(card_fun1 E).
intros E h1.
pose proof (normal_form _ h1) as h2.
unfold atom_set.
destruct h2 as [h2l h2r].
assert (h0: Finite
     (Im (Full_set (Fin_map E signe mns))
        (fun f : Fin_map E signe mns =>
         exist (Ensembles.In (Gen_Ens E)) (el_prod f) (el_prod_in_gen E f)))).
 apply finite_image.
  apply finite_fin_maps. assumption.
  apply signe_finite.
rewrite (sat_iff _ _ h2l).
assert (h4: [x : Btype (Bc (Gen E))
     | exists f : Fin_map E signe mns, proj1_sig x = el_prod f] = 
              Im (Full_set (Fin_map E signe mns)) (fun f => (exist _ _ (el_prod_in_gen E f)))).
    apply Extensionality_Ensembles.
    red. split.
    red. intros x h3.
    destruct h3 as [h3]. destruct h3 as [F h3].
    apply Im_intro with F. constructor.
    apply proj1_sig_injective.
    simpl. apply h3.
    red. intros x h3.
    destruct h3 as [F h3]. subst.
    constructor. simpl.
    exists F. reflexivity.
assert (h3: card_fun1
     [x : Btype (Bc (Gen E))
     | x <> 0 /\ (exists f : Fin_map E signe mns, proj1_sig x = el_prod f)] <= 
            card_fun1
     [x : Btype (Bc (Gen E))
     | exists f : Fin_map E signe mns, proj1_sig x = el_prod f]).
apply incl_card_fun1.  
rewrite h4.
apply h0.
red. intros x h3. destruct h3 as [h3]. destruct h3 as [? h3].
constructor. assumption.
rewrite h4 in h3.
pose proof (finite_cardinal _ _ h0) as h5.
destruct h5 as [n h5].
pose proof signe_finite as h6.
pose proof (card_fin_maps _ _ mns h1 h6) as h7.
rewrite card_signe in h7.
rewrite FiniteT_nat_cardinal_card_fun1_compat in h7.
pose proof (card_fun1_compat (Full_set (Fin_map E signe mns))) as h8.
destruct h8 as [h8l h8r].
specialize (h8l (finite_fin_maps _ _ mns h1 h6)).
rewrite h7 in h8l.
pose proof (cardinal_decreases _ _ _ _ _ h8l _ h5) as h9.
pose proof (card_fun1_compat  (Im (Full_set (Fin_map E signe mns))
            (fun f : Fin_map E signe mns =>
             exist (Ensembles.In (Gen_Ens E)) (el_prod f)
               (el_prod_in_gen E f)))) as h10.
destruct h10 as [h10l h10r].
specialize (h10l h0).
pose proof (cardinal_is_functional _ _ _ h10l _ _ h5 (eq_refl _)).
subst.
clear h10r h5 h8r. 
clear h0 h4 h6 h7. clear h2l h2r. clear h8l h10l.
eapply le_trans. apply h3. apply h9.
Qed.




Corollary card_fin_gen : 
  forall E:Ensemble At, 
    Finite E -> card_fun1 (Gen_Ens E) = 2^(card_fun1 (atom_set (Gen E))).
intros E h1.
pose proof (normal_form _ h1) as h2.
destruct h2 as [h2l h2r].
assert 
  (h3: forall 
         S:Ensemble (Btype (Bc (Gen E))), 
         Ensembles.In (power_set (atom_set (Gen E))) S ->
         Finite S).
  intros S h3.
  destruct h3 as [h3]. eapply Finite_downward_closed; auto.
  apply finite_atom_set_fin_gen. apply h1. apply h3. 
assert (h4:forall (S:Ensemble (Btype (Bc (Gen E)))),
             Ensembles.In (power_set (atom_set (Gen E))) S ->
          exists ! b, 
            Ensembles.In (Full_set (Btype (Bc (Gen E)))) b /\
            exists pf:Finite S, 
              (forall x : Btype (Bc (Gen E)), Ensembles.In S x -> atom x) /\ 
              (plus_set S pf = b)).
intros S h4. exists (plus_set S (h3 S h4)).
  red. split. split. constructor. exists (h3 S h4). split. 
  destruct h4 as [h4]. red in h4.  apply h4. reflexivity.
  intros b h5. destruct h5 as [h5 h6]. subst. destruct h6 as [h6 h7]. destruct h7; subst.
  apply plus_set_functional. reflexivity.
assert (h5:forall b:Btype (Bc (Gen E)), 
             Ensembles.In (Full_set (Btype (Bc (Gen E)))) b ->
           exists ! S:Ensemble (Btype (Bc (Gen E))),
             Ensembles.In (power_set (atom_set (Gen E))) S /\
             exists pf : Finite S,
               (forall x : Btype (Bc (Gen E)), Ensembles.In S x -> atom x) /\
           plus_set S pf = b).
  intros b ?. specialize (h2r b). destruct h2r as [S h2r].
  red in h2r. destruct h2r as [h2a h2b]. destruct h2a as [h2c h2d].
  exists S. red. split. split. constructor. red. 
  destruct h2d as [h2d ?]. intros x h5.  constructor. apply h2d; auto. 
   exists h2c. subst. assumption.
  intros x h5. destruct h5 as [? h5]. apply h2b; auto. 
clear h2r.
pose proof (finite_atom_set_fin_gen _ h1) as h6.
pose proof (power_set_finite _ h6) as h7.
pose proof (bij_ex_impl_eq_card' _ _ h7 _ h4 h5) as h8.
rewrite card_power_set in h8.
rewrite h8.
simpl.
destruct (classic_dec (Finite (Gen_Ens E))) as [h9 | h10]. 
pose proof h9 as h9'.
apply Finite_ens_type in h9. 
unfold SubBtype. unfold sig_set in h9.
rewrite <- FiniteT_nat_cardinal_card_fun1_compat with h9.
rewrite FiniteT_nat_cardinal_card_fun1_compat.
pose proof (card_fun_full_sig_eq _ h9') as h10.
do 2 rewrite card_fun_card_fun1_compat in h10.
assumption.
pose proof (card_fun1_compat (Gen_Ens E)) as h11.
destruct h11 as [h11l h11r].
specialize (h11r h10). rewrite h11r.
pose proof (card_fun1_compat  (Full_set (SubBtype A (Gen_Ens E)))) as h12.
destruct h12 as [h12l h12r].
assert (h13: ~ Finite (Full_set (SubBtype A (Gen_Ens E))) ).
  rewrite Finite_FiniteT_iff.
  intro h13. apply FiniteT_sig_Finite in h13. contradiction.
rewrite h12r; auto.
assumption.
Qed.


Corollary finite_fin_gen : 
  forall E:Ensemble At, Finite E -> Finite (Gen_Ens E).
intros E h1.
pose proof (card_fin_gen _ h1) as h2.
apply NNPP.
intro h3.
pose proof (card_fun1_compat (Gen_Ens E)) as h4.
destruct h4 as [h4l h4r].
specialize (h4r h3).
rewrite h4r in h2.
assert (h5:0 < 2). omega.
pose proof (O_lt_pow _ (card_fun1 (atom_set (Gen E))) h5).
omega.
Qed.

Corollary card_fin_gen_le : 
  forall E:Ensemble At, 
    Finite E -> card_fun1 (Gen_Ens E) <= 2 ^ (2 ^ (card_fun1 E)).
intros E h1.
pose proof (card_atom_set_fin_gen _ h1) as h2.
rewrite card_fin_gen; auto.
apply pow_mono. auto with arith.
assumption.
Qed.

Lemma incl_comp_set_closed : 
  forall {B:Bool_Alg} (D E:Ensemble (Btype (Bc B))),
    alg_closed E ->
    Included D E ->
    Included (comp_set D) E.
intros B D E h1 hi. red. intros x h2.
destruct h2 as [x h2]. subst.
apply comp_closed; auto.
Qed.


Definition comp_add_set {B:Bool_Alg} (A:Ensemble (Btype (Bc B))) :=
  Union A (Im A (Bcomp (Bc B))). 

Lemma incl_comp_add_set : 
  forall {B:Bool_Alg} (D:Ensemble (Btype (Bc B))),
    Included D (comp_add_set D).
intros B D. unfold comp_add_set. auto with sets.
Qed.

Lemma incl_comp_add_set_closed : 
  forall {B:Bool_Alg} (D E:Ensemble (Btype (Bc B))),
    alg_closed E ->
    Included D (comp_add_set E) ->
    Included D E.
intros B D E h1 h2. 
red. red in h2.
intros x h3. specialize (h2 _ h3).
destruct h2 as [x h2l | x h2r]. assumption.
destruct h2r as [x h4]. subst.
apply comp_closed; auto.
Qed.


Lemma comp_add_set_add : 
  forall {B:Bool_Alg} (D:Ensemble (Btype (Bc B))) (a:Btype (Bc B)),
    comp_add_set (Add D a) = Union (comp_add_set D) (Couple a (-a)).
intros B D a.
apply Extensionality_Ensembles.
red. split.
red. intros x h1.
destruct h1 as [x h1 | x h2].
destruct h1 as [x h1 | x h3].
left. left. assumption.
destruct h3; subst.
right. left.
destruct h2 as [x h2]. subst.
destruct h2 as [x h2l | x h2r].
left. right. econstructor. apply h2l.
reflexivity.
destruct h2r; subst.
right. right.
red. intros x h1.
destruct h1 as [x h1 | x h2].
destruct h1 as [x h1 | x h3].
constructor. left. assumption.
destruct h3 as [x h3]. subst.
right. econstructor. left. apply h3. reflexivity.
destruct h2. left. right. constructor.
right. econstructor. right. constructor.
reflexivity.
Qed.


Lemma comp_add_set_preserves_inclusion :
  forall {B:Bool_Alg} (D E:Ensemble (Btype (Bc B))),
    Included D E -> Included (comp_add_set D) (comp_add_set E).
intros B D E h1.
red.
intros x h2.
destruct h2 as [x h2l |x h2r].
left. apply (h1 _ h2l).
right.
destruct h2r as [x h2]. subst.
apply Im_intro with x; auto.
Qed.


Lemma im_eps_fin_map_incl_comp_add_set : 
  forall {D:Ensemble At} (f:Fin_map D signe mns),
    Included (Im D (fun i => eps i (f |-> i))) (comp_add_set D).
intros D f.
red. intros x h1.
destruct h1 as [x h1]. subst.
destruct (f |-> x).
simpl. constructor. assumption.
simpl. unfold comp_add_set. right.
apply Im_intro with x; auto.
Qed.

Lemma gen_ens_comp_add_set : 
  forall {B:Bool_Alg} (E:Ensemble (Btype (Bc B))),
    Gen_Ens (comp_add_set E) = Gen_Ens E.
intros B E.
pose proof (closed_gen_ens _ E) as h0.
pose proof (gen_ens_includes _ E) as hin.
apply Extensionality_Ensembles.
red. split.
red.
intros x h1. 
destruct h1 as [x h1].
apply (h1 (Gen_Ens E)).
constructor. split.
red.
intros b h2.
destruct h2 as [b h2l | b h2r].
apply hin. assumption.
destruct h2r as [b h2r]. subst.
apply comp_closed; auto. assumption.
red.
intros x h1.
pose proof (incl_comp_add_set E) as h2.
pose proof (gen_ens_preserves_inclusion _ _ h2) as h3.
auto with sets.
Qed.

(*This theorem says that every element in a  
  subalgebra generated by E can be written as a finite join of 
  finite meets of elements and complements of elements
  from E*)    
Theorem gen_ens_eq : 
  forall (E:Ensemble At), 
    Gen_Ens E = [x : At | exists (S:Ensemble At) (pfs:Finite S), 
                          x = plus_set S pfs /\
                          forall s:At, Ensembles.In S s ->
                                       exists (R:Ensemble At) 
                                        (pfr:Finite R), 
                                         Included R 
                                                  (comp_add_set E) /\
                                         s = times_set R pfr].
intro E.
pose (fun (F : Ensemble At) (pf: Finite F) (pfinc:Included F E) => Gen_Ens F) as BF_set.
pose (fun (F: Ensemble At) (pf:Finite F) => (non_zero_el_prod_maps F)) as non_zero_el_prod_maps_f.
assert (hfin: forall (F:Ensemble At) (pf:Finite F), Finite (
                                      non_zero_el_prod_maps F)). 
  intros. apply non_zero_el_prod_maps_fin; auto.
assert (h1:forall (F:Ensemble At) (pf:Finite F) (pfinc:Included F E), 
             BF_set F pf pfinc = [a:At | exists X (pfinc2:Included X (non_zero_el_prod_maps_f F pf)), a = plus_set (Im X el_prod) (finite_image _ _ _ el_prod (Finite_downward_closed _ _ (hfin F pf) _ pfinc2))]).
  intros F h1 h2.
  pose proof (normal_form' _ h1) as h3.
  unfold BF_set.
  destruct h3 as [h3l h3r].
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h4.
  specialize (h3r (exist _ _ h4)).
  destruct h3r as [X h3r].
  red in h3r. destruct h3r as [h3a h3b].
  destruct h3a as [h3a h3c].
  constructor.
  exists X. exists h3a.
  simpl in h3c. rewrite <- h3c.
  unfold plus_subset_non_zero_el_prod_maps.
  apply plus_set_functional. reflexivity.
  red. intros x h4. destruct h4 as [h4]. destruct h4 as [S h5].
  destruct h5 as [h5 h6].
assert (h7:forall f, Ensembles.In S f -> Ensembles.In (Gen_Ens F) (el_prod f)).
  intros f h7.
  apply el_prod_in_gen; auto. 
rewrite h6.
pose proof (closed_gen_ens _ F) as h8.
apply plus_set_closed. assumption.
red.
intros x' h9.
destruct h9 as [G h9]. subst.
apply h7. assumption.

simpl in BF_set.
pose proof (family_included_subs_directed_eq_B' E) as h2. 
destruct h2 as [h2l h2r]. 
rewrite <- h2r. 

apply Extensionality_Ensembles.
red. split. 
red. intros C h3.
destruct h3 as [C x h4 h5].
destruct h4 as [h4]. destruct h4 as [F h4]. destruct h4 as [h4a [h4b h4c]]. subst.
pose proof (normal_form _ h4c) as h6.
destruct h6 as [h6l h6r]. 
destruct (h6r (exist _ _ h5)) as [S h6b].
destruct h6b as [h6a h6b].
destruct h6a as [h6a h6c].
destruct h6c as [h6c h6d].
constructor. 
pose proof (f_equal (@proj1_sig _ _) h6d) as h7.
simpl in h7.
pose proof (plus_set_sub_compat _ _ _ S h6a (finite_image _ _ S (@proj1_sig _ _) h6a)) as h8.
simpl in h7. simpl in h8. 
unfold Gen in h7. rewrite h7 in h8.
exists  (Im S
               (proj1_sig
                  (P:=fun a : Btype (Bc A) => Ensembles.In (Gen_Ens F) a))).
exists (finite_image _ _ S (@proj1_sig _ _) h6a).
rewrite <- h8.
split. 
reflexivity. 
intros a h9.
destruct h9 as [a h9]. subst. 
specialize (h6c _ h9).
rewrite h6l in h6c.
destruct h6c as [h10 h11].
destruct h11 as [f h11].
unfold el_prod in h11.
exists  (Im F (fun i : At => eps i (f |-> i))).
exists (finite_image _ _ _ _ h4c).
split.
pose proof (im_eps_fin_map_incl_comp_add_set f) as h12.
pose proof (comp_add_set_preserves_inclusion _ _ h4b) as h13.
auto with sets.
unfold At in h11. unfold At.
assert (h12:h4c = fin_map_fin_dom f). apply proof_irrelevance. subst.
assumption.
red. intros x h3.   
destruct h3 as [h3]. destruct h3 as [S h3]. destruct h3 as [h3 h4].
rewrite h2r.
destruct h4 as [h4 h5]. subst. 
assert (h6:Included S (Gen_Ens E)).
  red. intros x h6. specialize (h5 _ h6).
  destruct h5 as [R h5]. destruct h5 as [h5 h7].
  destruct h7 as [h7l h7r]. subst.
  assert (h8:Ensembles.In (Gen_Ens R) (times_set R h5)).
  pose proof (closed_gen_ens _ R) as h7.
  pose proof (gen_ens_includes _ R) as h8'.
  pose proof (times_set_closed _ R (Gen_Ens R) h5 h7 h8') as h10.
  assumption.
  pose proof (gen_ens_preserves_inclusion _ _ h7l) as h9.
  rewrite gen_ens_comp_add_set in h9.
  auto with sets.
pose proof (closed_gen_ens _ E) as h7.
apply plus_set_closed; auto.
Qed.
End NormalForm.

Lemma gen_ens_subalg_eq : 
  forall {B:Bool_Alg} {C:Ensemble (bt B)} (pf:alg_closed C),
    let D:=Subalg C pf in 
    forall (E:Ensemble (bt D)),
      im_proj1_sig (Gen_Ens E) = Gen_Ens (im_proj1_sig E).
intros B C h0 D E.
do 2 rewrite gen_ens_eq.
apply Extensionality_Ensembles.
red. split.
red. intros x h1.
destruct h1 as [x h1]. subst.
destruct h1 as [h1]. destruct h1 as [S [h2 [h3 h4]]]. subst.
constructor.
exists (im_proj1_sig S).
exists (finite_image _ _ _ _ h2). 
split.
rewrite (plus_set_sub_compat B C h0 S h2).
reflexivity.
intros b h5. destruct h5 as [b h5]. subst. destruct b as [b h6].
simpl.
specialize (h4 _ h5).
destruct h4 as [R [h4a [h4b h4c]]].
exists (im_proj1_sig R).
exists (finite_image _ _ _ _ h4a). split.
red. red in h4b. intros x h7. destruct h7 as [x h7]. subst.
destruct x as [x h8]. simpl. specialize (h4b _ h7).
inversion h4b as [y h9 | y h10]; subst. clear h4b.
left. apply Im_intro with  (exist (fun x0 : Btype (Bc B) => Ensembles.In C x0) x h8); auto.
right. 
inversion h10 as [y h11 ? h12]. subst.
destruct y as [y h13].
unfold im_proj1_sig.
rewrite im_im.
apply Im_intro with  (exist (fun a : Btype (Bc B) => Ensembles.In C a) y h13); auto.
simpl.
simpl in h12.
apply exist_injective in h12. subst.
unfold Bcomp_sub. simpl.
reflexivity.
rewrite (times_set_sub_compat B C h0 R h4a).
pose proof (f_equal (@proj1_sig _ _) h4c) as h7.
simpl in h7.
assumption. 
red. intros x h1.

destruct h1 as [h1]. destruct h1 as [S [h2 [h3 h4]]].  subst.  
assert (h5:Included [x:(bt B) | 
                     exists (S0 : Ensemble (bt B))
                            (pfs:Finite S0),
                     x = plus_set S0 pfs /\
                     forall (s:(bt B)),
                               Ensembles.In S0 s ->
                               exists (R:Ensemble (bt B))
                                      (pfr:Finite R),
                                 Included R (comp_add_set (im_proj1_sig E)) /\ s = times_set R pfr] C).
  red. intros x h5.
  destruct h5 as [h5]. destruct h5 as [R [h5 [h6 h7]]].
  subst.
  apply plus_set_closed; auto.
  red. intros x h8. specialize (h7 _ h8).
  destruct h7 as [R' [h9 [h10 h11]]]. subst.
  apply times_set_closed; auto.
  intros r h11. specialize (h10 _ h11).
  destruct h10 as [r h10 | r h12].
  destruct h10 as [r h12 ? h13]. subst. apply proj2_sig.
  destruct h12 as [r h13 ? h14]. subst. apply comp_closed; auto.
  destruct h13 as [r h13]. subst. apply proj2_sig. 
assert (h6: [x : Btype (Bc D)
        | exists (S0 : Ensemble (Btype (Bc D))) (pfs : Finite S0),
            x = plus_set S0 pfs /\
            (forall s : Btype (Bc D),
             Ensembles.In S0 s ->
             exists (R : Ensemble (Btype (Bc D))) (pfr : Finite R),
               Included R (comp_add_set E) /\ s = times_set R pfr)] = im_proj2_sig _ h5).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h6. destruct x as [x h7].  
  assert (h8:Ensembles.In  [x0 : bt B
        | exists (S0 : Ensemble (bt B)) (pfs : Finite S0),
            x0 = plus_set S0 pfs /\
            (forall s : bt B,
             Ensembles.In S0 s ->
             exists (R : Ensemble (bt B)) (pfr : Finite R),
               Included R (comp_add_set (im_proj1_sig E)) /\
               s = times_set R pfr)] x).
    constructor.
    destruct h6 as [h6]. destruct h6 as [R [h6 [h8 h9]]]. 
    exists (im_proj1_sig R). exists (finite_image _ _ _ _ h6).
    rewrite (plus_set_sub_compat B C h0 R h6). split.
    pose proof (f_equal (@proj1_sig _ _) h8) as h10. simpl in  h10.
    assumption.
    intros s h10. destruct h10 as [s h10]. subst.
    specialize (h9 _ h10).
    destruct h9 as [R' [h11 [h12 h13]]]. subst.
    exists (im_proj1_sig R'). exists (finite_image _ _ _ _ h11).
    split.
    red. intros r h13. destruct h13 as [r h13]. subst.
    specialize (h12 _ h13).
    destruct h12 as [r h12a | r h12b]. left. apply Im_intro with r; auto.
    destruct h12b as [r h12b]. subst.
    right. unfold im_proj1_sig.  rewrite im_im. 
    apply Im_intro with r; auto.
    rewrite (times_set_sub_compat B C h0 R' h11).
    reflexivity.
  apply Im_intro with (exist _ _ h8).
  constructor. simpl.
  apply proj1_sig_injective. simpl. reflexivity.  
  red. intros x h6.   destruct h6 as [x h6]. subst. clear h6.
  destruct x as [x h6]. simpl. 
  destruct h6 as [h6]. destruct h6 as [R [h6 [h7 h8]]]. subst.
  constructor. 
  assert (h9:Included R C).
    red. intros r h9. specialize (h8 _ h9).
    destruct h8 as [R' [h10 [h11 h12]]]. subst.
    apply times_set_closed; auto.
    red. intros r h13. specialize (h11 _ h13).
    destruct h11 as [r h14 | r h15].
    destruct h14 as [r h14]. subst. apply proj2_sig.
    destruct h15 as [r h15]. subst. destruct h15 as [r h15]. subst.
    apply comp_closed; auto. apply proj2_sig.
  exists (im_proj2_sig _ h9).
  assert (h10:Finite (im_proj2_sig R h9)).
    eapply finite_image. rewrite <- finite_full_sig_iff.
    assumption.
  exists h10. split.
  apply proj1_sig_injective. simpl. 
  pose proof (plus_set_sub_compat B C h0 (im_proj2_sig R h9) h10 (finite_image _ _ _ _ h10)) as h11. 
  unfold D.
  rewrite <- h11.
  apply plus_set_functional.
  unfold im_proj2_sig. rewrite im_im. simpl.
  rewrite <- im_full_sig_proj1_sig.
  reflexivity. 
  intros s h11.
  destruct s as [s h12]. 
  inversion h11 as [s' h13 ? h14]. subst.
  apply exist_injective in h14. subst. clear h11.
  destruct s' as [s' h14].  simpl in h12.
  specialize (h8 _ h14). simpl.
  destruct h8 as [R' [h15 [h16 h17]]]. subst. 
  assert (h17:Included R' C).
    red. intros x h18.
    specialize (h16 _ h18).
    destruct h16 as [x h16a | x h16b].
    destruct h16a as [x h16a]. subst. apply proj2_sig.
    destruct h16b as [x h16b]. subst. destruct h16b as [x h16b]. 
    subst. apply comp_closed; auto. apply proj2_sig.
  exists (im_proj2_sig _ h17).
  assert (h18:Finite (im_proj2_sig R' h17)).
    eapply finite_image. rewrite <- finite_full_sig_iff.
    assumption.
  exists h18. split.
  red. intros x h19.
  destruct h19 as [x h19]. subst. clear h19.
  destruct x as [x h19]. simpl.
  specialize (h16 _ h19).
  destruct h16 as [x h16a | x h16b]. 
  destruct h16a as [x h16a]. subst. left.
  rewrite unfold_sig in h16a.
  assert (h20:h17 (proj1_sig x) h19 = proj2_sig x). apply proof_irrelevance.
  subst. rewrite <- h20 in h16a at 1.
  assumption.
  destruct h16b as [x h16b]. subst. destruct h16b as [x h16b].
  subst. 
  right. apply Im_intro with x. assumption.
  apply proj1_sig_injective. simpl.
  unfold Bcomp_sub. reflexivity.  
  apply proj1_sig_injective. simpl.
  pose proof (times_set_sub_compat B C h0 (im_proj2_sig R' h17) h18 (finite_image _ _ _ _ h18)) as h19.
  unfold D. rewrite <- h19.
  apply times_set_functional.
  unfold im_proj2_sig. rewrite im_im. simpl.
  rewrite <- im_full_sig_proj1_sig.
  reflexivity.
rewrite h6 at 1.
rewrite <- im_proj1_sig_undoes_im_proj2_sig at 1.
constructor.
exists S. exists h2. split; auto.
Qed.



Section NormalForm'.
Variable A:Bool_Alg.
Let At := Btype (Bc A).

Lemma gen_ens_closed_add_plus_set_aux : 
  forall (B:Ensemble At) (r:At), 
    alg_closed B ->
    forall (D:Ensemble At)
         (pf:Finite D), Included D  [d : At
         | exists p q : At,
             Ensembles.In B p /\ Ensembles.In B q /\ d = p * r + q * - r] ->
     exists p q : At,
     Ensembles.In B p /\ Ensembles.In B q /\ plus_set D pf = p * r + q * - r.
intros B r h1 D h2. 
pose proof (finite_set_list_no_dup _ h2) as h3.
destruct h3 as [l h3]. destruct h3 as [h3 h4]. 
revert h4 h3. revert h2. revert D r h1. revert B.
induction l as [|a l h5].
intros B D r h1 h2 h4 h3. simpl in h3. subst. intros. exists 0. exists 0. split.
apply zero_closed; auto. split. apply zero_closed; auto.
rewrite plus_set_empty'. rewrite comm_prod. rewrite zero_prod.
rewrite comm_prod. rewrite zero_prod. rewrite zero_sum. reflexivity.
intros B D r h2 h3 h4 h1. 
pose proof (no_dup_cons _ _ h4) as h7.
pose proof (no_dup_cons_nin _ _ h4) as h8.
pose proof (subtract_preserves_finite _ a h3) as h9.
specialize (h5 B (Subtract D a) r h2 h9 h7).
subst. simpl in h5.
rewrite list_to_set_in_iff in h8. 
pose proof (sub_add_compat_nin _ _ h8) as h10.
specialize (h5 h10).
simpl in h9. simpl in h3. simpl.
assert (h11:Finite (list_to_set l)). eapply Finite_downward_closed. apply h3.
  auto with sets.
pose proof (subsetT_eq_compat _ _ _ _ h9 h11 h10) as h12.
dependent rewrite -> h12 in h5.
intros h13.
assert (h14:Included (list_to_set l)  [d : At
          | exists p q : At,
              Ensembles.In B p /\ Ensembles.In B q /\ d = p * r + q * - r]).
  auto with sets.
specialize (h5 h14).
destruct h5 as [p h5]. destruct h5 as [q h5]. destruct h5 as [h5l [h5r h5c]].
specialize (h13 a). 
assert (h15:Ensembles.In (Add (list_to_set l) a) a). right. constructor.
specialize (h13 h15).
destruct h13 as [h13]. destruct h13 as [p' h13]. destruct h13 as [q' h13].
destruct h13 as [h13a [h13b h13c]].
assert (h16:plus_set  (Add (list_to_set l) a) h3 = a + (plus_set (list_to_set l) h11)).
  pose proof (plus_set_add' (list_to_set l) h11 a h3) as h16.
  unfold At in h16. unfold At. rewrite h16 at 1. reflexivity.
rewrite h13c in h16 at 2. rewrite h5c in h16.
rewrite <- assoc_sum in h16.
rewrite  (assoc_sum _ (q' * -r) (p*r) (q*-r)) in h16.
rewrite  (comm_sum _ (q' * -r) (p*r)) in h16.
rewrite <- (assoc_sum _ (p*r) (q' * -r) (q * -r)) in h16.
rewrite <- (dist_sum_r  q' q (-r)) in h16.
rewrite assoc_sum in h16.
rewrite <- dist_sum_r in h16.
exists (p' + p). exists (q' + q).
split. apply plus_closed; auto. split. apply plus_closed; auto.
assumption.
Qed.


Lemma gen_ens_closed_add : 
  forall B:Ensemble At, 
    alg_closed B -> forall r:At, 
                    Gen_Ens (Add B r) = 
                              [x:At | exists p q,
                                      Ensembles.In B p /\
                                      Ensembles.In B q /\
                                      x = p*r + q*-r].
intros B h1 r.
pose proof (gen_ens_eq _ (Add B r)) as h2.
assert (h3':forall (C:Ensemble At)
             (pf:Finite C),
             Included C (comp_add_set B) -> 
             Ensembles.In B (times_set C pf)).
  intros C h3 h4.
  apply times_set_closed; auto.
  red.
  intros x h5.
  specialize (h4 _ h5).
  destruct h4 as [x h4l | x h4r].
  assumption.
  destruct h4r as [x h4r]. subst.
  apply comp_closed; auto. 
assert (heq:Gen_Ens (Add B r) = 
            [x:At | exists D (pf:Finite D),
                     x = plus_set _ pf /\
                     (Included D (Union B (Union [a:At | exists p, Ensembles.In B p /\ a = p * r] 
                                                 [a:At | exists q, Ensembles.In B q /\ a = q * -r])))]).
rewrite h2.   
apply Extensionality_Ensembles.
red. split.
red. intros x h3.   
constructor.
destruct h3 as [h3]. destruct h3 as [S h3]. destruct h3 as [h3 h4]. 
destruct h4 as [h4 h5].
exists S. exists h3.  split; auto.
red.
intros s h6.
specialize (h5 _ h6).
destruct h5 as [R h5]. destruct h5 as [h5 h7]. destruct h7 as [h7 h8].
subst.
pose proof (comp_add_set_add B r) as h8. unfold At in h8. unfold At in h7.  simpl in h8. simpl in h7. rewrite h8 in h7.  
apply incl_union_setminus in h7.
pose proof (decompose_int_setminus R (Couple r (-r))) as h9.
assert (h10:Included (Intersection R (Couple r (-r))) R).
  auto with sets.
pose proof (Finite_downward_closed _ _ h5 _ h10) as h11.
assert (h12:Included (Setminus R (Couple r (-r))) R). 
  apply setminus_inc.
pose proof (Finite_downward_closed _ _ h5 _ h12) as h13.
pose proof (times_set_union _ _ h11 h13) as h14.
pose proof (subsetT_eq_compat _ _ _ _ h5  
  (Union_preserves_Finite (Btype (Bc A))
             (Intersection R (Couple r (- r))) (Setminus R (Couple r (- r)))
             h11 h13) h9) as h15.
dependent rewrite -> h15.
unfold At in h14. unfold At. rewrite h14 at 1.
assert (h16:Included (Intersection R (Couple r (-r))) (Couple r (-r))). auto with sets.
apply incl_couple_inv in h16.
destruct h16 as [h16a | [h16b | [h16c | h16d]]].
pose proof (Empty_is_finite At) as h17.
pose proof (subsetT_eq_compat _ _ _ _ h11 h17 h16a) as h18.
unfold At in h18. unfold At. dependent rewrite -> h18. 
rewrite times_set_empty'.
rewrite comm_prod. rewrite one_prod.
left. apply times_set_closed; auto. apply incl_comp_add_set_closed; auto with sets.
pose proof (Singleton_is_finite _ r) as h17.
pose proof (subsetT_eq_compat _ _ _ _ h11 h17 h16b) as h18.
unfold At in h18. unfold At. dependent rewrite -> h18.
rewrite times_set_sing'. rewrite comm_prod.
right. left. constructor.
exists (times_set (Setminus R (Couple r (- r))) h13); auto.
pose proof (Singleton_is_finite _ (-r)) as h17.
pose proof (subsetT_eq_compat _ _ _ _ h11 h17 h16c) as h18.
unfold At in h18. unfold At. dependent rewrite -> h18.
rewrite times_set_sing'. rewrite comm_prod.
right. right. constructor.
exists (times_set (Setminus R (Couple r (- r))) h13); auto.
pose proof (finite_couple r (-r)) as h17.
pose proof (subsetT_eq_compat _ _ _ _ h11 h17 h16d) as h18.
unfold At in h18. unfold At. dependent rewrite -> h18.
rewrite times_set_couple'.
rewrite comp_prod. rewrite comm_prod. rewrite zero_prod.
left. apply zero_closed; auto.
red. intros x h4.
destruct h4 as [h4]. destruct h4 as [D h4]. destruct h4 as [h4 h5].
destruct h5 as [h5l h5r].
constructor. exists D. exists h4. split; auto.
intros d h6.
apply h5r in h6.
destruct h6 as [d h6a | d h6b].
exists (Singleton d). exists (Singleton_is_finite _ d).
split.
red. intros d' h7. destruct h7. left. left. assumption.
rewrite times_set_sing'. reflexivity.
destruct h6b as [d h6b | d h6c].
destruct h6b as [h6b]. destruct h6b as [p h6b].
destruct h6b as [h7  h8].
exists (Couple p r). exists (finite_couple p r).
split. red.
intros a h9.
destruct h9. left. left. assumption. left. right. constructor.
subst. rewrite times_set_couple'. reflexivity.
destruct h6c as [h6c].
destruct h6c as [q h6c]. destruct h6c as [h7  h8].
exists (Couple q (-r)). exists (finite_couple q (-r)).
split. red. intros a h9. destruct h9. left. left. assumption.
right. apply Im_intro with r. right. constructor. reflexivity.
subst. rewrite times_set_couple'. reflexivity.
assert (h4:Included (Union B
                 (Union
                    [a : At | exists p : At, Ensembles.In B p /\ a = p * r]
                    [a : At | exists q : At, Ensembles.In B q /\ a = q * - r]))  [x : At
   | exists p q : At,
       Ensembles.In B p /\ Ensembles.In B q /\ x = p * r + q * - r]).
  red. intros x h4.
  destruct h4 as [x h4a | x h4b].
  constructor.
  exists x. exists x. repeat split; auto.
  rewrite <- dist_sum. rewrite comp_sum. rewrite one_prod.
  reflexivity.
  destruct h4b as [x h4b | x h4c].
  destruct h4b as [h4b]. destruct h4b as [a h4b].
  destruct h4b as [h5 h6].
  constructor. exists a. exists 0. split; auto.
  split. apply zero_closed; auto. rewrite (comm_prod _ 0 (-r)).
  rewrite zero_prod. rewrite zero_sum. assumption.
  destruct h4c as [h4c]. destruct h4c as [a h4c].
  destruct h4c as [h5 h6]. constructor.
  exists 0. exists a. repeat split; auto. apply zero_closed; auto.
  rewrite (comm_prod _ 0 r). rewrite zero_prod. rewrite comm_sum.
  rewrite zero_sum. assumption. 
assert (h5:Included (Gen_Ens (Add B r))
           [x : At | 
            exists (D : Ensemble (Btype (Bc A))) (pf : Finite D),
             x = plus_set D pf /\
             Included D [d:At |
             exists p q : At,
                 Ensembles.In B p /\ Ensembles.In B q /\ d = p * r + q * - r]]).
rewrite heq.
    red. intros x h5. destruct h5 as [h5]. destruct h5 as [D h5].
    destruct h5 as [h5 h6]. destruct h6 as [h6a h6b].
    constructor. exists D. exists h5. split; auto.
    intros d h7.
    assert (h8:Ensembles.In  [x : At
         | exists p q : At,
             Ensembles.In B p /\ Ensembles.In B q /\ x = p * r + q * - r] d). auto with sets.
    destruct h8 as [h8]. constructor. assumption. 
apply Extensionality_Ensembles.
red. split. Focus 2.
red. intros x h6. destruct h6 as [h6]. destruct h6 as [p h6]. destruct h6 as [q h6].
destruct h6 as [h6a [h6b h6c]].
subst. apply plus_closed. apply closed_gen_ens.
apply times_closed. apply closed_gen_ens. apply gen_ens_includes.
left. assumption. apply gen_ens_includes. right. constructor.
apply times_closed. apply closed_gen_ens. apply gen_ens_includes.
left. assumption. apply comp_closed. apply closed_gen_ens. 
apply gen_ens_includes. right. constructor.
red. intros x h6. apply h5 in h6. destruct h6 as [h6].
destruct h6 as [D h6]. destruct h6 as [h6 h7]. destruct h7 as [h7 h8].
constructor. subst.
apply gen_ens_closed_add_plus_set_aux; auto.
Qed.

(*Temporarily? ommitting the duplicate proof of gen_ens_closed_add from the text that
  doesn't use gen_ens_eq*)

                         

End NormalForm'.

Arguments eps_map [A] [E] _ _.
Arguments eps_map_compose [A] [B] [E] _ _ _.






Section CompleteAndRegularSubalgebras.
Variable B:Bool_Alg.
Let Bt := Btype (Bc B).

Inductive complete_subalg (A:Ensemble Bt) : Prop :=
  | complete_subalg_intro : 
      alg_closed A -> forall (pf:set_complete B), 
      (forall (C:Ensemble Bt), Included C A -> 
                               Ensembles.In A
                                            (sup_set_complete C pf)) -> complete_subalg A. 

Lemma complete_subalg_complete_supalg :
  forall (A:Ensemble Bt), complete_subalg A ->
                          set_complete B.
intros A h1.
destruct h1; auto.
Qed.


Lemma complete_subalg_cont_inf : 
  forall (A:Ensemble Bt) 
         (pf:complete_subalg A),
  forall (C:Ensemble Bt) (p:Bt), 
    Included C A -> Ensembles.In A (inf_set_complete C 
                                                     (complete_subalg_complete_supalg _ pf)).
intros A h1 C p h2.
destruct h1 as [h3 h4 h5]. 
pose proof (incl_comp_set_closed  _ _ h3 h2) as h6.
pose proof (h5 _ h6) as h7.
pose proof (sup_set_complete_compat (comp_set C) h4) as h8.
rewrite doub_neg in h8.
rewrite <-inf_sup_compat_iff in h8. 
unfold inf_set_complete.
destruct Description.constructive_definite_description as [q h10].
simpl.
pose proof (inf_unq _ _ _ h8 h10). subst.
apply comp_closed; auto.
Qed.

Inductive regular_subalg (A:Ensemble Bt) : Prop :=
  | regular_subalg_intro : 
      (forall (pfc:alg_closed A) (C:Ensemble Bt) (pfi:Included C A), 
                     (forall (p:Bt) (pfp:Ensembles.In A p),
                        sup (B:=Subalg _ pfc) (subset_sig _ _ pfi) (exist _ _ pfp) ->
                                                                                                     sup C p)) -> regular_subalg A.

(*Perhaps do -- complete implies regular*)

Lemma regular_subalg_iff :
  forall (A:Ensemble Bt) (pfc:alg_closed A),
    regular_subalg A <-> (forall (E:Ensemble Bt) (pfi:Included E A),
                            sup (B:=Subalg _ pfc) (subset_sig _ _ pfi)
                         (exist _ _ (one_closed _ _ pfc)) ->
                     sup E 1).
intros A h1. 
split.
intros h2.
destruct h2 as [h2].
intros E h5 h6.
specialize (h2 h1 _ h5 1 (one_closed B A h1) h6).
assumption.
intros h2.
constructor. 
intros hs E0 h3 p h4 h5.
pose (Subalg _ hs) as A'.
assert (h6:Included (Add E0 (-p)) A).
  red. intros x h6.
  destruct h6 as [x h6l | x h6r].
  auto with sets. destruct h6r. apply comp_closed; auto.
assert (h7:sup (B:=A') (subset_sig _ _ h6) (exist _ _ (one_closed _ _ h1))). 
  pose proof (comp_sum A' (exist _ _ h4)) as h7.
  pose proof (@sup_add_assoc  A' (subset_sig _ _ h3) ((Bcomp (Bc A') (exist _ _ h4))) _ h5) as h8.
  rewrite h7 in h8.
  simpl in h8. 
  assert (h9: (Add (subset_sig E0 A h3)
            (exist (Ensembles.In A)
               (Bcomp_sub B A (exist (Ensembles.In A) p h4))
               (C_c B A hs (exist (Ensembles.In A) p h4)))) =
              subset_sig (Add E0 (-p)) A h6).
    apply Extensionality_Ensembles.
    red. split. 
    red. intros x h9.  
    destruct h9 as [x h9l | x h9r]. 
    rewrite subset_sig_compat. rewrite subset_sig_compat in h9l.
    left. assumption.
    destruct h9r.
    rewrite subset_sig_compat. simpl. 
    right. constructor.
    red. intros x h9.
    rewrite subset_sig_compat in h9.
    inversion h9 as [x' h10 | x' h11].  subst. 
    left. rewrite subset_sig_compat. assumption.
    subst.
    inversion h11 as [h12]. right. 
    destruct x as [x h13]. simpl in h12. subst.
    simpl. unfold Bcomp_sub.  simpl.  
    assert (h15:h13 = (C_c B A hs (exist (Ensembles.In A) p h4))). 
      apply proof_irrelevance.  subst. constructor. 
  rewrite <- h9.
  assert (h10:one_closed B A h1 = O_c B A hs). apply proof_irrelevance.
  rewrite h10.
  assumption.
specialize (h2 _ h6). 
unfold A' in h2. unfold A' in h7.
assert (h8:hs = h1). apply proof_irrelevance. subst.
specialize (h2 h7).
pose proof (@set_infnt_distr_1_sup B _ p _ h2) as h8.
rewrite Im_add in h8.
rewrite comp_prod in h8.
rewrite one_prod in h8.
assert (h9: Im E0 (fun x : Btype (Bc B) => p * x) = E0).
  apply Extensionality_Ensembles.
  red. split.
  red.
  intros x h9.
  destruct h9 as [x h9]. subst.
  pose proof (le_sup _ _ _ h5 (exist _ _ (h3 _ h9))) as h10.
  assert (h11:Ensembles.In (subset_sig E0 A h3) (exist (Ensembles.In A) x (h3 _ h9))).
    rewrite subset_sig_compat.
    simpl. assumption.
  specialize (h10 h11).
  rewrite subset_sig_compat in h11. simpl in h11.
  red in h10. rewrite eq_ord in h10.
  simpl in h10. unfold Btimes_sub in h10. simpl in h10.
  apply exist_injective in h10.
  rewrite comm_prod. rewrite h10. assumption.
  red.
  intros x h9.
  pose proof (le_sup _ _ _ h5 (exist _ _ (h3 _ h9))) as h10.
  assert (h11: Ensembles.In (subset_sig E0 A h3)
          (exist (Ensembles.In A) x (h3 x h9))).
    rewrite subset_sig_compat.
    simpl. assumption.
  specialize (h10 h11).
  red in h10. rewrite eq_ord in h10.
  simpl in h10. unfold Btimes_sub in h10. simpl in h10.
  apply exist_injective in h10.
  rewrite <- h10.
  rewrite comm_prod.
  apply Im_intro with x; auto.
rewrite h9 in h8.
apply sup_subtract_zero in h8.
destruct (classic (Ensembles.In E0 0)) as [h9' | h10'].
rewrite sub_add_compat_in in h8.
rewrite sup_subtract_zero.
assumption. assumption.
rewrite sub_add_compat_nin in h8; auto.
Qed.
                                          
                        
End CompleteAndRegularSubalgebras.


Section Examples.
Variable fos:Field_of_Sets.
Definition psa_fos := psa (Xt fos).
Let Bt := Btype (Bc psa_fos).


Lemma fos_closed : @alg_closed psa_fos (F fos).
constructor.
(* + *)
red.
intros A B.
destruct A.
destruct B. 
unfold Bplus_sub.
simpl.
apply (Union_closed fos); assumption.
(* * *)
red.
intros A B.
destruct A.
destruct B.
unfold Btimes_sub.
simpl.
apply (Int_closed fos); assumption.
(* 1 *)
red. simpl. apply full_in_F.
(* 0 *)
red. simpl. apply empty_in_F.
(* - *)
red.
intro A.
destruct A as [? h1].
unfold Bcomp_sub.
simpl.
apply (Comp_closed fos). assumption.
Qed.

Definition fos_sub_psa := @Subalg psa_fos (F fos) fos_closed.

Variable Xt:Type.
Definition fin_cof_fos' := fin_cof_fos Xt.

Lemma fin_cof_closed : @alg_closed (psa Xt) (F_fc Xt).
constructor.
(* + *)
red.
intros A B.
destruct A.
destruct B. 
unfold Bplus_sub.
simpl.
apply (Union_closed_fc); assumption.
(* * *)
red.
intros A B.
destruct A.
destruct B.
unfold Btimes_sub.
simpl.
apply (Int_closed_fc); assumption.
(* 1 *)
red. simpl. apply full_in_F_fc.
(* 0 *)
red. simpl. apply empty_in_F_fc.
(* - *)
red.
intro A.
destruct A.
unfold Bcomp_sub.
simpl.
apply (Comp_closed_fc); assumption.
Qed.

Definition fin_cof_sub_psa := @Subalg (psa Xt) (F_fc Xt) fin_cof_closed. 

(* ". . . consider the field A of finite and cofinite subsets of
a set X. We shall prove that it is generated in P(X) by the set E of one-
element (that is, singleton) subsets of X." *)

Lemma fin_cof_gen_singletons' : @Gen (psa Xt) (SingF (Full_set Xt)) = fin_cof_sub_psa.
unfold fin_cof_sub_psa.
unfold Gen.
pose (@SA_cont_A (psa Xt) (SingF (Full_set Xt))) as F.
unfold SingF in F.
pose (int_saf_A F) as S.
simpl.
pose fin_cof_closed as h1.
pose proof (P_c _ _ h1) as h2.
assert (h3:Included (SingF (Full_set Xt)) (F_fc Xt)).
  red.
  intros A h4.
  inversion h4 as [x ? ? h5].
  rewrite h5.
  pose proof (Singleton_is_finite _ x) as h6.
  constructor. left; assumption.
assert (h4:int_saf_A (@SA_cont_A (psa Xt) (SingF (Full_set Xt))) = 
  (F_fc Xt)).
  apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red.
  intros sing h5.
  inversion h5 as [? h6].
  apply h6.
  constructor.
  split.
    (* left *)  
    red.
    intros sing'.
    unfold Bt in sing'.
    simpl in sing'.
    intro h7.
    inversion h7 as [y ? ? h8].
    rewrite h8.
    constructor.
    left.
    apply Singleton_is_finite.
    (* right *) assumption.
  (* >= *)
  red.
  intros S' h4.
  pose proof (union_singF_eq S') as h5.
  inversion h4 as [h6].
  assert (hfin : forall fin:(Btype (Bc (psa Xt))), Ensembles.In (F_fc Xt) fin -> Finite fin
    -> Ensembles.In S fin).
    intros fin h4' h6'.
    pose proof (union_singF_eq fin) as h5'. 
  induction h6' as [|fin' h7 h8 x h9].
  (* Empty *)
    constructor.
    intros E h6''.
    inversion h6'' as [h7].
    destruct h7 as [? h8].  
    pose proof (Z_c _ _ h8) as h9.
    red in h9.
    simpl in h9.
    assumption.
  (*Add*)
  unfold SingF in h5'.
  pose proof (Im_add _ _ fin' x (Singleton (U:=Xt))) as h10.
  unfold SingF.
  rewrite h10 in h5'.
  pose proof (family_union_add 
    (Im fin' (Singleton (U:=Xt))) (Singleton x)) as h11.
  rewrite h11 in h5'.
  rewrite h5'.
  pose proof (union_singF_eq fin') as h12.
  unfold SingF in h12.
  assert (h13:Ensembles.In (F_fc Xt) fin').
    constructor.
    left; assumption.
  pose proof (h8 h13 h12) as h14.
  rewrite <- h12.
  pose (Im (Full_set Xt) (Singleton (U:=Xt))) as J.
  assert (h15:Ensembles.In (int_saf_A (@SA_cont_A (psa Xt) J)) (Singleton x)).
    assert (h16:Ensembles.In J (Singleton x)).
      unfold J.
      apply Im_intro with x. apply Full_intro. trivial. 
      unfold SA_cont_A in F.
    pose (saf_A F) as EF.
    unfold Bt in EF.
    simpl in EF.
    unfold int_saf_A.
    assert (h17:forall (G:Family Xt), Ensembles.In EF G -> Included J G).
      intros G h18.
      inversion h18 as [h19].
      destruct h19 as [h19l].
      unfold J.
      assumption.
    constructor.
    unfold Bt. simpl.
    intros G h18.
    unfold EF in h17.
    pose proof (h17 G h18) as h19.
    auto with sets.

  pose proof (saf_closed F) as h20.    
  constructor.
  intros S0 h21.
  pose proof (h20 S0 h21) as h22.
  pose proof (P_c _ _ h22) as h23.
  red in h23.
  unfold SubBtype in h23.  unfold Bplus_sub in h23.
  unfold Bt in h23.
  simpl in h23.
  assert (h24: Ensembles.In S0 (Singleton x)).
    inversion h15 as [? h16].
    apply (h16 S0 h21).
  assert (h25: Ensembles.In S0 fin').
    inversion h14.
    apply (H S0 h21).
    apply (h23 (exist _ (Singleton x) h24) (exist _ fin' h25)).
destruct h6 as [h6l | h6r].
  (*h6l*)
  apply hfin; assumption.
  (*h6r*)
  assert (h7:Ensembles.In (F_fc Xt) (Complement S')).
    constructor.
    left; assumption.
  pose proof (hfin _ h7 h6r) as h8.
  pose proof (int_saf_alg_closed F) as h9.
  pose proof (C_c _ _ h9) as h10.
  red in h10.
  unfold SubBtype in h10. unfold Bcomp_sub in h10.
  unfold Bt in h10. simpl in h10.
  pose proof (h10 (exist _ (Complement S') h8)) as h11.
  simpl in h11.
  pose proof (Complement_Complement _ S') as h12.
  rewrite <- h12.
  assumption. 
unfold Gen_Ens.
unfold SA_cont_A in h4. unfold int_saf_A in h4. simpl in h4.  
apply subalg_functional.
assumption.
Qed.


End Examples.

(*Bool_alg_t analogues*)
Section ParametricAnalogues.
Section AlgClosed_p.
Variable T:Type.
Variable Bp:Bool_Alg_p T.
Let Btp := Btype_p T (Bc_p T Bp).
Variable A_p':Ensemble T.
Hypothesis incl_subalg : Included A_p' (A_p T (Bc_p T Bp)).
Definition SubBtype_p := sig_set A_p'.

(*This converts A_p' (an [Ensemble T]) to an [Ensemble Btype (Bc (ba_conv Bp))]*)
Definition ba_conv_und_subalg :=
  (ba_conv_set (Im (full_sig A_p') 
                   (fun x=>exist _ (proj1_sig x) (incl_subalg _ (proj2_sig x))))).






Definition SubBtype_p_conv (x:SubBtype_p) : SubBtype _ ba_conv_und_subalg.
unfold SubBtype, ba_conv. simpl.
unfold SubBtype_p in x.
pose (exist _ (proj1_sig x) (incl_subalg _ (proj2_sig x))) as x'.
assert (h1:Ensembles.In ba_conv_und_subalg x').
  apply Im_intro with x. constructor. reflexivity.
refine (exist _ _ h1).
Defined.


Lemma SubBtype_p_conv_inj :
  forall (x y:SubBtype_p),
    SubBtype_p_conv x = SubBtype_p_conv y ->
    x = y.
intros x y.
destruct x as [x h1]. destruct y as [y h2].
unfold SubBtype_p_conv.
intro h3.
apply exist_injective in h3.
apply exist_injective in h3. simpl in h3.
apply proj1_sig_injective. simpl.
assumption.
Qed.


Definition Bplus_sub_p (x y:SubBtype_p) : Btp.
destruct x as [x h1]. destruct y as [y h2].
pose proof (incl_subalg _ h1) as h3.
pose proof (incl_subalg _ h2) as h4.
pose (exist _ _ h3) as x'.
pose (exist _ _ h4) as y'.
refine (x' %+ y').
Defined.

Lemma Bplus_sub_p_eq : forall (x y:SubBtype_p),
                         Bplus_sub_p x y =
                         Bplus_sub _ _
                                   (SubBtype_p_conv x)
                                   (SubBtype_p_conv y).
intros x y. 
destruct x as [x h1]. destruct y as [y h2].
unfold Bplus_sub_p, Bplus_sub.  simpl.
reflexivity.
Qed.


Definition Btimes_sub_p (x y:SubBtype_p) : Btp.
destruct x as [x h1]. destruct y as [y h2].
pose proof (incl_subalg _ h1) as h3.
pose proof (incl_subalg _ h2) as h4.
pose (exist _ _ h3) as x'.
pose (exist _ _ h4) as y'.
refine (x' %* y').
Defined.

Lemma Btimes_sub_p_eq : forall (x y:SubBtype_p),
                         Btimes_sub_p x y =
                         Btimes_sub _ _
                                   (SubBtype_p_conv x)
                                   (SubBtype_p_conv y).
intros x y. 
destruct x as [x h1]. destruct y as [y h2].
unfold Btimes_sub_p, Btimes_sub.  simpl.
reflexivity.
Qed.

Definition Bcomp_sub_p (x:SubBtype_p) : Btp.
destruct x as [x h1]. 
pose proof (incl_subalg _ h1) as h3.
pose (exist _ _ h3) as x'.
refine (%-x').
Defined.

Lemma Bcomp_sub_p_eq : forall (x:SubBtype_p),
                         Bcomp_sub_p x =
                         Bcomp_sub _ _
                                   (SubBtype_p_conv x).
                                  
intro x. 
destruct x as [x h1]. 
unfold Bcomp_sub_p, Bcomp_sub.  simpl.
reflexivity.
Qed.



Notation  "x %+|' y" := (Bplus_sub_p x y) (at level 50, left associativity).
Notation  "x %*|' y" := (Btimes_sub_p x y) (at level 40, left associativity).
Notation "%-|' x" := (Bcomp_sub_p x) (at level 30).

Definition Plus_closed_sub_p : Prop := (forall (x y:SubBtype_p),
               Ensembles.In A_p' (proj1_sig (x %+|' y))).

Lemma Plus_closed_sub_p_iff : 
  Plus_closed_sub_p <->
  Plus_closed_sub ba_conv_und_subalg.
split.
intro h1.
red.
intros x y. unfold ba_conv_set. unfold ba_conv_type.
unfold transfer_dep. unfold eq_rect_r. simpl.
red in h1.
destruct x as [x h2]. destruct y as [y h3].
destruct h2 as [x h2]. destruct h3 as [y h3].
subst.
specialize (h1 x y).
simpl.
apply Im_intro with (exist _ _ h1).
constructor.
simpl.
unfold Bplus_sub. simpl.
apply proj1_sig_injective. simpl.
destruct x as [x h4]. destruct y as [y h5].
simpl.
reflexivity.
intro h1.
red.
intros x y.
red in h1.
unfold SubBtype in h1.
destruct x as [x h2]. destruct y as [y h3]. simpl.
pose (exist _ _ (incl_subalg _ h2)) as x'.  
pose (exist _ _ (incl_subalg _ h3)) as y'.  
assert (h4:Ensembles.In 
             (ba_conv_set
                (Im (full_sig A_p')
                    (fun x : sig_set A_p' =>
                       exist (Ensembles.In (A_p T (Bc_p T Bp))) 
                             (proj1_sig x)
                             (incl_subalg (proj1_sig x) (proj2_sig x))))) x').
  apply Im_intro with (exist _ _ h2).
  constructor. simpl. unfold x'. reflexivity.
assert (h5:Ensembles.In 
             (ba_conv_set
                (Im (full_sig A_p')
                    (fun x : sig_set A_p' =>
                       exist (Ensembles.In (A_p T (Bc_p T Bp))) 
                             (proj1_sig x)
                             (incl_subalg (proj1_sig x) (proj2_sig x))))) y').
  apply Im_intro with (exist _ _ h3).
  constructor. simpl. unfold y'. reflexivity.
specialize (h1 (exist _ _ h4) (exist _ _ h5)).
inversion h1 as [z h7 h8 h9]. subst.
unfold Bplus_sub in h9. simpl in h9.
unfold x', y' in h9.
rewrite h9.
simpl.
apply proj2_sig.
Qed.



Definition Times_closed_sub_p : Prop := (forall (x y:SubBtype_p),
               Ensembles.In A_p' (proj1_sig (x %*|' y))).

Lemma Times_closed_sub_p_iff : 
  Times_closed_sub_p <->
  Times_closed_sub ba_conv_und_subalg.
split.
intro h1.
red.
intros x y. unfold ba_conv_set. unfold ba_conv_type.
unfold transfer_dep. unfold eq_rect_r. simpl.
red in h1.
destruct x as [x h2]. destruct y as [y h3].
destruct h2 as [x h2]. destruct h3 as [y h3].
subst.
specialize (h1 x y).
simpl.
apply Im_intro with (exist _ _ h1).
constructor.
simpl.
unfold Bplus_sub. simpl.
apply proj1_sig_injective. simpl.
destruct x as [x h4]. destruct y as [y h5].
simpl.
reflexivity.
intro h1.
red.
intros x y.
red in h1.
unfold SubBtype in h1.
destruct x as [x h2]. destruct y as [y h3]. simpl.
pose (exist _ _ (incl_subalg _ h2)) as x'.  
pose (exist _ _ (incl_subalg _ h3)) as y'.  
assert (h4:Ensembles.In 
             (ba_conv_set
                (Im (full_sig A_p')
                    (fun x : sig_set A_p' =>
                       exist (Ensembles.In (A_p T (Bc_p T Bp))) 
                             (proj1_sig x)
                             (incl_subalg (proj1_sig x) (proj2_sig x))))) x').
  apply Im_intro with (exist _ _ h2).
  constructor. simpl. unfold x'. reflexivity.
assert (h5:Ensembles.In 
             (ba_conv_set
                (Im (full_sig A_p')
                    (fun x : sig_set A_p' =>
                       exist (Ensembles.In (A_p T (Bc_p T Bp))) 
                             (proj1_sig x)
                             (incl_subalg (proj1_sig x) (proj2_sig x))))) y').
  apply Im_intro with (exist _ _ h3).
  constructor. simpl. unfold y'. reflexivity.
specialize (h1 (exist _ _ h4) (exist _ _ h5)).
inversion h1 as [z h7 h8 h9]. subst.
unfold Btimes_sub in h9. simpl in h9.
unfold x', y' in h9.
rewrite h9.
simpl.
apply proj2_sig.
Qed.


Definition One_closed_sub_p  : Prop := Ensembles.In (A_p') (proj1_sig (Bone_p T (Bc_p T Bp))).


Lemma One_closed_sub_p_iff : 
  One_closed_sub_p <->
  One_closed_sub 
    ba_conv_und_subalg.
split.
intro h1. red in h1. 
pose (exist _ _ h1) as one.
apply Im_intro with one. constructor.
unfold one. simpl.
apply proj1_sig_injective.
simpl.
reflexivity.
intro h1. red in h1. 
inversion h1 as [one h2 ? h4]. subst.
red.
rewrite ba_conv_one. rewrite h4.
simpl.
apply proj2_sig.
Qed.


Definition Zero_closed_sub_p  : Prop := Ensembles.In (A_p') (proj1_sig (Bzero_p T (Bc_p T Bp))).

Lemma Zero_closed_sub_p_iff : 
  Zero_closed_sub_p <->
  Zero_closed_sub 
    ba_conv_und_subalg.
split.
intro h1. red in h1. 
pose (exist _ _ h1) as zero.
apply Im_intro with zero. constructor.
unfold zero. simpl.
apply proj1_sig_injective.
simpl.
reflexivity.
intro h1. red in h1. 
inversion h1 as [zero h2 ? h4]. subst.
red.
rewrite ba_conv_zero. rewrite h4.
simpl.
apply proj2_sig.
Qed.


Definition Comp_closed_sub_p : Prop := (forall x:SubBtype_p,
               Ensembles.In A_p' (proj1_sig (%-|' x))).


Lemma Comp_closed_sub_p_iff : 
  Comp_closed_sub_p <->
  Comp_closed_sub ba_conv_und_subalg.
split.
intro h1.
red.
intros x. unfold ba_conv_set. unfold ba_conv_type.
unfold transfer_dep. unfold eq_rect_r. simpl.
red in h1.
destruct x as [x h2]. 
destruct h2 as [x h2].
subst.
specialize (h1 x).
simpl.
apply Im_intro with (exist _ _ h1).
constructor.
simpl.
unfold Bcomp_sub. simpl.
apply proj1_sig_injective. simpl.
destruct x as [x h4]. 
simpl.
reflexivity.
intro h1.
red.
intros x.
red in h1.
unfold SubBtype in h1.
destruct x as [x h2]. simpl.
pose (exist _ _ (incl_subalg _ h2)) as x'.  
assert (h4:Ensembles.In 
             (ba_conv_set
                (Im (full_sig A_p')
                    (fun x : sig_set A_p' =>
                       exist (Ensembles.In (A_p T (Bc_p T Bp))) 
                             (proj1_sig x)
                             (incl_subalg (proj1_sig x) (proj2_sig x))))) x').
  apply Im_intro with (exist _ _ h2).
  constructor. simpl. unfold x'. reflexivity.
specialize (h1 (exist _ _ h4)).
inversion h1 as [z h7 h8 h9]. subst.
unfold Bcomp_sub in h9. simpl in h9.
unfold x' in h9.
rewrite h9.
simpl.
apply proj2_sig.
Qed.


Definition Plus_closed_p'' (x y:Btp) :
  Plus_closed_sub_p -> Ensembles.In ba_conv_und_subalg x -> 
  Ensembles.In ba_conv_und_subalg y -> Ensembles.In ba_conv_und_subalg (x%+y).
rewrite Plus_closed_sub_p_iff.
apply (@Plus_closed'' (ba_conv Bp)).
Qed.



Definition Times_closed_p'' (x y:Btp) :
  Times_closed_sub_p -> Ensembles.In ba_conv_und_subalg x -> 
  Ensembles.In ba_conv_und_subalg y -> Ensembles.In ba_conv_und_subalg (x%*y).
rewrite Times_closed_sub_p_iff.
apply (@Times_closed'' (ba_conv Bp)).
Qed.


Definition Comp_closed_p'' (b:Btp) :
  Comp_closed_sub_p -> Ensembles.In ba_conv_und_subalg b -> 
  Ensembles.In ba_conv_und_subalg (%-b).
rewrite Comp_closed_sub_p_iff.
apply (@Comp_closed'' (ba_conv Bp)).
Qed.


Record alg_closed_p : Prop := {
  P_c_p : Plus_closed_sub_p;
  T_c_p : Times_closed_sub_p;
  O_c_p : One_closed_sub_p;
  Z_c_p : Zero_closed_sub_p;
  C_c_p : Comp_closed_sub_p}.

Lemma alg_closed_p_iff :
  alg_closed_p <-> alg_closed ba_conv_und_subalg.
split.
intro h1.
destruct h1 as [h1a h1b h1c h1d h1e].
rewrite Plus_closed_sub_p_iff in h1a.
rewrite Times_closed_sub_p_iff in h1b.
rewrite One_closed_sub_p_iff in h1c.
rewrite Zero_closed_sub_p_iff in h1d.
rewrite Comp_closed_sub_p_iff in h1e.
constructor; auto.
intro h1.
destruct h1 as [h1a h1b h1c h1d h1e].
rewrite <- Plus_closed_sub_p_iff in h1a.
rewrite <- Times_closed_sub_p_iff in h1b.
rewrite <- One_closed_sub_p_iff in h1c.
rewrite <- Zero_closed_sub_p_iff in h1d.
rewrite <- Comp_closed_sub_p_iff in h1e.
constructor; auto.
Qed.




Variable Acp : alg_closed_p.

Definition Bc_p' :=
  Build_Bconst_p  
    T
    A_p'
    (Full_set SubBtype_p)
    (fun (x y:SubBtype_p) => exist (Ensembles.In A_p') (proj1_sig (x %+|' y))
      (P_c_p Acp x y))
    (fun (x y:SubBtype_p) => exist (Ensembles.In A_p') (proj1_sig (x %*|' y))
      (T_c_p Acp x y))
    (exist (Ensembles.In A_p') (proj1_sig %1) (O_c_p Acp))
    (exist (Ensembles.In A_p') (proj1_sig %0) (Z_c_p Acp))
    (fun (x:SubBtype_p) => (exist (Ensembles.In A_p') (proj1_sig (%-|' x))
      (C_c_p Acp x))).


Definition Bc_p_conv' := Bc' _ _ (iff1 alg_closed_p_iff Acp). 

Section Bc_p''.

Infix "%+|" := (Bplus_p T Bc_p')  (at level 50, left associativity).
Infix "%*|" := (Btimes_p T Bc_p') (at level 40, left associativity).
Notation "%0|" := (Bzero_p T Bc_p').
Notation "%1|" := (Bone_p T Bc_p').
Notation "%-| x" := ((Bcomp_p T Bc_p') x) (at level 30).

Lemma bplus_p_bc_p_eq' : forall (x y:SubBtype_p), 
                           SubBtype_p_conv (x %+| y) =
                           Bplus Bc_p_conv' (SubBtype_p_conv x)
                                 (SubBtype_p_conv y).
intros x y.
unfold SubBtype_p_conv. simpl.
apply proj1_sig_injective. simpl.
apply proj1_sig_injective. simpl.
unfold Bplus_sub. simpl. simpl.
unfold Bplus_sub_p.
destruct x as [x h1]. destruct y as [y h2]. 
simpl.
reflexivity.
Qed.


Lemma btimes_p_bc_p_eq' : forall (x y:SubBtype_p), 
                           SubBtype_p_conv (x %*| y) =
                           Btimes Bc_p_conv' (SubBtype_p_conv x)
                                 (SubBtype_p_conv y).
intros x y.
unfold SubBtype_p_conv. simpl.
apply proj1_sig_injective. simpl.
apply proj1_sig_injective. simpl.
unfold Btimes_sub. simpl. simpl.
unfold Btimes_sub_p.
destruct x as [x h1]. destruct y as [y h2]. 
simpl.
reflexivity.
Qed.


Lemma bzero_p_bc_p_eq' : 
  SubBtype_p_conv %0| = Bzero Bc_p_conv'.
unfold SubBtype_p_conv.
apply proj1_sig_injective. simpl.
apply proj1_sig_injective. simpl.
reflexivity.
Qed.


Lemma bone_p_bc_p_eq' : 
  SubBtype_p_conv %1| = Bone Bc_p_conv'.
unfold SubBtype_p_conv.
apply proj1_sig_injective. simpl.
apply proj1_sig_injective. simpl.
reflexivity.
Qed.



Lemma bcomp_p_bc_p_eq' : forall (x:SubBtype_p), 
                           SubBtype_p_conv (%-| x) =
                           Bcomp Bc_p_conv' (SubBtype_p_conv x).
intro x.
unfold SubBtype_p_conv. simpl.
apply proj1_sig_injective. simpl.
apply proj1_sig_injective. simpl.
unfold Bcomp_sub. simpl. simpl.
unfold Bcomp_sub_p.
destruct x as [x h1].
simpl.
reflexivity.
Qed.
                                


Lemma assoc_sum_p' : forall n m p:SubBtype_p, n %+| (m %+| p) = n %+| m %+| p. 
intros n m p.
pose proof (assoc_sum' _ ba_conv_und_subalg (iff1 alg_closed_p_iff Acp) (SubBtype_p_conv n) (SubBtype_p_conv m) (SubBtype_p_conv p)) as h1.  
apply SubBtype_p_conv_inj. 
do 2 rewrite bplus_p_bc_p_eq'.
rewrite h1 at 1.
do 2 rewrite bplus_p_bc_p_eq'.
reflexivity.
Qed.


Lemma assoc_prod_p' : forall n m p:SubBtype_p, n %*| (m %*| p) = n %*| m %*| p.
intros n m p.
pose proof (assoc_prod' _ ba_conv_und_subalg (iff1 alg_closed_p_iff Acp) (SubBtype_p_conv n) (SubBtype_p_conv m) (SubBtype_p_conv p)) as h1.  
apply SubBtype_p_conv_inj. 
do 2 rewrite btimes_p_bc_p_eq'.
rewrite h1 at 1.
do 2 rewrite btimes_p_bc_p_eq'.
reflexivity.
Qed.


Lemma comm_sum_p' : forall n m:SubBtype_p, n %+| m = m %+| n.
intros n m.
pose proof (comm_sum' _ ba_conv_und_subalg (iff1 alg_closed_p_iff Acp) (SubBtype_p_conv n) (SubBtype_p_conv m)) as h1.  
apply SubBtype_p_conv_inj. 
rewrite bplus_p_bc_p_eq'.
rewrite h1 at 1.
rewrite bplus_p_bc_p_eq'.
reflexivity.
Qed.

Lemma comm_prod_p' : forall n m:SubBtype_p, n %*| m = m %*| n.
intros n m.
pose proof (comm_prod' _ ba_conv_und_subalg (iff1 alg_closed_p_iff Acp) (SubBtype_p_conv n) (SubBtype_p_conv m)) as h1.  
apply SubBtype_p_conv_inj. 
rewrite btimes_p_bc_p_eq'.
rewrite h1 at 1.
rewrite btimes_p_bc_p_eq'.
reflexivity.
Qed.

Lemma abs_sum_p'  : forall n m:SubBtype_p, n %+| (n %*| m) = n.
intros n m.
pose proof (abs_sum' _ ba_conv_und_subalg (iff1 alg_closed_p_iff Acp) (SubBtype_p_conv n) (SubBtype_p_conv m)) as h1.  
apply SubBtype_p_conv_inj.
rewrite bplus_p_bc_p_eq'. rewrite btimes_p_bc_p_eq'.
rewrite h1 at 1.
reflexivity.
Qed.

Lemma abs_prod_p':  forall n m:SubBtype_p, n %*| (n %+| m) = n.
intros n m.
pose proof (abs_prod' _ ba_conv_und_subalg (iff1 alg_closed_p_iff Acp) (SubBtype_p_conv n) (SubBtype_p_conv m)) as h1.  
apply SubBtype_p_conv_inj.
rewrite btimes_p_bc_p_eq'. rewrite bplus_p_bc_p_eq'. 
rewrite h1 at 1.
reflexivity.
Qed.

Lemma dist_sum_p' : forall n m p:SubBtype_p, p %*| (n %+| m) = p %*| n %+ p %*| m.
intros n m p.
pose proof (dist_sum' _ ba_conv_und_subalg (iff1 alg_closed_p_iff Acp) (SubBtype_p_conv n) (SubBtype_p_conv m) (SubBtype_p_conv p)) as h1.  
apply SubBtype_p_conv_inj.
rewrite btimes_p_bc_p_eq'. rewrite bplus_p_bc_p_eq'.
rewrite h1 at 1.
rewrite bplus_p_bc_p_eq'.
rewrite btimes_p_bc_p_eq'.
rewrite btimes_p_bc_p_eq'. 
reflexivity.
Qed.


Lemma dist_prod_p': forall n m p:SubBtype_p, p %+| (n %*| m) = (p %+| n) %*| (p %+| m).
intros n m p.
pose proof (dist_prod' _ ba_conv_und_subalg (iff1 alg_closed_p_iff Acp) (SubBtype_p_conv n) (SubBtype_p_conv m) (SubBtype_p_conv p)) as h1.  
apply SubBtype_p_conv_inj.
rewrite bplus_p_bc_p_eq'. rewrite btimes_p_bc_p_eq'.
rewrite h1 at 1.
rewrite btimes_p_bc_p_eq'.
rewrite bplus_p_bc_p_eq'.
rewrite bplus_p_bc_p_eq'. 
reflexivity.
Qed.


Lemma comp_sum_p':  forall n:SubBtype_p, n %+| (%-| n) = %1.
intro n.
pose proof (comp_sum' _ ba_conv_und_subalg (iff1 alg_closed_p_iff Acp) (SubBtype_p_conv n)) as h1.
apply SubBtype_p_conv_inj.
rewrite bplus_p_bc_p_eq'. rewrite bcomp_p_bc_p_eq'.
rewrite h1 at 1.
rewrite bone_p_bc_p_eq'.
reflexivity.
Qed.

Lemma comp_prod_p': forall n:SubBtype_p, n %*| (%-| n) = %0.
intro n.
pose proof (comp_prod' _ ba_conv_und_subalg (iff1 alg_closed_p_iff Acp) (SubBtype_p_conv n)) as h1.
apply SubBtype_p_conv_inj.
rewrite btimes_p_bc_p_eq'. rewrite bcomp_p_bc_p_eq'.
rewrite h1 at 1.
rewrite bzero_p_bc_p_eq'.
reflexivity.
Qed.

Lemma und_set_p' : (BS_p T Bc_p') = Full_set (Btype_p T Bc_p').
simpl. unfold SubBtype_p. unfold Btype_p. simpl.
reflexivity.
Qed.


Definition Subalg_p := Build_Bool_Alg_p T Bc_p' und_set_p' assoc_sum_p'
  assoc_prod_p' comm_sum_p' comm_prod_p' abs_sum_p' abs_prod_p' dist_sum_p'
  dist_prod_p' comp_sum_p' comp_prod_p'.

Lemma two_ops_imp_times_p : Comp_closed_sub_p -> Plus_closed_sub_p -> Times_closed_sub_p.
intros h1 h2.
rewrite Comp_closed_sub_p_iff in h1.
rewrite Plus_closed_sub_p_iff in h2.
pose proof (two_ops_imp_times _ _ h1 h2) as h3.
rewrite <- Times_closed_sub_p_iff in h3.
assumption.
Qed.


Lemma two_ops_imp_plus_p : Comp_closed_sub_p -> Times_closed_sub_p -> Plus_closed_sub_p.
intros h1 h2.
rewrite Comp_closed_sub_p_iff in h1.
rewrite Times_closed_sub_p_iff in h2.
pose proof (two_ops_imp_plus _ _ h1 h2) as h3.
rewrite <- Plus_closed_sub_p_iff in h3.
assumption.
Qed.
  

Definition ba_conv_subalg : Bool_Alg.    
rewrite alg_closed_p_iff in Acp.
refine (Subalg _ Acp).
Defined.


Lemma im_proj1_sig_ba_conv_subalg_eq : 
  im_proj1_sig (ba_ens (ba_conv Subalg_p)) = im_proj1_sig (im_proj1_sig (ba_ens ba_conv_subalg)).
apply Extensionality_Ensembles.
red. split.
red. intros x h1. destruct h1 as [x h1]. subst.
destruct x as [x h2]. simpl.
unfold Subalg_p in h2. simpl in h2. 
assert (h3:Ensembles.In ba_conv_und_subalg (exist _ _ (incl_subalg _ h2))).
  apply Im_intro with (exist _ _ h2). constructor.
  apply proj1_sig_injective; simpl. reflexivity.
apply Im_intro with (exist _ _ (incl_subalg _ h2)). 
unfold ba_conv_subalg.
apply Im_intro with (exist _ _ h3). constructor.
simpl.
reflexivity. simpl.
reflexivity.
red. intros x h1.
destruct h1 as [x h1]. subst.
destruct x as [x h2]. simpl.
inversion h1 as [x' h3 ? h4]. subst.
destruct x' as [x' h5].
destruct h5 as [x' h5]. subst.
simpl in h4. 
pose proof (f_equal (@proj1_sig _ _) h4) as h4'. clear h4. 
simpl in h4'. subst.
apply Im_intro with x'. 
unfold ba_ens, ba_conv, Subalg_p. simpl. unfold bt. simpl.
constructor.
reflexivity.
Qed.


End Bc_p''. 




End AlgClosed_p.


Arguments alg_closed_p [T] [Bp] _ _.
Arguments alg_closed_p_iff [T] [Bp] _ _.
Arguments Subalg_p [T] _  _ _ _.
Arguments Plus_closed_sub_p [T] [Bp] _ _.
Arguments Times_closed_sub_p [T] [Bp] _ _.
Arguments Comp_closed_sub_p [T] [Bp] _ _.
Arguments Zero_closed_sub_p [T] [Bp] _.
Arguments One_closed_sub_p [T] [Bp] _.
Arguments ba_conv_und_subalg [T] [Bp] _ _ _.





Lemma incl_ens_btp : 
  forall {T:Type} {Bp:Bool_Alg_p T} (E:Ensemble (btp Bp)),
    Included (im_proj1_sig E) (ba_p_ens Bp).
intros T Bp E.
red.
intros x h1.
destruct h1 as [x h1]. subst.
apply proj2_sig.
Qed.



Lemma in_ba_conv_und_subalg_iff :
  forall {T:Type} (Bp:Bool_Alg_p T) (E:Ensemble T)
         (pf:Included E (ba_p_ens Bp)) (x:btp Bp),
    Ensembles.In (ba_conv_und_subalg E pf) x <-> 
    Ensembles.In E (proj1_sig x).
intros T Bp E h1 x. split.
intro h2. destruct h2 as [x h2]. subst. simpl.
apply proj2_sig.
intro h2. apply Im_intro with (exist _ _ h2). constructor.
simpl. destruct x as [x h3]. simpl.
apply proj1_sig_injective.
simpl.
reflexivity.
Qed.



Lemma ba_conv_und_subalg_im_proj1_sig : 
  forall {T:Type} {Bp:Bool_Alg_p T} (S:Ensemble (btp Bp))
         (pf:Included (im_proj1_sig S) (ba_p_ens Bp)),
    ba_conv_und_subalg (im_proj1_sig S) pf = S.
intros T Bp S h1.
apply Extensionality_Ensembles.
red. split.
red.
intros x h2.
destruct h2 as [x h2]. subst. clear h2.
destruct x as [x h2]. simpl.
destruct h2 as [x h2]. subst. simpl.
destruct x as [x h3]. simpl.
generalize  (h1 x
           (Im_intro (sig_set (A_p T (Bc_p T Bp))) T S
              (proj1_sig
                 (P:=fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0))
              (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) x h3)
              h2 x eq_refl)).
intro h4. assert(h4 = h3). apply proof_irrelevance. subst.
assumption.
red.
intros x h2. unfold ba_conv_und_subalg.  
assert (h3:Ensembles.In (im_proj1_sig S) (proj1_sig x)).
  apply Im_intro with x; auto.
apply Im_intro with (exist _ _ h3). constructor.
apply proj1_sig_injective. simpl.
reflexivity.
Qed.



Lemma alg_closed_p_ba_p_ens_refl : 
  forall {T:Type} (Bp:Bool_Alg_p T), 
    alg_closed_p (ba_p_ens Bp) (inclusion_reflexive _).
intros T Bp.
constructor.
red. 
intros. unfold Bplus_sub_p.
destruct x as [x h1].  destruct y as [y h2]. 
apply in_ba_p_ens_plus.
destruct x as [x h1].  destruct y as [y h2]. 
apply in_ba_p_ens_times.
red. apply in_ba_p_ens_one.
red. apply in_ba_p_ens_zero.
red.
intros. destruct x as [x h1]. 
apply in_ba_p_ens_comp.
Qed.





Lemma subalg_functional_p :
  forall {T:Type} {Bp:Bool_Alg_p T}
         (A A':Ensemble T) 
         (pfeq:A = A')
         (pfia:Included A (A_p T (Bc_p T Bp)))
         (pfia':Included A' (A_p T (Bc_p T Bp))),
    forall (pfa:alg_closed_p A pfia)
      (pfa':alg_closed_p A' pfia'),
        Subalg_p _ A pfia pfa = Subalg_p _ A' pfia' pfa'. 
intros T Bp A A' h1 h2 h3 h4 h5.
subst. 
assert (h2 = h3). apply proof_irrelevance. subst.
assert (h4 = h5). apply proof_irrelevance.
subst.
reflexivity.
Qed.

Lemma in_sup_alg_p : 
  forall {T:Type} {Bp:Bool_Alg_p T} {A:Ensemble T}
         {pfa:Included A (ba_p_ens Bp)}
         {pfcl:alg_closed_p _ pfa}
         (x:Btype_p T (Bc_p T (Subalg_p _ _ pfa pfcl))),
            Ensembles.In (ba_p_ens Bp) (proj1_sig x).
intros T Bp A h1 h2 x.
destruct x as [x h3].
simpl.
simpl in h3.
apply (h1 _ h3).
Qed.


Section Closed_p.
Variable T:Type.
Variable Bp:Bool_Alg_p T.
Let Btp := Btype_p T (Bc_p T Bp).

Lemma plus_closed_p :
  forall (S:Ensemble T)
         (pf:Included S (ba_p_ens Bp)),
    alg_closed_p S pf ->
    forall (x y:Btp) (pfx:Ensembles.In (ba_conv_und_subalg _ pf) x)
           (pfy:Ensembles.In (ba_conv_und_subalg _ pf) y),
      Ensembles.In (ba_conv_und_subalg _ pf) (x %+ y). 
intros S h1 h2 x y h3 h4.
rewrite (alg_closed_p_iff _ h1) in h2.
pose proof (plus_closed _ _ h2) as h5.
specialize (h5 x y h3 h4).
assumption.
Qed.

Lemma times_closed_p :
  forall (S:Ensemble T)
         (pf:Included S (ba_p_ens Bp)),
    alg_closed_p S pf ->
    forall (x y:Btp) (pfx:Ensembles.In (ba_conv_und_subalg _ pf) x)
           (pfy:Ensembles.In (ba_conv_und_subalg _ pf) y),
      Ensembles.In (ba_conv_und_subalg _ pf) (x %* y). 
intros S h1 h2 x y h3 h4.
rewrite (alg_closed_p_iff _ h1) in h2.
pose proof (times_closed _ _ h2) as h5.
specialize (h5 x y h3 h4).
assumption.
Qed.


Lemma comp_closed_p :
  forall (S:Ensemble T)
         (pf:Included S (ba_p_ens Bp)),
    alg_closed_p S pf ->
    forall (x:Btp) (pfx:Ensembles.In (ba_conv_und_subalg _ pf) x),
      Ensembles.In (ba_conv_und_subalg _ pf) (%-x).     
intros S h1 h2 x h3.
rewrite (alg_closed_p_iff _ h1) in h2.
pose proof (comp_closed _ _ h2) as h5.
specialize (h5 x h3).
assumption.
Qed.


Lemma zero_closed_p :
  forall (S:Ensemble T)
         (pf:Included S (ba_p_ens Bp)),
    alg_closed_p S pf ->
    Ensembles.In (ba_conv_und_subalg _ pf) (%0).     
intros S h1 h2.
rewrite (alg_closed_p_iff _ h1) in h2.
pose proof (zero_closed _ _ h2) as h3.
assumption.
Qed.


Lemma one_closed_p :
  forall (S:Ensemble T)
         (pf:Included S (ba_p_ens Bp)),
    alg_closed_p S pf ->
    Ensembles.In (ba_conv_und_subalg _ pf) (%1).
intros S h1 h2.
rewrite (alg_closed_p_iff _ h1) in h2.
pose proof (one_closed _ _ h2) as h3.
assumption.
Qed.


Lemma closed_fun2_plus : 
  forall (Cp:Bool_Alg_p T)
         (pf:Included (ba_p_ens Bp) (ba_p_ens Cp)),
    alg_closed_p _ pf ->
    closed_fun2 (Bplus_p _ (Bc_p _ Cp)) pf.
intros Cp h0 h1. 
red.
intros x y h2 h3. 
destruct h1 as [h1]. red in h1. 
pose proof (h1 (exist _ _ h2) (exist _ _ h3)) as h4.
simpl in h4.
assumption.
Qed.


Lemma closed_fun2_times : 
  forall (Cp:Bool_Alg_p T)
         (pf:Included (ba_p_ens Bp) (ba_p_ens Cp)),
    alg_closed_p _ pf ->
    closed_fun2 (Btimes_p _ (Bc_p _ Cp)) pf.
intros Cp h0 h1. 
red.
intros x y h2 h3. 
destruct h1 as [? h1]. red in h1. 
pose proof (h1 (exist _ _ h2) (exist _ _ h3)) as h4.
simpl in h4.
assumption.
Qed.


Lemma closed_fun_comp : 
  forall (Cp:Bool_Alg_p T)
         (pf:Included (ba_p_ens Bp) (ba_p_ens Cp)),
    alg_closed_p _ pf ->
    closed_fun (Bcomp_p _ (Bc_p _ Cp)) pf.
intros Cp h0 h1. 
red.
intros x h2. 
destruct h1 as [? ? ? ? h1]. red in h1. 
pose proof (h1 (exist _ _ h2)) as h4.
simpl in h4.
assumption.
Qed.


End Closed_p.

Arguments closed_fun2_plus [T] _ _ _ _ _ _ _ _.
Arguments closed_fun2_times [T] _ _ _ _ _ _ _ _.
Arguments closed_fun_comp [T] _ _ _ _ _ _.



Section Subalg_p.

Lemma ba_p_ens_subalg_p_compat : 
  forall {T:Type} (Bp:Bool_Alg_p T) (E:Ensemble T)
         (pfi:Included E (ba_p_ens Bp))
         (pfac:alg_closed_p _ pfi),
    ba_p_ens (Subalg_p _ _ pfi pfac) = E.
intros T Bp E h1 h2.
apply Extensionality_Ensembles.
red. split.
red. intros x h3.
unfold Subalg_p, ba_p_ens in h3. simpl in h3.
assumption.
red. intros x h3.
unfold Subalg_p, ba_p_ens. simpl.
assumption.
Qed.



Definition subalg_of_p {T:Type} (Ap Bp:Bool_Alg_p T) :=
  exists (pfin:Included (ba_p_ens Ap) (ba_p_ens Bp))
         (pfac:alg_closed_p _ pfin),
    Ap = Subalg_p Bp _ pfin pfac.


Lemma refl_subalg_of_p : 
  forall (T:Type), 
    Reflexive (@subalg_of_p T).
intro T.
red.
intro B.
red.
exists (inclusion_reflexive _).
exists (alg_closed_p_ba_p_ens_refl _).
apply bc_inj_p.
assert (h1:A_p T (Bc_p T B) = A_p T  (Bc_p T
     (Subalg_p B (ba_p_ens B) (inclusion_reflexive (ba_p_ens B))
        (alg_closed_p_ba_p_ens_refl B)))).
  simpl.
  reflexivity.
assert (h2:(sig_set_eq (A_p T (Bc_p T B)) (ba_p_ens B) h1) = eq_refl (sig_set (ba_p_ens B))). apply proof_irrelevance.
apply (bconst_ext_p _ _ h1); simpl.
simpl in h1.
clear h2.
destruct h1.
unfold sig_set_eq. unfold eq_ind_r. unfold eq_ind. simpl.
rewrite transfer_dep_r_eq_refl.
apply und_set_p. 
apply functional_extensionality. intro x. apply functional_extensionality.
intro y.
rewrite transfer_dep_r_fun2_eq.
rewrite h2.
unfold transfer_r. unfold eq_rect_r. simpl. 
destruct x as [x h3], y as [y h4]. 
apply proj1_sig_injective. simpl.
simpl.
unfold transfer. unfold eq_rect_r. simpl. 
assert (h5:h3 = (inclusion_reflexive (ba_p_ens B) x h3)). apply proof_irrelevance.
assert (h6:h4 = (inclusion_reflexive (ba_p_ens B) y h4)). apply proof_irrelevance.
rewrite h5, h6.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
apply functional_extensionality. intro x. apply functional_extensionality.
intro y.
rewrite transfer_dep_r_fun2_eq.
rewrite h2.
unfold transfer_r. unfold eq_rect_r. simpl. 
destruct x as [x h3], y as [y h4]. 
apply proj1_sig_injective. simpl.
simpl.
unfold transfer. unfold eq_rect_r. simpl. 
assert (h5:h3 = (inclusion_reflexive (ba_p_ens B) x h3)). apply proof_irrelevance.
assert (h6:h4 = (inclusion_reflexive (ba_p_ens B) y h4)). apply proof_irrelevance.
rewrite h5, h6.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
rewrite h2. rewrite transfer_dep_r_eq_refl.
apply proj1_sig_injective.
simpl.
reflexivity.
rewrite h2. rewrite transfer_dep_r_eq_refl.
apply proj1_sig_injective.
simpl.
reflexivity.
apply functional_extensionality. intro x. 
rewrite transfer_dep_r_fun1_eq.
rewrite h2.
unfold transfer_r. unfold eq_rect_r. simpl. 
destruct x as [x h3].
apply proj1_sig_injective. simpl.
simpl.
unfold transfer. unfold eq_rect_r. simpl. 
assert (h5:h3 = (inclusion_reflexive (ba_p_ens B) x h3)). apply proof_irrelevance.
rewrite h5.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
Qed.


Lemma trans_subalg_of_p : 
  forall (T:Type), 
    Transitive (@subalg_of_p T).
intro T. red.
intros Ap Bp Cp h1 h2.
red in h1, h2. destruct h1 as [h3 [h4 h5]], h2 as [h6 [h7 h8]].
red. 
exists (Inclusion_is_transitive _ _ _ _ h3 h6).  
destruct h4 as [h4a h4b h4c h4d h4e], h7 as [h7a h7b h7c h7d h7e].
assert (h9:alg_closed_p (ba_p_ens Ap)
              (Inclusion_is_transitive T (ba_p_ens Ap) 
                 (ba_p_ens Bp) (ba_p_ens Cp) h3 h6)).
  constructor. 
  red. intros x y. unfold Bplus_sub_p.
  pose proof (h4a x y) as h9. unfold Bplus_sub_p in h9.
  destruct x as [x h11], y as [y h12].
  pose proof (ba_p_subst_plus _ _ h8 _ _ (h3 x h11) (h3 y h12)) as h13.
  simpl in h13.
  assert (h14: Ensembles.In
                    (ba_p_ens
                       (Subalg_p Cp (ba_p_ens Bp) h6
                          {|
                          P_c_p := h7a;
                          T_c_p := h7b;
                          O_c_p := h7c;
                          Z_c_p := h7d;
                          C_c_p := h7e |})) x).
    rewrite <- h8. apply (h3 x h11).
  assert (h15 : Ensembles.In
                    (ba_p_ens
                       (Subalg_p Cp (ba_p_ens Bp) h6
                          {|
                          P_c_p := h7a;
                          T_c_p := h7b;
                          O_c_p := h7c;
                          Z_c_p := h7d;
                          C_c_p := h7e |})) y).
    rewrite <- h8. apply (h3 y h12).
  specialize (h13 h14 h15).
  rewrite h13 in h9 at 1.
  assert (h16:h6 x h14 = Inclusion_is_transitive T (ba_p_ens Ap) 
              (ba_p_ens Bp) (ba_p_ens Cp) h3 h6 x h11).
    apply proof_irrelevance.
  assert (h17:h6 y h15 = Inclusion_is_transitive T (ba_p_ens Ap) 
                 (ba_p_ens Bp) (ba_p_ens Cp) h3 h6 y h12).
    apply proof_irrelevance.
  rewrite h16, h17 in h9. 
  assumption.
  red. intros x y.  
  pose proof (h4b x y) as h9. unfold Btimes_sub_p in h9.
  destruct x as [x h11], y as [y h12].
  pose proof (ba_p_subst_times _ _ h8 _ _ (h3 x h11) (h3 y h12)) as h13.
  simpl in h13.
  assert (h14: Ensembles.In
                    (ba_p_ens
                       (Subalg_p Cp (ba_p_ens Bp) h6
                          {|
                          P_c_p := h7a;
                          T_c_p := h7b;
                          O_c_p := h7c;
                          Z_c_p := h7d;
                          C_c_p := h7e |})) x).
    rewrite <- h8. apply (h3 x h11).
  assert (h15 : Ensembles.In
                    (ba_p_ens
                       (Subalg_p Cp (ba_p_ens Bp) h6
                          {|
                          P_c_p := h7a;
                          T_c_p := h7b;
                          O_c_p := h7c;
                          Z_c_p := h7d;
                          C_c_p := h7e |})) y).
    rewrite <- h8. apply (h3 y h12).
  specialize (h13 h14 h15).
  rewrite h13 in h9 at 1.
  assert (h16:h6 x h14 = Inclusion_is_transitive T (ba_p_ens Ap) 
              (ba_p_ens Bp) (ba_p_ens Cp) h3 h6 x h11).
    apply proof_irrelevance.
  assert (h17:h6 y h15 = Inclusion_is_transitive T (ba_p_ens Ap) 
                 (ba_p_ens Bp) (ba_p_ens Cp) h3 h6 y h12).
    apply proof_irrelevance.
  rewrite h16, h17 in h9. 
  assumption.
  red. red in h4c, h7c.
  pose proof (ba_p_subst_one _ _ h8) as h9.
  pose proof h4c as h4c'. rewrite h9 in h4c'.
  pose proof (ba_p_subst_one _ _ h5) as h10.
  simpl in h10, h4c'.
  rewrite h8 in h10.
  assumption.
  red. red in h4d, h7d.
  pose proof (ba_p_subst_zero _ _ h8) as h9.
  pose proof h4d as h4d'. rewrite h9 in h4d'.
  pose proof (ba_p_subst_zero _ _ h5) as h10.
  simpl in h10, h4d'.
  rewrite h8 in h10.
  assumption.
  red. intros. unfold Bcomp_sub_p. destruct x as [x h11].
  red in h4e, h7e. pose proof (h4e (exist _ _ h11)) as h12.
  simpl in h12.
  pose proof (ba_p_subst_comp _ _ h8 _ (h3 x h11)) as h13.
  simpl in h13.
  assert (h14:Ensembles.In
                   (ba_p_ens
                      (Subalg_p Cp (ba_p_ens Bp) h6
                         {|
                         P_c_p := h7a;
                         T_c_p := h7b;
                         O_c_p := h7c;
                         Z_c_p := h7d;
                         C_c_p := h7e |})) x).
    rewrite <- h8. apply (h3 x h11).
  specialize (h13 h14).
  rewrite h13 in h12 at 1. 
  assert (h15:h6 x h14 = Inclusion_is_transitive T (ba_p_ens Ap) 
                (ba_p_ens Bp) (ba_p_ens Cp) h3 h6 x h11).
    apply proof_irrelevance.
  rewrite h15 in h12.
  assumption.
exists h9.
apply bc_inj_p.
assert (h10:A_p T (Bc_p T Ap) = A_p T (Bc_p T (Subalg_p Cp (ba_p_ens Ap)
        (Inclusion_is_transitive T (ba_p_ens Ap) (ba_p_ens Bp) 
           (ba_p_ens Cp) h3 h6) h9))).
  simpl. reflexivity.
assert (h0:sig_set_eq (A_p T (Bc_p T Ap)) (ba_p_ens Ap) h10 = eq_refl).
  apply proof_irrelevance. 
apply (bconst_ext_p _ _ h10); simpl; rewrite h0; try rewrite transfer_dep_r_eq_refl.
apply und_set_p.
apply functional_extensionality. intro x. apply functional_extensionality. intro y.
apply proj1_sig_injective. simpl. destruct x as [x h11], y as [y h12].
red in h4a, h7a. unfold Bplus_sub_p. simpl.
pose proof (ba_p_subst_plus _ _ h5 _ _ h11 h12) as h13. simpl in h13.
assert (h14: Ensembles.In
               (ba_p_ens
                  (Subalg_p Bp (ba_p_ens Ap) h3
                            {|
                              P_c_p := h4a;
                              T_c_p := h4b;
                              O_c_p := h4c;
                              Z_c_p := h4d;
                              C_c_p := h4e |})) x).
  rewrite <- h5. assumption.
assert (h15 : Ensembles.In
                 (ba_p_ens
                    (Subalg_p Bp (ba_p_ens Ap) h3
                              {|
                                P_c_p := h4a;
                                T_c_p := h4b;
                                O_c_p := h4c;
                                Z_c_p := h4d;
                                C_c_p := h4e |})) y).
  rewrite <- h5. assumption.
specialize (h13 h14 h15).
rewrite h13 at 1.
pose proof (ba_p_subst_plus _ _ h8 _ _ (h3 x h14) (h3 y h15)) as h16.
simpl in h16.
assert (h17 : Ensembles.In
                    (ba_p_ens
                       (Subalg_p Cp (ba_p_ens Bp) h6
                          {|
                          P_c_p := h7a;
                          T_c_p := h7b;
                          O_c_p := h7c;
                          Z_c_p := h7d;
                          C_c_p := h7e |})) x).
  rewrite <- h8. apply (h3 x h14).
assert (h18 : Ensembles.In
                    (ba_p_ens
                       (Subalg_p Cp (ba_p_ens Bp) h6
                          {|
                          P_c_p := h7a;
                          T_c_p := h7b;
                          O_c_p := h7c;
                          Z_c_p := h7d;
                          C_c_p := h7e |})) y).
  rewrite <- h8. apply (h3 y h15).
specialize (h16 h17 h18).
rewrite h16 at 1.
assert (h19:h6 x h17 = Inclusion_is_transitive T (ba_p_ens Ap) (ba_p_ens Bp) 
           (ba_p_ens Cp) h3 h6 x h11). apply proof_irrelevance.
assert (h20:h6 y h18 = Inclusion_is_transitive T (ba_p_ens Ap) 
              (ba_p_ens Bp) (ba_p_ens Cp) h3 h6 y h12).
  apply proof_irrelevance.
rewrite h19, h20.
reflexivity.
apply functional_extensionality. intro x. apply functional_extensionality. intro y.
apply proj1_sig_injective. simpl. destruct x as [x h11], y as [y h12].
red in h4a, h7a. unfold Bplus_sub_p. simpl.
pose proof (ba_p_subst_times _ _ h5 _ _ h11 h12) as h13. simpl in h13.
assert (h14: Ensembles.In
               (ba_p_ens
                  (Subalg_p Bp (ba_p_ens Ap) h3
                            {|
                              P_c_p := h4a;
                              T_c_p := h4b;
                              O_c_p := h4c;
                              Z_c_p := h4d;
                              C_c_p := h4e |})) x).
  rewrite <- h5. assumption.
assert (h15 : Ensembles.In
                 (ba_p_ens
                    (Subalg_p Bp (ba_p_ens Ap) h3
                              {|
                                P_c_p := h4a;
                                T_c_p := h4b;
                                O_c_p := h4c;
                                Z_c_p := h4d;
                                C_c_p := h4e |})) y).
  rewrite <- h5. assumption.
specialize (h13 h14 h15).
rewrite h13 at 1.
pose proof (ba_p_subst_times _ _ h8 _ _ (h3 x h14) (h3 y h15)) as h16.
simpl in h16.
assert (h17 : Ensembles.In
                    (ba_p_ens
                       (Subalg_p Cp (ba_p_ens Bp) h6
                          {|
                          P_c_p := h7a;
                          T_c_p := h7b;
                          O_c_p := h7c;
                          Z_c_p := h7d;
                          C_c_p := h7e |})) x).
  rewrite <- h8. apply (h3 x h14).
assert (h18 : Ensembles.In
                    (ba_p_ens
                       (Subalg_p Cp (ba_p_ens Bp) h6
                          {|
                          P_c_p := h7a;
                          T_c_p := h7b;
                          O_c_p := h7c;
                          Z_c_p := h7d;
                          C_c_p := h7e |})) y).
  rewrite <- h8. apply (h3 y h15).
specialize (h16 h17 h18).
rewrite h16 at 1.
assert (h19:h6 x h17 = Inclusion_is_transitive T (ba_p_ens Ap) (ba_p_ens Bp) 
           (ba_p_ens Cp) h3 h6 x h11). apply proof_irrelevance.
assert (h20:h6 y h18 = Inclusion_is_transitive T (ba_p_ens Ap) 
              (ba_p_ens Bp) (ba_p_ens Cp) h3 h6 y h12).
  apply proof_irrelevance.
rewrite h19, h20.
reflexivity.
apply proj1_sig_injective. simpl.
rewrite h5. simpl. rewrite h8. simpl.
reflexivity.
apply proj1_sig_injective. simpl.
rewrite h5. simpl. rewrite h8. simpl.
reflexivity.
apply functional_extensionality. intro x.
destruct x as [x h11]. simpl.
apply proj1_sig_injective.
simpl.
pose proof (ba_p_subst_comp _ _ h5 _ h11) as h12. simpl in h12.
assert (h13: Ensembles.In
                   (ba_p_ens
                      (Subalg_p Bp (ba_p_ens Ap) h3
                         {|
                         P_c_p := h4a;
                         T_c_p := h4b;
                         O_c_p := h4c;
                         Z_c_p := h4d;
                         C_c_p := h4e |})) x).
  rewrite <- h5. assumption.
specialize (h12 h13).
rewrite h12 at 1.
pose proof (ba_p_subst_comp _ _ h8 _ (h3 x h13)) as h14.
assert (h15: Ensembles.In
                   (ba_p_ens
                      (Subalg_p Cp (ba_p_ens Bp) h6
                         {|
                         P_c_p := h7a;
                         T_c_p := h7b;
                         O_c_p := h7c;
                         Z_c_p := h7d;
                         C_c_p := h7e |})) x).
  rewrite <- h8. apply (h3 x h13).
specialize (h14 h15).
rewrite h14 at 1. simpl.
assert (h16:h6 x h15 = Inclusion_is_transitive T (ba_p_ens Ap) 
             (ba_p_ens Bp) (ba_p_ens Cp) h3 h6 x h11).
  apply proof_irrelevance.
rewrite h16.
reflexivity.
Qed.



Lemma alg_closed_p_subalg_p_of : 
  forall {T:Type} (Bp Cp:Bool_Alg_p T) (S:Ensemble T) 
         (pfb:Included S (ba_p_ens Bp)) (pfc:Included S (ba_p_ens Cp)),
    subalg_of_p Bp Cp ->
    alg_closed_p S pfb -> alg_closed_p S pfc.
intros T Bp Cp S h1 h2 h3 h4.
red in h3. destruct h3 as [h3 [h5 h6]]. 
destruct h4 as [h4a h4b h4c h4d h4e].
constructor.
red. intros x y. destruct x as [x h7], y as [y h8]. 
specialize (h4a (exist _ _ h7) (exist _ _ h8)).
assert (h10: Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h3 h5)) x).
  rewrite <- h6. apply h1; auto.
assert (h11:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h3 h5)) y).
  rewrite <- h6. apply h1; auto.
pose proof (ba_p_subst_plus _ _ h6 _ _ (h1 _ h7) (h1 _ h8) h10 h11) as h9.
unfold Bplus_sub_p in h4a. unfold Bplus_sub_p.
rewrite h9 in h4a at 1. simpl in h4a.
assert (h12:h3 x h10 = h2 x h7). apply proof_irrelevance.
assert (h13:h3 y h11 = h2 y h8). apply proof_irrelevance.
rewrite h12, h13 in h4a. assumption.

red. intros x y. destruct x as [x h7], y as [y h8]. 
specialize (h4b (exist _ _ h7) (exist _ _ h8)).
assert (h10: Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h3 h5)) x).
  rewrite <- h6. apply h1; auto.
assert (h11:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h3 h5)) y).
  rewrite <- h6. apply h1; auto.
pose proof (ba_p_subst_times _ _ h6 _ _ (h1 _ h7) (h1 _ h8) h10 h11) as h9.
unfold Btimes_sub_p in h4b. unfold Btimes_sub_p.
rewrite h9 in h4b at 1. simpl in h4b.
assert (h12:h3 x h10 = h2 x h7). apply proof_irrelevance.
assert (h13:h3 y h11 = h2 y h8). apply proof_irrelevance.
rewrite h12, h13 in h4b. assumption.

red. red in h4c.
pose proof (ba_p_subst_one _ _ h6) as h7. rewrite <- h7 at 1.
assumption.
red. red in h4d.
pose proof (ba_p_subst_zero _ _ h6) as h7. rewrite <- h7 at 1.
assumption.

red. intro x. destruct x as [x h7].
specialize (h4e (exist _ _ h7)).
assert (h10: Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h3 h5)) x).
  rewrite <- h6. apply h1; auto.
pose proof (ba_p_subst_comp _  _ h6 _  (h1 _ h7) h10) as h9.
unfold Bcomp_sub_p in h4b. unfold Bcomp_sub_p.
rewrite h9 in h4e at 1. simpl in h4e.
assert (h12:h3 x h10 = h2 x h7). apply proof_irrelevance.
rewrite h12 in h4e. assumption.
Qed.



Lemma subalg_of_p_alg_closed_subalg_compat : 
  forall {T:Type} (Bp Cp:Bool_Alg_p T),
    subalg_of_p Bp Cp ->
    forall (B:Ensemble (bt (ba_conv Bp)))
           (C:Ensemble (bt (ba_conv Cp)))
           (pfb:alg_closed B) (pfc:alg_closed C)
           (S:Ensemble  (bt (Subalg B pfb)))
           (S':Ensemble (bt (Subalg C pfc))),
      im_proj1_sig (im_proj1_sig S) =
      im_proj1_sig (im_proj1_sig S') ->
      (alg_closed S <-> alg_closed S').
intros T Bp Cp h1 B C h2 h3 S S' h4.
split.

(* -> *)
intro h5.
destruct h5 as [h5a h5b h5c h5d h5e].
red in h5a, h5b, h5c, h5d, h5e.
constructor; red.
 
intros x y.
destruct x as [x h6], y as [y h7].
assert (h8:Ensembles.In (im_proj1_sig (im_proj1_sig S')) (proj1_sig (proj1_sig x))).
  destruct x as [x h8]. simpl.
  destruct x as [x h9]. simpl.
  assert (h10:Ensembles.In (im_proj1_sig S')  (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Cp)) x0) x h9)).
    apply Im_intro with (exist (fun a : Btype (Bc (ba_conv Cp)) => Ensembles.In C a)
            (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Cp)) x0) x h9)
            h8).
    assumption.
    apply proj1_sig_injective. simpl. reflexivity.
  apply Im_intro with (exist _ _ h9). 
  assumption. simpl. reflexivity.
rewrite <- h4 in h8.
assert (h8':Ensembles.In (im_proj1_sig (im_proj1_sig S')) (proj1_sig (proj1_sig y))).
  destruct y as [y h8']. simpl.
  destruct y as [y h9']. simpl.
  assert (h10:Ensembles.In (im_proj1_sig S')  (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Cp)) x0) y h9')).
    apply Im_intro with (exist (fun a : Btype (Bc (ba_conv Cp)) => Ensembles.In C a)
            (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Cp)) x0) y h9')
            h8').
    assumption.
    apply proj1_sig_injective. simpl. reflexivity.
  apply Im_intro with (exist _ _ h9'). 
  assumption. simpl. reflexivity.
rewrite <- h4 in h8'.
inversion h8 as [x' h10 ? h11]. subst.  
inversion h8' as [y' h10' ? h11']. subst. 
destruct h10 as [x' h10]. subst.
destruct h10' as [y' h10']. subst. 
specialize (h5a (exist _ _ h10) (exist _ _ h10')). 
destruct x as [x h20], x' as [x' h21], y as [y h22], y' as [y' h23].

destruct x as [x h24], x' as [x' h25], y as [y h26], y' as [y' h27].
simpl in h8, h8', h11, h11'.
subst.
assert (h12:Ensembles.In (im_proj1_sig S)
                         (proj1_sig (Bplus_sub (Subalg B h2) S
             (exist (Ensembles.In S)
                (exist (fun x : Btype (Bc (ba_conv Bp)) => Ensembles.In B x)
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)
                      x' h25) h21) h10)
             (exist (Ensembles.In S)
                (exist (fun x : Btype (Bc (ba_conv Bp)) => Ensembles.In B x)
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)
                      y' h27) h23) h10')))).
  eapply Im_intro. apply h5a. reflexivity.
assert (h13:Ensembles.In (im_proj1_sig (im_proj1_sig S))
                         (proj1_sig
                              (proj1_sig
             (Bplus_sub (Subalg B h2) S
                (exist (Ensembles.In S)
                   (exist
                      (fun x : Btype (Bc (ba_conv Bp)) => Ensembles.In B x)
                      (exist
                         (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) x'
                         h25) h21) h10)
                (exist (Ensembles.In S)
                   (exist
                      (fun x : Btype (Bc (ba_conv Bp)) => Ensembles.In B x)
                      (exist
                         (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) y'
                         h27) h23) h10'))))).
  eapply Im_intro. apply h12. reflexivity.
rewrite h4 in h13.
inversion h13 as [z h14 ? h15]. subst.
destruct z as [z h16]. simpl in h15. subst.
inversion h14 as [z h17 ? h18]. subst.
destruct z as [z h19]. simpl in h18. subst.
subst. subst.
unfold Bplus_sub, Subalg. simpl.
unfold Bplus_sub. simpl.  unfold Bplus_sub in h17. simpl in h17.
unfold Bplus_sub in h17. 
simpl in h17.
assert (h28: exist (fun x : sig_set (A_p T (Bc_p T Cp)) => Ensembles.In C x)
             (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                (proj1_sig
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)
                      x' h25
                    %+ exist
                         (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) y'
                         h27)) h16) h19 =
              exist (Ensembles.In C)
        (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) x' h24
         %+ exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) y' h26)
        (P_c (ba_conv Cp) C h3
           (exist (fun a : sig_set (A_p T (Bc_p T Cp)) => Ensembles.In C a)
              (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) x' h24)
              h20)
           (exist (fun a : sig_set (A_p T (Bc_p T Cp)) => Ensembles.In C a)
              (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) y' h26)
              h22))).
  apply proj1_sig_injective. simpl.
  apply proj1_sig_injective. simpl.
  red in h1. destruct h1 as [h1a [h1b h1c]].
  assert (h30:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                    x').
    rewrite <- h1c. assumption.
  assert (h31:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                    y').
    rewrite <- h1c. assumption.
  pose proof (ba_p_subst_plus _ _ h1c _ _ h25 h27 h30 h31) as h29.
  rewrite h29 at 1.
  simpl.
  f_equal. f_equal. apply proj1_sig_injective. reflexivity.
  apply proj1_sig_injective. reflexivity.
rewrite <- h28 at 1.
assumption.


intros x y.
destruct x as [x h6], y as [y h7].
assert (h8:Ensembles.In (im_proj1_sig (im_proj1_sig S')) (proj1_sig (proj1_sig x))).
  destruct x as [x h8]. simpl.
  destruct x as [x h9]. simpl.
  assert (h10:Ensembles.In (im_proj1_sig S')  (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Cp)) x0) x h9)).
    apply Im_intro with (exist (fun a : Btype (Bc (ba_conv Cp)) => Ensembles.In C a)
            (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Cp)) x0) x h9)
            h8).
    assumption.
    apply proj1_sig_injective. simpl. reflexivity.
  apply Im_intro with (exist _ _ h9). 
  assumption. simpl. reflexivity.
rewrite <- h4 in h8.
assert (h8':Ensembles.In (im_proj1_sig (im_proj1_sig S')) (proj1_sig (proj1_sig y))).
  destruct y as [y h8']. simpl.
  destruct y as [y h9']. simpl.
  assert (h10:Ensembles.In (im_proj1_sig S')  (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Cp)) x0) y h9')).
    apply Im_intro with (exist (fun a : Btype (Bc (ba_conv Cp)) => Ensembles.In C a)
            (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Cp)) x0) y h9')
            h8').
    assumption.
    apply proj1_sig_injective. simpl. reflexivity.
  apply Im_intro with (exist _ _ h9'). 
  assumption. simpl. reflexivity.
rewrite <- h4 in h8'.
inversion h8 as [x' h10 ? h11]. subst.  
inversion h8' as [y' h10' ? h11']. subst. 
destruct h10 as [x' h10]. subst.
destruct h10' as [y' h10']. subst. 
specialize (h5b (exist _ _ h10) (exist _ _ h10')). 
destruct x as [x h20], x' as [x' h21], y as [y h22], y' as [y' h23].

destruct x as [x h24], x' as [x' h25], y as [y h26], y' as [y' h27].
simpl in h8, h8', h11, h11'.
subst. 
assert (h12:Ensembles.In (im_proj1_sig S)
                         (proj1_sig (Btimes_sub (Subalg B h2) S
             (exist (Ensembles.In S)
                (exist (fun x : Btype (Bc (ba_conv Bp)) => Ensembles.In B x)
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)
                      x' h25) h21) h10)
             (exist (Ensembles.In S)
                (exist (fun x : Btype (Bc (ba_conv Bp)) => Ensembles.In B x)
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)
                      y' h27) h23) h10')))).
  eapply Im_intro. apply h5b. reflexivity.
assert (h13:Ensembles.In (im_proj1_sig (im_proj1_sig S))
                         (proj1_sig
                              (proj1_sig
             (Btimes_sub (Subalg B h2) S
                (exist (Ensembles.In S)
                   (exist
                      (fun x : Btype (Bc (ba_conv Bp)) => Ensembles.In B x)
                      (exist
                         (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) x'
                         h25) h21) h10)
                (exist (Ensembles.In S)
                   (exist
                      (fun x : Btype (Bc (ba_conv Bp)) => Ensembles.In B x)
                      (exist
                         (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) y'
                         h27) h23) h10'))))).
  eapply Im_intro. apply h12. reflexivity.
rewrite h4 in h13.
inversion h13 as [z h14 ? h15]. subst.
destruct z as [z h16]. simpl in h15. subst.
inversion h14 as [z h17 ? h18]. subst.
destruct z as [z h19]. simpl in h18. subst.
subst. subst.
unfold Btimes_sub, Subalg. simpl.
unfold Btimes_sub. simpl.  unfold Btimes_sub in h17. simpl in h17.
unfold Btimes_sub in h17. 
simpl in h17.
assert (h28: exist (fun x : sig_set (A_p T (Bc_p T Cp)) => Ensembles.In C x)
             (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                (proj1_sig
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)
                      x' h25
                    %* exist
                         (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) y'
                         h27)) h16) h19 =
              exist (Ensembles.In C)
        (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) x' h24
         %* exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) y' h26)
        (T_c (ba_conv Cp) C h3
           (exist (fun a : sig_set (A_p T (Bc_p T Cp)) => Ensembles.In C a)
              (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) x' h24)
              h20)
           (exist (fun a : sig_set (A_p T (Bc_p T Cp)) => Ensembles.In C a)
              (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) y' h26)
              h22))).
  apply proj1_sig_injective. simpl.
  apply proj1_sig_injective. simpl.
  red in h1. destruct h1 as [h1a [h1b h1c]].
  assert (h30:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                    x').
    rewrite <- h1c. assumption.
  assert (h31:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                    y').
    rewrite <- h1c. assumption.
  pose proof (ba_p_subst_times _ _ h1c _ _ h25 h27 h30 h31) as h29.
  rewrite h29 at 1.
  simpl.
  f_equal. f_equal. apply proj1_sig_injective. reflexivity.
  apply proj1_sig_injective. reflexivity.
rewrite <- h28 at 1.
assumption.


destruct h1 as [h1a [h1b h1c]].
pose proof (ba_p_subst_one _ _ h1c) as h6.
simpl.
simpl in h5c. 
assert (h7:Ensembles.In (im_proj1_sig S) %1).
  apply Im_intro with
  (exist (Ensembles.In B) %1 (O_c (ba_conv Bp) B h2)).
  assumption. simpl. reflexivity.
assert (h8:Ensembles.In (im_proj1_sig (im_proj1_sig S)) (proj1_sig (Bone_p _ (Bc_p _ Bp)))).
  apply Im_intro with %1. assumption.
  reflexivity.
rewrite h4 in h8 at 1.
rewrite h6 in h8.
inversion h8 as [a h9 ? h10]. clear h1c. subst.
destruct a as [a h11]. subst. simpl in h10.
subst.
inversion h9 as [a h12 ? h13]. subst.
destruct a as [a h14]. simpl in h13. subst.
assert (h15:
         exist (Ensembles.In C) %1 (O_c (ba_conv Cp) C h3) =
         exist (fun x : Btype (Bc (ba_conv Cp)) => Ensembles.In C x)
             (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                (proj1_sig %1) h11) h14).
  apply proj1_sig_injective. simpl.
  apply proj1_sig_injective. simpl. reflexivity.
rewrite h15 at 1. assumption.


destruct h1 as [h1a [h1b h1c]].
pose proof (ba_p_subst_zero _ _ h1c) as h6.
simpl.
simpl in h5d. 
assert (h7:Ensembles.In (im_proj1_sig S) %0).
  apply Im_intro with
  (exist (Ensembles.In B) %0 (Z_c (ba_conv Bp) B h2)).
  assumption. simpl. reflexivity.
assert (h8:Ensembles.In (im_proj1_sig (im_proj1_sig S)) (proj1_sig (Bzero_p _ (Bc_p _ Bp)))).
  apply Im_intro with %0. assumption.
  reflexivity.
rewrite h4 in h8 at 1.
rewrite h6 in h8.
inversion h8 as [a h9 ? h10]. clear h1c. subst.
destruct a as [a h11]. subst. simpl in h10.
subst.
inversion h9 as [a h12 ? h13]. subst.
destruct a as [a h14]. simpl in h13. subst.
assert (h15:
         exist (Ensembles.In C) %0 (Z_c (ba_conv Cp) C h3) =
         exist (fun x : Btype (Bc (ba_conv Cp)) => Ensembles.In C x)
             (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                (proj1_sig %0) h11) h14).
  apply proj1_sig_injective. simpl.
  apply proj1_sig_injective. simpl. reflexivity.
rewrite h15 at 1. assumption.


intro x.
destruct x as [x h6].
assert (h8:Ensembles.In (im_proj1_sig (im_proj1_sig S')) (proj1_sig (proj1_sig x))).
  destruct x as [x h8]. simpl.
  destruct x as [x h9]. simpl.
  assert (h10:Ensembles.In (im_proj1_sig S')  (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Cp)) x0) x h9)).
        apply Im_intro with (exist (fun a : Btype (Bc (ba_conv Cp)) => Ensembles.In C a)
            (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Cp)) x0) x h9)
            h8).
        assumption.
    apply proj1_sig_injective. simpl. reflexivity.
  apply Im_intro with (exist _ _ h9). 
  assumption. simpl. reflexivity.
rewrite <- h4 in h8.

inversion h8 as [x' h10 ? h11]. subst.  
destruct h10 as [x' h10]. subst.
specialize (h5e (exist _ _ h10)).
destruct x as [x h20], x' as [x' h21].
destruct x as [x h24], x' as [x' h25].
simpl in h8, h11.
subst.  
assert (h12:Ensembles.In (im_proj1_sig S)
                         (proj1_sig (Bcomp_sub (Subalg B h2) S
             (exist (Ensembles.In S)
                (exist (fun x : Btype (Bc (ba_conv Bp)) => Ensembles.In B x)
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)
                      x' h25) h21) h10)))). 
  eapply Im_intro. apply h5e. reflexivity.
assert (h13:Ensembles.In (im_proj1_sig (im_proj1_sig S))
                         (proj1_sig
                              (proj1_sig
             (Bcomp_sub (Subalg B h2) S
                (exist (Ensembles.In S)
                   (exist
                      (fun x : Btype (Bc (ba_conv Bp)) => Ensembles.In B x)
                      (exist
                         (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) x'
                         h25) h21) h10))))).
  eapply Im_intro. apply h12. reflexivity.
rewrite h4 in h13.
inversion h13 as [z h14 ? h15]. subst.
destruct z as [z h16]. simpl in h15. subst.
inversion h14 as [z h17 ? h18]. subst.
destruct z as [z h19]. simpl in h18. subst.
unfold Bcomp_sub, Subalg. simpl.
unfold Bcomp_sub. simpl.  unfold Btimes_sub in h17. simpl in h17.
unfold Bcomp_sub in h17. 
simpl in h17.
assert (h28:exist (fun x : sig_set (A_p T (Bc_p T Cp)) => Ensembles.In C x)
             (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                (proj1_sig
                   (%-exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)
                        x' h25)) h16) h19 =
            exist (Ensembles.In C)
        (%-exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) x' h24)
        (C_c (ba_conv Cp) C h3
           (exist (fun a : sig_set (A_p T (Bc_p T Cp)) => Ensembles.In C a)
              (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) x' h24)
              h20))).
  apply proj1_sig_injective. simpl.
  apply proj1_sig_injective. simpl.
  red in h1. destruct h1 as [h1a [h1b h1c]].
  assert (h30:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                    x').
    rewrite <- h1c. assumption.
  pose proof (ba_p_subst_comp _ _ h1c _  h25 h30) as h29.
  rewrite h29 at 1.
  simpl.
  f_equal. f_equal. apply proj1_sig_injective. reflexivity.
rewrite <- h28 at 1.
assumption.

(* <- *)
intro h5.
destruct h5 as [h5a h5b h5c h5d h5e].
red in h5a, h5b, h5c, h5d, h5e.
constructor; red.

intros x y.
destruct x as [x h6], y as [y h7].
assert (h8:Ensembles.In (im_proj1_sig (im_proj1_sig S)) (proj1_sig (proj1_sig x))).
  destruct x as [x h8]. simpl.
  destruct x as [x h9]. simpl.
  assert (h10:Ensembles.In (im_proj1_sig S)  (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) x h9)).
    apply Im_intro with (exist (fun a : Btype (Bc (ba_conv Bp)) => Ensembles.In B a)
            (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) x h9)
            h8).
    assumption.
    apply proj1_sig_injective. simpl. reflexivity.
  apply Im_intro with (exist _ _ h9). 
  assumption. simpl. reflexivity.
rewrite h4 in h8.
assert (h8':Ensembles.In (im_proj1_sig (im_proj1_sig S)) (proj1_sig (proj1_sig y))).
  destruct y as [y h8']. simpl.
  destruct y as [y h9']. simpl.
  assert (h10:Ensembles.In (im_proj1_sig S)  (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) y h9')).
    apply Im_intro with (exist (fun a : Btype (Bc (ba_conv Bp)) => Ensembles.In B a)
            (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) y h9')
            h8').
    assumption.
    apply proj1_sig_injective. simpl. reflexivity.
  apply Im_intro with (exist _ _ h9'). 
  assumption. simpl. reflexivity.
rewrite  h4 in h8'.
inversion h8 as [x' h10 ? h11]. subst.  
inversion h8' as [y' h10' ? h11']. subst. 
destruct h10 as [x' h10]. subst.
destruct h10' as [y' h10']. subst. 
specialize (h5a (exist _ _ h10) (exist _ _ h10')). 
destruct x as [x h20], x' as [x' h21], y as [y h22], y' as [y' h23].

destruct x as [x h24], x' as [x' h25], y as [y h26], y' as [y' h27].
simpl in h8, h8', h11, h11'.
subst.
assert (h12:Ensembles.In (im_proj1_sig S')
                         (proj1_sig
                             (Bplus_sub (Subalg C h3) S'
             (exist (Ensembles.In S')
                (exist (fun x : Btype (Bc (ba_conv Cp)) => Ensembles.In C x)
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                      x' h25) h21) h10)
             (exist (Ensembles.In S')
                (exist (fun x : Btype (Bc (ba_conv Cp)) => Ensembles.In C x)
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                      y' h27) h23) h10')))).
  eapply Im_intro. apply h5a. reflexivity.
assert (h13:Ensembles.In (im_proj1_sig (im_proj1_sig S'))
                         (proj1_sig (proj1_sig
                            (Bplus_sub (Subalg C h3) S'
                (exist (Ensembles.In S')
                   (exist
                      (fun x : Btype (Bc (ba_conv Cp)) => Ensembles.In C x)
                      (exist
                         (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) x'
                         h25) h21) h10)
                (exist (Ensembles.In S')
                   (exist
                      (fun x : Btype (Bc (ba_conv Cp)) => Ensembles.In C x)
                      (exist
                         (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) y'
                         h27) h23) h10'))))).
  eapply Im_intro. apply h12. reflexivity.
rewrite <- h4 in h13.
inversion h13 as [z h14 ? h15]. subst.
destruct z as [z h16]. simpl in h15. subst.
inversion h14 as [z h17 ? h18]. subst.
destruct z as [z h19]. simpl in h18. subst.
subst. subst.
unfold Bplus_sub, Subalg. simpl.
unfold Bplus_sub. simpl.  unfold Bplus_sub in h17. simpl in h17.
unfold Bplus_sub in h17. 
simpl in h17.
assert (h28: 
          exist (fun x : sig_set (A_p T (Bc_p T Bp)) => Ensembles.In B x)
                (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)
                (proj1_sig
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                          x' h25
                    %+ exist
                    (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) y'
                    h27)) h16) h19 =
          exist (Ensembles.In B)
        (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) x' h24
         %+ exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) y' h26)
        (P_c (ba_conv Bp) B h2
           (exist (fun a : sig_set (A_p T (Bc_p T Bp)) => Ensembles.In B a)
              (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) x' h24)
              h20)
           (exist (fun a : sig_set (A_p T (Bc_p T Bp)) => Ensembles.In B a)
              (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) y' h26)
              h22))).
  apply proj1_sig_injective. simpl.
  apply proj1_sig_injective. simpl.
  red in h1. destruct h1 as [h1a [h1b h1c]]. 
  assert (h30:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                    x').
    rewrite <- h1c. assumption.
  assert (h31:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                    y').
    rewrite <- h1c. assumption.
  pose proof (ba_p_subst_plus _ _ (eq_sym h1c) _ _ h30 h31 h24 h26) as h29.
  rewrite <- h29 at 1.
  simpl.
  f_equal. f_equal. apply proj1_sig_injective. reflexivity.
  apply proj1_sig_injective. reflexivity.
rewrite <- h28 at 1.
assumption.

intros x y.
destruct x as [x h6], y as [y h7].
assert (h8:Ensembles.In (im_proj1_sig (im_proj1_sig S)) (proj1_sig (proj1_sig x))).
  destruct x as [x h8]. simpl.
  destruct x as [x h9]. simpl.
  assert (h10:Ensembles.In (im_proj1_sig S)  (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) x h9)).
    apply Im_intro with (exist (fun a : Btype (Bc (ba_conv Bp)) => Ensembles.In B a)
            (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) x h9)
            h8).
    assumption.
    apply proj1_sig_injective. simpl. reflexivity.
  apply Im_intro with (exist _ _ h9). 
  assumption. simpl. reflexivity.
rewrite h4 in h8.
assert (h8':Ensembles.In (im_proj1_sig (im_proj1_sig S)) (proj1_sig (proj1_sig y))).
  destruct y as [y h8']. simpl.
  destruct y as [y h9']. simpl.
  assert (h10:Ensembles.In (im_proj1_sig S)  (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) y h9')).
    apply Im_intro with (exist (fun a : Btype (Bc (ba_conv Bp)) => Ensembles.In B a)
            (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) y h9')
            h8').
    assumption.
    apply proj1_sig_injective. simpl. reflexivity.
  apply Im_intro with (exist _ _ h9'). 
  assumption. simpl. reflexivity.
rewrite  h4 in h8'.
inversion h8 as [x' h10 ? h11]. subst.  
inversion h8' as [y' h10' ? h11']. subst. 
destruct h10 as [x' h10]. subst.
destruct h10' as [y' h10']. subst. 
specialize (h5b (exist _ _ h10) (exist _ _ h10')). 
destruct x as [x h20], x' as [x' h21], y as [y h22], y' as [y' h23].

destruct x as [x h24], x' as [x' h25], y as [y h26], y' as [y' h27].
simpl in h8, h8', h11, h11'.
subst.
assert (h12:Ensembles.In (im_proj1_sig S')
                         (proj1_sig
                             (Btimes_sub (Subalg C h3) S'
             (exist (Ensembles.In S')
                (exist (fun x : Btype (Bc (ba_conv Cp)) => Ensembles.In C x)
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                      x' h25) h21) h10)
             (exist (Ensembles.In S')
                (exist (fun x : Btype (Bc (ba_conv Cp)) => Ensembles.In C x)
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                      y' h27) h23) h10')))).
  eapply Im_intro. apply h5b. reflexivity.
assert (h13:Ensembles.In (im_proj1_sig (im_proj1_sig S'))
                         (proj1_sig (proj1_sig
                            (Btimes_sub (Subalg C h3) S'
                (exist (Ensembles.In S')
                   (exist
                      (fun x : Btype (Bc (ba_conv Cp)) => Ensembles.In C x)
                      (exist
                         (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) x'
                         h25) h21) h10)
                (exist (Ensembles.In S')
                   (exist
                      (fun x : Btype (Bc (ba_conv Cp)) => Ensembles.In C x)
                      (exist
                         (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) y'
                         h27) h23) h10'))))).
  eapply Im_intro. apply h12. reflexivity.
rewrite <- h4 in h13.
inversion h13 as [z h14 ? h15]. subst.
destruct z as [z h16]. simpl in h15. subst.
inversion h14 as [z h17 ? h18]. subst.
destruct z as [z h19]. simpl in h18. subst.
subst. subst.
unfold Btimes_sub, Subalg. simpl.
unfold Btimes_sub. simpl.  unfold Bplus_sub in h17. simpl in h17.
unfold Btimes_sub in h17. 
simpl in h17.
assert (h28: 
          exist (fun x : sig_set (A_p T (Bc_p T Bp)) => Ensembles.In B x)
                (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)
                (proj1_sig
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                          x' h25
                    %* exist
                    (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x) y'
                    h27)) h16) h19 =
          exist (Ensembles.In B)
        (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) x' h24
         %* exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) y' h26)
        (T_c (ba_conv Bp) B h2
           (exist (fun a : sig_set (A_p T (Bc_p T Bp)) => Ensembles.In B a)
              (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) x' h24)
              h20)
           (exist (fun a : sig_set (A_p T (Bc_p T Bp)) => Ensembles.In B a)
              (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) y' h26)
              h22))).
  apply proj1_sig_injective. simpl.
  apply proj1_sig_injective. simpl.
  red in h1. destruct h1 as [h1a [h1b h1c]]. 
  assert (h30:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                    x').
    rewrite <- h1c. assumption.
  assert (h31:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                    y').
    rewrite <- h1c. assumption.
  pose proof (ba_p_subst_times _ _ (eq_sym h1c) _ _ h30 h31 h24 h26) as h29.
  rewrite <- h29 at 1.
  simpl.
  f_equal. f_equal. apply proj1_sig_injective. reflexivity.
  apply proj1_sig_injective. reflexivity.
rewrite <- h28 at 1.
assumption.


destruct h1 as [h1a [h1b h1c]].
pose proof (ba_p_subst_one _ _ h1c) as h6.
simpl.
simpl in h5c.  
assert (h7:Ensembles.In (im_proj1_sig S') %1).
  apply Im_intro with
  (exist (Ensembles.In C) %1 (O_c (ba_conv Cp) C h3)).
  assumption. simpl. reflexivity.
assert (h8:Ensembles.In (im_proj1_sig (im_proj1_sig S')) (proj1_sig (Bone_p _ (Bc_p _ Cp)))).
  apply Im_intro with %1. assumption.
  reflexivity.
rewrite <- h4 in h8 at 1.
rewrite <- h6 in h8 at 1.
inversion h8 as [a h9 ? h10]. clear h1c. subst.
destruct a as [a h11]. subst. simpl in h10.
subst.
inversion h9 as [a h12 ? h13]. subst.
destruct a as [a h14]. simpl in h13. subst.
assert (h15:
          exist (fun x : Btype (Bc (ba_conv Bp)) => Ensembles.In B x)
             (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)
                    (proj1_sig %1) h11) h14 =
          exist (Ensembles.In B) %1 (O_c (ba_conv Bp) B h2)).
  apply proj1_sig_injective. simpl.
  apply proj1_sig_injective. simpl. reflexivity.
rewrite <- h15 at 1. assumption.

destruct h1 as [h1a [h1b h1c]].
pose proof (ba_p_subst_zero _ _ h1c) as h6.
simpl.
simpl in h5d.  
assert (h7:Ensembles.In (im_proj1_sig S') %0).
  apply Im_intro with
  (exist (Ensembles.In C) %0 (Z_c (ba_conv Cp) C h3)).
  assumption. simpl. reflexivity.
assert (h8:Ensembles.In (im_proj1_sig (im_proj1_sig S')) (proj1_sig (Bzero_p _ (Bc_p _ Cp)))).
  apply Im_intro with %0. assumption.
  reflexivity.
rewrite <- h4 in h8 at 1.
rewrite <- h6 in h8 at 1.
inversion h8 as [a h9 ? h10]. clear h1c. subst.
destruct a as [a h11]. subst. simpl in h10.
subst.
inversion h9 as [a h12 ? h13]. subst.
destruct a as [a h14]. simpl in h13. subst.
assert (h15:
          exist (fun x : Btype (Bc (ba_conv Bp)) => Ensembles.In B x)
             (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)
                    (proj1_sig %0) h11) h14 =
          exist (Ensembles.In B) %0 (Z_c (ba_conv Bp) B h2)).
  apply proj1_sig_injective. simpl.
  apply proj1_sig_injective. simpl. reflexivity.
rewrite <- h15 at 1. assumption.


intro x.
destruct x as [x h6].
assert (h8:Ensembles.In (im_proj1_sig (im_proj1_sig S)) (proj1_sig (proj1_sig x))).
  destruct x as [x h8]. simpl.
  destruct x as [x h9]. simpl.
  assert (h10:Ensembles.In (im_proj1_sig S)  (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) x h9)).
        apply Im_intro with (exist (fun a : Btype (Bc (ba_conv Bp)) => Ensembles.In B a)
            (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) x h9)
            h8).
        assumption.
    apply proj1_sig_injective. simpl. reflexivity.
  apply Im_intro with (exist _ _ h9). 
  assumption. simpl. reflexivity.
rewrite h4 in h8.

inversion h8 as [x' h10 ? h11]. subst.  
destruct h10 as [x' h10]. subst.
specialize (h5e (exist _ _ h10)).
destruct x as [x h20], x' as [x' h21].
destruct x as [x h24], x' as [x' h25].
simpl in h8, h11.
subst.  
assert (h12:Ensembles.In (im_proj1_sig S')
                         (proj1_sig
                            (Bcomp_sub (Subalg C h3) S'
             (exist (Ensembles.In S')
                (exist (fun x : Btype (Bc (ba_conv Cp)) => Ensembles.In C x)
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                      x' h25) h21) h10)))).
  eapply Im_intro. apply h5e. reflexivity.
assert (h13:Ensembles.In (im_proj1_sig (im_proj1_sig S'))
                         (proj1_sig
                         (proj1_sig
                            (Bcomp_sub (Subalg C h3) S'
             (exist (Ensembles.In S')
                (exist (fun x : Btype (Bc (ba_conv Cp)) => Ensembles.In C x)
                   (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                      x' h25) h21) h10))))).
  eapply Im_intro. apply h12. reflexivity.
rewrite <- h4 in h13.
inversion h13 as [z h14 ? h15]. subst.
destruct z as [z h16]. simpl in h15. subst.
inversion h14 as [z h17 ? h18]. subst.
destruct z as [z h19]. simpl in h18. subst.
unfold Bcomp_sub, Subalg. simpl.
unfold Bcomp_sub. simpl.  unfold Btimes_sub in h17. simpl in h17.
unfold Bcomp_sub in h17. 
simpl in h17.
assert (h28:
          exist (fun x : sig_set (A_p T (Bc_p T Bp)) => Ensembles.In B x)
             (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)
                (proj1_sig
                   (%-exist (fun x : T => Ensembles.In (A_p T (Bc_p T Cp)) x)
                        x' h25)) h16) h19 =
          exist (Ensembles.In B)
        (%-exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) x' h24)
        (C_c (ba_conv Bp) B h2
           (exist (fun a : sig_set (A_p T (Bc_p T Bp)) => Ensembles.In B a)
              (exist (fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x) x' h24)
              h20))).
  apply proj1_sig_injective. simpl.
  apply proj1_sig_injective. simpl.
  red in h1. destruct h1 as [h1a [h1b h1c]].
  assert (h30:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                    x').
    rewrite <- h1c. assumption.
  pose proof (ba_p_subst_comp _ _ (eq_sym h1c) _  h30 h24) as h29.
  rewrite <- h29 at 1.
  simpl.
  f_equal. f_equal. apply proj1_sig_injective. reflexivity.
rewrite <- h28 at 1.
assumption.
Qed.



Lemma sig_set_im_proj1_sig_ba_ens_ba_conv_eq : 
  forall {T:Type} (Bp:Bool_Alg_p T),
    sig_set (im_proj1_sig (ba_ens (ba_conv Bp))) =
    bt (ba_conv Bp).
intros T Bp.
unfold bt, ba_conv, ba_ens, bt. simpl.
f_equal.
apply Extensionality_Ensembles.
red. split.
red.
intros x h1. destruct h1 as [x h1]. subst.
apply proj2_sig.
red.
intros x h1.
apply Im_intro with (exist _ _ h1). constructor.
simpl.
reflexivity.
Defined.


Lemma ba_conv_und_subalg_im_proj1_Sig_ba_ens_ba_conv_eq : 
  forall {T:Type} (Bp Cp:Bool_Alg_p T)
         (hin:Included (im_proj1_sig (ba_ens (ba_conv Bp)))
                 (ba_p_ens Cp))
         (hin':Included (im_proj1_sig (ba_ens (ba_conv Bp)))
          (im_proj1_sig (ba_ens (ba_conv Cp)))),
    ba_conv_und_subalg (im_proj1_sig (ba_ens (ba_conv Bp))) hin = transfer_dep (sig_set_im_proj1_sig_ba_ens_ba_conv_eq Cp)
        (im_proj2_sig (im_proj1_sig (ba_ens (ba_conv Bp))) hin').
intros T Bp Cp h1 h2.
apply Extensionality_Ensembles.
red. split.
red. intros x h5.
destruct h5 as [x h5].  subst. clear h5.
destruct x as [x h5]. simpl.
destruct h5 as [x h5]. subst. 
destruct x as [x h6]. simpl.
rewrite <- (transfer_undoes_transfer_r  (sig_set_im_proj1_sig_ba_ens_ba_conv_eq Cp)).
rewrite <- transfer_in.
unfold im_proj1_sig, im_proj2_sig.
assert (h7:Ensembles.In (Im (ba_ens (ba_conv Bp))
                            (proj1_sig
                               (P:=fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0)))
                        ((proj1_sig
                            (P:=fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0)  (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) x h6)))).
simpl.
eapply Im_intro. apply h5. simpl. reflexivity.
simpl in h7. 
apply Im_intro with (exist _ _ h7). constructor.
simpl.
apply proj1_sig_injective. simpl.
pose (bt (ba_conv Cp)) as test.
unfold bt in test. simpl in test.
pose proof (sig_set_im_proj1_sig_ba_ens_ba_conv_eq Cp) as h8.
unfold bt in h8. simpl in h8.
assert (h9:im_proj1_sig (ba_ens (ba_conv Cp)) =
           A_p T (Bc_p T Cp)).
unfold im_proj1_sig. 
unfold ba_conv, ba_ens, bt.  simpl. 
fold (full_sig (A_p T (Bc_p T Cp))). 
rewrite <- im_full_sig_proj1_sig at 1.
reflexivity.
pose proof (transfer_r_sig_set_eq _ _ h9 h8
                                  (exist (Ensembles.In (A_p T (Bc_p T Cp))) x
                                         (h1 x
                                              (Im_intro (sig_set (A_p T (Bc_p T Bp))) T 
                                                        (ba_ens (ba_conv Bp))
                                                        (proj1_sig
                                                           (P:=fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0))
                                                        (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) x
                                                               h6) h5 x eq_refl)))) as h10.
assert (h11:h8 = sig_set_im_proj1_sig_ba_ens_ba_conv_eq Cp).
apply proof_irrelevance.
subst.
simpl in h10. simpl. 
rewrite h10 at 1.
simpl.
reflexivity.
red.
intros x h3.
rewrite <- (transfer_undoes_transfer_r  (sig_set_im_proj1_sig_ba_ens_ba_conv_eq Cp)) in h3.
rewrite <- transfer_in in h3.
inversion h3 as [a h4 ? h5]. subst. clear h4.
destruct a as [a h4]. simpl in h5.
destruct h4 as [a h4]. subst. destruct a as [a h6].
simpl in h5.
pose proof (f_equal (@proj1_sig _ _) h5) as h7. clear h5.
simpl in h7.
unfold ba_conv_und_subalg, im_proj1_sig.
unfold ba_conv_set, ba_conv_type. rewrite transfer_dep_eq_refl.
assert (h8:Ensembles.In (Im (ba_ens (ba_conv Bp)) (proj1_sig
                 (P:=fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0))) ((proj1_sig
                 (P:=fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0)) (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Bp)) x0) a h6))).
  simpl. subst.
  eapply Im_intro. apply h4. simpl.
  reflexivity.
simpl in h8.
apply Im_intro with (exist _ _ h8). constructor.
simpl.
apply proj1_sig_injective. simpl. subst.
destruct x as [x h9]. simpl.
pose proof (sig_set_im_proj1_sig_ba_ens_ba_conv_eq Cp) as h11.
unfold bt in h11. simpl in h11.
assert (h10:im_proj1_sig (ba_ens (ba_conv Cp)) =
           A_p T (Bc_p T Cp)).
unfold im_proj1_sig. 
unfold ba_conv, ba_ens, bt.  simpl. 
fold (full_sig (A_p T (Bc_p T Cp))). 
rewrite <- im_full_sig_proj1_sig at 1.
reflexivity.
pose proof (transfer_r_sig_set_eq _ _ h10 h11 (exist (fun x0 : T => Ensembles.In (A_p T (Bc_p T Cp)) x0) x h9)) as h13.
assert (h12:h11 = sig_set_im_proj1_sig_ba_ens_ba_conv_eq Cp).
apply proof_irrelevance. 
subst. 
rewrite h13 at 1.
simpl.
reflexivity.
Qed.


Inductive subalg_p_bc_compat 
          {T:Type} (Bp Cp:Bool_Alg_p T) 
          (pf:Included (ba_p_ens Bp) (ba_p_ens Cp)) 
          (pfac:alg_closed_p (ba_p_ens Bp) pf) : Prop :=
  | subalg_p_bc_compat_intro : 
      Bplus_p T (Bc_p T Bp)  =
      restriction_sig2' (Bplus_p T (Bc_p T Cp)) (ba_p_ens Bp) pf 
      (closed_fun2_plus _ _ pf pfac) ->
      Btimes_p T (Bc_p T Bp)  =
      restriction_sig2' (Btimes_p T (Bc_p T Cp)) (ba_p_ens Bp) pf 
      (closed_fun2_times _ _ pf pfac) ->
      proj1_sig (Bzero_p _ (Bc_p _ Bp)) = proj1_sig (Bzero_p _ (Bc_p _ Cp)) ->
      proj1_sig (Bone_p _ (Bc_p _ Bp)) = proj1_sig (Bone_p _ (Bc_p _ Cp)) ->
      Bcomp_p T (Bc_p T Bp)  =
      restriction_sig' (Bcomp_p T (Bc_p T Cp)) (ba_p_ens Bp) pf 
      (closed_fun_comp _ _ pf pfac) ->
      subalg_p_bc_compat Bp Cp pf pfac.


Lemma bs_p_eq_transfer_dep_r_sig_set_eq_full_sig :
  forall {T:Type} (Bp Cp:Bool_Alg_p T)
         (pfi:Included (ba_p_ens Bp) (ba_p_ens Cp)) 
         (pfac:alg_closed_p (ba_p_ens Bp) pfi)
         (pfeq: ba_p_ens Bp =
                ba_p_ens (Subalg_p Cp (ba_p_ens Bp) pfi pfac)),
    BS_p T (Bc_p T Bp) = 
 transfer_dep_r
         (sig_set_eq (A_p T (Bc_p T Bp))
            (A_p T (Bc_p T (Subalg_p Cp (ba_p_ens Bp) pfi pfac))) pfeq)
         (BS_p T (Bc_p T (Subalg_p Cp (ba_p_ens Bp) pfi pfac))).
intros T Bp Cp h1 h2 h3. simpl.
apply Extensionality_Ensembles.
red. split.
red. intros x h4.
rewrite <- (transfer_r_undoes_transfer  (sig_set_eq (A_p T (Bc_p T Bp)) (ba_p_ens Bp) h3)).
rewrite <- transfer_in_r.
constructor.
red. intros.
apply in_bs_p.
Qed.


Lemma subalg_of_p_iff : 
  forall {T:Type} (Bp Cp:Bool_Alg_p T),
    subalg_of_p Bp Cp <->
    exists (pf:Included (ba_p_ens Bp) (ba_p_ens Cp)) 
           (pfac:alg_closed_p (ba_p_ens Bp) pf),
      subalg_p_bc_compat _ _ pf pfac.
intros T Bp Cp. split.
intro h1. red in h1. destruct h1 as [h1 [h2 h3]].
exists h1. exists h2.
constructor. 

apply functional_extensionality. intro x. apply functional_extensionality. intro y.
unfold restriction_sig2'.
apply proj1_sig_injective. simpl.
destruct x as [x h4], y as [y h5]. simpl. 
assert (h7:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1 h2)) x).
  unfold ba_p_ens, Subalg_p. simpl. assumption.
assert (h8:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1 h2)) y).
  unfold ba_p_ens, Subalg_p. simpl. assumption.
pose proof (ba_p_subst_plus _ _ h3  _ _ h4 h5 h7 h8) as h6.
simpl in h6. 
rewrite h6 at 1. f_equal. f_equal. apply proj1_sig_injective; auto.
apply proj1_sig_injective; auto.

apply functional_extensionality. intro x. apply functional_extensionality. intro y.
unfold restriction_sig2'.
apply proj1_sig_injective. simpl.
destruct x as [x h4], y as [y h5]. simpl. 
assert (h7:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1 h2)) x).
  unfold ba_p_ens, Subalg_p. simpl. assumption.
assert (h8:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1 h2)) y).
  unfold ba_p_ens, Subalg_p. simpl. assumption.
pose proof (ba_p_subst_times _ _ h3  _ _ h4 h5 h7 h8) as h6.
simpl in h6. 
rewrite h6 at 1. f_equal. f_equal. apply proj1_sig_injective; auto.
apply proj1_sig_injective; auto.

pose proof (ba_p_subst_zero _ _ h3) as h4. assumption.

pose proof (ba_p_subst_one _ _ h3) as h4. assumption.

apply functional_extensionality. intro x.
unfold restriction_sig2'.
apply proj1_sig_injective. simpl.
destruct x as [x h4]. simpl. 
assert (h7:Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1 h2)) x).
  unfold ba_p_ens, Subalg_p. simpl. assumption.
pose proof (ba_p_subst_comp _ _ h3  _ h4 h7) as h6.
simpl in h6. 
rewrite h6 at 1. f_equal. f_equal. apply proj1_sig_injective; auto.
intro h1.
destruct h1 as [h1 [h2 h3]].
destruct h3 as [h3a h3b h3c h3d h3e].
red. exists h1, h2.
apply bc_inj_p.
assert (h6: A_p T (Bc_p T Bp) =
              A_p T (Bc_p T (Subalg_p Cp (ba_p_ens Bp) h1 h2))).
  unfold Subalg_p. simpl. reflexivity.
pose proof (bconst_ext_p  (Bc_p T Bp) (Bc_p T (Subalg_p Cp (ba_p_ens Bp) h1 h2)) h6) as h5.
apply h5; clear h5.
unfold Subalg_p. simpl.
erewrite bs_p_eq_transfer_dep_r_sig_set_eq_full_sig.
reflexivity.

rewrite h3a.
apply functional_extensionality. intro x. apply functional_extensionality. intro y.
simpl.
unfold restriction_sig2'. 
apply proj1_sig_injective.
f_equal.
apply proj1_sig_injective. simpl.
f_equal.  
rewrite transfer_dep_r_fun2_eq. 
rewrite (transfer_r_sig_set_eq _ _ h6 
                                (sig_set_eq (A_p T (Bc_p T Bp)) (ba_p_ens Bp) h6)) at 1. simpl.
do 2 rewrite (transfer_sig_set_eq _ _ h6
                              (sig_set_eq (A_p T (Bc_p T Bp)) (ba_p_ens Bp) h6)).
simpl.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. reflexivity.

rewrite h3b.
apply functional_extensionality. intro x. apply functional_extensionality. intro y.
simpl.
unfold restriction_sig2'. 
apply proj1_sig_injective.
f_equal.
apply proj1_sig_injective. simpl.
f_equal.  
rewrite transfer_dep_r_fun2_eq.
rewrite (transfer_r_sig_set_eq _ _ h6 
                                (sig_set_eq (A_p T (Bc_p T Bp)) (ba_p_ens Bp) h6)) at 1. simpl.
do 2 rewrite (transfer_sig_set_eq _ _ h6
                              (sig_set_eq (A_p T (Bc_p T Bp)) (ba_p_ens Bp) h6)).
simpl.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. reflexivity.

simpl. 
symmetry. 
assert (h9: (sig_set_eq (A_p T (Bc_p T Bp)) (ba_p_ens Bp) h6) = eq_refl _).
  apply proof_irrelevance.
rewrite h9.
rewrite transfer_dep_r_eq_refl. 
apply proj1_sig_injective. simpl.
symmetry. assumption.

simpl. 
symmetry. 
assert (h9: (sig_set_eq (A_p T (Bc_p T Bp)) (ba_p_ens Bp) h6) = eq_refl _).
  apply proof_irrelevance.
rewrite h9.
rewrite transfer_dep_r_eq_refl. 
apply proj1_sig_injective. simpl.
symmetry. assumption.

rewrite h3e.
apply functional_extensionality. intro x.
simpl.
unfold restriction_sig'. 
apply proj1_sig_injective.
f_equal.
apply proj1_sig_injective. simpl.
rewrite transfer_dep_r_fun1_eq.
rewrite (transfer_r_sig_set_eq _ _ h6 
                                (sig_set_eq (A_p T (Bc_p T Bp)) (ba_p_ens Bp) h6)) at 1. simpl.
rewrite (transfer_sig_set_eq _ _ h6
                              (sig_set_eq (A_p T (Bc_p T Bp)) (ba_p_ens Bp) h6)).
simpl.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
Qed.



(*Hopefully this is not a duplicate of alg_closed_p_iff*)
Lemma subalg_p_alg_closed_lift_sig_type : 
  forall {T:Type} (Bp Cp:Bool_Alg_p T),
    subalg_of_p Bp Cp ->
    exists (pf:Included (ba_p_ens Bp) (ba_p_ens Cp)),
      alg_closed (im_proj2_sig _ pf) (B:=ba_conv Cp).
intros T Bp Cp h1.
red in h1. destruct h1 as [h1 [h2 h3]].
exists h1.
assert (h5:Included (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1 h2)) (ba_p_ens Cp)).
  red. intros x h5.
  rewrite <- (in_ba_p_ens_eq _ _ _ h3) in h5 at 1.
  simpl in h5. apply h1; auto.
pose proof (subsetT_eq_compat _ (fun D => Included (ba_p_ens D) (ba_p_ens Cp))  _ _ h1 h5 h3) as h6. 
pose proof h2 as h2'. rewrite alg_closed_p_iff in h2'.
assumption. 
Qed.

  
End Subalg_p.

Section Gen_Ens_p.
Definition S_clo_cont_A_p {T:Type} {B:Bool_Alg_p T} 
           (A:Ensemble T) (pfa:Included A (ba_p_ens B)) := 
  [S:Ensemble T | Included A S /\ 
                  exists pfs : Included S (ba_p_ens B),
                    alg_closed_p S pfs].

Lemma S_clo_cont_A_p_eq : 
  forall {T:Type} {Bp:Bool_Alg_p T} 
         (A:Ensemble T) (pfa:Included A (ba_p_ens Bp)),
    S_clo_cont_A_p A pfa = 
    Im (S_clo_cont_A (B:=ba_conv Bp) (im_proj2_sig A pfa)) im_proj1_sig.
intros T Bp A h1.
apply Extensionality_Ensembles.
red. split.
red. intros S h2. destruct h2 as [h2]. destruct h2 as [h2 h3].
destruct h3 as [h3 h4].
apply Im_intro with (im_proj2_sig S h3).
constructor. split.
red.
intros x h5. destruct h5 as [x h5]. subst.
destruct x as [x h6]. simpl. clear  h5.
apply Im_intro with (exist _ _ (h2 _ h6)).
constructor. f_equal. apply proof_irrelevance.
rewrite alg_closed_p_iff in h4. assumption.
rewrite <- im_proj1_sig_undoes_im_proj2_sig.
reflexivity.
red.
intros S h2.
destruct h2 as [S h2]. subst.
destruct h2 as [h2]. destruct h2 as [h2 h3]. 
constructor. split. red in h2.
red. intros x h4. specialize (h2 (exist _ _ (h1 _ h4))).
assert (h5: Ensembles.In (im_proj2_sig A h1)
         (exist (Ensembles.In (ba_p_ens Bp)) x (h1 x h4)) ).
  apply Im_intro with (exist _ _ h4). constructor.
  apply proj1_sig_injective. reflexivity.
specialize (h2 h5).
apply Im_intro with (exist _ _ (h1 x h4)). assumption. reflexivity. 
assert (h4: Included (im_proj1_sig S) (ba_p_ens Bp)).
  red; intros x h4. destruct h4 as [x h4]. subst.
  apply proj2_sig.
exists h4.
rewrite alg_closed_p_iff.
rewrite ba_conv_und_subalg_im_proj1_sig.
assumption.
Qed.

 
Definition Gen_Ens_p {T:Type} {Bp:Bool_Alg_p T} (A:Ensemble T) 
           (pf:Included A (ba_p_ens Bp))  : Ensemble T :=
  FamilyIntersection (S_clo_cont_A_p A pf).


Lemma incl_gen_ens_p : 
  forall {T:Type} {Bp:Bool_Alg_p T} (A:Ensemble T)
         (pf:Included A (ba_p_ens Bp)),
    Included (Gen_Ens_p _ pf) (ba_p_ens Bp).
intros T Bp A h1. red.
intros x h2. destruct h2 as [x h2].
apply h2. constructor. split; auto.
exists (inclusion_reflexive _).
apply alg_closed_p_ba_p_ens_refl.
Qed.



Definition gen_ens_p_functional : 
  forall {T:Type} {Bp:Bool_Alg_p T} (A A':Ensemble T) 
         (pfa:Included A (ba_p_ens Bp))
         (pfa':Included A' (ba_p_ens Bp)),
    A = A' ->
    Gen_Ens_p _ pfa = Gen_Ens_p _ pfa'.
intros T Bp A A' h1 h2 h3.
subst.
assert (h4:h1 = h2). apply proof_irrelevance.
subst.
reflexivity.
Qed.


Lemma gen_ens_minimal_p : 
  forall {T:Type} {Bp:Bool_Alg_p T} 
         (A S:Ensemble T)
         (pf:Included A (ba_p_ens Bp)),
  let F := (S_clo_cont_A_p A pf) in Ensembles.In F S -> Included (Gen_Ens_p A pf) S.
intros T Bp A S h1 F h0.
unfold Gen_Ens_p.
red.
intros x h2.
inversion h2 as [? h3]. subst.
apply h3.  assumption.
Qed. 

Lemma gen_ens_minimal_p' : 
  forall {T:Type} {Bp:Bool_Alg_p T} 
         (A S:Ensemble T)
         (pfa:Included A (ba_p_ens Bp))
         (pfs:Included S (ba_p_ens Bp)),
    alg_closed_p S pfs ->
    Included A S -> 
    Included (Gen_Ens_p A pfa) S. 
intros T Bp A S h1 h2 h3 h4.
apply gen_ens_minimal_p. constructor. split; auto.
assert (h5:Included S (ba_p_ens Bp)).
  red; intros. apply h2; auto.
exists h5.
assert (h5 = h2). apply proof_irrelevance. subst.
assumption.
Qed.


 
Lemma gen_ens_ba_conv_und_subalg : 
  forall {T:Type} {Bp:Bool_Alg_p T} (E:Ensemble T)
         (pf:Included E (ba_p_ens Bp)),
    Gen_Ens (ba_conv_und_subalg E pf) = 
    ba_conv_und_subalg (Gen_Ens_p E pf) (incl_gen_ens_p E pf).
intros T Bp E h1.  
apply Extensionality_Ensembles. 
red. split. 
red. 
intros x h2.
pose proof (in_ba_conv_und_subalg_iff _ _ (incl_gen_ens_p _ h1) x) as h0.  
rewrite h0.
destruct h2 as [x h2].  destruct x as [x h3]. simpl.
simpl in h0. constructor.
intros S h4. 
rewrite S_clo_cont_A_p_eq in h4.
destruct h4 as [S h4]. subst.
apply Im_intro with (exist _ _ h3).
apply h2. assumption.
simpl. reflexivity. 
red.
intros x h2. 
pose proof (in_ba_conv_und_subalg_iff _ _ (incl_gen_ens_p _ h1)) as h0. 
rewrite h0 in h2. destruct x as [x hx]. simpl in h2.
destruct h2 as [x h2]. 
constructor.
intros S h3.
specialize (h2 (im_proj1_sig S)). 
rewrite S_clo_cont_A_p_eq in h2.
assert (h4:Ensembles.In (Im (S_clo_cont_A (im_proj2_sig E h1) (B:=ba_conv Bp)) im_proj1_sig)
         (im_proj1_sig S)).
  apply Im_intro with S. assumption. reflexivity.
specialize (h2 h4).
destruct h2 as [x h2]. subst.
destruct x as [x h5]. simpl. simpl in hx.
assert (hx = h5). apply proof_irrelevance. subst.
assumption.
Qed.


Lemma closed_gen_ens_p : 
  forall {T:Type} {Bp:Bool_Alg_p T} (E:Ensemble T)
         (pf:Included E (ba_p_ens Bp)),
    alg_closed_p (Gen_Ens_p E pf) (incl_gen_ens_p _ pf).
intros T Bp E h1.
pose proof closed_gen_ens.
pose proof (closed_gen_ens _ (ba_conv_und_subalg E h1)) as h2. 
rewrite gen_ens_ba_conv_und_subalg in h2.
rewrite <- alg_closed_p_iff in h2.
assumption.
Qed.

Lemma gen_ens_p_eq : 
  forall {T:Type} {Bp:Bool_Alg_p T} (A:Ensemble T) 
         (pf:Included A (ba_p_ens Bp)),
    Gen_Ens_p A pf = im_proj1_sig (Gen_Ens (im_proj2_sig _ pf) (B:=ba_conv Bp)).
intros T Bp A h1.
apply Extensionality_Ensembles.
red. split.
red. intros x h2. 
apply Im_intro with (exist _ _ (incl_gen_ens_p _ _ _ h2)).
destruct h2 as [x h2].
constructor.
intros S h3.
pose proof (h2 (im_proj1_sig S)) as h4. 
assert (h5:Ensembles.In (S_clo_cont_A_p A h1) (im_proj1_sig S)).
  rewrite S_clo_cont_A_p_eq.
  apply Im_intro with S. assumption. reflexivity.
specialize (h4 h5).
destruct h4 as [x h4]. subst.
destruct x as [x h6]. simpl. simpl in h2.
assert (h6 = incl_gen_ens_p A h1 x
           (family_intersection_intro T (S_clo_cont_A_p A h1) x h2)). apply proof_irrelevance.
subst.
assumption. simpl. reflexivity.
red. intros x h2.
destruct h2 as [x h2]. subst.
destruct h2 as [x h2]. constructor.
intros S h3. rewrite S_clo_cont_A_p_eq in h3.
destruct h3 as [S h3]. subst.
specialize (h2 _ h3).
apply Im_intro with x; auto.
Qed.



Lemma gen_ens_p_eq' : 
  forall {T:Type} {Bp:Bool_Alg_p T} (A:Ensemble T) 
         (pf:Included A (ba_p_ens Bp)),
    Gen_Ens_p A pf = im_proj1_sig (Gen_Ens (im_proj2_sig _ (incl_gen_ens_p _ pf)) (B:=ba_conv Bp)).
intros T Bp A h1.
apply Extensionality_Ensembles.
red. split.
red.
intros x h2.  pose proof h2 as h2'.
apply incl_gen_ens_p in h2'.
apply Im_intro with (exist _ _ h2').
constructor.
intros S h3. destruct h3 as [h3].
destruct h3 as [h3l h3r]. 
apply h3l.
apply Im_intro with (exist _ _ h2). constructor.
apply proj1_sig_injective; simpl; reflexivity. reflexivity.
red.
intros x h2.
destruct h2 as [x h2]. subst.
destruct h2 as [x h2]. destruct x as [x h3]. simpl.
specialize (h2 (ba_conv_und_subalg _ (incl_gen_ens_p _ h1))). 
assert (h4: Ensembles.In
         (S_clo_cont_A (B:=ba_conv Bp) (im_proj2_sig (Gen_Ens_p A h1) (incl_gen_ens_p A h1)))
         (ba_conv_und_subalg (Gen_Ens_p A h1) (incl_gen_ens_p A h1))).
  constructor. split. apply inclusion_reflexive.
  rewrite <- alg_closed_p_iff. 
  apply closed_gen_ens_p.
specialize (h2 h4). 
inversion h2 as [x' h5 S h6].  clear h4. subst.
apply exist_injective in h6. subst.
apply proj2_sig.
Qed.



Definition Gen_p {T:Type} {Bp:Bool_Alg_p T} (E:Ensemble T)
           (pf:Included E (ba_p_ens Bp)) : Bool_Alg_p T := 
  Subalg_p _ _ (incl_gen_ens_p _ pf) (closed_gen_ens_p _ pf).


Lemma ba_p_ens_gen_p_eq : 
  forall {T:Type} {Bp:Bool_Alg_p T} (E:Ensemble T)
         (pf:Included E (ba_p_ens Bp)),
    ba_p_ens (Gen_p E pf) = Gen_Ens_p E pf.
intros T Bp E h1.
unfold Gen_p. unfold Gen_Ens_p. unfold ba_p_ens, Subalg_p. simpl.
reflexivity.
Qed.


Lemma gen_p_subalg_of_p_compat : 
  forall {T:Type} {Bp:Bool_Alg_p T} (E:Ensemble T)
         (pf:Included E (ba_p_ens Bp)),
    subalg_of_p (Gen_p _ pf) Bp.
intros T Bp E h1.
red.
assert (h2:Included (ba_p_ens (Gen_p E h1)) (ba_p_ens Bp)).
  rewrite ba_p_ens_gen_p_eq.
  apply incl_gen_ens_p.
exists h2.
assert (h3: alg_closed_p (ba_p_ens (Gen_p E h1)) h2).
  assert (h3:Included (Gen_Ens_p _ h1) (ba_p_ens Bp)).
    apply incl_gen_ens_p.
  pose proof (subsetT_eq_compat _ (fun S=>Included S (ba_p_ens Bp))  _ _ h2 h3 (ba_p_ens_gen_p_eq _ h1)) as h4.
  dependent rewrite -> h4.
  assert (h5:h3 = incl_gen_ens_p _  h1). apply proof_irrelevance.
  subst.
  apply closed_gen_ens_p.
exists h3.
unfold Gen_p.
pose proof (ba_p_ens_subalg_p_compat _ _  (incl_gen_ens_p E h1)
           (closed_gen_ens_p E h1)) as h4.
f_equal.
apply subalg_functional_p.
rewrite h4.
reflexivity.
Qed.



Lemma gen_ens_includes_p : 
  forall {T:Type} {Bp:Bool_Alg_p T} (S:Ensemble T)
    (pf:Included S (ba_p_ens Bp)),
    Included S (Gen_Ens_p S pf).
intros T Bp S h1.
pose proof (gen_ens_includes (ba_conv Bp) (ba_conv_und_subalg S h1)) as h2.
red in h2. red. intros x h3.
specialize (h2 (exist _ _ (h1 _ h3))).
assert (h4: Ensembles.In (ba_conv_und_subalg S h1)
         (exist (Ensembles.In (ba_p_ens Bp)) x (h1 x h3))).
  apply Im_intro with (exist _ _ h3). constructor. simpl.
  f_equal.
specialize (h2 h4). clear h4.
rewrite gen_ens_ba_conv_und_subalg in h2.
inversion h2 as [x' h4 S' h5]. subst. apply exist_injective in h5.
subst.
apply proj2_sig.
Qed.

Lemma gen_ens_preserves_inclusion_p : 
  forall {T:Type} {Bp:Bool_Alg_p T} 
         (S U: Ensemble T) (pfs:Included S (ba_p_ens Bp))
         (pfu:Included U (ba_p_ens Bp)),
    Included S U -> Included (Gen_Ens_p S pfs) (Gen_Ens_p U pfu).
intros T Bp S U h1 h2 h3. 
pose proof (gen_ens_preserves_inclusion (ba_conv_und_subalg S h1)
                                        (ba_conv_und_subalg U h2)) as h4.
assert (h5:Included (ba_conv_und_subalg S h1) (ba_conv_und_subalg U h2)).
  red. intros x h5. destruct x as [x h6]. 
  inversion h5 as [x' h7 S' h8]. apply exist_injective in h8. subst.
  clear h7. destruct x' as [x' h7]. pose proof (h3 _ h7) as h8.
  simpl.
  apply Im_intro with (exist _ _ h8). constructor.
  apply proj1_sig_injective. simpl. reflexivity.
specialize (h4 h5).    
do 2 rewrite  gen_ens_p_eq. 
assert (h6: Included (im_proj1_sig (Gen_Ens (ba_conv_und_subalg S h1)))
         (im_proj1_sig (Gen_Ens (ba_conv_und_subalg U h2)))).
  red. intros x h6. destruct h6 as [x h6]. subst.
  apply h4 in h6. apply Im_intro with x. assumption.
  reflexivity. 
assumption.
Qed.


Lemma in_s_clo_cont_a_p_gen_ens_p: 
  forall {T:Type} {Bp:Bool_Alg_p T} (A:Ensemble T)
         (pf:Included A (ba_p_ens Bp)),
    Ensembles.In (S_clo_cont_A_p A pf) (Gen_Ens_p A pf).
intros T Bp A h1.
constructor. split. 
apply gen_ens_includes_p.
exists (incl_gen_ens_p _ _).
apply closed_gen_ens_p.
Qed.



Definition two_p {T:Type} (Bp:Bool_Alg_p T) :=
  (Couple (proj1_sig (Bzero_p T (Bc_p T Bp))) 
                     (proj1_sig (Bone_p T (Bc_p T Bp)))).

Lemma two_p_eq : 
  forall {T:Type} (Bp:Bool_Alg_p T),
   two_p Bp = im_proj1_sig (Couple (Bzero (Bc (ba_conv Bp)))
                                   (Bone (Bc (ba_conv Bp)))).
intros T  Bp.
apply Extensionality_Ensembles.
red. split.
red.
intros x h1. destruct h1.
apply Im_intro with %0. left. reflexivity.
apply Im_intro with %1. right. reflexivity.
red.
intros x h1. destruct h1 as [x h1]. subst.
destruct h1. left. right.
Qed.



Lemma incl_two_p : 
  forall {T:Type} (Bp:Bool_Alg_p T),
    Included (two_p Bp) (ba_p_ens Bp).
intros T Bp.
red. intros x h1. destruct h1; apply proj2_sig.
Qed.


Lemma ba_conv_und_subalg_two_p : 
  forall {T:Type} (Bp:Bool_Alg_p T),
    ba_conv_und_subalg (two_p Bp) (incl_two_p Bp) =
    Couple 0 1.
intros T Bp.
apply Extensionality_Ensembles.
red. split.
red. intros x h1. destruct x as [x h2].
rewrite in_ba_conv_und_subalg_iff in h1. simpl in h1. 
destruct h1.
simpl. 
rewrite (unfold_sig _ %0) at 1.
assert (h3:h2 = proj2_sig %0).  apply proof_irrelevance. subst. 
left.
simpl. 
rewrite (unfold_sig _ %1) at 1.
assert (h3:h2 = proj2_sig %1).  apply proof_irrelevance. subst. 
right. 
red. 
intros x h1.
rewrite in_ba_conv_und_subalg_iff.
destruct h1. left. right.
Qed.


Lemma alg_closed_two_p : 
  forall {T:Type} (Bp:Bool_Alg_p T),
         alg_closed_p (two_p Bp) (incl_two_p Bp).
intros T Bp.
pose proof (alg_closed_two (ba_conv Bp)) as h1.
rewrite alg_closed_p_iff.
rewrite ba_conv_und_subalg_two_p.
assumption.
Qed.

 
Definition two_bap {T:Type} (Bp:Bool_Alg_p T) :=
  Subalg_p Bp _ (incl_two_p Bp) (alg_closed_two_p Bp).


Lemma gen_ens_closed_eq_p : 
  forall {T:Type} {Bp:Bool_Alg_p T} (S:Ensemble T)
         (pf:Included S (ba_p_ens Bp)),
    alg_closed_p S pf -> Gen_Ens_p S pf = S.
intros T Bp S h1 h2.
rewrite alg_closed_p_iff in h2.
apply gen_ens_closed_eq in h2.
rewrite gen_ens_p_eq.
rewrite h2 at 1.
pose proof (im_proj1_sig_undoes_im_proj2_sig _ _ h1) as h3.
rewrite <- h3 at 1.
reflexivity.
Qed.


Lemma gen_ens_empty_p : 
  forall {T:Type} (Bp:Bool_Alg_p T),
    Gen_Ens_p (Bp:=Bp) (Empty_set T) (empty_inclusion _) = two_p Bp.
intros T Bp.
pose proof (gen_ens_empty (ba_conv Bp)) as h1. 
rewrite gen_ens_p_eq. 
pose proof (gen_ens_ba_conv_und_subalg _ (empty_inclusion (ba_p_ens Bp))) as h2.
rewrite h2 at 1.

pose proof (im_proj1_sig_undoes_im_proj2_sig  _ _ 
              (incl_gen_ens_p (Empty_set T) (empty_inclusion (ba_p_ens Bp)))) as h3.
rewrite <- h3 at 1.
rewrite gen_ens_p_eq.
assert (h4:Empty_set _ = im_proj2_sig (Empty_set T) (empty_inclusion (ba_p_ens Bp))).
  apply Extensionality_Ensembles; red; split; auto with sets.
  red. intros x h4. destruct h4 as [x h4]. destruct x. contradiction.
rewrite <- h4.
rewrite h1 at 1.
rewrite two_p_eq.
reflexivity.
Qed.


Theorem gen_empty_p : 
  forall {T:Type} (Bp:Bool_Alg_p T),
 Gen_p (Empty_set T) (empty_inclusion (ba_p_ens Bp)) = two_bap Bp.
intros T Bp. 
unfold Gen_p, two_bap. 
f_equal.
pose proof (gen_ens_empty_p Bp) as h1.
pose proof (subsetT_eq_compat _ (fun S=>Included S (ba_p_ens Bp)) _ _  (incl_gen_ens_p (Empty_set T) (empty_inclusion (ba_p_ens Bp))) (incl_two_p Bp) h1) as h2.
generalize (closed_gen_ens_p (Empty_set T) (empty_inclusion (ba_p_ens Bp))).
dependent rewrite -> h2.
intro h3. f_equal.
apply proof_irrelevance.
Qed.


Lemma two_incl_gen_ens_p : 
  forall {T:Type} {Bp:Bool_Alg_p T} 
         (E:Ensemble T)
         (pf:Included E (ba_p_ens Bp)),
    Included (two_p Bp) (Gen_Ens_p E pf).
intros T Bp E h0.
assert (h1:Included (Empty_set _) E). auto with sets.
pose proof (gen_ens_preserves_inclusion_p _ _ (empty_inclusion (ba_p_ens Bp)) h0  h1 ) as h2.
rewrite gen_ens_empty_p in h2 at 1.
assumption.
Qed.


Lemma inhabited_gen_ens_p : 
  forall {T:Type} {Bp:Bool_Alg_p T} 
         (E:Ensemble T) (pf:Included E (ba_p_ens Bp)),
    Inhabited (Gen_Ens_p E pf).
intros T Bp E h0.
pose proof (two_incl_gen_ens_p E h0) as h1.
assert (h2:Ensembles.In (two_p Bp) (proj1_sig (Bzero_p T (Bc_p T Bp)))).
constructor.
apply Inhabited_intro with (proj1_sig (Bzero_p T (Bc_p T Bp))).
auto with sets.
Qed.


Lemma bt_subalg_ba_conv_und_subalg_eq : 
  forall {T:Type} {Bp:Bool_Alg_p T} (E:Ensemble T)
         (pfi:Included E (ba_p_ens Bp))
         (pfc:alg_closed (ba_conv_und_subalg E pfi)),
    bt (Subalg (ba_conv_und_subalg E pfi) pfc) = sig_set (im_proj2_sig E pfi).
intros T Bp E h1 h2.
unfold bt, Subalg. simpl.
unfold SubBtype, sig_set.
unfold ba_conv. simpl.   
assert (h3:forall a:sig_set (ba_p_ens Bp), 
             Ensembles.In (ba_conv_und_subalg E h1) a <->
             Ensembles.In (im_proj2_sig E h1) a).
  intro a. destruct a. tauto.
apply proj2_sig_iff_sig_eq; auto.
Qed.




Lemma im_proj1_sig_gen_ens_im_proj2_sig_im_proj2_sig : 
  forall {T:Type} {Bp:Bool_Alg_p T} (E:Ensemble T)
         (pf:Included E (ba_p_ens Bp))
         (pf':Included (im_proj2_sig E pf) (full_sig (ba_p_ens Bp))),
    im_proj1_sig (Gen_Ens (im_proj2_sig (im_proj2_sig E pf) pf')
                  (B:=@Subalg (@ba_conv T Bp) (ba_ens (@ba_conv T Bp))
           (alg_closed_ba_ens (@ba_conv T Bp)))) = 
    Gen_Ens (im_proj2_sig E pf) (B:=ba_conv Bp).
intros T Bp E h1 h2.
apply Extensionality_Ensembles.
red. split.
red. intros x h3.
destruct h3 as [x h3]. subst.
destruct h3 as [x h3].
destruct x as [x h4]. destruct x as [x h5].
simpl. constructor. 
intros S h6. 
assert (h7:forall x:Btype (Bc (ba_conv Bp)), 
             Ensembles.In (ba_ens (ba_conv Bp)) x).
  intro a. destruct a as [a h7].  constructor.
pose (Im S (fun x => (exist _ _ (h7 x)))) as S'.
specialize (h3 S').
assert (h8: Ensembles.In (S_clo_cont_A (im_proj2_sig (im_proj2_sig E h1) h2) (B:=@Subalg (@ba_conv T Bp) (ba_ens (@ba_conv T Bp))
               (alg_closed_ba_ens (@ba_conv T Bp)))) S').
  destruct h6 as [h6]. destruct h6 as [h6a h6b].
  constructor. split.
  red. intros a h8. destruct h8 as [a h8]. subst.
  clear h8. destruct a as [a h8]. destruct h8 as [a h8]. subst.
  simpl. destruct a as [a h9]. simpl.
  assert (h10:Ensembles.In (im_proj2_sig E h1) (exist _ _ (h1 _ h9))).
    eapply Im_intro. apply h8. apply proj1_sig_injective.
    simpl. reflexivity.
  apply h6a in h10.
  unfold  S'.
  apply Im_intro with (exist (Ensembles.In (ba_p_ens Bp)) a (h1 a h9)).
  assumption. 
  apply proj1_sig_injective. simpl. apply proj1_sig_injective; auto.
  rewrite <- alg_closed_improper_subalg. apply h6b.
  apply Extensionality_Ensembles.
  red. split.
  red. intros a h8.
  unfold S', im_proj1_sig. rewrite im_im.
  apply Im_intro with a; auto.
  red. intros a h8.
  destruct h8 as [a h8]. subst. destruct h8 as [a h8]. subst.
  simpl. assumption.
specialize (h3 h8).
inversion h3 as [a h9 ? h10]. subst.
apply exist_injective in h10. subst.
assumption.
red. intros x h3. 
assert (h7:forall x:Btype (Bc (ba_conv Bp)), 
             Ensembles.In (ba_ens (ba_conv Bp)) x).
  intro a. destruct a as [a h7].  constructor.
pose (exist _ _ (h7 x)) as x'.
assert (h8:Ensembles.In (Gen_Ens (im_proj2_sig (im_proj2_sig E h1) h2)
               (B:= (@Subalg (@ba_conv T Bp) (ba_ens (@ba_conv T Bp))
              (alg_closed_ba_ens (@ba_conv T Bp))))) x').
  destruct h3 as [x h3].
  constructor. intros S h4.
  specialize (h3 (im_proj1_sig S)).
  assert (h8: Ensembles.In (S_clo_cont_A (im_proj2_sig E h1) (B:=ba_conv Bp)) (im_proj1_sig S)).
    destruct h4 as [h4]. destruct h4 as [h4 h5].
    constructor. split.
    red. intros a h6. destruct h6 as [a h6]. subst. clear h6.
    destruct a as [a h6]. simpl.
    assert (h8:Ensembles.In (im_proj2_sig E h1) (exist _ _ (h1 _ h6))).
      assert (h9:Ensembles.In (full_sig E) (exist _ _ h6)).
        constructor.
      eapply Im_intro. apply h9.
      apply proj1_sig_injective. simpl. reflexivity.
    assert (h9: Ensembles.In (im_proj2_sig (im_proj2_sig E h1) h2)
                             (exist _ _ (h2 _ h8))).
      assert (h9:Ensembles.In (full_sig (im_proj2_sig E h1))
                              (exist _ _ h8)).
        constructor.
      eapply Im_intro. apply h9.
    apply proj1_sig_injective. simpl. reflexivity.
    apply h4 in h9.
    apply Im_intro with  (exist (Ensembles.In (full_sig (ba_p_ens Bp)))
            (exist (Ensembles.In (ba_p_ens Bp)) a (h1 a h6))
            (h2 (exist (Ensembles.In (ba_p_ens Bp)) a (h1 a h6)) h8)).
    assumption. apply proj1_sig_injective. simpl. reflexivity.
    rewrite alg_closed_improper_subalg. apply h5. reflexivity.
specialize (h3 h8). 
destruct h3 as [x h3].
assert (h9:x = x'). unfold x'. apply proj1_sig_injective. simpl.
  rewrite H. reflexivity. rewrite <- h9. assumption.
apply Im_intro with x'. assumption.
unfold x'. simpl. reflexivity.
Qed.


Lemma gen_ens_p_subalg_of_p : 
  forall {T:Type} {Bp Cp:Bool_Alg_p T},
    subalg_of_p Bp Cp ->
    forall (E:Ensemble T) 
           (pfb:Included E (ba_p_ens Bp))
           (pfc:Included E (ba_p_ens Cp)),
      Gen_Ens_p E pfb = Gen_Ens_p E pfc.
intros T Bp Cp h1 E h2 h3. 
rewrite (gen_ens_p_eq E h3).
pose proof h1 as h0.
rewrite subalg_of_p_iff in h1.
destruct h1 as [h1a [h1b h1c]].
pose proof h1b as h4.
rewrite alg_closed_p_iff in h4.
assert (h7:Included (im_proj2_sig _ h3) (im_proj2_sig _ h1a)).
  red. intros x h7. 
  destruct h7 as [x h7].  subst. clear h7. 
  destruct x as [x h7]. simpl.
  apply Im_intro with (exist _ _ (h2 _ h7)). constructor.
  apply proj1_sig_injective. simpl. reflexivity. 
pose (im_proj2_sig (im_proj2_sig _ h3) h7) as E'. 
pose proof (gen_ens_subalg_eq h4 E') as h5.  
unfold E' in h5.
rewrite <- im_proj1_sig_undoes_im_proj2_sig in h5 at 1.   
rewrite <- h5 at 1.
assert (h8:Included (im_proj2_sig E h2) (full_sig (ba_p_ens Bp))).
  red; intros; constructor.
assert (he:im_proj1_sig (im_proj1_sig (Gen_Ens (im_proj2_sig (im_proj2_sig E h3) h7) (B:=@Subalg (@ba_conv T Cp)
              (@ba_conv_und_subalg T Cp (@ba_p_ens T Bp) h1a) h4)))
           = 
           im_proj1_sig (
           im_proj1_sig (Gen_Ens (im_proj2_sig (im_proj2_sig E h2)h8) (B:=Subalg _ (alg_closed_ba_ens (ba_conv Bp)))))).
  apply Extensionality_Ensembles. 
  red. split.
  red.
  intros x h9.
  destruct h9 as [x h9]. subst. destruct h9 as [x h9]. subst.
  destruct x as [x h10]. simpl. destruct h10 as [x h10]. subst. 
  inversion h9 as [a h11 q]. subst. simpl. clear h9.
  apply Im_intro with x.
  apply Im_intro with (exist _ _ h10).
  constructor.
  intros S h13.  simpl  in S. unfold SubBtype in S.    
  simpl in h11. unfold SubBtype in h11. simpl in h11.  
  pose proof h1a as h1'. rewrite ba_p_ens_eq in h1'.
    assert (h14:forall d: {a : Btype (Bc (ba_conv Bp)) | Ensembles.In (ba_ens (ba_conv Bp)) a},
                  Ensembles.In (ba_p_ens Cp)
                               (proj1_sig (proj1_sig d))).
    intro d.  destruct d as [d h15].  destruct d as [d h16]. 
      pose proof h16 as h16'. apply h1a in h16'.  simpl. 
      assumption.
      assert (h15:forall d:{a:Btype (Bc (ba_conv Bp)) | Ensembles.In (ba_ens (ba_conv Bp)) a},
                  Ensembles.In (ba_conv_und_subalg (ba_p_ens Bp) h1a) (exist _ _ (h14 d))).
      intro d. unfold ba_conv_und_subalg. unfold ba_conv_set, ba_conv_type.
      rewrite transfer_dep_eq_refl.
      destruct d as [d h15].
      apply Im_intro with d. constructor. simpl.
      apply proj1_sig_injective. simpl.
      reflexivity.
 pose (Im S (fun d => (exist _ _ (h15 d)))) as S'.
 specialize (h11 S').  
 assert (h16:Ensembles.In (S_clo_cont_A  (B:=@Subalg (@ba_conv T Cp)
                (@ba_conv_und_subalg T Cp (@ba_p_ens T Bp) h1a) h4) (im_proj2_sig (im_proj2_sig E h3) h7)) S').
   destruct h13 as [h13]. destruct h13 as [h13a h13b].
   constructor. split. 
   red. intros a h16.  destruct h16 as [a h16]. subst. clear h16.
   destruct a as [a h16]. simpl. destruct h16 as [a h16]. subst.
   destruct a as [a h17]. simpl.  
   pose (exist _ _ (h2 _ h17)) as a'.
   assert (h18:Ensembles.In (im_proj2_sig E h2) a').
     apply Im_intro with (exist _ _ h17). constructor.
     unfold a'. apply proj1_sig_injective. simpl. reflexivity.
   pose (exist _ _ (h8 _ h18)) as a''.
   specialize (h13a a''). 
   assert (h19: Ensembles.In (im_proj2_sig (im_proj2_sig E h2) h8) a'' ). unfold a''.
     apply Im_intro with (exist _ _ h18). constructor.
     apply proj1_sig_injective. simpl. reflexivity.
   specialize (h13a h19). unfold S'.
   apply Im_intro with a''. assumption.
   apply proj1_sig_injective. simpl.
   apply proj1_sig_injective. simpl. reflexivity.
   assert (h20 :im_proj1_sig (im_proj1_sig S) =
               im_proj1_sig (im_proj1_sig S')).
     apply Extensionality_Ensembles.
     red. split.
     red. intros t h21. destruct h21 as [t h21]. subst.
     destruct h21 as [t h21]. subst. destruct t as [t h22].
     destruct t as [t h23]. simpl.
     apply Im_intro with (exist _ _ (h1a _ h23)).
     unfold S'. unfold im_proj1_sig. rewrite im_im. simpl.
     apply Im_intro with (exist _ _ h22). assumption.
     apply proj1_sig_injective. simpl. reflexivity.
     simpl. reflexivity.
     red. intros t h21.
     destruct h21 as [t h21]. subst.
     destruct h21 as [t h21]. subst. destruct t as [t h22].
     destruct h22 as [t h22]. subst. simpl.
     destruct t as [t h23]. simpl. simpl in h21.
     apply Im_intro with (exist _ _ h23).
     apply Im_intro with (exist _ _ h22).
     inversion h21 as [a h24 ? h25]. subst.
     apply exist_injective in h25. apply exist_injective in h25.
     subst.
     assert (h25:a = (exist (Ensembles.In (full_sig (ba_p_ens Bp)))
        (exist (fun x0 : T => Ensembles.In (ba_p_ens Bp) x0)
           (proj1_sig (proj1_sig a)) h23) h22)).
       apply proj1_sig_injective. simpl. apply proj1_sig_injective.
       simpl. reflexivity.
     rewrite <- h25 at 1. assumption.
     simpl. apply proj1_sig_injective. simpl. reflexivity.
     simpl. reflexivity.
  erewrite <- subalg_of_p_alg_closed_subalg_compat.
  apply h13b. assumption. assumption.
  specialize (h11 h16).
  inversion h11 as [a h17 ? h18]. subst. clear h11.
  apply exist_injective in h18. apply exist_injective in h18.
  destruct x as [x h19]. simpl in h18. subst.
  assert (h20:a = exist (Ensembles.In (full_sig (ba_p_ens Bp)))
        (exist (fun x : T => Ensembles.In (ba_p_ens Bp) x)
           (proj1_sig (proj1_sig a)) h19) h10).
  apply proj1_sig_injective; apply proj1_sig_injective; auto.
  rewrite <- h20 at 1. assumption.
  simpl. reflexivity. reflexivity.
 
  red. intros x h9. destruct h9 as [x h9]. subst. 
  destruct h9 as [x h9]. subst. destruct h9 as [x h9].
  destruct x as [x h10]. destruct x as [x h11].
  simpl. 

  apply Im_intro with (exist _ _ (h1a _ h11)).
  assert (h12:Ensembles.In (ba_conv_und_subalg (ba_p_ens Bp) h1a) (exist _ _ (h1a _ h11))).
    apply Im_intro with (exist _ _ h11). constructor.
    apply proj1_sig_injective. simpl. reflexivity.
    assert (h14:Ensembles.In (Gen_Ens (im_proj2_sig (im_proj2_sig E h3) h7) (B:=(@Subalg (@ba_conv T Cp)
              (@ba_conv_und_subalg T Cp (@ba_p_ens T Bp) h1a) h4))) (exist _ _ h12)).
      constructor.
      intros S h13.
      assert (h14:forall x:(SubBtype (ba_conv Cp) (ba_conv_und_subalg (ba_p_ens Bp) h1a)),
                    Ensembles.In (ba_p_ens Bp) (proj1_sig (proj1_sig x))).
        intro a. destruct a as [a h14]. simpl.
        destruct h14 as [a h14]. subst. clear h14.
        destruct a as [a h14]. simpl. assumption. 
      assert (h15:forall x:(SubBtype (ba_conv Cp) (ba_conv_und_subalg (ba_p_ens Bp) h1a)),
                    Ensembles.In (ba_ens (ba_conv Bp)) (exist _ _ (h14 x))).
        intro a. destruct a as [a h15]. simpl.
        destruct h15 as [a h15]. subst. destruct a as [a h16].
        simpl. pose proof h16 as h16'. rewrite ba_p_ens_eq in h16'.
        destruct h16' as [a h16']. subst.
        assert (h17:a = exist (Ensembles.In (ba_p_ens Bp)) (proj1_sig a)
        (h14
           (exist
              (fun a0 : sig_set (A_p T (Bc_p T Cp)) =>
               Ensembles.In (ba_conv_und_subalg (ba_p_ens Bp) h1a) a0)
              (exist (Ensembles.In (A_p T (Bc_p T Cp))) 
                 (proj1_sig a) (h1a (proj1_sig a) h16))
              (Im_intro (sig_set (ba_p_ens Bp))
                 {x | Ensembles.In (A_p T (Bc_p T Cp)) x}
                 (full_sig (ba_p_ens Bp))
                 (fun x0 : sig_set (ba_p_ens Bp) =>
                  exist (Ensembles.In (A_p T (Bc_p T Cp))) 
                    (proj1_sig x0) (h1a (proj1_sig x0) (proj2_sig x0)))
                 (exist (fun x0 : T => Ensembles.In (ba_p_ens Bp) x0)
                    (proj1_sig a) h16) h15
                 (exist (Ensembles.In (A_p T (Bc_p T Cp))) 
                    (proj1_sig a) (h1a (proj1_sig a) h16)) eq_refl)))).
          apply proj1_sig_injective.  simpl. reflexivity.
        rewrite <- h17 at 1. assumption.
      pose (Im S (fun a => (exist _ _ (h15 a)))) as S'.
      specialize (h9 S').
    assert (h16: Ensembles.In (S_clo_cont_A (im_proj2_sig (im_proj2_sig E h2) h8) (B:=(@Subalg (@ba_conv T Bp) (ba_ens (@ba_conv T Bp))
               (alg_closed_ba_ens (@ba_conv T Bp))))) S').
      destruct h13 as [h13]. destruct h13 as [h13a h13b].
      constructor. split.
      red. intros a h16. destruct h16 as [a h16]. subst.
      clear h16. destruct a as [a h16]. simpl. 
      destruct h16 as [a h16]. subst. 
      destruct a as [a h17].  simpl.
      pose (exist _ _ (h3 _ h17)) as a'.
      assert (h18:Ensembles.In (im_proj2_sig  E h3) a').
        eapply Im_intro. apply h16. unfold a'.
        apply proj1_sig_injective. simpl. reflexivity.
      pose (exist _ _ (h7 _ h18)) as a''.
      assert (h19:Ensembles.In (im_proj2_sig (im_proj2_sig E h3) h7) a'').
      eapply Im_intro. 
      assert (h19:Ensembles.In (full_sig (im_proj2_sig E h3)) (exist _ _ h18)).
        constructor.
      apply h19. unfold a''. apply proj1_sig_injective.
      simpl. reflexivity.
      apply h13a in h19.
      unfold S'.
      apply Im_intro with a''; auto.
      apply proj1_sig_injective. simpl. apply proj1_sig_injective.
      simpl. reflexivity.
      erewrite subalg_of_p_alg_closed_subalg_compat. apply h13b.
      assumption.
      apply Extensionality_Ensembles.
      red. split.
      red. intros a h16. destruct h16 as [a h16]. subst.
      destruct h16 as [a h16]. subst. destruct h16 as [a h16].
      subst. simpl.
      destruct a as [a h17]. simpl.
      destruct h17 as [a h17]. subst. destruct a as [a h18].
      simpl. simpl in h16.
      apply Im_intro with (exist _ _ (h1a _ h18)).
      assert (h19:Ensembles.In (ba_conv_und_subalg (ba_p_ens Bp) h1a) (exist _ _ (h1a _ h18))).
        eapply Im_intro. apply h17. apply proj1_sig_injective.
        simpl. reflexivity.
      apply Im_intro with (exist _ _ h19).
      assert (h20: exist
             (fun a0 : sig_set (A_p T (Bc_p T Cp)) =>
              Ensembles.In (ba_conv_und_subalg (ba_p_ens Bp) h1a) a0)
             (exist (Ensembles.In (A_p T (Bc_p T Cp))) a (h1a a h18))
             (Im_intro (sig_set (ba_p_ens Bp))
                {x | Ensembles.In (A_p T (Bc_p T Cp)) x}
                (full_sig (ba_p_ens Bp))
                (fun x0 : sig_set (ba_p_ens Bp) =>
                 exist (Ensembles.In (A_p T (Bc_p T Cp))) 
                   (proj1_sig x0) (h1a (proj1_sig x0) (proj2_sig x0)))
                (exist (fun x0 : T => Ensembles.In (ba_p_ens Bp) x0) a h18)
                h17 (exist (Ensembles.In (A_p T (Bc_p T Cp))) a (h1a a h18))
                eq_refl) = 
                   exist (Ensembles.In (ba_conv_und_subalg (ba_p_ens Bp) h1a))
        (exist (Ensembles.In (ba_p_ens Cp)) a (h1a a h18)) h19).
        apply proj1_sig_injective. simpl. apply proj1_sig_injective.
        simpl. reflexivity.
      rewrite <- h20 at 1. assumption.
    apply proj1_sig_injective. simpl. reflexivity. simpl.
    reflexivity.
    red. intros a h16.
    destruct h16 as [a h16]. subst. destruct h16 as [a h16].
    subst. destruct  a as [a h17]. simpl.
    destruct h17 as [a h17]. subst. destruct a as [a h18].
    simpl. simpl in h16.
    unfold S'. unfold im_proj1_sig. rewrite im_im. rewrite im_im.
    simpl.
    apply Im_intro with (exist
             (fun x0 : sig_set (A_p T (Bc_p T Cp)) =>
              Ensembles.In (ba_conv_und_subalg (ba_p_ens Bp) h1a) x0)
             (exist (Ensembles.In (A_p T (Bc_p T Cp))) a (h1a a h18))
             (Im_intro (sig_set (ba_p_ens Bp))
                {x | Ensembles.In (A_p T (Bc_p T Cp)) x}
                (full_sig (ba_p_ens Bp))
                (fun x0 : sig_set (ba_p_ens Bp) =>
                 exist (Ensembles.In (A_p T (Bc_p T Cp))) 
                   (proj1_sig x0) (h1a (proj1_sig x0) (proj2_sig x0)))
                (exist (fun x0 : T => Ensembles.In (ba_p_ens Bp) x0) a h18)
                h17 (exist (Ensembles.In (A_p T (Bc_p T Cp))) a (h1a a h18))
                eq_refl)).
      assumption. simpl. reflexivity.
  specialize (h9 h16).
  inversion h9 as [a h17 ? h18]. subst. clear h9.
  apply exist_injective in h18. apply exist_injective in h18.
  subst.
  assert (h21:a = (exist (Ensembles.In (ba_conv_und_subalg (ba_p_ens Bp) h1a))
        (exist (Ensembles.In (ba_p_ens Cp)) (proj1_sig (proj1_sig a))
           (h1a (proj1_sig (proj1_sig a)) h11)) h12)).
    destruct a as [a h19]. simpl.
    apply proj1_sig_injective. simpl. destruct a as [a h20].
    apply proj1_sig_injective. simpl. reflexivity.
    rewrite <- h21 at 1. assumption.
  apply Im_intro with  (exist (Ensembles.In (ba_conv_und_subalg (ba_p_ens Bp) h1a))
             (exist (Ensembles.In (ba_p_ens Cp)) x (h1a x h11)) h12).
  assumption. apply proj1_sig_injective. simpl.
  reflexivity. simpl. reflexivity.
rewrite he at 1.
rewrite gen_ens_p_eq.
f_equal. symmetry.
apply im_proj1_sig_gen_ens_im_proj2_sig_im_proj2_sig.
Qed.


Lemma gen_p_subalg_of_p : 
  forall {T:Type} {Bp Cp:Bool_Alg_p T},
    subalg_of_p Bp Cp ->
    forall (E:Ensemble T) 
           (pfb:Included E (ba_p_ens Bp))
           (pfc:Included E (ba_p_ens Cp)),
      Gen_p E pfb = Gen_p E pfc.
intros T Bp Cp h1 E h2 h3.
pose proof h1 as h1'.
destruct h1' as [h1a [h1b h1c]].
apply bc_inj_p.
pose proof (gen_ens_p_subalg_of_p h1 E h2 h3) as h4.
do 2 rewrite <- ba_p_ens_gen_p_eq in h4.
apply (bconst_ext_p _ _ h4).

apply Extensionality_Ensembles.
red. split.
red.
intros x h5.
rewrite <- (transfer_r_undoes_transfer (sig_set_eq (A_p T (Bc_p T (Gen_p E h2))) (A_p T (Bc_p T (Gen_p E h3))) h4) x).
rewrite <- transfer_in_r. constructor.
red.
intros x h5. constructor.

apply functional_extensionality. intro x. apply functional_extensionality. intro y.
rewrite <- transfer_fun2_r_transfer_dep_r_compat'.
rewrite <- (transfer_r_undoes_transfer (sig_set_eq (A_p T (Bc_p T (Gen_p E h2))) (A_p T (Bc_p T (Gen_p E h3))) h4) x) at 2.
rewrite <- (transfer_r_undoes_transfer (sig_set_eq (A_p T (Bc_p T (Gen_p E h2))) (A_p T (Bc_p T (Gen_p E h3))) h4) y) at 2.
rewrite transfer_fun2_r_compat'. 
simpl.
apply proj1_sig_injective.
simpl.
destruct x as [x h5], y as [y h6]. simpl.
rewrite (transfer_r_sig_set_eq _ _ h4 (sig_set_eq (A_p T (Bc_p T (Gen_p E h2))) (A_p T (Bc_p T (Gen_p E h3))) h4)).
simpl.
do  2 rewrite (transfer_sig_set_eq _ _ h4 (sig_set_eq (A_p T (Bc_p T (Gen_p E h2))) (A_p T (Bc_p T (Gen_p E h3))) h4)).
simpl.
assert (h8: Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                   x).
  rewrite <- h1c. apply (incl_gen_ens_p E h2 x h5).
assert (h9 : Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                   y).
  rewrite <- h1c. apply (incl_gen_ens_p E h2 y h6).
pose proof (ba_p_subst_plus _ _ h1c _ _ (incl_gen_ens_p E h2 x h5)
                            (incl_gen_ens_p E h2 y h6) h8 h9) as h7.
rewrite h7 at 1.
unfold Subalg_p. simpl.
f_equal.
f_equal. apply proj1_sig_injective. reflexivity. 
apply proj1_sig_injective. reflexivity.

apply functional_extensionality. intro x. apply functional_extensionality. intro y.
rewrite <- transfer_fun2_r_transfer_dep_r_compat'.
rewrite <- (transfer_r_undoes_transfer (sig_set_eq (A_p T (Bc_p T (Gen_p E h2))) (A_p T (Bc_p T (Gen_p E h3))) h4) x) at 2.
rewrite <- (transfer_r_undoes_transfer (sig_set_eq (A_p T (Bc_p T (Gen_p E h2))) (A_p T (Bc_p T (Gen_p E h3))) h4) y) at 2.
rewrite transfer_fun2_r_compat'. 
simpl.
apply proj1_sig_injective.
simpl.
destruct x as [x h5], y as [y h6]. simpl.
rewrite (transfer_r_sig_set_eq _ _ h4 (sig_set_eq (A_p T (Bc_p T (Gen_p E h2))) (A_p T (Bc_p T (Gen_p E h3))) h4)).
simpl.
do  2 rewrite (transfer_sig_set_eq _ _ h4 (sig_set_eq (A_p T (Bc_p T (Gen_p E h2))) (A_p T (Bc_p T (Gen_p E h3))) h4)).
simpl.
assert (h8: Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                   x).
  rewrite <- h1c. apply (incl_gen_ens_p E h2 x h5).
assert (h9 : Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                   y).
  rewrite <- h1c. apply (incl_gen_ens_p E h2 y h6).
pose proof (ba_p_subst_times _ _ h1c _ _ (incl_gen_ens_p E h2 x h5)
                            (incl_gen_ens_p E h2 y h6) h8 h9) as h7.
rewrite h7 at 1.
unfold Subalg_p. simpl.
f_equal.
f_equal. apply proj1_sig_injective. reflexivity. 
apply proj1_sig_injective. reflexivity.


symmetry. simpl. 
pose proof (ba_p_subst_one _ _ h1c) as h5. 
apply proj1_sig_injective. simpl.
rewrite transfer_dep_r_id_transfer_r_compat.
rewrite (transfer_r_sig_set_eq _ _ h4 (sig_set_eq (Gen_Ens_p E h2) (Gen_Ens_p E h3) h4)).
simpl.
rewrite h5.
reflexivity.


symmetry. simpl. 
pose proof (ba_p_subst_zero _ _ h1c) as h5. 
apply proj1_sig_injective. simpl.
rewrite transfer_dep_r_id_transfer_r_compat.
rewrite (transfer_r_sig_set_eq _ _ h4 (sig_set_eq (Gen_Ens_p E h2) (Gen_Ens_p E h3) h4)).
simpl.
rewrite h5.
reflexivity.

apply functional_extensionality. intro x.
rewrite <- transfer_fun_r_transfer_dep_r_compat'.
rewrite <- (transfer_r_undoes_transfer (sig_set_eq (A_p T (Bc_p T (Gen_p E h2))) (A_p T (Bc_p T (Gen_p E h3))) h4) x) at 2.
rewrite transfer_fun_r_compat'. 
simpl.
apply proj1_sig_injective.
simpl.
destruct x as [x h5].
rewrite (transfer_r_sig_set_eq _ _ h4 (sig_set_eq (A_p T (Bc_p T (Gen_p E h2))) (A_p T (Bc_p T (Gen_p E h3))) h4)).
simpl.
rewrite (transfer_sig_set_eq _ _ h4 (sig_set_eq (A_p T (Bc_p T (Gen_p E h2))) (A_p T (Bc_p T (Gen_p E h3))) h4)).
simpl.
assert (h8: Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h1a h1b))
                   x).
  rewrite <- h1c. apply (incl_gen_ens_p E h2 x h5).
pose proof (ba_p_subst_comp _ _ h1c _ (incl_gen_ens_p E h2 x h5)
                            h8) as h7.
rewrite h7 at 1.
unfold Subalg_p. simpl.
f_equal.
f_equal. apply proj1_sig_injective. reflexivity. 
Qed.


Lemma subalg_of_p_gen_p_gen_p : 
  forall {T:Type} (Bp:Bool_Alg_p T) (E F:Ensemble T)
         (pfi:Included E F)
         (pfe:Included E (ba_p_ens Bp))
         (pff:Included F (ba_p_ens Bp)),
    subalg_of_p (Gen_p E pfe) (Gen_p F pff).
intros T Bp E F h0 h1 h2.
red.
assert (h3:Included (ba_p_ens (Gen_p E h1)) (ba_p_ens (Gen_p F h2))).
  do 2 rewrite ba_p_ens_gen_p_eq.
  apply gen_ens_preserves_inclusion_p; auto.
exists h3. 
assert (h4: alg_closed_p (ba_p_ens (Gen_p E h1)) h3).
  assert (h8:subalg_of_p (Gen_p F h2) Bp).
    apply gen_p_subalg_of_p_compat.
    assert (h5:Included (Gen_Ens_p E h1) (ba_p_ens (Gen_p F h2))).
      pose proof (ba_p_ens_gen_p_eq E h1) as h4.
      rewrite <- h4. assumption.
    assert (h9:Included E (ba_p_ens (Gen_p F h2))).
      rewrite ba_p_ens_gen_p_eq.
      pose proof (gen_ens_includes_p F h2) as h10.
      auto with sets.
    pose proof (gen_ens_p_subalg_of_p  h8 E h9 h1) as h10.
    assert (h11:ba_p_ens (Gen_p E h1) = Gen_Ens_p E h9).
      rewrite ba_p_ens_gen_p_eq. rewrite h10. reflexivity.
    assert (h12:Included (Gen_Ens_p E h9) (ba_p_ens (Gen_p F h2))).
      rewrite <- h11. assumption.
    pose proof (subsetT_eq_compat _ (fun S => Included S (ba_p_ens (Gen_p F h2))) _ _ h12 h3 h10) as h13.
    dependent rewrite <- h13.
    assert (h14:h12 = incl_gen_ens_p E h9). apply proof_irrelevance.
    subst.
    apply closed_gen_ens_p.
exists h4. 
apply bc_inj_p. 
assert (h5: A_p T (Bc_p T (Gen_p E h1)) =
            A_p T
                (Bc_p T (Subalg_p (Gen_p F h2) (ba_p_ens (Gen_p E h1)) h3 h4))).
  unfold Bc_p, Subalg_p. simpl.
  rewrite ba_p_ens_gen_p_eq. reflexivity.
apply (bconst_ext_p  (Bc_p T (Gen_p E h1))
                          (Bc_p T (Subalg_p (Gen_p F h2) (ba_p_ens (Gen_p E h1)) h3 h4)) h5).
 
apply Extensionality_Ensembles.
red. split.
red.
intros x h6.
rewrite <- (transfer_r_undoes_transfer (sig_set_eq (A_p T (Bc_p T (Gen_p E h1))) (A_p T (Bc_p T (Gen_p E h1))) h5) x).
rewrite <- transfer_in_r. constructor.
red.
intros x h6. constructor.

apply functional_extensionality. intro x. apply functional_extensionality. intro y.
rewrite <- transfer_fun2_r_transfer_dep_r_compat'. 
rewrite <- (transfer_r_undoes_transfer 
 (sig_set_eq (A_p T (Bc_p T (Gen_p E h1)))
        (A_p T (Bc_p T (Subalg_p (Gen_p F h2) (ba_p_ens (Gen_p E h1)) h3 h4))) h5) x).

rewrite <- (transfer_r_undoes_transfer 
 (sig_set_eq (A_p T (Bc_p T (Gen_p E h1)))
        (A_p T (Bc_p T (Subalg_p (Gen_p F h2) (ba_p_ens (Gen_p E h1)) h3 h4))) h5) y).
rewrite transfer_fun2_r_compat'. 
simpl.
do 2 rewrite transfer_r_undoes_transfer.
apply proj1_sig_injective.
simpl.
destruct x as [x h5'], y as [y h6]. simpl.
rewrite (transfer_r_sig_set_eq _ _ h5 (sig_set_eq (Gen_Ens_p E h1) (ba_p_ens (Gen_p E h1)) h5)).
simpl.
do  2 rewrite (transfer_sig_set_eq _ _ h5 (sig_set_eq (Gen_Ens_p E h1) (ba_p_ens (Gen_p E h1)) h5)).
simpl.
f_equal. f_equal. apply proj1_sig_injective; auto. apply proj1_sig_injective; auto.

apply functional_extensionality. intro x. apply functional_extensionality. intro y.
rewrite <- transfer_fun2_r_transfer_dep_r_compat'. 
rewrite <- (transfer_r_undoes_transfer 
 (sig_set_eq (A_p T (Bc_p T (Gen_p E h1)))
        (A_p T (Bc_p T (Subalg_p (Gen_p F h2) (ba_p_ens (Gen_p E h1)) h3 h4))) h5) x).

rewrite <- (transfer_r_undoes_transfer 
 (sig_set_eq (A_p T (Bc_p T (Gen_p E h1)))
        (A_p T (Bc_p T (Subalg_p (Gen_p F h2) (ba_p_ens (Gen_p E h1)) h3 h4))) h5) y).
rewrite transfer_fun2_r_compat'. 
simpl.
do 2 rewrite transfer_r_undoes_transfer.
apply proj1_sig_injective.
simpl.
destruct x as [x h5'], y as [y h6]. simpl.
rewrite (transfer_r_sig_set_eq _ _ h5 (sig_set_eq (Gen_Ens_p E h1) (ba_p_ens (Gen_p E h1)) h5)).
simpl.
do  2 rewrite (transfer_sig_set_eq _ _ h5 (sig_set_eq (Gen_Ens_p E h1) (ba_p_ens (Gen_p E h1)) h5)).
simpl.
f_equal. f_equal. apply proj1_sig_injective; auto. apply proj1_sig_injective; auto.

rewrite transfer_dep_r_id_transfer_r_compat.
rewrite (transfer_r_sig_set_eq _ _ h5  (sig_set_eq (A_p T (Bc_p T (Gen_p E h1)))
        (A_p T (Bc_p T (Subalg_p (Gen_p F h2) (ba_p_ens (Gen_p E h1)) h3 h4))) h5)).
apply proj1_sig_injective.
simpl. reflexivity.

rewrite transfer_dep_r_id_transfer_r_compat.
rewrite (transfer_r_sig_set_eq _ _ h5  (sig_set_eq (A_p T (Bc_p T (Gen_p E h1)))
        (A_p T (Bc_p T (Subalg_p (Gen_p F h2) (ba_p_ens (Gen_p E h1)) h3 h4))) h5)).
apply proj1_sig_injective.
simpl. reflexivity.

apply functional_extensionality. intro x.
rewrite <- transfer_fun_r_transfer_dep_r_compat'. 
rewrite <- (transfer_r_undoes_transfer 
 (sig_set_eq (A_p T (Bc_p T (Gen_p E h1)))
        (A_p T (Bc_p T (Subalg_p (Gen_p F h2) (ba_p_ens (Gen_p E h1)) h3 h4))) h5) x).
rewrite transfer_fun_r_compat'. 
simpl.
rewrite transfer_r_undoes_transfer.
apply proj1_sig_injective.
simpl.
destruct x as [x h5'].
rewrite (transfer_r_sig_set_eq _ _ h5 (sig_set_eq (Gen_Ens_p E h1) (ba_p_ens (Gen_p E h1)) h5)).
simpl.
rewrite (transfer_sig_set_eq _ _ h5 (sig_set_eq (Gen_Ens_p E h1) (ba_p_ens (Gen_p E h1)) h5)).
simpl.
f_equal. f_equal. apply proj1_sig_injective; auto.
Qed.


Lemma gen_p_functional : 
  forall {T:Type} (B:Bool_Alg_p T) (E E':Ensemble T)
         (pfeq:E = E') 
         (pf:Included E (ba_p_ens B))
         (pf':Included E' (ba_p_ens B)),
    Gen_p E pf = Gen_p E' pf'.
intros; subst; auto.
assert (pf=pf'). apply proof_irrelevance.
subst.
reflexivity.
Qed.



End Gen_Ens_p.

(*This takes a Bool_Alg and creates the corresponding 
  Bool_Alg_p with the underlying type of the value the same
  as the Boolean type of the argument.*)

Section BA_to_BA_p.

Definition ba_to_ba_p (B:Bool_Alg) : Bool_Alg_p (bt B).
refine (ba_sig_set_conv _ _ _).
pose (Subalg (Full_set (bt B)) (alg_closed_ba_ens B)) as B'.
rewrite (eq_refl (bt B')) at 1. reflexivity.
Defined.


Definition ba_p_conv_elt {B:Bool_Alg} (x:bt B) :
  (btp (ba_to_ba_p B)). 
unfold ba_to_ba_p, btp, ba_sig_set_conv, Btype_p, ba_ens.
simpl.
refine (exist _ _ (Full_intro _ x)).
Defined.


Lemma btp_ba_to_ba_p_eq :
  forall (B:Bool_Alg),
    btp (ba_to_ba_p B) = sig_set (ba_ens B).
intro B.
unfold ba_to_ba_p, btp, ba_sig_set_conv, Btype_p, ba_ens.
simpl. reflexivity.
Qed.



Lemma ba_conv_ba_to_ba_p_eq :
  forall (B:Bool_Alg),
    ba_conv (ba_to_ba_p B) = Subalg (Full_set (bt B)) (alg_closed_ba_ens B).
intro B.
apply bc_inj.
assert (h1:Btype (Bc (ba_conv (ba_to_ba_p B))) = 
           Btype (Bc (Subalg (Full_set (bt B)) (alg_closed_ba_ens B)))).
  simpl. reflexivity. 
apply (bconst_ext _ _ h1).
simpl in h1.
assert (h1 = eq_refl _). apply proof_irrelevance. subst.
rewrite transfer_dep_r_eq_refl. simpl. 
assert (h1:Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
        (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B))) eq_refl
        (sig_set (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
        (sig_set (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
        (Morphisms.eq_proper_proxy Type
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
        eq_refl = eq_refl). apply proof_irrelevance.
rewrite h1 at 1.
rewrite transfer_dep_eq_refl.
reflexivity.
simpl in h1. assert (h1 = eq_refl _). apply proof_irrelevance.
subst.
rewrite transfer_dep_r_eq_refl. simpl.
assert (h1: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
        (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B))) eq_refl
        (sig_set (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
        (sig_set (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
        (Morphisms.eq_proper_proxy Type
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
        eq_refl = eq_refl).
  apply proof_irrelevance.
rewrite h1 at 1.
rewrite transfer_dep_eq_refl.
reflexivity.
simpl in h1. assert (h1 = eq_refl _). apply proof_irrelevance.
subst.
rewrite transfer_dep_r_eq_refl. simpl.
assert (h1:Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
        (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B))) eq_refl
        (sig_set (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
        (sig_set (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
        (Morphisms.eq_proper_proxy Type
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
        eq_refl = eq_refl).  apply proof_irrelevance.
rewrite h1 at 1.
rewrite transfer_dep_eq_refl. reflexivity.
simpl in h1. assert (h1 = eq_refl _). apply proof_irrelevance. subst.
rewrite transfer_dep_r_eq_refl.  simpl.
assert (h1: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
        (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B))) eq_refl
        (sig_set (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
        (sig_set (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
        (Morphisms.eq_proper_proxy Type
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
        eq_refl = eq_refl). apply proof_irrelevance.
rewrite h1 at 1.
rewrite transfer_eq_refl.
reflexivity.
simpl in h1. assert (h1 = eq_refl _). apply proof_irrelevance. subst.
rewrite transfer_dep_r_eq_refl. simpl.
assert (h1:Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
        (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B))) eq_refl
        (sig_set (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
        (sig_set (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
        (Morphisms.eq_proper_proxy Type
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
        eq_refl = eq_refl).  apply proof_irrelevance.
rewrite h1 at 1. rewrite transfer_eq_refl.
reflexivity.
simpl in h1. assert (h1 = eq_refl _). apply proof_irrelevance. subst.
rewrite transfer_dep_r_eq_refl. simpl.
assert (h1: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
        (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B))) eq_refl
        (sig_set (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
        (sig_set (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
        (Morphisms.eq_proper_proxy Type
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
        eq_refl = eq_refl). apply proof_irrelevance.
rewrite h1 at 1.
rewrite transfer_dep_eq_refl.
reflexivity.
Qed.




Lemma ba_p_ens_ba_to_ba_p_eq :
  forall {B:Bool_Alg},
    ba_p_ens (ba_to_ba_p B) = ba_ens B.
intro B.
rewrite ba_p_ens_eq.
apply Extensionality_Ensembles.
red. split.
red. intros x h1.
destruct h1 as [x h1]. subst. clear h1.
destruct x as [x h1]. simpl. constructor.
red. intros x h1. 
apply Im_intro with (ba_p_conv_elt x).
constructor. simpl.
reflexivity.
Qed.



Lemma proj1_sig_btimes_p_eq : 
  forall {B:Bool_Alg} {Bp:Bool_Alg_p (bt B)}
         (x y:sig_set (ba_p_ens Bp)),
    subalg_of_p Bp (ba_to_ba_p B) ->
    proj1_sig (x %* y) =
    (proj1_sig x) * (proj1_sig y).
intros B Bp x y h1.
red in h1. destruct h1 as [h1a [h1b h1c]].
destruct x as [x h2], y as [y h3].
assert (h5:Ensembles.In
                   (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Bp) h1a h1b))
                   x).
  rewrite <- h1c. assumption.
assert (h6:Ensembles.In
                   (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Bp) h1a h1b))
                   y).
  rewrite <- h1c. assumption.
pose proof (ba_p_subst_times _ _ h1c _ _ h2 h3 h5 h6) as h4.
rewrite h4 at 1.
simpl.
assert (h7: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl).
  apply proof_irrelevance.
rewrite h7 at 1. rewrite transfer_dep_eq_refl. simpl.
unfold Btimes_sub. simpl.
reflexivity.
Qed.


Lemma proj1_sig_bplus_p_eq : 
  forall {B:Bool_Alg} {Bp:Bool_Alg_p (bt B)}
         (x y:sig_set (ba_p_ens Bp)),
    subalg_of_p Bp (ba_to_ba_p B) ->
    proj1_sig (x %+ y) =
    (proj1_sig x) + (proj1_sig y).
intros B Bp x y h1.
red in h1. destruct h1 as [h1a [h1b h1c]].
destruct x as [x h2], y as [y h3].
assert (h5:Ensembles.In
                   (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Bp) h1a h1b))
                   x).
  rewrite <- h1c. assumption.
assert (h6:Ensembles.In
                   (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Bp) h1a h1b))
                   y).
  rewrite <- h1c. assumption.
pose proof (ba_p_subst_plus _ _ h1c _ _ h2 h3 h5 h6) as h4.
rewrite h4 at 1.
simpl.
assert (h7: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl).
  apply proof_irrelevance.
rewrite h7 at 1. rewrite transfer_dep_eq_refl. simpl.
unfold Bplus_sub. simpl.
reflexivity.
Qed.


Lemma proj1_sig_bone_p_eq : 
  forall {B:Bool_Alg} {Bp:Bool_Alg_p (bt B)},
    subalg_of_p Bp (ba_to_ba_p B) ->
    proj1_sig (Bone_p _ (Bc_p _ Bp)) = (Bone (Bc B)).
intros B Bp h1.
red in h1.
destruct h1 as [h1 [h2 h3]].
pose proof (ba_p_subst_one _ _ h3) as h4.
rewrite h4.
simpl.
assert (h5: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl).
  apply proof_irrelevance.
rewrite h5 at 1.
rewrite transfer_eq_refl. simpl.
reflexivity.
Qed.


Lemma proj1_sig_bzero_p_eq : 
  forall {B:Bool_Alg} {Bp:Bool_Alg_p (bt B)},
    subalg_of_p Bp (ba_to_ba_p B) ->
    proj1_sig (Bzero_p _ (Bc_p _ Bp)) = (Bzero (Bc B)).
intros B Bp h1.
red in h1.
destruct h1 as [h1 [h2 h3]].
pose proof (ba_p_subst_zero _ _ h3) as h4.
rewrite h4.
simpl.
assert (h5: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl).
  apply proof_irrelevance.
rewrite h5 at 1.
rewrite transfer_eq_refl. simpl.
reflexivity.
Qed.



Lemma proj1_sig_bcomp_p_eq : 
  forall {B:Bool_Alg} {Bp:Bool_Alg_p (bt B)}
         (x:sig_set (ba_p_ens Bp)),
    subalg_of_p Bp (ba_to_ba_p B) ->
    proj1_sig (%- x) =
    - (proj1_sig x).
intros B Bp x h1.
red in h1. destruct h1 as [h1a [h1b h1c]].
destruct x as [x h2].
assert (h5:Ensembles.In
                   (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Bp) h1a h1b))
                   x).
  rewrite <- h1c. assumption.
pose proof (ba_p_subst_comp _ _ h1c _ h2 h5) as h4.
rewrite h4 at 1.
simpl.
assert (h7: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl).
  apply proof_irrelevance.
rewrite h7 at 1. rewrite transfer_dep_eq_refl. simpl.
unfold Bcomp_sub. simpl.
reflexivity.
Qed.



Lemma proj1_sig_times_list_transfer_dep_ba_conv_type_eq : 
  forall {B:Bool_Alg} {Bp:Bool_Alg_p (bt B)},
    subalg_of_p Bp (ba_to_ba_p B) ->
    forall (l:list (bt B)) (l':list (btp Bp)),
      NoDup l -> NoDup l' ->
      list_to_set l = im_proj1_sig (list_to_set l') ->
      proj1_sig  (times_list (B0 := ba_conv Bp) (transfer_dep (ba_conv_type Bp) l')) = 
  
    times_list l.
intros B Bp h1 l. 
pose proof h1 as h1'.
red in h1.
destruct h1 as [h1 [h2 h3]].
destruct h2 as [h2a h2b h2c h2d h2e]. 
red in h2a, h2b, h2c, h2d, h2e.
induction l as [|a l ih]; simpl; auto.
intros l' h4 h5 h6.
symmetry in h6.
apply empty_image in h6.
apply empty_set_nil in h6. rewrite h6.
simpl.
pose proof (ba_p_subst_one _ _ h3) as h0.
rewrite h0.
simpl.
assert (h7:(Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl) = eq_refl). apply proof_irrelevance.
rewrite h7 at 1. rewrite transfer_eq_refl. simpl. reflexivity.
intros l' h4 h5 h6.
pose proof (no_dup_cons _ _ h4) as h7.
apply no_dup_cons_nin in h4.
assert (h8:list_to_set l = Subtract (im_proj1_sig (list_to_set l')) a).
rewrite list_to_set_in_iff in h4.
pose proof (f_equal (fun S=> Subtract S a) h6) as h8. simpl in h8.
rewrite sub_add_compat_nin in h8.
assumption. assumption.  
assert (h10:Ensembles.In (ba_p_ens Bp) a).
  assert (h9:Ensembles.In (Add (list_to_set l) a) a).
  right; constructor; auto.
  rewrite h6 in h9. destruct h9 as [a h9]. clear h3. subst.
  apply proj2_sig.
assert (h9:list_to_set l = im_proj1_sig (list_to_set (remove eq_dec (exist _ _ h10) l'))).
  rewrite h8.
  rewrite <- subtract_remove_compat.
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h11.
  destruct h11 as [h11 h12].
  destruct h11 as [x h11]. clear h3. subst.
  apply Im_intro with x. constructor. assumption.
  intro h13. destruct h13. simpl in h12. contradict h12.
  constructor. reflexivity.
  red. intros x h11.
  destruct h11 as [x h11]. clear h3. subst.
  destruct h11 as [h11 h12]. constructor.
  apply Im_intro with x. assumption.
  reflexivity. intro h13. inversion h13. clear h13. subst.
  contradict h12.
  rewrite unfold_sig.
  assert (h13:proj2_sig x = h10). apply proof_irrelevance.
  subst. constructor.
assert (h11:NoDup (remove eq_dec (exist (Ensembles.In (ba_p_ens Bp)) a h10) l')).
  apply no_dup_remove. assumption.
specialize (ih _ h7 h11 h9).
rewrite <- ih.
unfold ba_conv_type. do 2 rewrite transfer_dep_eq_refl.
assert (h12:Ensembles.In (list_to_set l') (exist _ _ h10)).
  assert (h12:Ensembles.In (Add (list_to_set l) a) a). right; constructor; auto.
  rewrite h6 in h12.
  destruct h12 as [a h12]. clear h3. subst.
  assert (h13:h10 = proj2_sig a). apply proof_irrelevance. 
  subst. rewrite <- unfold_sig.
  assumption. 
assert (h13:times_list l' (B0:=ba_conv Bp) = (exist _ _ h10) %*
                             (times_list (B0:=ba_conv Bp)
        (remove eq_dec (exist (Ensembles.In (ba_p_ens Bp)) a h10) l'))).
  rewrite ba_conv_times. unfold ba_conv_elt, ba_conv_type.
  do 2 rewrite transfer_eq_refl. 
  rewrite <- list_to_set_in_iff in h12.
  rewrite <- times_times_list_remove at 1; auto. 
rewrite h13 at 1.  
rewrite proj1_sig_btimes_p_eq.
simpl.
reflexivity.
assumption. 
Qed.



Lemma alg_closed_ba_p_ens :
  forall (B:Bool_Alg) (Bp:Bool_Alg_p (bt B)),
    subalg_of_p Bp (ba_to_ba_p B) ->
  alg_closed (ba_p_ens Bp).
intros B Bp h1.
constructor; red.

intros x y. 
unfold Bplus_sub.
rewrite <- (proj1_sig_bplus_p_eq x y h1) at 1.
apply proj2_sig.

intros x y. 
unfold Btimes_sub.
rewrite <- (proj1_sig_btimes_p_eq x y h1) at 1. 
apply proj2_sig.

rewrite <- (proj1_sig_bone_p_eq h1) at 1. 
apply proj2_sig.

rewrite <- (proj1_sig_bzero_p_eq h1) at 1. 
apply proj2_sig.

intro x. 
unfold Bcomp_sub.
rewrite <- (proj1_sig_bcomp_p_eq x h1) at 1. 
apply proj2_sig.
Qed.


Definition fam_ba_p_bt_p_compat 
           {B:Bool_Alg}   (F:fam_ba_p (bt B))
           (Bp:Bool_Alg_p (bt B))
: Prop :=
  forall (Ap:Bool_Alg_p (bt B)),
    Ensembles.In F Ap -> subalg_of_p Ap (ba_to_ba_p B).



End BA_to_BA_p.



Section Closed_Finite_p.

Lemma plus_list_closed_p :
  forall {T:Type} {Bp:Bool_Alg_p T} 
         (S:Ensemble (btp Bp)) (l:list (btp Bp)),
    alg_closed_p _ (incl_ens_btp S) -> Included (list_to_set l) S ->
    Ensembles.In S (plus_list_p l).
intros T Bp S l h1 h2.
rewrite alg_closed_p_iff in h1.
rewrite ba_conv_und_subalg_im_proj1_sig in h1.
assert (h3:Included (list_to_set (ba_conv_list l)) (ba_conv_set S)).
  rewrite list_to_set_ba_conv_list.
  rewrite <- incl_ba_conv_set_iff.
  assumption.
pose proof (plus_list_closed _ (ba_conv_set S) _ h1 h3) as h4. 
unfold ba_conv_set, ba_conv_list in h4. unfold ba_conv_type in h4.
do 2 rewrite transfer_dep_eq_refl in h4.
rewrite plus_list_p_eq. unfold ba_conv_list, ba_conv_type.
rewrite transfer_dep_eq_refl.
assumption.
Qed.



Lemma times_list_closed_p :
  forall {T:Type} {Bp:Bool_Alg_p T} 
         (S:Ensemble (btp Bp)) (l:list (btp Bp)),
    alg_closed_p _ (incl_ens_btp S) -> Included (list_to_set l) S ->
    Ensembles.In S (times_list_p l).
intros T Bp S l h1 h2.
rewrite alg_closed_p_iff in h1.
rewrite ba_conv_und_subalg_im_proj1_sig in h1.
assert (h3:Included (list_to_set (ba_conv_list l)) (ba_conv_set S)).
  rewrite list_to_set_ba_conv_list.
  rewrite <- incl_ba_conv_set_iff.
  assumption.
pose proof (times_list_closed _ (ba_conv_set S) _ h1 h3) as h4. 
unfold ba_conv_set, ba_conv_list in h4. unfold ba_conv_type in h4.
do 2 rewrite transfer_dep_eq_refl in h4.
rewrite times_list_p_eq. unfold ba_conv_list, ba_conv_type.
rewrite transfer_dep_eq_refl.
assumption.
Qed.


Lemma plus_set_closed_p :
  forall {T:Type} {Bp:Bool_Alg_p T} 
         (R S:Ensemble (btp Bp)) (pf:Finite R),
    alg_closed_p _ (incl_ens_btp S) -> Included R S ->
    Ensembles.In S (plus_set_p R pf).
intros T Bp R S h1 h2 h3.
rewrite alg_closed_p_iff in h2.
rewrite ba_conv_und_subalg_im_proj1_sig in h2.
rewrite incl_ba_conv_set_iff in h3.
pose proof (plus_set_closed _ (ba_conv_set R) (ba_conv_set S) h1 h2 h3) as h4.
rewrite plus_set_p_eq.
assumption.
Qed.


Lemma times_set_closed_p :
  forall {T:Type} {Bp:Bool_Alg_p T} 
         (R S:Ensemble (btp Bp)) (pf:Finite R),
    alg_closed_p _ (incl_ens_btp S) -> Included R S ->
    Ensembles.In S (times_set_p R pf).
intros T Bp R S h1 h2 h3.
rewrite alg_closed_p_iff in h2.
rewrite ba_conv_und_subalg_im_proj1_sig in h2.
rewrite incl_ba_conv_set_iff in h3.
pose proof (times_set_closed _ (ba_conv_set R) (ba_conv_set S) h1 h2 h3) as h4.
rewrite times_set_p_eq.
assumption.
Qed.


End Closed_Finite_p.

Section NormalForm_p.
Variable T:Type.
Variable Ap:Bool_Alg_p T.
Let Atp := btp Ap.



Definition eps_p {T':Type} {Bp:Bool_Alg_p T'} 
           (b:(btp Bp)) (sgn:sign) := if sgn then b else %-b.
Definition eps_p' {T':Type}  {Bp:Bool_Alg_p T'} 
           (pr:(btp Bp)*sign) := eps_p (fst pr) (snd pr).



Lemma eps_eps'_compat_p : 
  forall {T':Type}  {Bp:Bool_Alg_p T'} (x:(btp Bp)), 
    eps_p x = (fun y:sign => eps_p' (x, y)).
intro x.
unfold eps_p, eps_p'. simpl.  unfold eps_p.
reflexivity.
Qed.

Lemma eps_p_eq : 
  forall {T':Type}  {Bp:Bool_Alg_p T'} 
         (b:(btp Bp)) (sgn:sign),
    eps_p b sgn = eps (ba_conv_elt b) sgn.
intros; auto.
Qed.

Lemma eps_p_eq' : 
  forall {T':Type}  {Bp:Bool_Alg_p T'} 
         (pr:(btp Bp)*sign),
    eps_p' pr = eps' (ba_conv_elt (fst pr), snd pr).
intros; auto.
Qed.


Lemma eps_covers_p :
  forall {T':Type}  {Bp:Bool_Alg_p T'} (E:Ensemble (btp Bp)) 
         (pf:Finite E) (x:btp Bp),
    plus_set_p (Im signe (eps_p x)) (finite_image _ _ signe (eps_p x) (signe_finite)) = %1.
intros T' Bp E h1 x.
pose proof (eps_covers (ba_conv_set E) h1 (ba_conv_elt x)) as h2.
rewrite plus_set_p_eq.
assumption.
Qed.


Definition eps_map_p {E:Ensemble Atp} (f:Fin_map E signe mns) :
  Atp->Atp :=
  (fun i:Atp => eps_p  i (f |-> i)).

Lemma eps_map_p_eq :
  forall {E:Ensemble Atp} (f:Fin_map E signe mns),
    eps_map_p f = eps_map (ba_conv_fin_map_dom f).
intros; auto.
Qed.



Lemma eps_maps_dec_p :
  forall {E:Ensemble Atp} (f g:Fin_map E signe mns)
         (x:Atp), Ensembles.In E x ->
    {eps_map_p f x = eps_map_p g x} + {eps_map_p f x = %-(eps_map_p g x)}.

intros E f g x h1.
pose proof (eps_maps_dec _ (ba_conv_fin_map_dom f) (ba_conv_fin_map_dom g) (ba_conv_elt x) h1) as h2.
assumption.
Qed.


Definition eps_map_compose_p {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
           (f:Fin_map E signe mns) (g:(btp Bp)->Atp) : (btp Bp) -> Atp :=
  (fun i:(btp Bp) => eps_p (g i) (f |-> i)).

Lemma eps_map_compose_p_eq : 
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (f:Fin_map E signe mns) (g:(btp Bp)->Atp),
    eps_map_compose_p f g = eps_map_compose (ba_conv_fin_map_dom f) (ba_conv_fun1 g)  (A:=ba_conv Ap).
intros; auto.
Qed.


Lemma eps_maps_compose_dec_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (f f':Fin_map E signe mns) (g:(btp Bp)->Atp)
         (x:(btp Bp)), Ensembles.In E x ->
         {eps_map_compose_p f g x = eps_map_compose_p f' g x} +
         {eps_map_compose_p f g x = %-(eps_map_compose_p f' g x)}.
intros Bp E f f' g x h1.
pose proof (eps_maps_compose_dec (ba_conv Ap) (ba_conv_fin_map_dom f) (ba_conv_fin_map_dom f') (ba_conv_fun1 g) (ba_conv_elt x) h1) as h2.
assumption.
Qed.


Definition im_eps_p {E:Ensemble Atp}
           (f:Fin_map E signe mns) : Ensemble Atp :=
Im E (eps_map_p f).


Lemma im_eps_p_eq : 
  forall {E:Ensemble Atp}
         (f:Fin_map E signe mns),
    im_eps_p f = im_eps _ (ba_conv_fin_map_dom f).
intros; auto.
Qed.


Definition im_eps_compose_p
           {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
           (f:Fin_map E signe mns) (g:(btp Bp)->Atp) : 
  Ensemble Atp :=
  Im E (eps_map_compose_p f g).


Lemma im_eps_compose_p_eq : 
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (f:Fin_map E signe mns) (g:(btp Bp)->Atp),
    im_eps_compose_p f g = im_eps_compose (ba_conv Ap) (ba_conv_fin_map_dom f) (ba_conv_fun1 g).
intros; auto.
Qed.



Lemma finite_im_eps_p :
  forall {E:Ensemble Atp}
         (f:Fin_map E signe mns),
    Finite (im_eps_p f).
intros E f. apply finite_image.
apply (fin_map_fin_dom f).
Qed.



Definition finite_im_eps_compose_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (f:Fin_map E signe mns) (g:(btp Bp)->Atp),
    Finite (im_eps_compose_p f g).
intros B E f g. apply finite_image. apply (fin_map_fin_dom f).
Qed.


Lemma im_eps_im_eps_compose_compat_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (g:sig_set E->Atp)
         (f : Fin_map (Im (full_sig E) g) signe mns)
         (pf:Finite E),
   im_eps_p f =
   im_eps_compose_p (fin_map_im_full_sig_eq f pf %0) (sig_fun_app g %0).
intros Bp E g f h1.
pose proof (im_eps_im_eps_compose_compat (ba_conv Ap) (ba_conv_sig_fun1 g) (ba_conv_fin_map_dom f) h1) as h2.
assumption.
Qed.



Definition el_prod_p {E:Ensemble Atp}
           (f:Fin_map E signe mns) : Atp.
pose (fun i:Atp => eps_p  i (f |-> i)) as p.
pose proof (fin_map_fin_dom f) as h1.
pose proof (finite_image _ _ E p h1) as h2.
refine (times_set_p _ h2).
Defined.

Lemma el_prod_p_eq : 
  forall {E:Ensemble Atp} (f:Fin_map E signe mns),
    el_prod_p f = el_prod _ (ba_conv_fin_map_dom f).
intros E f.
unfold el_prod_p, el_prod, ba_conv_fin_map_dom.
rewrite times_set_p_eq.
unfold ba_conv_elt, ba_conv_set. unfold ba_conv_type.
rewrite transfer_eq_refl. 
f_equal. 
Qed.




Definition el_sum_p {E:Ensemble Atp}
           (f:Fin_map E signe mns) : Atp.
pose (fun i:Atp => eps_p  i (f |-> i)) as p.
pose proof (fin_map_fin_dom f) as h1.
pose proof (finite_image _ _ E p h1) as h2.
refine (plus_set_p _ h2).
Defined.

Lemma el_sum_p_eq : 
  forall {E:Ensemble Atp} (f:Fin_map E signe mns),
    el_sum_p f = el_sum _ (ba_conv_fin_map_dom f).
intros E f.
unfold el_sum_p, el_sum, ba_conv_fin_map_dom.
rewrite plus_set_p_eq.
unfold ba_conv_elt, ba_conv_set. unfold ba_conv_type.
rewrite transfer_eq_refl. 
f_equal. 
Qed.


Definition el_prod_compose_p {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
           (g:(btp Bp)->Atp) (f:Fin_map E signe mns)  : Atp.
pose (fun i:(btp Bp) => eps_p (g i) (f |-> i)) as p.
pose proof (fin_map_fin_dom f) as h1.
pose proof (finite_image _ _ E p h1) as h2.
refine (times_set_p _ h2).
Defined.


Lemma el_prod_compose_p_eq : 
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (g:(btp Bp)->Atp) (f:Fin_map E signe mns),
    el_prod_compose_p g f =
    el_prod_compose (ba_conv Ap) (ba_conv_fun1 g) f.
intros Bp E g f.
unfold el_prod_compose_p, el_prod_compose, ba_conv_fun1.
rewrite times_set_p_eq.
unfold ba_conv_elt, ba_conv_set. unfold ba_conv_type.
rewrite transfer_eq_refl.
f_equal.
Qed.



Definition el_prod_compose_p' 
           {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)} 
           (g:(btp Bp)->Atp) (f:Fin_map E signe mns) : Atp.
pose (fun i:sig_set E => g (proj1_sig i)) as g'.
pose (fun i:sig_set E => eps_p (g' i) (f |-> (proj1_sig i))) as p.
pose proof (fin_map_fin_dom f) as h1.
rewrite finite_full_sig_iff in h1.
pose proof (finite_image _ _ (full_sig E) p h1) as h2.
refine (times_set_p _ h2).
Defined.

Lemma el_prod_compose_p_eq' : 
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)} 
         (g:(btp Bp)->Atp) (f:Fin_map E signe mns),
    el_prod_compose_p' g f =
    el_prod_compose' (ba_conv Ap) (ba_conv_fun1 g) (ba_conv_fin_map_dom f).
intros Bp E g f.
unfold el_prod_compose_p', el_prod_compose', ba_conv_fun1, ba_conv_fin_map_dom.
rewrite times_set_p_eq. unfold ba_conv_elt. unfold ba_conv_type.
rewrite transfer_eq_refl.
f_equal.
Qed.


Definition el_prod_compose_p1 {A:Bool_Alg} {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
           (g:(btp Bp)->bt A) (f:Fin_map E signe mns)  : bt A.
pose (fun i:(btp Bp) => eps (g i) (f |-> i)) as p.
pose proof (fin_map_fin_dom f) as h1.
pose proof (finite_image _ _ E p h1) as h2.
refine (times_set _ h2).
Defined.



Definition el_prod_compose_p1_eq :
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (g:(btp Bp)->bt A) (f:Fin_map E signe mns),
    el_prod_compose_p1 g f = el_prod_compose _ (ba_conv_fun1 g) (ba_conv_fin_map_dom f).
intros A Bp E g f.
unfold  el_prod_compose_p1, el_prod_compose.
apply times_set_functional.
f_equal.
Qed.



Lemma el_prod_compose_p1_eq' : 
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (g:(btp Bp)->Atp) (f:Fin_map E signe mns),
    el_prod_compose_p g f = el_prod_compose_p1 (ba_conv_fun2 g) f.
intros Bp E g f.
unfold el_prod_compose_p, el_prod_compose_p1.
rewrite times_set_p_eq. unfold ba_conv_elt, ba_conv_type.
rewrite transfer_eq_refl.
unfold ba_conv_set, ba_conv_type.
generalize  (finite_image (btp Bp) (btp Ap) E
        (fun i : btp Bp => eps_p (g i) (f |-> i)) (fin_map_fin_dom f)).  intro h0.
assert (h1:times_set
     (transfer_dep eq_refl (Im E (fun i : btp Bp => eps_p (g i) (f |-> i))))
     h0 (B:=ba_conv Ap) = 
           times_set
     (Im E (fun i : btp Bp => eps_p (g i) (f |-> i))) h0 (B:=ba_conv Ap)).
  apply times_set_functional. rewrite transfer_dep_eq_refl.
  reflexivity.
rewrite h1 at 1.
apply (times_set_functional (B:=ba_conv Ap)).
reflexivity.
Qed.





Lemma el_prod_compose_el_prod_compose'_compat_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (g:(btp Bp)->Atp),
    el_prod_compose_p g = el_prod_compose_p' (E:=E) g.
intros B E g.
pose proof (el_prod_compose_el_prod_compose'_compat (E:=(ba_conv_set E)) (ba_conv Ap) (ba_conv_fun1 g)) as h1. 
unfold ba_conv_set, ba_conv_type in h1.
rewrite transfer_dep_eq_refl in h1.
apply functional_extensionality.
intro x.
rewrite el_prod_compose_p_eq, el_prod_compose_p_eq'.
rewrite h1.
reflexivity.
Qed.


Definition el_sum_compose_p {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
           (g:(btp Bp)->Atp) (f:Fin_map E signe mns)  : Atp.
pose (fun i:(btp Bp) => eps_p (g i) (f |-> i)) as p.
pose proof (fin_map_fin_dom f) as h1.
pose proof (finite_image _ _ E p h1) as h2.
refine (plus_set_p _ h2).
Defined.


Lemma el_sum_compose_p_eq : 
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (g:(btp Bp)->Atp) (f:Fin_map E signe mns),
    el_sum_compose_p g f =
    el_sum_compose (ba_conv Ap) (ba_conv_fun1 g) f.
intros Bp E g f.
unfold el_sum_compose_p, el_sum_compose, ba_conv_fun1.
rewrite plus_set_p_eq.
unfold ba_conv_elt, ba_conv_set. unfold ba_conv_type.
rewrite transfer_eq_refl.
f_equal.
Qed.


Lemma el_prod_el_prod_compose_compat_p :
  forall {Bp:Bool_Alg_p T} 
         {E:Ensemble (btp Bp)} {g:sig_set E->Atp}
         (f:Fin_map (Im (full_sig E) g) signe mns)
         (pf:Finite E),
    el_prod_p f = el_prod_compose_p (sig_fun_app g %0)
                                (fin_map_im_full_sig_eq f pf %0).
intros Bp E g f h1.
assert (h2:exists! f':Fin_map (Im (full_sig (ba_conv_set E)) (ba_conv_sig_fun1 g)) signe mns,
             forall y, Ensembles.In (Im (full_sig E) g) y -> 
                       f |-> y = f' |-> y). 
  assert (h2:Im (full_sig (ba_conv_set E)) (ba_conv_sig_fun1 g) =
             Im (full_sig E) g).
  unfold ba_conv_set, ba_conv_type.
  apply im_ext_in.  intros x h2. reflexivity. 
  rewrite h2.
  exists f. red; split; auto.
  intros f' h3. apply fin_map_ext_in.
  assumption.
pose (proj1_sig (constructive_definite_description _ h2)) as f'.
pose proof (el_prod_el_prod_compose_compat (ba_conv Ap) f' h1) as h3.
rewrite el_prod_p_eq, el_prod_compose_p_eq.
unfold f' in h3.
destruct constructive_definite_description as [f'' h4].
simpl in h3. simpl in f'.
assert (h5:f'' = ba_conv_fin_map_dom f).
  apply fin_map_ext_in.
  intros y h5. rewrite h4. reflexivity.
  destruct h5 as [y h5]. subst. apply Im_intro with y. constructor.
  reflexivity.
subst. 
 unfold btp, bt in h3.  unfold bt, Btype_p, Atp, btp. simpl in h3. simpl.
rewrite h3 at 1.
f_equal.
Qed.


Lemma el_sum_el_sum_compose_compat_p :
  forall {Bp:Bool_Alg_p T} 
         {E:Ensemble (btp Bp)} {g:sig_set E->Atp}
         (f:Fin_map (Im (full_sig E) g) signe mns)
         (pf:Finite E),
    el_sum_p f = el_sum_compose_p (sig_fun_app g %0)
                                (fin_map_im_full_sig_eq f pf %0).
intros Bp E g f h1.
assert (h2:exists! f':Fin_map (Im (full_sig (ba_conv_set E)) (ba_conv_sig_fun1 g)) signe mns,
             forall y, Ensembles.In (Im (full_sig E) g) y -> 
                       f |-> y = f' |-> y). 
  assert (h2:Im (full_sig (ba_conv_set E)) (ba_conv_sig_fun1 g) =
             Im (full_sig E) g).
  unfold ba_conv_set, ba_conv_type.
  apply im_ext_in.  intros x h2. reflexivity. 
  rewrite h2.
  exists f. red; split; auto.
  intros f' h3. apply fin_map_ext_in.
  assumption.
pose (proj1_sig (constructive_definite_description _ h2)) as f'.
pose proof (el_sum_el_sum_compose_compat (ba_conv Ap) f' h1) as h3.
rewrite el_sum_p_eq, el_sum_compose_p_eq.
unfold f' in h3.
destruct constructive_definite_description as [f'' h4].
simpl in h3. simpl in f'.
assert (h5:f'' = ba_conv_fin_map_dom f).
  apply fin_map_ext_in.
  intros y h5. rewrite h4. reflexivity.
  destruct h5 as [y h5]. subst. apply Im_intro with y. constructor.
  reflexivity.
subst. 
 unfold btp, bt in h3.  unfold bt, Btype_p, Atp, btp. simpl in h3. simpl.
rewrite h3 at 1.
f_equal.
Qed.


Lemma el_prod_el_prod_compose_compat_p' :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (f : Fin_map E signe mns) (g:sig_set E->Atp)
         (f': Fin_map (Im (full_sig E) g) signe mns),
     (forall b:(btp Bp),
      Ensembles.In E b -> f' |-> ((g, (Bzero_p T (Bc_p T Ap))) ||-> b) =
                          f |-> b) ->
  el_prod_compose_p (sig_fun_app g %0) f = el_prod_p f'.
intros Bp E f g f' h1.
assert (h2:exists! f'':Fin_map (Im (full_sig (ba_conv_set E)) (ba_conv_sig_fun1 g)) signe mns,
             forall y, Ensembles.In (Im (full_sig E) g) y -> 
                       f' |-> y = f'' |-> y). 
  assert (h2:Im (full_sig (ba_conv_set E)) (ba_conv_sig_fun1 g) =
             Im (full_sig E) g).
  unfold ba_conv_set, ba_conv_type.
  apply im_ext_in.  intros x h2. reflexivity. 
  rewrite h2.
  exists f'. red; split; auto.
  intros f'' h3. apply fin_map_ext_in.
  assumption.
pose (proj1_sig (constructive_definite_description _ h2)) as f''.
assert (hf:f'' = f').
   unfold f''. destruct constructive_definite_description as [f''' h6].
  simpl. simpl in f''. apply fin_map_ext_in.
  intros y h7. rewrite h6. reflexivity.
  destruct h7 as [y h7]. subst. apply Im_intro with y. constructor.
  reflexivity.
 
assert (h4: (forall b : bt (ba_conv Bp),
        Ensembles.In (ba_conv_set E) b ->
        f'' |-> ((ba_conv_sig_fun1 g, (Bzero_p T (Bc_p T Ap))) ||-> b) =
        ba_conv_fin_map_dom f |-> b) ).
  intros b h5.
  specialize (h1 b h5). rewrite <- h1 at 1. f_equal. 
  apply hf.
pose proof (el_prod_el_prod_compose_compat' (ba_conv Ap) (ba_conv_fin_map_dom f) (ba_conv_sig_fun1 g) f'' h4) as h3.
rewrite el_prod_compose_p_eq, el_prod_p_eq.
 unfold btp, bt in h3.  unfold bt, Btype_p, Atp, btp. simpl in h3. simpl.
rewrite h3 at 1.
f_equal. apply hf.
Qed.


Lemma el_prod_le_ai_p : forall {E:Ensemble Atp}
                               (a:Fin_map E signe mns) (i:Atp),
                        Ensembles.In E i ->
                        le_p (el_prod_p a) (eps_p i (a |-> i)).
intros E a i h1.
unfold el_prod_p.
apply le_times_set_p.
apply Im_intro with i; auto.
Qed.

Lemma ai_le_el_sum_p : forall {E:Ensemble Atp}
                              (a:Fin_map E signe mns) (i:Atp),
                         Ensembles.In E i ->
                         le_p (eps_p i (a |-> i))  (el_sum_p a).
intros E a i h1.
unfold el_sum_p.
apply le_plus_set_p.
apply Im_intro with i; auto.
Qed.


Lemma non_zero_el_prod_inj_p :
  forall {E:Ensemble Atp} (f g:Fin_map E signe mns),
    el_prod_p f <> %0 ->
    el_prod_p f = el_prod_p g ->
    im_eps_p f = im_eps_p g.
intros E f g h1 h2. 
rewrite im_eps_p_eq. rewrite (im_eps_p_eq g).
rewrite el_prod_p_eq in h1.
rewrite el_prod_p_eq in h2. rewrite (el_prod_p_eq g) in h2.
rewrite ba_conv_zero in h1.
pose proof (non_zero_el_prod_inj _ (ba_conv_fin_map_dom f) (ba_conv_fin_map_dom g) h1 h2) as h3.
rewrite h3 at 1.
reflexivity.
Qed.

Lemma el_prod_disjoint_p :
  forall (E:Ensemble Atp)
         (a b:Fin_map E signe mns), a <> b ->
             (el_prod_p a) %* (el_prod_p b) = %0.
intros E a b h1.
pose proof (el_prod_disjoint _ _ (ba_conv_fin_map_dom a) (ba_conv_fin_map_dom b) h1) as h2.
rewrite el_prod_p_eq. rewrite (el_prod_p_eq b).
rewrite h2 at 1.
reflexivity.
Qed.


Lemma el_prod_compose_le_ai_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (a:Fin_map E signe mns)
         (g:(btp Bp)->Atp) (i:(btp Bp)),
    Ensembles.In E i ->
    le_p (el_prod_compose_p g a) (eps_p (g i) (a |-> i)).
intros B E a g i h1.
unfold el_prod_compose_p.
apply le_times_set_p.
apply Im_intro with i; auto.
Qed.


Lemma ai_le_el_sum_compose_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (a:Fin_map E signe mns)
         (g:(btp Bp)->Atp) (i:(btp Bp)),
    Ensembles.In E i ->
    le_p (eps_p (g i) (a |-> i)) (el_sum_compose_p g a).
intros B E a g i h1.
unfold el_sum_compose_p.
apply le_plus_set_p.
apply Im_intro with i; auto.
Qed.


Lemma el_prod_compose_disjoint_p :
  forall {Bp:Bool_Alg_p T} (E:Ensemble (btp Bp))
         (g:(btp Bp)->Atp)
         (a b:Fin_map E signe mns), a <> b ->
             (el_prod_compose_p g a) %* (el_prod_compose_p g b) = %0.
intros Bp E g a b h1.
pose proof (el_prod_compose_disjoint (ba_conv Ap) _ (ba_conv_fun1 g) (ba_conv_fin_map_dom a) (ba_conv_fin_map_dom b) h1) as h2.
rewrite el_prod_compose_p_eq. rewrite (el_prod_compose_p_eq g b).
rewrite h2 at 1.
reflexivity.
Qed.

Lemma non_zero_el_prod_compose_inj_p :
  forall
    {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
    (f f':Fin_map E signe mns)
    (g:(btp Bp)->Atp),
    el_prod_compose_p g f <> %0 ->
    el_prod_compose_p g f = el_prod_compose_p g f' ->
    im_eps_compose_p f g = im_eps_compose_p f' g.
intros Bp E f f' g h1 h2. 
rewrite im_eps_compose_p_eq. rewrite (im_eps_compose_p_eq f').
rewrite el_prod_compose_p_eq in h1.
rewrite el_prod_compose_p_eq in h2. rewrite (el_prod_compose_p_eq g f') in h2.
rewrite ba_conv_zero in h1.
pose proof (non_zero_el_prod_compose_inj  (ba_conv Ap) (ba_conv_fin_map_dom f) (ba_conv_fin_map_dom f') (ba_conv_fun1 g) h1 h2) as h3.
rewrite h3 at 1.
reflexivity.
Qed.


Lemma non_zero_el_prod_compose_constant_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (g:btp Bp->Atp) (f:Fin_map E signe mns),
    el_prod_compose_p g f <> %0 ->
    forall (x x':btp Bp),
      Ensembles.In E x -> Ensembles.In E x' ->
      g x = g x' ->
      f |-> x = f |-> x'.
intros Bp E g f h1 x x' h2 h3 h4.
rewrite el_prod_compose_p_eq in h1.
pose proof (non_zero_el_prod_compose_constant (ba_conv Ap) (ba_conv_fun1 g) (ba_conv_fin_map_dom f) h1 x x' h2 h3 h4) as h5.
assumption.
Qed.

Lemma el_prod_disjoint_p' :
  forall (E:Ensemble Atp) (X Y:Ensemble (Fin_map E signe mns)),
    Inhabited [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x] ->
         (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod_p (fst x) %* el_prod_p (snd x))) = Singleton %0.
intros E X Y h1.
pose (Im X (ba_conv_fin_map_dom (def:=mns))) as X'.
pose (Im Y (ba_conv_fin_map_dom (def:=mns))) as Y'.
assert (h2:Inhabited
         [x : Fin_map E signe mns * Fin_map E signe mns
         | Ensembles.In (cart_prod  X' Y') x /\ fst x <> snd x]).
  destruct h1 as [f h1].
  destruct f as [fx fy].
  apply Inhabited_intro with (ba_conv_fin_map_dom fx, ba_conv_fin_map_dom fy).
  constructor. simpl. 
  inversion h1 as [h2]. simpl in h2. destruct h2 as [h2a h2b].
  destruct h2a as [h2a]. simpl in h2a; destruct h2a as [h2a h2c].
  split. constructor; simpl. split. 
  apply Im_intro with fx; auto. apply Im_intro with fy; auto.
  assumption.
pose proof (el_prod_disjoint' (ba_conv Ap) (ba_conv_set E) (Im X (ba_conv_fin_map_dom (def:=mns))) (Im Y (ba_conv_fin_map_dom (def:=mns))) h2) as h3.
rewrite ba_conv_zero.
rewrite <- h3 at 1.
f_equal.
apply Extensionality_Ensembles.
red. split.
red. intros f h4.
destruct h4 as [h4]. destruct f as [fx fy]. destruct h4 as [h4 h5].
simpl in h5.
constructor. simpl. split; auto.
constructor; simpl.
destruct h4 as [h4]. simpl in h4; destruct h4 as [h4a h4b].
split. apply Im_intro with fx; auto. apply Im_intro with fy; auto.
red.
intros f h4.  destruct h4 as [h4]. destruct h4 as [h4a h4b].
destruct h4a as [h4a]. destruct h4a as [h4a h4c].
destruct f as [fx fy]. simpl in h4a, h4b, h4c.
constructor; simpl.
destruct h4a as [fx h4a]. subst. destruct h4c as [fy h4c]. subst.
constructor. constructor; auto.
assumption.
apply functional_extensionality.
intro f. destruct f as [fx fy]. simpl.
rewrite el_prod_p_eq. rewrite (el_prod_p_eq fy).
reflexivity.
Qed.



Lemma el_prod_compose_disjoint_p' :
  forall {Bp:Bool_Alg_p T} (E:Ensemble (btp Bp))
         (g:(btp Bp)->Atp)
         (X Y:Ensemble (Fin_map E signe mns)),
    Inhabited [x : Fin_map E signe mns * Fin_map E signe mns
              | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x] ->
    (Im
       [x : Fin_map E signe mns * Fin_map E signe mns
       | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
       (fun x : Fin_map E signe mns * Fin_map E signe mns =>
          el_prod_compose_p g (fst x) %* el_prod_compose_p g (snd x))) = Singleton %0.
intros Bp E g X Y h1.
pose (Im X (ba_conv_fin_map_dom (def:=mns))) as X'.
pose (Im Y (ba_conv_fin_map_dom (def:=mns))) as Y'.
assert (h2:Inhabited
         [x : Fin_map E signe mns * Fin_map E signe mns
         | Ensembles.In (cart_prod  X' Y') x /\ fst x <> snd x]).
  destruct h1 as [f h1].
  destruct f as [fx fy].
  apply Inhabited_intro with (ba_conv_fin_map_dom fx, ba_conv_fin_map_dom fy).
  constructor. simpl. 
  inversion h1 as [h2]. simpl in h2. destruct h2 as [h2a h2b].
  destruct h2a as [h2a]. simpl in h2a; destruct h2a as [h2a h2c].
  split. constructor; simpl. split. 
  apply Im_intro with fx; auto. apply Im_intro with fy; auto.
  assumption.
pose proof (el_prod_compose_disjoint' (ba_conv Ap) (ba_conv_set E) (ba_conv_fun1 g) (Im X (ba_conv_fin_map_dom (def:=mns))) (Im Y (ba_conv_fin_map_dom (def:=mns))) h2) as h3.
rewrite ba_conv_zero.
rewrite <- h3 at 1.
f_equal.
apply Extensionality_Ensembles.
red. split.
red. intros f h4.
destruct h4 as [h4]. destruct f as [fx fy]. destruct h4 as [h4 h5].
simpl in h5.
constructor. simpl. split; auto.
constructor; simpl.
destruct h4 as [h4]. simpl in h4; destruct h4 as [h4a h4b].
split. apply Im_intro with fx; auto. apply Im_intro with fy; auto.
red.
intros f h4.  destruct h4 as [h4]. destruct h4 as [h4a h4b].
destruct h4a as [h4a]. destruct h4a as [h4a h4c].
destruct f as [fx fy]. simpl in h4a, h4b, h4c.
constructor; simpl.
destruct h4a as [fx h4a]. subst. destruct h4c as [fy h4c]. subst.
constructor. constructor; auto.
assumption.
apply functional_extensionality.
intro f. destruct f as [fx fy]. simpl.
rewrite el_prod_compose_p_eq. rewrite (el_prod_compose_p_eq (ba_conv_fun1 g) fy) at 1.
reflexivity.
Qed.



Lemma plus_set_el_prod_disjoint_p :
  forall (E:Ensemble Atp) (X Y:Ensemble (Fin_map E signe mns))
         (pf:Finite [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]),
         plus_set_p (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod_p (fst x) %* el_prod_p (snd x))) (finite_image _ _ _ _ pf) = %0.
intros E X Y h1.
pose (Im X (ba_conv_fin_map_dom (def:=mns))) as X'.
pose (Im Y (ba_conv_fin_map_dom (def:=mns))) as Y'.
  assert (h0: [x : Fin_map E signe mns * Fin_map E signe mns
     | Ensembles.In (cart_prod X' Y') x /\ fst x <> snd x]
              = Im [x : Fin_map E signe mns * Fin_map E signe mns
         | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
                   (fun pr=> (ba_conv_fin_map_dom (fst pr), ba_conv_fin_map_dom (snd pr)))).
    apply Extensionality_Ensembles.
    red. split.
    red. intros pr h2. apply Im_intro with pr; auto.
    destruct h2 as [h2]. destruct h2 as [h2a h2b]. destruct h2a as [h2a].
    destruct h2a as [h2a h2c]. destruct pr as [fx fy]. 
    simpl in h2a, h2b, h2c.
    destruct h2a as [fx h2a]. subst. destruct h2c as [fy h2c].
    subst.
    constructor; simpl. split. constructor. simpl.
    split; auto. assumption. 
    apply surjective_pairing.
    red. intros f h2.
    destruct h2 as [pr h2]. subst.
    destruct h2 as [h2].  destruct h2 as [h2a h2b]. destruct h2a as [h2a].
    destruct h2a as [h2a h2c]. destruct pr as [fx fy]. 
    simpl in h2a, h2b, h2c.  simpl.
    constructor; simpl. split. constructor; simpl.
    split. apply Im_intro with fx; auto. apply Im_intro with fy; auto.
    assumption.
assert (h2:Finite
         [x : Fin_map E signe mns * Fin_map E signe mns
         | Ensembles.In (cart_prod X' Y') x /\ fst x <> snd x]).
  rewrite h0.
  apply finite_image; auto.
pose proof (plus_set_el_prod_disjoint (ba_conv Ap) E X' Y' h2) as h3.
 
rewrite plus_set_p_eq.
rewrite ba_conv_zero.
rewrite <- h3.
 
unfold ba_conv_elt. unfold ba_conv_type. rewrite transfer_eq_refl.
unfold ba_conv_set. unfold ba_conv_type.
apply (plus_set_functional (B:=ba_conv Ap)).
rewrite transfer_dep_eq_refl.
f_equal. 
unfold Atp, bt. unfold Atp, bt in h0.
rewrite h0 at 1.
pose proof (im_id [x : Fin_map E signe mns * Fin_map E signe mns
   | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]) as h4.
unfold Atp, bt in h4. unfold Atp, bt.
rewrite h4 at 1.
f_equal.
apply functional_extensionality.
intro x. unfold ba_conv_fin_map_dom. unfold id.
apply surjective_pairing.
apply functional_extensionality.
intro f. destruct f as [fx fy].
simpl.
rewrite el_prod_p_eq. rewrite (el_prod_p_eq fy).
reflexivity.
Qed.


Lemma plus_set_el_prod_disjoint_p' :
  forall (E:Ensemble Atp) (X Y:Ensemble (Fin_map E signe mns))
         (pf:Finite [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x])
         (pfi:Finite (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod_p (fst x) %* el_prod_p (snd x)))),
         plus_set_p (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod_p (fst x) %* el_prod_p (snd x))) pfi = %0.
intros E X Y h1 h2.
pose proof (plus_set_el_prod_disjoint_p _ X Y h1) as h3.
rewrite <- h3.
apply plus_set_functional_p. reflexivity.
Qed.

Lemma plus_set_el_prod_compose_disjoint_p :
  forall {Bp:Bool_Alg_p T} (E:Ensemble (btp Bp)) (g:(btp Bp)->Atp) (X Y:Ensemble (Fin_map E signe mns))
         (pf:Finite [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]),
         plus_set_p (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod_compose_p g (fst x) %* el_prod_compose_p g (snd x))) (finite_image _ _ _ _ pf) = %0.
intros Bp E g X Y h1.
pose (Im X (ba_conv_fin_map_dom (def:=mns))) as X'.
pose (Im Y (ba_conv_fin_map_dom (def:=mns))) as Y'.
  assert (h0: [x : Fin_map E signe mns * Fin_map E signe mns
     | Ensembles.In (cart_prod X' Y') x /\ fst x <> snd x]
              = Im [x : Fin_map E signe mns * Fin_map E signe mns
         | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
                   (fun pr=> (ba_conv_fin_map_dom (fst pr), ba_conv_fin_map_dom (snd pr)))).
    apply Extensionality_Ensembles.
    red. split.
    red. intros pr h2. apply Im_intro with pr; auto.
    destruct h2 as [h2]. destruct h2 as [h2a h2b]. destruct h2a as [h2a].
    destruct h2a as [h2a h2c]. destruct pr as [fx fy]. 
    simpl in h2a, h2b, h2c.
    destruct h2a as [fx h2a]. subst. destruct h2c as [fy h2c].
    subst.
    constructor; simpl. split. constructor. simpl.
    split; auto. assumption. 
    apply surjective_pairing.
    red. intros f h2.
    destruct h2 as [pr h2]. subst.
    destruct h2 as [h2].  destruct h2 as [h2a h2b]. destruct h2a as [h2a].
    destruct h2a as [h2a h2c]. destruct pr as [fx fy]. 
    simpl in h2a, h2b, h2c.  simpl.
    constructor; simpl. split. constructor; simpl.
    split. apply Im_intro with fx; auto. apply Im_intro with fy; auto.
    assumption.
assert (h2:Finite
         [x : Fin_map E signe mns * Fin_map E signe mns
         | Ensembles.In (cart_prod X' Y') x /\ fst x <> snd x]).
  rewrite h0.
  apply finite_image; auto.
pose proof (plus_set_el_prod_compose_disjoint (ba_conv Ap) (ba_conv_set E) (ba_conv_fun1 g) X' Y' h2) as h3.
 
rewrite plus_set_p_eq.
rewrite ba_conv_zero.
rewrite <- h3.
 
unfold ba_conv_elt. unfold ba_conv_type. rewrite transfer_eq_refl.
unfold ba_conv_set. unfold ba_conv_type.
apply (plus_set_functional (B:=ba_conv Ap)).
rewrite transfer_dep_eq_refl.
f_equal. 
unfold Atp, bt. unfold Atp, bt in h0.
rewrite h0 at 1.
pose proof (im_id [x : Fin_map E signe mns * Fin_map E signe mns
   | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]) as h4.
unfold Atp, bt in h4. unfold Atp, bt.
rewrite h4 at 1.
f_equal.
apply functional_extensionality.
intro x. unfold ba_conv_fin_map_dom. unfold id.
apply surjective_pairing.
apply functional_extensionality.
intro f. destruct f as [fx fy].
simpl.
rewrite el_prod_compose_p_eq. rewrite (el_prod_compose_p_eq g fy).
reflexivity.
Qed.

Lemma plus_set_el_prod_compose_disjoint_p' :
  forall {Bp:Bool_Alg_p T} (E:Ensemble (btp Bp)) (g:(btp Bp)->Atp) (X Y:Ensemble (Fin_map E signe mns))
         (pf:Finite [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x])
         (pfi:Finite (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod_compose_p g (fst x) %* el_prod_compose_p g (snd x)))),
         plus_set_p (Im
            [x : Fin_map E signe mns * Fin_map E signe mns
            | Ensembles.In (cart_prod X Y) x /\ fst x <> snd x]
            (fun x : Fin_map E signe mns * Fin_map E signe mns =>
             el_prod_compose_p g (fst x) %* el_prod_compose_p g (snd x))) pfi = %0.
intros Bp E g X Y h1 h2.
pose proof (plus_set_el_prod_compose_disjoint_p _ g X Y h1) as h3.
rewrite <- h3.
apply plus_set_functional_p. reflexivity.
Qed.

Lemma el_prod_covers_p :
  forall (E:Ensemble Atp) (pf:Finite E),
    plus_set_p (Im (Full_set (Fin_map E signe mns)) el_prod_p)
    (finite_image _ _ _ _ (finite_fin_maps _ _ mns pf signe_finite)) = %1.
intros E h1.
pose proof (el_prod_covers _ (ba_conv_set E) h1) as h2.
rewrite ba_conv_one. rewrite <- h2.
rewrite plus_set_p_eq.
unfold ba_conv_elt, ba_conv_set, ba_conv_type.
rewrite transfer_eq_refl.
apply (plus_set_functional (B:=ba_conv Ap)).
rewrite transfer_dep_eq_refl.
f_equal.
apply functional_extensionality.
intro x. rewrite el_prod_p_eq. reflexivity.
Qed.



Lemma el_prod_covers_p' :
  forall (E:Ensemble Atp) (pf:Finite E)
         (pf:Finite (Im (Full_set (Fin_map E signe mns)) el_prod_p)),
    plus_set_p (Im (Full_set (Fin_map E signe mns)) el_prod_p) pf = %1.
intros E h1 h2.
pose proof (el_prod_covers_p _ h1) as h3.
rewrite <- h3.
apply plus_set_functional_p. reflexivity.
Qed.


Lemma el_prod_ex_non_zero_el_prod_compose_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (g:btp Bp->Atp) (a:Fin_map E signe mns),
    el_prod_compose_p g a <> %0 ->
    exists a':Fin_map (Im E g) signe mns,
      el_prod_p a' = el_prod_compose_p g a.
intros Bp E g a h1.
rewrite el_prod_compose_p_eq, ba_conv_zero in h1.
pose proof (el_prod_ex_non_zero_el_prod_compose (ba_conv Ap) (ba_conv_fun1 g) (ba_conv_fin_map_dom a) h1) as h2.
destruct h2 as [a' h2].
assert (h3:Im E g = Im (ba_conv_set E) (ba_conv_fun1 g)).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h3. destruct h3 as [x h3]. subst.
  apply Im_intro with x; auto.
  red. intros x h3. destruct h3 as [x h3]. subst.
  apply Im_intro with x; auto.
rewrite h3.
exists a'.
rewrite el_prod_p_eq, el_prod_compose_p_eq.
unfold ba_conv_fin_map_dom.
unfold ba_conv_set, bt, ba_conv_type, Atp in h2.
unfold ba_conv_set, bt, ba_conv_type, Atp.
rewrite h2 at 1.
reflexivity.
Qed.


Lemma non_zero_el_prod_compose_ex_el_prod_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)},
    Finite E ->
    forall (g:btp Bp->Atp) (a:Fin_map (Im E g) signe mns),
    el_prod_p a <> %0 ->
    exists a':Fin_map E signe mns,
      el_prod_compose_p g a' = el_prod_p a.
intros Bp E h1 g a h2.
rewrite el_prod_p_eq, ba_conv_zero in h2.
assert (h3:Included (Im (ba_conv_set E) (ba_conv_fun1 g)) (Im E g)).
  red. intros x h4.
  destruct h4 as [x h4]. subst. apply Im_intro with x; auto.
pose proof (@non_zero_el_prod_compose_ex_el_prod (ba_conv Ap) (ba_conv Bp) _  h1 (ba_conv_fun g) (ba_conv_fin_map_dom (restriction a h3))) as h4.
assert (h5:el_prod (ba_conv Ap) (ba_conv_fin_map_dom (restriction a h3)) <> 0).
  intro h5. rewrite <- h5 in h2. contradict h2.
  f_equal. apply fin_map_ext_in.
  intros x h6. unfold ba_conv_fin_map_dom. rewrite restriction_compat.
  reflexivity.
  destruct h6 as [x h6]. subst. apply Im_intro with x; auto.
specialize (h4 h5).
destruct h4 as [a' h4].
exists a'. rewrite el_prod_compose_p_eq, el_prod_p_eq. rewrite h4 at 1.
unfold ba_conv_fin_map_dom. f_equal.
apply fin_map_ext_in.
intros x h6. rewrite restriction_compat; auto.
Qed.


Lemma el_prod_compose_covers_p :
  forall {Bp:Bool_Alg_p T} (E:Ensemble (btp Bp)) (pf:Finite E)
         (g:(btp Bp)->Atp),
    plus_set_p (Im (Full_set (Fin_map E signe mns)) (el_prod_compose_p g))
    (finite_image _ _ _ _ (finite_fin_maps _ _ mns pf signe_finite)) = %1.
intros Bp E h1 g.
rewrite plus_set_p_eq. unfold ba_conv_elt, ba_conv_type.
rewrite transfer_eq_refl.
pose proof (el_prod_compose_covers (ba_conv Ap) (ba_conv_set E) h1 (ba_conv_fun g)) as h2.
rewrite ba_conv_one. rewrite <- h2.
apply (plus_set_functional (B:=ba_conv Ap)).
unfold ba_conv_set, ba_conv_type. rewrite transfer_dep_eq_refl.
f_equal. apply functional_extensionality. 
intros f. rewrite el_prod_compose_p_eq.
reflexivity.
Qed.

Definition non_zero_el_prod_maps_p (E:Ensemble Atp) :=
  [a:Fin_map E signe mns | el_prod_p a <> %0].

Lemma non_zero_el_prod_maps_p_eq : 
    forall (E:Ensemble Atp),
      non_zero_el_prod_maps_p E = non_zero_el_prod_maps (ba_conv Ap) (ba_conv_set E).
intro E.
unfold non_zero_el_prod_maps_p, non_zero_el_prod_maps.
apply Extensionality_Ensembles.
red. split.
red. intros x h1.
destruct h1 as [h1]. constructor.
rewrite el_prod_p_eq in h1.
assumption.
red. intros x h1.
destruct h1 as [h1]. constructor.
rewrite el_prod_p_eq.
assumption.
Qed.

Lemma non_zero_el_prod_maps_fin_p :
  forall (E:Ensemble Atp), Finite E ->
                          Finite (non_zero_el_prod_maps_p E).
intros E h1.
assert (h2:Included (non_zero_el_prod_maps_p E) (Full_set (Fin_map E signe mns))). red. intros; constructor.
pose proof (finite_fin_maps _ _ mns h1 signe_finite) as h3.
apply (Finite_downward_closed _ _  h3 _ h2).
Qed.


Definition non_zero_el_prod_compose_maps_p
           {Bp:Bool_Alg_p T} (g:(btp Bp)->Atp) (E:Ensemble (btp Bp)) :=
  [a:Fin_map E signe mns | el_prod_compose_p g a <> %0].

Lemma non_zero_el_prod_compose_maps_p_eq : 
  forall {Bp:Bool_Alg_p T} (g:(btp Bp)->Atp) (E:Ensemble (btp Bp)),
    non_zero_el_prod_compose_maps_p g E = 
    non_zero_el_prod_compose_maps (ba_conv Ap) (ba_conv_fun g) (ba_conv_set E).
intros Bp g E.
unfold non_zero_el_prod_compose_maps_p, non_zero_el_prod_compose_maps.
apply Extensionality_Ensembles.
red. split.
red. intros x h1.
destruct h1 as [h1]. constructor.
rewrite el_prod_compose_p_eq in h1. assumption.
red. intros x h1.
destruct h1 as [h1]. constructor.
rewrite el_prod_compose_p_eq. assumption.
Qed.


Lemma non_zero_el_prod_compose_maps_fin_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (g:btp Bp->Atp),
    Finite E -> Finite (non_zero_el_prod_compose_maps_p g E).
intros B E g h1.
assert (h2:Included (non_zero_el_prod_compose_maps_p g E) (Full_set (Fin_map E signe mns))). red. intros; constructor.
pose proof (finite_fin_maps _ _ mns h1 signe_finite) as h3.
apply (Finite_downward_closed _ _  h3 _ h2).
Qed.


Definition plus_subset_non_zero_el_prod_maps_p 
           (E:Ensemble Atp) (pfe:Finite E) (S:Ensemble (Fin_map E signe mns))
           (pfi:Included S (non_zero_el_prod_maps_p E)) : Atp :=
  plus_set_p (Im S el_prod_p) (finite_image _ _ _ el_prod_p (Finite_downward_closed _ _ (non_zero_el_prod_maps_fin_p _ pfe) _ pfi)).


Lemma plus_subset_non_zero_el_prod_maps_p_eq : 
  forall (E:Ensemble Atp) (pfe:Finite E) 
         (S:Ensemble (Fin_map E signe mns))
         (pfi:Included S (non_zero_el_prod_maps_p E)),
    plus_subset_non_zero_el_prod_maps_p E pfe S pfi =
    plus_subset_non_zero_el_prod_maps _ (ba_conv_set E) pfe (ba_conv_ens_fin_map_dom S) (eq_rect _ _ (iff1 (incl_ba_conv_ens_fin_map_dom_iff (def:=mns) _ _ ) pfi) _ (non_zero_el_prod_maps_p_eq _ )).
intros E h1 S h2.
unfold plus_subset_non_zero_el_prod_maps_p, plus_subset_non_zero_el_prod_maps, ba_conv_set, ba_conv_ens_fin_map_dom.
simpl.
rewrite plus_set_p_eq.
apply (plus_set_functional (B:=ba_conv Ap)).
unfold ba_conv_set, ba_conv_type. erewrite transfer_dep_eq_refl.
unfold eq_rect_r. simpl.
f_equal.
apply functional_extensionality.
intro f.
rewrite el_prod_p_eq. reflexivity.
Qed.

Definition plus_subset_non_zero_el_prod_compose_maps_p
           {Bp:Bool_Alg_p T} (E:Ensemble (btp Bp)) (pfe:Finite E)
           (g:(btp Bp)->Atp) (S:Ensemble (Fin_map E signe mns))
           (pfi:Included S (non_zero_el_prod_compose_maps_p g E)) : Atp :=
  plus_set_p (Im S (el_prod_compose_p g)) (finite_image _ _ _ (el_prod_compose_p g) (finite_fin_map_ens S pfe signe_finite)).

Lemma plus_subset_non_zero_el_prod_compose_maps_p_eq : 
  forall {Bp:Bool_Alg_p T} (E:Ensemble (btp Bp)) (pfe:Finite E)
         (g:(btp Bp)->Atp) (S:Ensemble (Fin_map E signe mns))
         (pfi:Included S (non_zero_el_prod_compose_maps_p g E)),
    plus_subset_non_zero_el_prod_compose_maps_p E pfe g S pfi =
    plus_subset_non_zero_el_prod_compose_maps (ba_conv Ap) (ba_conv_set E) pfe (ba_conv_fun g) (ba_conv_ens_fin_map_dom S) (eq_rect _ _ (iff1 (incl_ba_conv_ens_fin_map_dom_iff (def:=mns) _ _ ) pfi) _ (non_zero_el_prod_compose_maps_p_eq _ _ )).
intros Bp E h1 g S h2.
unfold plus_subset_non_zero_el_prod_compose_maps_p, plus_subset_non_zero_el_prod_compose_maps, ba_conv_set, ba_conv_ens_fin_map_dom.
simpl.
rewrite plus_set_p_eq.
apply (plus_set_functional (B:=ba_conv Ap)).
unfold ba_conv_set, ba_conv_type. erewrite transfer_dep_eq_refl.
unfold eq_rect_r. simpl.
f_equal.
apply functional_extensionality.
intro f.
rewrite el_prod_compose_p_eq. reflexivity.
Qed.
    

Lemma inclusion_plus_subset_non_zero_el_prod_maps_preserves_int_p :
  forall (E:Ensemble Atp) (pfe:Finite E) (X Y:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_maps_p E) ->
    Included Y (non_zero_el_prod_maps_p E) ->
    Included (Intersection X Y) (non_zero_el_prod_maps_p E).
intros E h1 X Y h2 h3.
red.
intros a h4.
destruct h4 as [a h4l h4r].
constructor.
red in h3.
specialize (h3 _ h4r).
destruct h3.
assumption.
Qed.



Lemma inclusion_plus_subset_non_zero_el_prod_maps_preserves_union_p :
  forall (E:Ensemble Atp) (pfe:Finite E) (X Y:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_maps_p E) ->
    Included Y (non_zero_el_prod_maps_p E) ->
    Included (Union X Y) (non_zero_el_prod_maps_p E).
intros E h1 X Y h2 h3. red.
intros a h4.
destruct h4 as [a h4l | a h4r].
red in h2. specialize (h2 _ h4l). assumption.
red in h3. specialize (h3 _ h4r). assumption.
Qed.


Lemma inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp_p :
  forall (E:Ensemble Atp) (pfe:Finite E) (X:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_maps_p E) ->
    Included (Setminus (non_zero_el_prod_maps_p E) X) (non_zero_el_prod_maps_p E).
intros E h1 X h2. red.
intros a h4.
destruct h4 as [h4l].
assumption.
Qed.

Lemma inclusion_plus_subset_non_zero_el_prod_maps_preserves_setminus_p :
  forall (E:Ensemble Atp) (pfe:Finite E) (X Y:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_maps_p E) ->
    Included (Setminus X Y) (non_zero_el_prod_maps_p E).
intros E h1 X Y h2. red.
intros a h3.
destruct h3 as [h3a h3b].
auto with sets.
Qed.

Lemma inclusion_plus_subset_non_zero_el_prod_maps_preserves_symdiff_full_p :
  forall (E:Ensemble Atp) (pfe:Finite E) (X Y:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_maps_p E) ->
    Included Y (non_zero_el_prod_maps_p E) ->
    Included (Symdiff_full (non_zero_el_prod_maps_p E) X Y) (non_zero_el_prod_maps_p E).
intros E h1 X Y h2 h3. red.
intros a h4.
destruct h4 as [a h4l | a h4r].
destruct h4l as [a h4a h4b].  auto with sets.
destruct h4r as [a h4a h4b]. auto with sets.
Qed.




Lemma inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_union_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (pfe:Finite E) (g:(btp Bp)->Atp)
         (X Y:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_compose_maps_p g E) ->
    Included Y (non_zero_el_prod_compose_maps_p g E) ->
    Included (Union X Y) (non_zero_el_prod_compose_maps_p g E).
intros B E h1 g X Y h2 h3. red.
intros a h4.
destruct h4 as [a h4l | a h4r].
red in h2. specialize (h2 _ h4l). assumption.
red in h3. specialize (h3 _ h4r). assumption.
Qed.


Lemma inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_int_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (pfe:Finite E) (g:(btp Bp)->Atp)
         (X Y:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_compose_maps_p g E) ->
    Included Y (non_zero_el_prod_compose_maps_p g E) ->
    Included (Intersection X Y) (non_zero_el_prod_compose_maps_p g E).
intros B E h1 g X Y h2 h3.
red.
intros a h4.
destruct h4 as [a h4l h4r].
constructor.
red in h3.
specialize (h3 _ h4r).
destruct h3.
assumption.
Qed.


Lemma inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_comp_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (pfe:Finite E) (g:(btp Bp)->Atp)
         (X:Ensemble (Fin_map E signe mns)),
    Included X (non_zero_el_prod_compose_maps_p g E) ->
    Included (Setminus (non_zero_el_prod_compose_maps_p g E) X) (non_zero_el_prod_compose_maps_p g E).
intros B E h1 g X h2. red.
intros a h4.
destruct h4 as [h4l].
assumption.
Qed.


Lemma plus_subset_non_zero_el_prod_maps_empty_p :
  forall (E:Ensemble Atp) (pfe:Finite E),
    plus_subset_non_zero_el_prod_maps_p E pfe _ (empty_inclusion _) = %0.
intros E h1.
rewrite plus_subset_non_zero_el_prod_maps_p_eq.
rewrite ba_conv_zero.
rewrite <- (plus_subset_non_zero_el_prod_maps_empty (ba_conv Ap) _ h1).
f_equal. apply proof_irrelevance.
Qed.

Lemma le_plus_subset_non_zero_el_prod_maps_p :
  forall (E:Ensemble Atp) (pfe:Finite E) 
         (X:Ensemble (Fin_map E signe mns))
         (pf:Included X (non_zero_el_prod_maps_p E)) 
         (a:Fin_map E signe mns),
    Ensembles.In X a -> 
    le_p (el_prod_p a) (plus_subset_non_zero_el_prod_maps_p E pfe _ pf).
intros E h1 X h2 a h3.
unfold plus_subset_non_zero_el_prod_maps_p.
apply le_plus_set_p.
apply Im_intro with a; auto.
Qed.


Lemma plus_subset_non_zero_el_prod_maps_empty_rev_p :
  forall (E:Ensemble Atp) (pfe:Finite E) (X:Ensemble (Fin_map E signe mns))
         (pf:Included X (non_zero_el_prod_maps_p E)),
         plus_subset_non_zero_el_prod_maps_p E pfe _ pf = %0 -> X = Empty_set _.
intros E h1 X h2 h3.
assert (h5: Included (ba_conv_ens_fin_map_dom X)
                (non_zero_el_prod_maps (ba_conv Ap) E)). 
  red. intros f h5.
  pose proof h2 as h2'. rewrite non_zero_el_prod_maps_p_eq in h2'.
  apply h2'. assumption.
assert (h6: plus_subset_non_zero_el_prod_maps (ba_conv Ap) E h1
         (ba_conv_ens_fin_map_dom X) h5 = 0).
  rewrite plus_subset_non_zero_el_prod_maps_p_eq in h3.
  rewrite ba_conv_zero in h3. rewrite <- h3.
  f_equal. apply proof_irrelevance.
pose proof (plus_subset_non_zero_el_prod_maps_empty_rev (ba_conv Ap) _ h1 (ba_conv_ens_fin_map_dom X) h5 h6) as h4.
assumption.
Qed.

Lemma plus_set_el_prod_zero_p : 
  forall (E:Ensemble Atp) (pf:Finite _), 
    plus_set_p (Im [x : Fin_map E signe mns | el_prod_p x = %0] el_prod_p) pf = %0.
intros E h1. 
rewrite plus_set_p_eq. 
  assert (h3:
    Im [x : Fin_map E signe mns | el_prod_p x = %0] el_prod_p =
    Im [x : Fin_map E signe mns | el_prod (ba_conv Ap) x = 0]
        (el_prod (ba_conv Ap))).
    apply Extensionality_Ensembles.
    red. split.
    red. intros x h2.
    destruct h2 as [x h2]. subst.
    destruct h2 as [h2]. rewrite el_prod_p_eq in h2.
    apply Im_intro with x. constructor. assumption.
    rewrite el_prod_p_eq. reflexivity.
    red. intros x h2.
    destruct h2 as [x h2]. subst.
    destruct h2 as [h2]. apply Im_intro with x.
    constructor. rewrite el_prod_p_eq. assumption.
    rewrite el_prod_p_eq. reflexivity.
assert (h2:Finite  (Im [x : Fin_map E signe mns | el_prod (ba_conv Ap) x = 0] (el_prod (ba_conv Ap)))).
  rewrite <- h3. assumption.
pose proof (plus_set_el_prod_zero _ (ba_conv_set E) h2) as h4.
rewrite ba_conv_zero. rewrite <- h4.
unfold ba_conv_elt, ba_conv_type. rewrite transfer_eq_refl.
apply (plus_set_functional (B:=ba_conv Ap)).
unfold ba_conv_set, ba_conv_type. rewrite transfer_dep_eq_refl.
f_equal.
apply Extensionality_Ensembles.
red. split.
red. intros x h5. destruct h5 as [h5]. constructor.
rewrite el_prod_p_eq in h5. assumption.
red. intros x h5. destruct h5 as [h5]. constructor.
rewrite el_prod_p_eq. assumption.
apply functional_extensionality.
intro x. rewrite el_prod_p_eq. reflexivity.
Qed.


Lemma plus_set_el_prod_compose_zero_p :
  forall {Bp:Bool_Alg_p T}  (g:btp Bp->Atp) (E:Ensemble (btp Bp))
         (pf:Finite _),
         plus_set_p (Im [x : Fin_map E signe mns | el_prod_compose_p g x = %0] (el_prod_compose_p g)) pf = %0.
intros Bp g E h1. 
rewrite plus_set_p_eq. 
  assert (h3:
    Im [x : Fin_map E signe mns | el_prod_compose_p g x = %0] (el_prod_compose_p g) =
    Im [x : Fin_map E signe mns | el_prod_compose (ba_conv Ap) (ba_conv_fun g) x = 0]
        (el_prod_compose (ba_conv Ap) (ba_conv_fun g))).
    apply Extensionality_Ensembles.
    red. split.
    red. intros x h2.
    destruct h2 as [x h2]. subst.
    destruct h2 as [h2]. rewrite el_prod_compose_p_eq in h2.
    apply Im_intro with x. constructor. assumption.
    rewrite el_prod_compose_p_eq. reflexivity.
    red. intros x h2.
    destruct h2 as [x h2]. subst.
    destruct h2 as [h2]. apply Im_intro with x.
    constructor. rewrite el_prod_compose_p_eq. assumption.
    rewrite el_prod_compose_p_eq. reflexivity.
assert (h2:Finite  (Im [x : Fin_map E signe mns | el_prod_compose (ba_conv Ap) (B:=ba_conv Bp) (ba_conv_fun g) x = 0] (el_prod_compose _ (ba_conv_fun g)))).
  rewrite <- h3. assumption.
pose proof (plus_set_el_prod_compose_zero _ (ba_conv_fun g) (ba_conv_set E) h2) as h4.
rewrite ba_conv_zero. rewrite <- h4.
unfold ba_conv_elt, ba_conv_type. rewrite transfer_eq_refl.
apply (plus_set_functional (B:=ba_conv Ap)).
unfold ba_conv_set, ba_conv_type. rewrite transfer_dep_eq_refl.
f_equal.
apply Extensionality_Ensembles.
red. split.
red. intros x h5. destruct h5 as [h5]. constructor.
rewrite el_prod_compose_p_eq in h5. assumption.
red. intros x h5. destruct h5 as [h5]. constructor.
rewrite el_prod_compose_p_eq. assumption.
apply functional_extensionality.
intro x. rewrite el_prod_compose_p_eq. reflexivity.
Qed.

Lemma plus_subset_non_zero_el_prod_maps_full_p :
  forall (E:Ensemble Atp) (pfe:Finite E),
    plus_subset_non_zero_el_prod_maps_p E pfe _ (inclusion_reflexive _) = %1.
intros E h1.
rewrite plus_subset_non_zero_el_prod_maps_p_eq.
rewrite ba_conv_one.
pose proof (plus_subset_non_zero_el_prod_maps_full (ba_conv Ap) E h1) as h2.
rewrite <- h2 at 1. 
generalize (eq_rect (ba_conv_ens_fin_map_dom (non_zero_el_prod_maps_p E))
        (Included (ba_conv_ens_fin_map_dom (non_zero_el_prod_maps_p E)))
        (iff1
           (incl_ba_conv_ens_fin_map_dom_iff (non_zero_el_prod_maps_p E)
              (non_zero_el_prod_maps_p E))
           (inclusion_reflexive (non_zero_el_prod_maps_p E)))
        (non_zero_el_prod_maps (ba_conv Ap) (ba_conv_set (ba_conv_set E)))
         (non_zero_el_prod_maps_p_eq (ba_conv_set E))).
generalize (inclusion_reflexive (non_zero_el_prod_maps (ba_conv Ap) E)). 
rewrite non_zero_el_prod_maps_p_eq. 
intros h3 h4.
assert (h3 = h4). apply proof_irrelevance.
subst.
f_equal.
Qed.


Lemma plus_subset_non_zero_el_prod_maps_int_p :
  forall (E:Ensemble Atp) (pfe:Finite E)
         (X Y:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_maps_p E))
         (pfy:Included Y (non_zero_el_prod_maps_p E)),
         (plus_subset_non_zero_el_prod_maps_p E pfe _ pfx) %*
         (plus_subset_non_zero_el_prod_maps_p E pfe _ pfy) =
         plus_subset_non_zero_el_prod_maps_p E pfe (Intersection X Y) (inclusion_plus_subset_non_zero_el_prod_maps_preserves_int_p _ pfe X Y pfx pfy).
intros E h1 X Y h2 h3.
pose proof (iff1 (incl_ba_conv_ens_fin_map_dom_iff _ _) h2) as h4.
pose proof (iff1 (incl_ba_conv_ens_fin_map_dom_iff _ _) h3) as h5.
rewrite non_zero_el_prod_maps_p_eq in h4, h5.
pose proof (plus_subset_non_zero_el_prod_maps_int (ba_conv Ap) _ h1 _ _ h4 h5) as h6.
rewrite plus_subset_non_zero_el_prod_maps_p_eq.
rewrite (plus_subset_non_zero_el_prod_maps_p_eq E h1 Y h3).
rewrite (plus_subset_non_zero_el_prod_maps_p_eq E h1 (Intersection X Y)  (inclusion_plus_subset_non_zero_el_prod_maps_preserves_int_p E h1 X Y h2
        h3)).
assert (h7:inclusion_plus_subset_non_zero_el_prod_maps_preserves_int
            (ba_conv Ap) E h1 (ba_conv_ens_fin_map_dom X)
            (ba_conv_ens_fin_map_dom Y) h4 h5 =
           (eq_rect (ba_conv_ens_fin_map_dom (non_zero_el_prod_maps_p E))
        (Included (ba_conv_ens_fin_map_dom (Intersection X Y)))
        (iff1
           (incl_ba_conv_ens_fin_map_dom_iff (Intersection X Y)
              (non_zero_el_prod_maps_p E))
           (inclusion_plus_subset_non_zero_el_prod_maps_preserves_int_p E h1
              X Y h2 h3))
        (non_zero_el_prod_maps (ba_conv Ap) (ba_conv_set (ba_conv_set E)))
        (non_zero_el_prod_maps_p_eq (ba_conv_set E)))).
  apply proof_irrelevance.
rewrite <- h7.
unfold btp, bt, Atp in h6. unfold Atp, btp, bt. simpl in h6. simpl.
rewrite <- h6 at 1.
f_equal. f_equal. apply proof_irrelevance. f_equal. apply proof_irrelevance.
Qed.

Lemma plus_subset_non_zero_el_prod_maps_union_p :
  forall (E:Ensemble Atp) (pfe:Finite E)
         (X Y:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_maps_p E))
         (pfy:Included Y (non_zero_el_prod_maps_p E)),
         (plus_subset_non_zero_el_prod_maps_p E pfe _ pfx) %+
         (plus_subset_non_zero_el_prod_maps_p E pfe _ pfy) =
         plus_subset_non_zero_el_prod_maps_p E pfe (Union X Y) (inclusion_plus_subset_non_zero_el_prod_maps_preserves_union_p _ pfe X Y pfx pfy).
intros E h1 X Y h2 h3.
unfold plus_subset_non_zero_el_prod_maps_p.
generalize (finite_image (Fin_map E signe mns) Atp (Union X Y) el_prod_p
        (Finite_downward_closed (Fin_map E signe mns)
           (non_zero_el_prod_maps_p E) (non_zero_el_prod_maps_fin_p E h1)
           (Union X Y)
           (inclusion_plus_subset_non_zero_el_prod_maps_preserves_union_p E h1
              X Y h2 h3))).
rewrite <- plus_set_union_p.
intro h4.
apply plus_set_functional_p.
rewrite im_union.
reflexivity.
Qed.

Lemma plus_subset_non_zero_el_prod_maps_comp_p :
  forall (E:Ensemble Atp) (pfe:Finite E)
         (X:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_maps_p E)),
         %- (plus_subset_non_zero_el_prod_maps_p E pfe _ pfx) =
         plus_subset_non_zero_el_prod_maps_p E pfe _ (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp_p _ pfe X pfx).
intros E h1 X h2.
pose proof (incl_ba_conv_ens_fin_map_dom_iff X (non_zero_el_prod_maps_p E)) as h3.
unfold Atp in h2.
pose proof h2 as h2'. rewrite h3 in h2'.
rewrite non_zero_el_prod_maps_p_eq in h2'.
pose proof (plus_subset_non_zero_el_prod_maps_comp (ba_conv Ap) _ h1 _ h2') as h4.
rewrite ba_conv_comp.
rewrite plus_subset_non_zero_el_prod_maps_p_eq.
rewrite (plus_subset_non_zero_el_prod_maps_p_eq E h1).
unfold ba_conv_elt, ba_conv_type. rewrite transfer_eq_refl.
assert (h5:h2' = (eq_rect (ba_conv_ens_fin_map_dom (non_zero_el_prod_maps_p E))
        (Included (ba_conv_ens_fin_map_dom X))
        (iff1
           (incl_ba_conv_ens_fin_map_dom_iff X (non_zero_el_prod_maps_p E))
           h2)
        (non_zero_el_prod_maps (ba_conv Ap) (ba_conv_set (ba_conv_set E)))
        (non_zero_el_prod_maps_p_eq (ba_conv_set E)))).
  apply proof_irrelevance.
subst.
rewrite h4 at 1.
generalize (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp 
        (ba_conv Ap) E h1 (ba_conv_ens_fin_map_dom X)
        (eq_rect (ba_conv_ens_fin_map_dom (non_zero_el_prod_maps_p E))
           (Included (ba_conv_ens_fin_map_dom X))
           (iff1
              (incl_ba_conv_ens_fin_map_dom_iff X (non_zero_el_prod_maps_p E))
              h2)
           (non_zero_el_prod_maps (ba_conv Ap) (ba_conv_set (ba_conv_set E)))
           (non_zero_el_prod_maps_p_eq (ba_conv_set E)))).
generalize  (eq_rect (ba_conv_ens_fin_map_dom (non_zero_el_prod_maps_p E))
        (Included
           (ba_conv_ens_fin_map_dom (Setminus (non_zero_el_prod_maps_p E) X)))
        (iff1
           (incl_ba_conv_ens_fin_map_dom_iff
              (Setminus (non_zero_el_prod_maps_p E) X)
              (non_zero_el_prod_maps_p E))
           (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp_p E h1
              X h2))
        (non_zero_el_prod_maps (ba_conv Ap) (ba_conv_set (ba_conv_set E)))
        (non_zero_el_prod_maps_p_eq (ba_conv_set E))).
rewrite non_zero_el_prod_maps_p_eq.
intros h5 h6.
f_equal. apply proof_irrelevance.
Qed.

Lemma plus_subset_non_zero_el_prod_maps_setminus_p :
  forall (E:Ensemble Atp) (pfe:Finite E)
         (X Y:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_maps_p E))
         (pfy:Included Y (non_zero_el_prod_maps_p E)),
         (plus_subset_non_zero_el_prod_maps_p E pfe _ pfx) %*
         %- (plus_subset_non_zero_el_prod_maps_p E pfe _ pfy) =
         plus_subset_non_zero_el_prod_maps_p E pfe
                                           (Intersection X (Setminus (non_zero_el_prod_maps_p E) Y))
                                                         (inclusion_plus_subset_non_zero_el_prod_maps_preserves_int_p _ pfe X _ pfx
                                                         (inclusion_plus_subset_non_zero_el_prod_maps_preserves_comp_p _ pfe Y pfy)).
intros E h1 X Y h2 h3.
rewrite <- plus_subset_non_zero_el_prod_maps_int_p.
rewrite <- plus_subset_non_zero_el_prod_maps_comp_p.
reflexivity.
Qed.


Lemma plus_subset_non_zero_el_prod_maps_symdiff_full_p :
  forall (E:Ensemble Atp) (pfe:Finite E)
         (X Y:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_maps_p E))
         (pfy:Included Y (non_zero_el_prod_maps_p E)),
         (plus_subset_non_zero_el_prod_maps_p E pfe _ pfx) /%\
         (plus_subset_non_zero_el_prod_maps_p E pfe _ pfy) =
         plus_subset_non_zero_el_prod_maps_p E pfe (Symdiff_full (non_zero_el_prod_maps_p E) X Y) (inclusion_plus_subset_non_zero_el_prod_maps_preserves_symdiff_full_p _ pfe X Y pfx pfy).
intros E h1 X Y h2 h3.
unfold sym_diff_p.
do 2 rewrite plus_subset_non_zero_el_prod_maps_comp_p.
do 2 rewrite plus_subset_non_zero_el_prod_maps_int_p.
rewrite plus_subset_non_zero_el_prod_maps_union_p.
unfold Symdiff_full. unfold Setminus_full.
unfold plus_subset_non_zero_el_prod_maps.
apply plus_set_functional_p. reflexivity.
Qed.


Definition non_zero_el_prod_maps_of_set_p
           {E:Ensemble Atp} (pfe:Finite E)
           (S:Ensemble (Fin_map E signe mns)) :=
  Intersection S (non_zero_el_prod_maps_p E).

Lemma non_zero_el_prod_maps_of_set_p_eq : 
  forall {E:Ensemble Atp} (pfe:Finite E)
         (S:Ensemble (Fin_map E signe mns)),
    non_zero_el_prod_maps_of_set_p pfe S =
    non_zero_el_prod_maps_of_set (ba_conv Ap) pfe (ba_conv_ens_fin_map_dom S).
intros E h1 S.
unfold non_zero_el_prod_maps_of_set_p, non_zero_el_prod_maps_of_set.
rewrite non_zero_el_prod_maps_p_eq.
reflexivity.
Qed.

Lemma non_zero_el_prod_maps_of_set_eq_p :
  forall {E:Ensemble Atp} (pfe:Finite E)
         (S:Ensemble (Fin_map E signe mns)),
    non_zero_el_prod_maps_of_set_p pfe S =
  [f:Fin_map E signe mns | Ensembles.In S f /\ el_prod_p f <> %0].
intros E h1 S.
pose proof (non_zero_el_prod_maps_of_set_eq (ba_conv Ap) h1 (ba_conv_ens_fin_map_dom S)) as h2.
rewrite non_zero_el_prod_maps_of_set_p_eq.
rewrite h2 at 1.
apply Extensionality_Ensembles.
red. split.
red. intros x h3.
destruct h3 as [h3]. destruct h3 as [h3 h4].
constructor. rewrite ba_conv_zero. rewrite el_prod_p_eq.
split; auto.
red. intros x h3.
destruct h3 as [h3]. destruct h3 as [h3 h4].
constructor. rewrite ba_conv_zero in h4. rewrite el_prod_p_eq in h4. split; auto.
Qed.

Lemma non_zero_el_prod_maps_of_set_decompose_p :
  forall {E:Ensemble Atp} (pfe:Finite E)
         (S:Ensemble (Fin_map E signe mns)),
    S = Union (non_zero_el_prod_maps_of_set_p pfe S)
              [f:Fin_map E signe mns | Ensembles.In S f /\ el_prod_p f = %0].
intros E h1 S.
apply Extensionality_Ensembles.
red. split.
red. intros f h2.
destruct (classic (el_prod_p f = %0)) as [h3 | h4].
right. constructor. split; auto.
left. constructor; auto. constructor; auto.
red. intros f h2.
destruct h2 as [f h2 | f h3].
destruct h2; auto.
destruct h3 as [h3]. destruct h3; auto.
Qed.



Lemma incl_non_zero_el_prod_maps_of_set_p :
  forall {E:Ensemble Atp} (pfe:Finite E)
         (S:Ensemble (Fin_map E signe mns)),
    Included (non_zero_el_prod_maps_of_set_p pfe S) (non_zero_el_prod_maps_p E).
intros E h1 S.
unfold non_zero_el_prod_maps_of_set_p.
auto with sets.
Qed.


Lemma finite_non_zero_el_prod_maps_of_set_p :
    forall {E:Ensemble Atp} (pfe:Finite E)
           (S:Ensemble (Fin_map E signe mns)),
      Finite (non_zero_el_prod_maps_of_set_p pfe S).
intros E h1 S.
eapply Finite_downward_closed.
apply (non_zero_el_prod_maps_fin_p _ h1).
apply incl_non_zero_el_prod_maps_of_set_p.
Qed.


Definition non_zero_el_prod_compose_maps_of_set_p
           {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
           (pfe:Finite E) (g:btp Bp->Atp)
           (S:Ensemble (Fin_map E signe mns)) :=
  Intersection S (non_zero_el_prod_compose_maps_p g E).

Lemma non_zero_el_prod_compose_maps_of_set_p_eq : 
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (pfe:Finite E) (g:btp Bp->Atp)
         (S:Ensemble (Fin_map E signe mns)),
    non_zero_el_prod_compose_maps_of_set_p pfe g S = 
    non_zero_el_prod_compose_maps_of_set (ba_conv Ap) (E:=ba_conv_set E) pfe (ba_conv_fun g) (ba_conv_ens_fin_map_dom S).
intros Bp E h1 g S.
unfold non_zero_el_prod_compose_maps_of_set_p, non_zero_el_prod_compose_maps_of_set.
rewrite non_zero_el_prod_compose_maps_p_eq.
reflexivity.
Qed.



Lemma non_zero_el_prod_compose_maps_of_set_eq_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (pfe:Finite E) (g:btp Bp->Atp)
         (S:Ensemble (Fin_map E signe mns)),
    non_zero_el_prod_compose_maps_of_set_p pfe g S =
  [f:Fin_map E signe mns | Ensembles.In S f /\ el_prod_compose_p g f <> %0].
intros B E h1 g S.
apply Extensionality_Ensembles.
red. split.
red. intros x h2. destruct h2 as [f h2 h3].
destruct h3 as [h3].
constructor. split; auto.
red. intros x h2. destruct h2 as [h2]. destruct h2 as [h2a h2b].
constructor; auto. constructor. assumption.
Qed.


Lemma non_zero_el_prod_compose_maps_of_set_decompose_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (pfe:Finite E) (g:btp Bp->Atp)
         (S:Ensemble (Fin_map E signe mns)),
    S = Union (non_zero_el_prod_compose_maps_of_set_p pfe g S)
              [f:Fin_map E signe mns | Ensembles.In S f /\ el_prod_compose_p g f = %0].
intros B E h1 g S.
apply Extensionality_Ensembles.
red. split.
red. intros f h2.
destruct (classic (el_prod_compose_p g f = %0)) as [h3 | h4].
right. constructor. split; auto.
left. constructor; auto. constructor; auto.
red. intros f h2.
destruct h2 as [f h2 | f h3].
destruct h2; auto.
destruct h3 as [h3]. destruct h3; auto.
Qed.



Lemma incl_non_zero_el_prod_compose_maps_of_set_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
         (pfe:Finite E) (g:btp Bp->Atp)
         (S:Ensemble (Fin_map E signe mns)),
    Included (non_zero_el_prod_compose_maps_of_set_p pfe g S) (non_zero_el_prod_compose_maps_p g E).
intros B E h1 g S.
unfold non_zero_el_prod_compose_maps_of_set_p.
auto with sets.
Qed.


Lemma finite_non_zero_el_prod_compose_maps_of_set_p :
    forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)}
           (pfe:Finite E) (g:btp Bp->Atp)
           (S:Ensemble (Fin_map E signe mns)),
      Finite (non_zero_el_prod_compose_maps_of_set_p pfe g S).
intros B E h1 g S.
eapply Finite_downward_closed.
apply (non_zero_el_prod_compose_maps_fin_p g h1).
apply incl_non_zero_el_prod_compose_maps_of_set_p.
Qed.
 

Lemma plus_set_zero_el_prod_maps_of_set_p :
  forall {E:Ensemble Atp},
    Finite E ->
    forall (S:Ensemble (Fin_map E signe mns))
         (pf:Finite (Im [f : Fin_map E signe mns | Ensembles.In S f /\ el_prod_p f = %0]
             el_prod_p)),
    plus_set_p _ pf = %0.
intros E h1 S h0.
rewrite plus_set_p_eq.
pose proof h0 as h2'.
  assert (h3:
    Im [x : Fin_map E signe mns | Ensembles.In S x /\ el_prod_p x = %0] el_prod_p =
    Im [x : Fin_map E signe mns | Ensembles.In S x /\ el_prod (ba_conv Ap) x = 0]
        (el_prod (ba_conv Ap))).
    apply Extensionality_Ensembles.
    red. split.
    red. intros x h2.
    destruct h2 as [x h2]. subst.
    destruct h2 as [h2]. rewrite el_prod_p_eq in h2.
    apply Im_intro with x. constructor. assumption.
    rewrite el_prod_p_eq. reflexivity.
    red. intros x h2.
    destruct h2 as [x h2]. subst.
    destruct h2 as [h2]. apply Im_intro with x.
    constructor. rewrite el_prod_p_eq. assumption.
    rewrite el_prod_p_eq. reflexivity.
rewrite h3 in h2'.
pose proof (plus_set_zero_el_prod_maps_of_set (ba_conv Ap) h1 S h2') as h4.
rewrite ba_conv_zero. rewrite <- h4.
unfold ba_conv_elt, ba_conv_type. rewrite transfer_eq_refl.
apply (plus_set_functional (B:=ba_conv (Ap))).
unfold ba_conv_set, ba_conv_type. rewrite transfer_dep_eq_refl.
f_equal.
apply Extensionality_Ensembles.
red. split.
red. intros x h5. destruct h5 as [h5]. constructor.
rewrite el_prod_p_eq in h5. assumption.
red. intros x h5.
destruct h5 as [h5]. constructor.
rewrite el_prod_p_eq. rewrite ba_conv_zero.
assumption. 
apply functional_extensionality. intro x. rewrite el_prod_p_eq; auto.
Qed.

Lemma plus_subset_el_prod_compose_maps_eq_same_non_zero_p :
  forall {Bp:Bool_Alg_p T} {E:Ensemble (btp Bp)} (pfe:Finite E)
         (g:btp Bp->Atp) (S:Ensemble (Fin_map E signe mns)),
    plus_set_p (Im S (el_prod_compose_p g)) (finite_image _ _ _ (el_prod_compose_p g) (finite_fin_map_ens S pfe signe_finite)) =
    plus_set_p (Im (non_zero_el_prod_compose_maps_of_set_p pfe g S)
                 (el_prod_compose_p g))
            (finite_image _ _ _ (el_prod_compose_p g)
                           (finite_non_zero_el_prod_compose_maps_of_set_p pfe g S)).
intros Bp E h1 g S.
rewrite plus_set_p_eq. rewrite plus_set_p_eq.
f_equal.
pose proof (plus_subset_el_prod_compose_maps_eq_same_non_zero 
              (ba_conv Ap) (B:=(ba_conv Bp)) (E:=ba_conv_set E) h1 (ba_conv_fun g)
              (ba_conv_ens_fin_map_dom S)) as h2.
assert (h4:Im S (el_prod_compose_p g) =  
           Im (ba_conv_ens_fin_map_dom S)
            (el_prod_compose (ba_conv Ap) (ba_conv_fun g))).
 f_equal. apply functional_extensionality.
 intro x. rewrite el_prod_compose_p_eq. reflexivity. 
generalize  (finite_image (Fin_map E signe mns) Atp S (el_prod_compose_p g) (finite_fin_map_ens S h1 signe_finite)).
rewrite h4.
intro h5.
assert (h5 = finite_image (Fin_map (ba_conv_set E) signe mns)
            (Btype (Bc (ba_conv Ap))) (ba_conv_ens_fin_map_dom S)
            (el_prod_compose (ba_conv Ap) (ba_conv_fun g))
            (finite_fin_map_ens (ba_conv_ens_fin_map_dom S) h1 signe_finite)). apply proof_irrelevance.
subst.
rewrite h2 at 1.
apply plus_set_functional.
unfold ba_conv_set, ba_conv_type. rewrite transfer_dep_eq_refl.
f_equal.
rewrite non_zero_el_prod_compose_maps_of_set_p_eq.
reflexivity.
apply functional_extensionality.
intro f. rewrite el_prod_compose_p_eq.
reflexivity.
Qed.

Lemma plus_subset_non_zero_el_prod_compose_maps_empty_p :
  forall {Bp:Bool_Alg_p T} (E:Ensemble (btp Bp)) (pfe:Finite E)
         (g:(btp Bp)->Atp),
    plus_subset_non_zero_el_prod_compose_maps_p E pfe g _ (empty_inclusion _) = %0.
intros. unfold plus_subset_non_zero_el_prod_compose_maps_p.
generalize ((finite_image (Fin_map E signe mns) Atp (Empty_set (Fin_map E signe mns))
        (el_prod_compose_p g)
        (finite_fin_map_ens (Empty_set (Fin_map E signe mns)) pfe
                            signe_finite))).
pose proof (image_empty (Fin_map E signe mns) _ (el_prod_compose_p g)) as h1.
rewrite h1.
intro. apply plus_set_empty_p'.
Qed.



Lemma plus_subset_non_zero_el_prod_compose_maps_full_p :
  forall {Bp:Bool_Alg_p T} (E:Ensemble (btp Bp)) (pfe:Finite E)
         (g:(btp Bp)->Atp),
    plus_subset_non_zero_el_prod_compose_maps_p E pfe g _ (inclusion_reflexive _) = %1.
intros Bp E h1 g.
pose proof (plus_subset_non_zero_el_prod_compose_maps_full (ba_conv Ap) (ba_conv_set E) h1 (ba_conv_fun g)) as h2.
rewrite ba_conv_one.
rewrite <- h2.
rewrite plus_subset_non_zero_el_prod_compose_maps_p_eq.
generalize (eq_rect (ba_conv_ens_fin_map_dom (non_zero_el_prod_compose_maps_p g E))
        (Included
           (ba_conv_ens_fin_map_dom (non_zero_el_prod_compose_maps_p g E)))
        (iff1
           (incl_ba_conv_ens_fin_map_dom_iff
              (non_zero_el_prod_compose_maps_p g E)
              (non_zero_el_prod_compose_maps_p g E))
           (inclusion_reflexive (non_zero_el_prod_compose_maps_p g E)))
        (non_zero_el_prod_compose_maps (ba_conv Ap) 
           (ba_conv_fun g) (ba_conv_set (ba_conv_set E)))
        (non_zero_el_prod_compose_maps_p_eq g (ba_conv_set E))).
rewrite non_zero_el_prod_compose_maps_p_eq.
intro h3.
f_equal.
apply proof_irrelevance.
Qed.



Lemma plus_subset_non_zero_el_prod_compose_maps_int_p :
  forall {Bp:Bool_Alg_p T} (E:Ensemble (btp Bp)) (pfe:Finite E)
         (g:(btp Bp)->Atp) (X Y:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_compose_maps_p g E))
         (pfy:Included Y (non_zero_el_prod_compose_maps_p g E)),
    plus_subset_non_zero_el_prod_compose_maps_p E pfe g (Intersection X Y)
    (inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_int_p pfe g X Y pfx pfy) =
    plus_subset_non_zero_el_prod_compose_maps_p E pfe g X pfx %*
    plus_subset_non_zero_el_prod_compose_maps_p E pfe g Y pfy.
intros Bp E h1 g X Y h2 h3.
pose proof h2 as h2'. pose proof h3 as h3'.
apply (iff1 (incl_ba_conv_ens_fin_map_dom_iff  X (non_zero_el_prod_compose_maps_p g E))) in h2'.
apply (iff1 (incl_ba_conv_ens_fin_map_dom_iff Y (non_zero_el_prod_compose_maps_p g E))) in h3'.
rewrite non_zero_el_prod_compose_maps_p_eq in h2', h3'.
pose proof (plus_subset_non_zero_el_prod_compose_maps_int (ba_conv Ap) (ba_conv_set E) h1 (ba_conv_fun g) _ _ h2' h3') as h4.
do 2 rewrite plus_subset_non_zero_el_prod_compose_maps_p_eq.
generalize  (eq_rect (ba_conv_ens_fin_map_dom (non_zero_el_prod_compose_maps_p g E))
        (Included (ba_conv_ens_fin_map_dom (Intersection X Y)))
        (iff1
           (incl_ba_conv_ens_fin_map_dom_iff (Intersection X Y)
              (non_zero_el_prod_compose_maps_p g E))
           (inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_int_p
              h1 g X Y h2 h3))
        (non_zero_el_prod_compose_maps (ba_conv Ap) 
           (ba_conv_fun g) (ba_conv_set (ba_conv_set E)))
        (non_zero_el_prod_compose_maps_p_eq g (ba_conv_set E))).
intro h5. 
assert (h5= inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_int
            (ba_conv Ap) h1 (ba_conv_fun g) (ba_conv_ens_fin_map_dom X)
            (ba_conv_ens_fin_map_dom Y) h2' h3' (B:=ba_conv Bp)).
  apply proof_irrelevance. subst.
rewrite h4 at 1.
rewrite ba_conv_times.
f_equal.
rewrite plus_subset_non_zero_el_prod_compose_maps_p_eq.
reflexivity.
Qed.  

Lemma plus_subset_non_zero_el_prod_compose_maps_union_p :
  forall {Bp:Bool_Alg_p T} (E:Ensemble (btp Bp)) (pfe:Finite E)
         (g:(btp Bp)->Atp) (X Y:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_compose_maps_p g E))
         (pfy:Included Y (non_zero_el_prod_compose_maps_p g E)),
    plus_subset_non_zero_el_prod_compose_maps_p E pfe g (Union X Y)
    (inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_union_p pfe g X Y pfx pfy) =
    plus_subset_non_zero_el_prod_compose_maps_p E pfe g X pfx %+
    plus_subset_non_zero_el_prod_compose_maps_p E pfe g Y pfy.
intros Bp E h1 g X Y h2 h3.
pose proof h2 as h2'. pose proof h3 as h3'.
apply (iff1 (incl_ba_conv_ens_fin_map_dom_iff  X (non_zero_el_prod_compose_maps_p g E))) in h2'.
apply (iff1 (incl_ba_conv_ens_fin_map_dom_iff Y (non_zero_el_prod_compose_maps_p g E))) in h3'.
rewrite non_zero_el_prod_compose_maps_p_eq in h2', h3'.
pose proof (plus_subset_non_zero_el_prod_compose_maps_union (ba_conv Ap) (ba_conv_set E) h1 (ba_conv_fun g) _ _ h2' h3') as h4.
do 2 rewrite plus_subset_non_zero_el_prod_compose_maps_p_eq.
generalize  (eq_rect (ba_conv_ens_fin_map_dom (non_zero_el_prod_compose_maps_p g E))
        (Included (ba_conv_ens_fin_map_dom (Union X Y)))
        (iff1
           (incl_ba_conv_ens_fin_map_dom_iff (Union X Y)
              (non_zero_el_prod_compose_maps_p g E))
           (inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_union_p
              h1 g X Y h2 h3))
        (non_zero_el_prod_compose_maps (ba_conv Ap) 
           (ba_conv_fun g) (ba_conv_set (ba_conv_set E)))
        (non_zero_el_prod_compose_maps_p_eq g (ba_conv_set E))).
intro h5. 
assert (h5= inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_union
            (ba_conv Ap) h1 (ba_conv_fun g) (ba_conv_ens_fin_map_dom X)
            (ba_conv_ens_fin_map_dom Y) h2' h3' (B:=ba_conv Bp)).
  apply proof_irrelevance. subst.
rewrite h4 at 1.
rewrite ba_conv_plus.
f_equal.
rewrite plus_subset_non_zero_el_prod_compose_maps_p_eq.
reflexivity.
Qed.  

Lemma plus_subset_non_zero_el_prod_compose_maps_comp_p :
  forall {Bp:Bool_Alg_p T} (E:Ensemble (btp Bp)) (pfe:Finite E)
         (g:(btp Bp)->Atp) (X:Ensemble (Fin_map E signe mns))
         (pfx:Included X (non_zero_el_prod_compose_maps_p g E)),
     plus_subset_non_zero_el_prod_compose_maps_p 
       E pfe g 
       (Setminus (non_zero_el_prod_compose_maps_p g E) X)
       (inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_comp_p pfe g X pfx) = 
     %-plus_subset_non_zero_el_prod_compose_maps_p E pfe g X pfx.
intros Bp E h1 g X h2.
pose proof h2 as h2'.
apply (iff1 (incl_ba_conv_ens_fin_map_dom_iff  X (non_zero_el_prod_compose_maps_p g E))) in h2'.
rewrite non_zero_el_prod_compose_maps_p_eq in h2'.
pose proof (plus_subset_non_zero_el_prod_compose_maps_comp (ba_conv Ap) (ba_conv_set E) h1 (ba_conv_fun g) _ h2') as h4.
do 2 rewrite plus_subset_non_zero_el_prod_compose_maps_p_eq.
generalize  (eq_rect (ba_conv_ens_fin_map_dom (non_zero_el_prod_compose_maps_p g E))
        (Included
           (ba_conv_ens_fin_map_dom
              (Setminus (non_zero_el_prod_compose_maps_p g E) X)))
        (iff1
           (incl_ba_conv_ens_fin_map_dom_iff
              (Setminus (non_zero_el_prod_compose_maps_p g E) X)
              (non_zero_el_prod_compose_maps_p g E))
           (inclusion_plus_subset_non_zero_el_prod_compose_maps_preserves_comp_p
              h1 g X h2))
        (non_zero_el_prod_compose_maps (ba_conv Ap)
           (ba_conv_fun g) (ba_conv_set (ba_conv_set E)))
        (non_zero_el_prod_compose_maps_p_eq g (ba_conv_set E))).
intro h5.
pose proof (non_zero_el_prod_compose_maps_p_eq g E) as h6.
simpl in h6, h5.
pose proof h5 as h5'. rewrite h6 in h5'.
pose proof (f_equal (fun S =>
         (ba_conv_ens_fin_map_dom
            (Setminus S X)))
                    h6) as h7. simpl in h7.
pose proof (subsetT_eq_compat
              _ (fun S=>Included S
 (non_zero_el_prod_compose_maps (ba_conv Ap) 
            (ba_conv_fun g) (ba_conv_set (ba_conv_set E))))
              _ _ h5 h5' h7) as h8.
dependent rewrite -> h8.
rewrite h4 at 1.
rewrite ba_conv_comp.
f_equal.
Qed.

Lemma incl_im_proj1_sig_ex_ens_fin_map_plus_subset_non_zero_el_prod_maps_p : 
  forall (E:Ensemble Atp) (pfe:Finite E),
    Included
      (im_proj1_sig [a : Atp
                    | exists
                        (S : Ensemble (Fin_map E signe mns)) (pfi : 
                                                                Included S
                                                                         (non_zero_el_prod_maps_p E)),
                      a = plus_subset_non_zero_el_prod_maps_p E pfe S pfi])
      (A_p T (Bc_p T Ap)).
  intros E h0. red. intros x h2. destruct h2 as [x h2]. subst.
  apply proj2_sig.
Qed.



Lemma closed_all_plus_subsets_non_zero_el_prod_maps_p :
  forall (E:Ensemble Atp) (pfe:Finite E),
    alg_closed_p 
                 (im_proj1_sig [a:Atp | exists (S:Ensemble (Fin_map E signe mns))
                            (pfi:Included S (non_zero_el_prod_maps_p E)),
                     a = plus_subset_non_zero_el_prod_maps_p E pfe S pfi]) (incl_im_proj1_sig_ex_ens_fin_map_plus_subset_non_zero_el_prod_maps_p _ pfe).
intros E h1.
assert (h3:  [a : Atp
            | exists
                (S : Ensemble (Fin_map E signe mns)) 
              (pfi : Included S (non_zero_el_prod_maps_p E)),
                a = plus_subset_non_zero_el_prod_maps_p E h1 S pfi] =
              [a : (bt (ba_conv Ap))
            | exists
                (S : Ensemble (Fin_map E signe mns)) 
              (pfi : Included S (non_zero_el_prod_maps (ba_conv Ap) E)),
                a = plus_subset_non_zero_el_prod_maps (ba_conv Ap) E h1 S pfi]).
  apply Extensionality_Ensembles. red. split.
  red. intros x h3. destruct h3 as [h3]. destruct h3 as [S [h3 h4]].
  subst.
  constructor. 
  pose proof h3 as h3'. rewrite non_zero_el_prod_maps_p_eq in h3'.
  exists S, h3'. rewrite plus_subset_non_zero_el_prod_maps_p_eq.
  f_equal. apply proof_irrelevance.
  red. intros x h3. destruct h3 as [h3]. destruct h3 as [S [h3 h4]].
  subst. constructor.
  pose proof h3 as h3'. 
  rewrite <- (non_zero_el_prod_maps_p_eq E) in h3' at 1.
  exists S, h3'. rewrite plus_subset_non_zero_el_prod_maps_p_eq.
  f_equal. apply proof_irrelevance.
rewrite alg_closed_p_iff.
pose proof (f_equal (fun S => (im_proj1_sig S)) h3) as h4. simpl in h4.
assert (h2':Included 
               (im_proj1_sig
         [a : bt (ba_conv Ap)
         | exists
             (S : Ensemble (Fin_map E signe mns)) (pfi : 
                                                  Included S
                                                  (non_zero_el_prod_maps
                                                  (ba_conv Ap) E)),
             a = plus_subset_non_zero_el_prod_maps (ba_conv Ap) E h1 S pfi]) (A_p T (Bc_p T Ap))).
  rewrite <- h4. apply  (incl_im_proj1_sig_ex_ens_fin_map_plus_subset_non_zero_el_prod_maps_p E
         h1).
pose proof (subsetT_eq_compat _ (fun S => Included S (A_p T (Bc_p T Ap))) _ _  (incl_im_proj1_sig_ex_ens_fin_map_plus_subset_non_zero_el_prod_maps_p E
         h1) h2' h4) as h5.
dependent rewrite -> h5. 
rewrite ba_conv_und_subalg_im_proj1_sig.
apply closed_all_plus_subsets_non_zero_el_prod_maps.
Qed.


Lemma el_prod_sing_inc_p :
  forall (E:Ensemble Atp) (pf:Finite E) (a:Fin_map E signe mns),
    el_prod_p a <> %0 -> Included (Singleton a) (non_zero_el_prod_maps_p E).
intros E h1 a h2.
red.
intros a' h3.
destruct h3; subst.
unfold non_zero_el_prod_maps_p. constructor.
assumption.
Qed.

Lemma el_prod_sing_p : 
  forall (E:Ensemble Atp) (pf:Finite E) (a:Fin_map E signe mns)
         (pfel:el_prod_p a <> %0), 
    el_prod_p a = plus_subset_non_zero_el_prod_maps_p _ pf (Singleton a) (el_prod_sing_inc_p _ pf _ pfel).
intros E h1 a h2.
unfold plus_subset_non_zero_el_prod_maps_p.
assert (h3:Im (Singleton a) el_prod_p = (Singleton (el_prod_p a))).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x' h13.
  inversion h13. subst. destruct H. subst. constructor.
  red. intros x' h13. destruct h13.
  apply Im_intro with a. constructor.
  reflexivity.
generalize dependent (
     (finite_image (Fin_map E signe mns) Atp (Singleton a) el_prod_p
        (Finite_downward_closed (Fin_map E signe mns)
           (non_zero_el_prod_maps_p E) (non_zero_el_prod_maps_fin_p E h1)
           (Singleton a) (el_prod_sing_inc_p E h1 a h2)))).
rewrite h3.
intro h4.
rewrite plus_set_sing_p'.
reflexivity.
Qed.



End NormalForm_p.



End ParametricAnalogues.

Arguments alg_closed_p [T] [Bp] _ _.
Arguments Subalg_p [T] _ _ _ _.

Section DirectedFamilyOfAlgebras.
Variable T:Type.

Definition directed (S:fam_ba_p T) :=
  forall Ap Bp:Bool_Alg_p T,
    Ensembles.In S Ap ->
    Ensembles.In S Bp ->
    exists Cp:Bool_Alg_p T,
      Ensembles.In S Cp /\ 
      subalg_of_p Ap Cp /\
      subalg_of_p Bp Cp.
 

Lemma directed_times_ex :
  forall {S:fam_ba_p T},
    directed S ->
    forall (a b:sig_set (FamilyUnion (Im S (@ba_p_ens T)))),
      exists! c:sig_set (FamilyUnion (Im S (@ba_p_ens T))),
        forall (Ap Bp:Bool_Alg_p T)
               (pfina:Ensembles.In (ba_p_ens Ap) (proj1_sig a))
               (pfinb:Ensembles.In (ba_p_ens Bp) (proj1_sig b)),
          Ensembles.In S Ap ->
          Ensembles.In S Bp ->
          exists (Cp:Bool_Alg_p T)
                 (pfinca:Included (ba_p_ens Ap) (ba_p_ens Cp))
                 (pfincb:Included (ba_p_ens Bp) (ba_p_ens Cp))
                 (pfaca:alg_closed_p _  pfinca)
                 (pfacb:alg_closed_p _ pfincb),
            Ensembles.In (ba_p_ens Cp) (proj1_sig c) /\
            Ap = Subalg_p _ _ pfinca pfaca /\
            Bp = Subalg_p _ _ pfincb pfacb /\
            Ensembles.In S Cp /\
            proj1_sig c =
            proj1_sig (exist _ _ (pfinca _ pfina) %* 
                             exist _ _ (pfincb _ pfinb)).
intros S h1 a b.
destruct a as [a h2], b as [b h3].  destruct h2 as [A a h2 h4], h3 as [B b h3 h5].
destruct h2 as [Ap h2], h3 as [Bp h3]. subst. simpl.
red in h1.
pose proof (h1 _ _ h2 h3) as h6. destruct h6 as [Cp [h6 [h7 h8]]].
red in h7, h8. destruct h7 as [h7 [h9 h10]], h8 as [h8 [h11 h12]].
pose (exist _ _ (h7 _ h4) %* exist _ _ (h8 _ h5)) as c'.
assert (h13:Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T))))
                         (proj1_sig c')).
  unfold c'.  apply family_union_intro with (ba_p_ens Cp).
  apply Im_intro with Cp. assumption. reflexivity.
  apply proj2_sig. 
exists (exist _ _ h13).  
red. split. 
intros A B h14 h15 h16 h17.
pose proof (h1 _ _ h6 h16) as h18. pose proof (h1 _ _ h6 h17) as h19. 
destruct h18 as [C [h18a [h18b h18c]]], h19 as [D [h19a [h19b h19c]]]. 
pose proof (h1 _ _ h18a h19a) as h20.
destruct h20 as [Ep [h20a [h20b h20c]]]. 
pose proof h18b as h18b'. pose proof h19b as h19b'.
pose proof h18c as h18c'. pose proof h19c as h19c'.  
pose proof h20b as h20b'. pose proof h20c as h20c'.
red in h18c, h19c, h20b, h20c. 
destruct h18c as [h21a [h21b h21c]], 
                 h20b as [h23a [h23b h23c]],
                         h19c as [h22a [h22b h22c]],
                                 h20c as [h24a [h24b h24c]].
exists Ep. 
pose proof (Inclusion_is_transitive _ _ _  _ h21a h23a) as h25.
pose proof (Inclusion_is_transitive _ _ _ _ h22a h24a) as h26.
pose proof (trans_subalg_of_p _ _ _ _ h18c' h20b') as h27.
pose proof (trans_subalg_of_p _ _ _ _ h19c' h20c') as h28.
destruct h27 as [h27a [h27b h27c]], h28 as [h28a [h28b h28c]].
exists h27a, h28a, h27b, h28b.
simpl. repeat split.
apply h24a. destruct h19b as [h29a [h29b h29c]]. apply h29a.
apply proj2_sig. assumption. assumption. assumption.
unfold c'.
red in h19b. destruct h19b as [h29 [h30 h31]].
pose proof (ba_p_subst_times _ _ h31 _ _ (h7 a h4) (h8 b h5)) as h32.
assert (h33:Ensembles.In (ba_p_ens (Subalg_p D (ba_p_ens Cp) h29 h30))
                   a).
  rewrite <- h31. apply (h7 a h4).
assert (h34 : Ensembles.In 
                (ba_p_ens (Subalg_p D (ba_p_ens Cp) h29 h30)) b).
  rewrite <- h31.  apply (h8 b h5).
specialize (h32 h33 h34).
rewrite h32 at 1.
simpl.
pose proof (ba_p_subst_times _ _ h24c _ _ (h29 a h33) (h29 b h34)) as h35.
assert (h36:Ensembles.In
              (ba_p_ens (Subalg_p Ep (ba_p_ens D) h24a h24b)) a).
  rewrite <- h24c. apply (h29 a h33).
assert (h37 : Ensembles.In
                (ba_p_ens (Subalg_p Ep (ba_p_ens D) h24a h24b)) b).
  rewrite <- h24c. apply (h29 b h34).
specialize (h35 h36 h37).
rewrite h35 at 1.
simpl.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
intros c h14.
specialize (h14 _ _ h4 h5 h2 h3).
destruct h14 as [Dp [h15 [h16 [h17 [h18 [h19 [h20 [h21 [h22 h23]]]]]]]]].
apply proj1_sig_injective. simpl.
rewrite h23 at 1.
unfold c'.
pose proof (h1 _ _ h6 h22) as h24.
destruct h24 as [D [h24 [h25 h26]]].
red in h25, h26. destruct h25 as [h25a [h25b h25c]], h26 as [h26a [h26b h26c]].
pose proof (ba_p_subst_times _ _ h25c _ _ (h7 a h4) (h8 b h5)) as h27.
assert (h28: Ensembles.In
               (ba_p_ens (Subalg_p D (ba_p_ens Cp) h25a h25b)) a).
  rewrite <- h25c. apply (h7 a h4).
assert (h29 : Ensembles.In
                (ba_p_ens (Subalg_p D (ba_p_ens Cp) h25a h25b)) b). 
  rewrite <- h25c. apply (h8 b h5).
specialize (h27 h28 h29).
rewrite h27 at 1. simpl.
pose proof (ba_p_subst_times _ _ h26c _ _ (h15 a h4) (h16 b h5)) as h30.
assert (h31:Ensembles.In
              (ba_p_ens (Subalg_p D (ba_p_ens Dp) h26a h26b)) a).
  rewrite <- h26c. apply (h15 a h4).
assert (h32 : Ensembles.In
                (ba_p_ens (Subalg_p D (ba_p_ens Dp) h26a h26b)) b).
  rewrite <- h26c. apply (h16 b h5).
specialize (h30 h31 h32).
rewrite h30 at 1. simpl.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
Qed.


Definition directed_times {S:fam_ba_p T}
           (pf:directed S) :
  sig_set (FamilyUnion (Im S (@ba_p_ens T))) ->
  sig_set (FamilyUnion (Im S (@ba_p_ens T))) ->
  sig_set (FamilyUnion (Im S (@ba_p_ens T))) :=
  (fun a b =>
     proj1_sig (constructive_definite_description _
                                                  (directed_times_ex pf a b))).

Lemma directed_times_compat :
  forall {S:fam_ba_p T}
         (pf:directed S)
         (a b:sig_set (FamilyUnion (Im S (@ba_p_ens T)))),
      let c:=directed_times pf a b in
      forall (Ap Bp:Bool_Alg_p T)
             (pfina:Ensembles.In (ba_p_ens Ap) (proj1_sig a))
             (pfinb:Ensembles.In (ba_p_ens Bp) (proj1_sig b)),
        Ensembles.In S Ap ->
        Ensembles.In S Bp ->
        exists (Cp:Bool_Alg_p T)
               (pfinca:Included (ba_p_ens Ap) (ba_p_ens Cp))
               (pfincb:Included (ba_p_ens Bp) (ba_p_ens Cp))
               (pfaca:alg_closed_p _  pfinca)
               (pfacb:alg_closed_p _ pfincb),
          Ensembles.In (ba_p_ens Cp) (proj1_sig c) /\
          Ap = Subalg_p _ _ pfinca pfaca /\
          Bp = Subalg_p _ _ pfincb pfacb /\
          Ensembles.In S Cp /\
          proj1_sig c =
          proj1_sig (exist _ _ (pfinca _ pfina) %* 
                             exist _ _ (pfincb _ pfinb)).
intros S pf a b c.  unfold directed_times in c.
destruct constructive_definite_description as [c' h2].
simpl in c. simpl.
assumption.
Qed.




Lemma directed_plus_ex :
  forall {S:fam_ba_p T},
    directed S ->
    forall (a b:sig_set (FamilyUnion (Im S (@ba_p_ens T)))),
      exists! c:sig_set (FamilyUnion (Im S (@ba_p_ens T))),
        forall (Ap Bp:Bool_Alg_p T)
               (pfina:Ensembles.In (ba_p_ens Ap) (proj1_sig a))
               (pfinb:Ensembles.In (ba_p_ens Bp) (proj1_sig b)),
          Ensembles.In S Ap ->
          Ensembles.In S Bp ->
          exists (Cp:Bool_Alg_p T)
                 (pfinca:Included (ba_p_ens Ap) (ba_p_ens Cp))
                 (pfincb:Included (ba_p_ens Bp) (ba_p_ens Cp))
                 (pfaca:alg_closed_p _  pfinca)
                 (pfacb:alg_closed_p _ pfincb),
            Ensembles.In (ba_p_ens Cp) (proj1_sig c) /\
            Ap = Subalg_p _ _ pfinca pfaca /\
            Bp = Subalg_p _ _ pfincb pfacb /\
            Ensembles.In S Cp /\
            proj1_sig c =
            proj1_sig (exist _ _ (pfinca _ pfina) %+ 
                             exist _ _ (pfincb _ pfinb)).
intros S h1 a b.
destruct a as [a h2], b as [b h3].  destruct h2 as [A a h2 h4], h3 as [B b h3 h5].
destruct h2 as [Ap h2], h3 as [Bp h3]. subst. simpl.
red in h1.
pose proof (h1 _ _ h2 h3) as h6. destruct h6 as [Cp [h6 [h7 h8]]].
red in h7, h8. destruct h7 as [h7 [h9 h10]], h8 as [h8 [h11 h12]].
pose (exist _ _ (h7 _ h4) %+ exist _ _ (h8 _ h5)) as c'.
assert (h13:Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T))))
                         (proj1_sig c')).
  unfold c'.  apply family_union_intro with (ba_p_ens Cp).
  apply Im_intro with Cp. assumption. reflexivity.
  apply proj2_sig. 
exists (exist _ _ h13).  
red. split. 
intros A B h14 h15 h16 h17.
pose proof (h1 _ _ h6 h16) as h18. pose proof (h1 _ _ h6 h17) as h19. 
destruct h18 as [C [h18a [h18b h18c]]], h19 as [D [h19a [h19b h19c]]]. 
pose proof (h1 _ _ h18a h19a) as h20.
destruct h20 as [Ep [h20a [h20b h20c]]]. 
pose proof h18b as h18b'. pose proof h19b as h19b'.
pose proof h18c as h18c'. pose proof h19c as h19c'.  
pose proof h20b as h20b'. pose proof h20c as h20c'.
red in h18c, h19c, h20b, h20c. 
destruct h18c as [h21a [h21b h21c]], 
                 h20b as [h23a [h23b h23c]],
                         h19c as [h22a [h22b h22c]],
                                 h20c as [h24a [h24b h24c]].
exists Ep. 
pose proof (Inclusion_is_transitive _ _ _  _ h21a h23a) as h25.
pose proof (Inclusion_is_transitive _ _ _ _ h22a h24a) as h26.
pose proof (trans_subalg_of_p _ _ _ _ h18c' h20b') as h27.
pose proof (trans_subalg_of_p _ _ _ _ h19c' h20c') as h28.
destruct h27 as [h27a [h27b h27c]], h28 as [h28a [h28b h28c]].
exists h27a, h28a, h27b, h28b.
simpl. repeat split.
apply h24a. destruct h19b as [h29a [h29b h29c]]. apply h29a.
apply proj2_sig. assumption. assumption. assumption.
unfold c'.
red in h19b. destruct h19b as [h29 [h30 h31]].
pose proof (ba_p_subst_plus _ _ h31 _ _ (h7 a h4) (h8 b h5)) as h32.
assert (h33:Ensembles.In (ba_p_ens (Subalg_p D (ba_p_ens Cp) h29 h30))
                   a).
  rewrite <- h31. apply (h7 a h4).
assert (h34 : Ensembles.In 
                (ba_p_ens (Subalg_p D (ba_p_ens Cp) h29 h30)) b).
  rewrite <- h31.  apply (h8 b h5).
specialize (h32 h33 h34).
rewrite h32 at 1.
simpl.
pose proof (ba_p_subst_plus _ _ h24c _ _ (h29 a h33) (h29 b h34)) as h35.
assert (h36:Ensembles.In
              (ba_p_ens (Subalg_p Ep (ba_p_ens D) h24a h24b)) a).
  rewrite <- h24c. apply (h29 a h33).
assert (h37 : Ensembles.In
                (ba_p_ens (Subalg_p Ep (ba_p_ens D) h24a h24b)) b).
  rewrite <- h24c. apply (h29 b h34).
specialize (h35 h36 h37).
rewrite h35 at 1.
simpl.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
intros c h14.
specialize (h14 _ _ h4 h5 h2 h3).
destruct h14 as [Dp [h15 [h16 [h17 [h18 [h19 [h20 [h21 [h22 h23]]]]]]]]].
apply proj1_sig_injective. simpl.
rewrite h23 at 1.
unfold c'.
pose proof (h1 _ _ h6 h22) as h24.
destruct h24 as [D [h24 [h25 h26]]].
red in h25, h26. destruct h25 as [h25a [h25b h25c]], h26 as [h26a [h26b h26c]].
pose proof (ba_p_subst_plus _ _ h25c _ _ (h7 a h4) (h8 b h5)) as h27.
assert (h28: Ensembles.In
               (ba_p_ens (Subalg_p D (ba_p_ens Cp) h25a h25b)) a).
  rewrite <- h25c. apply (h7 a h4).
assert (h29 : Ensembles.In
                (ba_p_ens (Subalg_p D (ba_p_ens Cp) h25a h25b)) b). 
  rewrite <- h25c. apply (h8 b h5).
specialize (h27 h28 h29).
rewrite h27 at 1. simpl.
pose proof (ba_p_subst_plus _ _ h26c _ _ (h15 a h4) (h16 b h5)) as h30.
assert (h31:Ensembles.In
              (ba_p_ens (Subalg_p D (ba_p_ens Dp) h26a h26b)) a).
  rewrite <- h26c. apply (h15 a h4).
assert (h32 : Ensembles.In
                (ba_p_ens (Subalg_p D (ba_p_ens Dp) h26a h26b)) b).
  rewrite <- h26c. apply (h16 b h5).
specialize (h30 h31 h32).
rewrite h30 at 1. simpl.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
Qed.



Definition directed_plus {S:fam_ba_p T}
           (pf:directed S) :
  sig_set (FamilyUnion (Im S (@ba_p_ens T))) ->
  sig_set (FamilyUnion (Im S (@ba_p_ens T))) ->
  sig_set (FamilyUnion (Im S (@ba_p_ens T))) :=
  (fun a b =>
     proj1_sig (constructive_definite_description _
                                                  (directed_plus_ex pf a b))).

Lemma directed_plus_compat :
  forall {S:fam_ba_p T}
         (pf:directed S)
         (a b:sig_set (FamilyUnion (Im S (@ba_p_ens T)))),
      let c:=directed_plus pf a b in
      forall (Ap Bp:Bool_Alg_p T)
             (pfina:Ensembles.In (ba_p_ens Ap) (proj1_sig a))
             (pfinb:Ensembles.In (ba_p_ens Bp) (proj1_sig b)),
        Ensembles.In S Ap ->
        Ensembles.In S Bp ->
        exists (Cp:Bool_Alg_p T)
               (pfinca:Included (ba_p_ens Ap) (ba_p_ens Cp))
               (pfincb:Included (ba_p_ens Bp) (ba_p_ens Cp))
               (pfaca:alg_closed_p _  pfinca)
               (pfacb:alg_closed_p _ pfincb),
          Ensembles.In (ba_p_ens Cp) (proj1_sig c) /\
          Ap = Subalg_p _ _ pfinca pfaca /\
          Bp = Subalg_p _ _ pfincb pfacb /\
          Ensembles.In S Cp /\
          proj1_sig c =
          proj1_sig (exist _ _ (pfinca _ pfina) %+ 
                             exist _ _ (pfincb _ pfinb)).
intros S pf a b c.  unfold directed_plus in c.
destruct constructive_definite_description as [c' h2].
simpl in c. simpl.
assumption.
Qed.




Lemma directed_comp_ex :
  forall {S:fam_ba_p T},
    directed S ->
    forall a:sig_set (FamilyUnion (Im S (@ba_p_ens T))),
      exists! c:sig_set (FamilyUnion (Im S (@ba_p_ens T))),
        forall (Ap:Bool_Alg_p T)
               (pfina:Ensembles.In (ba_p_ens Ap) (proj1_sig a)),
          Ensembles.In S Ap ->
          exists (Cp:Bool_Alg_p T),
            Ensembles.In (ba_p_ens Cp) (proj1_sig c) /\
            subalg_of_p Ap Cp /\
            Ensembles.In S Cp /\
            proj1_sig c =
            proj1_sig (%- exist _ _ pfina).
intros S h1 a.
destruct a as [a h2]. 
destruct h2 as [A a h2 h4].
destruct h2 as [A h2]. subst. 
pose (%- exist _ _ h4) as c'.
assert (h12:Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) (proj1_sig c')).
  apply family_union_intro with (ba_p_ens A). apply Im_intro with A.
  assumption. reflexivity. apply in_ba_p_ens_comp. 
inversion h12 as [C ? hc hd]. subst.
destruct hc as [Cp he]. subst.
exists (exist _ _ h12). simpl. 
red. split.
intros Ap h0 h0'. simpl. 
pose proof (h1 _ _ h0' he) as hf.
destruct hf as [Dp [hg [hh hi]]].
exists Dp. repeat split; auto. 
red in hi. destruct hi as [hi [hj hk]]. apply hi; auto.
unfold c'.
pose proof (h1 _ _ h2 h0') as hj. destruct hj as [Ep [hk [hl hm]]].
red in hl, hm. destruct hl as [hla [hlb hlc]], hm as [hma [hmb hmc]].
pose proof (ba_p_subst_comp _ _ hlc _ h4) as hn.
assert (ho:Ensembles.In (ba_p_ens (Subalg_p Ep (ba_p_ens A) hla hlb)) a).
  rewrite <- hlc. auto.
specialize (hn ho).
pose proof (ba_p_subst_comp _ _ hmc _ h0) as hp.
assert (hq: Ensembles.In (ba_p_ens (Subalg_p Ep (ba_p_ens Ap) hma hmb)) a).
  rewrite <- hmc. auto.
specialize (hp hq).
rewrite hn, hp.
simpl. f_equal. f_equal. apply proj1_sig_injective. simpl.
reflexivity.
intros c h13.
specialize (h13 _ h4 h2).
destruct h13 as [Dp [h14 [h15 [h16 h17]]]]. 
apply proj1_sig_injective.  simpl. 
rewrite h17 at 1.
unfold c'.
reflexivity.
Qed.


Definition directed_comp {S:fam_ba_p T}
           (pf:directed S) :
  sig_set (FamilyUnion (Im S (@ba_p_ens T))) ->
  sig_set (FamilyUnion (Im S (@ba_p_ens T))) :=
  (fun a =>
     proj1_sig (constructive_definite_description _
                                                  (directed_comp_ex pf a))).


Lemma directed_comp_compat :
  forall {S:fam_ba_p T}
         (pf:directed S),
  forall a:sig_set (FamilyUnion (Im S (@ba_p_ens T))),
    let c := directed_comp pf a in 
    forall (Ap:Bool_Alg_p T)
           (pfina:Ensembles.In (ba_p_ens Ap) (proj1_sig a)),
      Ensembles.In S Ap ->
      exists (Cp:Bool_Alg_p T),
        Ensembles.In (ba_p_ens Cp) (proj1_sig c) /\
        subalg_of_p Ap Cp /\
        Ensembles.In S Cp /\
            proj1_sig c =
            proj1_sig (%- exist _ _ pfina).
intros. unfold directed_comp in c.
destruct constructive_definite_description as [c' h1].
simpl in c.
unfold c.
apply h1; auto.
Qed.



Lemma directed_zero_ex :
  forall {S:fam_ba_p T},
    directed S ->
    Inhabited S ->
    exists! zr:sig_set (FamilyUnion (Im S (@ba_p_ens T))),
      forall (Ap:Bool_Alg_p T)
             (pfinas:Ensembles.In S Ap),
      exists (Cp:Bool_Alg_p T),
             Ensembles.In S Cp /\
             Ensembles.In (ba_p_ens Cp) (proj1_sig zr) /\
             subalg_of_p Ap Cp /\
        proj1_sig (Bzero_p T (Bc_p T Ap)) =
        proj1_sig zr.
intros S h1 h2. 
destruct h2 as [Cp h2].
assert (h3:Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) (proj1_sig (Bzero_p T (Bc_p T Cp)))).
  apply family_union_intro with (ba_p_ens Cp).
  apply Im_intro with Cp. assumption. reflexivity.
  apply proj2_sig.
exists (exist _ _ h3).
red. split.
intros Ap h4.
red in h1.
specialize (h1 _ _ h4 h2).
destruct h1 as [Dp [h1a [h1b h1c]]].
destruct h1c as [h5 [h6 h7]].
exists Dp. repeat split; simpl; auto.

pose proof (ba_p_subst_zero _ _ h7) as h8. rewrite h8 at 1.
simpl. 
apply in_ba_p_ens_zero. 
destruct h1b as [h9 [h10 h11]].
pose proof (ba_p_subst_zero _ _ h7) as h8. rewrite h8 at 1.
pose proof (ba_p_subst_zero _ _ h11) as h12. rewrite h12 at 1.
reflexivity.
intros zr h4. apply proj1_sig_injective.
specialize (h4 _ h2).
destruct h4 as [Dp [h5 [h6 [h7 h8]]]].
simpl.
rewrite h8.
reflexivity.
Qed.


Definition directed_zero {S:fam_ba_p T}
           (pf:directed S) (pfinh:Inhabited S) :=
  (proj1_sig (constructive_definite_description _ (directed_zero_ex pf pfinh))).


Lemma directed_zero_compat :
  forall {S:fam_ba_p T}
         (pfd:directed S)
         (pfinh:Inhabited S),
    let zr := directed_zero pfd pfinh in
    forall (Ap:Bool_Alg_p T)
           (pfinas:Ensembles.In S Ap),
    exists (Cp:Bool_Alg_p T),
             Ensembles.In S Cp /\
             Ensembles.In (ba_p_ens Cp) (proj1_sig zr) /\
             subalg_of_p Ap Cp /\
        proj1_sig (Bzero_p T (Bc_p T Ap)) =
        proj1_sig zr.
intros S h1 h2 zr. unfold directed_zero in zr.
destruct constructive_definite_description as [zr' h3].
simpl in zr. unfold zr.
assumption.
Qed.



Lemma directed_one_ex :
  forall {S:fam_ba_p T},
    directed S ->
    Inhabited S ->
    exists! one:sig_set (FamilyUnion (Im S (@ba_p_ens T))),
      forall (Ap:Bool_Alg_p T)
             (pfinas:Ensembles.In S Ap),
      exists (Cp:Bool_Alg_p T),
             Ensembles.In S Cp /\
             Ensembles.In (ba_p_ens Cp) (proj1_sig one) /\
             subalg_of_p Ap Cp /\
        proj1_sig (Bone_p T (Bc_p T Ap)) =
        proj1_sig one.
intros S h1 h2. 
destruct h2 as [Cp h2].
assert (h3:Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) (proj1_sig (Bone_p T (Bc_p T Cp)))).
  apply family_union_intro with (ba_p_ens Cp).
  apply Im_intro with Cp. assumption. reflexivity.
  apply proj2_sig.
exists (exist _ _ h3).
red. split.
intros Ap h4.
red in h1.
specialize (h1 _ _ h4 h2).
destruct h1 as [Dp [h1a [h1b h1c]]].
destruct h1c as [h5 [h6 h7]].
exists Dp. repeat split; simpl; auto.

pose proof (ba_p_subst_one _ _ h7) as h8. rewrite h8 at 1.
simpl. 
apply in_ba_p_ens_one. 
destruct h1b as [h9 [h10 h11]].
pose proof (ba_p_subst_one _ _ h7) as h8. rewrite h8 at 1.
pose proof (ba_p_subst_one _ _ h11) as h12. rewrite h12 at 1.
reflexivity.
intros zr h4. apply proj1_sig_injective.
specialize (h4 _ h2).
destruct h4 as [Dp [h5 [h6 [h7 h8]]]].
simpl.
rewrite h8.
reflexivity.
Qed.


Definition directed_one {S:fam_ba_p T}
           (pf:directed S) (pfinh:Inhabited S) :=
  (proj1_sig (constructive_definite_description _ (directed_one_ex pf pfinh))).


Lemma directed_one_compat :
  forall {S:fam_ba_p T}
         (pfd:directed S)
         (pfinh:Inhabited S),
    let one := directed_one pfd pfinh in
    forall (Ap:Bool_Alg_p T)
           (pfinas:Ensembles.In S Ap),
    exists (Cp:Bool_Alg_p T),
             Ensembles.In S Cp /\
             Ensembles.In (ba_p_ens Cp) (proj1_sig one) /\
             subalg_of_p Ap Cp /\
        proj1_sig (Bone_p T (Bc_p T Ap)) =
        proj1_sig one.
intros S h1 h2 one. unfold directed_one in one.
destruct constructive_definite_description as [one' h3].
simpl in one. unfold one.
assumption.
Qed.

Definition directed_bcp 
           {S:fam_ba_p T} (pfd:directed S) (pfinh:Inhabited S) :=
Build_Bconst_p _ (FamilyUnion (Im S (@ba_p_ens T)))
               (full_sig (FamilyUnion (Im S (@ba_p_ens T))))
               (directed_plus pfd) (directed_times pfd) 
               (directed_one pfd pfinh) (directed_zero pfd pfinh)
               (directed_comp pfd).



Lemma directed_und_set_p : 
  forall {S:fam_ba_p T} (pfd:directed S) 
         (pfinh:Inhabited S),
    let Bc_p := directed_bcp pfd pfinh in
    BS_p T Bc_p =
    Full_set (Btype_p T Bc_p).
intros S h1 h2 Bc_p.
unfold Bc_p.
simpl.
reflexivity.
Qed.

Lemma directed_assoc_sum_p : 
    forall {S:fam_ba_p T} (pfd:directed S) 
         (pfinh:Inhabited S),
    let Bc_p := directed_bcp pfd pfinh in
    forall n m p:(Btype_p T Bc_p), n %+ (m %+ p) = n %+ m %+ p.
intros S h1 h2 Bc_p n m p.
pose proof (directed_plus_compat h1 m p) as hmp.
pose proof (directed_plus_compat h1 n (m %+ p)) as hn'mp.
pose proof (directed_plus_compat h1 n m) as hnm.
pose proof (directed_plus_compat h1 (n %+ m) p) as hnm'p.
simpl in hmp, hn'mp, hnm, hnm'p.
destruct n as [n h3]. simpl in h3.
destruct m as [m h4]. simpl in h4.
destruct p as [p h9]. simpl in h9.
destruct h3 as [N n h5 h6]. destruct h5 as [Np h5]. subst.
destruct h4 as [M m h7 h8]. destruct h7 as [Mp h7]. subst.
destruct h9 as [P p h10 h11]. destruct h10 as [Pp h10]. subst.
simpl.
specialize (hnm _ _ h6 h8 h5 h7). simpl in hnm.
specialize (hmp _ _ h8 h11 h7 h10). simpl in hmp.
destruct hnm as [Ap [h13 [h14 [h15 [h16 [h17 [h18 [h19 [h20 h21]]]]]]]]].
destruct hmp as [Ap' [h13' [h14' [h15' [h16' [h17' [h18' [h19' [h20' h21']]]]]]]]].
specialize (hn'mp _ _ h6 h17' h5 h20'). simpl in hn'mp.
specialize (hnm'p _ _ h17 h11 h20 h10). simpl in hnm'p.
destruct hn'mp as [Ap'' [h13'' [h14'' [h15'' [h16'' [h17'' [h18'' [h19'' [h20'' h21'']]]]]]]]].
destruct hnm'p as [Ap''' [h13''' [h14''' [h15''' [h16''' [h17''' [h18''' [h19''' [h20''' h21''']]]]]]]]].
simpl in h21, h21', h21'', h21'''. simpl.
apply (proj1_sig_injective  (fun x : T =>
                   Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)).
rewrite h21'', h21''' at 1.
generalize  (h14''
              (proj1_sig
                 (directed_plus h1
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       m
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Mp) m
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Mp h7
                             (ba_p_ens Mp) eq_refl) h8))
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       p
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Pp) p
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Pp h10
                             (ba_p_ens Pp) eq_refl) h11)))) h17').
intro h22.
generalize  (h13'''
                   (proj1_sig
                      (directed_plus h1
                         (exist
                            (fun x : T =>
                             Ensembles.In
                               (FamilyUnion (Im S (ba_p_ens (T:=T)))) x) n
                            (family_union_intro T (Im S (ba_p_ens (T:=T)))
                               (ba_p_ens Np) n
                               (Im_intro (Bool_Alg_p T) 
                                  (Ensemble T) S (ba_p_ens (T:=T)) Np h5
                                  (ba_p_ens Np) eq_refl) h6))
                         (exist
                            (fun x : T =>
                             Ensembles.In
                               (FamilyUnion (Im S (ba_p_ens (T:=T)))) x) m
                            (family_union_intro T (Im S (ba_p_ens (T:=T)))
                               (ba_p_ens Mp) m
                               (Im_intro (Bool_Alg_p T) 
                                  (Ensemble T) S (ba_p_ens (T:=T)) Mp h7
                                  (ba_p_ens Mp) eq_refl) h8)))) h17) as h23.
intro h23.  
assert (h24:Ensembles.In (ba_p_ens Ap'') 
                         (proj1_sig
           (exist (Ensembles.In (ba_p_ens Ap')) m (h13' m h8)
            %+ exist (Ensembles.In (ba_p_ens Ap')) p (h14' p h11)))).
  rewrite <- h21'. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h22 h24 h21') as h25.
dependent rewrite -> h25. 
pose proof (ba_p_subst_plus _ _ h19'' _ _ (h13' m h8) (h14' p h11)) as h26.
assert (h27: Ensembles.In
                    (ba_p_ens (Subalg_p Ap'' (ba_p_ens Ap') h14'' h16'')) m).
  rewrite <- h19''. apply (h13' m h8).
assert (h28 : Ensembles.In
                (ba_p_ens (Subalg_p Ap'' (ba_p_ens Ap') h14'' h16'')) p).
  rewrite <- h19''. apply (h14' p h11).
specialize (h26 h27 h28).
assert (h29:Ensembles.In (ba_p_ens Ap'') 
                           (proj1_sig (exist
             (Ensembles.In
                (ba_p_ens (Subalg_p Ap'' (ba_p_ens Ap') h14'' h16''))) m h27
           %+ exist
                (Ensembles.In
                   (ba_p_ens (Subalg_p Ap'' (ba_p_ens Ap') h14'' h16''))) p
                h28))).
  apply in_sup_alg_p.
pose proof (subsetT_eq_compat _ _ _ _ h24 h29 h26) as h30.
dependent rewrite -> h30.
simpl. rewrite simpl_sig.
assert (h31:Ensembles.In (ba_p_ens Ap''')  (proj1_sig
          (exist (Ensembles.In (ba_p_ens Ap)) n (h13 n h6)
           %+ exist (Ensembles.In (ba_p_ens Ap)) m (h14 m h8)))).
  rewrite <- h21. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h23 h31 h21) as h32.
dependent rewrite -> h32.
pose proof (ba_p_subst_plus _ _ h18''' _ _ (h13 n h6) (h14 m h8)) as h33.
assert (h34:Ensembles.In
              (ba_p_ens (Subalg_p Ap''' (ba_p_ens Ap) h13''' h15''')) n).
  rewrite <- h18'''. apply (h13 n h6).
assert (h35 : Ensembles.In
                    (ba_p_ens (Subalg_p Ap''' (ba_p_ens Ap) h13''' h15''')) m).
  rewrite <- h18'''. apply (h14 m h8).
specialize (h33 h34 h35).
assert (h36:Ensembles.In (ba_p_ens Ap''')  (proj1_sig
          (exist
             (Ensembles.In
                (ba_p_ens (Subalg_p Ap''' (ba_p_ens Ap) h13''' h15'''))) n
             h34
           %+ exist
                (Ensembles.In
                   (ba_p_ens (Subalg_p Ap''' (ba_p_ens Ap) h13''' h15'''))) m
                h35))).
  apply in_sup_alg_p.
pose proof (subsetT_eq_compat _ _ _ _ h31 h36 h33) as h37.
dependent rewrite -> h37.
simpl. rewrite simpl_sig.
rewrite assoc_sum_p at 1.
pose proof (h1 _ _ h20'' h20''') as h38.
destruct h38 as [Dp [h38 [h39 h40]]].
destruct h39 as [h39a [h39b h39c]]. destruct h40 as [h40a [h40b h40c]].
pose proof (ba_p_subst_plus3 _ _ h39c _ _ _ (h13'' n h6) (h14'' m h27) (h14'' p h28)) as h41.
assert (h42: Ensembles.In
               (ba_p_ens (Subalg_p Dp (ba_p_ens Ap'') h39a h39b)) n).
  rewrite <- h39c. apply (h13'' n h6).
assert (h43 : Ensembles.In
                 (ba_p_ens (Subalg_p Dp (ba_p_ens Ap'') h39a h39b)) m).
  rewrite <- h39c. apply (h14'' m h27).
assert (h44 : Ensembles.In
                (ba_p_ens (Subalg_p Dp (ba_p_ens Ap'') h39a h39b)) p).
  rewrite <- h39c. apply (h14'' p h28).
specialize (h41 h42 h43 h44).
rewrite h41 at 1.
pose proof (ba_p_subst_plus3 _ _ h40c _ _ _ (h13''' n h34) (h13''' m h35) (h14''' p h11)) as h45.
assert (h46: Ensembles.In
                    (ba_p_ens (Subalg_p Dp (ba_p_ens Ap''') h40a h40b)) n).
  rewrite <- h40c. apply (h13''' n h34).
assert (h47 : Ensembles.In
                    (ba_p_ens (Subalg_p Dp (ba_p_ens Ap''') h40a h40b)) m).
  rewrite <- h40c. apply (h13''' m h35).
assert (h48 : Ensembles.In
                (ba_p_ens (Subalg_p Dp (ba_p_ens Ap''') h40a h40b)) p).
  rewrite <- h40c. apply (h14''' p h11).
specialize (h45 h46 h47 h48).
rewrite h45 at 1.
simpl.
f_equal. f_equal. apply proj1_sig_injective.
simpl. f_equal. f_equal. apply proj1_sig_injective. simpl.
reflexivity. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
Qed.


Lemma directed_assoc_prod_p : 
    forall {S:fam_ba_p T} (pfd:directed S) 
         (pfinh:Inhabited S),
    let Bc_p := directed_bcp pfd pfinh in
    forall n m p:(Btype_p T Bc_p), n %* (m %* p) = n %* m %* p.
intros S h1 h2 Bc_p n m p.
pose proof (directed_times_compat h1 m p) as hmp.
pose proof (directed_times_compat h1 n (m %* p)) as hn'mp.
pose proof (directed_times_compat h1 n m) as hnm.
pose proof (directed_times_compat h1 (n %* m) p) as hnm'p.
simpl in hmp, hn'mp, hnm, hnm'p.
destruct n as [n h3]. simpl in h3.
destruct m as [m h4]. simpl in h4.
destruct p as [p h9]. simpl in h9.
destruct h3 as [N n h5 h6]. destruct h5 as [Np h5]. subst.
destruct h4 as [M m h7 h8]. destruct h7 as [Mp h7]. subst.
destruct h9 as [P p h10 h11]. destruct h10 as [Pp h10]. subst.
simpl.
specialize (hnm _ _ h6 h8 h5 h7). simpl in hnm.
specialize (hmp _ _ h8 h11 h7 h10). simpl in hmp.
destruct hnm as [Ap [h13 [h14 [h15 [h16 [h17 [h18 [h19 [h20 h21]]]]]]]]].
destruct hmp as [Ap' [h13' [h14' [h15' [h16' [h17' [h18' [h19' [h20' h21']]]]]]]]].
specialize (hn'mp _ _ h6 h17' h5 h20'). simpl in hn'mp.
specialize (hnm'p _ _ h17 h11 h20 h10). simpl in hnm'p.
destruct hn'mp as [Ap'' [h13'' [h14'' [h15'' [h16'' [h17'' [h18'' [h19'' [h20'' h21'']]]]]]]]].
destruct hnm'p as [Ap''' [h13''' [h14''' [h15''' [h16''' [h17''' [h18''' [h19''' [h20''' h21''']]]]]]]]].
simpl in h21, h21', h21'', h21'''. simpl.
apply (proj1_sig_injective  (fun x : T =>
                   Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)).
rewrite h21'', h21''' at 1.
generalize  (h14''
              (proj1_sig
                 (directed_times h1
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       m
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Mp) m
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Mp h7
                             (ba_p_ens Mp) eq_refl) h8))
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       p
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Pp) p
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Pp h10
                             (ba_p_ens Pp) eq_refl) h11)))) h17').
intro h22.
generalize  (h13'''
                   (proj1_sig
                      (directed_times h1
                         (exist
                            (fun x : T =>
                             Ensembles.In
                               (FamilyUnion (Im S (ba_p_ens (T:=T)))) x) n
                            (family_union_intro T (Im S (ba_p_ens (T:=T)))
                               (ba_p_ens Np) n
                               (Im_intro (Bool_Alg_p T) 
                                  (Ensemble T) S (ba_p_ens (T:=T)) Np h5
                                  (ba_p_ens Np) eq_refl) h6))
                         (exist
                            (fun x : T =>
                             Ensembles.In
                               (FamilyUnion (Im S (ba_p_ens (T:=T)))) x) m
                            (family_union_intro T (Im S (ba_p_ens (T:=T)))
                               (ba_p_ens Mp) m
                               (Im_intro (Bool_Alg_p T) 
                                  (Ensemble T) S (ba_p_ens (T:=T)) Mp h7
                                  (ba_p_ens Mp) eq_refl) h8)))) h17) as h23.
intro h23.  
assert (h24:Ensembles.In (ba_p_ens Ap'') 
                         (proj1_sig
           (exist (Ensembles.In (ba_p_ens Ap')) m (h13' m h8)
            %* exist (Ensembles.In (ba_p_ens Ap')) p (h14' p h11)))).
  rewrite <- h21'. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h22 h24 h21') as h25.
dependent rewrite -> h25. 
pose proof (ba_p_subst_times _ _ h19'' _ _ (h13' m h8) (h14' p h11)) as h26.
assert (h27: Ensembles.In
                    (ba_p_ens (Subalg_p Ap'' (ba_p_ens Ap') h14'' h16'')) m).
  rewrite <- h19''. apply (h13' m h8).
assert (h28 : Ensembles.In
                (ba_p_ens (Subalg_p Ap'' (ba_p_ens Ap') h14'' h16'')) p).
  rewrite <- h19''. apply (h14' p h11).
specialize (h26 h27 h28).
assert (h29:Ensembles.In (ba_p_ens Ap'') 
                           (proj1_sig (exist
             (Ensembles.In
                (ba_p_ens (Subalg_p Ap'' (ba_p_ens Ap') h14'' h16''))) m h27
           %* exist
                (Ensembles.In
                   (ba_p_ens (Subalg_p Ap'' (ba_p_ens Ap') h14'' h16''))) p
                h28))).
  apply in_sup_alg_p.
pose proof (subsetT_eq_compat _ _ _ _ h24 h29 h26) as h30.
dependent rewrite -> h30.
simpl. rewrite simpl_sig.
assert (h31:Ensembles.In (ba_p_ens Ap''')  (proj1_sig
          (exist (Ensembles.In (ba_p_ens Ap)) n (h13 n h6)
           %* exist (Ensembles.In (ba_p_ens Ap)) m (h14 m h8)))).
  rewrite <- h21. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h23 h31 h21) as h32.
dependent rewrite -> h32.
pose proof (ba_p_subst_times _ _ h18''' _ _ (h13 n h6) (h14 m h8)) as h33.
assert (h34:Ensembles.In
              (ba_p_ens (Subalg_p Ap''' (ba_p_ens Ap) h13''' h15''')) n).
  rewrite <- h18'''. apply (h13 n h6).
assert (h35 : Ensembles.In
                    (ba_p_ens (Subalg_p Ap''' (ba_p_ens Ap) h13''' h15''')) m).
  rewrite <- h18'''. apply (h14 m h8).
specialize (h33 h34 h35).
assert (h36:Ensembles.In (ba_p_ens Ap''')  (proj1_sig
          (exist
             (Ensembles.In
                (ba_p_ens (Subalg_p Ap''' (ba_p_ens Ap) h13''' h15'''))) n
             h34
           %* exist
                (Ensembles.In
                   (ba_p_ens (Subalg_p Ap''' (ba_p_ens Ap) h13''' h15'''))) m
                h35))).
  apply in_sup_alg_p.
pose proof (subsetT_eq_compat _ _ _ _ h31 h36 h33) as h37.
dependent rewrite -> h37.
simpl. rewrite simpl_sig.
rewrite assoc_prod_p at 1.
pose proof (h1 _ _ h20'' h20''') as h38.
destruct h38 as [Dp [h38 [h39 h40]]].
destruct h39 as [h39a [h39b h39c]]. destruct h40 as [h40a [h40b h40c]].
pose proof (ba_p_subst_times3 _ _ h39c _ _ _ (h13'' n h6) (h14'' m h27) (h14'' p h28)) as h41.
assert (h42: Ensembles.In
               (ba_p_ens (Subalg_p Dp (ba_p_ens Ap'') h39a h39b)) n).
  rewrite <- h39c. apply (h13'' n h6).
assert (h43 : Ensembles.In
                 (ba_p_ens (Subalg_p Dp (ba_p_ens Ap'') h39a h39b)) m).
  rewrite <- h39c. apply (h14'' m h27).
assert (h44 : Ensembles.In
                (ba_p_ens (Subalg_p Dp (ba_p_ens Ap'') h39a h39b)) p).
  rewrite <- h39c. apply (h14'' p h28).
specialize (h41 h42 h43 h44).
rewrite h41 at 1.
pose proof (ba_p_subst_times3 _ _ h40c _ _ _ (h13''' n h34) (h13''' m h35) (h14''' p h11)) as h45.
assert (h46: Ensembles.In
                    (ba_p_ens (Subalg_p Dp (ba_p_ens Ap''') h40a h40b)) n).
  rewrite <- h40c. apply (h13''' n h34).
assert (h47 : Ensembles.In
                    (ba_p_ens (Subalg_p Dp (ba_p_ens Ap''') h40a h40b)) m).
  rewrite <- h40c. apply (h13''' m h35).
assert (h48 : Ensembles.In
                (ba_p_ens (Subalg_p Dp (ba_p_ens Ap''') h40a h40b)) p).
  rewrite <- h40c. apply (h14''' p h11).
specialize (h45 h46 h47 h48).
rewrite h45 at 1.
simpl.
f_equal. f_equal. apply proj1_sig_injective.
simpl. f_equal. f_equal. apply proj1_sig_injective. simpl.
reflexivity. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
Qed.


Lemma directed_comm_sum_p : 
  forall {S:fam_ba_p T} (pfd:directed S)
         (pfinh:Inhabited S),
    let Bc_p := directed_bcp pfd pfinh in
    forall n m:(Btype_p T Bc_p), n %+ m = m %+ n.
intros S h1 h2 Bc_p n m.
pose proof (directed_plus_compat h1 n m) as hnm.
pose proof (directed_plus_compat h1 m n) as hmn.
destruct n as [n h3]. simpl in h3.
destruct m as [m h4]. simpl in h4.
destruct h3 as [N n h5 h6]. destruct h5 as [Np h5]. subst.
destruct h4 as [M m h7 h8]. destruct h7 as [Mp h7]. subst.
specialize (hnm _ _ h6 h8 h5 h7). simpl in hnm.
specialize (hmn _ _ h8 h6 h7 h5). simpl in hmn.
destruct hnm as [Ap [h13 [h14 [h15 [h16 [h17 [h18 [h19 [h20 h21]]]]]]]]].
destruct hmn as [Ap' [h13' [h14' [h15' [h16' [h17' [h18' [h19' [h20' h21']]]]]]]]].
simpl in h21, h21'. simpl.
apply (proj1_sig_injective  (fun x : T =>
                   Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)).
rewrite h21, h21' at 1.
rewrite comm_sum_p.
pose proof (h1 _ _ h20 h20') as h22.
destruct h22 as [Bp [h22 [h23 h24]]].
destruct h23 as [h23a [h23b h23c]], h24 as [h24a [h24b h24c]].
pose proof (ba_p_subst_plus _ _ h23c _ _ (h14 m h8) (h13 n h6)) as h25.
assert (h26 : Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h23a h23b)) m).
  rewrite <- h23c. apply (h14 m h8).
assert (h27 : Ensembles.In
                (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h23a h23b)) n).
  rewrite <- h23c. apply (h13 n h6).
specialize (h25 h26 h27).
rewrite h25 at 1.
pose proof (ba_p_subst_plus _ _ h24c _ _ (h13' m h8) (h14' n h6)) as h28.
assert (h29: Ensembles.In
               (ba_p_ens (Subalg_p Bp (ba_p_ens Ap') h24a h24b)) m).
  rewrite <- h24c. apply (h13' m h8).
assert (h30 : Ensembles.In
                (ba_p_ens (Subalg_p Bp (ba_p_ens Ap') h24a h24b)) n).
  rewrite <- h24c. apply (h14' n h6).
specialize (h28 h29 h30).
rewrite h28 at 1.
simpl. f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
Qed.


Lemma directed_comm_prod_p : 
  forall {S:fam_ba_p T} (pfd:directed S)
         (pfinh:Inhabited S),
    let Bc_p := directed_bcp pfd pfinh in
    forall n m:(Btype_p T Bc_p), n %* m = m %* n.
intros S h1 h2 Bc_p n m.
pose proof (directed_times_compat h1 n m) as hnm.
pose proof (directed_times_compat h1 m n) as hmn.
destruct n as [n h3]. simpl in h3.
destruct m as [m h4]. simpl in h4.
destruct h3 as [N n h5 h6]. destruct h5 as [Np h5]. subst.
destruct h4 as [M m h7 h8]. destruct h7 as [Mp h7]. subst.
specialize (hnm _ _ h6 h8 h5 h7). simpl in hnm.
specialize (hmn _ _ h8 h6 h7 h5). simpl in hmn.
destruct hnm as [Ap [h13 [h14 [h15 [h16 [h17 [h18 [h19 [h20 h21]]]]]]]]].
destruct hmn as [Ap' [h13' [h14' [h15' [h16' [h17' [h18' [h19' [h20' h21']]]]]]]]].
simpl in h21, h21'. simpl.
apply (proj1_sig_injective  (fun x : T =>
                   Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)).
rewrite h21, h21' at 1.
rewrite comm_prod_p.
pose proof (h1 _ _ h20 h20') as h22.
destruct h22 as [Bp [h22 [h23 h24]]].
destruct h23 as [h23a [h23b h23c]], h24 as [h24a [h24b h24c]].
pose proof (ba_p_subst_times _ _ h23c _ _ (h14 m h8) (h13 n h6)) as h25.
assert (h26 : Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h23a h23b)) m).
  rewrite <- h23c. apply (h14 m h8).
assert (h27 : Ensembles.In
                (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h23a h23b)) n).
  rewrite <- h23c. apply (h13 n h6).
specialize (h25 h26 h27).
rewrite h25 at 1.
pose proof (ba_p_subst_times _ _ h24c _ _ (h13' m h8) (h14' n h6)) as h28.
assert (h29: Ensembles.In
               (ba_p_ens (Subalg_p Bp (ba_p_ens Ap') h24a h24b)) m).
  rewrite <- h24c. apply (h13' m h8).
assert (h30 : Ensembles.In
                (ba_p_ens (Subalg_p Bp (ba_p_ens Ap') h24a h24b)) n).
  rewrite <- h24c. apply (h14' n h6).
specialize (h28 h29 h30).
rewrite h28 at 1.
simpl. f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
Qed.


Lemma directed_abs_sum_p  :
    forall {S:fam_ba_p T} (pfd:directed S)
         (pfinh:Inhabited S),
    let Bc_p := directed_bcp pfd pfinh in
    forall n m:(Btype_p T Bc_p), n %+ (n %* m) = n.
intros S h1 h2 Bc_p n m.
pose proof (directed_times_compat h1 n m) as hnm.
pose proof (directed_plus_compat h1 n (n %* m)) as hn'nm.
simpl in hnm, hn'nm.
destruct n as [n h3]. simpl in h3.
destruct m as [m h4]. simpl in h4.
destruct h3 as [N n h5 h6]. destruct h5 as [Np h5]. subst.
destruct h4 as [M m h7 h8]. destruct h7 as [Mp h7]. subst.
simpl.
specialize (hnm _ _ h6 h8 h5 h7). simpl in hnm.
destruct hnm as [Ap [h13 [h14 [h15 [h16 [h17 [h18 [h19 [h20 h21]]]]]]]]].
specialize (hn'nm _ _ h6 h17 h5 h20). simpl in hn'nm.
destruct hn'nm as [Ap' [h13' [h14' [h15' [h16' [h17' [h18' [h19' [h20' h21']]]]]]]]].
simpl in h21, h21'. simpl.
apply (proj1_sig_injective  (fun x : T =>
                   Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)).
rewrite h21' at 1.
generalize (h14'
              (proj1_sig
                 (directed_times h1
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       n
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Np) n
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Np h5
                             (ba_p_ens Np) eq_refl) h6))
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       m
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Mp) m
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Mp h7
                             (ba_p_ens Mp) eq_refl) h8)))) h17).
intro h22.
simpl.
assert (h23:Ensembles.In (ba_p_ens Ap')
                         (proj1_sig
                            (exist (Ensembles.In (ba_p_ens Ap)) n (h13 n h6)
                                   %* exist (Ensembles.In (ba_p_ens Ap)) m (h14 m h8)))).
  rewrite <- h21. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h22 h23 h21) as h24.
dependent rewrite -> h24.
pose proof (h1 _ _ h20 h20') as h25.
destruct h25 as [Bp [h25 [h26 h27]]].
destruct h26 as [h26a [h26b h26c]], h27 as [h27a [h27b h27c]]. 
pose proof (ba_p_subst_times _ _ h26c _ _ (h13 n h6) (h14 m h8)) as h28.
assert (h29 : Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h26a h26b)) n).
  rewrite <- h26c. apply (h13 n h6).
assert (h30 : Ensembles.In
                (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h26a h26b)) m).
  rewrite <- h26c. apply (h14 m h8).
specialize (h28 h29 h30). 
assert (h31:Ensembles.In (ba_p_ens Ap')  (proj1_sig
          (exist
             (Ensembles.In (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h26a h26b)))
             n h29
           %* exist
                (Ensembles.In
                   (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h26a h26b))) m h30))).
  rewrite <- h28. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h23 h31 h28) as h32.
dependent rewrite -> h32.
simpl.
pose proof (ba_p_subst_plus _ _ h27c _ _ (h13' n h6) h31) as h33.
assert (h34: Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap') h27a h27b)) n).
  rewrite <- h27c. apply (h13' n h6).
assert (h35 : Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap') h27a h27b))
                    (proj1_sig
                       (exist
                          (Ensembles.In
                             (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h26a h26b)))
                          n h29
                        %* exist
                             (Ensembles.In
                                (ba_p_ens
                                   (Subalg_p Bp (ba_p_ens Ap) h26a h26b))) m
                             h30))).
  rewrite <- h27c. apply h31.
specialize (h33 h34 h35).
rewrite h33 at 1.
simpl.
rewrite simpl_sig.
assert (h36:h27a n h34 = h26a n h29). apply proof_irrelevance.
rewrite h36.
rewrite abs_sum_p at 1.
simpl.
reflexivity.
Qed.



Lemma directed_abs_prod_p  :
    forall {S:fam_ba_p T} (pfd:directed S)
         (pfinh:Inhabited S),
    let Bc_p := directed_bcp pfd pfinh in
    forall n m:(Btype_p T Bc_p), n %* (n %+ m) = n.
intros S h1 h2 Bc_p n m.
pose proof (directed_plus_compat h1 n m) as hnm.
pose proof (directed_times_compat h1 n (n %+ m)) as hn'nm.
simpl in hnm, hn'nm.
destruct n as [n h3]. simpl in h3.
destruct m as [m h4]. simpl in h4.
destruct h3 as [N n h5 h6]. destruct h5 as [Np h5]. subst.
destruct h4 as [M m h7 h8]. destruct h7 as [Mp h7]. subst.
simpl.
specialize (hnm _ _ h6 h8 h5 h7). simpl in hnm.
destruct hnm as [Ap [h13 [h14 [h15 [h16 [h17 [h18 [h19 [h20 h21]]]]]]]]].
specialize (hn'nm _ _ h6 h17 h5 h20). simpl in hn'nm.
destruct hn'nm as [Ap' [h13' [h14' [h15' [h16' [h17' [h18' [h19' [h20' h21']]]]]]]]].
simpl in h21, h21'. simpl.
apply (proj1_sig_injective  (fun x : T =>
                   Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)).
rewrite h21' at 1.
generalize (h14'
              (proj1_sig
                 (directed_plus h1
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       n
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Np) n
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Np h5
                             (ba_p_ens Np) eq_refl) h6))
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       m
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Mp) m
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Mp h7
                             (ba_p_ens Mp) eq_refl) h8)))) h17).
intro h22.
simpl.
assert (h23:Ensembles.In (ba_p_ens Ap')
                         (proj1_sig
                            (exist (Ensembles.In (ba_p_ens Ap)) n (h13 n h6)
                                   %+ exist (Ensembles.In (ba_p_ens Ap)) m (h14 m h8)))).
  rewrite <- h21. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h22 h23 h21) as h24.
dependent rewrite -> h24.
pose proof (h1 _ _ h20 h20') as h25.
destruct h25 as [Bp [h25 [h26 h27]]].
destruct h26 as [h26a [h26b h26c]], h27 as [h27a [h27b h27c]]. 
pose proof (ba_p_subst_plus _ _ h26c _ _ (h13 n h6) (h14 m h8)) as h28.
assert (h29 : Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h26a h26b)) n).
  rewrite <- h26c. apply (h13 n h6).
assert (h30 : Ensembles.In
                (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h26a h26b)) m).
  rewrite <- h26c. apply (h14 m h8).
specialize (h28 h29 h30). 
assert (h31:Ensembles.In (ba_p_ens Ap')  (proj1_sig
          (exist
             (Ensembles.In (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h26a h26b)))
             n h29
           %+ exist
                (Ensembles.In
                   (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h26a h26b))) m h30))).
  rewrite <- h28. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h23 h31 h28) as h32.
dependent rewrite -> h32.
simpl.
pose proof (ba_p_subst_times _ _ h27c _ _ (h13' n h6) h31) as h33.
assert (h34: Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap') h27a h27b)) n).
  rewrite <- h27c. apply (h13' n h6).
assert (h35 : Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap') h27a h27b))
                    (proj1_sig
                       (exist
                          (Ensembles.In
                             (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h26a h26b)))
                          n h29
                        %+ exist
                             (Ensembles.In
                                (ba_p_ens
                                   (Subalg_p Bp (ba_p_ens Ap) h26a h26b))) m
                             h30))).
  rewrite <- h27c. apply h31.
specialize (h33 h34 h35).
rewrite h33 at 1.
simpl.
rewrite simpl_sig.
assert (h36:h27a n h34 = h26a n h29). apply proof_irrelevance.
rewrite h36.
rewrite abs_prod_p at 1.
simpl.
reflexivity.
Qed.


Lemma directed_dist_sum_p :
  forall {S:fam_ba_p T} (pfd:directed S)
         (pfinh:Inhabited S),
    let Bc_p := directed_bcp pfd pfinh in
    forall n m p:(Btype_p T Bc_p), p%*(n %+ m) = p %* n %+ p %* m.
intros S h1 h2 Bc_p n m p.
pose proof (directed_plus_compat h1 n m) as hnm.
pose proof (directed_times_compat h1 p (n %+ m)) as hp'nm.
pose proof (directed_times_compat h1 p n) as hpn.
pose proof (directed_times_compat h1 p m) as hpm.
pose proof (directed_plus_compat h1 (p %* n) (p %* m)) as hpn'pm.
simpl in hnm, hp'nm, hpn, hpm, hpn'pm.
destruct n as [n h3]. simpl in h3.
destruct m as [m h4]. simpl in h4.
destruct p as [p h9]. simpl in h9.
destruct h3 as [N n h5 h6]. destruct h5 as [Np h5]. subst.
destruct h4 as [M m h7 h8]. destruct h7 as [Mp h7]. subst.
destruct h9 as [P p h10 h11]. destruct h10 as [Pp h10]. subst.
simpl.
specialize (hnm _ _ h6 h8 h5 h7). simpl in hnm.
specialize (hpn _ _ h11 h6 h10 h5). simpl in hpn.
specialize (hpm _ _ h11 h8 h10 h7). simpl in hpm.
destruct hnm as [Ap [h13 [h14 [h15 [h16 [h17 [h18 [h19 [h20 h21]]]]]]]]].
destruct hpn as [Ap' [h13' [h14' [h15' [h16' [h17' [h18' [h19' [h20' h21']]]]]]]]].
destruct hpm as [Ap'' [h13'' [h14'' [h15'' [h16'' [h17'' [h18'' [h19'' [h20'' h21'']]]]]]]]].
specialize (hp'nm _ _ h11 h17 h10 h20). simpl in hp'nm.
specialize (hpn'pm _ _ h17' h17'' h20' h20''). simpl in hpn'pm.
destruct hp'nm as [Ap''' [h13''' [h14''' [h15''' [h16''' [h17''' [h18''' [h19''' [h20''' h21''']]]]]]]]].
destruct hpn'pm as [Ap'''' [h13'''' [h14'''' [h15'''' [h16'''' [h17'''' [h18'''' [h19'''' [h20'''' h21'''']]]]]]]]].
simpl in h21, h21', h21'', h21''', h21''''. simpl.
apply (proj1_sig_injective  (fun x : T =>
                   Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)).
rewrite h21''', h21'''' at 1.
generalize (h14'''
              (proj1_sig
                 (directed_plus h1
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       n
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Np) n
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Np h5
                             (ba_p_ens Np) eq_refl) h6))
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       m
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Mp) m
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Mp h7
                             (ba_p_ens Mp) eq_refl) h8)))) h17).
intro h22.
generalize  (h13''''
           (proj1_sig
              (directed_times h1
                 (exist
                    (fun x : T =>
                     Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x) p
                    (family_union_intro T (Im S (ba_p_ens (T:=T)))
                       (ba_p_ens Pp) p
                       (Im_intro (Bool_Alg_p T) (Ensemble T) S
                          (ba_p_ens (T:=T)) Pp h10 
                          (ba_p_ens Pp) eq_refl) h11))
                 (exist
                    (fun x : T =>
                     Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x) n
                    (family_union_intro T (Im S (ba_p_ens (T:=T)))
                       (ba_p_ens Np) n
                       (Im_intro (Bool_Alg_p T) (Ensemble T) S
                          (ba_p_ens (T:=T)) Np h5 (ba_p_ens Np) eq_refl) h6)))) h17').
intro h23.
generalize  (h14''''
              (proj1_sig
                 (directed_times h1
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       p
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Pp) p
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Pp h10
                             (ba_p_ens Pp) eq_refl) h11))
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       m
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Mp) m
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Mp h7
                             (ba_p_ens Mp) eq_refl) h8)))) h17'').
intro h24.
assert (h25:Ensembles.In (ba_p_ens Ap''')
                         (proj1_sig
          (exist (Ensembles.In (ba_p_ens Ap)) n (h13 n h6)
           %+ exist (Ensembles.In (ba_p_ens Ap)) m (h14 m h8)))).
  rewrite <- h21.
  apply h14'''. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h22 h25 h21) as h26.
dependent rewrite -> h26.
assert (h27:Ensembles.In (ba_p_ens Ap'''')
                          (proj1_sig
           (exist (Ensembles.In (ba_p_ens Ap')) p (h13' p h11)
            %* exist (Ensembles.In (ba_p_ens Ap')) n (h14' n h6)))).
  rewrite <- h21'. apply h13''''. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h23 h27 h21') as h28.
dependent rewrite -> h28.
assert (h29:Ensembles.In (ba_p_ens Ap'''')
                         ( proj1_sig
            (exist (Ensembles.In (ba_p_ens Ap'')) p (h13'' p h11)
             %* exist (Ensembles.In (ba_p_ens Ap'')) m (h14'' m h8)))).
  rewrite <- h21''. apply h14''''. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h24 h29 h21'') as h30.
dependent rewrite -> h30.
pose proof (h1 _ _ h20 h20''') as h31.
destruct h31 as [Bp [h32 [h33 h34]]].
destruct h33 as [h33a [h33b h33c]], h34 as [h34a [h34b h34c]].
pose proof (ba_p_subst_times _ _ h34c _ _ (h13''' p h11) h25) as h35.
assert (h36:Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap''') h34a h34b)) p).
  rewrite <- h34c. apply (h13''' p h11).
assert (h37 : Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap''') h34a h34b))
                    (proj1_sig
                       (exist (Ensembles.In (ba_p_ens Ap)) n (h13 n h6)
                        %+ exist (Ensembles.In (ba_p_ens Ap)) m (h14 m h8)))).
  rewrite <- h34c. assumption.
specialize (h35 h36 h37).
rewrite h35 at 1. 
pose proof (ba_p_subst_plus _ _ h33c _ _ (h13 n h6) (h14 m h8)) as h38.
assert (h39: Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h33a h33b)) n).
  rewrite <- h33c. apply (h13 n h6).
assert (h40 : Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h33a h33b)) m).
  rewrite <- h33c. apply (h14 m h8).
specialize (h38 h39 h40).
assert (h41:Ensembles.In (ba_p_ens (Subalg_p Bp (ba_p_ens Ap''') h34a h34b))
                       (proj1_sig
          (exist
             (Ensembles.In (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h33a h33b)))
             n h39
           %+ exist
                (Ensembles.In
                   (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h33a h33b))) m h40))).
  rewrite <- h38. rewrite <- h34c. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h37 h41 h38) as h42.
dependent rewrite -> h42.
simpl.
rewrite simpl_sig.
rewrite dist_sum_p.
pose proof (h1 _ _ h20' h20'''') as h43.
destruct h43 as [Cp [h44 [h45 h46]]].
pose proof (h1 _ _ h20'' h20'''') as h47.
destruct h47 as [Dp [h47 [h48 h49]]].
pose proof (h1 _ _ h44 h47) as h50.
destruct h50 as [Ep [h51 [h52 h53]]].
pose proof (trans_subalg_of_p _ _ _ _ h45 h52) as h54.
pose proof (trans_subalg_of_p _ _ _ _ h48 h53) as h55.
pose proof (trans_subalg_of_p _ _ _ _ h46 h52) as h56.
destruct h54 as [h54a [h54b h54c]].
destruct h55 as [h55a [h55b h55c]].
destruct h56 as [h56a [h56b h56c]]. 
pose proof (ba_p_subst_plus _ _ h56c _ _ h27 h29) as h57.
assert (ha:Ensembles.In
                    (ba_p_ens (Subalg_p Ep (ba_p_ens Ap'''') h56a h56b))
                    (proj1_sig
                       (exist (Ensembles.In (ba_p_ens Ap')) p (h13' p h11)
                        %* exist (Ensembles.In (ba_p_ens Ap')) n (h14' n h6)))).
  rewrite <- h56c. assumption.
assert (hb : Ensembles.In
                    (ba_p_ens (Subalg_p Ep (ba_p_ens Ap'''') h56a h56b))
                    (proj1_sig
                       (exist (Ensembles.In (ba_p_ens Ap'')) p (h13'' p h11)
                        %* exist (Ensembles.In (ba_p_ens Ap'')) m
                             (h14'' m h8)))).
  rewrite <- h56c. assumption.
specialize (h57 ha hb).
rewrite h57 at 1.
pose proof (ba_p_subst_times _ _ h54c _ _ (h13' p h11) (h14' n h6)) as h58. 
assert (h59: Ensembles.In
                    (ba_p_ens (Subalg_p Ep (ba_p_ens Ap') h54a h54b)) p).
  rewrite <- h54c. apply (h13' p h11).
assert (h60 : Ensembles.In
                    (ba_p_ens (Subalg_p Ep (ba_p_ens Ap') h54a h54b)) n).
  rewrite <- h54c. apply (h14' n h6).
specialize (h58 h59 h60).
simpl. 
assert (h61:Ensembles.In  (ba_p_ens Ep)
                          (proj1_sig
          (exist
             (Ensembles.In (ba_p_ens (Subalg_p Ep (ba_p_ens Ap') h54a h54b)))
             p h59
           %* exist
                (Ensembles.In
                   (ba_p_ens (Subalg_p Ep (ba_p_ens Ap') h54a h54b))) n h60))).
  apply in_sup_alg_p.
pose proof (subsetT_eq_compat _ _ _ _  (h56a
           (proj1_sig
              (exist (Ensembles.In (ba_p_ens Ap')) p (h13' p h11)
               %* exist (Ensembles.In (ba_p_ens Ap')) n (h14' n h6))) ha) h61 h58) as h62.
dependent rewrite -> h62.
pose proof (ba_p_subst_times _ _ h55c _ _ (h13'' p h11) (h14'' m h8)) as h63.
assert (h64:Ensembles.In
                    (ba_p_ens (Subalg_p Ep (ba_p_ens Ap'') h55a h55b)) p).
  rewrite <- h55c. apply (h13'' p h11).
assert (h65 : Ensembles.In
                    (ba_p_ens (Subalg_p Ep (ba_p_ens Ap'') h55a h55b)) m).
  rewrite <- h55c. apply (h14'' m h8).
specialize (h63 h64 h65).
assert (h66:Ensembles.In (ba_p_ens Ep)
                          (proj1_sig
          (exist
             (Ensembles.In (ba_p_ens (Subalg_p Ep (ba_p_ens Ap'') h55a h55b)))
             p h64
           %* exist
                (Ensembles.In
                   (ba_p_ens (Subalg_p Ep (ba_p_ens Ap'') h55a h55b))) m h65))).
  apply in_sup_alg_p.
pose proof (subsetT_eq_compat _ _ _ _  (h56a
              (proj1_sig
                 (exist (Ensembles.In (ba_p_ens Ap'')) p (h13'' p h11)
                  %* exist (Ensembles.In (ba_p_ens Ap'')) m (h14'' m h8))) hb) h66 h63) as h67.
dependent rewrite -> h67.
simpl.
do 2 rewrite simpl_sig.
pose proof (h1 _ _ h32 h51) as h68.
destruct h68 as [Fp [h68 [h69 h70]]].
red in h69, h70.
destruct h69 as [h69a [h69b h69c]], h70 as [h70a [h70b h70c]].
pose proof (ba_p_subst_times_plus_times _ _ h69c _ _ _ _ (h34a p h36) (h33a n h39) (h34a p h36) (h33a m h40)) as h71.
assert (h72:Ensembles.In
              (ba_p_ens (Subalg_p Fp (ba_p_ens Bp) h69a h69b)) p).
  rewrite <- h69c. apply (h34a p h36).
assert (h73 : Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Bp) h69a h69b)) n).
  rewrite <- h69c. apply (h33a n h39).
assert (h74 : Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Bp) h69a h69b)) p).
  rewrite <- h69c. apply (h34a p h36).
assert (h75 : Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Bp) h69a h69b)) m).
  rewrite <- h69c. apply (h33a m h40).
specialize (h71 h72 h73 h74 h75).
rewrite h71 at 1.
pose proof (ba_p_subst_times_plus_times _ _ h70c _ _ _ _ (h54a p h59) (h54a n h60) (h55a p h64) (h55a m h65)) as h76.
assert (h77:Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Ep) h70a h70b)) p).
  rewrite <- h70c. apply (h54a p h59).
assert (h78 : Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Ep) h70a h70b)) n).
  rewrite <- h70c. apply (h54a n h60).
assert (h79 : Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Ep) h70a h70b)) p).
  rewrite <- h70c. apply (h55a p h64).
assert (h80 : Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Ep) h70a h70b)) m).
  rewrite <- h70c. apply (h55a m h65).
specialize (h76 h77 h78 h79 h80).
rewrite h76 at 1.
simpl.
f_equal. f_equal. apply proj1_sig_injective. simpl.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
Qed.



Lemma directed_dist_prod_p :
  forall {S:fam_ba_p T} (pfd:directed S)
         (pfinh:Inhabited S),
    let Bc_p := directed_bcp pfd pfinh in
    forall n m p:(Btype_p T Bc_p), p%+(n %* m) = (p %+ n) %* (p %+ m).
intros S h1 h2 Bc_p n m p.
pose proof (directed_times_compat h1 n m) as hnm.
pose proof (directed_plus_compat h1 p (n %* m)) as hp'nm.
pose proof (directed_plus_compat h1 p n) as hpn.
pose proof (directed_plus_compat h1 p m) as hpm.
pose proof (directed_times_compat h1 (p %+ n) (p %+ m)) as hpn'pm.
simpl in hnm, hp'nm, hpn, hpm, hpn'pm.
destruct n as [n h3]. simpl in h3.
destruct m as [m h4]. simpl in h4.
destruct p as [p h9]. simpl in h9.
destruct h3 as [N n h5 h6]. destruct h5 as [Np h5]. subst.
destruct h4 as [M m h7 h8]. destruct h7 as [Mp h7]. subst.
destruct h9 as [P p h10 h11]. destruct h10 as [Pp h10]. subst.
simpl.
specialize (hnm _ _ h6 h8 h5 h7). simpl in hnm.
specialize (hpn _ _ h11 h6 h10 h5). simpl in hpn.
specialize (hpm _ _ h11 h8 h10 h7). simpl in hpm.
destruct hnm as [Ap [h13 [h14 [h15 [h16 [h17 [h18 [h19 [h20 h21]]]]]]]]].
destruct hpn as [Ap' [h13' [h14' [h15' [h16' [h17' [h18' [h19' [h20' h21']]]]]]]]].
destruct hpm as [Ap'' [h13'' [h14'' [h15'' [h16'' [h17'' [h18'' [h19'' [h20'' h21'']]]]]]]]].
specialize (hp'nm _ _ h11 h17 h10 h20). simpl in hp'nm.
specialize (hpn'pm _ _ h17' h17'' h20' h20''). simpl in hpn'pm.
destruct hp'nm as [Ap''' [h13''' [h14''' [h15''' [h16''' [h17''' [h18''' [h19''' [h20''' h21''']]]]]]]]].
destruct hpn'pm as [Ap'''' [h13'''' [h14'''' [h15'''' [h16'''' [h17'''' [h18'''' [h19'''' [h20'''' h21'''']]]]]]]]].
simpl in h21, h21', h21'', h21''', h21''''. simpl.
apply (proj1_sig_injective  (fun x : T =>
                   Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)).
rewrite h21''', h21'''' at 1.
generalize (h14'''
              (proj1_sig
                 (directed_times h1
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       n
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Np) n
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Np h5
                             (ba_p_ens Np) eq_refl) h6))
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       m
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Mp) m
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Mp h7
                             (ba_p_ens Mp) eq_refl) h8)))) h17).
intro h22.
generalize  (h13''''
           (proj1_sig
              (directed_plus h1
                 (exist
                    (fun x : T =>
                     Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x) p
                    (family_union_intro T (Im S (ba_p_ens (T:=T)))
                       (ba_p_ens Pp) p
                       (Im_intro (Bool_Alg_p T) (Ensemble T) S
                          (ba_p_ens (T:=T)) Pp h10 
                          (ba_p_ens Pp) eq_refl) h11))
                 (exist
                    (fun x : T =>
                     Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x) n
                    (family_union_intro T (Im S (ba_p_ens (T:=T)))
                       (ba_p_ens Np) n
                       (Im_intro (Bool_Alg_p T) (Ensemble T) S
                          (ba_p_ens (T:=T)) Np h5 (ba_p_ens Np) eq_refl) h6)))) h17').
intro h23.
generalize  (h14''''
              (proj1_sig
                 (directed_plus h1
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       p
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Pp) p
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Pp h10
                             (ba_p_ens Pp) eq_refl) h11))
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       m
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Mp) m
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Mp h7
                             (ba_p_ens Mp) eq_refl) h8)))) h17'').
intro h24.
assert (h25:Ensembles.In (ba_p_ens Ap''')
                         (proj1_sig
          (exist (Ensembles.In (ba_p_ens Ap)) n (h13 n h6)
           %* exist (Ensembles.In (ba_p_ens Ap)) m (h14 m h8)))).
  rewrite <- h21.
  apply h14'''. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h22 h25 h21) as h26.
dependent rewrite -> h26.
assert (h27:Ensembles.In (ba_p_ens Ap'''')
                          (proj1_sig
           (exist (Ensembles.In (ba_p_ens Ap')) p (h13' p h11)
            %+ exist (Ensembles.In (ba_p_ens Ap')) n (h14' n h6)))).
  rewrite <- h21'. apply h13''''. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h23 h27 h21') as h28.
dependent rewrite -> h28.
assert (h29:Ensembles.In (ba_p_ens Ap'''')
                         ( proj1_sig
            (exist (Ensembles.In (ba_p_ens Ap'')) p (h13'' p h11)
             %+ exist (Ensembles.In (ba_p_ens Ap'')) m (h14'' m h8)))).
  rewrite <- h21''. apply h14''''. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h24 h29 h21'') as h30.
dependent rewrite -> h30.
pose proof (h1 _ _ h20 h20''') as h31.
destruct h31 as [Bp [h32 [h33 h34]]].
destruct h33 as [h33a [h33b h33c]], h34 as [h34a [h34b h34c]].
pose proof (ba_p_subst_plus _ _ h34c _ _ (h13''' p h11) h25) as h35.
assert (h36:Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap''') h34a h34b)) p).
  rewrite <- h34c. apply (h13''' p h11).
assert (h37 : Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap''') h34a h34b))
                    (proj1_sig
                       (exist (Ensembles.In (ba_p_ens Ap)) n (h13 n h6)
                        %* exist (Ensembles.In (ba_p_ens Ap)) m (h14 m h8)))).
  rewrite <- h34c. assumption.
specialize (h35 h36 h37).
rewrite h35 at 1. 
pose proof (ba_p_subst_times _ _ h33c _ _ (h13 n h6) (h14 m h8)) as h38.
assert (h39: Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h33a h33b)) n).
  rewrite <- h33c. apply (h13 n h6).
assert (h40 : Ensembles.In
                    (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h33a h33b)) m).
  rewrite <- h33c. apply (h14 m h8).
specialize (h38 h39 h40).
assert (h41:Ensembles.In (ba_p_ens (Subalg_p Bp (ba_p_ens Ap''') h34a h34b))
                       (proj1_sig
          (exist
             (Ensembles.In (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h33a h33b)))
             n h39
           %* exist
                (Ensembles.In
                   (ba_p_ens (Subalg_p Bp (ba_p_ens Ap) h33a h33b))) m h40))).
  rewrite <- h38. rewrite <- h34c. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h37 h41 h38) as h42.
dependent rewrite -> h42.
simpl.
rewrite simpl_sig.
rewrite dist_prod_p.
pose proof (h1 _ _ h20' h20'''') as h43.
destruct h43 as [Cp [h44 [h45 h46]]].
pose proof (h1 _ _ h20'' h20'''') as h47.
destruct h47 as [Dp [h47 [h48 h49]]].
pose proof (h1 _ _ h44 h47) as h50.
destruct h50 as [Ep [h51 [h52 h53]]].
pose proof (trans_subalg_of_p _ _ _ _ h45 h52) as h54.
pose proof (trans_subalg_of_p _ _ _ _ h48 h53) as h55.
pose proof (trans_subalg_of_p _ _ _ _ h46 h52) as h56.
destruct h54 as [h54a [h54b h54c]].
destruct h55 as [h55a [h55b h55c]].
destruct h56 as [h56a [h56b h56c]]. 
pose proof (ba_p_subst_times _ _ h56c _ _ h27 h29) as h57.
assert (ha:Ensembles.In
                    (ba_p_ens (Subalg_p Ep (ba_p_ens Ap'''') h56a h56b))
                    (proj1_sig
                       (exist (Ensembles.In (ba_p_ens Ap')) p (h13' p h11)
                        %+ exist (Ensembles.In (ba_p_ens Ap')) n (h14' n h6)))).
  rewrite <- h56c. assumption.
assert (hb : Ensembles.In
                    (ba_p_ens (Subalg_p Ep (ba_p_ens Ap'''') h56a h56b))
                    (proj1_sig
                       (exist (Ensembles.In (ba_p_ens Ap'')) p (h13'' p h11)
                        %+ exist (Ensembles.In (ba_p_ens Ap'')) m
                             (h14'' m h8)))).
  rewrite <- h56c. assumption.
specialize (h57 ha hb).
rewrite h57 at 1.
pose proof (ba_p_subst_plus _ _ h54c _ _ (h13' p h11) (h14' n h6)) as h58. 
assert (h59: Ensembles.In
                    (ba_p_ens (Subalg_p Ep (ba_p_ens Ap') h54a h54b)) p).
  rewrite <- h54c. apply (h13' p h11).
assert (h60 : Ensembles.In
                    (ba_p_ens (Subalg_p Ep (ba_p_ens Ap') h54a h54b)) n).
  rewrite <- h54c. apply (h14' n h6).
specialize (h58 h59 h60).
simpl. 
assert (h61:Ensembles.In  (ba_p_ens Ep)
                          (proj1_sig
          (exist
             (Ensembles.In (ba_p_ens (Subalg_p Ep (ba_p_ens Ap') h54a h54b)))
             p h59
           %+ exist
                (Ensembles.In
                   (ba_p_ens (Subalg_p Ep (ba_p_ens Ap') h54a h54b))) n h60))).
  apply in_sup_alg_p.
pose proof (subsetT_eq_compat _ _ _ _  (h56a
           (proj1_sig
              (exist (Ensembles.In (ba_p_ens Ap')) p (h13' p h11)
               %+ exist (Ensembles.In (ba_p_ens Ap')) n (h14' n h6))) ha) h61 h58) as h62.
dependent rewrite -> h62.
pose proof (ba_p_subst_plus _ _ h55c _ _ (h13'' p h11) (h14'' m h8)) as h63.
assert (h64:Ensembles.In
                    (ba_p_ens (Subalg_p Ep (ba_p_ens Ap'') h55a h55b)) p).
  rewrite <- h55c. apply (h13'' p h11).
assert (h65 : Ensembles.In
                    (ba_p_ens (Subalg_p Ep (ba_p_ens Ap'') h55a h55b)) m).
  rewrite <- h55c. apply (h14'' m h8).
specialize (h63 h64 h65).
assert (h66:Ensembles.In (ba_p_ens Ep)
                          (proj1_sig
          (exist
             (Ensembles.In (ba_p_ens (Subalg_p Ep (ba_p_ens Ap'') h55a h55b)))
             p h64
           %+ exist
                (Ensembles.In
                   (ba_p_ens (Subalg_p Ep (ba_p_ens Ap'') h55a h55b))) m h65))).
  apply in_sup_alg_p.
pose proof (subsetT_eq_compat _ _ _ _  (h56a
              (proj1_sig
                 (exist (Ensembles.In (ba_p_ens Ap'')) p (h13'' p h11)
                  %+ exist (Ensembles.In (ba_p_ens Ap'')) m (h14'' m h8))) hb) h66 h63) as h67.
dependent rewrite -> h67.
simpl.
do 2 rewrite simpl_sig.
pose proof (h1 _ _ h32 h51) as h68.
destruct h68 as [Fp [h68 [h69 h70]]].
red in h69, h70.
destruct h69 as [h69a [h69b h69c]], h70 as [h70a [h70b h70c]].
pose proof (ba_p_subst_plus_times_plus _ _ h69c _ _ _ _ (h34a p h36) (h33a n h39) (h34a p h36) (h33a m h40)) as h71.
assert (h72:Ensembles.In
              (ba_p_ens (Subalg_p Fp (ba_p_ens Bp) h69a h69b)) p).
  rewrite <- h69c. apply (h34a p h36).
assert (h73 : Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Bp) h69a h69b)) n).
  rewrite <- h69c. apply (h33a n h39).
assert (h74 : Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Bp) h69a h69b)) p).
  rewrite <- h69c. apply (h34a p h36).
assert (h75 : Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Bp) h69a h69b)) m).
  rewrite <- h69c. apply (h33a m h40).
specialize (h71 h72 h73 h74 h75).
rewrite h71 at 1.
pose proof (ba_p_subst_plus_times_plus _ _ h70c _ _ _ _ (h54a p h59) (h54a n h60) (h55a p h64) (h55a m h65)) as h76.
assert (h77:Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Ep) h70a h70b)) p).
  rewrite <- h70c. apply (h54a p h59).
assert (h78 : Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Ep) h70a h70b)) n).
  rewrite <- h70c. apply (h54a n h60).
assert (h79 : Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Ep) h70a h70b)) p).
  rewrite <- h70c. apply (h55a p h64).
assert (h80 : Ensembles.In
                    (ba_p_ens (Subalg_p Fp (ba_p_ens Ep) h70a h70b)) m).
  rewrite <- h70c. apply (h55a m h65).
specialize (h76 h77 h78 h79 h80).
rewrite h76 at 1.
simpl.
f_equal. f_equal. apply proj1_sig_injective. simpl.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
Qed.


Lemma directed_comp_sum_p :
 forall {S:fam_ba_p T} (pfd:directed S)
         (pfinh:Inhabited S),
    let Bc_p := directed_bcp pfd pfinh in
    forall n:(Btype_p T Bc_p), n %+ (%- n) = %1.
intros S h1 h2 Bc_p n.
pose proof (directed_plus_compat h1 n (%- n)) as hn. 
pose proof (directed_comp_compat h1 n) as hn'. 
pose proof (directed_one_compat h1 h2) as ho.
destruct n as [n h3]. simpl in h3.
destruct h3 as [N n h5 h6].  destruct h5 as [Np h5]. subst.
simpl in hn, hn'.
specialize (hn' _ h6  h5).
destruct hn' as [Cp [h7 [h8 [h9 h10]]]].
apply proj1_sig_injective. simpl.
specialize (hn _ _ h6 h7 h5 h9).
destruct hn as [Ap [h13 [h14 [h15 [h16 [h17 [h18 [h19 [h20 h21]]]]]]]]].
rewrite h21 at 1.
generalize  (h14
              (proj1_sig
                 (directed_comp h1
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       n
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Np) n
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Np h5
                             (ba_p_ens Np) eq_refl) h6)))) h7).
intro h22.
assert (h23:Ensembles.In (ba_p_ens Ap)
                         (proj1_sig (%-exist (Ensembles.In (ba_p_ens Np)) n h6))).
  rewrite <- h10. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h22 h23 h10) as h24.
dependent rewrite -> h24.
pose proof (ba_p_subst_comp _ _ h18 _ h6) as h25.
assert (h26:Ensembles.In (ba_p_ens (Subalg_p Ap (ba_p_ens Np) h13 h15))
                   n).
  rewrite <- h18. assumption.
specialize (h25 h26).
assert (h27:Ensembles.In (ba_p_ens Ap) 
                          (proj1_sig
          (%-exist
               (Ensembles.In (ba_p_ens (Subalg_p Ap (ba_p_ens Np) h13 h15)))
               n h26))).
  apply in_sup_alg_p.
pose proof (subsetT_eq_compat _ _ _ _ h23 h27 h25) as h28.
dependent rewrite -> h28.
simpl.
rewrite simpl_sig.
assert (h29:h13 n h6 = h13 n h26). apply proof_irrelevance.
rewrite h29. 
rewrite comp_sum_p at 1.
specialize (ho Ap h20).
destruct ho as [Dp [ho1 [ho2 [ho3 ho4]]]].
rewrite <- ho4 at 1.
reflexivity.
Qed.
 


Lemma directed_comp_prod_p :
 forall {S:fam_ba_p T} (pfd:directed S)
         (pfinh:Inhabited S),
    let Bc_p := directed_bcp pfd pfinh in
    forall n:(Btype_p T Bc_p), n %* (%- n) = %0.
intros S h1 h2 Bc_p n.
pose proof (directed_times_compat h1 n (%- n)) as hn. 
pose proof (directed_comp_compat h1 n) as hn'. 
pose proof (directed_zero_compat h1 h2) as ho.
destruct n as [n h3]. simpl in h3.
destruct h3 as [N n h5 h6].  destruct h5 as [Np h5]. subst.
simpl in hn, hn'.
specialize (hn' _ h6  h5).
destruct hn' as [Cp [h7 [h8 [h9 h10]]]].
apply proj1_sig_injective. simpl.
specialize (hn _ _ h6 h7 h5 h9).
destruct hn as [Ap [h13 [h14 [h15 [h16 [h17 [h18 [h19 [h20 h21]]]]]]]]].
rewrite h21 at 1.
generalize  (h14
              (proj1_sig
                 (directed_comp h1
                    (exist
                       (fun x : T =>
                        Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T)))) x)
                       n
                       (family_union_intro T (Im S (ba_p_ens (T:=T)))
                          (ba_p_ens Np) n
                          (Im_intro (Bool_Alg_p T) 
                             (Ensemble T) S (ba_p_ens (T:=T)) Np h5
                             (ba_p_ens Np) eq_refl) h6)))) h7).
intro h22.
assert (h23:Ensembles.In (ba_p_ens Ap)
                         (proj1_sig (%-exist (Ensembles.In (ba_p_ens Np)) n h6))).
  rewrite <- h10. assumption.
pose proof (subsetT_eq_compat _ _ _ _ h22 h23 h10) as h24.
dependent rewrite -> h24.
pose proof (ba_p_subst_comp _ _ h18 _ h6) as h25.
assert (h26:Ensembles.In (ba_p_ens (Subalg_p Ap (ba_p_ens Np) h13 h15))
                   n).
  rewrite <- h18. assumption.
specialize (h25 h26).
assert (h27:Ensembles.In (ba_p_ens Ap) 
                          (proj1_sig
          (%-exist
               (Ensembles.In (ba_p_ens (Subalg_p Ap (ba_p_ens Np) h13 h15)))
               n h26))).
  apply in_sup_alg_p.
pose proof (subsetT_eq_compat _ _ _ _ h23 h27 h25) as h28.
dependent rewrite -> h28.
simpl.
rewrite simpl_sig.
assert (h29:h13 n h6 = h13 n h26). apply proof_irrelevance.
rewrite h29. 
rewrite comp_prod_p at 1.
specialize (ho Ap h20).
destruct ho as [Dp [ho1 [ho2 [ho3 ho4]]]].
rewrite <- ho4 at 1.
reflexivity.
Qed.


Definition directed_ba_p {S:fam_ba_p T} (pfd:directed S)
      (pfinh:Inhabited S) :=
  Build_Bool_Alg_p T (directed_bcp pfd pfinh)
                   (directed_und_set_p pfd pfinh)
                   (directed_assoc_sum_p pfd pfinh)
                   (directed_assoc_prod_p pfd pfinh)
                   (directed_comm_sum_p pfd pfinh)
                   (directed_comm_prod_p pfd pfinh)
                   (directed_abs_sum_p pfd pfinh)
                   (directed_abs_prod_p pfd pfinh)
                   (directed_dist_sum_p pfd pfinh)
                   (directed_dist_prod_p pfd pfinh)
                   (directed_comp_sum_p pfd pfinh)
                   (directed_comp_prod_p pfd pfinh).



Lemma directed_ba_p_subalg : 
  forall {S:fam_ba_p T} (pfd:directed S)
         (pfinh:Inhabited S) (Bp:Bool_Alg_p T),
    Ensembles.In S Bp -> 
    subalg_of_p Bp (directed_ba_p pfd pfinh).
intros S h1 h2 Bp h3.
assert (h4: Included (A_p T (Bc_p T Bp))
                     (ba_p_ens (directed_ba_p h1 h2))).
  unfold directed_ba_p. unfold ba_p_ens. simpl.
  red. intros x h4.
  apply family_union_intro with (ba_p_ens Bp).
  apply Im_intro with Bp. assumption. reflexivity. assumption.
exists h4.   
assert (h5:alg_closed_p (A_p T (Bc_p T Bp)) h4).  
   assert (h5:Comp_closed_sub_p _ _ _ h4).  
    red.  intro z. destruct z as [z h5].
    unfold directed_ba_p. unfold Bcomp_sub_p. simpl.   
    pose proof (directed_comp_compat h1 (exist (Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T))))) z
              (h4 z h5))) as h6. simpl in h6.
    specialize (h6 _ h5 h3).
    destruct h6 as [Cp [h8 [h9 [h10 h11]]]].
    rewrite h11.
    apply in_ba_p_ens_comp.
  assert (h6:Plus_closed_sub_p _ _ _ h4). 
    red. intros x y. destruct x as [x h6], y as [y h7].
    unfold directed_ba_p. unfold Bplus_sub_p. simpl.
    pose proof (directed_plus_compat h1 (exist (Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T))))) x
              (h4 x h6))
           (exist (Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T))))) y
              (h4 y h7))) as h8.
    simpl in h8.
    specialize (h8 _ _ h6 h7 h3 h3).
    destruct h8 as [Cp [h8 [h9 [h10 [h11 [h12 [h13 [h14 [h15 h16]]]]]]]]].   
    rewrite h16.
    pose ((exist _ _ h6) %+ (exist _ _ h7)) as z.
    pose proof (proj2_sig z) as h17. simpl in h17. unfold z in h17.
    pose proof (ba_p_subst_plus _ _  h13 _ _ h6 h7) as h18.
    assert (h19: Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h8 h10))
                    x).
      rewrite <- h13. assumption.
    assert (h20 : Ensembles.In (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h8 h10)) y).
      rewrite <- h13. assumption.
   specialize (h18 h19 h20).
   rewrite h18 in h17 at 1.
   simpl in h17.
   assert (h21:h19 = h6). apply proof_irrelevance.
   assert (h22:h20 = h7). apply proof_irrelevance.
   assert (h23:h8 = h9). apply proof_irrelevance.
   rewrite h21, h22, h23 in h17. rewrite h23.
   assumption.
   constructor; auto.
   apply two_ops_imp_times_p; auto.
   red.
   unfold directed_ba_p. simpl.
   pose proof (directed_one_compat h1 h2) as h7. simpl in h7.
   specialize (h7 _ h3).
   destruct h7 as [Cp [h8 [h9 [h10 h11]]]].
   rewrite <- h11.
   apply in_ba_p_ens_one.
   red.
   unfold directed_ba_p. simpl. 
   pose proof (directed_zero_compat h1 h2) as h7. simpl in h7. 
   specialize (h7 _ h3).
   destruct h7 as [Cp [h8 [h9 [h10 h11]]]].
   rewrite <- h11.
   apply in_ba_p_ens_zero.
exists h5.
apply bc_inj_p.
simpl.
assert (h6:A_p T (Bc_p T Bp) = A_p T (Bc_p' T (directed_ba_p h1 h2) (ba_p_ens Bp) h4 h5)).
  simpl. reflexivity.
assert (h7: (sig_set_eq (A_p T (Bc_p T Bp)) (ba_p_ens Bp) h6) = eq_refl).
  apply proof_irrelevance.
apply (bconst_ext_p _ _ h6); simpl. 
unfold SubBtype_p.
rewrite h7. rewrite transfer_dep_r_eq_refl.
rewrite und_set_p. simpl. unfold Btype_p.
reflexivity.
apply functional_extensionality.
intro x. apply functional_extensionality. intro y.
rewrite transfer_dep_r_fun2_eq. simpl.
rewrite h7.
rewrite transfer_r_eq_refl. 
destruct x as [x h8], y as [y h9].
simpl.
pose proof (directed_plus_compat h1  (exist (Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T))))) x
              (h4 x h8))
           (exist (Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T))))) y
              (h4 y h9))) as h10.
simpl in h10.
specialize (h10 _ _ h8 h9 h3 h3).
destruct h10 as [Cp [h8' [h9' [h10' [h11' [h12' [h13' [h14' [h15' h16']]]]]]]]].   
apply proj1_sig_injective. simpl.
rewrite h16'.
pose proof (ba_p_subst_plus _ _ h13' _ _ h8 h9) as h17.
assert (h18:Ensembles.In
                    (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h8' h10')) x).
  rewrite <- h13'; auto.
assert (h19 : Ensembles.In
                    (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h8' h10')) y).
  rewrite <- h13'; auto.
specialize (h17 h18 h19).
simpl in h17. rewrite h17 at 1.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.

apply functional_extensionality.
intro x. apply functional_extensionality. intro y.
rewrite transfer_dep_r_fun2_eq. simpl.
rewrite h7.
rewrite transfer_r_eq_refl. 
destruct x as [x h8], y as [y h9].
simpl.
pose proof (directed_times_compat h1  (exist (Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T))))) x
              (h4 x h8))
           (exist (Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T))))) y
              (h4 y h9))) as h10.
simpl in h10.
specialize (h10 _ _ h8 h9 h3 h3).
destruct h10 as [Cp [h8' [h9' [h10' [h11' [h12' [h13' [h14' [h15' h16']]]]]]]]].   
apply proj1_sig_injective. simpl.
rewrite h16'.
pose proof (ba_p_subst_times _ _ h13' _ _ h8 h9) as h17.
assert (h18:Ensembles.In
                    (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h8' h10')) x).
  rewrite <- h13'; auto.
assert (h19 : Ensembles.In
                    (ba_p_ens (Subalg_p Cp (ba_p_ens Bp) h8' h10')) y).
  rewrite <- h13'; auto.
specialize (h17 h18 h19).
simpl in h17. rewrite h17 at 1.
f_equal. f_equal. apply proj1_sig_injective. simpl. reflexivity.
apply proj1_sig_injective. simpl. reflexivity.
rewrite h7. rewrite transfer_dep_r_eq_refl.
apply proj1_sig_injective. simpl.
pose proof (directed_one_compat h1 h2 _ h3) as h8. simpl in h8.
destruct h8 as [Cp [h8 [h9 [h10 h11]]]].
assumption.
rewrite h7. rewrite transfer_dep_r_eq_refl.
apply proj1_sig_injective. simpl.
pose proof (directed_zero_compat h1 h2 _ h3) as h8. simpl in h8.
destruct h8 as [Cp [h8 [h9 [h10 h11]]]].
assumption.

apply functional_extensionality. intro x.
rewrite transfer_dep_r_fun1_eq. simpl.
rewrite h7.
rewrite transfer_r_eq_refl. 
destruct x as [x h8].
simpl. 
pose proof (directed_comp_compat h1  (exist (Ensembles.In (FamilyUnion (Im S (ba_p_ens (T:=T))))) x
              (h4 x h8))) as h10.           
simpl in h10.
specialize (h10 _ h8 h3).
destruct h10 as [Cp [h8' [h9' [h10' h11']]]].
apply proj1_sig_injective. simpl.
rewrite h11'.
f_equal.
Qed.
End DirectedFamilyOfAlgebras.

Section DirectedFamilyOfAlgebras'.


 
Lemma subalg_of_p_directed_ba_p_ba_to_ba_p :
  forall {B:Bool_Alg}
         (S:fam_ba_p (bt B)) (pfd:directed _ S)
         (pfinh:Inhabited S),
  fam_ba_p_bt_p_compat S (ba_to_ba_p B) ->
    subalg_of_p (directed_ba_p (bt B) pfd pfinh) (ba_to_ba_p B).
intros B S h1 h2 h0.
red.
assert (h3: Included (ba_p_ens (directed_ba_p (bt B) h1 h2))
               (ba_p_ens (ba_to_ba_p B))).
  red. intros x h3.
  unfold ba_p_ens, ba_to_ba_p. simpl. 
  apply Full_intro.
exists h3.
assert (h4: alg_closed_p (Bp:=ba_to_ba_p B) (ba_p_ens (directed_ba_p (bt B) h1 h2)) h3).   
  simpl.
  unfold ba_p_ens, directed_ba_p. simpl.
  constructor. 

  intros x y.  
  unfold SubBtype_p in x, y.
  destruct x as [x h4], y as [y h5].
  unfold Plus_closed_sub_p, Bplus_sub_p. simpl.
  assert (h6: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
              (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
              eq_refl
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
              (Morphisms.eq_proper_proxy Type
                 (sig_set
                    (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
              eq_refl = eq_refl). apply proof_irrelevance. 
  rewrite h6 at 1. clear h6. rewrite transfer_dep_eq_refl. simpl.
  unfold Bplus_sub. simpl. 
  red in h0.
  pose proof (directed_plus_compat _ h1 (exist _ _ h4) (exist _ _ h5)) as hd.
  destruct h4 as [Ap x h4a h4b], h5 as [Bp y h5a h5b].
  destruct h4a as [Ap h4a], h5a as [Bp h5a]. subst.
  specialize (hd _ _ h4b h5b h4a h5a).
  simpl in hd.
    destruct hd as [Cp [h15 [h16 [h17 [h18 [h19 [h20 [h21 [h22 h23]]]]]]]]]. 
    simpl in h23.
    simpl in h19.
    rewrite h23 in h19. clear h23.
  specialize (h0 _ h22).
  red in h0.
  destruct h0 as [h0a [h0b h0c]].
  assert (h28:Ensembles.In
                    (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Cp) h0a h0b))
                    x).
    rewrite <- h0c. apply (h15 x h4b).
  assert (h29 : Ensembles.In
                    (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Cp) h0a h0b))
                    y).
    rewrite <- h0c. apply (h16 y h5b).
  pose proof (ba_p_subst_plus _ _ h0c _ _ (h15 x h4b) (h16 y h5b) h28 h29) as h27.
  rewrite h27 in h19.
  simpl in h19.
  assert (h30: (Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
                   (SubBtype B (Full_set (bt B)))
                   (SubBtype B (Full_set (bt B))) eq_refl
                   (sig_set
                      (fun a : Btype (Bc B) =>
                       Ensembles.In (Full_set (bt B)) a))
                   (sig_set
                      (fun a : Btype (Bc B) =>
                       Ensembles.In (Full_set (bt B)) a))
                   (Morphisms.eq_proper_proxy Type
                      (sig_set
                         (fun a : Btype (Bc B) =>
                          Ensembles.In (Full_set (bt B)) a))) eq_refl) = eq_refl).  apply proof_irrelevance.
rewrite h30 in h19 at 1.
rewrite transfer_dep_eq_refl in h19.
simpl in h19.
unfold Bplus_sub in h19. simpl in h19.
apply family_union_intro with (ba_p_ens Cp).
apply Im_intro with Cp; auto.
assumption.

  intros x y.  
  unfold SubBtype_p in x, y.
  destruct x as [x h4], y as [y h5].
  unfold Times_closed_sub_p, Btimes_sub_p. simpl.
  assert (h6: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
              (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
              eq_refl
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
              (Morphisms.eq_proper_proxy Type
                 (sig_set
                    (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
              eq_refl = eq_refl). apply proof_irrelevance. 
  rewrite h6 at 1. clear h6. rewrite transfer_dep_eq_refl. simpl.
  unfold Btimes_sub. simpl. 
  red in h0.
  pose proof (directed_times_compat _ h1 (exist _ _ h4) (exist _ _ h5)) as hd.
  destruct h4 as [Ap x h4a h4b], h5 as [Bp y h5a h5b].
  destruct h4a as [Ap h4a], h5a as [Bp h5a]. subst.
  specialize (hd _ _ h4b h5b h4a h5a).
  simpl in hd.
    destruct hd as [Cp [h15 [h16 [h17 [h18 [h19 [h20 [h21 [h22 h23]]]]]]]]]. 
    simpl in h23.
    simpl in h19.
    rewrite h23 in h19. clear h23.
  specialize (h0 _ h22).
  red in h0.
  destruct h0 as [h0a [h0b h0c]].
  assert (h28:Ensembles.In
                    (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Cp) h0a h0b))
                    x).
    rewrite <- h0c. apply (h15 x h4b).
  assert (h29 : Ensembles.In
                    (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Cp) h0a h0b))
                    y).
    rewrite <- h0c. apply (h16 y h5b).
  pose proof (ba_p_subst_times _ _ h0c _ _ (h15 x h4b) (h16 y h5b) h28 h29) as h27.
  rewrite h27 in h19.
  simpl in h19.
  assert (h30: (Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
                   (SubBtype B (Full_set (bt B)))
                   (SubBtype B (Full_set (bt B))) eq_refl
                   (sig_set
                      (fun a : Btype (Bc B) =>
                       Ensembles.In (Full_set (bt B)) a))
                   (sig_set
                      (fun a : Btype (Bc B) =>
                       Ensembles.In (Full_set (bt B)) a))
                   (Morphisms.eq_proper_proxy Type
                      (sig_set
                         (fun a : Btype (Bc B) =>
                          Ensembles.In (Full_set (bt B)) a))) eq_refl) = eq_refl).  apply proof_irrelevance.
rewrite h30 in h19 at 1.
rewrite transfer_dep_eq_refl in h19.
simpl in h19.
unfold Btimes_sub in h19. simpl in h19.
apply family_union_intro with (ba_p_ens Cp).
apply Im_intro with Cp; auto.
assumption.

red.
pose proof (directed_one_compat _ h1 h2) as hd.
simpl in hd.
destruct h2 as [C h2a].
specialize (hd _ h2a).
destruct hd as [Ap [h4 [h5 [h6 h7]]]]. 
red in h0.
specialize (h0 _ h4).
destruct h0 as [h0a [h0b h0c]].
pose proof (ba_p_subst_one _ _ h0c) as h8.
rewrite <-  h8 at 1. 
apply family_union_intro with (ba_p_ens Ap).
apply Im_intro with Ap; auto.
apply proj2_sig.

red.
pose proof (directed_zero_compat _ h1 h2) as hd.
simpl in hd.
destruct h2 as [C h2a].
specialize (hd _ h2a).
destruct hd as [Ap [h4 [h5 [h6 h7]]]]. 
red in h0.
specialize (h0 _ h4).
destruct h0 as [h0a [h0b h0c]].
pose proof (ba_p_subst_zero _ _ h0c) as h8.
rewrite <-  h8 at 1. 
apply family_union_intro with (ba_p_ens Ap).
apply Im_intro with Ap; auto.
apply proj2_sig.

red. intros x.
unfold SubBtype_p in x.
destruct x as [x h4].
unfold Comp_closed_sub_p, Bcomp_sub_p. simpl.
assert (h6: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
                                                    (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
                                                    eq_refl
                                                    (sig_set
                                                       (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
                                                    (sig_set
                                                       (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
                                                    (Morphisms.eq_proper_proxy Type
                                                                               (sig_set
                                                                                  (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
                                                    eq_refl = eq_refl). apply proof_irrelevance. 
rewrite h6 at 1. clear h6. rewrite transfer_dep_eq_refl. simpl.
unfold Bcomp_sub. simpl. 
red in h0.
pose proof (directed_comp_compat _ h1 (exist _ _ h4)) as hd.
destruct h4 as [Ap x h4a h4b].
destruct h4a as [Ap h4a]. subst.
specialize (hd Ap h4b h4a).
simpl in hd.
destruct hd as [Cp [h15 [h16 [h17 h18]]]]. 
rewrite h18 in h15. clear h18.
specialize (h0 _ h17).
destruct h0 as [h0a [h0b h0c]].
destruct h16 as [h16a [h16b h16c]].
assert (h19: Ensembles.In
                   (ba_p_ens (Subalg_p Cp (ba_p_ens Ap) h16a h16b)) x).
  rewrite <- h16c. assumption.
pose proof (ba_p_subst_comp _ _ h16c _ h4b h19) as h18.  
apply family_union_intro with (ba_p_ens Cp).
apply Im_intro with Cp; auto. 
unfold ba_p_ens, Subalg_p in h18. simpl in h18.
unfold ba_p_ens, Subalg_p in h19. simpl in h19.
pose proof (h16a _ h19) as h20.
destruct h0b as [h0d h0e h0f h0g h0h].
red in h0h.
pose proof (h0h (exist _ _ h20)) as h21.
unfold Bcomp_sub_p in h21. simpl in h21.
assert (h22:(Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
                   (SubBtype B (Full_set (bt B)))
                   (SubBtype B (Full_set (bt B))) eq_refl
                   (sig_set
                      (fun a : Btype (Bc B) =>
                       Ensembles.In (Full_set (bt B)) a))
                   (sig_set
                      (fun a : Btype (Bc B) =>
                       Ensembles.In (Full_set (bt B)) a))
                   (Morphisms.eq_proper_proxy Type
                      (sig_set
                         (fun a : Btype (Bc B) =>
                          Ensembles.In (Full_set (bt B)) a))) eq_refl) = eq_refl). apply proof_irrelevance.
rewrite h22 in h21 at 1. rewrite transfer_dep_eq_refl in h21.
simpl in h21.
unfold Bcomp_sub in h21. simpl in h21.
assumption.
exists h4.  
apply bc_inj_p.
assert (h6: A_p (bt B) (Bc_p (bt B) (directed_ba_p (bt B) h1 h2)) =
              A_p (bt B)
                (Bc_p (bt B)
                   (Subalg_p (ba_to_ba_p B)
                      (ba_p_ens (directed_ba_p (bt B) h1 h2)) h3 h4))).
  unfold Subalg_p, directed_ba_p, ba_p_ens. simpl.
  reflexivity.
apply (bconst_ext_p  (Bc_p (bt B) (directed_ba_p (bt B) h1 h2)) (Bc_p (bt B)
     (Subalg_p (ba_to_ba_p B) (ba_p_ens (directed_ba_p (bt B) h1 h2)) h3 h4)) h6).
apply Extensionality_Ensembles.
red. split.
red. intros x h7.
rewrite <- (transfer_r_undoes_transfer  (sig_set_eq (A_p (bt B) (Bc_p (bt B) (directed_ba_p (bt B) h1 h2)))
           (A_p (bt B)
              (Bc_p (bt B)
                 (Subalg_p (ba_to_ba_p B)
                    (ba_p_ens (directed_ba_p (bt B) h1 h2)) h3 h4))) h6) x).
rewrite <- transfer_in_r at 1.
constructor.


red. intros x h7. constructor.
apply functional_extensionality. intro x. apply functional_extensionality. intro y. 
rewrite <- transfer_fun2_r_transfer_dep_r_compat'.  
rewrite <- (transfer_r_undoes_transfer 
 (sig_set_eq (A_p (bt B) (Bc_p (bt B) (directed_ba_p (bt B) h1 h2)))
        (A_p (bt B)
           (Bc_p (bt B)
              (Subalg_p (ba_to_ba_p B)
                 (ba_p_ens (directed_ba_p (bt B) h1 h2)) h3 h4))) h6) x) at 2.
rewrite <- (transfer_r_undoes_transfer 
 (sig_set_eq (A_p (bt B) (Bc_p (bt B) (directed_ba_p (bt B) h1 h2)))
        (A_p (bt B)
           (Bc_p (bt B)
              (Subalg_p (ba_to_ba_p B)
                 (ba_p_ens (directed_ba_p (bt B) h1 h2)) h3 h4))) h6) y) at 2.
rewrite transfer_fun2_r_compat'.
simpl.
rewrite (transfer_r_sig_set_eq _ _ h6).
simpl.
apply proj1_sig_injective.
simpl.
do 2 rewrite (transfer_sig_set_eq _ _ h6) at 1.
simpl.
assert (h7:Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl). apply proof_irrelevance.
rewrite h7 at 1. rewrite transfer_dep_eq_refl.
simpl.
unfold Bplus_sub. simpl.
pose proof (directed_plus_compat _ h1 x y) as h8.
pose proof (proj2_sig x) as h11. pose proof (proj2_sig y) as h12.
unfold Btype_p, Bc_p, directed_ba_p in x, y.
simpl in h11, h12.
inversion h11 as [E a h13 h14]. subst.
inversion h12 as [F a h15 h16]. subst.
destruct h13 as [Ep h13a E]. destruct h15 as [Fp h15a F].
subst.
specialize (h8 _ _ h14 h16 h13a h15a).
destruct h8 as [Dp [h17 [h18 [h19 [h20 [h21 [h22 [h23 [h24 h25]]]]]]]]].
rewrite h25 at 1.
specialize (h0 _ h24).
destruct h0 as [h0a [h0b h0c]].
assert (h27:Ensembles.In
                    (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Dp) h0a h0b))
                    (proj1_sig x)).
  rewrite <- h0c. apply (h17 (proj1_sig x) h14).
assert (h28: Ensembles.In
                    (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Dp) h0a h0b))
                    (proj1_sig y)).
  rewrite <- h0c. apply (h18 (proj1_sig y) h16).
pose proof (ba_p_subst_plus _ _ h0c _ _ (h17 (proj1_sig x ) h14) (h18 (proj1_sig y) h16) h27 h28) as h26.
rewrite h26 at 1.
simpl.
assert (h29:Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl). apply proof_irrelevance.
rewrite h29 at 1.
rewrite transfer_dep_eq_refl. simpl.
unfold Bplus_sub. simpl.
reflexivity.

apply functional_extensionality. intro x. apply functional_extensionality. intro y. 
rewrite <- transfer_fun2_r_transfer_dep_r_compat'.  
rewrite <- (transfer_r_undoes_transfer 
 (sig_set_eq (A_p (bt B) (Bc_p (bt B) (directed_ba_p (bt B) h1 h2)))
        (A_p (bt B)
           (Bc_p (bt B)
              (Subalg_p (ba_to_ba_p B)
                 (ba_p_ens (directed_ba_p (bt B) h1 h2)) h3 h4))) h6) x) at 2.
rewrite <- (transfer_r_undoes_transfer 
 (sig_set_eq (A_p (bt B) (Bc_p (bt B) (directed_ba_p (bt B) h1 h2)))
        (A_p (bt B)
           (Bc_p (bt B)
              (Subalg_p (ba_to_ba_p B)
                 (ba_p_ens (directed_ba_p (bt B) h1 h2)) h3 h4))) h6) y) at 2.
rewrite transfer_fun2_r_compat'.
simpl.
rewrite (transfer_r_sig_set_eq _ _ h6).
simpl.
apply proj1_sig_injective.
simpl.
do 2 rewrite (transfer_sig_set_eq _ _ h6) at 1.
simpl.
assert (h7:Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl). apply proof_irrelevance.
rewrite h7 at 1. rewrite transfer_dep_eq_refl.
simpl.
unfold Btimes_sub. simpl.
pose proof (directed_times_compat _ h1 x y) as h8.
pose proof (proj2_sig x) as h11. pose proof (proj2_sig y) as h12.
unfold Btype_p, Bc_p, directed_ba_p in x, y.
simpl in h11, h12.
inversion h11 as [E a h13 h14]. subst.
inversion h12 as [F a h15 h16]. subst.
destruct h13 as [Ep h13a E]. destruct h15 as [Fp h15a F].
subst.
specialize (h8 _ _ h14 h16 h13a h15a).
destruct h8 as [Dp [h17 [h18 [h19 [h20 [h21 [h22 [h23 [h24 h25]]]]]]]]].
rewrite h25 at 1.
specialize (h0 _ h24).
destruct h0 as [h0a [h0b h0c]].
assert (h27:Ensembles.In
                    (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Dp) h0a h0b))
                    (proj1_sig x)).
  rewrite <- h0c. apply (h17 (proj1_sig x) h14).
assert (h28: Ensembles.In
                    (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Dp) h0a h0b))
                    (proj1_sig y)).
  rewrite <- h0c. apply (h18 (proj1_sig y) h16).
pose proof (ba_p_subst_times _ _ h0c _ _ (h17 (proj1_sig x ) h14) (h18 (proj1_sig y) h16) h27 h28) as h26.
rewrite h26 at 1.
simpl.
assert (h29:Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl). apply proof_irrelevance.
rewrite h29 at 1.
rewrite transfer_dep_eq_refl. simpl.
unfold Bplus_sub. simpl.
reflexivity.
  
rewrite (transfer_dep_r_id_transfer_r_compat _ _ 
                                             (sig_set_eq (A_p (bt B) (Bc_p (bt B) (directed_ba_p (bt B) h1 h2)))
                                                         (A_p (bt B)
           (Bc_p (bt B)
                 (Subalg_p (ba_to_ba_p B)
                           (ba_p_ens (directed_ba_p (bt B) h1 h2)) h3 h4))) h6)).
rewrite (transfer_r_sig_set_eq _ _ h6  (sig_set_eq (A_p (bt B) (Bc_p (bt B) (directed_ba_p (bt B) h1 h2)))
                                                         (A_p (bt B)
           (Bc_p (bt B)
                 (Subalg_p (ba_to_ba_p B)
                           (ba_p_ens (directed_ba_p (bt B) h1 h2)) h3 h4))) h6)). 
apply proj1_sig_injective.
simpl.
 
assert (h7: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl). apply proof_irrelevance.
rewrite h7 at 1.

rewrite transfer_eq_refl. simpl. 

pose proof (directed_one_compat _ h1 h2) as h8. 
simpl in h8.
destruct h2 as [Ap h2]. clear h7.   
pose proof (h8 _ h2) as h8'. clear h8.
destruct h8' as [Cp [h9 [h10 [h11 h12]]]].
rewrite <- h12 at 1. 


 
specialize (h0 _ h2).
destruct h0 as [h0a [h0b h0c]]. 
pose proof (ba_p_subst_one _ _ h0c) as h13.
rewrite h13 at 1.
simpl. 
assert (h14:Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl).
  apply proof_irrelevance.
rewrite h14 at 1. clear h14.

rewrite transfer_eq_refl.
simpl.
reflexivity.


rewrite (transfer_dep_r_id_transfer_r_compat _ _ 
                                             (sig_set_eq (A_p (bt B) (Bc_p (bt B) (directed_ba_p (bt B) h1 h2)))
                                                         (A_p (bt B)
           (Bc_p (bt B)
                 (Subalg_p (ba_to_ba_p B)
                           (ba_p_ens (directed_ba_p (bt B) h1 h2)) h3 h4))) h6)).
rewrite (transfer_r_sig_set_eq _ _ h6).
apply proj1_sig_injective.
simpl.
assert (h7: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl). apply proof_irrelevance.
rewrite h7 at 1.
rewrite transfer_eq_refl. simpl.
pose proof (directed_zero_compat _ h1 h2) as h8.
simpl in h8.
destruct h2 as [Ap h2]. clear h7. 
pose proof (h8 _ h2) as h8'.
destruct h8' as [Cp [h9 [h10 [h11 h12]]]].
rewrite <- h12 at 1. 

specialize (h0 _ h2).
destruct h0 as [h0a [h0b h0c]]. 
pose proof (ba_p_subst_zero _ _ h0c) as h13.
rewrite h13 at 1.
simpl.
assert (h14:Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl).
  apply proof_irrelevance.
rewrite h14 at 1.
rewrite transfer_eq_refl.
simpl.
reflexivity.


apply functional_extensionality. intro x.
rewrite <- transfer_fun_r_transfer_dep_r_compat'.  
rewrite <- (transfer_r_undoes_transfer 
 (sig_set_eq (A_p (bt B) (Bc_p (bt B) (directed_ba_p (bt B) h1 h2)))
        (A_p (bt B)
           (Bc_p (bt B)
              (Subalg_p (ba_to_ba_p B)
                 (ba_p_ens (directed_ba_p (bt B) h1 h2)) h3 h4))) h6) x) at 2.
rewrite transfer_fun_r_compat'.
simpl.
rewrite (transfer_r_sig_set_eq _ _ h6).
simpl.
apply proj1_sig_injective.
simpl.
rewrite (transfer_sig_set_eq _ _ h6) at 1.
simpl.
assert (h7:Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl). apply proof_irrelevance.
rewrite h7 at 1. rewrite transfer_dep_eq_refl.
simpl.
unfold Btimes_sub. simpl.
pose proof (directed_comp_compat _ h1 x) as h8.
pose proof (proj2_sig x) as h11. 
unfold Btype_p, Bc_p, directed_ba_p in x.
simpl in h11.
inversion h11 as [E a h13 h14]. subst.
destruct h13 as [Ep h13a E]. 
subst. 
specialize (h8 Ep h14 h13a).
destruct h8 as [Dp [h17 [h18 [h19 h20]]]].
rewrite h20 at 1.
specialize (h0 _ h19).
destruct h0 as [h0a [h0b h0c]].
unfold Bcomp_sub. simpl.
red in h18.
destruct h18 as [h18a [h18b h18c]].

assert (h27: Ensembles.In
                   (ba_p_ens (Subalg_p Dp (ba_p_ens Ep) h18a h18b))
                   (proj1_sig x)).
  rewrite <- h18c. assumption.
pose proof (ba_p_subst_comp _ _ h18c _ h14 h27) as h26.
rewrite h26 at 1.
simpl.
assert (h29: Ensembles.In
                   (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Dp) h0a h0b))
                   (proj1_sig x)).
  rewrite <- h0c. apply (h18a (proj1_sig x) h27).
pose proof (ba_p_subst_comp _ _ h0c _ (h18a (proj1_sig x) h27) h29) as h28. 
rewrite h28 at 1.
simpl.
assert (h30:Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl). apply proof_irrelevance.
rewrite h30 at 1.
rewrite transfer_dep_eq_refl. simpl.
unfold Bcomp_sub. simpl.
reflexivity.
Qed.



   
End DirectedFamilyOfAlgebras'.


Arguments directed [T] _.
Arguments directed_bcp [T] [S] _ _.
Arguments directed_ba_p [T] [S] _ _.
