(* Copyright (C) 2014, Daniel Wyckoff *)
(*This file is part of BooleanAlgebrasIntro2.

BooleanAlgebrasIntro2 is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

BooleanAlgebrasIntro2 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with BooleanAlgebrasIntro2.  If not, see <http://www.gnu.org/licenses/>.*)

 
Require Import LogicUtilities.
Require Import TypeUtilities.
Require Import EnsemblesImplicit.
Require Import Finite_sets_facts. 
Require Export List.
Require Import FunctionProperties.
Require Import Constructive_sets.
Require Import DecidableDec.
Require Import FunctionalExtensionality.
Require Import SetUtilities.
Require Import ArithUtilities.
Require Import NaryFunctions.
Require Import ProofIrrelevance.  
Require Import Description.
 

Lemma in_not_nil : forall {T:Type} (l:list T) (x:T), 
                     In x l -> l <> nil.
intros; intro; subst; contradiction.
Qed.

Lemma not_nil_cons : forall {T:Type} (l:list T), 
                       l <> nil -> exists (a:T) (l':list T),
                                     l = a::l'.
intros T l h1.
destruct l as [|a l'].  contradict h1. auto.
exists a.  exists l'. auto.
Qed.

Lemma all_singleton_in : forall {T:Type} (l:list T) (a:T),
                     (l <> nil) ->
                     (forall x : T, In x l -> x = a) ->
                     In a l.
intros T l a h1.
destruct l as [|b l']. contradict h1; auto.
intro h2.
pose proof (in_eq b l') as h3.
specialize (h2 _ h3). subst. auto.
Qed.

Lemma in_list2_comm : forall {T:Type} (l:list T) (a x y:T),
                        In a (x::y::l) -> In a (y::x::l).
intros T l a x y h1.
destruct h1 as [h1a | h1b].
subst.
right. left. auto.
destruct h1b as [h2a | h2b].
subst. left. auto.
right. right. auto.
Qed.

Lemma remove_neq_cons : forall {T:Type} (l:list T) (a b:T),
                          a <> b -> 
                          remove eq_dec a (b::l) =
                          b :: remove eq_dec a l.
intros T l a b h1.
simpl.
destruct (eq_dec a b). contradiction. auto.
Qed.


Lemma remove_a_from_2cons_nil : 
  forall {T:Type} (l:list T) (a x y:T),
         remove eq_dec a (x::y::l) = nil ->
         remove eq_dec a (y::x::l) = nil.
  intros T l a x y h1.
  simpl in h1. simpl.
  destruct (eq_dec a x); destruct (eq_dec a y); auto; try discriminate.
Qed.

Lemma remove_not_in' : forall {T:Type} (l:list T) (a:T),
                        ~ In a l -> l = remove eq_dec a l.
intros T l a h1.
induction l. simpl. auto.
simpl.
destruct (eq_dec a a0).
subst.
pose proof (in_eq a0 l). contradiction.
f_equal.
simpl in h1.
assert (h2: ~ In a l). tauto.
apply IHl; auto.
Qed.



Lemma remove2_eq : forall {T:Type} (l:list T) (a b:T),
                       remove eq_dec a (remove eq_dec b l) =
                       remove eq_dec b (remove eq_dec a l).
intros T l a b.
induction l as [|c l h1].
simpl. auto.
simpl.
destruct (eq_dec b c) as [hc | hb]; destruct (eq_dec a c) as [hc' | ha].
subst; auto.
subst.
simpl.
destruct (eq_dec c c) as [h3 | h4]. auto.
contradict h4. auto.
subst. simpl.
destruct (eq_dec c c) as [h3 | h4]. auto.
contradict h4. auto.
simpl.
destruct (eq_dec a c) as [hc | ha']; destruct (eq_dec b c) as [hc' | hb']; subst; try contradiction; auto.
contradict ha. auto. contradict hb; auto.
f_equal. auto.
Qed.


Lemma in_remove_neq' : forall {T:Type} (l:list T) (x:T),
                        In x l -> l <> remove eq_dec x l.
intros T l x h1 h2.
rewrite h2 in h1.
pose proof (remove_In eq_dec l x).
contradiction.
Qed.


Lemma remove_a_in_eq : 
  forall {T:Type} (l:list T) (a x:T),
    In x l -> x <> a ->
    In x (remove eq_dec a l).
intros T l a x h1 h3.
destruct (in_dec eq_dec a l) as [h2 | h0].
induction l as [|b l h7].
contradiction.
simpl.
destruct (eq_dec a b) as [h4 | h5]. subst.
destruct h1 as [h1a | h1b]. 
symmetry in h1a.
contradiction.
destruct (in_dec eq_dec b l) as [h8 | h9].
apply h7; auto.
pose proof (remove_not_in' _ _ h9) as h10.
rewrite <- h10. auto.
destruct h1 as [h1a | h1b]; destruct h2 as [h2a | h2b]. 
subst. contradiction.
subst. left. auto.
subst.
contradict h5. auto.
pose proof (h7 h1b h2b) as h8.
right. auto.
rewrite <- remove_not_in'; assumption.
Qed.

(*Kinda weak, but it's in use.*)
Lemma in_remove_neq_in_l : 
  forall {T:Type} (l:list T) (a b:T),
    In a (remove eq_dec b l) ->
    a <> b -> In a l.
intros T l a b h1 h2.
induction l as [|c l h3].
simpl in h1. auto.
simpl in h1.
destruct (eq_dec b c). subst.
specialize (h3 h1). right; auto.
destruct h1 as [h1a | h1b]. subst. left. auto.
specialize (h3 h1b). right. auto.
Qed.

Lemma in_remove_inv : 
  forall {T:Type} (l:list T) (a b:T),
    In a (remove eq_dec b l) ->
    In a l /\ a <> b.
intros T l a b h1. 
pose proof remove_In.
destruct (eq_dec a b) as [h2 | h3]. subst.
pose proof (remove_In eq_dec l b). contradiction.
split; auto.
eapply in_remove_neq_in_l; auto. apply h1. assumption.
Qed.

Lemma in_remove_iff : 
  forall {T:Type} (l:list T) (a b:T),
    In a (remove eq_dec b l) <->
    In a l /\ a <> b.
intros T l a b. split. intro h1.
apply in_remove_inv; auto.
intro h1. destruct h1.
apply remove_a_in_eq; auto.
Qed.

Lemma nin_nin_remove : forall {T:Type} (l:list T) (x a:T),
                         ~In x l -> ~In x (remove eq_dec a l).
intros T l x a h1.
induction l as [|b l h2]. simpl. auto.
simpl in h1. 
assert (h3:b <> x /\ ~ In x l). tauto.
destruct h3 as [h3a h3b].
pose proof (h2 h3b) as h4.
simpl.
destruct (eq_dec a b) as [h5 | h6]. subst. auto.
simpl. tauto.
Qed.


Lemma nin_remove_eq_nin : 
  forall {T:Type} (l:list T) (a b:T),
    ~ In a (remove eq_dec b l) ->
    a = b \/ ~ In a l.
intros T l a b h1.
induction l as [|c l h2].
right. intro; auto.
simpl in h1.
destruct (eq_dec b c) as [h3 | h4]. subst.
specialize (h2 h1). simpl.
rewrite eq_sym_iff. rewrite eq_sym_iff.
tauto.
simpl in h1.
assert (h3:c <> a /\ ~ In a (remove eq_dec b l)). tauto.
destruct h3 as [h3l h3r].
specialize (h2 h3r).
destruct h2; [left; auto | idtac].
right. simpl. tauto.
Qed.

(*A list of all sequence lists obtained by choosing one element from each 
  list in [l], in the same order as the lists in [l]*)

Fixpoint list_of_lists_seqs {T:Type} (l:list (list T)) : list (list T) :=
  match l with 
  | nil => cons nil nil
  | al::l' => map (fun x:(T * (list T)) => cons (fst x) (snd x)) 
    (list_prod al (list_of_lists_seqs l'))
  end.

Fixpoint list_to_set {T:Type} (l:list T) : Ensemble T :=
  match l with
  | nil => Empty_set _
  | cons a l' => Add (list_to_set l') a
  end.

Lemma empty_set_nil : 
  forall {T:Type} (l:list T),
    list_to_set l = Empty_set _ -> l = nil.
intros T l h1.
induction l as [|a l]. reflexivity.
simpl in h1.
pose proof (Add_intro2 _ (list_to_set l) a) as h2.
rewrite h1 in h2.
contradiction.
Qed.

Fixpoint lrep {T:Type} (x:T) (n:nat) :=
  match n with
    | O => nil
    | S n => x :: lrep x n
  end.
    
Lemma singleton_all_a :
  forall {T:Type} (l:list T) (a:T),
    list_to_set l = Singleton a ->
    forall x:T, In x l -> x = a.
  intros T l a h1 x h2.
  induction l as [|a' l h3].
  (* nil *)
  simpl in h1.
  pose proof (In_singleton _ a) as h4.
  rewrite <- h1 in h4.
  contradiction.
  (* cons *)
  simpl in h1.
  simpl in h2.
  pose proof (Add_intro2 _ (list_to_set l) a') as h5.
  rewrite h1 in h5.
  destruct h5; subst.
  assert (h4:Included (list_to_set l) (Singleton a)).
    red. intros y h6.
    pose proof (Add_intro1 _ (list_to_set l) a y h6) as h7.
    rewrite h1 in h7. assumption.
  pose proof (singleton_inc _ _ h4)as h5.
  destruct h5 as [h5l | h5r].
    (*h5l*)
    pose proof (empty_set_nil _ h5l). subst.
    destruct h2; [subst | contradiction].
    reflexivity.
    (*h5r*)
    symmetry.
    destruct h2 as [|h2]; try assumption.
    symmetry.
    apply h3; assumption.
Qed.

Lemma list_to_set_finite : forall {T:Type} (l:list T), 
  Finite (list_to_set l).
intros T l.
induction l as [|a l h1].
simpl. auto with sets.
simpl. apply Add_preserves_Finite; assumption.
Qed.

Lemma list_to_set_in_iff : forall {T:Type} (l:list T) (x:T),
  In x l <-> Ensembles.In (list_to_set l) x.
intros T l.
induction l as [|a l h2].
(* nil *)
intros x. split. intro. contradiction. 
intro. contradiction.
(* cons *)
intro x. split.
  (* -> *)
  intro h1. simpl in h1. destruct h1 as [h1l | h1r]. 
    (* h1l *)
    subst. apply Add_intro2.
    (* h1r *)
    rewrite h2 in h1r.
    simpl. apply Add_intro1; assumption.
  (* <- *)
  intro h1. simpl in h1. inversion h1 as [y h3|y h4]. 
    (* h3 *)
    subst. rewrite <- h2 in h3. simpl. right; assumption.
    (* h4 *)
    subst. inversion h4. subst. simpl. left; reflexivity.
Qed.

Lemma list_to_set_bij : forall {T:Type} (l:list T), 
  exists f:{x:T | In x l} -> {x:T | Ensembles.In (list_to_set l) x}, 
  bijective f.
intros T l.
pose proof (list_to_set_in_iff l) as h1.
assert (h2:forall x : T, In x l -> Ensembles.In (list_to_set l) x).
  intros. rewrite <- h1; assumption.
exists (fun t:{x : T | In x l} => exist _ (proj1_sig t) (h2 _ (proj2_sig t))).
red. split.
(* inj *)
red. intros ? ? h3.  apply exist_injective in h3. 
apply proj1_sig_injective. assumption.
(* surj *)
red. intros y.
destruct (h1 (proj1_sig y)) as [? h1r].
specialize (h1r (proj2_sig y)).
exists (exist _ (proj1_sig y) h1r).
simpl. rewrite unfold_sig.
apply existTexist. apply subsetT_eq_compat. reflexivity.
Qed.

Lemma list_to_set_finitet : forall {T:Type} (l:list T), 
  FiniteT {x:T | In x l}.
intros T l.
pose proof (list_to_set_bij l) as h1.
pose proof (list_to_set_finite l) as h2.
pose proof (Finite_ens_type _ h2) as h3.
destruct h1 as [f h1].
pose proof (bijective_impl_invertible _ h1) as h4.
pose proof (invertible_impl_inv_invertible _ h4) as h5.
apply (bij_finite _ _ _ h3 h5).
Qed.

Lemma finite_set_list : forall {T:Type} (E:Ensemble T), Finite E ->
  exists (l:list T), E = list_to_set l.
intros T E h1.
induction h1 as [| E h2 h3 t h4].
(* nil *)
exists nil.
unfold list_to_set. reflexivity.
(* Add *)
destruct h3 as [l h5].
exists (t::l).
unfold list_to_set.
unfold list_to_set in h5.
rewrite <- h5.
reflexivity.
Qed.

Lemma add_cons_compat : forall {T:Type} (l:list T) (a:T),
                          Add (list_to_set l) a =
                          list_to_set (a::l).
intros. simpl. auto.
Qed.


Lemma subtract_remove_compat : forall {T:Type} (l:list T) (x:T), 
                              Subtract (list_to_set l) x =
                              list_to_set (remove eq_dec x l).
intros T l.
induction l.
(* nil *)
(* Mayve move to SetUtilities*)
intros x. simpl.
apply Extensionality_Ensembles. red. split.
red. intros y h1.
destruct h1; contradiction.
auto with sets.
(* cons *)
intro x. simpl.
destruct (eq_dec x a) as [h2 | h3].
subst.
destruct (in_dec eq_dec a l) as [h4 | h5].
rewrite list_to_set_in_iff in h4.
pose proof (sub_add_compat_in _ _ h4) as h6.
rewrite h6.
apply IHl.
rewrite <- (remove_not_in' l a).
rewrite list_to_set_in_iff in h5.
apply sub_add_compat_nin; auto. auto.
rewrite sub_add_comm.
rewrite IHl.
apply add_cons_compat.
apply neq_sym_iff; auto.
Qed.


Definition nat_seg_list (n:nat) := seq 0 n.

Definition map_sig {T:Type} (l:list T) : list {x:T | In x l}.
induction l as [|a l lsig].
refine nil.
pose (map (fun i:{x:T | In x l} => 
  exist (fun x:T => In x (a::l)) 
    (proj1_sig i) (in_cons a _ _ (proj2_sig i))) lsig) as lsig'.
pose (exist (fun x:T => In x (a::l)) a (in_eq a l)) as a'.
refine (a'::lsig').
Defined.

Lemma map_sig_in : forall {T:Type} (l:list T) (i:{x:T | In x l}), 
  In i (map_sig l).
intros T l i.
induction l as [|a l h1]. 
(* nil *)
pose proof (proj2_sig i) as h5. simpl in h5. contradiction.    
(* a :: l *)
simpl. 
pose proof (proj2_sig i) as h5.
simpl in h5.
  destruct h5 as [h5l | h5r].
  (* h5l *)
  left.
  rewrite unfold_sig.
  apply existTexist.
  apply subsetT_eq_compat. assumption.
  (* h5r *)
  right.
  rewrite in_map_iff.
  pose (exist (fun (x:T) => In x l) (proj1_sig i) h5r) as i'.
  exists i'.
  split.
  unfold i'.
  simpl. rewrite unfold_sig. apply existTexist. apply subsetT_eq_compat. 
  reflexivity.
  apply h1.
Qed.


(* Same as above *)
Lemma in_map_sig : forall {T:Type} (l:list T) (x:{t:T | In t l}),
                     In x (map_sig l) -> In (proj1_sig x) l.
intros; apply proj2_sig.
Qed.


Lemma map_map_sig_compat : forall {T:Type} (l:list T),
 map (fun x : {t : T | In t l} => proj1_sig x) (map_sig l) = l. 
intros T l.
induction l as [|a l h1].
simpl. reflexivity.
simpl.
rewrite map_map.
simpl.
rewrite h1.
reflexivity.
Qed.


Lemma cons_inj : forall {T:Type} (a1 a2:T) (l1 l2:list T), a1::l1 = a2::l2 ->
  a1 = a2 /\ l1 = l2.
intros ? ? ? ? ? h1.
inversion h1. split; trivial.
Qed.

Lemma cons_eq : forall {T:Type} (a1 a2:T) (l1 l2:list T), 
  a1::l1 = a2::l2 <->  a1 = a2 /\ l1 = l2.
intros T a1 a2 l1 l2.
split.
intro h1. apply cons_inj; assumption.
intro h1. destruct h1; subst. reflexivity.
Qed.



Lemma map_im_compat : forall {T U:Type} (f:T->U) (l:list T),
                        list_to_set (map f l) = Im (list_to_set l) f.
intros T U f l.
apply Extensionality_Ensembles.
red. split.
red.
intros y h1.
rewrite <- list_to_set_in_iff in h1.
rewrite in_map_iff in h1.
destruct h1 as [x h2].
destruct h2 as [h2l h2r].
apply Im_intro with x.
rewrite <- list_to_set_in_iff.
assumption. subst. reflexivity.
red.
intros x h1.
destruct h1 as [x h1]. subst.
rewrite <- list_to_set_in_iff.
rewrite <- list_to_set_in_iff in h1.
rewrite in_map_iff.
exists x. split; auto.
Qed.

Lemma in_list_power1 : forall {T1 T2:Type} {l1:list T1} {l2:list T2}
  (pl:list (T1 * T2)), In pl (list_power l1 l2) -> forall (pr:T1*T2),
  In pr pl -> In (fst pr) l1.
intros T1 T2 l1 l2. 
induction l1 as [|b l1 h4]. 
intros pl h1. 
simpl in h1.
destruct h1 as [h1l | h1r].
subst. contradiction. contradiction.
intros pl h1.
simpl in h1.
rewrite in_flat_map in h1.
destruct h1 as [pl' h5].
destruct h5 as [h5l h5r].
rewrite in_map_iff in h5r.
destruct h5r as [t2 h6].
destruct h6 as [h6l h6r].
specialize (h4 pl' h5l).
intros pr h7.
subst.
destruct h7 as [h7l | h7r].
subst. simpl. left. reflexivity.
specialize (h4 pr h7r).
right.
assumption.
Qed.

Lemma in_list_power2 : forall {T1 T2:Type} {l1:list T1} {l2:list T2}
  (pl:list (T1 * T2)), In pl (list_power l1 l2) -> forall (pr:T1*T2),
  In pr pl -> In (snd pr) l2.
intros T1 T2 l1 l2.
induction l1 as [|b l1 h4]. 
intros pl h1. 
simpl in h1.
intros pr h2.
destruct h1 as [h1l | h1r].
subst. contradiction. contradiction.
intros pl h2.
simpl in h2.
rewrite in_flat_map in h2.
destruct h2 as [pl' h3].
destruct h3 as [h3l h3r].
rewrite in_map_iff in h3r.
destruct h3r as [t2 h5].
destruct h5 as [h5l h5r].
subst.
intros pr h6.
destruct h6 as [h6l | h6r].
subst. simpl. assumption.
specialize (h4 _ h3l _ h6r).
assumption.
Qed.

Inductive functionally_paired_l {T U:Type} (la:list T)
          (lb:list U) (lp:list (T*U)) : Prop :=
  functional_pairs_intro :
    (forall (x:T),
      In x la -> (exists! y:U,
                  In y lb /\ In (x, y) lp)) ->
    (forall (pr:T*U), In pr lp -> In (fst pr) la /\
    In (snd pr) lb) ->
                  functionally_paired_l la lb lp.

Lemma fpl_in_dom :
  forall {T U:Type} (la:list T)
         (lb:list U) (lp:list (T*U)),
    functionally_paired_l la lb lp ->
    forall pr:T*U, In pr lp -> In (fst pr) la.
intros T U la lb lp h1 pr h2.
destruct h1 as [h1a h1b].
specialize (h1b _ h2).
destruct h1b; assumption.
Qed.

Lemma fpl_in_ran :
  forall {T U:Type} (la:list T)
         (lb:list U) (lp:list (T*U)),
    functionally_paired_l la lb lp ->
    forall pr:T*U, In pr lp -> In (snd pr) lb.
intros T U la lb lp h1 pr h2.
destruct h1 as [h1a h1b].
specialize (h1b _ h2).
destruct h1b; assumption.
Qed.

Lemma fpl_functional :
  forall {T U:Type} {la:list T} {lb:list U}
         {lp:list (T*U)},
    functionally_paired_l la lb lp ->
    forall (x:T) (y1 y2:U), In (x, y1) lp ->
                            In (x, y2) lp ->
                            y1 = y2.
intros T U la lb lp h1 x y1 y2 h2 h3.
inversion h1 as [h4 h5].
pose proof (h5 _ h2) as h6.
pose proof (h5 _ h3) as h7.
simpl in h6. simpl in h7.
destruct h6 as [h6a h6b].
destruct h7 as [h7a h7b].
pose proof (h4 _ h6a) as h8.
destruct h8 as [y h9].
red in h9.
destruct h9 as [h9a h9b].
pose proof (h9b _ (conj h6b h2)).
pose proof (h9b _ (conj h7b h3)).
subst.
assumption.
Qed.

Lemma fpl_empty1 : forall (T U:Type) (lb:list U),
                        functionally_paired_l (@nil T) lb nil. 
intros; constructor; intros; contradiction.
Qed.

Lemma fpl_empty1_s : forall {T U:Type} (lb:list U) (lp:list (T*U)),
                      functionally_paired_l (@nil T) lb lp ->
                      lp = nil.
intros T U lb lp h1.
destruct h1 as [h2 h3].
destruct lp as [|pr lp].
reflexivity.
pose proof (in_eq pr lp) as h4.
specialize (h3 _ h4).
destruct h3; contradiction.
Qed.

Lemma no_fpl_empty2 : forall {T U:Type} (la:list T)
                            (lp:list (T*U)),
                        functionally_paired_l la (@nil U) lp ->
                        la = nil.
intros T U la.
destruct la as [|a la].
intros; reflexivity.
intros lp h2.
destruct h2 as [h2 h3].
pose proof (in_eq a la) as h4.
specialize (h2 _ h4).
destruct h2 as [y h2].
red in h2.
destruct h2 as [[]]; contradiction.
Qed.

Lemma no_fpl_empty2' : forall {T U:Type} (la:list T)
                            (lp:list (T*U)),
                        functionally_paired_l la (@nil U) lp ->
                        lp = nil.
intros T U la lp.
destruct lp as [|pr lp].
intros; reflexivity.
intro h1.
pose proof (no_fpl_empty2 _ _ h1).
subst.
pose proof (fpl_empty1_s _ _ h1).
discriminate.
Qed.

Lemma fpl_app_ex : forall {T U:Type} {la:list T} {lb:list U}
                          {lp:list (T*U)},
                     functionally_paired_l la lb lp ->
                     forall (def:U) (a:T), exists! b:U,
                       if (in_dec eq_dec a la) then In (a, b) lp
                       else b = def.
intros T U la lb lp h1 def a.
inversion h1 as [h0 h2].
destruct (in_dec eq_dec a la) as [h3 | h4].
specialize (h0 _ h3).
destruct h0 as [b h0].
red in h0.
exists b. red. 
destruct h0 as [h0l h0r].
destruct h0l as [h0a h0b].
split.
assumption.
intros x' h4.
pose proof (@fpl_in_ran).
pose proof (fpl_in_ran _ _ _ h1 _ h4) as h5.
simpl in h5.
apply h0r. split; assumption.
exists def. red. split; auto.
Qed.

Definition fpl_app {T U:Type} {la:list T} {lb:list U}
           {lp:list (T*U)} (pf:functionally_paired_l la lb lp)
           (def:U) (a:T) : U.
pose proof (fpl_app_ex pf def a) as h1.
refine (proj1_sig (constructive_definite_description 
                      _ h1)). 
Defined.

Notation "pr 'l->' a" := (fpl_app (fst pr) (snd pr) a) (at level 20).

Lemma fpl_app_compat : forall {T U:Type} {la:list T} {lb:list U}
           {lp:list (T*U)} (pf:functionally_paired_l la lb lp)
           (a:T) (def:U),
                         if (in_dec eq_dec a la) then
                           In (a, (pf, def) l-> a) lp
                         else (pf, def) l-> a = def.
intros T U la lb lp pf a def.
unfold fpl_app.
destruct constructive_definite_description as [b h1].
simpl.
simpl in h1.
assumption.
Qed.


Lemma in_fpl_snd : 
  forall {T U:Type} (la:list T) (lb:list U)
         (lp:list (T*U)) (pf:functionally_paired_l la lb lp) 
         (pr:T*U) (def:U), In pr lp -> snd pr = (pf, def) l-> (fst pr).
intros T U la lb lp h1 pr def h2. 
pose proof (@fpl_app_compat).
pose proof (fpl_app_compat h1 (fst pr) def) as h3.
destruct in_dec as [h4 | h5].
rewrite (surjective_pairing pr) in h2.
pose proof (fpl_functional h1 _ _ _ h2 h3).
assumption.
pose proof (@fpl_in_dom).
pose proof (fpl_in_dom _ _ _ h1 _ h2).
contradiction.
Qed.



Definition im1l {T U V:Type} {la:list T}
           {lb:list U} {lc:list V} {lp:list (T*U*V)} 
           (pf:functionally_paired_l (list_prod la lb) lc lp) 
           (def:V) (x:T) :
  list V := map (fun y:U => (pf, def) l-> (x, y)) lb.

Definition im2l {T U V:Type} {la:list T}
           {lb:list U} {lc:list V} {lp:list (T*U*V)} 
           (pf:functionally_paired_l (list_prod la lb) lc lp) 
           (def:V) (y:U) :
  list V := map (fun x:T => (pf, def) l-> (x, y)) la.


(*When invoked as [list_singularize l nil], it returns
  the list without the duplicate entries, in the same order as
  each new element appearing starting backwards, i.e.
  list_singularize (3::4::2::4::2::3::3::3::nil) = 4::2::3.*)

Fixpoint list_singularize {T:Type} (l:list T) (acc:list T) :=
  match l with
    | nil => nil
    | a::l' => if (in_dec eq_dec a l') then
                 (list_singularize l' acc)
               else (a::(list_singularize l' (a::acc)))
  end.

Lemma list_singularize_not_in :
  forall {T:Type} (l:list T) (acc:list T),
    (forall (x:T), In x acc -> ~In x l) ->
    list_singularize l acc =
    list_singularize l nil.
intros T l.
induction l as [|a l h1].
intros. simpl. auto.
intros acc h2.
simpl.
destruct (in_dec eq_dec a l) as [h3 | h4].
assert (h4:forall x:T, In x acc -> ~ In x l).
  intros x h4.
  specialize (h2 x h4). simpl in h2. tauto.
apply h1; auto.
assert (h5: forall x:T, In x (a::acc) -> ~In x l).
  intros x h6.
  destruct h6 as [|h6]. subst. auto.
  specialize (h2 _ h6).
  simpl in h2. tauto.
pose proof (h1 _ h5) as h7.
rewrite h7.
assert (h8:forall x:T, In x (a::nil) -> ~In x l).
  intros x h9.
  destruct h9. subst. auto. contradiction.
rewrite (h1 _ h8). reflexivity.
Qed.

(* Returns the list [l] with all occurrences of
   [x] (in my usage there will only be
   one) removed*)
Fixpoint new_head_aux {T:Type} (l:list T) (x:T)
: list T :=
  match l with
    | nil => nil
    | a::l' => if (eq_dec a x) then (new_head_aux l' x)
               else (a::new_head_aux l' x)
    end.

(* Returns the list [l] with [x] at the head of it,
   and all occurrences of [x] (in my usage there will only be
   one) removed*)

Definition new_head {T:Type} (l:list T) (x:T)
: list T := x::new_head_aux l x.

Lemma list_singularize_new_head_not_in_compat :
  forall {T:Type} (l:list T) (x:T), ~In x l ->
                         list_singularize l nil =
                         new_head_aux (list_singularize l nil) x.
intros T l.
induction l as [|a l h1].
intros x h1.
simpl. auto.
intros x h2.
simpl.
simpl in h2.
assert (h3:~In x l). tauto.
destruct (in_dec eq_dec a l) as [h4 | h5].
apply h1; auto.
assert (h6:forall x:T, In x (a::nil) -> ~In x l).
  intros y h7.
  destruct h7; [subst | contradiction]; auto.
rewrite (list_singularize_not_in _ _ h6).
simpl.
destruct (eq_dec a x); try tauto.
rewrite <- h1. auto. auto.
Qed.





Lemma nin_new_head_aux_eq :
  forall {T:Type} (l:list T) (x:T),
    ~In x l ->
    new_head_aux l x = l.
intros T l. induction l as [|a l h1].
intros; simpl; auto.
intros x h2.
simpl in h2.
destruct (eq_dec a x). tauto.
assert (h3: ~In x l).
  simpl in h2. tauto.
simpl.
destruct (eq_dec a x) as [h4 | h5]. contradiction.
rewrite h1; auto.
Qed.


Lemma subtract_new_head_compat : 
  forall {T:Type} (l:list T) (x:T),
    Subtract (list_to_set l) x = list_to_set (new_head_aux l x).
intros T l. induction l as [|a l h1].
(* nil *)
intros x. simpl.
apply Extensionality_Ensembles.
red. split. red. intros y h1.
destruct h1; contradiction.
auto with sets.
(* cons *)
intro x.
simpl.
destruct (eq_dec a x) as [h2 | h3]. subst.
pose proof (in_dec eq_dec x l) as h4.
destruct h4 as [h4l | h4r].
rewrite list_to_set_in_iff in h4l.
rewrite sub_add_compat_in. apply h1. auto.
rewrite list_to_set_in_iff in h4r.
rewrite sub_add_compat_nin.
rewrite nin_new_head_aux_eq; auto.
rewrite list_to_set_in_iff. auto. auto.
simpl.
rewrite sub_add_comm.
rewrite h1. auto. auto.
Qed.


Lemma add_list_to_set_new_head : 
  forall {T:Type} (l:list T) (A:Ensemble T) (x:T),
    ~ Ensembles.In A x ->
    Add A x = list_to_set (list_singularize l nil) ->
    A = list_to_set (new_head_aux (list_singularize l nil) x).
intros T l. induction l as [|a l h1].
intros A x h1 h2. simpl in h2.
pose proof (add_not_empty A x). contradiction.
intros A x h2 h3.
simpl in h3.
destruct (in_dec eq_dec a l) as [h4 | h5].
rewrite (h1 _ x).
simpl.
destruct (in_dec eq_dec a l) as [h5|?]. reflexivity.
contradiction. assumption. assumption.
simpl. 
destruct (in_dec eq_dec a l) as [h6|h7].
contradiction. simpl.
destruct (eq_dec a x) as [h8 | h9].
subst.
assert (h6:forall c:T, In c (x::nil) -> ~In c l).
  intros y h8.
  destruct h8; [subst | contradiction]; auto.
simpl in h3.
pose proof (add_nin_sub_compat _ _ _ h2 h3) as h8.
rewrite subtract_new_head_compat in h8. auto.
assert (h6:forall c:T, In c (a::nil) -> ~In c l).
  intros y h8.
  destruct h8; [subst | contradiction]; auto.
rewrite list_singularize_not_in in h3.
rewrite list_singularize_not_in.
assert (h10:Subtract (Add A x) x = Subtract (list_to_set (a::list_singularize l nil)) x).
  f_equal. auto.
rewrite sub_add_compat_nin in h10.
rewrite h10.
simpl.
rewrite sub_add_comm.
rewrite subtract_new_head_compat.
auto. auto. auto.
intros y h8.
destruct h8; [subst | contradiction]; auto.
intros y h8.
destruct h8; [subst | contradiction]; auto.
Qed.




Lemma in_l_neq_a_x_in_new_head_aux :
  forall {T:Type} (l:list T) (a x:T),
    In a l -> a <> x -> In a (new_head_aux l x).
intros T l. induction l as [|b l h2].
intros; contradiction.
intros a x h3 h4. simpl.
destruct (eq_dec b x) as [h5 | h6]. subst.
destruct h3; subst.  contradict h4. auto.
apply h2; auto.
destruct h3 as [h3l | h3r]. subst. left. auto.
right. apply h2; auto.
Qed.

Lemma nin_nin_new_head_aux :
  forall {T:Type} (l:list T) (a x:T),
    ~In a l -> ~In a (new_head_aux l x).
intros T l. induction l as [|b l h2].
intros; simpl. intro. auto.
intros a x h3. intro h4.
simpl in h4.
destruct (eq_dec b x) as [h5 | h6].
subst.
simpl in h3.
assert (h5:~ In a l). tauto.
specialize (h2 _ x h5). contradiction.
simpl in h3.
destruct h4 as [h4l | h4r]. subst.
assert (h7:a <> a). intro h8. auto.
contradict h7. auto.
assert (h7:~In a l). tauto.
specialize (h2 _ x h7). contradiction.
Qed.

Lemma new_head_aux_cons :
  forall {T:Type} (l:list T) (x:T),
    new_head_aux (x::l) x = new_head_aux l x.
intros T l x.
simpl.
destruct (eq_dec x x) as [h1 | h2]. auto.
contradict h2. auto.
Qed.

Lemma new_head_singularize_comm :
  forall {T:Type} (l:list T) (x:T),
    new_head_aux (list_singularize l nil) x =
    list_singularize (new_head_aux l x) nil.
intros T l.
induction l as [|a l h1].
intros x. simpl. auto.
intros x. simpl.
destruct (in_dec eq_dec a l) as [h2 | h3];
  destruct (eq_dec a x) as [h4 | h5].
subst. auto.
simpl.
destruct (in_dec eq_dec a (new_head_aux l x)) as [h6 | h7].
auto.
pose proof (in_l_neq_a_x_in_new_head_aux _ _ _ h2 h5).
contradiction.
subst.
assert (h6:forall c:T, In c (x::nil) -> ~In c l).
  intros y h8.
  destruct h8; [subst | contradiction]; auto.
rewrite list_singularize_not_in.
rewrite new_head_aux_cons. apply h1. auto.
simpl.
destruct (eq_dec a x) as [h6 | h7];
  destruct (in_dec eq_dec a (new_head_aux l x)) as [h8 | h9].
contradiction. contradiction.
assert (h6:forall c:T, In c (a::nil) -> ~In c l).
  intros y h9.
  destruct h9; [subst | contradiction]; auto.
pose proof (nin_nin_new_head_aux _ _ x h3). contradiction.
assert (h6:forall c:T, In c (a::nil) -> ~In c l).
  intros y h8.
  destruct h8; [subst | contradiction]; auto.
rewrite list_singularize_not_in.
assert (h10:forall c:T, In c (a::nil) -> ~In c (new_head_aux l x)).
  intros y h8.
  destruct h8; [subst | contradiction]; auto.
rewrite (list_singularize_not_in _ _ h10).
rewrite h1. auto. auto.
Qed.

Lemma list_singularize_in_iff :
  forall {T:Type} (l:list T) (x:T), 
    In x l <-> In x (list_singularize l nil).
intros T l.
induction l as [|a l h1].
(* nil *)
split.
(* -> *)
intro; contradiction.
simpl. tauto.
(* <- *)
(* split *)
intro x.
split.
(* -> *)
intros h2.
destruct h2 as [h2 | h3]. subst.
simpl.
destruct (in_dec eq_dec x l) as [h2 | h3].
apply h1. assumption.
constructor. reflexivity.
simpl.
destruct (in_dec eq_dec a l) as [h4 | h5].
apply h1. assumption.
rewrite list_singularize_not_in.
right.
apply h1. assumption.
intros y h6.
destruct h6; try subst.
assumption. contradiction.
intro h2.
simpl in h2.
destruct (in_dec eq_dec a l) as [h3 | h4].
right.
apply h1. assumption.
rewrite list_singularize_not_in in h2.
destruct h2 as [h2a | h2b].
subst.
constructor. reflexivity.
right. apply h1. assumption.
intros y h3.
destruct h3; subst. assumption.
contradiction.
Qed.



Lemma remove_app : forall {T:Type} (a:T) (l l':list T),
                     remove eq_dec a l ++ (remove eq_dec a l') =
                     remove eq_dec a (l ++ l').
intros T a l. revert a.
induction l as [|b l h1].
simpl.
intros; auto.
intros a l'.
simpl.
destruct (eq_dec a b) as [h3 | h4].
subst.
apply h1.
simpl.
rewrite h1.
reflexivity.
Qed.

Lemma list_singularize_rev_cons : 
  forall {T:Type} (a:T) (l:list T), 
    list_singularize (l++(a::nil)) nil =
    (remove eq_dec a (list_singularize l nil))++(a::nil).
intros T a l. revert a.
induction l as [|b l h0].
intro a.
simpl. reflexivity.
simpl.
intro a.
destruct (in_dec eq_dec b (l++a::nil)) as [h1|h2];
  destruct (in_dec eq_dec b l) as [h3|h4].
rewrite h0.  reflexivity.
pose proof (in_app_or _ _ _ h1) as h5.
destruct h5 as [h5a | h5b].
contradiction.
destruct h5b; subst.
assert (h6:forall x:T, In x (b::nil) -> ~In x l).
  intros x h7.
  destruct h7; subst.
  assumption. contradiction.
pose proof (list_singularize_not_in _ _ h6) as h7.
rewrite h7.
simpl.
destruct (eq_dec b b) as [h8 | h9].
apply h0.
contradict h9. reflexivity.
contradiction.
contradict h2.
apply in_or_app. left; assumption.
assert (h5:a <> b).
  intro h5.
  subst.
  contradict h2.
  apply in_or_app. right; constructor.  reflexivity.
rewrite list_singularize_not_in.
pose proof (list_singularize_not_in l (b::nil)) as h6.
assert (h7:(forall x:T, In x (b::nil) -> ~In x l)).
  intros x h7. destruct h7; subst. assumption. contradiction.
specialize (h6 h7); clear h7.
rewrite h6.
pose proof (app_comm_cons (list_singularize l nil) (a::nil) b) as h7.
rewrite remove_neq_cons.
rewrite <- app_comm_cons.
rewrite <- h0.
reflexivity.
assumption.
intros x h6.
destruct h6; subst. assumption. contradiction.
Qed.

Lemma no_dup_list_singularize : 
  forall {T:Type} (l:list T),
    NoDup (list_singularize l nil).
intros T l.
induction l as [|a l h1].
simpl. constructor.
simpl.
destruct (in_dec eq_dec a l) as [h2 | h3].
assumption.
rewrite list_singularize_not_in.
constructor.
rewrite <- list_singularize_in_iff.
assumption. assumption.
intros x h2.
destruct h2; subst. assumption. contradiction.
Qed.

Lemma list_singularize_no_dup : 
  forall {T:Type} (l:list T),
    NoDup l <-> list_singularize l nil = l.
intros T l.
split.
(* -> *)
intro h1.
induction h1 as [|x l h2 h3 h4]. simpl. reflexivity.
simpl.
destruct (in_dec eq_dec x l) as [h6 | h7].
contradiction.
rewrite list_singularize_not_in.
rewrite h4. reflexivity.
intros y h5.
destruct h5 as [h5l | h5r].
rewrite h5l in h7. assumption.
contradiction.
(* <- *)
intro h1.
rewrite <- h1.
apply no_dup_list_singularize.
Qed.

Lemma remove_hd_no_dup : 
  forall {T:Type} (l:list T) (def:T), NoDup l -> 
    remove eq_dec (hd def l) l = tl l.
intros T l.
induction l as [|a l h1].
simpl. auto.
simpl.
intros def h2.  
destruct (eq_dec (T:=T)) as [h5 | h6]. 
inversion h2 as [|x l' h3 h4]. subst.
symmetry.
apply remove_not_in'; auto.
contradiction h6.
reflexivity.
Qed.



Lemma in_hd_not_nil : forall {T:Type} (l:list T) (def:T),
                l <> nil -> In (hd def l) l.
intros T l.
induction l as [|a l' h1].
intros def h2.
contradict h2. reflexivity.
simpl.
intros def h2.
left. reflexivity.
Qed.


Lemma hd_map : 
  forall {T U:Type} (f:T->U) (def:T) (l:list T),
    hd (f def) (map f l) = f (hd def l).
intros T U f def l.
induction l as [|a l h1]; simpl; auto.
Qed.                    



Lemma in_tl : forall {T:Type} (l:list T) (x:T), In x (tl l) -> In x l.
intros T l.
induction l; simpl; auto.
Qed.



Lemma list_decompose : 
  forall {T:Type} (l:list T) (def:T), 
    l <> nil -> l = (hd def l) :: (tl l).
intros T l.
induction l; auto.
intros def h1. contradict h1; auto.
Qed.




Lemma removelast_cons_cons : forall {T:Type} (l:list T)
                               (a a':T),
                          removelast (a::a'::l) = a::removelast (a'::l).
intros; simpl; auto.
Qed.


Lemma list_power_no_dup : 
  forall {T U:Type} (la:list T) (lb:list U) (lp:list (T*U)),
    NoDup la -> NoDup lb ->
    (In lp (list_power la lb)) ->
    NoDup lp.
intros T U la.
induction la as [|a la h1].
simpl.
intros lb lp h1 h2 h3.
destruct h3 as [h3l | h3r].
rewrite <- h3l. constructor. contradiction.
intros lb lp h2 h3 h4.
rewrite <- app_nil_l in h2.
pose proof (NoDup_remove_1 _ _ _ h2) as h5.
rewrite app_nil_l in h5.
simpl in h4.
rewrite in_flat_map in h4.
destruct h4 as [lp' h6].
destruct h6 as [h6l h6r].
specialize (h1 _ _ h5 h3 h6l).
rewrite in_map_iff in h6r.
destruct h6r as [b h7].
destruct h7 as [h7l h7r].
rewrite <- h7l.
constructor.
intro h8.
rewrite app_nil_l in h2.
inversion h2.
pose proof (@in_list_power1).
pose proof (in_list_power1 _ h6l _ h8) as h9.
simpl in h9.
contradiction.
assumption.
Qed.

Lemma no_dup_map_inj :
  forall {T U:Type}  (l:list T) (f:T->U),
    NoDup l -> FunctionProperties.injective f ->
    NoDup (map f l).
intros T U l.
induction l as [|a l h1].
simpl.
intros f h1 h2. constructor.
intros f h2 h3.
simpl.
inversion h2. subst.
specialize (h1 _ H2 h3).
constructor.
rewrite in_map_iff.
intro h4.
destruct h4 as [x h5].
destruct h5 as [h5l h5r].
red in h3.
specialize (h3 _ _ h5l).
subst.
contradiction.
assumption.
Qed.


Lemma no_dup_app : 
  forall {T:Type} (l1 l2:list T),
    NoDup l1 -> NoDup l2 ->
    Intersection (list_to_set l1) (list_to_set l2) =
    Empty_set _ -> NoDup (l1 ++ l2).
intros T l1.
induction l1 as  [|a l1 h1].
simpl.
intros. assumption.
intros l2 h2 h3 h4.
rewrite <- app_comm_cons.
constructor.
intro h5.
pose proof (in_app_or _ _ _ h5) as h6.
destruct h6 as [h6l | h6r].
inversion h2. contradiction.
rewrite list_to_set_in_iff in h6r.
pose proof (in_eq a l1) as h7.
rewrite list_to_set_in_iff in h7.
pose proof (Intersection_intro _ _ _ _ h7 h6r) as h8.
rewrite h4 in h8.
contradiction.
assert (h5:Included (list_to_set l1) (list_to_set (a::l1))).
  red. simpl.
  intros x h5.
  left. assumption.
pose proof (intersection_preserves_inclusion _ _ (list_to_set l2) h5) as h6.
rewrite comm_prod_psa in h6.
rewrite (comm_prod_psa (list_to_set l2) (list_to_set (a::l1))) in h6.
rewrite h4 in h6.
assert (h7:(Intersection (list_to_set l1) (list_to_set l2)) =
         (Empty_set T)).
  apply Extensionality_Ensembles.
  red; split; auto with sets.
inversion h2.
apply h1; assumption.
Qed.

Lemma no_dup_list_prod : 
  forall {T U:Type} (la:list T) (lb:list U),
    NoDup la -> NoDup lb -> NoDup (list_prod la lb).
intros T U la.
induction la as [|a la h1].
simpl. intros lb h1 h2. constructor.
simpl.
intros lb h2 h3.
inversion h2. subst.
specialize (h1 _ H2 h3).
assert (h4: FunctionProperties.injective (fun y:U => (a, y))).
  red.
  intros x1 x2 h4.
  inversion h4. subst. reflexivity.
pose proof (no_dup_map_inj _ _ h3 h4) as h5.
apply no_dup_app; try assumption.
apply Extensionality_Ensembles.
red. split.
red.
intros x h6.
destruct h6 as [pr h6l h6r].
rewrite <- list_to_set_in_iff in h6l.
rewrite <- list_to_set_in_iff in h6r.
rewrite in_map_iff in h6l.
destruct h6l as [x h7].
destruct h7; subst.
rewrite in_prod_iff in h6r.
destruct h6r; contradiction.
auto with sets.
Qed.


Lemma list_to_set_singularize_compat :
  forall {T:Type} (l:list T), list_to_set l =
                   list_to_set (list_singularize l nil).
intros T l.
induction l as [|a l h1]. simpl. reflexivity.
simpl.
destruct (in_dec eq_dec a l) as [h2 | h3].
rewrite list_to_set_in_iff in h2.
rewrite h1 in h2. rewrite h1.
rewrite in_add_eq. reflexivity. assumption.
simpl. rewrite h1.
assert (h9:forall c:T, In c (a::nil) -> ~In c l).
  intros y h10.
  destruct h10; [subst | contradiction]; auto.
rewrite (list_singularize_not_in _ _ h9).
reflexivity.
Qed.



Lemma finite_set_list_no_dup : 
  forall {T:Type} (E:Ensemble T), 
    Finite E ->
    exists (l:list T), E = list_to_set l /\
                       NoDup l.
intros T E h1.
pose proof (finite_set_list _ h1) as h2.
destruct h2 as [l h3].
exists (list_singularize l nil).
split.
rewrite h3.
apply list_to_set_singularize_compat.
apply no_dup_list_singularize.
Qed.

Lemma list_prod_cart_prod_compat :
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U)
         (la:list T) (lb:list U),
    list_to_set la = A ->
    list_to_set lb = B ->
    list_to_set (list_prod la lb) = cart_prod A B.
intros T U A B la.
induction la as [|a la h1].
simpl.
intros lb h1 h2.
rewrite <- h1.
rewrite cart_prod_empty.
reflexivity.
simpl.
intros lb h2 h3.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros pr h4.
rewrite <- list_to_set_in_iff in h4.
pose proof (in_app_or _ _ _ h4) as h5.
destruct h5 as [h5l | h5r].
rewrite in_map_iff  in h5l.
destruct h5l as [b h6].
destruct h6 as [h6l h6r]. subst.
constructor.
split.
simpl.
apply Add_intro2.
simpl.
rewrite <- list_to_set_in_iff. assumption.
rewrite (surjective_pairing pr) in h5r.
rewrite in_prod_iff in h5r.
constructor.
do 2 rewrite list_to_set_in_iff in h5r.
destruct h5r as [h5a h5b].
pose proof (Add_intro1 _ _ a _ h5a) as h6.
rewrite h2 in h6.
rewrite h3 in h5b.
split; assumption.
(* >= *)
red.
intros pr h4.
rewrite <- list_to_set_in_iff.

apply in_or_app.
rewrite <- h2 in h4.
rewrite <- h3 in h4.
destruct h4 as [h4].
destruct h4 as [h4l h4r].
inversion h4l as [? h5|? h6]. 
subst.
right.
rewrite (surjective_pairing pr).
rewrite <- list_to_set_in_iff in h4r.
rewrite <- list_to_set_in_iff in h5.
apply in_prod; assumption.
subst.
inversion h6.
rewrite (surjective_pairing pr).
left.
simpl.
rewrite in_map_iff.
exists (snd pr). subst.
split; auto.
rewrite list_to_set_in_iff.
assumption.
Qed.

Lemma no_dup_cons : forall {T:Type} (l:list T) (a:T),
                      NoDup (a::l) -> NoDup l.
intros T l a h1.
rewrite <- app_nil_l in h1.
pose proof (NoDup_remove_1 _ _ _ h1) as h2.
rewrite app_nil_l in h2.
assumption.
Qed.

Lemma no_dup_cons_nin : forall {T:Type} (l:list T) (a:T),
                          NoDup (a::l) -> ~In a l.
intros T l a h1.
rewrite <- app_nil_l in h1.
pose proof (NoDup_remove_2 _ _ _ h1) as h2.
rewrite app_nil_l in h2.
assumption.
Qed.

Lemma l_lt_length_cons :
  forall {T:Type} (l:list T) (a:T) (n:nat),
    S n < length (a::l) -> n < length l.
intros T l a n h1.
simpl in h1.
omega.
Qed.

Lemma nth_lt_ex : 
  forall {T:Type} (l:list T), 
    NoDup l ->
    forall n:nat, 
      n < length l ->
  exists! x:T, In x l /\ nth n l x = x. 
intros T l.
induction l as [|a l h1]; simpl.
intros; omega. 
intros h2 n h3.
pose proof (no_dup_cons _ _ h2) as h7.
pose proof (no_dup_cons_nin _ _ h2) as hnin.
destruct n as [|n].
exists a. red. split;intros; tauto. 
pose proof (lt_S_n _ _ h3) as h6.
specialize (h1 h7 _ h6).
destruct h1 as [x h1]. red in h1. destruct h1 as [h1l h1r]. destruct h1l as [h1a h1b].
destruct (eq_dec a x) as [h8 | h9].
rewrite h8 in hnin. contradiction. 
exists x. red. split.
split. right. assumption.
assumption.  
intros x' h8.
destruct h8 as [h8l h8r]. 
destruct h8l as [h8a | h8b].
rewrite h8a in hnin.  
pose proof (nth_indep _ x x' h6). congruence.
specialize (h1r _ (conj h8b h8r)).
assumption.
Qed.

Definition nth_lt {T:Type} (l:list T)
           (pfnd:NoDup l) (n:nat) (pflt:n < length l) :=
  proj1_sig (constructive_definite_description _ (nth_lt_ex _ pfnd _ pflt)).

Lemma nth_lt_compat : 
  forall {T:Type} (l:list T) (pfnd:NoDup l) 
         (n:nat) (pflt:n < length l),
    let x := (nth_lt _ pfnd _ pflt) in
    In x l /\ nth n l x = x. 
intros T l h1 n h2 x.
unfold nth_lt in x.
destruct constructive_definite_description. simpl in x. 
assumption.
Qed.

Lemma nth_lt_cons : 
  forall {T:Type} (l:list T) (a:T) (pfnd:NoDup (a::l))
         (n:nat) (pflt:S n < S (length l)),
    nth_lt _ pfnd _ pflt =
    nth_lt _ (no_dup_cons _ _ pfnd) _ (lt_S_n _ _ pflt).
intros T l a h1 n h2.
pose proof (nth_lt_compat _ h1 _ h2) as h3. destruct h3 as [h3l h3r].
pose proof (nth_lt_compat _ (no_dup_cons _ _ h1) _ (lt_S_n _ _ h2)) as h4. destruct h4 as [h4l h4r].
simpl in h3r.
pose proof (lt_S_n _ _ h2) as h5.
pose proof (nth_indep _  (nth_lt l (no_dup_cons l a h1) n (lt_S_n n (length l) h2)) (nth_lt (a :: l) h1 (S n) h2) h5) as h6.
congruence.
Qed.


(*Perhaps move to another spot.*)
Lemma nth_inj : 
  forall {T:Type} (l:list T),
    NoDup l ->
    forall (m n:nat) (d1 d2:T),
      m < length l -> n < length l ->
      nth m l d1 = nth n l d2 ->
      m = n.
intros T l.
induction l as [|a l h1]; simpl.
intros; omega. 
intros h2 m n d1 d2 h3 h4 h5.
pose proof (no_dup_cons_nin _ _ h2) as hnin.
pose proof (no_dup_cons _ _ h2) as hdup.
destruct (le_lt_eq_dec _ _ h3) as [h6 | h7]; 
  destruct (le_lt_eq_dec _ _ h4) as [h8 | h9].
apply lt_S_n in h6. apply lt_S_n in h8.
apply (h1 hdup _ _ d1 d2 h6 h8).
destruct m; destruct n; auto.
apply nth_indep. assumption.
apply lt_S_n in h4.
pose proof (nth_In  _ d1 h4) as h11.
subst. 
pose proof (nth_indep). 
pose proof (nth_indep _  d1 d2 h4) as hi.
rewrite hi in h11. contradiction. subst.
apply lt_S_n in h3.
pose proof (nth_In _ d2 h3) as h12.
pose proof (nth_indep _ d1 d2 h3) as hi.
rewrite <- hi in h12.
contradiction.
apply h1 in h5. subst.
apply nth_indep; auto. 
assumption. apply lt_S_n. assumption. apply lt_S_n. assumption.  
apply S_inj in h9.
apply lt_S_n in h6. subst.
destruct (zerop (length l)) as [h8 | h9]. omega.
pose proof (S_pred _ _ h9) as h10.
rewrite h10 in h5.   
destruct m as [|m].
assert (h11:pred (length l) < length l). omega. 
pose proof (nth_In _ d2 h11) as h12.
rewrite <- h5 in h12.
contradiction.
assert (h11:pred (length l) < length l). omega.
apply lt_S_n in h3.
pose proof (h1 hdup _ _ d1 d2 h3 h11 h5). omega.
apply lt_S_n in h8. apply S_inj in h7. subst.
destruct (zerop (length l)) as [h9 | h10]. omega.
rewrite (S_pred _ _ h10) in h5.
destruct n as [|n].
assert (h11:pred (length l) < length l). omega.
pose proof (nth_In _ d1 h11) as h12. rewrite h5 in h12. contradiction.
apply lt_S_n in h4.
assert (h11:pred (length l) < length l). omega.
pose proof (h1 hdup _ _ d1 d2 h11 h4 h5) as h12.
omega.
apply S_inj in h7. apply S_inj in h9. congruence.
Qed.


Lemma nth_lt_inj : 
  forall {T:Type} (l:list T) (pfnd:NoDup l) (m n:nat) 
         (pfm: m < length l) (pfn: n < length l),
    nth_lt _ pfnd _ pfm = nth_lt _ pfnd _ pfn ->
    m = n.
intros T l h1 m n h2 h3 h4.
pose proof (nth_lt_compat _ h1 _ h2) as h5.
pose proof (nth_lt_compat _ h1 _ h3) as h6.
destruct h5 as [h5l h5r].
destruct h6 as [h6l h6r].
rewrite <- h6r in h4.
rewrite <- h5r in h4.
eapply nth_inj; auto. apply h1. apply h2. apply h3. apply h4.
Qed.


Lemma lind_ex : 
  forall {T:Type} (l:list T)
         (pf:NoDup l) (x:T),
      In x l -> 
      exists! i:{m:nat | m < (length l)},
        nth_lt _ pf _ (proj2_sig i) = x.
intros T l h1 x h2.
induction l as [|a l h3]; try contradiction.
pose proof (no_dup_cons _ _ h1) as h4.
pose proof (no_dup_cons_nin _ _ h1) as h5.
destruct h2 as [h2l | h2r]. subst.
destruct (zerop (length (x::l))) as [h6 | h7].
simpl in h6. omega.
exists (exist _ 0 h7).
red. split. simpl.
pose proof (nth_lt_compat _ h1 _ h7) as h8.
destruct h8 as [h8l h8r].
simpl in h8r. symmetry. assumption.
intros i h8.
destruct i as [i h9]. simpl in h8.
pose proof (nth_lt_compat _ h1 _ h9) as h10.
destruct h10 as [h10l h10r].
apply proj1_sig_injective. simpl.
rewrite h8 in h10r.
assert (h11:nth 0 (x::l) x = x). simpl. reflexivity.
rewrite <- h10r in h11 at 3.
apply nth_inj in h11; auto.
specialize (h3 h4 h2r).
destruct h3 as [i h3]. red in h3. destruct h3 as [h3l h3r].
simpl. destruct i as [i h6].
simpl in h3l.
assert (h7:S i < S (length l)). omega.
exists (exist _ (S i) h7).  
red. split. simpl.
pose proof (nth_lt_compat _ h1 _ h7) as h8.
destruct h8 as [h8l h8r].
rewrite nth_lt_cons. 
rewrite <- h3l. f_equal; apply proof_irrelevance.
pose proof (lt_S_n _ _ h7) as h8.
intros i' h9. assert (h10:h6 = h8).  apply proof_irrelevance. subst.
apply proj1_sig_injective. simpl.
destruct i' as [i' h9]. simpl in h3l. simpl in h3r. simpl. simpl in h2r.
destruct (zerop i') as [h10 | h11]. subst.
pose proof (nth_lt_compat _ h1 _ h9) as h10.
destruct h10 as [h10l h10r]. simpl in h10r.
rewrite <- h10r in h2r at 1.
contradiction.
pose proof (S_pred _ _ h11) as h12. 
assert (h13: S (pred i') < S (length l)). omega. 
pose proof (subsetT_eq_compat _ (fun x => x < S (length l)) _ _ h9 h13 h12) as h14. 
dependent rewrite -> h14 in h3l. 
pose proof (lt_S_n _ _ h13) as h15.
specialize (h3r (exist _ (pred i') h15)). simpl in h3r.
rewrite nth_lt_cons in h3l. 
pose proof (nth_lt_compat _ h4 _  h8) as h16.
destruct h16 as [h16l h16r].
pose proof (nth_lt_compat _ (no_dup_cons l a h1) _ (lt_S_n (pred i') (length l) h13)) as h17.
destruct h17 as [h17l h17r].
rewrite <- h17r in h3l.
rewrite <- h16r in h3l.
apply nth_inj in h3l; auto.
omega.
Qed.

Definition lind {T:Type} (l:list T)
           (pfnd:NoDup l) (x:T)
           (pfin:In x l) : {m:nat | m < (length l)} :=
  proj1_sig (constructive_definite_description _ (lind_ex l pfnd x pfin)).

Lemma lind_compat : 
  forall {T:Type} (l:list T) (pfnd:NoDup l) (x:T) (pfin:In x l),
    let i := lind l pfnd x pfin in
    nth_lt _ pfnd _ (proj2_sig i) = x.
intros T l h1 x h2 i.
unfold lind in i.
destruct constructive_definite_description.
simpl in i.
assumption.
Qed.



Lemma no_dup_remove : 
  forall {T:Type} (l:list T) (x:T),
    NoDup l -> NoDup (remove eq_dec x l).
intros T l.
induction l as [|a l h1]; simpl; auto.
intros x h2.
destruct (eq_dec x a) as [h3 | h4]. subst.
apply no_dup_cons in h2.
specialize (h1 a h2). assumption.
pose proof (no_dup_cons _ _ h2) as h5. 
pose proof (no_dup_cons_nin _ _ h2) as h6. clear h2.
specialize (h1 x h5).
constructor; auto.
apply nin_nin_remove.
assumption.
Qed.


Lemma no_dup_tail : forall {T:Type} (l:list T),
                      NoDup l -> NoDup (tl l).
intros T l.
induction l as [|a l h1].
simpl. auto.
intro h2.
simpl.
apply no_dup_cons with a.
assumption.
Qed.

Fixpoint remove1 {T U:Type} (l:list (T*U)) (x:T) :=
  match l with 
    | nil => nil
    | pr::l' => if (eq_dec (fst pr) x) then 
                  (remove1 l' x) else
                  pr::(remove1 l' x)
  end.

Fixpoint remove2 {T U:Type} (l:list (T*U)) (y:U) :=
  match l with 
    | nil => nil
    | pr::l' => if (eq_dec (snd pr) y) then 
                  (remove2 l' y) else
                  pr::(remove2 l' y)
  end.

Lemma remove1_nin : forall {T U:Type} (l:list (T*U)) (x:T) (y:U),
                      ~In (x, y) (remove1 l x).
intros T U l.
induction l as [|l pr h1].
simpl. auto.
simpl.
intros x y.
destruct (eq_dec (fst l) x) as [h2 | h3].
apply h1.
intro h4.
destruct h4 as [h4 | h5].
contradict h3.
rewrite h4. simpl. reflexivity.
pose proof (h1 x y).
contradiction.
Qed.

Lemma remove2_nin : forall {T U:Type} (l:list (T*U)) (x:T) (y:U),
                      ~In (x, y) (remove2 l y).
intros T U l.
induction l as [|l pr h1].
simpl. auto.
simpl.
intros x y.
destruct (eq_dec (snd l) y) as [h2 | h3].
apply h1.
intro h4.
destruct h4 as [h4 | h5].
contradict h3.
rewrite h4. simpl. reflexivity.
pose proof (h1 x y).
contradiction.
Qed.


Lemma remove1_in :
  forall {T U:Type} (l:list (T*U)) (pr:T*U) (x:T),
         fst pr <> x -> In pr l -> In pr (remove1 l x).
intros T U l.
induction l as [|a l h1].
simpl.
auto.
simpl.
intros pr x h2 h3.
destruct (eq_dec (fst a) x) as [h4 | h5].
subst.
destruct h3 as [h3l | h3r].
subst.
contradict h2. reflexivity.
apply h1; assumption.
destruct h3 as [h3l | h3r].
subst. left. reflexivity.
right.
apply h1; assumption.
Qed.

Lemma remove2_in :
  forall {T U:Type} (l:list (T*U)) (pr:T*U) (y:U),
         snd pr <> y -> In pr l -> In pr (remove2 l y).
intros T U l.
induction l as [|a l h1].
simpl.
auto.
simpl.
intros pr y h2 h3.
destruct (eq_dec (snd a) y) as [h4 | h5].
subst.
destruct h3 as [h3l | h3r].
subst.
contradict h2. reflexivity.
apply h1; assumption.
destruct h3 as [h3l | h3r].
subst. left. reflexivity.
right.
apply h1; assumption.
Qed.


Lemma list_to_set_remove1_inc : 
  forall {T U:Type} (lp:list (T*U)) (x:T),
         Included (list_to_set (remove1 lp x)) (list_to_set lp). 
intros T U lp.
induction lp as [|pr lp h1].
simpl. auto with sets.
intros x.
red.
simpl.
intros pr' h2.
destruct (eq_dec (fst pr) x) as [h3 | h4].
subst.
specialize (h1 (fst pr)).
left.
auto with sets.
simpl in h2.
specialize (h1 x).
destruct h2 as [pr'' h5 | pr'' h6].
left.
auto with sets.
destruct h6.
right. constructor.
Qed.

Lemma list_to_set_remove2_inc : 
  forall {T U:Type} (lp:list (T*U)) (y:U),
         Included (list_to_set (remove2 lp y)) (list_to_set lp). 
intros T U lp.
induction lp as [|pr lp h1].
simpl. auto with sets.
intros y.
red.
simpl.
intros pr' h2.
destruct (eq_dec (snd pr) y) as [h3 | h4].
subst.
specialize (h1 (snd pr)).
left.
auto with sets.
simpl in h2.
specialize (h1 y).
destruct h2 as [pr'' h5 | pr'' h6].
left.
auto with sets.
destruct h6.
right. constructor.
Qed.

Lemma remove1_inc : 
  forall {T U:Type} (lp:list (T*U)) (x:T) (pr:T*U),
    In pr (remove1 lp x) -> In pr lp.
intros T U lp x pr h1.
rewrite list_to_set_in_iff in h1.
rewrite list_to_set_in_iff.
pose proof (list_to_set_remove1_inc lp x) as h2.
auto with sets.
Qed.


Lemma remove2_inc : 
  forall {T U:Type} (lp:list (T*U)) (y:U) (pr:T*U),
    In pr (remove2 lp y) -> In pr lp.
intros T U lp y pr h1.
rewrite list_to_set_in_iff in h1.
rewrite list_to_set_in_iff.
pose proof (list_to_set_remove2_inc lp y) as h2.
auto with sets.
Qed.


Lemma remove1_list_to_set_union :
  forall {T U:Type} (lab:list (T*U)) (x:T)
  (A:Ensemble T) (B:Ensemble U),
    ~ Ensembles.In A x -> 
    list_to_set lab = 
    Union (cart_prod (Singleton x) B) (cart_prod A B) -> 
    list_to_set (remove1 lab x) = cart_prod A B. 
intros T U lab x A B h0  h2.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros pr h4.
pose proof (list_to_set_remove1_inc lab x) as h5.
rewrite h2 in h5.
assert (h6:Ensembles.In  (Union (cart_prod (Singleton x) B) (cart_prod A B)) pr). auto with sets.
destruct h6 as [pr h7 | pr h8].
destruct h7 as [h7].
destruct h7 as [h7l h7r].
rewrite surjective_pairing in h4.
destruct h7l. 
rewrite <- list_to_set_in_iff in h4.
pose proof (remove1_nin lab x (snd pr)).
contradiction.
assumption.
(* >= *)
red.
intros pr h3.
rewrite <- list_to_set_in_iff.
assert (h4:fst pr <> x).
  intro h4.
  destruct h3 as [h3].
  destruct h3 as [h3l h3r].
  subst.
  contradiction.
apply remove1_in; try assumption.
rewrite list_to_set_in_iff.
rewrite h2.
right. assumption.
Qed.

Lemma remove1_no_dup : forall {T U:Type} (l:list (T*U)) (x:T),
                         NoDup l -> NoDup (remove1 l x).
intros T U l.
induction l as [|a l h1].
simpl. auto.
simpl.
intros x h2.
pose proof (no_dup_cons _ _ h2) as h3.
specialize (h1 x h3).
destruct (eq_dec (fst a) x) as [h4 | h5].
assumption.
constructor.
intro h6.
pose proof (remove1_inc _ _ _ h6).
inversion h2.
contradiction.
assumption.
Qed.

Lemma remove2_no_dup : forall {T U:Type} (l:list (T*U)) (y:U),
                         NoDup l -> NoDup (remove2 l y).
intros T U l.
induction l as [|a l h1].
simpl. auto.
simpl.
intros y h2.
pose proof (no_dup_cons _ _ h2) as h3.
specialize (h1 y h3).
destruct (eq_dec (snd a) y) as [h4 | h5].
assumption.
constructor.
intro h6.
pose proof (remove2_inc _ _ _ h6).
inversion h2.
contradiction.
assumption.
Qed.


Lemma destruct_list_cart_prod : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U) 
         (lab:list (T * U)),
    lab <> nil ->
    NoDup lab -> list_to_set lab = cart_prod A B ->
    exists (la:list T) (lb:list U),
      list_to_set la = A /\ list_to_set lb = B /\            
      NoDup la /\ NoDup lb /\
      list_to_set lab = list_to_set (list_prod la lb).
intros T U A B lab h1 h2 h3.
pose proof (list_to_set_finite lab) as h4.
assert (h5:list_to_set lab <> Empty_set _).
  intro h5.
  pose proof (empty_set_nil _ h5).
  contradiction.
rewrite h3 in h5.
rewrite h3 in h4.
assert (h6: A <> Empty_set _).
  intro h6.
  rewrite h6 in h5.
  rewrite cart_prod_empty in h5.
  contradict h5. reflexivity.
assert (h7: B <> Empty_set _).
  intro h7.
  rewrite h7 in h5.
  rewrite cart_prod_empty' in h5.
  contradict h5. reflexivity. 
pose proof (not_empty_Inhabited _ _ h6) as h8.
pose proof (not_empty_Inhabited _ _ h7) as h9.
pose proof (cart_prod_fin_rev1 _ _ h4 h9) as h10.
pose proof (cart_prod_fin_rev2 _ _ h4 h8) as h11. 
pose proof (finite_set_list_no_dup _ h10) as h12.
pose proof (finite_set_list_no_dup _ h11) as h13.
destruct h12 as [la h12].
destruct h13 as [lb h13].
exists la, lb.
destruct h12 as [h12l h12r].
destruct h13 as [h13l h13r]. 
symmetry in h12l. symmetry in h13l.
repeat split; try assumption.
rewrite h3.
symmetry.
apply list_prod_cart_prod_compat; assumption.
Qed.


Lemma in_list_power_add :
  forall {T U:Type} (la:list T) (lb:list U)
         (lp:list (T*U)) (a:T) (b:U),
    In b lb ->
    In lp (list_power la lb) ->
    In ((a, b)::lp) (list_power (a::la) lb).
intros T U la lb lp a b h2 h3.
simpl.
rewrite in_flat_map.
exists lp. split; auto.
rewrite in_map_iff.
exists b. split; auto.
Qed.
 

Lemma fpl_cons : forall {T U:Type} (la:list T) (lb:list U)
                        (lp:list (T*U)) (pr:T*U),
                   In pr lp ->
                   NoDup ((fst pr)::la) ->
                   functionally_paired_l ((fst pr)::la) lb lp ->
                   functionally_paired_l la lb (remove eq_dec pr lp). 
intros T U la lb lp pr h0 h1 h4.
constructor.
inversion h4 as [h5 h6].
intros x h7.
specialize (h5 x (in_cons (fst pr) x la h7)).
destruct h5 as [y h5].
exists y.
red.  red in h5.
pose proof NoDup_remove_2.
rewrite <- app_nil_l in h1.
pose proof (NoDup_remove_2 _ _ _ h1) as h8.
rewrite app_nil_l in h8.
assert (h9:(fst pr) <> x).
  intro; subst; contradiction.
split.
destruct h5 as [h5l h5r].
destruct h5l as [h5a h5b].
split; try assumption.
assert (h10:(x,y) <> pr).
  intro h11.
  rewrite surjective_pairing in h11.
  inversion h11.
  subst.
  contradict h9.
  reflexivity.
apply remove_a_in_eq; assumption.
intros u h10.
destruct h5 as [h5l h5r].
destruct h10 as [h10l h10r].
assert (h11:(x,u) <> pr).
  intro h12.
  rewrite surjective_pairing in h12.
  inversion h12.
  subst.
  contradict h9.
  reflexivity.
pose proof (in_remove_neq_in_l _ _ _ h10r h11) as h12.
apply (h5r _ (conj h10l h12)).

intros pr' h5.
destruct (eq_dec pr pr') as [h6 | h7].
subst.
pose proof (remove_In eq_dec lp pr').
contradiction.
apply neq_sym_iff in h7.
pose proof (in_remove_neq_in_l _ _ _ h5 h7) as h8.
pose proof (fpl_in_dom _ _ _ h4 _ h8) as h9.
pose proof (fpl_in_ran _ _ _ h4 _ h8) as h10.
split.
destruct h9 as [h9l | h9r].
rewrite (surjective_pairing pr) in h0.
rewrite h9l in h0.
rewrite (surjective_pairing pr') in h8.
pose proof (fpl_functional h4 _ _ _ h0 h8) as h11.
contradict h7.
rewrite (surjective_pairing pr). rewrite (surjective_pairing pr').
rewrite h9l. rewrite h11.
reflexivity.
assumption.
assumption.
Qed.


Lemma fpl_cons_nin : 
  forall {T U:Type} (la:list T) (lb:list U)
         (lp:list (T*U)) (a:T) (b:U), 
    ~In a la -> In b lb ->  
    functionally_paired_l la lb lp ->
    functionally_paired_l (a::la) lb ((a, b) :: lp).
intros T U la lb lp a b h1 h2 h3.
constructor.
intros x h4.
destruct h4 as [h4l | h4r]. subst.
exists b. red.
split. split; auto. left. reflexivity.
intros b' h4.
destruct h4 as [h4l h4r].
destruct h4r as [h5 | h6].
inversion h5; subst. reflexivity.
destruct h3 as [h3l h3r].
specialize (h3r _ h6).
simpl in h3r.
destruct h3r; contradiction. 
inversion h3 as [h3l h3r].
specialize (h3l _ h4r).
destruct h3l as [y h3l].
red in h3l.
exists y. red.
destruct h3l as [h5 h6].
destruct h5 as [h5l h5r].
split.
split; auto. right. auto.
intros y' h7. 
destruct h7 as [h7l h7r].
destruct h7r as [h8 | h9].
inversion h8.  subst.
contradiction.
specialize (h3r _ h9).
specialize (h6 _ (conj h7l h9)).
assumption.
intros pr h4.
destruct h4 as [h4l | h4r].
rewrite (surjective_pairing pr) in h4l.
inversion h4l. subst.
split. left. reflexivity. assumption.
destruct h3 as [h3l h3r].
specialize (h3r _ h4r).
destruct h3r as [h5  h6].
split. right. auto. assumption.
Qed.



Inductive synced {T U:Type} (la:list T) (lp:list (T*U)) :=
  | synced_intro : map (fun pr => (fst pr)) lp = la ->
                   synced la lp.

Lemma synced_hd : forall {T U:Type} (def:T*U) (la:list T) 
                           (lp:list (T*U)),
                      synced la lp ->
                      fst (hd def lp) = hd (fst def) la.
intros T U def la lp.
induction la as [|a la h1]; induction lp as [|pr lp h2].
simpl. auto.
intro h3.
destruct h3 as [h3].
simpl in h3.
discriminate.
simpl.
intro h2.
destruct h2 as [h2].
simpl in h2.
discriminate.
intro h3.
simpl.
destruct h3 as [h3].
simpl in h3.
inversion h3.
reflexivity.
Qed.

Lemma synced_nil1 : forall {T U:Type} (lp:list (T*U)),
                      synced nil lp -> lp = nil.
intros T U lp.
induction lp as [|pr lp h1].
auto.
intro h2.
destruct h2 as [h2].
simpl in h2.
discriminate.
Qed.

Lemma synced_nil2 : forall {T U:Type} (la:list T),
                      synced la (@nil (T*U)) -> la = nil.
intros T U la.
induction la as [|a la h1].
auto.
intro h2.
destruct h2 as [h2].
simpl in h2.
discriminate.
Qed.


Lemma synced_cons : forall {T U:Type}  (la:list T)
                           (lp:list (T*U)) (a:T) (pr:(T*U)),
                      synced (a::la) (pr::lp) ->
                      synced la lp.
intros T U la lp a pr h1.
constructor.
destruct h1 as [h1].
simpl in h1.
pose proof (cons_inj _ _ _ _ h1) as h2.
destruct h2; assumption.
Qed.


Lemma synced_cons' : forall {T U:Type}  (la:list T)
                           (lp:list (T*U)) (a:T),
                      synced (a::la) lp ->
                      synced la (tail lp).
intros T U la lp.
revert la.
induction lp as [|pr lp h1].
simpl.
intros la a h1.
destruct h1 as [h1].
simpl in h1.
discriminate.
intros la a h2. simpl.
apply (synced_cons _ _ a pr).
assumption.
Qed.


Lemma synced_cons'' : 
  forall {T U:Type} (la:list T) 
         (lp:list (T*U)) (a:T) (b:U), 
    synced la lp -> synced (a::la) ((a, b) :: lp).
intros T U la lp a b h1.
constructor.
destruct h1. simpl.
f_equal. assumption.
Qed.



Lemma remove_synced_cons : forall {T U:Type} (la:list T) (lb:list U)
                                  (lp:list (T*U)) (pr:(T*U)),
                           synced ((fst pr)::la) lp ->
                           functionally_paired_l ((fst pr)::la) lb lp ->
                           In pr lp -> NoDup lp ->
                           remove eq_dec pr lp = tl lp.
intros T U la lb lp. revert la.
induction lp as [|pr' lp h1].
simpl. auto.
intros la pr h2 h2' h3 h4.
simpl.
destruct (eq_dec pr pr') as [h5 | h6].
subst.
pose proof (no_dup_cons_nin _ _ h4) as h5.
symmetry.
apply remove_not_in'.
assumption.
destruct h3 as [h7 | h8].
symmetry in h7.
contradiction.
pose proof (synced_hd pr _ _ h2) as h7.
simpl in h7.
pose proof (in_cons pr' pr lp h8) as h9.
pose proof (in_eq pr' lp) as h10.
rewrite (surjective_pairing pr) in h9.
rewrite (surjective_pairing pr') in h10 at 1.
rewrite h7 in h10.
pose proof (fpl_functional  h2' _ _ _ 
                            h9 h10) as h11.
assert (h12:pr = pr').
  rewrite (surjective_pairing pr).
  rewrite (surjective_pairing pr').
  rewrite h7, h11.
  reflexivity.
contradiction.
Qed.


Lemma fpl_cons_synced : 
  forall {T U:Type} (la:list T) (lb:list U)
         (lp:list (T*U)) (a:T),
    synced (a::la) lp ->
    NoDup (a::la) ->
    NoDup lp ->
    functionally_paired_l (a::la) lb lp ->
    functionally_paired_l la lb (tl lp).
intros T U la lb lp a h1 h2 h2' h3.
inversion h3 as [h4 h5].
specialize (h4 a (in_eq a la)).
destruct h4 as [b h6].
red in h6.
destruct h6 as [h6 h7].
destruct h6 as [h6a h6b].
pose proof (fpl_cons _ _  _ (a, b) h6b h2 h3) as h8.
rewrite (remove_synced_cons la lb) in h8.
assumption.
simpl. assumption.
simpl. assumption. assumption.
assumption.
Qed.


Lemma in_tail_list_power_synced :
  forall {T U:Type} (la:list T) (lb:list U) (lp:list (T*U))
         (a:T),
    In (tl lp) (list_power la lb) ->
    synced (a :: la) lp ->
    functionally_paired_l (a :: la) lb lp ->
    In lp (list_power (a::la) lb).
intros T U la lb lp a h1 h2 h2'.
revert h1 h2 h2'. revert a lb la.
induction lp as [|pr lp h3].
simpl.
intros a lb la h1 h2.
destruct h2 as [h2].
simpl in h2. discriminate.
simpl.
intros a lb la h1 h2 h4.
rewrite in_flat_map. exists lp.
split. assumption.
rewrite in_map_iff.
exists (snd pr).
destruct h2 as [h2].
simpl in h2.
pose proof (cons_inj _ _ _ _ h2) as h5.
destruct h5 as [h5l h5r].
rewrite <- h5l.
rewrite <- surjective_pairing.
split. reflexivity.
destruct h4 as [h4 h5].
specialize (h5 _ (in_eq pr lp)).
destruct h5; assumption.
Qed.

Lemma in_list_power_synced_tail:
  forall {T U:Type} (la:list T) (lb:list U) (lp:list (T*U))
         (a:T),
    synced (a :: la) lp ->
    functionally_paired_l (a :: la) lb lp ->
    In lp (list_power (a::la) lb) ->
    In (tl lp) (list_power la lb).
intros T U la lb lp a h1 h2 h2'.
revert h1 h2 h2'. revert a lb la.
destruct lp as [|pr lp].
simpl.
intros a lb la h1.
destruct h1 as [h1].
simpl in h1. discriminate.
simpl.
intros a lb la h1 h2 h4.
rewrite in_flat_map in h4.
destruct h4 as [lp' h4].
destruct h4 as [h4 h5].
rewrite in_map_iff in h5.
destruct h5 as [b h6].
destruct h6 as [h6l h6r].
inversion h6l.
subst.
assumption.
Qed.

Lemma in_tail_list_power_synced_iff :
  forall {T U:Type} (la:list T) (lb:list U) (lp:list (T*U))
         (a:T),
    synced (a :: la) lp ->
    functionally_paired_l (a :: la) lb lp ->
    (In (tl lp) (list_power la lb) <->
    In lp (list_power (a::la) lb)).
intros T U la lb lp a h1 h2.
split.
intro; apply in_tail_list_power_synced; assumption.
intro; apply in_list_power_synced_tail with a; assumption.
Qed.


Lemma synced_no_dup : 
  forall {T U:Type} (l:list T) (lp:list (T*U)),
    NoDup l -> synced l lp -> NoDup lp.
intros T U l lp h1 h2.
revert lp h2. 
induction h1 as [|a l h3 h4 h5]. 
intros lp h1.
pose proof (synced_nil1 _ h1). subst.
constructor.
intros lp. induction lp as [|pr lp h2].
intros; constructor.
intro h6.
pose proof (synced_cons' _ _ _ h6) as h7.
simpl in h7.
specialize (h5 _ h7).
constructor.
pose proof (synced_hd pr _ _  h6) as h8.
simpl in h8.
intro h9.
inversion h7 as [h10].
pose proof (in_map (fun pr0:T*U => fst pr0) lp pr h9) as h11.
rewrite h10 in h11.
rewrite h8 in h11.
contradiction.
assumption.
Qed.

Lemma in_list_power_synced : 
  forall {T U:Type} (la:list T) (lb:list U) (lp:list (T*U)),
    In lp (list_power la lb) ->
    synced la lp.
intros T U la.
induction la as [|a la h1].
intros  lb lp h1.
simpl in h1.
destruct h1; try contradiction. subst.
constructor.
simpl. reflexivity.
intros lb lp h2.
simpl in h2.
rewrite in_flat_map in h2.
destruct h2 as [lp' h2].
destruct h2 as [h2l h2r].
constructor.
rewrite in_map_iff in h2r.
destruct h2r as [b h2r].
destruct h2r as [h2a h2b].
specialize (h1 _ _ h2l).
destruct h1 as [h1].
rewrite <- h2a.
simpl.
rewrite h1.
reflexivity.
Qed.

Lemma synced_length : 
  forall {T U:Type} (la:list T) (lp:list (T*U)),
    synced la lp -> length la = length lp.
intros T U la lp h1.
destruct h1 as [h1].
subst.
apply map_length.
Qed.


Lemma fpl_in_list_power : 
  forall {T U:Type} (la:list T) (lb:list U) (lp:list (T*U)),
    synced la lp ->
    NoDup la ->  NoDup lp ->
    (functionally_paired_l la lb lp <->
     In lp (list_power la lb)).
intros T U la lb lp h0 h1 h3.  split.
(* -> *)
revert h0 h1  h3. revert lb lp.
induction la as [|a la h1].
intros lb lp h0 h1  h3.
simpl. left. symmetry.
apply fpl_empty1_s with lb. assumption.
intros lb lp h0  h3 h4 h5.
inversion h5 as [h6 h7].
specialize (h6 a (in_eq a la)).
destruct h6 as [b h6].
red in h6.
destruct h6 as [h6a h6b]. 
destruct h6a as [h8 h9].
pose proof (fpl_cons_synced _ _ _ _ h0 h3 h4 h5) as h10.
pose proof (synced_cons' _ _ _ h0) as h11.
pose proof (no_dup_cons _ _ h3) as h12.
pose proof (no_dup_tail _ h4) as h13.
specialize (h1 _ _ h11 h12 h13 h10).
apply in_tail_list_power_synced; assumption.
(* <- *) 
intro h4.
revert lb lp h0 h1 h3 h4.
induction la as [|a la h1].
(* nil *)
simpl.
intros lb lp h1 h2 h3 h4.
destruct h4 as [h4 | h5].
subst.
apply fpl_empty1. contradiction.
(* cons *)
intros lb lp h2 h3 h4 h5.
simpl in h5.
rewrite in_flat_map in h5.
destruct h5 as [lp' h6].
destruct h6 as [h6l h6r].
rewrite in_map_iff in h6r.
destruct h6r as [b h7].
destruct h7 as [h7a h7b].
constructor.
intros x h8.
destruct h8 as [h8 | h9].
subst.
exists b. red.
split.
split; try constructor; auto.
intros b' h8.
destruct h8 as [h8l h8r].
pose proof (no_dup_cons_nin _ _ h3) as h9.
destruct h8r as [h10 | h11].
inversion h10. reflexivity.
pose proof (in_list_power1 _ h6l _ h11) as h12.
simpl in h12.
contradiction.
rewrite <- h7a in h2.
pose proof (synced_cons _ _ _ _ h2) as h10.
rewrite <- h7a in h4.
pose proof (no_dup_cons _ _ h3) as h11.
pose proof (no_dup_cons _ _ h4) as h12.
specialize (h1 _ _ h10 h11 h12 h6l).
destruct h1 as [h13 h14].
specialize (h13 _ h9).
destruct h13 as [y h13].
exists y.
red. red in h13.
destruct h13 as [h13a h13b].
split. split. destruct h13a as [h13aa h13ab].
assumption.
destruct h13a as [h13aa h13ab].
pose proof (in_cons (a, b) (x, y) lp' h13ab) as h15.
rewrite h7a in h15. assumption.
intros b' h15.
destruct h15 as [h15l h15r].
rewrite <- h7a in h15r.
destruct h15r as [h16 | h17].
inversion h16. subst.
pose proof (no_dup_cons_nin _ _ h3).
contradiction.
apply h13b.
split; assumption.
intros pr h8. split.
rewrite <- h7a in h8.
destruct h8 as [h8 | h9].
left. rewrite surjective_pairing in h8.
inversion h8. reflexivity.
pose  proof (in_list_power1  _ h6l _ h9).
right. assumption.
rewrite <- h7a in h8.
destruct h8 as [h8l | h8r]. rewrite surjective_pairing in h8l.
inversion h8l. subst. assumption.
pose proof (in_list_power2 _ h6l _ h8r). assumption.
Qed.

Lemma in_list_power_fpl : 
  forall {T U:Type} (la:list T) (lb:list U) (lp:list (T*U)),
    NoDup la ->  NoDup lp ->
    In lp (list_power la lb) ->
    functionally_paired_l la lb lp.
intros T U la lb lp h1 h2 h3.
rewrite fpl_in_list_power; try assumption.
apply in_list_power_synced with lb; assumption.
Qed.
 
Lemma in_list_of_lists_seqs_map_map_pair :
  forall {T U V:Type} (p:T->U->V) (li:list T) (lj:list U) (l:list V),
    In l (list_of_lists_seqs (map (fun i : T => map (p i) lj) li)) ->
    exists 
      lp:list (T*U), In lp (list_of_lists_seqs
                              (map (fun i:T => map (pair i) lj) li)) /\ l = map (fun pr:(T*U) => p (fst pr) (snd pr)) lp.
intros T U V p li lj.
induction li as [|i li h1]; induction lj as [|j lj h2].
simpl.
intros l h3. 
exists nil. split. left. reflexivity.
simpl. destruct h3 as [h3l | h3r]. subst. reflexivity.
contradiction.
simpl.
intros l h3.
exists nil. split.
left. reflexivity. simpl. destruct h3; auto. contradiction.
simpl. intros; contradiction.
intros l h5.
simpl in h5.
rewrite in_map_iff in h5.
destruct h5 as [vl h5].
destruct h5 as [h5l h5r].
destruct (in_app_or _ _ _ h5r) as [h6 | h7].
clear h5r.
rewrite in_map_iff in h6.
destruct h6 as [lv h6].
destruct h6 as [h6l h6r].
specialize (h1  _ h6r).
destruct h1 as [lp h1].
exists ((i, j) :: lp).
simpl.
rewrite in_map_iff.
split.
exists (i, j, lp).
simpl. split. reflexivity.
apply in_or_app.
left.
rewrite in_map_iff. exists lp.
split. reflexivity.
destruct h1 as [h1l h1r].
assumption.  
destruct h1 as [h1l h1r].
subst. simpl. reflexivity.
rewrite (surjective_pairing vl) in h7.
rewrite in_prod_iff in h7. 
destruct h7 as [h7l h7r].
specialize (h1  _ h7r).
destruct h1 as [lp h1]. 
rewrite in_map_iff in h7l.
destruct h7l as [u h8].
destruct h8 as [h8l h8r].
exists ((i, u)::lp).
simpl.
rewrite in_map_iff.
split.
exists ((i, u), lp).
simpl. split.
reflexivity.
apply in_or_app.
right.
rewrite in_prod_iff.
split.
rewrite in_map_iff.
exists u. split.
reflexivity.
assumption.
destruct h1 as [h1l h1r].
assumption.
rewrite <- h5l.
f_equal. rewrite h8l. reflexivity.
destruct h1 as [h1l h1r].
assumption.
Qed.



Lemma in_list_of_lists_seqs_map_map_pair_in_list_power :
  forall {T U V:Type} (p:T->U->V) (li:list T) (lj:list U) (lp:list (T*U)),
    In lp (list_of_lists_seqs (map (fun i:T => map (pair i) lj) li)) ->
    In lp (list_power li lj).
intros T U V p li lj.
induction li as [|i li h1]; induction lj as [|j lj h2].
simpl. auto.
simpl. auto.
simpl. intros; contradiction.
simpl. intros lp h3.
rewrite in_flat_map.
rewrite in_map_iff in h3.
destruct h3 as [tr h4].
destruct h4 as [h4l h4r].
destruct (in_app_or _ _ _ h4r) as [h5 | h6].
clear h4r.
rewrite in_map_iff in h5.
destruct h5 as [lp' h6].
destruct h6 as [h6l h6r].
specialize (h1 _ h6r).
exists lp'. split; auto.
simpl.
subst. simpl. left.
reflexivity.
subst.
rewrite (surjective_pairing tr) in h6.
pose proof (in_prod_iff).
rewrite in_prod_iff in h6.
destruct h6 as [h6l h6r].
specialize (h1 _ h6r).
exists (snd tr).
split; auto.
simpl. right.
rewrite in_map_iff.
rewrite in_map_iff in h6l.
destruct h6l as [u h6l].
destruct h6l as [h6a h6b].
exists u. rewrite h6a.
split; auto. 
Qed.

Lemma map_ext_in : forall {T U:Type} (f g:T->U) (l:list T),
                     (forall x:T, In x l -> f x = g x) ->
                          map f l = map g l.
intros T U f g l.
induction l as [|a l h1].
simpl. auto.
simpl. intro h2.
pose proof (h2 a) as h3.
rewrite h3.
f_equal. apply h1.
intros x h4.
apply h2. right. assumption.
left. reflexivity.
Qed.


Lemma fpl_impl_map : 
  forall {T U:Type} (li:list T) (lj:list U) (pfi:NoDup li)
         (lp:list (T*U))
         (pffp:functionally_paired_l li lj lp) (def:U),
    synced li lp ->
    lp = map 
           (fun i : T =>
              (i, (pffp, def) l-> i)) li.
intros T U li lj h1 lp h3 h4 h5.
destruct h5 as [h5].
rewrite <- map_id at 1.
rewrite <- h5 at 1.
rewrite map_map.
apply map_ext_in.
intros pr h6. simpl.
rewrite (surjective_pairing pr).
simpl. 
f_equal.
unfold fpl_app.
destruct ((constructive_definite_description
        (fun b : U =>
         if in_dec eq_dec (fst pr) li then In (fst pr, b) lp else b = h4)
        (fpl_app_ex h3 h4 (fst pr)))) as [u h7].
simpl.
inversion h3 as [h3l h3r].
specialize (h3r _ h6).
destruct h3r as [h3a h3b].
destruct (in_dec eq_dec (fst pr) li) as [h8 | h9].
rewrite (surjective_pairing pr) in h6.
rewrite (fpl_functional h3 _ _ _ h6 h7). 
reflexivity.
contradiction.
Qed.


Lemma in_list_of_lists_seqs_map_map : 
  forall {T U V:Type} (p:T->U->V) (li:list T) (lj:list U) (pfi:NoDup li)
         (pfj:NoDup lj) (l:list V) (def:U),
    In l (list_of_lists_seqs (map (fun i : T => map (p i) lj) li)) ->
  exists (lp:list (T*U)) (pfin:In lp (list_power li lj)),
    l = map (fun i:T => p i  ((in_list_power_fpl li lj lp pfi
                 (list_power_no_dup li lj lp pfi pfj pfin)
                 pfin, def) l-> i)) li.
intros T U V p li lj h1 h2 l def h3.
pose proof (in_list_of_lists_seqs_map_map_pair p li lj l h3) as h4.
destruct h4 as [lp h4].
destruct h4 as [h4l h4r].
pose proof (in_list_of_lists_seqs_map_map_pair_in_list_power p li lj lp h4l) as h5.
exists lp, h5.
pose proof (in_list_power_synced _ _ _ h5) as h6.
pose proof (list_power_no_dup _ _ _  h1 h2 h5) as h7.
pose proof (in_list_power_fpl _ _ _ h1 h7 h5) as h8.
pose proof (fpl_impl_map _ _ h1 _ h8 def h6) as h9.
rewrite h4r.
rewrite h9 at 1.
rewrite map_map.
apply map_ext_in.
simpl.
intros x h10.
f_equal.
f_equal.
apply proof_irrelevance.
Qed.

Lemma in_list_of_lists_seqs_map_map_pair_rev :
  forall {T U V:Type} (p:T->U->V) (li:list T) (lj:list U) (lp:list (T*U)),
    In lp (list_of_lists_seqs
             (map (fun i:T => map (pair i) lj) li)) ->
    exists l:list V,
    In l (list_of_lists_seqs (map (fun i : T => map (p i) lj) li)) /\ l = map (fun pr:(T*U) => p (fst pr) (snd pr)) lp.
intros T U V p li lj.
induction li as [|i li h1]; induction lj as [|j lj h2].
simpl.
intros lp h1.
destruct h1 as [h1l | h1r].
subst.
simpl.
exists nil. split; auto. contradiction.
simpl.
intros lp h3.
destruct h3 as [h3 | h4].
subst. simpl. exists nil. split; auto. contradiction.
simpl. intros; contradiction.
intros lp h3.
simpl. simpl in h3.
rewrite in_map_iff in h3.
destruct h3 as [tr h3].
destruct h3 as [h3l h3r]. 

destruct (in_app_or _ _ _  h3r) as [h4 | h5].
clear h3r.
rewrite in_map_iff in h4.
destruct h4 as [lp' h4].
destruct h4 as [h4l h4r].
rewrite surjective_pairing in h4l.
inversion h4l.
specialize (h1 _ h4r).
destruct h1 as [l h1].
exists ((p i j) :: l).
split.
rewrite in_map_iff.
exists (p i j, l).
simpl.
split; auto.
apply (in_or_app).
left.
rewrite in_map_iff.
exists l.
split; auto.
destruct h1 as [h1l h1r].
simpl in h1l.
assumption. 
rewrite <- h3l.
simpl. f_equal. 
f_equal. 
inversion H0. auto. inversion H0. auto.
destruct h1 as [h1l h1r].
rewrite H1 in h1r.
assumption.
rewrite (surjective_pairing tr) in h5.
rewrite in_prod_iff in h5.
destruct h5 as [h5l h5r].
rewrite in_map_iff in h5l.
destruct h5l as [u h5l].
destruct h5l as [h5a h5b].
specialize (h1 _ h5r).
destruct h1 as [l h1].
destruct h1 as [h1l h1r].
exists ((p i u) :: l).
split.
rewrite in_map_iff.
exists (p i u, l).
simpl.
split; auto.
apply in_or_app.
right.
rewrite in_prod_iff.
split. 
rewrite in_map_iff.
exists u. split; auto.
assumption.
rewrite <- h3l.
simpl.
f_equal.
f_equal.
inversion h5a. simpl. auto.
inversion h5a. simpl. auto.
assumption.
Qed.


Lemma in_list_power_in_list_of_lists_seqs_map_map_pair :
  forall {T U:Type} (li:list T) (lj:list U) (lp:list (T*U)),
    In lp (list_power li lj) ->
    In lp (list_of_lists_seqs (map (fun i:T => map (pair i) lj) li)).
intros T U li lj.
induction li as [|i li h1]; induction lj as [|j lj h2].
simpl. auto.
simpl. auto.
simpl. simpl in h1.
intros lp h2.
rewrite in_flat_map in h2.
destruct h2 as [? h2].
destruct h2; contradiction.
simpl.
intros lp h3.
rewrite in_map_iff.
rewrite in_flat_map in h3.
destruct h3 as [lp' h3].
destruct h3 as [h3l h3r].
simpl in h3r.
destruct h3r as [h3a | h3b].
exists (i, j, lp'). simpl.
split; auto.
apply in_or_app.
left.
rewrite in_map_iff.
exists lp'. split; auto.
rewrite in_map_iff in h3b.
destruct h3b as [u h3b].
destruct h3b as [h4 h5].
exists (i, u, lp'). simpl. split; auto.
apply in_or_app.
right.
rewrite in_prod_iff.
split.
rewrite in_map_iff. exists u.
split; auto.
auto.
Qed.


Lemma in_map_list_of_lists_seqs_map :
  forall {T U V:Type} (p:T->U->V) (li:list T) (lj:list U)
         (ls:{pr:list (T*U) | In pr (list_power li lj)})
         (pfi:NoDup li) (pfj:NoDup lj) (def:U),
    In ls (map_sig (list_power li lj)) ->
    exists l: list V, In l (list_of_lists_seqs (map (fun i : T => map (p i) lj) li)) /\
      l =  (map
             (fun i : T =>
              p i
                ((in_list_power_fpl li lj (proj1_sig ls) pfi
                    (list_power_no_dup li lj (proj1_sig ls) pfi pfj
                       (proj2_sig ls)) (proj2_sig ls), def) l-> i)) li).
intros T U V p li lj ls h1 h2 def h3.
pose proof (in_list_power_in_list_of_lists_seqs_map_map_pair _ _ _ (proj2_sig ls)) as h4.
pose proof (in_list_of_lists_seqs_map_map_pair_rev p _ _ _  h4) as h5.
destruct h5 as [l h5].
destruct h5 as [h5l h5r].
exists l.
split; auto.
pose proof (in_list_power_synced _ _ _ (proj2_sig ls)) as h6.
pose proof (list_power_no_dup _ _ _  h1 h2 (proj2_sig ls)) as h7.
pose proof (in_list_power_fpl _ _ _ h1 h7 (proj2_sig ls) ) as h8.
pose proof (fpl_impl_map _ _ h1 _ h8 def h6) as h9.
rewrite h5r.
rewrite h9 at 1.
rewrite map_map.
apply map_ext_in.
simpl.
intros x h10. 
f_equal. f_equal.
apply proof_irrelevance.
Qed.

Lemma in_list_of_lists_seqs_map_map_pair_in_list_power' :
  forall {T U:Type}  (li:list T) (lj:list U) (lp:list (U*T)),
    In lp (list_of_lists_seqs
          (map (fun j : U => map (fun i : T => (j, i)) li) lj)) ->
    In lp (list_power lj li).
intros T U li lj. 
induction lj as [|j lj h1]; simpl; auto.
intros lp h2.
rewrite in_map_iff in h2.
destruct h2 as [tr h2].
destruct h2 as [h2a h2b].
rewrite (surjective_pairing tr) in h2b.
rewrite in_prod_iff in h2b.
destruct h2b as [h2l h2r].
specialize (h1 _ h2r).
rewrite in_flat_map.
exists (snd tr). split; auto.
rewrite in_map_iff.
rewrite in_map_iff in h2l.
destruct h2l as [x h2l]. destruct h2l as [h3 h4].
exists x. split. rewrite <- h2a.
f_equal. assumption.
assumption.
Qed.

Lemma in_list_power_in_list_of_lists_seqs_map_map_pair' :
  forall {T U:Type}  (li:list T) (lj:list U) (lp:list (U*T)),
    In lp (list_power lj li) ->
    In lp (list_of_lists_seqs
          (map (fun j : U => map (fun i : T => (j, i)) li) lj)).
intros T U li lj.
induction lj as [|j lj h1]; simpl; auto.
intros lp h2.
rewrite in_flat_map in h2.
destruct h2 as [lp' h2].
destruct h2 as [h2l h2r].
rewrite in_map_iff.
specialize (h1 _ h2l).
rewrite in_map_iff in h2r.
destruct h2r as [i h2r].
destruct h2r as [h2a h2b].
exists (j,i, lp').
simpl. split. assumption.
rewrite in_prod_iff.
split.
rewrite in_map_iff.
exists i. split; auto.
assumption.
Qed.


Lemma in_list_of_lists_seqs_map_map_pair' : 
  forall {T U V:Type} (p:T->U->V) (li:list T) (lj:list U)
         (l:list V),
    In l
       (list_of_lists_seqs
          (map (fun j : U => map (fun i : T => p i j) li) lj)) ->
    exists lp:list (U*T),
       In lp (list_of_lists_seqs
          (map (fun j : U => map (fun i : T => (j, i)) li) lj))
       /\ l = map (fun pr:(U*T) => p (snd pr) (fst pr)) lp.
intros T U V p li lj. revert li.
induction lj as [|j lj h1].
simpl.
intros li l h2.
exists nil. simpl. split. left; auto. destruct h2; [subst; auto | contradiction]. 
intros li l h5.
simpl in h5.
rewrite in_map_iff in h5.
destruct h5 as [pr  h5].
destruct h5 as [h5l h5r].
rewrite (surjective_pairing pr) in h5r.
rewrite in_prod_iff in h5r.
destruct h5r as [h5a h5b].
specialize (h1 _ _  h5b).
destruct h1 as [lp h1].
destruct h1 as [h1a h1b].
pose proof (in_list_of_lists_seqs_map_map_pair_in_list_power' _ _  _ h1a) as h7.
rewrite in_map_iff in h5a.
destruct h5a as [i h5a]. destruct h5a as [h9 h10].
pose proof (in_list_power_add _ _ _ j i h10 h7) as h11.
pose proof (in_list_power_in_list_of_lists_seqs_map_map_pair' _ _ _  h11) as h12.
exists ((j, i)::lp).
split; auto.
simpl. subst.
f_equal. rewrite h9. reflexivity.
assumption.
Qed.


Lemma in_list_of_lists_seqs_map_map' : 
  forall {T U V:Type} (p:T->U->V) (li:list T) (lj:list U)
         (pfi:NoDup li) (pfj:NoDup lj) (l:list V) (def:T),
    In l
       (list_of_lists_seqs
          (map (fun j : U => map (fun i : T => p i j) li) lj)) ->
    exists (lp:list (U*T)) (pfin:In lp (list_power lj li)),
      l =  (map
          (fun j : U =>
           p
             ((in_list_power_fpl lj li lp pfj
                 (list_power_no_dup lj li lp pfj pfi pfin)
                 pfin, def) l-> j) j) lj).
intros T U V p li lj h1 h2 l def h3.
pose proof (in_list_of_lists_seqs_map_map_pair _ _ _ _ h3) as h4.
destruct h4 as [lp h4].
destruct h4 as [h4l h4r].
pose proof (in_list_of_lists_seqs_map_map_pair_in_list_power' li lj lp h4l) as h5.
exists lp, h5.
pose proof (in_list_power_synced _ _ _ h5) as h6.
pose proof (list_power_no_dup _ _ _  h2 h1 h5) as h7.
pose proof (in_list_power_fpl _ _ _ h2 h7 h5) as h8.
pose proof (fpl_impl_map _ _ h2 _ h8 def h6) as h9.
rewrite h4r.
rewrite h9 at 1.
rewrite map_map.
apply map_ext_in.
simpl.
intros x h10.
f_equal.
f_equal.
apply proof_irrelevance.
Qed.

Lemma in_list_of_lists_seqs_map_map_pair_rev' :
  forall {T U V:Type} (p:T->U->V) (li:list T) (lj:list U) (lp:list (U*T)),
    In lp (list_of_lists_seqs
             (map (fun j : U => map (fun i : T => (j, i)) li) lj)) ->
    exists l:list V,
      In l (list_of_lists_seqs
          (map (fun j : U => map (fun i : T => p i j) li) lj)) /\
         l = map (fun pr:(U*T) => p (snd pr) (fst pr)) lp.
intros T U V p li lj.
induction lj as [|j lj h1]; destruct li as [|i li]; simpl.
intros lp h1. exists nil. split. left. reflexivity.
destruct h1 as [h1l | h1r]. subst. simpl. reflexivity.
contradiction.
intros lp h1. exists nil. split. left. reflexivity.
destruct h1 as [h1l | h1r]. subst. simpl. reflexivity.
contradiction.
intros; contradiction.
intros lp h2.
rewrite in_map_iff in h2.
destruct h2 as [tr [h2 h3]].
destruct (in_app_or _ _ _ h3) as [h4 | h5]. clear h3.
rewrite in_map_iff in h4. 
destruct h4 as [lp' [h4 h5]].
specialize (h1 _ h5).
destruct h1 as [l h1].
destruct h1 as [h1l h1r].
exists ((p i j)::l).
rewrite in_map_iff.
split.
exists ((p i j), l).
simpl.
split. reflexivity.
apply in_or_app.
left.
rewrite in_map_iff.
exists l. split; auto.  rewrite <- h2.
simpl. rewrite <- h4.
simpl. f_equal.
assumption.
rewrite (surjective_pairing tr) in h5. clear h3.
rewrite in_prod_iff in h5.
destruct h5 as [h5l h5r].
specialize (h1 _ h5r).
destruct h1 as [l [h1l h1r]]. 
rewrite in_map_iff in h5l.
destruct h5l as [a h5l]. destruct h5l as [h5a h5b].
exists ((p a j)::l). split.
rewrite in_map_iff.
exists ((p a j), l). split.
simpl. reflexivity.
apply in_or_app. right.
rewrite in_prod_iff. split.
rewrite in_map_iff. exists a. split; auto.
assumption. 
rewrite <- h2.
simpl. rewrite <- h5a. simpl. f_equal.
assumption.
Qed.


Lemma in_map_list_of_lists_seqs_map' :
  forall {T U V:Type} (p:T->U->V) (li:list T) (lj:list U)
         (ls:{lp:list (U*T) | In lp (list_power lj li)})
         (pfi:NoDup li) (pfj:NoDup lj) (def:T),
    In ls (map_sig (list_power lj li)) ->
    exists l:list V,
      In l (list_of_lists_seqs
          (map (fun j : U => map (fun i : T => p i j) li) lj)) /\
      l = (map
             (fun j : U =>
              p
                ((in_list_power_fpl lj li (proj1_sig ls) pfj
                    (list_power_no_dup lj li (proj1_sig ls) pfj pfi
                       (proj2_sig ls)) (proj2_sig ls), def) l-> j) j) lj).
intros T U V p li lj ls h1 h2 def h3.
pose proof (in_list_power_in_list_of_lists_seqs_map_map_pair' _ _ _ (proj2_sig ls)) as h4.
pose proof (in_list_of_lists_seqs_map_map_pair_rev' p _ _ _  h4) as h5.
destruct h5 as [l h5].
destruct h5 as [h5l h5r].
exists l.
split; auto.
pose proof (in_list_power_synced _ _ _ (proj2_sig ls)) as h6.
pose proof (list_power_no_dup _ _ _  h2 h1 (proj2_sig ls)) as h7.
pose proof (in_list_power_fpl _ _ _ h2 h7 (proj2_sig ls) ) as h8.
pose proof (fpl_impl_map _ _ h2 _ h8 def h6) as h9.
rewrite h5r.
rewrite h9 at 1.
rewrite map_map.
apply map_ext_in.
simpl.
intros x h10.
f_equal. f_equal.
apply proof_irrelevance.
Qed.


Lemma sync_fpl : forall {T U:Type} {la:list T} {lb:list U}
                        {lp:list (T*U)},
                   functionally_paired_l la lb lp ->
                   exists lp':list (T*U),
                     functionally_paired_l la lb lp' /\
                     synced la lp' /\
                     list_to_set lp = list_to_set lp'.
intros T U la lb lp h1.
inversion h1 as [h2 h3].
pose (fun (x:{t:T | In t la}) => 
        proj1_sig (constructive_definite_description 
                     _ (h2 (proj1_sig x) (proj2_sig x)))) as f.
pose (fun (x:{t:T | In t la}) => (proj1_sig x, f x)) as g.
pose (map g (map_sig la)) as lp'.
exists lp'.
repeat split.
intros x h4.
exists (f (exist _ x h4)).
red.
repeat split.
unfold f.
destruct constructive_definite_description as [b h6].
simpl in h6. simpl. destruct h6 as [h6l h6r].
assumption.
unfold lp'.
rewrite in_map_iff. unfold g.
exists (exist _ x h4). simpl. split.
reflexivity.
apply map_sig_in.
intros b h5.
unfold f.
destruct constructive_definite_description as [b' h6].
simpl in h6. simpl.
unfold lp' in h5.
destruct h5 as [h5l h5r].
rewrite in_map_iff in h5r.
destruct h5r as [x' h7].
destruct h7 as [h7l h7r].
unfold g in h7l.
inversion h7l as [h8].
unfold f in H.
unfold f.
destruct constructive_definite_description as [a h9].
simpl. simpl in H. subst.
destruct h9 as [h9l h9r]. destruct h6 as [h6l h6r].
apply (fpl_functional h1 _ _ _ h6r h9r).
unfold lp' in H.
unfold g in H.
rewrite in_map_iff in H.
destruct H as [x h4].
destruct h4 as [h4l h4r].
rewrite surjective_pairing in h4l.
inversion h4l.
apply proj2_sig.
unfold lp' in H. unfold g in H.
rewrite in_map_iff in H.
destruct H as [x h4].
destruct h4 as [h4l h4r].
rewrite surjective_pairing in h4l.
inversion h4l.
unfold f.
destruct constructive_definite_description as [x' h6].
simpl. destruct h6; assumption.
unfold lp'.
rewrite map_map. simpl.
apply map_map_sig_compat.
apply Extensionality_Ensembles.
red. split.
red.
(* <= *)
intros pr h4.
rewrite <- list_to_set_in_iff.
rewrite <- list_to_set_in_iff in h4.
unfold lp'.
unfold g.
rewrite in_map_iff.
rewrite (surjective_pairing pr) in h4.
pose proof (fpl_in_dom _ _ _ h1 _ h4) as h5.
simpl in h5.
exists (exist _ (fst pr) h5). 
simpl. split.
rewrite surjective_pairing.
unfold f.
destruct constructive_definite_description as [x h6]. simpl. simpl in h6.
destruct h6 as [h6l h6r].
pose proof (fpl_functional  h1 _ _ _ h4 h6r). subst.
reflexivity.
apply map_sig_in.
red.
intros pr h4.
rewrite <- list_to_set_in_iff.
rewrite <- list_to_set_in_iff in h4.
unfold lp' in h4.
unfold g in h4. rewrite in_map_iff in h4.
destruct h4 as [x h4].
destruct h4 as [h4l h4r].
rewrite <- h4l.
unfold f.
destruct constructive_definite_description as [b h5].
simpl.
destruct h5; assumption.
Qed.

Lemma cardinal_length_compat : forall {T:Type} (l:list T), NoDup l -> cardinal _ (list_to_set l) (length l).
intros T l.
induction l as [|a l h1].
intros; simpl. constructor.
simpl; intro h2.
inversion h2 as [|x l' h4 h5]. clear h2. subst.
constructor.
apply h1; auto.
rewrite <- list_to_set_in_iff.
assumption.
Qed.


Lemma list_to_set_no_dup_length : forall {T:Type} (l:list T) (E:Ensemble T),
                                    list_to_set l = E -> NoDup l ->
                                    exists !n, length l = n.
intros T l.
induction l as [|a l h1].
simpl. intros. exists 0. red. split; auto.
simpl.
intros E h2 h3.
inversion h3 as [|x l' h4 h5].
subst. clear h3.
specialize (h1 _ (eq_refl _ )).
specialize (h1 h5).
destruct h1 as [n h1]. red in h1. destruct h1 as [h1 h2].
exists (S n). red. split.
f_equal. assumption.
intros x h6.
specialize (h2 _ (eq_refl _)).
rewrite <- h2 in h6.
assumption.
Qed.

Fixpoint maxl (l:list nat) :=
  match l with 
    | nil => 0
    | a::l' => max a (maxl l')
  end.

Lemma maxl_compat : 
  forall (l:list nat) (n:nat),
    In n l -> n <= maxl l.
intro l.
induction l as [|a l h1]; simpl.
intros; contradiction. 
intros n h2.
destruct h2 as [h2l | h2r]. subst.
rewrite le_max_iff. left. auto with arith.
specialize (h1 _ h2r).
rewrite le_max_iff.
right. assumption.
Qed.

Fixpoint minl_aux (l:list nat) (acc:nat) :=
  match l with
    | nil => acc
    | a::l' => minl_aux l' (min acc a)
  end.

Lemma minl_aux_compat :
  forall (l:list nat) (a:nat),
    minl_aux l a <= a.
intro l. induction l as [|b l h1].
simpl. auto with arith.
intros a. simpl.
unfold min.
destruct (le_lt_dec a b) as [h2 | h3].
apply h1. specialize (h1 b).
omega.
Qed.

Definition minl (l:list nat) :=
  match l with 
    | nil => O
    | a::l' => minl_aux l' a
  end.


Lemma minl_aux_O : 
  forall (l:list nat), minl_aux l 0 = 0.
intro l. induction l as [|a l h1].
simpl. reflexivity.
simpl. rewrite min_comm. rewrite min_O.
assumption.
Qed.

Lemma minl_aux_mono : 
  forall (l:list nat) (a b:nat),
    a <= b ->
    minl_aux l a <= minl_aux l b.
intro l. induction l as [|a l h1].
simpl. auto.
intros a' b h2.
simpl.
unfold min.
destruct (le_lt_dec a' a) as [h3 | h4]; destruct (le_lt_dec b a) as [h5 | h6].
apply (h1 _ _ h2).
apply (h1 _ _ h3).
assert (h6:a <= b). omega.
apply (h1 _ _ h6).
omega.
Qed.


Lemma minl_aux_le_minl : 
  forall (l:list nat) (a:nat),
    l <> nil ->
    minl_aux l a <= minl l.
intro l.
induction l as [|a l h1].
simpl. intros ? h1. contradict h1. reflexivity.
intros n h2.
simpl.
apply minl_aux_mono.
rewrite min_le_iff. right.
auto with arith.
Qed.


Lemma minl_compat : 
  forall (l:list nat) (n:nat),
    In n l -> minl l <= n.
intro l; induction l as [|a l h1].
intros; contradiction.
intros n h2. simpl.
destruct h2 as [h3 | h4]. subst.
apply minl_aux_compat.
specialize (h1 _ h4).
assert (h5:l <> nil). intro h5. subst. contradiction.
pose proof (minl_aux_le_minl l a h5). 
omega.
Qed.

Fixpoint plusl (l:list nat) :=
  match l with
    | nil => 0
    | a::l' => a + (plusl l')
  end.



Lemma plusl_map_remove : 
  forall {T:Type} (l:list T) (f:T->nat) (x:T), 
    NoDup l -> In x l ->
    plusl (map f l) = (f x) + (plusl (map f (remove eq_dec x l))).
intros T l. induction l as [|a l h1]; simpl. intros; contradiction.
intros f x h2 h3.
destruct h3 as [h3l | h3r]. subst.
destruct (eq_dec x x) as [h3 | h4].
pose proof (no_dup_cons_nin _ _ h2) as h4.
rewrite <- remove_not_in'; auto. 
contradict h4. reflexivity.
pose proof (no_dup_cons _ _ h2) as h4.
specialize (h1 f x h4 h3r).
rewrite h1.
destruct (eq_dec x a) as [h5 | h6]. subst.
pose proof (no_dup_cons_nin _ _ h2). contradiction.
simpl.
rewrite plus_assoc.
rewrite (plus_comm (f a) (f x)).
rewrite plus_assoc.
reflexivity.
Qed.


Lemma plusl_nat_valued_fun_functional :
  forall {T:Type} (l l':list T), 
    NoDup l -> NoDup l' ->
    list_to_set l = list_to_set l' ->
    forall f:T->nat,
      plusl (map f l) = plusl (map f l').
intros T l.
induction l as [|a l h1];simpl; auto.
intros l' h1 h2 h3. symmetry in h3.
pose proof (empty_set_nil _ h3).
subst. simpl. reflexivity.
intros l' h2 h3 h4.
pose proof (subtract_remove_compat l' a) as h5.
rewrite <- h4 in h5.
pose proof (no_dup_cons_nin _ _ h2) as h6.
rewrite list_to_set_in_iff in h6.
rewrite sub_add_compat_nin in h5; auto.
pose proof (no_dup_remove _ a h3) as h7.
pose proof (no_dup_cons _ _ h2) as h8.
specialize (h1 _ h8 h7 h5).
intro f. rewrite h1.
rewrite (plusl_map_remove _ _ a h3).
reflexivity.
pose proof (Add_intro2 _ (list_to_set l) a) as h9.
rewrite h4 in h9.
rewrite list_to_set_in_iff.
assumption.
Qed.


(************************)
Lemma nprod_inj : 
  forall {T:Type} {n:nat} (x1 x2:T^n) (def:T),
    x1 = x2 -> forall m:nat, m < n ->
    nth m (nprod_to_list _ _ x1) def = 
    nth m (nprod_to_list _ _ x2) def.
intros; subst; auto.
Qed.

Fixpoint in_n {T:Type} {n:nat} (l:list T) : T^n -> Prop :=
  match n return (T^n-> Prop) with
    | O => (fun x => True)
    | S n' => (fun x => In (fst x) l /\ (in_n l (snd x)))
  end.

Lemma in_n_in_nprod_to_list_compat : 
  forall {T:Type} {n:nat} (l:list T) (t:T^n),
    in_n l t -> (forall x:T, In x (nprod_to_list _ _ t) -> In x l).
intros T n.
induction n as [|n h1]; simpl.
intros l tt. 
tauto.
intros l t.
intros h2 ft' h3.
destruct t as [ft st].
destruct h3 as [h3l | h3r].
subst.
simpl in h2. 
destruct h2; auto.
simpl in h2.
destruct h2 as [h2l h2r].
specialize (h1 _ _  h2r _ h3r).
assumption.
Qed.

Lemma in_nprod_to_list_in_n_compat : 
  forall {T:Type} {n:nat} (l:list T) (t:T^n),
    (forall x:T, In x (nprod_to_list _ _ t) -> In x l) -> in_n l t.
intros T n.
induction n as [|n h1]; simpl.
intros. tauto.  
intros l t h2.
destruct t as [ft st]. simpl.
assert (h3:forall x:T, In x (nprod_to_list T n st) -> In x l).
  intros x h3.
  pose proof (in_cons ft x _ h3) as h4.
apply h2; auto.
specialize (h1 _ _ h3). 
specialize (h2 ft (in_eq _ _)).
split; auto.
Qed.

Lemma in_n_in_nprod_to_list_compat_iff : 
  forall {T:Type} {n:nat} (l:list T) (t:T^n),
    in_n l t <-> (forall x:T, In x (nprod_to_list _ _ t) -> In x l).
intros; split; [apply in_n_in_nprod_to_list_compat | apply in_nprod_to_list_in_n_compat].
Qed.


Lemma in_n_cons : 
  forall {T:Type} {n:nat} (l:list T) (a:T) (t:T^n),
    in_n l t -> in_n (a::l) t.
intros T n l a t h1.
rewrite in_n_in_nprod_to_list_compat_iff in h1.
rewrite in_n_in_nprod_to_list_compat_iff.
intros x h2.
specialize (h1 _ h2).
right.
assumption.
Qed.


Lemma in_n_nprod_to_list : 
  forall {T:Type} {n:nat} (t:T^n), in_n (nprod_to_list _ _ t) t.
intros T n.
induction n as [|n h1]; simpl. auto.
intro t. split.
destruct t; simpl. left. auto.
destruct t; simpl. apply in_n_cons. apply h1.
Qed.


(* maybe mobe elsewhere*)
Lemma map_fpl_app_cons_compat : 
  forall {T U:Type} {la:list T} {lb:list U} {lp:list (T*U)} {a:T}
        (pf':functionally_paired_l (a::la) lb lp) 
        (pf:functionally_paired_l la lb (tl lp))
        (def:U), NoDup lp ->
    map (fpl_app pf def) la = map (fpl_app pf' def) la.
intros T U la lb lp a h1 h2 def h3.
apply map_ext_in.
intros x h11.
unfold fpl_app.
destruct constructive_definite_description as [y' h15].
destruct constructive_definite_description as [y h13].
simpl.
destruct in_dec as [h16 | h17].
destruct in_dec as [h18 | h19].
simpl. 
pose proof (in_tl _ _  h15) as h19.
apply (fpl_functional h1 _ _ _ h19 h13).
contradict h19. right. assumption.
contradiction.
Qed.


Lemma nprod_to_fpl_ex : 
  forall {T U:Type} (la:list T) (lb:list U) (def:U),
    NoDup la ->
    let n := length la in 
    forall (t:U^n), 
      in_n lb t ->
      exists! lp:list (T*U), NoDup lp /\ 
        synced la lp /\
        exists (pf:functionally_paired_l la lb lp),
          map (fpl_app pf def) la = nprod_to_list _ _ t.
intros T U la.
induction la as [|a la h1]; simpl.
intros.
exists (list_prod nil lb). 
red. simpl.
split. split. constructor. split; try constructor. simpl. reflexivity.
constructor. intros; contradiction. intros; contradiction.
reflexivity. 
intros pl h1.
destruct h1 as [h1l [h1c h1r]].
destruct h1r as [h1r].
symmetry.
apply (fpl_empty1_s _ _ h1r).
intros lb def h0 t h2.
destruct h2 as [h2l h2r].
inversion h0 as [|? ? h0l h0r].  subst.
specialize (h1 lb def h0r _ h2r).
destruct h1 as [lp h1].
red in h1. destruct h1 as [h1l h1r].
destruct h1l as [h3 h4].
destruct h4 as [h4 h5]. 
exists ((a, (fst t))::lp).
red.
split. split.
constructor.
intro hi. destruct h5 as [h5l h5r].
pose proof (fpl_in_dom _ _ _ h5l _ hi) as h0'.
simpl in h0'. contradiction. assumption. 
split.
apply synced_cons''. assumption.
destruct h5 as [h5 h6].
pose proof (fpl_cons_nin _ _ _ _ _ h0l h2l h5) as h8.
exists h8.
assert (h10:map (fpl_app h8 def) la = map (fpl_app h5 def) la).
  apply map_ext_in.
  intros x h11.
  unfold fpl_app.
  destruct constructive_definite_description as [y h13].
  destruct constructive_definite_description as [y' h15].
  destruct in_dec as [h16 | h17].
  destruct in_dec as [h18 | h19].
  simpl.
  destruct h13 as [h13l | h13r].
  inversion h13l. subst. contradiction.
  apply (fpl_functional h5 _ _ _ h13r h15).
  contradiction. simpl.
  destruct in_dec.
  contradict h17.
  right. assumption. subst. reflexivity. 
unfold nprod. unfold nprod in h10.
rewrite h10.
rewrite h6.
pose proof (fpl_app_compat h8 a def) as h11. 
destruct in_dec.
destruct h11 as [h11l | h11r].
inversion h11l.
destruct t. simpl. reflexivity.  
pose proof (fpl_in_dom _ _ _ h5 _ h11r) as h12.
simpl in h12. contradiction.
contradict n. left. reflexivity.
intros lp' h6.
destruct h6 as [h6l [h6c h6r]].
destruct h6r as [h7 h8].
pose proof (synced_cons' _ _ _ h6c) as h9.
assert (h10:synced la (tl lp') /\ (exists pf : functionally_paired_l la lb (tl lp'), map (fpl_app pf def) la = nprod_to_list U (length la) (snd t))).
  split; auto.
  assert (h10:In (a, (h7, def) l-> a) lp').
    pose proof (fpl_app_compat h7 a def) as h11.
    destruct in_dec.
    assumption.
    contradict n. left. reflexivity.
  pose proof (fpl_cons _ _ _ (a, (h7, def) l-> a) h10 h0 h7) as h11.
  assert (h12:remove eq_dec (a, (h7, def) l->a) lp' = tl lp').

    assert (h13:hd (a, (h7, def) l-> a) lp' = (a, (h7, def) l-> a)).  
    inversion h6c as [h12]. 
    apply injective_projections. simpl. 
    pose proof (synced_hd (a, fpl_app h7 def a) _ _ h6c) as h13.
    simpl in h13.
    assumption.
    simpl.
    assert (h13:lp' <> nil). intro h14. subst. contradiction.
    pose proof (in_hd_not_nil _ (a, fpl_app h7 def a) h13) as h14.
    pose proof (in_fpl_snd _ _ _ h7 _ def h14) as h15.  
    rewrite h15.
    f_equal. 
    assert (h13': (hd a (a::la)) = a). simpl. reflexivity. 
    rewrite <- h12 in h13'.
    assert (h16:a = fst (a, fpl_app h7 def a)). auto.
    rewrite h16 in h13' at 1.
    rewrite hd_map in h13'.
    assumption.
    pose proof (remove_hd_no_dup _ (a, (h7, def) l-> a) h6l) as h14.
    rewrite h13 in h14.
    assumption.
  rewrite h12 in h11.
  exists h11.
  destruct t.
  inversion h8 as [h13]. simpl.
  rewrite (map_fpl_app_cons_compat h7 h11 def h6l).
  assumption.
  pose proof (no_dup_tail _  h6l) as h11.
specialize (h1r _ (conj h11 h10)).
subst.
destruct t. simpl.
inversion h8.
assert (h12:lp' <> nil).
  intro h13.
  subst.
  simpl in h9. destruct h6c as [h6c]. simpl in h6c. discriminate.
pose proof (list_decompose _ (a, u) h12) as h13.
rewrite h13.
f_equal.
rewrite surjective_pairing.
f_equal. 
rewrite (synced_hd _ _ _ h6c).
simpl. reflexivity.
pose proof (in_hd_not_nil _ (a, u) h12 ) as h14. 
pose proof (in_fpl_snd _ _ _ h7 _ def h14) as h15.
rewrite h15. 
inversion h6c as [h16].
pose proof (in_eq a la) as h17.
rewrite <- h16 in h17.
pose proof (hd_map (fun pr:T*U=>fst pr) (a, u) lp') as h18.
rewrite h16 in h18.
simpl in h18.
rewrite <- h18.
unfold fpl_app.
simpl.
reflexivity.
Qed.

Definition nprod_to_fpl {T U:Type} (la:list T) (lb:list U) (def:U)
           (pfla:NoDup la) (t:U^(length la)) (pfin:in_n lb t) := 
  proj1_sig (constructive_definite_description _ (nprod_to_fpl_ex la lb def pfla t pfin)).

Lemma nprod_to_fpl_compat : 
  forall {T U:Type} (la:list T) (lb:list U) (def:U)
         (pfla:NoDup la),
    let n:= length la in 
    forall (t:U^(length la)) (pfin:in_n lb t),
      let lp := (nprod_to_fpl la lb def pfla t pfin) in
      NoDup lp /\ synced la lp /\
      (exists pf : functionally_paired_l la lb lp,
         map (fpl_app pf def) la = nprod_to_list _ n t).
intros.
unfold nprod_to_fpl in lp.
destruct constructive_definite_description.
simpl in lp. unfold lp.
assumption.
Qed.

Lemma nprod_to_list_inj : 
  forall {T:Type} (n:nat) (t t':T^n),
    nprod_to_list _ n t = nprod_to_list _ n t' ->
    t = t'.
intros T n.
induction n as [|n h0].
simpl. intros t t' h1. destruct t; destruct t'. reflexivity.
simpl.
intros t t' h1.
destruct t as [ft st]; destruct t' as [ft' st'].
inversion h1.  subst. clear h1.
specialize (h0 st st' H1).
apply injective_projections; auto.
Qed.


Lemma nprod_to_fpl_inj : 
  forall {T U:Type} (la:list T) (lb:list U) (def:U) 
         (pfla:NoDup la) (t t':U^(length la)) (pfin:in_n lb t) 
         (pfin':in_n lb t'),
    nprod_to_fpl la lb def pfla t pfin = nprod_to_fpl la lb def pfla t' pfin' -> t = t'.
intros T U la lb def h1 t t' h2 h3 h4.
pose proof (nprod_to_fpl_compat la lb def h1 t h2) as h5.
pose proof (nprod_to_fpl_compat la lb def h1 t' h3) as h6.
destruct h5 as [h5a [h5b h5c]].
destruct h6 as [h6a [h6b h6c]].
destruct h5c as [h5c h5d].
destruct h6c as [h6c h6d].
generalize dependent h6c.
rewrite <- h4.
intro h6c.
assert (h7:h5c = h6c). apply proof_irrelevance. subst.
intro h7.
rewrite h5d in h7.
apply (nprod_to_list_inj _ _ _ h7).
Qed.


Lemma nprod_to_fpl_nil : 
  forall {T U:Type} (lb:list U) (def:U) (pfla:NoDup (nil (A:=T))) (t:unit) (pfin:True), 
    nprod_to_fpl nil lb def pfla t pfin = nil.
intros T U lb def h1 t h2.
pose proof (nprod_to_fpl_compat nil lb def h1 t h2) as h3.
destruct h3 as [h3a [h3b h3c]].
apply synced_nil1. assumption.
Qed.


Lemma nprod_to_list_nprod_of_list_eq : 
  forall {T:Type} (l:list T), 
    nprod_to_list _ (length l) (nprod_of_list _ l) = l.
intros T l.
induction l as [|a l h1]; simpl.
reflexivity.
f_equal; auto.
Qed.


Lemma fpl_in_map_snd_impl_in_lb : 
  forall {T U:Type} (la:list T) (lb:list U) 
         (lp:list (T*U)),
    functionally_paired_l la lb lp ->
  forall (y:U),
    In y (map (snd (B:=U)) lp) -> In y lb.
intros T U la lb lp h1 y.
intro h2.
rewrite in_map_iff in h2.
destruct h2 as [pr [h2l h2r]].
destruct h1 as [h1l h1r].
specialize (h1r _ h2r).
destruct h1r as [h1a h1b].
subst.
assumption.
Qed.

Lemma map_fpl_app : forall {T U:Type} {la:list T} {lb:list U}
                           {lp:list (T*U)} {def:U}
                           (pf:functionally_paired_l la lb lp),
                      synced la lp ->
                      map (fpl_app pf def) la = map (snd (B:=U)) lp.
intros T U la lb lp def h1 h2.
destruct h2 as [h2].
pose proof (f_equal (map (fpl_app h1 def)) h2) as h3.
rewrite <- h3.
rewrite map_map.
apply map_ext_in.
intros pr h4.
rewrite (in_fpl_snd _ _ _ h1 _ def h4).
unfold fpl_app.
reflexivity.
Qed.


Lemma list_fst_snd_eq : 
  forall {T U:Type} (lp lp':list (T*U)),
    map (fst (B:=U)) lp = map (fst (B:=U)) lp' ->
    map (snd (B:=U)) lp = map (snd (B:=U)) lp' ->
    lp = lp'.
intros T U lp. 
induction lp as [|a lp h1]; auto; simpl.
intros lp' h2 h3. 
symmetry in h3.
pose proof (map_eq_nil _ _ h3).
subst. reflexivity. 
intro lp'.
induction lp' as [|a' lp' h2]; simpl.
intros h2 h3. discriminate.
intros h3 h4.
inversion h3; inversion h4.
specialize (h1 _ H1 H3).
f_equal.
apply injective_projections; auto.
assumption.
Qed.


Lemma fpl_app_eq : 
  forall {T U:Type} {la:list T} {lb:list U}
         {lp lp':list (T*U)}
         (pf: functionally_paired_l la lb lp)
         (pf':functionally_paired_l la lb lp')
         (def:U),
    lp = lp' ->
    fpl_app pf def = fpl_app pf' def.
intros T U la lb lp lp' h1 h2 def h3.
subst.
assert (h3:h1 = h2). apply proof_irrelevance.
subst.
reflexivity.
Qed.

Lemma fpl_app_eq' : 
  forall {T U:Type} {la:list T} {lb:list U}
         {lp lp':list (T*U)}
         (pf: functionally_paired_l la lb lp)
         (pf':functionally_paired_l la lb lp')
         (def:U),
    lp = lp' ->
    forall x:T,
    (pf, def) l-> x = (pf', def) l-> x.
intros T U la lb lp lp' h1 h2 def h3 x.
simpl.
pose proof (fpl_app_eq h1 h2 def h3) as h4.
rewrite h4.
reflexivity.
Qed.


Fixpoint nprod_of_list_n 
         {T:Type} (l:list T) (n:nat) (def:T) {struct n} : 
  T ^ n :=
  match n return T^n with 
    | O => tt
    | S n' => ((hd def l), (nprod_of_list_n (tl l) n' def))
  end.



Lemma nprod_to_fpl_surj : 
  forall {T U:Type} (la:list T) (lb:list U) (lp:list (T*U))
    (pf:functionally_paired_l la lb lp) (pfla:NoDup la), 
      NoDup lp -> synced la lp ->
    forall (def:U),
      let n := length la in 
    exists (t:U^n) (pfin:in_n lb t),
      map (fpl_app pf def) la = nprod_to_list _ n t /\
      lp = (nprod_to_fpl la lb def pfla t pfin).
intros T U la lb lp h1 h2 h3 h4 def.
pose proof (synced_length _ _ h4) as h5.  
pose (nprod_of_list _ (map (@snd _ _) lp)) as t.
intro n.
 assert (h6':length (map (snd (B:=U)) lp) = n).
  unfold n. rewrite h5. apply map_length.
subst.
pose (transfer_dep h6' t) as t'.
exists t'.
assert (h8:in_n lb t).
  rewrite in_n_in_nprod_to_list_compat_iff.
  intros x h8.
  unfold t in h8.
  rewrite nprod_to_list_nprod_of_list_eq in h8.
  eapply fpl_in_map_snd_impl_in_lb; auto.
  apply h1; auto. assumption. 
pose (fun n':{n:nat & U^n} => (in_n (n:=(projT1 n')) lb (projT2 n'))) as P.
pose proof (transfer_dep_prop _ _ h6' t P) as h9.
unfold P in h9. simpl in h9.
unfold t'.
rewrite h9 in h8.
exists h8.
split. 
pose proof (fpl_impl_map _ _ h2 _ h1 def h4) as h10.
pose proof (f_equal (map (@snd T U)) h10) as h11.
rewrite map_map in h11. simpl in h11.
assert (h12:(fun x :T => fpl_app h1 def x) = (fpl_app h1 def)).
  apply functional_extensionality. auto.
rewrite h12 in h11.
rewrite <- h11.
unfold t. 
pose proof (transfer_dep_existT _ _ h6' (nprod_of_list U (map (snd(B:=U)) lp))) as h13.
dependent rewrite <- h13.
rewrite nprod_to_list_nprod_of_list_eq.
reflexivity.
pose proof (nprod_to_fpl_compat _ _ def h2 _ h8) as h10. 
destruct h10 as [h10a [h10b h10c]].
destruct h10c as [h10 h11].
unfold t in h11.
pose proof (transfer_dep_existT _ _ h6' (nprod_of_list U (map (snd (B:=U)) lp))) as h13.
rewrite <- nprod_to_list_nprod_of_list_eq in h11 at 1.
pose proof (fpl_impl_map _ _ h2 _ h10 def h10b) as h14.
pose proof (f_equal (map (@snd T U)) h14) as h15. 
rewrite h14.
pose proof (fpl_impl_map _ _ h2 _ h1 def h4) as h10'.
pose proof (f_equal (map (@snd T U)) h10') as h11'.
rewrite map_map in h11'. simpl in h11'.
rewrite h10' at 1.
f_equal.
apply functional_extensionality.
intro x.
f_equal.
assert (h16:lp = (nprod_to_fpl la lb def h2
       (transfer_dep  h6' t) h8)).   
  unfold t. 
  unfold nprod_to_fpl.
  destruct constructive_definite_description as [lp' h18]. simpl.
  destruct h18 as [h18a [h18b h18c]].
  destruct h18c as [h19 h20].
  dependent rewrite <- h13 in h20.
  rewrite nprod_to_list_nprod_of_list_eq in h20.
  rewrite map_fpl_app in h20.
  apply list_fst_snd_eq.
  destruct h18b as [h18b].
  pose proof h4 as h4'. clear h4.
  destruct h4' as [h4'].
  rewrite <- h18b in h4'.
  assert (h21:(fun pr:T*U=>fst pr) = (fst (B:=U))).
    apply functional_extensionality; auto.
  rewrite h21 in h4'.
  assumption.
  symmetry. assumption.
  assumption.  clear h10'. clear h11'.  
  apply fpl_app_eq'.
  assumption.
Qed.

Definition im_nprod {T U:Type} {n:nat} (t:T^n) (f:T->U) : U^n.
induction n as [|n h1]; simpl.
constructor.
simpl in t.
refine (f (fst t), (h1 (snd t))).
Defined.

Lemma in_n_Sn : forall {T:Type} {n:nat} (l:list T) (t:T^n) (x:T),
                  in_n (n:=S n) l (x, t) -> In x l /\ in_n l t.
intros T n l t x h1.
rewrite in_n_in_nprod_to_list_compat_iff in h1.
split.
apply h1. simpl. left. reflexivity.
assert (h2:forall x0 : T, In x0 (nprod_to_list T n t) -> In x0 l).
  intros x' h2.
  specialize (h1 x').
  simpl in h1.
  apply h1. right. assumption.
rewrite <- in_n_in_nprod_to_list_compat_iff in h2.
assumption.
Qed.


Lemma im_nprod_im_nprod : 
  forall {T U V:Type} {n:nat} (t:T^n) (f:T->U) (g:U->V),
    im_nprod (im_nprod t f) g = im_nprod t (fun x=> g (f x)).
intros T U V n.
induction n as [|n h1].
intros t f g; simpl. reflexivity.
intros t f g; simpl.
apply injective_projections; simpl. reflexivity.
destruct t as [x t]. simpl.
apply h1.
Qed.

Lemma im_nprod_inj : 
  forall {T U:Type} {n:nat} (t1 t2:T^n) (f:T->U),
    FunctionProperties.injective f -> im_nprod t1 f = im_nprod t2 f ->
    t1 = t2.
intros T U.
induction n as [|n h1]; intros t1 t2 f h2 h3; simpl.
simpl in h3.
destruct t1. destruct t2. reflexivity.
destruct t1 as [ft1 st1].
destruct t2 as [ft2 st2].
simpl in h3.
inversion h3 as [h4].
specialize (h1 _ _ _ h2 H).
red in h2.
specialize (h2 _ _ h4).
apply injective_projections; auto.
Qed.


Lemma im_nprod_new_proj2_ex : 
  forall {T:Type} {l:list T} {n:nat} (t':{t:T^n | in_n l t}),
         exists! t:{x:T|In x (nprod_to_list _ _ (proj1_sig t'))}^n,
           proj1_sig t' = im_nprod t (@proj1_sig _ _).
intros T l n t'.           
induction n as [|n ret]. 
(* 0 *)
simpl. exists tt.  red.  simpl in t'.  destruct t'.  simpl.  destruct x.  split. reflexivity. intros tt' ?. destruct tt'. assumption.
(* S n*)
destruct t' as [t' h2].  destruct t' as [ft' st'];  simpl. 
apply in_n_Sn in h2.
destruct h2 as [h2l h2r].
specialize (ret (exist _ _ h2r)). simpl in ret. 
pose (exist (fun x=>ft'=x \/ In x (nprod_to_list T n st')) ft' (or_introl (In ft' (nprod_to_list T n st')) (eq_refl _))) as ft.
destruct ret as [ret h3].
red in h3. 
pose  (im_nprod ret (U:={x:T|ft' = x \/ In x (nprod_to_list T n st')}) (fun x=>(exist _ (proj1_sig x) (or_intror  (ft' = (proj1_sig x)) (proj2_sig x))))) as st.
exists (ft, st).
red. simpl. split.
unfold st. 
apply injective_projections; simpl. reflexivity. 
rewrite im_nprod_im_nprod. simpl.
destruct h3 as [h3l h3r].
rewrite h3l at 1.
f_equal.
intros x h4.
destruct x as [fx sx].
simpl in h4.
inversion h4 as [h5].
destruct fx as [fx h6].
simpl in h4.
simpl in h5.
unfold ft. unfold st.
apply injective_projections.
simpl.
destruct h6.
apply proj1_sig_injective.
simpl.
assumption.
apply proj1_sig_injective.
simpl.
assumption.
simpl.  
destruct h3 as [h3l h3r]. 
rewrite h3l in H at 1. 
apply im_nprod_inj with (f:=(proj1_sig (P:=fun x:T=>ft' = x \/ In x (nprod_to_list T n st')))).
red.
apply proj1_sig_injective.
rewrite im_nprod_im_nprod.
simpl.
assumption.
Qed.


Definition im_nprod_new_proj2 {T:Type} {l:list T} {n:nat} 
           (t':{t:T^n | in_n l t}) : {x:T|In x (nprod_to_list _ _ (proj1_sig t'))}^n := 
  (proj1_sig (constructive_definite_description _ (im_nprod_new_proj2_ex t'))).

Lemma im_nprod_new_proj2_compat : 
  forall {T:Type} {l:list T} {n:nat} 
           (t':{t:T^n | in_n l t}),
    proj1_sig t' = im_nprod (im_nprod_new_proj2 t') (@proj1_sig _ _).
intros T l n t'.
unfold im_nprod_new_proj2.
destruct constructive_definite_description.
simpl.
assumption.
Qed.


Lemma nprod_in_n_to_nprod_in_l_ex : 
  forall {T:Type} {l:list T} {n:nat}
         (t':{t:T^n|in_n l t}), exists! ret:{x:T|In x l}^n,
    proj1_sig t' = im_nprod ret (@proj1_sig _ _).
intros T l n t'.
pose (im_nprod_new_proj2 t') as t''.
pose proof (im_nprod_new_proj2_compat t') as h0.
destruct t' as [t' h1].
simpl in h0. 
pose proof h1 as h1'.
rewrite in_n_in_nprod_to_list_compat_iff in h1'.
simpl in t''.
simpl.
exists (im_nprod t'' (fun x=>(exist _ (proj1_sig x) (h1' _ (proj2_sig x))))).
red.
split.
rewrite im_nprod_im_nprod.
simpl.
unfold t''.
assumption.
intros x' h2.
unfold t''.
apply im_nprod_inj with (f:=(proj1_sig (P:=fun x:T => In x l))).
red.
apply proj1_sig_injective.
rewrite im_nprod_im_nprod.
simpl.
rewrite <- h2.
symmetry. assumption.
Qed.

Definition nprod_in_n_to_nprod_in_l {T:Type} (l:list T) (n:nat) 
           (t':{t:T^n | in_n l t}) : {x:T|In x l}^n :=
  (proj1_sig (constructive_definite_description _ (nprod_in_n_to_nprod_in_l_ex t'))).

Lemma nprod_in_n_to_nprod_in_l_compat : 
  forall {T:Type} {l:list T} {n:nat}
         (t':{t:T^n|in_n l t}),
    proj1_sig t' = im_nprod (nprod_in_n_to_nprod_in_l _ _ t') (@proj1_sig _ _).
intros T l n t'.
unfold nprod_in_n_to_nprod_in_l.
destruct constructive_definite_description.
simpl.
assumption.
Qed.

Lemma nprod_in_n_to_nprod_in_l_bij : 
  forall {T:Type} (l:list T) (n:nat), 
    bijective (nprod_in_n_to_nprod_in_l l n).
intros T l n.
red.
split.
red.
intros x1 x2 h1.
pose proof (f_equal (fun x => im_nprod x (@proj1_sig _ _)) h1) as h2.
simpl in h2.
do 2 rewrite <- nprod_in_n_to_nprod_in_l_compat in h2.
apply proj1_sig_injective. assumption.
red. 
intro t'.
induction n as [|n h1]. 
simpl in t'. simpl.
exists (exist _ tt I).
unfold nprod_in_n_to_nprod_in_l.
destruct constructive_definite_description.
simpl.
simpl in x. destruct t'; destruct x.
reflexivity.
simpl in t'.
destruct t' as [ft' st'].
specialize (h1 st').
destruct h1 as [t h2].
simpl.
destruct t as [t h1].
destruct ft' as [x h3].
exists (exist _ (x, t) (conj h3 h1)).
apply injective_projections.
simpl.
apply proj1_sig_injective.
simpl.
unfold nprod_in_n_to_nprod_in_l.
destruct constructive_definite_description as [y h4].
simpl. simpl in h4.
inversion h4.
reflexivity.
simpl.
unfold nprod_in_n_to_nprod_in_l.
destruct constructive_definite_description as [y h4].
simpl. simpl in h4. 
subst.
inversion h4.
apply im_nprod_inj with (f:=proj1_sig (P:=fun x0:T => In x0 l)).
red.
apply proj1_sig_injective.
subst.
rewrite <- nprod_in_n_to_nprod_in_l_compat.
simpl.
reflexivity.
Qed.

Lemma in_sig_fun_ex : 
  forall {T:Type} {l:list T}
         (x:{t:T|In t l}), 
            exists! y:{t:T|Ensembles.In (list_to_set l) t},
  proj1_sig x = proj1_sig y.
intros T l x.
destruct x as [x h1].
pose proof h1 as h1'.
rewrite list_to_set_in_iff in h1'.
exists (exist _ _ h1').
red.
simpl. split; auto.
intros y h2.
destruct y as [y h3].
simpl in h2. subst.
assert (h4:h3 = h1'). apply proof_irrelevance.
subst.
f_equal.
Qed.

Definition in_sig_fun {T:Type} {l:list T}
  (x:{t:T|In t l}):= 
  proj1_sig (constructive_definite_description _ (in_sig_fun_ex x)).

Lemma in_sig_fun_compat : forall {T:Type} {l:list T}
      (x:{t:T|In t l}), proj1_sig x = proj1_sig (in_sig_fun x).
intros T l x.
unfold in_sig_fun. destruct constructive_definite_description as [x' h1].
simpl. assumption.
Qed.

Lemma in_sig_fun_bij : forall {T:Type} (l:list T),
                         bijective (@in_sig_fun _ l).
intros T l.
red. split.
red.
intros x1 x2 h1.
pose proof (f_equal (@proj1_sig _ _) h1) as h2.
do 2 rewrite <- in_sig_fun_compat in h2.
apply proj1_sig_injective; auto.
red.
intro y.
destruct y as [y h1].
pose proof h1 as h1'.
rewrite <- list_to_set_in_iff in h1'.
exists (exist _ y h1').
apply proj1_sig_injective.
simpl.
rewrite <- in_sig_fun_compat.
simpl.
reflexivity.
Qed.

Lemma finite_list_sig : forall {T:Type} (l:list T),
                           FiniteT {x:T | In x l}.
intros T l.
pose proof (in_sig_fun_bij l) as h1.
apply bijective_impl_invertible in h1.

pose proof (list_to_set_finite l) as h2.
pose proof (Finite_ens_type _ h2) as h3.
apply (bij_finite _ _ 
                  (proj1_sig (function_inverse (@in_sig_fun _ l) h1))
                  h3 (invertible_impl_inv_invertible _ h1)).
Qed.

Lemma cardinal_full_set_in_l : 
  forall {T:Type} (l:list T),
    NoDup l ->
    cardinal _ (Full_set {x:T | In x l}) (length l).
intros T l h0.
pose (list_to_set l) as A.
induction l as [|a l h1].
simpl.
rewrite full_false_empty.
constructor.
simpl.
inversion h0. subst.
specialize (h1 H2).
pose (fun x':{x:T|In x l} => (exist (fun x=>a=x \/ In x l) (proj1_sig x') (or_intror (a=(proj1_sig x')) (proj2_sig x')))) as f.
assert (h3:FunctionProperties.injective f).
  red. unfold f.
  intros x1 x2 h4.
  destruct x1; destruct x2.
  simpl in h4.
  apply proj1_sig_injective.
  simpl.
  pose proof (f_equal (@proj1_sig _ _) h4) as h5. simpl in h5.
  assumption.
pose proof (injection_preserves_cardinal _ _ f _ _ h1 h3) as h4.
assert (h5:Im (Full_set {x:T | In x l}) f = [x':{x:T | a = x \/ In x l} | In (proj1_sig x') l]).
  apply Extensionality_Ensembles.
  red.  split.
  red.
  intros y h5.
  destruct h5 as [x h5 y]. subst.
  constructor.
  unfold f.
  simpl. apply proj2_sig.
  red.
  intros y h5.
  destruct h5 as [h5].
  destruct y as [y h6].
  simpl in h5.
  apply Im_intro with (exist _ y h5).
  constructor.
  unfold f.
  apply proj1_sig_injective.
  simpl. reflexivity.
rewrite h5 in h4.
assert (h6:Full_set {x:T | a = x \/ In x l} = 
           Add [x':{x:T | a = x \/ In x l} | In (proj1_sig x') l] (exist _ a (or_introl (In a l) (eq_refl _)))).
  apply Extensionality_Ensembles.
  red. split.
  red.
  intros x h6.
  destruct x as [x h7]. 
  destruct h7 as [h7l | h7r].
  right. rewrite h7l.
  constructor.
  constructor.
  constructor. simpl.
  assumption.
  red. constructor.
rewrite h6.
apply card_add.
assumption.
intro h7.
destruct h7 as [h7].
simpl in h7.
contradiction.
Qed.

Lemma cardinal_in_l : 
  forall {T:Type} (l:list T), 
    NoDup l -> 
    FiniteT_nat_cardinal {x:T | In x l} (finite_list_sig l) =
    length l.
intros T l h1.
apply FiniteT_nat_cardinal_cond.
apply cardinal_full_set_in_l; auto.
Qed.


Definition l_exp_seg_to_nprod_in_l 
           {T:Type} (l:list T) (n:nat) 
           (f:{m:nat | m < n} -> {x:T|In x l}) : {x:T | In x l} ^ n.
induction n as [|n h1].
simpl. refine tt.
simpl.
pose (fun m':{m:nat | m < n} => f (exist _ (proj1_sig m') (lt_S _ _ (proj2_sig m')))) as f'.
refine ((f (exist _ n (lt_n_Sn n)), (h1 f'))).
Defined.


Lemma l_exp_seg_to_nprod_in_l_compat : 
  forall {T:Type} (l:list T) (n:nat) 
         (f:{m:nat | m < n} -> {x:T|In x l})
         (def:{x:T|In x l})
         (x:nat) (pf:x < n),
    nth (n - x -1) (nprod_to_list _ _ (l_exp_seg_to_nprod_in_l _ _ f)) def =
    f (exist _ x pf).
intros T l n f def x.
revert f def.
induction n as [|n h1]; induction x as [|x h2].
simpl; intros; try omega.
intros; omega.
intros f def h2.
destruct (zerop n) as [h3 | h4].
subst; simpl.
assert (h3:h2 = lt_n_Sn 0). apply proof_irrelevance. subst.
reflexivity. 
simpl.
assert (h5:n-0 = n). auto with arith.
rewrite h5.
destruct n as [|n]. omega.
specialize (h1 (fun m' : {m : nat | m < S n} =>
            f
              (exist (fun m : nat => m < S (S n)) (proj1_sig m')
                 (lt_S (proj1_sig m') (S n) (proj2_sig m')))) def h4).
assert (h6:S n - 0 -1 = n). omega.
rewrite h6 in h1.
rewrite h1.
simpl.
assert (h7:h2 = (lt_S 0 (S n) h4)). apply proof_irrelevance.
f_equal.
f_equal. auto. 
intros f def h3.
assert (h4:x < n). omega.
assert (h5:S n - S x - 1 = n - x -1). omega. 
rewrite h5.
destruct (zerop (n - x -1)) as [h6 | h7].
rewrite h6. 
simpl.
assert (h7:n = S x). omega.
subst.
assert (h7:h3 = lt_n_Sn (S x)). apply proof_irrelevance.
subst.
reflexivity. 
destruct (lt_eq_lt_dec x (n-1)) as [[h10' | h11] | h12].
pose proof (S_pred _ _ h7) as h8.
rewrite h8.
simpl. 
assert (h9:pred (n - x -1) = n - S x - 1).  
  apply S_inj.  
  rewrite <- (S_pred _ _ h7).
  assert (h10:1 <= n - S x). 
  pose proof (lt_le_S _ _ h7) as h9.
  rewrite (minus_plus_simpl_l_reverse (n-x) 1 x) in h9.
  assert (h11:x <= n).  
    pose proof (lt_S _ _ h4) as h11.
    unfold lt in h11.
    apply le_S_n. assumption.
  rewrite <- (le_plus_minus _ _ h11) in h9.
  rewrite <- (S_compat x).
  assumption.
  rewrite (minus_Sn_m _ _ h10).
  do 2 rewrite <- pred_of_minus.
  f_equal.
  unfold lt in h4.
  rewrite (minus_Sn_m _ _ h4).
  auto with arith.
rewrite <- h9 in h1.

assert (h13 : S x < n). unfold lt in h10'.
  pose proof (le_lt_n_Sm _ _ h10') as h14.
  rewrite <- pred_of_minus in h14.
  rewrite <- (S_pred _ _ h4) in h14.
  assumption.
specialize (h1 (fun m' : {m : nat | m < n} =>
            f
              (exist (fun m : nat => m < S n) (proj1_sig m')
                 (lt_S (proj1_sig m') n (proj2_sig m')))) def h13).
rewrite h1.
f_equal.
f_equal.
apply proof_irrelevance. 
subst. 
destruct (zerop (n-1)) as [h8 | h9].
rewrite h8 in h7.
rewrite <- (minus_n_O n) in h7.
rewrite h8 in h7.
contradict h7. auto with arith.
rewrite <- pred_of_minus in h9.
rewrite <- (pred_of_minus n) in h7.
rewrite (n_minus_pred_n _ h9) in h7.
assert (h10:1 - 1 = 0). auto with arith.
rewrite h10 in h7.
contradict h7. auto with arith.
unfold lt in h12. rewrite <- pred_of_minus in h12.
rewrite <- (S_pred _ _ h4) in h12.
clear h5 h7.
omega.
Qed.


Lemma l_exp_seg_to_nprod_in_l_bij : 
  forall {T:Type} (l:list T) (n:nat),
    (bijective (l_exp_seg_to_nprod_in_l l n)).
intros T l n.
induction n as [|n h1].
(* 0 *)
red. split.
red.
intros x1 x2 h1.
simpl in h1.
apply functional_extensionality.
intro x. destruct x. 
pose proof (lt_n_0 x). contradiction.
red.
intro y.
exists (fun x:{m:nat | m < 0} => False_rect {x:T | In x l} (lt_n_0 _ (proj2_sig x))).
simpl. simpl in y. destruct y. reflexivity.
(* S n *)
red. split. 
(* injective *)
red.
intros f1 f2 h2.
simpl in h2. 
inversion h2 as [h0]. 
apply functional_extensionality.
intro x.
destruct x as [x h3]. 
destruct (lt_eq_lt_dec x n) as [[h4 | h5] | h6].
pose proof (f_equal (@nprod_to_list {x:T|In x l} n) H) as h5.
pose proof (f_equal (fun l => nth (n-x-1) l (f1 (exist _ x h3))) h5) as h6.
simpl in h6.
do 2 rewrite (l_exp_seg_to_nprod_in_l_compat _ _ _ _ _ h4) in h6.
simpl in h6.
assert (h7:h3 = lt_S x n h4).  apply proof_irrelevance.
subst.
assumption.
subst.
assert (h4:h3 = lt_n_Sn n). apply proof_irrelevance.
subst.
assumption.
omega.
(* surjective *)
red.
simpl.
intro y.
red in h1.
destruct h1 as [h1l h1r].
red in h1r.
destruct y as [fy sy].
specialize (h1r sy).
destruct h1r as [f h2].
pose (fun m':{m:nat | m <= n} => 
        let pf := (le_lt_eq_dec (proj1_sig m') n (proj2_sig m')) in
        match pf with
          | left pf' => (f (exist (fun m=>m<n) (proj1_sig m') pf'))
          | right _ => fy
        end) as f'.
pose (fun m':{m:nat | m < S n} => f' (exist _ _ (le_S_n _ _ (proj2_sig m')))) as f''.
exists f''.
apply injective_projections.
simpl.
unfold f''.
simpl.
unfold f'.
simpl.
destruct le_lt_eq_dec as [h3 | h4].
contradict h3. auto with arith.
reflexivity.
simpl.
unfold f''.
simpl.
unfold f'.
simpl.
assert (h3:(fun m' : {m : nat | m < n} =>
      match
        le_lt_eq_dec (proj1_sig m') n
          (le_S_n (proj1_sig m') n (lt_S (proj1_sig m') n (proj2_sig m')))
      with
      | left pf' => f (exist (fun m : nat => m < n) (proj1_sig m') pf')
      | right _ => fy
      end) = f).
  apply functional_extensionality.
  intro x.
  destruct x as [x h3].
  simpl.
  destruct le_lt_eq_dec as [h4 | h5].
  assert (h3 = h4).  apply proof_irrelevance. subst.
  reflexivity.
  omega.
rewrite h3.
exact h2.
Qed.

Lemma finite_l_exp_seg : 
  forall {T:Type} (l:list T) (n:nat),
    FiniteT ({m:nat | m < n} -> {x:T|In x l}).
intros T l n.
pose proof (finite_nat_initial_segment n) as h1.
pose proof (list_to_set_finitet l) as h2.
apply finite_exp; auto.
Qed.


Lemma finite_nprod_in_l : 
  forall {T:Type} (l:list T) (n:nat),
    FiniteT ({x:T|In x l}^n).
intros T l n.
pose proof (finite_nat_initial_segment n) as h1.
pose proof (finite_list_sig l) as h2.
pose proof (finite_exp _ _ h1 h2) as h3.
pose proof (l_exp_seg_to_nprod_in_l_bij l n) as h4.
apply bijective_impl_invertible in h4. 
apply (bij_finite _ _ _ h3 h4).
Qed.


Lemma card_in_l : 
  forall {T:Type} (l:list T),
    NoDup l ->
    FiniteT_nat_cardinal _ (list_to_set_finitet l) = length l.
intros T l h0.
induction l as [|a l h1].
simpl. 
rewrite <- FiniteT_nat_cardinal_False.
pose (fun x:{_:T | False} => False_rect False (proj2_sig x)) as f.
assert (h1:invertible f). apply bijective_impl_invertible.
  red; split; red; intros; contradiction.
pose proof (FiniteT_nat_cardinal_bijection _ _ (list_to_set_finitet nil) f h1) as h2.
simpl in h2.
rewrite <- h2.
assert (h3:(bij_finite {_ : T | False} False f (list_to_set_finitet nil) h1) = empty_finite). apply proof_irrelevance.
rewrite h3. reflexivity.
pose proof (no_dup_cons_nin _ _ h0) as h0'.
simpl. 
pose proof (FiniteT_nat_cardinal_option _ (list_to_set_finitet l)) as h3.
pose (fun x:(option {x : T | In x l}) =>
        let f := (fun x => a = x \/ In x l) in
        match x with
          | Some e => exist f (proj1_sig e) (or_intror (proj2_sig e)) 
          | None => exist f a (or_introl (eq_refl _))
        end) as g.
assert (h4:invertible g).
  apply bijective_impl_invertible.
  red; split; red.
  intros o p h4.
  destruct o; destruct p. simpl in h4.
  apply exist_injective in h4. f_equal.  apply proj1_sig_injective. assumption.
  simpl in h4. apply exist_injective in h4.
  contradict h0'.
  destruct s. simpl in h4. subst. assumption.
  simpl in h4. apply exist_injective in h4. contradict h0'.
  destruct s. simpl in h4. subst. assumption.
  reflexivity.
  intro y.
  destruct y as [y h4].
  destruct h4 as [h4l | h4r]. exists None.
  simpl. apply proj1_sig_injective. simpl. assumption.
  exists (Some (exist _ y h4r)). simpl. reflexivity.
rewrite <- (FiniteT_nat_cardinal_bijection _ _ (add_finite _ (list_to_set_finitet l)) g h4) in h3.
pose proof (no_dup_cons _ _ h0) as h5.
specialize (h1 h5).
rewrite h1 in h3.
rewrite <- h3.
f_equal. apply proof_irrelevance.
Qed.
