(* Copyright (C) 2015, Daniel Wyckoff *)

(*This file is part of BooleanAlgebrasIntro2.

BooleanAlgebrasIntro2 is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

BooleanAlgebrasIntro2 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with BooleanAlgebrasIntro2.  If not, see <http://www.gnu.org/licenses/>.*)
  
Require Import BoolAlgBasics.
Require Import FunctionProperties.
Require Import Subalgebras.
Require Import SetUtilities.
Require Import FunctionalExtensionality.
Require Import LogicUtilities.
Require Import Description. 
Require Import FieldsOfSets.
Require Import TypeUtilities.
Require Import Basics.
Require Import FiniteOperations.
Require Import ListUtilities.
 
(*Note I have ommitted the exchange principle for embeddings for now, since I hadn't yet developed Bool_Alg_ps, but now that I have I will return to it real soon!*)
(*Also, I'll eventually do fields of periodic integers discussed
in this chapter.*)


Record homomorphism {A B:Bool_Alg} (f:bt A->bt B) : Prop :=
  {
    homo_times : forall x y, f (x*y) = (f x) * (f y);
    homo_plus : forall x y,   f (x+y) = (f x) + (f y);
    homo_comp : forall x,   f (-x) = - (f x)
  }.


Record homomorphism_general {A:Bool_Alg} {T:Type} (f:(bt A)->T) (homo_ut : T -> T -> T)
      (homo_up : T -> T -> T)
      (homo_uc : T -> T)
: Prop :=
  {
    homo_gen_times : forall x y, f (x*y) = homo_ut (f x) (f y);
    homo_gen_plus : forall x y,   f (x+y) = homo_up (f x) (f y);
    homo_gen_comp : forall x,   f (-x) = homo_uc (f x)
  }.

Lemma homo_sym_diff : 
  forall {A B:Bool_Alg} (f:bt A->bt B),
    homomorphism f -> forall x y, f (x /_\ y) = (f x) /_\ (f y).
intros A B f h1 x y.
unfold sym_diff.
destruct h1 as [h1a h1b h1c].
rewrite h1b. do 2 rewrite h1a.  do 2 rewrite h1c.
reflexivity.
Qed.

Lemma homo_id : 
  forall A:Bool_Alg,
    homomorphism id (A:=A).
intros; constructor; auto.
Qed.



Lemma universal_homomorphic_ops_imply_bool : 
  forall {A:Bool_Alg} {T:Type} (f:(bt A)->T)
         (ut:T->T->T) (up:T->T->T) (uc:T->T)
         (pf:homomorphism_general f ut up uc), 
    surjective f ->
    exists! B:Bool_Alg,
      Bc B = Build_Bconst T (Full_set T) up
                          ut (f 1) (f 0) uc.
intros A T f ut up uc h1 h2.
red in h2.
destruct h1 as [h1a h1b h1c].
pose (Build_Bconst T (Full_set T) up ut (f 1) (f 0) uc) as Bc'.
assert (my_und_set : BS Bc' = Full_set (Btype Bc')).
  simpl. reflexivity.
assert (my_assoc_sum : forall n m p : Btype Bc', n + (m + p) = n + m + p).
  intros p q r. simpl.
  pose proof (h2 p) as h3. pose proof (h2 q) as h4. pose proof (h2 r) as h5.
  destruct h3 as [u h3]. destruct h4 as [v h4]. destruct h5 as [w h5].
  subst.
  rewrite <- h1b at 1.
  rewrite <- h1b at 1.
  rewrite assoc_sum.
  rewrite h1b. rewrite h1b.
  reflexivity.
assert (my_assoc_prod : forall n m p : Btype Bc', n * (m * p) = n * m * p).
  intros p q r. simpl.
  pose proof (h2 p) as h3. pose proof (h2 q) as h4. pose proof (h2 r) as h5.
  destruct h3 as [u h3]. destruct h4 as [v h4]. destruct h5 as [w h5].
  subst.
  rewrite <- h1a at 1.
  rewrite <- h1a at 1.
  rewrite assoc_prod.
  rewrite h1a. rewrite h1a.
  reflexivity.
assert (my_comm_sum : forall n m : Btype Bc', n + m = m + n).
  intros p q. simpl.
  pose proof (h2 p) as h3. pose proof (h2 q) as h4. 
  destruct h3 as [u h3]. destruct h4 as [v h4]. 
  subst.
  rewrite <- h1b. rewrite comm_sum. rewrite h1b.
  reflexivity.
assert (my_comm_prod : forall n m : Btype Bc', n * m = m * n).
  intros p q. simpl.
  pose proof (h2 p) as h3. pose proof (h2 q) as h4. 
  destruct h3 as [u h3]. destruct h4 as [v h4]. 
  subst.
  rewrite <- h1a. rewrite comm_prod. rewrite h1a.
  reflexivity.
assert (my_abs_sum : forall n m : Btype Bc', n + n * m = n).
  intros p q. simpl.
  pose proof (h2 p) as h3. pose proof (h2 q) as h4. 
  destruct h3 as [u h3]. destruct h4 as [v h4]. 
  subst.
  rewrite <- h1a.
  rewrite <- h1b.
  rewrite abs_sum.
  reflexivity.
assert (my_abs_prod : forall n m : Btype Bc', n * (n + m) = n).
  intros p q. simpl.
  pose proof (h2 p) as h3. pose proof (h2 q) as h4. 
  destruct h3 as [u h3]. destruct h4 as [v h4]. 
  subst.
  rewrite <- h1b.
  rewrite <- h1a.
  rewrite abs_prod.
  reflexivity.
assert (my_dist_sum :  forall n m p : Btype Bc', p * (n + m) = p * n + p * m).
  intros p q r.
  pose proof (h2 p) as h3. pose proof (h2 q) as h4. pose proof (h2 r) as h5.
  destruct h3 as [u h3]. destruct h4 as [v h4]. destruct h5 as [w h5].
  subst. simpl.  
  rewrite <- h1b at 1.
  rewrite <- h1a at 1.
  rewrite dist_sum.
  rewrite h1b at 1.
  do 2 rewrite h1a at 1.
  reflexivity.
assert (my_dist_prod :  forall n m p : Btype Bc', p + (n * m) = (p + n) * (p + m)).
  intros p q r.
  pose proof (h2 p) as h3. pose proof (h2 q) as h4. pose proof (h2 r) as h5.
  destruct h3 as [u h3]. destruct h4 as [v h4]. destruct h5 as [w h5].
  subst. simpl.   
  rewrite <- h1a at 1.
  rewrite <- h1b at 1.
  rewrite dist_prod.
  rewrite h1a at 1.
  do 2 rewrite h1b at 1.
  reflexivity.
assert (my_comp_sum : forall n : Btype Bc', n + - n = 1).
  intro n.
  pose proof (h2 n) as h3.
  destruct h3 as [u h3]. 
  subst. simpl.   
  rewrite <- h1c.
  rewrite <- h1b.
  rewrite comp_sum.
  reflexivity.
assert (my_comp_prod : forall n : Btype Bc', n * - n = 0).
  intro n.
  pose proof (h2 n) as h3.
  destruct h3 as [u h3]. 
  subst. simpl.   
  rewrite <- h1c.
  rewrite <- h1a.
  rewrite comp_prod.
  reflexivity.
exists (Build_Bool_Alg _ my_und_set my_assoc_sum my_assoc_prod my_comm_sum my_comm_prod my_abs_sum my_abs_prod my_dist_sum my_dist_prod my_comp_sum my_comp_prod).
red. split.
simpl.
reflexivity.
intros B h1.
apply bc_inj. simpl.
rewrite h1.
reflexivity.
Qed.


Lemma homo_zero : 
forall {A B:Bool_Alg} (f:bt A->bt B),
  homomorphism f ->
  f 0 = 0.
intros A B f h1.
destruct h1 as [h1 h2 h3].
pose proof (comp_prod A 0) as h4.
pose proof (f_equal f (eq_refl (0*-0))) as h5.
rewrite h4 in h5 at 1.
rewrite h1 in h5.
rewrite h3 in h5.
rewrite comp_prod in h5.
assumption.
Qed.

Lemma homo_one : 
forall {A B:Bool_Alg} (f:bt A->bt B),
  homomorphism f ->
  f 1 = 1.
intros A B f h1.
destruct h1 as [h1 h2 h3].
pose proof (comp_sum A 1) as h4.
pose proof (f_equal f (eq_refl ((1+(-(1)))))) as h5.
rewrite h4 in h5 at 1.
rewrite h2 in h5.
rewrite h3 in h5.
rewrite comp_sum in h5.
assumption.
Qed.

Lemma homo_two_ops_plus : 
  forall {A B:Bool_Alg} (f:bt A->bt B),
    (forall x y, f (x + y) = (f x) + (f y)) ->
    (forall x, f (-x) = - (f x)) ->
    homomorphism f.
intros A B f h1 h2.
constructor; auto.
intros x y.
rewrite (doub_neg (f x)). rewrite (doub_neg (f y)).
rewrite <- de_mor_sum.
rewrite <- h2. rewrite <- h2.
rewrite <- h1.
rewrite <- h2.
rewrite de_mor_sum.
do 2 rewrite <- doub_neg.
reflexivity.
Qed.

Lemma homo_two_ops_times : 
  forall {A B:Bool_Alg} (f:bt A->bt B),
    (forall x y, f (x * y) = (f x) * (f y)) ->
    (forall x, f (-x) = - (f x)) ->
    homomorphism f.
intros A B f h1 h2.
constructor; auto.
intros x y.
rewrite (doub_neg (f x)). rewrite (doub_neg (f y)).
rewrite <- de_mor_prod.
rewrite <- h2. rewrite <- h2.
rewrite <- h1.
rewrite <- h2.
rewrite de_mor_prod.
do 2 rewrite <- doub_neg.
reflexivity.
Qed.




Lemma homo_times_list : 
  forall {A B:Bool_Alg} (f:bt A->bt B),
    homomorphism f ->
    forall (l:list (bt A)),
      NoDup l ->
      f (times_list l) = times_list (map f l).
intros A B f h1 l.
induction l as [|a l h2].
intros; simpl; auto.
apply homo_one; auto.
intro h3; simpl.
rewrite homo_times; auto.
f_equal.
apply h2.
apply (no_dup_cons l a).
assumption.
Qed.


Lemma homo_plus_list : 
  forall {A B:Bool_Alg} (f:bt A->bt B),
    homomorphism f ->
    forall (l:list (bt A)),
      NoDup l ->
      f (plus_list l) = plus_list (map f l).
intros A B f h1 l.
induction l as [|a l h2].
intros; simpl; auto.
apply homo_zero; auto.
intro h3; simpl.
rewrite homo_plus; auto.
f_equal.
apply h2.
apply (no_dup_cons l a).
assumption.
Qed.


Lemma homo_times_set : 
  forall {A B:Bool_Alg} (f:bt A->bt B),
    homomorphism f ->
    forall (E:Ensemble (bt A)) (pf:Finite E),
      f (times_set E pf) = times_set (Im E f) (finite_image _ _ _ f pf).
intros A B f h1 E h2.
pose proof (finite_set_list_no_dup _ h2) as h3.
destruct h3 as [l [h3 h4]]. subst.
pose proof (times_set_compat' (list_to_set l) _ h2 (eq_refl _)) as h3.
rewrite h3 at 1.
pose proof (map_im_compat f l) as h6. symmetry in h6.
pose proof (times_set_compat' (Im (list_to_set l) f) (map f l) (finite_image (bt A) (bt B) (list_to_set l) f h2) h6) as h5.
rewrite h5 at 1.
apply homo_times_list; auto.
Qed.

Lemma homo_plus_set : 
  forall {A B:Bool_Alg} (f:bt A->bt B),
    homomorphism f ->
    forall (E:Ensemble (bt A)) (pf:Finite E),
      f (plus_set E pf) = plus_set (Im E f) (finite_image _ _ _ f pf).
intros A B f h1 E h2.
pose proof (finite_set_list_no_dup _ h2) as h3.
destruct h3 as [l [h3 h4]]. subst.
pose proof (plus_set_compat' (list_to_set l) _ h2 (eq_refl _)) as h3.
rewrite h3 at 1.
pose proof (map_im_compat f l) as h6. symmetry in h6.
pose proof (plus_set_compat' (Im (list_to_set l) f) (map f l) (finite_image (bt A) (bt B) (list_to_set l) f h2) h6) as h5.
rewrite h5 at 1.
apply homo_plus_list; auto.
Qed.





Lemma homo_compose : 
  forall {A B C:Bool_Alg}
         (f:bt A->bt B)
         (g:bt B->bt C),
    homomorphism f -> homomorphism g -> 
    homomorphism (compose g f).
intros A B C f g h1 h2.
apply homo_two_ops_times.
intros x y.
unfold compose.
rewrite (homo_times f); auto. rewrite (homo_times g); auto.
intro x.
unfold compose.
rewrite (homo_comp f); auto. rewrite (homo_comp g); auto.
Qed.

Lemma homo_inv : 
  forall {A B:Bool_Alg} (f:bt A->bt B)
         (pf:invertible f),
    homomorphism f -> 
    homomorphism (proj1_sig (function_inverse _ pf)).
intros A B f h1 h0.
pose proof h1 as h2. 
apply invertible_impl_bijective in h2.
destruct h2 as [h2l h2r].
apply homo_two_ops_plus.
intros p q.
pose proof (h2r p) as h3. pose proof (h2r q) as h4.
destruct h3 as [p' h3]. destruct h4 as [q' h4].
subst. rewrite <- homo_plus.
do 3 rewrite (match (proj2_sig (function_inverse f h1)) with
             | conj pf _ => pf 
         end).
reflexivity. assumption.
intro x.
specialize (h2r x).
destruct h2r as [x' hr].
subst.
rewrite <- homo_comp.
do 2 rewrite (match (proj2_sig (function_inverse f h1)) with
                 | conj pf _ => pf
         end).
reflexivity.
assumption.
Qed.

Lemma homo_mono : 
  forall {A B:Bool_Alg} (f:bt A->bt B),
    homomorphism f -> forall x y, le x y -> le (f x) (f y).
intros A B f h1.
destruct h1 as [h1a h1b h1c].
intros x y h2.
red. rewrite <- h1b. red in h2. rewrite h2.
reflexivity.
Qed.

Lemma homo_im_closed : 
  forall {A B:Bool_Alg} (f:bt A->bt B),
    homomorphism f ->
    alg_closed (Im (BS (Bc A)) f).
intros A B f h1.
pose proof h1 as h1'.
destruct h1 as [h1a h1b h1c].
constructor.
red.
intros x y.
unfold Bplus_sub.
destruct x as [x h2]. destruct y as [y h3].
simpl.
destruct h2 as [x h2]. destruct h3 as [y h3]. subst.
rewrite <- h1b.
apply Im_intro with (x+y); auto. rewrite und_set. constructor.
intros x y.
unfold Btimes_sub.
destruct x as [x h2]. destruct y as [y h3].
simpl.
destruct h2 as [x h2]. destruct h3 as [y h3]. subst.
rewrite <- h1a.
apply Im_intro with (x*y); auto. rewrite und_set. constructor.
red.
apply Im_intro with 1. rewrite und_set. constructor.
rewrite homo_one; auto.
red.
apply Im_intro with 0. rewrite und_set. constructor.
rewrite homo_zero; auto.
intro x.
unfold Bcomp_sub.
destruct x as [x h2]. 
simpl.
destruct h2 as [x h2]. subst.
rewrite <- h1c.
apply Im_intro with (-x); auto. rewrite und_set. constructor.
Qed.

(*This returns the same homomorphism, only with the range replaced
  with its image.*)
Definition homo_onto {A B:Bool_Alg}
           (f:bt A->bt B) 
           (pf:homomorphism f) :
  bt A->bt (Subalg _ (homo_im_closed _ pf)).
intros x. unfold bt, Subalg, SubBtype. simpl.
rewrite und_set.
refine (exist _ _ (proj2_sig (sig_im_fun f x))).
Defined.


Lemma homo_onto_compat :
  forall {A B:Bool_Alg}
         (f:bt A->bt B) 
         (pf:homomorphism f),
    surjective (homo_onto _ pf) /\
    forall (x:bt A),
      f x = proj1_sig (homo_onto _ pf x).
intros A B f h1.  
assert (h2: (forall x : bt A, f x = proj1_sig (homo_onto f h1 x))). 
unfold homo_onto. unfold eq_rect_r. unfold eq_rect. simpl.
destruct (eq_sym (und_set A)). simpl.
reflexivity.
split; auto.
pose proof (surj_sig_im_fun f) as h0.
red in h0. red. intro y. simpl in y. 
destruct y as [y h3]. 
pose proof h3 as h3'. rewrite und_set in h3'.
specialize (h0 (exist _ _ h3')).
destruct h0 as [x h0].
exists x.
apply proj1_sig_injective. simpl.
rewrite <- h2.
destruct h3. subst.
pose proof (f_equal (@proj1_sig _ _) h0) as h4. simpl in h4.
assumption.
Qed.

Lemma homo_homo_onto : 
  forall {A B:Bool_Alg}
         (f:bt A->bt B) 
         (pf:homomorphism f),
    homomorphism (homo_onto _ pf).
intros A B f h1.
pose proof (homo_onto_compat _ h1) as h2. destruct h2 as [h2l h2r].
apply homo_two_ops_times.
intros x y.
apply proj1_sig_injective. simpl.
rewrite <- h2r. rewrite homo_times; auto.
unfold Btimes_sub. do 2 rewrite <- h2r.
reflexivity.
intro x.
apply proj1_sig_injective. simpl.
rewrite <- h2r. rewrite homo_comp; auto. unfold Bcomp_sub.
f_equal. apply h2r.
Qed.    

Lemma inj_homo_onto_iff : 
  forall {A B:Bool_Alg} (f:bt A->bt B)
    (pf:homomorphism f),
  FunctionProperties.injective f <-> 
  FunctionProperties.injective (homo_onto _ pf).
intros A B f h1. 
pose proof (homo_onto_compat _ h1) as h2. 
destruct h2 as [h2l h2r].
split.
intro h2.
red in h2. red. intros a b h3. 
apply h2. do 2 rewrite h2r.
f_equal. assumption.
intro h3. red in h3. red.
intros a b h4.
do 2 rewrite h2r in h4. apply proj1_sig_injective in h4.
apply h3; auto.
Qed.

Lemma homo_transfer_iff : 
  forall (A B C:Bool_Alg),
    A = B ->
    forall (f:bt A -> bt C)
           (pf:bt A = bt B),
    homomorphism f <-> homomorphism (transfer_fun pf f).
intros A B C h1 f h2. subst.
assert (h2=eq_refl). apply proof_irrelevance.
subst.
rewrite transfer_fun_eq_refl.
tauto.
Qed.

Definition rel_homo {A:Bool_Alg} (p:(bt A)) := (fun x=>x*p). 

Definition relativization_set {A:Bool_Alg} (p:bt A) := 
  [x:(bt A) | le x p].

Definition relativization_set_zero : 
  forall {A:Bool_Alg} (p:bt A),
    Ensembles.In (relativization_set p) 0.
intros A p. constructor. apply zero_min.
Qed.

Definition relativization_set_one : 
  forall {A:Bool_Alg} (p:bt A),
    Ensembles.In (relativization_set p) p.
intros A p. constructor. apply refl_le.
Qed.



Lemma rel_homo_im_in_rel_set : 
  forall {A:Bool_Alg} (p:(bt A)),
    Included (Im (BS (Bc A)) (rel_homo p)) (relativization_set p).
intros A p. red. intros x h1.
destruct h1 as [x h1]. subst.
constructor.
unfold rel_homo.
rewrite comm_prod.
apply times_le.
Qed.

Lemma rel_homo_sig_ex : 
  forall {A:Bool_Alg} (p:bt A),
    exists! f:(bt A)->sig_set (relativization_set p),
      forall x,
        proj1_sig (f x) = rel_homo p x.
intros A p.
assert (h1:forall x:(bt A), Ensembles.In (Im (BS (Bc A)) (rel_homo p)) (x*p)).
  intros x.
  apply Im_intro with x.  apply in_bs.
  reflexivity.
exists (fun x => exist _ _ (rel_homo_im_in_rel_set _ _ ((h1 x)))).
red.
simpl.
split. auto.
intros f h2.
apply functional_extensionality.
intro x.
apply proj1_sig_injective.
simpl.
rewrite h2.
reflexivity.
Qed.


Definition rel_homo_sig {A:Bool_Alg} (p:bt A) : 
  (bt A) -> sig_set (relativization_set p) :=
  proj1_sig (constructive_definite_description _ (rel_homo_sig_ex p)).


Lemma rel_homo_sig_compat : 
  forall {A:Bool_Alg} (p:bt A),
    let f:=rel_homo_sig p in 
    forall x,
      proj1_sig (f x) = rel_homo p x.
intros A p f.
unfold f. unfold rel_homo_sig.
destruct constructive_definite_description as [f' h2].
simpl.
assumption.
Qed.


Definition rel_times_closed : 
  forall {B:Bool_Alg} (p:bt B) (x y:bt B),
    Ensembles.In (relativization_set p) x ->
    Ensembles.In (relativization_set p) y ->
    Ensembles.In (relativization_set p) (x * y).
intros B p x y h1 h2.
destruct h1 as [h1]. destruct h2 as [h2].
constructor.
rewrite <- (idem_prod p).
apply mono_prod; auto.
Qed.


Definition rel_plus_closed : 
  forall {B:Bool_Alg} (p:bt B) (x y:bt B),
    Ensembles.In (relativization_set p) x ->
    Ensembles.In (relativization_set p) y ->
    Ensembles.In (relativization_set p) (x + y).
intros B p x y h1 h2.
destruct h1 as [h1]. destruct h2 as [h2].
constructor.
rewrite <- (idem_sum p).
apply mono_sum; auto.
Qed.

Definition rel_comp_closed : 
  forall {B:Bool_Alg} (p:bt B) (x:bt B),
    Ensembles.In (relativization_set p) x ->
    Ensembles.In (relativization_set p) (-x * p).
intros B p x h1.
constructor.
rewrite comm_prod.
apply times_le.
Qed.

Lemma rel_times_ex : 
  forall {B:Bool_Alg} (p:bt B),
    exists! (rt:sig_set (relativization_set p) ->
               sig_set (relativization_set p) -> 
               sig_set (relativization_set p)),
      forall x y, proj1_sig (rt x y) = (proj1_sig x) * (proj1_sig y).
intros B p.
exists (fun (x y:sig_set (relativization_set p))  => exist _ _ (rel_times_closed p (proj1_sig x) (proj1_sig y) (proj2_sig x) (proj2_sig y))).
red. simpl. split. auto.
intros f h1.
apply functional_extensionality.
intro x.
apply functional_extensionality.
intro y.
apply proj1_sig_injective. simpl.
rewrite h1.
reflexivity.
Qed.

Definition rel_times {B:Bool_Alg} (p:bt B) :=
  proj1_sig (constructive_definite_description _ (rel_times_ex p)).

Lemma rel_times_compat : 
  forall {B:Bool_Alg} (p:bt B),
    let rt := rel_times p in
    forall x y, proj1_sig (rt x y) = (proj1_sig x) * (proj1_sig y).
intros B p rt x y.
unfold rt. unfold rel_times.
destruct constructive_definite_description as [h1 h2].
simpl.
apply h2.
Qed.

Lemma rel_times_compat' : 
  forall {B:Bool_Alg} (p:bt B),
    rel_times p = fun x y => 
                    exist _ _ (rel_times_closed p (proj1_sig x) (proj1_sig y) (proj2_sig x) (proj2_sig y)).
intros B p.
apply functional_extensionality. intro x.
apply functional_extensionality. intro y.
apply proj1_sig_injective. simpl.
apply rel_times_compat.
Qed.


Lemma rel_plus_ex : 
  forall {B:Bool_Alg} (p:bt B),
    exists! (rp:sig_set (relativization_set p) ->
               sig_set (relativization_set p) -> 
               sig_set (relativization_set p)),
      forall x y, proj1_sig (rp x y) = (proj1_sig x) + (proj1_sig y).
intros B p.
exists (fun (x y:sig_set (relativization_set p))  => exist _ _ (rel_plus_closed p (proj1_sig x) (proj1_sig y) (proj2_sig x) (proj2_sig y))).
red. simpl. split. auto.
intros f h1.
apply functional_extensionality.
intro x.
apply functional_extensionality.
intro y.
apply proj1_sig_injective. simpl.
rewrite h1.
reflexivity.
Qed.

Definition rel_plus {B:Bool_Alg} (p:bt B) :=
  proj1_sig (constructive_definite_description _ (rel_plus_ex p)).

Lemma rel_plus_compat : 
  forall {B:Bool_Alg} (p:bt B),
    let rp := rel_plus p in
    forall x y, proj1_sig (rp x y) = (proj1_sig x) + (proj1_sig y).
intros B p rp x y.
unfold rp. unfold rel_plus.
destruct constructive_definite_description as [h1 h2].
simpl.
apply h2.
Qed.

Lemma rel_plus_compat' : 
  forall {B:Bool_Alg} (p:bt B),
    rel_plus p = fun x y => 
                    exist _ _ (rel_plus_closed p (proj1_sig x) (proj1_sig y) (proj2_sig x) (proj2_sig y)).
intros B p.
apply functional_extensionality. intro x.
apply functional_extensionality. intro y.
apply proj1_sig_injective. simpl.
apply rel_plus_compat.
Qed.


Lemma rel_comp_ex : 
  forall {B:Bool_Alg} (p:bt B),
    exists! (rc:sig_set (relativization_set p) ->
                sig_set (relativization_set p)),
      forall x, proj1_sig (rc x) = - (proj1_sig x) * p.
intros B p.
exists (fun x:sig_set (relativization_set p)  => exist _ _ (rel_comp_closed p (proj1_sig x) (proj2_sig x))).
red. simpl. split. auto.
intros f h1.
apply functional_extensionality.
intro x.
apply proj1_sig_injective.
rewrite h1.
simpl.
reflexivity.
Qed.

Definition rel_comp {B:Bool_Alg} (p:bt B) :=
  proj1_sig (constructive_definite_description _ (rel_comp_ex p)).

Lemma rel_comp_compat : 
  forall {B:Bool_Alg} (p:bt B),
    let rc := rel_comp p in
    forall x, proj1_sig (rc x) = - (proj1_sig x) * p.
intros B p rp x.
unfold rp. unfold rel_comp. 
destruct constructive_definite_description as [h1 h2].
simpl.
apply h2.
Qed.

Lemma rel_comp_compat' : 
  forall {B:Bool_Alg} (p:bt B),
    rel_comp p = fun x => 
                    exist _ _ (rel_comp_closed p (proj1_sig x) (proj2_sig x)).
intros B p.
apply functional_extensionality. intro x.
apply proj1_sig_injective. simpl.
apply rel_comp_compat.
Qed.





Definition rel_bc {B:Bool_Alg} (p:bt B) :=
           (Build_Bconst (sig_set (relativization_set p))
                         (full_sig (relativization_set p))
                         (rel_plus p) (rel_times p)
                         (exist _ _ (relativization_set_one p))
                         (exist _ _ (relativization_set_zero p))
                         (rel_comp p)).



Lemma homo_gen_rel_homo_sig : 
  forall {A:Bool_Alg} (p:bt A),
         homomorphism_general (rel_homo_sig p) (rel_times p) (rel_plus p) (rel_comp p).
intros A p. 
constructor.
intros x y. 
pose proof (rel_homo_sig_compat p (x*y)) as h2.
apply proj1_sig_injective. 
rewrite h2 at 1.
pose proof (rel_times_compat p (rel_homo_sig p x) (rel_homo_sig p y)) as h1.  
rewrite h1 at 1.
pose proof (rel_homo_sig_compat p x) as h3.
pose proof (rel_homo_sig_compat p y) as h4.
rewrite h3. rewrite h4.
unfold rel_homo.
rewrite (comm_prod _ y p).
rewrite assoc_prod.
rewrite <- (assoc_prod _ x p p).
rewrite idem_prod.
rewrite <- assoc_prod.
rewrite (comm_prod _ y p).
rewrite assoc_prod.
reflexivity.
intros x y.
pose proof (rel_homo_sig_compat p (x+y)) as h2.
apply proj1_sig_injective. 
rewrite h2 at 1.
pose proof (rel_plus_compat p (rel_homo_sig p x) (rel_homo_sig p y)) as h1.  
rewrite h1 at 1.
pose proof (rel_homo_sig_compat p x) as h3.
pose proof (rel_homo_sig_compat p y) as h4.
rewrite h3. rewrite h4.
unfold rel_homo.
apply dist_sum_r.
intros x.
pose proof (rel_homo_sig_compat p (-x)) as h1.
apply proj1_sig_injective.
rewrite h1 at 1.
pose proof (rel_comp_compat p (rel_homo_sig p x)) as h2.
rewrite h2 at 1.
pose proof (rel_homo_sig_compat p x) as h3.
rewrite h3.
unfold rel_homo.
rewrite de_mor_prod.
rewrite dist_sum_r.
rewrite (comm_prod _ (-p) p).
rewrite comp_prod.
rewrite zero_sum.
reflexivity.
Qed.

Lemma surj_rel_homo_sig : 
  forall {A:Bool_Alg} (p:bt A),
         surjective (rel_homo_sig p).
intros A p.
red.
intros b.
destruct b as [b h1].
destruct h1 as [h1].
red in h1.
pose proof h1 as h1'.
pose proof (eq_ord b p) as h2.
rewrite h2 in h1'.
exists b.
apply proj1_sig_injective.
simpl.
pose proof (rel_homo_sig_compat p b) as h3.
rewrite h3 at 1.
rewrite <- h1' at 2. reflexivity.
Qed.

Definition rel_ba {B:Bool_Alg} (p:bt B) := 
  proj1_sig (constructive_definite_description
               _ (universal_homomorphic_ops_imply_bool 
                  _ _ _ _   (homo_gen_rel_homo_sig p)
                  (surj_rel_homo_sig p))).
             
Lemma rel_ba_compat : 
  forall {B:Bool_Alg} (p:bt B),
  let B' := rel_ba p in
  Bc B' = rel_bc p.
intros B p B'.
unfold B'. 
unfold rel_ba.
destruct constructive_definite_description as 
    [A h1].
simpl.
rewrite h1.
unfold rel_bc.
assert (h2:Full_set (sig_set (relativization_set p)) = full_sig (relativization_set p)). reflexivity.
rewrite h2. 
f_equal; try apply proj1_sig_injective; simpl. 
pose proof (rel_homo_sig_compat p 1) as h3.
unfold rel_homo in h3.
rewrite comm_prod in h3. rewrite one_prod in h3.
assumption.
pose proof (rel_homo_sig_compat p 0) as h3.
unfold rel_homo in h3.
rewrite comm_prod in h3. rewrite zero_prod in h3.
assumption.
Qed.   

Lemma btype_bc_rel_ba : 
  forall {B:Bool_Alg} (p:bt B),
    bt (rel_ba p) = sig_set (relativization_set p).
intros B p.
unfold bt. simpl.
rewrite rel_ba_compat.
simpl.
reflexivity.
Qed.


Lemma rel_ba_times_compat : 
  forall {B:Bool_Alg} (p:bt B),
    (Btimes (Bc (rel_ba p))) =
    (transfer_dep_r (btype_bc_rel_ba p)
                    (U:=fun T=>T->T->T)
                    (Btimes (rel_bc p))).
intros B p.
simpl. 
rewrite rel_times_compat'.
rewrite <- (transfer_dep_transfer_dep_r_compat (eq_sym (btype_bc_rel_ba p)) (btype_bc_rel_ba p)).    
symmetry. 
assert (h1:@transfer_dep Type (fun x:Type => x-> x-> x) 
                         _ _ (eq_sym (btype_bc_rel_ba p))
                         (fun x0 y0 : sig_set (relativization_set p) =>
                            exist (Ensembles.In (relativization_set p)) (proj1_sig x0 * proj1_sig y0)
                                  (rel_times_closed p (proj1_sig x0) (proj1_sig y0) 
                                                    (proj2_sig x0) (proj2_sig y0))) =
           (Btimes (Bc (rel_ba p)))). 
rewrite <- (@transfer_dep_eq_iff Type (fun x:Type => x->x->x) _ _  (eq_sym (btype_bc_rel_ba p))
                                 (fun x0 y0 : sig_set (relativization_set p) =>
                                    exist (Ensembles.In (relativization_set p)) (proj1_sig x0 * proj1_sig y0)
                                          (rel_times_closed p (proj1_sig x0) (proj1_sig y0) 
                                                            (proj2_sig x0) (proj2_sig y0)))
                                 (Btimes (Bc (rel_ba p)))).  
assert (h1:(fun x0 y0 : sig_set (relativization_set p) =>
              exist (Ensembles.In (relativization_set p)) (proj1_sig x0 * proj1_sig y0)
        (rel_times_closed p (proj1_sig x0) (proj1_sig y0) 
                          (proj2_sig x0) (proj2_sig y0))) = 
           (rel_times p)).
  apply functional_extensionality. intro x'.
  apply functional_extensionality. intro y'.
  apply proj1_sig_injective. simpl.
  rewrite <- (rel_times_compat p x' y').
  reflexivity. 
rewrite h1. unfold bt.
rewrite (rel_ba_compat p). simpl.
reflexivity.
assumption.
Qed.

Lemma rel_ba_plus_compat : 
  forall {B:Bool_Alg} (p:bt B),
    (Bplus (Bc (rel_ba p))) =
    (transfer_dep_r (btype_bc_rel_ba p)
                    (U:=fun T=>T->T->T)
                    (Bplus (rel_bc p))).
intros B p.
simpl. 
rewrite rel_plus_compat'.
rewrite <- (transfer_dep_transfer_dep_r_compat (eq_sym (btype_bc_rel_ba p)) (btype_bc_rel_ba p)).    
symmetry. 
assert (h1:@transfer_dep Type (fun x:Type => x-> x-> x) 
                         _ _ (eq_sym (btype_bc_rel_ba p))
                         (fun x0 y0 : sig_set (relativization_set p) =>
                            exist (Ensembles.In (relativization_set p)) (proj1_sig x0 + proj1_sig y0)
                                  (rel_plus_closed p (proj1_sig x0) (proj1_sig y0) 
                                                    (proj2_sig x0) (proj2_sig y0))) =
           (Bplus (Bc (rel_ba p)))). 
rewrite <- (@transfer_dep_eq_iff Type (fun x:Type => x->x->x) _ _  (eq_sym (btype_bc_rel_ba p))
                                 (fun x0 y0 : sig_set (relativization_set p) =>
                                    exist (Ensembles.In (relativization_set p)) (proj1_sig x0 + proj1_sig y0)
                                          (rel_plus_closed p (proj1_sig x0) (proj1_sig y0) 
                                                            (proj2_sig x0) (proj2_sig y0)))
                                 (Bplus (Bc (rel_ba p)))).  
assert (h1:(fun x0 y0 : sig_set (relativization_set p) =>
              exist (Ensembles.In (relativization_set p)) (proj1_sig x0 + proj1_sig y0)
        (rel_plus_closed p (proj1_sig x0) (proj1_sig y0) 
                          (proj2_sig x0) (proj2_sig y0))) = 
           (rel_plus p)).
  apply functional_extensionality. intro x'.
  apply functional_extensionality. intro y'.
  apply proj1_sig_injective. simpl.
  rewrite <- (rel_plus_compat p x' y').
  reflexivity. 
rewrite h1.  unfold bt.
rewrite (rel_ba_compat p). simpl.
reflexivity.
assumption.
Qed.

Lemma rel_ba_comp_compat : 
  forall {B:Bool_Alg} (p:bt B),
    (Bcomp (Bc (rel_ba p))) =
    (transfer_dep_r (btype_bc_rel_ba p)
                    (U:=fun T=>T->T)
                    (Bcomp (rel_bc p))).
intros B p.
simpl. 
rewrite rel_comp_compat'.
rewrite <- (transfer_dep_transfer_dep_r_compat (eq_sym (btype_bc_rel_ba p)) (btype_bc_rel_ba p)).    
symmetry. 
assert (h1:@transfer_dep Type (fun x:Type => x-> x) 
                         _ _ (eq_sym (btype_bc_rel_ba p))
                         (fun x0: sig_set (relativization_set p) =>
                            exist (
                                Ensembles.In (relativization_set p)) (- proj1_sig x0 * p)
                                  (rel_comp_closed p (proj1_sig x0) 
                                                   (proj2_sig x0) )) =
           (Bcomp (Bc (rel_ba p)))). 
rewrite <- (@transfer_dep_eq_iff Type (fun x:Type => x->x) _ _  (eq_sym (btype_bc_rel_ba p))
                                 (fun x0 : sig_set (relativization_set p) =>
                                    exist (Ensembles.In (relativization_set p)) (-proj1_sig x0 * p)
                                          (rel_comp_closed p (proj1_sig x0) 
                                                            (proj2_sig x0)))
                                 (Bcomp (Bc (rel_ba p)))).  
assert (h1:(fun x0 : sig_set (relativization_set p) =>
              exist (Ensembles.In (relativization_set p)) (- proj1_sig x0 * p)
        (rel_comp_closed p (proj1_sig x0)  
                          (proj2_sig x0))) = 
           (rel_comp p)).
  apply functional_extensionality. intro x'.
  apply proj1_sig_injective. simpl.
  rewrite <- (rel_comp_compat p x').
  reflexivity. 
rewrite h1. unfold bt.
rewrite (rel_ba_compat p). simpl.
reflexivity.
assumption.
Qed.



Definition point_induced_homo 
           (F:Field_of_Sets) (x0:Xt F) :
  bt (fos_ba F) -> (bt two_bool_ba).
unfold bt. 
rewrite two_bool_ba_compat. simpl. 
intro P. destruct P as [P].
refine (char_fun P x0).
Defined.


Lemma homo_point_induced_homo : 
  forall (F:Field_of_Sets) (x0:Xt F),
  homomorphism (point_induced_homo F x0).
intros F x0. apply homo_two_ops_times.    
unfold fos_ba. simpl.
intros P Q. simpl in P. simpl in Q. simpl.   
destruct P as [P h1]. destruct Q as [Q h2]. 
unfold Int_fos.  simpl. unfold point_induced_homo. simpl.
unfold eq_rect_r. unfold eq_rect.
unfold bt.
destruct (eq_sym two_bool_ba_compat).
simpl.
apply char_fun_int.
intro P. simpl in P.
destruct P as [P h1].  
unfold point_induced_homo.  simpl.
unfold eq_rect_r, eq_rect.
unfold bt.
destruct (eq_sym two_bool_ba_compat).
unfold Comp_fos. simpl.
apply char_fun_comp.
Qed.

Definition function_induced_homo 
           (A B:Field_of_Sets) (f:(Xt A) -> (Xt B))
           (pf:forall P:(bt (fos_ba B)),
                 Ensembles.In (F A) (inv_im (proj1_sig P) f)) :
  (bt (fos_ba B)) -> (bt (fos_ba A)).
intro P. specialize (pf P).
refine (exist _ _ pf).
Defined.

Lemma homo_function_induced_homo : 
  forall (A B:Field_of_Sets) (f:(Xt A) -> (Xt B))
         (pf:forall P:(bt (fos_ba B)),
               Ensembles.In (F A) (inv_im (proj1_sig P) f)),
    homomorphism (function_induced_homo A B f pf).
intros A B f h1.
unfold function_induced_homo.
apply homo_two_ops_times.
intros P Q.
simpl. unfold Int_fos. simpl.
destruct P as [P h2]. destruct Q as [Q h3].
simpl.
apply proj1_sig_injective. simpl.
apply inv_im_int.
intro P.
simpl. unfold Comp_fos. simpl.
destruct P as [P h2]. simpl.
apply proj1_sig_injective. simpl.
apply inv_im_comp.
Qed.

Inductive monomorphism {A B:Bool_Alg}
          (f:(bt A)->(bt B)) : Prop :=
| monomorphism_intro : homomorphism f -> FunctionProperties.injective f -> monomorphism f.

Lemma mono_homo : 
  forall {A B:Bool_Alg}
         (f:(bt A)->(bt B)),
    monomorphism f -> homomorphism f.
intros A B f h1. destruct h1; auto.
Qed.

Inductive epimorphism {A B:Bool_Alg}
          (f:(bt A)->(bt B)) : Prop :=
| epimorphism_intro : homomorphism f -> surjective f -> epimorphism f.

Lemma epi_homo : 
  forall {A B:Bool_Alg}
         (f:(bt A)->(bt B)),
    epimorphism f -> homomorphism f.
intros A B f h1. destruct h1; auto.
Qed.

Inductive isomorphism {A B:Bool_Alg}
          (f:(bt A)->(bt B)) : Prop :=
| isomorphism_intro : homomorphism f -> bijective f -> isomorphism f.

Lemma iso_homo : 
  forall {A B:Bool_Alg} 
         (f:(bt A)->(bt B)), 
         isomorphism f -> homomorphism f.
intros A B f h1. destruct h1; auto.
Qed.

Lemma iso_epi_mono_compat_iff : 
  forall {A B:Bool_Alg}
         (f:(bt A)->(bt B)),
    isomorphism f <-> monomorphism f /\ epimorphism f.
intros A B f.
split.
intro h1.
destruct h1 as [h1 h2].
red in h2.
destruct h2 as [h2l h2r].
split. constructor; auto.
constructor; auto.
intro h1.
destruct h1 as [h1 h2].
destruct h1 as [h1 h3]. destruct h2 as [h1' h4].
constructor; auto.
red. split; auto.
Qed.

Definition iso_inv {A B:Bool_Alg} (f:bt A->bt B)
           (pf:isomorphism f) : bt B -> bt A.
destruct pf as [h1 h2].
apply bijective_impl_invertible in h2.
refine (proj1_sig (function_inverse _ h2)).
Defined.

Lemma iso_inv_compat : 
  forall {A B:Bool_Alg} (f:bt A->bt B)
         (pf:isomorphism f),
    isomorphism (iso_inv f pf) /\
    (forall x, (iso_inv f pf) (f x) = x) /\
    (forall x, f (iso_inv f pf x) = x).
    
intros A B f h1.
destruct h1 as [h1 h2].
unfold iso_inv. 
split. split.
unfold iso_inv.
apply homo_inv; auto.
apply invertible_impl_bijective.
apply invertible_impl_inv_invertible.
unfold function_inverse.
destruct constructive_definite_description. simpl. 
assumption.
Qed.


Definition automorphism {A:Bool_Alg}
      (f:(bt A)->(bt A)) : Prop :=
  isomorphism f.

Lemma automorphism_id : 
  forall A:Bool_Alg, 
    @automorphism A id.
intro A. red.
constructor; [apply homo_id | apply bij_id].
Qed.

Definition isomorphic (A B:Bool_Alg) : Prop :=
    exists phi:(bt A)->(bt B),
      isomorphism phi.

Notation "A =~ B" := (isomorphic A B) (at level 30).



Lemma refl_iso : Reflexive _ isomorphic.
red.
intro A.
red.
exists id.
apply automorphism_id.
Qed.

Lemma symm_iso : Symmetric _ isomorphic.
red. intros A B h1.

destruct h1 as [f h1]. destruct h1 as [h1 h2].
apply bijective_impl_invertible in h2.
exists (proj1_sig (function_inverse _ h2)). 
constructor.
apply homo_inv.
assumption.
apply invertible_impl_bijective.
apply invertible_impl_inv_invertible; auto.
Qed.

Lemma trans_iso : Transitive _ isomorphic.
red.
intros A B C h1 h2.
destruct h1 as [f h1].
destruct h2 as [g h2].
destruct h1 as [h1l h1r]. destruct h2 as [h2l h2r].
exists (compose g f). 
constructor.
apply homo_compose; auto.
apply bij_compose; auto.
Qed.

Lemma equivalence_isomorphic : 
  Equivalence _ isomorphic.
constructor.
apply refl_iso.
apply trans_iso.
apply symm_iso.
Qed.

Definition sub_isomorphic (A B:Bool_Alg) :=
  exists (C:Ensemble (bt B))
         (pf:alg_closed C),
    A =~ Subalg _ pf.

Notation "A <=~ B" := (sub_isomorphic A B) (at level 30).


Lemma rel_homo_psa : 
  forall {Xt:Type} 
         (Y P:bt (psa Xt)),
    rel_homo Y P = Intersection P Y.
intros Xt Y P.
unfold rel_homo; simpl; auto.
Qed.


Lemma rel_set_psa : 
  forall {Xt:Type} (Y:bt (psa Xt)),
    relativization_set Y = power_set Y.
intros Xt Y.
unfold relativization_set. simpl.
unfold le. simpl.
assert (h1:forall P:bt (psa Xt),
             Union P Y = Y <-> Included P Y).
  intros; rewrite inclusion_iff_union. tauto.
simpl in Y. simpl in h1.
rewrite (sat_iff _ _ h1).
unfold power_set.
reflexivity.
Qed.


Lemma btype_bc_rel_ba_psa : 
  forall {Xt:Type} (Y:bt (psa Xt)),
    bt (rel_ba Y) = sig_set (power_set Y).
intros Xt Y. unfold bt.
pose proof (rel_ba_compat Y) as h1.
rewrite h1.
simpl.
f_equal.
apply rel_set_psa.
Qed.

Lemma sig_set_psa : 
  forall {Xt:Type} (Y:bt (psa Xt)),
    sig_set (relativization_set Y) = sig_set (power_set Y).
intros Xt Y. f_equal.
apply rel_set_psa.
Qed.


Definition proj1_sig_rel_ba_psa 
           {Xt:Type} (Y:bt (psa Xt))
           (N:bt (rel_ba Y)) : Ensemble Xt :=
  (proj1_sig (transfer (btype_bc_rel_ba_psa Y) N)).

Lemma proj1_sig_rel_ba_psa_compat :
  forall {Xt:Type} (Y:bt (psa Xt))
         (N:bt (rel_ba Y)),
       Included (proj1_sig_rel_ba_psa Y N) Y. 
intros Xt Y N.
apply (proj2_sig (transfer (btype_bc_rel_ba_psa Y) N)).
Qed.

Lemma proj1_sig_rel_ba_psa_compat' :
  forall {Xt:Type} (Y:bt (psa Xt))
         (N:bt (rel_ba Y)),
       Ensembles.In (power_set Y) (proj1_sig_rel_ba_psa Y N).
intros Xt Y N. 
constructor. 
apply proj1_sig_rel_ba_psa_compat.
Qed.

Lemma proj1_sig_rel_ba_psa_inj :
  forall {Xt:Type} (Y:bt (psa Xt))
    (A B:bt (rel_ba Y)),
    proj1_sig_rel_ba_psa Y A =
    proj1_sig_rel_ba_psa Y B ->
    A = B.
intros Xt Y A B h1.
unfold proj1_sig_rel_ba_psa in h1.
apply proj1_sig_injective in h1.
apply transfer_inj in h1.
assumption.
Qed.


Lemma rel_ba_psa_times_compat :
  forall {Xt:Type} (Y:bt (psa Xt)),
    Btimes (Bc (rel_ba Y)) =
    transfer_dep_r (btype_bc_rel_ba_psa Y)
                    (U:=fun T=>T->T->T)
                    (fun A B : sig_set (power_set Y) =>
                       exist _ _ 
                             (in_power_intersection (proj1_sig A) (proj1_sig B) Y (proj2_sig A))). 
intros Xt Y. pose proof (rel_ba_times_compat Y) as h1. 
simpl in h1. 
rewrite rel_times_compat' in h1.
rewrite h1. simpl. 
rewrite (@transfer_dep_r_hetero _ (fun y=>y->y->y) _ _ _ (btype_bc_rel_ba Y) 
                               (btype_bc_rel_ba_psa Y)).   
apply functional_extensionality. intro A.
apply functional_extensionality. intro B.
apply proj1_sig_injective. 
rewrite transfer_dep_r_fun2_eq.
pose proof (rel_set_psa Y) as h2.
do 2 rewrite (transfer_sig_set_eq _ _ h2 (eq2 (btype_bc_rel_ba Y) (btype_bc_rel_ba_psa Y))).
simpl.
destruct A as [A h3]. destruct B as [B h4]. simpl. 
pose proof (eq2 (btype_bc_rel_ba Y) (btype_bc_rel_ba_psa Y)) as h5. 
rewrite (transfer_r_sig_set_eq _ _ h2 (eq2 (btype_bc_rel_ba Y) (btype_bc_rel_ba_psa Y))).
simpl.
reflexivity.
Qed.


Lemma rel_ba_psa_times_compat' :
  forall {Xt : Type} (Y:bt (psa Xt))
    (A B:bt (rel_ba Y)),
    A * B = transfer_r (btype_bc_rel_ba_psa Y) 
                       (exist _ _ (in_power_intersection 
                              (proj1_sig_rel_ba_psa Y A)
                              (proj1_sig_rel_ba_psa Y B) Y
                              (proj1_sig_rel_ba_psa_compat' _ A))).
intros Xt Y A B.
rewrite rel_ba_psa_times_compat.
rewrite transfer_dep_r_fun2_eq.
f_equal.
apply proj1_sig_injective. simpl.
unfold proj1_sig_rel_ba_psa.
reflexivity.
Qed.



Lemma rel_ba_psa_plus_compat :
  forall {Xt:Type} (Y:bt (psa Xt)),
    Bplus (Bc (rel_ba Y)) =
    transfer_dep_r (btype_bc_rel_ba_psa Y)
                    (U:=fun T=>T->T->T)
                    (fun A B : sig_set (power_set Y) =>
                       exist _ _ 
                             (in_power_union (proj1_sig A) (proj1_sig B) Y (proj2_sig A) (proj2_sig B))). 
intros Xt Y. pose proof (rel_ba_plus_compat Y) as h1. 
simpl in h1. 
rewrite rel_plus_compat' in h1.
rewrite h1. simpl. 
rewrite (@transfer_dep_r_hetero _ (fun y=>y->y->y) _ _ _ (btype_bc_rel_ba Y) 
                               (btype_bc_rel_ba_psa Y)).   
apply functional_extensionality. intro A.
apply functional_extensionality. intro B.
apply proj1_sig_injective. 
rewrite transfer_dep_r_fun2_eq.
pose proof (rel_set_psa Y) as h2.
do 2 rewrite (transfer_sig_set_eq _ _ h2 (eq2 (btype_bc_rel_ba Y) (btype_bc_rel_ba_psa Y))).
simpl.
destruct A as [A h3]. destruct B as [B h4]. simpl. 
pose proof (eq2 (btype_bc_rel_ba Y) (btype_bc_rel_ba_psa Y)) as h5. 
rewrite (transfer_r_sig_set_eq _ _ h2 (eq2 (btype_bc_rel_ba Y) (btype_bc_rel_ba_psa Y))).
simpl.
reflexivity.
Qed.



Lemma rel_ba_psa_plus_compat' :
  forall {Xt : Type} (Y:bt (psa Xt))
    (A B:bt (rel_ba Y)),
    A + B = transfer_r (btype_bc_rel_ba_psa Y) 
                       (exist _ _ (in_power_union 
                              (proj1_sig_rel_ba_psa Y A)
                              (proj1_sig_rel_ba_psa Y B) Y
                              (proj1_sig_rel_ba_psa_compat' _ A)
                              (proj1_sig_rel_ba_psa_compat' _ B)

)).
intros Xt Y A B.
rewrite rel_ba_psa_plus_compat.
rewrite transfer_dep_r_fun2_eq.
f_equal.
apply proj1_sig_injective. simpl.
unfold proj1_sig_rel_ba_psa.
reflexivity.
Qed.



Lemma rel_ba_psa_comp_compat :
  forall {Xt:Type} (Y:bt (psa Xt)),
    Bcomp (Bc (rel_ba Y)) =
    transfer_dep_r (btype_bc_rel_ba_psa Y)
                    (U:=fun T=>T->T)
                    (fun A: sig_set (power_set Y) =>
                       exist _ _ 
                             (in_power_comp (proj1_sig A) Y (proj2_sig A))).
intros Xt Y. pose proof (rel_ba_comp_compat Y) as h1. 
simpl in h1. 
rewrite rel_comp_compat' in h1.
rewrite h1. simpl. 
rewrite (@transfer_dep_r_hetero _ (fun y=>y->y) _ _ _ (btype_bc_rel_ba Y) 
                               (btype_bc_rel_ba_psa Y)).   
apply functional_extensionality. intro A.
apply proj1_sig_injective. 
rewrite transfer_dep_r_fun1_eq.
pose proof (rel_set_psa Y) as h2.
rewrite (transfer_sig_set_eq _ _ h2 (eq2 (btype_bc_rel_ba Y) (btype_bc_rel_ba_psa Y))).
simpl.
destruct A as [A h3].
pose proof (eq2 (btype_bc_rel_ba Y) (btype_bc_rel_ba_psa Y)) as h5. 
rewrite (transfer_r_sig_set_eq _ _ h2 (eq2 (btype_bc_rel_ba Y) (btype_bc_rel_ba_psa Y))).
simpl.
rewrite setminus_int_complement.
rewrite comm_prod_psa.
reflexivity.
Qed.


Lemma rel_ba_psa_comp_compat' :
  forall {Xt : Type} (Y:bt (psa Xt))
    (A:bt (rel_ba Y)),
    - A = transfer_r (btype_bc_rel_ba_psa Y) 
                       (exist _ _ (in_power_comp
                              (proj1_sig_rel_ba_psa Y A) Y
                              (proj1_sig_rel_ba_psa_compat' _ A)
                       )).
intros Xt Y A.
rewrite rel_ba_psa_comp_compat.
rewrite transfer_dep_r_fun1_eq.
f_equal.
apply proj1_sig_injective. simpl.
unfold proj1_sig_rel_ba_psa.
reflexivity.
Qed.


Lemma proj1_sig_rel_ba_psa_times : 
  forall {Xt : Type} (Y:bt (psa Xt))
    (A B:bt (rel_ba Y)),
  proj1_sig_rel_ba_psa _ (A * B) = 
  Intersection (proj1_sig_rel_ba_psa _ A)
               (proj1_sig_rel_ba_psa _ B).
intros Xt Y A B. 
pose proof (rel_ba_psa_times_compat' Y A B) as h0.
pose proof (f_equal (transfer (btype_bc_rel_ba_psa Y)) h0) as h2.
rewrite transfer_undoes_transfer_r in h2.
pose proof (f_equal (@proj1_sig _ _) h2) as h3.
simpl in h3.
unfold proj1_sig_rel_ba_psa.
assumption.
Qed.

Lemma proj1_sig_rel_ba_psa_plus : 
  forall {Xt : Type} (Y:bt (psa Xt))
    (A B:bt (rel_ba Y)),
  proj1_sig_rel_ba_psa _ (A + B) = 
  Union (proj1_sig_rel_ba_psa _ A)
               (proj1_sig_rel_ba_psa _ B).
intros Xt Y A B. 
pose proof (rel_ba_psa_plus_compat' Y A B) as h0.
pose proof (f_equal (transfer (btype_bc_rel_ba_psa Y)) h0) as h2.
rewrite transfer_undoes_transfer_r in h2.
pose proof (f_equal (@proj1_sig _ _) h2) as h3.
simpl in h3.
unfold proj1_sig_rel_ba_psa.
assumption.
Qed.

Lemma proj1_sig_rel_ba_psa_comp : 
  forall {Xt : Type} (Y:bt (psa Xt))
    (A:bt (rel_ba Y)),
  proj1_sig_rel_ba_psa _ (- A) = 
  Setminus Y (proj1_sig_rel_ba_psa _ A).
intros Xt Y A. 
pose proof (rel_ba_psa_comp_compat' Y A) as h0.
pose proof (f_equal (transfer (btype_bc_rel_ba_psa Y)) h0) as h2.
rewrite transfer_undoes_transfer_r in h2.
pose proof (f_equal (@proj1_sig _ _) h2) as h3.
simpl in h3.
unfold proj1_sig_rel_ba_psa.
assumption.
Qed.


Lemma rel_ba_psa_sig_set_iso : 
  forall {Xt:Type} (Y:(bt (psa Xt))),
    rel_ba Y =~ psa (sig_set Y).
intros Xt Y. red.
pose proof (rel_ba_compat Y) as h1. simpl in h1.
assert (h2:forall P, exists! V:Btype (Bc (psa (sig_set Y))),
             Im V (@proj1_sig _ _) = rel_homo P Y).
  intro P.
  assert (h2:Included (rel_homo P Y) Y). unfold rel_homo. simpl. auto with sets.
  exists (Im (full_sig (rel_homo P Y)) (fun v=>(exist _ _ (h2  _ (proj2_sig v))))).
  red. split.
  rewrite im_im. simpl.
  apply Extensionality_Ensembles.
  red. split.
  red. intros a h3.
  destruct h3 as [a h3]. subst.
  unfold rel_homo. simpl.
  apply proj2_sig.
  red.
  intros a h3.
  apply Im_intro with (exist _ _ h3).
  constructor. simpl.
  reflexivity. 
  intros V h3. 
  apply Extensionality_Ensembles.
  red. split.
  red.
  intros y h4.  
  destruct y as [y h5]. simpl.
  inversion h4 as [v h4b]. subst.
  apply exist_injective in H. subst. 
  pose proof (proj2_sig v) as h7. simpl in h7.
  rewrite <- h3 in h7 at 1.
  inversion h7 as [v' h8]. subst. 
  simpl in V. 
  destruct v.  simpl.  simpl in H. subst. 
  destruct v'.  simpl. simpl in h5. 
  assert (h9:i0 = h5). apply proof_irrelevance. subst.
  assumption. 
  red. intros x h4.
  destruct x as [x h5].
  assert (h6:Ensembles.In (rel_homo P Y) x). rewrite <- h3 at 1.
  apply Im_intro with (exist (fun x0:Xt => Ensembles.In Y x0) x h5).
  assumption.  simpl.  reflexivity. 
  apply Im_intro with (exist _ _ h6).
  constructor. apply proj1_sig_injective.  simpl. 
  reflexivity.   

pose (fun P:Btype (Bc (rel_ba Y)) =>
        (proj1_sig (constructive_definite_description 
                      _ (h2 (proj1_sig_rel_ba_psa Y P))))) as f.
unfold psa. simpl.
exists f. 
assert (h3:bijective f).

red. split.
red.
unfold f.
intros A B.
destruct constructive_definite_description as [A' h4].
destruct constructive_definite_description as [B' h5].
simpl.
intro h6. subst.
rewrite h5 in h4.
unfold rel_homo in h4. simpl in h4.
pose proof (proj1_sig_rel_ba_psa_compat _ A) as h7.
pose proof (proj1_sig_rel_ba_psa_compat _ B) as h8.
rewrite inclusion_iff_intersection_eq in h7.
rewrite inclusion_iff_intersection_eq in h8. 
rewrite comm_prod_psa in h7. rewrite comm_prod_psa in h8.
rewrite h7 in h4. rewrite h8 in h4.
apply proj1_sig_rel_ba_psa_inj in h4. subst.
reflexivity.
red. simpl.
intro B.
simpl in f. 
assert (h3:Ensembles.In (relativization_set Y) (Im B (@proj1_sig _ _))).
  constructor. unfold le. simpl.
  rewrite <- inclusion_iff_union.
  red.
  intros x h3.
  destruct h3 as [x h3]. subst.
  apply proj2_sig.
exists (transfer_r (btype_bc_rel_ba Y) (exist _ _ h3)).
unfold f.
destruct constructive_definite_description as [V h4].
simpl.
unfold rel_homo in h4.
simpl in h4.
pose proof (proj1_sig_rel_ba_psa_compat _ 
 (transfer_r (btype_bc_rel_ba Y)
               (exist (Ensembles.In (relativization_set Y))
                  (Im B (proj1_sig (P:=fun x : Xt => Ensembles.In Y x))) h3))) as h5.
rewrite inclusion_iff_intersection_eq in h5.
rewrite comm_prod_psa in h5. 
rewrite h5 in h4 at 1. 
unfold proj1_sig_rel_ba_psa in h4.  
rewrite transfer_transfer_r_hetero in h4.
pose proof (rel_set_psa Y) as h6.
pose proof (transfer_sig_set_eq _ _ h6 
                             (eq2 (btype_bc_rel_ba Y) (btype_bc_rel_ba_psa Y))  (exist (Ensembles.In (relativization_set Y))
               (Im B (proj1_sig (P:=fun x : Xt => Ensembles.In Y x))) h3)) as h7.
rewrite h7 in h4 at 1.
simpl in h4.
apply im_inj_inj in h4; auto.
red.
intros; apply proj1_sig_injective; auto. 
constructor.
Focus 2.
assumption.

apply homo_two_ops_times.
intros A B.
unfold f.
assert (h0:FunctionProperties.injective (proj1_sig (P:=fun x : Xt => Ensembles.In Y x))).
  red; intros; apply proj1_sig_injective; auto.

pose proof (proj1_sig_rel_ba_psa_times Y A B) as h7.
destruct constructive_definite_description as [A' h4].
destruct constructive_definite_description as [B' h5].
destruct constructive_definite_description as [V  h6].
simpl. 
unfold rel_homo in h4. simpl in h4.
unfold rel_homo in h5. simpl in h5.
unfold rel_homo in h6. simpl in h6.
pose proof (proj1_sig_rel_ba_psa_compat Y (A*B)) as h8.
pose proof (proj1_sig_rel_ba_psa_compat Y A) as h9.
pose proof (proj1_sig_rel_ba_psa_compat Y B) as h10.
rewrite inclusion_iff_intersection_eq in h8.
rewrite inclusion_iff_intersection_eq in h9.
rewrite inclusion_iff_intersection_eq in h10.
rewrite comm_prod_psa in h8 at 1. rewrite comm_prod_psa in h9 at 1.
rewrite comm_prod_psa in h10 at 1.
rewrite h8 in h4. rewrite h9 in h5. rewrite h10 in h6.
clear h8 h9 h10.
rewrite h7 in h4 at 1.
rewrite <- h6 in h4. rewrite <- h5 in h4.
rewrite <- im_intersection_inj in h4; auto.
apply im_inj_inj in h4; auto.

intro A. 
pose proof (proj1_sig_rel_ba_psa_comp Y A) as h7.
unfold f.
destruct constructive_definite_description as [A' h4].
destruct constructive_definite_description as [V  h6].
simpl. 
unfold rel_homo in h4. simpl in h4.
unfold rel_homo in h6. simpl in h6.
pose proof (proj1_sig_rel_ba_psa_compat Y (-A)) as h8.
pose proof (proj1_sig_rel_ba_psa_compat Y A) as h9.
rewrite inclusion_iff_intersection_eq in h8.
rewrite inclusion_iff_intersection_eq in h9.
rewrite comm_prod_psa in h8 at 1. rewrite comm_prod_psa in h9 at 1.
rewrite h8 in h4. rewrite h9 in h6.
clear h8 h9.
rewrite h7 in h4 at 1.
rewrite <- h6 in h4. 
simpl in V. simpl in Y.
assert (h9: Setminus Y (Im V (proj1_sig (P:=fun x : Xt => Ensembles.In Y x))) = Im (Ensembles.Complement V) (@proj1_sig _ _)).
  apply Extensionality_Ensembles.
  red. split.
  intros x h10.
  destruct h10 as [h10 h11].
  apply Im_intro with (exist _ _ h10).
  intro h12.
  contradict h11.
  apply Im_intro with (exist _ _ h10).
  assumption. simpl. reflexivity.
  simpl. reflexivity.
  red. intros x h10.
  destruct h10 as [x h10]. subst.
  constructor. apply proj2_sig.
  intro h11. 
  inversion h11 as [x' h12 y h13]. subst. apply proj1_sig_injective in h13.
  subst.
  contradiction.
rewrite h9 in h4.
apply im_inj_inj in h4; auto.
red. intros; apply proj1_sig_injective; auto.
Qed.


Lemma same_card_type_iso_psa :
  forall (Xt Yt:Type),
    (exists f:Xt->Yt, bijective f) ->
      psa Yt =~ psa Xt.
intros Wt Yt h1.  
destruct h1 as [f h1].
pose proof (fos_psa_compat Wt) as h3.  pose proof (fos_psa_compat Yt) as h4.
destruct h3 as [h3l h3r]. destruct h4 as [h4l h4r].
simpl. 
pose (fun P:Ensemble Yt => inv_im P f) as f''.
pose proof h3l as h3l'. symmetry in h3l'. pose proof h4l as h4l'. symmetry in h4l'.
pose (fun x:(Xt (fos_psa Wt)) => (transfer h4l' (f (transfer h3l x)))) as f'. 
pose proof (transfer_fun_pred' h3l h4l' f (fun pr:{pT:Type*Type & fst pT -> snd pT} => bijective (projT2 pr))) as ht.
simpl in ht.
pose proof h1 as h1'.
rewrite ht in h1'.
assert (hbij:bijective f').
  unfold f'.
  unfold transfer at 1 in h1'.
  unfold eq_rect_r in h1'.
  rewrite <- eq_rect_eq in h1'. 
  assumption.
assert (h5:(forall P : bt (fos_ba (fos_psa Yt)),
         Ensembles.In (F (fos_psa Wt)) (inv_im (proj1_sig P) f'))).
  rewrite h3r. intros. constructor. red; intros; constructor.
pose (function_induced_homo (fos_psa Wt) (fos_psa Yt) f' h5) as phi.
simpl in phi.
unfold Ft in phi.
pose (fun B:Ensemble Yt => proj1_sig_fos_psa (phi (exist_fos_psa B))) as g.
exists g.   
constructor. 
pose proof (homo_function_induced_homo (fos_psa Wt) (fos_psa Yt) f' h5) as h6.
destruct h6 as [h6a h6b h6c].
apply homo_two_ops_times. 
intros A B.
simpl in A, B.
simpl in h6a.
unfold Ft in h6a.
specialize (h6a (exist_fos_psa A) (exist_fos_psa B)).
unfold Int_fos in h6a. simpl in h6a.
unfold function_induced_homo in h6a. 
simpl in h6a.
pose proof (f_equal (@proj1_sig _ _) h6a) as h6a'.
simpl in h6a'.
unfold g.  simpl.
destruct (fos_psa_compat Wt) as [h8l h8r].
rewrite <- h8l.
do 3 rewrite <- eq_rect_eq. 
rewrite proj1_sig_exist_fos_psa_int.
rewrite h6a'.
reflexivity.
intro A. simpl in A.
simpl in h6c.
specialize (h6c (exist_fos_psa A)).
unfold Comp_fos in h6c. simpl in h6c.
unfold function_induced_homo in h6c. simpl in h6c.
pose proof (f_equal (@proj1_sig _ _) h6c) as h6c'.
simpl in h6c'.
unfold g. simpl.
destruct (fos_psa_compat Wt) as [h8l h8r].
rewrite <- h8l.
do 2 rewrite <- eq_rect_eq.
rewrite proj1_sig_exist_fos_psa_comp.
assumption. 
pose proof hbij as hbij'.
red in hbij'. destruct hbij' as [hb1 hb2].
red in h1.
destruct h1 as [h1l h1r].
red in h1l. red in h1r.  
red. split.
red.
intros A B h1.
unfold g in h1. 
unfold phi in h1. unfold exist_fos_psa in h1.
destruct (fos_psa_compat Yt) as [h2 h3].
unfold proj1_sig_fos_psa in h1.
destruct (fos_psa_compat Wt) as [h6 h7]. simpl in h1.
destruct h6.  
do 2 rewrite <- eq_rect_eq in h1.
unfold eq_rect_r in h1.
pose proof (inv_im_surj_inj f' hb2 _ _ h1) as hp.
apply proj1_sig_injective in hp.
rewrite h3 in hp.
simpl in hp.
pose proof (f_equal (@proj1_sig _ _) hp) as hp'.
rewrite h2 in hp'. simpl in hp'.
assumption. 
red.
intro A.  
unfold g. 
red in hb2. simpl in A. 
unfold phi.
simpl.
exists (Im A f).    
destruct (fos_psa_compat Wt) as [h7 h8].     
apply Extensionality_Ensembles.
red. split.
red. intros x h9.
rewrite in_eq_rect in h9.
destruct h9 as [h9].
pose proof (exist_fos_psa_compat (Im A f)) as h10.
rewrite h10 in h9.
unfold f' in h9.
destruct (fos_psa_compat Yt) as [h11 h12].
pose proof h11 as h11'. symmetry in h11'.
rewrite <- (transfer_dep_transfer_dep_r_compat h11' h11) in h9.
assert (h13:h4l' = h11'). apply proof_irrelevance.
rewrite h13 in h9.
rewrite <- transfer_in in h9.
inversion h9 as [w h15 y h17]. 
apply h1l in h17.
rewrite <- h17 in h15.  
assert (h18:h7=h3l). apply proof_irrelevance. rewrite h18 in h15.
rewrite <- transfer_r_eq in h15.
rewrite transfer_undoes_transfer_r in h15.
assumption.
red.
intros x h9.
rewrite in_eq_rect.
constructor.
pose proof (exist_fos_psa_compat (Im A f)) as h10.
rewrite h10.
destruct (fos_psa_compat Yt) as [h11 h12].
pose proof h11 as h11'. symmetry in h11'.
rewrite <- (transfer_dep_transfer_dep_r_compat h11' h11).
simpl. 
unfold f'.
assert (h13:h11' = h4l'). apply proof_irrelevance.
rewrite h13.
rewrite <- transfer_in.
apply Im_intro with (transfer h3l (eq_rect_r id x h7)).
rewrite <- transfer_r_eq.
assert (h14:h3l = h7). apply proof_irrelevance. rewrite h14.
rewrite transfer_undoes_transfer_r.
assumption. reflexivity.
Qed.

Lemma same_card_type_iso_psa' :
  forall (Xt Yt:Type),
    (exists f:Xt->Yt, bijective f) ->
    psa Xt =~ psa Yt.
intros Xt Yt h1.
assert (h2:exists g:Yt -> Xt, bijective g).
destruct h1 as [f h1].
apply bijective_impl_invertible in h1.
exists (proj1_sig (function_inverse f h1)).
apply invertible_impl_bijective.
apply invertible_impl_inv_invertible.
eapply same_card_type_iso_psa; auto.
Qed.


Inductive complete_homo {A B:Bool_Alg} 
  (f:bt A->bt B) : Prop :=
  complete_homo_intro : 
    homomorphism f -> 
    (forall (S:Ensemble (bt A)) p,
       sup S p -> sup (Im S f) (f p)) ->
    complete_homo f.

Inductive complete_homo_inv {A B:Bool_Alg}
  (f:bt A->bt B) : Prop :=
  complete_homo_inv_intro : 
    homomorphism f -> 
    (forall (S:Ensemble (bt B)) p,
       sup S (f p) -> sup (inv_im S f) p) ->
    complete_homo_inv f.
          

Lemma iso_complete : 
  forall {A B:Bool_Alg} (f:bt A->bt B),
    isomorphism f -> complete_homo f.
intros A B f h1.
destruct h1 as [h2 h3].
constructor. assumption.
intros S p h4. 
red in h4. destruct h4 as [h4l h4r]. 
assert (h5:ub (Im S f) (f p)). 
  red. intros s h5. 
  destruct h5 as [s h5]. subst.
  red in h4l. specialize (h4l _ h5).
  apply homo_mono; auto. 

red. split. auto.
intros b' h6. 
apply bijective_impl_invertible in h3.
pose (function_inverse _ h3) as f'.  
red in h6. 
apply NNPP.
intro h7.
assert (h8: ~ le p (proj1_sig f' b')).
  intro h8.
  apply (homo_mono _ h2) in h8.
  pose proof (proj2_sig (function_inverse f h3)) as h9. simpl in h9.
  destruct h9 as [h9l h9r].
  rewrite h9r in h8.
  contradiction.
specialize (h4r (proj1_sig f' b')).
assert (h9:~ub S (proj1_sig f' b')). tauto. 
unfold ub in h9.
apply not_all_ex_not in h9.
destruct h9 as [c h9].
assert (h10:Ensembles.In S c /\ ~le c (proj1_sig f' b')). tauto. clear h9.
destruct h10 as [h10l h10r].
specialize (h4l _ h10l).
assert (h11:Ensembles.In (Im S f) (f c)). apply Im_intro with c; auto.
specialize (h6 _ h11).
pose proof (homo_inv _ h3 h2) as h12.
apply (homo_mono _ h12) in h6.
rewrite (match (proj2_sig (function_inverse f h3)) with
             | conj P  _ => P end) in h6.
contradiction.
Qed.

Lemma iso_complete_inv : 
  forall {A B:Bool_Alg} (f:bt A->bt B),
    isomorphism f -> complete_homo_inv f.
intros A B f h1.
pose proof (iso_inv_compat _ h1) as h2.
destruct h2 as [h2l h2r].
apply iso_complete in h2l.
destruct h2l as [h3 h4].
constructor.
destruct h1; auto.
intros S p h5.
specialize (h4 _ _ h5).
destruct h2r as [h2a h2b].
rewrite h2a in h4.
assert (h6:Im S (iso_inv f h1) = inv_im S f).
  apply Extensionality_Ensembles.
  red. split.
  red.
  intros x h6.
  destruct h6 as [y h6]. subst.
  constructor. rewrite h2b.
  assumption.
  red. intros x h6. destruct h6 as [h6].
  apply Im_intro with (f x); auto.
rewrite h6 in h4.
assumption.
Qed.


Lemma mono_iso_bij_sig_im_fun : 
  forall {A B:Bool_Alg} (f:bt A->bt B)
         (pf:monomorphism f),
         isomorphism (homo_onto _ (mono_homo _ pf)).
intros A B f h1.
generalize (mono_homo f h1). intro h2.
destruct h1 as [h1l h1r].
assert (h3:h1l = h2). apply proof_irrelevance. subst.
constructor.
apply homo_homo_onto. 
red. split.
rewrite <- (inj_homo_onto_iff f h2); auto.
pose proof (homo_onto_compat _ h2) as h3.
destruct h3; assumption.
Qed.


Lemma mono_iso_bij_sig_im_fun_iff : 
  forall {A B:Bool_Alg} (f:bt A->bt B)
         (pf:homomorphism f),
    FunctionProperties.injective f <-> isomorphism (homo_onto _ pf).
intros A B f h1.
pose proof (homo_onto_compat _ h1) as h0.
destruct h0 as [h0l h0r].
split.
intro h2. constructor; auto. 
apply homo_homo_onto. red. split; auto.
rewrite (inj_homo_onto_iff _ h1) in h2.
assumption.
intro h2. destruct h2 as [h2l h2r]. red in h2r.
destruct h2r as [h2a h2b].
rewrite <- (inj_homo_onto_iff f h1) in h2a. 
assumption.
Qed.


Lemma mono_complete_regular_image : 
  forall {A B:Bool_Alg} (f:bt B->bt A),
    monomorphism f ->
    (complete_homo f <->
     regular_subalg _ (Im (BS (Bc B)) f)).
intros A B f h1. 
split. Focus 2. intro h2.
pose proof (mono_iso_bij_sig_im_fun _ h1) as h3.
constructor; auto.
destruct h1 as [h1l h1r]. assumption.
intros Pi p h4.
apply iso_complete in h3.
destruct h3 as [h3a h3b].
specialize (h3b _ _ h4). 
destruct h2 as [h2]. 
destruct h1 as [h1a h1b]. 
specialize (h2 (homo_im_closed _ h1a)).
assert (h5:Included (Im Pi f) (Im (BS (Bc B)) f)).
  red. intros x h5.
  destruct h5 as [x h5]. subst.
  apply Im_intro with x. rewrite und_set. constructor. reflexivity.
specialize (h2 _ h5).
pose proof (homo_onto_compat f (mono_homo f (monomorphism_intro f h1a h1b))) as h7.
destruct h7 as [h7l h7r].
assert (h6:subset_sig (Im Pi f) (Im (BS (Bc B)) f) h5 =
           (Im Pi (homo_onto f (mono_homo f (monomorphism_intro f h1a h1b))))).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h6.
  destruct h6 as [x h6]. subst.
  destruct x as [x h7]. simpl.
  destruct h7 as [x h8]. subst.
  apply Im_intro with x. assumption.
  apply proj1_sig_injective. simpl.
  apply h7r.
  red. intros x h8.
  destruct h8 as [x h8]. subst.
  unfold subset_sig.
  assert (h9:Ensembles.In (Im Pi f) (f x)).
    apply Im_intro with x; auto.
  assert (h10:Ensembles.In (full_sig (Im Pi f)) (exist _ _ h9)).
    constructor.
  apply Im_intro with  (exist (Ensembles.In (Im Pi f)) (f x) h9).
  assumption.
  apply proj1_sig_injective. simpl. symmetry.
  apply h7r.
rewrite <- h6 in h3b at 1.
assert (h8: Ensembles.In (Im (BS (Bc B)) f) (f p)).
  apply Im_intro with p. rewrite und_set. constructor. reflexivity.
specialize (h2 _ h8).
assert (h9:homo_onto f (mono_homo f (monomorphism_intro f h1a h1b)) p = 
           (exist (Ensembles.In (Im (BS (Bc B)) f)) (f p) h8)).
  apply proj1_sig_injective. simpl.
  symmetry. apply h7r.
rewrite h9 in h3b.
apply h2. simpl in h3b. unfold sup. unfold sup in h3b. 
assert (h10:h1a = mono_homo f (monomorphism_intro f h1a h1b)).
  apply proof_irrelevance.
rewrite h10.
assumption.
intro h2.
constructor.
intros h3 Qi h4 q h5 h6. 
destruct h5 as [p h5]. subst.
destruct h2  as [h2l h2r].
pose proof (mono_iso_bij_sig_im_fun _ h1) as h7.
pose (homo_onto f (mono_homo f h1)) as f'. 
assert (h8:h3 = homo_im_closed f (mono_homo f h1)). apply proof_irrelevance. subst.
apply iso_complete_inv in h7.
destruct h7 as [h7ll h7r].
specialize (h7r (subset_sig Qi (Im (BS (Bc B)) f) h4)).
assert (h8: homo_onto f (mono_homo f h1) p =
            exist (Ensembles.In (Im (BS (Bc B)) f)) (f p)
            (Im_intro (bt B) (bt A) 
               (BS (Bc B)) f p h5 (f p) eq_refl)).
  apply proj1_sig_injective. simpl.
pose proof (homo_onto_compat f (mono_homo f h1)) as h7.
destruct h7 as [h8 h9].  symmetry. apply h9.
rewrite <- h8 in h6 at 1.
specialize (h7r _ h6).
apply h2r in h7r.
pose proof (homo_onto_compat f (mono_homo f h1)) as h10.
destruct h10 as [h10l h10r]. 
assert (h9: Im
             (inv_im (subset_sig Qi (Im (BS (Bc B)) f) h4)
                (homo_onto f (mono_homo f h1))) f =
            Qi).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h9.
  destruct h9 as [y h9]. subst.
  destruct h9 as [h9]. 
  rewrite subset_sig_compat in h9.
  rewrite <- h10r in h9.
  assumption. 
  red. intros x h9.
  red in h10l.  pose proof h9 as h9'.
  apply h4 in h9.
  specialize (h10l (exist _ _ h9)).
  destruct h10l as [b h10l].
  apply Im_intro with b.
  constructor. rewrite subset_sig_compat. 
  pose proof (f_equal (@proj1_sig _ _) h10l) as h12. simpl in h12.
  subst. assumption. 
  destruct h9. subst.
  pose proof (f_equal (@proj1_sig _ _) h10l) as h12. simpl in h12.  rewrite <- h10r in h12. 
  symmetry. assumption. 
simpl in h9. simpl in h7r.  
rewrite h9 in h7r at 1.
assumption.
Qed.



(*Analogues of all the above theorems for Bool_Alg_p instead of just Bool_Alg*)
Section ParametricAnalogues.
Variable T:Type.

Record homomorphism_p {Ap Bp:Bool_Alg_p T} (f:btp Ap->btp Bp) : Prop :=
  {
    homo_times_p : forall x y, f (x%*y) = (f x) %* (f y);
    homo_plus_p : forall x y,   f (x%+y) = (f x) %+ (f y);
    homo_comp_p : forall x,   f (%-x) = %- (f x)
  }.

Record homomorphism_p1 {Ap:Bool_Alg_p T} {B:Bool_Alg} 
       (f:btp Ap->bt B) : Prop :=
  {
    homo_times_p1 : forall x y, f (x%*y) = (f x) * (f y);
    homo_plus_p1 : forall x y,   f (x%+y) = (f x) + (f y);
    homo_comp_p1 : forall x,   f (%-x) = - (f x)
  }.

Record homomorphism_p2 {A:Bool_Alg} {Bp:Bool_Alg_p T} 
       (f:bt A->btp Bp) : Prop :=
  {
    homo_times_p2 : forall x y, f (x*y) = (f x) %* (f y);
    homo_plus_p2 : forall x y,   f (x+y) = (f x) %+ (f y);
    homo_comp_p2 : forall x,   f (-x) = %- (f x)
  }.


Lemma homomorphism_p_iff : 
  forall {Ap Bp:Bool_Alg_p T} (f:btp Ap->btp Bp),
    homomorphism_p f <-> homomorphism (ba_conv_fun f).
intros Ap Bp f. split.
intro h1.
destruct h1.
constructor; auto.
intro h1.
destruct h1.
constructor; auto.
Qed.                        

Lemma homomorphism_p1_iff : 
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg} 
         (f:btp Ap->bt B),
    homomorphism_p1 f <-> homomorphism (ba_conv_fun1 f).
intros Ap Bp f. split.
intro h1.
destruct h1.
constructor; auto.
intro h1.
destruct h1.
constructor; auto.
Qed.                        


Lemma homomorphism_p2_iff : 
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T} 
         (f:bt A->btp Bp),
    homomorphism_p2 f <-> homomorphism (ba_conv_fun2 f).
intros A Bp f. split.
intro h1.
destruct h1.
constructor; auto.
intro h1.
destruct h1.
constructor; auto.
Qed.                        


Lemma homo_transfer_iff_p : 
  forall (A B C:Bool_Alg_p T)
         (pf:A = B) (f:btp A -> btp C),
         homomorphism_p f <-> homomorphism_p (transfer (f_equal 
(fun D => btp D->btp C) pf) f).
intros A B C h1 f.
subst. simpl.
rewrite transfer_eq_refl.
tauto.
Qed.


Lemma homo_transfer_r_iff_p : 
  forall (A B C:Bool_Alg_p T)
         (pf:A = B) (f:btp B -> btp C),
         homomorphism_p f <-> homomorphism_p (transfer_r (f_equal 
(fun D => btp D->btp C) pf) f).
intros A B C h1 f.
subst. simpl.
rewrite transfer_r_eq_refl.
tauto.
Qed.

Lemma homo_transfer_iff_p1 : 
  forall (A B:Bool_Alg_p T) (C:Bool_Alg)
         (pf:A = B) (f:btp A -> bt C),
         homomorphism_p1 f <-> homomorphism_p1 (transfer (f_equal 
(fun D => btp D->bt C) pf) f).
intros A B C h1 f.
subst. simpl.
rewrite transfer_eq_refl.
tauto.
Qed.


Lemma homo_transfer_r_iff_p1 : 
  forall (A B:Bool_Alg_p T) (C:Bool_Alg)
         (pf:A = B) (f:btp B -> bt C),
         homomorphism_p1 f <-> homomorphism_p1 (transfer_r (f_equal 
(fun D => btp D->bt C) pf) f).
intros A B C h1 f.
subst. simpl.
rewrite transfer_r_eq_refl.
tauto.
Qed.


       
Lemma homo_transfer_fun_iff_p : 
  forall (A B C:Bool_Alg_p T),
    A = B ->
    forall (f:btp A -> btp C)
           (pf:btp A = btp B),
    homomorphism_p f <-> homomorphism_p (transfer_fun pf f).
intros A B C h1 f h2.
subst.
assert (h3:h2 = eq_refl). apply proof_irrelevance.
subst.
rewrite transfer_fun_eq_refl.
tauto.
Qed.


Lemma homo_transfer_fun_iff_p1 : 
  forall (A B:Bool_Alg_p T) (C:Bool_Alg),
    A = B ->
    forall (f:btp A -> bt C)
           (pf:btp A = btp B),
    homomorphism_p1 f <-> homomorphism_p1 (transfer_fun pf f).
intros A B C h1 f h2.
subst.
assert (h2=eq_refl). apply proof_irrelevance.
subst.
rewrite transfer_fun_eq_refl.
tauto.
Qed.






Record homomorphism_general_p 
       {Ap:Bool_Alg_p T} {A:Ensemble T} 
       (f:btp Ap->sig_set A) 
       (homo_ut_p : sig_set A -> sig_set A -> sig_set A)
       (homo_up_p : sig_set A -> sig_set A -> sig_set A)
       (homo_uc_p : sig_set A-> sig_set A)
: Prop :=
  {
    homo_gen_times_p : forall x y, f (x%*y) = homo_ut_p (f x) (f y);
    homo_gen_plus_p : forall x y,   f (x%+y) = homo_up_p (f x) (f y);
    homo_gen_comp_p : forall x,   f (%-x) = homo_uc_p (f x)
  }.

Lemma homomorphism_general_p_iff :
  forall 
  {Ap:Bool_Alg_p T} {A:Ensemble T} 
  (f:btp Ap->sig_set A) 
  (homo_ut_p : sig_set A -> sig_set A -> sig_set A)
  (homo_up_p : sig_set A -> sig_set A -> sig_set A)
  (homo_uc_p : sig_set A-> sig_set A),
    homomorphism_general_p f homo_ut_p homo_up_p homo_uc_p <->
    homomorphism_general (ba_conv_fun1 f) homo_ut_p homo_up_p homo_uc_p.
intros Ap T' f t p c.
split.
intro h1. destruct h1; constructor; auto.
intro h1. destruct h1; constructor; auto.
Qed.

Lemma homo_sym_diff_p :
  forall {Ap Bp:Bool_Alg_p T} 
         (f:btp Ap->btp Bp),
    homomorphism_p f -> forall x y, f (x /%\ y) = (f x) /%\ (f y).
intros Ap Bp f h1 x y.
rewrite homomorphism_p_iff in h1.
apply (@homo_sym_diff (ba_conv Ap) (ba_conv Bp)); auto.
Qed.

Lemma homo_sym_diff_p1 :
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg}
         (f:btp Ap->bt B),
    homomorphism_p1 f -> forall x y, f (x /%\ y) = (f x) /_\ (f y).
intros Ap B f h1 x y.
rewrite homomorphism_p1_iff in h1.
apply (@homo_sym_diff (ba_conv Ap) B); auto.
Qed.


Lemma homo_sym_diff_p2 :
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T} 
         (f:bt A->btp Bp),
    homomorphism_p2 f -> forall x y, f (x /_\ y) = (f x) /%\ (f y).
intros A Bp f h1 x y.
rewrite homomorphism_p2_iff in h1.
apply (@homo_sym_diff A (ba_conv Bp)); auto.
Qed.



Lemma homo_id_p :
  forall Ap:Bool_Alg_p T,
    homomorphism_p id (Ap:=Ap).
intros; constructor; auto.
Qed.



Lemma universal_homomorphic_ops_imply_bool_p :
  forall 
  {Ap:Bool_Alg_p T} {A:Ensemble T} 
  (f:btp Ap->sig_set A) 
  (ut_p : sig_set A -> sig_set A -> sig_set A)
  (up_p : sig_set A -> sig_set A -> sig_set A)
  (uc_p : sig_set A-> sig_set A)
  (pf:homomorphism_general_p f ut_p up_p uc_p),
    surjective f ->
    exists! Bp:Bool_Alg_p T,
      Bc_p T Bp = Build_Bconst_p T A (full_sig A)
                                 up_p
                          ut_p (f %1) (f %0) uc_p.
intros Ap A f t p c h1 h2.
rewrite homomorphism_general_p_iff in h1.
pose proof (@universal_homomorphic_ops_imply_bool (ba_conv Ap) (sig_set A) f t p c h1 h2) as h3. 
destruct h3 as [B h3].  red in h3. destruct h3 as [h3l h3r]. 
assert (h4:bt B = sig_set A). unfold bt. rewrite h3l. simpl.
  reflexivity.
exists (ba_sig_set_conv A _ h4).
red. split.
rewrite ba_sig_set_conv_bc_p_compat. 
destruct B. simpl. simpl in h3l. simpl in h4.
subst.
unfold bc_sig_set_conv. simpl.
simpl in h4.
assert (h5:h4 = eq_refl _). apply proof_irrelevance.
subst.
do 4 rewrite transfer_dep_eq_refl.
do 2 rewrite transfer_eq_refl.
f_equal.   
intros Bp h5.
specialize (h3r (ba_conv Bp)).
simpl in h3r.
rewrite h5 in h3r. unfold bconst_conv in h3r. simpl in h3r.
specialize (h3r (eq_refl _)).
subst.
apply bc_inj_p. 
rewrite ba_sig_set_conv_bc_p_compat.
simpl. unfold bconst_conv. unfold bc_sig_set_conv.
simpl.
destruct Bp. destruct Bc_p.
simpl. simpl in h4. simpl in h5. simpl in h3l. 
rewrite h5.
inversion h3l as [ha].
unfold bt in h4. simpl in h4.
rewrite (transfer_dep_eq_iff _ _ h4) in H.
rewrite (transfer_dep_eq_iff _ _ h4) in H0.
rewrite (transfer_dep_eq_iff _ _ h4) in H1.
rewrite (transfer_dep_eq_iff _ _ h4) in H2.
rewrite (transfer_dep_eq_iff _ _ h4) in H3.
rewrite (transfer_dep_eq_iff _ _ h4) in H4.
rewrite H, H0, H1, H2, H3, H4 at 1.
reflexivity.
Qed.

Lemma homo_zero_p :
forall {Ap Bp:Bool_Alg_p T} (f:btp Ap->btp Bp),
  homomorphism_p f ->
  f %0 = %0.
intros Ap Bp f h1.
rewrite homomorphism_p_iff in h1.
apply (@homo_zero (ba_conv Ap) (ba_conv Bp)); auto.
Qed.

Lemma homo_zero_p1 :
forall {Ap:Bool_Alg_p T} {B:Bool_Alg} 
       (f:btp Ap->bt B),
  homomorphism_p1 f ->
  f %0 = 0.
intros Ap B f h1.
rewrite homomorphism_p1_iff in h1.
apply (@homo_zero (ba_conv Ap) B); auto.
Qed.


Lemma homo_zero_p2 :
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T} 
         (f:bt A->btp Bp),
  homomorphism_p2 f ->
  f 0 = %0.
intros A Bp f h1.
rewrite homomorphism_p2_iff in h1.
apply (@homo_zero A (ba_conv Bp)); auto.
Qed.



Lemma homo_one_p :
forall {Ap Bp:Bool_Alg_p T} (f:btp Ap->btp Bp),
  homomorphism_p f ->
  f %1 = %1.
intros Ap Bp f h1.
rewrite homomorphism_p_iff in h1.
apply (@homo_one (ba_conv Ap) (ba_conv Bp)); auto.
Qed.


Lemma homo_one_p1 :
forall {Ap:Bool_Alg_p T} {B:Bool_Alg} 
       (f:btp Ap->bt B),
  homomorphism_p1 f ->
  f %1 = 1.
intros Ap B f h1.
rewrite homomorphism_p1_iff in h1.
apply (@homo_one (ba_conv Ap) B); auto.
Qed.

Lemma homo_one_p2 :
forall {A:Bool_Alg} {Bp:Bool_Alg_p T} 
       (f:bt A->btp Bp),
  homomorphism_p2 f ->
  f 1 = %1.
intros A Bp f h1.
rewrite homomorphism_p2_iff in h1.
apply (@homo_one A (ba_conv Bp)); auto.
Qed.



Lemma homo_two_ops_plus_p :
  forall {Ap Bp:Bool_Alg_p T} (f:btp Ap->btp Bp),
    (forall x y, f (x %+ y) = (f x) %+ (f y)) ->
    (forall x, f (%-x) = %- (f x)) ->
    homomorphism_p f.
intros Ap Bp f h1 h2.
rewrite homomorphism_p_iff.
apply (homo_two_ops_plus (ba_conv_fun f) h1 h2).
Qed.

Lemma homo_two_ops_plus_p1 :
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg} 
         (f:btp Ap->bt B),
    (forall x y, f (x %+ y) = (f x) + (f y)) ->
    (forall x, f (%-x) = - (f x)) ->
    homomorphism_p1 f.
intros Ap B f h1 h2.
rewrite homomorphism_p1_iff.
apply (homo_two_ops_plus (ba_conv_fun1 f) h1 h2).
Qed.

Lemma homo_two_ops_plus_p2 :
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T} 
         (f:bt A->btp Bp),
    (forall x y, f (x + y) = (f x) %+ (f y)) ->
    (forall x, f (-x) = %- (f x)) ->
    homomorphism_p2 f.
intros Ap B f h1 h2.
rewrite homomorphism_p2_iff.
apply (homo_two_ops_plus (ba_conv_fun2 f) h1 h2).
Qed.



Lemma homo_two_ops_times_p :
  forall {Ap Bp:Bool_Alg_p T} 
         (f:btp Ap->btp Bp),
    (forall x y, f (x %* y) = (f x) %* (f y)) ->
    (forall x, f (%-x) = %- (f x)) ->
    homomorphism_p f.
intros Ap Bp f h1 h2.
rewrite homomorphism_p_iff.
apply (homo_two_ops_times (ba_conv_fun f) h1 h2).
Qed.


Lemma homo_two_ops_times_p1 :
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg} 
         (f:btp Ap->bt B),
    (forall x y, f (x %* y) = (f x) * (f y)) ->
    (forall x, f (%-x) = - (f x)) ->
    homomorphism_p1 f.
intros Ap B f h1 h2.
rewrite homomorphism_p1_iff.
apply (homo_two_ops_times (ba_conv_fun1 f) h1 h2).
Qed.

Lemma homo_two_ops_times_p2 :
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T} 
         (f:bt A->btp Bp),
    (forall x y, f (x * y) = (f x) %* (f y)) ->
    (forall x, f (-x) = %- (f x)) ->
    homomorphism_p2 f.
intros Ap B f h1 h2.
rewrite homomorphism_p2_iff.
apply (homo_two_ops_times (ba_conv_fun2 f) h1 h2).
Qed.


Lemma ba_conv_fun_compose :
  forall {Ap Bp Cp:Bool_Alg_p T}
         (f:btp Ap->btp Bp)
         (g:btp Bp->btp Cp),
    ba_conv_fun (compose g f) = compose (ba_conv_fun g) (ba_conv_fun f).
intros Ap Bp Cp f g.
unfold ba_conv_fun. unfold compose.
reflexivity.
Qed.

Lemma homo_compose_p :
  forall {Ap Bp Cp:Bool_Alg_p T}
         (f:btp Ap->btp Bp)
         (g:btp Bp->btp Cp),
    homomorphism_p f -> homomorphism_p g ->
    homomorphism_p (compose g f).
intros Ap Bp Cp f g h1 h2.
rewrite homomorphism_p_iff in h1.
rewrite homomorphism_p_iff in h2.
pose proof (homo_compose _ _ h1 h2) as h3.
rewrite homomorphism_p_iff.
rewrite ba_conv_fun_compose.
assumption.
Qed.

Lemma ba_conv_fun_inv_iff : 
  forall {Ap Bp:Bool_Alg_p T} 
         (f:btp Ap->btp Bp),
    invertible f <-> invertible (ba_conv_fun f).
intros Ap Bp f. split.
intro h1.
destruct h1 as [g h2 h3].
apply (intro_invertible (ba_conv_fun f) (ba_conv_fun g)); auto.
intro h1.
destruct h1 as [g h2 h3].
apply (intro_invertible f g); auto.
Qed.


Lemma ba_conv_fun1_inv_iff : 
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg}
         (f:btp Ap->bt B),
    invertible f <-> invertible (ba_conv_fun1 f).
intros Ap B f. split.
intro h1.
destruct h1 as [g h2 h3].
apply (intro_invertible (ba_conv_fun1 f) (ba_conv_fun2 g)); auto.
intro h1.
destruct h1 as [g h2 h3].
apply (intro_invertible f g); auto.
Qed.

Lemma ba_conv_fun2_inv_iff : 
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T}
         (f:bt A->btp Bp),
    invertible f <-> invertible (ba_conv_fun2 f).
intros A Bp f. split.
intro h1.
destruct h1 as [g h2 h3].
apply (intro_invertible (ba_conv_fun2 f) (ba_conv_fun1 g)); auto.
intro h1.
destruct h1 as [g h2 h3].
apply (intro_invertible f g); auto.
Qed.



Lemma ba_conv_fun_function_inverse : 
  forall {Ap Bp:Bool_Alg_p T} 
         (f:btp Ap->btp Bp)
         (pf:invertible f),
    ba_conv_fun (proj1_sig (function_inverse f pf)) =
    proj1_sig (function_inverse (ba_conv_fun f) (iff1 (ba_conv_fun_inv_iff f) pf)).
intros Ap Bp f h1.
unfold ba_conv_fun, function_inverse.
apply functional_extensionality.
intro x.
assert (h2: unique_inverse f h1 = 
(unique_inverse (fun x0 : Btype (Bc (ba_conv Ap)) => f x0) (iff1 (ba_conv_fun_inv_iff f) h1))).
  apply proof_irrelevance.
rewrite h2.
reflexivity.
Qed.

Lemma ba_conv_fun1_function_inverse : 
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg}
         (f:btp Ap->bt B)
         (pf:invertible f),
    ba_conv_fun2 (proj1_sig (function_inverse f pf)) =
    proj1_sig (function_inverse (ba_conv_fun1 f) (iff1 (ba_conv_fun1_inv_iff f) pf)).
intros Ap B f h1.
unfold ba_conv_fun1, ba_conv_fun2, function_inverse.
apply functional_extensionality.
intro x.
assert (h2: unique_inverse f h1 = 
(unique_inverse (fun x0 : Btype (Bc (ba_conv Ap)) => f x0) (iff1 (ba_conv_fun1_inv_iff f) h1))).
  apply proof_irrelevance.
rewrite h2.
reflexivity.
Qed.

Lemma ba_conv_fun2_function_inverse : 
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T}
         (f:bt A->btp Bp)
         (pf:invertible f),
    ba_conv_fun1 (proj1_sig (function_inverse f pf)) =
    proj1_sig (function_inverse (ba_conv_fun2 f) (iff1 (ba_conv_fun2_inv_iff f) pf)).
intros A Bp f h1.
unfold ba_conv_fun1, ba_conv_fun2, function_inverse.
apply functional_extensionality.
intro x.
assert (h2: unique_inverse f h1 = 
(unique_inverse (fun x0 : bt A => f x0) (iff1 (ba_conv_fun2_inv_iff f) h1))).
  apply proof_irrelevance.
rewrite h2.
reflexivity.
Qed.



Lemma homo_inv_p :
  forall {Ap Bp:Bool_Alg_p T} 
         (f:btp Ap->btp Bp)
         (pf:invertible f),
    homomorphism_p f ->
    homomorphism_p (proj1_sig (function_inverse _ pf)).
intros Ap Bp f h1 h2.
rewrite homomorphism_p_iff in h2.
pose proof (homo_inv (ba_conv_fun f) (iff1 (ba_conv_fun_inv_iff f) h1) h2) as h3.
rewrite homomorphism_p_iff.
rewrite ba_conv_fun_function_inverse.
assumption.
Qed.


Lemma homo_inv_p1 :
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg}
         (f:btp Ap->bt B)
         (pf:invertible f),
    homomorphism_p1 f ->
    homomorphism_p2 (proj1_sig (function_inverse _ pf)).
intros Ap B f h1 h2.
rewrite homomorphism_p1_iff in h2.
pose proof (homo_inv (ba_conv_fun1 f) (iff1 (ba_conv_fun1_inv_iff f) h1) h2) as h3.
rewrite homomorphism_p2_iff.
rewrite ba_conv_fun1_function_inverse.
assumption.
Qed.


Lemma homo_inv_p2 :
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T}
         (f:bt A->btp Bp)
         (pf:invertible f),
    homomorphism_p2 f ->
    homomorphism_p1 (proj1_sig (function_inverse _ pf)).
intros A Bp f h1 h2.
rewrite homomorphism_p2_iff in h2.
pose proof (homo_inv (ba_conv_fun2 f) (iff1 (ba_conv_fun2_inv_iff f) h1) h2) as h3.
rewrite homomorphism_p1_iff.
rewrite ba_conv_fun2_function_inverse.
assumption.
Qed.



Lemma homo_mono_p :
  forall {Ap Bp:Bool_Alg_p T} (f:btp Ap->btp Bp),
    homomorphism_p f -> forall x y, le_p x y -> le_p (f x) (f y).
intros Ap Bp f h1 x y h2.
rewrite homomorphism_p_iff in h1.
rewrite le_p_iff in h2. rewrite le_p_iff.
apply (homo_mono _ h1); auto.
Qed.

Lemma homo_mono_p1 :
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg} 
         (f:btp Ap->bt B),
    homomorphism_p1 f -> forall x y, le_p x y -> le (f x) (f y).
intros Ap B f h1 x y h2.
rewrite homomorphism_p1_iff in h1.
rewrite le_p_iff in h2.
apply (homo_mono _ h1); auto.
Qed.


Lemma homo_mono_p2 :
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T} 
         (f:bt A->btp Bp),
    homomorphism_p2 f -> forall x y, le x y -> le_p (f x) (f y).
intros A Bp f h1 x y h2.
rewrite homomorphism_p2_iff in h1.
rewrite le_p_iff.
apply (homo_mono _ h1); auto.
Qed.



Lemma im_ba_conv_fun_eq : 
  forall {Ap Bp:Bool_Alg_p T} 
         (f:btp Ap->btp Bp)
         (pf:Included
            (Im (Im (BS_p T (Bc_p T Ap)) f)
               (proj1_sig (P:=fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)))
            (A_p T (Bc_p T Bp))),
    Im (BS (Bc (ba_conv Ap))) (ba_conv_fun f) = 
    (ba_conv_und_subalg T Bp
                        (Im (Im (BS_p T (Bc_p T Ap)) f)
                            (proj1_sig (P:=fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x))) pf).
intros Ap Bp f h1 .
apply Extensionality_Ensembles.
red. split.
red. intros x h2.
destruct h2 as [x h2]. subst.
unfold ba_conv_und_subalg. unfold ba_conv_set. unfold transfer_dep. unfold eq_rect_r.
simpl.
pose proof (Im_intro _ _ _ f _ h2 _ (eq_refl _)) as h3.
pose proof (Im_intro _ _ _ (@proj1_sig _ _) _ h3 _ (eq_refl _)) as h4.
pose (exist _ _ h4) as x'.
apply Im_intro with x'.
constructor. unfold x'. simpl. unfold ba_conv_fun.
apply proj1_sig_injective. simpl. reflexivity.
red. intros x h2.
destruct h2 as [x h2]. subst.
destruct x as [x h3]. simpl. 
inversion h3 as [y h4].  subst.
destruct h4 as [y h4]. subst. 
apply Im_intro with y. assumption.
apply proj1_sig_injective. simpl.
reflexivity.
Qed.


Lemma im_ba_conv_fun1_eq : 
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg}
         (f:btp Ap->bt B),
    Im (BS (Bc (ba_conv Ap))) (ba_conv_fun1 f) = 
    (Im (BS_p T (Bc_p T Ap)) f).
intros Ap B f.
apply Extensionality_Ensembles.
red. split.
red. intros x h2.
destruct h2 as [x h2]. subst.  
destruct x as [x h0]. 
apply Im_intro with (exist _ _ h0).
apply in_bs_p.
unfold ba_conv_fun1. f_equal.
red. intros x h1.
destruct h1 as [x h1]. subst.
apply Im_intro with x. 
apply in_bs.
unfold ba_conv_fun1.
reflexivity.
Qed.


Lemma im_ba_conv_fun2_eq : 
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T} (f:bt A-> btp Bp)
         (pf:Included (Im (Im (ba_ens A) f) (@proj1_sig _ _))
                       (A_p T (Bc_p T Bp))),
    Im (BS (Bc A)) (ba_conv_fun2 f) = 
    ba_conv_und_subalg T Bp
        (Im (Im (ba_ens A) f)
           (proj1_sig (P:=fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x))) pf.
intros A Bp f h1.
apply Extensionality_Ensembles.
red. split.
red. intros x h2.
destruct h2 as [x h2]. subst.
unfold ba_conv_und_subalg. unfold ba_conv_set. unfold transfer_dep. unfold eq_rect_r.
simpl.
rewrite und_set in h2.
pose proof (Im_intro _ _ _ f _ h2 _ (eq_refl _)) as h3. 
pose proof (Im_intro _ _ _ (@proj1_sig _ _) _ h3 _ (eq_refl _)) as h4.
pose (exist _ _ h4) as x'.
apply Im_intro with x'.
constructor. unfold x'. simpl. unfold ba_conv_fun.
apply proj1_sig_injective. simpl. reflexivity.
red. intros x h2.
destruct h2 as [x h2]. subst.
destruct x as [x h3]. simpl. 
inversion h3 as [y h4].  subst.
destruct h4 as [y h4]. subst. 
apply Im_intro with y. unfold ba_ens, bt in h4. 
rewrite <- (und_set A) in h4. assumption.
apply proj1_sig_injective. simpl.
reflexivity.
Qed.



Lemma homo_im_closed_p :
  forall {Ap Bp:Bool_Alg_p T} 
         (f:btp Ap->btp Bp),
    homomorphism_p f ->
    exists pf:Included (Im (Im (BS_p T (Bc_p T Ap)) f) (@proj1_sig _ _))
                       (A_p T (Bc_p T Bp)),
      alg_closed_p (Im (Im (BS_p T (Bc_p T Ap)) f) (@proj1_sig _ _)) pf.
intros Ap Bp f h1.
rewrite homomorphism_p_iff in h1.
assert (h2:Included
            (Im (Im (BS_p T (Bc_p T Ap)) f)
               (proj1_sig (P:=fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)))
            (A_p T (Bc_p T Bp))).
  rewrite im_im.
  red. intros x h2.
  destruct h2 as [x h2]. subst.
  apply proj2_sig.
exists h2.
rewrite alg_closed_p_iff.
pose proof (homo_im_closed _ h1) as h3. 
rewrite (im_ba_conv_fun_eq _ h2) in h3 at 1.
assumption.
Qed.


Lemma homo_im_closed_p1 :
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg} 
         (f:btp Ap->bt B),
    homomorphism_p1 f ->
      alg_closed (Im (BS_p T (Bc_p T Ap)) f).
intros Ap B f h1.
rewrite homomorphism_p1_iff in h1.
pose proof (homo_im_closed _ h1) as h3.
pose proof (im_ba_conv_fun1_eq f) as h4.
unfold bt in h3, h4. rewrite h4 in h3.
assumption.
Qed.


Lemma homo_im_closed_p2 :
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T} 
         (f:bt A->btp Bp),
    homomorphism_p2 f ->
    exists pf:Included (Im (Im (ba_ens A) f) (@proj1_sig _ _))
                       (A_p T (Bc_p T Bp)),
      alg_closed_p (Im (Im (ba_ens A) f) (@proj1_sig _ _)) pf.
intros A Bp f h1.
rewrite homomorphism_p2_iff in h1.
assert (h2:Included
             (Im (Im (ba_ens A) f)
                 (proj1_sig (P:=fun x : T => Ensembles.In (A_p T (Bc_p T Bp)) x)))
             (A_p T (Bc_p T Bp))).
  rewrite im_im.
  red. intros x h2. destruct h2 as [x h2]. subst. apply proj2_sig.
exists h2.
rewrite alg_closed_p_iff.
pose proof (homo_im_closed _ h1) as h3.
pose proof (im_ba_conv_fun2_eq f h2) as h4. unfold bt in h3, h4.
rewrite h4 in h3.
assumption.
Qed.



Lemma homo_subalg_im_p_ex : 
  forall {Ap Bp:Bool_Alg_p T} 
         (f:btp Ap->btp Bp),
    homomorphism_p f ->
    exists! (Cp:Bool_Alg_p T),
      exists (pfincl:Included (Im (Im (BS_p T (Bc_p T Ap)) f) (@proj1_sig _ _))
                          (A_p T (Bc_p T Bp)))
             (pfcl:alg_closed_p (Im (Im (BS_p T (Bc_p T Ap)) f) (@proj1_sig _ _)) pfincl),
        Cp = Subalg_p _ _ pfincl pfcl.
intros Ap Bp f h1.
pose proof (homo_im_closed_p _ h1) as h2.
destruct h2 as [h2 h3].
exists (Subalg_p _ _ h2 h3).
red.  split.
exists h2. exists h3.
reflexivity.
intros Cp h4.
destruct h4 as [h4 [h5 h6]]. subst.
assert (h4 = h2). apply proof_irrelevance. subst.
assert (h3 = h5). apply proof_irrelevance. subst.
reflexivity.
Qed.

Definition homo_subalg_im_p  
           {Ap Bp:Bool_Alg_p T} 
           (f:btp Ap->btp Bp)
           (pf:homomorphism_p f) : Bool_Alg_p T :=
  proj1_sig (constructive_definite_description _ (homo_subalg_im_p_ex f pf)).


Lemma homo_subalg_im_p_compat : 
  forall {Ap Bp:Bool_Alg_p T} 
         (f:btp Ap->btp Bp)
         (pf:homomorphism_p f),
         let Cp := homo_subalg_im_p _ pf in 
         exists (pfincl:Included (Im (Im (BS_p T (Bc_p T Ap)) f) (@proj1_sig _ _))
                          (A_p T (Bc_p T Bp)))
             (pfcl:alg_closed_p (Im (Im (BS_p T (Bc_p T Ap)) f) (@proj1_sig _ _)) pfincl),
        Cp = Subalg_p _ _ pfincl pfcl.
intros Ap f h1 h2. simpl.
unfold homo_subalg_im_p.
destruct constructive_definite_description as [Cp h3].
simpl.
assumption.
Qed.

Lemma homo_subalg_im_p1_ex : 
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg}
         (f:btp Ap->bt B),
    homomorphism_p1 f ->
    exists! (C:Bool_Alg),
      exists (pfcl:alg_closed (Im (BS_p T (Bc_p T Ap)) f)),
        C = Subalg _ pfcl.
intros Ap B f h1.
pose proof (homo_im_closed_p1 _ h1) as h2. 
exists (Subalg _  h2).
red.  split.
exists h2.  reflexivity.
intros Cp h4.
destruct h4 as [h4 h5]. subst.
assert (h4 = h2). apply proof_irrelevance. subst.
reflexivity.
Qed.

Definition homo_subalg_im_p1  
           {Ap:Bool_Alg_p T} {B:Bool_Alg}
           (f:btp Ap->bt B)
           (pf:homomorphism_p1 f) : Bool_Alg :=
  proj1_sig (constructive_definite_description _ (homo_subalg_im_p1_ex f pf)).


Lemma homo_subalg_im_p1_compat : 
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg}
         (f:btp Ap->bt B) (pf:homomorphism_p1 f),
    let C:= homo_subalg_im_p1 f pf in 
      exists (pfcl:alg_closed (Im (BS_p T (Bc_p T Ap)) f)),
        C = Subalg _ pfcl.
intros Ap B f h1 C. unfold C. unfold homo_subalg_im_p1.
destruct constructive_definite_description as [C' h2].
simpl. assumption.
Qed.


Lemma homo_subalg_im_p2_ex : 
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T} 
         (f:bt A->btp Bp),
    homomorphism_p2 f ->
    exists! (Cp:Bool_Alg_p T),
      exists (pfincl:Included (Im (Im (ba_ens A) f) (@proj1_sig _ _))
                          (A_p T (Bc_p T Bp)))
             (pfcl:alg_closed_p (Im (Im (ba_ens A) f) (@proj1_sig _ _)) pfincl),
        Cp = Subalg_p _ _ pfincl pfcl.
intros A Bp f h1.
pose proof (homo_im_closed_p2 _ h1) as h2.
destruct h2 as [h2 h3].
exists (Subalg_p _ _ h2 h3).
red. split. exists h2. exists h3. reflexivity.
intros x h4. destruct h4 as [h4 [h5 h6]]. subst.
assert (h2 = h4). apply proof_irrelevance. subst. 
assert (h3 = h5). apply proof_irrelevance. subst.
reflexivity.
Qed.

Definition homo_subalg_im_p2 
           {A:Bool_Alg} {Bp:Bool_Alg_p T} (f:bt A->btp Bp)
           (pf:homomorphism_p2 f) :=
  proj1_sig (constructive_definite_description _ (homo_subalg_im_p2_ex f pf)).


Lemma homo_subalg_im_p2_compat : 
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T} 
         (f:bt A->btp Bp) (pf:homomorphism_p2 f),
    let Cp := homo_subalg_im_p2 f pf in 
      exists (pfincl:Included (Im (Im (ba_ens A) f) (@proj1_sig _ _))
                          (A_p T (Bc_p T Bp)))
             (pfcl:alg_closed_p (Im (Im (ba_ens A) f) (@proj1_sig _ _)) pfincl),
        Cp = Subalg_p _ _ pfincl pfcl.
intros A Bp f h1 Cp.
unfold Cp, homo_subalg_im_p2.
destruct constructive_definite_description as [C' h2]. simpl.
assumption.
Qed.



Lemma transfer_homo_subalg_im_p_ex : 
  forall {Ap Bp:Bool_Alg_p T} 
         (f:btp  Ap->btp Bp)
         (pf:homomorphism_p f) (x:btp Ap),
  exists! y:btp (homo_subalg_im_p f pf),
    proj1_sig y = proj1_sig (f x).
intros Ap Bp f h1 x.
pose proof (homo_subalg_im_p_compat f h1) as h2.
simpl in h2. destruct h2 as [h2 [h3 h4]].
assert (h5:Ensembles.In (ba_p_ens (homo_subalg_im_p f h1))
              (proj1_sig (f x))).
unfold ba_p_ens. rewrite h4. unfold Subalg_p. simpl.
  apply Im_intro with (f x); auto. apply Im_intro with x; auto.
  rewrite und_set_p. constructor.
exists (exist _ _ h5).
red. split. simpl. reflexivity.
intros y h6.
apply proj1_sig_injective. simpl.
rewrite h6 at 1.
reflexivity.
Qed.


Definition transfer_homo_subalg_im_p 
           {Ap Bp:Bool_Alg_p T} 
           (f:btp Ap->btp Bp)
           (pf:homomorphism_p f) (x:btp Ap) :
  btp (homo_subalg_im_p f pf) :=
  proj1_sig (constructive_definite_description _ (transfer_homo_subalg_im_p_ex _ pf x)).

Lemma transfer_homo_subalg_im_p_compat :
  forall {Ap Bp:Bool_Alg_p T} 
         (f:btp Ap->btp Bp)
         (pf:homomorphism_p f) (x:btp Ap),
    let y := transfer_homo_subalg_im_p f pf x in
    proj1_sig y = proj1_sig (f x).
intros Ap Bp f h1 x y. 
unfold transfer_homo_subalg_im_p in y.
destruct constructive_definite_description as [z h2].
simpl in y. unfold y.
assumption.
Qed.


Lemma transfer_homo_subalg_im_p1_ex :
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg}
         (f:btp Ap->bt B)
         (pf:homomorphism_p1 f) (x:btp Ap),
    let C:=homo_subalg_im_p1 f pf in
  exists! y:bt C,
    forall pfcl: alg_closed (Im (BS_p T (Bc_p T Ap)) f),
      exists (pfeq:C = Subalg _ pfcl),
        proj1_sig (transfer_ba_elt pfeq y) = f x.
intros Ap B f h1 x C.
pose proof (homo_subalg_im_p1_compat f h1) as h2. simpl in h2.
destruct h2 as [h2 h3].
assert (h4:Ensembles.In (Im (BS_p T (Bc_p T Ap)) f) (f x)).
  apply Im_intro with x. apply in_bs_p. reflexivity.
exists (transfer_ba_elt_r h3 (exist _ _ h4)).
red. split.
intros h5. assert (h5 = h2). apply proof_irrelevance. subst.
exists h3.
rewrite transfer_ba_elt_undoes_transfer_ba_elt_r. simpl.
reflexivity.
intros x' h5. specialize (h5 h2).
destruct h5 as [h5 h6]. subst.
apply (transfer_ba_elt_inj h3).
rewrite transfer_ba_elt_undoes_transfer_ba_elt_r.
apply proj1_sig_injective.
simpl. assert (h3 = h5). apply proof_irrelevance. subst.
rewrite h6 at 1. reflexivity.
Qed.

Definition transfer_homo_subalg_im_p1
           {Ap:Bool_Alg_p T} {B:Bool_Alg}
           (f:btp Ap->bt B)
           (pf:homomorphism_p1 f) (x:btp Ap) :=
  proj1_sig (constructive_definite_description _
               (transfer_homo_subalg_im_p1_ex f pf x)).


Lemma transfer_homo_subalg_im_p1_compat :
  forall {Ap:Bool_Alg_p T} {B:Bool_Alg}
         (f:btp Ap->bt B)
         (pf:homomorphism_p1 f) (x:btp Ap),
    let C := homo_subalg_im_p1 f pf in
    let y := transfer_homo_subalg_im_p1 f pf x in
    forall pfcl: alg_closed (Im (BS_p T (Bc_p T Ap)) f),
      exists (pfeq:C = Subalg _ pfcl),
        proj1_sig (transfer_ba_elt pfeq y) = f x.
intros Ap B f h1 x C y.
unfold y, C, transfer_homo_subalg_im_p1.
destruct constructive_definite_description as [y' h2]. simpl.
assumption.
Qed.


Lemma transfer_homo_subalg_im_p2_ex : 
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T} 
         (f:bt A->btp Bp)
         (pf:homomorphism_p2 f) (x:bt A),
  exists! y:btp (homo_subalg_im_p2 f pf),
    proj1_sig y = proj1_sig (f x).
intros Ap Bp f h1 x.
pose proof (homo_subalg_im_p2_compat f h1) as h2.
simpl in h2. destruct h2 as [h2 [h3 h4]].
assert (h5:Ensembles.In (ba_p_ens (homo_subalg_im_p2 f h1))
              (proj1_sig (f x))).
unfold ba_p_ens. rewrite h4. unfold Subalg_p. simpl.
  apply Im_intro with (f x); auto. apply Im_intro with x; auto.
  constructor.
exists (exist _ _ h5).
red. split. simpl. reflexivity.
intros y h6.
apply proj1_sig_injective. simpl.
rewrite h6 at 1.
reflexivity.
Qed.

Definition transfer_homo_subalg_im_p2 
           {A:Bool_Alg} {Bp:Bool_Alg_p T} 
           (f:bt A->btp Bp)
           (pf:homomorphism_p2 f) (x:bt A) :=
  proj1_sig (constructive_definite_description _ (transfer_homo_subalg_im_p2_ex f pf x)).


Lemma transfer_homo_subalg_im_p2_compat : 
  forall {A:Bool_Alg} {Bp:Bool_Alg_p T} 
         (f:bt A->btp Bp)
         (pf:homomorphism_p2 f) (x:bt A),
    let y:= transfer_homo_subalg_im_p2 f pf x in 
    proj1_sig y = proj1_sig (f x).
intros A Bp f h1 x y. unfold y, transfer_homo_subalg_im_p2.
destruct constructive_definite_description as [y' h2].
simpl.
assumption.
Qed.


Definition homo_onto_p {Ap Bp:Bool_Alg_p T}
           (f:btp Ap->btp Bp)
           (pf:homomorphism_p f) :
  btp Ap->btp (homo_subalg_im_p _ pf) :=
  (fun x => (transfer_homo_subalg_im_p f pf x)).

Definition homo_onto_p1 {Ap:Bool_Alg_p T} {B:Bool_Alg}
           (f:btp Ap->bt B)
           (pf:homomorphism_p1 f) :
  btp Ap->bt (homo_subalg_im_p1 _ pf) :=
  (fun x => (transfer_homo_subalg_im_p1 f pf x)).

Definition homo_onto_p2 {A:Bool_Alg} {Bp:Bool_Alg_p T}
           (f:bt A->btp Bp)
           (pf:homomorphism_p2 f) :
  bt A->btp (homo_subalg_im_p2 _ pf) :=
  (fun x => (transfer_homo_subalg_im_p2 f pf x)).


Lemma homomorphism_p1_restriction_sig : 
  forall {Bp:Bool_Alg_p T} (A:Bool_Alg)
         (f:(btp Bp)->(bt A)),
    homomorphism_p1 f ->
    forall (Cp:Bool_Alg_p T)
           (pf:Included (ba_p_ens Cp) (ba_p_ens Bp)),
      subalg_of_p Cp Bp ->
      homomorphism_p1 (restriction_sig f _ pf).
intros Bp A f h1 Cp h2 h3. 
red in h3.
destruct h3 as [h3a [h3b h3c]].
destruct h1 as [h1a h1b h1c]. 
apply homo_two_ops_plus_p1.

intros x y. destruct x as [x h4], y as [y h5].
unfold restriction_sig.
simpl.
assert (h8:Ensembles.In (ba_p_ens (Subalg_p Bp (ba_p_ens Cp) h3a h3b)) x).
  rewrite <- h3c. assumption.
assert (h9: Ensembles.In (ba_p_ens (Subalg_p Bp (ba_p_ens Cp) h3a h3b)) y).
  rewrite <- h3c. assumption.
pose proof (ba_p_subst_plus _ _ h3c _ _ h4 h5 h8  h9) as h6.
rewrite <- h1b.
f_equal.
apply proj1_sig_injective.
simpl.
rewrite h6 at 1.  unfold Subalg_p. simpl.
f_equal. f_equal. apply proj1_sig_injective; auto. apply proj1_sig_injective; auto.

intros x. destruct x as [x h4].
unfold restriction_sig.
simpl.
assert (h8:Ensembles.In (ba_p_ens (Subalg_p Bp (ba_p_ens Cp) h3a h3b)) x).
  rewrite <- h3c. assumption.
pose proof (ba_p_subst_comp _ _ h3c _  h4 h8) as h6.
rewrite <- h1c.
f_equal.
apply proj1_sig_injective. simpl.
rewrite h6 at 1.  unfold Subalg_p. simpl.
f_equal. f_equal. apply proj1_sig_injective; auto.
Qed.


Lemma ba_p_subst_homomorphism_p1 : 
  forall (Ap Bp:Bool_Alg_p T) (C:Bool_Alg)
         (pf:Ap = Bp) (pf':btp Ap = btp Bp) (f:btp Ap->bt C),
    homomorphism_p1 f <-> homomorphism_p1 (transfer_fun pf' f).
intros Ap Bp C h0 h1 f.
subst.
assert (h2:h1 = eq_refl). apply proof_irrelevance.
subst.
rewrite transfer_fun_eq_refl.
tauto.
Qed.
  
(*Still have a lot of compatibility lemmas to connect lemmas
  and definition to their parametric analogues!*)



End ParametricAnalogues.

Arguments homomorphism_p [T] [Ap] [Bp] _.
Arguments homomorphism_p_iff [T] [Ap] [Bp] _.
Arguments homomorphism_general_p [T] [Ap] [A] _ _ _ _.
Arguments homomorphism_general_p_iff [T] [Ap] [A] _ _ _ _.
Arguments homo_sym_diff_p [T] [Ap] [Bp] _ _ _ _.
Arguments homo_id_p [T] _.
Arguments universal_homomorphic_ops_imply_bool_p [T] [Ap] [A] _ _ _ _ _ _.
Arguments homo_zero_p [T] [Ap] [Bp] _ _. 
Arguments homo_one_p [T] [Ap] [Bp] _ _.
Arguments homo_two_ops_plus_p [T] [Ap] [Bp] _ _ _.
Arguments homo_two_ops_times_p [T] [Ap] [Bp] _ _ _.
Arguments ba_conv_fun_compose [T] [Ap] [Bp] _ _ _.
Arguments homo_compose_p [T] [Ap] [Bp] _ _ _ _ _.
Arguments ba_conv_fun_inv_iff [T] [Ap] [Bp] _.
Arguments ba_conv_fun_function_inverse [T] [Ap] [Bp] _ _.
Arguments homo_inv_p [T] [Ap] [Bp] _ _ _.
Arguments homo_mono_p [T] [Ap] [Bp] _ _ _ _ _.
Arguments im_ba_conv_fun_eq [T] [Ap] [Bp] _ _.
Arguments homo_im_closed_p [T] [Ap] [Bp] _ _.
Arguments homo_subalg_im_p_ex [T] [Ap] [Bp] _ _.
Arguments homo_subalg_im_p [T] [Ap] [Bp] _ _.
Arguments homo_subalg_im_p_compat [T] [Ap] [Bp] _ _.
Arguments transfer_homo_subalg_im_p_ex [T] [Ap] [Bp] _ _ _.
Arguments transfer_homo_subalg_im_p [T] [Ap] [Bp] _ _ _.
Arguments transfer_homo_subalg_im_p_compat [T] [Ap] [Bp] _ _ _.
Arguments homo_onto_p [T] [Ap] [Bp] _ _ _.



Section ParametricAnalogues'.

Lemma homomorphism_ba_to_ba_p_iff :
  forall (A B:Bool_Alg) (f:btp (ba_to_ba_p B) -> bt A),
    homomorphism_p1 _ f <->
    homomorphism (fun x:bt B => f (exist _ _ (Full_intro _ x))).
intros A B f.

assert (h2: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl). apply proof_irrelevance.
split. 
intro h1.
destruct h1 as [h1a h1b h1c].
apply homo_two_ops_times.
intros x y.
specialize (h1a (exist _ _ (Full_intro _ x)) (exist _ _ (Full_intro _ y))).
rewrite <- h1a at 1.
simpl.

rewrite h2 at 1.
rewrite transfer_dep_eq_refl.
unfold Btimes_sub. simpl.
f_equal.
apply proj1_sig_injective.
simpl. reflexivity.

intro x.
specialize (h1c (exist _ _ (Full_intro _ x))).
rewrite <- h1c at 1.
simpl.
  
rewrite h2 at 1.
rewrite transfer_dep_eq_refl.
unfold Bcomp_sub. simpl.
f_equal.
apply proj1_sig_injective.
simpl. reflexivity.

intro h1.
destruct h1 as [h1a h1b h1c].
apply homo_two_ops_times_p1.

intros x y.
destruct x as [x h3], y as [y h4].
specialize (h1a x y).
simpl. 
assert (h6:h3 = Full_intro _ _). apply proof_irrelevance.
assert (h5:h4 = Full_intro _ _). apply proof_irrelevance.
subst.
rewrite <- h1a at 1.
f_equal.
rewrite h2 at 1.
rewrite transfer_dep_eq_refl.
unfold Btimes_sub. simpl.
apply proj1_sig_injective; auto.


intro x.
destruct x as [x h3].
specialize (h1c x).
simpl. 
assert (h6:h3 = Full_intro _ _). apply proof_irrelevance.
subst.
rewrite <- h1c at 1.
f_equal.
rewrite h2 at 1.
rewrite transfer_dep_eq_refl.
unfold Btimes_sub. simpl.
apply proj1_sig_injective; auto.
Qed.


Lemma homomorphism_p1_restriction_fun : 
  forall {A B:Bool_Alg} (f:bt B->bt A),
          homomorphism f ->
          forall  (Cp:Bool_Alg_p (bt B)),
            subalg_of_p Cp (ba_to_ba_p B) ->
            homomorphism_p1 (Ap:=Cp) _ (restriction_fun f (ba_p_ens Cp)).
intros A B f h1 Cp h2.
destruct h1 as [h1a h1b h1c].
destruct h2 as [h2a [h2b h2c]].
apply homo_two_ops_times_p1.
intros x y.
destruct x as [x h3], y as [y h4].
specialize (h1a x y).
unfold restriction_fun. simpl.
rewrite <- h1a at 1.
f_equal.

assert (h6:Ensembles.In
                   (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Cp) h2a h2b))
                   x).
  rewrite <- h2c. assumption.
assert (h7 : Ensembles.In
                   (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Cp) h2a h2b))
                   y).
  rewrite <- h2c. assumption.
pose proof (ba_p_subst_times _ _ h2c _ _ h3 h4 h6 h7) as h5.
rewrite h5 at 1.
simpl.
assert (h8: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl). apply proof_irrelevance.

rewrite h8 at 1. clear h8.
rewrite transfer_dep_eq_refl.
simpl.
unfold Btimes_sub. simpl.
reflexivity.

intros x.
destruct x as [x h3].
specialize (h1c x).
unfold restriction_fun. simpl.
rewrite <- h1c at 1.
f_equal.

assert (h6:Ensembles.In
                   (ba_p_ens (Subalg_p (ba_to_ba_p B) (ba_p_ens Cp) h2a h2b))
                   x).
  rewrite <- h2c. assumption.
pose proof (ba_p_subst_comp _ _ h2c _  h3 h6) as h5.
rewrite h5 at 1.
simpl.
assert (h8: Morphisms.trans_co_eq_inv_impl_morphism eq_Transitive
           (SubBtype B (Full_set (bt B))) (SubBtype B (Full_set (bt B)))
           eq_refl
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (sig_set
              (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a))
           (Morphisms.eq_proper_proxy Type
              (sig_set
                 (fun a : Btype (Bc B) => Ensembles.In (Full_set (bt B)) a)))
           eq_refl = eq_refl). apply proof_irrelevance.
rewrite h8 at 1. clear h8.
rewrite transfer_dep_eq_refl.
simpl.
unfold Bcomp_sub. simpl.
reflexivity.
Qed.


End ParametricAnalogues'.