(* Copyright (C) 2014-2015 Daniel Wyckoff *)
(*This file is part of BooleanAlgebrasIntro2.

BooleanAlgebrasIntro2 is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

BooleanAlgebrasIntro2 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with BooleanAlgebrasIntro2.  If not, see <http://www.gnu.org/licenses/>.*)

Require Import SetUtilities.
Require Import FunctionalExtensionality.
Require Import ProofIrrelevance.
Require Import Setoid.
Require Import Description.
Require Import DecidableDec.
Require Import ProofIrrelevance.
Require Import FunctionProperties.
Require Import TypeUtilities.
Require Import LogicUtilities.
Require Import Equality.
 
Inductive functionally_paired {T U:Type} (A:Ensemble T)
          (B:Ensemble U) (S:Ensemble (T*U)) : Prop :=
  functional_pairs_intro : 
    (forall (x:T), 
      In A x -> (exists! y:U, 
                  In B y /\ In S (x, y))) ->
    (forall (pr:T*U), In S pr -> In A (fst pr) /\
    In B (snd pr)) ->
                  functionally_paired A B S.



(* In theory the above is less elegant than the below, but 
  in practice it's not as easy to work with.*)


(*
Inductive functionally_paired {T U:Type} (A:Ensemble T)
          (B:Ensemble U) (S:Ensemble (T*U)) : Prop :=
  functional_pairs_intro :
    (forall pr:T*U, 
       In S pr <-> In A (fst pr) /\
                   In B (snd pr) /\
                   unique (fun pr':T*U => fst pr = fst pr')
                          pr) -> functionally_paired A B S.
*)  



Lemma fp_in_dom : 
  forall {T U:Type} (A:Ensemble T) 
         (B:Ensemble U) (S:Ensemble (T*U)),
    functionally_paired A B S -> 
    forall pr:T*U, In S pr -> In A (fst pr).
  intros T U A B S h1 pr h2.
  destruct h1 as [? h1].
  apply h1; assumption.
Qed.

Lemma fp_in_ran :
  forall {T U:Type} (A:Ensemble T) 
         (B:Ensemble U) (S:Ensemble (T*U)),
    functionally_paired A B S -> 
    forall pr:T*U, In S pr -> In B (snd pr).
  intros T U A B S h1 pr h2.
  destruct h1 as [? h1].
  apply h1; assumption.
Qed.

Lemma fp_functional :
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {S:Ensemble (T*U)},
    functionally_paired A B S ->
    forall (x:T) (y1 y2:U), In S (x, y1) ->
                            In S (x, y2) ->
                            y1 = y2.
intros T U A B S h1 x y1 y2 h3 h4.
inversion h1 as [h5 h6].
pose proof (fp_in_dom _ _ _ h1 _ h3) as h7.
simpl in h7.
pose proof (h5 x h7) as h8.
destruct h8 as [y h9].
red in h9.
destruct h9 as [h9a h9b].
pose proof (fp_in_ran _ _ _ h1 _ h3) as h10.
pose proof (fp_in_ran _ _ _ h1 _ h4) as h11.
simpl in h10, h11.
pose proof (h9b y1) as h12.
pose proof (h9b y2) as h13.
cut (y = y1).
cut (y = y2).
congruence. tauto. tauto.
Qed.
 
(*This constructs an explicit function from the inductive definition
  of [functionally_paired] *)

Definition fps_to_f  {T U:Type} 
           {A:Ensemble T} {B:Ensemble U} (S:Ensemble (T*U))
           (pf:functionally_paired A B S) (def:U) : T->U.
  intro x.
  destruct (classic_dec (In A x)) as [h1 | h2].
  destruct pf as [h3].
  refine (proj1_sig (constructive_definite_description 
                       _ (h3 _ h1))).
  refine def.
Defined.

Lemma fps_to_f_in_ran : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {S:Ensemble (T*U)}  
         (pf:functionally_paired A B S) (def:U) (x:T),
    In A x ->
    In B (fps_to_f _ pf def x).
intros T U A B S h1 def x h2.
unfold fps_to_f. destruct (classic_dec (In A x)) as [h3 | h4].
destruct h1 as [h1l h1r].
destruct constructive_definite_description as [y h4].
simpl. destruct h4; assumption.
contradiction.
Qed.

Lemma fps_to_f_def : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {S:Ensemble (T*U)}  
         (pf:functionally_paired A B S) (def:U) (x:T),
    ~ In A x ->
    fps_to_f _ pf def x = def.
intros T U A B S h1 def x h2.
unfold fps_to_f.
destruct (classic_dec (In A x)) as [h3 | h4].
contradiction.
reflexivity.
Qed.

    
Definition self_fp {T U:Type} (S:Ensemble (T*U)) :=
  functionally_paired (dom_rel S) (ran_rel S) S.

Lemma self_fp_empty : forall (T U:Type),
                        self_fp (Empty_set (T*U)).
intros T U. red.
rewrite dom_rel_empty. rewrite ran_rel_empty.
constructor; intros; contradiction.
Qed.

        
Lemma self_fp_im: forall {T U:Type} (A:Ensemble T) (f:T->U),
                         self_fp (Im A (fun x=>(x, f x))).
intros T U A f.
red. constructor.
intros x h1. 
rewrite dom_rel_eq in h1.
rewrite im_im in h1.
simpl in h1.
destruct h1. subst.
exists (f x). red.
split. split.
constructor. exists x. apply Im_intro with x. assumption.
reflexivity.
apply Im_intro with x. assumption. reflexivity. 
intros y h1. 
destruct h1 as [h1l h1r]. 
rewrite ran_rel_eq in h1l. 
rewrite im_im in h1l. simpl in h1l. destruct h1l. subst.
inversion h1r. inversion H2. reflexivity.
intros pr h1. split.
rewrite dom_rel_eq. rewrite im_im. simpl.
destruct h1. subst.
simpl. apply Im_intro with x; auto.
rewrite ran_rel_eq. rewrite im_im. simpl.
destruct h1. subst. simpl. apply Im_intro with x; auto.
Qed.

Lemma self_fp_incl : 
  forall {T U:Type} (R S:Ensemble (T*U)),
    Included R S -> self_fp S -> self_fp R.
intros T U R S h1 h2. 
constructor. 
intros x h3. 
destruct h3 as [h3].
destruct h3 as [y h3].
exists y. red. split. split. constructor. exists x. assumption.
assumption.
intros y' h4.
inversion h2 as [h2l h2r].
assert (h5:In S (x, y)). auto with sets.
specialize (h2r _ h5).
simpl in h2r. destruct h2r as [h2a h2b].
pose proof (h2l _ h2a) as h2'.
destruct h2' as [y'' h2']. red in h2'. destruct h2' as [h6 h7].
specialize (h7 _ (conj h2b h5)). subst. 
destruct h4 as [h4l h4r]. destruct h6 as [h6l h6r].
assert (h7: In S (x, y')). auto with sets.
red in h2.
pose proof (fp_functional h2 _ _ _ h6r h7). assumption.
intros pr h3. split.
constructor. exists (snd pr). rewrite (surjective_pairing pr) in h3. assumption.
constructor. exists (fst pr). rewrite (surjective_pairing pr) in h3. assumption.
Qed.




    


Lemma fp_sub_add : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {S:Ensemble (T*U)}
         (x:T) (y y':U),
    y <> y' ->
    functionally_paired A B (Add S (x, y)) -> 
    Subtract (Add S (x, y)) (x, y') = Add S (x, y).
intros T U A B S  x y y' h0 h1. 
apply Extensionality_Ensembles. red. split.
apply incl_subtract. 
red. 
intros pr h2. 
constructor. assumption.
intro h3. inversion h3. subst. clear h3.
pose proof (Add_intro2 _ S (x, y)) as h3.
pose proof (fp_functional h1 _ _ _ h2 h3). subst.
contradict h0. reflexivity.
Qed.


Lemma fps_to_f_compat : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {S:Ensemble (T*U)} (pf:functionally_paired A B S)
         (def:U),
    let f := fps_to_f _ pf def in 
    S = [pr:T*U | snd pr = f (fst pr) /\
                  In A (fst pr)].
  intros T U A B S h1 def f.
  destruct h1 as [h1 h2].
  apply Extensionality_Ensembles.
  red. split.
    (* <= *)
    red.
    intros pr h4.
    constructor.
    split.
    pose proof (h2 pr h4) as h5.
    destruct h5 as [h5l h5r].
    unfold f.
    unfold fps_to_f.
    destruct (classic_dec (In A (fst pr))) as [h3l | h3r].
      (*h3l*)
      destruct constructive_definite_description as [u h6].
      simpl.
      destruct h6 as [h6l h6r].
      pose proof (h1 (fst pr) h5l) as h7.
      destruct h7 as [u' h7].
      red in h7.
      destruct h7 as [h7l h7r].
      pose proof (h7r u (conj h6l h6r)).
      subst.
      pose proof (surjective_pairing pr) as h8.
      rewrite h8 in h4.
      symmetry.
      apply h7r. split; assumption.
      (* h3r *)
      contradiction.
    pose proof (h2 pr h4) as h5.
    apply h5.
    (* >= *)
    red.
    intros pr h4.
    destruct h4 as [h4].
    destruct h4 as [h4l h4r].
    pose proof (h1 (fst pr) h4r) as h5.
    destruct h5 as [u h5].
    red in h5.
    destruct h5 as [h5l h5r].
    unfold f in h4l.
    unfold fps_to_f  in h4l.
    destruct (classic_dec (In A (fst pr))) as [h6 | h7].
    simpl in h4l.
    destruct constructive_definite_description as [u' h8].
    pose proof (h5r _ h8) as h9.
    subst.
    simpl in h4l.
    subst.
    destruct h8.
    rewrite surjective_pairing.
    assumption.
    contradiction.
Qed.

Lemma fp_dom_rel :
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} 
         {S:Ensemble (T*U)},
    functionally_paired A B S ->
    dom_rel S = A.
intros T U A B S h1.
apply Extensionality_Ensembles.
red. split.
red. intros x h2.
destruct h2 as [h2]. destruct h2 as [y h2].
destruct h1 as [h1l h1r].
specialize (h1r _ h2). simpl in h1r. destruct h1r; auto.
red. intros x h2.
constructor.
destruct h1 as [h1l h1r].
specialize (h1l _ h2).
destruct h1l as [y h1l].
red in h1l. destruct h1l as [[h1l ?] ?].
exists y; auto.
Qed.


Lemma fp_ran_rel_im : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} 
         {S:Ensemble (T*U)} (pf:functionally_paired A B S)
         (def:U),
    ran_rel S = Im (dom_rel S) (fps_to_f _ pf def).
intros T U A B S h1 def.
apply Extensionality_Ensembles.
red. split.
red. intros y h2.
destruct h2 as [h2].
destruct h2 as [x h2].
apply Im_intro with x.
constructor. exists y. assumption.
rewrite (fps_to_f_compat h1 def) in h2.
destruct h2 as [h2]. simpl in h2.
destruct h2; auto.
red.
intros y h2.
destruct h2 as [x h2]. subst.
destruct h2 as [h2]. destruct h2 as [y h2].
constructor.
exists x.
rewrite (fps_to_f_compat h1 def) at 1.
constructor. simpl; split. reflexivity.
destruct h1 as [h1l h1r]. specialize (h1r _ h2). simpl in h1r.
destruct h1r; assumption.
Qed.

Lemma fp_ran_rel_im' : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} 
         {S:Ensemble (T*U)} (pf:functionally_paired A B S)
         (def:U),
    ran_rel S = Im A (fps_to_f _ pf def).
intros T U A B S h1 def.
pose proof (fp_ran_rel_im h1 def) as h2.
rewrite (fp_dom_rel h1) in h2.
assumption.
Qed.

Lemma fp_sub_fps_to_f : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {S:Ensemble (T*U)}
         (pf:functionally_paired A B S)
         (x:T) (def:U),
    In A x ->
    functionally_paired (Subtract A x) B (Subtract S (x, (fps_to_f _ pf def x))).
intros T U A B S h1 x def h2.
constructor.
intros x' h3.
pose proof (incl_subtract A x) as h4.
assert (h5:In A x'). auto with sets.
exists (fps_to_f _ h1 def x'). red. split. split.
apply fps_to_f_in_ran. assumption.
constructor. rewrite (fps_to_f_compat h1 def) at 1.
constructor. simpl; split; auto.
intro h6.
inversion h6. subst.
destruct h3. contradict H0. constructor.
intros y h6.
destruct h6 as [h6l h6r].
inversion h6r as [h7].
rewrite (fps_to_f_compat h1 def) in h7.
destruct h7 as [h7]. simpl in h7. destruct h7; subst. reflexivity.
intros pr h3.
destruct h3 as [h3l h3r].
inversion h1 as [h1l h1r].
specialize (h1r _ h3l).
destruct h1r as [h1a h1b].
split.
constructor; auto. intro h5. inversion h5. subst.
rewrite (fps_to_f_compat h1 def) in h3l.
destruct h3l as [h3l].
destruct h3l as [h3a h3b].
rewrite <- h3a in h3r.
contradict h3r.
rewrite surjective_pairing.
constructor.
assumption.
Qed.


Lemma fp_add : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {S:Ensemble (T*U)}
         (pf:functionally_paired A B S)
         (x:T) (y:U),
    ~ In A x -> In B y -> ~ In S (x, y) ->
    functionally_paired (Add A x) B (Add S (x, y)).
intros T U A B S h1 x y h2 h3 h4.
constructor.
intros x' h5.
destruct h5 as [x' h5 | x' h6].
destruct h1 as [h1l h1r].
specialize (h1l _ h5).
destruct h1l as [y' h1l].
red in h1l.
exists y'.
red.
destruct h1l as [h1a h1b]. split.
destruct h1a as [h1aa h1ab].
split; auto. left. assumption.
intros y'' h6.
destruct h6 as [h6l h6r].
inversion h6r. subst.
specialize (h1b _ (conj h6l H)). assumption.
subst.
inversion H. subst.
contradiction.
inversion h6. subst.
exists y. red.
split. split; try right; auto. constructor.
intros y' h7. destruct h7 as [h7l h7r].
inversion h7r. subst. 
destruct h1 as [h1l h1r].
specialize (h1r _ H).
simpl in h1r. destruct h1r; contradiction. subst.
inversion H. reflexivity.
intros pr h5. destruct h5 as [pr h5l |pr  h5r].
destruct h1 as [h1l h1r].
specialize (h1r _ h5l).
destruct h1r as [h6  h7].
split; try left; auto.
inversion h5r. subst. simpl. split; try right; auto.
constructor.
Qed.


Lemma self_fp_add : 
  forall {T U:Type} {S:Ensemble (T*U)},
    self_fp S ->
    forall (x:T) (y:U), 
      ~In (dom_rel S) x ->
      self_fp (Add S (x, y)).
intros T U S h1 x y h2.
red. constructor.
intros a h3.
rewrite dom_rel_add in h3. simpl in h3.
destruct h3 as [a h3l | a h3r].
destruct h3l as [h3l]. destruct h3l as [b h3l].
exists b. red. split.
split.
rewrite ran_rel_add. simpl. left. constructor. exists a.
assumption.
left. assumption.
intros y' h4.
destruct h4 as [h4l h4r]. 
inversion h4r as [? h5 | ? h6]. 
subst. 
apply (fp_functional h1 _ _ _ h3l h5). subst.
inversion h6. subst. 
pose proof (fp_in_dom _ _ _ h1 _ h3l) as h7.
simpl in h7. contradiction.
inversion h3r. subst. clear h3r.
exists y. red.
split. split. rewrite ran_rel_add. simpl. right. constructor.
right. constructor.
intros y' h3. destruct h3 as [h3l h3r].
inversion h3r as [? h3a | ? h3b]. subst.
pose proof (fp_in_dom _ _ _ h1 _ h3a) as h4. simpl in h4. contradiction.
subst.
inversion h3b. reflexivity.
intros pr h3. split. constructor. exists (snd pr). rewrite <- surjective_pairing. assumption.
constructor. exists (fst pr). rewrite <- surjective_pairing. assumption.
Qed.

Lemma self_fp_subtract : 
  forall {T U:Type} {S:Ensemble (T*U)},
    self_fp S ->
    forall pr:T*U,
      self_fp (Subtract S pr).
intros T U S h1 pr.
red. constructor.
intros x h2.
destruct h2 as [h2]. destruct h2 as [y h2].
inversion h2 as [h3 h4].
red in h1.
exists y. red. split. split.
constructor. exists x. assumption. assumption.
intros y' h5.
destruct h5 as [h5l h5r].
inversion h5r as [h6 h7].
apply (fp_functional h1 _ _ _ h3 h6).
intros pr' h2. split.
constructor. exists (snd pr'). rewrite <- surjective_pairing.
assumption.
constructor. exists (fst pr'). rewrite <- surjective_pairing.
assumption.
Qed.


Lemma fp_sub_add' :
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {S:Ensemble (T*U)}
         (pf:functionally_paired A B S),
    forall (x:T) (y:U) (def:U),
      In A x -> In B y ->
      ~ Ensembles.In S (x, y) ->
      functionally_paired A B 
                          (Subtract (Add S (x, y)) (x, (fps_to_f _ pf def x))).
intros T U A B S h1 x y def h0' hb h0. 
assert (h2:y <> fps_to_f _ h1 def x).
intro h2.
subst.
contradict h0. 
rewrite (fps_to_f_compat h1 def) at 1. 
constructor. simpl.
split; auto.
rewrite sub_add_comm. 
pose proof (fp_sub_fps_to_f h1 _ def h0') as h3.
assert (h4:~ In (Subtract A x) x).  intro h4. destruct h4. contradict H0. constructor.
assert (h5:~In (Subtract S (x,fps_to_f _ h1 def x)) (x, y)).
  intro h6.
  destruct h6. contradiction.
pose proof (fp_add h3 _ _ h4 hb h5) as h6.
rewrite add_sub_compat_in in h6.
assumption. assumption.
intro h3.
inversion h3. contradiction.
Qed.



Lemma fps_to_f_s_compat : forall {T U:Type} {A:Ensemble T}
                                 {B:Ensemble U} {S:Ensemble (T*U)}
                                 (pf:functionally_paired A B S)
                                 (def:U),
                          forall x:T, In A x ->
                                      In S (x, fps_to_f _ pf def x).
intros T U A B S h1 def x h2.
pose proof (fps_to_f_compat h1 def) as h3.
simpl in h3.
assert (h4:In [pr : T * U | snd pr = fps_to_f _ h1 def (fst pr) /\ In A (fst pr)] 
              (x, fps_to_f _ h1 def x)).
  constructor.
  simpl. split; auto.
rewrite <- h3 in h4.
assumption.
Qed.


Inductive Fin_map {T U:Type} (A:Ensemble T) (B:Ensemble U)
          (def:U) : Type :=
  fin_map_intro : Finite A -> Finite B ->
    forall (S:Ensemble (T*U)),
                    functionally_paired A B S ->
                    Fin_map A B def.

Lemma fin_map_fin_dom : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U} (f:Fin_map A B def), Finite A.
  intros T U A B def f.
  destruct f; assumption.
Qed.

Lemma fin_map_fin_ran : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U} (f:Fin_map A B def), Finite B.
  intros T U A B def f.
  destruct f; assumption.
Qed.


Definition fin_map_app {T U:Type} {A:Ensemble T} {B:Ensemble U}
           {def:U} (F:(Fin_map A B def)): T->U.
  intro x.
  destruct F as [? ? S h1].
  refine ((fps_to_f _ h1 def) x).
Defined.

Notation "F |-> x" := (fin_map_app F x) (at level 20).

Lemma fin_map_app_in : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U}
         (f:Fin_map A B def) (x:T), In A x -> In B (f |-> x).
  intros T U A B def f x h1.
  destruct f as [h2 h3 S h4].
  destruct h4 as [h4 h5].
  unfold fin_map_app.
  unfold fps_to_f.
  destruct (classic_dec (In A x)) as [h6 | h7].
  destruct constructive_definite_description.
  simpl.
  tauto.
  contradiction.
Qed.

Lemma fin_map_app_def :
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U}
         (f:Fin_map A B def) (x:T), ~In A x -> f |-> x = def.
  intros T U A B def f x h1.
  destruct f as [h2 h3 S h4].
  unfold fin_map_app.
  unfold fps_to_f.    
  destruct (classic_dec (In A x)) as [h5 | h6].
  contradiction.
  reflexivity.
Qed.


Lemma fin_map_app_s_compat : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} (def:U)
         (pfa:Finite A) (pfb:Finite B)
         (S1 S2:Ensemble (T*U))
         (pf1:functionally_paired A B S1)
         (pf2:functionally_paired A B S2), 
         S1 = S2 <->
         fin_map_intro A B def pfa pfb S1 pf1 = 
         fin_map_intro A B def pfa pfb S2 pf2.

  intros T U A B def pfa pfb S1 S2 h1 h2.
  (* -> *)
  split. intro h3.
  subst.
  pose proof (proof_irrelevance _ h1 h2).
  subst.
  reflexivity.
  (* <- *)
  intro h3.
  apply NNPP.
  intro h4.
  destruct h1 as [h1l h1r].
  destruct h2 as [h2l h2r].
  assert (h5:(exists pr:(T*U), In S1 pr /\ ~In S2 pr) \/
             (exists pr:(T*U), In S2 pr /\ ~In S1 pr)).
    apply NNPP.
    intro h6.
    pose proof (not_or_and _ _ h6) as h7.
    destruct h7 as [h7l h7r].
    pose proof (not_ex_all_not _ _ h7l) as h8.
    pose proof (not_ex_all_not _ _ h7r) as h9.
    simpl in h8.
    simpl in h9.
    assert (h10:S1 = S2).
      apply Extensionality_Ensembles.
      red. split.
      (* <= *)
        red. intros pr h1.
        specialize (h8 pr).
        tauto.
     (* >= *)
        red. intros pr h1.
        specialize (h9 pr).
        tauto.
    contradiction.
  assert (h6:S1 = S2).
    injection h3. tauto.
  contradiction.
Qed.

Definition im_fin_map {T U:Type} {A:Ensemble T} {B:Ensemble U}
           {def:U} (F:Fin_map A B def) :=
  Im A (fin_map_app F).

Lemma in_im_fin_map : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U} (F:Fin_map A B def) (x:T),
    In A x -> In (im_fin_map F) (F|->x).
intros T U A B def F x h1.
apply Im_intro with x; auto.  
Qed.

Lemma im_fin_map_finite : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U} (F:Fin_map A B def),
    Finite (im_fin_map F).
intros T U A B def F. 
unfold im_fin_map.
pose proof (fin_map_fin_dom F) as h1.
apply finite_image. assumption.
Qed.

Lemma im_fin_map_ran :
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U} (f:Fin_map A B def), 
    Included (im_fin_map f) B.
intros T U A B def f.
unfold im_fin_map.
red. intros u h1.
destruct h1 as [x h1]. subst. 
apply fin_map_app_in. assumption.
Qed.


Lemma fin_map_card_im_le_dom : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U} 
         (F:Fin_map A B def),
    (card_fun (im_fin_map F) (im_fin_map_finite F)) <=
    (card_fun A (fin_map_fin_dom F)).
intros T U A B def F.
pose proof (card_fun_compat A (fin_map_fin_dom F)) as h1.
pose proof (card_fun_compat (im_fin_map F) (im_fin_map_finite F)) as h2.
unfold im_fin_map.
apply (cardinal_decreases _ _ _ (fin_map_app F) _ h1 _  h2).
Qed.

Lemma fin_map_card_im_le_dom' : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U} 
         (F:Fin_map A B def),
    (card_fun1 (im_fin_map F)) <=
    (card_fun1 A).
intros T U A B def F.
pose proof (card_fun1_compat A) as h1.
pose proof (card_fun1_compat (im_fin_map F)) as h2.
destruct h1 as [h1l h1r]. destruct h2 as [h2l h2r].
specialize (h1l (fin_map_fin_dom F)).
specialize (h2l (im_fin_map_finite F)).
unfold im_fin_map.
eapply cardinal_decreases. apply h1l. apply h2l.
Qed.



Lemma fin_map_s_compat : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U) 
         (S:Ensemble (T*U)) (def:U) (pr:T*U)
       (pf:functionally_paired A B S),
         In S pr ->
         (fps_to_f _ pf def) (fst pr) = snd pr. 
intros T U A B S def pr h1 h2.
unfold fps_to_f.
destruct (classic_dec (In A (fst pr))) as [h3 | h4].
destruct h1 as [h1a h1b].
destruct constructive_definite_description as [u h4].
simpl.
specialize (h1b _ h2).
specialize (h1a _ h3).
destruct h1a as [y h5 h6].
red in h5.
destruct h5 as [h5l h5r].
pose proof (h5r _ h4).
subst.
apply h5r.
rewrite surjective_pairing in h2.
tauto.
destruct h1 as [h1a h1b].
specialize (h1b _ h2).
tauto.
Qed.


Lemma fps_to_f_inj : 
  forall {T U:Type} 
         {A:Ensemble T} {B:Ensemble U}
         (S1 S2:Ensemble (T*U)) (def:U) 
         (pf1:functionally_paired A B S1)
         (pf2:functionally_paired A B S2),
    fps_to_f _ pf1 def = fps_to_f _ pf2 def ->
    S1 = S2.
  intros T U A B S1 S2 def h1 h2 h3.
assert (h4:forall x:T, (fps_to_f _ h1 def) x = 
                       (fps_to_f _ h2 def) x).
  intros; rewrite h3; reflexivity.
apply Extensionality_Ensembles.
red. split.
  (* <= *)
  red.
  intros pr h5.
  specialize (h4 (fst pr)).
  unfold fps_to_f in h4.
  pose proof (fp_in_dom _ _ _ h1 _ h5) as h6.
  destruct (classic_dec (In A (fst pr))) as [h7 | h8].
  pose proof h1 as h2'.
  destruct h1 as [h7a h7b].
  destruct constructive_definite_description as [u hu].
  simpl in h4.
  destruct h2 as [h3a h3b].
  destruct constructive_definite_description  as [u' hu'].
  simpl in h4.
  subst.
  pose proof (h7a _ h6) as h8.
  destruct h8 as [u h8].
  red in h8.
  destruct h8 as [h8l h8r].
  pose proof (h8r _ hu) as h9.
  subst.
  rewrite surjective_pairing in h5.
  pose proof (fp_in_ran _ _ _ h2' _ h5) as h9.
  simpl in h9.
  assert (h10:In B (snd pr) /\ In S1 (fst pr, snd pr)).
    split; assumption.
  pose proof (h8r _ h10).
  subst.
  destruct hu'. rewrite surjective_pairing. assumption.
  contradiction.
  (* >= *)
  (* More or less copy and paste of the above.*)
  red.
  intros pr h5.
  specialize (h4 (fst pr)).
  unfold fps_to_f in h4.
  pose proof (fp_in_dom _ _ _ h2 _ h5) as h6.
  destruct (classic_dec (In A (fst pr))) as [h7 | h8].
  pose proof h2 as h3'.
  destruct h1 as [h7a h7b].
  destruct constructive_definite_description as [u hu].
  simpl in h4.
  destruct h2 as [h3a h3b].
  destruct constructive_definite_description as [u' hu'].
  simpl in h4.
  subst.

  pose proof (h3a _ h6) as h8.
  destruct h8 as [u h8].
  red in h8.
  destruct h8 as [h8l h8r].
  pose proof (h8r _ hu') as h9.
  subst.
  rewrite surjective_pairing in h5.
  pose proof (fp_in_ran _ _ _ h3' _ h5) as h9.
  simpl in h9.
  assert (h10:In B (snd pr) /\ In S2 (fst pr, snd pr)).
    split; assumption.
  pose proof (h8r _ h10).
  subst.
  destruct hu. rewrite surjective_pairing. assumption.
  contradiction.
Qed.


Lemma fps_to_f_inj' : forall {T U:Type} (A:Ensemble T) 
                             (B C:Ensemble U) 
                             (S1 S2:Ensemble (T*U)) (def:U)
                             (pf1:functionally_paired A B S1)
                             (pf2:functionally_paired A C S2),
                             fps_to_f _ pf1 def=
                             fps_to_f _ pf2 def-> S1 = S2.
intros T U A B C S1 S2 def h1 h2 h3.
assert (h4:forall (A':Ensemble T) (B' C':Ensemble U)
                  (S1' S2':Ensemble (T*U))
                  (h1':functionally_paired A' B' S1')
                  (h2':functionally_paired A' C' S2')
                  (h3':fps_to_f _ h1' def = 
                       fps_to_f _ h2' def),
             Included S1' S2').
intros A' B' C' S1' S2' h1' h2' h3'.
(* <= *)
red.
intros pr h4.
inversion h1' as [h1a h1b]; inversion h2' as [h2a h2b].
pose (fps_to_f _ h1' def (fst pr)). 
pose proof (fp_in_dom _ _ _ h1' _ h4) as h5.
(* First establish that (fst pr) in *)
specialize (h2a _ h5).
destruct h2a as [y h6].
red in h6.
destruct h6 as [h6l h6r].
destruct h6l as [h6la h6lb].
pose proof (fin_map_s_compat _ _ _ def _ h2' h6lb) as h6.
simpl in h6.
rewrite <- h3' in h6.
unfold fps_to_f in h6.
destruct (classic_dec (In A' (fst pr))) as [h7 | h8].
pose proof h1' as h1''.
destruct h1'.
destruct constructive_definite_description as [c h8].
simpl in h6. subst.
destruct h8 as [h8l h8r].
pose proof (surjective_pairing pr) as h9.
rewrite h9 in h4.

pose proof (fp_functional  h1'' _ _ _ h8r h4) as h10.
rewrite h10 in h6lb.
rewrite surjective_pairing.
apply h6lb. contradiction.
(* <= *)
apply Extensionality_Ensembles.
red; split.
apply (h4 _ _ _ _ _ h1 h2 h3).
symmetry in h3.
apply (h4 _ _ _ _ _ h2 h1 h3).
Qed.




Lemma fin_map_app_inj : forall {T U : Type} {A : Ensemble T}
                           {B : Ensemble U} {def : U}
                           (f1 f2:Fin_map A B def),
                          fin_map_app f1 = fin_map_app f2 ->
                          f1 = f2.
intros T U A B def f1 f2 h1.
unfold fin_map_app in h1.

destruct f1 as [h1a h1b S1 h2].
destruct f2 as [h2a h2b S2 h3].

assert (h1a = h2a). apply  proof_irrelevance.
assert (h1b = h2b). apply  proof_irrelevance.
subst.
rewrite <- fin_map_app_s_compat.

assert (h4:(fun x : T => fps_to_f _ h2 def x) = 
           fps_to_f _ h2 def).
  apply functional_extensionality. reflexivity.
assert (h5:(fun x : T => fps_to_f _ h3 def x) = 
           fps_to_f _ h3 def).
  apply functional_extensionality. reflexivity.
rewrite h4 in h1.
rewrite h5 in h1.
clear h4 h5.
apply fps_to_f_inj with def h2 h3.
assumption.
Qed.




Lemma fin_map_ext : forall {T U:Type} {A:Ensemble T} 
                           {B:Ensemble U} {def:U}
                           (f1 f2:Fin_map A B def),
                      (forall x:T, f1 |-> x = f2 |-> x) ->
                                  f1 = f2.
  intros T U A B def f1 f2 h1.
  destruct f1 as [h3 h4 S1 h5].
  destruct f2 as [h6 h7 S2 h8].
  pose proof (functional_extensionality _ _ h1) as h9.
  apply fin_map_app_inj; assumption.
Qed.







Lemma distinct_fin_maps_differ_at_point : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U}
         (f1 f2:Fin_map A B def), 
    f1 <> f2 -> 
    exists x:T, In A x /\ f1 |-> x <> f2 |-> x.
  intros T U A B def f1 f2 h1.
  pose proof (fin_map_app_inj f1 f2).
  assert (h3:fin_map_app f1 <> fin_map_app f2).
    tauto.
  pose proof (distinct_functions_differ_at_point _ _ h3) as h4.
  destruct h4 as [x h4].
  exists x.
  destruct (classic_dec (In A x)) as [h5 | h6].
  (* h5 *)
    split; assumption.
  (* h6 *)
    pose proof (fin_map_app_def f1 _ h6).
    pose proof (fin_map_app_def f2 _ h6).
    congruence.
Qed.


Definition inj {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U} 
           (F:Fin_map A B def) : Prop :=
  forall x y:T, In A x -> In A y -> F |-> x = F |-> y -> x = y.

Definition surj {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U} 
           (F:Fin_map A B def) : Prop :=
  forall b:U, In B b -> exists a:T, In A a /\ F |-> a = b.

Definition bij {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U} 
           (F:Fin_map A B def) : Prop :=
  inj F /\ surj F.

Lemma inj_im_eq :
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U} 
         (F:Fin_map A B def),
    inj F ->
    card_fun A (fin_map_fin_dom F) =
    card_fun (im_fin_map F) (im_fin_map_finite F).
intros T U A B def F h0.
pose proof (card_fun_compat (full_sig A) (iff1 (finite_full_sig_iff A) (fin_map_fin_dom F))) as h1.
pose proof (card_fun_compat (full_sig (im_fin_map F)) (iff1 (finite_full_sig_iff (im_fin_map F)) (im_fin_map_finite F))) as h2.
rewrite card_fun_full_sig_eq.  rewrite (card_fun_full_sig_eq (im_fin_map F)).  
pose (fun x:(sig_set A) => exist _ (F |-> (proj1_sig x)) (in_im_fin_map F _ (proj2_sig x))) as f.
pose proof (injective_preserves_cardinal).
assert (h3:Image.injective _ _ f). red. intros x y.
  unfold f. intro h4. apply exist_injective in h4.
  apply h0 in h4. apply proj1_sig_injective. assumption.
  apply proj2_sig. apply proj2_sig. 
unfold sig_set in h1. unfold sig_set in h2.
assert (h4:full_sig (im_fin_map F) =
                    (Im (full_sig A) f)).
  apply Extensionality_Ensembles.
  red. split.
  red. intros y h4. unfold sig_set in y.
  destruct y as [y h5].
  destruct h5 as [x h6 y h5]. subst.
  apply Im_intro with (exist _ x h6).
  constructor. unfold f. apply proj1_sig_injective. simpl.
  reflexivity.
  red. intros y h5. constructor. 
assert (h5:cardinal {x : U | In (im_fin_map F) x} (Im (full_sig A) f)  (card_fun (full_sig (im_fin_map F))
            (iff1 (finite_full_sig_iff (im_fin_map F)) (im_fin_map_finite F)))). rewrite <- h4. assumption.
pose proof (injective_preserves_cardinal _ _ _ _ _ h3 h1 _ h5) as h6. symmetry.
assumption.
Qed.

Lemma inj_im_eq' :
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U} 
         (F:Fin_map A B def),
    inj F ->
    card_fun1 A = 
    card_fun1 (im_fin_map F).
intros T U A B def F h1.
do 2 erewrite <- card_fun_card_fun1_compat.
apply inj_im_eq; auto.
Qed.

Lemma surj_im_ran_eq : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U} 
         (F:Fin_map A B def),
    surj F ->
    (im_fin_map F) = B.
intros T U A B def F h1.
red in h1.
apply Extensionality_Ensembles.
red. split.
red. intros y h2.
pose proof (im_fin_map_ran F) as h3. red in h3. apply h3; auto.
red. intros y h2.
specialize (h1 _ h2).
destruct h1 as [x h1].
destruct h1; subst.
apply in_im_fin_map; auto.
Qed.

Lemma surj_im_ran_card_eq : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U} 
         (F:Fin_map A B def),
    surj F ->
    card_fun (im_fin_map F) (im_fin_map_finite F) = 
    card_fun B (fin_map_fin_ran F).
intros T U A B def F h1.
pose proof (surj_im_ran_eq F h1) as h2.
pose proof (subsetT_eq_compat _ _ _ _ (im_fin_map_finite F) (fin_map_fin_ran F) h2) as h3.
dependent rewrite -> h3.
reflexivity.
Qed.

Lemma surj_im_ran_card_eq' : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U} 
         (F:Fin_map A B def),
    surj F ->
    card_fun1 (im_fin_map F) = 
    card_fun1 B.
intros T U A B def F h1.
do 2 erewrite <- card_fun_card_fun1_compat.
apply surj_im_ran_card_eq; auto.
Qed.

Lemma bij_dom_ran_card_eq : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U} 
         (F:Fin_map A B def),
    bij F ->
    card_fun A (fin_map_fin_dom F) =
    card_fun B (fin_map_fin_ran F).
intros T U A B def F h1. red in h1.
destruct h1 as [h1l h1r].
rewrite inj_im_eq; auto.
rewrite surj_im_ran_card_eq; auto.
Qed.

Lemma bij_dom_ran_card_eq' : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U} 
         (F:Fin_map A B def),
    bij F ->
    card_fun1 A = 
    card_fun1 B.
intros T U A B def F h1.
do 2 erewrite <- card_fun_card_fun1_compat.
eapply bij_dom_ran_card_eq; auto. apply h1.
Qed.

Definition fin_map_to_fun {T U:Type} {A:Ensemble T} 
           {B:Ensemble U} {def:U} (f:Fin_map A B def) : T->U.
destruct f as [h1 h2 S h3].
refine (fps_to_f _ h3 def).
Defined.

Lemma fin_map_to_fun_compat : 
  forall {T U:Type} {A:Ensemble T}
         {B:Ensemble U} (def:U) (f:Fin_map A B def),
  forall x:T, f |-> x = (fin_map_to_fun f) x.
intros T U A B def f x.
unfold fin_map_app, fin_map_to_fun.
destruct f. reflexivity.
Qed.


Lemma fin_map_new_ran_lemma :
  forall {T U:Type} {A:Ensemble T} {B C:Ensemble U}
         {def:U} (F:Fin_map A B def),
    Finite C -> Included B C -> exists! F':Fin_map A C def,
                      forall x:T, F |-> x = F' |-> x.
intros T U A B C def F h1 h2.
pose (fin_map_app F) as f.
destruct F as [h3 h4 S h6].
assert (h7:functionally_paired A C S).
  constructor.
  inversion h6 as [h6a h6b].
  intros x h7.
  specialize (h6a x h7).
  destruct h6a as [y h6a].
  red in h6a.
  destruct h6a as [h8 h9].
  assert (h10:In C y /\ In S (x, y)).  intuition auto with sets.
    exists y. red.
    split. assumption.
    intros u h13.  
  destruct h13 as [h13l h13r].
  pose proof (fp_in_ran  _ _  _ h6 _ h13r) as h14.
  simpl in h14.
  specialize (h9 u). tauto.
inversion h6 as [h6a h6b].
intros pr h13.
specialize (h6b _ h13).
intuition auto with sets.
pose (fin_map_intro _ _ def h3 h1 _ h7) as F'.
exists F'.
red. split.
intro x.
unfold F'.
unfold fin_map_app.
destruct (classic_dec (In A x)) as [ha | ha'].

pose proof (fps_to_f_s_compat h6 def _ ha) as h9.
pose proof (fps_to_f_s_compat h7 def _ ha) as h10.
pose proof (fp_functional h6 _ _ _ h9 h10) as h11.
assumption.
unfold fps_to_f.
destruct (classic_dec (In A x)); try contradiction. reflexivity.


intros G h8.
unfold F'.
apply fin_map_ext.
intro x.
rewrite <- h8.
unfold fin_map_app.
destruct (classic_dec (In A x)) as [ha | ha'].

pose proof (fps_to_f_s_compat h6 def _ ha) as h9.
pose proof (fps_to_f_s_compat h7 def _ ha) as h10.
pose proof (fp_functional h6 _ _ _ h9 h10) as h11.
rewrite h11. reflexivity.
unfold fps_to_f.
destruct (classic_dec (In A x)); try contradiction. reflexivity.
Qed.

Definition fin_map_new_ran {T U:Type} {A:Ensemble T} 
           {B C:Ensemble U} {def:U} (F:Fin_map A B def)
           (pfc:Finite C) (pfinc:Included B C) :
  Fin_map A C def.  
refine (proj1_sig 
          (constructive_definite_description 
             _ (fin_map_new_ran_lemma F pfc pfinc))).
Defined.

Lemma fin_map_new_ran_compat :
  forall {T U:Type} {A:Ensemble T} 
           {B C:Ensemble U} {def:U} (F:Fin_map A B def)
           (pfc:Finite C) (pfinc:Included B C),
  let F' := fin_map_new_ran F pfc pfinc in
  forall x:T, F |-> x = F' |-> x.
intros  T U A B C def F h1 h2 F'.
unfold fin_map_new_ran in F'.
destruct constructive_definite_description as [F'' h4].
simpl in F'.
unfold F'.
apply h4.
Qed.

Lemma fin_map_new_ran_compat' : 
  forall {T U:Type} {A:Ensemble T} 
           {B C:Ensemble U} {def:U} (F:Fin_map A B def)
           (pfc:Finite C) (pfinc:Included B C),
    fin_map_app F = fin_map_app (fin_map_new_ran F pfc pfinc).
intros T U A B C def F pfc pfinc.
apply functional_extensionality.
apply fin_map_new_ran_compat.
Qed.


Lemma fun_to_fin_map_lemma : forall  
           {T U:Type} (A:Ensemble T) (def:U)
           (pfa:Finite A) (f:T->U),
                            exists! F:Fin_map A (Im A f) def,
                              forall x:T, In A x ->
                              (fin_map_to_fun F) x = f x.
intros T U A def h1 f.
pose [pr:T*U | snd pr = f (fst pr) /\ In A (fst pr)]
  as S.
assert (h3:functionally_paired A (Im A f) S).
  constructor.
  intros x h3.
  exists (f x). red.
  repeat split.
  apply Im_intro with x. assumption. reflexivity.
  simpl. assumption.
  intros y h4.
  destruct h4 as [h4l h4r].
  destruct h4l. subst.
  destruct h4r as [h5].
  simpl in h5. symmetry. destruct h5; assumption.
  
  intros pr h3.
  destruct h3 as [h3].
  destruct h3 as [h3l h3r]. 
  split; try assumption. 
  rewrite h3l.
  apply Im_intro with (fst pr). assumption. reflexivity.
pose proof (finite_image _ _ _ f h1) as h4.
pose (fin_map_intro _ _ def h1 h4 S h3) as F.
exists F.
red. split. simpl.
pose proof (fps_to_f_compat h3 def) as h5.
intros x h6.
assert (h7:In S (x, (f x))).
  constructor. simpl. split; auto.
rewrite h5 in h7.
destruct h7 as [h8].
simpl in h8. symmetry. destruct h8; assumption.

intros F' h5.
apply fin_map_ext.
intro x.
destruct (classic_dec (In A x)).
assert (h6:f x = fin_map_to_fun F x).
  unfold F. unfold fin_map_to_fun.
  unfold fps_to_f.
  destruct (classic_dec (In A x)). destruct h3.
destruct constructive_definite_description.
  simpl.
  destruct a0 as [h6 ?].
  destruct h6. subst.
  destruct H.
  simpl in H. destruct H; auto.
  contradiction.
specialize (h5 _ i).
rewrite h6 in h5.
do 2 rewrite fin_map_to_fun_compat.
auto.

pose proof (fin_map_app_def F _ n).
pose proof (fin_map_app_def F' _ n).
rewrite H. rewrite H0. auto.
Qed.

Definition fun_to_fin_map  
           {T U:Type} (A:Ensemble T) (def:U)
           (pfa:Finite A) (f:T->U) : Fin_map A (Im A f) def.
pose proof (fun_to_fin_map_lemma A def pfa f) as h1.
refine (proj1_sig
          (constructive_definite_description _ h1)).
Defined.

Lemma fun_to_fin_map_compat : 
  forall {T U:Type} (A:Ensemble T) (def:U)
         (pfa:Finite A) (f:T->U), 
      forall x:T, In A x ->
                  (fun_to_fin_map A def pfa f) |-> x =
                  f x.
intros T U A def h1 f x h2.
unfold fun_to_fin_map.
destruct constructive_definite_description as [F h4].
simpl.
specialize (h4 _ h2).
rewrite <- h4.
apply fin_map_to_fun_compat.
Qed.

Lemma fin_map_to_fun_undoes_fun_to_fin_map :
  forall {T U:Type} (A:Ensemble T) (pf:Finite A) (def:U)
         (p:T->U) (x:T), In A x ->
    fin_map_to_fun (fun_to_fin_map A def pf p) x = p x.
intros T U A h1 def p x h2.
rewrite <- fin_map_to_fun_compat.
rewrite fun_to_fin_map_compat;auto.
Qed.


Lemma im_fin_map_app_undoes_fun_to_fin_map : 
  forall {T U:Type} (E:Ensemble T) (f:T->U) (def:U) 
    (pf:Finite E),
    Im E (fin_map_app (fun_to_fin_map E def pf f)) =
    Im E f.
intros T U E f def h1.
apply Extensionality_Ensembles.
red. split.
red.
intros y h2.
destruct h2 as [y h2 x]. subst.
apply Im_intro with y. assumption.
apply fun_to_fin_map_compat. assumption.
red. intros y h2.
destruct h2 as [y h2 x]. subst.
apply Im_intro with y.
assumption.
rewrite fun_to_fin_map_compat; auto.
Qed.


Lemma fp_cart_prod_sing : 
  forall {T U:Type} (A:Ensemble T) (b:U),
    functionally_paired A (Singleton b) 
                        (cart_prod A (Singleton b)).
intros T U A b.
constructor.
intros x h1.
exists b.
red. split. split. constructor. constructor.
split. simpl.  assumption. simpl. constructor.
intros u h2.
destruct h2 as [h2l h2r].
destruct h2l; subst. reflexivity.
intros pr h1.
destruct h1 as [h1].
assumption.
Qed.

Lemma fin_map_cart_prod_fin1 : 
  forall {T U V:Type} {A:Ensemble T} 
         {B:Ensemble U} {C:Ensemble V} {def:V} 
         (F:Fin_map (cart_prod A B) C def),
    Inhabited (cart_prod A B) -> Finite A /\ Finite B.
intros T U V A B C def F h1.
destruct F as [h4 h5 S h6].
apply cart_prod_fin_rev; assumption.
Qed.

Lemma fin_map_cart_prod_fin2 : 
  forall {T U V:Type} {A:Ensemble T} 
         {B:Ensemble U} {C:Ensemble V} {def:V} 
         (F:Fin_map (cart_prod A B) C def),
    A <> Empty_set _ -> Finite B.
intros T U V A B C def F h1.
inversion F as [h4 h5 S h6].
pose proof (not_empty_Inhabited _ _ h1) as h3.
apply cart_prod_fin_rev2 with A; assumption.
Qed.


Definition fin_map_sing {T U:Type} (A:Ensemble T) (pfa:Finite A)
           (def val:U) := 
           fin_map_intro _ _ def pfa (Singleton_is_finite _ val) _ (fp_cart_prod_sing A val).




(* fun y => {(F (x,y) |x in A)} *)

Definition im2 {T U V:Type} {A:Ensemble T} 
           {B:Ensemble U} {C:Ensemble V} {def:V} 
           (F:Fin_map (cart_prod A B) C def) (y:U): Ensemble V.
destruct (eq_dec B (Empty_set _)) as [h1 | h2].
refine (Empty_set V).
pose (fun x:T => F |-> (x, y)) as f.
refine (Im A f).
Defined.

Lemma im2_fin : 
  forall {T U V:Type} {A:Ensemble T} 
         {B:Ensemble U} {C:Ensemble V} {def:V} 
         (F:Fin_map (cart_prod A B) C def) (y:U),
    Finite (im2 F y).
intros T U V A B C def F y.
unfold im2.
destruct (eq_dec B (Empty_set U)) as [h1 | h2].
constructor.
inversion F as [h4 h5 S h6].
pose proof (not_empty_Inhabited _ _ h2) as h3.
apply finite_image.
apply (cart_prod_fin_rev1 _ _ h4 h3).
Qed.


Lemma im2_inc : forall {T U V:Type} {A:Ensemble T}
                       {B:Ensemble U} {C:Ensemble V}
                       {def:V}
                       (F:Fin_map (cart_prod A B) C def) (y:U),
                  In B y -> Included (im2 F y) (Im (cart_prod A B)
                                         (fin_map_app F)).
intros T U V A B C def F y h0.
red. intros z h1.
unfold im2 in h1.
destruct (eq_dec B (Empty_set U)) as [|h2]; try contradiction.
destruct h1 as [x h1]; subst.
apply Im_intro with (x, y).
constructor. simpl. split; assumption. 
reflexivity.
Qed.

Definition im1 {T U V:Type} {A:Ensemble T} 
           {B:Ensemble U} {C:Ensemble V} {def:V} 
           (F:Fin_map (cart_prod A B) C def) (x:T): Ensemble V.
destruct (eq_dec A (Empty_set _)) as [h1 | h2].
refine (Empty_set _).
pose (fun y:U => F |-> (x, y)) as f.
refine (Im B f).
Defined.

Lemma im1_fin : 
  forall {T U V:Type} {A:Ensemble T} 
         {B:Ensemble U} {C:Ensemble V} {def:V} 
         (F:Fin_map (cart_prod A B) C def) (x:T),
    Finite (im1 F x).
intros T U V A B C def F x.
unfold im1.
destruct (eq_dec A (Empty_set T)) as [h1 | h2].
constructor.
inversion F as [h4 h5 S h6].
pose proof (not_empty_Inhabited _ _ h2) as h3.
apply finite_image.
apply (cart_prod_fin_rev2 _ _ h4 h3).
Qed.


Lemma im1_inc : forall {T U V:Type} {A:Ensemble T}
                       {B:Ensemble U} {C:Ensemble V}
                       {def:V}
                       (F:Fin_map (cart_prod A B) C def) (x:T),
                  In A x -> Included (im1 F x) (Im (cart_prod A B)
                                         (fin_map_app F)).
intros T U V A B C def F x h0.
red. intros z h1.
unfold im1 in h1.
destruct (eq_dec A (Empty_set T)) as [|h2].
contradiction.
destruct h1 as [y h1]; subst.
apply Im_intro with (x, y).
constructor. simpl. split; assumption. 
reflexivity.
Qed.




Lemma fps_inc_cart_prod :
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {S:Ensemble (T*U)},
  functionally_paired A B S -> Included S (cart_prod A B).
intros T U A B S h1.
red.
intros pr h2.
constructor.
destruct h1 as [h3 h4].
apply h4. assumption.
Qed.


Lemma fin_map_to_fps_lemma : 
  forall {T U:Type} {A:Ensemble T}
         {B:Ensemble U} {def:U}
         (F:Fin_map A B def),
  exists! (S:Ensemble (T*U)),
    exists (pf:functionally_paired A B S),
      fps_to_f _ pf def = fin_map_app F.
intros T U A B def h1.
destruct h1 as [h1 h2 S h3].
exists S. red.
split.
exists h3.
unfold fin_map_app.
reflexivity.
intros S' h4.
destruct h4 as [h4 h5].
unfold fin_map_app in h5.
symmetry.
apply (fps_to_f_inj _ _ def h4 h3 h5).
Qed.

Lemma fin_map_to_fps {T U:Type} {A:Ensemble T}
         {B:Ensemble U} {def:U}
         (F:Fin_map A B def) : Ensemble (T*U).
refine 
  (proj1_sig 
     (constructive_definite_description _ 
                                        (fin_map_to_fps_lemma F))).
Defined.

Lemma fin_map_to_fps_compat : 
  forall {T U:Type} {A:Ensemble T}
         {B:Ensemble U} {def:U}
         (F:Fin_map A B def), 
  exists (pf:functionally_paired A B
         (fin_map_to_fps F)),
    fps_to_f _ pf def = fin_map_app F.
intros T U A B def F.
unfold fin_map_to_fps.
destruct constructive_definite_description as [S h1].
simpl.
assumption.
Qed.

Lemma fin_map_to_fps_inj : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U},
    injective (@fin_map_to_fps T U A B def).
intros T U A B def.
red.
intros F G h1.
pose proof (fin_map_to_fps_compat F) as h2.
pose proof (fin_map_to_fps_compat G) as h3.
destruct h2 as [h2a h2b].
destruct h3 as [h3a h3b].
apply fin_map_app_inj.
rewrite <- h2b.
rewrite <- h3b.
pose proof (subsetT_eq_compat _ _ _ _ h2a h3a h1) as h4.
dependent rewrite -> h4.
reflexivity.
Qed.



Lemma fp_fin_map_to_fps :
  forall {T U:Type} {A:Ensemble T}
         {B:Ensemble U} {def:U}
         (F:Fin_map A B def), 
    functionally_paired A B (fin_map_to_fps F).
intros T U A B def F.
pose proof (fin_map_to_fps_compat F) as h1.
destruct h1; auto.
Qed.


Lemma in_fin_map_to_fps : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U}
         (F:Fin_map A B def),
    forall x:T, In A x -> In (fin_map_to_fps F) (x, F |-> x).
intros T U A B def F x h1.
unfold fin_map_to_fps. 
destruct constructive_definite_description as [S h2].
simpl.
destruct h2 as [h2 h3].
pose proof (fps_to_f_s_compat h2 def _ h1) as h4.
unfold fin_map_app. 
destruct F as [h4' h5 S' h6]. 
unfold fin_map_app in h3. simpl in h3.
pose proof (fps_to_f_inj _ _ def h2 h6 h3). subst.
assert (h7:h2 = h6). apply proof_irrelevance.
subst.
assumption.
Qed.





Definition fin_map_eq {T U:Type} {A:Ensemble T} {B C:Ensemble U}
           {def:U} (F:Fin_map A B def) (G:Fin_map A C def) := 
  exists pf:Included B C, fin_map_new_ran F (fin_map_fin_ran G)
                                          pf = G.



Lemma fps_eq_fin_map_eq : forall {T U:Type} {A:Ensemble T} 
                                 {B C:Ensemble U} {def:U} 
                                 (F:Fin_map A B def) 
                                 (G:Fin_map A C def),
                            Included B C ->
                            fin_map_to_fps F =
                            fin_map_to_fps G ->
                            fin_map_eq F G.
intros T U A B C def F G h0 h1.
unfold fin_map_to_fps in h1.

destruct  constructive_definite_description as [S h2];
  destruct constructive_definite_description as [S' h3].
simpl in h1.
destruct h2 as [h2 h2'].
destruct h3 as [h3 h3'].
red. exists h0.

unfold fin_map_new_ran.

destruct constructive_definite_description as [G' h4].
simpl.
apply fin_map_ext.
intros x.
rewrite <- h4.
rewrite <- h2'.
rewrite <- h3'.
destruct (classic_dec (In A x)) as [h5 | h6].
pose proof (fps_to_f_s_compat h2 def _ h5) as h6.
pose proof (fps_to_f_s_compat h3 def _ h5) as h7.
rewrite h2', h3'.
rewrite h2' in h6.
rewrite h3' in h7.
rewrite <- h1 in h7.
destruct h2 as [h8 h9].
pose proof (h9 (x, G |-> x) h7) as h10.
simpl in h10.
destruct h10 as [? h11].
pose proof (fin_map_app_in F x h5) as h12.
pose proof (h8 x h5) as h10.
destruct h10 as [y h10].
red in h10.
destruct h10 as [h10a h10b].
pose proof (h10b _ (conj h12 h6)) as h13.
pose proof (h10b _ (conj h11 h7)) as h14.
rewrite <- h13, h14.
reflexivity.
unfold fps_to_f.
destruct (classic_dec (In A x)) as [h7 | h8].
contradiction.
reflexivity.
Qed.

Definition extends {T U:Type} {A A':Ensemble T} {B:Ensemble U} {def:U}
           (F':Fin_map A' B def) (F:Fin_map A B def) :=
  Included A A' /\ forall x:T, In A x -> F |-> x = F' |-> x.


Lemma functionally_paired_restriction : 
  forall {T U:Type} (A A':Ensemble T) (B:Ensemble U)
         (S':Ensemble (T*U)),
    Included A A' ->
    let S := [pr : T * U | In S' pr /\ In A (fst pr)] : Ensemble (T * U) in
    functionally_paired A' B S' ->
    functionally_paired A B S.
intros T U A A' B S' h1 S h2.
destruct h2 as [h2l h2r].
constructor. 
intros x h3.
assert (h4:In A' x). auto with sets.
specialize (h2l _ h4). destruct h2l as [y h2l].
red in h2l. destruct h2l as [h2a h2b].
destruct h2a as [h2a h2a'].
exists y. red. split. unfold S. split; auto.
constructor. split; auto.
intros u h5.
destruct h5 as [h5l h5r].
specialize (h2b u).
destruct h5r as [h5r]. simpl in h5r.
destruct h5r as [h5ra h5rb].
specialize (h2b (conj h5l h5ra)). assumption.
intros pr h3.
destruct h3 as [h3].
destruct h3 as [h3l h3r].
specialize (h2r _ h3l).
destruct h2r.
split; auto.
Qed.

Lemma restriction_ex : 
  forall {T U:Type} {A A':Ensemble T} {B:Ensemble U} {def:U}
         (F':Fin_map A' B def),
         Included A A' -> exists! F:Fin_map A B def,
                            forall x:T, In A x -> F |-> x = F' |-> x.
intros T U A A' B def F h1.
destruct F as [h2 h3 S' h4].
pose [pr:T*U | In S' pr /\ In A (fst pr)] as S.
pose proof (functionally_paired_restriction _ _ B S' h1 h4) as h5.
exists (fin_map_intro _ _ _ (Finite_downward_closed _ _ h2 _ h1) h3 _ h5).
red.
split.
intros x h6. 
unfold fin_map_app.
unfold fps_to_f. 
destruct (classic_dec (In A x)) as [h7 | h8].
destruct (classic_dec (In A' x)) as [h9 | h10]. 
pose proof h4 as h4'.
destruct h5 as [h5a h5b]; destruct h4 as [h4a h4b].
destruct constructive_definite_description as [u h11]. 
destruct  constructive_definite_description as [u' h12]. simpl.
destruct h11 as [h11l h11r].
destruct h12 as [h12l h12r].
destruct h11r as [h11r]. destruct h11r as [h11a h11b].
apply (fp_functional  h4' _ _ _ h11a h12r). 
red in h1. specialize (h1 _ h6). contradiction. contradiction.
intros F h6. 
apply fin_map_ext.   
intro x. unfold fin_map_app.
destruct F as [h7 h8 S'' h9].
unfold fps_to_f. 
destruct (classic_dec (In A x)) as [h10 | h11]. 
specialize (h6 _ h10).
unfold fin_map_app in h6. unfold fps_to_f in h6.  pose proof h9 as h9'.
destruct h5 as [h5a h5b]; destruct h9 as [h9a h9b].
destruct constructive_definite_description  as [u h11].
destruct  constructive_definite_description as [u' h12]. simpl.
destruct (classic_dec (In A x)) as [h13 | h14].
destruct (classic_dec (In A' x)) as [h15 | h16].
destruct constructive_definite_description as [u'' h17].
pose proof h4 as h4'.
destruct h4 as [h18 h19].
destruct  constructive_definite_description as [u''' h20]. simpl in h6. subst.
destruct h11 as [h11l h11r].
destruct h11r as [h11r]. simpl in h11r.
destruct h20 as [h20l h20r].
destruct h12 as [h12l h12r]. destruct h17 as [h17l h17r].
pose proof (fp_functional h9' _ _ _ h17r h12r). subst.
destruct h11r as [h11a h11b].
pose proof (fp_functional h4' _ _ _ h11a h20r). assumption.
red in h1. specialize (h1 _ h13). contradiction. contradiction.
reflexivity.
Qed.


Definition restriction  {T U:Type} {A A':Ensemble T} {B:Ensemble U} {def:U}
         (F':Fin_map A' B def)
         (pf:Included A A') : Fin_map A B def.
refine (proj1_sig (constructive_definite_description _ (restriction_ex F' pf))).
Defined.

Lemma restriction_compat :   
  forall {T U:Type} {A A':Ensemble T} 
         {B:Ensemble U} {def:U}
         (F':Fin_map A' B def)
         (pf:Included A A'),  forall x:T, In A x -> (restriction F' pf) |-> x = F' |-> x.
intros T U A A' B def F' h1 x h2.
unfold restriction.
destruct constructive_definite_description as [F h3]. simpl.
apply h3; auto.
Qed.

Lemma restriction_extends_compat :
  forall {T U:Type} {A A':Ensemble T} {B:Ensemble U} {def:U} 
         (F:Fin_map A' B def) 
    (pf:Included A A'), extends F (restriction F pf).
intros T U A A' B def F h1.
red. split; auto.
apply restriction_compat.
Qed.

Lemma restriction_extends_compat_iff :
  forall {T U:Type} {A A':Ensemble T} {B:Ensemble U} {def:U} 
         (F':Fin_map A' B def) (F:Fin_map A B def)
    (pf:Included A A'), extends F' F <-> 
    (restriction F' pf) = F.
intros T U A A' B def F' F h1.
split.
intro h2.
apply fin_map_ext.
red in h2.
destruct h2 as [h2l h2r].
intro x.
destruct (classic_dec (In A x)) as [h3 |h4].
rewrite restriction_compat.
symmetry. apply h2r; auto. assumption.
rewrite fin_map_app_def. rewrite fin_map_app_def.
reflexivity. assumption. assumption.
intro h2. subst.
apply restriction_extends_compat.
Qed.


Lemma not_extends_compat : 
  forall {T U:Type} {A A':Ensemble T} {B:Ensemble U} {def:U} 
         (F':Fin_map A' B def) (F:Fin_map A B def)
    (pf:Included A A'), ~ extends F' F -> 
    (restriction F' pf) <> F.
intros T U A A' B def F' F pf h1.
intro h2.
contradict h1.
rewrite <- restriction_extends_compat_iff in h2.
assumption.
Qed.


Lemma not_extends_differs_at_point : 
  forall {T U:Type} {A A':Ensemble T} {B:Ensemble U} {def:U} 
         (F':Fin_map A' B def) (F:Fin_map A B def),
    Included A A' -> ~ extends F' F -> 
    exists x:T, In A x /\ F |-> x <> F' |-> x.
intros T U A A' B def f1 f2 h1 h2.
pose proof (not_extends_compat _ _ h1 h2) as h3.
pose proof (fin_map_app_inj (restriction f1 h1) f2) as h0.
assert (h3':fin_map_app (restriction f1 h1) <> fin_map_app f2).
    tauto. 
pose proof (distinct_functions_differ_at_point _ _ h3') as h4.
destruct h4 as [x h4].
exists x.
split.
destruct (classic_dec (In A x)) as [h5 | h6].  assumption.
rewrite fin_map_app_def in h4. rewrite fin_map_app_def in h4.
contradict h4. reflexivity. assumption. assumption. 
destruct (classic_dec (In A x)) as [h5 | h6].  
rewrite restriction_compat in h4. 
apply neq_sym in h4.
assumption.  assumption.
rewrite fin_map_app_def in h4. rewrite fin_map_app_def in h4.
contradict h4. reflexivity.  assumption. assumption.
Qed.



Lemma fin_map_to_fps_ext : forall {T U:Type} {A:Ensemble T}
                                  {B C:Ensemble U} {def:U}
                                  (F:Fin_map A B def) 
                                  (G:Fin_map A C def),
                             (forall x:T, F |-> x = G |-> x) ->
                             fin_map_to_fps F =
                             fin_map_to_fps G.
intros T U A B C def F G h1.
pose proof (fin_map_to_fps_compat F) as h2.
pose proof (fin_map_to_fps_compat G) as h3.
destruct h2 as [h2a h2b].
destruct h3 as [h3a h3b].
pose proof (fps_to_f_inj' A B C _ _ def h2a h3a) as h4.
apply h4.
rewrite h2b. rewrite h3b.
apply functional_extensionality.
assumption.
Qed.


Lemma fun_to_fin_map_sing_im : 
  forall {T U:Type} (A:Ensemble T) (def val:U) (pfa:Finite A),
         fin_map_eq (fun_to_fin_map A def pfa (fun x:T=>val)) 
                    (fin_map_sing A pfa def val).
intros T U A def val h1.
red.
assert (h2:Included (Im A (fun _ : T => val)) (Singleton val)).
  red. intros u h2.
  destruct h2 as [x h2 u]. subst. constructor.
exists h2.
apply fin_map_to_fps_inj.
apply fin_map_to_fps_ext.
intro x.
rewrite <- fin_map_new_ran_compat.
unfold fin_map_sing.
destruct (classic_dec (In A x)) as [h3 | h4].
pose proof (fun_to_fin_map_compat A def h1 (fun _ : T => val) x h3) as h4.
rewrite h4.
unfold fin_map_app.
unfold fps_to_f.
destruct (classic_dec (In A x)) as [h5 | h6].
destruct (fp_cart_prod_sing A val) as [h7 h8].
destruct constructive_definite_description as [u h10].
simpl.
destruct h10 as [h10].
destruct h10; subst. reflexivity.
contradiction.
pose proof (fin_map_app_def (fun_to_fin_map A def h1 (fun _ : T => val)) _ h4) as h5.
rewrite h5.
pose proof (fin_map_app_def ( fin_map_intro A (Singleton val) def h1 (Singleton_is_finite U val)
     (cart_prod A (Singleton val)) (fp_cart_prod_sing A val)) _ h4) as h6.
rewrite h6.
reflexivity.
Qed.

Lemma fin_map_sing_const : 
  forall {T U:Type} (A:Ensemble T) (def val:U) (pfa:Finite A)
         (x:T), In A x ->
                (fin_map_sing A pfa def val) |-> x = val.
intros T U A def val h1 x h0.
unfold fin_map_app.
destruct (fin_map_sing A h1 def val) as [h2 h3 S h4].
unfold fps_to_f.
destruct (classic_dec (In A x)) as [h5 | h6].
destruct h4 as [h4a h4b].
destruct constructive_definite_description as [u h8].
simpl.
destruct h8 as [h8a h8b].
destruct h8a. reflexivity.
contradiction.
Qed.


Lemma im_fin_map_sing : forall {T U:Type} (A:Ensemble T) 
                               (def val:U)
                               (pf:Finite A),
                          A <> Empty_set _ ->
                          im_fin_map (fin_map_sing A pf def val) = Singleton val. 
intros T U A def val h1 h0.
unfold im_fin_map.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red. 
intros x h2.
destruct h2 as [x h2 y h3].
rewrite h3. 
rewrite fin_map_sing_const.
constructor.
assumption.
(* >= *)
red.
intros x h2.
destruct h2; subst.
pose proof h1 as h2.
induction h2 as [|A h2 h3 x h4].
contradict h0. reflexivity.
apply Im_intro with x. right. constructor.
rewrite fin_map_sing_const. 
reflexivity.
right. constructor.
Qed.


Lemma fin_map_to_fps_compat_s : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U) (def:U)
         (pfa:Finite A) (pfb:Finite B) (S:Ensemble (T*U))
         (pffp:functionally_paired A B S),
    S = fin_map_to_fps (fin_map_intro A B def pfa pfb S pffp).
intros T U A B def h1 h2 S h3.
unfold fin_map_to_fps.
destruct constructive_definite_description as [S' h4].
simpl.
destruct h4 as [h4 h5].
rewrite (fin_map_app_s_compat def h1 h2 _ _ h3 h4).
apply fin_map_ext.
intro x.
rewrite <- h5.
unfold fin_map_app.
reflexivity.
Qed.


Lemma fin_map_ext_in : forall {T U:Type} {A:Ensemble T} 
                           {B:Ensemble U} {def:U}
                           (f1 f2:Fin_map A B def),
                      (forall x:T, In A x -> f1 |-> x = f2 |-> x) ->
                                  f1 = f2.
intros T U A B def f1 f2 h1.
apply fin_map_to_fps_inj.
apply Extensionality_Ensembles.
red. split.
red. intros pr h2.
destruct f1 as [h3 h4 S h6].
rewrite <- fin_map_to_fps_compat_s in h2.
pose proof (fp_in_dom _ _ _ h6) as h7.
specialize (h7 _ h2).
apply h1 in h7.
destruct f2 as [h8 h9 S' h10].
rewrite <- fin_map_to_fps_compat_s.
simpl in h7.
rewrite (fps_to_f_compat h10 def).
constructor.
rewrite (fps_to_f_compat  h6 def) in h2.
destruct h2 as [h2]. destruct h2 as [h2l h2r]. rewrite <- h7.
split.
assumption. assumption.
red. intros pr h2.
destruct f2 as [h3 h4 S' h6].
rewrite <- fin_map_to_fps_compat_s in h2.
pose proof (fp_in_dom _ _ _ h6) as h7.
specialize (h7 _ h2).
apply h1 in h7.
destruct f1 as [h8 h9 S h10].
rewrite <- fin_map_to_fps_compat_s.
simpl in h7.
rewrite (fps_to_f_compat h10 def).
constructor.
rewrite (fps_to_f_compat  h6 def) in h2.
destruct h2 as [h2]. destruct h2 as [h2l h2r]. rewrite h7.
split.
assumption. assumption.
Qed.



Lemma sig_fun_to_fin_map_ex :
  forall {T U:Type} {A:Ensemble T} 
         (f:sig_set A->U),
    Finite A -> forall def:U,
    exists! F:Fin_map A (Im (full_sig A) f) def,
      forall (x:T) (pf:In A x),
        F |-> x = f (exist _ _ pf).
intros T U A f h0 def.
pose [pr:T*U | exists pf:(In A (fst pr)),
                         f (exist _ _ pf) = snd pr] as S.
assert (h1:functionally_paired A (Im (full_sig A) f) S).
  constructor.
  intros x h1.
  exists (f (exist _ _ h1)).
  red. split. split.
  apply Im_intro with (exist _ _ h1). constructor. reflexivity.
  constructor. simpl. exists h1. reflexivity.
  intros x' h2. destruct h2 as [h2a h2b].
  inversion h2b as [h3]. simpl in h3.
  destruct h3 as [h3 h4].
  assert (h3 = h1). apply proof_irrelevance. subst.
  reflexivity.
  intros pr h1.
  destruct h1 as [h1].
  destruct h1 as [h1 h2].
  split; auto. rewrite <- h2.
  apply Im_intro with (exist _ _ h1).
  constructor. reflexivity.
pose proof h0 as h0'.
rewrite finite_full_sig_iff in h0'.
pose proof (finite_image _ _ _ f h0') as h2.
exists (fin_map_intro _ _ _ h0 h2 S h1).
red. split.
intros x h3.
simpl. 
pose proof (fps_to_f_s_compat h1 def _ h3) as h4. simpl in h4. 
inversion h4 as [h5].
destruct h5 as [h5 h6]. simpl in h6, h5.
assert (h7:h3 = h5). apply proof_irrelevance. subst.
rewrite h6. reflexivity. 
intros F h3.
apply fin_map_ext_in.
intros x h4.
specialize (h3 _ h4).
rewrite h3. simpl.
pose proof (fps_to_f_s_compat h1 def _ h4) as h5.
inversion h5 as [h6]. clear h5.
simpl in h6. destruct h6 as [h6 h7].
assert (h4 = h6). apply proof_irrelevance. subst.
rewrite h7.
reflexivity.
Qed.


Definition sig_fun_to_fin_map 
      {T U:Type} {A:Ensemble T} 
      (f:sig_set A->U)
      (pf:Finite A) (def:U) :=
  proj1_sig (constructive_definite_description _ (sig_fun_to_fin_map_ex f pf def)).

Lemma sig_fun_to_fin_map_compat :
  forall {T U:Type} {A:Ensemble T} 
         (f:sig_set A->U)
         (pf:Finite A) (def:U),
    let F:= sig_fun_to_fin_map f pf def in
    forall (x:T) (pf:In A x),
        F |-> x = f (exist _ _ pf).
intros T U A f h1 def F.
unfold F, sig_fun_to_fin_map.
destruct constructive_definite_description as [F' h2].
simpl.
assumption.
Qed.

Lemma sig_fun_to_fin_map_ran_ex :
  forall {T U:Type} {A:Ensemble T} 
         (f:sig_set A->U),
    Finite A -> 
    forall (def:U)
           (B:Ensemble U),
      Finite B ->
      Included (Im (full_sig A) f) B ->
    exists! F:Fin_map A B def,
      forall (x:T) (pf:In A x),
        F |-> x = f (exist _ _ pf).
intros T U A f h1 def B h0 h2.
pose (sig_fun_to_fin_map f h1 def) as F.
pose (fin_map_new_ran F h0 h2) as F'.
pose proof (fin_map_new_ran_compat F h0 h2) as h4. simpl in h4.
pose proof (sig_fun_to_fin_map_compat f h1 def) as h5.
exists F'.
red. split.
intros x h3.
unfold F'.
specialize (h4 x).
rewrite <- h4.
specialize (h5 _ h3). unfold F.
assumption.
intros G h3.
apply fin_map_ext_in.
intros x h6.
specialize (h3 _ h6).
rewrite h3. unfold F'.
specialize (h5 _ h6). specialize (h4 x).
rewrite <- h4. unfold F.
assumption.
Qed.


Definition sig_fun_to_fin_map_ran
  {T U:Type} {A:Ensemble T} 
  (f:sig_set A->U) (pfa:Finite A) (def:U)
  (B:Ensemble U) (pfb:Finite B)
  (pfi:Included (Im (full_sig A) f) B) :=
  proj1_sig (constructive_definite_description _ (sig_fun_to_fin_map_ran_ex f pfa def B pfb pfi)).


Lemma sig_fun_to_fin_map_ran_compat :
  forall {T U:Type} {A:Ensemble T} 
         (f:sig_set A->U) (pfa:Finite A)
         (def:U) (B:Ensemble U) (pfb:Finite B)
         (pfi:Included (Im (full_sig A) f) B),
    let F := sig_fun_to_fin_map_ran f pfa def B pfb pfi in
      forall (x:T) (pf:In A x),
        F |-> x = f (exist _ _ pf).
intros T U A f h1 def B h2 h3 F.
unfold F, sig_fun_to_fin_map_ran.
destruct constructive_definite_description as [F' h4].
simpl.
assumption.
Qed.

Lemma fin_map_compose_inj_ex :
  forall {T U V:Type} {A:Ensemble T} {B:Ensemble V}
         {defv:V} (F:(Fin_map A B defv)) (deft:T) (g:T->U),
    injective g ->
    exists! F':Fin_map (Im A g) B defv,
      forall x:T, In A x ->
                  F |-> x = F' |-> (g x).
intros T U V A B defv F deft g h1.
pose proof (incl_im_im_left_inverse_compose _ h1 A deft 
                                            (fin_map_app F)) as h2.
pose proof (im_fin_map_ran F) as h3.
assert (h4:Included  (Im (Im A g)
            (fun u : U => F |-> ((left_inverse g h1 A, deft) ||-> u))) B). auto with sets.
exists (fin_map_new_ran (fun_to_fin_map (Im A g) defv (finite_image _ _ _ g (fin_map_fin_dom F)) 
                       (fun u => F |-> (((left_inverse _ h1 A), deft) ||-> u))) (fin_map_fin_ran F) h4).
red. split.
intros x h5.
rewrite <- fin_map_new_ran_compat.
rewrite fun_to_fin_map_compat.
simpl. unfold sig_fun_app.
destruct classic_dec as [h6 | h7].
assert (h7:h6 = Im_intro T U A g x h5 (g x) eq_refl). apply proof_irrelevance.
subst.
rewrite left_inverse_compat.
reflexivity.
contradict h7. apply Im_intro with x; auto.
apply Im_intro with x; auto.
intros F' h5.
apply fin_map_ext_in.
intros u h6.
destruct h6 as [x h6]. subst.
rewrite <- fin_map_new_ran_compat.
rewrite fun_to_fin_map_compat.
rewrite <- h5; auto. f_equal.
simpl. unfold sig_fun_app.
destruct classic_dec as [h7 | h8].
assert (h9:h7 = Im_intro T U A g x h6 (g x) eq_refl). apply proof_irrelevance. subst.
rewrite left_inverse_compat.
reflexivity.
contradict h8.
apply Im_intro with x; auto.
apply Im_intro with x; auto.
Qed.

Definition fin_map_compose_inj 
  {T U V:Type} {A:Ensemble T} {B:Ensemble V}
  {defv:V} (F:(Fin_map A B defv)) (deft:T) (g:T->U)
  (pf:injective g) : Fin_map (Im A g) B defv :=
  proj1_sig (constructive_definite_description _ (fin_map_compose_inj_ex F deft g pf)).

Lemma fin_map_compose_inj_compat :
  forall {T U V:Type} {A:Ensemble T} {B:Ensemble V}
         {defv:V} (F:(Fin_map A B defv)) (deft:T) (g:T->U)
         (pf:injective g),
    let F' := fin_map_compose_inj F deft g pf in
      forall x:T, In A x ->
                  F |-> x = F' |-> (g x).
intros T U V A B defv F deft g pf F'.
unfold F'. unfold fin_map_compose_inj.
destruct constructive_definite_description as [K h2]. simpl.
assumption.
Qed.





Lemma fin_map_app_compat : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U} (F:(Fin_map A B def)) (x:T),
    F |-> x =
    fps_to_f _ (fp_fin_map_to_fps F) def x.
intros T U A B def F x. 
unfold fin_map_app. 
destruct F as [h1 h2 S h3].
pose proof (fin_map_to_fps_compat_s _ _ def h1 h2 _ h3) as h4.
pose (fun pr:{S:(Ensemble (T*U)) | functionally_paired A B S} =>
        fps_to_f (proj1_sig pr) (proj2_sig pr) def x) as f.
pose proof (subsetT_eq_compat _ _ _ _ h3 (fp_fin_map_to_fps (fin_map_intro A B def h1 h2 S h3)) h4) as h6.
pose proof (existTexist _ _ _ _ _ _ h6) as h7. clear h6.
pose proof (f_equal f h7) as h8.
unfold f in h8. simpl in h8.
assumption.
Qed.


Lemma fin_map_to_fps_dom_rel : 
  forall {T U:Type} {A:Ensemble T}
         {B:Ensemble U} {def:U}
         (F:Fin_map A B def), 
    dom_rel (fin_map_to_fps F) = A.
intros T U A B def F.
rewrite dom_rel_eq.
destruct F as [h1 h2 S h3].
apply Extensionality_Ensembles.
red. split.
red.
intros x h4.
destruct h4 as [pr h4 x h5]. subst.
rewrite <- fin_map_to_fps_compat_s in h4.
destruct h3 as [h3l h3r].
specialize (h3r _ h4).
destruct h3r; auto.
red.
intros x h4.
destruct h3 as [h3l h3r].
pose proof h3l as h3l'.
specialize (h3l' _ h4).
destruct h3l' as [y h3l'].
apply Im_intro with (x, y).
rewrite <- fin_map_to_fps_compat_s.
red in h3l'.
destruct h3l' as [h3a h3b].
destruct h3a; auto. auto.
Qed.


Lemma fin_map_to_fps_ran_rel : 
  forall {T U:Type} {A:Ensemble T}
         {B:Ensemble U} {def:U}
         (F:Fin_map A B def), 
    ran_rel (fin_map_to_fps F) = Im A (fin_map_to_fun F).
intros T U A B def F.
unfold ran_rel.
apply Extensionality_Ensembles.
red. split.
red. intros y h1.
destruct h1 as [h1].
destruct h1 as [x h1].
apply Im_intro with x. 
pose proof (fp_fin_map_to_fps F) as h2.
destruct h2 as [h2l h2r].
specialize (h2r _ h1).
simpl in h2r.
destruct h2r; auto.
unfold fin_map_to_fun. destruct F as [h2 h3 S h4].
rewrite <- fin_map_to_fps_compat_s in h1.
pose proof (fps_to_f_compat h4 def) as h5.
rewrite h5 in h1.
destruct h1 as [h1]. simpl in h1. destruct h1; auto.
red.
intros y h1.
constructor.
destruct h1 as [x h1 y h2].
subst.
exists x.
unfold fin_map_to_fun. destruct F as [h2 h3 S h4].
pose proof (fps_to_f_compat h4 def) as h5.
rewrite <- fin_map_to_fps_compat_s.
rewrite h5 at 1.
constructor.
simpl. split; auto.
Qed.

Lemma fp_fin_map_to_fps' : 
  forall {T U:Type} {A:Ensemble T}
         {B:Ensemble U} {def:U}
         (F:Fin_map A B def),
    functionally_paired 
      (dom_rel (fin_map_to_fps F)) (ran_rel (fin_map_to_fps F))
      (fin_map_to_fps F).
intros T U A B def F.
destruct F as [h1 h2 S h3].
rewrite fin_map_to_fps_dom_rel.
rewrite fin_map_to_fps_ran_rel.
rewrite <- fin_map_to_fps_compat_s.
constructor.
intros x h4.
inversion h3 as [h3l h3r].
pose proof (h3l _ h4) as h3l'.
destruct h3l' as [y h3l'].
red in h3l'.
exists y. red.
split. split.
apply Im_intro with x. assumption.
simpl.
destruct h3l' as [h3a h3b].
destruct h3a as [h5 h6].
rewrite (fps_to_f_compat h3 def) in h6.
destruct h6 as [h6].
simpl in h6.
destruct h6; auto.
destruct h3l' as [[]]; auto.
intros y' h5.
destruct h5 as [h5l h5r].
specialize (h3r _ h5r).
simpl in h3r. destruct h3r as [h6 h7].
destruct h3l' as [h3a h3b].
specialize (h3b _ (conj h7 h5r)).
assumption.
intros pr h4.
inversion h3 as [h5 h6].
split.
specialize (h6 _ h4).
destruct h6; auto.
apply Im_intro with (fst pr).
specialize (h6 _ h4). destruct h6; auto.
simpl.
rewrite (fps_to_f_compat h3 def) in h4.
destruct h4 as [h4].
destruct h4; auto.
Qed.



Lemma fin_map_to_fps_fin_map_app_compat : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U} (F:Fin_map A B def) (pr:T*U),
    In (fin_map_to_fps F) pr ->
    F |-> (fst pr) = snd pr.
intros T U A B ef F pr h1.
pose proof (fin_map_to_fps_compat F) as h2.
destruct h2 as [h2 h3].
unfold fin_map_app.
destruct F. 
rewrite <- fin_map_to_fps_compat_s in h1.
apply fin_map_s_compat.
assumption.
Qed.


Lemma fin_map_app_incl_fps : 
  forall {T U:Type} {A A':Ensemble T} {B B':Ensemble U} {def:U}
         (F:Fin_map A B def) (F':Fin_map A' B' def),
    Included (fin_map_to_fps F) (fin_map_to_fps F') ->
    forall x:T, Ensembles.In A x -> F |-> x = F' |-> x.
intros T U A A' B B' def F F' h1 x h2.
unfold fin_map_app.
destruct F as [h3 h4 S h5]; destruct F' as [h6 h7 S' h8]. 
do 2 rewrite <- fin_map_to_fps_compat_s in h1.
rewrite (fps_to_f_compat h5 def) in h1.
rewrite (fps_to_f_compat h8 def) in h1.
specialize (h1 (x, fps_to_f S h5 def x)).
assert (h9: Ensembles.In
         [pr : T * U
         | snd pr = fps_to_f S h5 def (fst pr) /\ Ensembles.In A (fst pr)]
         (x, fps_to_f S h5 def x)).
  constructor. simpl. split; auto.
specialize (h1 h9).
inversion h1 as [h10]. simpl in h10.
destruct h10; auto.
Qed.



Lemma fin_map_app_fin_map_to_fps_compat : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U} (F:Fin_map A B def) (pr:T*U),
    In A (fst pr) ->
    F |-> (fst pr) = snd pr ->
    In (fin_map_to_fps F) pr.
intros T U A B def F pr h0 h1.
destruct F as [h2 h3 S h4].
rewrite <- fin_map_to_fps_compat_s.
unfold fin_map_app in h1.
rewrite (fps_to_f_compat h4 def).
constructor.
symmetry in h1.
split; auto.
Qed.


Lemma fin_map_to_fps_finite : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U}
         (F:Fin_map A B def),
    Finite (fin_map_to_fps F).
intros T U A B def F.
destruct F as [h1 h2 S h3].
rewrite <- fin_map_to_fps_compat_s.
pose proof (fps_inc_cart_prod h3) as h4.
pose proof (cart_prod_fin _ _ h1 h2) as h5.
apply Finite_downward_closed with (cart_prod A B); assumption.
Qed.



Lemma fps_to_f_undoes_fin_map_to_fps : 
  forall {T U:Type} {def:U} {A:Ensemble T} {B:Ensemble U}
         (F:Fin_map A B def),
    let S := fin_map_to_fps F in 
              forall (pf:functionally_paired A B S)
                     (x:T), fps_to_f _ pf def x = F |-> x. 
intros T U def A B F S h0 x.
pose proof (fin_map_to_fun_compat def F) as h1.
rewrite  h1.
unfold fin_map_to_fun.
destruct F as [h2 h3 S' h4].
pose proof (fin_map_to_fps_compat_s A B def h2 h3 _ h4) as h5.
unfold S.
symmetry in h5.
pose proof (subsetT_eq_compat _ _ _ _ h0 h4 h5) as h6.
dependent rewrite -> h6.
reflexivity.
Qed.

Lemma fin_map_to_fps_new_ran_compat : 
  forall {T U:Type} (def:U) (A:Ensemble T) (B C:Ensemble U)
         (F:Fin_map A B def) (pfc:Finite C) (pfinc:Included B C),
  fin_map_to_fps (fin_map_new_ran F pfc pfinc) =
         fin_map_to_fps F.

intros T U def A B C F h1 h2.
pose proof (fin_map_to_fps_compat F) as h3.
destruct h3 as [h3a h3b].
pose proof (fin_map_to_fps_compat (fin_map_new_ran F h1 h2)) as h4.
destruct h4 as [h4a h4b].
symmetry.
apply (fps_to_f_inj' A B C _ _ def h3a h4a).
apply functional_extensionality.
intro x.
do 2 rewrite (fps_to_f_undoes_fin_map_to_fps).
apply fin_map_new_ran_compat.
Qed.


Lemma fin_map_sing_ran_ex : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U)
         (pfa:Finite A) (pfb:Finite B) 
         (def val:U) (pfin:Ensembles.In B val),
  exists! F:Fin_map A B def,
    forall x:T, Ensembles.In A x -> F |-> x = val.
intros T U A B h1 h2 def val h3.
pose (fin_map_sing _ h1 def val) as F.
assert (h4:Included (Singleton val) B). red. intros x h4.
  destruct h4; subst. assumption.
pose (fin_map_new_ran F h2 h4) as F'.
exists F'.
red. split.
intros x h5. unfold F'. 
rewrite <- (fin_map_new_ran_compat F h2 h4). 
unfold F. rewrite fin_map_sing_const. reflexivity. assumption.
intros G h5.
apply fin_map_to_fps_inj.
unfold F'. rewrite fin_map_to_fps_new_ran_compat.
unfold F. unfold fin_map_sing.
rewrite <- fin_map_to_fps_compat_s.
apply Extensionality_Ensembles.
red. split.
red.
intros pr h6.
destruct h6 as [h6]. destruct h6 as [h6l h6r]. 
rewrite surjective_pairing.
inversion h6r as [h9]. rewrite <- h9.
rewrite <- (h5 _ h6l).
apply in_fin_map_to_fps. assumption.
red. intros pr h6.
destruct G as [h7 h8 S h9].
rewrite <- fin_map_to_fps_compat_s in h6.
pose proof (fp_in_dom _ _ _ h9 _ h6) as h10.
rewrite surjective_pairing. constructor. simpl; split. assumption.
apply h5 in h10. simpl in h10.
rewrite <- h10.
rewrite fin_map_s_compat; auto. constructor.
Qed.


Definition fin_map_sing_ran 
           {T U:Type} (A:Ensemble T) (B:Ensemble U)
           (pfa:Finite A) (pfb:Finite B) 
           (def val:U) (pfin:Ensembles.In B val) : Fin_map A B def :=
  proj1_sig (constructive_definite_description _ (fin_map_sing_ran_ex A B pfa pfb def val pfin)).

Lemma fin_map_sing_ran_compat :
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U)
         (pfa:Finite A) (pfb:Finite B) 
         (def val:U) (pfin:Ensembles.In B val),
    let F := fin_map_sing_ran A B pfa pfb def val pfin in
    forall x:T, Ensembles.In A x -> F |-> x = val.
intros T U A B h1 h2 def val h3 F x h4.
unfold F. unfold fin_map_sing_ran. destruct constructive_definite_description as [F' h5]. simpl.
apply h5. assumption.
Qed.


Lemma fin_map_im_full_sig_eq_ex : 
  forall {T U V:Type} {A:Ensemble T} {g:sig_set A->U} {B:Ensemble V}
         {defv:V}  (F:Fin_map (Im (full_sig A) g) B defv),
    Finite A -> 
    forall defu:U, 
       exists! F':Fin_map A B defv,
        forall x:T,
          In A x ->
          F' |-> x = F |-> ((g, defu) ||-> x).
intros T U V A g B defv F h1 defu.
pose (fun x:T => (F|-> ((g, defu) ||-> x))) as f. 
pose (fun_to_fin_map A defv h1 f) as G. 
assert (h2:Included (Im A f) B).
  red.
  intros b h2.
  destruct h2 as [b h2].  subst.
  unfold f. simpl. 
  unfold sig_fun_app. 
  destruct (classic_dec (In A b)) as [h6 | h7].
  apply fin_map_app_in. 
  apply Im_intro with (exist _ _ h6). constructor.
  reflexivity. contradiction.
pose proof (fin_map_fin_ran F) as h3.
pose (fin_map_new_ran G h3 h2) as G'.
exists G'.
red. split. 
intros x h0. simpl. unfold  sig_fun_app.
destruct (classic_dec (In A x)) as [h4 | h5].
rewrite <- (fin_map_new_ran_compat G h3 h2).
unfold G. unfold f. simpl.
rewrite fun_to_fin_map_compat.
unfold sig_fun_app. 
destruct (classic_dec (In A x)) as [h5 | h6].
assert (h4 = h5). apply proof_irrelevance. subst. reflexivity.
contradiction. assumption. contradiction.
intros K h4.
unfold G'.
apply fin_map_ext_in.
intros x h5. specialize (h4 _ h5). rewrite h4.
rewrite <- (fin_map_new_ran_compat G h3 h2).
unfold G, f. rewrite fun_to_fin_map_compat. reflexivity.
assumption.
Qed.


Definition fin_map_im_full_sig_eq  
           {T U V:Type} {A:Ensemble T} {g:sig_set A->U} {B:Ensemble V}
           {defv:V}  (F:Fin_map (Im (full_sig A) g) B defv)
           (pf:Finite A) (defu:U) : Fin_map A B defv :=
  proj1_sig (constructive_definite_description _ (fin_map_im_full_sig_eq_ex F pf defu)).

Lemma fin_map_im_full_sig_eq_compat : 
  forall {T U V:Type} {A:Ensemble T} {g:sig_set A->U} {B:Ensemble V}
         {defv:V}  (F:Fin_map (Im (full_sig A) g) B defv)
         (pf:Finite A) (defu:U),
    let F' := fin_map_im_full_sig_eq F pf defu in 
    forall x:T,
      In A x ->
      F' |-> x = F |-> ((g, defu) ||-> x).
intros T U V A g B defv F h1 defu F'.
unfold F'. unfold fin_map_im_full_sig_eq.
destruct constructive_definite_description as [G h3].
simpl.
assumption.
Qed.



Lemma fin_map_to_fin_map_im_full_sig_comp : 
  forall {T U V:Type} {A:Ensemble T} {B:Ensemble V}
         {defv:V} (F:Fin_map A B defv)
         (g:sig_set A->U) (defu:U),
    injective g ->
    exists (F':(Fin_map (Im (full_sig A) g) B defv)),
      forall x:T, 
        In A x ->
        F' |-> ((g, defu) ||-> x) =  F |-> x.
intros T U V A B defv F g defu h1.
assert (h3:forall u:U, In (Im (full_sig A) g) u -> exists! x, u = g x).
  intros u h3. destruct h3 as [u h3]. subst.
  exists u; auto. 
  red. split. reflexivity.
  intros x h4. apply h1. assumption.
pose (fun x:sig_set (Im (full_sig A) g) => (proj1_sig (proj1_sig (constructive_definite_description _ (h3 _ (proj2_sig x)))))) as f.
pose (fun x:sig_set (Im (full_sig A) g) => F |-> (f x)) as k.
assert (h4:Included (Im (full_sig (Im (full_sig A) g)) k) B).
  red. intros x h4.
  destruct h4 as [x h4]. subst. destruct x as [x h5].
  destruct h5 as [x h5]. subst. unfold k.
  apply fin_map_app_in. unfold f.
  simpl.
  destruct constructive_definite_description as [a h6].
  simpl. apply proj2_sig.
pose proof (fin_map_fin_dom F) as h5.
pose proof (fin_map_fin_ran F) as h6.
rewrite finite_full_sig_iff in h5.
pose proof (finite_image _ _ _ g h5) as h7.
pose (sig_fun_to_fin_map_ran k h7 defv B h6 h4) as F'.
exists F'.
intros x h8. unfold F'.
pose proof (sig_fun_to_fin_map_ran_compat k h7 defv B h6 h4) as h9. simpl in h9.
assert (h10:In (Im (full_sig A) g) ((g, defu) ||-> x)).
  apply Im_intro with (exist _ _ h8). constructor.
  simpl. unfold sig_fun_app. destruct (classic_dec (In A x)) as [hi | hni].
  assert (h8 = hi). apply proof_irrelevance.
  subst. reflexivity.
  contradiction.
specialize (h9 _ h10). 
rewrite h9.  
unfold k. simpl. 
unfold f. 
f_equal.
destruct constructive_definite_description as [x' h11].
simpl. simpl in h11.
unfold sig_fun_app in h11.
destruct (classic_dec (In A x)) as [h12 | h13].
apply h1 in h11.
destruct x' as [x' h13].
simpl. apply exist_injective in h11. subst. reflexivity.
contradiction.
Qed.



Lemma cart_prod_subsets_finite : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U),
    Finite A -> Finite B -> 
    FiniteT {S:Ensemble (T*U) | Included S (cart_prod A B)}.
intros T U A B h1 h2.
apply power_set_finitet.
apply cart_prod_fin; assumption.
Qed.


Lemma finitet_fin_maps : forall {T U:Type} (A:Ensemble T)
                                (B:Ensemble U) (def:U),
                           Finite A -> Finite B ->
                           FiniteT (Fin_map A B def).
intros T U A B def h1 h2.
pose (@fin_map_to_fps T U A B def) as f.
assert (h4:forall F:(Fin_map A B def), Included (f F) (cart_prod A B)). 
  intro F.
  apply fps_inc_cart_prod. 
  pose proof (fin_map_to_fps_compat F) as h5.
  destruct h5.
  unfold f.
  assumption.
pose (fun F:(Fin_map A B def) => 
        exist 
          (fun S:Ensemble (T*U) => Included S (cart_prod A B))  
          (f F) (h4 F)) as f'.
pose proof (cart_prod_fin _ _ h1 h2) as h5.
pose proof (power_set_finitet _ h5) as h6.
assert (h7:injective f').
  red.
  intros F1 F2 h8.
  unfold f' in h8.
  unfold f in h8.
  pose proof (exist_injective _ _ _ _ _ h8) as h9.
  apply fin_map_to_fps_inj. assumption.
assert (h8: (forall y : {S:Ensemble (T*U) | Included S (cart_prod A B)}, (exists x, f' x = y) \/ ~ (exists x, f' x = y))).
  intros; tauto.
apply (inj_finite _ _ f' h6 h7 h8).
Qed.

Lemma finite_fin_maps : forall {T U:Type} (A:Ensemble T)
                               (B:Ensemble U) (def:U),
                          Finite A -> Finite B ->
                          Finite (Full_set (Fin_map A B def)).
intros T U A B def h1 h2.
pose proof (finitet_fin_maps A B def h1 h2).
apply FiniteT_Finite.  assumption.
Qed.


Lemma finite_fin_maps_squared : forall {T U:Type} (A:Ensemble T)
                                       (B:Ensemble U) (def:U),
                                  Finite A -> Finite B ->
                                  Finite (Full_set ((Fin_map A B def) * (Fin_map A B def))).
intros T U A B def h1 h2.
pose proof (finite_fin_maps A B def h1 h2) as h3.
assert (h4:Full_set (Fin_map A B def * Fin_map A B def) = 
           cart_prod (Full_set (Fin_map A B def)) (Full_set (Fin_map A B def))).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x ?. constructor. split; constructor.
  red. split.
rewrite h4.
apply cart_prod_fin; auto.
Qed.


Lemma finite_fin_map_ens : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U}
    (S:Ensemble (Fin_map A B def)), 
    Finite A -> Finite B -> Finite S.
intros T U A B def S h1 h2.
assert (h0:Included S (Full_set (Fin_map A B def))). red.
  intros; constructor.
apply Finite_downward_closed with (A:=Full_set (Fin_map A B def)).
apply finite_fin_maps; auto.
assumption.
Qed.


Lemma finite_fin_map_squared_ens :   forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U}
    (S:Ensemble ((Fin_map A B def) * (Fin_map A B def))),
    Finite A -> Finite B -> Finite S.
intros T U A B def S h1 h2.
assert (h0:Included S (Full_set ((Fin_map A B def) * (Fin_map A B def)))). red; intros; constructor.
apply Finite_downward_closed with (A:=Full_set (Fin_map A B def * Fin_map A B def)).
apply finite_fin_maps_squared; auto.
assumption.
Qed.


Lemma fin_map_empty1 : forall {T U:Type} {B:Ensemble U} {def:U}
                              (F:Fin_map (Empty_set T) B def),
                         fin_map_to_fps F = Empty_set _.
intros T U B def f.
pose proof (fin_map_to_fps_compat f) as h1.
destruct h1 as [h1 h2].
apply Extensionality_Ensembles.
red; split; auto with sets.
red.
intros pr h3.
destruct h1 as [h4 h5].
pose proof (h5 _ h3) as h6.
destruct h6; contradiction.
Qed.


Lemma fin_map_cart_empty11 : 
  forall {T U V:Type} {A:Ensemble U} {B:Ensemble V} {def:V}
                              (F:Fin_map (cart_prod (Empty_set T) A) B def),
                         fin_map_to_fps F = Empty_set _.
intros T U V A B def F.
pose proof (fin_map_to_fps_compat F) as h1.
destruct h1 as [h1 h2].
apply Extensionality_Ensembles.
red; split; auto with sets.
red. intros pr h3.
destruct h1 as [h4 h5].
pose proof (h5 _ h3) as h6.
destruct h6 as [h6l h6r]. 
rewrite cart_prod_empty in h6l. 
contradiction.
Qed.


Lemma fin_map_cart_empty21 : forall {T U V:Type} {A:Ensemble T} {B:Ensemble V} {def:V}
                              (F:Fin_map (cart_prod A (Empty_set U)) B def),
                         fin_map_to_fps F = Empty_set _.
intros T U V A B def F.
pose proof (fin_map_to_fps_compat F) as h1.
destruct h1 as [h1 h2].
apply Extensionality_Ensembles.
red; split; auto with sets.
red. intros pr h3.
destruct h1 as [h4 h5].
pose proof (h5 _ h3) as h6.
destruct h6 as [h6l h6r]. 
rewrite cart_prod_empty' in h6l. 
contradiction.
Qed.


Lemma fin_map_empty2 : forall {T U:Type} {A:Ensemble T} {def:U}
                              (F:Fin_map A (Empty_set U) def),
                         fin_map_to_fps F = Empty_set _.
intros T U A def f.
pose proof (fin_map_to_fps_compat f) as h1.
destruct h1 as [h1 h2].
inversion f as [h3 h4 S h5].
induction h3 as [|A h6 h7 x h8].
(*Empty*)
destruct h1 as [h9 h10].
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros pr h11.
pose proof (h10 _ h11) as h12.
destruct h12; contradiction.
(* >= *)
auto with sets.
(* Add *)
destruct h1 as [h9 h10].
pose proof (h9 x (Add_intro2 _ A x)) as h11.
destruct h11 as [y h12].
red in h12.
destruct h12 as [h12].
destruct h12; contradiction.
Qed.


Lemma fp_empty1 : forall (T U:Type) (B:Ensemble U),
                        functionally_paired (Empty_set T) 
                                            B (Empty_set (T*U)).
intros; constructor; intros; contradiction.
Qed.

 
Lemma fp_cart_empty11 : forall {T U V:Type} (A:Ensemble U) (B:Ensemble V),
                          functionally_paired (cart_prod (Empty_set T) A)
                                              B (Empty_set (T*U*V)).
intros T U V A B.
constructor. intros ? h1. destruct h1 as [h1].
destruct h1; contradiction.
intros; contradiction.
Qed.

Lemma fp_cart_empty21 : forall {T U V:Type} (A:Ensemble T) (B:Ensemble V),
                        functionally_paired (cart_prod A (Empty_set U))
                                            B (Empty_set (T*U*V)).
intros T U V A B.
constructor. intros ? h1. destruct h1 as [h1].
destruct h1; contradiction.
intros; contradiction.
Qed.


Lemma fp_empty1_s : forall {T U:Type} (B:Ensemble U) (S:Ensemble (T*U)),
                      functionally_paired (Empty_set T) B S ->
                      S = Empty_set (T*U).
intros T U B S h1.
apply Extensionality_Ensembles; red; split; auto with sets.
red.
intros pr h2.
destruct h1 as [h3 h4].
specialize (h4 _ h2).
destruct h4; contradiction.
Qed.


Lemma no_fp_empty2 : forall {T U:Type} (A:Ensemble T) 
                            (S:Ensemble (T*U)),
                        functionally_paired A (Empty_set U) S ->
                        A = Empty_set _.
intros T U A S h1.
apply Extensionality_Ensembles.
red; split; auto with sets; red.
intros x h2.
destruct h1 as [h3 h4].
specialize (h3 _ h2).
destruct h3 as [? h3].
red in h3.
destruct h3 as [h3].
destruct h3; contradiction.
Qed.


Lemma no_fp_empty2' : forall {T U:Type} (A:Ensemble T) 
                            (S:Ensemble (T*U)),
                        functionally_paired A (Empty_set U) S ->
                        S = Empty_set _.
intros T U A S h1.
apply Extensionality_Ensembles.
red; split; auto with sets; red.
intros pr h2.
destruct h1 as [h3 h4].
specialize (h4 _ h2).
destruct h4; contradiction.
Qed.


Lemma empty_map_fps : forall {T U:Type} {A:Ensemble T} {def:U}
                            (F:(Fin_map A (Empty_set U) def)),
                        functionally_paired A (Empty_set U) 
                                            (Empty_set (T*U)).
intros T U A def F.
destruct F as [h1 h2 S h3].
pose proof (no_fp_empty2 _ _ h3) as h4.
rewrite h4.
apply fp_empty1.
Qed.


Lemma empty_map_unq1 : forall {T U:Type} {B:Ensemble U} {def:U}
                              (F:Fin_map (Empty_set T) B def),
                         F = fin_map_intro _ _ def
                                           (Empty_is_finite T)
                                           (fin_map_fin_ran F)
                                           _
                                           (fp_empty1 T U B).
intros T U B def F.
apply fin_map_app_inj.
apply functional_extensionality.
intro x.

pose proof (Noone_in_empty _ x) as h1.
pose proof (fin_map_app_def F _ h1) as h2.
rewrite h2.
unfold fin_map_app.
unfold fps_to_f.
destruct (classic_dec (In (Empty_set T) x)) as [h3 | h4].
destruct h3.
reflexivity.
Qed.





Lemma empty_map_unq1' : forall {T U:Type} {B:Ensemble U} {def:U}
                              (F:Fin_map (Empty_set T) B def)
                              (pft:Finite (Empty_set T))
                              (pfb:Finite B)
                              (pffp:functionally_paired (Empty_set _)
                                                       B
                                                       (Empty_set _)),
                         F = fin_map_intro _ _ def
                                           pft pfb _ pffp.
intros T U B def F h1 h2 h3.
pose proof (empty_map_unq1 F) as h4.
rewrite h4.
assert (h5:h1 = Empty_is_finite T). apply proof_irrelevance.
assert (h6:h2 = (fin_map_fin_ran F)). apply proof_irrelevance.
assert (h7:h3 = fp_empty1 T U B). apply proof_irrelevance.
rewrite h5, h6, h7.
reflexivity.
Qed.


Lemma cart_empty_map_unq11 : forall {T U V:Type} {A:Ensemble U} {B:Ensemble V} {def:V}
                              (F:Fin_map (cart_prod (Empty_set T) A) B def),
                         F = fin_map_intro _ _ def
                                           (fin_map_fin_dom F)
                                           (fin_map_fin_ran F)
                                           (Empty_set _)
                                           (fp_cart_empty11 A B).
intros T U V A B def F.
apply fin_map_app_inj.
apply functional_extensionality.
intro pr.

pose proof (Noone_in_empty _ pr) as h1.
rewrite <- (cart_prod_empty A) in h1.
pose proof (fin_map_app_def F _ h1) as h2.
rewrite h2.
unfold fin_map_app.
unfold fps_to_f.
destruct (classic_dec (In (cart_prod (Empty_set T) A) pr)) as [h3 | h4].
destruct h3 as [h3]. destruct h3 as [h3l h3r]. contradict h3r. 
intro h4. contradiction.
reflexivity.
Qed.


Lemma cart_empty_map_unq21 : forall {T U V:Type} {A:Ensemble T} {B:Ensemble V} {def:V}
                              (F:Fin_map (cart_prod A (Empty_set U)) B def),
                         F = fin_map_intro _ _ def
                                           (fin_map_fin_dom F)
                                           (fin_map_fin_ran F)
                                           (Empty_set _)
                                           (fp_cart_empty21 A B).
intros T U V A B def F.
apply fin_map_app_inj.
apply functional_extensionality.
intro pr.

pose proof (Noone_in_empty _ pr) as h1.
rewrite <- (cart_prod_empty' A) in h1.
pose proof (fin_map_app_def F _ h1) as h2.
rewrite h2.
unfold fin_map_app.
unfold fps_to_f.
destruct (classic_dec (In (cart_prod A (Empty_set U)) pr)) as [h3 | h4].
destruct h3 as [h3]. destruct h3 as [h3l h3r]. contradict h3r. 
intro h4. contradiction.
reflexivity.
Qed.



Lemma cart_empty_map_unq11' : forall {T U V:Type} {A:Ensemble U} {B:Ensemble V} {def:V}
                              (F:Fin_map (cart_prod (Empty_set T) A) B def)
                              (pfa:Finite (cart_prod (Empty_set T) A))
                              (pfb:Finite B)
                              (pffp:functionally_paired (cart_prod (Empty_set T) A)
                                                       B
                                                       (Empty_set _)),
                         F = fin_map_intro _ _ def
                                           pfa pfb _ pffp.
intros T U V A B def F h1 h2 h3.
pose proof (cart_empty_map_unq11 F) as h4.
rewrite h4.
assert (h5:h1 = (fin_map_fin_dom F)). apply proof_irrelevance.
assert (h6:h2 = (fin_map_fin_ran F)). apply proof_irrelevance.
assert (h7:h3 = fp_cart_empty11 A B). apply proof_irrelevance.
rewrite h5, h6, h7.
reflexivity.
Qed.


Lemma cart_empty_map_unq21' : forall {T U V:Type} {A:Ensemble T} {B:Ensemble V} {def:V}
                              (F:Fin_map (cart_prod A (Empty_set U)) B def)
                              (pfa:Finite (cart_prod A (Empty_set U)))
                              (pfb:Finite B)
                              (pffp:functionally_paired (cart_prod A (Empty_set U))
                                                       B
                                                       (Empty_set _)),
                         F = fin_map_intro _ _ def
                                           pfa pfb _ pffp.
intros T U V A  B def F h1 h2 h3.
pose proof (cart_empty_map_unq21 F) as h4.
rewrite h4.
assert (h5:h1 = (fin_map_fin_dom F)). apply proof_irrelevance.
assert (h6:h2 = (fin_map_fin_ran F)). apply proof_irrelevance.
assert (h7:h3 = fp_cart_empty21 A B). apply proof_irrelevance.
rewrite h5, h6, h7.
reflexivity.
Qed.


Lemma empty_fin_map_ex1 : forall (T U:Type) (def:U) (B:Ensemble U), 
                            Finite B ->
                            exists ! F:Fin_map (Empty_set T) 
                                               B
                                               def,
                              True.
intros T U def B h1.
exists (fin_map_intro _ _ def (Empty_is_finite T) h1 
                      (Empty_set _) (fp_empty1 T U B)).
red. split; auto.
intros F ?.
symmetry.
apply empty_map_unq1'.
Qed.


Lemma cart_empty_fin_map_ex21 : forall (T U V:Type) (def:V) 
                                       (A:Ensemble T) (B:Ensemble V),
                                  Finite A ->
                                  Finite B ->
                                  exists ! F:Fin_map (cart_prod 
                                                        A (Empty_set U))
                                                     B def,
                                    True.
intros T U V def A B h1 h2.
exists (fin_map_intro _ _ def (cart_prod_fin _ _ h1 (Empty_is_finite U)) h2 
                      (Empty_set _) (fp_cart_empty21 A B)).
red. split; auto.
intros F ?.
symmetry.
apply cart_empty_map_unq21'.
Qed.

Lemma cart_empty_fin_map_ex11 : forall (T U V:Type) (def:V) 
                                       (A:Ensemble U) (B:Ensemble V),
                                  Finite A ->
                                  Finite B ->
                                  exists ! F:Fin_map (cart_prod 
                                                        (Empty_set T) A)
                                                     B def,
                                    True.
intros T U V def A B h1 h2.
exists (fin_map_intro _ _ def (cart_prod_fin _ _ (Empty_is_finite T) h1) h2 
                      (Empty_set _) (fp_cart_empty11 A B)).
red. split; auto.
intros F ?.
symmetry.
apply cart_empty_map_unq11'.
Qed.


Definition empty_map1 (T U:Type) (def:U) (B:Ensemble U) (pf:Finite B) :
  Fin_map (Empty_set T) B def.
refine (proj1_sig (constructive_definite_description _ (empty_fin_map_ex1 T U def B pf))).
Defined.


Definition cart_empty_map11 (T U V:Type) (def:V) (A:Ensemble U) 
           (B:Ensemble V) (pfa:Finite A) (pfb:Finite B) :
  Fin_map (cart_prod (Empty_set T) A) B def.
refine (proj1_sig (constructive_definite_description _ (cart_empty_fin_map_ex11 T U V def A B pfa pfb))).
Defined.


Definition cart_empty_map21 (T U V:Type) (def:V) (A:Ensemble T) 
           (B:Ensemble V) (pfa:Finite A) (pfb:Finite B) :
  Fin_map (cart_prod A (Empty_set U)) B def.
refine (proj1_sig (constructive_definite_description _ (cart_empty_fin_map_ex21 T U V def A B pfa pfb))).
Defined.


Lemma empty_map1_compat : forall {T U:Type} {def:U} {B:Ensemble U}
                                 (F:Fin_map (Empty_set T) B def) 
                                 (pf:Finite B),
                            F = empty_map1 T U def B pf.
intros T U def B F pf.
unfold empty_map1.
destruct constructive_definite_description as [F'].
simpl.
destruct F' as [h1 h2 S h3].
pose proof (fp_empty1_s _ _ h3) as h4.
pose proof h3 as h5.
rewrite h4 in h5.
pose proof (empty_map_unq1' F h1 pf h5) as h6.
rewrite h6.
assert (h7:h2 = pf). apply proof_irrelevance.
rewrite h7.
pose proof (subsetT_eq_compat _ _ _ _ h3 h5 h4) as h8.
dependent rewrite -> h8.
reflexivity.
Qed.


Lemma empty_map1_def : forall (T U:Type) (def:U) (B:Ensemble U)
                              (pf:Finite B) (x:T),
                         (empty_map1 T U def B pf) |-> x = def.
intros T U def B h1 x.
unfold fin_map_app.
destruct (empty_map1 T U def B h1) as [h2 h3 S h4].
pose proof (fp_empty1_s _ _ h4) as h5.
pose proof (fp_empty1 T U B) as h6.
pose proof (subsetT_eq_compat _ _ _ _ h4 h6 h5) as h7.
dependent rewrite -> h7.
unfold fps_to_f.
destruct (classic_dec (In (Empty_set T) x)) as [h8 | h9].
destruct h6.
destruct constructive_definite_description.
simpl.
contradiction.
reflexivity.
Qed.


Lemma full_set_empty_map_sing1: forall (T U:Type) (def:U)
                                       (B:Ensemble U) (pf:Finite B),
                               Full_set (Fin_map (Empty_set T)
                                          B
                                          def) =
                                       Singleton (empty_map1 T U def B pf).
intros T U def B pf.
apply Extensionality_Ensembles; red; split.
red.
intros F ?.
pose proof (empty_map1_compat F pf) as h2.
rewrite h2.
constructor.
red. intros; constructor.
Qed.



Lemma cart_empty_map11_compat : forall {T U V:Type} {def:V} {A:Ensemble U} 
                                  {B:Ensemble V}
                                  (F:Fin_map (cart_prod (Empty_set _) A) B def)
                                  (pfa:Finite A),
                             F = cart_empty_map11 T U V def A B pfa (fin_map_fin_ran F).
intros T U V def A B F h1.
unfold cart_empty_map11.
destruct constructive_definite_description as [F'].
simpl.
destruct F' as [h3 h4 S h5].
pose proof (@cart_prod_empty T U A) as h6.
pose proof h5 as h7.
rewrite h6 in h7.
pose proof (fp_empty1_s _ _ h7) as h8.
rewrite h8 in h7.
rewrite <- h6 in h7.
pose proof (cart_prod_fin _ _ (Empty_is_finite T) h1) as h9.
pose proof (cart_empty_map_unq11' F h9 h4 h7) as h10.
rewrite h10.
assert (h11:h9 = h3).  apply proof_irrelevance.
pose proof (subsetT_eq_compat _ _ _ _ h5 h7 h8) as h12.
dependent rewrite -> h12.
rewrite h11.
reflexivity.
Qed.



Lemma cart_empty_map11_def : 
  forall (T U V:Type) (def:V) (A:Ensemble U) 
         (B:Ensemble V)
         (pfa:Finite A) (pfb:Finite B)
         (pr:(T*U)),
    (cart_empty_map11 T U V def A B pfa pfb) |-> pr = def.
intros T U V def A B h1 h2 pr.
unfold fin_map_app.
destruct (cart_empty_map11 T U V def A B h1 h2) as [h3 h4 S h5].
unfold fps_to_f.
destruct (classic_dec (In (cart_prod (Empty_set T) A) pr)) as 
    [h6 | h7].
destruct h6 as [h6].
destruct h6 as [h6l h6r].
destruct h5. 
destruct constructive_definite_description.
simpl.
contradiction. reflexivity.
Qed.


Lemma cart_empty_map21_compat : 
  forall {T U V:Type} {def:V} {A:Ensemble T} 
         {B:Ensemble V}
         (F:Fin_map (cart_prod A (Empty_set _)) B def)
                                  (pfa:Finite A),
                             F = cart_empty_map21 T U V def A B pfa (fin_map_fin_ran F).
intros T U V def A B F h1.
unfold cart_empty_map21.
destruct constructive_definite_description as [F'].
simpl.
destruct F' as [h3 h4 S h5].
pose proof (@cart_prod_empty' T U A) as h6.
pose proof h5 as h7.
rewrite h6 in h7.
pose proof (fp_empty1_s _ _ h7) as h8.
rewrite h8 in h7.
rewrite <- h6 in h7.
pose proof (cart_prod_fin _ _ h1 (Empty_is_finite U) ) as h9.
pose proof (cart_empty_map_unq21' F h9 h4 h7) as h10.
rewrite h10.
assert (h11:h9 = h3).  apply proof_irrelevance.
pose proof (subsetT_eq_compat _ _ _ _ h5 h7 h8) as h12.
dependent rewrite -> h12.
rewrite h11.
reflexivity.
Qed.

Lemma cart_empty_map21_def : 
  forall (T U V:Type) (def:V) (A:Ensemble T) 
         (C:Ensemble V)
         (pfa:Finite A) (pfc:Finite C)
         (pr:(T*U)),
    (cart_empty_map21 T U V def A C pfa pfc) |-> pr = def.
intros T U V def A B h1 h2 pr.
unfold fin_map_app.
destruct (cart_empty_map21 T U V def A B h1 h2) as [h3 h4 S h5].
unfold fps_to_f.
destruct (classic_dec (In (cart_prod A (Empty_set U)) pr)) as 
    [h6 | h7].
destruct h6 as [h6].
destruct h6 as [h6l h6r].
destruct h5. 
destruct constructive_definite_description.
simpl.
contradiction. reflexivity.
Qed.


Lemma empty_map_unq2: forall {T U:Type} {A:Ensemble T} {def:U}
               (F:(Fin_map A (Empty_set U) def)),
                       F = fin_map_intro _ _ def
                                         (fin_map_fin_dom F)
                                         (Empty_is_finite U)
                                         _
                                         (empty_map_fps F).
intros T U A def F.
apply fin_map_app_inj.
apply functional_extensionality.
intros x.
destruct (classic_dec (In A x)) as [h1 | h2].
pose proof (fin_map_app_in F _ h1). contradiction.
pose proof (fin_map_app_def F _ h2) as h3.
rewrite h3.
unfold fin_map_app.
unfold fps_to_f.
destruct (classic_dec (In A x)) as [h4 | h5].
contradiction.
reflexivity.
Qed.


Lemma empty_map_unq2' : forall {T U:Type} {A:Ensemble T} {def:U}
                              (F:(Fin_map A (Empty_set U) def))
                              (pfa:Finite A) 
                              (pfe:Finite (Empty_set U)) 
                              (pffp:functionally_paired A
                                                        (Empty_set _)
                                                        (Empty_set _)),
                                    F = fin_map_intro 
                                          _ _ _ pfa pfe _ pffp.
intros T U A def F h1 h2 h3.
pose proof (empty_map_unq2 F) as h4.
rewrite h4.
assert (h5:h1 = (fin_map_fin_dom F)).
  apply proof_irrelevance.
rewrite h5.
assert (h6:h2 = Empty_is_finite U). apply proof_irrelevance.
rewrite h6.
assert (h7:h3 = (empty_map_fps F)). apply proof_irrelevance.
rewrite h7.
reflexivity.
Qed.



Lemma empty_fin_map_ex2 : forall (T U:Type) (def:U), 
                               exists ! F:Fin_map (Empty_set T)
                                                  (Empty_set U)
                                                  def,
                                 True.
intros T U def.
exists (fin_map_intro _ _ _ (Empty_is_finite T) (Empty_is_finite U) 
                      (Empty_set _) (fp_empty1 T U _)).
red. split; auto.
intros F ?.
symmetry.
apply empty_map_unq2'.
Qed.

Definition empty_map (T U:Type) (def:U) : Fin_map (Empty_set T)
                                                  (Empty_set U)
                                                  def.
refine (proj1_sig (constructive_definite_description _ (empty_fin_map_ex2 T U def))).
Defined.

Lemma empty_map_compat : forall {T U:Type} {def:U} 
                               (F:Fin_map (Empty_set T)
                                          (Empty_set U)
                                          def),
  F = empty_map T U def.
intros T U def F.
unfold empty_map.
destruct constructive_definite_description as [F'].
simpl.
destruct F' as [h1 h2 S h3].
pose proof (no_fp_empty2' _ _ h3). subst.
apply empty_map_unq2'.
Qed.

Lemma empty_map_def : forall (T U:Type) (def:U),
                        forall x:T, 
                          (empty_map T U def) |-> x = def.
intros T U def x.
unfold fin_map_app.
destruct (empty_map T U def) as [h1 h2 S h3].
pose proof (fp_empty1_s _ _ h3) as h4.
pose proof (fp_empty1 T U (Empty_set _)) as h5.
pose proof (subsetT_eq_compat _ _ _ _ h3 h5 h4) as h6.
dependent rewrite -> h6.
unfold fps_to_f.
destruct (classic_dec (In (Empty_set T) x)) as [h7 | h8].
destruct h5. destruct constructive_definite_description.
simpl.
contradiction. reflexivity.
Qed.

Lemma full_set_empty_map_sing_empty: forall (T U:Type) (def:U),
                               Full_set (Fin_map (Empty_set T)
                                          (Empty_set U)
                                          def) =
                                       Singleton (empty_map T U def).
intros T U def.
apply Extensionality_Ensembles.
red. split.
red.
intros F ?.
pose proof (empty_map_compat F) as h1.
rewrite h1. constructor.
red. intros; constructor.
Qed.

                         
Lemma empty_map_ex2_fin : forall {T U:Type} (A:Ensemble T) (def:U),
                        Finite A -> A <> Empty_set T ->
                        forall F:Fin_map A (Empty_set U) def,
                          False.
intros T U A def h1.
induction h1. intro h2. contradict h2. reflexivity.
intros ? F.
pose proof (fin_map_app_in F x (Add_intro2 _ A x)).
contradiction.
Qed.


Lemma full_set_non_empty_map_empty : 
  forall (T U:Type) (def:U)
    (A:Ensemble T), A <> Empty_set T ->
      Full_set (Fin_map A (Empty_set U) def) =
      Empty_set _.
intros T U def A h1.
apply Extensionality_Ensembles; red;split; auto with sets.
red.
intros F ?.
inversion F as [h2 h3 S h4].
pose proof (empty_map_ex2_fin _ _ h2 h1 F).
contradiction.
Qed. 


Lemma fun_to_fin_map_empty_set1 : 
  forall {T U:Type} (def:U)
         (pfe:Finite (Empty_set _)) (f:T->U),
         fun_to_fin_map _ def pfe f = 
         empty_map1 T U def (Im (Empty_set _) f) 
                        (finite_image _ _ _ f pfe). 
intros; apply empty_map1_compat.
Qed.

Lemma fun_to_fin_map_empty_set11 : 
  forall {T U V:Type} (def:V) (B:Ensemble U)
         (pfb:Finite B)   
         (pfce:Finite (cart_prod (Empty_set _) B)) (f:T*U->V),
         fun_to_fin_map _ def pfce f = 
         cart_empty_map11 T U V def _ _ pfb 
                          (finite_image _ _ _ f pfce). 
intros T U V def B h1 h3 f. 
pose proof (cart_empty_map11_compat ( fun_to_fin_map (cart_prod (Empty_set T) B) def h3 f) h1) as h4.
assert (h5: (fin_map_fin_ran
            (fun_to_fin_map (cart_prod (Empty_set T) B) def h3 f))
              = (finite_image (T * U) V (cart_prod (Empty_set T) B) f h3)). apply proof_irrelevance.
rewrite h5 in h4.
assumption.
Qed.

Lemma fun_to_fin_map_empty_set21 : 
  forall {T U V:Type} (def:V) (A:Ensemble T) (pfa:Finite A)    
         (pfce:Finite (cart_prod A (Empty_set _))) (f:T*U->V),
         fun_to_fin_map _ def pfce f = 
         cart_empty_map21 T U V def _ _ pfa 
                          (finite_image _ _ _ f pfce). 
intros T U V def A h1 h3 f. 
pose proof (cart_empty_map21_compat ( fun_to_fin_map (cart_prod A (Empty_set U)) def h3 f) h1) as h4.
assert (h5: (fin_map_fin_ran
            (fun_to_fin_map (cart_prod A (Empty_set U)) def h3 f))
              = (finite_image (T * U) V (cart_prod A (Empty_set U)) f h3)). apply proof_irrelevance.
rewrite h5 in h4. 
assumption.
Qed.


Lemma im1_fun_to_fin_map : 
  forall {T U V:Type} {A:Ensemble T} {B:Ensemble U} (f:(T*U)->V)
         (def:V) (pf:Finite (cart_prod A B)) (x:T),
    Finite A -> In A x ->
    im1 (fun_to_fin_map (cart_prod A B) def pf f) x =
    Im B (fun y => f (x, y)).
intros T U V A B f def h1 x h2 h3.
induction h2 as [|A h4 h5 a h6]. 
contradiction.
unfold im1.
destruct (eq_dec (Add A a) (Empty_set T)) as [h7 | h8].
pose proof (Add_intro2 _ A a) as h8.
rewrite h7 in h8. contradiction.
apply Extensionality_Ensembles.
red. split.
red. 
intros v h2.
destruct h2 as [y h9 v  h10].
rewrite fun_to_fin_map_compat in h10.
apply Im_intro with y; auto.
constructor. split; auto.
red.
intros v h2.
destruct h2 as [y h9 v h10].
apply Im_intro with y; auto.
rewrite fun_to_fin_map_compat. assumption.
constructor. split; auto.
Qed.


Lemma im2_fun_to_fin_map : 
  forall {T U V:Type} {A:Ensemble T} {B:Ensemble U} (f:(T*U)->V)
         (def:V) (pf:Finite (cart_prod A B)) (y:U),
    Finite B -> In B y ->
    im2 (fun_to_fin_map (cart_prod A B) def pf f) y =
    Im A (fun x => f (x, y)).
intros T U V A B f def h1 y h2 h3.
induction h2 as [|B h4 h5 b h6]. 
contradiction.
unfold im2.
destruct (eq_dec (Add B b) (Empty_set U)) as [h7 | h8].
pose proof (Add_intro2 _ B b) as h8.
rewrite h7 in h8. contradiction.
apply Extensionality_Ensembles.
red. split.
red. 
intros v h2.
destruct h2 as [x h9 v  h10].
rewrite fun_to_fin_map_compat in h10.
apply Im_intro with x; auto.
constructor. split; auto.
red.
intros v h2.
destruct h2 as [x h9 v h10].
apply Im_intro with x; auto.
rewrite fun_to_fin_map_compat. assumption.
constructor. split; auto.
Qed.


Lemma fin_map_im_eq_ex : 
  forall {T U V:Type} {A:Ensemble T} {g:T->U} {B:Ensemble V}
         {defv:V}  (F:Fin_map (Im A g) B defv),
    Finite A ->  
       exists! F':Fin_map A B defv,
        forall x:T,
          In A x ->
          F' |-> x = F |-> (g x).
intros T U V A g B defv F h1.
destruct (classic (Inhabited A)) as [hi | hni].
destruct hi as [a hi].
pose (fun x:sig_set A => g (proj1_sig x)) as g'.
assert (h2:Im A g = Im (full_sig A) g').
  apply Extensionality_Ensembles.
  red. split.
  red. intros u h2. destruct h2 as [u h2]. subst.
  apply Im_intro with (exist _ _ h2). constructor.
  unfold g'. simpl. reflexivity.
  red. 
  intros x h2. destruct h2 as [x h2]. subst. unfold g'.
  clear h2. destruct x as [x h2]. simpl.
  apply Im_intro with x; auto.
assert (h3:Included (Im (full_sig A) g') (Im A g)). 
  rewrite h2. auto with sets.
pose (restriction F h3) as F'.
exists (fin_map_im_full_sig_eq F' h1 (g a)).
red. split.
intros x h4.
rewrite fin_map_im_full_sig_eq_compat.
unfold F'. rewrite restriction_compat.
f_equal.
unfold g'.
simpl.
unfold sig_fun_app.
destruct classic_dec as [h5 | h6]. simpl. reflexivity.
contradiction.
apply Im_intro with (exist _ _ h4). constructor.
simpl.
unfold sig_fun_app. destruct classic_dec as [h5 | h6].
assert (h4 = h5). apply proof_irrelevance.
subst.
reflexivity.
contradiction. assumption.
intros F'' h4.
apply fin_map_ext_in.
intros x h5.
rewrite fin_map_im_full_sig_eq_compat.
unfold F'. rewrite restriction_compat.
rewrite h4.
f_equal. simpl.
unfold sig_fun_app. destruct classic_dec as [h6 | h7].
unfold g'. simpl. reflexivity.
contradiction.
assumption.
apply Im_intro with (exist _ _ h5).
constructor. simpl.
unfold sig_fun_app.
destruct classic_dec as [h6 | h7].
f_equal. apply proj1_sig_injective. reflexivity.
contradiction.
assumption.
apply not_inhabited_empty in hni. subst.
exists (empty_map1 T V defv B (fin_map_fin_ran F)).
red. split.
intros; contradiction.
intros.
pose proof (empty_fin_map_ex1) as h2. 
rewrite <- (empty_map1_compat x').
reflexivity.
Qed.


Definition fin_map_im_eq 
           {T U V:Type} {A:Ensemble T} {g:T->U} {B:Ensemble V}
           {defv:V} (F:Fin_map (Im A g) B defv)
           (pf:Finite A) :=
  proj1_sig (constructive_definite_description _ (fin_map_im_eq_ex F pf)).



Lemma fin_map_im_eq_compat :
  forall {T U V:Type} {A:Ensemble T} {g:T->U} {B:Ensemble V}
         {defv:V}  (F:Fin_map (Im A g) B defv)
         (pf:Finite A),
    let F' := fin_map_im_eq F pf in 
    forall x:T,
      In A x ->
      F' |-> x = F |-> (g x).
intros T U V A g B defv F h1 F'.
unfold F'.
unfold fin_map_im_eq.
destruct constructive_definite_description as [F'' h2].
simpl.
assumption.
Qed.  

Lemma fin_map_to_fin_map_im_eq : 
  forall {T U V:Type} {A:Ensemble T} {B:Ensemble V} {defv:V}
         (F : Fin_map A B defv)
         (g : T -> U),
       injective g ->
       exists F' : Fin_map (Im A g) B defv,
         forall x : T,
         In A x -> F' |-> (g x) = F |-> x.
intros T U V A B defv F g h1.
destruct (classic (Inhabited A)) as [h2 | h3].
destruct h2 as [a h2].
pose (fun x:sig_set A => g (proj1_sig x)) as g'.
assert (h3:injective g').
  red. intros x y h3. unfold g' in h3. apply h1 in h3.
  apply proj1_sig_injective. assumption. 
pose proof (fin_map_to_fin_map_im_full_sig_comp F _ (g' (exist _ _ h2)) h3) as h4.
destruct h4 as [F' h4]. 
assert (h5:Included (Im A g) (Im (full_sig A) g')).
  red. intros u h5. destruct h5 as [u h5]. subst. apply Im_intro with (exist _ _ h5).
  constructor. unfold g'. simpl. reflexivity.
exists (restriction F' h5).
intros x h6. rewrite restriction_compat. 
rewrite <- h4.
f_equal. simpl. unfold sig_fun_app. destruct classic_dec as [h7 | h8].
unfold g'. simpl. reflexivity.
contradiction. assumption. apply Im_intro with x; auto.
apply not_inhabited_empty in h3. subst.
pose (empty_map1 U V defv B (fin_map_fin_ran F)) as f.
pose proof (image_empty T U g) as h2.
assert (h3:Included (Im (Empty_set T) g) (Empty_set U)).
  rewrite h2. auto with sets.
exists (restriction f h3).
intros; contradiction.
Qed.


Lemma im_full_set_fin_map_im_inj : 
  forall {T U V W:Type} {A:Ensemble T} {B:Ensemble V} {defv:V}
         (deft:T),
    Finite A -> forall (g:T->U) (pfi:injective g) 
                       (k:Fin_map (Im A g) B defv->W),
  Im (Full_set (Fin_map (Im A g) B defv)) k =
  Im (Full_set (Fin_map A B defv)) 
       (fun f:Fin_map A B defv =>
          (k (fin_map_compose_inj f deft _ pfi))).
intros T U V W A B defv deft h1 g h2 k.
apply Extensionality_Ensembles.
red. split.
red.
intros v h3.
destruct h3 as [v h3]. subst. clear h3.
pose (fin_map_im_eq v h1) as F. 
apply Im_intro with F. constructor.
f_equal.
apply fin_map_ext_in.
intros x h3.
destruct h3 as [x h3]. subst.
rewrite <- fin_map_compose_inj_compat.
unfold F. rewrite fin_map_im_eq_compat.
reflexivity.
assumption. assumption.
red.
intros v h3.
destruct h3 as [v h3].
subst. clear h3.
pose proof (fin_map_to_fin_map_im_eq v _ h2) as h3.
destruct h3 as [F h3].
apply Im_intro with F. constructor.
f_equal.
apply fin_map_ext_in.
intros x h4. destruct h4 as [x h4]. subst.
rewrite h3; auto. rewrite <- fin_map_compose_inj_compat.
reflexivity.
assumption.
Qed.


Definition fin_map_dom_subst 
           {T U:Type} {A A':Ensemble T} {B:Ensemble U} {def:U}
           (pf:A = A') (F:Fin_map A B def) : Fin_map A' B def.
subst. refine F.
Defined.

Definition fin_map_dom_subst_fun 
  {T U V:Type} {A A':Ensemble T} {B:Ensemble U} {def:U}
  (pf:A = A') (k:Fin_map A B def->V) : Fin_map A' B def->V.
intro f. subst. refine (k f).
Defined.

Lemma im_full_set_fin_map_dom_subst : 
  forall {T U V:Type} {A A':Ensemble T} {B:Ensemble U} {def:U}
         (pf:A = A') (k:Fin_map A B def->V),
    Im (Full_set (Fin_map A  B def)) k =
    Im (Full_set (Fin_map A' B def)) (fin_map_dom_subst_fun pf k).
intros T U V A A' B def h1 k. subst.
unfold fin_map_dom_subst_fun. unfold eq_rect_r. simpl.
f_equal.
Qed.


Require Import ListUtilities.

Lemma fp_fpl_compat : forall {T U:Type} (A:Ensemble T) (B:Ensemble U)
                             (S:Ensemble (T*U)) 
                             (la:list T) (lb:list U) (lp:list (T*U)),
                        list_to_set la = A ->
                        list_to_set lb = B ->
                        list_to_set lp = S ->
                        ((functionally_paired A B S) <->
                        (functionally_paired_l la lb lp)).
intros T U A B S la lb lp h1 h2 h3.
split.
intro h4.
destruct h4 as [h5 h6].
constructor. 
intros x h7.
rewrite list_to_set_in_iff in h7.
rewrite h1 in h7.
specialize (h5 _ h7).
destruct h5 as [y h5].
exists y.
red in h5. red.
destruct h5 as [h5l h5r].
rewrite <- h2 in h5l. rewrite <- h3 in h5l.
do 2 rewrite <- list_to_set_in_iff in h5l.
split; try assumption.
intros b h8.
do 2 rewrite list_to_set_in_iff in h8.
rewrite h2 in h8. rewrite h3 in h8.
specialize (h5r _ h8). assumption.
intros pr h7.
rewrite list_to_set_in_iff in h7.
rewrite h3 in h7.
specialize (h6 _ h7).
rewrite  <- h1 in h6.  rewrite <- h2 in h6.
do 2 rewrite <- list_to_set_in_iff in h6.
assumption.
intro h4.
destruct h4 as [h4 h5].
constructor.
intros x h6.
rewrite <- h1 in h6.
rewrite <- list_to_set_in_iff in h6.
specialize (h4 _ h6).
destruct h4 as [y h4].
exists y.
red in h4. red.
destruct h4 as [h4l h4r].
split.
do 2 rewrite list_to_set_in_iff in h4l.
rewrite  h2 in h4l. rewrite h3 in h4l.
assumption.
intros b h7.
rewrite <- h2 in h7. rewrite <- h3 in h7.
do 2 rewrite <- list_to_set_in_iff in h7.
apply h4r; assumption.
intros pr h6.
rewrite <- h3 in h6.
rewrite <- list_to_set_in_iff in h6.
specialize (h5 _ h6).
do 2 rewrite list_to_set_in_iff in h5.
rewrite h1 in h5. rewrite h2 in h5.
assumption.
Qed.


Lemma fin_map_to_fps_list_power_compat :
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U) (def:U)
         (F:Fin_map A B def) (la:list T) (lb:list U)
         (lp:list (T*U)),
         list_to_set la = A -> list_to_set lb = B ->
         list_to_set lp = fin_map_to_fps F ->
         NoDup la -> NoDup lp ->
         synced la lp ->
         In lp (list_power la lb).
intros T U A B def F la lb lp h1 h2 h3 h4 h5 h6.
pose proof (fin_map_to_fps_compat F) as h7.
destruct h7 as [h8 h9].
clear h9.
rewrite (fp_fpl_compat _ _ _ _ _ _ h1 h2 h3) in h8.
rewrite fpl_in_list_power in h8; assumption.
Qed.


Lemma fin_map_to_list_pairs_lemma : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U}
         (F:Fin_map A B def) (la:list T) (lb:list U),
    NoDup la -> 
   list_to_set la = A -> list_to_set lb = B ->   
    exists (lp:list (T*U)), 
      list_to_set lp = fin_map_to_fps F /\
      NoDup lp /\
      In lp (list_power la lb).
      
intros T U A B def F la lb h0 h1 h2.
pose proof (fin_map_to_fps_compat F) as h3.
destruct h3 as [h4 h5].
clear h5.
pose proof (fin_map_to_fps_finite F) as h5.
pose proof (finite_set_list _ h5) as h6.
destruct h6 as [lp h7].
symmetry in h7.
rewrite (fp_fpl_compat _ _ _ _ _ _ h1 h2 h7) in h4.
pose proof (sync_fpl h4) as h6.
destruct h6 as [lp' h6].
destruct h6 as [h6a [h6b h6c]].
exists lp'.
repeat split.
rewrite <- h7.
rewrite h6c. reflexivity.
apply (synced_no_dup _ _ h0 h6b).
pose proof (synced_no_dup _ _ h0 h6b) as h8.
rewrite h6c in h7.
apply (fin_map_to_fps_list_power_compat _ _ _ _ _ _ _ h1 h2 h7);
assumption.
Qed.


Lemma no_dup_self_fp_fst : 
  forall {T U:Type} (S:Ensemble (T*U)) (l:list (T*U)),
    S = list_to_set l -> NoDup l ->self_fp S -> NoDup (map (@fst _ _) l).
intros T U S l h0 h1 h2. generalize dependent S. revert h1.
induction l as [|pr l h1].
simpl. intros; constructor.
intros h2 S h3 h4.
simpl.
constructor.
assert (h5:Ensembles.In S pr). rewrite h3. rewrite <- list_to_set_in_iff. left. reflexivity.
intro h6. rewrite in_map_iff in h6. destruct h6 as [pr' h6].
destruct h6 as [h6l h6r].
assert (h6:Ensembles.In S pr'). rewrite h3. rewrite <- list_to_set_in_iff. right. assumption.
rewrite surjective_pairing in h5. rewrite surjective_pairing in h6. rewrite h6l in h6.
pose proof (fp_functional h4 _ _ _ h5 h6) as h7.
assert (h8:pr = pr'). apply injective_projections; auto.
subst.
pose proof (no_dup_cons_nin _ _ h2).
contradiction.
simpl in h3.
pose proof (f_equal (fun S=>Subtract S pr) h3) as h5.
simpl in h5.
rewrite sub_add_compat_nin in h5.
pose proof (no_dup_cons _ _ h2) as h6.
pose proof (incl_subtract  S pr) as h7.
pose proof (self_fp_incl _ _ h7 h4) as h8.
specialize (h1 h6 _ h5 h8).
assumption.
intro h6.
rewrite <- list_to_set_in_iff in h6.
pose proof (no_dup_cons_nin _ _ h2).
contradiction.
Qed.


Record nice_map_lists {T U:Type} {A:Ensemble T} {B:Ensemble U} 
       {def:U} (F:Fin_map A B def) :=
{n_la:list T;
 n_lb:list U;
 n_lp:list (T*U);
 n_im := map (@snd T U) n_lp;
 la_compat : list_to_set n_la = A;
 lb_compat : list_to_set n_lb = B;
 lp_compat : list_to_set n_lp = fin_map_to_fps F;
 nda : NoDup n_la;
 ndb : NoDup n_lb;
 ndp: NoDup n_lp;
 in_lp: In n_lp (list_power n_la n_lb);
 fpl : functionally_paired_l n_la n_lb n_lp}.


(*The complications relating to testing if the cartesian product
  is empty accounts for such aberrations as the cartesian product
  of an infinite set and the empty set, which would qualify 
  as the domain of a finite map according to my definition.*)
 

Record nice_map_lists2 {T U V:Type} {A:Ensemble T} {B:Ensemble U}
       {C:Ensemble V} {def:V} (F:Fin_map (cart_prod A B) C def) :=
{n_la2:list T;
 n_lb2:list U;
 n_lab2:list (T*U) := list_prod n_la2 n_lb2;
 n_lc2:list V;
 n_lp2:list (T*U*V);
 n_im2 := map (@snd (T*U) V) n_lp2;
 la_lb_compat2 : Finite A -> Finite B -> 
                list_to_set n_la2 = A /\ list_to_set n_lb2 = B;
 lab_compat2 : list_to_set n_lab2 = cart_prod A B;
 lab_empty2 : 
   if (eq_dec n_lab2 nil) then (n_la2 = nil /\ A = Empty_set _) \/
                               (n_lb2 = nil /\ B = Empty_set _) 
   else (list_to_set n_la2 = A /\ list_to_set n_lb2 = B);
 lab_empty2' :    
   if (eq_dec n_lab2 nil) then (A = Empty_set _ -> n_la2 = nil) /\
                               (B = Empty_set _ -> n_lb2 = nil)
   else True;
 lc_compat2 : list_to_set n_lc2 = C;
 lp_compat2 : list_to_set n_lp2 = fin_map_to_fps F;
 nda2 : NoDup n_la2;
 ndb2 : NoDup n_lb2;
 ndab2 : NoDup n_lab2;
 ndc : NoDup n_lc2;
 ndp2 : NoDup n_lp2;
 in_lp2 : In n_lp2 (list_power n_lab2 n_lc2);
 fpl2 : functionally_paired_l n_lab2 n_lc2 n_lp2}.


Lemma fin_map_ex_nice_map_lists :
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U} {def:U}
         (F:Fin_map A B def),
    exists _:(nice_map_lists F), True.
intros T U A B def F.
pose proof (fin_map_fin_dom F) as h1.
pose proof (fin_map_fin_ran F) as h2.
pose proof (finite_set_list_no_dup _ h1) as h3.
pose proof (finite_set_list_no_dup _ h2) as h4.
destruct h3 as [la h3].
destruct h3 as [h3l h3r].
destruct h4 as [lb h4].
destruct h4 as [h4l h4r].
symmetry in h3l.
symmetry in h4l.
pose proof (fin_map_to_list_pairs_lemma F  _ _ h3r h3l h4l) as h5.
destruct h5 as [lp h5].
destruct h5 as [h5a [h5b h5c]].
pose proof (in_list_power_synced _ _ _ h5c) as h6.
pose proof (fpl_in_list_power _ lb _ h6 h3r h5b) as h7. 
pose proof h5c as h8.
rewrite <- h7 in h8.
exists (Build_nice_map_lists _ _ _ _ _ F  _ _ _ h3l h4l h5a h3r h4r h5b h5c h8).
constructor.
Qed.


Lemma fin_map_ex_nice_map_lists_list_to_set :
  forall {T U:Type} {la:list T} {lb:list U} {def:U}
         (F:Fin_map (list_to_set la) (list_to_set lb) def),
            NoDup la -> NoDup lb ->
            exists nml:(nice_map_lists F),
              la = n_la _ nml /\
              lb = n_lb _ nml.
intros T U la lb def F h1 h2.
pose proof (fin_map_to_list_pairs_lemma F la lb h1 (eq_refl _)
                                        (eq_refl _)) as h3.
destruct h3 as [lp h3].
destruct h3 as [h3a [h3b h3c]].
pose proof (in_list_power_synced _ _ _ h3c) as h4.
pose proof (fpl_in_list_power _ lb _ h4 h1 h3b) as h5.
pose proof h3c as h8.
rewrite <- h5 in h8.
exists (Build_nice_map_lists _ _ _ _ _ F la lb lp (eq_refl _) 
                             (eq_refl _) h3a h1 h2 h3b h3c h8).
simpl. split; auto.
Qed.


Lemma fin_map_ex_nice_map_lists_list_to_set' :
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U} (F:Fin_map A B def)
         (la:list T) (lb:list U),
    list_to_set la = A ->
    list_to_set lb = B ->
    NoDup la -> NoDup lb ->
    exists nml:(nice_map_lists F),
      la = n_la _ nml /\
      lb = n_lb _ nml.
intros T U A B def F la lb h1 h2 h3 h4.
pose proof (fin_map_to_list_pairs_lemma F la lb h3 h1 h2) as h5.
destruct h5 as [lp h5].
destruct h5 as [h5a [h5b h5c]].
pose proof (in_list_power_synced _ _ _ h5c) as h4'.
pose proof (fpl_in_list_power _ lb _ h4' h3 h5b) as h5'.
pose proof h5c as h8.
rewrite <- h5' in h8.
exists (Build_nice_map_lists _ _ _ _ _ F la lb lp h1 h2 h5a h3 h4 h5b h5c h8).
simpl.
split; auto.
Qed.


Lemma fin_map_ex_nice_map_lists_list_to_set_dom :
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U} (F:Fin_map A B def)
         (la:list T),
    list_to_set la = A -> NoDup la ->     
    exists nml:(nice_map_lists F),
      la = n_la _ nml.
intros T U A B def F la h1 h2.
pose proof (fin_map_fin_ran F) as h3.
pose proof (finite_set_list_no_dup _ h3) as h4.
destruct h4 as [lb h4].
destruct h4 as [h4l h4r].
symmetry in h4l.
pose proof (fin_map_to_list_pairs_lemma F la lb h2 h1 h4l) as h5.
destruct h5 as [lp h5].
destruct h5 as [h5a [h5b h5c]].
pose proof (in_list_power_synced _ _ _ h5c) as h4'.
pose proof (fpl_in_list_power _ lb _ h4' h2 h5b) as h5'.
pose proof h5c as h8.
rewrite <- h5' in h8.
exists (Build_nice_map_lists _ _ _ _ _ F la lb lp h1 h4l h5a h2 h4r h5b h5c h8).
simpl. reflexivity.
Qed.





Lemma fin_map_ex_nice_map_lists_intro :
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U)
         (la:list T) (lb:list U) (lp:list (T*U)) (def:U)
         (pfa:Finite A) (pfb:Finite B)
         (pfin:In lp (list_power la lb))
         (pffp:functionally_paired A B (list_to_set lp)),
    list_to_set la = A ->
    list_to_set lb = B ->
    NoDup la -> NoDup lb ->
    exists nml:(nice_map_lists (fin_map_intro A B def pfa pfb (list_to_set lp) pffp)),
      la = n_la _ nml /\
      lb = n_lb _ nml /\
      lp = n_lp _ nml.
intros T U A B la lb lp def h1 h2 h3 h4 h1' h2' h3' h4'.
pose proof (in_list_power_synced _ _ _ h3) as h4''.
pose proof (list_power_no_dup _ _ _ h3' h4' h3) as h5.
pose proof (fpl_in_list_power _ lb _ h4'' h3' h5) as h5'.
pose proof (fp_fpl_compat _ _ _ _ _ _ h1' h2' (eq_refl (list_to_set lp))) as h6.
pose proof h4 as h8.
rewrite h6 in h8.
pose proof h8 as h9.
rewrite h5' in h9.
pose proof (fin_map_to_fps_compat_s _ _ def h1 h2 _ h4) as h10.
exists (Build_nice_map_lists _ _ _ _ _ _ la lb lp h1' h2' h10 h3' h4' h5 h9 h8).
simpl.
split; auto.
Qed.



Lemma fin_map_ex_nice_map_lists2 :
  forall {T U V:Type} {A:Ensemble T} {B:Ensemble U} 
         {C:Ensemble V} {def:V}
         (F:Fin_map (cart_prod A B) C def),
    exists _:(nice_map_lists2 F), True.
intros T U V A B C def F.
pose proof (fin_map_fin_dom F) as h1.
dependent induction h1. 
pose proof (fin_map_fin_ran F) as h1.
pose proof (finite_set_list_no_dup _ h1) as h3.
destruct h3 as [lc h3].
destruct h3 as [h3l h3r].
symmetry in h3l.
pose proof (fin_map_to_fps_finite F) as h4.
pose proof (finite_set_list_no_dup _ h4) as h5.
destruct h5 as [lp h5].
destruct h5 as [h5l h5r].
symmetry in h5l.
destruct (classic_dec (Finite A /\ Finite B)) as [hf | nhf].
destruct hf as [hfa hfb].
pose proof (finite_set_list_no_dup _ hfa) as h0.
pose proof (finite_set_list_no_dup _ hfb) as h0'.
destruct h0 as [la h0].
destruct h0' as [lb h0'].
destruct h0 as [h0l h0r].
destruct h0' as [h0l' h0r'].
assert (hn: Finite A -> Finite B -> 
                list_to_set la = A /\ list_to_set lb = B).
  symmetry in h0l. symmetry in h0l'. tauto.
assert (h6':list_to_set (list_prod la lb) = cart_prod A B).
  apply list_prod_cart_prod_compat; auto.
assert (h9':NoDup (list_prod la lb)).
  apply no_dup_list_prod; assumption.
assert (h10':lp = nil). 
  symmetry in x.
  pose proof (cart_prod_empty_rev A B x) as h10.
  destruct h10 as [h10l | h10r].
  generalize dependent F.
  rewrite h10l.
  intros F h11 h12.
  rewrite fin_map_cart_empty11 in h12.
  pose proof (empty_set_nil _ h12).
  subst. reflexivity.
  generalize dependent F.
  rewrite h10r.
  intros F h11 h12.
  rewrite fin_map_cart_empty21 in h12.
  pose proof (empty_set_nil _ h12). subst. reflexivity.
assert (h11':In lp (list_power (list_prod la lb) lc)).
  rewrite <- x in h6'.
  pose proof (empty_set_nil _ h6') as h12.
  rewrite h12. simpl. left. rewrite h10'. reflexivity.
assert (h12':functionally_paired_l (list_prod la lb) lc lp).
  rewrite <- x in h6'.
  pose proof (empty_set_nil _ h6') as h12.
  rewrite h12. rewrite h10'.
  apply fpl_empty1.
assert (h13': if eq_dec (list_prod la lb) nil
                then (la = nil /\ A = Empty_set _) \/
                     (lb = nil /\ B = Empty_set _)
                else 
                  (list_to_set la = A /\ list_to_set lb = B)).
  destruct (eq_dec (list_prod la lb) nil) as [h14 | h15].
  symmetry in x.
  pose proof (@cart_prod_empty_rev _ _ _ _ x) as h15.
  destruct h15 as [h15l | h15r].
  left. rewrite h15l in h0l. symmetry in h0l. 
  pose proof (empty_set_nil _ h0l).
  split; auto.
  right. 
  rewrite h15r in h0l'. symmetry in h0l'.
  pose proof (empty_set_nil _ h0l').
  split; auto.
  contradict h15.
  rewrite <- x in h6'.
  apply empty_set_nil. assumption.
assert (h13'': if (eq_dec (list_prod la lb) nil) then
                    (A = Empty_set _ -> la = nil) /\
                    (B = Empty_set _ -> lb = nil)
              else True).
  destruct (eq_dec (list_prod la lb) nil) as [h14 | h15].
  split.
  intro h16. rewrite h16 in h0l. symmetry in h0l.
  apply empty_set_nil. assumption.
  intro h16. rewrite h16 in h0l'. symmetry in h0l'.
  apply empty_set_nil. assumption.
  constructor.
exists (Build_nice_map_lists2 _ _ _ _ _ _ _ F la lb lc lp hn h6' h13' h13'' h3l h5l h0r h0r' h9' h3r h5r h11' h12'). constructor.
assert (h6:list_to_set (list_prod nil nil) = cart_prod A B).
  simpl. assumption.
pose proof (NoDup_nil T) as h7.
pose proof (NoDup_nil U) as h8.
assert (h9:NoDup (list_prod (@nil T) (@nil U))).
  simpl. constructor.
assert (h10:lp = nil).
symmetry in x.
  pose proof (cart_prod_empty_rev A B x) as h10.
  destruct h10 as [h10l | h10r].
  generalize dependent F.
  rewrite h10l.
  intros F h11 h12.
  rewrite fin_map_cart_empty11 in h12.
  pose proof (empty_set_nil _ h12). 
  subst.  reflexivity.
  generalize dependent F.
  rewrite h10r.
  intros F h11 h12.
  rewrite fin_map_cart_empty21 in h12.
  pose proof (empty_set_nil _ h12).
  subst. reflexivity.
assert (h11:In lp (list_power (list_prod nil nil) lc)).
  simpl. left. subst. reflexivity.
assert (h12: functionally_paired_l (list_prod nil nil) lc lp).
  simpl. subst.
  apply fpl_empty1.
assert (h13: if eq_dec (list_prod (@nil T) (@nil U)) nil
                then ((@nil T) = nil /\ A = Empty_set _) \/
                     ((@nil U) = nil /\ B = Empty_set _)
                else 
                  (list_to_set nil = A /\ list_to_set nil = B)).
  simpl.
  destruct (eq_dec nil nil) as [h13 | h14].
  symmetry in x.
  pose proof (cart_prod_empty_rev _ _ x) as h14.
  destruct h14.
  left. split; auto. right. split; auto.
  contradict h14. reflexivity.
assert (h13': if (eq_dec (list_prod (@nil T) (@nil U)) nil) then
                    (A = Empty_set _ -> (@nil T) = nil) /\
                    (B = Empty_set _ -> (@nil U) = nil)
              else True).
  simpl. 
  destruct (eq_dec nil nil) as [h13' | h14'].
  tauto. constructor.
assert (hn: Finite A -> Finite B -> 
                list_to_set nil = A /\ list_to_set nil = B).
  tauto.
exists (Build_nice_map_lists2 _ _ _ _ _ _ _ F nil nil lc lp hn h6 h13 h13' h3l h5l h7 h8 h9 h3r h5r h11 h12).
constructor.
(* Add *)
pose proof (fin_map_fin_dom F) as h2.
pose proof (fin_map_fin_ran F) as h3.
pose proof (Add_intro2 _ A0 x0) as h4.
rewrite x in h4.
pose proof (Inhabited_intro _ _ _ h4) as h5.
pose proof (cart_prod_fin_rev _ _ h2 h5) as h6.
destruct h6 as [h6l h6r].
pose proof (finite_set_list_no_dup _ h6l) as h7.
pose proof (finite_set_list_no_dup _ h6r) as h8.
pose proof (finite_set_list_no_dup _ h3) as h9.
destruct h7 as [la h7].
destruct h8 as [lb h8].
destruct h9 as [lc h9].
destruct h7 as [h7l h7r].
destruct h8 as [h8l h8r].
destruct h9 as [h9l h9r].
symmetry in h7l.
symmetry in h8l.
symmetry in h9l.
pose proof (list_prod_cart_prod_compat A B _ _ h7l h8l) as h10.
pose proof (no_dup_list_prod  _ _ h7r h8r) as h11.
pose proof (fin_map_to_list_pairs_lemma F _ lc h11 h10 h9l) as h12.
destruct h12 as [lp h12].
destruct h12 as [h12a [h12b h12c]].
pose proof (in_list_power_synced _ _ _ h12c) as h13.
pose proof (fpl_in_list_power _ lc _ h13 h11 h12b) as h7. 
pose proof h12c as h14.
rewrite <- h7 in h14.
assert (h15: if (eq_dec (list_prod la lb) nil)
                then (la = nil /\ A = Empty_set _) \/
                     (lb = nil /\ B = Empty_set _)
                else list_to_set la = A /\ list_to_set lb = B).
  destruct (eq_dec (list_prod la lb) nil) as [h16 | h17].
  pose proof h5 as h15.
  rewrite <- h10 in h15.
  rewrite h16 in h15.
  simpl in h15.
  inversion h15.
  contradiction.
  split; assumption.
assert (h15': if (eq_dec (list_prod la lb) nil) 
                   then (A = Empty_set _ -> la = nil) /\
                        (B = Empty_set _ -> lb = nil)
              else True).
  destruct (eq_dec (list_prod la lb) nil) as [h16 | h17].
  pose proof h5 as h15'.
  rewrite <- h10 in h15'.
  rewrite h16 in h15'.
  simpl in h15'.
  inversion h15'.
  contradiction.
  constructor.
assert (hn: Finite A -> Finite B -> 
                list_to_set la = A /\ list_to_set lb = B).
  tauto.
exists (Build_nice_map_lists2 _ _ _ _ _ _ _ F la lb lc lp hn h10 h15 h15' h9l h12a h7r h8r h11 h9r h12b h12c h14).
constructor.
Qed.

Lemma fin_map_ex_nice_map_lists2_list_to_set_dom :
  forall {T U V:Type} {la:list T} {lb:list U} (C:Ensemble V) 
         {def:V}
         (F:Fin_map (cart_prod (list_to_set la) (list_to_set lb))
                    C def),
            NoDup la -> NoDup lb ->
            exists nml2:(nice_map_lists2 F),
              la = n_la2 _ nml2 /\
              lb = n_lb2 _ nml2. 
intros T U V la lb C def F ha hb.
pose proof (fin_map_fin_ran F) as h1.
pose proof (finite_set_list_no_dup _ h1) as h3.
destruct h3 as [lc h3].
destruct h3 as [h3l h3r].
symmetry in h3l.
assert (h4:list_to_set (list_prod la lb) = cart_prod (list_to_set la) (list_to_set lb)).
  apply list_prod_cart_prod_compat; auto.
assert (h5: if eq_dec (list_prod la lb) nil
                 then
                  la = nil /\ list_to_set la = Empty_set T \/
                  lb = nil /\ list_to_set lb = Empty_set U
                 else list_to_set la = list_to_set la /\ 
                      list_to_set lb = list_to_set lb).
  destruct (eq_dec (list_prod la lb)) as [h5 | h6].
  assert (h6:list_to_set (list_prod la lb) = Empty_set _).
    rewrite h5. simpl. reflexivity.
  rewrite h4 in h6.
  pose proof (cart_prod_empty_rev _ _ h6) as h7.
  destruct h7 as [h7l | h7r].
  left.
  split; auto. apply empty_set_nil. assumption.
  right. split; auto. apply empty_set_nil. assumption.
  split; auto.
assert (h6:if eq_dec (list_prod la lb) nil then
             (list_to_set la = Empty_set T -> la = nil) /\
             (list_to_set lb = Empty_set U -> lb = nil)
             else True).
  destruct (eq_dec (list_prod la lb)).
  split.
  intro h6. pose proof (empty_set_nil _ h6). assumption.
  intro h7. pose proof (empty_set_nil _ h7). assumption.
  constructor.
pose proof (no_dup_list_prod _ _ ha hb) as h7.
pose proof (fin_map_to_list_pairs_lemma F (list_prod la lb) _ h7 h4 h3l) as h3.
destruct h3 as [lp h3].
destruct h3 as [h3a [h3b h3c]].
pose proof (in_list_power_synced _ _ _ h3c) as h4'.
pose proof (fpl_in_list_power _ lc _ h4' h7 h3b) as h5'.
pose proof h3c as h8.
rewrite <- h5' in h8.
assert (h9:Finite (list_to_set la) -> Finite (list_to_set lb) ->
           list_to_set la = list_to_set la /\ 
           list_to_set lb = list_to_set lb).
auto.
                                               
exists (Build_nice_map_lists2 _ _ _ _ _ _ _ F la lb lc lp h9 h4 h5 h6 h3l h3a ha hb h7 h3r h3b h3c h8).
simpl. split; auto.
Qed.

Lemma fin_map_ex_nice_map_lists2_dom :
  forall {T U V:Type} {A:Ensemble T} {B:Ensemble U} 
         {C:Ensemble V} {def:V} 
         (F:Fin_map (cart_prod A B) C def)
         (la:list T) (lb:list U),
    list_to_set la = A -> list_to_set lb = B ->
            NoDup la -> NoDup lb ->
            exists nml2:(nice_map_lists2 F),
              la = n_la2 _ nml2 /\
              lb = n_lb2 _ nml2.
intros T U V A B C def F la lb hla hlb ha hb.
pose proof (fin_map_fin_ran F) as h1.
pose proof (finite_set_list_no_dup _ h1) as h3.
destruct h3 as [lc h3].
destruct h3 as [h3l h3r].
symmetry in h3l.
assert (h4:list_to_set (list_prod la lb) = cart_prod A B).
  apply list_prod_cart_prod_compat; auto.
assert (h5: if eq_dec (list_prod la lb) nil
                 then
                  la = nil /\ A = Empty_set T \/
                  lb = nil /\ B = Empty_set U
                 else list_to_set la = A /\ 
                      list_to_set lb = B).
  destruct (eq_dec (list_prod la lb)) as [h5 | h6].
  assert (h6:list_to_set (list_prod la lb) = Empty_set _).
    rewrite h5. simpl. reflexivity.
  rewrite h4 in h6.
  pose proof (cart_prod_empty_rev _ _ h6) as h7.
  destruct h7 as [h7l | h7r].
  left. split; auto. apply empty_set_nil. rewrite <- h7l. assumption.
  right. split; auto. apply empty_set_nil. rewrite <- h7r. assumption.
  split; auto.
assert (h6:if eq_dec (list_prod la lb) nil then
             (A = Empty_set T -> la = nil) /\
             (B = Empty_set U -> lb = nil)
             else True).
  destruct (eq_dec (list_prod la lb)).
  split.
  intro h6. rewrite h6 in hla. apply empty_set_nil. assumption.
  intro h6. rewrite h6 in hlb. apply empty_set_nil. assumption.
  constructor.
pose proof (no_dup_list_prod _ _ ha hb) as h7.
pose proof (fin_map_to_list_pairs_lemma F (list_prod la lb) _ h7 h4 h3l) as h3.
destruct h3 as [lp h3].
destruct h3 as [h3a [h3b h3c]].
pose proof (in_list_power_synced _ _ _ h3c) as h4'.
pose proof (fpl_in_list_power _ lc _ h4' h7 h3b) as h5'.
pose proof h3c as h8.
rewrite <- h5' in h8.
assert (h9:Finite A -> Finite B ->
           list_to_set la = A /\ list_to_set lb = B). auto.
exists (Build_nice_map_lists2 _ _ _ _ _ _ _ F la lb lc lp h9 h4 h5 h6 h3l h3a ha hb h7 h3r h3b h3c h8).
simpl. split; auto.
Qed.


Lemma n_im_im_fin_map_compat : 
  forall {T U:Type} {A:Ensemble T} 
         {B:Ensemble U} {def:U}
         (F:Fin_map A B def) (nml:nice_map_lists F), 
    list_to_set (n_im F nml) = im_fin_map F.
intros T U A B def F nml.
unfold n_im. unfold im_fin_map.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros y h1.
rewrite <- list_to_set_in_iff in h1.
rewrite in_map_iff in h1.

destruct h1 as [pr h1].
destruct h1 as [h1l h1r].
rewrite list_to_set_in_iff in h1r.
rewrite (lp_compat F nml) in h1r.
pose proof (fin_map_to_fps_compat F) as h2.
subst.
apply Im_intro with (fst pr).
destruct h2 as [h1 h2].
destruct h1 as [h3 h4].
pose proof (h4 _ h1r) as h5. 
destruct h5 as [h5l h5r]; assumption.
symmetry.
apply fin_map_to_fps_fin_map_app_compat. assumption.
(* >= *)
red.
intros y h1.
destruct h1 as [x h2 y].
subst.
rewrite <- list_to_set_in_iff.
rewrite in_map_iff.
exists (x, F |-> x). simpl. split.
reflexivity. 
pose proof (lp_compat F nml) as h3.
rewrite list_to_set_in_iff.
rewrite h3.
pose proof (fin_map_to_fps_compat F) as h4.
destruct h4 as [h4 h5].
rewrite (fps_to_f_compat h4 def).
constructor.
simpl.
split.
rewrite h5. reflexivity.
assumption.
Qed.


Lemma fpl_f_compat : 
  forall {T U:Type} {A:Ensemble T} {B:Ensemble U}
         {def:U} (F:Fin_map A B def) 
         (nml:nice_map_lists F) (x:T),
         (fpl F nml, def) l-> x = F |-> x.
intros T U A B def F nml x.
unfold fin_map_app.
destruct F as [h1 h2 S h3].
unfold fpl_app. simpl.
destruct constructive_definite_description as [y h4]. simpl.
destruct (in_dec eq_dec x (n_la (fin_map_intro A B def h1 h2 S h3) nml)) as [h5 | h6].
pose proof (la_compat _ nml) as h6.
rewrite list_to_set_in_iff in h5.
rewrite h6 in h5.
pose proof (lp_compat _ nml) as h7.
rewrite list_to_set_in_iff in h4.
rewrite h7 in h4.
rewrite <- fin_map_to_fps_compat_s in h4.
rewrite (fps_to_f_compat h3 def) in h4.
destruct h4 as [h4l  h4r].
simpl in h4l.
destruct h4l; assumption.
subst. symmetry.
unfold fps_to_f.
pose proof (la_compat _ nml) as h7.
rewrite list_to_set_in_iff in h6.
rewrite h7 in h6.
destruct (classic_dec (Ensembles.In A x)).
contradiction.
reflexivity.
Qed.


Lemma fpl_f_compat_list_to_set : 
  forall {T U:Type} (li:list T) (lj:list U) (def:U)
         (F:Fin_map (list_to_set li) (list_to_set lj) def)
         (nml:nice_map_lists F) 
         (pf:In (n_lp F nml) (list_power li lj))
         (pfi:NoDup li) (pfj:NoDup lj) (x:T),
    fpl_app
      (in_list_power_fpl li lj (n_lp F nml) pfi
                         (list_power_no_dup li lj (n_lp F nml) pfi pfj pf) pf) def x = 
    F |-> x.
intros T U li lj def F nml h1 h2 h3 x.
destruct F.
unfold fin_map_app.
unfold fpl_app.
destruct  constructive_definite_description as [y h4]. simpl.
destruct (in_dec eq_dec x li) as [h5 | h6].
pose proof (lp_compat _ nml) as h6.
rewrite list_to_set_in_iff in h4.
rewrite h6 in h4.
rewrite <- fin_map_to_fps_compat_s in h4.
rewrite (fps_to_f_compat f1 def) in h4.
destruct h4 as [h4l h4r].
simpl in h4l.
destruct h4l; auto.
unfold fps_to_f.
destruct (classic_dec (Ensembles.In (list_to_set li) x)) as [h7 | h8].
destruct f1.
destruct constructive_definite_description. simpl.
rewrite <- list_to_set_in_iff in h7.
contradiction. auto.
Qed.


Lemma fpl_f_compat_pseudo_list_to_set : 
  forall {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
         (F:Fin_map I J def)
         (li:list T) (lj:list U) (lp:list (T*U))
         (pf:In lp (list_power li lj))
         (pfi:NoDup li) (pfj:NoDup lj) (x:T),
    list_to_set li = I -> list_to_set lj = J ->
    list_to_set lp = fin_map_to_fps F ->
    fpl_app
      (in_list_power_fpl li lj lp pfi
                         (list_power_no_dup li lj lp pfi pfj pf) pf) def x = 
    F |-> x.
intros T U I J def F li lj lp h1 h2 h3 x h2' h3' h4'.
destruct F.
unfold fin_map_app.
unfold fpl_app.
destruct constructive_definite_description as [y h4].  simpl.
destruct (in_dec eq_dec x li) as [h5 | h6]. 
rewrite list_to_set_in_iff in h4.
rewrite h4' in h4.
rewrite <- fin_map_to_fps_compat_s in h4.
rewrite (fps_to_f_compat f1 def) in h4. 
destruct h4 as [h4].
destruct h4 as [h4l h4r].
simpl in h4l.
assumption.
unfold fps_to_f.
rewrite list_to_set_in_iff in h6.
rewrite h2' in h6.
destruct (classic_dec (Ensembles.In I x)) as [h8 | h9].
contradiction.
assumption.
Qed.


Lemma fpl_f_compat_pseudo_list_to_set' : 
  forall {T U:Type} {I:Ensemble T} {J:Ensemble U} {def:U}
         (F:Fin_map I J def)
         (li:list T) (lj:list U)
         (nml:nice_map_lists F) 
         (pf:In (n_lp F nml) (list_power li lj))
         (pfi:NoDup li) (pfj:NoDup lj) (x:T),
    list_to_set li = I -> list_to_set lj = J ->
    fpl_app
      (in_list_power_fpl li lj (n_lp F nml) pfi
                         (list_power_no_dup li lj (n_lp F nml) pfi pfj pf) pf) def x = 
    F |-> x.
intros T U I J def F li lj nml h1 h2 h3 x h2' h3'.
destruct F.
unfold fin_map_app.
unfold fpl_app.
destruct constructive_definite_description as [y h4]. simpl.
destruct (in_dec eq_dec x li) as [h5 | h6]. 
pose proof (lp_compat _ nml) as h6.
rewrite list_to_set_in_iff in h4.
rewrite h6 in h4.
rewrite <- fin_map_to_fps_compat_s in h4.
rewrite (fps_to_f_compat f1 def) in h4.
destruct h4 as [h4l h4r].
simpl in h4l.
destruct h4l; auto.
rewrite list_to_set_in_iff in h6.
rewrite h2' in h6.
unfold fps_to_f. 
destruct (classic_dec (Ensembles.In I x)) as [h8 | h9].
contradiction.
assumption.
Qed.


Lemma fpl2_f_compat : 
  forall {T U V:Type} {A:Ensemble T} {B:Ensemble U} {C:Ensemble V}
         {def:V} (F:Fin_map (cart_prod A B) C def) 
         (nml:nice_map_lists2 F) (pr:T*U),
         (fpl2 F nml, def) l-> pr = F |-> pr.
intros T U V A B C def F nml pr.
unfold fin_map_app.
dependent destruction F.
symmetry.
destruct (classic_dec (Ensembles.In S 
                    (pr, (fpl2 (fin_map_intro (cart_prod A B) C def f f0 S f1) nml, def) l-> pr))) as [h0 | h0'].
pose proof (fin_map_s_compat  _ _ _ def  _ f1 h0) as h2.
simpl in h2.
assumption.
unfold fpl_app.
destruct constructive_definite_description as [z h2].
simpl.
destruct (in_dec eq_dec pr) as [h8 | h9].
rewrite list_to_set_in_iff in h2.
rewrite lp_compat2 in h2.
pose proof (fin_map_to_fps_compat_s (cart_prod A B) C def f f0 _ f1) as h9.
rewrite <- h9 in h2.
pose proof (fin_map_s_compat _ _ _ def _ f1 h2) as h10.
simpl in h10.
assumption.
simpl in h2. subst.
rewrite list_to_set_in_iff in h9.
rewrite (lab_compat2 _ nml) in h9.
unfold fps_to_f.
destruct (classic_dec (Ensembles.In (cart_prod A B) pr)).
contradiction.
reflexivity.
Qed.

Lemma im1_im1l_compat :
  forall {T U V:Type} {A:Ensemble T} {B:Ensemble U}
         {C:Ensemble V} {def:V}
         (F:Fin_map (cart_prod A B) C def)
         (nml:(nice_map_lists2 F)),
      (forall x:T, Ensembles.In A x -> im1 F x = list_to_set (im1l (fpl2 F nml) def x)).
intros T U V A B C def F nml x h0.
apply Extensionality_Ensembles.
red. split.
red.
(* <= *)
intros z h1.
rewrite <- list_to_set_in_iff.
unfold im1l. unfold im1 in h1.
destruct (eq_dec A (Empty_set T)).
rewrite e in h0.
contradiction.
destruct h1 as [y h1].
subst.
rewrite in_map_iff.
exists y.
split.
apply (fpl2_f_compat F nml (x, y)).
pose proof (not_empty_Inhabited _ _ n) as h2.
destruct h2 as [a h2]. 
assert (h3:Ensembles.In (cart_prod A B) (a, y)).
  constructor; split; assumption.
rewrite <- (lab_compat2 _ nml) in h3.
rewrite <- list_to_set_in_iff in h3.
unfold n_lab2 in h3.
rewrite in_prod_iff in h3.
destruct h3; assumption. 
(* >= *)
red.
intros z h1.
pose proof (lab_empty2 _ nml) as h2.
destruct (eq_dec (n_lab2 F nml) nil) as [h3 | h4]. 
rewrite <- list_to_set_in_iff in h1.
unfold im1l in h1.
rewrite in_map_iff in h1.
unfold im1.
destruct h1 as [y h1].
destruct h1  as [h1l h1r].   
pose proof (fpl_app_compat  (fpl2 _ nml) (x,y) def) as h2'.
destruct (in_dec eq_dec (x, y) (n_lab2 F nml)) as [h3' | h4']. 
rewrite h3 in h3'. 
contradiction. 
destruct h2 as [h2l | h2r].
destruct h2l as [h2a h2b].
rewrite h2b in h0.
contradiction.
destruct (eq_dec A (Empty_set _ )) as [h5 | h6].
rewrite h5 in h0.
contradiction.
destruct h2r as [h2a h2b].
rewrite h2a in h1r.
contradiction.
destruct h2 as [h2l h2r].
unfold im1. 
rewrite <- list_to_set_in_iff in h1.
unfold im1l in h1.
destruct (eq_dec A (Empty_set _)) as [h5 | h6].
rewrite h5 in h0.
contradiction. 
rewrite in_map_iff in h1.
destruct h1 as [y h1].
destruct h1 as [h1l h1r].
rewrite list_to_set_in_iff in h1r.
rewrite h2r in h1r.
exists y. assumption.
rewrite <- h1l.
apply (fpl2_f_compat _ nml (x, y)).
Qed.


Lemma im2_im2l_compat :
  forall {T U V:Type} {A:Ensemble T} {B:Ensemble U}
         {C:Ensemble V} {def:V}
         (F:Fin_map (cart_prod A B) C def)
         (nml:(nice_map_lists2 F)),
      (forall y:U, Ensembles.In B y -> im2 F y = list_to_set (im2l (fpl2 F nml) def y)).
intros T U V A B C def F nml y h0.
apply Extensionality_Ensembles.
red. split.
red.
(* <= *)
intros z h1.
rewrite <- list_to_set_in_iff.
unfold im2l. unfold im2 in h1.
destruct (eq_dec B (Empty_set U)).
rewrite e in h0.
contradiction.
destruct h1 as [x h1].
subst.
rewrite in_map_iff.
exists x.
split.
apply (fpl2_f_compat F nml (x, y)).
pose proof (not_empty_Inhabited _ _ n) as h2.
destruct h2 as [b h2]. 
assert (h3:Ensembles.In (cart_prod A B) (x, b)).
  constructor; split; assumption.
rewrite <- (lab_compat2 _ nml) in h3.
rewrite <- list_to_set_in_iff in h3.
unfold n_lab2 in h3.
rewrite in_prod_iff in h3.
destruct h3; assumption. 
(* >= *)
red.
intros z h1.
pose proof (lab_empty2 _ nml) as h2.
destruct (eq_dec (n_lab2 F nml) nil) as [h3 | h4]. 
rewrite <- list_to_set_in_iff in h1.
unfold im2l in h1. 
rewrite in_map_iff in h1.
unfold im2.
destruct h1 as [x h1].
destruct h1  as [h1l h1r].   
pose proof (fpl_app_compat  (fpl2 _ nml) (x,y) def) as h2'.
destruct (in_dec eq_dec (x, y) (n_lab2 F nml)) as [h3' | h4']. 
rewrite h3 in h3'. 
contradiction.  
destruct h2 as [h2l | h2r].
Focus 2.
destruct h2r as [h2a h2b].
rewrite h2b in h0.
contradiction.
destruct (eq_dec B (Empty_set _ )) as [h5 | h6].
rewrite h5 in h0.
contradiction.
destruct h2l as [h2a h2b].
rewrite h2a in h1r.
contradiction.
destruct h2 as [h2l h2r].
unfold im2. 
rewrite <- list_to_set_in_iff in h1.
unfold im2l in h1.
destruct (eq_dec B (Empty_set _)) as [h5 | h6].
rewrite h5 in h0.
contradiction. 
rewrite in_map_iff in h1.
destruct h1 as [x h1].
destruct h1 as [h1l h1r].
rewrite list_to_set_in_iff in h1r.
rewrite h2l in h1r.
exists x. assumption.
rewrite <- h1l.
apply (fpl2_f_compat _ nml (x, y)).
Qed.