(* Copyright (C) 2014-2015, Daniel Wyckoff.*)
(*This file is part of BooleanAlgebrasIntro2.

BooleanAlgebrasIntro2 is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

BooleanAlgebrasIntro2 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with BooleanAlgebrasIntro2.  If not, see <http://www.gnu.org/licenses/>.*)

Require Export Bool.
Require Export Ensembles.
Require Export Constructive_sets.
Require Export Setoid.
Require Import List.
Require Import EnsemblesImplicit.
Require Import ImageImplicit.
Require Import Description.
Require Import SetUtilities.
Require Import FiniteMaps.
Require Import FunctionalExtensionality.
Require Import TypeUtilities.
Require Import LogicUtilities.
Require Import DecidableDec.
Require ListUtilities.

(* I use two records to describe a single 
   Boolean algebra for notational reasons; I haven't figured out if 
   I can combine Bconst and Bool_Alg using notations that don't
   get confused with Arith.*)

Record Bconst : Type :=
  {Btype : Type;
   BS : Ensemble Btype;
   Bplus : Btype -> Btype -> Btype;
   Btimes : Btype -> Btype -> Btype;
   Bone : Btype;
   Bzero : Btype;
   Bcomp : Btype -> Btype}.

Infix "+" := (Bplus _) (at level 50, left associativity).
Infix "*" := (Btimes _) (at level 40, left associativity).

Notation "0" := (Bzero _).
Notation "1" := (Bone _).
Notation "- x" := ((Bcomp _) x).

Record Bool_Alg : Type :=
  {Bc : Bconst;   
   und_set : (BS Bc) = Full_set (Btype Bc);
   assoc_sum : forall n m p:(Btype Bc), n + (m + p) = n + m + p;
   assoc_prod : forall n m p:(Btype Bc), n * (m * p) = n * m * p;
   comm_sum : forall n m:(Btype Bc), n + m = m + n;
   comm_prod : forall n m:(Btype Bc), n * m = m * n;
   abs_sum  : forall n m:(Btype Bc), n + (n * m) = n;
   abs_prod:  forall n m:(Btype Bc), n * (n + m) = n;
   dist_sum : forall n m p:(Btype Bc), p*(n + m) = p * n + p * m;
   dist_prod: forall n m p:(Btype Bc), p+(n * m) = (p+n) * (p+m);
   comp_sum:  forall n:(Btype Bc), n + (- n) = 1;
   comp_prod: forall n:(Btype Bc), n * (- n) = 0}.


Hint Resolve assoc_sum assoc_prod comm_sum comm_prod abs_sum
             abs_prod dist_sum dist_prod comp_sum comp_prod.

Lemma bconst_ext : 
  forall (Bc Bc':Bconst)
         (pf:Btype Bc = Btype Bc'),
    BS Bc = transfer_dep_r pf (BS Bc') ->
    Bplus Bc = transfer_dep_r (U:=fun T=>T->T->T) pf (Bplus Bc') ->
    Btimes Bc = transfer_dep_r (U:=fun T=>T->T->T) pf (Btimes Bc') ->
    Bone Bc = transfer_dep_r  (U:=id) pf (Bone Bc') ->
    Bzero Bc = transfer_dep_r  (U:=id) pf (Bzero Bc') ->
    Bcomp Bc = transfer_dep_r  (U:=fun T=>T->T) pf (Bcomp Bc') ->
    Bc = Bc'.
intros Bc Bc' h1 h2 h3 h4 h5 h6 h7.
destruct Bc, Bc'. simpl.
simpl in h1, h2, h3, h4, h5, h6, h7.
subst.
do 6 rewrite transfer_dep_r_eq_refl.
reflexivity.
Qed.


Definition bt (B:Bool_Alg) := Btype (Bc B).
Definition ba_ens (B:Bool_Alg) := Full_set (bt B).


(*Throughout the development there's a need for handling
  families of separate algebras or finding extensions of a given algebra,
  and for this purpose, [Bool_Alg] doesn't suffice, since 
  [Ensemlbe Bool_Alg] yields a universe inconsistency error.
  I introduce a "dual" record pair
  Bconst_p / Bool_Alg_p, the p standing for parametric.
  The difference is that in these structures there is a parametric
  type T, which is necessary to compare elements in distinct 
  algebras or taking extensions.  Every statment about a Bool_Alg
  has a corresponding statement about Bool_Alg_p (but not
  vice versa).*)

Record Bconst_p (T:Type) : Type :=
  {A_p : Ensemble T;
   Btype_p := sig_set A_p;
   BS_p : Ensemble Btype_p;
   Bplus_p : Btype_p -> Btype_p -> Btype_p;
   Btimes_p : Btype_p -> Btype_p -> Btype_p;
   Bone_p : Btype_p;
   Bzero_p : Btype_p;
   Bcomp_p : Btype_p -> Btype_p}.
   

Infix "%+" := (Bplus_p _ _) (at level 50, left associativity).
Infix "%*" := (Btimes_p _ _) (at level 40, left associativity).

Notation "%0" := (Bzero_p _ _).
Notation "%1" := (Bone_p _ _).
Notation "%- x" := ((Bcomp_p _ _) x) (at level 30).

Definition bconst_conv {T:Type} (Bc_p:Bconst_p T) : Bconst :=
  Build_Bconst (sig_set (A_p T Bc_p)) (BS_p T Bc_p) (Bplus_p T Bc_p) (Btimes_p T Bc_p) (Bone_p T Bc_p) (Bzero_p T Bc_p) (Bcomp_p T Bc_p).


Lemma bconst_ext_p :
  forall {T:Type} (Bcp Bcp':Bconst_p T)
    (pf:A_p T Bcp = A_p T Bcp'),
    BS_p T Bcp = transfer_dep_r (sig_set_eq _ _ pf) (BS_p T Bcp') ->
    Bplus_p T Bcp = transfer_dep_r  (sig_set_eq _ _ pf) (U:=fun T=>T->T->T) (Bplus_p T Bcp') ->
    Btimes_p T Bcp = transfer_dep_r  (sig_set_eq _ _ pf) (U:=fun T=>T->T->T) (Btimes_p T Bcp') ->
    Bone_p T Bcp = transfer_dep_r   (U:=id) (sig_set_eq _ _ pf) (Bone_p T Bcp') ->
    Bzero_p T Bcp = transfer_dep_r  (U:=id) (sig_set_eq _ _ pf) (Bzero_p T Bcp') ->
    Bcomp_p T Bcp = transfer_dep_r  (U:=fun T=>T->T) (sig_set_eq _ _ pf) (Bcomp_p T Bcp')
    -> Bcp = Bcp'.
intros T Bcp Bcp' h0  h1 h2 h3 h4 h5 h6.
destruct Bcp, Bcp'. simpl.
simpl in h0, h1.
destruct h0.
simpl in h1, h2, h3, h4, h5, h6.
unfold sig_set_eq in h1, h2, h3, h4, h5, h6.
unfold eq_ind_r in h1, h2, h3, h4, h5, h6.
unfold eq_ind in h1, h2, h3, h4, h5, h6.
simpl in h1, h2, h3, h4, h5, h6.
rewrite transfer_dep_r_eq_refl in h1, h2, h3, h4, h5, h6.
subst.
reflexivity.
Qed.


Record Bool_Alg_p (T:Type) : Type :=
  {Bc_p : Bconst_p T;   
   und_set_p : (BS_p T Bc_p) = Full_set (Btype_p T Bc_p);
   assoc_sum_p : forall n m p:(Btype_p T Bc_p), n %+ (m %+ p) = n %+ m %+ p;
   assoc_prod_p : forall n m p:(Btype_p T Bc_p), n %* (m %* p) = n %* m %* p;
   comm_sum_p : forall n m:(Btype_p T Bc_p), n %+ m = m %+ n;
   comm_prod_p : forall n m:(Btype_p T Bc_p), n %* m = m %* n;
   abs_sum_p  : forall n m:(Btype_p T Bc_p), n %+ (n %* m) = n;
   abs_prod_p :  forall n m:(Btype_p T Bc_p), n %* (n %+ m) = n;
   dist_sum_p : forall n m p:(Btype_p T Bc_p), p%*(n %+ m) = p %* n %+ p %* m;
   dist_prod_p : forall n m p:(Btype_p T Bc_p), p%+(n %* m) = (p%+n) %* (p%+m);
   comp_sum_p :  forall n:(Btype_p T Bc_p), n %+ (%- n) = %1;
   comp_prod_p : forall n:(Btype_p T Bc_p), n %* (%- n) = %0}.

Definition ba_p_ens {T:Type} (Bp:Bool_Alg_p T) := A_p T (Bc_p T Bp).
Definition btp {T:Type} (Bp:Bool_Alg_p T) := Btype_p T (Bc_p T Bp).

Lemma incl_im_proj1_sig_ba_p_ens : 
  forall {T:Type} {Bp:Bool_Alg_p T} (S:Ensemble (btp Bp)),
    Included (im_proj1_sig S) (ba_p_ens Bp).
intros T Bp S. red. intros x h1. 
destruct h1 as [x h1]. subst.
apply proj2_sig.
Qed.


Definition ba_conv {T:Type} (B:Bool_Alg_p T) : Bool_Alg.
pose (bconst_conv (Bc_p T B)) as Bc.
refine (Build_Bool_Alg Bc (und_set_p T B) (assoc_sum_p T B) (assoc_prod_p T B) (comm_sum_p T B) (comm_prod_p T B) (abs_sum_p T B) (abs_prod_p T B) (dist_sum_p T B) (dist_prod_p T B) (comp_sum_p T B) (comp_prod_p T B)).
Defined.


Lemma ba_p_ens_eq :
  forall {T:Type} (Bp:Bool_Alg_p T),
    ba_p_ens Bp = im_proj1_sig (ba_ens (ba_conv Bp)).
intros T Bp.
unfold ba_p_ens, ba_ens.
apply Extensionality_Ensembles.
red. split.
red.
intros x h1.
apply Im_intro with (exist _ _ h1). constructor. simpl. reflexivity.
red. intros x h1. destruct h1 as [x h1]. subst.
apply proj2_sig.
Qed.



Lemma ba_conv_type : 
  forall {T:Type} (Bp:Bool_Alg_p T),
         btp Bp = bt (ba_conv Bp).
auto.
Defined.

Definition ba_conv_elt {T:Type} {Bp:Bool_Alg_p T}
           (x:btp Bp) :=
  transfer (ba_conv_type Bp) x.

Lemma ba_conv_elt_eq : 
  forall {T:Type} {Bp:Bool_Alg_p T}
    (x:btp Bp),
  x = ba_conv_elt x.
intros T B x. unfold ba_conv_type.
unfold transfer. unfold eq_rect_r. simpl.
reflexivity.
Qed.


Lemma ba_conv_plus : forall {T:Type} (Bp:Bool_Alg_p T)
                     (x y:btp Bp),
                     x %+ y = (ba_conv_elt x) + 
                              (ba_conv_elt y).
simpl. intros T B x y.
pose proof (ba_conv_elt_eq x) as h1. simpl in h1.
pose proof (ba_conv_elt_eq y) as h2. simpl in h2.
rewrite <- h1. rewrite <- h2.
reflexivity.
Qed.

Lemma ba_conv_times : forall {T:Type} (Bp:Bool_Alg_p T)
                     (x y:btp Bp),
                     x %* y = (ba_conv_elt x) *
                              (ba_conv_elt y).
simpl. intros T B x y.
pose proof (ba_conv_elt_eq x) as h1. simpl in h1.
pose proof (ba_conv_elt_eq y) as h2. simpl in h2.
rewrite <- h1. rewrite <- h2.
reflexivity.
Qed.

Lemma ba_conv_comp : forall {T:Type} (Bp:Bool_Alg_p T)
                     (x:btp Bp),
                     %- x = -(ba_conv_elt x).
simpl. intros T B x.
pose proof (ba_conv_elt_eq x) as h1. simpl in h1.
rewrite <- h1.
reflexivity.
Qed.

Lemma ba_conv_one : 
  forall {T:Type} (Bp:Bool_Alg_p T),
    Bone_p T (Bc_p T Bp) = Bone (Bc (ba_conv Bp)).
simpl. auto.
Qed.

Lemma ba_conv_zero : 
  forall {T:Type} (Bp:Bool_Alg_p T),
    Bzero_p T (Bc_p T Bp) = Bzero (Bc (ba_conv Bp)).
simpl. auto.
Qed.

Definition ba_conv_set {T:Type} {Bp:Bool_Alg_p T} 
           (A:Ensemble (btp Bp)) := 
  (transfer_dep (ba_conv_type Bp) A).

Lemma bc_subst : forall (Bc Bc':Bconst)
                        (P:Bconst->Prop),
                   Bc = Bc' -> (P Bc <-> P Bc').
intros; subst. tauto.
Qed.


Definition ba_conv_list {T:Type} {Bp:Bool_Alg_p T} 
           (l:list (Btype_p T (Bc_p T Bp))) :=
  transfer_dep (ba_conv_type Bp) l.


Definition ba_conv_list_list {T:Type} {Bp:Bool_Alg_p T} 
           (l:list (List.list (btp Bp))) :=
  transfer_dep (U:=(fun V=>list (list V))) (ba_conv_type Bp) l.




Definition ba_conv_fun {T:Type} {Ap Bp:Bool_Alg_p T} 
           (f:btp Ap->btp Bp) : bt (ba_conv Ap)->bt (ba_conv Bp).
intro x.
refine (f x).
Defined.


Definition ba_conv_fun1 {T:Type} {Ap:Bool_Alg_p T} {T':Type} 
           (f:btp Ap->T') : bt (ba_conv Ap)->T'.
intro x.
refine (f x).
Defined.


Definition ba_conv_fun2 {T':Type} {T:Type} {Ap:Bool_Alg_p T}  
           (f:T'->btp Ap) : T'-> bt (ba_conv Ap).
intro x.
refine (f x).
Defined.



Definition ba_conv_sig_fun1 {T:Type} {Bp:Bool_Alg_p T} 
           {E:Ensemble (btp Bp)} {U:Type} 
           (g:sig_set E->U) : sig_set (ba_conv_set E)->U.
intro p.
refine (g p).
Defined.


Definition bc_sig_set_conv 
  {T:Type} (A:Ensemble T) (Bc:Bconst)
  (pf:Btype Bc = sig_set A) : Bconst_p T :=
  Build_Bconst_p T A (transfer_dep pf (BS Bc)) 
                 (transfer_dep (U:=fun T=>T->T->T) pf (Bplus Bc))
                 (transfer_dep (U:=fun T=>T->T->T) pf (Btimes Bc))
                 (transfer pf (Bone Bc))
                 (transfer pf (Bzero Bc))
                 (transfer_dep (U:=fun T=>T->T) pf (Bcomp Bc)).



Lemma ba_sig_set_conv_und_set : 
  forall {T:Type} {A:Ensemble T} {B:Bool_Alg}
         {pf:bt B = sig_set A},
    BS_p T (bc_sig_set_conv _ _ pf) = Full_set (Btype_p T (bc_sig_set_conv _ _ pf)).
intros T A B h1.
simpl. apply Extensionality_Ensembles.
red. split.
red. intros x h2. constructor.
red. intros x h2. clear h2.  
unfold bc_sig_set_conv, Btype_p in x. simpl in x.
rewrite <- (transfer_undoes_transfer_r h1).
rewrite <- transfer_in.
rewrite und_set.
constructor.
Qed.



Lemma transfer_dep_fun_plus : 
  forall {T:Type} {A:Ensemble T} {Bc:Bconst}
         {pf:Btype Bc = sig_set A},
    transfer_dep (U:=fun T=>T->T->T) pf (Bplus Bc) =
    fun (x:sig_set A) (y:sig_set A) =>
      transfer pf ((transfer_r pf x) + (transfer_r pf y)).
intros T A Bc h1.
destruct Bc. simpl in h1. subst.
rewrite transfer_dep_eq_refl. simpl.
apply functional_extensionality. intro x. apply functional_extensionality. intro y.
rewrite transfer_eq_refl. do 2 rewrite transfer_r_eq_refl.
reflexivity.
Qed.

Lemma transfer_dep_fun_times : 
  forall {T:Type} {A:Ensemble T} {Bc:Bconst}
         {pf:Btype Bc = sig_set A},
    transfer_dep (U:=fun T=>T->T->T) pf (Btimes Bc) =
    fun (x:sig_set A) (y:sig_set A) =>
      transfer pf ((transfer_r pf x) * (transfer_r pf y)).
intros T A Bc h1.
destruct Bc. simpl in h1. subst.
rewrite transfer_dep_eq_refl. simpl.
apply functional_extensionality. intro x. apply functional_extensionality. intro y.
rewrite transfer_eq_refl. do 2 rewrite transfer_r_eq_refl.
reflexivity.
Qed.


Lemma transfer_dep_fun_comp : 
  forall {T:Type} {A:Ensemble T} {Bc:Bconst}
         {pf:Btype Bc = sig_set A},
    transfer_dep (U:=fun T=>T->T) pf (Bcomp Bc) =
    fun (x:sig_set A)  =>
      transfer pf (- (transfer_r pf x)).
intros T A Bc h1.
destruct Bc. simpl in h1. subst.
rewrite transfer_dep_eq_refl. simpl.
apply functional_extensionality. intro x.
rewrite transfer_eq_refl, transfer_r_eq_refl.
reflexivity.
Qed.


Lemma ba_sig_set_conv_assoc_sum : 
  forall {T:Type} {A:Ensemble T} {B:Bool_Alg}
         {pf:bt B = sig_set A},
    forall n m p : Btype_p T (bc_sig_set_conv _ _ pf) , n %+ (m %+ p) = n %+ m %+ p.  
intros T A B h1 n m p. simpl. 
rewrite transfer_dep_fun_plus. 
do 2 rewrite transfer_r_undoes_transfer.
rewrite assoc_sum.
reflexivity.
Qed.

Lemma ba_sig_set_conv_assoc_prod : 
  forall {T:Type} {A:Ensemble T} {B:Bool_Alg}
         {pf:bt B = sig_set A},
   forall n m p : Btype_p T (bc_sig_set_conv _ _ pf), 
     n %* (m %* p) = n %* m %* p.
intros T A B h1 n m p. simpl. 
rewrite transfer_dep_fun_times. 
do 2 rewrite transfer_r_undoes_transfer.
rewrite assoc_prod.
reflexivity.
Qed.


Lemma ba_sig_set_conv_comm_sum : 
  forall {T:Type} {A:Ensemble T} {B:Bool_Alg}
         {pf:bt B = sig_set A},
    forall n m : Btype_p T (bc_sig_set_conv _ _ pf), 
      n %+ m = m %+ n.
intros T A B h1 n m. simpl. 
rewrite transfer_dep_fun_plus. 
rewrite comm_sum.
reflexivity.
Qed.

Lemma ba_sig_set_conv_comm_prod :
  forall {T:Type} {A:Ensemble T} {B:Bool_Alg}
         {pf:bt B = sig_set A},
    forall n m : Btype_p T (bc_sig_set_conv _ _ pf), 
      n %* m = m %* n.
intros T A B h1 n m. simpl. 
rewrite transfer_dep_fun_times. 
rewrite comm_prod.
reflexivity.
Qed.

Lemma ba_sig_set_conv_abs_sum :
  forall {T:Type} {A:Ensemble T} {B:Bool_Alg}
         {pf:bt B = sig_set A},
  forall n m : Btype_p T (bc_sig_set_conv _ _ pf), n %+ n %* m = n.
intros T A B h1 n m. simpl.  
rewrite transfer_dep_fun_plus, transfer_dep_fun_times. 
rewrite transfer_r_undoes_transfer.
rewrite abs_sum. rewrite transfer_undoes_transfer_r.
reflexivity.
Qed.


Lemma ba_sig_set_conv_abs_prod :
  forall {T:Type} {A:Ensemble T} {B:Bool_Alg}
         {pf:bt B = sig_set A},
  forall n m : Btype_p T (bc_sig_set_conv _ _ pf), n %* (n %+ m) = n.
intros T A B h1 n m. simpl.  
rewrite transfer_dep_fun_plus, transfer_dep_fun_times. 
rewrite transfer_r_undoes_transfer.
rewrite abs_prod. rewrite transfer_undoes_transfer_r.
reflexivity.
Qed.

Lemma ba_sig_set_conv_dist_sum : 
  forall {T:Type} {A:Ensemble T} {B:Bool_Alg}
         {pf:bt B = sig_set A},
  forall n m p : Btype_p T (bc_sig_set_conv _ _ pf),
                 p %* (n %+ m) = p %* n %+ p %* m.
intros T A B h1 n m p. simpl. 
rewrite transfer_dep_fun_plus, transfer_dep_fun_times.
do 3 rewrite transfer_r_undoes_transfer.
rewrite dist_sum.
reflexivity.
Qed.


Lemma ba_sig_set_conv_dist_prod : 
  forall {T:Type} {A:Ensemble T} {B:Bool_Alg}
         {pf:bt B = sig_set A},
  forall n m p : Btype_p T (bc_sig_set_conv _ _ pf),
                 p %+ n %* m = (p %+ n) %* (p %+ m).
intros T A B h1 n m p. simpl. 
rewrite transfer_dep_fun_plus, transfer_dep_fun_times.
do 3 rewrite transfer_r_undoes_transfer.
rewrite dist_prod.
reflexivity.
Qed.

Lemma ba_sig_set_conv_comp_sum : 
  forall {T:Type} {A:Ensemble T} {B:Bool_Alg}
         {pf:bt B = sig_set A},
  forall n : Btype_p T (bc_sig_set_conv _ _ pf), n %+ %-n = %1.
intros T A B h1 n. simpl.
rewrite transfer_dep_fun_plus, transfer_dep_fun_comp.
rewrite transfer_r_undoes_transfer.
rewrite comp_sum.
reflexivity.
Qed.

Lemma ba_sig_set_conv_comp_prod : 
  forall {T:Type} {A:Ensemble T} {B:Bool_Alg}
         {pf:bt B = sig_set A},
  forall n : Btype_p T (bc_sig_set_conv _ _ pf), n %* %-n = %0.
intros T A B h1 n. simpl.
rewrite transfer_dep_fun_times, transfer_dep_fun_comp.
rewrite transfer_r_undoes_transfer.
rewrite comp_prod.
reflexivity.
Qed.



Definition ba_sig_set_conv   
           {T:Type} (A:Ensemble T) (B:Bool_Alg)
           (pf:bt B = sig_set A) : Bool_Alg_p T :=
  Build_Bool_Alg_p T (bc_sig_set_conv _ _ pf) 
                         ba_sig_set_conv_und_set
                         ba_sig_set_conv_assoc_sum
                         ba_sig_set_conv_assoc_prod
                         ba_sig_set_conv_comm_sum
                         ba_sig_set_conv_comm_prod
                         ba_sig_set_conv_abs_sum
                         ba_sig_set_conv_abs_prod
                         ba_sig_set_conv_dist_sum 
                         ba_sig_set_conv_dist_prod
                         ba_sig_set_conv_comp_sum 
                         ba_sig_set_conv_comp_prod.



Lemma ba_sig_set_conv_bc_p_compat : 
  forall 
    {T:Type} (A:Ensemble T) (B:Bool_Alg)
    (pf:bt B = sig_set A),
    Bc_p T (ba_sig_set_conv A B pf) = bc_sig_set_conv A (Bc B) pf.
intros; auto.
Qed.

Section Basics.
Variable B:Bool_Alg.
Let Bt := bt B.


(*We start out with some tactics for simplifying Boolean algebraic expressions,
but it turns out that to use these tactics requires module functionality, which 
is very inconvenient, and using these tactics is like the equivalent of doing a quicksort on a small list.  
so I have included these for reference purposes only.  I use them
in a few places in this file only.
Someone can make these commutativity/associativity tactics more ubiquitious if they want,
by incoporating them into modules.*)
Section assoc.

(*************)
(* All these functions/tactics in "Section assoc" and "Section commut" 
  are adapted from the one-operation versions in Coq'Art Section 16.3.*)


(* Binary tree with two types of nodes corresponding to two binary operations, plus and times,
   and a leaf which contains an integer rank corresponding to its position in a derived
   list.  *)
Inductive bin : Set := pnode : bin -> bin -> bin | 
                        tnode : bin -> bin -> bin |
                        leaf : nat-> bin.

Fixpoint flatten_aux_p (p fin:bin){struct p} : bin :=
  match p with
  | pnode x1 x2 => flatten_aux_p x1 (flatten_aux_p x2 fin)
  | x => pnode x fin
  end.

Fixpoint flatten_aux_t (t fin:bin){struct t} : bin :=
  match t with
  | tnode x1 x2 => flatten_aux_t x1 (flatten_aux_t x2 fin)
  | x => tnode x fin
  end.

Fixpoint flatten (t:bin) : bin :=
  match t with
  | pnode (tnode x1 x2 as y1) (tnode x3 x4 as y2) => pnode (flatten y1) (flatten y2)
  | tnode (pnode x1 x2 as y1) (pnode x3 x4 as y2) => tnode (flatten y1) (flatten y2)
  | tnode x1 x2 => flatten_aux_t (flatten x1) (flatten x2)
  | pnode x1 x2 => flatten_aux_p (flatten x1) (flatten x2)
  | x => x
  end.

Fixpoint bin_Bt (l: list Bt)(def:Bt)(t:bin){struct t}: Bt :=
  match t with
  | pnode t1 t2 => (bin_Bt l def t1) + (bin_Bt l def t2)
  | tnode t1 t2 => (bin_Bt l def t1) * (bin_Bt l def t2)
  | leaf n=> nth n l def
  end.

Theorem flatten_aux_valid_Bt_p :
  forall (l:list Bt)(def:Bt)(t t':bin),
  (bin_Bt l def t) + (bin_Bt l def t') = 
  bin_Bt l def (flatten_aux_p t t').
intros l def t.  elim t; simpl; auto. intros t1 iht1 t2 iht2 t'.
rewrite <- iht1. rewrite <- iht2. rewrite (assoc_sum B).
reflexivity.
Qed.

Theorem flatten_aux_valid_Bt_t :
  forall (l:list Bt)(def:Bt)(t t':bin),
  (bin_Bt l def t) * (bin_Bt l def t') =
  bin_Bt l def (flatten_aux_t t t').
intros l def t.  elim t; simpl; auto.  intros t1 iht1 t2 iht2 t'.
rewrite <- iht1.  rewrite <- iht2. rewrite (assoc_prod B).
reflexivity.
Qed.

Theorem flatten_valid_Bt :
  forall (l:list Bt)(def:Bt)(t:bin),
  bin_Bt l def t = bin_Bt l def (flatten t).
intros l def t. elim t; simpl; auto.
intros t1 iht1 t2 iht2. destruct t1. destruct t2.
rewrite iht1. rewrite iht2. rewrite flatten_aux_valid_Bt_p. reflexivity.
rewrite iht1. rewrite iht2. rewrite flatten_aux_valid_Bt_p. reflexivity.
rewrite iht1. rewrite iht2. rewrite flatten_aux_valid_Bt_p. reflexivity.
destruct t2.  rewrite iht1. rewrite iht2. rewrite flatten_aux_valid_Bt_p. reflexivity.
rewrite iht1. rewrite iht2. unfold bin_Bt at 3. fold bin_Bt. reflexivity.
rewrite iht1. rewrite iht2. rewrite flatten_aux_valid_Bt_p. reflexivity.
rewrite iht1. rewrite iht2. rewrite flatten_aux_valid_Bt_p. reflexivity.


intros t1 iht1 t2 iht2. destruct t1. destruct t2.
unfold bin_Bt at 3.  fold bin_Bt.
rewrite <- iht1; rewrite <- iht2; reflexivity.
rewrite iht1; rewrite iht2; rewrite flatten_aux_valid_Bt_t;  reflexivity.
rewrite iht1. rewrite iht2. rewrite flatten_aux_valid_Bt_t.  reflexivity.
rewrite iht1. rewrite iht2. rewrite flatten_aux_valid_Bt_t.  reflexivity.
rewrite iht1. rewrite iht2. rewrite flatten_aux_valid_Bt_t.  reflexivity.
Qed.



Theorem flatten_valid_Bt_2 :
  forall (t t':bin)(l:list Bt)(def:Bt),
    bin_Bt l def (flatten t) = bin_Bt l def (flatten t') ->
    bin_Bt l def t = bin_Bt l def t'.
intros t t' l def; rewrite (flatten_valid_Bt l def t); rewrite (flatten_valid_Bt l def t').
trivial.
Qed.
End assoc.

Ltac term_list l v :=
  match v with
  | (?X1 + ?X2) => let l1 := term_list l X2 in term_list l1 X1
  | (?X1 * ?X2) => let l1 := term_list l X2 in term_list l1 X1
  | ?X1 => constr:(cons X1 l)
  end.

Ltac compute_rank l n v :=
  match l with
  | (cons ?X1 ?X2) =>
    let t1 := constr:X2 in
    match constr:(X1 = v) with
    | (?X1 = ?X1) => n
    | _ => compute_rank t1 (S n) v
    end
  end.

Ltac model_aux l v :=
  match v with
  | (?X1 + ?X2) =>
    let r1 := model_aux l X1 with r2 := model_aux l X2 in
      constr:(pnode r1 r2)
  | (?X1 * ?X2) =>
    let r1 := model_aux l X1 with r2 := model_aux l X2 in
      constr:(tnode r1 r2)
  | ?X1 => let n := compute_rank l 0%nat X1 in constr:(leaf n)
  | _ => constr:(leaf 0%nat)
  end.
 
Ltac model_Bt def v :=
  let l := term_list (nil (A:=Bt)) v in
  let t := model_aux l v in
  constr:(bin_Bt l def t).

Ltac assoc_eq:=
  match goal with
  | [|- (?X1 = ?X2) ] =>
  let term1 := model_Bt X1 X1
  with term2 := model_Bt X1 X2 in
  (change (term1 = term2);
  apply flatten_valid_Bt_2; auto)
  end.

(*
Variables a b c d e f g h i j k l: Bt.

Goal ((a+(b+c))*(d+(e+f)))*(g+(h+i)) = ((a+b)+c) * (((d+e)+f)*((g+h)+i)).
assoc_eq.
Qed.
(*Checks*)

Goal ((a*(b*c))+(d*(e*f)))+(g*(h*i)) = ((a*b)*c) + (((d*e)*f)+((g*h)*i)).
assoc_eq.
Qed.
(*Checks*)

Goal (a+(b*(c*d)))+(e*f*(g+h))+(i*j*(k+l)) = (a + ((b*c)*d)) + ((e*(f*(g+h))) + (i*(j*(k+l)))).
assoc_eq.
(*Checks*)

Goal (a+b+c+d)*(e+f+g+h)*(i+j+k+l) = ((a+b)+(c+d))*(((e+f)+(g+h))*((i+j)+(k+l))).
assoc_eq.
(*Checks*)

Goal (a*b*c*d)+(e*f*g*h)+(i*j*k*l) = ((a*b)*(c*d))+(((e*f)*(g*h))+((i*j)*(k*l))).
assoc_eq.
(*Checks*)
*)

Section commut.

Fixpoint nat_eq_bool (m n:nat){struct n} : bool :=
  match m, n with
  | 0, 0 => true
  | S m, 0 => false
  | 0, S n => false
  | S m, S n => nat_eq_bool m n
  end.

Fixpoint nat_lt_bool (m n:nat){struct n} : bool :=
  match m, n with
  | _, 0 => false
  | 0, S n => true
  | S m, S n => nat_lt_bool m n
  end.

Fixpoint bin_eq_bool (t t':bin){struct t'} : bool :=
  match t, t' with
  | (leaf m), (leaf n) => nat_eq_bool m n
  | (pnode p1 p2), (pnode p'1 p'2) => andb (bin_eq_bool p1 p'1) (bin_eq_bool p2 p'2)
  | (tnode t1 t2), (tnode t'1 t'2) => andb (bin_eq_bool t1 t'1) (bin_eq_bool t2 t'2)
  | _, _ => false
  end.

(*The order is arbitrary.  Leaves are less than pnodes are less than tnodes, all other cases
 depend on the constructors. *)

Fixpoint bin_lt_bool (t t':bin){struct t'} : bool :=
  match t, t' with
  | (leaf m), (leaf n) => nat_lt_bool m n
  | (leaf m), _=> true
  | (pnode p1 p2), (pnode p'1 p'2) => if (bin_eq_bool p1 p'1) then (bin_lt_bool p2 p'2)
                                        else (bin_lt_bool p1 p'1)
  | (pnode p1 p2), (leaf m) => false
  | (pnode _ _), (tnode _ _) => true
  | (tnode t1 t2), (tnode t'1 t'2) => if (bin_eq_bool t1 t'1) then (bin_lt_bool t2 t'2)
                                        else (bin_lt_bool t1 t'1)
  | (tnode t1 t2), _ => false
  end.

Fixpoint insert_bin_p (t0:bin)(t:bin){struct t} : bin :=
  match t with
  | leaf m => match (bin_lt_bool t0 t) with
              | true => pnode t0 t
              | false => pnode t t0
              end
  | tnode _ _ => match (bin_lt_bool t0 t) with
                 | true => pnode t0 t
                 | false => pnode t t0
                 end
  | pnode ((leaf m) as l) t' =>
    match (bin_lt_bool t0 l) with
    | true => pnode t0 t
    | false => pnode l (insert_bin_p t0 t')
    end
  | pnode ((tnode _ _) as x) t' =>
    match (bin_lt_bool t0 x) with
    | true => pnode t0 t
    | false => pnode x (insert_bin_p t0 t')
    end
  | t => pnode t0 t
  end.


Fixpoint insert_bin_t (t0:bin)(t:bin){struct t} : bin :=
  match t with
  | leaf m => match (bin_lt_bool t0 t) with
              | true => tnode t0 t
              | false => tnode t t0
              end
  | pnode _ _ => match (bin_lt_bool t0 t) with
                 | true => tnode t0 t
                 | false => tnode t t0
                 end
  | tnode ((leaf m) as l) t' =>
    match (bin_lt_bool t0 l) with
    | true => tnode t0 t
    | false => tnode l (insert_bin_t t0 t')
    end
  | tnode ((pnode _ _) as x) t' =>
    match (bin_lt_bool t0 x) with
    | true => tnode t0 t
    | false => tnode x (insert_bin_t t0 t')
    end
  | t => tnode t0 t
  end.

Fixpoint sort_bin (t:bin) : bin :=
  match t with
  | tnode ((leaf n) as l) t' => insert_bin_t l (sort_bin t')
  | tnode (pnode _ _ as p) t' => insert_bin_t (sort_bin p) (sort_bin t')
  | pnode ((leaf n) as l) p' => insert_bin_p l (sort_bin p')
  | pnode (tnode _ _ as t') p' => insert_bin_p (sort_bin t') (sort_bin p')
  | t => t
end.

Theorem bin_Bt_ref_p : forall (l:list Bt)(def:Bt)(t1 t2:bin), 
  bin_Bt l def (pnode t1 t2) = (bin_Bt l def t1) + (bin_Bt l def t2).
intros l def t1 t2.  unfold bin_Bt. fold bin_Bt. reflexivity.
Qed.

Theorem bin_Bt_ref_t : forall (l:list Bt)(def:Bt)(t1 t2:bin),
  bin_Bt l def (tnode t1 t2) = (bin_Bt l def t1) * (bin_Bt l def t2).
intros l def t1 t2. unfold bin_Bt. fold bin_Bt. reflexivity.
Qed.

Theorem insert_eq_p :
  forall (l:list Bt)(def:Bt)(t0 t:bin),
  bin_Bt l def (insert_bin_p t0 t) = (bin_Bt l def t0) + (bin_Bt l def t).
intros l def t0 t.  induction t; unfold insert_bin_p. case t1.
intros b b0. apply bin_Bt_ref_p.
(*next subgoal*)
fold insert_bin_p. 
intros b b0. case (bin_lt_bool t0 (tnode b b0)).
apply bin_Bt_ref_p.
(*next subgoal*)
rewrite bin_Bt_ref_p. rewrite IHt2.
unfold bin_Bt. fold bin_Bt. 
rewrite (assoc_sum B) with (n:= bin_Bt l def t0).
rewrite (comm_sum B) with (n := bin_Bt l def t0) 
  (m := bin_Bt l def b * bin_Bt l def b0).
apply (assoc_sum B).
(*next subgoal*)
fold insert_bin_p.
intros n.
case (bin_lt_bool t0 (leaf n)).
rewrite bin_Bt_ref_p. reflexivity.
rewrite bin_Bt_ref_p. rewrite IHt2.
rewrite bin_Bt_ref_p.
rewrite (assoc_sum B) with (n := bin_Bt l def t0).
rewrite (comm_sum B) with (n := bin_Bt l def t0)
  (m := bin_Bt l def (leaf n)).
rewrite (assoc_sum B). reflexivity.
case (bin_lt_bool t0 (tnode t1 t2)).
rewrite bin_Bt_ref_p. reflexivity.
rewrite bin_Bt_ref_p. apply (comm_sum B). 
case (bin_lt_bool t0 (leaf n)).
rewrite bin_Bt_ref_p. reflexivity.
rewrite bin_Bt_ref_p. apply (comm_sum B).
Qed.

Theorem insert_eq_t :
  forall (l:list Bt)(def:Bt)(t0 t:bin),
  bin_Bt l def (insert_bin_t t0 t) = (bin_Bt l def t0) * (bin_Bt l def t).
intros l def t0 t.  induction t; unfold insert_bin_t. 
case (bin_lt_bool t0 (pnode t1 t2)).
rewrite bin_Bt_ref_t. reflexivity.
rewrite bin_Bt_ref_t. apply (comm_prod B).
fold insert_bin_t. destruct t1.
case (bin_lt_bool t0 (pnode t1_1 t1_2)).
apply bin_Bt_ref_t.
rewrite bin_Bt_ref_t. 
rewrite IHt2.
unfold bin_Bt. fold bin_Bt.
rewrite (assoc_prod B) with (n := bin_Bt l def t0).
rewrite (comm_prod B) with (n := bin_Bt l def t0)
  (m := (bin_Bt l def t1_1 + bin_Bt l def t1_2)).
apply (assoc_prod B).
unfold bin_Bt.  fold bin_Bt. reflexivity.
case (bin_lt_bool t0 (leaf n)).
apply bin_Bt_ref_t. 
rewrite bin_Bt_ref_t.
rewrite IHt2.
rewrite bin_Bt_ref_t.
rewrite (assoc_prod B) with (n := bin_Bt l def t0).
rewrite (comm_prod B) with (n := bin_Bt l def t0)
  (m := bin_Bt l def (leaf n)).
apply (assoc_prod B).
case (bin_lt_bool t0 (leaf n)).
apply bin_Bt_ref_t.
rewrite bin_Bt_ref_t. apply (comm_prod B).
Qed.

Theorem sort_eq : forall (l: list Bt)(def:Bt)(t:bin),
  bin_Bt l def (sort_bin t) = bin_Bt l def t.
intros l def. induction t.
unfold sort_bin. destruct t1. reflexivity.
fold sort_bin. unfold sort_bin in IHt1. fold sort_bin in IHt1.
rewrite insert_eq_p. rewrite IHt1.  rewrite IHt2. rewrite <- bin_Bt_ref_p.
reflexivity.
fold sort_bin. rewrite insert_eq_p. rewrite IHt2.
rewrite bin_Bt_ref_p. reflexivity.
unfold sort_bin.  destruct t1. fold sort_bin.
unfold sort_bin in IHt1. fold sort_bin in IHt1. rewrite insert_eq_t.
rewrite IHt1. rewrite IHt2. rewrite <- bin_Bt_ref_t.  reflexivity.
reflexivity.
fold sort_bin. rewrite insert_eq_t. rewrite bin_Bt_ref_t. rewrite IHt2.
reflexivity.
unfold sort_bin. reflexivity.
Qed.

Theorem sort_eq_2 :
  forall (l:list Bt)(def:Bt)(t1 t2:bin),
  bin_Bt l def (sort_bin t1) = bin_Bt l def (sort_bin t2) ->
  bin_Bt l def t1 = bin_Bt l def t2.
intros l def t1 t2.
rewrite sort_eq. rewrite sort_eq. trivial.
Qed.

End commut.


Ltac comm_eq :=
  match goal with
  | [ |- (?X1 = ?X2) ] =>
  let l := term_list (nil (A := Bt)) X1 in
  let term1 := model_aux l X1
  with term2 := model_aux l X2 in
  (change (bin_Bt l X1 term1 = bin_Bt l X1 term2);
    apply flatten_valid_Bt_2; 
    apply sort_eq_2;
    auto)
  end.

(*
(*All check!*)
Variables a b c d e f g h i j k l : Bt.
Goal (a*b)+(c*d) = (b*a) +(d*c).
comm_eq.
Goal (a*b)+(d*c)+(e*f*g) = (g*f*e) + (c*d) + (b*a).
comm_eq.
Goal (a*b*c+(d*e*f+(h*i*j)+k) = k + c*b*a + (f*d*e) + (j*i*h)).
comm_eq.
Goal a*(b+c)*d = (c+b)*d*a.
comm_eq.
Goal a+(b*(c+(d*e*f))*g) = ((f*e*d) + c)*b*g + a.
comm_eq.
*)

(*Theorems *)

Lemma in_bs : forall x:Bt, Ensembles.In (BS (Bc B)) x.
intros; rewrite und_set; constructor.
Qed.



Lemma bc_inj : 
  forall {A A':Bool_Alg}, 
    Bc A = Bc A' -> A = A'.
intros A A' h1.
destruct A; destruct A'.
simpl in h1. subst.
f_equal; apply proof_irrelevance.
Qed.


Lemma dist_sum_r : forall (n m p: Bt), (n + m) * p = n*p + m*p.
intros n m p.  
assert (h1: (n + m) * p = p * (n + m)).  apply (comm_prod B).
assert (h2: p * (n + m) = p*n + p*m).  apply (dist_sum B).
assert (h3: p*n + p*m = n*p + m*p). comm_eq. congruence.
Qed.
Hint Resolve dist_sum_r.



Lemma dist_prod_r : forall (n m p: Bt), (n*m) + p = (n+p) * (m + p).
intros n m p.  assert (h1: (n*m) + p = p + (n * m)).  apply (comm_sum B).
assert (h2: p + (n * m) = (p + n) * (p + m)).  apply (dist_prod B). 
assert (h3: (p+n)*(p+m) = (n+p) * (m+p)). comm_eq. congruence.
Qed.
Hint Resolve dist_prod_r.

Lemma idem_sum : forall b : Bt, b + b = b.
intros b.  assert (h1: (b+b = b + b * (b + b))).
assert (h2: b * (b + b) = b). rewrite (abs_prod B). reflexivity. 
rewrite h2. reflexivity. rewrite (abs_sum B) in h1. assumption. 
Qed.
Hint Resolve idem_sum.

Lemma idem_prod: forall (b: Bt), b * b = b.
intros. assert (s: (b*b = b * (b + (b * b)))).
assert (s: b + (b * b) = b). apply (abs_sum B). rewrite s.
reflexivity. rewrite s. rewrite (abs_prod B). reflexivity. 
Qed.
Hint Resolve idem_prod.

Lemma eq_ord: forall (x y: Bt),
  x + y = y <-> x * y = x.                                                     
intros. assert (h: x + y = y -> x * y = x).
intros h1. assert (h2: x * y = x * (x + y)). rewrite h1.
reflexivity. assert (h3: x * (x + y) = x).  apply abs_prod.
congruence.
assert (h4: x * y = x -> x + y = y).
intro h5.
assert (h4: x + x * y = x). rewrite h5. apply idem_sum.
assert (h6: y + y * x = y).  apply abs_sum.
assert (h7: y + y * x = x*y + y).  comm_eq.
congruence. tauto.
Qed.
Hint Resolve eq_ord.

Definition le (x y:Bt): Prop :=
x + y = y.
Hint Unfold le.


Lemma refl_le : forall (x: Bt), le x x.
intros.  assert (x + x = x).  rewrite idem_sum. reflexivity.
auto.
Qed.
Hint Resolve refl_le.

Lemma anti_sym_le: forall (x y: Bt), 
  le x y -> le y x -> x = y.
intros x y h1 h2.  assert (h3: x + y = y). intuition.
assert (h4:y + x = x). intuition.  assert (h5: x + y = y + x).
rewrite comm_sum. reflexivity. rewrite <- h3.
rewrite h5. rewrite h4. reflexivity.
Qed.
Hint Resolve anti_sym_le.

Lemma trans_le: forall (x y z : Bt),
  le x y -> le y z -> le x z.
intros x y z h1 h2.  assert (h3: x + z = x + (y + z)). rewrite h2. 
reflexivity.  assert (h4: x + (y + z) = x + y + z). assoc_eq.
assert (h5: x + y + z = y + z). rewrite h1. reflexivity.
assert (h6: y + z = z). rewrite h2. reflexivity. assert (h7: x + z = z).
congruence. auto.
Qed.
Hint Resolve trans_le.

Lemma times_le : forall (x y : Bt), le (x * y) x.
intros x y.
red.
rewrite eq_ord.
rewrite <- assoc_prod.
rewrite (comm_prod _ y x).
rewrite assoc_prod.
rewrite idem_prod.
reflexivity.
Qed.

Lemma le_plus : forall (x y : Bt), le x (x + y).
intros x y.
red.
rewrite assoc_sum.
rewrite idem_sum.
reflexivity.
Qed.

Definition lt (x y:Bt) := le x y /\ x <> y.
 
(* b is a lower bound of S in B*)
Definition lb (S: Ensemble Bt) (b: Bt) : Prop :=
  forall (s:Bt), Ensembles.In S s -> le b s.
Hint Unfold lb.

(* b is an upper bound of S in B*)
Definition ub (S: Ensemble Bt) (b: Bt) : Prop :=   
  forall (s:Bt), Ensembles.In S s -> le s b.
Hint Unfold ub.

(* b is the infimum (greatest lower bound) of S in B *)
Definition inf (S: Ensemble Bt) (b: Bt) : Prop :=
  lb S b /\ forall b':Bt, lb S b' -> le b' b.
Hint Unfold inf.

(* b is the supremum (least upper bound) of S in B*)
Definition sup (S: Ensemble Bt) (b: Bt) : Prop :=
  ub S b /\ forall b':Bt, ub S b' -> le b b'.
Hint Unfold sup.


Lemma inf_le : forall (A:Ensemble Bt) (b:Bt),
                 inf A b -> forall b':Bt, Ensembles.In A b' -> le b b'.
intros A b h1 b' h2.
red in h1.
destruct h1 as [h1].
red in h1.
apply h1; auto.
Qed.

Lemma le_sup : forall (A:Ensemble Bt) (b:Bt),
                 sup A b -> forall b':Bt, Ensembles.In A b' -> le b' b.
intros A b h1 b' h2.
red in h1. destruct h1 as [h1].
red in h1.
apply h1; auto.
Qed.




Lemma inf_singleton : forall (x:Bt), inf (Singleton x) x.
intro x.
red. split.
(* lower bound *)
red.
intros y h1.
inversion h1.
subst.
auto.
(* greatest lower bound *)
intros b' h1.
red in h1.
pose proof (Singleton_intro _ _ _ (eq_refl x)) as h2.
apply h1. assumption.
Qed.

Lemma sup_singleton : forall (x:Bt), sup (Singleton x) x.
intro x.
red. split.
(* lower bound *)
red.
intros y h1.
inversion h1.
subst.
auto.
(* greatest lower bound *)
intros b' h1.
red in h1.
pose proof (Singleton_intro _ _ _ (eq_refl x)) as h2.
apply h1. assumption.
Qed.


(* B is a lattice under partial order le (sum portion) *)
Lemma lat_sum : forall (x y: Bt), 
  sup (Couple x y) (x + y).
intros x y.  assert (h1: x + (x + y) = (x + x) + y).
assoc_eq. 
assert (h2: x + x + y = x + y). rewrite idem_sum. reflexivity.
assert (h3: le x (x + y)). unfold le. rewrite h1. rewrite h2. reflexivity.
assert (h4: le y (y + x)). unfold le.  rewrite (assoc_sum B). rewrite idem_sum.
reflexivity. assert (h5: ub (Couple x y) (x+y)). unfold ub. intros s.  
intros h7.  assert (h8: s = x \/ s = y). apply Couple_inv.
apply h7.  destruct h8 as [h8l | h8r]. rewrite h8l. apply h3.  rewrite h8r.
rewrite (comm_sum B). apply h4.  unfold sup.  split.  assumption. intros b' h11.
unfold le.  assert (h15: y + b' = b').  apply h11.  apply (Couple_r Bt).  
assert (h16: x + y + b' = x + (y + b')).  assoc_eq. 
rewrite h16. rewrite h15.  apply h11. apply (Couple_l Bt).
Qed.
Hint Resolve lat_sum.

(* B is a lattice under partial order le (product portion) *)
Lemma lat_prod : forall (x y: Bt),
  inf (Couple x y) (x * y).
intros x y.  assert (h0: lb (Couple x y) (x * y)).  
unfold lb.  intros s h2.  assert (h3: s = x \/ s = y).  apply Couple_inv.
assumption.  destruct h3 as [h3l | h3r]. rewrite h3l.  unfold le.
apply eq_ord.  assert (h4: y * x = x * y). apply (comm_prod B).
assert (h5: x * y * x = x * (y * x)). assoc_eq.   
rewrite h5.  rewrite h4.  assert (h6: x * (x * y) = x * x * y). 
rewrite (assoc_prod B). reflexivity.  rewrite h6.  rewrite idem_prod.
reflexivity.  rewrite h3r.  unfold le.  apply eq_ord.
assert (h7: x * y * y = x * (y * y)).  rewrite (assoc_prod B). reflexivity.
rewrite h7.  rewrite idem_prod. reflexivity. unfold inf.  split.
assumption.  intros z h10. unfold le.  apply eq_ord.  
assert (h11: z * (x * y) = z * x * y).  rewrite <- (assoc_prod B).
reflexivity. rewrite h11.  assert (h12: le z x). apply h10.
apply (Couple_l Bt). assert (h13: z * x = z).
apply eq_ord. apply h12. rewrite h13.  assert (h14: le z y).
apply h10.  apply (Couple_r Bt). apply eq_ord. 
apply h14.
Qed.
Hint Resolve lat_prod.

(* infimum unique *)
Lemma inf_unq : forall (S:Ensemble Bt) (x y: Bt), 
  inf S x -> inf S y -> x = y.
intros S x y h2 h3.  assert (h4: le x y).  
assert (h5: lb S y). apply h3. apply h3. apply h2.
assert (h6: le y x). assert (h7: lb S x).  apply h2. apply h2.
apply h3.  apply anti_sym_le. assumption.
assumption. 
Qed.
Hint Resolve inf_unq.

(* supremum unique *)
Lemma sup_unq : forall (S:Ensemble Bt) (x y: Bt), 
  sup S x -> sup S y -> x = y.
intros S x y h2 h3.  assert (h4: le x y). assert (h5: ub S x).
apply h2. apply h2.  apply h3. assert (h6: le y x).  assert (h7: ub S y).
apply h3. apply h3.  apply h2.  apply anti_sym_le.  assumption.
assumption.
Qed.
Hint Resolve sup_unq.

Lemma ex_inf_unq : 
  forall (A:Ensemble Bt),
    (exists p:Bt, inf A p) ->
    (exists! p:Bt, inf A p).
intros A h1.
destruct h1 as [p h1].
exists p.
red. split; auto.
intros q h2.
apply inf_unq with A; auto.
Qed.



Lemma ex_sup_unq : 
  forall (A:Ensemble Bt),
    (exists p:Bt, sup A p) ->
    (exists! p:Bt, sup A p).
intros A h1.
destruct h1 as [p h1].
exists p.
red. split; auto.
intros q h2.
apply sup_unq with A; auto.
Qed.



(*0 <= x*)
Lemma zero_min : forall (x : Bt), le 0 x.
intros x.  assert (h1: inf (Couple x (-x)) 0).
assert (h2: inf (Couple x (-x)) (x * - x)).
apply lat_prod.  assert (h3: x * - x = 0).  apply (comp_prod B).
rewrite <- h3. assumption.  apply h1. apply (Couple_l Bt).
Qed.
Hint Resolve zero_min.

Lemma le_x_0 : forall (x:Bt), le x 0 -> x = 0.
intros x h1.
pose proof (zero_min x) as h2.
apply anti_sym_le; assumption.
Qed.

(* x <= 1 *)
Lemma one_max : forall (x : Bt), le x 1.
intros x.  assert (h1: sup (Couple x (-x)) 1).
assert (h2: sup (Couple x (-x)) (x + (- x))).
apply lat_sum.  assert (h3: x + (-x) = 1).  apply (comp_sum B).
rewrite <- h3. assumption. apply h1.  apply (Couple_l Bt).
Qed.
Hint Resolve one_max.

Lemma le_1_x : forall (x:Bt), le 1 x -> x = 1.
intros x h1.
pose proof (one_max x) as h2.
apply anti_sym_le; assumption.
Qed.

(* x + 0 = x *)
Lemma zero_sum : forall (x : Bt), x + 0 = x.
intros x.  assert (h1: ub (Couple x 0) x).
unfold ub. intros s h2.  assert (h3: s = x \/ s = 0). apply Couple_inv. 
assumption.  destruct h3 as [h3l | h3r].  rewrite h3l. apply idem_sum.
rewrite h3r.  apply zero_min.  assert (h4 : sup (Couple x 0) (x + 0)).
apply lat_sum.  assert (h5: le (x + 0) x). apply h4. assumption. 
assert (h6: le x (x + 0)). apply lat_sum. apply (Couple_l Bt). 
apply anti_sym_le. assumption.  assumption.
Qed.
Hint Resolve zero_sum.

(*  x * 0 = 0 *)
Lemma zero_prod : forall (x : Bt), x * 0 = 0.
intros x.  assert (h1: lb (Couple x 0) 0).
unfold lb.  intros s h2. assert (h3: s = x \/ s = 0). apply Couple_inv.
assumption. destruct h3 as [h3l | h3r].  rewrite h3l. apply zero_min.
rewrite h3r.  unfold le. apply zero_sum.  
assert (h4: inf (Couple x 0) (x * 0)). apply lat_prod.
assert (h5: le 0 (x * 0)).  apply h4. assumption.
assert (h6: le (x * 0) 0). apply lat_prod. apply (Couple_r Bt).
apply anti_sym_le.  assumption. assumption.
Qed.
Hint Resolve zero_prod.

(* x + 1 = 1 *)
Lemma one_sum : forall (x : Bt), x + 1 = 1.
intros x.  assert (h1: ub (Couple x 1) 1).
unfold ub. intros s h2.  assert (h3 : s = x \/ s = 1). apply Couple_inv.
assumption.  destruct h3 as [h3l | h3r].  rewrite h3l.  apply one_max.
rewrite h3r.  unfold le.  apply idem_sum.  
assert (h4: sup (Couple x 1) (x + 1)).  apply lat_sum.
assert (h5: le (x + 1) 1). apply h4. assumption.
assert (h6: le 1 (x+1)).  apply h4. apply (Couple_r Bt).
apply anti_sym_le. assumption. assumption.
Qed.
Hint Resolve one_sum.

(* x * 1 = x *)
Lemma one_prod : forall x : Bt, x * 1 = x.
intros x.  assert (h1: lb (Couple x 1) x).
unfold lb. intros s h2.  assert (h3 : s = x \/ s = 1).  apply Couple_inv.
assumption.  destruct h3 as [h3l | h3r].  rewrite h3l.  apply idem_sum.
rewrite h3r.  apply one_max.  assert (h4: inf (Couple x 1) (x * 1)).
apply lat_prod.  assert (h5: le x (x * 1)). apply h4. assumption.
assert (h6: le (x * 1) x).  apply lat_prod. apply (Couple_l Bt).
apply anti_sym_le.  assumption.  assumption.
Qed.
Hint Resolve one_prod.

Lemma inf_empty : inf (Empty_set Bt) 1.
red. split.
red.
intros.
contradiction.
auto.
Qed.

Lemma sup_empty : sup (Empty_set Bt) 0.
red. split.
red. intros. 
contradiction.
auto.
Qed.

Lemma inf_subtract_one : 
  forall (A:Ensemble Bt) (b:Bt),
    inf A b <-> inf (Subtract A 1) b.
intros A b.
split.
intro h1.  red in h1.
destruct h1 as [h1 h2]. 
red in h1. 
red. split.
red.
intros s h3.
destruct h3 as [h3 h4].
apply h1; auto.
intros b' h3.
assert (h4:lb A b'). 
  red. intros s h4.
  destruct (eq_dec s 1) as [h5 | h6].
  subst.
  apply one_max.
  red in h3.
  assert (h7:Ensembles.In (Subtract A 1) s).
    constructor; auto. intro h7. destruct h7. contradict h6.
    reflexivity.
  apply h3; auto.
apply h2; auto.
intro h1. red in h1.
destruct h1 as [h1 h2]. red in h1.
red. split.
red.
intros s h3.
destruct (eq_dec s 1) as [h4 | h5].
subst. apply one_max.
assert (h6:Ensembles.In (Subtract A 1) s).
  constructor; auto. intro h6. destruct h6. contradict h5.
  reflexivity.
apply h1; auto.
intros b' h3.
red in h3.
assert (h4:lb (Subtract A 1) b').
  red. intros s h4. destruct h4 as [h4 h5].
  apply h3; auto.
apply h2; auto.
Qed.


Lemma sup_subtract_zero : 
  forall (A:Ensemble Bt) (b:Bt),
    sup A b <-> sup (Subtract A 0) b.
intros A b.
split.
intro h1.  red in h1.
destruct h1 as [h1 h2]. 
red in h1. 
red. split.
red.
intros s h3.
destruct h3 as [h3 h4].
apply h1; auto.
intros b' h3.
assert (h4:ub A b'). 
  red. intros s h4.
  destruct (eq_dec s 0) as [h5 | h6].
  subst.
  apply zero_min.
  red in h3.
  assert (h7:Ensembles.In (Subtract A 0) s).
    constructor; auto. intro h7. destruct h7. contradict h6.
    reflexivity.
  apply h3; auto.
apply h2; auto.
intro h1. red in h1.
destruct h1 as [h1 h2]. red in h1.
red. split.
red.
intros s h3.
destruct (eq_dec s 0) as [h4 | h5].
subst. apply zero_min.
assert (h6:Ensembles.In (Subtract A 0) s).
  constructor; auto. intro h6. destruct h6. contradict h5.
  reflexivity.
apply h1; auto.
intros b' h3.
red in h3.
assert (h4:ub (Subtract A 0) b').
  red. intros s h4. destruct h4 as [h4 h5].
  apply h3; auto.
apply h2; auto.
Qed.


Lemma lb_subtract_one : 
  forall (A:Ensemble Bt) (b:Bt),
    lb A b <-> lb (Subtract A 1) b.
intros A b.
split.
intro h1.
red in h1. red.
intros s h2. destruct h2 as [h2 h3].
apply h1; auto.
intro h1.
red in h1. red.
intros s h2.
destruct (eq_dec s 1) as [h3 | h4].
subst.
apply one_max.
apply h1.
constructor; auto.
intro h5. destruct h5. contradict h4.
reflexivity.
Qed.


Lemma ub_subtract_zero : 
  forall (A:Ensemble Bt) (b:Bt),
    ub A b <-> ub (Subtract A 0) b.
intros A b.
split.
intro h1.
red in h1. red.
intros s h2. destruct h2 as [h2 h3].
apply h1; auto.
intro h1.
red in h1. red.
intros s h2.
destruct (eq_dec s 0) as [h3 | h4].
subst.
apply zero_min.
apply h1.
constructor; auto.
intro h5. destruct h5. contradict h4.
reflexivity.
Qed.



(*The goal of this section is to develop a tactic that reduces equalities that contain
  any 0 or 1 in a disjunction/conjunction or any adjacent combination of x/-x, so that
  0s and 1s and inverses get eliminated on both sides of the equation *)
Section const_eq.
Inductive bin2 : Set := pnode2 : bin2 -> bin2 -> bin2 |
                        tnode2 : bin2 -> bin2 -> bin2 |
                        leaf2 : nat-> bin2 |
                        zero : bin2 |
                        one  : bin2 |
                        popps : nat -> bool -> bin2 |
                        topps : nat -> bool -> bin2.

Fixpoint bin2_Bt (l: list Bt)(def:Bt)(t:bin2){struct t}: Bt :=
  match t with
  | pnode2 t1 t2 => (bin2_Bt l def t1) + (bin2_Bt l def t2)
  | tnode2 t1 t2 => (bin2_Bt l def t1) * (bin2_Bt l def t2)
  | leaf2 n => nth n l def
  | zero => 0
  | one  => 1
  | popps n b => let x := (nth n l def) in 
                    if b then (x + -x) else (-x + x)
  | topps n b => let x := (nth n l def) in 
                    if b then (x * -x) else (-x * x)
  end.

Fixpoint elim_bin2_aux (t:bin2) : bin2 :=
  match t with 
    | pnode2 zero t' => elim_bin2_aux t'
    | pnode2 t' zero => elim_bin2_aux t'
    | pnode2 one t' => one
    | pnode2 t' one => one
    | tnode2 zero t' => zero
    | tnode2 t' zero => zero
    | tnode2 one t' => elim_bin2_aux t'
    | tnode2 t' one => elim_bin2_aux t'
    | pnode2 t1 t2 => pnode2 (elim_bin2_aux t1) (elim_bin2_aux t2)
    | tnode2 t1 t2 => tnode2 (elim_bin2_aux t1) (elim_bin2_aux t2)
    | topps _ _ => zero
    | popps _ _ => one
    | _ => t
  end.

Fixpoint elim_bin2 (t: bin2) : bin2 :=
  match t with
  | pnode2 t1 t2 => elim_bin2_aux (pnode2 (elim_bin2 t1) (elim_bin2 t2))
  | tnode2 t1 t2 => elim_bin2_aux (tnode2 (elim_bin2 t1) (elim_bin2 t2))
  | topps _ _ => zero
  | popps _ _ => one
  | _ => t
  end.

Theorem bin_Bt2_ref_p : forall (l:list Bt)(def:Bt)(t1 t2:bin2),
  bin2_Bt l def (pnode2 t1 t2) = (bin2_Bt l def t1) + (bin2_Bt l def t2).
intros l def t1 t2.  unfold bin2_Bt. fold bin2_Bt. reflexivity.
Qed.

Theorem bin_Bt2_ref_t : forall (l:list Bt)(def:Bt)(t1 t2:bin2),
  bin2_Bt l def (tnode2 t1 t2) = (bin2_Bt l def t1) * (bin2_Bt l def t2).
intros l def t1 t2. unfold bin2_Bt. fold bin2_Bt. reflexivity.
Qed.

Theorem elim_bin2_aux_p : forall (l:list Bt) (def : Bt) (t1 t2:bin2), 
  bin2_Bt l def (elim_bin2_aux (pnode2 t1 t2)) = 
  bin2_Bt l def (pnode2 (elim_bin2_aux t1) (elim_bin2_aux t2)).
intros l def t1 t2.  unfold elim_bin2_aux. fold elim_bin2_aux.
destruct t1. destruct t2. reflexivity. 
(* next subgoal *) reflexivity. 
(* next subgoal *) reflexivity.
(* next subgoal *)
unfold elim_bin2_aux at 3.  rewrite bin_Bt2_ref_p. unfold bin2_Bt at 3.
rewrite zero_sum. reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 2.  rewrite bin_Bt2_ref_p. unfold bin2_Bt at 3.
unfold bin2_Bt at 1. rewrite one_sum. reflexivity.
(*next subgoal*)
reflexivity.
(*next subgoal*)
reflexivity.
(*next subgoal*)
destruct t2.  reflexivity.
(*next subgoal*)
reflexivity.
(*next subgoal*)
reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 3.  rewrite bin_Bt2_ref_p. unfold bin2_Bt at 3.
rewrite zero_sum. reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 2. unfold bin2_Bt. fold bin2_Bt. 
rewrite one_sum. reflexivity.
(*next subgoal*)
reflexivity.
(*next subgoal*)
reflexivity.
(*next subgoal*)
destruct t2. reflexivity.
(*next subgoal *)
reflexivity.
(*next subgoal *)
reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 3. unfold bin2_Bt. fold bin2_Bt.
rewrite zero_sum. reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 2. unfold bin2_Bt. fold bin2_Bt.
rewrite one_sum. reflexivity.
(*next subgoal*)
reflexivity.
(*next subgoal*)
reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 2. unfold bin2_Bt. fold bin2_Bt.
rewrite (comm_sum B). rewrite zero_sum. reflexivity.
(*next subgoal*)
destruct t2;
unfold elim_bin2_aux at 1; unfold bin2_Bt; fold bin2_Bt;
rewrite (comm_sum B); rewrite one_sum; reflexivity.
(*next subgoal*)
destruct t2; try reflexivity.
unfold elim_bin2_aux at 3. unfold bin2_Bt. fold bin2_Bt. 
rewrite zero_sum. reflexivity.
unfold elim_bin2_aux at 2.  unfold bin2_Bt. fold bin2_Bt.
rewrite one_sum. reflexivity.
destruct t2; try reflexivity.
unfold elim_bin2_aux. unfold bin2_Bt. fold bin2_Bt. rewrite zero_sum.
reflexivity.
unfold elim_bin2_aux at 2. unfold bin2_Bt. fold bin2_Bt.
rewrite one_sum. reflexivity.
Qed.

Theorem elim_bin2_aux_t : forall (l:list Bt) (def : Bt) (t1 t2:bin2), 
  bin2_Bt l def (elim_bin2_aux (tnode2 t1 t2)) = 
  bin2_Bt l def (tnode2 (elim_bin2_aux t1) (elim_bin2_aux t2)).
intros l def t1 t2.  unfold elim_bin2_aux. fold elim_bin2_aux.
destruct t1. destruct t2; try reflexivity.
unfold elim_bin2_aux at 2. unfold bin2_Bt. fold bin2_Bt.
rewrite zero_prod. reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 3. unfold bin2_Bt. fold bin2_Bt.
rewrite one_prod. reflexivity.
(*next subgoal*)
destruct t2; try reflexivity.
unfold elim_bin2_aux at 2. unfold bin2_Bt. fold bin2_Bt.
rewrite zero_prod. reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 3. unfold bin2_Bt. fold bin2_Bt.
rewrite one_prod. reflexivity.
(*next subgoal*)
destruct t2; try reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 2. unfold bin2_Bt. fold bin2_Bt.
rewrite zero_prod. reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 3. unfold bin2_Bt. fold bin2_Bt.
rewrite one_prod. reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 1. unfold bin2_Bt. fold bin2_Bt.
rewrite (comm_prod B). rewrite zero_prod. reflexivity.
unfold elim_bin2_aux at 5. unfold bin2_Bt. fold bin2_Bt.
rewrite (comm_prod B). rewrite (one_prod). 
destruct t2; try reflexivity.
(*next subgoal*)
destruct t2; try reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 2.  unfold bin2_Bt. fold bin2_Bt.
rewrite zero_prod. reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 3.  unfold bin2_Bt. fold bin2_Bt.
rewrite one_prod. reflexivity.
(*next subgoal*)
destruct t2; try reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 2. unfold bin2_Bt. fold bin2_Bt.
rewrite zero_prod. reflexivity.
(*next subgoal*)
unfold elim_bin2_aux at 3. unfold bin2_Bt. fold bin2_Bt.
rewrite one_prod. reflexivity.
Qed.


Theorem elim_bin2_aux_valid : forall (l:list Bt)(def:Bt)(t:bin2),
   bin2_Bt l def t = bin2_Bt l def (elim_bin2_aux t).
intros l def t.  induction t; unfold bin2_Bt; fold bin2_Bt.
rewrite IHt1.  rewrite IHt2. rewrite <- bin_Bt2_ref_p. 
rewrite elim_bin2_aux_p. reflexivity.
(*next subgoal*)
rewrite IHt1.  rewrite IHt2. rewrite <- bin_Bt2_ref_t.
rewrite elim_bin2_aux_t. reflexivity.
unfold elim_bin2_aux. unfold bin2_Bt. reflexivity.
(*next subgoal*)
unfold elim_bin2_aux. unfold bin2_Bt. reflexivity.
(*next subgoal*)
unfold elim_bin2_aux. unfold bin2_Bt. reflexivity.
(*next subgoal*)
case b.  unfold elim_bin2_aux. unfold bin2_Bt. apply (comp_sum B). 
(*next subgoal*)
unfold elim_bin2_aux. unfold bin2_Bt. rewrite (comm_sum B).
apply (comp_sum B).
(*next subgoal*)
case b.  unfold elim_bin2_aux. unfold bin2_Bt. apply (comp_prod B).
(*next subgoal*)
unfold elim_bin2_aux. unfold bin2_Bt. rewrite (comm_prod B). 
apply (comp_prod B). 
Qed.

Theorem  elim_bin2_valid :
  forall (l:list Bt)(def:Bt)(t:bin2),
  bin2_Bt l def t = bin2_Bt l def (elim_bin2 t).
intros l def t.  
induction t; unfold elim_bin2; fold elim_bin2; 
try rewrite <- elim_bin2_aux_valid; unfold bin2_Bt; fold bin2_Bt;
try (rewrite IHt1; rewrite IHt2; reflexivity) || reflexivity.
case b.  apply (comp_sum B).  
(*next subgoal*) 
rewrite (comm_sum B). apply (comp_sum B).
case b. apply (comp_prod B). 
(*next subgoal*)
rewrite (comm_prod B). apply (comp_prod B).
Qed.

Theorem elim_bin2_valid_2 :
  forall (t t':bin2)(l:list Bt)(def:Bt),
    bin2_Bt l def (elim_bin2 t) = bin2_Bt l def (elim_bin2 t') ->
    bin2_Bt l def t = bin2_Bt l def t'.
intros t t' l def.  
rewrite <- elim_bin2_valid.  rewrite <- elim_bin2_valid.
tauto.
Qed.

End const_eq.


(*Tactics to build the const_eq tactic*)
Ltac term_list2 l v :=
  match v with
  | (?X1 + (-?X1)) => constr:(cons X1 l)
  | ((-?X1) + ?X1) => constr:(cons X1 l)
  | (?X1 * (-?X1)) => constr:(cons X1 l)
  | ((-?X1) * ?X1) => constr:(cons X1 l)
  | (?X1 + ?X2) => let l1 := term_list2 l X2 in term_list2 l1 X1
  | (?X1 * ?X2) => let l1 := term_list2 l X2 in term_list2 l1 X1
  | ?X1 => constr:(cons X1 l)
  end.

Ltac model_aux2 l v :=
  match v with
  | (?X1 + (-?X1)) => let  n := compute_rank l 0%nat X1 in 
                    constr:(popps n true)
  | ((-?X1) + ?X1) => let  n := compute_rank l 0%nat X1 in
                    constr:(popps n false)
  | (?X1 * (-?X1)) => let  n := compute_rank l 0%nat X1 in
                    constr:(topps n true)
  | ((-?X1) * ?X1) => let  n := compute_rank l 0%nat X1 in 
                    constr:(topps n false)
  | (?X1 + ?X2) =>
    let r1 := model_aux2 l X1 with r2 := model_aux2 l X2 in
      constr:(pnode2 r1 r2)
  | (?X1 * ?X2) =>
    let r1 := model_aux2 l X1 with r2 := model_aux2 l X2 in
      constr:(tnode2 r1 r2)
  | 0 => constr:(zero)
  | 1 => constr:(one)
  | ?X1 => let n := compute_rank l 0%nat X1 in constr:(leaf2 n)
  | _ => constr:(leaf2 0%nat)
  end.

Ltac const_eq :=
  match goal with
  | [ |- (?X1 = ?X2) ] =>
  
  let l1 := term_list2 (nil (A := Bt)) X1 in
  let l2 := term_list2 l1 X2 in
  
  let term1 := model_aux2 l2 X1
  with term2 := model_aux2 l2 X2 in
  (change (bin2_Bt l2 X1 term1 = bin2_Bt l2 X1 term2);
   apply elim_bin2_valid_2;
   auto)
  end.

(*
(*All tests check *)
Variables a b c d e f g h : Bt.

Goal ((a + 0) = a).
const_eq.
Qed.
Goal ((a + (b*0)) = a).
const_eq.
Qed.
Goal ((a+b+(c*1)*(d+0)) = a + b +(c*d)).
const_eq.
Qed.
Goal (0*(a+b) + c*1) = c.
const_eq.
Qed.
Goal (a+(b*-b)) = a.
const_eq.
Qed.
Goal (a+(c+d)*(-(c+d))) + (b*(c+d*((e+f) + (-(e+f))))) = a + b*(c+d).
const_eq.
Qed.
*)

(* More Theorems *)
(* complement unique *)

Lemma comp_unq : forall (x y : Bt), -x = -y -> x = y.
intros x y h1.  
assert (h2: x = x*y).
  assert (h3: x*y = x*y + x * -x). const_eq.
  assert (h4: x*y + x*-x = x * (y + -x)).  
    rewrite <- (dist_sum B). reflexivity.
  rewrite h1 in h4.
  assert (h5: x * (y + -y) = x).  const_eq.
  congruence.
assert (h6: y = y *x).
  assert (h7 : y*x = y*x + y*-y). const_eq.
  assert (h8: y*x + y*-y = y*(x + -y)).
    rewrite <- (dist_sum B). reflexivity.
  rewrite <- h1 in h8.
  assert (h9: y*(x + -x) = y). const_eq.
  congruence.
rewrite h2. rewrite h6 at 2.  comm_eq.
Qed.
Hint Resolve comp_unq.

(* x = --x *)
Lemma doub_neg : forall (x : Bt), x = --x.
intros x.  assert (h1 : le (---x) (-x)). assert (h2 : -x + --x = 1).
apply (comp_sum B). assert (h3: ---x * 1 = ---x). apply one_prod.
assert (h4: ---x * 1 = ---x * (-x + --x)). rewrite <- h2. reflexivity. 
assert (h5: --x * ---x = 0). apply (comp_prod B). assert (h6: ---x * --x = 0).
assert (h7: ---x * --x = --x * ---x). apply (comm_prod B).  rewrite <- h5. 
assumption. assert (h8: ---x * (-x + --x) = ---x * -x + ---x*--x).
apply (dist_sum B). assert (h9: ---x * --x = 0). rewrite <- h5.
apply (comm_prod B). unfold le. apply eq_ord.  
assert (h10: ---x * (-x + --x) = ---x * -x + 0).  rewrite <- h9.
rewrite h8. reflexivity. assert (h11 : ---x * -x + 0 = ---x * -x).
apply zero_sum. rewrite <- h11. rewrite <- h6. rewrite <- h8.  
rewrite <- h4. assumption.
assert (h12 : le (-x) (---x)).  unfold le.  apply eq_ord.
assert (h13 : -x * 1 = -x).  apply one_prod.
assert (h14: -x * 1 = -x * (--x + ---x)).  assert (h15: --x + ---x = 1).
apply (comp_sum B). rewrite <- h15. reflexivity. 
assert (h16: -x * (--x + ---x) = -x*--x + -x*---x).  apply (dist_sum B).
assert (h17: -x * --x = 0).  apply (comp_prod B).
assert (h18: -x * --x + -x * ---x = 0 + -x * ---x). rewrite <- h17.
reflexivity. assert (h19: 0 + -x*---x = -x*---x).  
assert (h20: -x*---x + 0 = -x*---x). apply zero_sum.
assert (h21: -x * ---x + 0 = 0 + -x * ---x). apply (comm_sum B).
rewrite <- h21. assumption.  rewrite <- h19. rewrite <- h17.
rewrite <- h16. rewrite <- h14. assumption.
assert (h22 : -x = ---x).  apply anti_sym_le. 
assumption.  assumption.  apply comp_unq. assumption.
Qed.
Hint Resolve doub_neg.

(* zero is characterized by the property x + y = x for all y*)
Lemma zero_char : forall (y : Bt), 
  (forall (x : Bt), x + y= x) -> y = 0.
intros y h1.  assert (h2 : 0 + y = 0). apply h1.
assert (h3: y + 0 = y). apply zero_sum.  assert (h4: y + 0 = 0 + y).
apply (comm_sum B).  rewrite <- h3. rewrite h4. assumption.
Qed.
Hint Resolve zero_char.

(* one is characterized by the property x * y = x for all y*)
Lemma one_char : forall (y : Bt), 
  (forall (x : Bt), x * y = x) -> y = 1.
intros y h1.  assert (h2 : 1 * y = 1). apply h1.
assert (h3: y * 1 = y). apply one_prod.  assert (h4: y * 1 = 1 * y).
apply (comm_prod B).  rewrite <- h3.  rewrite h4. assumption.
Qed.
Hint Resolve one_char.

(* complement is characterized by  x * y = 0, x + y =1 *)
Lemma comp_char : forall (x y : Bt), 
  x * y = 0 -> x + y = 1 -> y = -x.
intros x y h1 h2.
assert (h3: y = y * (x + -x)).  const_eq.
assert (h4: y * (x + -x) = (y * x) + (y * -x)). apply (dist_sum B).
assert (h5: (y * x) + (y * -x) = (x * y) + (y * -x)).
comm_eq. rewrite h1 in h5.
assert (h6 : 0 + y * -x = (-x * y) + 0).  comm_eq.
assert (h7: (-x * y) + 0 = -x * y + -x * x).  const_eq.
assert (h8: -x * y + -x * x = -x * (y + x)). 
rewrite (dist_sum B). reflexivity.
assert (h9: -x * (y + x) = -x * (x + y)).  comm_eq.
rewrite h2 in h9.
assert (h10 : -x * 1 = -x).  const_eq.
congruence.
Qed.
Hint Resolve comp_char.

Lemma comp_eq : forall (x y : Bt), -x = -y -> x = y.
intros x y h. assert (h1: x = --x).  apply doub_neg.
assert (h2: --x = --y). rewrite h. reflexivity.  assert (h3 : y = --y).
apply doub_neg. rewrite h3. rewrite <- h2. assumption.
Qed.
Hint Resolve comp_eq.

(* -0 = 1 *)
Lemma zero_comp: -0 = Bone (Bc B).
assert (h1: 0 * 1 = Bzero (Bc B)). apply one_prod.
assert (h2: 0 + 1 = Bone (Bc B)). apply one_sum.
assert (h3: 1 = -(Bzero (Bc B))). apply comp_char.  assumption.  assumption.
congruence.
Qed.
Hint Resolve zero_comp.

(* -1 = 0 *)
Lemma one_comp: -(1) = Bzero (Bc B).
assert (h1: 1 * 0 = Bzero (Bc B)). apply zero_prod. assert (h2: 1 + 0 = Bone (Bc B)).
apply zero_sum. assert (h3: 0 = -(Bone (Bc B))). apply comp_char.
assumption. assumption. rewrite h3. reflexivity.
Qed.
Hint Resolve one_comp.

Lemma de_mor_sum : forall (x y : Bt), -(x + y) = -x * -y.
intros x y.  assert (h0: (x + y ) * (-x * -y) = 0). 
assert (h1: (x + y) * (-x * -y) = x * (-x * -y) + y * (-x * -y)).
apply dist_sum_r.  rewrite h1.  assert (h2: x * (-x * -y) = (x * -x) * -y).
apply (assoc_prod B). rewrite h2.  assert (h3: x * -x = 0). apply (comp_prod B).
rewrite h3.  assert (h4: 0 * -y = 0). assert (h5: -y * 0 = 0).  apply zero_prod.
rewrite <- h5 at 2.  apply (comm_prod B). rewrite h4.
assert (h6: -x * -y = -y * -x).  apply (comm_prod B). rewrite h6.
assert (h7: y * (-y * -x) = (y * -y) * -x). apply (assoc_prod B).
rewrite h7. assert (h8: y * -y = 0).  apply (comp_prod B). rewrite h8.
assert (h9: 0 * -x = 0). assert (h10: (-x * 0) = 0). apply zero_prod.
rewrite <- h10 at 2. apply (comm_prod B). rewrite h9. apply idem_sum.
assert (h11: x + y + -x * -y = 1).  
  assert (h12: x + y + -x * -y = x * (y + -y) + y + -x * -y).
    assert (h13: y + -y = 1).  auto.
    assert (h14: x * 1 = x).  auto.
    congruence.
  assert (h15: x * (y + -y) + y + -x * -y = x * y + x * -y + y + -x * -y).
    assert (h16: x * (y + -y) = x * y + x * -y). auto.
    congruence.
  assert (h16: x * y + x * -y + y + -x * -y = y + x * -y + -x * -y).
    assert (h17: x * y + x * -y + y = x*y + y + x * -y).  comm_eq.
    assert (h18: (x + 1) * y = y).
      assert (h19: x + 1 = 1). auto.
      assert (h20: 1 * y = y * 1). auto.
      assert (h21: y * 1 = y).  auto.
    congruence.
    assert (h22: (x + 1) * y = x * y + y).
      assert (h23: (x + 1) * y = x * y + 1 * y).  auto.
      assert (h24: 1 * y = y * 1).  auto.
      assert (h25: y * 1 = y).  auto.
    congruence.
  congruence.
  assert (h26: y + x * -y + -x * -y = y + -y).
    assert (h27: y + (x * -y + -x * -y) = y + x * -y + -x * -y). auto.
    assert (h28: (x + -x) * -y = x * -y + -x * -y).  auto.
    assert (h29: x + -x = 1).  auto.
    assert (h30: 1 * -y = -y * 1).  auto.
    assert (h31: -y * 1 = -y).  auto.
  congruence. 
  assert (y + -y = 1).  auto.
congruence.
assert (h32: -x * -y = - (x + y)).  auto.
congruence.
Qed.   
Hint Resolve de_mor_sum.

Lemma de_mor_prod : forall (x y : Bt), -(x*y) = -x + -y.
intros x y.
assert (h1 : (x * y) + (-x + -y) = 1).
  assert (h2 : (x * y) + (-x + -y) = (x + (-x + -y))*(y + (-x + -y))).
    apply dist_prod_r.
  assert (h3: (x + (-x + -y)) * (y + (-x + -y)) = 
              ((x + -x) + -y) * ((y + -y) + -x)).
    comm_eq.
  assert (h4 : ((x + -x) + -y) * ((y + -y) + -x) = 1).
    const_eq.
  congruence.
assert (h5 : (x * y) * (-x + -y) = 0).
  assert (h6: (x * y) * (-x + -y) = (x + (y * -y)) * y * (-x + -y)).
    const_eq.
  assert (h7: (x + (y * -y)) * y * (-x + -y) = (x + y) * (x + -y) * y * (-x + -y)).
    rewrite dist_prod with (p := x) (n := y) (m := -y). reflexivity.
  assert (h8: (x + y) * (x+ -y) * y * (-x + -y) = (x + y) * y * (x + -y) * (-x + -y)).
    comm_eq.
  assert (h9: (x + y) * y *(x + -y) *(-x + -y) = y * (x+-y) * (-x + -y)).
    assert (h10 : (x + y) * y*(x + -y) * (-x + -y) = 
                  y * (y + x)  * (x + -y) * (-x + -y)).
      comm_eq.
    assert (h11: y * (y+x) * (x + -y) * (-x + -y) = y*(x + -y) * (-x + -y)).
      rewrite (abs_prod B) with (n := y) (m := x). reflexivity.
    congruence.
  assert (h12: y * (x + -y) * (-x + -y) = y * ((x * -x ) + -y)).
    assert (h13: y * (x + -y) * (-x + -y) = y * ((x + -y) * (-x + -y))).
      assoc_eq.
    rewrite <- dist_prod_r with (n := x) (m := -x) (p := -y) in h13.
    assumption.
  assert (h13 : y * ((x * -x) + -y) = 0).
    assert (h14 : (x * -x) + -y = -y).
      const_eq.
    rewrite h14.
      const_eq.
  congruence.
symmetry.
apply comp_char; auto.
Qed.
Hint Resolve de_mor_prod.    

Lemma times_eq_plus : forall (x y: Bt), x*y = -((-x) + (-y)).
intros x y.
rewrite de_mor_sum with (x := -x) (y := -y).
rewrite <- doub_neg. rewrite <- doub_neg.
reflexivity.
Qed.

Lemma plus_eq_times : forall (x y:Bt), x+y = -((-x)*(-y)).
intros x y.
rewrite de_mor_prod with (x := -x) (y := -y).
rewrite <- doub_neg. rewrite <- doub_neg.
reflexivity.
Qed.

Lemma mono_sum : forall (x x' y y' : Bt), 
  le x x' -> le y y' -> le (x+y) (x' + y').
intros x x' y y' h1 h2.
assert (h3: (x + y) + (x' + y') = x' + y').
assert (h4: (x + y) + (x' + y') = (x+x') + (y + y')).  comm_eq.
assert (h8: x + x' = x').  auto.
assert (h9: y + y' = y').  auto.
congruence. auto.
Qed.

Lemma mono_prod : forall (x x' y y': Bt), 
  le x x' -> le y y' -> le (x * y) (x' * y').
intros x x' y y' h1 h2.
assert (h3: (x * y) * (x' * y') = x * y).
assert (h4: (x * y) * (x' * y') = (x * x') *(y * y')).  comm_eq.
assert (h8: x * x' = x).  
  assert (h8a: x + x' = x'). auto.  apply eq_ord. assumption.
  (* turn this functionality into a separate lemma, to avoid repetition *)
assert (h9: y * y' = y).
  assert (h9a: y + y' = y').  auto.  apply eq_ord. assumption.
congruence.
apply eq_ord.
assumption.
Qed.


Lemma le_iff' : forall (a b:Bt), le a (-b) <-> a * b = 0.
intros a b. split.
intro h1.
pose proof refl_le.
pose proof (refl_le b) as h2.
pose proof (mono_prod _ _ _ _ h1 h2) as h3.
rewrite (comm_prod _ (-b) b)  in h3.
rewrite comp_prod in h3.
apply le_x_0. assumption.
intro h1.
pose proof (zero_sum (a*-b)) as h2.
rewrite <- h1 in h2.
rewrite <- dist_sum in h2.
rewrite comm_sum in h2.
rewrite comp_sum in h2.
rewrite one_prod in h2.
red. rewrite eq_ord.
rewrite h2 at 2.
reflexivity.
Qed.

Lemma le_iff : forall (a b:Bt), le a b <-> a * -b = 0.
intros a b.
rewrite (doub_neg b) at 1.
rewrite le_iff'. tauto.
Qed.

Lemma mono_comp : forall (x x' : bt B), le x x' -> le (-x') (-x).
intros x x' h1.
assert (h2: -(x * x') = -x + -x').  apply de_mor_prod.
assert (h3: x * x' = x).
  assert (h4: x + x' = x').  auto.
  apply eq_ord.  assumption.
assert (h5: -x + -x' = -x).
  rewrite <- h2.  congruence.
unfold le.  
assert (h6: -x + -x' = -x' + -x).  auto.
congruence.
Qed.


Definition comp_set {B:Bool_Alg} (A:Ensemble (bt B)) :=
  Im A (Bcomp (Bc B)).


Lemma comp_set_comp_set : forall (A:Ensemble Bt), comp_set (comp_set A) = A.
intro A.
apply Extensionality_Ensembles. red. split.
red. intros x h1.
destruct h1 as [x h1]. subst.
destruct h1 as [x h1]. subst.
rewrite <- doub_neg. assumption.
red.
intros x h1. 
apply Im_intro with (-x).
apply Im_intro with x; auto.
apply doub_neg.
Qed.


Lemma lb_ub_compat_iff : 
  forall (A:Ensemble Bt) (p:Bt),
    lb A p <-> ub (comp_set A) (-p).
intros A p.
split.
intros h1.
red in h1. red.
intros s h2.
destruct h2 as [s h2]. subst.
specialize (h1 _ h2).
apply mono_comp; auto.
intro h1.
red in h1. red.
intros s h2.
specialize (h1 (-s)).
assert (h3:Ensembles.In (comp_set A) (-s)).
  apply Im_intro with s; auto.
specialize (h1 h3).
pose proof (mono_comp _ _ h1) as h4.
rewrite <- doub_neg in h4. rewrite <- doub_neg in h4.
assumption.
Qed.

Lemma ub_lb_compat_iff : 
  forall (A:Ensemble Bt) (p:Bt),
    ub A p <-> lb (comp_set A) (-p).
intros A p.
split.
intros h1.
red in h1. red.
intros s h2.
destruct h2 as [s h2]. subst.
specialize (h1 _ h2).
apply mono_comp; auto.
intro h1.
red in h1. red.
intros s h2.
specialize (h1 (-s)).
assert (h3:Ensembles.In (comp_set A) (-s)).
  apply Im_intro with s; auto.
specialize (h1 h3).
pose proof (mono_comp _ _ h1) as h4.
rewrite <- doub_neg in h4. rewrite <- doub_neg in h4.
assumption.
Qed.



Lemma sup_inf_compat_iff : 
  forall (A:Ensemble Bt) (p:Bt),
    sup A p <-> inf (comp_set A) (-p).
intros A p. split.
intro h1.
red in h1. red.
destruct h1 as [h1l h1r]. split.
red.
intros s h2.
destruct h2 as [s h2]. subst.
apply h1l in h2.
apply mono_comp; auto.
intros b' h2.
rewrite lb_ub_compat_iff in h2.
rewrite comp_set_comp_set in h2.
pose proof (h1r _ h2) as h3.
rewrite (doub_neg b').
apply mono_comp; auto.
intro h1. red in h1. red.
destruct h1 as [h1l h1r]. split.
red. intros s h2.
assert (h3:Ensembles.In (comp_set A) (-s)). apply Im_intro with s; auto.
apply h1l in h3.
rewrite doub_neg.  rewrite (doub_neg s). apply mono_comp; auto.
intros b' h2.
rewrite ub_lb_compat_iff in h2.
specialize (h1r _ h2).
rewrite doub_neg. rewrite (doub_neg p). apply mono_comp; auto.
Qed.

Lemma inf_sup_compat_iff : 
  forall (A:Ensemble Bt) (p:Bt),
    inf A p <-> sup (comp_set A) (-p).
intros A p. split.
intro h1.
red in h1. red.
destruct h1 as [h1l h1r]. split.
red.
intros s h2.
destruct h2 as [s h2]. subst.
apply h1l in h2.
apply mono_comp; auto.
intros b' h2.
rewrite ub_lb_compat_iff in h2.
rewrite comp_set_comp_set in h2.
pose proof (h1r _ h2) as h3.
rewrite (doub_neg b').
apply mono_comp; auto.
intro h1. red in h1. red.
destruct h1 as [h1l h1r]. split.
red. intros s h2.
assert (h3:Ensembles.In (comp_set A) (-s)). apply Im_intro with s; auto.
apply h1l in h3.
rewrite doub_neg.  rewrite (doub_neg p). apply mono_comp; auto.
intros b' h2.
rewrite lb_ub_compat_iff in h2.
specialize (h1r _ h2).
rewrite doub_neg. rewrite (doub_neg b'). apply mono_comp; auto.
Qed.


Definition atom (a:Bt) : 
  Prop := a <> 0 /\ forall b:Bt, le b a -> a = b \/ b = 0.

Lemma atom_iff : forall (a:Bt), atom a <-> (forall b:Bt, (le a b \/ a * b = 0) /\ ~ (le a b /\ a*b = 0)).
intro a. split.
(* -> *)
intros h1 b.
split.
red in h1. destruct h1 as [h1l h1r].
pose proof (times_le a b) as h2.
specialize (h1r _ h2).
destruct h1r. left. red. rewrite eq_ord. symmetry. assumption.
right. assumption.
intro h2.
destruct h2 as [h2l h2r].
rewrite <- le_iff' in h2r.
pose proof (mono_prod _ _ _ _ h2l h2r) as h3.
rewrite idem_prod in h3.
rewrite comp_prod in h3.
pose proof (le_x_0  _ h3). subst.
red in h1. destruct h1 as [h1].
contradict h1. apply reflexivity.
(* <- *)
intro h1. red.
split.
intro h2. subst.
specialize (h1 0).
destruct h1 as [h1l h1r].
contradict h1r.
split.
apply zero_min. apply zero_prod.
intros b h2.
specialize (h1 b).
destruct h1 as [h1l h1r].
destruct h1l as [h1a | h1b].
pose proof (anti_sym_le _ _ h1a h2) as h3.
subst. left. reflexivity.
rewrite comm_prod in h1b.
rewrite <- le_iff' in h1b.
pose proof (mono_prod _ _ _ _ h1b h2) as h3.
rewrite idem_prod in h3.
rewrite comm_prod in h3.
rewrite comp_prod in h3.
right.
apply le_x_0. assumption.
Qed.


Lemma atom_impl : forall (a:Bt), atom a -> a <> 0 /\ (forall b:Bt, a * b = a \/ a * b = 0).
intros a h1. red in h1. destruct h1 as [h1l h1r].
split; auto. 
intro b.
pose proof (times_le a b) as h2.
specialize (h1r _ h2).
destruct h1r. left. symmetry. assumption.
right. auto.
Qed.

End Basics.


(*Theorems*)

Arguments flatten_aux_valid_Bt_p [B] l def t t'.
Arguments flatten_aux_valid_Bt_t [B] l def t t'.
Arguments flatten_valid_Bt [B] l def t.
Arguments flatten_valid_Bt_2 [B] t t' l def _.
Arguments bin_Bt_ref_p [B] l def t1 t2.
Arguments bin_Bt_ref_t [B] l def t1 t2.
Arguments insert_eq_p [B] l def t0 t.
Arguments insert_eq_t [B] l def t0 t.
Arguments sort_eq [B] l def t.
Arguments sort_eq_2 [B] l def t1 t2 _.
Arguments dist_sum_r [B] n m p.
Arguments dist_prod_r [B] n m p.
Arguments idem_sum [B] b.
Arguments idem_prod [B] b.
Arguments eq_ord [B] x y.
Arguments refl_le [B] x.
Arguments anti_sym_le [B] x y _ _.
Arguments trans_le [B] x y z _ _.
Arguments lat_sum [B] x y.
Arguments lat_prod [B] x y.
Arguments inf_unq [B] S x y _ _.
Arguments sup_unq [B] S x y _ _.
Arguments zero_min [B] x.
Arguments one_max [B] x.
Arguments zero_sum [B] x.
Arguments zero_prod [B] x.
Arguments one_sum [B] x.
Arguments one_prod [B] x.
Arguments comp_unq [B] x y _.
Arguments doub_neg [B] x.
Arguments zero_char [B] y _.
Arguments one_char [B] y _.
Arguments comp_char [B] x y _ _.
Arguments comp_eq [B] x y _.
Arguments zero_comp [B].
Arguments one_comp [B].
Arguments de_mor_sum [B] x y.
Arguments de_mor_prod [B] x y.
Arguments mono_sum [B] x x' y y' _ _.
Arguments mono_prod [B] x x' y y' _ _.
Arguments mono_comp [B] x x' _.
Arguments le_plus [B] _ _.
Arguments le_x_0 [B] _ _.
Arguments le_1_x [B] _ _.
Arguments times_le [B] _ _.
Arguments atom_iff [B] _.
Arguments in_bs [B] _.
(*Definitions*)
Arguments le [B] x y.
Arguments lt [B] x y.
Arguments lb [B] S b.
Arguments ub [B] S b.
Arguments inf [B] S b.
Arguments sup [B] S b.
Arguments atom [B] _.




Definition sym_diff {B:Bool_Alg} (x y:(bt B)) := x * -y + y* -x.

Notation "x /_\ y" := (sym_diff x y) (at level 50, left associativity).

Section SymDiff.
Variable B:Bool_Alg.
Let Bt := bt B.


Lemma comm_sym_diff : forall (x y:Bt), x /_\ y = y /_\ x.
intros. unfold sym_diff. apply comm_sum.
Qed.

Lemma sym_diff_ref : forall x:Bt, x /_\ x = 0.
intros. unfold sym_diff. rewrite comp_prod. apply zero_sum.
Qed.

Lemma zero_sum_each_zero: forall (x y:Bt), x + y = 0 <-> x = 0 /\ y = 0.
intros x y. split.
intro h1.
pose proof (le_plus x y) as h2.
rewrite h1 in h2.
pose proof (le_x_0 _ h2). subst.
rewrite comm_sum in h1. rewrite zero_sum in h1. subst.
split; reflexivity.
intro h2.
destruct h2; subst.
rewrite zero_sum. reflexivity.
Qed.

Lemma one_prod_each_one: forall (x y:Bt), x * y = 1 <-> x = 1 /\ y = 1.
intros x y. split.
intro h1.
pose proof (times_le x y) as h2.
rewrite h1 in h2.
pose proof (le_1_x _ h2). subst.
rewrite comm_prod in h1. rewrite one_prod in h1. subst.
split; reflexivity.
intro h2.
destruct h2; subst.
rewrite one_prod. reflexivity.
Qed.


Lemma sym_diff_iff : forall (x y:Bt), x /_\ y = 0 <-> x = y.
intros x y. split.
intro h1.
unfold sym_diff in h1.
rewrite zero_sum_each_zero in h1.
destruct h1 as [h1l h1r].
unfold Bt, bt in h1l. rewrite <- le_iff in h1l.
unfold Bt, bt in h1r. rewrite <- le_iff in h1r.
apply anti_sym_le; assumption.
intros; subst.
apply sym_diff_ref.
Qed.
End SymDiff.

(*This section deals with the common task of establishing
  equivalent predictaes dependent on equal Bool_Algs.*)

Section BASubst.

(*Tentative.*)
Lemma ba_subst_pred : 
  forall (A B:Bool_Alg),
    A = B ->
    forall (P:Bool_Alg->Prop),
      P A <-> P B.
intros; subst; tauto.
Qed.


End BASubst.


Section ParametricAnalogues.
Variable T:Type.
Variable Bp:Bool_Alg_p T.
Let Btp := Btype_p T (Bc_p T Bp).


Lemma in_ba_p_ens_times : 
  forall (x y:T) (pfx:Ensembles.In (ba_p_ens Bp) x) 
         (pfy:Ensembles.In (ba_p_ens Bp) y),
    Ensembles.In (ba_p_ens Bp) (proj1_sig ((exist _ _ pfx) %* (exist _ _ pfy))).
  intros. apply proj2_sig. 
Qed.

Lemma in_ba_p_ens_plus : 
  forall (x y:T) (pfx:Ensembles.In (ba_p_ens Bp) x) 
         (pfy:Ensembles.In (ba_p_ens Bp) y),
    Ensembles.In (ba_p_ens Bp) (proj1_sig ((exist _ _ pfx) %+ (exist _ _ pfy))).
  intros. apply proj2_sig. 
Qed.


Lemma in_ba_p_ens_comp : 
  forall (x:T) (pfx:Ensembles.In (ba_p_ens Bp) x),
           Ensembles.In (ba_p_ens Bp) (proj1_sig (%-(exist _ _ pfx))).
  intros. apply proj2_sig. 
Qed.

Lemma in_ba_p_ens_one : 
  Ensembles.In (ba_p_ens Bp) (proj1_sig (Bone_p T (Bc_p T Bp))).
apply proj2_sig.
Qed.

Lemma in_ba_p_ens_zero : 
  Ensembles.In (ba_p_ens Bp) (proj1_sig (Bzero_p T (Bc_p T Bp))).
apply proj2_sig.
Qed.

Lemma in_ba_p_ens_plus' : 
  forall (x y:Btp),
    Ensembles.In (ba_p_ens Bp) (proj1_sig (x %+ y)).
destruct x, y.
apply in_ba_p_ens_plus.
Qed.

Lemma in_ba_p_ens_times' : 
  forall (x y:Btp),
    Ensembles.In (ba_p_ens Bp) (proj1_sig (x %* y)).
destruct x, y.
apply in_ba_p_ens_times.
Qed.


Lemma in_ba_p_ens_comp' : 
  forall x:Btp,
    Ensembles.In (ba_p_ens Bp) (proj1_sig (%- x)).
destruct x.
apply in_ba_p_ens_comp.
Qed.




Lemma in_bs_p : forall x:Btp, Ensembles.In (BS_p T (Bc_p T Bp)) x.
intros; rewrite und_set_p; constructor.
Qed.


Lemma bc_inj_p : 
  forall {A A':Bool_Alg_p T}, 
    Bc_p T A = Bc_p T A' -> A = A'.
intros A A' h1.
destruct A; destruct A'.
simpl in h1. subst.
f_equal; apply proof_irrelevance.
Qed.


Lemma dist_sum_r_p : forall (n m p: Btp), (n %+ m) %* p = n%*p %+ m%*p.
intros n m p.    
do 2 rewrite ba_conv_plus. do 3 rewrite ba_conv_times.  
repeat rewrite <- ba_conv_elt_eq. 
apply (@dist_sum_r (ba_conv Bp)).
Qed.

Lemma dist_prod_r_p : forall (n m p: Btp), (n %* m) %+ p = (n%+p) %* (m%+p).
intros n m p.    
do 3 rewrite ba_conv_plus. do 2 rewrite ba_conv_times.  
repeat rewrite <- ba_conv_elt_eq. 
apply (@dist_prod_r (ba_conv Bp)).
Qed.

Lemma idem_sum_p : forall b : Btp, b %+ b = b.
intro b. rewrite ba_conv_plus.
repeat rewrite <- ba_conv_elt_eq.
apply (@idem_sum (ba_conv Bp)).
Qed.

Lemma idem_prod_p: forall (b: Btp), b %* b = b.
intro b. rewrite ba_conv_times.
repeat rewrite <- ba_conv_elt_eq.
apply (@idem_prod (ba_conv Bp)).
Qed.

Lemma eq_ord_p: forall (x y: Btp),
  x %+ y = y <-> x %* y = x.
intros x y. rewrite ba_conv_times. rewrite ba_conv_plus.
repeat rewrite <- ba_conv_elt_eq.
apply (@eq_ord (ba_conv Bp)).
Qed.

Definition le_p (x y:Btp): Prop :=
  x %+ y = y.

Lemma le_p_iff : 
  forall (x y:Btp),
    le_p x y <-> le (ba_conv_elt x) (ba_conv_elt y).
intros x y.
unfold le_p, le.
repeat rewrite <- ba_conv_elt_eq. simpl.
tauto.
Qed.

Lemma refl_le_p : forall (x: Btp), le_p x x.
intro x. rewrite le_p_iff.
repeat rewrite <- ba_conv_elt_eq.
apply refl_le.
Qed.

Lemma anti_sym_le_p: forall (x y: Btp),
  le_p x y -> le_p y x -> x = y.
intros x y. rewrite le_p_iff.
repeat rewrite <- ba_conv_elt_eq.
apply anti_sym_le.
Qed.

Lemma trans_le_p: forall (x y z : Btp),
  le_p x y -> le_p y z -> le_p x z.
intros x y z. rewrite le_p_iff.
repeat rewrite <- ba_conv_elt_eq.
apply trans_le.
Qed.

Lemma times_le_p : forall (x y : Btp), le_p (x %* y) x.
intros x y. rewrite le_p_iff. rewrite ba_conv_times.
repeat rewrite <- ba_conv_elt_eq.
apply times_le.
Qed.

Lemma le_plus_p : forall (x y : Btp), le_p x (x %+ y).
intros x y. rewrite le_p_iff. rewrite ba_conv_plus.
repeat rewrite <- ba_conv_elt_eq.
apply le_plus.
Qed.

Definition lt_p (x y:Btp) := le_p x y /\ x <> y.

Lemma lt_p_iff :
  forall (x y:Btp),
    lt_p x y <-> lt (transfer (ba_conv_type Bp) x) (transfer (ba_conv_type Bp) y).
intros x y. unfold lt_p. rewrite le_p_iff. unfold lt.
tauto.
Qed.

(* b is a lower bound of S in B*)
Definition lb_p (S: Ensemble Btp) (b: Btp) : Prop :=
  forall (s:Btp), Ensembles.In S s -> le_p b s.

Lemma lb_p_iff :
  forall (S:Ensemble Btp) (b:Btp),
    lb_p S b <-> lb (ba_conv_set S) (ba_conv_elt b).
intros S b. 
unfold ba_conv_set. unfold ba_conv_type.
rewrite transfer_dep_eq_refl. unfold transfer. unfold eq_rect_r. simpl.
unfold lb_p. unfold lb. simpl.
split.
intros h1 s h2. specialize (h1 _ h2). rewrite le_p_iff in h1.
repeat rewrite <- ba_conv_elt_eq in h1.
assumption.
intros h1 s h2.
rewrite le_p_iff.
repeat rewrite <- ba_conv_elt_eq.
apply h1; auto.
Qed.

(* b is an upper bound of S in B*)
Definition ub_p (S: Ensemble Btp) (b: Btp) : Prop :=
  forall (s:Btp), Ensembles.In S s -> le_p s b.

Lemma ub_p_iff :
  forall (S:Ensemble Btp) (b:Btp),
    ub_p S b <-> ub (ba_conv_set S) (ba_conv_elt b).
intros S b. 
unfold ba_conv_set. unfold ba_conv_type.
rewrite transfer_dep_eq_refl. unfold transfer. unfold eq_rect_r. simpl.
unfold lb_p. unfold lb. simpl.
split.
intros h1 s h2. specialize (h1 _ h2). rewrite le_p_iff in h1.
repeat rewrite <- ba_conv_elt_eq in h1.
assumption.
intros h1 s h2.
rewrite le_p_iff.
repeat rewrite <- ba_conv_elt_eq.
apply h1; auto.
Qed.


(* b is the infimum (greatest lower bound) of S in B *)
Definition inf_p (S: Ensemble Btp) (b: Btp) : Prop :=
  lb_p S b /\ forall b':Btp, lb_p S b' -> le_p b' b.

Lemma inf_p_iff :
  forall (S:Ensemble Btp) (b:Btp),
    inf_p S b <-> inf (ba_conv_set S) (ba_conv_elt b).
intros S b. unfold ba_conv_set. unfold ba_conv_type. 
rewrite transfer_dep_eq_refl. unfold transfer. unfold eq_rect_r. simpl.
unfold inf_p. unfold inf. simpl.
split.
intros h1. destruct h1 as [h1l h1r]. split.
rewrite lb_p_iff in h1l.  rewrite <- ba_conv_elt_eq in h1l. 
unfold ba_conv_set in h1l. unfold ba_conv_type in h1l. rewrite transfer_dep_eq_refl in h1l.
assumption.
intros b' h2. 
pose proof (lb_p_iff S b') as h3.
rewrite <- ba_conv_elt_eq in h3. unfold ba_conv_set in h3. unfold ba_conv_type in h3. 
rewrite transfer_dep_eq_refl in h3.
rewrite <- h3 in h2.
pose proof (h1r _ h2) as h4.
rewrite le_p_iff in h4.
repeat rewrite <- ba_conv_elt_eq in h4.
assumption.
intro h1.
destruct h1 as [h1l h1r].
split.
rewrite lb_p_iff.
repeat rewrite <- ba_conv_elt_eq.
unfold ba_conv_set. unfold ba_conv_type. rewrite transfer_dep_eq_refl.
assumption.
intros b' h2.
rewrite lb_p_iff in h2.
repeat rewrite <- ba_conv_elt_eq in h2.
unfold ba_conv_set in h2. unfold ba_conv_type in h2. rewrite transfer_dep_eq_refl in h2.
specialize (h1r _ h2).
rewrite le_p_iff.
do 2 rewrite <- ba_conv_elt_eq.
assumption.
Qed.

(* b is the supremum (least upper bound) of S in B*)
Definition sup_p (S: Ensemble Btp) (b: Btp) : Prop :=
  ub_p S b /\ forall b':Btp, ub_p S b' -> le_p b b'. 


Lemma sup_p_iff :
  forall (S:Ensemble Btp) (b:Btp),
    sup_p S b <-> sup (ba_conv_set S) (ba_conv_elt b).
intros S b. unfold ba_conv_set. unfold ba_conv_type. 
rewrite transfer_dep_eq_refl. unfold transfer. unfold eq_rect_r. simpl.
unfold sup_p. unfold sup. simpl.
split.
intros h1. destruct h1 as [h1l h1r]. split.
rewrite ub_p_iff in h1l.  rewrite <- ba_conv_elt_eq in h1l. 
unfold ba_conv_set in h1l. unfold ba_conv_type in h1l. rewrite transfer_dep_eq_refl in h1l.
assumption.
intros b' h2. 
pose proof (ub_p_iff S b') as h3.
rewrite <- ba_conv_elt_eq in h3. unfold ba_conv_set in h3. unfold ba_conv_type in h3. 
rewrite transfer_dep_eq_refl in h3.
rewrite <- h3 in h2.
pose proof (h1r _ h2) as h4.
rewrite le_p_iff in h4.
repeat rewrite <- ba_conv_elt_eq in h4.
assumption.
intro h1.
destruct h1 as [h1l h1r].
split.
rewrite ub_p_iff.
repeat rewrite <- ba_conv_elt_eq.
unfold ba_conv_set. unfold ba_conv_type. rewrite transfer_dep_eq_refl.
assumption.
intros b' h2.
rewrite ub_p_iff in h2.
repeat rewrite <- ba_conv_elt_eq in h2.
unfold ba_conv_set in h2. unfold ba_conv_type in h2. rewrite transfer_dep_eq_refl in h2.
specialize (h1r _ h2).
rewrite le_p_iff.
do 2 rewrite <- ba_conv_elt_eq.
assumption.
Qed.

Lemma inf_le_p : forall (A:Ensemble Btp) (b:Btp),
                 inf_p A b -> forall b':Btp, Ensembles.In A b' -> le_p b b'.
intros A b h1 b' h2.
rewrite le_p_iff. do 2 rewrite <- ba_conv_elt_eq. 
rewrite inf_p_iff in h1.
rewrite <- ba_conv_elt_eq in h1. unfold ba_conv_set in h1. unfold ba_conv_type in h1.
rewrite transfer_dep_eq_refl in h1.
apply inf_le with A; auto.
Qed.

Lemma le_sup_p : forall (A:Ensemble Btp) (b:Btp),
                   sup_p A b -> forall b':Btp, Ensembles.In A b' -> le_p b' b.
intros A b h1 b' h2.
rewrite le_p_iff. do 2 rewrite <- ba_conv_elt_eq. 
rewrite sup_p_iff in h1.
rewrite <- ba_conv_elt_eq in h1. unfold ba_conv_set in h1. unfold ba_conv_type in h1.
rewrite transfer_dep_eq_refl in h1.
apply le_sup with A; auto.
Qed.

Lemma inf_singleton_p : forall (x:Btp), inf_p (Singleton x) x.
intro x. rewrite inf_p_iff. rewrite <- ba_conv_elt_eq.
unfold ba_conv_set. unfold ba_conv_type. rewrite transfer_dep_eq_refl.
apply inf_singleton.
Qed.

Lemma sup_singleton_p : forall (x:Btp), sup_p (Singleton x) x.
intro x. rewrite sup_p_iff. rewrite <- ba_conv_elt_eq.
unfold ba_conv_set. unfold ba_conv_type. rewrite transfer_dep_eq_refl.
apply sup_singleton.
Qed.

Lemma lat_sum_p : forall (x y: Btp),
  sup_p (Couple x y) (x %+ y).
intros x y.
rewrite sup_p_iff. 
rewrite ba_conv_plus. 
repeat rewrite <- ba_conv_elt_eq.
unfold ba_conv_set. unfold ba_conv_type.
rewrite transfer_dep_eq_refl. 
apply lat_sum; auto.
Qed.

Lemma lat_prod_p : 
  forall (x y: Btp),
    inf_p (Couple x y) (x %* y).
intros x y.
rewrite inf_p_iff. 
rewrite ba_conv_times. 
repeat rewrite <- ba_conv_elt_eq.
unfold ba_conv_set. unfold ba_conv_type.
rewrite transfer_dep_eq_refl. 
apply lat_prod; auto.
Qed.

Lemma inf_unq_p : forall (S:Ensemble Btp) (x y: Btp),
  inf_p S x -> inf_p S y -> x = y.
intros S x y h1 h2.
rewrite inf_p_iff in h1. rewrite inf_p_iff in h2.
rewrite <- ba_conv_elt_eq in h1. rewrite <- ba_conv_elt_eq in h2.
unfold ba_conv_set in h1. unfold ba_conv_type in h1. 
unfold ba_conv_set in h2. unfold ba_conv_type in h2.
rewrite transfer_dep_eq_refl in h1. rewrite transfer_dep_eq_refl in h2.
apply (@inf_unq (ba_conv Bp) S); auto.
Qed.

Lemma sup_unq_p : forall (S:Ensemble Btp) (x y: Btp),
  sup_p S x -> sup_p S y -> x = y.
intros S x y h1 h2.
rewrite sup_p_iff in h1. rewrite sup_p_iff in h2.
rewrite <- ba_conv_elt_eq in h1. rewrite <- ba_conv_elt_eq in h2.
unfold ba_conv_set in h1. unfold ba_conv_type in h1. 
unfold ba_conv_set in h2. unfold ba_conv_type in h2.
rewrite transfer_dep_eq_refl in h1. rewrite transfer_dep_eq_refl in h2.
apply (@sup_unq (ba_conv Bp) S); auto.
Qed.

Lemma ex_inf_unq_p :
  forall (A:Ensemble Btp),
    (exists p:Btp, inf_p A p) ->
    (exists! p:Btp, inf_p A p).
intros A h1.
destruct h1 as [p h1].
rewrite inf_p_iff in h1.
rewrite <- ba_conv_elt_eq in h1.
unfold ba_conv_set in h1. unfold ba_conv_type in h1.
rewrite transfer_dep_eq_refl in h1.
pose proof (ex_inf_unq _ (transfer_dep (ba_conv_type Bp) A)) as h2. simpl in h2.
specialize (h2 (ex_intro _ p h1)).
destruct h2 as [b h2].
exists b.
red in h2. red.
destruct h2 as [h2 h3]. split.
rewrite inf_p_iff. rewrite <- ba_conv_elt_eq. assumption.
intros x h4.
rewrite inf_p_iff in h4. rewrite <- ba_conv_elt_eq in h4.
apply h3; auto.
Qed.

Lemma ex_sup_unq_p :
  forall (A:Ensemble Btp),
    (exists p:Btp, sup_p A p) ->
    (exists! p:Btp, sup_p A p).
intros A h1.
destruct h1 as [p h1].
rewrite sup_p_iff in h1.
rewrite <- ba_conv_elt_eq in h1.
unfold ba_conv_set in h1. unfold ba_conv_type in h1.
rewrite transfer_dep_eq_refl in h1.
pose proof (ex_sup_unq _ (transfer_dep (ba_conv_type Bp) A)) as h2. simpl in h2.
specialize (h2 (ex_intro _ p h1)).
destruct h2 as [b h2].
exists b.
red in h2. red.
destruct h2 as [h2 h3]. split.
rewrite sup_p_iff. rewrite <- ba_conv_elt_eq. assumption.
intros x h4.
rewrite sup_p_iff in h4. rewrite <- ba_conv_elt_eq in h4.
apply h3; auto.
Qed.


Lemma zero_min_p : forall (x : Btp), le_p %0 x.
intro x. rewrite le_p_iff. rewrite ba_conv_zero.
do 2 rewrite <- ba_conv_elt_eq.
apply zero_min.
Qed.

Lemma le_x_0_p : forall (x:Btp), le_p x %0 -> x = %0.
intros x h1. rewrite le_p_iff in h1. rewrite ba_conv_zero in h1.
do 2 rewrite <- ba_conv_elt_eq in h1. rewrite ba_conv_zero.
apply (@le_x_0 (ba_conv Bp)); auto.
Qed.

Lemma one_max_p : forall (x : Btp), le_p x %1.
intro x. rewrite le_p_iff. rewrite ba_conv_one.
do 2 rewrite <- ba_conv_elt_eq.
apply one_max.
Qed.

Lemma le_1_x_p : forall (x:Btp), le_p %1 x -> x = %1.
intros x h1. rewrite le_p_iff in h1. rewrite ba_conv_one in h1.
do 2 rewrite <- ba_conv_elt_eq in h1. rewrite ba_conv_one.
apply (@le_1_x (ba_conv Bp)); auto.
Qed.

Lemma zero_sum_p : forall (x : Btp), x %+ %0 = x.
intro x. rewrite ba_conv_plus. rewrite ba_conv_zero.
repeat rewrite <- ba_conv_elt_eq.
apply (@zero_sum (ba_conv Bp)).
Qed.

Lemma zero_prod_p : forall (x : Btp), x %* %0 = %0.
intro x. rewrite ba_conv_times. rewrite ba_conv_zero.
repeat rewrite <- ba_conv_elt_eq.
apply (@zero_prod (ba_conv Bp)).
Qed.

Lemma one_sum_p : forall (x : Btp), x %+ %1 = %1.
intro x. rewrite ba_conv_plus. rewrite ba_conv_one.
repeat rewrite <- ba_conv_elt_eq.
apply (@one_sum (ba_conv Bp)).
Qed.

Lemma one_prod_p : forall x : Btp, x %* %1 = x.
intros x. rewrite ba_conv_times. rewrite ba_conv_one.
repeat rewrite <- ba_conv_elt_eq.
apply (@one_prod (ba_conv Bp)).
Qed.

Lemma inf_empty_p : inf_p (Empty_set Btp) %1.
rewrite inf_p_iff. rewrite ba_conv_one.
repeat rewrite <- ba_conv_elt_eq.
unfold ba_conv_set. unfold ba_conv_type. rewrite transfer_dep_eq_refl.
apply inf_empty.
Qed.

Lemma sup_empty_p : sup_p (Empty_set Btp) %0.
rewrite sup_p_iff. rewrite ba_conv_zero.
repeat rewrite <- ba_conv_elt_eq.
unfold ba_conv_set. unfold ba_conv_type. rewrite transfer_dep_eq_refl.
apply sup_empty.
Qed.

Lemma inf_subtract_one_p :
  forall (A:Ensemble Btp) (b:Btp),
    inf_p A b <-> inf_p (Subtract A %1) b.
intros A b. 
rewrite inf_p_iff.
apply inf_subtract_one.
Qed.

Lemma sup_subtract_zero_p :
  forall (A:Ensemble Btp) (b:Btp),
    sup_p A b <-> sup_p (Subtract A %0) b.
intros A b. 
rewrite sup_p_iff.
apply sup_subtract_zero.
Qed.

Lemma lb_subtract_one_p :
  forall (A:Ensemble Btp) (b:Btp),
    lb_p A b <-> lb_p (Subtract A %1) b.
intros A b.
rewrite lb_p_iff.
apply lb_subtract_one.
Qed.

Lemma ub_subtract_zero_p :
  forall (A:Ensemble Btp) (b:Btp),
    ub_p A b <-> ub_p (Subtract A %0) b.
intros A b.
rewrite ub_p_iff.
apply ub_subtract_zero.
Qed.

Lemma comp_unq_p : forall (x y : Btp), %-x = %-y -> x = y.
intros x y.
rewrite ba_conv_comp. rewrite ba_conv_comp.
apply (@comp_unq  (ba_conv Bp)).
Qed.

Lemma doub_neg_p : forall (x : Btp), x = %- %-x.
intro x.
do 2 rewrite ba_conv_comp.
apply (@doub_neg (ba_conv Bp)).
Qed.

Lemma zero_char_p : forall (y : Btp),
  (forall (x : Btp), x %+ y = x) -> y = %0.
intros y h1.
assert (h2:forall x:(Btype (Bc (ba_conv Bp))), x + y = x).
  intro x.
  apply h1.
rewrite ba_conv_zero.
apply (@zero_char (ba_conv Bp)); auto.
Qed.

Lemma one_char_p : forall (y : Btp),
  (forall (x : Btp), x %* y = x) -> y = %1.
intros y h1.
assert (h2:forall x:(Btype (Bc (ba_conv Bp))), x * y = x).
  intro x.
  apply h1.
rewrite ba_conv_one.
apply (@one_char (ba_conv Bp)); auto.
Qed.

Lemma comp_char_p : forall (x y : Btp),
  x %* y = %0 -> x %+ y = %1 -> y = %-x.
intros x y.
rewrite ba_conv_zero. rewrite ba_conv_one. rewrite ba_conv_times.
rewrite ba_conv_comp. rewrite ba_conv_plus.
apply (@comp_char (ba_conv Bp)).
Qed.

Lemma comp_eq_p : forall (x y : Btp), %-x = %-y -> x = y.
intros x y. do 2 rewrite ba_conv_comp.
apply (@comp_eq (ba_conv Bp)).
Qed.

Lemma zero_comp_p: %-%0 = Bone_p T (Bc_p T Bp).
rewrite ba_conv_zero. rewrite ba_conv_comp. rewrite ba_conv_one.
apply (@zero_comp (ba_conv Bp)).
Qed.

Lemma one_comp_p: %-(%1) = Bzero_p T (Bc_p T Bp).
rewrite ba_conv_zero. rewrite ba_conv_comp. rewrite ba_conv_one.
apply (@one_comp (ba_conv Bp)).
Qed.

Lemma de_mor_sum_p : forall (x y : Btp), %-(x %+ y) = %-x %* %-y.
intros x y.
do 3 rewrite ba_conv_comp. rewrite ba_conv_plus. rewrite ba_conv_times.
apply (@de_mor_sum (ba_conv Bp)).
Qed.

Lemma de_mor_prod_p : forall (x y : Btp), %-(x%*y) = %-x %+ %-y.
intros x y.
do 3 rewrite ba_conv_comp. rewrite ba_conv_plus. rewrite ba_conv_times.
apply (@de_mor_prod (ba_conv Bp)).
Qed.

Lemma times_eq_plus_p : forall (x y: Btp), x%*y = %-((%-x) %+ (%-y)).
intros x y.
rewrite ba_conv_times. do 3 rewrite ba_conv_comp. rewrite ba_conv_plus.
apply (@times_eq_plus (ba_conv Bp)).
Qed.

Lemma plus_eq_times_p : forall (x y:Btp), x%+y = %-((%-x)%*(%-y)).
intros x y.
rewrite ba_conv_times. do 3 rewrite ba_conv_comp. rewrite ba_conv_plus.
apply (@plus_eq_times (ba_conv Bp)).
Qed.

Lemma mono_sum_p : forall (x x' y y' : Btp),
  le_p x x' -> le_p y y' -> le_p (x%+y) (x' %+ y').
intros x x' y y'. rewrite le_p_iff.
do 2 rewrite ba_conv_plus.
apply (@mono_sum (ba_conv Bp)).
Qed.

Lemma mono_prod_p : forall (x x' y y': Btp),
  le_p x x' -> le_p y y' -> le_p (x %* y) (x' %* y').
intros x x' y y'. rewrite le_p_iff.
do 2 rewrite ba_conv_times.
apply (@mono_prod (ba_conv Bp)).
Qed.

Lemma le_iff_p' : forall (a b:Btp), le_p a (%-b) <-> a %* b = %0.
intros a b. rewrite le_p_iff.
rewrite ba_conv_times. rewrite ba_conv_comp. rewrite ba_conv_zero.
apply (@le_iff' (ba_conv Bp)).
Qed.

Lemma le_iff_p : forall (a b:Btp), le_p a b <-> a %* %-b = %0.
intros a b. rewrite le_p_iff.
rewrite ba_conv_times. rewrite ba_conv_comp. rewrite ba_conv_zero.
apply (@le_iff (ba_conv Bp)).
Qed.

Lemma mono_comp_p : forall (x x' : Btp), le_p x x' -> le_p (%-x') (%-x).
intros x x'.
rewrite le_p_iff. do 2 rewrite ba_conv_comp.
apply (@mono_comp (ba_conv Bp)).
Qed.

Definition comp_set_p (A:Ensemble Btp) :=
  Im A (Bcomp_p T (Bc_p T Bp)).

Lemma comp_set_p_eq : 
  forall (A:Ensemble Btp),
    comp_set_p A = comp_set (ba_conv_set A).
intros A.
unfold comp_set_p. unfold comp_set. unfold ba_conv_set.
unfold ba_conv_type. rewrite transfer_dep_eq_refl.
f_equal.
Qed.


Lemma comp_set_p_comp_set_p : forall (A:Ensemble Btp), comp_set_p (comp_set_p A) = A.
intro A. do 2 rewrite comp_set_p_eq.
unfold ba_conv_set. unfold ba_conv_type. 
do 2 rewrite transfer_dep_eq_refl.
apply (@comp_set_comp_set (ba_conv Bp)).
Qed.

Lemma lb_ub_compat_iff_p :
  forall (A:Ensemble Btp) (p:Btp),
    lb_p A p <-> ub_p (comp_set_p A) (%-p).
intros A p.
rewrite lb_p_iff. rewrite ub_p_iff. rewrite comp_set_p_eq. rewrite ba_conv_comp.
apply (@lb_ub_compat_iff (ba_conv Bp)).
Qed.

Lemma ub_lb_compat_iff_p :
  forall (A:Ensemble Btp) (p:Btp),
    ub_p A p <-> lb_p (comp_set_p A) (%-p).
intros A p.
rewrite lb_p_iff. rewrite ub_p_iff. rewrite comp_set_p_eq. rewrite ba_conv_comp.
apply (@ub_lb_compat_iff (ba_conv Bp)).
Qed.

Lemma sup_inf_compat_iff_p :
  forall (A:Ensemble Btp) (p:Btp),
    sup_p A p <-> inf_p (comp_set_p A) (%-p).
intros A p.
rewrite sup_p_iff. rewrite inf_p_iff. rewrite comp_set_p_eq. rewrite ba_conv_comp.
apply (@sup_inf_compat_iff (ba_conv Bp)).
Qed.

Lemma inf_sup_compat_iff_p :
  forall (A:Ensemble Btp) (p:Btp),
    inf_p A p <-> sup_p (comp_set_p A) (%-p).
intros A p.
rewrite sup_p_iff. rewrite inf_p_iff. rewrite comp_set_p_eq. rewrite ba_conv_comp.
apply (@inf_sup_compat_iff (ba_conv Bp)).
Qed.

Definition atom_p (a:Btp) :
  Prop := a <> %0 /\ forall b:Btp, le_p b a -> a = b \/ b = %0.

Lemma atom_p_iff : 
  forall (a:Btp), atom_p a <-> atom (ba_conv_elt a).
intro a. unfold atom_p, atom.
rewrite ba_conv_zero. rewrite <- ba_conv_elt_eq.
tauto.
Qed.

Lemma atom_iff_p : forall (a:Btp), atom_p a <-> (forall b:Btp, (le_p a b \/ a %* b = %0) /\ ~ (le_p a b /\ a%*b = %0)).
intro a.
rewrite atom_p_iff.
rewrite <- ba_conv_elt_eq.
split.
intros h1 b.
rewrite atom_iff in h1.
specialize (h1 b).
rewrite le_p_iff. 
assumption.
intros h1.
rewrite atom_iff.
assumption.
Qed.

Lemma atom_impl_p : forall (a:Btp), atom_p a -> a <> %0 /\ (forall b:Btp, a %* b = a \/ a %* b = %0).
intro a.
rewrite atom_p_iff. rewrite <- ba_conv_elt_eq.
intro h1.
apply atom_impl in h1.
assumption.
Qed.

Definition sym_diff_p (x y:Btp) := x %* %-y %+ y %* %-x.


Lemma incl_ba_conv_set_iff : 
  forall (E F:Ensemble (btp Bp)),
    Included E F <-> Included (ba_conv_set E) (ba_conv_set F).
intros E F. split.
intro h1. red. intros x h2.
unfold ba_conv_set in h2. unfold ba_conv_type in h2. rewrite transfer_dep_eq_refl in h2.
apply h1; auto. 
intros h1. red. intros x h2.
apply h1; auto.
Qed.


Lemma eq_ba_conv_set_iff : 
  forall (E F:Ensemble (btp Bp)),
    E = F <-> ba_conv_set E = ba_conv_set F.
intros; subst; tauto.
Qed.



Lemma list_to_set_ba_conv_list : 
  forall (l:list (btp Bp)),
    ListUtilities.list_to_set (ba_conv_list l) =
    ba_conv_set (ListUtilities.list_to_set l).
intro l.
apply Extensionality_Ensembles.
red. split.
red. intros x h1.
unfold ba_conv_set. unfold ba_conv_type. rewrite transfer_dep_eq_refl.
unfold ba_conv_list in h1. unfold ba_conv_type in h1.
rewrite transfer_dep_eq_refl in h1.
assumption.
red. intros x h1.
unfold ba_conv_set in h1. unfold ba_conv_type in h1. rewrite transfer_dep_eq_refl in h1.
unfold ba_conv_list. unfold ba_conv_type. rewrite transfer_dep_eq_refl.
assumption.
Qed.

      

End ParametricAnalogues.


(*Definitions*)
Arguments le_p [T] [Bp] x y.
Arguments lt_p [T] [Bp] x y.
Arguments lb_p [T] [Bp] S b.
Arguments ub_p [T] [Bp] S b.
Arguments inf_p [T] [Bp] S b.
Arguments sup_p [T] [Bp] S b.
Arguments atom_p [T] [Bp] _.
Arguments sym_diff_p [T] [Bp] _ _.
Arguments comp_set_p [T] [Bp] _ _.

Notation "x /%\ y" := (sym_diff_p x y) (at level 50, left associativity).

(*broken up for top-level Notation*)
Section ParametricAnalogues2.
Variable T:Type.
Variable Bp:Bool_Alg_p T.
Let Btp := btp Bp.

Lemma sym_diff_p_eq : 
  forall (x y:Btp), x /%\ y = (ba_conv_elt x) /_\ (ba_conv_elt y).
intros x y.
unfold sym_diff_p, sym_diff.
do 2 rewrite ba_conv_times. rewrite ba_conv_plus. do 2 rewrite ba_conv_comp.
repeat rewrite <- ba_conv_elt_eq.
reflexivity.
Qed.


Lemma comm_sym_diff_p : forall (x y:Btp), x /%\ y = y /%\ x.
intros x y. rewrite sym_diff_p_eq. rewrite (sym_diff_p_eq y x).
apply (@comm_sym_diff (ba_conv Bp)).
Qed.

Lemma sym_diff_ref_p : forall x:Btp, x /%\ x = %0.
intros. rewrite sym_diff_p_eq. rewrite ba_conv_zero.
apply (@sym_diff_ref (ba_conv Bp)).
Qed.

Lemma zero_sum_each_zero_p: forall (x y:Btp), x %+ y = %0 <-> x = %0 /\ y = %0.
intros x y. rewrite ba_conv_plus. rewrite ba_conv_zero.
apply (@zero_sum_each_zero (ba_conv Bp)).
Qed.

Lemma one_prod_each_one_p: forall (x y:Btp), x %* y = %1 <-> x = %1 /\ y = %1.
intros x y. rewrite ba_conv_times. rewrite ba_conv_one.
apply (@one_prod_each_one (ba_conv Bp)).
Qed.

Lemma sym_diff_iff_p : forall (x y:Btp), x /%\ y = %0 <-> x = y.
intros x y. rewrite sym_diff_p_eq. rewrite ba_conv_zero.
apply (@sym_diff_iff (ba_conv Bp)).
Qed.

End ParametricAnalogues2.


(*Theorems*)
Arguments dist_sum_r_p [T] [Bp] n m p.
Arguments dist_prod_r_p [T] [Bp] n m p.
Arguments idem_sum_p [T] [Bp] b.
Arguments idem_prod_p [T] [Bp] b.
Arguments eq_ord_p [T] [Bp] x y.
Arguments refl_le_p [T] [Bp] x.
Arguments anti_sym_le_p [T] [Bp] x y _ _.
Arguments trans_le_p [T] [Bp] x y z _ _.
Arguments lat_sum_p [T] [Bp] x y.
Arguments lat_prod_p [T] [Bp] x y.
Arguments inf_unq_p [T] [Bp] S x y _ _.
Arguments sup_unq_p [T] [Bp] S x y _ _.
Arguments zero_min_p [T] [Bp] x.
Arguments one_max_p [T] [Bp] x.
Arguments zero_sum_p [T] [Bp] x.
Arguments zero_prod_p [T] [Bp] x.
Arguments one_sum_p [T] [Bp] x.
Arguments one_prod_p [T] [Bp] x.
Arguments comp_unq_p [T] [Bp] x y _.
Arguments doub_neg_p [T] [Bp] x.
Arguments zero_char_p [T] [Bp] y _.
Arguments one_char_p [T] [Bp] y _.
Arguments comp_char_p [T] [Bp] x y _ _.
Arguments comp_eq_p [T] [Bp] x y _.
Arguments zero_comp_p [T] [Bp].
Arguments one_comp_p [T] [Bp].
Arguments de_mor_sum_p [T] [Bp] x y.
Arguments de_mor_prod_p [T] [Bp] x y.
Arguments mono_sum_p [T] [Bp] x x' y y' _ _.
Arguments mono_prod_p [T] [Bp] x x' y y' _ _.
Arguments mono_comp_p [T] [Bp] x x' _.
Arguments le_plus_p [T] [Bp] _ _.
Arguments le_x_0_p [T] [Bp] _ _.
Arguments le_1_x_p [T] [Bp] _ _.
Arguments times_le_p [T] [Bp] _ _.
Arguments atom_iff_p [T] [Bp] _.
Arguments in_bs_p [T] [Bp] _.
Arguments comm_sym_diff_p [T] [Bp] _ _.
Arguments sym_diff_ref_p [T] [Bp] _.
Arguments zero_sum_each_zero_p [T] [Bp] _ _.
Arguments one_prod_each_one_p [T] [Bp] _ _.
Arguments sym_diff_iff_p [T] [Bp] _ _.
Arguments comp_set_p_eq [T] [Bp] _.
Arguments ba_p_ens [T] _ _.
Arguments btp [T] _.

Section MoreParametricBAs.

Lemma in_ba_p_ens_eq : 
  forall {T:Type} (x:T) (C D:Bool_Alg_p T),
    C = D ->
    (Ensembles.In (ba_p_ens C) x <->
    Ensembles.In (ba_p_ens D) x).
intros; subst; tauto.
Qed.


Lemma ba_p_subst_times : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    forall (x y:T) 
           (pfxa:Ensembles.In (ba_p_ens Ap) x)
           (pfya:Ensembles.In (ba_p_ens Ap) y)
           (pfxb:Ensembles.In (ba_p_ens Bp) x)
           (pfyb:Ensembles.In (ba_p_ens Bp) y),
           proj1_sig (exist _ _ pfxa %* exist _ _ pfya) = 
           proj1_sig (exist _ _ pfxb %* exist _ _ pfyb).
intros; subst.
assert (pfxa = pfxb). apply proof_irrelevance. assert (pfya = pfyb). apply proof_irrelevance. subst.
reflexivity.
Qed.

Lemma ba_p_subst_times3 : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    forall (x y z:T) 
           (pfxa:Ensembles.In (ba_p_ens Ap) x)
           (pfya:Ensembles.In (ba_p_ens Ap) y) 
           (pfza:Ensembles.In (ba_p_ens Ap) z)
           (pfxb:Ensembles.In (ba_p_ens Bp) x)
           (pfyb:Ensembles.In (ba_p_ens Bp) y)
           (pfzb:Ensembles.In (ba_p_ens Bp) z),
           proj1_sig (exist _ _ pfxa %* exist _ _ pfya %* exist _ _ pfza) = 
           proj1_sig (exist _ _ pfxb %* exist _ _ pfyb %* exist _ _ pfzb).
intros; subst.
assert (pfxa = pfxb). apply proof_irrelevance. assert (pfya = pfyb). apply proof_irrelevance.
assert (pfza = pfzb). apply proof_irrelevance.
subst.
reflexivity.
Qed.

Lemma ba_p_subst_times3' : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    forall (x y z:T) 
           (pfxa:Ensembles.In (ba_p_ens Ap) x)
           (pfya:Ensembles.In (ba_p_ens Ap) y) 
           (pfza:Ensembles.In (ba_p_ens Ap) z)
           (pfxb:Ensembles.In (ba_p_ens Bp) x)
           (pfyb:Ensembles.In (ba_p_ens Bp) y)
           (pfzb:Ensembles.In (ba_p_ens Bp) z),
           proj1_sig (exist _ _ pfxa %* (exist _ _ pfya %* exist _ _ pfza)) = 
           proj1_sig (exist _ _ pfxb %* (exist _ _ pfyb %* exist _ _ pfzb)).
intros; subst.
assert (pfxa = pfxb). apply proof_irrelevance. assert (pfya = pfyb). apply proof_irrelevance.
assert (pfza = pfzb). apply proof_irrelevance.
subst.
reflexivity.
Qed.


Lemma ba_p_subst_plus : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    forall (x y:T) 
           (pfxa:Ensembles.In (ba_p_ens Ap) x)
           (pfya:Ensembles.In (ba_p_ens Ap) y)
           (pfxb:Ensembles.In (ba_p_ens Bp) x)
           (pfyb:Ensembles.In (ba_p_ens Bp) y),
           proj1_sig (exist _ _ pfxa %+ exist _ _ pfya) = 
           proj1_sig (exist _ _ pfxb %+ exist _ _ pfyb).
intros; subst.
assert (pfxa = pfxb). apply proof_irrelevance. assert (pfya = pfyb). apply proof_irrelevance. subst.
reflexivity.
Qed.

Lemma ba_p_subst_plus3 : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    forall (x y z:T) 
           (pfxa:Ensembles.In (ba_p_ens Ap) x)
           (pfya:Ensembles.In (ba_p_ens Ap) y) 
           (pfza:Ensembles.In (ba_p_ens Ap) z)
           (pfxb:Ensembles.In (ba_p_ens Bp) x)
           (pfyb:Ensembles.In (ba_p_ens Bp) y)
           (pfzb:Ensembles.In (ba_p_ens Bp) z),
           proj1_sig (exist _ _ pfxa %+ exist _ _ pfya %+ exist _ _ pfza) = 
           proj1_sig (exist _ _ pfxb %+ exist _ _ pfyb %+ exist _ _ pfzb).
intros; subst.
assert (pfxa = pfxb). apply proof_irrelevance. assert (pfya = pfyb). apply proof_irrelevance.
assert (pfza = pfzb). apply proof_irrelevance.
subst.
reflexivity.
Qed.


Lemma ba_p_subst_plus3' : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    forall (x y z:T) 
           (pfxa:Ensembles.In (ba_p_ens Ap) x)
           (pfya:Ensembles.In (ba_p_ens Ap) y) 
           (pfza:Ensembles.In (ba_p_ens Ap) z)
           (pfxb:Ensembles.In (ba_p_ens Bp) x)
           (pfyb:Ensembles.In (ba_p_ens Bp) y)
           (pfzb:Ensembles.In (ba_p_ens Bp) z),
           proj1_sig (exist _ _ pfxa %+ (exist _ _ pfya %+ exist _ _ pfza)) = 
           proj1_sig (exist _ _ pfxb %+ (exist _ _ pfyb %+ exist _ _ pfzb)).
intros; subst.
assert (pfxa = pfxb). apply proof_irrelevance. assert (pfya = pfyb). apply proof_irrelevance.
assert (pfza = pfzb). apply proof_irrelevance.
subst.
reflexivity.
Qed.


Lemma ba_p_subst_times_plus : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    forall (x y z:T) 
           (pfxa:Ensembles.In (ba_p_ens Ap) x)
           (pfya:Ensembles.In (ba_p_ens Ap) y) 
           (pfza:Ensembles.In (ba_p_ens Ap) z)
           (pfxb:Ensembles.In (ba_p_ens Bp) x)
           (pfyb:Ensembles.In (ba_p_ens Bp) y)
           (pfzb:Ensembles.In (ba_p_ens Bp) z),
           proj1_sig (exist _ _ pfxa %* exist _ _ pfya %+ exist _ _ pfza) = 
           proj1_sig (exist _ _ pfxb %* exist _ _ pfyb %+ exist _ _ pfzb).
intros; subst.
assert (pfxa = pfxb). apply proof_irrelevance. assert (pfya = pfyb). apply proof_irrelevance.
assert (pfza = pfzb). apply proof_irrelevance.
subst.
reflexivity.
Qed.

Lemma ba_p_subst_times_plus' : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    forall (x y z:T) 
           (pfxa:Ensembles.In (ba_p_ens Ap) x)
           (pfya:Ensembles.In (ba_p_ens Ap) y) 
           (pfza:Ensembles.In (ba_p_ens Ap) z)
           (pfxb:Ensembles.In (ba_p_ens Bp) x)
           (pfyb:Ensembles.In (ba_p_ens Bp) y)
           (pfzb:Ensembles.In (ba_p_ens Bp) z),
           proj1_sig (exist _ _ pfxa %* (exist _ _ pfya %+ exist _ _ pfza)) = 
           proj1_sig (exist _ _ pfxb %* (exist _ _ pfyb %+ exist _ _ pfzb)).
intros; subst.
assert (pfxa = pfxb). apply proof_irrelevance. assert (pfya = pfyb). apply proof_irrelevance.
assert (pfza = pfzb). apply proof_irrelevance.
subst.
reflexivity.
Qed.



Lemma ba_p_subst_plus_times : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    forall (x y z:T) 
           (pfxa:Ensembles.In (ba_p_ens Ap) x)
           (pfya:Ensembles.In (ba_p_ens Ap) y) 
           (pfza:Ensembles.In (ba_p_ens Ap) z)
           (pfxb:Ensembles.In (ba_p_ens Bp) x)
           (pfyb:Ensembles.In (ba_p_ens Bp) y)
           (pfzb:Ensembles.In (ba_p_ens Bp) z),
           proj1_sig ((exist _ _ pfxa %+ exist _ _ pfya) %* exist _ _ pfza) = 
           proj1_sig ((exist _ _ pfxb %+ exist _ _ pfyb) %* exist _ _ pfzb).
intros; subst.
assert (pfxa = pfxb). apply proof_irrelevance. assert (pfya = pfyb). apply proof_irrelevance.
assert (pfza = pfzb). apply proof_irrelevance.
subst.
reflexivity.
Qed.

Lemma ba_p_subst_plus_times' : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    forall (x y z:T) 
           (pfxa:Ensembles.In (ba_p_ens Ap) x)
           (pfya:Ensembles.In (ba_p_ens Ap) y) 
           (pfza:Ensembles.In (ba_p_ens Ap) z)
           (pfxb:Ensembles.In (ba_p_ens Bp) x)
           (pfyb:Ensembles.In (ba_p_ens Bp) y)
           (pfzb:Ensembles.In (ba_p_ens Bp) z),
           proj1_sig (exist _ _ pfxa %+ exist _ _ pfya %* exist _ _ pfza) = 
           proj1_sig (exist _ _ pfxb %+ exist _ _ pfyb %* exist _ _ pfzb).
intros; subst.
assert (pfxa = pfxb). apply proof_irrelevance. assert (pfya = pfyb). apply proof_irrelevance.
assert (pfza = pfzb). apply proof_irrelevance.
subst.
reflexivity.
Qed.

Lemma ba_p_subst_times_plus_times : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    forall (x y z w:T) 
           (pfxa:Ensembles.In (ba_p_ens Ap) x)
           (pfya:Ensembles.In (ba_p_ens Ap) y) 
           (pfza:Ensembles.In (ba_p_ens Ap) z)
           (pfwa:Ensembles.In (ba_p_ens Ap) w)
           (pfxb:Ensembles.In (ba_p_ens Bp) x)
           (pfyb:Ensembles.In (ba_p_ens Bp) y)
           (pfzb:Ensembles.In (ba_p_ens Bp) z)
           (pfwb:Ensembles.In (ba_p_ens Bp) w),
           proj1_sig (exist _ _ pfxa %* exist _ _ pfya %+ exist _ _ pfza %* exist _ _ pfwa) = 
           proj1_sig (exist _ _ pfxb %* exist _ _ pfyb %+ exist _ _ pfzb %* exist _ _ pfwb).
intros; subst.
assert (pfxa = pfxb). apply proof_irrelevance. assert (pfya = pfyb). apply proof_irrelevance.
assert (pfza = pfzb). apply proof_irrelevance.  assert (pfwa = pfwb). apply proof_irrelevance.
subst.
reflexivity.
Qed.

Lemma ba_p_subst_plus_times_plus : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    forall (x y z w:T) 
           (pfxa:Ensembles.In (ba_p_ens Ap) x)
           (pfya:Ensembles.In (ba_p_ens Ap) y) 
           (pfza:Ensembles.In (ba_p_ens Ap) z)
           (pfwa:Ensembles.In (ba_p_ens Ap) w)
           (pfxb:Ensembles.In (ba_p_ens Bp) x)
           (pfyb:Ensembles.In (ba_p_ens Bp) y)
           (pfzb:Ensembles.In (ba_p_ens Bp) z)
           (pfwb:Ensembles.In (ba_p_ens Bp) w),
           proj1_sig ((exist _ _ pfxa %+ exist _ _ pfya) %* (exist _ _ pfza %+ exist _ _ pfwa)) = 
           proj1_sig ((exist _ _ pfxb %+ exist _ _ pfyb) %* (exist _ _ pfzb %+ exist _ _ pfwb)).
intros; subst.
assert (pfxa = pfxb). apply proof_irrelevance. assert (pfya = pfyb). apply proof_irrelevance.
assert (pfza = pfzb). apply proof_irrelevance.  assert (pfwa = pfwb). apply proof_irrelevance.
subst.
reflexivity.
Qed.



Lemma ba_p_subst_comp : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    forall (x:T) 
           (pfxa:Ensembles.In (ba_p_ens Ap) x)
           (pfxb:Ensembles.In (ba_p_ens Bp) x),
           proj1_sig (%- exist _ _ pfxa) =
           proj1_sig (%- exist _ _ pfxb).
intros; subst.
assert (pfxa = pfxb). apply proof_irrelevance. subst.
reflexivity.
Qed.


Lemma ba_p_subst_one : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    proj1_sig (Bone_p T (Bc_p T Ap)) = proj1_sig (Bone_p T (Bc_p T Bp)).
intros; subst; reflexivity.
Qed.

Lemma ba_p_subst_zero : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T), 
    Ap = Bp ->
    proj1_sig (Bzero_p T (Bc_p T Ap)) = proj1_sig (Bzero_p T (Bc_p T Bp)).
intros; subst; reflexivity.
Qed.

End MoreParametricBAs.


Section TwoBoolBa.



Definition two_bool_bc := Build_Bconst bool (Full_set bool)
                                       orb andb true false negb.

Definition two_bool_ex :
  exists! (B:Bool_Alg),
    Bc B = two_bool_bc.
assert (h1: BS (two_bool_bc) = Full_set bool). reflexivity.
exists (Build_Bool_Alg two_bool_bc h1 orb_assoc andb_assoc 
                       orb_comm andb_comm absorption_orb 
                       absorption_andb 
                       (fun a b c  => andb_orb_distrib_r c a b)
                       (fun a b c  => orb_andb_distrib_r c a b)
                       orb_negb_r andb_negb_r).
red. split.
simpl. reflexivity.
intros B h2.
apply bc_inj.
simpl.
auto.
Qed.


Definition two_bool_ba := proj1_sig (constructive_definite_description _ two_bool_ex).

Lemma two_bool_ba_compat : Bc two_bool_ba = two_bool_bc.
unfold two_bool_ba.
destruct constructive_definite_description. simpl.
assumption.
Qed.

End TwoBoolBa.

Section ExpBoolBa.


Lemma exp_bool_times_ex : 
  forall {T:Type} {A:Ensemble T}
         (F F':Fin_map A (Full_set bool) false),
  exists! (G:Fin_map A (Full_set bool) false),
    forall x:T, G |-> x = F |-> x && F' |-> x.
intros T A F F'.
pose (fun_to_fin_map _ false (fin_map_fin_dom F) 
                     (fun x => F |-> x && F' |-> x)) as G'.

assert (h1:Included (Im A (fun x:T=>F|->x && F'|->x)) (Full_set bool)). 
  red. intros x h1. constructor. 
pose (fin_map_new_ran G' finite_bool h1) as G.
exists G. red. split.
intro x.
unfold G.
rewrite <- fin_map_new_ran_compat.
unfold G'. 
destruct (classic (Ensembles.In A x)).
rewrite fun_to_fin_map_compat; auto.
rewrite fin_map_app_def; auto.
rewrite fin_map_app_def; auto.
intros G'' h2.
apply fin_map_app_inj.
apply functional_extensionality.
unfold G. 
intro x. rewrite <- fin_map_new_ran_compat. unfold G'.
destruct (classic (Ensembles.In A x)).
rewrite fun_to_fin_map_compat; auto.
rewrite fin_map_app_def; auto. rewrite fin_map_app_def; auto.
Qed.

Definition exp_bool_times {T:Type} {A:Ensemble T}
         (F F':Fin_map A (Full_set bool) false) :=
    proj1_sig (constructive_definite_description _ (exp_bool_times_ex F F')).

Lemma exp_bool_times_compat : 
  forall {T:Type} (A:Ensemble T)
         (F F':Fin_map A (Full_set bool) false),
    let G := exp_bool_times F F' in
    forall x:T, G |-> x = F |-> x && F' |-> x.
intros T A F F' G x.
unfold G. unfold exp_bool_times. destruct constructive_definite_description as [G' h2].
simpl.
apply h2.
Qed.


Definition exp_bool_plus_ex : 
  forall {T:Type} {A:Ensemble T}
         (F F':Fin_map A (Full_set bool) false),
  exists! (G:Fin_map A (Full_set bool) false),
    forall x:T, G |-> x = F |-> x || F' |-> x.
intros T A F F'.
pose (fun_to_fin_map _ false (fin_map_fin_dom F) 
                     (fun x => F |-> x || F' |-> x)) as G'.

assert (h1:Included (Im A (fun x:T=>F|->x || F'|->x)) (Full_set bool)). 
  red. intros x h1. constructor. 
pose (fin_map_new_ran G' finite_bool h1) as G.
exists G. red. split.
intro x.
unfold G.
rewrite <- fin_map_new_ran_compat.
unfold G'. 
destruct (classic (Ensembles.In A x)).
rewrite fun_to_fin_map_compat; auto.
rewrite fin_map_app_def; auto.
rewrite fin_map_app_def; auto.
rewrite fin_map_app_def; auto.
intros G'' h2.
apply fin_map_app_inj.
apply functional_extensionality.
unfold G. 
intro x. rewrite <- fin_map_new_ran_compat. unfold G'.
destruct (classic (Ensembles.In A x)).
rewrite fun_to_fin_map_compat; auto.
rewrite fin_map_app_def; auto. rewrite fin_map_app_def; auto.
Qed.
                             
Definition exp_bool_plus {T:Type} {A:Ensemble T}
         (F F':Fin_map A (Full_set bool) false) :=
    proj1_sig (constructive_definite_description _ (exp_bool_plus_ex F F')).

Lemma exp_bool_plus_compat : 
  forall {T:Type} (A:Ensemble T)
         (F F':Fin_map A (Full_set bool) false),
    let G := exp_bool_plus F F' in
    forall x:T, G |-> x = F |-> x || F' |-> x.
intros T A F F' G x.
unfold G. unfold exp_bool_plus. destruct constructive_definite_description as [G' h2].
simpl.
apply h2.
Qed.

Definition exp_bool_comp_ex : 
  forall {T:Type} {A:Ensemble T}
         (F:Fin_map A (Full_set bool) false),
  exists! (G:Fin_map A (Full_set bool) false),
    forall x:T, Ensembles.In A x ->
                G |-> x = negb (F |-> x).
intros T A F.
pose (fun_to_fin_map _ false (fin_map_fin_dom F) 
                     (fun x => negb( F |-> x))) as G'.
assert (h1:Included (Im A (fun x:T=>negb (F|->x))) (Full_set bool)).
  red. intros x h1. constructor. 
pose (fin_map_new_ran G' finite_bool h1) as G.
exists G. red. split.
intro x.
unfold G.
rewrite <- fin_map_new_ran_compat.
unfold G'. 
destruct (classic (Ensembles.In A x)).
rewrite fun_to_fin_map_compat; auto.
intros; contradiction.
intros G'' h2. 
apply fin_map_ext_in.
intros x h3. rewrite h2; auto.
unfold G.
rewrite <- fin_map_new_ran_compat. unfold G'. 
destruct (classic (Ensembles.In A x)).
rewrite fun_to_fin_map_compat; auto.
contradiction.
Qed.

                              
Definition exp_bool_comp {T:Type} {A:Ensemble T}
         (F:Fin_map A (Full_set bool) false) :=
    proj1_sig (constructive_definite_description _ (exp_bool_comp_ex F)).

Lemma exp_bool_comp_compat : 
  forall {T:Type} (A:Ensemble T)
         (F:Fin_map A (Full_set bool) false),
    let G := exp_bool_comp F in
    forall x:T, Ensembles.In A x -> G |-> x = negb (F |-> x).
intros T A F G x.
unfold G. unfold exp_bool_comp. destruct constructive_definite_description as [G' h2].
simpl.
apply h2; auto.
Qed.
 

Definition exp_bool_bc {T:Type} (A:Ensemble T) (pf:Finite A) := 
  Build_Bconst (Fin_map A (Full_set bool) false) 
               (Full_set (Fin_map A (Full_set bool) false))
               exp_bool_plus exp_bool_times
               (fin_map_sing_ran A (Full_set bool) pf finite_bool false true 
                                 (Full_intro _ true))
               (fin_map_sing_ran A (Full_set bool) pf finite_bool false false 
                                 (Full_intro _ false))
               exp_bool_comp.


Lemma exp_bool_assoc_sum : 
  forall {T:Type} {A:Ensemble T} {pf:Finite A}
         (n m p : Btype (exp_bool_bc A pf)), 
    n + (m + p) = n + m + p.
intros T A h1 f g k. simpl.
apply fin_map_ext_in.
intros x h2.   
do 4 rewrite exp_bool_plus_compat.
apply orb_assoc.
Qed.

Lemma exp_bool_assoc_prod : 
  forall {T:Type} {A:Ensemble T} {pf:Finite A}
         (n m p : Btype (exp_bool_bc A pf)), 
    n * (m * p) = n * m * p.
intros T A h1 f g k. simpl.
apply fin_map_ext_in.
intros x h2.   
do 4 rewrite exp_bool_times_compat.
apply andb_assoc.
Qed.

Lemma exp_bool_comm_sum : 
  forall {T:Type} {A:Ensemble T} {pf:Finite A} 
         (n m : Btype (exp_bool_bc A pf)),
    n + m = m + n.
intros T A h1 f g. simpl. apply fin_map_ext_in. intros x h2.
do 2 rewrite exp_bool_plus_compat.
apply orb_comm.
Qed.

Lemma exp_bool_comm_prod : 
  forall {T:Type} {A:Ensemble T} {pf:Finite A} 
         (n m : Btype (exp_bool_bc A pf)),
    n * m = m * n.
intros T A h1 f g. simpl. apply fin_map_ext_in. intros x h2.
do 2 rewrite exp_bool_times_compat.
apply andb_comm.
Qed.


Lemma exp_bool_abs_sum : 
  forall {T:Type} {A:Ensemble T} {pf:Finite A}
         (n m : Btype (exp_bool_bc A pf)), n + n * m = n.
intros T A h1 n m.  apply fin_map_ext_in. intros x h2.
simpl.
rewrite exp_bool_plus_compat. rewrite exp_bool_times_compat.
apply absorption_orb.
Qed.


Lemma exp_bool_abs_prod : 
  forall {T:Type} {A:Ensemble T} {pf:Finite A}
         (n m : Btype (exp_bool_bc A pf)), n * (n + m) = n.
intros T A h1 n m.  apply fin_map_ext_in. intros x h2.
simpl.
rewrite exp_bool_times_compat. rewrite exp_bool_plus_compat.
apply absorption_andb.
Qed.

Lemma exp_bool_dist_sum : 
  forall {T:Type} {A:Ensemble T} {pf:Finite A} 
         (n m p : Btype (exp_bool_bc A pf)), 
    p * (n + m) = p * n + p * m.
intros T A h1 f g k. apply fin_map_ext_in. intros x h2.
simpl. 
rewrite exp_bool_times_compat. do 2 rewrite exp_bool_plus_compat.
do 2 rewrite exp_bool_times_compat.
apply (fun a b c  => andb_orb_distrib_r c a b).
Qed.

Lemma exp_bool_dist_prod : 
  forall {T:Type} {A:Ensemble T} {pf:Finite A} 
         (n m p : Btype (exp_bool_bc A pf)), 
    p + n * m = (p + n) * (p + m).
intros T A h1 f g k. apply fin_map_ext_in. intros x h2.
simpl. 
rewrite exp_bool_plus_compat. do 2 rewrite exp_bool_times_compat.
do 2 rewrite exp_bool_plus_compat.
apply  (fun a b c  => orb_andb_distrib_r c a b).
Qed.


Lemma exp_bool_comp_sum : 
  forall {T:Type} {A:Ensemble T} {pf:Finite A}
         (n : Btype (exp_bool_bc A pf)) , n + - n = 1.
intros T A h1 f. apply fin_map_ext_in. intros x h2.
simpl. 
rewrite fin_map_sing_ran_compat; auto.
rewrite exp_bool_plus_compat; auto. rewrite exp_bool_comp_compat; auto.
apply orb_negb_r.
Qed.

Lemma exp_bool_comp_prod : 
  forall {T:Type} {A:Ensemble T} {pf:Finite A}
         (n : Btype (exp_bool_bc A pf)) , n * - n = 0.
intros T A h1 f. apply fin_map_ext_in. intros x h2.
simpl. 
rewrite fin_map_sing_ran_compat; auto.
rewrite exp_bool_times_compat; auto. rewrite exp_bool_comp_compat; auto.
apply andb_negb_r.
Qed.


Definition exp_bool_ba {T:Type} (A:Ensemble T) (pf:Finite A) :=
  Build_Bool_Alg (exp_bool_bc A pf) (eq_refl _) 
                 exp_bool_assoc_sum exp_bool_assoc_prod
                 exp_bool_comm_sum exp_bool_comm_prod
                 exp_bool_abs_sum exp_bool_abs_prod
                 exp_bool_dist_sum exp_bool_dist_prod
                 exp_bool_comp_sum exp_bool_comp_prod.
                 
           
End ExpBoolBa.

Section FamBa.
(*Before I had employed parametric Bool_Alg records, I couldn't
define [Ensemble Bool_Alg] (univesere inconsistency), 
so I had to dance around a universe inconsistency with a clumsy
definition.  Now that I have defined parametric analogues, 
the below defintion [fam_ba_p] works great, I can easily define
but I decided to leave the old [fam_ba],
just in case for some perverse reason I need to have an ensemble
of Bool_Alg records without a parametric underlying type.
*)

Definition fam_ba_p (T:Type) := (Ensemble (Bool_Alg_p T)).

(*the family of underlying sets in a family of parametric
  Boolean algebras*)
Definition fam_ba_p_family {T:Type} (E:fam_ba_p T) :=
  Im E (fun A=>(A_p T (Bc_p T A))).


(*probably obsolete*)
Inductive fam_ba : Type :=
  ens_ba_intro :forall (A:Ensemble Type),
    (forall T:Type, Ensembles.In A T -> exists B:Bool_Alg, bt B = T) -> fam_ba.

(*probably obsolete*)
Definition fam_ba_types (S:fam_ba) : Ensemble Type.
destruct S. refine A.
Defined.


(*another fumbled syntax required to avoid universe 
  inconsistencies*)
Definition in_fam_ba (S:fam_ba) (B:Bool_Alg) : Prop :=
  exists T:Type, Ensembles.In (fam_ba_types S) T /\
                 T = bt B.

End FamBa.

Section TransferBA.

Definition transfer_ba_elt 
           {A B:Bool_Alg} (pf:A = B) (x:bt A) : bt B.
subst; refine x.
Defined.

Definition transfer_ba_elt_r 
           {A B:Bool_Alg} (pf:A = B) (x:bt B) : bt A.
subst; refine x.
Defined.

Lemma transfer_ba_elt_undoes_transfer_ba_elt_r : 
  forall {A B:Bool_Alg} (pf:A = B) (x:bt B),
    transfer_ba_elt pf (transfer_ba_elt_r pf x) = x.
intros; subst; auto.
Qed.

Lemma transfer_ba_elt_r_undoes_transfer_ba_elt : 
  forall {A B:Bool_Alg} (pf:A = B) (x:bt A),
    transfer_ba_elt_r pf (transfer_ba_elt pf x) = x.
intros; subst; auto.
Qed.

Lemma transfer_ba_elt_inj : 
  forall {A B:Bool_Alg} (pf:A = B) (x y:bt A),
    transfer_ba_elt pf x = transfer_ba_elt pf y ->
    x = y.
intros A B h1 x y h2. subst.
unfold transfer_ba_elt in h2. unfold eq_rect_r in h2. simpl in h2.
assumption.
Qed.

Lemma transfer_ba_elt_r_inj : 
  forall {A B:Bool_Alg} (pf:A = B) (x y:bt B),
    transfer_ba_elt_r pf x = transfer_ba_elt_r pf y ->
    x = y.
intros A B h1 x y h2. subst.
unfold transfer_ba_elt_r in h2. unfold eq_rect_r in h2. simpl in h2.
assumption.
Qed.  



Lemma transfer_dep_ba_p_eq : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T)
         (pf:Ap = Bp)
         (x:btp Ap),
     (@transfer_dep _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf x) =
     exist _ _ (iff1 (in_ba_p_ens_eq _ _ _ pf) (proj2_sig x)).
intros; subst.
rewrite transfer_dep_eq_refl.
destruct x.
apply proj1_sig_injective.
simpl.
reflexivity.
Qed.

Lemma transfer_dep_r_ba_p_eq : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T)
         (pf:Ap = Bp)
         (x:btp Bp),
     (@transfer_dep_r _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf x) =
     exist _ _ (iff2 (in_ba_p_ens_eq _ _ _ pf) (proj2_sig x)).
intros; subst.
rewrite transfer_dep_r_eq_refl.
destruct x.
apply proj1_sig_injective.
simpl.
reflexivity.
Qed.



Lemma transfer_dep_times : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T)
         (pf:Ap = Bp)
         (x y:btp Ap),
    @transfer_dep _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf (x %* y) =
    (@transfer_dep _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf x) %*
    (@transfer_dep _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf y).
intros. subst.
do 3 rewrite transfer_dep_eq_refl.
reflexivity.
Qed.

Lemma transfer_dep_r_times : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T)
         (pf:Ap = Bp)
         (x y:btp Bp),
    @transfer_dep_r _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf (x %* y) =
    (@transfer_dep_r _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf x) %*
    (@transfer_dep_r _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf y).
intros. subst.
do 3 rewrite transfer_dep_r_eq_refl.
reflexivity.
Qed.



Lemma transfer_dep_plus : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T)
         (pf:Ap = Bp)
         (x y:btp Ap),
    @transfer_dep _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf (x %+ y) =
    (@transfer_dep _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf x) %+
    (@transfer_dep _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf y).
intros. subst.
do 3 rewrite transfer_dep_eq_refl.
reflexivity.
Qed.

Lemma transfer_dep_r_plus : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T)
         (pf:Ap = Bp)
         (x y:btp Bp),
    @transfer_dep_r _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf (x %+ y) =
    (@transfer_dep_r _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf x) %+
    (@transfer_dep_r _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf y).
intros. subst.
do 3 rewrite transfer_dep_r_eq_refl.
reflexivity.
Qed.



Lemma transfer_dep_comp : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T)
         (pf:Ap = Bp)
         (x:btp Ap),
    @transfer_dep _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf (%- x) =
    %- (@transfer_dep _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf x).
intros. subst.
do 2 rewrite transfer_dep_eq_refl.
reflexivity.
Qed.


Lemma transfer_dep_r_comp : 
  forall {T:Type} (Ap Bp:Bool_Alg_p T)
         (pf:Ap = Bp)
         (x:btp Bp),
    @transfer_dep_r _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf (%- x) =
    %- (@transfer_dep_r _ (fun B=>{a:T | Ensembles.In (ba_p_ens B) a}) _ _ pf x).
intros. subst.
do 2 rewrite transfer_dep_r_eq_refl.
reflexivity.
Qed.


End TransferBA.                                                  