// Interp.C  -*- C++ -*-
// Copyright (c) 1998 Etienne BERNARD
// Copyright (C) 2002,2005,2008 Clinton Ebadi

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
// 02110-1301, USA.

#include "Interp.H"

#ifdef USESCRIPTS

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libguile.h>
extern "C" {
#include <libguile/hooks.h>
}

#include "Bot.H"
#include "BotInterp.H"
#include "Commands.H"
#include "Macros.H"
#include "ScriptCommands.H"
#include "ShitEntry.H"
#include "User.H"

// static class member initial definitions
Bot * Interp::bot = 0;
SCM Interp::bot_module = 0;

#ifdef MULTITHREAD
pthread_mutex_t Interp::mutex = PTHREAD_MUTEX_INITIALIZER;
#endif

typedef scm_t_subr SCMFunc;

SCM
Interp::ScmApplyWrapper (void *data)
{
#ifdef MULTITHREAD
  //  pthread_mutex_lock (&Interp::mutex);
#endif

  wrapper_data * wd = static_cast<wrapper_data *> (data);
  scm_apply(wd->func, wd->args, SCM_EOL);

#ifdef MULTITHREAD
  //  pthread_mutex_unlock (&Interp::mutex);
#endif

  return SCM_BOOL_T;
}

SCM
Interp::LazyHandler (void *data, SCM tag, SCM throw_args)
{
  SCM log_port = Interp::bot->botInterp->ScriptLog();
  SCM eport = scm_set_current_error_port(log_port);

  scm_handle_by_message_noexit((void *)"bobot++", tag, throw_args);
  scm_force_output(log_port);
  scm_set_current_error_port(eport);
  scm_ithrow(tag, throw_args, 1);
  return SCM_UNSPECIFIED; /* never returns */
}

SCM
Interp::EmptyHandler(void *data, SCM tag, SCM args)
{
  return SCM_UNSPECIFIED;
}


SCM
Interp::LazyApplyWrapper(void *data)
{
  return scm_c_with_throw_handler(SCM_BOOL_T,
				  (scm_t_catch_body) Interp::ScmApplyWrapper,
				  data,
				  (scm_t_catch_handler) Interp::LazyHandler,
				  0,
				  0);
}


static SCM
lazy_eval_file(char *filename)
{
  return scm_c_with_throw_handler(SCM_BOOL_T,
				  (scm_t_catch_body) scm_c_primitive_load_path,
				  filename,
				  (scm_t_catch_handler) Interp::LazyHandler,
				  0,
				  0);
}

static SCM
lazy_eval_string(char *str)
{
  return scm_c_with_throw_handler(SCM_BOOL_T,
				  (scm_t_catch_body) scm_c_eval_string,
				  str,
				  (scm_t_catch_handler) Interp::LazyHandler,
				  0,
				  0);
}



#define bot_new_procedure(a, b, c, d, e) scm_c_define_gsubr (a, c, d, e, b); scm_c_export (a, 0)
#define scm_c_define_gsubr(a, b, c, d, e) scm_c_define_gsubr (a, b, c, d, e); scm_c_export (a, 0)
#define scm_c_define(a, b) scm_c_define (a, b); scm_c_export (a, 0)

void
interp_init_helper (void* unused)
{
  scm_c_use_module ("guile-user");
  // Hooks
  scm_c_define ("bot:exit-hook", scm_make_hook (scm_from_int (0)));

  // Symbols
  // bot:user-*
  scm_c_define ("bot:user-none", scm_from_int (User::NONE));
  scm_c_define ("bot:user-user", scm_from_int (User::USER));
  scm_c_define ("bot:user-trusted", scm_from_int (User::TRUSTED_USER));
  scm_c_define ("bot:user-friend", scm_from_int (User::FRIEND));
  scm_c_define ("bot:user-master", scm_from_int (User::MASTER));

  // user channel modes
  scm_c_define ("bot:mode/op", scm_from_int (User::OP_MODE));
  scm_c_define ("bot:mode/voice", scm_from_int (User::VOICE_MODE));
  scm_c_define ("bot:mode/away", scm_from_int (User::AWAY_MODE));
  scm_c_define ("bot:mode/ircop", scm_from_int (User::IRCOP_MODE));

  // protection
  scm_c_define ("bot:protection/none", scm_from_int (User::NO_PROT));
  scm_c_define ("bot:protection/no-ban", scm_from_int (User::NO_BAN));
  scm_c_define ("bot:protection/no-kick", scm_from_int (User::NO_KICK));
  scm_c_define ("bot:protection/no-deop", scm_from_int (User::NO_DEOP));

  // auto-op
  scm_c_define ("bot:aop/yes", scm_from_int (true));
  scm_c_define ("bot:aop/no", scm_from_int (false));

  // shit-list
  scm_c_define ("bot:shit/none", scm_from_int (ShitEntry::SHIT_NOSHIT));
  scm_c_define ("bot:shit/no-op", scm_from_int (ShitEntry::SHIT_NOOP));
  scm_c_define ("bot:shit/no-join", scm_from_int (ShitEntry::SHIT_NOJOIN));
  scm_c_define ("bot:shit/no-deban", 
		scm_from_int (ShitEntry::SHIT_NODEBAN));
  
  // sys-dir
  scm_c_define ("bot:sys-scripts-dir", 
		scm_from_locale_string ((String(PREFIX) + 
					 "/share/bobotpp/scripts/").c_str ()));
  // Hooks
  scm_c_define ("hooks/action", scm_from_int(Hook::ACTION));
  scm_c_define ("hooks/nickname", scm_from_int(Hook::NICKNAME));
  scm_c_define ("hooks/signoff", scm_from_int(Hook::SIGNOFF));
  scm_c_define ("hooks/ctcp", scm_from_int(Hook::CTCP));
  scm_c_define ("hooks/ctcp-reply", scm_from_int(Hook::CTCP_REPLY));
  scm_c_define ("hooks/disconnect", scm_from_int(Hook::DISCONNECT));
  scm_c_define ("hooks/flood", scm_from_int(Hook::FLOOD));
  scm_c_define ("hooks/invite", scm_from_int(Hook::INVITE));
  scm_c_define ("hooks/join", scm_from_int(Hook::JOIN));
  scm_c_define ("hooks/kick", scm_from_int(Hook::KICK));
  scm_c_define ("hooks/mode", scm_from_int(Hook::MODE));
  scm_c_define ("hooks/message", scm_from_int(Hook::MESSAGE));
  scm_c_define ("hooks/notice", scm_from_int(Hook::NOTICE));
  scm_c_define ("hooks/part", scm_from_int(Hook::PART));
  scm_c_define ("hooks/public", scm_from_int(Hook::PUBLIC));
  scm_c_define ("hooks/public-notice", scm_from_int(Hook::PUBLIC_NOTICE));
  scm_c_define ("hooks/raw", scm_from_int(Hook::RAW));
  scm_c_define ("hooks/timer", scm_from_int(Hook::TIMER));
  scm_c_define ("hooks/topic", scm_from_int(Hook::TOPIC));
  // send hooks
  scm_c_define ("hooks/send/public", scm_from_int (Hook::SEND_PUBLIC));
  scm_c_define ("hooks/send/message", scm_from_int (Hook::SEND_MESSAGE));
  scm_c_define ("hooks/send/action", scm_from_int (Hook::SEND_ACTION));
  scm_c_define ("hooks/send/ctcp", scm_from_int (Hook::SEND_CTCP));
  scm_c_define ("hooks/send/who", scm_from_int (Hook::SEND_WHO));
  scm_c_define ("hooks/send/whois", scm_from_int (Hook::SEND_WHOIS));
  // dcc hooks
  scm_c_define ("hooks/dcc/chat-begin", 
		scm_from_int (Hook::DCC_CHAT_BEGIN));
  scm_c_define ("hooks/dcc/chat-end",
		scm_from_int (Hook::DCC_CHAT_END));
  scm_c_define ("hooks/dcc/chat-message", 
		scm_from_int (Hook::DCC_CHAT_MESSAGE));

  // procedures
  bot_new_procedure ("bot:action", (SCMFunc)ScriptCommands::Action, 2, 0, 0);
  scm_c_define_gsubr ("bot:adduser", 5, 2, 0,
		      (SCMFunc)ScriptCommands::AddUser);
  bot_new_procedure ("bot:addserver", (SCMFunc)ScriptCommands::Action, 
		     3, 4, 0);
  scm_c_define_gsubr ("bot:addshit", 3, 2, 0, 
		      (SCMFunc)ScriptCommands::AddShit);
  bot_new_procedure ("bot:ban", (SCMFunc)ScriptCommands::Action, 2, 0, 0);
  bot_new_procedure ("bot:change-command-level", 
		     (SCMFunc)ScriptCommands::ChangeCommandLevel,
		     2, 0, 0);
  scm_c_define_gsubr ("bot:channel-users", 1, 0, 0,
		      (SCMFunc)ScriptCommands::ChannelUsers);
  scm_c_define_gsubr ("bot:ctcp", 3, 0, 0,
		      (SCMFunc)ScriptCommands::CTCP);
  scm_c_define_gsubr ("bot:ctcp-reply", 3, 0, 0,
		      (SCMFunc)ScriptCommands::CTCPReply);
  bot_new_procedure ("bot:cycle", (SCMFunc)ScriptCommands::Action, 1, 0, 0);
  bot_new_procedure ("bot:deban", (SCMFunc)ScriptCommands::Deban, 2, 0, 0);
  bot_new_procedure ("bot:delserver", (SCMFunc)ScriptCommands::DelServer, 
		     1, 0, 0);
  bot_new_procedure ("bot:deluser", (SCMFunc)ScriptCommands::DelUser, 2, 0, 0);
  bot_new_procedure ("bot:delshit", (SCMFunc)ScriptCommands::DelShit, 2, 0, 0);
  bot_new_procedure ("bot:deop", (SCMFunc)ScriptCommands::Deop, 2, 0, 0);
  bot_new_procedure ("bot:die", (SCMFunc)ScriptCommands::Die, 1, 0, 0);
  bot_new_procedure ("bot:do", (SCMFunc)ScriptCommands::Do, 1, 0, 0);
  bot_new_procedure ("bot:invite", (SCMFunc)ScriptCommands::Invite, 2, 0, 0);
  bot_new_procedure ("bot:join", (SCMFunc)ScriptCommands::Join, 1, 1, 0);
  bot_new_procedure ("bot:keep", (SCMFunc)ScriptCommands::Keep, 2, 0, 0);
  bot_new_procedure ("bot:kick", (SCMFunc)ScriptCommands::Kick, 2, 1, 0);
  bot_new_procedure ("bot:kickban", (SCMFunc)ScriptCommands::KickBan, 2, 1, 0);
  bot_new_procedure ("bot:lock", (SCMFunc)ScriptCommands::Lock, 1, 0, 0);
  bot_new_procedure ("bot:logport", (SCMFunc)ScriptCommands::LogPort, 0, 0, 0);
  bot_new_procedure ("bot:mode", (SCMFunc)ScriptCommands::Mode, 2, 0, 0);
  bot_new_procedure ("bot:msg", (SCMFunc)ScriptCommands::Msg, 2, 0, 0);
  bot_new_procedure ("bot:nextserver", (SCMFunc)ScriptCommands::NextServer, 
		     0, 0, 0);
  bot_new_procedure ("bot:nick", (SCMFunc)ScriptCommands::Nick, 1, 0, 0);

  scm_c_define_gsubr ("bot:notice", 2, 0, 0, 
		      (SCMFunc)ScriptCommands::Notice);

  bot_new_procedure ("bot:op", (SCMFunc)ScriptCommands::Op, 2, 0, 0);
  bot_new_procedure ("bot:part", (SCMFunc)ScriptCommands::Part, 1, 0, 0);
  bot_new_procedure ("bot:reconnect", (SCMFunc)ScriptCommands::Reconnect, 
		     0, 0, 0);
  bot_new_procedure ("bot:say", (SCMFunc)ScriptCommands::Say, 2, 0, 0);
  bot_new_procedure ("bot:server", (SCMFunc)ScriptCommands::Server, 1, 0, 0);
  scm_c_define_gsubr ("bot:setfloodrate", 1, 0, 0,
		      (SCMFunc)ScriptCommands::SetFloodRate);
  bot_new_procedure ("bot:setversion", (SCMFunc)ScriptCommands::SetVersion, 
		     1, 0, 0);
  bot_new_procedure ("bot:tban", (SCMFunc)ScriptCommands::TBan, 3, 0, 0);
  bot_new_procedure ("bot:tkban", (SCMFunc)ScriptCommands::TKBan, 3, 1, 0);
  bot_new_procedure ("bot:topic", (SCMFunc)ScriptCommands::Topic, 2, 0, 0);
  bot_new_procedure ("bot:unlock", (SCMFunc)ScriptCommands::Unlock, 1, 0, 0);
  scm_c_define_gsubr ("bot:who", 1, 0, 0, (SCMFunc)ScriptCommands::Who);
  scm_c_define_gsubr ("bot:whois", 1, 0, 0, (SCMFunc)ScriptCommands::Whois);

  bot_new_procedure ("bot:getnickname", (SCMFunc)ScriptCommands::getNickname, 
		     0, 0, 0);
  bot_new_procedure ("bot:getserver", (SCMFunc)ScriptCommands::getServer, 
		     0, 0, 0);
  bot_new_procedure ("bot:getserverlist", 
		     (SCMFunc)ScriptCommands::getServerList, 0, 0, 0);
  bot_new_procedure ("bot:flush", (SCMFunc)ScriptCommands::flushQueue, 
		     0, 0, 0);
  bot_new_procedure ("bot:flushport", (SCMFunc)ScriptCommands::flushPort, 
		     0, 0, 0);
  bot_new_procedure ("bot:random", (SCMFunc)ScriptCommands::random, 
		     1, 0, 0);
  bot_new_procedure ("bot:addcommand", (SCMFunc)ScriptCommands::addCommand, 
		     5, 0, 0);
  bot_new_procedure ("bot:delcommand", (SCMFunc)ScriptCommands::delCommand, 
		     1, 0, 0);
  bot_new_procedure ("bot:addhook", (SCMFunc)ScriptCommands::AddHook, 
		     3, 3, 0);
  bot_new_procedure ("bot:addtimer", (SCMFunc)ScriptCommands::AddTimer, 
		     2, 0, 0);
  bot_new_procedure ("bot:deltimer", (SCMFunc)ScriptCommands::DelTimer, 
		     1, 0, 0);

  scm_c_define_gsubr ("bot:dcc-chat-send", 2, 0, 0,
		      (SCMFunc)ScriptCommands::sendDCCChatMessage);
}

#undef bot_new_procedure
#undef scm_c_define_gsubr
#undef scm_c_define


SCM
interp_post_startup_helper (void *bot_module)
{
  SCM module = static_cast<SCM> (bot_module);
  scm_c_define ("the-bot-module", module);
  scm_c_export ("the-bot-module", 0);

  // load bobot-utils
  return scm_primitive_load_path
    (scm_from_locale_string ((String(PREFIX) + 
			      "/share/bobotpp/scripts/bobot-utils.scm").c_str ()));
}

void
Interp::Startup()
{
  bot_module = scm_c_define_module ("the-bot-module", 
				    interp_init_helper, 0);

}

void Interp::Startup2 (Bot *b)
{
  bot = b;

  scm_c_call_with_current_module (bot_module,
				  interp_post_startup_helper,
				  bot_module);
}

void
Interp::Shutdown()
{ 
  scm_c_run_hook (scm_variable_ref (scm_c_module_lookup (bot_module, 
							 "bot:exit-hook")), 
		  SCM_EOL);
}

void
Interp::Execute(Bot *b, String command)
{
#ifdef MULTITHREAD
  // We get the lock
  pthread_mutex_lock(&mutex);
#endif
  
  bot = b;
  scm_c_with_throw_handler(SCM_BOOL_T,
			   (scm_t_catch_body) lazy_eval_string,
			   (void *) static_cast<const char *> (command.c_str ()),
			   (scm_t_catch_handler) Interp::EmptyHandler,
			   0,
			   0);
  
#ifdef MULTITHREAD
  // We release the lock
  pthread_mutex_unlock(&mutex);
#endif
}


void
Interp::LoadScript(Bot *b, String filename)
{
#ifdef MULTITHREAD
  // We get the lock
  pthread_mutex_lock(&mutex);
#endif
  bot = b;
  scm_c_with_throw_handler(SCM_BOOL_T,
			   (scm_t_catch_body) lazy_eval_file,
			   (void *)static_cast<const char * >(filename.c_str ()),
			   (scm_t_catch_handler) Interp::EmptyHandler,
			   0,
			   0);
#ifdef MULTITHREAD
  // We release the lock
  pthread_mutex_unlock(&mutex);
#endif
}

#endif
