/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <sstream>
#include <iomanip>
#include <algorithm>
#include <iterator>
#include "variablesTable.h"

namespace bmEval
{
  namespace 
  {
    const int highestPrecision = 12;
    const int defaultWidth = 26;
  }

  std::string variablesTable::fmtVar (const std::pair<std::string, storedValue>& variable)
  {
    std::ostringstream oss;
  
    oss << std::setw(defaultWidth) << variable.first << " = " 
	<< std::setprecision(highestPrecision) 
	<< std::showpos << variable.second.getValue();
    if ( (variable.second.isReadOnly()) )
      {
	oss << " (const)";
      }
    return oss.str();
  }

  std::vector<variablesTable::entry>
  variablesTable::getVariables (const std::string& key, 
				match::type mType) const
  {
    entry ee;
    std::vector<entry> vv;
    
    if ( key.empty() == true)
      {
	for (std::map<std::string, storedValue >::const_iterator it = m_table.begin(); it != m_table.end(); ++it)
	  {
	    ee.m_id = it->first;
	    ee.m_value = it->second.getValue();
	    ee.m_isReadOnly = it->second.isReadOnly();
	    vv.push_back (ee);
	  }
      }
    else
      {
	match varMatch(key, mType);
	std::map<std::string, storedValue>::const_iterator startIt = m_table.begin();
	std::map<std::string, storedValue>::const_iterator endMark = m_table.end();
	std::map<std::string, storedValue>::const_iterator it;
      
	while ( startIt != endMark )
	  {
	    it = find_if (startIt, endMark, varMatch);
	    if ( it != endMark)
	      {
		ee.m_id = it->first;
		ee.m_value = it->second.getValue();
		ee.m_isReadOnly = it->second.isReadOnly();
		vv.push_back (ee);
		startIt = ++it;
	      }
	    else
	      {
		startIt = endMark;
	      }
	  } // end while START != ENDMARK
      } // end of KEY not empty
    return vv;
  }
  
  void variablesTable::listVariables (std::ostream& os, 
				      const std::string& key, 
				      match::type mType) const
  {
    if ( key.empty() == true)
      {
	std::ostream_iterator<std::string> osIt (os, "\n");
      
	transform (m_table.begin(), m_table.end(), osIt, fmtVar);
      }
    else
      {
	match varMatch(key, mType);
	std::map<std::string, storedValue>::const_iterator startIt = m_table.begin();
	std::map<std::string, storedValue>::const_iterator endMark = m_table.end();
	std::map<std::string, storedValue>::const_iterator it;
      
	while ( startIt != endMark )
	  {
	    it = find_if (startIt, endMark, varMatch);
	    if ( it != endMark)
	      {
		os << fmtVar(*it) << std::endl;
		startIt = ++it;
	      }
	    else
	      {
		startIt = endMark;
	      }
	  } // end while START != ENDMARK
      } // end of KEY not empty
  }

  size_t variablesTable::eraseVariables (const std::string& key, 
					 match::type mType)
  {
    size_t rv = 0;
    
    if ( key.empty() == true)
      {
	rv = m_table.size();
	m_table.clear();
      }
    else
      {
	match varMatch(key, mType);
	std::map<std::string, storedValue>::iterator startIt = m_table.begin();
	std::map<std::string, storedValue>::iterator endMark = m_table.end();
	std::map<std::string, storedValue>::iterator it;
      
	while ( startIt != endMark )
	  {
	    it = find_if (startIt, endMark, varMatch);
	    if ( it != endMark)
	      {
		++(startIt = it);
		m_table.erase (it);
		rv++;
	      }
	    else
	      {
		startIt = endMark;
	      }
	  } // end while START != ENDMARK
      } // end of KEY not empty
    return rv;
  }
} // end of namespace bmEval
