/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "translator.h"
#include "evalError.h"

/*

  Priorities                                                 Grouping

  HIGHEST
  |
  |
  9|      ~  (unary operator)
  8|      ^                                                  Right to left
  7|      !  +  -  (unary operators)                         Right to left
  6|      <>  ><                                             Left to right
  5|      *  /  %  //  %%                                    Left to right
  4|      +  -                                               Left to right
  3|      <  >  <=  >=  ==  !=                               Left to right
  2|      &&  ||  ^^                                         Left to right
  1|      :=  =  +=  -=  *=  /=  %=  //=  %%=  ^=  <>=  ><=  Right to left
  |
  V
  LOWEST

*/

namespace bmEval
{
  std::map<mathToken::tokenType, translator::grouping> translator::createGroupingTable ()
  {
    std::map <mathToken::tokenType, translator::grouping> table;

    table[mathToken::OP_POS] = RIGHT_TO_LEFT;
    table[mathToken::OP_NEG] = RIGHT_TO_LEFT;
    table[mathToken::OP_NOT] = RIGHT_TO_LEFT;
    table[mathToken::OP_DEL] = RIGHT_TO_LEFT;
    table[mathToken::OP_AND] = LEFT_TO_RIGHT;
    table[mathToken::OP_OR]  = LEFT_TO_RIGHT;
    table[mathToken::OP_XOR] = LEFT_TO_RIGHT;
    table[mathToken::OP_LT]  = LEFT_TO_RIGHT;
    table[mathToken::OP_GT]  = LEFT_TO_RIGHT;
    table[mathToken::OP_LE]  = LEFT_TO_RIGHT;
    table[mathToken::OP_GE]  = LEFT_TO_RIGHT;
    table[mathToken::OP_EQ]  = LEFT_TO_RIGHT;
    table[mathToken::OP_NE]  = LEFT_TO_RIGHT;
    table[mathToken::OP_ADD] = LEFT_TO_RIGHT;
    table[mathToken::OP_SUB] = LEFT_TO_RIGHT;
    table[mathToken::OP_MUL] = LEFT_TO_RIGHT;
    table[mathToken::OP_DIV] = LEFT_TO_RIGHT;
    table[mathToken::OP_PERCENT] = LEFT_TO_RIGHT;
    table[mathToken::OP_MOD]  = LEFT_TO_RIGHT;
    table[mathToken::OP_IDIV] = LEFT_TO_RIGHT;
    table[mathToken::OP_POW]  = RIGHT_TO_LEFT;
    table[mathToken::OP_MAX]  = LEFT_TO_RIGHT;
    table[mathToken::OP_MIN]  = LEFT_TO_RIGHT;
    table[mathToken::OP_DEF]  = RIGHT_TO_LEFT;
    table[mathToken::OP_STO]  = RIGHT_TO_LEFT;
    table[mathToken::OP_ADD_STO] = RIGHT_TO_LEFT;
    table[mathToken::OP_SUB_STO] = RIGHT_TO_LEFT;
    table[mathToken::OP_MUL_STO] = RIGHT_TO_LEFT;
    table[mathToken::OP_DIV_STO] = RIGHT_TO_LEFT;
    table[mathToken::OP_PERCENT_STO] = RIGHT_TO_LEFT;
    table[mathToken::OP_MOD_STO] = RIGHT_TO_LEFT;
    table[mathToken::OP_IDIV_STO] = RIGHT_TO_LEFT;
    table[mathToken::OP_POW_STO] = RIGHT_TO_LEFT;
    table[mathToken::OP_MAX_STO] = RIGHT_TO_LEFT;
    table[mathToken::OP_MIN_STO] = RIGHT_TO_LEFT;
    return table;
  }

  std::map<mathToken::tokenType, translator::grouping> translator::sm_groupingTable = translator::createGroupingTable ();

  std::map<mathToken::tokenType, int> translator::createPriorityTable ()
  {
    std::map <mathToken::tokenType, int> table;

    table[mathToken::OP_POS] = 7;
    table[mathToken::OP_NEG] = 7;
    table[mathToken::OP_NOT] = 7;
    table[mathToken::OP_DEL] = 9;
    table[mathToken::OP_AND] = 2;
    table[mathToken::OP_OR]  = 2;
    table[mathToken::OP_XOR] = 2;
    table[mathToken::OP_LT]  = 3;
    table[mathToken::OP_GT]  = 3;
    table[mathToken::OP_LE]  = 3;
    table[mathToken::OP_GE]  = 3;
    table[mathToken::OP_EQ]  = 3;
    table[mathToken::OP_NE]  = 3;
    table[mathToken::OP_ADD] = 4;
    table[mathToken::OP_SUB] = 4;
    table[mathToken::OP_MUL] = 5;
    table[mathToken::OP_DIV] = 5;
    table[mathToken::OP_PERCENT] = 5;
    table[mathToken::OP_MOD]  = 5;
    table[mathToken::OP_IDIV] = 5;
    table[mathToken::OP_POW]  = 8;
    table[mathToken::OP_MAX]  = 6;
    table[mathToken::OP_MIN]  = 6;
    table[mathToken::OP_DEF]  = 1;
    table[mathToken::OP_STO]  = 1;
    table[mathToken::OP_ADD_STO] = 1;
    table[mathToken::OP_SUB_STO] = 1;
    table[mathToken::OP_MUL_STO] = 1;
    table[mathToken::OP_DIV_STO] = 1;
    table[mathToken::OP_PERCENT_STO] = 1;
    table[mathToken::OP_MOD_STO] = 1;
    table[mathToken::OP_IDIV_STO] = 1;
    table[mathToken::OP_POW_STO] = 1;
    table[mathToken::OP_MAX_STO] = 1;
    table[mathToken::OP_MIN_STO] = 1;
    return table;
  }

  std::map<mathToken::tokenType, int> translator::sm_priorityTable = translator::createPriorityTable();

  // This function implements the shunting-yard algorithm

  std::queue<mathToken> translator::translate (const std::vector<mathToken>& tokenSequence)
  {
    std::queue<mathToken> outputQueue;
    mathToken currentToken; // initially, the type of this is mathToken::UNDEFINED

    for (size_t i = 0; 
	 i < tokenSequence.size() && 
	   currentToken.Type() != mathToken::END_TOKEN;
	 i++)
      {
	// read the current token
	currentToken = tokenSequence[i];

	if (currentToken.Type() == mathToken::NUMBER ||
	    currentToken.Type() == mathToken::VARIABLE ||
	    currentToken.Type() == mathToken::INVALID_TOKEN)
	  {
	    // If the token is a number, a variable, or an invalid
	    // one, then add it to the output queue
	    outputQueue.push (currentToken);
	  }
	else if (currentToken.Type() == mathToken::OPEN_PARENTHESIS || 
		 currentToken.isFunction() == true)
	  {
	    // If the token is a function or open parenthesis
	    // then push it onto the operator stack
	    m_operatorStack.push(currentToken);
	  }
	else if (currentToken.isOperator() == true)
	  {
	    // In case the current token is an operator
	    // things get a bit complicate
	    bool outputTopOperator = true;

	    do
	      {
		if ( (m_operatorStack.empty()) )
		  {
		    // If the operator stack is empty, then
		    // exit from the cycle
		    outputTopOperator = false;
		  }
		else
		  {
		    mathToken topOfStack(m_operatorStack.top());

		    // As long as there is an operator at the top
		    // of the stack, and...
		    if ( (topOfStack.isOperator()) )
		      {
			int priority1 = getPriorityOf (currentToken);
			int priority2 = getPriorityOf (topOfStack);

			// either the current token is left-associative and its
			// priority is less or equal to that of the top operator
			// of the stack, or the current token is right associative
			// and its precedence is less than that one of the top of the stack 
			outputTopOperator = ( getGroupingOf(currentToken) == LEFT_TO_RIGHT ?
					      (priority1 <= priority2) : (priority1 < priority2) ); 
			if ( outputTopOperator == true )
			  {
			    // then pop the operator on top of the stack
			    // and add it to the output queue
			    m_operatorStack.pop();
			    outputQueue.push (topOfStack);
			  }
		      }
		    else
		      {
			// If the token at the top of the stack is not
			// an operator, then exit from the cycle
			outputTopOperator = false;
		      }
		  } // end of: the stack is not empty
	      } while ( outputTopOperator == true );
	    // push the current token onto the operator stack
	    m_operatorStack.push(currentToken);
	  } // end of: the token is an operator
	else if (currentToken.Type() == mathToken::CLOSED_PARENTHESIS)
	  {
	    // If the token is a closed parenthesis, then
	    while (!m_operatorStack.empty() && 
		   m_operatorStack.top().Type() != mathToken::OPEN_PARENTHESIS)
	      {
		// pop tokens from the stack and push them onto
		// the output queue until the token at the top of 
		// the stack is a left parenthesis
		outputQueue.push (m_operatorStack.top());
		m_operatorStack.pop();
	      }
	    if ( (m_operatorStack.empty()) )
	      {
		// If the stack has run out without fiding a left
		// parenthesis ==> Error: Mismatched parentheses
		// The operator stack is empty, just throw an error.
		throw evalError ("", "Found parenthesis mismatch",
				 "Remove or add a parenthesis where needed", 
				 currentToken.startPosition());
	      }
	    else
	      {
		// otherwise, the top of the stack is an open parenthesis ==> remove it
		m_operatorStack.pop();
		// If a function token follows, remove it from the
		// stack after putting it onto the queue
		if ( !m_operatorStack.empty() && m_operatorStack.top().isFunction() == true )
		  {
		    outputQueue.push (m_operatorStack.top());
		    m_operatorStack.pop();
		  }
	      } 
	  } // end of: the token is a closed parenthesis
      } // end of: while there are tokens to be read

    // If no error has been detected, then the algorithm
    // must be completed. While there are still
    // tokens in the operator stack,
    while (!m_operatorStack.empty())
      {
	// check if the top token is a parenthesis or not.
	if (m_operatorStack.top().Type() == mathToken::OPEN_PARENTHESIS)
	  {
	    // If it is ==> Error: Mismatched parentheses
	    size_t mismatchPosition = m_operatorStack.top().startPosition();
	  
	    // Clear the operator stack if needed
	    purgeOperatorStack();
	    // and then throw an error.
	    throw evalError ("", "Found parenthesis mismatch",
			     "Remove or add a parenthesis where needed", 
			     mismatchPosition);
	  }
	else
	  {
	    // otherwise, pop the top token and push it
	    // onto the output queue.
	    outputQueue.push (m_operatorStack.top());
	    m_operatorStack.pop();	      
	  }
      }
    // return the output queue to the caller
    return outputQueue;
  }
} // end of namespace bmEval
