/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <cstdlib>
#include <string>
#include <algorithm>
#include "evaluator.h"
#include "Utils.h"
#ifdef _USE_READLINE_
#include <readline/readline.h>
#include <readline/history.h>
#endif // _USE_READLINE_

#ifndef VERSION
#define VERSION "x.x (unknown version)"
#endif

#define CMD_PREFIX ':'
#define TYPESCRIPT_NAME "/.ssc_typescript"

using namespace std;
using namespace bmEval;

void printHelp (const char* progName)
{
  cout << progName
       << " - evaluate expressions and print their results to standard output\n" <<endl;
  
  cout << "Usage:  " << progName
       << " [-h|--help] [-v|--version] [-a|--append] [EXPR]...\n" << endl;

  cout << "If any mathematical expression EXPR is provided, evaluate it,\n"
       << "print its result to standard output, and terminate after evaluating\n"
       << "the last EXPRession.\n"
       << "Also, print to standard error any error occurred during the evaluation\n"
       << "of any expression. Quit in case of a parsing error.\n"
       << "If no mathematical EXPRession is provided, start in interactive mode:\n"
       << "wait for the user typing an expression followed by <Enter>,\n"
       << "then evaluate the expression and display its result, or alternatively\n"
       << "the errors occurred during its evaluation.\n" << endl;

  cout << "The user may enter a command instead of an expression.\n"
       << "In such a case the command should be executed and its\n"
       << "outcome reported to the user.\n"
       << "The special command " << CMD_PREFIX << "help\n"
       << "should print a list of all available commands.\n" << endl;
  
  cout << "Every working session should be recorded to the file" << endl;
  cout << "$HOME" << TYPESCRIPT_NAME << endl;
  cout << "where $HOME stays for the home directory of the user.\n" << endl;
  
  cout << "Options:\n" << endl;
  cout << "-h, --help       Print this help" << endl;
  cout << "-v, --version    Show version number, Copyright, Distribution Terms" << endl;
  cout << "                 and NO-Warranty" << endl;
  cout << "-a, --append     Append the input/output of the new working session" << endl;
  cout << "                 at the end of the file $HOME" << TYPESCRIPT_NAME << endl;
  cout << "                 instead of overwriting its current contents\n" << endl;
  
#ifdef _USE_READLINE_
  cout << "This version of " << progName << " has been built against\n"
       << "the GNU Readline library, thus GNU Readline shortcuts and history\n"
       << "support are available when editing the expressions in interactive mode.\n"
       << "For more information about GNU Readline capabilities and shortcuts\n"
       << "see  http://cnswww.cns.cwru.edu/php/chet/readline/readline.html\n" << endl;
#endif
}

void printInfo (const char* progName)
{
  cout << progName << ' ' << VERSION << endl;
  cout << "Copyright (C) 2015 Ivano Primi <ivprimi@libero.it>\n" 
       << "License GPLv3+: GNU GPL version 3 or later,\n"
       << "see <http://gnu.org/licenses/gpl.html>.\n"
       << "This is free software: you are free to change and redistribute it.\n"
       << "There is NO WARRANTY, to the extent permitted by law.\n" << endl;
}

inline int handleOptions (int argc, char* argv[], bool& appendMode)
{
  string argument;
  bool helpRequested = false;
  bool infoRequested = false;
  int ii;
  
  for (ii = 1; ii < argc; ii++)
    {
      argument = string(argv[ii]);
      if (argument.compare("--") == 0)
	{
	  ii++;
	  break;
	}
      else if (argument.compare("-h") == 0 ||
	       argument.compare("--help") == 0)
	{
	  helpRequested = true;
	}
      else if (argument.compare("-v") == 0 ||
	       argument.compare("--version") == 0)
	{
	  infoRequested = true;
	}
      else if (argument.compare("-a") == 0 ||
	       argument.compare("--append") == 0)
	{
	  appendMode = true;
	}
      else
	{
	  break;
	}
    }
  if ((infoRequested))
    printInfo (argv[0]);	    
  if ((helpRequested))
    printHelp (argv[0]);
  return ii;
}

#ifdef _USE_READLINE_

inline void setHistorySize (int maxNEntries = 0)
{
  if (maxNEntries > 0)
    stifle_history(maxNEntries);
  else
    unstifle_history();
}

inline int initReadline (const char* progName,
			 const char* historyFile,
			 int historySize)
{
  int errorCode = 0;
  
  rl_readline_name = progName;
  rl_instream = (FILE*)(0); // standard input
  rl_bind_key ('\t', rl_insert);
  using_history();
  if ( !(errorCode = read_history (historyFile)) ) {
      setHistorySize (historySize);
    }
  return errorCode;
}

inline int endReadline (const char* historyFile)
{
  int errorCode = write_history (historyFile);
  clear_history();
  return errorCode;
}

#endif // _USE_READLINE_ 

inline void readLineInString (const char* prompt,
			      string& str,
			      ostream* ptypescript,
			      bool& wasLineRead)
{
#ifdef _USE_READLINE_
  char *line = readline (prompt);
  
  if (!line)
    {
      wasLineRead = false;
    }
  else
    {
      str = line;
      wasLineRead = true;
      if ((ptypescript)) {
	*ptypescript << prompt << str << endl;
      }
      add_history (line);
      free ((void*)line);
    }
#else
  if ((prompt) && *prompt != '\0')
    {
      cout << prompt;
      if ((ptypescript)) {
	*ptypescript << prompt;
      }
    }
  getline (cin, str);
  wasLineRead = (cin.eof() == false && 
		 cin.fail() == false && 
		 cin.bad() == false);
  if ((ptypescript) && (wasLineRead)) {
    *ptypescript << str << endl;
  }
#endif // _USE_READLINE_
}

void display (ostream* ptypescript, cValue result)
{
  if ( result.real() != 0)
  {
    cout << result.real();
    if ((ptypescript)) {
      *ptypescript << result.real();
    }
    if (result.imag() != 0)
    {
      cout << showpos << result.imag() << 'i' << noshowpos;
      if ((ptypescript)) {
	*ptypescript << showpos << result.imag() << 'i' << noshowpos;
      }
    }
  }
  else // result.real() == 0
  {
    if (result.imag() != 0)
    {		      
      cout << result.imag() << 'i';
      if ((ptypescript)) {
	*ptypescript << result.imag() << 'i';
      }
    }
    else
    {
      cout << '0';
      if ((ptypescript)) {
	*ptypescript << '0';
      }
    }
  } // end of result.real() == 0
  cout << endl;
  if ((ptypescript)) {
    *ptypescript << endl;
  }
}

void reportFatalError (const string expr, const evalError& e, ostream* ptypescript)
{
  
#ifdef _USE_ASCII_ESCAPES_
  // ASCII escape for bold writing on red background: \e[1;39;41m
  // ASCII escape for reset: \e[00m
  cerr << "\n*** Fatal error occurred while evaluating\n";
  for (size_t idx = 0; idx < expr.size(); idx++) {
    if (idx == e.position())
      cerr << "\e[" << 1 << ';' << 39 << ';' << 41 << 'm'
	   << expr[idx] << "\e[00m";
    else
      cerr.put(expr[idx]);
  }
#else
  cerr << "\n*** Fatal error occurred while evaluating\n" << expr << endl;
  
  for (size_t idx = 0; idx < expr.size(); idx++) {
    cerr.put((idx == e.position() ? '^' : ' '));
  }
#endif // _USE_ASCII_ESCAPES_
  cerr << "\n\n*** " <<  e.what() << endl;
  cerr << "*** " << e.hint() << '\n' << endl;
  if ((ptypescript))
    {
      *ptypescript << "\n*** Fatal error occurred while evaluating\n" << expr << endl;
      
      for (size_t idx = 0; idx < expr.size(); idx++) {
	ptypescript->put((idx == e.position() ? '^' : ' '));
      }
      *ptypescript << "\n\n*** " <<  e.what() << endl;
      *ptypescript << "*** " << e.hint() << '\n' << endl;
    }
}

class location {
public:
  location (size_t idx): m_idx(idx) {}
  bool operator() (evalError e) {
    return (e.position() == m_idx);
  }
private:
  size_t m_idx;
};

void reportErrors (const string& expr, const vector<evalError>& vv, ostream* ptypescript)
{
#ifdef _USE_ASCII_ESCAPES_
  // ASCII escape for bold blue write: \e[1;34;49m
  // ASCII escape for reset: \e[00m
  cerr << "\n*** Errors occurred while evaluating\n";
  for (size_t idx = 0; idx < expr.size(); idx++) {
    if ( find_if(vv.begin(), vv.end(), location(idx)) != vv.end() )
      cerr << "\e[" << 1 << ';' << 34 << ';' << 49 << 'm'
	   << expr[idx] << "\e[00m";
    else
      cerr.put(expr[idx]);
  }
#else
  cerr << "\n*** Errors occurred while evaluating\n" << expr << endl;
  for (size_t idx = 0; idx < expr.size(); idx++)
    {
      if ( find_if(vv.begin(), vv.end(), location(idx)) != vv.end() )
	cerr.put('^');
      else
	cerr.put(' ');
    }
#endif // _USE_ASCII_ESCAPES_  
  for (size_t idx = 0; idx < vv.size(); idx++)
    {
      cerr << "\n\n*** " <<  vv[idx].what() << endl;
      cerr << "*** " << vv[idx].hint() << endl;
    }
  cerr.put('\n');
  if ((ptypescript))
    {
      *ptypescript << "\n*** Errors occurred while evaluating\n" << expr << endl;
      for (size_t idx = 0; idx < expr.size(); idx++)
	{
	  if ( find_if(vv.begin(), vv.end(), location(idx)) != vv.end() )
	    ptypescript->put('^');
	  else
	    ptypescript->put(' ');
	}
      for (size_t idx = 0; idx < vv.size(); idx++)
	{
	  *ptypescript << "\n\n*** " <<  vv[idx].what() << endl;
	  *ptypescript << "*** " << vv[idx].hint() << endl;
	}
      ptypescript->put('\n');
    }
}

vector<string> splitString (const string& str)
{
  istringstream iss (str);
  vector<string> vv;
  string field;
  
  while ((iss >> field)) {
    vv.push_back (field);
  }
  return vv;
}

void displayEntries (ostream* ptypescript,
		     const vector<variablesTable::entry>& vv)
{
  for (vector<variablesTable::entry>::const_iterator cit = vv.begin();
       cit != vv.end(); ++cit)
    {
      string head = (cit->m_isReadOnly == true ? "(const) " : "        ");
      cout << head;
      if ((ptypescript))
	{
	  (*ptypescript) << head;
	}
      cout << setw(20) << cit->m_id << " = ";
      if ((ptypescript))
	{
	  (*ptypescript) << setw(20) << cit->m_id << " = ";
	}
      display (ptypescript, cit->m_value);
    }
  cout << "+++ Number of variables listed: " << vv.size() << endl;
  if ((ptypescript))
    {
      (*ptypescript) << "+++ Number of variables listed: "
		     <<	vv.size() << endl;
    }
}

int handleCommand (const string& inputLine, evaluator& myEvaluator,
		   ostream* ptypescript)
{
  const string commandPrefix(1, CMD_PREFIX);
  const string help = commandPrefix + "help";
  const string listByPrefix = commandPrefix + "ls";
  const string listBySuffix = commandPrefix + "LS";  
  const string removeByPrefix = commandPrefix + "rm";
  const string removeBySuffix = commandPrefix + "RM";  

  variablesTable& vTable = myEvaluator.internalVartable();
  vector<string> vv = splitString (inputLine);
  string key = (vv.size() == 2 ? vv.back() : "");
  
  if ((vv.front() == help && vv.size() != 1) ||
      (vv.front() != help && vv.size() > 2))
    {
      cerr << "*** Unknown command or invalid number of arguments" << endl;
      if ((ptypescript))
	{
	  (*ptypescript) << "*** Unknown command or invalid number of arguments" << endl;
	}
      return -1;
    }
  else if (vv.front() == help)
    {
      cout << "+++ Available commands:" << endl;
      //      cout << help << " - display this help" << endl;
      cout << listByPrefix << " <prefix>" << endl;
      cout << "  List all variables whose identifier matches the given <prefix>" << endl;
      cout << "  or list them all if no <prefix> is supplied" << endl;
      cout << listBySuffix << " <suffix>" << endl;
      cout << "  List all variables whose identifier matches the given <suffix>" << endl;
      cout << "  or list them all if no <suffix> is supplied" << endl;      
      cout << removeByPrefix << " <prefix>" << endl;
      cout << "  Remove all variables whose identifier matches the given <prefix>" << endl;
      cout << "  or remove them all if no <prefix> is supplied" << endl;
      cout << removeBySuffix << " <suffix>" << endl;
      cout << "  Remove all variables whose identifier matches the given <suffix>" << endl;
      cout << "  or remove them all if no <suffix> is supplied" << endl;  
    }
  else if (vv.front() == listByPrefix)
    {
      vector<variablesTable::entry> vv = vTable.getVariables (key, match::PREFIX);
      displayEntries (ptypescript, vv);
    }
  else if (vv.front() == listBySuffix)
    {
      vector<variablesTable::entry> vv = vTable.getVariables (key, match::SUFFIX);
      displayEntries (ptypescript, vv);      
    }
  else if (vv.front() == removeByPrefix)
    {
      size_t nn = vTable.eraseVariables (key, match::PREFIX);
      
      cout << "+++ " << nn << " variable(s) deleted" << endl;
      if ((ptypescript))
	{
	  (*ptypescript) << "+++ " << nn
			 << " variable(s) deleted" << endl;
	}
    }
  else if (vv.front() == removeBySuffix)
    {
      size_t nn = vTable.eraseVariables (key, match::SUFFIX);
      
      cout << "+++ " << nn << " variable(s) deleted" << endl;
      if ((ptypescript))
	{
	  (*ptypescript) << "+++ " << nn
			 << " variable(s) deleted" << endl;
	}
    }
  else
    {
      cerr << "*** " << vv.front() << " - Unknown command" << endl;
      if ((ptypescript))
	{
	  (*ptypescript) << "*** " << vv.front()
			 << " - Unknown command" << endl;
	}
      return -1;
    }
  return 0;
}

// Return value: -1 in case of fatal error detected, 0 otherwise
int handleInputWith (const string& inputLine, evaluator& myEvaluator, ostream* ptypescript)
{
  const char startOfComment = '#';
  cValue result;
  size_t idx;
  
  idx = Utils::posOfFirstNonSpace (inputLine);
  if ( idx != string::npos && inputLine[idx] == CMD_PREFIX)
    {
      // The input line contains a command
      handleCommand (inputLine, myEvaluator, ptypescript);
    }
  else if ( idx != string::npos && inputLine[idx] != startOfComment )
    {
      // The input line does not contain only spaces,
      // and the first non-space character is not a comment mark
      try
	{
	  result = myEvaluator.evaluate (inputLine.c_str());
	}
      catch (const evalError& e)
	{
	  reportFatalError (inputLine, e, ptypescript);
	  return -1;
	} 
      if (myEvaluator.listOfComputationalErrors().size() != 0)
	{
	  reportErrors (inputLine,
			myEvaluator.listOfComputationalErrors(),
			ptypescript);
	}
      else
	{
	  display (ptypescript, result);
	}
    } // end of: the input line is not empty and is not a comment
  return 0;
}

int main (int argc, char* argv[])
{
  string inputLine;
  evaluator myEvaluator;
  bool appendMode;
  string typescriptPath;
  char* homeDir = getenv("HOME");
  ofstream* ptypescript;
  ofstream typescript;
  int firstArgIndex = handleOptions (argc, argv, appendMode);

  myEvaluator.internalVartable().setVariable (variableDefinition("_E_", cValue (2.718281828459, 0.0), true));
  myEvaluator.internalVartable().setVariable (variableDefinition("_PI_", cValue (3.141592653590, 0.0), true));
  if (!homeDir)
    {
      cerr << "*** This session will not be recorded into the typescript file:" << endl;
      cerr << "    the path of the home directory could not be retrieved" << endl;
      ptypescript = (ofstream*)0;
    }
  else
    {
      typescriptPath = string(homeDir);
      typescriptPath += TYPESCRIPT_NAME;
      if ((appendMode)) {
	typescript.open (typescriptPath.c_str(), std::ofstream::out | std::ofstream::app);
      }
      else {
	typescript.open (typescriptPath.c_str(), std::ofstream::out | std::ofstream::trunc);
      }
      if ( (typescript.is_open()) ) {
	ptypescript = &typescript;
      }
      else {
	cerr << "*** This session will not be recorded into the typescript file:" << endl;
	cerr << "    the file could not be opened for writing" << endl;
	ptypescript = (ofstream*)0;
      }
    }
  cout.precision(12);
  if ((ptypescript)) {
    ptypescript->precision(12);
  }
  if (firstArgIndex == argc)
    {
      // Interactive usage
      bool wasLineRead = false;
#ifdef _USE_READLINE_
      string historyFile;
      int errorCode = 0;

      const char *userHome = getenv("HOME");
      if ((userHome))
	{
	  historyFile = string(userHome) + "/.schistory";
	}
      else
	{
	  historyFile = "./.schistory";
	}
      if ( (errorCode = initReadline (argv[0], historyFile.c_str(), 1024)) )
	{
	  cerr << "*** Could not load history from " << historyFile
	       << ':' << endl;
	  cerr << strerror (errorCode) << endl;
	}
#endif // _USE_READLINE_
      cout << "Type an expression followed by <Enter> to evaluate\n"
	   << "it and display its result. To quit the program\n"
	   << "press <Ctrl> and 'D' together on an empty line.\n"
	   << "To obtain a list of all available commands\n"
	   << "type :help followed by <Enter>.\n" << endl;
      do
	{
	  readLineInString (">>> ", inputLine, ptypescript, wasLineRead);
	  if ((wasLineRead))
	    handleInputWith (inputLine, myEvaluator, ptypescript);
	} while ((wasLineRead));
#ifdef _USE_READLINE_
      if ( (errorCode = endReadline (historyFile.c_str())) )
	{
	  cerr << "*** Could not save history to " << historyFile
	       << ':' << endl;
	  cerr << strerror (errorCode) << endl;
	}
#endif // _USE_READLINE_
    } // end of: Interactive usage
  else
    {
      // Non interactive usage
      for (int idx = firstArgIndex; idx < argc; idx++)
	{
	  inputLine = string(argv[idx]);
	  if ((ptypescript)) {
	    *ptypescript << ">>> " << inputLine << endl;
	  }
	  if (handleInputWith (inputLine, myEvaluator, ptypescript) != 0) {
	      return EXIT_FAILURE;
	  }
	}
    } // end of: Non interactive usage
  if ((ptypescript)) {
    ptypescript->close();
  }
  return EXIT_SUCCESS;
}
