/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _MATHTOKEN_H_
#define _MATHTOKEN_H_

#include <string>
#include <vector>
#include <algorithm>
#include <iostream>
#include "numTypes.h"

namespace bmEval
{
  class mathToken
  {
    friend std::ostream& operator<< (std::ostream& os, const mathToken& token);

  public:
    typedef enum {
      UNDEFINED = 0,
      NUMBER = 1,
      VARIABLE = 2,
      OPEN_PARENTHESIS = 3,
      CLOSED_PARENTHESIS = 4,
      UNARY_OPERATOR=6,
      OP_POS = 6,
      OP_NEG = 7,
      OP_NOT = 8,
      OP_DEL = 9,
      BINARY_OPERATOR = 10,
      OP_AND = 10,
      OP_OR  = 11,
      OP_XOR = 12,
      OP_LT = 13,
      OP_GT = 14,
      OP_LE = 15,
      OP_GE = 16,
      OP_EQ = 17,
      OP_NE = 18,
      OP_ADD = 19,
      OP_SUB = 20,
      OP_MUL = 21,
      OP_DIV = 22,
      OP_PERCENT = 23,
      OP_MOD  = 24,
      OP_IDIV = 25,
      OP_POW  = 26,
      OP_MAX  = 27,
      OP_MIN  = 28,
      ASSIGNMENT_OPERATOR = 50,
      OP_DEF  = 50,
      OP_STO  = 51,
      OP_ADD_STO = 52,
      OP_SUB_STO = 53,
      OP_MUL_STO = 54,
      OP_DIV_STO = 55,
      OP_PERCENT_STO = 56,
      OP_MOD_STO  = 57,
      OP_IDIV_STO = 58,
      OP_POW_STO  = 59,
      OP_MAX_STO  = 60,
      OP_MIN_STO  = 61,
      FUNCTION = 100,
      FN_RE    = 100,
      FN_IM    = 101,
      FN_ARG   = 102,
      FN_ABS   = 103,
      FN_CONJ  = 104,
      FN_EXP   = 105,
      FN_SQRT  = 106,
      FN_CBRT  = 107,
      FN_LOG   = 108,
      FN_LOG2  = 109,
      FN_LOG10 = 110,
      FN_SIN   = 111,
      FN_COS   = 112,
      FN_TAN   = 113,
      FN_ASIN  = 114,
      FN_ACOS  = 115,  
      FN_ATAN  = 116,
      FN_SINH  = 117,
      FN_COSH  = 118,
      FN_TANH  = 119,
      FN_ASINH = 120,
      FN_ACOSH = 121,
      FN_ATANH = 122,
      FN_ERF   = 123,
      FN_ERFC  = 124,
      FN_GAMMA = 125,
      FN_FLOOR = 126,
      FN_CEIL  = 127,
      FN_ROUND = 128,
      FN_FIX   = 129,
      FN_FRAC  = 130,
      FN_STEP  = 131,
      FN_OSTEP = 132, 
      FN_SIGN  = 133,
      FN_X01CC = 134,
      FN_X01OO = 135,
      FN_X01CO = 136,
      FN_X01OC = 137,
      FN_DISP =  138,
      INVALID_TOKEN = 200,
      END_TOKEN = 300
    } tokenType;

    mathToken(size_t startPosition = 0, tokenType type = UNDEFINED, std::string id = "", cValue value = 0) :
    m_startPosition (startPosition),
    m_type (type),
    m_id(id),
    m_value(value)
    { }
    
    mathToken (const std::string& str, bool allowForComplexInput, size_t& afterEndToken, size_t startWith = 0) :
    m_startPosition (startWith),
    m_type (UNDEFINED),
    m_id(""),
    m_value(0.0)
    {
      extractTokenFrom (str, allowForComplexInput, m_startPosition, m_type, m_id, m_value, afterEndToken);
    }

    mathToken (const mathToken& token) :
    m_startPosition(token.m_startPosition),
    m_type(token.m_type),
    m_id(token.m_id),
    m_value(token.m_value)
    { }

    mathToken& operator= (const mathToken& rhs)
      {
	if (&rhs != this)
	  {
	    m_startPosition = rhs.m_startPosition;
	    m_type = rhs.m_type;
	    m_id = rhs.m_id;
	    m_value = rhs.m_value;
	  }
	return *this;
      }
    // Default destructor is fine

    // Getting methods
    size_t startPosition() const { return m_startPosition; }
    tokenType Type () const { return m_type; }
    std::string Id() const { return m_id; }
    cValue Value() const { return m_value; } 

    // Setting methods
    void Type (tokenType newType) { m_type = newType; }
    void Value (cValue newValue)  { m_value = newValue; } 

    // Property methods
    bool isOperator () const { return (m_type >= UNARY_OPERATOR &&
				       m_type < FUNCTION); }

    bool isFunction () const { return (m_type >= FUNCTION &&
				       m_type < INVALID_TOKEN); }

    bool isUnaryOperator () const { return (m_type >= UNARY_OPERATOR &&
					    m_type < BINARY_OPERATOR); }

    bool isBinaryOperator () const { return (m_type >= BINARY_OPERATOR &&
					     m_type < ASSIGNMENT_OPERATOR); }

    bool isAssignmentOperator () const { return (m_type >= ASSIGNMENT_OPERATOR &&
						 m_type < FUNCTION); }

    static const char oParenthesis = '(';
    static const char cParenthesis = ')';
    static const char* multiplicationSign;

  private:

    struct association
    {
    association(const char* str, tokenType type) : m_str(str), m_type(type) {}
      std::string m_str;
      tokenType m_type;
    };

    friend bool operator< (const association& key1, const association& key2)
    {
      return (key1.m_str < key2.m_str); 
    }

    static std::vector <association> sm_fnLookupTable;
    static std::vector <association> sm_opLookupTable;
    static std::vector <association> createFnLookupTable();
    static std::vector <association> createOpLookupTable();

    static tokenType lookForKeyInOperatorsTable (const std::string& key)
    {
      std::vector<association>::const_iterator constIt;
    
      constIt = std::lower_bound (sm_opLookupTable.begin(), sm_opLookupTable.end(), association(key.c_str(), UNDEFINED));
      return ( (constIt != sm_opLookupTable.end() && constIt->m_str == key) ? constIt->m_type : INVALID_TOKEN );
    }

    static tokenType lookForKeyInFunctionsTable (const std::string& key)
    {
      std::vector<association>::const_iterator constIt;
    
      constIt = std::lower_bound (sm_fnLookupTable.begin(), sm_fnLookupTable.end(), association(key.c_str(), UNDEFINED));
      return ( (constIt != sm_fnLookupTable.end() && constIt->m_str == key) ? constIt->m_type : INVALID_TOKEN );
    }

    void extractTokenFrom (const std::string& str, bool isComplexInputAllowed,
			   size_t& startPosition, tokenType& type, 
			   std::string& id, cValue& value, 
			   size_t& afterEndToken);

    size_t m_startPosition;
    tokenType m_type;
    std::string m_id;
    cValue m_value; // Only used for numbers 
  };
} // end of namespace bmEval

#endif // _MATHTOKEN_H_
