/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _FUNCTION_LIST_H_
#define _FUNCTION_LIST_H_

inline rValue ROUND (rValue x)
{
  if (x >= 0)
    {
      return (x - floor (x) < 0.5 ? floor (x) : ceil (x));
    }
  else
    {
      return (x - floor (x) <= 0.5 ? floor (x) : ceil (x));
    }
}

inline rValue TRUNC (rValue x)
{
  return (x >= 0 ? floor(x) : ceil(x));
}

inline rValue FRAC (rValue x)
{
  return (x - (x >= 0 ? floor(x) : ceil(x)));
}

// The arg() function of GLIBC from <complex.h> is buggy.
// This forces a redefinition not only of this function, but also
// of the functions which make use of it.
inline rValue ARG (cValue z)
{
  return ( (z.real() < 0 && z.imag() < EPS && z.imag() > -EPS) ? 
           MATH_PI : atan2 (z.imag(), z.real()) );
} 

// The sign function
inline rValue SGN (rValue x)
{
  return (x >= 0 ? 1 : -1);
}

inline cValue SQRT (cValue z)
{
  rValue mod = abs(z);
  return cValue(sqrt ((mod + z.real()) * 0.5), SGN (z.imag()) * sqrt ((mod - z.real()) * 0.5));
}

inline cValue LOG (cValue z)
{
  return cValue(log(abs(z)), ARG(z));
}

inline cValue GAMMA (cValue z)
{
  // This is not the right definition
  // of the Gamma function for complex numbers.
  // But I did not have time to implement
  // it in a proper way.
  return cValue(tgamma(z.real()), 0);
}

mathToken _pos (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", z);
}

mathToken _neg (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", -z);
}

mathToken _not (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", (z == Zero ? One : Zero));
}

mathToken _re (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", z.real());
}

mathToken _im (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", z.imag());
}

mathToken _arg (cValue z, size_t fnPos)
{
  if (z != Zero)
    {
      return mathToken (fnPos, mathToken::NUMBER, "", cValue(ARG(z)));      
    }
  else
    {
      throw evalError ("", "Out of domain", 
		       "The function argument is defined only for non-null complex numbers", fnPos);
    }
}

mathToken _abs (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", cValue(abs(z)));
}

mathToken _conj (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", cValue(conj(z)));
}

mathToken _exp (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", exp(z));
}

mathToken _sqrt (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", SQRT(z));
}

/*
  ROOT(Z, I, N) is the Ith branch of the Nth root of Z.
  Precondition: N > 0
*/

inline cValue ROOT (cValue z, unsigned long i, unsigned long n)
{
  rValue mod = pow (abs(z), 1.0 / n);
  rValue arg = (ARG(z) + i * 2 * MATH_PI) / n;
  
  return cValue(mod*cos(arg), mod*sin(arg));
}

mathToken _cbrt (cValue z, size_t fnPos)
{
  unsigned long i = (z.real() >= 0.0 ? 0 : 1);
  return mathToken (fnPos, mathToken::NUMBER, "", ROOT(z, i, 3));
}

mathToken _log (cValue z, size_t fnPos)
{
  if (z != Zero)
    {
      return mathToken (fnPos, mathToken::NUMBER, "", LOG(z));
    }
  else
    {
      throw evalError ("", "Out of domain", 
		       "The argument of the logarithmic function should always be a non-null number", fnPos);
    }
}

mathToken _log2 (cValue z, size_t fnPos)
{
  if (z != Zero)
    {
      return mathToken (fnPos, mathToken::NUMBER, "", LOG(z) / LogTwo);
    }
  else
    {
      throw evalError ("", "Out of domain", 
		       "The argument of the logarithmic function should always be a non-null number", fnPos);
    }
}

mathToken _log10 (cValue z, size_t fnPos)
{
  if (z != Zero)
    {
      return mathToken (fnPos, mathToken::NUMBER, "", LOG(z) / LogTen);
    }
  else
    {
      throw evalError ("", "Out of domain", 
		       "The argument of the logarithmic function should always be a non-null number", fnPos);
    }
}

mathToken _sin (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", sin(z));
}

mathToken _cos (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", cos(z));
}

mathToken _tan (cValue z, size_t fnPos)
{
  cValue cos_z = cos(z);

  if ( cos_z == Zero )
    {
      throw evalError ("", "Out of domain", 
		       "The cosine of the argument of the tangent function should always be non-zero", fnPos);      
    }
  else
    {
      return mathToken (fnPos, mathToken::NUMBER, "", sin(z) / cos_z);
    }
}

mathToken _asin (cValue z, size_t fnPos)
{
  cValue w = SQRT (One - z*z);

  if ( abs(ImagUnit * z + w) < VSV  * abs(z) )
    {
      return mathToken (fnPos, mathToken::NUMBER, "", 
                        LOG (w - ImagUnit * z) * ImagUnit);
    }
  else
    {
      return mathToken (fnPos, mathToken::NUMBER, "", 
                        -ImagUnit * LOG (ImagUnit * z + w));
    }
}

mathToken _acos (cValue z, size_t fnPos)
{
  cValue w = SQRT (One - z*z);

  if ( abs(z + ImagUnit * w) < VSV  * abs(z) )
    {
      return mathToken (fnPos, mathToken::NUMBER, "", 
                        LOG (z - ImagUnit * w) * ImagUnit);
    }
  else
    {
      return mathToken (fnPos, mathToken::NUMBER, "",
                        -ImagUnit * LOG (z + ImagUnit * w));
    }
}

mathToken _atan (cValue z, size_t fnPos)
{
  if ( abs(z.real()) < EPS && 
      (abs(z.imag()+1) < EPS || abs(z.imag()-1) < EPS) )
    {
      throw evalError ("", "Out of domain", 
		       "The argument of the arc tangent function should always differ from +/-1i", fnPos);            
    }
  else
    {
      if ( abs(z) > VGV )
	{
	  cValue w(-ImagUnit * LOG((One+ImagUnit*z) / SQRT(One + z*z)));
	  return mathToken (fnPos, mathToken::NUMBER, "", w);
	}
      else
	{
	  cValue w(-ImagUnit * LOG( SQRT((One + ImagUnit * z)/(One - ImagUnit * z)) ));
	  return mathToken (fnPos, mathToken::NUMBER, "", w);
	}
    }
}

mathToken _sinh (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", sinh(z));
}

mathToken _cosh (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", cosh(z));
}

mathToken _tanh (cValue z, size_t fnPos)
{
  cValue cosh_z = cosh(z);

  if ( cosh_z == Zero )
    {
      throw evalError ("", "Out of domain", 
		       "The hyperbolic cosine of the argument of the hyperbolic tangent\nshould always be non-zero", fnPos);      
    }
  else
    {
      return mathToken (fnPos, mathToken::NUMBER, "", sinh(z) / cosh_z);
    }
}

mathToken _asinh (cValue z, size_t fnPos)
{
  // In this way, _asinh() works fine also with real numbers
  // very near to -Inf.
  cValue w ( SQRT(One+z*z) );

  if ( abs(z+w) < VSV * abs(z) )
    {
      return mathToken (fnPos, mathToken::NUMBER, "", -LOG(w-z));      
    }
  else
    {
      return mathToken (fnPos, mathToken::NUMBER, "", LOG(w+z));
    }
}

mathToken _acosh (cValue z, size_t fnPos)
{
  cValue w ( SQRT(z*z-One) );

  if ( abs(z+w) < VSV * abs(z) )
    {
      return mathToken (fnPos, mathToken::NUMBER, "", -LOG(z-w));      
    }
  else
    {
      return mathToken (fnPos, mathToken::NUMBER, "", LOG(z+w));
    }
}

mathToken _atanh (cValue z, size_t fnPos)
{
  if ( abs(z.imag()) < EPS && 
      (abs(z.real()+1) < EPS || abs(z.real()-1) < EPS) )
    {
      throw evalError ("", "Out of domain", 
		       "The argument of the hyperbolic arc tangent should always differ from +/-1", fnPos);            
    }
  else
    {
      return mathToken (fnPos, mathToken::NUMBER, "", LOG( SQRT((One + z)/(One - z)) ));
    }
}

mathToken _erf (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", 
                    static_cast<cValue>(Faddeeva::erf(z)));
}

mathToken _erfc (cValue z, size_t fnPos)
{
  return mathToken (fnPos, mathToken::NUMBER, "", 
                    static_cast<cValue>(Faddeeva::erfc(z)));
}

mathToken _gamma (cValue z, size_t fnPos)
{
  if (z.real() < EPS && FRAC(z.real()) > -EPS)
    {
      throw evalError ("", "Out of domain", 
		       "The real part of the argument of gamma cannot be a non-positive integer number", fnPos);
    }
  else
    {
      return mathToken (fnPos, mathToken::NUMBER, "", GAMMA(z));
    }
}

mathToken _floor (cValue z, size_t fnPos)
{
  return mathToken ( fnPos, mathToken::NUMBER, "", 
                     cValue(floor(z.real()), floor(z.imag())) );  
}

mathToken _ceil (cValue z, size_t fnPos)
{
  return mathToken ( fnPos, mathToken::NUMBER, "", 
                     cValue(ceil(z.real()), ceil(z.imag())) );  
}

mathToken _round (cValue z, size_t fnPos)
{
  return mathToken ( fnPos, mathToken::NUMBER, "", 
                     cValue(ROUND(z.real()), ROUND(z.imag())) );  
}

mathToken _fix (cValue z, size_t fnPos)
{
  return mathToken ( fnPos, mathToken::NUMBER, "", 
                     cValue(TRUNC(z.real()), TRUNC(z.imag())) );  
}

mathToken _frac (cValue z, size_t fnPos)
{
  return mathToken ( fnPos, mathToken::NUMBER, "", 
                     cValue(FRAC(z.real()), FRAC(z.imag())) );  
}

mathToken _step (cValue z, size_t fnPos)
{
  if ( z.imag() == 0 && z.real() >= 0 )
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", One);
    }
  else
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", Zero);
    }
}

mathToken _ostep (cValue z, size_t fnPos)
{
  if ( z.imag() == 0 && z.real() > 0 )
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", One);
    }
  else
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", Zero);
    }
}

mathToken _sign (cValue z, size_t fnPos)
{
  if (z.real() > 0)
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", One);
    }
  else if (z.real() < 0)
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", -One);
    }
  else
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", Zero);
    }  
}

mathToken _X01cc (cValue z, size_t fnPos)
{
  if ( z.imag() == 0 && z.real() >= 0 && z.real() <= 1)
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", One);
    }
  else
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", Zero);
    }
}

mathToken _X01oo (cValue z, size_t fnPos)
{
  if ( z.imag() == 0 && z.real() > 0 && z.real() < 1)
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", One);
    }
  else
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", Zero);
    }
}

mathToken _X01co (cValue z, size_t fnPos)
{
  if ( z.imag() == 0 && z.real() >= 0 && z.real() < 1)
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", One);
    }
  else
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", Zero);
    }
}

mathToken _X01oc (cValue z, size_t fnPos)
{
  if ( z.imag() == 0 && z.real() > 0 && z.real() <= 1)
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", One);
    }
  else
    {
      return mathToken ( fnPos, mathToken::NUMBER, "", Zero);
    }
}

mathToken _disp (cValue z, size_t fnPos)
{
  std::cout << "\ndisp() at position " << fnPos << ": " << z << std::endl;
  return mathToken ( fnPos, mathToken::NUMBER, "", z);
}

mathToken _nop (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", z2);
}

mathToken _and (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", 
		    (z1 != Zero && z2 != Zero ? One : Zero));
}

mathToken _or (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", 
		    (z1 != Zero || z2 != Zero ? One : Zero));
}

mathToken _xor (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", 
		    (z1 != Zero && z2 == Zero) ||
		    (z1 == Zero && z2 != Zero) ? One : Zero);
}

mathToken _lt (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", 
		    (z1.real() < z2.real() ? One : Zero));
}

mathToken _gt (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", 
		    (z1.real() > z2.real() ? One : Zero));
}

mathToken _le (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", 
		    (z1.real() <= z2.real() ? One : Zero));
}

mathToken _ge (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", 
		    (z1.real() >= z2.real() ? One : Zero));
}

mathToken _eq (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", 
		    (z1 == z2 ? One : Zero));
}

mathToken _ne (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", 
		    (z1 != z2 ? One : Zero));
}

mathToken _add (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", z1+z2);
}

mathToken _sub (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", z1-z2);
}

mathToken _mul (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", z1*z2);
}

mathToken _div (cValue z1, cValue z2, size_t opPos)
{
  if (z2 != Zero)
    {
	return mathToken (opPos, mathToken::NUMBER, "", z1 / z2);
    }
  else
    {
      throw evalError ("", "Division by zero", "The divisor should always be a non-null number", opPos);
    }
}

mathToken _percent (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", (z1/OneHundred)*z2);
}

mathToken _mod (cValue z1, cValue z2, size_t opPos)
{
  //  rValue z2Mod2 ( z2.real() * z2.real() + z2.imag() * z2.imag() );

  if (z2 != Zero)
    {
      cValue quotient = z1/z2;
      cValue roundedQuotient (TRUNC(quotient.real()), TRUNC(quotient.imag()));
      cValue z (z1 - roundedQuotient * z2);

      return mathToken ( opPos, mathToken::NUMBER, "", z );
    }
  else
    {
      throw evalError ("", "Division by zero", "The divisor should always be a non-null number", opPos);
    }
}

mathToken _idiv (cValue z1, cValue z2, size_t opPos)
{
  //  rValue z2Mod2 ( z2.real() * z2.real() + z2.imag() * z2.imag() );

  if (z2 != Zero)
    {
      cValue quotient = z1/z2;

      return mathToken ( opPos, mathToken::NUMBER, "", 
			 cValue(TRUNC(quotient.real()), TRUNC(quotient.imag())) );
    }
  else
    {
      throw evalError ("", "Division by zero", "The divisor should always be a non-null number", opPos);
    }
}

mathToken _pow (cValue z1, cValue z2, size_t opPos)
{
  if (z1 != Zero)
    {
      rValue mod1 = abs(z1);
      rValue arg1 = ARG(z1);
      rValue a = z2.real() * log (mod1) - z2.imag() * arg1;
      rValue b = z2.real() * arg1 + z2.imag() * log (mod1);

      return mathToken (opPos, mathToken::NUMBER, "", cValue(exp(a)*cos(b), exp(a)*sin(b)));
    }
  else
    {
	if (z2.real() <= 0)
	  {
            throw evalError ("", "Bad exponent for operator", 
			     "Zero cannot be raised to a power with non-positive real part", opPos);
	  }
	else
	  {
	    return mathToken (opPos, mathToken::NUMBER, "", Zero);
	  }
    }
}

mathToken _max (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", 
		    (z1.real() >= z2.real() ? z1 : z2));
}

mathToken _min (cValue z1, cValue z2, size_t opPos)
{
  return mathToken (opPos, mathToken::NUMBER, "", 
		    (z1.real() <= z2.real() ? z1 : z2));
}

#endif // _FUNCTION_LIST_H_
