/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <cstdlib>
#include <cerrno>
#include <cctype>
#include <algorithm>
#include "Utils.h"

namespace bmEval
{
  bool Utils::isAlpha (char ch)
  {
#ifdef USE_ASCII
    return ((ch >= 'A' && ch <= 'Z') || (ch >= 'a' && ch <= 'z'));
#else
    switch (ch)
      {
      case 'a':
      case 'b':
      case 'c':
      case 'd':
      case 'e':
      case 'f':
      case 'g':
      case 'h':
      case 'i':
      case 'j':
      case 'k':
      case 'l':
      case 'm':
      case 'n':
      case 'o':
      case 'p':
      case 'q':
      case 'r':
      case 's':
      case 't':
      case 'u':
      case 'v':
      case 'w':
      case 'x':
      case 'y':
      case 'z':
      case 'A':
      case 'B':
      case 'C':
      case 'D':
      case 'E':
      case 'F':
      case 'G':
      case 'H':
      case 'I':
      case 'J':
      case 'K':
      case 'L':
      case 'M':
      case 'N':
      case 'O':
      case 'P':
      case 'Q':
      case 'R':
      case 'S':
      case 'T':
      case 'U':
      case 'V':
      case 'W':
      case 'X':
      case 'Y':
      case 'Z':       
	return true;
      default:
	return false;
      }
#endif
  }

  bool Utils::doesStartWithId (const std::string& str, size_t& idx)
  {
    if (str.length() == 0)
      {
	idx = std::string::npos;
	return false;
      }
    else if ( str[0] != '_' && !isAlpha(str[0]) )
      {
	idx = 0;
	return false;
      }
    else
      {
	for (idx = 1;
	     idx < str.length() &&
	       (str[idx] == '_' || isDigit(str[idx]) || isAlpha(str[idx])); 
	     idx++);
	return true;
      }
  }

  std::string Utils::n2str (size_t n)
  {
    std::string str;
    
    do
      {
	str.push_back('0'+ n % 10);
      } while ((n /= 10));
    reverse (str.begin(), str.end());
    return str;
  }

  long Utils::str2l (const std::string& str, size_t& idx, int& errorCode, int base)
  {
    const char* cstr = str.c_str(); // it always ends with \0
    size_t startPosition;
    long value;
    char* endptr;

    for (startPosition = 0; isspace(cstr[startPosition]) != 0; startPosition++);
    errno = 0;
    value = strtol (cstr + startPosition, &endptr, base);
    errorCode = errno;
    idx = static_cast<size_t> (endptr -  cstr);
    if (errorCode == ERANGE)
      {
	errorCode = OVERFLOW_OCCURRED;
      }
    else
      {
	errorCode = (idx > startPosition ? CONVERSION_DONE : NOT_A_VALUE);
      }
    return value;
  }

  unsigned long Utils::str2ul (const std::string& str, size_t& idx, int& errorCode, int base)
  {
    const char* cstr = str.c_str(); // it always ends with \0
    size_t startPosition;
    unsigned long value;
    char* endptr;
  
    // First check that the first character after the
    // (possibly empty) initial sequence of white spaces
    // is not a minus. The function strtoul() accepts
    // also negative numbers, which then are converted
    // to unsigned long values. But I do not like
    // this default behaviour.
    for (startPosition = 0; isspace(cstr[startPosition]) != 0; startPosition++);
    if (cstr[startPosition] == '-')
      {
	idx = startPosition;
	errorCode = NOT_A_VALUE;
	return 0;
      }
    else
      {
	errno = 0;
	value = strtoul (cstr + startPosition, &endptr, base);
	errorCode = errno;
	idx = static_cast<size_t> (endptr -  cstr);
	if (errorCode == ERANGE)
	  {
	    errorCode = OVERFLOW_OCCURRED;
	  }
	else
	  {
	    errorCode = (idx > startPosition ? CONVERSION_DONE : NOT_A_VALUE);
	  }
	return value;
      }
  }

  rValue Utils::str2r (const std::string& str, size_t& idx, int& errorCode)
  {
    const char* cstr = str.c_str(); // it always ends with \0
    size_t startPosition;
    double value;
    char* endptr;

    for (startPosition = 0; isspace(cstr[startPosition]) != 0; startPosition++);
    errno = 0;
    value = strtod (cstr + startPosition, &endptr);
    errorCode = errno;
    idx = static_cast<size_t> (endptr -  cstr);
    if (errorCode == ERANGE)
      {
	if (value > 1.0 || value < -1.0)
	  {
	    errorCode = OVERFLOW_OCCURRED;
	  }
	else
	  {
	    // Underflow cases are just handled by rounding to zero
	    value = 0.0f;
	    errorCode = CONVERSION_DONE;
	  }
      }
    else
      {
	errorCode = (idx > startPosition ? CONVERSION_DONE : NOT_A_VALUE);
      }
#ifdef _SINGLE_PRECISION_
    // rValue is an alias of float
    return static_cast<rValue> (value);
#else
    // rValue is an alias of double
    return value; 
#endif
  }

  cValue Utils::str2c (const std::string& str, size_t& idx, int& errorCode)
  {
    rValue rvalue;
  
    rvalue = str2r (str, idx, errorCode);
    if (errorCode == NOT_A_VALUE)
      {
	return cValue(rvalue, 0.0);
      }
    else
      {
	// This is adequate both if errorCode == CONVERSION_DONE or
	// errorCode == OVERFLOW_OCCURRED 
	if (idx < str.length() && str[idx] == imaginaryUnit)
	  {
	    idx++;
	    return cValue(0.0, rvalue);
	  }
	else
	  {
	    return cValue(rvalue, 0.0);
	  }
      }
  }
} // end of namespace bmEval
