// Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef TAPIOCA_SESSION_H
#define TAPIOCA_SESSION_H

#include <gtkmm.h>

#include <sigc++/sigc++.h>
#include <libsoupmm/session.h>
#include <string>
#include <list>
#include <time.h>

#include "profile.h"
#include "document.h"
#include "gallerylist.h"
#include "gallery.h"
#include "changed-gallery.h"
#include "changed-document.h"

#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>
#include <boost/serialization/list.hpp>

namespace Tapioca
{
class SessionChanges
  {
public:
    SessionChanges(){};
    ~SessionChanges(){};
    bool empty() {return (modified_galleries.empty() == true &&
                          modified_docs.empty() == true && 
                          removed_galleries.empty() == true &&
                          added_galleries.empty() == true && 
                          removed_documents.empty() == true &&
                          added_documents.empty() == true);}
    std::list<ChangedGallery> modified_galleries;
    std::list<ChangedDocument> modified_docs;
    std::list<ChangedGallery> removed_galleries;
    std::list<ChangedGallery> added_galleries;
    std::list<ChangedDocument> removed_documents;
    std::list<ChangedDocument> added_documents;
  };
class Session: public GalleryList
{
  static double timeout; //in minutes.
    public:

	//! Copy constructor.
        Session(const Session& profile);

	//! Create an empty profile.
	Session();

	//! Destructor.
        virtual ~Session();

        // Set Methods
        
        void set_profile(Profile *p) {profile = p;}

        //Get Methods

        Profile* get_profile() const {return profile;}

        static Session* create(Profile *profile);

        typedef sigc::slot2<void, Document *, std::string> SlotDownloadDocument;
        typedef sigc::slot2<void, Gallery*, std::list<std::string>& > SlotPullGallery;
        typedef sigc::slot1<void, Document*> SlotPullThumbnail;
        void download_document(Document *document, const Session::SlotDownloadDocument slot);
        void pull_thumbnail(std::string url, const Session::SlotPullThumbnail slot);

        sigc::signal<void> pulled_gallery(Gallery *g);
        sigc::signal<void> pulled_documents_for_gallery(Gallery *g);
        sigc::signal<void> pulled_thumbnail_for_document(Document *d);
        sigc::signal<void> last_document_page_processed;
        sigc::signal<void, Document*> downloaded_document;
        sigc::signal<void, Document*> downloaded_chunk_of_document;
        sigc::signal<void, Document*> download_document_failed;
        sigc::signal<void, Session*> pull_galleries_failed;
        sigc::signal<void, std::list<Gallery*> > pulled_galleries;

        void pull_galleries();
        void login(sigc::slot<void, bool> slot);

        void copy_galleries_from_profile();
        void merge_changes(Session *session);
        
        void pull_documents_for_gallery(Gallery *gallery, const Session::SlotPullGallery slot, std::list<std::string> &thumbnail_urls);
        SessionChanges get_changes() const;

        void revert(SessionChanges reverts);
        void apply(SessionChanges changes);
    protected:

    private:
        Profile *profile;

        friend class boost::serialization::access;
                             
        template<class Archive>
          void serialize(Archive & ar, const unsigned int version)
            {
              ar & profile;
              ar & static_cast<GalleryList&>(*this);
            }

        Glib::RefPtr<Soup::Session> web;

        void on_login_attempted(Glib::RefPtr<Soup::Message> &msg,
                                sigc::slot<void, bool> slot);
        void on_document_downloaded(Glib::RefPtr<Soup::Message> msg, Document *document, const Session::SlotDownloadDocument slot, sigc::connection timer);

        void on_new_document_downloaded(Glib::RefPtr<Soup::Message> msg);

        std::string get_document_link_from_link_listing(Glib::RefPtr<Soup::Message> msg, Document *document) const;

        void on_pull_galleries_attempted(Glib::RefPtr<Soup::Message> &msg);

        void on_pull_documents_attempted(Glib::RefPtr<Soup::Message> &msg, Gallery *gallery, const Session::SlotPullGallery slot, std::list<std::string> &thumbnail_urls);
        void pull_all_documents_from_all_galleries();
        time_t last_login;
        bool is_login_necessary();
        std::string extract_document_id_from_thumbnail_url(std::string u) const;
        void on_thumbnail_downloaded(Glib::RefPtr<Soup::Message> &msg, Document *document, const Session::SlotPullThumbnail slot);

        bool on_download_document_timeout(Glib::RefPtr<Soup::Message> &msg);
        Glib::RefPtr<Soup::Message> login();

        void revert_modified_galleries(std::list<ChangedGallery> l);
        void revert_modified_documents(std::list<ChangedDocument> l);
        void revert_added_galleries(std::list<ChangedGallery> l);
        void revert_removed_galleries(std::list<ChangedGallery> l);
        void revert_added_documents(std::list<ChangedDocument> l);
        void revert_removed_documents(std::list<ChangedDocument> l);
        void revert_modified_gallery(ChangedGallery g);
        void revert_modified_document(ChangedDocument d);
        void revert_added_gallery(ChangedGallery g);
        void revert_removed_gallery(ChangedGallery g);
        void revert_added_document(ChangedDocument d);
        void revert_removed_document(ChangedDocument d);
        void update_modified_galleries(std::list<ChangedGallery> l);
        void update_modified_documents(std::list<ChangedDocument> l);
        void update_added_galleries(std::list<ChangedGallery> l);
        void update_added_documents(std::list<ChangedDocument> l);
        void update_modified_gallery(ChangedGallery g);
        void update_modified_document(ChangedDocument d);
        void update_added_gallery(ChangedGallery g);
        void update_added_document(ChangedDocument d);
};

}
#endif // TAPIOCA_SESSION_H

