//  Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>
#include <gtkmm.h>

#include "preferences-dialog.h"
#include "main.h"

PreferencesDialog::PreferencesDialog()
{
  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(Main::get_glade_path()
				    + "/preferences-dialog.gtk");

  xml->get_widget("dialog", dialog);
  dialog->set_icon_from_file(Main::get_data_path() + "/icon.png");

  xml->get_widget("homepage_entry", homepage_entry);
  homepage_entry->signal_key_release_event().connect
    (sigc::mem_fun(*this,&PreferencesDialog::on_homepage_changed));
  xml->get_widget("image_viewer_button", image_viewer_button);
  xml->get_widget("video_viewer_button", video_viewer_button);

  conf_client = Gnome::Conf::Client::get_default_client();
  conf_client->add_dir("/apps/" PACKAGE_NAME "/general");

  video_viewer_path = conf_client->get_string("/apps/" PACKAGE_NAME 
                                              "/general/video_viewer_path");
  if (video_viewer_path == "")
    {
      video_viewer_path = Glib::find_program_in_path("totem");
      conf_client->set("/apps/" PACKAGE_NAME "/general/video_viewer_path", 
                       video_viewer_path);
    }
  update_viewer_button(video_viewer_button, video_viewer_path);

  image_viewer_path = conf_client->get_string("/apps/" PACKAGE_NAME 
                                              "/general/image_viewer_path");
  if (image_viewer_path == "")
    {
      image_viewer_path = Glib::find_program_in_path("eog");
      conf_client->set("/apps/" PACKAGE_NAME "/general/image_viewer_path", 
                       image_viewer_path);
    }
  update_viewer_button(image_viewer_button, image_viewer_path);

  homepage = conf_client->get_string("/apps/" PACKAGE_NAME "/general/homepage");
  if (homepage == "")
    {
      homepage = "http://www.atpic.com";
      conf_client->set("/apps/" PACKAGE_NAME "/general/homepage", homepage);
    }

  homepage_entry->set_text(homepage);
  image_viewer_button->signal_clicked().connect
    (sigc::mem_fun(*this,&PreferencesDialog::on_image_viewer_clicked));
  video_viewer_button->signal_clicked().connect
    (sigc::mem_fun(*this,&PreferencesDialog::on_video_viewer_clicked));

  conf_client->notify_add("/apps/" PACKAGE_NAME "/general/video_viewer_path",
                          sigc::mem_fun(*this, 
                                        &PreferencesDialog::key_changed));
  conf_client->notify_add("/apps/" PACKAGE_NAME "/general/image_viewer_path",
                            sigc::mem_fun(*this, 
                                          &PreferencesDialog::key_changed));
  conf_client->notify_add("/apps/" PACKAGE_NAME "/general/homepage",
                          sigc::mem_fun(*this, 
                                        &PreferencesDialog::key_changed));
}

PreferencesDialog::~PreferencesDialog()
{
  delete dialog;
}
void PreferencesDialog::set_parent_window(Gtk::Window &parent)
{
    dialog->set_transient_for(parent);
    //dialog->set_position(Gtk::WIN_POS_CENTER_ON_PARENT);
}

void PreferencesDialog::hide()
{
  dialog->hide();
}

void PreferencesDialog::run()
{
  dialog->run();
}

void
PreferencesDialog::key_changed(guint connection_id, Gnome::Conf::Entry entry)
{
  // Make sure the preference has a valid value
  if (entry.get_value().get_type() == Gnome::Conf::VALUE_STRING) 
    {
      if (entry.get_key() == "/apps/" PACKAGE_NAME "/general/video_viewer_path")
        {
          Glib::ustring new_setting = entry.get_value().get_string();
          if (new_setting != video_viewer_path) 
            {
              video_viewer_path = new_setting;
              update_viewer_button(video_viewer_button, video_viewer_path);
            }
        }
      if (entry.get_key() == "/apps/" PACKAGE_NAME "/general/image_viewer_path")
        {
          Glib::ustring new_setting = entry.get_value().get_string();
          if (new_setting != image_viewer_path) 
            {
              image_viewer_path = new_setting;
              update_viewer_button(image_viewer_button, image_viewer_path);
            }
        }
      if (entry.get_key() == "/apps/" PACKAGE_NAME "/general/homepage")
        {
          Glib::ustring new_setting = entry.get_value().get_string();
          if (new_setting != homepage) 
            {
              homepage = new_setting;
              homepage_entry->set_text(homepage);
            }
        }
    }
}
    
bool PreferencesDialog::on_homepage_changed(GdkEventKey *event)
{
  homepage = homepage_entry->get_text();
  conf_client->set("/apps/" PACKAGE_NAME "/general/homepage", homepage);
  return true;
}

void PreferencesDialog::on_video_viewer_clicked()
{
  Gtk::FileChooserDialog chooser(*dialog, "Choose Program To View Videos");
  chooser.set_current_folder("/usr/bin");

  chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  chooser.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_ACCEPT);
  chooser.set_default_response(Gtk::RESPONSE_ACCEPT);
  chooser.show_all();
  int response = chooser.run();
    
  if (response == Gtk::RESPONSE_ACCEPT)
    {
      video_viewer_path = chooser.get_filename();
      conf_client->set("/apps/" PACKAGE_NAME "/general/video_viewer_path", 
                       video_viewer_path);
      update_viewer_button(video_viewer_button, video_viewer_path);
    }
}

void PreferencesDialog::on_image_viewer_clicked()
{
  Gtk::FileChooserDialog chooser(*dialog, "Choose Program To View Images");
  chooser.set_current_folder("/usr/bin");

  chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  chooser.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_ACCEPT);
  chooser.set_default_response(Gtk::RESPONSE_ACCEPT);
  chooser.show_all();
  int response = chooser.run();
    
  if (response == Gtk::RESPONSE_ACCEPT)
    {
      image_viewer_path = chooser.get_filename();
      conf_client->set("/apps/" PACKAGE_NAME "/general/image_viewer_path", 
                       image_viewer_path);
      update_viewer_button(image_viewer_button, image_viewer_path);
    }
}

void PreferencesDialog::update_viewer_button(Gtk::Button *button, Glib::ustring file)
{
  //There's a bug that happens when 2 filechooserbuttons appear on the same
  //dialog or window.
  if (file == "")
    button->set_label("no viewer program set");
  else
    button->set_label(Glib::path_get_basename(file));
}

