// Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <fstream>
#include "gallery.h"
#include "document.h"

using namespace Tapioca;

Gallery::Gallery(const Gallery& g)
: Item(g), document_thumbnail_id(g.document_thumbnail_id)
{
  for (Gallery::const_iterator it = g.begin(); it != g.end(); it++)
    push_back(new Document(*(*it)));
}

Gallery::Gallery()
: Item(), document_thumbnail_id("")
{
}

void Gallery::remove_all_documents()
{
  for (iterator it = begin(); it != end(); it++)
    delete (*it);
  clear();
}

Gallery::~Gallery()
{
  remove_all_documents();
}

        
Glib::RefPtr<Gdk::Pixbuf> Gallery::get_thumbnail_image() const
{
  for (Gallery::const_iterator it = begin(); it != end(); it++)
    {
      if ((*it)->get_id() == document_thumbnail_id)
        return (*it)->get_thumbnail();
    }
  Gtk::Image image(Gtk::Stock::MISSING_IMAGE, Gtk::ICON_SIZE_DIALOG);
  return image.render_icon(Gtk::Stock::MISSING_IMAGE, Gtk::ICON_SIZE_DIALOG);
}

std::string Gallery::get_next_in_order_sequence(std::string seq)
{
  std::string new_seq;
  if (seq.length() == 0)
    return " ";
  char last_letter = seq.at(seq.length() - 1);
  //starts with !, ends with ~
  if (last_letter >= '~')
    new_seq = seq + ' ';
  else
    {
      char new_letter = last_letter + 1;
      new_seq = seq;
      new_seq[new_seq.length() - 1] = new_letter;
    }
  return new_seq;
}

void Gallery::set_document_order(Document *document, Document *after)
{
  std::string new_order = "";
  if (after == NULL && front() != document && document)
    new_order = "";
  else if (after == NULL && front() == document && document)
    return;
  else
    {
      while (true)
        {
          std::string new_order = 
            get_next_in_order_sequence(after->get_order());
          if (has_order(new_order) == false)
            break;
        }
    }
  if (document)
    document->set_order(new_order);
  return;
}

bool Gallery::has_order(std::string o) const
{
  for (Gallery::const_iterator it = begin(); it != end(); it++)
    {
      if ((*it)->get_order() == o)
        return true;
    }
  return false;
}

void Gallery::remove_document(Document *document)
{
  remove(document);
  if (document_thumbnail_id == document->get_id())
    {
      if (size() > 0)
        document_thumbnail_id = front()->get_id();
      else
        document_thumbnail_id = "";
    }
}

void Gallery::update_thumbnails()
{
  for (iterator it = begin(); it != end(); it++)
    (*it)->update_thumbnail_from_loaded_data();
}

void Gallery::add_document(Document *document)
{
  push_back(document);
  document->set_gallery_id(get_id());
  if (size() == 1)
    document_thumbnail_id = front()->get_id();
}

void Gallery::find_deleted_documents(std::list<Document*> &removed)
{
  for (iterator it = begin(); it != end(); it++)
    {
      Document *document = (*it);
      if (document->get_image_filename() != "")
        {
          std::ifstream ifs(document->get_image_filename().c_str());
          if (ifs.is_open() == false)
            removed.push_back(document);
          else
            ifs.close();
        }
    }
}

void Gallery::export_html_page(std::string export_dir) const
{
  std::string top_index = export_dir + "/" + get_id() + "/index.html";
  std::ofstream ofs(top_index.c_str());
  if (ofs.is_open() == false)
    return;
  ofs << 
    "<html>\n\t<title>" << Glib::Markup::escape_text(get_title()) << 
    "</title>\n\t<body>" << std::endl;
  ofs << "\t</body>\n</html>" << std::endl;
  ofs << "\t\t<div class=\"gallerytitleblock\">" << std::endl;
  ofs <<"\t\t\t" << Glib::Markup::escape_text(get_title()) << std::endl;
  for (const_iterator it = begin(); it != end(); it++)
    {
      Document *doc = *it;
      if (doc->get_export_filename() == "")
        continue;
      ofs << "\t\t\t<div class=\"documentblock\">" << std::endl;
      ofs << "\t\t\t\t";
      ofs << "\t\t\t\t<div class=\"imgblock\">" << std::endl;
      ofs << "\t\t\t\t\t<img width=\"20%\" src=\"" << doc->get_export_filename() << "\"/>" << std::endl;
      ofs << "\t\t\t\t</div>" << std::endl;
      ofs << "<a href=\"" << doc->get_export_filename() << "\">" << Glib::Markup::escape_text(doc->get_title()) << "</a>" << std::endl;
      if (doc->get_description() != "")
        {
          ofs << "\t\t\t\t<div class=\"docdescriptionblock\">" << std::endl;
          ofs << "\t\t\t\t\t" << 
            Glib::Markup::escape_text(doc->get_description()) << std::endl;
          ofs << "\t\t\t\t</div>" << std::endl;
        }
      if (doc->get_latitude() != 0.0 && doc->get_longitude() != 0.0)
        {
          ofs << "\t\t\t\t<div class=\"docposition\">" << std::endl;
          ofs << "\t\t\t\t\tLatitude: " << doc->get_latitude() << std::endl;
          ofs << "\t\t\t\t\tLongitude: " << doc->get_longitude() << std::endl;
          ofs << "\t\t\t\t</div>" << std::endl;
        }
      std::list<std::string> tags = doc->get_tags();
      if (tags.size())
        {
          ofs << "\t\t\t\t<div class=\"doctags\">" << std::endl;
          ofs << "\t\t\t\t\tTags:";
          for (std::list<std::string>::iterator t = tags.begin(); 
               t != tags.end(); t++)
            ofs << " " << Glib::Markup::escape_text(*t);
          ofs << std::endl;
          ofs << "\t\t\t\t</div>" << std::endl;
        }
      ofs << "\t\t\t</div>" << std::endl;
    }
  ofs << "\t\t</div>" << std::endl;
  ofs.close();
}

void Gallery::set_id(std::string i)
{
  id = i;
  for (iterator it = begin(); it != end(); it++)
    (*it)->set_gallery_id(i);
}
        
void Gallery::add(const Gallery*gallery)
{
  for (const_iterator i = gallery->begin(); i != gallery->end(); i++)
    push_back(new Document(*(*i)));
}
