#!/bin/bash
# Limit extension usage to given time for each day

# Global variables
USAGE="daily_use.sh extension start|end|check [limit]"
LOGTAG="phonepatch[daily_use]"
SPOOLDIR=/var/spool/asterisk/phonepatch
LOCKEXEC="lockfile -1 -r2 -l5 -s2"
SPOOLNAME="limit-"
DATE=$(date +%Y%m%d)
SEC=$(date +%s)
START_TAG="start"
END_TAG="end"

##############
info() { echo $@ 1>&2; }

##############
log() { info $@; logger -t $LOGTAG $@; }

##############
error() { 
	log $@
	remove_lockfile $LOCKFILE
	exit 1
}

##############
get_usage()
{
	# $1: file
	CURRENT=0
	STATE="start"
	TEMP=$(tempfile)
	cat $1 | grep ^$DATE > $TEMP
	while read LINE; do
		OPT=$(echo $LINE | awk '{print $3}')
		if [ "$OPT" = $START_TAG ]; then
			SEC1=$(echo $LINE | awk '{print $2}')
			STATE="end"
		elif [ "$STATE" = "end" -a "$OPT" = $END_TAG ]; then
			SEC2=$(echo $LINE | awk '{print $2}')
			((CURRENT=CURRENT+SEC2-SEC1))
			STATE="start"
		fi
	done < $TEMP
	rm -f $TEMP
	info "get_usage: $CURRENT"
	RET=$CURRENT
}

####################
check_usage()
{
	FILE=$1
	LIMIT=$2
	get_usage $FILE
	if [ $RET -gt $LIMIT ]; then 
		TEXT="$DATE $SEC denied ($RET > $LIMIT)"
		echo "$TEXT" >> $FILE
		log "$EXTENSION - $TEXT"
		info "daily usage reached: $RET > $LIMIT"
		return 1
	fi
	return 0
}

#############
create_lockfile() { $LOCKEXEC $1; }
remove_lockfile() { [ ! -z "$1" ] && rm -f $1; }

################
# Main
################

# Get parameters
EXTENSION=$1
OP=$2
LIMIT=$3

# Sanity checks
if [ -z "$EXTENSION" -o -z "$OP" ]; then error "usage: $USAGE"; fi
if [ "$OP" != "end" -a -z "$LIMIT" ]; then error "usage: $USAGE"; fi
if [ ! -d $SPOOLDIR ]; then error "spool directory not found: $SPOOLDIR"; fi
FILE="$SPOOLDIR/$SPOOLNAME$EXTENSION"
LOCKFILE="$FILE.lock"

# Lock the spool file
create_lockfile $LOCKFILE

# info
info "spool file: $FILE"
info "lock file: $LOCKFILE"

# Create spool file if not found
if [ ! -e $FILE ]; then touch $FILE; fi

# Available operations = start | end | check
RETVAL=0
if [ "$OP" = "start" ]; then 
	if check_usage $FILE $LIMIT; then 
		TEXT="$DATE $SEC $START_TAG"
		echo "$TEXT" >> $FILE
		log "$EXTENSION - $TEXT"
	else RETVAL=1; fi
elif [ "$OP" = "end" ]; then 
	TEXT="$DATE $SEC $END_TAG"
	echo "$TEXT" >> $FILE
	log "$EXTENSION - $TEXT"
elif [ "$OP" = "check" ]; then 
	if ! check_usage $FILE $LIMIT; then RETVAL=1; fi
else
	error "option not supported: $OP"
	RETVAL=1
fi

remove_lockfile $LOCKFILE
info "done"
exit $RETVAL