#!/usr/bin/python

# This file is part of asterisk-phonepatch

# Copyright (C) 2006 Arnau Sanchez
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License or any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

# Standard Python modules
import sys, os, optparse
import signal, select
import errno

# External phonepatch modules
sys.path.append("/usr/lib/asterisk-phonepatch")
import templateparser
import unixsocket

__version__ = "$Revision: 1.2 $"
__author__ = "Arnau Sanchez <arnau@ehas.org>"
__copyright__ = """Copyright (C) 2006 Arnau Sanchez <arnau@ehas.org>.
This code is distributed under the terms of the GNU General Public License."""

GLOBAL_SECTION = "general"
PIDFILE_DIR = "/var/run/asterisk"
DEFAULT_SECTION = None
OUTCALL_IDNAME = "_phonepatch_"

###############################
###############################
class AGIPhonepatch:		
	###################################
	def __init__(self, configuration, verbose=False):
		"""Configuration is a dictionary variable whose keys with sections:
		asterisk, soundcard, festival, telephony, dtmf, radio, incall and outcall
		"""
		self.configuration = configuration
		self.verbose = verbose
		self.modules_verbose = verbose
		self.set_state("phonepatch")

		self.data_size = 320
		self.phonepatch_extension = None
		self.phonepatch_phpconfig = None

		# Asterisk definitions
		self.pid = os.getpid()
		self.asterisk_fifo = os.path.join("/tmp/php_fifo_%d.raw"%self.pid)
		
		self.set_configuration_sections()

	####################################
	def signal_handler(self, signum, frame):
		"""Signal handler for:
		
		SIGHUP: Newer versions of asterisk send it to kill the AGI
		"""
		self.debug("signal_handler: received %s" %signum)
		
		if signum in (signal.SIGHUP, signal.SIGTERM, signal.SIGINT):
			if self.state == "daemon":
				self.debug("signal_handler: AGI phonepatch daemon killed with SIGHUP")
				#self.end_daemon()
				os._exit(0)
			self.debug("signal_handler: phonepatch stopped with SIGHUP")

	####################################
	def getconf(self, parameter, phpext=None):
		if not phpext:
			phpext = self.phonepatch_extension
		if phpext and phpext in self.phonepatch_extensions and parameter in self.configuration[phpext]:
			return self.configuration[phpext][parameter]
		phpconfig = self.phonepatch_phpconfig
		if phpconfig and phpconfig in self.phonepatch_configs and parameter in self.configuration[phpconfig]:
			return self.configuration[phpconfig][parameter]
		if self.phonepatch_global and parameter in self.configuration[self.phonepatch_global]:
			return self.configuration[self.phonepatch_global][parameter]
		if parameter in self.configuration[self.phonepatch_default]:
			value = self.configuration[self.phonepatch_default][parameter]
			self.debug("getconf: %s not defined, default returned: %s" %(parameter, value))
			return value
		self.debug("getconf: unknown parameter: %s" %parameter, 1)

	####################################
	def set_configuration_sections(self):
		"""Update configuration sections: phonepatch_conf, phonepatch_extension, phonepatch_global"""
		self.phonepatch_configs = []
		self.phonepatch_extensions = []
		self.phonepatch_global = None
		self.phonepatch_default = DEFAULT_SECTION
		for section in self.configuration:
			if section == GLOBAL_SECTION:
				self.phonepatch_global = section
			elif type(section) == str and section.find("phonepatch") == 0:
				self.phonepatch_configs.append(section)
			elif section != DEFAULT_SECTION:
				self.phonepatch_extensions.append(section)

	###############################
	def set_state(self, state):
		"""Set phonepatch state (incall/outcall/daemon)"""
		self.state = state
		if state: self.state_string = state
		else: self.state_string = ""

	###############################
	def debug(self, log, exit=None):
		"""Output debug lines in verbose mode"""
		if not self.verbose: return
		if exit: log = "fatal error - " + log
		log = "phonepatch[%s] - %s" %(self.state_string, log)
		self.secure_os(sys.stderr.write, log + "\n")
		self.secure_os(sys.stderr.flush)
		
		# Exit with code error if "exit" parameter given
		if exit is not None: 
			sys.exit(exit)

	###############################
	def secure_os(self, method, *args):
		while 1:
			try: data = method(*args)
			except IOError, e:
				if e.errno == errno.EINTR: continue
				else: raise
			else: break
		return data

	###############################
	def agi_command(self, command):
		"""Send an AGI command to asterisk (written to stdout and flushed)"""
		self.debug("agi_command: send AGI command: %s" %command)
		sc = command + "\n"
		self.secure_os(sys.stdout.write, sc)
		self.secure_os(sys.stdout.flush)
		
	###################
	def get_agi_keys(self, fd):
		"""Read the AGI keys given by asterisk (from stdin) -> dictionary"""
		self.debug ("get_agi_keys: reading AGI keys")
		
		# All AGI keys should begin with "agi_"
		begin_string = "agi_"
		
		# Init dictonary that will contain AGI keys
		agi_keys = {}
		while 1:
			line = fd.readline()
			if not line or not line.strip(): break
			line = line.strip()
			try: key, value = line.split(":")	
			except: continue
			if key.find(begin_string) != 0: continue
			key = key[len(begin_string):]
			agi_keys[key] = value.strip()
			self.debug("get_agi_keys: %s = %s" %(key, agi_keys[key]))
		
		self.agi_keys = agi_keys

			
	###################################
	def open_interface(self):
		"""Opens phonepatch interface between Asterisk and radio:
		
		- Soundcard -> Phonepatch FIFO -> Asterisk (GET DATA command)
		- Asterisk (file descriptor 3) > Soundcard
		"""
		fifo_file = self.asterisk_fifo
		# Way 1: soundcard -> php_fifo -> asterisk
		if os.path.exists(fifo_file):
			os.unlink(fifo_file)
		os.mkfifo(fifo_file)
		
		# GET DATA don't get extension file
		base_file = os.path.splitext(fifo_file)[0]
		command = 'GET DATA %s ""' %base_file
		
		try: self.agi_command(command)
		except: self.debug("open_interface: error sending AGI command: %s" %command); sys.exit(1)
			
		# Open fifo (asterisk input) for writing
		self.asterisk_in = open(fifo_file, "w")

		# Way 2: asterisk (channel 3) -> soundcard
		self.asterisk_out = 3

	###################################
	def daemon_connect(self, extension=None):
		self.debug(self.controlfile)
		try: s = unixsocket.get_unixsocket(self.controlfile)
		except: self.debug("daemon_connect: cannot connect to phonepatch control"); return
		if extension:
			s.send("incall|%s\n"%extension)
		else: s.send("outcall\n")
		response = s.recv(2)
		self.debug("daemon_connect: response = '%s'"%response)
		if response == "ok":
			self.debug("daemon_connect: server accepted connection")
			return s
		else: self.debug("daemon_connect: server denied access")
			
	###################################
	def bridge(self, fd1input, fd1output, fd2):
		while 1:
			s = select.select([fd1output, fd2], [], [])[0]
			if not s: break
			if fd1output in s:
				try: data = self.secure_os(os.read, fd1output, self.data_size)
				except: data = None
				if not data: break
				#self.draw_power(data)
				try: self.secure_os(os.write, fd2.fileno(), data)
				except: break
			if fd2 in s:
				try: data = self.secure_os(os.read, fd2.fileno(), self.data_size)
				except: data = None
				if not data: break
				#self.draw_power(data)
				try: self.secure_os(os.write, fd1input.fileno(), data)
				except: break
		fd2.close()
		

	###################################
	def draw_power(self, data):
		import audioop
		x=int(audioop.rms(data, 2) /20.0)
		if x > 40: self.debug("*"*x)
		
	###################################
	def daemon_call(self, extension=None):
		self.open_interface()
		dsocket = self.daemon_connect(extension)
		if not dsocket: 
			self.debug("daemon_call: daemon_connect() failed")
			return
		self.bridge(self.asterisk_in, self.asterisk_out, dsocket)
		self.close_interface()

	###################################
	def close_interface(self):
		"""Close asterisk interface (FIFO)"""
		self.asterisk_in.close()
		os.close(self.asterisk_out)
		os.unlink(self.asterisk_fifo)
		
	###################################
	def set_signals(self, signals):
		"""Bind a list of signal to default signal_handler"""
		for sig in signals:
			signal.signal(sig, self.signal_handler)

	####################################
	def init_php(self, phonepatch):	
		if not phonepatch: 
			self.debug("init_php: phonepatch name not given")
			return
		if phonepatch not in self.configuration: 
			self.debug("init_php: phonepatch name not found in configuration: %s" %phonepatch)
			return
		self.phonepatch_phpconfig = phonepatch
		self.pidfile = os.path.join(PIDFILE_DIR, self.phonepatch_phpconfig + ".pid")
		self.controlfile = os.path.join(PIDFILE_DIR, self.phonepatch_phpconfig + ".ctl")
		return phonepatch

	###################################
	def agicall(self, extension):
		"""Phonepatch acting as AGI"""
		self.set_state("agicall")
		self.get_agi_keys(sys.stdin)
		if extension not in self.configuration:
			self.debug("agicall: phonepatch extension not defined: %s" %extension, 1)
		self.phonepatch_extension = extension
		phonepatch = self.getconf("phonepatch")
		if not phonepatch:
			if not self.phonepatch_configs:
				self.debug("agicall: no phonepatchs defined", 1)
			phonepatch = self.phonepatch_configs[0]
			self.debug("agicall: no phonepatchs defined, selecting first: %s" %phonepatch)
		if not self.init_php(phonepatch): 
			self.debug("agicall: error in phonepatch init", 1)
			return
		acode = self.agi_keys.get("accountcode")
		if not acode: 
			self.debug("agicall: accountcode not given in AGI, assume incall")
		if acode and acode.find(OUTCALL_IDNAME) == 0: 
			self.call()
		else: self.call(extension)
		
	###################################
	def call(self, extension=None):
		"""Phonepatch to make/receive outgoing/ingoing calls"""
		direction = {False: "outcall", True: "incall"}[extension is not None]
		self.set_state(direction)
		if not self.getconf(direction):
			self.debug("call: %s disabled for extension: %s" %(direction, self.phonepatch_extension))
			return			
		self.daemon_call(extension=extension)
		self.debug("call: %s ended" %direction)

###################################
def error(line):
	sys.stderr.write(line+"\n")
	sys.stderr.flush()
	
###################################
def main():
	usage = """
phonepatch.agi [options] extension_name

AGI (Asterisk Gateway Interface) to control the radio phonepatch"""
	
	default_template = "/usr/share/asterisk-phonepatch/phonepatch.conf.template"
	default_configuration = "/etc/asterisk/phonepatch.conf"
	
	optpar = optparse.OptionParser(usage)
	optpar.add_option('-q', '--quiet', dest='verbose', default = True, action='store_false', help = 'Be quiet (disable verbose mode)')
	optpar.add_option('-f', '--configuration-file',  dest='configuration_file', type = "string", default = default_configuration, help = 'Use configuration file')

	options, args = optpar.parse_args()
	
	config = templateparser.Parser(verbose = True)
	config.read_template(default_template)
	configuration = config.read_configuration(options.configuration_file)
		
	if not args: error("fatal error: need a argument"); sys.exit(1)
	extname = args[0]
	php = AGIPhonepatch(configuration, verbose=options.verbose)
	php.agicall(extname)
	sys.exit(0)

##############################
## MAIN
#################

if __name__ == "__main__":
	main()
