#!/usr/bin/python

# This file is part of asterisk-phonepatch

# Copyright (C) 2006 Arnau Sanchez
#
# Asterisk-phonepatch is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License or any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

# Standard Python modules
import os, sys, struct, math, struct
import numarray,optparse, FFT

__version__ = "$Revision: 1.2 $"
__author__ = "Arnau Sanchez <arnau@ehas.org>"
__depends__ = ['OSSAudioDev', 'FFT', 'Numeric-Extension', 'Python-2.4']
__copyright__ = """Copyright (C) 2006 Arnau Sanchez <arnau@ehas.org>.
This code is distributed under the terms of the GNU General Public License."""

CTCSS_FREQS =[67.0, 69.3, 71.9, 74.4, 77.0, 79.7, 82.5, 85.4, 88.5, 91.5, 94.8, 97.4, 100.0,\
        103.5,  107.2, 110.9, 114.8, 118.8, 123.0, 127.3, 131.8, 136.5, 141.3, 146.2, 151.4, \
        156.7, 162.2, 167.9, 173.8, 179.9, 186.2, 192.8, 203.5, 206.5,  210.7, 218.1, 225.7, \
        229.2, 233.6, 241.8, 250.3, 254.1]

MIN_POWER = 0.0005
UPFACTOR = 1
DOWNFACTOR = 0.5
OVERLAPPING = 0.75

#########################
class Generator:
	#########################
	def __init__(self, samplerate, samplewidth):
		self.samplerate = samplerate
		self.samplewidth = samplewidth
		self.sinindex = 0
		self.samplemax = 2.0**(8*samplewidth) / 2.0

	#########################
	def generate(self, length, amplitude, freq):
		start = self.sinindex
		nbuffer = length
		anbuffer = nbuffer / self.samplewidth
		ctcss_signal = [self.samplemax*amplitude*math.sin(2*math.pi*freq*x/self.samplerate) \
			for x in range(start, start+anbuffer)]
		self.sinindex += anbuffer
		format = "<%dh" %anbuffer
		return struct.pack(format, *ctcss_signal)

#########################
class Decoder:
	#########################
	def __init__(self, samplerate, samplewidth, mintime):
		if samplerate < 8000: 
			raise ValueError, "Samplerate must be 8000sps or more: %s" %samplerate
		if samplewidth < 1 or samplewidth > 2: 
			raise ValueError, "Sample width not available: %s" %samplewidth
		self.samplerate = samplerate
		self.samplewidth = samplewidth
		
		self.fftlength = samplerate / 2
		self.fftlength = int(2**math.ceil(math.log(self.fftlength, 2)))
		self.samplemax = 2.0**(8*samplewidth) / 2.0
		self.buffer = ""
		self.tone_detected = self.tone_current = None
		self.ntone = 0
		self.upfactor = UPFACTOR
		self.downfactor = DOWNFACTOR
		self.overlapping = OVERLAPPING
		self.threshold = (samplerate / self.fftlength) * float(mintime) / (1-self.overlapping)
		self.min_freq = CTCSS_FREQS[0] - (CTCSS_FREQS[1] - CTCSS_FREQS[0])
		self.max_freq = CTCSS_FREQS[-1] + (CTCSS_FREQS[-1] - CTCSS_FREQS[-2])
	

	#########################
	def get_tone(self):
		return self.tone_detected

	#########################
	def decode_buffer(self, buffer):
		self.buffer += buffer
		length = self.samplewidth * self.fftlength
		advance = int(length * (1 - self.overlapping))
		while len(self.buffer) >= length: 
			buffer = self.buffer[:length]
			self.buffer = self.buffer[advance:]
			format = {1: "b", 2: "h"}[self.samplewidth]
			window = struct.unpack("%d%s" %(len(buffer)/self.samplewidth, format), buffer)
			fft = FFT.real_fft(window)
			begin = int(CTCSS_FREQS[0] * self.fftlength / self.samplerate) - 1
			end = int(CTCSS_FREQS[-1] * self.fftlength / self.samplerate) +1
			out = [(abs(fft[index]), index) for index in range(begin, end) if index != 0]
			out.sort()
			out.reverse()
			maxpower, maxindex = out[0]
			maxpower /= self.fftlength * self.samplemax
			freq = self.samplerate * float(maxindex) / self.fftlength
			if maxpower < MIN_POWER:
				self.tone_detected = self.tone_current = None
				self.ntone = 0
				continue
			mindiff = CTCSS_FREQS[-1]
			for f in CTCSS_FREQS:
				diff = abs(freq - f)
				if diff < mindiff:
					mindiff = diff
					minfreq = f
				else: break
			if self.tone_current == minfreq:
				self.ntone += self.upfactor
				if self.ntone >= self.threshold:
					self.ntone = self.threshold
					self.tone_detected = minfreq
			else:
				self.ntone -= self.downfactor
				if self.ntone < 0:
					self.tone_current = minfreq
					self.ntone = self.upfactor
					self.tone_detected = None



###########################
def main():
	usage = """
	ctcss.py [options]: CTCSS Decoder
	
	You must activate a generator or decoding option"""
	
	parser = optparse.OptionParser(usage)
	
	parser.add_option('-s', '--samplerate', dest='samplerate', default = 8000, metavar='SPS', type='int', help = 'Set sampling rate')
	parser.add_option('-w', '--samplewidth', dest='samplewidth', default = 2, metavar='BYTES', type='int', help = 'Set sample width')
	parser.add_option('-b', '--buffersize', dest='buffersize', default = 1024, metavar = "BYTES", type = int, help = 'Buffer size for input/output')
	parser.add_option('-m', '--mintime', dest='mintime', default = 1.0, metavar = "SECONDS", type = float, help = 'Threshold detection time')

	options, args = parser.parse_args()
	dec = Decoder(options.samplerate, options.samplewidth, options.mintime)	
	oldtone = None
	dec.decode_buffer("\x00" * 10000)
	while 1:
		buffer = os.read(0, options.buffersize)
		if not buffer: break
		dec.decode_buffer(buffer)
		tone = dec.get_tone()
		if tone != oldtone: 
			sys.stdout.write(str(tone) + "\n")
			sys.stdout.flush()
			oldtone = tone	

#########
############
if __name__ == "__main__":
	main()
