#!/usr/bin/python

# This file is part of asterisk-phonepatch

# Copyright (C) 2006 Arnau Sanchez
#
# Asterisk-phonepatch is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License or any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

import os, sys, re, array
import optparse, fcntl, math
import ossaudiodev as oss

## TODO:

## Implement half/full-duplex mode

__version__ = "$Revision: 1.2 $"
__author__ = "Arnau Sanchez <arnau@ehas.org>"
__depends__ = ['OSSAudioDev', 'Python-2.3']
__copyright__ = """Copyright (C) 2006 Arnau Sanchez <arnau@ehas.org>.
This code is distributed under the terms of the GNU General Public License."""
	
###
def create_dict(**args): return args
	
###############################################
class Soundcard:
	"""High-level wrapping for OSS (ossaudiodev)
	
	Default mode is full-duplex with CD quality (stereo, 44100 sps, s16_le: 16-bits 
	signed little-endian samples)"""

	#####
	## Generic functions
	####################################################	
	
	##################################################
	def __init__(self, **args):
		"""Open soundcard with the following args:
		
		channels: 1 or 2.
		samplerate: soundcard rate
		sampleformat: u8 | s8 | s16_le, u16_le, s16_be, u16_be, alaw, ulaw
		mode: "r" | "w" | "rw" (read, write or read/write modes)
		nonblock: whether writes to soundcard block (only OSS)
		fullduplex: full-duplex (send and write on same time) operation
		fragment_size: buffer fragment size (only OSS)
		"""
		self.channels = int(args.get("channels", 2))
		self.samplerate = int(args.get("samplerate", 44100))
		self.sampleformatstr = str(args.get("sampleformat", "s16_le")).lower()
		try: self.sampleformat = getattr(oss, "AFMT_" + self.sampleformatstr.upper())
		except: raise NameError, "Sample format not supported: %s" %self.sampleformatstr
		self.fullduplex = bool(args.get("fullduplex", True))
		self.fragment_size = int(args.get("fragment_size", 0))
		self.nonblock = bool(args.get("nonblock", False))
		self.mode = str(args.get("mode", "rw"))
		self.opened = False
		self.device = str(args.get("device", "/dev/dsp"))
		self.open()

	##################################################
	def getoptions(self):
		return create_dict(device = self.device, channels = self.channels, \
			samplerate = self.samplerate, sampleformat = self.sampleformatstr, \
			fullduplex = self.fullduplex, mode = self.mode, fragment_size = self.fragment_size)
			
	##################################################
	def read(self, max):
		if not self.opened:
			raise IOError, "cannot read from %s: device is not opened" %self.device
		if "r" not in self.mode:
			raise IOError, "cannot read from %s: device not opened in read mode" %self.device
		return self.fd.read(max)

	##################################################
	def write(self, buffer):
		if not self.opened:
			raise IOError, "cannot write to %s: device is not opened" %self.device
		if "w" not in self.mode:
			raise IOError, "cannot write to %s, device not opened in write mode" %self.device
		return self.fd.write(buffer)

	##################################################
	def open(self):
		if  self.opened:
			raise IOError, "cannot open %s: device already opened" %self.device
		self.opened = True
		self.fd = oss.open(self.device, self.mode)
		
		# Set fragment size operation is not supported by ossaudiodev, do with ioctl
		if self.fragment_size:
			arg = 0x7FFF0000 + int(math.log(self.fragment_size, 2))
			fragment = array.array('L', [arg])
			fcntl.ioctl(self.fd, oss.SNDCTL_DSP_SETFRAGMENT, fragment, 1)
		self.fd.setparameters(self.sampleformat, self.channels, self.samplerate)
		if self.nonblock: 
			self.fd.nonblock()

	##################################################
	def close(self):
		if not self.opened:
			raise IOError, "cannot close %s: device not opened" %self.device
		self.opened = False
		self.fd.close()

	##################################################
	def fileno(self):
		return self.fd.fileno()

	##################################################
	def obufcount(self):
		return self.fd.obufcount()

	##################################################
	def sync(self):
		return self.fd.sync()


###########################
def show_options(sc):
	for parameter, value in sc.getoptions().items():
		sys.stderr.write("%s = %s\n" %(parameter.replace("_", " "), value))
		sys.stderr.flush()

###########################
def main():
	usage = """
	soundcard.py [options] play | record
	
	Soundcard RAW audio player and recorder"""
	
	parser = optparse.OptionParser(usage)

	parser.add_option('-v', '--verbose', dest='verbose', default = False, action='store_true', help = 'enable verbose mode')
	parser.add_option('-d', '--device', dest='device', default = "", metavar='DEVICE', type='str', help = 'audio device')
	parser.add_option('-s', '--samplerate', dest='samplerate', default = 44100, metavar='SPS', type='int', help = 'sampling rate')
	parser.add_option('-f', '--sampleformat', dest='sampleformat', default = "s16_le", metavar='AUDIO_FORMAT', type='string', help = 'sample format (u8/s8/s16_le/u16_le/s16_be/u16_be/a_law/u_law)')
	parser.add_option('-c', '--channels', dest='channels', default = "2", metavar='NUMBER', type='int', help = 'audio channels')
	parser.add_option('-b', '--buffersize', dest='buffersize', default = 1024, metavar = "BYTES", type = int, help = 'Buffer size for input/output')

	options, args = parser.parse_args()

	if len(args) != 1: parser.print_help(); sys.exit(1)
	
	command = args[0]
	options.device = options.device or "/dev/dsp"
	
	if command == "play":
		sc = Soundcard(device = options.device, channels = options.channels, \
				mode = "w", samplerate = options.samplerate, \
				sampleformat = options.sampleformat)
		if options.verbose: show_options(sc)
		while 1:
			buffer = os.read(0, options.buffersize)
			if not buffer: break
			sc.write(buffer)
		sc.close()
				
	elif command == "record":
		sc = Soundcard(device = options.device, channels = options.channels, \
				mode = "r", samplerate = options.samplerate, \
				sampleformat = options.sampleformat)
		print sc.fd
		if options.verbose: show_options(sc)
		while 1:
			try: os.write(1, sc.read(options.buffersize))
			except: break
		sc.close()
			
	sys.exit(0)

############################
if __name__ == "__main__":
	main()