#!/usr/bin/python

# asterisk-phonepatch - Phonepatch for the Asterisk PBX

# Copyright (C) 2006 Arnau Sanchez
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License or any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.


# Standard Python modules
import sys, os, optparse
import time, select, re

# Addons Python modules
import serial, parallel

# External phonepatch modules
import processcomm

MODES = ("serial", "parallel", "command")
STATES = ("on", "off")
ONOFF = {False: "off", True: "on"}

###########################
def debug(text, verbose = True, exit = None):
	"""Print debug information to standard errror if verbose enabled"""
	if not verbose: return
	sys.stderr.write(text + "\n")
	sys.stderr.flush()
	if exit != None: 
		sys.exit(exit)

###################################
###################################
class RadioControl:
	"""Control PTT (Push-to-Talk) and carrier detection through the serial port, the
	parallel port, or a external command"""
	###################################
	def __init__(self, mode, device, command_options = None):
		"""Returns a RadioControl instance.
		
		mode -- "serial" | "parallel" | "command"
		device -- serial/parallel device or path to command
		command_options -- in command mode, enter a string with: PttOn, PttOff, GetCarrier, GetCarrierRetRegExp
		"""
		if mode not in MODES:
			raise NameError, "Mode error: %s. Available modes: %s" %(mode, ", ".join(list(MODES)))
		self.mode = mode
		self.device = device
		self.command_options = command_options
		if mode == "serial":
			self.serial = serial.Serial(device)
			self.serial.setDTR(1)
		elif mode == "parallel":
			self.parallel = parallel.Parallel("/dev/parport0")
			self.parallel.setDataStrobe(1)
		elif mode == "command":
			if self.command_options["get_carrier"]:
				try: self.onstring = re.findall("\((.*)\)", self.command_options["get_carrier_response"])[0].split("|")[0]
				except: raise ValueError, "Syntax error on get_carrier_response: %s" %self.command_options["get_carrier_response"]
			self.command = processcomm.Popen(device)
			try: self.command.read(timeout=0.5)
			except: 
				try: self.command.close()
				except: pass
				self.command = None

			if not self.command:
				raise IOError, "Command could not be started: %s" %self.device
					
	###################################
	def get_ptt(self):
		"""Get PTT state. It's not really readed from device, which is not 
		possible, just return the class variable stored by set_ptt()"""
		if self.mode == None: 
			raise IOError, "RadioControl is not opened"
		return self.ptt

	###################################
	def get_carrier(self, timeout = 0.5):
		"""Get carrier detection state"""
		if self.mode == None: 
			raise IOError, "RadioControl is not opened"
		if self.mode == "serial":
			return self.serial.getCD()
		elif self.mode == "parallel":
			return self.parallel.getInBusy()
		elif self.mode == "command":
			self.command.write(self.command_options["get_carrier"] + "\n")
			maxtime = time.time() + timeout
			while 1:
				try: response = self.command.read()
				except: raise IOError, "Command closed its descriptor"
				for line in response.splitlines():
					value = re.findall(self.command_options["get_carrier_response"], line)
					if value: return self.onstring == value[0]
				now = time.time()
				if now > maxtime: break
			raise IOError, "Cannot get carrier-detection state from command"
		
	###################################
	def set_ptt(self, state, timeout = 0.5):
		"""Set PTT state (True/False)"""
		if self.mode == None: 
			raise IOError, "RadioControl is not opened"
		self.ptt = state
		if self.mode == "serial":
			return self.serial.setRTS(state)
		elif self.mode == "parallel":
			return self.parallel.setData(state & 1)
		elif self.mode == "command":
			key = "set_ptt_%s" %(ONOFF[state])
			self.command.write(self.command_options[key]  + "\n")
		
	###################################
	def close(self):
		if self.mode == None: 
			raise IOError, "RadioControl is not opened"
		if self.mode == "serial":
			self.serial.close()
		elif self.mode == "parallel":
			del self.parallel
		elif self.mode == "command":
			self.command.close()
		self.mode = None

###################################
def output(text):
	sys.stdout.write(text + "\n")
	sys.stdout.flush()

###################################
def server(radio, verbose):
	"""Allowed commands:
	set ptt on: Set PTT on (return "done" if succesful)
	set ptt off: Set PTT off (return "done" if succesful)
	get carrier: Return Carrier-Detection state (return "carrier: 0" or "carrier: 1" if succesful)"""
	debug("starting server mode", verbose)
	while 1:
		line = sys.stdin.readline()
		if not line: break
		line = line.strip()
		if line.find("set ptt ") == 0:
			if line == "set ptt on": radio.set_ptt(True)
			elif line == "set ptt off": radio.set_ptt(False)
			else: output("syntax error: %s" %line); continue
			output("done: %s" %line)
		elif line == "get carrier":
			ret = radio.get_carrier()
			output("get carrier: %d" %int(ret))
		else: output("unknown command: %s" %line)
	debug("end of server mode", verbose)

###################################
def main():
	usage = """radio-control.py [options]

Control radio PTT (output) and carrier detection (input) 
using serial or parallel port:

Serial port -- RTS for PTT and CTS for Carrier-Detection.
Parallel port -- D0 for PTT and Busy for Carrier-Detecion.

Server mode accepts the following commands:

set ptt on --  Set PTT on
set ptt off --  Set PTT off
get carrier --  Get Carrier-Detection state. Returns: "get carrier: 0|1" """
	
	optpar = optparse.OptionParser(usage)
	optpar.add_option('-v', '--verbose', dest='verbose', default = False, action='store_true', help = 'Be verbose')
	optpar.add_option('-d', '--device',  dest='device', type = "string", default = "", metavar = 'DEVICE', help = 'Device file')
	optpar.add_option('-m', '--port-mode',  dest='mode', type = "string", default = "", metavar = 'MODE', help = 'Port mode: %s' %" | ". join(list(MODES)))
	optpar.add_option('-p', '--ptt',  dest='ptt', type = "string", default = "", metavar = 'STATE', help = 'Set new PTT state (on|off)')
	optpar.add_option('-w', '--wait-time',  dest='wait', type = "float", default = None, metavar = 'SECONDS', help = 'On PTT mode, time to wait before exit')
	optpar.add_option('-c', '--carrier-detection',  dest='carrier',  default = False, action = 'store_true', help = 'Get carrier-detection state')
	optpar.add_option('-s', '--server',  dest='server',  default = False, action = 'store_true', help = 'Start in server mode')
	optpar.add_option('-o', '--command-options',  dest='command_options', type = "string", default = "", metavar = 'OPTIONS', help = 'String for command mode: ptt-on,ptt-off,get-carrier,get-carrier-response-regexp')
	
	options, args = optpar.parse_args()
	verbose = options.verbose
	
	if not options.mode: 
		options.mode = "serial"
	if options.mode not in MODES:
		optpar.print_help()
		debug("\nSupported port modes: %s" %", ".join(list(MODES)), exit = 1)
	if options.ptt and options.ptt not in STATES:
		optpar.print_help()
		debug("\nSupported PTT states: %s" % ", ".join(list(STATES)), exit = 1)
	if not options.device:
		if options.mode == "serial": options.device = "/dev/ttyS0"
		elif options.mode == "serial":  options.device = "/dev/parport0"
		elif options.mode == "command": 
			optpar.print_help()
			debug("\nYou must specify a command (in device option) for command mode", exit = 1)
	
	if options.mode == "command":
		if not options.command_options:
			optpar.print_help()
			debug("\nYou must specify command options for command mode (leave it void if not used)", exit = 1)
		try: ptton, pttoff, getcarrier,getcarrierreturn = [x.strip() for x in options.command_options.split(",")]
		except: optpar.print_help(); debug("\nSyntax error on command options:%s" %options.command_options, exit = 1)
		command_options = {"set_ptt_on": ptton, "set_ptt_off": pttoff, "get_carrier": getcarrier, "get_carrier_response": getcarrierreturn}
	else:
		command_options = None
		
	debug("options - verbose mode: on", verbose)
	debug("options - mode: %s" %options.mode, verbose)
	debug("options - device: %s" %options.device, verbose)
	debug("options - wait time: %s" %str(options.wait), verbose)

	if not options.ptt and not options.carrier and not options.server:
		optpar.print_help()
		sys.exit(1)
		
	debug("opening device: %s" %options.device, verbose)
	rc = RadioControl(options.mode, options.device, command_options)
	debug("device opened: %s" %options.device, verbose)

	if options.server: 
		server(rc, verbose)
		rc.close()
		sys.exit(0)

	if options.ptt:
		ptt = (options.ptt == "on")
		rc.set_ptt(ptt)
		debug("ptt set: %s" %options.ptt, verbose)
		if not options.wait: 
			debug("press enter to exit")
			raw_input()
		elif options.wait:
			debug("waiting %d seconds" %options.wait, verbose)
			time.sleep(options.wait)
		
	if options.carrier:
		try: state = rc.get_carrier()
		except: output("error getting carrier state")
		else: output("get carrier: %d" %int(state))
	
	rc.close()
	sys.exit(0)

##############################
## MAIN
#################

if __name__ == "__main__":
	main()