/*
 * versioncontrol_monotone.cpp
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string>
#include <iostream>
#include <sstream>

#include <boost/tokenizer.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/operations.hpp>

#include "utils.h"
#include "cctools.h"
#include "versioncontrol_monotone.h"

namespace apso {


/**
 * Returns the path to the working copy.
 *
 * Commands are executed in the last set path. Use this method to find out which
 * working copy is being used.
 *
 * @return The path to the working copy.
 */
Path&
VersionControlMonotone::get_wc_path() {
	return wc_path;
}

/**
 * Sets the path tothe working copy.
 *
 * Commands are executed in the last set path, so changing the path will make the
 * object work on another working copy.
 *
 * @param path The path to the new working copy.
 */
void
VersionControlMonotone::set_wc_path(const Path& path) {
	wc_path = path;
}

/**
 * Executes one single monotone command.
 *
 * Receives three strings:
 * - The monotone command (without the "monotone" binary). Example: " add myfile.txt"
 * - An input string (used as stdin when called monotone)
 * - An output string (get the output from monotone there)
 *
 * Error output is not shown.
 *
 * FIXME:
 * - The timeout is set to zero. Is that reasonable?
 * - Exceptions should be reviewed
 */
void
VersionControlMonotone::monotone_execute(std::string& cmd,
				std::string& in,
				std::string& out) {
	int ret;
	Path old_dir = boost::filesystem::initial_path();
	std::string err;

	ret = chdir(wc_path.string().c_str());
	if (ret != 0)
		throw std::runtime_error ("Can't change directory into monotone working copy!"); // Bad
	ret = cctools::system (monotone_binary.string() 
			      + " -d " + db_path.string() 
			      + " -k " + private_key
			      + cmd, in, out, err, 0);
        if (ret != 0)
                throw std::runtime_error ("Error:\n" + err + "\nCommand:\n" + cmd);
	ret = chdir (old_dir.string().c_str());
	if (ret != 0)
		throw std::runtime_error ("Can't change directory out of monotone working copy!"); // Wrong!
}

void
VersionControlMonotone::parse_patch (std::istream& instream, std::ostream& outstream, std::string& line) {
	std::string err;
	std::string in;
	std::string out;
	std::string cmd;
	std::string filename = line.substr(7,line.size()-8);
	
	getline(instream,line);
	std::string file_id1 = line.substr(7,line.size()-8);
	
	getline(instream,line);
	std::string file_id2 = line.substr(7,line.size()-8);

	if (file_id1.empty())
		cmd += " automate packet_for_fdata " + file_id2;
	else
		cmd += " automate packet_for_fdelta " + file_id1 + " " + file_id2;
	
	monotone_execute(cmd, in, out);
	outstream << out;
}


void
VersionControlMonotone::parse_add_file (std::istream& in, std::ostream& out, std::string& line) {
	std::string filename = line.substr(10,line.size()-11);
	getline(in,line);
	std::string file_id2 = line.substr(10,line.size()-11);
	std::string cmd = " automate packet_for_fdata " + file_id2;
	std::string s;
        std::string output;
        monotone_execute(cmd, s, output);
	out << output;
	std::cerr << "Added file " + filename + "\n";
}

void
VersionControlMonotone::parse_add_parent(std::istream& in, std::list<std::string>& parents, std::string& line) {
	std::string parent_id = line.substr(14,line.size()-15);
	getline(in,line);
	parents.push_back(parent_id);
}

bool
VersionControlMonotone::begins_with(std::string& a, std::string b) {
	if (a.substr(0,b.size()).compare(b) == 0)
		return true;
	else
		return false;
}

/**
 * Parses the output of get_revision.
 *
 * This will call monotone to get the packets for the items in the
 * revision. The packets will be added to the out stream.
 *
 * @param in An instream where the output of get_revision can be read
 * @param out An outstream where the packets will be written
 * @param parents A list of strings with the revision IDs of all parents to this revision
 */
void
VersionControlMonotone::parse(std::istream& in, std::ostream& out, std::list<std::string>& parents) {
	
	std::string line;

	while(getline(in,line)) {
		if (line.empty())
			continue;
		if (begins_with(line,"patch")){
			parse_patch(in, out, line);
		}
		if (begins_with(line,"add_file")){
			parse_add_file(in,out,line);
		}
		if (begins_with(line,"old_revision")) {
			parse_add_parent(in,parents,line);
		}
	}

}


/**
 * Extracts the changed data in one revision from the database.
 *
 * This will get the file deltas (and file data, in the case of new files)
 * from a string. The output is represented as Monotone packets.
 *
 * @param in The output og get_revision
 * @return A string with the file deltas/data
 */
std::list<std::string>
VersionControlMonotone::get_delta_data(const std::string& in) {
	std::istringstream ins (in);
	std::ostringstream content;
	std::list<std::string> result;
	parse(ins, content, result);
	result.push_front(content.str());
	return result;
}

/**
 * Extrats a complete delta for one revision.
 *
 * The monotone packets will be stored in the Delta object content.
 * 
 * @param id The revision id.
 */
Delta_ptr
VersionControlMonotone::get_delta(const std::string& id) {
	std::string in;
	std::string out;
	std::string err;
	std::string cmd (" automate get_revision ");
	cmd += id;
	monotone_execute (cmd,in,out);
	std::list<std::string> result = get_delta_data(out);
	std::string content = result.front();
	Delta_ptr d (new Delta (content));
	d->set_id(id);
	result.pop_front();
	d->add_parents(result);
	return d;
}

/**
 * Gets the keys from one Monotone database.
 *
 * @return A string with packets representing the keys
 */
std::string
VersionControlMonotone::get_keys(std::string rev) {
	std::string in;
	std::string out;
	std::string cmd;
	cmd = " automate keys | grep name"; // FIXME: grep???
	monotone_execute (cmd,in,out);
	boost::tokenizer<boost::char_separator<char> >::iterator t;
	boost::char_separator<char> separator ("\n");
	boost::tokenizer<boost::char_separator<char> > tok(out);

	for(t=tok.begin(); t!=tok.end();++t)
		;
	
	std::string s ("");
	return s;

	
	//mtn pubkey key_id;


        // 1. monotone automate certs ID | grep "key"
        // 2. monotone automate keys | grep "name"
        // 3. ?
        // 4. monotone pubkey KEYID

}

/**
 * Extracts deltas from the repository.
 *
 * FIXME: stop using pointers
 * 
 * @param A shared_ptr to a list of pointers to strings, where the strings represent revision IDs
 */
boost::shared_ptr<std::list<Delta_ptr> >
VersionControlMonotone::get_deltas(const boost::shared_ptr<std::list<std::string> > ids) {
	boost::shared_ptr<std::list<Delta_ptr> > deltas
		(new std::list<Delta_ptr>);
	Delta_ptr delta;
	
	std::list <std::string>::iterator id;
	std::string in;
	std::string out;
	std::string cmd;
	std::string keys;

	for (id=ids->begin(); id!=ids->end(); id++) {
		delta = get_delta( *id );
		cmd = " automate packet_for_rdata " + *id;
		monotone_execute(cmd,in,out);
		delta->add(out);
		cmd = " automate packets_for_certs " + *id;
		monotone_execute(cmd,in,out);
		delta->add(out);
		delta->set_id(*id);
		keys = get_keys(*id);
		delta->add(keys);
		deltas->push_back(delta);
	}
	return deltas;
}


/**
 * Extracts the complete list of revision IDs from the repository, in topological order.
 *
 * @return A list of strings, each string being one ID of a revision, all in topological order.
 */
boost::shared_ptr<std::list<std::string> >
VersionControlMonotone::get_revision_ids() {
	boost::shared_ptr<std::list<std::string> > ids (new std::list <std::string>);
	std::string in;
	std::string out;
	std::string final;
	std::string cmd (" automate select i:");
	boost::tokenizer<boost::char_separator<char> >::iterator t;
	
	
	std::string toposort_cmd = *new std::string(" automate toposort -@-");
	monotone_execute(cmd,in,out);
	
	boost::char_separator<char> separator ("\n");
	boost::tokenizer<boost::char_separator<char> > tok(out);
	
	monotone_execute (toposort_cmd, out,final); // Put the output in toposort's input
	
	boost::tokenizer<boost::char_separator<char> > tok2(final);
	for(t=tok2.begin(); t!=tok2.end();++t) {
		ids->push_back(*new std::string(*t));
	}

	return ids;
}

/**
 * Accepts a list of deltas into the repository.
 *
 * This will be passed to "mtn read", so the deltas should be Monotone packets.
 *
 * @param deltas A list of Delta_ptr's with the deltas
 */
void
VersionControlMonotone::accept_deltas(const std::list<Delta_ptr>& deltas) {
	std::cout << "[ VCMonotone accept_deltas start ]\n";
	std::list<Delta_ptr>::const_iterator d;
	std::string in;
	std::string out;
	std::string cmd (" read ");
	for(d = deltas.begin(); d != deltas.end(); d++) {
		in += (*d)->get_string();
		in += "\n";
	}
	monotone_execute (cmd,in,out);
	std::cout << "[ VCMonotone accept_deltas end ]\n";
}


/**
 * Initializes a Monotone database
 */
void
VersionControlMonotone::init_db () {
	std::string in;
	std::string out;
	std::string err;
	{
		std::string cmd (" db init");
		int ret = cctools::system (monotone_binary.string() 
 				           + " -d " + db_path.string()
					   + " -k " + private_key
					   + cmd, in, out, err, 0);
		if (ret != 0)
			throw std::runtime_error ("Error:\n" + err + "\nCommand:\n" +
					          " -d " + db_path.string() + " -k " + private_key + cmd);
	}

}

/**
 * Checks out the (current) working copy.
 */
void
VersionControlMonotone::checkout () {
	std::string in;
	std::string out;
	std::string err;
	std::cout << "MTN: Issuing a checkout\n";
        {
		std::string cmd (" -b apso co " + wc_path.string());
		int ret = cctools::system (monotone_binary.string() 
				           + " -d " + db_path.string() 
					   + " -k " + private_key
					   + cmd, in, out, err, 0);
		if (ret != 0)
                        throw std::runtime_error ("Error:\n" + err + "\nCommand:\n" + 
						  " -d " + db_path.string() + " -k " + private_key + cmd);
        }
}

/**
 * Sets up the apso branch in the database, also creating the wc directory.
 */
void
VersionControlMonotone::setup () {
	std::string in;
        std::string out;
        std::string err;
	std::cout << "Setting up public database\n";
	{
		std::string cmd (" -b apso setup " + wc_path.string());
		int ret = cctools::system (monotone_binary.string() 
				           + " -d " + db_path.string()
					   + " -k " + private_key
					   + cmd, in, out, err, 0);
		if (ret != 0)
			throw std::runtime_error ("Error:\n" + err + "\nCommand:\n" +
						  " -d " + db_path.string() + " -k " + private_key + cmd);
	}
	add_directory ("deltas");
	add_directory ("users");
	commit();
}

/**
 * Constructor.
 *
 * This will:
 * - Check that the monotone binary exists, and the version is correct;
 * - Check that the database is OK;
 * - Check if the scratch working copy exists;
 * - Create the scratch working copy if it didn't exist.
 */
VersionControlMonotone::VersionControlMonotone(Path binary,
					       Path db,
					       Path wc,
					       std::string priv_key) {
	std::cout << "[ VersionControlMonotone start ]\n";
        monotone_binary = binary;
        db_path = boost::filesystem::complete(db);
        wc_path = boost::filesystem::complete(wc);
	std::cout << "db_path = " << db_path.string() << "\n";
	std::cout << "wc_path = " << wc_path.string() << "\n";
	std::cout << "private_key = " << priv_key << "\n";
	private_key = priv_key;


	if (!boost::filesystem::exists(binary)) {
		throw std::runtime_error ("Monotone binary does not exist");
	}
	
	if (!boost::filesystem::exists(db_path)) {
		std::cout << "Database didn't exist? " << db_path.string() << "\n";
		init_db();
	} 

	if (!boost::filesystem::exists(wc_path)) {
		std::cout << "Scratch directory didn't exist? " << wc_path.string() << "\n";
		checkout();
	}

	std::cout << "[ VersionControlMonotone end ]\n";
}


/**
 * Adds a file to version control.
 *
 * If the file does not exist, it will be created and left empty.
 *
 * @param path A path to the file to be added
 */
void
VersionControlMonotone::add_file(const Path& path) {
	std::string in;
        std::string out;
        std::string err;
        std::string cmd;
	// Touch the file:
	{
		int ret;
	        Path old_dir = boost::filesystem::initial_path();
        	std::string err;
	        ret = chdir(wc_path.string().c_str());
        	if (ret != 0) {
                	throw std::runtime_error ("Can't change directory into monotone working copy!"); // Bad
		}

		if (!exists(path)) {
			boost::filesystem::ofstream o (path);
			o.close();
		}
        
		ret = chdir (old_dir.string().c_str());
		if (ret != 0) 
			throw std::runtime_error ("Can't change directory out of monotone working copy!"); // Wrong
	}
	cmd += " add " + path.string();
        monotone_execute (cmd,in,out);
}

/**
 * Adds a file to version control, with content.
 *
 * If the file already exists, it will be overwritten.
 *
 * @param path A path to the file
 * @param content A bdata with the content to be put in the file
 */
void
VersionControlMonotone::add_file(const Path& path, const bdata& content) {
        std::string in;
        std::string out;
        std::string err;
        std::string cmd;
	Path complete_path (wc_path);
	complete_path /= path;
	if (exists(complete_path)) {
		std::cout << complete_path.string() << " exists -- overwriting!\n";
	}
	boost::filesystem::ofstream file_stream (complete_path);
	file_stream.write(content.get_data(), content.get_size());
	file_stream.close();
        cmd += " add " + path.string();
        monotone_execute (cmd,in,out);
}

/**
 * Adds a directory to version control.
 *
 * If the directory ddoes not exist, it will be created.
 * If the path points to a file, an exception is raised.
 *
 * @param path The path to the directory
 */
void
VersionControlMonotone::add_directory(const Path& path) {
	std::string in;
	std::string out;
	std::string err;
	std::string cmd;
	Path dir (wc_path);
	dir /= path;
	if (!exists(path)) {
		boost::filesystem::create_directory(dir);
	} else if (! boost::filesystem::is_directory(path)) {
		throw std::runtime_error ("I was told to add a directory, but it's a file!");
	}
	cmd += " add " + path.string();
	monotone_execute (cmd,in,out);
}


/**
 * Removes one directory, recursively.
 *
 * Also deletes the files and directory from the filesystem.
 *
 * @param path The path to the directory.
 */
void
VersionControlMonotone::remove_directory(const Path& path) {
        std::string in;
        std::string out;
        std::string err;
        std::string cmd;
        cmd += " drop --recursive --execute " + path.string();
        monotone_execute (cmd,in,out);
        boost::filesystem::remove_all(path);
}

/**
 * Removes one file from version control.
 *
 * Also deletes the file from the filesystem.
 *
 * @param path The path to the file.
 */
void
VersionControlMonotone::remove_file(const Path& path) {
	std::string in;
	std::string out;
	std::string err;
	std::string cmd;
	cmd += " drop --execute " + path.string();
	monotone_execute (cmd,in,out);
	boost::filesystem::remove(path);
}


/**
 * List all files that monotone is tracking.
 *
 * @param path The path of the subtree where the command will list the files.
 */
const std::list<Path_ptr>
VersionControlMonotone::list_known(const Path& path) {
	std::string in;
	std::string out;
	std::string err;
	std::string cmd (" list known " + path.string());
	monotone_execute (cmd,in,out);
	boost::shared_ptr<std::list<Path_ptr> > result (new std::list<Path_ptr>);
	
	boost::tokenizer<boost::char_separator<char> >::iterator t;
	boost::char_separator<char> separator ("\n");
	boost::tokenizer<boost::char_separator<char> > tok(out, separator);
	for(t=tok.begin(); t!=tok.end();++t) {
		Path_ptr file_path (new Path(*t));
		result->push_back(file_path);
	}
	return *result;
}

/**
 * List missing files from a working copy.
 *
 * FIXME: doesn't scale (should be a list of strings)
 *
 * @return A string with the list of files that are missing from the working copy.
 */
std::string
VersionControlMonotone::list_missing_files(const Path& path) {
	std::string in;
        std::string out;
        std::string err;
	std::string cmd (" list missing " + path.string());
	monotone_execute (cmd,in,out);
	return out;
}


/**
 * Finds all missing files in the scratch directory.
 *
 * FIXME: doesn't scale (should be a list of strings)
 *
 * @return A string with the list of files that are missing from the working copy.
 */
std::string
VersionControlMonotone::list_missing_files() {
        std::string in;
        std::string out;
        std::string err;
        std::string cmd (" list missing");
        monotone_execute (cmd,in,out);
        return out;
}

/**
 * Reverts (brings back) all missing files on the working copy.
 */
void
VersionControlMonotone::revert_missing() {
	std::string missing = list_missing_files();
	std::string in;
	std::string out;
	std::string err;
	std::string cmd (" revert " + missing);
	monotone_execute (cmd,in,out);
}

/**
 * Reverts (brings back) all missing files on a tree.
 *
 * @param path A Path to the tree.
 */
void
VersionControlMonotone::revert_missing(Path& path) {
        std::string missing = list_missing_files(path);
        std::string in;
        std::string out;
        std::string err;
        std::string cmd (" revert " + missing);
        monotone_execute (cmd,in,out);
}


/**
 * Updates the working copy.
 */
void
VersionControlMonotone::update_wc() {
	if (missing_files()) {
		revert_missing();
	}
	std::string in;
	std::string out;
	std::string err;
	std::string cmd (" up");
	monotone_execute (cmd,in,out);
}

/**
 * Updates one specific path in the working copy.
 *
 * The whole tree will be updated!
 *
 * @param path The path to the subtree to be updated.
 */
void
VersionControlMonotone::update_wc(Path& path) {
	if (missing_files(path)) {
		revert_missing(path);
	}
	std::string in;
	std::string out;
	std::string err;
	std::string cmd (" up ");
	monotone_execute (cmd,in,out);
}

/**
 * Checks if there are files missing from the tree.
 *
 * @param path A Path to the tree.
 */
bool
VersionControlMonotone::missing_files(const Path& path) {
        std::string in;
        std::string out;
        std::string err;
        std::string cmd (" list missing " + path.string());
        monotone_execute (cmd,in,out);
        if (out.size() > 0)
                return true;
        return false;
}


/**
 * Checks if there are files missing from the tree.
 */
bool
VersionControlMonotone::missing_files() {
	std::string in;
	std::string out;
	std::string err;
	std::string cmd (" list missing");
	monotone_execute (cmd,in,out);
	if (out.size() > 0)
		return true;
	return false;
}

/**
 * Checks is there are changes waiting for a commit.
 *
 * @param path The path to the subtree where this method will look for changes
 * @return True if changes were made to the subtree, but not commited
 */
bool
VersionControlMonotone::changes_to_commit(const Path& path) {
	std::string in;
	std::string out;
	std::string err;
	std::string cmd;
	cmd += " list changed " + path.string();
	monotone_execute (cmd,in,out);
	if (out.size() > 0)
		return true;
	return false;
}


/**
 * Checks is there are changes waiting for a commit.
 *
 * @return True if changes were made to the working copy, but not commited.
 */
bool
VersionControlMonotone::changes_to_commit() {
	std::string in;
        std::string out;
        std::string err;
        std::string cmd;
        cmd += " list changed";
        monotone_execute (cmd,in,out);
        if (out.size() > 0)
                return true;
        return false;
}

/**
 * Commits one path in the working copy to the database.
 *
 * If there are no changes made to the working copy, then nothing is done.
 *
 * @param path The path to be commited
 */
void
VersionControlMonotone::commit(const Path& path) {
	if (missing_files())
		throw std::runtime_error ("There are missinng files in the monotone working copy. Fix that first!");
	if (changes_to_commit(path)) {
		std::string in;
        	std::string out;
        	std::string err;
        	std::string cmd;
        	cmd += " ci " + path.string() + " -m 'Apso commit' ";
        	monotone_execute (cmd,in,out);
	}
}

/**
 * Commits the working copy to the database.
 *
 * If there are no changes made to the working copy, then nothing is done.
 */
void
VersionControlMonotone::commit() {
	if (missing_files())
		throw std::runtime_error ("There are missinng files in the monotone working copy. Fix that first!");
	if (changes_to_commit()) {
		std::string in;
        	std::string out;
        	std::string err;
        	std::string cmd;
        	cmd += " ci -m 'Apso commit'";
        	monotone_execute (cmd,in,out);
	}
}

}
