/*
 * pubrepository.cpp
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <list>
#include <string>
#include <typeinfo>
#include <iostream>
#include <fstream>


#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/operations.hpp>


#include "delta.h"
#include "pubrepository.h"

namespace apso {


/**
 * Decrypts deltas, given their rev id and content.
 *
 * @param content The content to be decrypted
 * @return A string with the decrypted delta
 */
bdata_ptr
PubRepository::decrypt (const bdata& content, Key& k) {
	bdata_ptr enc = crypto->sym_dec(content,k);
	return enc;
}

/**
 * Encrypts deltas, given their rev id and encrypted content.
 *
 * @param content The clear content
 * @return A string with the encrypted delta.
 */
bdata_ptr
PubRepository::encrypt (const bdata& content, Key& k) {
	bdata_ptr encrypted = crypto->sym_enc(content,k);
	return encrypted;
}


/**
 * Includes revision in the index.
 *
 * FIXME: opens and closes the file all the time.
 *
 * @param id A string with the revision id.
 */
void
PubRepository::index_delta(const std::string& id) {
	Path path (scratch);
	path /= "deltas_index";
	boost::filesystem::ofstream out (path, std::ofstream::out | std::ofstream::app);
	out << id << "\n";
	out.close();
}

/**
 * Saves the parents of a Delta.
 *
 * FIXME: Should this go into Delta?
 *
 * @param d A Delta_ptr, pointing to the Delta whose parents will be written.
 */
void
PubRepository::write_parents(Delta_ptr d) {
	Path path (deltas_dep_dir);
	path /= d->get_id();
	Path complete_path (scratch);
	complete_path /= path;
	boost::filesystem::ofstream out (complete_path);
	std::list<std::string> parents (d->get_parents());
	std::list<std::string>::const_iterator parent;
	for (parent = parents.begin(); parent != parents.end(); parent++)
		out << *parent << "\n";
	out.close();
	vc->add_file (path); // FIXME: this is not good. Should be close to the commit.
}

/**
 * Gets a list of deltas and includes them in the database.
 *
 * The list should be in topological order!
 *
 * @param deltas a list of Delta_ptr's that will be injected into this database.
 */
void
PubRepository::accept_deltas (const std::list<Delta_ptr>& deltas) {
	std::cout << "[ PubRepository accept_deltas start ]\n";
	
	// Read keystore, get one key
	if (!keystore.already_read()) {
	        Path keymap_complete (scratch);
	        keymap_complete /= my_dir;
	        std::cout << "Reading keystore from " << keymap_complete.string() << "\n";
	        keystore.read(boost::filesystem::complete(keymap_complete));
		keystore.set_encrypted(true);
	}

	if (keystore.encrypted()) {
	        decrypt_keystore(private_key);
		keystore.set_encrypted(false);
        	std::cout << "Decrypted keystore\n";
	}
        
	Path keypath (scratch);
        keypath /= my_dir;
        keypath /= "public_key";
        Key pubk (keypath);

        if (keystore.get_size() == 0) {
                std::cout << "No keys in keystore... Creating one\n";
                Key k;
                keystore.add(k);
        }

        Key k = keystore.get();
	std::cout << "Latest key is " << k.get_id() << "\n";

	std::list<Delta_ptr>::const_iterator d;

	for (d=deltas.begin(); d!=deltas.end(); d++) {
		Path path (deltas_dir);
		path /= (*d)->get_id();
		Path complete_path (scratch);
		complete_path /= path;

		bdata_ptr encrypted ( encrypt( (*d)->get_content(), k) );
		Delta_ptr new_delta ( new Delta(*encrypted) );
		new_delta->save(complete_path);

		std::cout << "Adding file " << path.string() << "\n";
		vc->add_file (path);
		write_parents(*d);
		keystore.set((*d)->get_id());
	}
	encrypt_keystore(pubk);
	keystore.set_encrypted(true);
	keystore.store_keymap(scratch); // FIXME: will store ALL mappings again! :-(
	Path mapdir ("keymap");
	vc->add_directory(mapdir);
	vc->commit();
	std::cout << "[ PubRepository accept_deltas end ]\n";
}

void
PubRepository::get_one_delta(std::list<Delta_ptr>& delta_list, std::string id, std::map<std::string,Delta_ptr>& finished) {
	//std::cout << "[ PubRepository: getting delta " << id << " start ]\n";
	// Get its parent list:
	
	Path deps_path (scratch);
	deps_path /= "delta_dependencies";
	deps_path /= id;
	if (!boost::filesystem::exists(deps_path)) {
		std::string str (deps_path.string());
		str += " does not exist!";
		throw std::runtime_error(str);
	}
	boost::filesystem::ifstream deps_file(deps_path);
	std::list<std::string> parents;
        unsigned int parent_count = 0;
        while(!deps_file.eof()) {
                        std::string line;
                        deps_file >> line;
                        if (line.empty())
                                continue;
                        parents.push_back(line);
                        parent_count++;
        }
        deps_file.close();

	// Load each parent
	std::list<std::string>::iterator p;
	for(p = parents.begin(); p != parents.end(); p++) {
		if (finished.count(*p) == 0 && (id.compare(*p) != 0)) {
			get_one_delta(delta_list, *p, finished);
		}
	}

	// And the content:
	//
        Path file_path (scratch);
	file_path /= "deltas";
        file_path /= id;
    
	boost::filesystem::ifstream file(file_path);
	Delta_ptr delta (new Delta);
	delta->read(file); // Read encrypted
	file.close();

	Key k = keystore.get (delta->get_id());

	delta->set_content ( *decrypt(delta->get_content(),k) );
	delta->set_id(id);
	delta->add_parents (parents);

	// Also count this one:
	finished[id] = delta;

	delta_list.push_back(delta);
	std::cout << "[ PubRepository: getting delta " << id << " end ]\n";
}


/**
 * Get a list of deltas from this public repository, in topological order.
 *
 * @return A list of Delta_ptr's, in the order that they should be applied in another database.
 */
std::list<Delta_ptr>
PubRepository::get_deltas() {
	std::cout << "[ PubRepository get_deltas start ]\n";
	boost::shared_ptr<std::list<Delta_ptr> > result     (new std::list<Delta_ptr>);
	boost::shared_ptr<std::list<Delta_ptr> > new_result (new std::list<Delta_ptr>);
	std::string deltas_d (deltas_dir.string() + "/");

	vc->update_wc(deltas_dir);
	vc->update_wc(deltas_dep_dir);
	vc->update_wc(my_keys_dir); // FIXME:

	Path deps_dir_path (scratch);
	deps_dir_path /= "delta_dependencies";

	// Decrypt keystore:
	Path keymap_complete (scratch);
	keymap_complete /= my_dir;

	if (!keystore.already_read()) {
	        std::cout << "Reading keystore from " << keymap_complete.string() << "\n";
		std::cout << "Before read: " << keystore.get_size() << " keys\n";
		keystore.read(boost::filesystem::complete(keymap_complete));
		keystore.set_encrypted(true);
		std::cout << "Read " << keystore.get_size() << " keys\n";
	}
	if (keystore.get_size() == 0) {
		Key k;
		keystore.add(k);
		keystore.set_encrypted(false);
	}
	if (keystore.encrypted()) {
	        decrypt_keystore(private_key);
		keystore.set_encrypted(false);
	}

	std::map<std::string,Delta_ptr> delta_hash;
	boost::filesystem::directory_iterator end_itr;
	for (boost::filesystem::directory_iterator itr( deps_dir_path ); itr != end_itr; ++itr) {
		get_one_delta(*result, itr->leaf(), delta_hash);
	}

	// Reencrypt keystore:
	if (keystore.encrypted()==false) {
	        Path keypath (scratch);
		keypath /= my_dir;
		keypath /= "public_key";
		Key pubk (keypath);
		encrypt_keystore(pubk);
		keystore.set_encrypted(true);
		keystore.store(boost::filesystem::complete(keymap_complete));
	}
	

	std::cout << "[ PubRepository get_deltas end ]\n";
	return *result;
}

/**
 * Destructor
 */
PubRepository::~PubRepository() {
}

/**
 * Constructor.
 *
 * @param v A version control object (already set with paths to repository and scratch working copy)
 * @param cr A cryptengine to be used
 * @param priv path to an Apso private key
 */
PubRepository::PubRepository(VCRepository_ptr v,
			     CryptEngine_ptr cr,
			     Path priv) {
	std::cout << "\n[ PubRepository start ]\n";

	// We want to remember where we were when the object was created
	// FIXME: is this safe?
	initial_dir = boost::filesystem::initial_path();
	
	crypto = cr;
	vc = v;
	
	bdata b;
	b.read (priv);
	private_key.set_value(b);
	private_key.set_id(priv.leaf().c_str());

	deltas_dir     = "deltas";
	deltas_dep_dir = "delta_dependencies";
	users_dir      = "users";
        my_dir         = users_dir;
	my_dir        /= private_key.get_id();
        my_keys_dir    = my_dir;
        my_keys_dir   /= "keys";
	my_keymap_dir  = my_dir;
        my_keymap_dir /= "keymap";

	scratch = vc->get_wc_path();

	
	std::cout << "\ndeltas_dir     = " << deltas_dir.string()
		  << "\ndeltas_dep_dir = " << deltas_dep_dir.string()
		  << "\nusers_dir      = " << users_dir.string()
		  << "\nmy_dir         = " << my_dir.string()
		  << "\nkeys_dir       = " << my_keys_dir.string()
		  << "\nkeymap_dir     = " << my_keymap_dir.string() << "\n";

	std::cout << "[ PubRepository end ]\n";
	
}

/**
 * Sets up a database.
 *
 * Directory tree created:
 *
 * deltas/
 * keymap/
 * users/MY_USER/public_key
 * users/MY_USER/keys/
 *
 * This will create the directories, if they are not there yet.
 */
void
PubRepository::setup() {
	if (boost::filesystem::exists(deltas_dir)) {
		throw std::runtime_error ("I will not set up a public database on" + 
					  deltas_dir.string() +
					  " because it already exists");
	}
	
	vc->add_directory(deltas_dir);
	vc->add_directory(deltas_dep_dir);
	vc->add_directory(users_dir);
	vc->add_directory(my_dir);
	vc->add_directory(my_keys_dir);
	vc->add_directory(my_keymap_dir);

	// FIXME: Copy public key to "users/MY_USER/public_key"
}

/**
 * Encrypts one key with a public key.
 *
 * The key itself is changed (nothing new is returned by the method).
 *
 * @param k The key that will be encrypted.
 * @param pubk The public key used for encryption.
 */
void
PubRepository::encrypt_key(Key& k, const Key& pubk) {
	k.set_value(*crypto->asym_enc(*(k.get_value()), pubk));
}

/**
 * Decrypts one key with a private key.
 *
 * The key itself is changed (nothing new is returned by the method).
 *
 * @param k The key that will be decrypted.
 * @param prik The private key used for decrypting.
 */
void
PubRepository::decrypt_key(Key& k, const Key& prik) {
	k.set_value(*crypto->asym_dec(*(k.get_value()), prik));
}

/**
 * Decrypts the keystore in this database.
 *
 * The keystore itself will change (nothing is returned by this method).
 *
 * @param prik The private key with wich all keys will be decrypted.
 */
void
PubRepository::decrypt_keystore(const Key& prik) {
	std::vector<Key_ptr> list = keystore.get_key_list();
	std::vector<Key_ptr>::iterator i;
	for (i=list.begin(); i<list.end(); i++)
		decrypt_key(**i,prik);
}

/**
 * Encrypts the keystore in this database.
 *
 * The keystore itself will change (nothing is returned by this method).
 *
 * @param pubk The public key with which all keys will be encrypted.
 */
void
PubRepository::encrypt_keystore(const Key& pubk) {
	std::cout << "[ PubRepository encrypt_keystore start ]\n";
	std::cout << "[ PubRepository pubk id = " << pubk.get_id() << " ]\n";
	std::vector<Key_ptr> list = keystore.get_key_list();
	std::vector<Key_ptr>::iterator i;
	for (i=list.begin(); i<list.end(); i++) {
		std::cerr << "Encrypting key " << (*i)->get_id() << "\n";
		encrypt_key(**i,pubk);
	}
	std::cout << "[ PubRepository encrypt_keystore end ]\n";
}


/**
 * Grants access to a user.
 *
 * This is for the case when we know that the user's public key is already in the
 * repository.
 *
 * @param user_id The user's ID (his public key will be used).
 */
void
PubRepository::grant(const std::string user_id) {
	Path user_key (users_dir);
	user_key /= "public_key";
	grant (user_id, user_key);
}

/**
 * Grants access to a user.
 *
 * @param user_id The user's ID in the Apso public repository.
 * @param user_pubk A path to the user's public key.
 */
void
PubRepository::grant(const std::string user_id, const Path user_pubk) {
	std::cout << "[ PubRepository grant start ]\n";
	// Get this users' public key, since we'll encrypt data for him:
	Key pubk (user_pubk);

	// Make sure the user's subdirectory is in the scratch dir, and add it:
	Path his_dir (users_dir);
	his_dir  /= user_id;

	Path his_key (scratch);
	his_key /= his_dir;
	his_key /= "public_key";


	std::cout << "his_key        = " << his_key.string() << "\n";
	Path complete_his_dir (scratch);
	complete_his_dir /= his_dir;
	
	// Now we decrypt the keystore with our private key, and encrypt it with his
	// public key. Then just write it in his keymap location at the database.
	
	Path my_keymap_dir_complete (scratch);
	my_keymap_dir_complete /= my_dir;
	std::cout << "Reading keystore from " << my_keymap_dir_complete.string() << "\n";
	keystore.read(my_keymap_dir_complete); // my keymap
	std::cout << "Found " << keystore.get_size() << " keys in the store!\n";
	decrypt_keystore(private_key);
	encrypt_keystore(pubk);

	std::cout << "Now at " << boost::filesystem::initial_path().string() << "\n";
	std::cout << "Saving keystore to " << complete_his_dir.string() << "\n";
	keystore.store_keys(complete_his_dir);
	
	std::cout << "Now at " << boost::filesystem::initial_path().string() << "\n";
	std::cout << "Will add to vc: " << his_dir.string() << "\n";

	// Also save his public key.
	std::cout << "Will save his key to " << his_key.string() << "\n";
	pubk.save(his_key);
	
	vc->add_directory (his_dir);
	vc->commit();
	std::cout << "[ PubRepository grant end ]\n";
}

/**
 * Sends the latest key to the directories of all users.
 *
 * This method should be called after we create a new key (in the case
 * of a symmetric key compromise, for example).
 */
void
PubRepository::send_current_key_to_users() {
	boost::filesystem::directory_iterator end_itr;
	Path complete_users_dir (scratch);
	complete_users_dir /= users_dir;
	for ( boost::filesystem::directory_iterator itr(complete_users_dir); itr != end_itr; ++itr ) {
		if (! boost::filesystem::is_directory(*itr) )
			continue;
		
		Key k = keystore.get();

		Path pubk_path (*itr);
		pubk_path /= "public_key";
		Key pubk (pubk_path);
		
		encrypt_key(k, pubk);
		
		Path key_path (*itr);
		key_path /= "keys";
		key_path /= k.get_id();

		k.save(key_path);
		
		Path relative_key_path (users_dir);
		relative_key_path /= itr->leaf();
		relative_key_path /= "keys";
		relative_key_path /= k.get_id();

		vc->add_file(relative_key_path);
	}
}

/**
 * Revokes access from a user.
 *
 * This will remove the users' directory from version control
 * and generate a new symmetric key for everyone to use, so the
 * user wil not be able to access new content.
 *
 * @param user_id The key ID of the user.
 */
void
PubRepository::revoke(std::string user_id) {
	std::cout << "[ PubRepository revoke start ]\n";
	Path his_dir (users_dir);
        his_dir  /= user_id;
	Path complete_his_dir(scratch);
	complete_his_dir /= his_dir;
	std::cout << "his_dir = " << his_dir.string() << "\n";
	std::cout << "complete_his_dir = " << complete_his_dir.string() << "\n";
	vc->remove_directory (his_dir);
	boost::filesystem::remove_all (complete_his_dir);
	keystore.newkey();

	send_current_key_to_users();

	Path current_key_path (scratch);
	current_key_path /= "current_key";
	boost::filesystem::ofstream current_key_file (current_key_path);
	current_key_file << keystore.get().get_id() << "\n";
	current_key_file.close();
	vc->commit();
	std::cout << "[ PubRepository revoke end ]\n";
}

/**
 * Does all necessary changes after a compromise.
 *
 * This will throw away all symmetric keys (by revoking access to all users), then
 * create a new key, encrypt it for all users, and then grant access to all users
 * again.
 */
void
PubRepository::compromise() {
// FIXME:
}

/**
 * Calculates Delta(a) - Delta(b).
 *
 * This will return a list of all deltas that are in A but not in B.
 *
 * @param a A database object.
 * @param b A database object.
 * @return a list of Delta objects.
 */
std::list<Delta_ptr>
operator- (Repository& a, Repository& b) {
	std::cout << "[ PubRepository - start ]\n";
	std::list<Delta_ptr> list_a ( a.get_deltas() );
	std::list<Delta_ptr> list_b ( b.get_deltas() );
	boost::shared_ptr<std::list<Delta_ptr> > result (new std::list<Delta_ptr>);
	bool found;

	std::list<Delta_ptr>::iterator ia;
	std::list<Delta_ptr>::iterator ib;
	for (ia = list_a.begin(); ia != list_a.end(); ia++) {
		found = false;
		for (ib = list_b.begin(); ib != list_b.end(); ib++) {
			if ((*ia)->get_id() == (*ib)->get_id()) {
				std::cout << "    " << (*ib)->get_id() << " already in!\n";
				found = true;
				continue;
			} 
		}
		if (! found) {
			std::cout << "    " << (*ia)->get_id() << " missing (and will be included)\n";
			result->push_back(*ia);
		}
	}
	std::cout << "[ PubRepository - end ]\n";
	return *result;
}


/**
 * Sends mising deltas to private database.
 *
 * This will calculate which deltas "this" database has,
 * which are not in a given private database, and then
 * include them there.
 *
 * @param priv A database object with the private database.
 */
void
PubRepository::push(PrivRepository_ptr priv) {
	std::cout << "[ PubRepository: push start ]\n";
	std::list<Delta_ptr> missing = *this - *priv;
	std::list<Delta_ptr>::iterator i;
	if (!missing.empty())
		priv->accept_deltas (missing);
	std::cerr << "[ PubRepository: push end ]\n";
}

/**
 * Brings missing deltas from private database.
 *
 * @param priv A database object with the private database.
 */
void
PubRepository::pull(PrivRepository_ptr priv) {
	std::cout << "[ PubRepository: pull start ]\n";
	std::list<Delta_ptr> missing (*priv - *this);
	std::list<Delta_ptr>::iterator i;
	if (!missing.empty())
		accept_deltas (missing);
	std::cerr << "[ PubRepository: pull end ]\n";
}

/**
 * Synchronizes this public database with a private database.
 *
 * This is NOT the usual sync from distributed version control!
 * One important assumption is that the deltas in THIS database are encrypted, and those
 * in the private database are not.
 *
 * @param priv A database object with the private database.
 */
void
PubRepository::sync(PrivRepository_ptr priv) {
	std::list<Delta_ptr> missing_this (*priv - *this);
	std::list<Delta_ptr> missing_priv (*this - *priv);
	while (! (missing_this.empty() && missing_priv.empty())) {
		pull(priv);
		push(priv);
		missing_this = (*priv - *this);
		missing_priv = (*this - *priv);
	}
}

}
