/*
 * key.cpp
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

extern "C" {
#include "stdlib.h"
#include "time.h"
#include "string.h"
}
#include "key.h"
#include "cryptengine.h"
#include "cryptengine_nettle.h"

#include <boost/filesystem/fstream.hpp>


namespace apso {

/**
 * Constructor.
 *
 * This will initialize the key with random data. The key will of course
 * be stored in clear, and should be encrypted if necessary.
 *
 * The key will come with a default random ID.
 */
Key::Key() {
	CryptEngineNettle_ptr cr (new CryptEngineNettle());

	/* FIXME: magic numbers? */
	size_t size = 32;
	size_t id_size = 80;

	value = bdata();
	cr->random_fill(value, size);

	bdata_ptr random_data (new bdata());
	cr->random_fill(*random_data, id_size);

	bdata_ptr hash = cr->hash(*random_data);

	bdata_ptr result = cr->encode16(*hash);

	id = result->get_string();
}

/**
 * Returns the size of this key.
 *
 * @return The size of this key.
 */
size_t
Key::get_size() const {
	return value.get_size();
}

/**
 * Returns a pointer to the value of this key.
 * 
 * Please check the size with get_size().
 * 
 * @return A COPY of the contents of this key.
 */
bdata_ptr
Key::get_value() const {
	bdata_ptr p (new bdata (value));
	return p;
}



/**
 * Returns a string representation of this key.
 *
 * This will add a '\0' to the end of the key value and make a string with it.
 * DON'T use this for binary data (hint: base64 encoded is OK)
 *
 * @return A string with the value of this key.
 */
std::string
Key::get_string() const {
	boost::shared_ptr<std::string> s (new std::string(value.get_data(),value.get_size()));
	return *s;
}

/**
 * Sets the value of this key.
 *
 * @param b A bdata with the new content of the key.
 */
void
Key::set_value(const bdata& b) {
	value = b;
}

/**
 * Gets the id of a key.
 *
 * This will return a string with the id of this key.
 *
 * @return A string with the ID of this key.
 */
std::string
Key::get_id() const {
	return id;
}

/**
 * Sets the id of the key.
 *
 * @param i A string with the new ID of this key.
 */
void
Key::set_id(const std::string& i) {
	id = i;
}


/**
 * Compares two keys.
 *
 * @param a A key.
 * @param b A key.
 * @return true if the IDs, sizes and values are equal for both keys.
 */
bool
operator== (Key& a, Key& b) {
	if (a.get_value() == b.get_value())
		return true;
	return false;
}

/**
 * Constructor that reads a key from a file.
 *
 * @param path The path to the file.
 */
Key::Key(const Path& path) {
        std::cout << "Will read key from " << path.string() << "\n";
        boost::filesystem::ifstream pub_in(path);

        size_t chunk_size = 4096;
        char * p = (char *) malloc (chunk_size * sizeof(char));
        size_t total = 0;
        size_t bytes_read;
        do {
                pub_in.read ((char *)p + total,chunk_size);
                bytes_read = pub_in.gcount();
                total += bytes_read;
                if (bytes_read == chunk_size)
                        p = (char *) realloc (p, total + chunk_size * sizeof (char));
        } while (bytes_read == chunk_size);
        p = (char *) realloc (p, total * sizeof (char));
        bdata b (p,total);
        set_value(b);
}

/**
 * Saves the key to a file.
 *
 * @param path The path to the file.
 */
void
Key::save(const Path& path) {
	boost::filesystem::ofstream out (path);
	std::cout << "KEYSAVE: " << path.string() << "\n";
	out.write(value.get_data(), value.get_size());
	out.close();
}

}

