/*
 * delta.cpp
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string>
#include <iostream>

#include <boost/filesystem/fstream.hpp>

#include "delta.h"

namespace apso {

/**
 * Constructor.
 */
Delta::Delta() {
}


/**
 * Constructor that also defines a first content of the delta.
 */
Delta::Delta(std::string c) {
	content.set_data((char *)c.data(), c.size());
}

/**
 * Constructor from bdata.
 */
Delta::Delta(bdata& d) {
	content = d;
}

/**
 * Sets the id of this delta.
 */
void
Delta::set_id(const std::string& i) {
	id = i;
}

/**
 * Returns the id of this delta.
 */
const std::string
Delta::get_id() {
	return id;
}

/**
 * Sets the content of this delta.
 *
 * @param c A string with the content.
 */
void
Delta::set_content(const std::string& c) {
	content.set_data(c.data(), c.size());
}

/**
 * Sets the content of this delta.
 *
 * @param c A bdata with the content.
 */
void
Delta::set_content(const bdata& c) {
        content.set_data(c.get_data(), c.get_size());
}


/**
 * Gets the content of this delta as a string.
 */
std::string
Delta::get_string() {
	boost::shared_ptr<std::string> s (new std::string (content.get_data(),content.get_size())); 
	return *s;
}

/**
 * Returns the content of the Delta object.
 *
 * @return A bdata object with the content.
 */
const bdata&
Delta::get_content() {
	return content;
}

/**
 * Adds a string to the content of the delta.
 */
void
Delta::add(const std::string& s) {
	content.add(s.data(), s.size());
}

/**
 * Appends binary data to the Delta.
 *
 * @param d A bdata with the content to be appended.
 */
void
Delta::add(bdata& d) {
	content.add(d);
}

/**
 * Adds one parent to this Delta.
 *
 * @param parent_id A string with the ID of the parent being added.
 */
void
Delta::add_parent(const std::string parent_id) {
	parents.push_back(parent_id);
}

/**
 * Adds a list of parents to this Delta.
 *
 * FIXME: overwrites old list!
 *
 * @param parent_list A list of strings with the parents IDs.
 */
void
Delta::add_parents(const std::list<std::string> parent_list) {
	parents = parent_list;
}

/**
 * Gets the list of parents of this Delta.
 *
 * @return A list of strings with the parents IDs.
 */
std::list<std::string>
Delta::get_parents() const {
	return parents;
}


/**
 * Reads the Delta from an istream.
 *
 * @param in The istream from which the Delta will be read.
 */
void
Delta::read(std::istream& in) {
	unsigned size;
	in >> size;
	char *i = (char *) malloc (size);
	in.read(i,1);    // The ' ' after the number representing the size
	in.read(i,size); // The rest of the relta, of which we know the size now
	std::string my_id (i,size);
	id=my_id;
	content.read(in);
}

/**
 * Saves a delta to an ostream.
 *
 * @param out The ostream to which the delta will be written.
 */
void
Delta::save(std::ostream& out) const {
	out << id.size();
	out << ' ';
	out.write(id.c_str(),id.size());
	content.save(out);
}

/**
 * Loads a delta from the disk.
 *
 * @param path The path from where to load the delta.
 */
void
Delta::read(Path& path) {
	boost::filesystem::ifstream in (path);
	read(in);
}

/**
 * Saves this delta to the disk.
 *
 * @param path The path where to save the delta.
 */
void
Delta::save(Path& path) const {
	boost::filesystem::ofstream out (path);
	save(out);
}

}
