#! /usr/bin/env python
#*******************************************************************************
# aping.py --> main program module                                             *
#                                                                              *
#*******************************************************************************
# Copyright (C) 2007 Kantor A. Zsolt <aping.project@gmail.com>                 *
#*******************************************************************************
# This file is part of APing.                                                  *    
#                                                                              *   
# APing is free software; you can redistribute it and/or modify it under the   *
# terms of the GNU General Public License as published by the Free Software    *
# Foundation;either version 2 of the License,or (at your option) any later     *
# version.                                                                     *
#                                                                              *    
# APing is distributed in the hope that it will be useful,but WITHOUT ANY      *
# WARRANTY;without even the implied warranty of MERCHANTABILITY or FITNESS     *
# FOR A PARTICULAR PURPOSE.See the GNU General Public License for more details.*              
#                                                                              *    
# You should have received a copy of the GNU General Public License            *    
# along with APing; if not, write to the Free Software                         *   
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA   *  
#*******************************************************************************
# encoding utf-8

from default import *

# Option parser, check for valid options
try:
    valid_options = getopt.gnu_getopt(sys.argv[1:], "Vt:r:w:p:o:v:s:P:hdT:b:", ("Probe=",\
    "rdns", "print-options", "ttl=", "pkg-trace", "size=", "packet=",\
    "retry=", "verbose=", "send-delay=", "listen=", "tos=", "time", "help", "Version", "bind="))
except getopt.GetoptError, bad_opt:
    sys.exit("\nAPing: %s \nTry -h or --help for a list of available options" %bad_opt)


def help():
    """Prints this help message to the stdout if the '-h/--help' option is used
    """
    sys.exit("""\nUsage: aping.py {target specification} [OPTIONS]\n
Target:
   {target specification}
      Specify the target address. The target can be a hostname like 
      www.probe.com, my.example.org or any IPv4 address like 192.168.0.1
Options:
  -P, --Probe <type> 
      Specify the ICMP probe type.<type> can be p, t, m or i where 
      p is for usual ping probes, t is for timestamp request m for 
      address mask request and i for information request (default
      is the ICMP echo request) 
  --time
      This option is used only with the timestamp requests. If a valid
      timestamp reply is received from the target and this option is 
      enabled it returns the time in the timestamp packets
  -d, --rdns 
      Make reverse DNS resolution if you specified a IPv4 address
  --print-options
      Print out all the options configured with this session before 
      sending any packets 
  -t, --ttl <num>
      Set up the time to live field. <num> is an integer and it is
      between 0 and 255 (inclusive). The default value is 64
  -b, --bind <IP number>
      Use this option to bind the created socket to an IP address. The
      argument <IP number> must to be a valid IPv4 address. This option
      if useful if you have multiple public IP's or when you probe your
      local address. In this case it's a good idea to bind the socket
      to annother local IP
  --pkg-trace
      If this option is specified it prints out all the packets sent to 
      the target not just the received ones
  -s, --size <byte>
      Data in packets to send. <byte> is the number of extra bytes to 
      send.Default behavior for APing is to send packets with no
      extra data
  -o, --listen <time>[s/m]
      Set the listening timeout. <time> is the argument in milliseconds 
      by default. With the 's'(seconds) or 'm'(minutes) appended to the
      end of the number and squeezed together you can specify times in
      seconds or minutes (eg. 2s)
  -p, --packet <pkg>
      Set the packets to send,then stop. <pkg> are the number of 
      packets to send default is to send a number of infinitive packets
      unless you stop from the keyboard, or retransmission exceeded 
  -w, --send-delay <time>[s/m]
      Adjust the send delay between probes. <time> is the delay time
      in milliseconds, the default send delay is 1 second.The 's' or
      'm' options are the same like in the case of the listening 
      timeout 
  -r, --retry <num>
      Set the probes retry if no packet is received. <num> is the 
      packets to send, the default probes retry is 10 after that  
      APing stops
  -T, --tos <num>
      Set the TOS (Type of Service) field in the IP header. The <num>
      argument can be an integer from 0 to 255 (inclusive) or specified
      as a hexadecimal number in format 0x (default value is 0)
  -v, --verbose <level>
      Verbose output for the DNS resolver. The <level> argument is 
      a number that can be 1, 2 or 3 (default is 0)
  -V, --Version           
      Print out the version and exit
  -h, --help                This help message""")


lista = [] # create an empty list
# run through arguments and rewrite the default variables
for opt,arg in valid_options[0]:
    if opt == "-s" or opt == "--size":
        extra_data = arg
        lista.append("size")
    elif opt == "-v" or opt == "--verbose":
        verbose = arg
        lista.append("verbose")
    elif opt == "-P" or opt == "--Probe":
        probe_type = arg
        lista.append("Probe")
    elif opt == "-o" or opt == "--listen":
        listen_timeout = arg
        lista.append("listen")
    elif opt == "-p" or opt == "--packet":
        probe_time = arg
        lista.append("packet")
    elif opt == "-d" or opt == "--rdns":
        rev_dns = True
        lista.append("rdns")
    elif opt == "-w" or opt == "--send-delay":
        send_delay = arg
        lista.append("send-delay")
    elif opt == "--print-options":
        print_opt = True
        lista.append("print-options")
    elif opt == "-t" or opt == "--ttl":
        time_to_live = arg
        lista.append("ttl")
    elif opt == "-r" or opt == "--retry":
        probes_retry = arg
        lista.append("retry")
    elif opt == "--pkg-trace":
        pkg_trace = True
        lista.append("pkg-trace")
    elif opt == "-T" or opt == "--tos":
        ip_tos = arg
        lista.append("tos")
    elif opt == "--time":
        return_time = True
        lista.append("time")
    elif opt == "-b" or opt == "--bind":
        bind_addr = arg
        lista.append("bind")
    elif opt == "-h" or opt == "--help":
        help()
    elif opt == "-V" or opt == "--Version":
        sys.exit("\nAPing version: 0.1 alpha4")

# Check for double or more options of the same type
for i in xrange(len(valid_options[0]) - 1):
    if lista.count(lista[i]) > 1:
        sys.exit("\nAPing: Duplicated options detected from a type")

# The first non-option should be the target address, if none specified exit
try:
    dst_address = valid_options[1][0]
    # Check if the user entered more then 1 non-option 
    if len(valid_options[1]) > 1:
        sys.exit("\nAPing: Only one target specification permitted at a time")
except IndexError:
    sys.exit("""\nAPing: At least specify a target address 
        \rTry -h or --help for a list of available options""")
 
def sighandler(signum, frame):
    """The keyboard interrupt handler"""
    sys.exit("APing: Interrupt from keyboard (SIGINT)")
signal.signal(signal.SIGINT, sighandler)

print

def calcsum(sum_data):
    """The packet checksum algorithm (the one's complement sum of 16-bit words)
    Generates a checksum of a (ICMP) packet. Based on the function found in 
    ping.c on FreeBSD.
    """
    # add byte if not dividable by 2
    if len(sum_data) & 1:              
        sum_data += '\0'
    # split into 16 bit word and insert into a binary array
    words = array.array('H', sum_data)
    sum = 0
    # perform ones complement arithmetic on 16-bit words
    for word in words:
        sum += (word & 0xffff) 
    hi = sum >> 16 
    lo = sum & 0xffff 
    sum = hi + lo
    sum += (sum >> 16)
    return struct.pack('H', (~sum & 0xffff))
    
class ICMPprobe:
    """The program main class handles all the important stuff. 
    From here are sent the ICMP packets
    """
    
    def __init__(self):
        """With this method function initiates the ICMPprobe class."""        
        # the signal handler for the ctrl + c keyboard interrupt (SIGINT)
        signal.signal(signal.SIGINT, self.sighandler)
        
        # creates the raw socket, if the user haze no root privileges then stop 
        try:
            self.rawicmp = socket.socket(socket.AF_INET, socket.SOCK_RAW, socket.IPPROTO_ICMP)
            self.rawicmp.bind((bind_addr, 0)) # bind the socket to an IP address
        except socket.error, error_msg:
            if error_msg[0] == 1:
                sys.exit("\nAPing: You must to have root (superuser) privileges to run APing")
            elif error_msg[0] == 99:
                sys.exit("""\nAPing: Can not bind socket to specified address: %s
                \rThe IP address must to exist on your interface""" %bind_addr)
        
        # set up some socket option 
        self.rawicmp.setsockopt(socket.IPPROTO_IP, socket.IP_TTL, time_to_live)
        self.rawicmp.setsockopt(socket.IPPROTO_IP, socket.IP_TOS, ip_tos)
        self.rawicmp.settimeout(listen_timeout)
        
        # initiating some variables
        self.send_delay = send_delay
        self.pkg_sent = 0
        self.pkg_recv = 0
        self.rtt_sum = 0
        self.rtt_max = 0
        self.rtt_min = 9000
        self.code = "\0"
        self.retrans = 0
        self.addr_mask = ''
        self.tmstamp_req = ''
        self.payload = ''
        
        # generate a random identifier and create a 16 bit big-endian structure
        self.ident = struct.pack('!H', random.randrange(1, 65536))
        
        # generates some extra bytes, if specified by the user
        self.data = self.data_gen() 
                
        # selecting, setting up the probe type
        if probe_type == 'p':
            self.types = "\x08"
            self.strint_type = "8(echo request)"
        elif probe_type == 'i':
            self.types = "\x0f"
            self.strint_type = "15(information request)"
        elif probe_type == 'm':
            self.types = "\x11"
            self.addr_mask = "\0\0\0\0"
            self.strint_type = "17(address mask request)"
        elif probe_type == 't':
            self.types = "\x0d"
            self.strint_type = "13(timestamp request)"
            self.tmstamp_req = "\0\0\0\0\0\0\0\0\0\0\0\0"
        
        # calculate the sent packets length
        self.length = 8 + extra_data + len(self.tmstamp_req + self.addr_mask)
        self.start_time = time.time() # read the time for the total elapsed time
    
    def data_gen(self):
        """Generates some 0 bytes of extra data if specified by the user"""
        for i in xrange(extra_data):
            self.payload += "\0"
        return self.payload
    
    def run(self):
        """The packet transmission loop"""
        while 1: 
            # if the last packet comes next set the send delay to 0
            # in the next loop if the required packets haze been sent then stop
            if probe_time == self.pkg_sent + 1: 
                self.send_delay = 0
            elif probe_time < self.pkg_sent + 1:
                self.reason = "Stop after %s packet(s) sent" %(probe_time)
                self.statistics()
            
            # generate the sequnce nr. then pack it to a 16 bit big-endian struct 
            self.intseq = self.pkg_sent + 1
            self.seq = struct.pack('!H', self.intseq)
            # calculate the checksum
            checksum = calcsum(self.types + "\0" + "\0\0" + self.ident + self.seq)
            
            # print this on every loop if the user uses the --pkg-trace option
            if pkg_trace:
                print '-' * 85,"\nsent: %s bytes ttl=%s icmp type=%s icmp seq=%s "\
                       %(self.length,  time_to_live, self.strint_type, \
                       self.intseq)
            
            # setting start time for the rtt, and sends a icmp packet to the target
            rtt_starttime = time.time()
            try:
                self.rawicmp.sendto(self.types + self.code + checksum +\
                self.ident + self.seq + self.addr_mask + self.tmstamp_req + self.data,\
                (ip_dst_address, dst_port))
            except socket.error, error_msg:
                if error_msg[0] == 101:
                    sys.exit("\nAPing: Unable to establish a connection.Verify your network connectivity !")

            self.pkg_sent += 1
            self.keyboardhalt = 1
            # listening for incomming packet, if it's not from us then loop again
            # if timeout exceeds set up some variables and send another packet
            # if retransmission exceeds then jump to statistics function  
            while 1:
                try:
                    self.recv_data, self.src_addr = self.rawicmp.recvfrom(56)          
                except socket.timeout:
                    self.retrans += 1
                    self.keyboardhalt = 0
                    if self.retrans >= probes_retry:
                        self.reason = "Retransmission exceeded after %s probe(s)" %(probes_retry)
                        self.statistics()
                    time.sleep(self.send_delay)
                    self.run()
                if self.recv_data[24:26] == self.ident  or self.recv_data[52:54] == self.ident:
                    break
            
            self.current_rtt = (time.time() - rtt_starttime) * 1000
            self.pkg_recv += 1
            self.retrans = 0 # always reset the retrans. if a packet is received 
            self.keyboardhalt = 0
            self.data_analize() # parse the captured packet
            
            # calculate minimum rtt , maximum rtt and the rtt sums
            self.rtt_sum += self.current_rtt
            self.rtt_min = min(self.current_rtt, self.rtt_min)
            self.rtt_max = max(self.current_rtt, self.rtt_max)
            
            time.sleep(self.send_delay)
            
    def data_analize(self):
        """Here are parsed the received packets,and printed to the stdout"""
        self.recv_data = binascii.hexlify(self.recv_data) # converting to hex data
        ttl = int(self.recv_data[16:18], 16) # getting the ttl
        self.src_addr = self.src_addr[0]
        icmp_type = int(self.recv_data[40:42], 16) # getting the icmp type
        
        # for every type of ICMP read the sequence number and set a string
        # expression for the message
        if icmp_type == 0:
            icmp_msg = "echo reply"
            icmp_seq = int(self.recv_data[52:56], 16)
        elif icmp_type == 14:
            icmp_msg = "timestamp reply"
            icmp_seq = int(self.recv_data[52:56], 16)
        elif icmp_type == 8:
            icmp_msg = "echo request"
            icmp_seq = int(self.recv_data[52:56], 16)
        elif icmp_type == 11:
            icmp_msg = "time exceeded"
            icmp_seq = int(self.recv_data[108:112], 16)
        elif icmp_type == 3:
            icmp_msg = "dest. unreachable"
            icmp_seq = int(self.recv_data[108:112], 16)
        else:
            icmp_msg = "unknown"
                    
        length = len(self.recv_data[40:]) / 2 # calculate the returned packet length
        # if the user do not use the --time option (usual probes)
        if return_time != 1:
            print "recv: %s bytes addr=%s ttl=%s icmp type=%s(%s) icmp seq=%s rtt=%.2f ms"\
            %(length, self.src_addr, ttl, icmp_type, icmp_msg, icmp_seq, self.current_rtt)
        # if the user uses the --time 
        else:
            # check if the returned packet is a valid timestamp reply 
            if icmp_type != 14: 
                sys.exit("""\nAPing: Invalid timestamp reply received
                \rSo can not use the --time option to parse the time""")
            print  "recv: timestamp reply addr=%s time -> %s - UTC" \
            %(dst_address, time.strftime("%H:%M:%S", time.gmtime(int(self.recv_data[72:80],16) /1000)))
               
    def sighandler(self, signum, frame):
        """The keyboard interrupt handler"""
        self.reason = "Interrupt from keyboard (SIGINT)"
        self.statistics()
    
    def statistics(self):
        """The statistics are printed to the stdout every time when the program
        stops for some reason (retrans. expired, interrupt from keyboard ...)
        """
        # calculate the total elapsed time
        time_elapsed = time.time() - self.start_time
        # calculate the lost packets
        pkg_lost = self.pkg_sent - self.pkg_recv 
                
        # calcualte the average time  
        if self.pkg_recv > 0:
            aver_time = self.rtt_sum / self.pkg_recv
        else: # if no packets are received set some variables to zero
            aver_time, self.rtt_min, self.src_addr = 0, 0, ip_dst_address
        
        # check if the received packet are from the target address
        if self.src_addr in ip_dst_address:
            from_where = ''
        else :
            from_where="\n - The received packets are not from the target address (%s)!"\
            %dst_address
        
        # check if APing was stopped in the listening time
        if self.keyboardhalt != 1 and  pkg_lost == 0:
            last_pkg = ''
        else:
            last_pkg = "\n - No answer for the last sent packet because interrupted in the listening time" 
        
        # if no error message was received
        if last_pkg == '' and from_where == '':
            msg = "\n - All is OK"
        else:
            msg = ''
        
        # calculate the elapsed time in milliseconds and seconds if time is less
        # then 1 minute, else in seconds and minutes
        if time_elapsed < 60:
            time1 = 1000 * time_elapsed; end1 = "ms"
            time2 = time_elapsed; end2 = 's'
        else:
            time1 = time_elapsed; end1 = 's'
            time2 = time_elapsed / 60; end2 = "m"
        
        #print the last line if --pkg-trace was specified
        if pkg_trace:
                print '-' * 85
        
        # print this out every time when the program stops 
        print "Halt reason: %s" %self.reason
        print "Status:%s%s%s" %(from_where, last_pkg, msg)
        print "\n+++++++++++++++  statistics  +++++++++++++++"
        print "Packets:"
        print "   Total sent:%s | lost:%s | received:%s"\
        %(self.pkg_sent, pkg_lost, self.pkg_recv)
        print "          | lost:%.2f%% | received:%.2f%%"\
        %(((100.0 * pkg_lost) / self.pkg_sent), ((100.0 * self.pkg_recv) / self.pkg_sent))
        print "Timing:"
        print "   rtt min:%.3f | aver:%.3f | max:%.3f ms"\
        %(self.rtt_min, aver_time, self.rtt_max)
        print "   Total time elapsed %.2f %s | %.2f %s" %(time1, end1, time2, end2)
        self.rawicmp.close(); sys.exit(0)

class Resolver:
    """The DNS resolver class.
    
    This class makes the resolution of the hostname or the reverse DNS resolution
    if an IPv4 is entered and the -d option used.(The class uses the system DNS 
    resolver).
    """
    
    def __init__(self, str_probe_type, isip) :
        global ip_dst_address
        
        # get the full information about the target host (address record, 
        # canonical names, ip addr.), if can't then target can not be resolved
        try:
            target_addr_info = socket.gethostbyname_ex(dst_address)
        except socket.gaierror:
            sys.exit("""APing: Target hostname can not be resolved (%s)
            \rAre you specified a valid hostname ? check the characters""" %dst_address)
        
        nr_of_ips = len(target_addr_info[2]) # count the number of ip(s)
        target_ips = str(target_addr_info[2]).strip("[]") # get the ip(s)
                
        # get the canonical names  
        addr_cnames = str(target_addr_info[1]).strip("[]")
        address_record = target_addr_info[0] # get the address record
        
        # select the destination ip         
        if nr_of_ips == 1 :
            ip_dst_address = (target_addr_info[2])[0]
        # pick an IP if there are more (randomly)
        else: 
            ip_dst_address = (target_addr_info[2])[random.randrange(0, nr_of_ips)]
        
        # if verbosity is 1 or 2 
        if verbose > 0 : 
            if not isip and nr_of_ips > 1 : # a hostname with multiple ip's
                print "%s resolves to multiple IP's (%s)"%(dst_address, nr_of_ips)
                if verbose == 2: # if verbosity is 2
                    print "The IP's are:", target_ips
            elif not isip and nr_of_ips == 1 : # a hostname with one ip 
                print dst_address,"resolves to", ip_dst_address
            
            if not isip and verbose == 2 :  # if verbosity is 2
                if addr_cnames == '':
                    addr_cnames = None
                print "Canonical names:", addr_cnames
                print "Address record:", address_record
            print "Trying with IP:", ip_dst_address
        
        # if reverse DNS resolution is true & the target is in IP format        
        if isip and rev_dns:
            try:
                print "Reverse DNS resolution: %s" %socket.gethostbyaddr(dst_address)[0]
            except socket.herror:
                print "Warning ! Reverse DNS resolution failed"  
        
        print "Sending", str_probe_type
        ICMPprobe().run()

class VerifyPrintpot:
    """Checks the entered options for validity and and if it's specified by the 
    user prints out to the stdout all the probing options
    """
    
    def printopt(self):
        """Checks the date for the time zone output and if it's specified by the
        user outputs to the stdout all the setups witch are used for the current
        session 
        """
        # get the current time zone 
        if time.localtime()[8] == 1:
            tzone=time.tzname[1]
        else:
            tzone = time.tzname[0]
        
        print "Starting APing at: %s %s" %(time.asctime(), tzone)
        
        # print this information if --print-options are specified
        if print_opt:
            print """\nICMP probe options:
                \r Target address: %s
                \r Probe type: %s
                \r Packets to send: %s
                \r Listening timeout: %s
                \r Send delay: %s
                \r Extra data: %s bytes
                \r Time to live: %s
                \r Probes retry %s (times)
                \r Verbosity level %s 
                \r Reverse DNS resolution: %s
                \r Packet trace: %s
                \r Print options: %s
                \r TOS field value: %s (int)
                \r Print timestamp time: %s
                \r Bond to address: %s \n""" %(dst_address, self.str_probe_type,\
                probe_time, self.print_listen_time, self.print_send_delay, extra_data,\
                time_to_live, probes_retry, verbose, rev_dns, pkg_trace, print_opt,\
                ip_tos, return_time, bool(bind_addr)) 
        Resolver(self.str_probe_type, self.isip)
    
    def probe_type_verif(self):
        """Verifies if the probe type entered by the user is correct"""
        global probe_type
        
        # assingns a string expression for the selected probe type used for the
        # printopt() function, and in the Resolver() class 
        if probe_type == 'p':
            self.str_probe_type = "ICMP Echo request"
        elif probe_type == 't':
            self.str_probe_type = "ICMP Timestamp request"
        elif probe_type == 'm':
            self.str_probe_type = "ICMP Address Mask request"
        elif probe_type == 'i':
            self.str_probe_type = "ICMP Information request"
        else: # print this if the user entered an invalid probe type
            sys.exit("""APing: Unknown probe type specified (%s)
            \rValid probe types are: p for echo request
                       t for timestamp request
                       m for address mask request
                       i for information request""" %probe_type)
        self.printopt()
    
    def ip_tos_verif(self):
        "Verifies the TOS argument"
        global ip_tos
        try:
            ip_tos = int(ip_tos) # check the ToS format (decimal or hexadecimal)
            # the ToS field is 8 bit (unsigned), so the maximum int value is 255
            if ip_tos > 255 or ip_tos < 0:
                sys.exit("""APing: Invalid TOS value specified (%s)
            \rArgument must to be between 0 and 255 (inclusive)""" %ip_tos)
        except ValueError:
            if ip_tos[:2] != '0x': # the hex format must to start with '0x'
                sys.exit("""APing: Invalid TOS value specified (%s)
            \rArgument must to be an integer or hex value""" %ip_tos)
            elif len(ip_tos) > 4: # for sure the entered hex value is to high
                sys.exit("""APing: Invalid TOS value specified (%s)
            \rThe maximum hex value is 0xff""" %ip_tos)
            # this is a valid ToS value in hexadecimal format
            ip_tos = int(ip_tos[2:], 16) 
        self.probe_type_verif()
    
    def probes_retry_verif(self):
        "Verifies the probes retry argument"
        global probes_retry
        try:
            probes_retry = int(probes_retry)
            if probes_retry <= 0:
                sys.exit("""APing: Invalid probes retry specified (%s)
                \rArgument must to be greater then 0""" %probes_retry)
        except ValueError:
            sys.exit("""APing: Invalid probes retry specified (%s)
            \rArgument must to be an integer value""" %probes_retry)
        self.ip_tos_verif()
            
    def time_to_live_verif(self):
        "Verifies the time to live value"
        global time_to_live
        try:
            time_to_live = int(time_to_live)
            # valid time to live value is maximum a 8 bit unsigned number
            if time_to_live < 1 or time_to_live > 255:
                sys.exit("""APing: Invalid time to live specified (%d)
                \rValid number range is between 0 and 255 (inclusive)""" %time_to_live)
        except ValueError:
            sys.exit("""APing: Invalid time to live specified (%s)
            \rArgument must to be an integer value""" %time_to_live)
        self.probes_retry_verif()
    
    def send_delay_verif(self):
        "Verifies the send delay and checks if an end option is used (m or s)"
        global send_delay
        send_delay = str(send_delay)
        
        # check if after the send delay number is appended a 'm', or 's' and 
        # sets up a divisor after create a string expression for the printopt()
        # method function  
        if send_delay[-1:] == 'm':
            send_delay = send_delay[:-1]
            self.print_send_delay = send_delay + " (min)"
            div = 1
        elif send_delay[-1:] == 's':
            send_delay = send_delay[:-1]
            self.print_send_delay = send_delay + " (sec)"
            div = 60
        else: 
            self.print_send_delay = send_delay + " (msec)"
            div = 60000
        var = send_delay
        try:    
            send_delay = float(send_delay) * 60 / div
            if send_delay < 0:
                sys.exit("""APing: Invalid send delay specified (%s)
                \rArgument must to be greater or equal with 0""" %var)
        except ValueError:
            sys.exit("""APing: Invalid send delay specified (%s)
            \rArgument must to be a float or integer value""" %send_delay)
        self.time_to_live_verif()
    
    def pkg_to_send_verif(self):
        "Verifies the package(s) to send specified"
        global probe_time
        try:
            probe_time = int(probe_time)
            if probe_time <= 0:
                sys.exit("""APing: Invalid number of packets to send specified (%s)
                \rArgument must to be integer value and above 0""" %probe_time)
        except ValueError:
            if probe_time == "Infinitive":
                self.send_delay_verif()
            sys.exit("""APing: Invalid packets to send specified (%s)
            \rArgument must to be an integer value not string of float""" %probe_time)
        self.send_delay_verif()
    
    def listen_timeout_verif(self):
        "Verifies the listen timeout and checks if an end option is used (m or s)"
        global listen_timeout
        listen_timeout = str(listen_timeout)
        
        # check if after the listen timeout number is appended a 'm', or 's' and 
        # sets up a divisor after create a string expression for the printopt()
        # method function  
        if listen_timeout[-1:] == 'm':
            listen_timeout = listen_timeout[:-1]
            self.print_listen_time = listen_timeout + " (min)"
            div = 1
        elif listen_timeout[-1:] == 's':
            listen_timeout = listen_timeout[:-1]
            self.print_listen_time = listen_timeout + " (sec)"
            div = 60
        else: 
            self.print_listen_time=listen_timeout + " (msec)"
            div = 60000        
        var = listen_timeout
        try:
            listen_timeout = float(listen_timeout) * 60 / div
            if listen_timeout <= 0:
                sys.exit("""Aping: Invalid listen timeout specified (%s)
                \rArgument must to be greater then 0""" %var)
        except ValueError:
            sys.exit("""Aping: Invalid listen timeout specified (%s)
            \rArgument must to be an integer or float value not string""" %listen_timeout)
        self.pkg_to_send_verif()
    
    def extra_data_verif(self):
        "Verifies the entered extra data in packets (payload data)"
        global extra_data
        try:
            extra_data = int(extra_data)
            if extra_data < 0:
                sys.exit("""APing: Invalid extra data specified (%s)
                \rArgument must to be greater or equal with 0""" %extra_data)
        except ValueError:
            sys.exit("""APing: Invalid extra data specified (%s)
                \rArgument must to be an integer not string or float value""" %extra_data)
        self.listen_timeout_verif()
    
    def verbose_verif(self):
        "Verifies the entered verbosity level"
        global verbose
        try:
            verbose = int(verbose)
            if verbose < 0 or verbose > 2:
                sys.exit("""APing: Invalid verbosity level specified (%s)
                \rValid number range is between 0 and 2 (inclusive)""" %verbose)
        except ValueError:
            sys.exit("""APing: Invalid verbosity level specified (%s)
            \rArgument must to be an integer not string or float value""" %verbose)
        self.extra_data_verif()
    
    def dest_addr_verif(self):
        """Verifies the entered IPv4 address (length and number range) or if the
        hostname is correctly specified
        """
        self.isip = 0
        try:
            # check if the target is a hostname or an IP address
            int(dst_address.replace('.', ''))
            
            # if we got here that means that the target must to be an IP address
            # so set the variable to 1 (True)
            self.isip = 1 
            
            # check if the specified IP can be sliced in a four group          
            a, b, c, d = dst_address.split('.')
            
            # check for valid number ranges in the address
            a = int(a); b = int(b); c = int(c); d = int(d)
            if a > 255 or a < 0 or b > 255 or b < 0 or c > 255 or c < 0 or d > 255 or d < 0:
                sys.exit("""APing: Invalid IP address number range specified (%s)
                \rValid number range is between 0 and 255 (inclusive)""" %dst_address)
        except ValueError, error_msg:
            if "need" in str(error_msg) or "many" in str(error_msg):
                sys.exit("""APing: Invalid IP address length specified (%s)
                \rThe address must to be a valid IPv4 address (0.0.0.0 - 255.255.255.255)"""\
                %dst_address)
        self.verbose_verif()

if __name__ == "__main__":
    VerifyPrintpot().dest_addr_verif()
