# This file is Copyright (c) 2014 Florent Kermarrec <florent@enjoy-digital.fr>
# License: BSD

import os
import subprocess
import sys

from migen.fhdl.structure import _Fragment
from migen.build.generic_platform import *
from migen.build import tools
from migen.build.xilinx import common


def _format_constraint(c):
    if isinstance(c, Pins):
        return "set_property LOC " + c.identifiers[0]
    elif isinstance(c, IOStandard):
        return "set_property IOSTANDARD " + c.name
    elif isinstance(c, Drive):
        return "set_property DRIVE " + str(c.strength)
    elif isinstance(c, Misc):
        return "set_property " + c.misc.replace("=", " ")
    else:
        raise ValueError("unknown constraint {}".format(c))


def _format_xdc(signame, resname, *constraints):
    fmt_c = [_format_constraint(c) for c in constraints]
    fmt_r = resname[0] + ":" + str(resname[1])
    if resname[2] is not None:
        fmt_r += "." + resname[2]
    r = " ## {}\n".format(fmt_r)
    for c in fmt_c:
        r += c + " [get_ports " + signame + "]\n"
    return r


def _build_xdc(named_sc, named_pc):
    r = ""
    for sig, pins, others, resname in named_sc:
        if len(pins) > 1:
            for i, p in enumerate(pins):
                r += _format_xdc(sig + "[" + str(i) + "]", resname, Pins(p), *others)
        elif pins:
            r += _format_xdc(sig, resname, Pins(pins[0]), *others)
        else:
            r += _format_xdc(sig, resname, *others)
    if named_pc:
        r += "\n" + "\n\n".join(named_pc)
    return r


def _run_vivado(build_name, vivado_path, source, ver=None):
    if sys.platform == "win32" or sys.platform == "cygwin":
        build_script_contents = "REM Autogenerated by Migen\n"
        build_script_contents += "vivado -mode batch -source " + build_name + ".tcl\n"
        build_script_file = "build_" + build_name + ".bat"
        tools.write_to_file(build_script_file, build_script_contents)
        command = build_script_file
    else:
        build_script_contents = "# Autogenerated by Migen\nset -e\n"
        settings = common.settings(vivado_path, ver)
        build_script_contents += "source " + settings + "\n"
        build_script_contents += "vivado -mode batch -source " + build_name + ".tcl\n"
        build_script_file = "build_" + build_name + ".sh"
        tools.write_to_file(build_script_file, build_script_contents)
        command = ["bash", build_script_file]
    r = tools.subprocess_call_filtered(command, common.colors)
    if r != 0:
        raise OSError("Subprocess failed")


class XilinxVivadoToolchain:
    attr_translate = {
        "keep": ("dont_touch", "true"),
        "no_retiming": ("dont_touch", "true"),
        "async_reg": ("async_reg", "true"),
        "mr_ff": ("mr_ff", "true"),  # user-defined attribute
        "mr_false_path": ("mr_false_path", "true"),  # user-defined attribute
        "ars_ff1": ("ars_ff1", "true"),  # user-defined attribute
        "ars_ff2": ("ars_ff2", "true"),  # user-defined attribute
        "ars_false_path": ("ars_false_path", "true"),  # user-defined attribute
        "no_shreg_extract": None
    }

    def __init__(self):
        self.bitstream_commands = []
        self.additional_commands = []
        self.pre_synthesis_commands = []
        self.with_phys_opt = False
        self.clocks = dict()
        self.false_paths = set()

    def _build_batch(self, platform, sources, build_name):
        tcl = []
        tcl.append("create_project -force -name {} -part {}".format(build_name, platform.device))
        tcl.append("create_property -type bool mr_ff cell")
        tcl.append("create_property -type bool mr_false_path net")
        tcl.append("create_property -type bool ars_ff1 cell")
        tcl.append("create_property -type bool ars_ff2 cell")
        tcl.append("create_property -type bool ars_false_path net")
        for filename, language, library in sources:
            filename_tcl = "{" + filename + "}"
            tcl.append("add_files " + filename_tcl)
            tcl.append("set_property library {} [get_files {}]"
                       .format(library, filename_tcl))

        tcl.append("read_xdc {}.xdc".format(build_name))
        tcl.extend(c.format(build_name=build_name) for c in self.pre_synthesis_commands)
        # "-include_dirs {}" crashes Vivado 2016.4
        if platform.verilog_include_paths:
            tcl.append("synth_design -top top -part {} -include_dirs {{{}}}".format(platform.device, " ".join(platform.verilog_include_paths)))
        else:
            tcl.append("synth_design -top top -part {}".format(platform.device))
        tcl.append("write_checkpoint -force {}_synth.dcp".format(build_name))
        tcl.append("report_timing_summary -file {}_timing_synth.rpt".format(build_name))
        tcl.append("report_utilization -hierarchical -file {}_utilization_hierarchical_synth.rpt".format(build_name))
        tcl.append("report_utilization -file {}_utilization_synth.rpt".format(build_name))
        tcl.append("opt_design")
        tcl.append("place_design")
        if self.with_phys_opt:
            tcl.append("phys_opt_design -directive AddRetime")
        tcl.append("write_checkpoint -force {}_place.dcp".format(build_name))
        tcl.append("report_utilization -hierarchical -file {}_utilization_hierarchical_place.rpt".format(build_name))
        tcl.append("report_utilization -file {}_utilization_place.rpt".format(build_name))
        tcl.append("report_io -file {}_io.rpt".format(build_name))
        tcl.append("report_control_sets -verbose -file {}_control_sets.rpt".format(build_name))
        tcl.append("report_clock_utilization -file {}_clock_utilization.rpt".format(build_name))
        tcl.append("route_design")
        tcl.append("write_checkpoint -force {}_route.dcp".format(build_name))
        tcl.append("report_route_status -file {}_route_status.rpt".format(build_name))
        tcl.append("report_drc -file {}_drc.rpt".format(build_name))
        tcl.append("report_timing_summary -datasheet -max_paths 10 -file {}_timing.rpt".format(build_name))
        tcl.append("report_power -file {}_power.rpt".format(build_name))
        for bitstream_command in self.bitstream_commands:
            tcl.append(bitstream_command.format(build_name=build_name))
        tcl.append("write_bitstream -force {}.bit ".format(build_name))
        for additional_command in self.additional_commands:
            tcl.append(additional_command.format(build_name=build_name))
        tcl.append("quit")
        tools.write_to_file(build_name + ".tcl", "\n".join(tcl))

    def _convert_clocks(self, platform):
        for clk, period in sorted(self.clocks.items(), key=lambda x: x[0].duid):
            platform.add_platform_command(
                "create_clock -name {clk} -period " + str(period) +
                " [get_nets {clk}]", clk=clk)
        for from_, to in sorted(self.false_paths,
                                key=lambda x: (x[0].duid, x[1].duid)):
            if (from_ not in self.clocks
                    or to not in self.clocks):
                raise ValueError("Vivado requires period "
                                 "constraints on all clocks used in false paths")
            platform.add_platform_command(
                "set_false_path -from [get_clocks {from_}] -to [get_clocks {to}]",
                from_=from_, to=to)

        # make sure add_*_constraint cannot be used again
        del self.clocks
        del self.false_paths

    def _constrain(self, platform):
        # The asynchronous input to a MultiReg is a false path
        platform.add_platform_command(
            "set_false_path -quiet "
            "-through [get_nets -hier -filter mr_false_path] "
            "-to [get_cells -hier -filter mr_ff]"
        )
        # The asychronous reset input to the AsyncResetSynchronizer is a false
        # path
        platform.add_platform_command(
            "set_false_path -quiet "
            "-through [get_nets -hier -filter ars_false_path] "
            "-to [get_cells -hier -filter {{ars_ff1 || ars_ff2}}]"
        )
        # clock_period-2ns to resolve metastability on the wire between the
        # AsyncResetSynchronizer FFs
        platform.add_platform_command(
            "set_max_delay 2 -quiet "
            "-from [get_cells -hier -filter ars_ff1] "
            "-to [get_cells -hier -filter ars_ff2]"
        )

    def build(self, platform, fragment, build_dir="build", build_name="top",
            toolchain_path="/opt/Xilinx/Vivado", source=True, run=True):
        os.makedirs(build_dir, exist_ok=True)
        cwd = os.getcwd()
        os.chdir(build_dir)

        if not isinstance(fragment, _Fragment):
            fragment = fragment.get_fragment()
        platform.finalize(fragment)
        self._convert_clocks(platform)
        self._constrain(platform)
        v_output = platform.get_verilog(fragment)
        named_sc, named_pc = platform.resolve_signals(v_output.ns)
        v_file = build_name + ".v"
        v_output.write(v_file)
        sources = platform.sources | {(v_file, "verilog", "work")}
        self._build_batch(platform, sources, build_name)
        tools.write_to_file(build_name + ".xdc", _build_xdc(named_sc, named_pc))
        if run:
            _run_vivado(build_name, toolchain_path, source)

        os.chdir(cwd)

        return v_output.ns

    def add_period_constraint(self, platform, clk, period):
        if clk in self.clocks:
            raise ValueError("A period constraint already exists")
        self.clocks[clk] = period

    def add_false_path_constraint(self, platform, from_, to):
        self.false_paths.add((from_, to))
