/*
 * Copyright (C) 1997 Red Hat Software, Inc.
 *	Cristian Gafton <gafton@redhat.com>
 *
 * Modify:  2014/01/07  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2013/03/19  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/10/03  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/08/09  Tommy Scheunemann <net@arrishq.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * WM configurator
 *
 * root menu management
 */

#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include <strings.h>

#include "wmconfig.h"

/*
 * External data access ... First things we are most concerned
 * about: groups
 */
extern struct group 	*root_group;
extern char		*root_menu_name;
extern struct package	*packages;
extern int		num_packages;
extern unsigned int 	flags;

/*This function initializes the root_group */
static int init_root_group(void)
{
    if (!root_group) {
	/* oops, let's initialize the root group */
	root_group = (struct group *)malloc(sizeof(struct group));
	if (root_group == NULL) {
	    fprintf(stderr, gettext ("%d@%s: out of memory !\n"), __LINE__-1, __FUNCTION__);
	    return -1;
	}
	memset(root_group, 0, sizeof(struct group));
	/* the name of the root group is "" */
	root_group->name = root_menu_name;
    }
    return 0;
}

/*
 * This will create room for a new item in the group
 * hierarchy/anarchy...
 */
static struct group *get_group(char *name)
{
    struct group *grp;
    struct item *item;
    char *ptr;

    if (!name) {
	return NULL;
    }

    grp = root_group;
    ptr = strchr(name, '/');

    /* break down each component */
    if (ptr != NULL) {
	*ptr='\0';
    }

    while ((name != NULL) && *name) {
	int found = 0;
	struct group *tmp;
	
	item = grp->items;
	while (item->type != 0) {
	    if (item->type == ITEM_MENU) {
		tmp = (struct group *)item->data;
		if (strcmp(tmp->name, name) == 0) {
		    /* we've found it */
		    grp = tmp;
		    found++;
		    name += strlen(name)+1;
		    /* if we don't have an hierarchy */
		    if (ptr == NULL) {
			name--;
		    }
		    if (*name) {
			ptr = name - 1;
			if (*ptr == '\0') {
			    *ptr = '/';
			}
			ptr = strchr(name, '/');
			if (ptr != NULL) {
			    *ptr = '\0';
			}
		    }
		    break;
		}
	    }
	    item++;
	}
	if (found) {
	    continue;
	}
	/* okay, we've searched and we've found nothin'. Add a new one */
	item->data = (struct group *)malloc(sizeof(struct group));
	if (item->data == NULL) {
	    fprintf(stderr, gettext ("%d@%s: out of memory!\n"), __LINE__, __FUNCTION__);
	    return NULL;
	}
	memset(item->data, 0, sizeof(struct group));
	item->type = ITEM_MENU;
	tmp = (struct group *)item->data;
	tmp->name = x_strdup(name);
	name += strlen(name)+1;
	/* if we don't have an hierarchy */
	if (ptr == NULL) {
	    name--;
	}
	if (*name) {
	    ptr = name - 1;
	    if (*ptr == '\0') {
		*ptr = '/';
	    }
	    ptr = strchr(name, '/');
	    if (ptr != NULL) {
		*ptr = '\0';
	    }
	}
	grp = tmp;
    }
    return grp;
}

/*
 Compare two packages: sort by type,name
 Directories are put to the start
*/
static int cmpitem(const struct item *a, const struct item *b) {
    if (a->type < b->type) return  1;
    if (a->type > b->type) return -1;
    if (a->type == ITEM_APP) {
	struct package *pa = (struct package *)a->data;
	struct package *pb = (struct package *)b->data;
	return strcasecmp(pa->name, pb->name);
    } else if (a->type == ITEM_MENU) {
	struct group *ga = (struct group *)a->data;
	struct group *gb = (struct group *)b->data;
	return strcasecmp(ga->name, gb->name);
    } else {
	/* other than APP,MENU */
	abort();
    }
}

/* Sort menu */
static void sort_group(struct group *g) {
    struct item *i;
    int n = 0;

    /* nr_items does not seem to count submenus. A bug? */
    for (i = g->items; i->type != 0; i++) {
	n++;
    }

    qsort(g->items, n, sizeof(struct item), (int (*)(const void *, const void *))cmpitem);
}

/* Sort groups recursively. */
static void sort_groups(struct group *g) {
    struct item *item;

    if (g == (struct group *)NULL) {
	return;
    }
    sort_group(g);

    item = g->items;
    while (item->type != 0) {
	if (item->type == ITEM_MENU) {
	    struct group *subgroup = (struct group *)item->data;
	    sort_groups(subgroup);
	}
	item++;
    }
}

/* Get rid of the menus with only one entry (application in them) */
static int promote_groups(struct group *g) {
    struct item *item;
    int n;
    if (g == (struct group *)NULL) {
	return 0;
    }

    /* First, deal recursively */
    item = g->items;
    while (item->type != 0) {
	int retval;
	if (item->type == ITEM_MENU) {
	    struct group *subgroup = (struct group *)item->data;
	    retval = promote_groups(subgroup);
	    if (retval > 0) {

		struct item *tmp;
		struct package *pkg;

		/* we need to promote it */
		tmp = subgroup->items;
		pkg = (struct package *)(tmp->data);

		fprintf(stderr, gettext ("Promoting %s from %s to %s\n"), pkg->name, pkg->group, g->name);

		/* promote it */
		item->data = pkg;
		item->type = ITEM_APP;
		pkg->group = g->name;
		free(subgroup);
	    }
	}
	item++;
    }

    /* now see what is left */
    for (n = 0, item = g->items; item->type != 0; item++) {
	if (item->type == ITEM_MENU) {
	    /* this menu have a submenu */
	    return 0;
	}
	n++;
	if (n > 1) {
	    /* more than one entry */
	    return 0;
	}
    }
    /* This menu have only one application entry */
    return 1;
    /* signal that we need to promote that entry */
}

/* Taking the package list, this will build the root menu */
int build_root(void)
{
    int	i;
    /* first initialize the root_group */
    if (init_root_group() != 0) {
	return -1;
    }

    /* fill in the rest of the packages according to their 'copy' field */
    complete_packages();

    /* Mark packages, which are deemed not to exist, to be skipped */
    if (!is_set(NO_CHECK_EXISTENCE)) {
	unselect_nonexistent_packages();
    }

    /* Take all the packages */
    for (i = 0; i < num_packages; i++) {
	struct group *grp = NULL;
	struct package *pkg = NULL;
	struct item *item = NULL;

	pkg = packages + i;
	if (pkg->group == NULL) {
	    fprintf(stderr, gettext ("ignoring groupless package: %s\n"), pkg->__package);
	    continue;
	}

	if (pkg->name == NULL) {
	    pkg->name = pkg->__package;
	}

	/* Try to ignore packages wich are choosen to be skipped */
	if (strcmp(pkg->group, "-") == 0) {
	    continue;
	}

	grp = get_group(pkg->group);
	if (grp == (struct group *)NULL) {
	    /* This is ugly and it shouldn't happen */
	    fprintf(stderr, gettext ("error processing package configuration...\nAborted.\n"));
	    exit(-2);
	}
	/*  make room for the application */
	item = grp->items;
	while (item->type != 0) {
	    item++;
	}
	item->type = ITEM_APP;
	item->data = pkg;
	grp->nr_items++;
    }

    /* Now walk through this structure and destroy submenus with only one
       application entry (and promote that to a higher level menu  */
    if (is_set(PROMOTE)) {
	promote_groups(root_group);
    }

    /* Final step means sorting the remaining menus */
    sort_groups(root_group);
    return 0;
}
