/* vifm
 * Copyright (C) 2001 Ken Steen.
 * Copyright (C) 2011 xaizek.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#ifdef _WIN32
#define _WIN32_WINNT 0x0500
#include <windows.h>
#endif

#include "vifm.h"

#include <curses.h>

#include <unistd.h> /* getcwd() */

#include <errno.h> /* errno */
#include <locale.h> /* setlocale */
#include <stddef.h> /* NULL size_t */
#include <stdio.h> /* fprintf() fputs() puts() snprintf() */
#include <stdlib.h> /* EXIT_FAILURE EXIT_SUCCESS exit() malloc() system() */
#include <string.h>

#include "cfg/config.h"
#include "cfg/info.h"
#include "engine/cmds.h"
#include "engine/keys.h"
#include "engine/mode.h"
#include "engine/options.h"
#include "engine/variables.h"
#include "modes/dialogs/msg_dialog.h"
#include "modes/modes.h"
#include "modes/view.h"
#include "ui/cancellation.h"
#include "ui/statusbar.h"
#include "ui/ui.h"
#include "utils/fs.h"
#include "utils/fs_limits.h"
#include "utils/log.h"
#include "utils/macros.h"
#include "utils/path.h"
#include "utils/str.h"
#include "utils/utils.h"
#include "args.h"
#include "background.h"
#include "bookmarks.h"
#include "bracket_notation.h"
#include "builtin_functions.h"
#include "color_manager.h"
#include "color_scheme.h"
#include "commands.h"
#include "commands_completion.h"
#include "dir_stack.h"
#include "event_loop.h"
#include "filelist.h"
#include "fileops.h"
#include "filetype.h"
#include "fuse.h"
#include "ipc.h"
#include "ops.h"
#include "opt_handlers.h"
#include "path_env.h"
#include "quickview.h"
#include "registers.h"
#include "running.h"
#include "signals.h"
#include "status.h"
#include "term_title.h"
#include "trash.h"
#include "undo.h"
#include "vim.h"

static int pair_in_use(short int pair);
static void move_pair(short int from, short int to);
static int undo_perform_func(OPS op, void *data, const char src[],
		const char dst[]);
static void parse_received_arguments(char *args[]);
static void remote_cd(FileView *view, const char *path, int handle);
static void check_path_for_file(FileView *view, const char path[], int handle);
static int need_to_switch_active_pane(const char lwin_path[],
		const char rwin_path[]);
static void load_scheme(void);
static void convert_configs(void);
static int run_converter(int vifm_like_mode);
static void exec_startup_commands(const args_t *args);
static void _gnuc_noreturn vifm_leave(int exit_code, int cquit);

/* Command-line arguments in parsed form. */
static args_t vifm_args;

int
main(int argc, char *argv[])
{
	/* TODO: refactor main() function */

	static const int quit = 0;

	char dir[PATH_MAX];
	int old_config;

	if(getcwd(dir, sizeof(dir)) == NULL)
	{
		perror("getcwd");
		return -1;
	}
#ifdef _WIN32
	to_forward_slash(dir);
#endif

	(void)vifm_chdir(dir);
	args_parse(&vifm_args, argc, argv, dir);
	args_process(&vifm_args, 1);

	(void)setlocale(LC_ALL, "");

	cfg_init();

	if(vifm_args.logging)
	{
		init_logger(1);
	}

	init_filelists();
	init_registers();
	cfg_discover_paths();
	reinit_logger();

	/* Commands module also initializes bracket notation and variables. */
	init_commands();

	init_builtin_functions();
	update_path_env(1);

	if(init_status(&cfg) != 0)
	{
		puts("Error during session status initialization.");
		return -1;
	}

	/* Tell file type module what function to use to check availability of
	 * external programs. */
	ft_init(&external_command_exists);
	/* This should be called before loading any configuration file. */
	ft_reset(curr_stats.exec_env_type == EET_EMULATOR_WITH_X);

	init_option_handlers();

	old_config = cfg_has_old_format();
	if(!old_config && !vifm_args.no_configs)
	{
		read_info_file(0);
	}

	ipc_init(&parse_received_arguments);
	args_process(&vifm_args, 0);

	init_background();

	init_fileops();

	set_view_path(&lwin, vifm_args.lwin_path);
	set_view_path(&rwin, vifm_args.rwin_path);

	if(need_to_switch_active_pane(vifm_args.lwin_path, vifm_args.rwin_path))
	{
		swap_view_roles();
	}

	load_initial_directory(&lwin, dir);
	load_initial_directory(&rwin, dir);

	/* Force split view when two paths are specified on command-line. */
	if(vifm_args.lwin_path[0] != '\0' && vifm_args.rwin_path[0] != '\0')
	{
		curr_stats.number_of_windows = 2;
	}

	/* Prepare terminal for further operations. */
	curr_stats.original_stdout = reopen_terminal();
	if(curr_stats.original_stdout == NULL)
	{
		return -1;
	}

	if(!setup_ncurses_interface())
	{
		return -1;
	}

	{
		const colmgr_conf_t colmgr_conf = {
			.max_color_pairs = COLOR_PAIRS,
			.max_colors = COLORS,
			.init_pair = &init_pair,
			.pair_content = &pair_content,
			.pair_in_use = &pair_in_use,
			.move_pair = &move_pair,
		};
		colmgr_init(&colmgr_conf);
	}

	init_modes();
	init_undo_list(&undo_perform_func, NULL, &ui_cancellation_requested,
			&cfg.undo_levels);
	load_local_options(curr_view);

	curr_stats.load_stage = 1;

	if(!old_config && !vifm_args.no_configs)
	{
		load_scheme();
		cfg_load();
	}
	/* Load colors in any case to load color pairs. */
	load_color_scheme_colors();

	write_color_scheme_file();
	setup_signals();

	if(old_config && !vifm_args.no_configs)
	{
		convert_configs();

		curr_stats.load_stage = 0;
		read_info_file(0);
		curr_stats.load_stage = 1;

		set_view_path(&lwin, vifm_args.lwin_path);
		set_view_path(&rwin, vifm_args.rwin_path);

		load_initial_directory(&lwin, dir);
		load_initial_directory(&rwin, dir);

		cfg_load();
	}

	/* Ensure trash directories exist, it might not have been called during
	 * configuration file sourcing if there is no `set trashdir=...` command. */
	(void)set_trash_dir(cfg.trash_dir);

	check_path_for_file(&lwin, vifm_args.lwin_path, vifm_args.lwin_handle);
	check_path_for_file(&rwin, vifm_args.rwin_path, vifm_args.rwin_handle);

	curr_stats.load_stage = 2;

	exec_startup_commands(&vifm_args);
	update_screen(UT_FULL);
	modes_update();

	/* Update histories of the views to ensure that their current directories,
	 * which might have been set using command-line parameters, are stored in the
	 * history.  This is not done automatically as history manipulation should be
	 * postponed until views are fully loaded, otherwise there is no correct
	 * information about current file and relative cursor position. */
	save_view_history(&lwin, NULL, NULL, -1);
	save_view_history(&rwin, NULL, NULL, -1);

	curr_stats.load_stage = 3;

	event_loop(&quit);

	return 0;
}

/* Checks whether pair is being used at the moment.  Returns non-zero if so and
 * zero otherwise. */
static int
pair_in_use(short int pair)
{
	int i;

	for(i = 0; i < MAXNUM_COLOR; ++i)
	{
		if(cfg.cs.pair[i] == pair || lwin.cs.pair[i] == pair ||
				rwin.cs.pair[i] == pair)
		{
			return 1;
		}
	}

	return 0;
}

/* Substitutes old pair number with the new one. */
static void
move_pair(short int from, short int to)
{
	int i;
	for(i = 0; i < MAXNUM_COLOR; ++i)
	{
		if(cfg.cs.pair[i] == from)
		{
			cfg.cs.pair[i] = to;
		}
		if(lwin.cs.pair[i] == from)
		{
			lwin.cs.pair[i] = to;
		}
		if(rwin.cs.pair[i] == from)
		{
			rwin.cs.pair[i] = to;
		}
	}
}

/* perform_operation() interface adaptor for the undo unit. */
static int
undo_perform_func(OPS op, void *data, const char src[], const char dst[])
{
	return perform_operation(op, NULL, data, src, dst);
}

/* Handles arguments received from remote instance. */
static void
parse_received_arguments(char *argv[])
{
	int argc = 0;
	args_t args = {};

	while(argv[argc] != NULL)
	{
		argc++;
	}

	(void)vifm_chdir(argv[0]);
	args_parse(&args, argc, argv, argv[0]);
	args_process(&args, 0);

	exec_startup_commands(&args);
	args_free(&args);

	if(NONE(vle_mode_is, NORMAL_MODE, VIEW_MODE))
	{
		return;
	}

#ifdef _WIN32
	SwitchToThisWindow(GetConsoleWindow(), TRUE);
	BringWindowToTop(GetConsoleWindow());
	SetForegroundWindow(GetConsoleWindow());
#endif

	if(view_needs_cd(&lwin, args.lwin_path))
	{
		remote_cd(&lwin, args.lwin_path, args.lwin_handle);
	}

	if(view_needs_cd(&rwin, args.rwin_path))
	{
		remote_cd(&rwin, args.rwin_path, args.rwin_handle);
	}

	if(need_to_switch_active_pane(args.lwin_path, args.rwin_path))
	{
		change_window();
	}

	clean_status_bar();
	curr_stats.save_msg = 0;
}

static void
remote_cd(FileView *view, const char *path, int handle)
{
	char buf[PATH_MAX];

	if(view->explore_mode)
	{
		leave_view_mode();
	}

	if(view == other_view && vle_mode_is(VIEW_MODE))
	{
		leave_view_mode();
	}

	if(curr_stats.view)
	{
		toggle_quick_view();
	}

	copy_str(buf, sizeof(buf), path);
	exclude_file_name(buf);

	(void)cd(view, view->curr_dir, buf);
	check_path_for_file(view, path, handle);
}

/* Navigates to/opens (handles) file specified by the path (and file only, no
 * directories). */
static void
check_path_for_file(FileView *view, const char path[], int handle)
{
	if(path[0] == '\0' || is_dir(path))
	{
		return;
	}

	load_dir_list(view, !(cfg.vifm_info&VIFMINFO_SAVEDIRS));
	if(ensure_file_is_selected(view, after_last(path, '/')))
	{
		if(handle)
		{
			open_file(view, FHE_RUN);
		}
	}
}

/* Decides whether active view should be switched based on paths provided for
 * panes on the command-line.  Returns non-zero if so, otherwise zero is
 * returned. */
static int
need_to_switch_active_pane(const char lwin_path[], const char rwin_path[])
{
	/* Forces view switch when path is specified for invisible pane. */
	return lwin_path[0] != '\0'
	    && rwin_path[0] == '\0'
	    && curr_view != &lwin;
}

/* Loads color scheme.  Converts old format to the new one if needed.
 * Terminates application with error message on error. */
static void
load_scheme(void)
{
	if(cfg_has_old_color_schemes())
	{
		const int err = run_converter(2);
		if(err != 0)
		{
			endwin();
			fputs("Problems with running vifmrc-converter\n", stderr);
			exit(err);
		}
	}
	else if(cs_have_no_extensions())
	{
		const int err = run_converter(3);
		if(err != 0)
		{
			endwin();
			fputs("Problems with running vifmrc-converter\n", stderr);
			exit(err);
		}
	}
	if(color_scheme_exists(curr_stats.color_scheme))
	{
		load_primary_color_scheme(curr_stats.color_scheme);
	}
}

/* Converts old versions of configuration files to new ones.  Terminates
 * application with error message on error or when user chooses to do not update
 * anything. */
static void
convert_configs(void)
{
	int vifm_like_mode;
	int err;

	if(!prompt_msg("Configuration update", "Your vifmrc will be "
			"upgraded to a new format.  Your current configuration will be copied "
			"before performing any changes, but if you don't want to take the risk "
			"and would like to make one more copy say No to exit vifm.  Continue?"))
	{
#ifdef _WIN32
		system("cls");
#endif
		endwin();
		exit(0);
	}

	vifm_like_mode = !prompt_msg("Configuration update", "This version of "
			"vifm is able to save changes in the configuration files automatically "
			"when quitting, as it was possible in older versions.  It is from now "
			"on recommended though, to save permanent changes manually in the "
			"configuration file as it is done in vi/vim.  Do you want vifm to "
			"behave like vi/vim?");

	err = run_converter(vifm_like_mode);
	if(err != 0)
	{
		endwin();
		fputs("Problems with running vifmrc-converter\n", stderr);
		exit(err);
	}

	show_error_msg("Configuration update", "Your vifmrc has been upgraded to "
			"new format, you can find its old version in " CONF_DIR "/vifmrc.bak.  "
			"vifm will not write anything to vifmrc, and all variables that are "
			"saved between runs of vifm are stored in " CONF_DIR "/vifminfo now "
			"(you can edit it by hand, but do it carefully).  You can control what "
			"vifm stores in vifminfo with 'vifminfo' option.");
}

/* Runs vifmrc-converter in mode specified by the vifm_like_mode argument.
 * Returns zero on success, non-zero otherwise. */
static int
run_converter(int vifm_like_mode)
{
#ifndef _WIN32
	char cmd[PATH_MAX];
	snprintf(cmd, sizeof(cmd), "vifmrc-converter %d", vifm_like_mode);
	return shellout(cmd, -1, 0);
#else
	char path[PATH_MAX];
	char cmd[2*PATH_MAX];
	int returned_exit_code;

	if(get_exe_dir(path, sizeof(path)) != 0)
	{
		return -1;
	}

	snprintf(cmd, sizeof(cmd), "%s/vifmrc-converter %d", path, vifm_like_mode);
	return win_exec_cmd(cmd, &returned_exit_code);
#endif
}

void
vifm_restart(void)
{
	FileView *tmp_view;

	curr_stats.restart_in_progress = 1;

	/* All user mappings in all modes. */
	clear_user_keys();

	/* User defined commands. */
	execute_cmd("comclear");

	/* Directory histories. */
	ui_view_clear_history(&lwin);
	ui_view_clear_history(&rwin);

	/* All kinds of history. */
	(void)hist_reset(&cfg.search_hist, cfg.history_len);
	(void)hist_reset(&cfg.cmd_hist, cfg.history_len);
	(void)hist_reset(&cfg.prompt_hist, cfg.history_len);
	(void)hist_reset(&cfg.filter_hist, cfg.history_len);
	cfg.history_len = 0;

	/* Session status.  Must be reset _before_ options, because options take some
	 * of values from status. */
	(void)reset_status(&cfg);

	/* Options of current pane. */
	reset_options_to_default();
	/* Options of other pane. */
	tmp_view = curr_view;
	curr_view = other_view;
	load_local_options(other_view);
	reset_options_to_default();
	curr_view = tmp_view;

	/* File types and viewers. */
	ft_reset(curr_stats.exec_env_type == EET_EMULATOR_WITH_X);

	/* Undo list. */
	reset_undo_list();

	/* Directory stack. */
	clean_stack();

	/* Registers. */
	clear_registers();

	/* Clear all bookmarks. */
	clear_all_bookmarks();

	/* Reset variables. */
	clear_variables();
	init_variables();
	/* This update is needed as clear_variables() will reset $PATH. */
	update_path_env(1);

	reset_views();
	read_info_file(1);
	save_view_history(&lwin, NULL, NULL, -1);
	save_view_history(&rwin, NULL, NULL, -1);

	/* Color schemes. */
	if(stroscmp(curr_stats.color_scheme, DEF_CS_NAME) != 0 &&
			color_scheme_exists(curr_stats.color_scheme))
	{
		if(load_primary_color_scheme(curr_stats.color_scheme) != 0)
		{
			load_def_scheme();
		}
	}
	else
	{
		load_def_scheme();
	}
	load_color_scheme_colors();

	cfg_load();
	exec_startup_commands(&vifm_args);

	curr_stats.restart_in_progress = 0;

	update_screen(UT_REDRAW);
}

/* Executes list of startup commands. */
static void
exec_startup_commands(const args_t *args)
{
	size_t i;
	for(i = 0; i < args->ncmds; ++i)
	{
		(void)exec_commands(args->cmds[i], curr_view, CIT_COMMAND);
	}
}

void
vifm_try_leave(int write_info, int cquit, int force)
{
	if(!force && bg_has_active_jobs())
	{
		if(!prompt_msg("Warning", "Some of backgrounded commands are still "
					"working.  Quit?"))
		{
			return;
		}
	}

	fuse_unmount_all();

	if(write_info)
	{
		write_info_file();
	}

	if(stats_file_choose_action_set())
	{
		vim_write_empty_file_list();
	}

#ifdef _WIN32
	system("cls");
#endif

	endwin();
	vifm_leave(EXIT_SUCCESS, cquit);
}

void _gnuc_noreturn
vifm_choose_files(const FileView *view, int nfiles, char *files[])
{
	int exit_code;

	/* As curses can do something with terminal on shutting down, disable it
	 * before writing anything to the screen. */
	endwin();

	exit_code = EXIT_SUCCESS;
	if(vim_write_file_list(view, nfiles, files) != 0)
	{
		exit_code = EXIT_FAILURE;
	}
	/* XXX: this ignores nfiles+files. */
	if(vim_run_choose_cmd(view) != 0)
	{
		exit_code = EXIT_FAILURE;
	}

	write_info_file();

	vifm_leave(exit_code, 0);
}

/* Single exit point for leaving vifm, performs only minimum common
 * deinitialization steps. */
static void _gnuc_noreturn
vifm_leave(int exit_code, int cquit)
{
	vim_write_dir(cquit ? "" : flist_get_dir(curr_view));

	if(cquit && exit_code == EXIT_SUCCESS)
	{
		exit_code = EXIT_FAILURE;
	}

	set_term_title(NULL);
	exit(exit_code);
}

void _gnuc_noreturn
vifm_finish(const char message[])
{
	endwin();
	write_info_file();
	fprintf(stderr, "%s\n", message);
	LOG_ERROR_MSG("Finishing: %s", message);
	exit(EXIT_FAILURE);
}

/* vim: set tabstop=2 softtabstop=2 shiftwidth=2 noexpandtab cinoptions-=(0 : */
/* vim: set cinoptions+=t0 : */
