/* vifm
 * Copyright (C) 2001 Ken Steen.
 * Copyright (C) 2011 xaizek.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "fuse.h"

#include <curses.h> /* werase() def_prog_mode() */

#include <sys/stat.h> /* S_IRWXU */
#include <unistd.h> /* rmdir() unlink() */

#include <errno.h> /* errno */
#include <stddef.h> /* NULL size_t */
#include <stdio.h> /* snprintf() fclose() */
#include <stdlib.h> /* EXIT_SUCCESS WIFEXITED free() malloc() */
#include <string.h> /* memmove() strcpy() strlen() strcmp() strcat() */

#include "cfg/config.h"
#include "compat/os.h"
#include "menus/menus.h"
#include "modes/dialogs/msg_dialog.h"
#include "ui/statusbar.h"
#include "ui/ui.h"
#include "utils/fs.h"
#include "utils/fs_limits.h"
#include "utils/log.h"
#include "utils/macros.h"
#include "utils/path.h"
#include "utils/str.h"
#include "utils/test_helpers.h"
#include "utils/utils.h"
#include "background.h"
#include "filelist.h"
#include "status.h"

/* Description of existing FUSE mounts. */
typedef struct fuse_mount_t
{
	char source_file_path[PATH_MAX]; /* Full path to source file. */
	char source_file_dir[PATH_MAX];  /* Full path to directory of source file. */
	char mount_point[PATH_MAX];      /* Full path to mount point. */
	int mount_point_id;              /* Identifier of mounts for unique dirs. */
	struct fuse_mount_t *next;       /* Pointer to the next mount in chain. */
}
fuse_mount_t;

static int fuse_mount(FileView *view, char file_full_path[], const char param[],
		const char program[], char mount_point[]);
static int get_last_mount_point_id(const fuse_mount_t *mounts);
static void register_mount(fuse_mount_t **mounts, const char file_full_path[],
		const char mount_point[], int id);
TSTATIC void format_mount_command(const char mount_point[],
		const char file_name[], const char param[], const char format[],
		size_t buf_size, char buf[], int *foreground);
static fuse_mount_t * get_mount_by_source(const char source[]);
static fuse_mount_t * get_mount_by_mount_point(const char dir[]);
static fuse_mount_t * get_mount_by_path(const char path[]);
static void updir_from_mount(FileView *view, fuse_mount_t *runner);

/* List of active mounts. */
static fuse_mount_t *fuse_mounts;

void
fuse_try_mount(FileView *view, const char program[])
{
	/* TODO: refactor this function fuse_try_mount() */

	fuse_mount_t *runner;
	char file_full_path[PATH_MAX];
	char mount_point[PATH_MAX];

	if(!path_exists(cfg.fuse_home, DEREF))
	{
		if(make_path(cfg.fuse_home, S_IRWXU) != 0)
		{
			show_error_msg("Unable to create FUSE mount home directory",
					cfg.fuse_home);
			return;
		}
	}

	get_current_full_path(view, sizeof(file_full_path), file_full_path);

	/* Check if already mounted. */
	runner = get_mount_by_source(file_full_path);

	if(runner != NULL)
	{
		strcpy(mount_point, runner->mount_point);
	}
	else
	{
		char param[PATH_MAX];
		param[0] = '\0';

		/* New file to be mounted. */
		if(starts_with(program, "FUSE_MOUNT2"))
		{
			FILE *f;
			if((f = os_fopen(file_full_path, "r")) == NULL)
			{
				show_error_msg("SSH mount failed", "Can't open file for reading");
				curr_stats.save_msg = 1;
				return;
			}

			if(fgets(param, sizeof(param), f) == NULL)
			{
				show_error_msg("SSH mount failed", "Can't read file content");
				curr_stats.save_msg = 1;
				fclose(f);
				return;
			}
			fclose(f);

			chomp(param);
			if(param[0] == '\0')
			{
				show_error_msg("SSH mount failed", "File is empty");
				curr_stats.save_msg = 1;
				return;
			}

		}
		if(fuse_mount(view, file_full_path, param, program, mount_point) != 0)
		{
			return;
		}
	}

	navigate_to(view, mount_point);
}

/* Searchers for mount record by source file path. */
static fuse_mount_t *
get_mount_by_source(const char source[])
{
	fuse_mount_t *runner = fuse_mounts;
	while(runner != NULL)
	{
		if(paths_are_equal(runner->source_file_path, source))
			break;
		runner = runner->next;
	}
	return runner;
}

/* mount_point should be an array of at least PATH_MAX characters
 * Returns non-zero on error. */
static int
fuse_mount(FileView *view, char file_full_path[], const char param[],
		const char program[], char mount_point[])
{
	/* TODO: refactor this function fuse_mount(). */

	int mount_point_id;
	char buf[2*PATH_MAX];
	char *escaped_filename;
	int foreground;
	char errors_file[PATH_MAX];
	int status;
	int cancelled;

	escaped_filename = escape_filename(get_current_file_name(view), 0);

	mount_point_id = get_last_mount_point_id(fuse_mounts);
	do
	{
		snprintf(mount_point, PATH_MAX, "%s/%03d_%s", cfg.fuse_home,
				++mount_point_id, get_current_file_name(view));
	}
	while(path_exists(mount_point, DEREF));
	if(os_mkdir(mount_point, S_IRWXU) != 0)
	{
		free(escaped_filename);
		show_error_msg("Unable to create FUSE mount directory", mount_point);
		return -1;
	}
	free(escaped_filename);

	/* Just before running the mount,
		 I need to chdir out temporarily from any FUSE mounted
		 paths, Otherwise the fuse-zip command fails with
		 "fusermount: failed to open current directory: permission denied"
		 (this happens when mounting JARs from mounted JARs) */
	if(vifm_chdir(cfg.fuse_home) != 0)
	{
		show_error_msg("FUSE MOUNT ERROR", "Can't chdir() to FUSE home");
		return -1;
	}

	format_mount_command(mount_point, file_full_path, param, program, sizeof(buf),
			buf, &foreground);

	status_bar_message("FUSE mounting selected file, please stand by..");

	if(foreground)
	{
		def_prog_mode();
		endwin();
	}

	generate_tmp_file_name("vifm.errors", errors_file, sizeof(errors_file));

	strcat(buf, " 2> ");
	strcat(buf, errors_file);
	LOG_INFO_MSG("FUSE mount command: `%s`", buf);
	status = background_and_wait_for_status(buf, !foreground, &cancelled);

	clean_status_bar();

	/* Check child process exit status. */
	if(!WIFEXITED(status) || WEXITSTATUS(status) != EXIT_SUCCESS)
	{
		FILE *ef;

		if(!WIFEXITED(status))
		{
			LOG_ERROR_MSG("FUSE mounter didn't exit!");
		}
		else
		{
			LOG_ERROR_MSG("FUSE mount command exit status: %d", WEXITSTATUS(status));
		}

		ef = os_fopen(errors_file, "r");
		if(ef == NULL)
		{
			LOG_SERROR_MSG(errno, "Failed to open temporary stderr file: %s",
					errors_file);
		}
		show_errors_from_file(ef, "FUSE mounter error");

		werase(status_bar);

		if(cancelled)
		{
			status_bar_message("FUSE mount cancelled");
			curr_stats.save_msg = 1;
		}
		else
		{
			show_error_msg("FUSE MOUNT ERROR", file_full_path);
		}

		if(unlink(errors_file) != 0)
		{
			LOG_SERROR_MSG(errno, "Error file deletion failure: %d", errors_file);
		}

		/* Remove the directory we created for the mount. */
		(void)rmdir(mount_point);

		(void)vifm_chdir(flist_get_dir(view));
		return -1;
	}
	unlink(errors_file);
	status_bar_message("FUSE mount success");

	register_mount(&fuse_mounts, file_full_path, mount_point, mount_point_id);

	return 0;
}

/* Gets last mount point id used.  Returns the id or 0 if list of mounts is
 * empty. */
static int
get_last_mount_point_id(const fuse_mount_t *mounts)
{
	/* As new entries are added at the front, first entry must have the largest
	 * value of the id. */
	return (mounts == NULL) ? 0 : mounts->mount_point_id;
}

/* Adds new entry to the list of *mounts. */
static void
register_mount(fuse_mount_t **mounts, const char file_full_path[],
		const char mount_point[], int id)
{
	fuse_mount_t *fuse_mount = malloc(sizeof(*fuse_mount));

	copy_str(fuse_mount->source_file_path, sizeof(fuse_mount->source_file_path),
			file_full_path);

	copy_str(fuse_mount->source_file_dir, sizeof(fuse_mount->source_file_dir),
			file_full_path);
	remove_last_path_component(fuse_mount->source_file_dir);

	canonicalize_path(mount_point, fuse_mount->mount_point,
			sizeof(fuse_mount->mount_point));

	fuse_mount->mount_point_id = id;

	fuse_mount->next = *mounts;
	*mounts = fuse_mount;
}

/* Builds the mount command based on the file type program.
 * Accepted formats are:
 *   FUSE_MOUNT|some_mount_command %SOURCE_FILE %DESTINATION_DIR [%FOREGROUND]
 * and
 *   FUSE_MOUNT2|some_mount_command %PARAM %DESTINATION_DIR [%FOREGROUND]
 * %CLEAR is an obsolete name of %FOREGROUND.
 * Always sets value of *foreground. */
TSTATIC void
format_mount_command(const char mount_point[], const char file_name[],
		const char param[], const char format[], size_t buf_size, char buf[],
		int *foreground)
{
	char *buf_pos;
	const char *prog_pos;
	char *escaped_path;
	char *escaped_mount_point;

	*foreground = 0;

	escaped_path = escape_filename(file_name, 0);
	escaped_mount_point = escape_filename(mount_point, 0);

	buf_pos = buf;
	buf_pos[0] = '\0';

	prog_pos = after_first(format, '|');
	while(*prog_pos != '\0')
	{
		if(*prog_pos == '%')
		{
			char cmd_buf[96];
			char *cmd_pos;

			cmd_pos = cmd_buf;
			while(*prog_pos != '\0' && *prog_pos != ' ')
			{
				*cmd_pos = *prog_pos;
				if((size_t)(cmd_pos - cmd_buf) < sizeof(cmd_buf))
				{
					++cmd_pos;
				}
				++prog_pos;
			}
			*cmd_pos = '\0';

			if(!strcmp(cmd_buf, "%SOURCE_FILE"))
			{
				copy_str(buf_pos, buf_size - (buf_pos - buf), escaped_path);
				buf_pos += strlen(buf_pos);
			}
			else if(!strcmp(cmd_buf, "%PARAM"))
			{
				copy_str(buf_pos, buf_size - (buf_pos - buf), param);
				buf_pos += strlen(buf_pos);
			}
			else if(!strcmp(cmd_buf, "%DESTINATION_DIR"))
			{
				copy_str(buf_pos, buf_size - (buf_pos - buf), escaped_mount_point);
				buf_pos += strlen(buf_pos);
			}
			else if(!strcmp(cmd_buf, "%FOREGROUND") || !strcmp(cmd_buf, "%CLEAR"))
			{
				*foreground = 1;
			}
		}
		else
		{
			*buf_pos = *prog_pos;
			if((size_t)(buf_pos - buf) < buf_size - 1)
			{
				++buf_pos;
			}
			++prog_pos;
		}
	}

	*buf_pos = '\0';
	free(escaped_mount_point);
	free(escaped_path);
}

void
fuse_unmount_all(void)
{
	fuse_mount_t *runner;

	if(fuse_mounts == NULL)
	{
		return;
	}

	if(vifm_chdir("/") != 0)
	{
		return;
	}

	runner = fuse_mounts;
	while(runner != NULL)
	{
		char buf[14 + PATH_MAX + 1];
		char *escaped_filename;

		escaped_filename = escape_filename(runner->mount_point, 0);
		snprintf(buf, sizeof(buf), "%s %s", curr_stats.fuse_umount_cmd,
				escaped_filename);
		free(escaped_filename);

		(void)vifm_system(buf);
		if(path_exists(runner->mount_point, DEREF))
		{
			rmdir(runner->mount_point);
		}

		runner = runner->next;
	}

	leave_invalid_dir(&lwin);
	leave_invalid_dir(&rwin);
}

int
fuse_try_updir_from_a_mount(const char path[], FileView *view)
{
	fuse_mount_t *const mount = get_mount_by_mount_point(path);
	if(mount == NULL)
	{
		return 0;
	}

	updir_from_mount(view, mount);
	return 1;
}

int
fuse_is_mount_point(const char path[])
{
	return get_mount_by_mount_point(path) != NULL;
}

/* Searches for mount record by path to mount point.  Returns mount point or
 * NULL on failure. */
static fuse_mount_t *
get_mount_by_mount_point(const char dir[])
{
	fuse_mount_t *runner = fuse_mounts;
	while(runner != NULL)
	{
		if(paths_are_equal(runner->mount_point, dir))
		{
			return runner;
		}
		runner = runner->next;
	}
	return NULL;
}

const char *
fuse_get_mount_file(const char path[])
{
	const fuse_mount_t *const mount = get_mount_by_path(path);
	return (mount == NULL) ? NULL : mount->source_file_path;
}

/* Searches for mount record by path inside one of mount points.  Picks the
 * longest match so that even nested mount points work.  Returns mount point or
 * NULL on failure. */
static fuse_mount_t *
get_mount_by_path(const char path[])
{
	size_t max_len = 0U;
	fuse_mount_t *mount = NULL;
	fuse_mount_t *runner = fuse_mounts;
	while(runner != NULL)
	{
		if(path_starts_with(path, runner->mount_point))
		{
			const size_t len = strlen(runner->mount_point);
			if(len > max_len)
			{
				max_len = len;
				mount = runner;
			}
		}
		runner = runner->next;
	}
	return mount;
}

int
fuse_try_unmount(FileView *view)
{
	char buf[14 + PATH_MAX + 1];
	fuse_mount_t *runner, *trailer;
	int status;
	fuse_mount_t *sniffer;
	char *escaped_mount_point;

	runner = fuse_mounts;
	trailer = NULL;
	while(runner)
	{
		if(paths_are_equal(runner->mount_point, view->curr_dir))
		{
			break;
		}

		trailer = runner;
		runner = runner->next;
	}

	if(runner == NULL)
	{
		return 0;
	}

	/* we are exiting a top level dir */
	escaped_mount_point = escape_filename(runner->mount_point, 0);
	snprintf(buf, sizeof(buf), "%s %s 2> /dev/null", curr_stats.fuse_umount_cmd,
			escaped_mount_point);
	LOG_INFO_MSG("FUSE unmount command: `%s`", buf);
	free(escaped_mount_point);

	/* Have to chdir to parent temporarily, so that this DIR can be unmounted. */
	if(vifm_chdir(cfg.fuse_home) != 0)
	{
		show_error_msg("FUSE UMOUNT ERROR", "Can't chdir to FUSE home");
		return -1;
	}

	status_bar_message("FUSE unmounting selected file, please stand by..");
	status = background_and_wait_for_status(buf, 0, NULL);
	clean_status_bar();
	/* check child status */
	if(!WIFEXITED(status) || WEXITSTATUS(status))
	{
		werase(status_bar);
		show_error_msgf("FUSE UMOUNT ERROR", "Can't unmount %s.  It may be busy.",
				runner->source_file_path);
		(void)vifm_chdir(flist_get_dir(view));
		return -1;
	}

	/* remove the directory we created for the mount */
	if(path_exists(runner->mount_point, DEREF))
		rmdir(runner->mount_point);

	/* remove mount point from fuse_mount_t */
	sniffer = runner->next;
	if(trailer)
		trailer->next = sniffer ? sniffer : NULL;
	else
		fuse_mounts = sniffer;

	updir_from_mount(view, runner);
	free(runner);
	return 1;
}

static void
updir_from_mount(FileView *view, fuse_mount_t *runner)
{
	char *file;
	int pos;

	if(change_directory(view, runner->source_file_dir) < 0)
		return;

	load_dir_list(view, 0);

	file = runner->source_file_path;
	file += strlen(runner->source_file_dir) + 1;
	pos = find_file_pos_in_list(view, file);
	flist_set_pos(view, pos);
}

int
fuse_is_mount_string(const char string[])
{
	return starts_with(string, "FUSE_MOUNT|") ||
		starts_with(string, "FUSE_MOUNT2|");
}

void
fuse_strip_mount_metadata(char string[])
{
	size_t prefix_len;
	if(starts_with(string, "FUSE_MOUNT|"))
	{
		prefix_len = ARRAY_LEN("FUSE_MOUNT|") - 1;
	}
	else if(starts_with(string, "FUSE_MOUNT2|"))
	{
		prefix_len = ARRAY_LEN("FUSE_MOUNT2|") - 1;
	}
	else
	{
		prefix_len = 0;
	}

	if(prefix_len != 0)
	{
		size_t new_len = strlen(string) - prefix_len;
		memmove(string, string + prefix_len, new_len + 1);
	}
}

/* vim: set tabstop=2 softtabstop=2 shiftwidth=2 noexpandtab cinoptions-=(0 : */
/* vim: set cinoptions+=t0 : */
