/*
   Copyright (C) 2008, 2010, 2011, 2012, 2013 Glad Deschrijver
     <glad.deschrijver@gmail.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 3 or later as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "katelatexthread.h"
//#include "katelatexthread.moc"

#include <QtCore/QDateTime>
#include <QtCore/QFileInfo>

#include <KConfigGroup>
#include <KGlobal>
#include <KLocalizedString>
#include <KProcess>
#include <KShell>
#include <KTextEdit>
#include <kate/mainwindow.h>
#include <ktexteditor/document.h>
#include <ktexteditor/view.h>

KateLatexThread::KateLatexThread(Kate::MainWindow *mw, QWidget *parent)
    : QThread(parent)
    , m_mw(mw)
    , m_kateLatexEdit(0)
    , m_autoRerun(true)
    , m_autoRunViewer(true)
    , m_whichTool(0)
    , m_proc(0)
    , m_processAborted(false)
    , m_latexErrorIndex(-1)
{
}

KateLatexThread::~KateLatexThread()
{
}

void KateLatexThread::setOutputBox(KTextEdit *edit)
{
	const QMutexLocker lock(&m_memberLock);
	m_kateLatexEdit = edit;
}

/*** Build ***/

bool KateLatexThread::checkLogForRegExp(const QString &regExp, int startLine)
{
	const QMutexLocker lock(&m_memberLock);
	QTextCursor cursor(m_kateLatexEdit->document());
	cursor.movePosition(QTextCursor::Start);
	cursor.movePosition(QTextCursor::Down, QTextCursor::MoveAnchor, startLine);

	if (!m_kateLatexEdit->document()->find(regExp, cursor.position()).isNull())
		return true;

	return false;
}

bool KateLatexThread::checkRerun(int startLine)
{
	return checkLogForRegExp("Rerun to get", startLine); // Rerun to get cross-references right
}

bool KateLatexThread::checkRunBibtex(const QString &bblFile)
{
	const QMutexLocker lock(&m_memberLock);
	if (!m_mw->activeView()->document()->text().contains(QRegExp("\\\\bibliography\\{[^\\}]+\\}")))
		return false;

	const QFileInfo fi(bblFile);
	if (!fi.exists())
		return true;

//	return checkLogForRegExp("Citation.*undefined"); // doesn't work if on multiple lines
	// search for a warning "LaTeX Warning: Citation ... on page ... undefined"; if the citation labels are long, this error can be written on two lines
	QTextCursor cursor = m_kateLatexEdit->document()->find("LaTeX Warning: Citation");
	while (!cursor.isNull())
	{
		const QTextBlock block = cursor.block();
		QString text = block.text() + block.next().text(); // we assume that the user's labels are not *that* long that this error needs 3 lines to be displayed
		if (text.contains(QRegExp("Citation `.*' on page .* undefined")))
			return true;
		cursor = m_kateLatexEdit->document()->find("LaTeX Warning: Citation", cursor);
	}
	return false;
}

bool KateLatexThread::checkRunMakeindex(const QString &texFile, const QString &indFile)
{
	const QMutexLocker lock(&m_memberLock);
	if (!m_mw->activeView()->document()->text().contains(QRegExp("\\\\usepackage\\{[^\\}]*makeidx")))
		return false;

	const QFileInfo fi(indFile);
	if (!fi.exists())
		return true;

	const QFileInfo fi2(texFile);
	if (fi.lastModified() < fi2.lastModified())
		return true;

	return false;
}

static QString parseOption(const QString &option, const QString &baseName, const QString &extension, int currentLine)
{
	QString optionLine = option;
	optionLine.replace("%%", baseName + '.' + extension);
	optionLine.replace('%', baseName);
	optionLine.replace('@', QString::number(currentLine));
	return optionLine;
}

void KateLatexThread::readFromStandardOutput()
{
	QByteArray output;
	m_memberLock.lock();
	if (m_proc)
		output = m_proc->readAllStandardOutput();
	m_memberLock.unlock();
	const QString outputString(output);
	if (!outputString.isEmpty())
		Q_EMIT outputAppendText(outputString);
}

bool KateLatexThread::runCommand(const QString &toolName, const QString &command, const QStringList &optionList, const QString &workingDir, bool runDetached)
{
	// Start process
kDebug() << command << optionList;
	m_memberLock.lock();
	m_proc = new KProcess;
	m_proc->setWorkingDirectory(workingDir);
	m_proc->setProgram(command, optionList);
	m_proc->setOutputChannelMode(KProcess::MergedChannels);
	connect(m_proc, SIGNAL(readyReadStandardOutput()), this, SLOT(readFromStandardOutput()));

	bool runDetachedSuccess = false;
	m_processAborted = false;
	if (runDetached)
		runDetachedSuccess = m_proc->startDetached();
	else
	{
		m_proc->start();
		Q_EMIT processRunning(true);
	}
	m_memberLock.unlock();
	m_memberLock.lock();
	if (!runDetachedSuccess && (!m_proc || !m_proc->waitForStarted(1000)))
	{
		Q_EMIT outputAppendText('[' + toolName + "] " + i18n("Error: could not start the command \"%1 %2\".", command, optionList.join(" ")));
		delete m_proc;
		m_proc = 0;
		m_memberLock.unlock();
		Q_EMIT processRunning(false);
		return false;
	}
	m_memberLock.unlock();

	// Finish process (do not protect this with a mutex because we want to see the output of m_proc and we want to be able to kill m_proc while m_proc is running)
	m_proc->waitForFinished();
	Q_EMIT processRunning(false);

	// Get result of process
	readFromStandardOutput(); // make sure the complete log is displayed
	m_memberLock.lock();
	QString result;
	bool success = false;
	if (m_processAborted)
	{
		Q_EMIT outputAppendText("\n\n[" + toolName + "] " + i18n("Process aborted."));
		success = false;
	}
	else if (m_proc && m_proc->exitCode() == 0)
	{
		Q_EMIT outputAppendText("\n\n[" + toolName + "] " + i18n("Process exited with success."));
		success = true;
	}
	else
	{
		Q_EMIT outputAppendText("\n\n[" + toolName + "] " + i18n("Process exited with error(s)."));
		success = false;
	}
	m_memberLock.unlock();

	m_memberLock.lock();
	delete m_proc;
	m_proc = 0;
	m_memberLock.unlock();

	return success;
}

/*** Thread functions ***/

void KateLatexThread::abortProcess()
{
	const QMutexLocker lock(&m_memberLock);
	if (m_proc)
	{
		m_proc->kill();
		m_processAborted = true;
	}
}

void KateLatexThread::runTool(int which)
{
	if (isRunning())
		return;

	m_whichTool = which;
	start();
}

void KateLatexThread::run()
{
	m_memberLock.lock();
	if (!m_mw->activeView())
	{
		m_memberLock.unlock();
		return;
	}

	const KUrl documentUrl = m_mw->activeView()->document()->url();
	const int currentLine = m_mw->activeView()->cursorPosition().line();
	if (!documentUrl.isValid() || !documentUrl.isLocalFile())
	{
		m_memberLock.unlock();
		return;
	}
	const QString latexCommand = m_latexCommand;
	const QString latexOptions = m_latexOptions;
	const QString bibtexCommand = m_bibtexCommand;
	const QString bibtexOptions = m_bibtexOptions;
	const QString makeindexCommand = m_makeindexCommand;
	const QString makeindexOptions = m_makeindexOptions;
	const bool autoRerun = m_autoRerun;
	const QString viewerCommand = m_viewerCommand;
	const QString viewerOptions = m_viewerOptions;
	const bool autoRunViewer = m_autoRunViewer;
	const int whichTool = m_whichTool;
	m_memberLock.unlock();

	Q_EMIT outputClear();

	const QFileInfo fi(documentUrl.fileName());
	const QString baseName = fi.completeBaseName();
	const QString workingDir = documentUrl.directory();
	const QString extension = fi.suffix();
	const QString absoluteBaseName = workingDir + '/' + baseName;
	const QString bblFile = absoluteBaseName + ".bbl";
	const QString indFile = absoluteBaseName + ".ind";

	QStringList optionList;
	bool success;
	switch (whichTool)
	{
		case 0: // run LaTeX (and rerun if necessary)
			optionList = KShell::splitArgs(latexOptions);
			for (int i = 0; i < optionList.size(); ++i)
				optionList[i] = parseOption(optionList.at(i), baseName, extension, currentLine);
			success = runCommand("LaTeX", latexCommand, optionList, workingDir);
			findLatexErrors();
			if (success && autoRerun)
			{
				bool rerunLatex = false;
				if (checkRunBibtex(bblFile)) // run BibTeX
				{
					optionList = KShell::splitArgs(bibtexOptions);
					for (int i = 0; i < optionList.size(); ++i)
						optionList[i] = parseOption(optionList.at(i), baseName, extension, currentLine);
					success = runCommand("BibTeX", bibtexCommand, optionList, workingDir);
					rerunLatex = true;
				}
				if (success && checkRunMakeindex(documentUrl.fileName(), indFile)) // run MakeIndex
				{
					optionList = KShell::splitArgs(makeindexOptions);
					for (int i = 0; i < optionList.size(); ++i)
						optionList[i] = parseOption(optionList.at(i), baseName, extension, currentLine);
					success = runCommand("MakeIndex", makeindexCommand, optionList, workingDir);
					rerunLatex = true;
				}
				int startLine = 0; // only the log of the last LaTeX run should be checked
				for (int i = 0; i < 3 && success && (checkRerun(startLine) || rerunLatex); ++i) // rerun LaTeX as much times as needed
				{
					optionList = KShell::splitArgs(latexOptions);
					for (int i = 0; i < optionList.size(); ++i)
						optionList[i] = parseOption(optionList.at(i), baseName, extension, currentLine);
					startLine = m_kateLatexEdit->document()->blockCount(); // the log of the following LaTeX run will start from here
					success = runCommand("LaTeX", latexCommand, optionList, workingDir);
					rerunLatex = false;
				}
				if (success && autoRunViewer) // run viewer
				{
					optionList = KShell::splitArgs(viewerOptions);
					for (int i = 0; i < optionList.size(); ++i)
						optionList[i] = parseOption(optionList.at(i), baseName, extension, currentLine);
					success = runCommand("View output", viewerCommand, optionList, workingDir, true);
				}
			}
			break;
		case 1: // run BibTeX
			optionList = KShell::splitArgs(bibtexOptions);
			for (int i = 0; i < optionList.size(); ++i)
				optionList[i] = parseOption(optionList.at(i), baseName, extension, currentLine);
			success = runCommand("BibTeX", bibtexCommand, optionList, workingDir);
			break;
		case 2: // run MakeIndex
			optionList = KShell::splitArgs(makeindexOptions);
			for (int i = 0; i < optionList.size(); ++i)
				optionList[i] = parseOption(optionList.at(i), baseName, extension, currentLine);
			success = runCommand("MakeIndex", makeindexCommand, optionList, workingDir);
			break;
		case 3: // run viewer
			optionList = KShell::splitArgs(viewerOptions);
			for (int i = 0; i < optionList.size(); ++i)
				optionList[i] = parseOption(optionList.at(i), baseName, extension, currentLine);
			success = runCommand("View output", viewerCommand, optionList, workingDir, true);
			break;
	}
}

/*** Configuration ***/

void KateLatexThread::readConfig()
{
	const QMutexLocker lock(&m_memberLock);
	// same defaults as in KateLatexConfigPage::reset()
	KConfigGroup config(KGlobal::config(), "LaTeX Plugin");
	m_latexCommand = config.readEntry("LatexCommand", "pdflatex");
	m_latexOptions = config.readEntry("LatexOptions", "--synctex=1 --interaction=nonstopmode %.tex");
	m_bibtexCommand = config.readEntry("BibtexCommand", "bibtex");
	m_bibtexOptions = config.readEntry("BibtexOptions", "%.aux");
	m_makeindexCommand = config.readEntry("MakeindexCommand", "makeindex");
	m_makeindexOptions = config.readEntry("MakeindexOptions", "%.idx");
	m_autoRerun = config.readEntry("AutoRerun", true);
	m_autoRunViewer = config.readEntry("AutoRunViewer", true);
	m_viewerCommand = config.readEntry("ViewerCommand", "okular");
	m_viewerOptions = config.readEntry("ViewerOptions", "--unique \"file:%.pdf#src:@ %.tex\"");
}

/*** Errors ***/

void KateLatexThread::nextError()
{
	goToError(true); // goToNext = true
}

void KateLatexThread::previousError()
{
	goToError(false); // goToNext = false
}

void KateLatexThread::goToError(bool goToNext)
{
	if (!m_latexErrorList.isEmpty())
	{
		if (goToNext && m_latexErrorIndex < m_latexErrorList.size() - 1)
			++m_latexErrorIndex;
		else if (!goToNext && m_latexErrorIndex > 0)
			--m_latexErrorIndex;
		const int line = m_latexErrorList.at(m_latexErrorIndex).logLine;
		Q_EMIT outputSetCursorPosition(line, 0);
		if (m_latexErrorList.at(m_latexErrorIndex).line != 0)
			Q_EMIT goToLineInFile(m_latexErrorList.at(m_latexErrorIndex).file, m_latexErrorList.at(m_latexErrorIndex).line - 1);
	}
}

void KateLatexThread::findLatexErrors()
{
	m_memberLock.lock();
	m_latexErrorIndex = -1;
	m_latexErrorList.clear();

	QStringList fileStack;
	const KUrl documentUrl = m_mw->activeView()->document()->url();
	const QString workingDir = QFileInfo(documentUrl.path()).absolutePath();

	int par = 0;
	int errorPar;

	const QRegExp rxWarning1("Warning: (.*) on.*line *(\\d+)");
	const QRegExp rxWarning2("Warning: (.*)");
	const QRegExp rxLatexError("(! )*(LaTeX Error:)* *(.*)\\.l\\.(\\d+) *(.*)");
	const QRegExp rxLineError("l\\.(\\d+)");
	const QRegExp rxBadBox("at (line|lines) ([0-9]+)");

	QTextBlock tb = m_kateLatexEdit->document()->begin();
	int skipped = 0;
	while (tb.isValid())
	{
		QString expression = tb.text();
		QStringList pile;

		errorPar = par;

		// get filenames of inputted/included files
		for (int j = 0; j < expression.length();)
		{
			QChar letter = expression.at(j);
			if (letter == QLatin1Char('(') || letter == QLatin1Char(')'))
			{
				pile.prepend(letter);
				++j;
			}
			else
			{
				QString word;
				while (j < expression.length())
				{
					letter = expression.at(j);
					if (letter == QLatin1Char('(') || letter == QLatin1Char(')'))
						break;
					word += letter;
					++j;
				}
				pile.prepend(word);
			}
		}
		while (!pile.isEmpty())
		{
			QString word = pile.takeLast();
			if (word == QLatin1String("(") && !pile.isEmpty())
			{
				QString fileName = pile.takeLast();
				const QFileInfo fi(fileName);
				if (fi.exists() || (fi.isRelative() && QFileInfo(workingDir + QLatin1Char('/') + fileName).exists()))
					fileStack.append(fileName.remove(QLatin1String("./")));
				else
					++skipped;
			}
			else if (word == QLatin1String(")"))
			{
				if (skipped > 0)
					--skipped;
				else if (!fileStack.isEmpty())
					fileStack.removeLast();
			}
		}
		pile.clear();

		// find warnings and errors
		if (expression.contains(QLatin1String("Warning")))
		{
			QString warning = expression.trimmed();
			// get the complete warning if split across several lines
			while (tb.isValid() && !expression.endsWith(QLatin1Char('.')))
			{
				++par;
				tb = tb.next();
				if (tb.isValid())
				{
					expression = tb.text();
					warning += expression.trimmed();
				}
			}

			LatexError latexErrorItem;
			latexErrorItem.file = (!fileStack.isEmpty()) ? fileStack.last() : QString();
			latexErrorItem.type = QLatin1String("Warning");
			if (rxWarning1.indexIn(warning) != -1)
			{
				latexErrorItem.line = rxWarning1.cap(2).toInt();
				latexErrorItem.message = rxWarning1.cap(1).remove(QLatin1Char('*'));
			}
			else if (rxWarning2.indexIn(warning) != -1)
			{
				latexErrorItem.line = 1;
				latexErrorItem.message = rxWarning2.cap(1).remove(QLatin1Char('*'));
			}
			else
			{
				latexErrorItem.line = 1;
				latexErrorItem.message = warning.remove(QLatin1Char('*'));
			}
			latexErrorItem.logLine = errorPar;
			m_latexErrorList.append(latexErrorItem);

			errorPar = par;
		}
		else if (expression.contains(QRegExp("^! (.*)")))
		{
			QString error = expression.trimmed();
			while (tb.isValid() && !expression.contains(rxLineError))
			{
				++par;
				tb = tb.next();
				if (tb.isValid())
				{
					expression = tb.text();
					error += expression.trimmed();
				}
			}

			LatexError latexErrorItem;
			latexErrorItem.file = (!fileStack.isEmpty()) ? fileStack.last() : QString();
			latexErrorItem.type = QLatin1String("Error");
//			if (rxLatexError.indexIn(error) != -1)
//			{
//				latexErrorItem.line = rxLatexError.cap(2).toInt();
//				latexErrorItem.message = rxLatexError.cap(3) + " :" + rxLatexError.cap(5);
//			}
			if (rxLineError.indexIn(error) != -1)
			{
				latexErrorItem.line = rxLineError.cap(1).toInt();
				latexErrorItem.message = error.remove(rxLineError).remove(QLatin1Char('*'));
			}
			else
			{
				latexErrorItem.line = 1;
				latexErrorItem.message = error.remove(QLatin1Char('*'));
			}
			latexErrorItem.logLine = errorPar;
			m_latexErrorList.append(latexErrorItem);

			errorPar = par;
		}
//		else if (expression.contains(QRegExp("^(Over|Under)(full \\\\[hv]box .*)")))
		else if (expression.startsWith(QLatin1String("Overfull \\hbox"))
		    || expression.startsWith(QLatin1String("Overfull \\vbox"))
		    || expression.startsWith(QLatin1String("Underfull \\hbox"))
		    || expression.startsWith(QLatin1String("Underfull \\vbox")))
		{
			QString badbox = expression.trimmed();
//			while (tb.isValid() && !expression.contains(QRegExp("(.*) at line")))
//			{
//				++par;
//				tb = tb.next();
//				if (tb.isValid())
//				{
//					expression = tb.next();
//					badbox += expression.trimmed();
//				}
//			}

			LatexError latexErrorItem;
			latexErrorItem.file = (!fileStack.isEmpty()) ? fileStack.last() : QString();
			latexErrorItem.type = QLatin1String("Badbox");
			if (rxBadBox.indexIn(badbox) != -1)
				latexErrorItem.line = rxBadBox.cap(2).toInt();
			else
				latexErrorItem.line = 1;
			latexErrorItem.message = badbox.remove(QLatin1Char('*'));
			latexErrorItem.logLine = errorPar;
			m_latexErrorList.append(latexErrorItem);

			errorPar = par;
		}

		if (tb.isValid())
		{
			++par;
			tb = tb.next();
		}
	}
	m_memberLock.unlock();
}
