/*
   Copyright (C) 2008, 2010, 2011, 2012, 2013 Glad Deschrijver
     <glad.deschrijver@gmail.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 3 or later as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "katelatex.h"
#include "katelatex.moc"

#include <QtGui/QApplication>
#include <QtCore/QDir>
#include <QtCore/QFileInfo>
#include <QtGui/QHBoxLayout>
#include <QtGui/QKeyEvent>
#include <QtGui/QScrollBar>

#include <KAction>
#include <KActionCollection>
#include <KConfigGroup>
#include <KLocalizedString>
#include <KTextEdit>
#include <KToolBar>

#include <kate/mainwindow.h>
#include <ktexteditor/document.h>
#include <ktexteditor/view.h>

#include "katelatexthread.h"
#include "loghighlighter.h"

KateLatex::KateLatex(Kate::MainWindow *mw, const KComponentData &componentData, QWidget *parent)
    : KVBox(parent)
    , Kate::XMLGUIClient(componentData)
    , m_mw(mw)
    , m_toolView(parent)
{
	KAction *action = actionCollection()->addAction("katelatex_tools_latex");
	action->setIcon(KIcon("system-run"));
	action->setText(i18nc("@action:inmenu", "Run &LaTeX"));
	connect(action, SIGNAL(triggered()), this, SLOT(runTool()));

	action = actionCollection()->addAction("katelatex_tools_bibtex");
	action->setIcon(KIcon("system-run"));
	action->setText(i18nc("@action:inmenu", "Run &BibTeX"));
	connect(action, SIGNAL(triggered()), this, SLOT(runTool()));

	action = actionCollection()->addAction("katelatex_tools_makeindex");
	action->setIcon(KIcon("system-run"));
	action->setText(i18nc("@action:inmenu", "Run &MakeIndex"));
	connect(action, SIGNAL(triggered()), this, SLOT(runTool()));

	action = actionCollection()->addAction("katelatex_tools_view_output");
	action->setIcon(KIcon("view-preview"));
	action->setText(i18nc("@action:inmenu", "&View Output"));
	connect(action, SIGNAL(triggered()), this, SLOT(runTool()));

	QWidget *mainWidget = new QWidget(this);
	QHBoxLayout *mainLayout = new QHBoxLayout(mainWidget);

	m_kateLatexEdit = new KTextEdit(mainWidget);
	m_kateLatexEdit->setReadOnly(true);
	LogHighlighter *highlighter = new LogHighlighter(m_kateLatexEdit->document());
	Q_UNUSED(highlighter);

	m_kateLatexThread = new KateLatexThread(m_mw, this);
	m_kateLatexThread->setOutputBox(m_kateLatexEdit);
	connect(m_kateLatexThread, SIGNAL(outputClear()), this, SLOT(outputClear()));
	connect(m_kateLatexThread, SIGNAL(outputSetCursorPosition(int,int)), this, SLOT(outputSetCursorPosition(int,int)));
	connect(m_kateLatexThread, SIGNAL(outputAppendText(QString)), this, SLOT(outputAppendText(QString)));
	connect(m_kateLatexThread, SIGNAL(outputInsertText(QString)), m_kateLatexEdit, SLOT(insertPlainText(QString)));
	connect(m_kateLatexThread, SIGNAL(processRunning(bool)), this, SLOT(setProcessRunning(bool)));
	connect(m_kateLatexThread, SIGNAL(goToLineInFile(QString,int)), this, SLOT(goToLineInFile(QString,int)));

	mainLayout->addWidget(logToolBar(mainWidget)); // this must be done after defining m_kateLatexThread, otherwize the connections for the actions trigger a crash
	mainLayout->addWidget(m_kateLatexEdit);

	m_procStopAction = actionCollection()->addAction("katelatex_tools_abort");
	m_procStopAction->setIcon(KIcon("process-stop"));
	m_procStopAction->setText(i18nc("@action:inmenu", "St&op LaTeX"));
	connect(m_procStopAction, SIGNAL(triggered()), m_kateLatexThread, SLOT(abortProcess()));
	m_procStopAction->setEnabled(false);

	m_mw->guiFactory()->addClient(this); // this must be done after defining the actions, otherwise the actions do not appear in the toolbar

	readConfig();
}

KateLatex::~KateLatex()
{
	m_mw->guiFactory()->removeClient(this);
}

KToolBar *KateLatex::logToolBar(QWidget *parent)
{
	KToolBar *toolBar = new KToolBar("LogToolBar", parent);
	toolBar->setFloatable(false);
	toolBar->setOrientation(Qt::Vertical);
	toolBar->setIconDimensions(IconSize(KIconLoader::Small));
	toolBar->setToolButtonStyle(Qt::ToolButtonIconOnly);

	KAction *action = actionCollection()->addAction("katelatex_tools_previous_error");
	action->setIcon(KIcon("arrow-left"));
	action->setText(i18nc("@action:intoolbar", "&Previous LaTeX Error"));
	connect(action, SIGNAL(triggered()), m_kateLatexThread, SLOT(previousError()));
	toolBar->addAction(action);

	action = actionCollection()->addAction("katelatex_tools_next_error");
	action->setIcon(KIcon("arrow-right"));
	action->setText(i18nc("@action:intoolbar", "&Next LaTeX Error"));
	connect(action, SIGNAL(triggered()), m_kateLatexThread, SLOT(nextError()));
	toolBar->addAction(action);

	return toolBar;
}

void KateLatex::keyPressEvent(QKeyEvent *event)
{
	if (event->key() == Qt::Key_Escape && parentWidget())
	{
		m_mw->hideToolView(m_toolView);
		m_mw->centralWidget()->setFocus();
	}
}

void KateLatex::outputClear()
{
	m_kateLatexEdit->clear();
}

void KateLatex::outputSetCursorPosition(int line, int column)
{
	QTextCursor kateLatexCursor = m_kateLatexEdit->textCursor();
	const QTextBlock p = m_kateLatexEdit->document()->findBlockByNumber(line);
	kateLatexCursor.setPosition(p.position() + column, QTextCursor::MoveAnchor);
	m_kateLatexEdit->setTextCursor(kateLatexCursor);
	QScrollBar *scrollBar = m_kateLatexEdit->verticalScrollBar();
	scrollBar->setValue(scrollBar->maximum()); // make sure that the current error is shown at the top of m_kateLatexEdit
	m_kateLatexEdit->ensureCursorVisible();
}

void KateLatex::outputAppendText(const QString &text)
{
	m_kateLatexEdit->insertPlainText(text);
	m_kateLatexEdit->ensureCursorVisible();
}

void KateLatex::runTool()
{
	QAction *action = qobject_cast<QAction*>(sender());
	int which;
	if (action->objectName() == "katelatex_tools_latex")
		which = 0;
	else if (action->objectName() == "katelatex_tools_bibtex")
		which = 1;
	else if (action->objectName() == "katelatex_tools_makeindex")
		which = 2;
	else if (action->objectName() == "katelatex_tools_view_output")
		which = 3;
	else
		return;

	const QFileInfo fi(m_mw->activeView()->document()->url().path());
	if (!fi.exists()) // return if editing an "untitled" document
		return;

	if (which < 3 && m_autoShowMessageBox)
	{
		if (m_mw->activeView()->document()->isModified())
			m_mw->activeView()->document()->documentSave(); // save document before typesetting
		m_mw->showToolView(m_toolView);
		m_kateLatexEdit->setFocus();
	}
	m_kateLatexThread->runTool(which);
}

/*
void KateLatex::slotToggleFocus()
{
  QAction *action = actionCollection()->action("katelatex_tools_toggle_focus");
  if ( ! m_part ) {
    m_mw->showToolView( parentWidget() );
    action->setText( i18n("Defocus Terminal") );
    return; // this shows and focuses the konsole
  }

  if ( ! m_part ) return;

  if (m_part->widget()->hasFocus()) {
    if (m_mw->activeView())
      m_mw->activeView()->setFocus();
      action->setText( i18n("Focus Terminal") );
  } else {
    // show the view if it is hidden
    if (parentWidget()->isHidden())
      m_mw->showToolView( parentWidget() );
    else // should focus the widget too!
      m_part->widget()->setFocus( Qt::OtherFocusReason );
    action->setText( i18n("Defocus Terminal") );
  }
}
*/

void KateLatex::setProcessRunning(bool isRunning)
{
	m_procStopAction->setEnabled(isRunning);
	if (isRunning)
		QApplication::setOverrideCursor(Qt::BusyCursor);
	else
		QApplication::restoreOverrideCursor();
}

void KateLatex::goToLineInFile(const QString &fileName, int lineNumber)
{
	const QString currentFile = m_mw->activeView()->document()->url().path();
	const QString workingDir = QFileInfo(currentFile).absolutePath();
	const QFileInfo fi(workingDir, fileName);
	if (currentFile != fi.absoluteFilePath())
	{
		KUrl url = KUrl::fromPath(fileName);
		m_mw->openUrl(url); // open file in new view; if file is already open, this activates its view
	}
	m_mw->activeView()->setCursorPosition(KTextEditor::Cursor(lineNumber, 0)); // XXX we assume that lineNumber is valid, i.e. between 0 and the total number of lines
}

void KateLatex::readConfig()
{
	KConfigGroup config(KGlobal::config(), "LaTeX Plugin");
	m_kateLatexEdit->setFont(QFont(config.readEntry("Font", qApp->font().toString())));
	m_autoShowMessageBox = config.readEntry("AutoShowMessageBox", true);

	m_kateLatexThread->readConfig();
}
