# -*- coding: utf-8 -*-

# Copyright(C) 2010-2011 Romain Bignon
#
# This file is part of weboob.
#
# weboob is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# weboob is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with weboob. If not, see <http://www.gnu.org/licenses/>.

import time
import logging
from PyQt4.QtGui import QWidget, QListWidgetItem, QImage, QIcon, QPixmap, \
                        QFrame, QMessageBox, QTabWidget, QVBoxLayout, \
                        QFormLayout, QLabel, QPushButton
from PyQt4.QtCore import SIGNAL, Qt

from weboob.tools.application.qt import QtDo, HTMLDelegate
from weboob.tools.misc import to_unicode
from weboob.capabilities.contact import ICapContact, Contact
from weboob.capabilities.chat import ICapChat
from weboob.capabilities.messages import ICapMessages, ICapMessagesPost, Message
from weboob.capabilities.base import NotLoaded

from .ui.contacts_ui import Ui_Contacts
from .ui.contact_thread_ui import Ui_ContactThread
from .ui.thread_message_ui import Ui_ThreadMessage
from .ui.profile_ui import Ui_Profile
from .ui.notes_ui import Ui_Notes

class ThreadMessage(QFrame):
    """
    This class represents a message in the thread tab.
    """

    def __init__(self, message, parent=None):
        QFrame.__init__(self, parent)
        self.ui = Ui_ThreadMessage()
        self.ui.setupUi(self)

        self.set_message(message)

    def set_message(self, message):
        self.message = message

        self.ui.nameLabel.setText(message.sender)
        header = time.strftime('%Y-%m-%d %H:%M:%S', message.date.timetuple())
        if message.flags & message.IS_NOT_ACCUSED:
            header += u' — <font color=#ff0000>Unread</font>'
        elif message.flags & message.IS_ACCUSED:
            header += u' — <font color=#00ff00>Read</font>'
        self.ui.headerLabel.setText(header)
        if message.flags & message.IS_HTML:
            content = message.content
        else:
            content = message.content.replace('&', '&amp;').replace('<', '&lt;').replace('>', '&gt;').replace('\n', '<br />')
        self.ui.contentLabel.setText(content)


    def __eq__(self, m):
        return self.message == m.message

class ContactThread(QWidget):
    """
    The thread of the selected contact.
    """

    def __init__(self, weboob, contact, support_reply, parent=None):
        QWidget.__init__(self, parent)
        self.ui = Ui_ContactThread()
        self.ui.setupUi(self)

        self.weboob = weboob
        self.contact = contact
        self.thread = None
        self.messages = []
        self.process_msg = None
        self.connect(self.ui.refreshButton, SIGNAL('clicked()'), self.refreshMessages)

        if support_reply:
            self.connect(self.ui.sendButton, SIGNAL('clicked()'), self.postReply)
        else:
            self.ui.frame.hide()

        self.refreshMessages()

    def refreshMessages(self, fillobj=False):
        if self.process_msg:
            return

        self.ui.refreshButton.setEnabled(False)
        self.process_msg = QtDo(self.weboob, self.gotThread, self.gotError)
        if fillobj and self.thread:
            self.process_msg.do('fillobj', self.thread, ['root'], backends=self.contact.backend)
        else:
            self.process_msg.do('get_thread', self.contact.id, backends=self.contact.backend)

    def gotError(self, backend, error, backtrace):
        self.ui.textEdit.setEnabled(False)
        self.ui.sendButton.setEnabled(False)
        self.ui.refreshButton.setEnabled(True)

    def gotThread(self, backend, thread):
        if not thread:
            #v = self.ui.scrollArea.verticalScrollBar()
            #print v.minimum(), v.value(), v.maximum(), v.sliderPosition()
            #self.ui.scrollArea.verticalScrollBar().setValue(self.ui.scrollArea.verticalScrollBar().maximum())
            self.process_msg = None
            return

        self.ui.textEdit.setEnabled(True)
        self.ui.sendButton.setEnabled(True)
        self.ui.refreshButton.setEnabled(True)

        self.thread = thread

        if thread.root is NotLoaded:
            self._insert_load_button(0)
        else:
            for message in thread.iter_all_messages():
                self._insert_message(message)

    def _insert_message(self, message):
        widget = ThreadMessage(message)
        if widget in self.messages:
            old_widget = self.messages[self.messages.index(widget)]
            if old_widget.message.flags != widget.message.flags:
                old_widget.set_message(widget.message)
            return

        for i, m in enumerate(self.messages):
            if widget.message.date > m.message.date:
                self.ui.scrollAreaContent.layout().insertWidget(i, widget)
                self.messages.insert(i, widget)
                if message.parent is NotLoaded:
                    self._insert_load_button(i)
                return

        self.ui.scrollAreaContent.layout().addWidget(widget)
        self.messages.append(widget)
        if message.parent is NotLoaded:
            self._insert_load_button(-1)

    def _insert_load_button(self, pos):
        button = QPushButton(self.tr('More messages...'))
        self.connect(button, SIGNAL('clicked()'), lambda: self._load_button_pressed(button))
        if pos >= 0:
            self.ui.scrollAreaContent.layout().insertWidget(pos, button)
        else:
            self.ui.scrollAreaContent.layout().addWidget(button)

    def _load_button_pressed(self, button):
        self.ui.scrollAreaContent.layout().removeWidget(button)
        button.hide()
        button.deleteLater()

        self.refreshMessages(fillobj=True)

    def postReply(self):
        text = unicode(self.ui.textEdit.toPlainText())
        self.ui.textEdit.setEnabled(False)
        self.ui.sendButton.setEnabled(False)
        m = Message(thread=self.thread,
                    id=0,
                    title=u'',
                    sender=None,
                    receivers=None,
                    content=text,
                    parent=self.messages[0].message if len(self.messages) > 0 else None)
        self.process_reply = QtDo(self.weboob, self._postReply_cb, self._postReply_eb)
        self.process_reply.do('post_message', m, backends=self.contact.backend)

    def _postReply_cb(self, backend, ignored):
        if not backend:
            return
        self.ui.textEdit.clear()
        self.ui.textEdit.setEnabled(True)
        self.ui.sendButton.setEnabled(True)
        self.refreshMessages()
        self.process_reply = None

    def _postReply_eb(self, backend, error, backtrace):
        content = unicode(self.tr('Unable to send message:\n%s\n')) % to_unicode(error)
        if logging.root.level == logging.DEBUG:
            content += '\n%s\n' % to_unicode(backtrace)
        QMessageBox.critical(self, self.tr('Error while posting reply'),
                             content, QMessageBox.Ok)
        self.process_reply = None




class ContactProfile(QWidget):

    def __init__(self, weboob, contact, parent=None):
        QWidget.__init__(self, parent)
        self.ui = Ui_Profile()
        self.ui.setupUi(self)

        self.connect(self.ui.previousButton, SIGNAL('clicked()'), self.previousClicked)
        self.connect(self.ui.nextButton, SIGNAL('clicked()'), self.nextClicked)

        self.weboob = weboob
        self.contact = contact
        self.loaded_profile = False
        self.displayed_photo_idx = 0
        self.process_photo = {}

        missing_fields = self.gotProfile(self.weboob.get_backend(contact.backend), contact)
        if len(missing_fields) > 0:
            self.process_contact = QtDo(self.weboob, self.gotProfile, self.gotError)
            self.process_contact.do('fillobj', self.contact, missing_fields, backends=self.contact.backend)

    def gotError(self, backend, error, backtrace):
        self.ui.frame_photo.hide()
        self.ui.descriptionEdit.setText('<h1>Unable to show profile</h1><p>%s</p>' % to_unicode(error))

    def gotProfile(self, backend, contact):
        if not backend:
            return []

        missing_fields = set()

        self.display_photo()

        self.ui.nicknameLabel.setText('<h1>%s</h1>' % contact.name)
        if contact.status == Contact.STATUS_ONLINE:
            status_color = 0x00aa00
        elif contact.status == Contact.STATUS_OFFLINE:
            status_color = 0xff0000
        elif contact.status == Contact.STATUS_AWAY:
            status_color = 0xffad16
        else:
            status_color = 0xaaaaaa

        self.ui.statusLabel.setText('<font color="#%06X">%s</font>' % (status_color, contact.status_msg))
        self.ui.contactUrlLabel.setText('<b>URL:</b> <a href="%s">%s</a>' % (contact.url, contact.url))
        if contact.summary is NotLoaded:
            self.ui.descriptionEdit.setText('<h1>Description</h1><p><i>Receiving...</i></p>')
            missing_fields.add('summary')
        else:
            self.ui.descriptionEdit.setText('<h1>Description</h1><p>%s</p>' % contact.summary.replace('\n', '<br />'))

        if not contact.profile:
            missing_fields.add('profile')
        elif not self.loaded_profile:
            self.loaded_profile = True
            for head in contact.profile.itervalues():
                if head.flags & head.HEAD:
                    widget = self.ui.headWidget
                else:
                    widget = self.ui.profileTab
                self.process_node(head, widget)

        return missing_fields

    def process_node(self, node, widget):
        # Set the value widget
        value = None
        if node.flags & node.SECTION:
            value = QWidget()
            value.setLayout(QFormLayout())
            for sub in node.value.itervalues():
                self.process_node(sub, value)
        elif isinstance(node.value, list):
            value = QLabel('<br />'.join(unicode(s) for s in node.value))
            value.setWordWrap(True)
        elif isinstance(node.value, tuple):
            value = QLabel(', '.join(unicode(s) for s in node.value))
            value.setWordWrap(True)
        elif isinstance(node.value, (basestring,int,long,float)):
            value = QLabel(unicode(node.value))
        else:
            logging.warning('Not supported value: %r' % node.value)
            return

        if isinstance(value, QLabel):
            value.setTextInteractionFlags(Qt.TextSelectableByMouse|Qt.TextSelectableByKeyboard|Qt.LinksAccessibleByMouse)

        # Insert the value widget into the parent widget, depending
        # of its type.
        if isinstance(widget, QTabWidget):
            widget.addTab(value, node.label)
        elif isinstance(widget.layout(), QFormLayout):
            label = QLabel(u'<b>%s:</b> ' % node.label)
            widget.layout().addRow(label, value)
        elif isinstance(widget.layout(), QVBoxLayout):
            widget.layout().addWidget(QLabel(u'<h3>%s</h3>' % node.label))
            widget.layout().addWidget(value)
        else:
            logging.warning('Not supported widget: %r' % widget)

    def previousClicked(self):
        if len(self.contact.photos) == 0:
            return
        self.displayed_photo_idx = (self.displayed_photo_idx - 1) % len(self.contact.photos)
        self.display_photo()

    def nextClicked(self):
        if len(self.contact.photos) == 0:
            return
        self.displayed_photo_idx = (self.displayed_photo_idx + 1) % len(self.contact.photos)
        self.display_photo()

    def display_photo(self):
        if self.displayed_photo_idx >= len(self.contact.photos):
            self.displayed_photo_idx = len(self.contact.photos) - 1
        if self.displayed_photo_idx < 0:
            self.ui.photoUrlLabel.setText('')
            return

        photo = self.contact.photos.values()[self.displayed_photo_idx]
        if photo.data:
            data = photo.data
            if photo.id in self.process_photo:
                self.process_photo.pop(photo.id)
        else:
            self.process_photo[photo.id] = QtDo(self.weboob, lambda b,p: self.display_photo())
            self.process_photo[photo.id].do('fillobj', photo, ['data'], backends=self.contact.backend)

            if photo.thumbnail_data:
                data = photo.thumbnail_data
            else:
                return

        img = QImage.fromData(data)
        img = img.scaledToWidth(self.width()/3)

        self.ui.photoLabel.setPixmap(QPixmap.fromImage(img))
        if photo.url is not NotLoaded:
            text = '<a href="%s">%s</a>' % (photo.url, photo.url)
            if photo.hidden:
                text += '<br /><font color=#ff0000><i>(Hidden photo)</i></font>'
            self.ui.photoUrlLabel.setText(text)

class ContactNotes(QWidget):
    """ Widget for storing notes about a contact """

    def __init__(self, weboob, contact, parent=None):
        QWidget.__init__(self, parent)
        self.ui = Ui_Notes()
        self.ui.setupUi(self)

        self.weboob = weboob
        self.contact = contact

        self.ui.textEdit.setEnabled(False)
        self.ui.saveButton.setEnabled(False)
        self.process = QtDo(self.weboob, self._getNotes_cb, self._getNotes_eb)
        self.process.do('get_notes', self.contact.id, backends=(self.contact.backend,))

        self.connect(self.ui.saveButton, SIGNAL('clicked()'), self.saveNotes)


    def _getNotes_cb(self, backend, data):
        if not backend or not data:
            self.process = None
            self.ui.textEdit.setEnabled(True)
            self.ui.saveButton.setEnabled(True)
            return

        self.ui.textEdit.setText(data)

    def _getNotes_eb(self, backend, error, backtrace):
        self.ui.textEdit.setEnabled(True)
        self.ui.saveButton.setEnabled(True)
        content = unicode(self.tr('Unable to load notes:\n%s\n')) % to_unicode(error)
        if logging.root.level == logging.DEBUG:
            content += '\n%s\n' % to_unicode(backtrace)
            QMessageBox.critical(self, self.tr('Error while loading notes'),
            content, QMessageBox.Ok)

    def saveNotes(self):
        text = unicode(self.ui.textEdit.toPlainText())
        self.ui.saveButton.setEnabled(False)
        self.ui.textEdit.setEnabled(False)

        self.process = QtDo(self.weboob, self._saveNotes_cb, self._saveNotes_eb)
        self.process.do('save_notes', self.contact.id, text, backends=(self.contact.backend,))

    def _saveNotes_cb(self, backend, data):
        self.ui.saveButton.setEnabled(True)
        self.ui.textEdit.setEnabled(True)
        pass

    def _saveNotes_eb(self, backend, error, backtrace):
        self.ui.saveButton.setEnabled(True)
        self.ui.textEdit.setEnabled(True)
        content = unicode(self.tr('Unable to save notes:\n%s\n')) % to_unicode(error)
        if logging.root.level == logging.DEBUG:
            content += '\n%s\n' % to_unicode(backtrace)
            QMessageBox.critical(self, self.tr('Error while saving notes'),
            content, QMessageBox.Ok)

class IGroup(object):
    def __init__(self, weboob, id, name):
        self.id = id
        self.name = name
        self.weboob = weboob

    def iter_contacts(self, cb):
        raise NotImplementedError()

class MetaGroup(IGroup):
    def iter_contacts(self, cb):
        if self.id == 'online':
            status = Contact.STATUS_ONLINE|Contact.STATUS_AWAY
        elif self.id == 'offline':
            status = Contact.STATUS_OFFLINE
        else:
            status = Contact.STATUS_ALL

        self.process = QtDo(self.weboob, lambda b, d: self.cb(cb, b, d))
        self.process.do('iter_contacts', status, caps=ICapContact)

    def cb(self, cb, backend, contact):
        if contact:
            cb(contact)
        elif not backend:
            self.process = None
            cb(None)

class ContactsWidget(QWidget):
    def __init__(self, weboob, parent=None):
        QWidget.__init__(self, parent)
        self.ui = Ui_Contacts()
        self.ui.setupUi(self)

        self.weboob = weboob
        self.contact = None
        self.ui.contactList.setItemDelegate(HTMLDelegate())

        self.url_process = None
        self.photo_processes = {}

        self.ui.groupBox.addItem('All', MetaGroup(self.weboob, 'all', self.tr('All')))
        self.ui.groupBox.addItem('Onlines', MetaGroup(self.weboob, 'online', self.tr('Online')))
        self.ui.groupBox.addItem('Offlines', MetaGroup(self.weboob, 'offline', self.tr('Offline')))
        self.ui.groupBox.setCurrentIndex(1)

        self.connect(self.ui.groupBox, SIGNAL('currentIndexChanged(int)'), self.groupChanged)
        self.connect(self.ui.contactList, SIGNAL('itemClicked(QListWidgetItem*)'), self.contactChanged)
        self.connect(self.ui.refreshButton, SIGNAL('clicked()'), self.refreshContactList)
        self.connect(self.ui.urlButton, SIGNAL('clicked()'), self.urlClicked)

    def load(self):
        self.refreshContactList()
        self.ui.backendsList.clear()
        for backend in self.weboob.iter_backends():
            self.ui.backendsList.addItem(backend.name)

    def groupChanged(self, i):
        self.refreshContactList()

    def refreshContactList(self):
        self.ui.contactList.clear()
        self.ui.refreshButton.setEnabled(False)
        i = self.ui.groupBox.currentIndex()
        group = self.ui.groupBox.itemData(i).toPyObject()
        group.iter_contacts(self.addContact)

    def setPhoto(self, contact, item):
        if not contact:
            return False

        try:
            self.photo_processes.pop(contact.id, None)
        except KeyError:
            pass

        img = None
        for photo in contact.photos.itervalues():
            if photo.thumbnail_data:
                img = QImage.fromData(photo.thumbnail_data)
                break

        if img:
            item.setIcon(QIcon(QPixmap.fromImage(img)))
            return True

        return False

    def addContact(self, contact):
        if not contact:
            self.ui.refreshButton.setEnabled(True)
            return

        status = ''
        if contact.status == Contact.STATUS_ONLINE:
            status = u'Online'
            status_color = 0x00aa00
        elif contact.status == Contact.STATUS_OFFLINE:
            status = u'Offline'
            status_color = 0xff0000
        elif contact.status == Contact.STATUS_AWAY:
            status = u'Away'
            status_color = 0xffad16
        else:
            status = u'Unknown'
            status_color = 0xaaaaaa

        if contact.status_msg:
            status += u' — %s' % contact.status_msg

        item = QListWidgetItem()
        item.setText('<h2>%s</h2><font color="#%06X">%s</font><br /><i>%s</i>' % (contact.name, status_color, status, contact.backend))
        item.setData(Qt.UserRole, contact)

        if contact.photos is NotLoaded:
            process = QtDo(self.weboob, lambda b, c: self.setPhoto(c, item))
            process.do('fillobj', contact, ['photos'], backends=contact.backend)
            self.photo_processes[contact.id] = process
        elif len(contact.photos) > 0:
            if not self.setPhoto(contact, item):
                photo = contact.photos.values()[0]
                process = QtDo(self.weboob, lambda b, p: self.setPhoto(contact, item))
                process.do('fillobj', photo, ['thumbnail_data'], backends=contact.backend)
                self.photo_processes[contact.id] = process

        for i in xrange(self.ui.contactList.count()):
            if self.ui.contactList.item(i).data(Qt.UserRole).toPyObject().status > contact.status:
                self.ui.contactList.insertItem(i, item)
                return

        self.ui.contactList.addItem(item)

    def contactChanged(self, current):
        if not current:
            return

        contact = current.data(Qt.UserRole).toPyObject()
        self.setContact(contact)

    def setContact(self, contact):
        if not contact or contact == self.contact:
            return

        if not isinstance(contact, Contact):
            return self.retrieveContact(contact)

        self.ui.tabWidget.clear()
        self.contact = contact
        backend = self.weboob.get_backend(self.contact.backend)

        self.ui.tabWidget.addTab(ContactProfile(self.weboob, self.contact), self.tr('Profile'))
        if backend.has_caps(ICapMessages):
            self.ui.tabWidget.addTab(ContactThread(self.weboob, self.contact, backend.has_caps(ICapMessagesPost)), self.tr('Messages'))
        if backend.has_caps(ICapChat):
            self.ui.tabWidget.setTabEnabled(self.ui.tabWidget.addTab(QWidget(), self.tr('Chat')),
                                            False)
        self.ui.tabWidget.setTabEnabled(self.ui.tabWidget.addTab(QWidget(), self.tr('Calendar')),
                                        False)
        self.ui.tabWidget.addTab(ContactNotes(self.weboob, self.contact), self.tr('Notes'))

    def urlClicked(self):
        url = unicode(self.ui.urlEdit.text())
        if not url:
            return

        self.retrieveContact(url)

    def retrieveContact(self, url):
        backend_name = unicode(self.ui.backendsList.currentText())
        self.ui.urlButton.setEnabled(False)
        self.url_process = QtDo(self.weboob, self.retrieveContact_cb, self.retrieveContact_eb)
        self.url_process.do('get_contact', url, backends=backend_name)

    def retrieveContact_cb(self, backend, contact):
        if not backend:
            self.url_process = None
            self.ui.urlButton.setEnabled(True)
            return

        self.ui.urlEdit.clear()
        self.setContact(contact)

    def retrieveContact_eb(self, backend, error, backtrace):
        content = unicode(self.tr('Unable to get contact:\n%s\n')) % to_unicode(error)
        if logging.root.level == logging.DEBUG:
            content += u'\n%s\n' % to_unicode(backtrace)
        QMessageBox.critical(self, self.tr('Error while getting contact'),
                             content, QMessageBox.Ok)


