"""
Copy from django restframework version 2.1.6, and apply patch from
https://github.com/tomchristie/django-rest-framework/commit/df957c8625c79e36c33f314c943a2c593f3a2701#diff-1
"""

from rest_framework.throttling import SimpleRateThrottle

class ScopedRateThrottle(SimpleRateThrottle):
    """
    Limits the rate of API calls by different amounts for various parts of
    the API.  Any view that has the `throttle_scope` property set will be
    throttled.  The unique cache key will be generated by concatenating the
    user id of the request, and the scope of the view being accessed.
    """
    scope_attr = 'throttle_scope'

    def __init__(self):
        pass

    def allow_request(self, request, view):
        self.scope = getattr(view, self.scope_attr, None)

        if not self.scope:
            return True

        self.rate = self.get_rate()
        self.num_requests, self.duration = self.parse_rate(self.rate)
        return super(ScopedRateThrottle, self).allow_request(request, view)

    def get_cache_key(self, request, view):
        """
        If `view.throttle_scope` is not set, don't apply this throttle.

        Otherwise generate the unique cache key by concatenating the user id
        with the '.throttle_scope` property of the view.
        """
        scope = getattr(view, self.scope_attr, None)

        if not scope:
            # Only throttle views if `.throttle_scope` is set on the view.
            return None

        if request.user.is_authenticated():
            ident = request.user.id
        else:
            ident = request.META.get('REMOTE_ADDR', None)

        return self.cache_format % {
            'scope': scope,
            'ident': ident
        }
