/*
 * Pidgin Authorization Blocker
 * Copyright (C) 2011 Konrad Gräfe
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02111-1301, USA.
 */

#include "authorization-blocker.h"

#include "authorization.h"
#include "request_info_hook.h"
#include "prefs.h"

#include <account.h>
#include <debug.h>
#include <gtkblist.h>
#include <minidialog.h>
#include <gtkutils.h>
#include <version.h>

extern PurplePlugin *plugin;

typedef struct _greylist_entry {
	PurpleAccount* account;
	char *sender;
} GreylistEntry;

GList *greylist;

static void show_user_info_cb(PidginMiniDialog *mini_dialog, GtkButton *button, gpointer data) {
	GreylistEntry *entry = (GreylistEntry *)data;

	pidgin_retrieve_user_info(purple_account_get_connection(entry->account), entry->sender);
}

static int request_authorization_cb (PurpleAccount* account, char *sender) {
	GtkWidget *mini_dialog;
	GreylistEntry *entry;
	GList *cur;
	gchar *msg;
	GdkPixbuf *prpl_icon;

	purple_debug_info(PLUGIN_STATIC_NAME, "account-authorization-requested()\n");

	if(!purple_prefs_get_bool(PLUGIN_PREFS_PREFIX "/block_requests") || purple_utf8_strcasecmp(purple_account_get_protocol_id(account), "prpl-icq")!=0 || purple_find_buddy(account, sender)) {
		return 0; /* business as usual */
	} else {
		/* okay. is he on our list? */
		cur = greylist;
		while(cur) {
			entry = (GreylistEntry *)cur->data;

			if(entry && entry->account == account && !purple_utf8_strcasecmp(entry->sender, sender)) {
				/* he is! let's prompt the user! */
				return 0;
			}

			cur = cur->next;
		}

		/* okay. he's not. let's greylist him! */

		entry = g_malloc(sizeof(GreylistEntry));
		entry->account = account;
		entry->sender = g_strdup(sender);

		greylist = g_list_append(greylist, entry);

		/* now Pidgin asks you for a reason..let's hook into this to answer it automatically */
		msg = prefs_get_deny_message();
		hook_into_request_info_once(msg);
		g_free(msg);

		if(purple_prefs_get_bool(PLUGIN_PREFS_PREFIX "/show_blocked_requests")) {
			msg = g_strdup_printf(_("Blocked authorization request from %s (%s)."), sender, purple_account_get_protocol_name(account));

#if PURPLE_VERSION_CHECK(2,7,4)
			prpl_icon = pidgin_create_prpl_icon(account, PIDGIN_PRPL_ICON_SMALL);
			mini_dialog = GTK_WIDGET(pidgin_mini_dialog_new_with_custom_icon(_("Authorization Blocker"), msg, prpl_icon));
#else
			mini_dialog = GTK_WIDGET(pidgin_mini_dialog_new(_("Authorization Blocker"), msg, ""));
#endif
			pidgin_mini_dialog_add_button(PIDGIN_MINI_DIALOG(mini_dialog), _("OK"), NULL, NULL);
			pidgin_mini_dialog_add_button(PIDGIN_MINI_DIALOG(mini_dialog), _("Info"), show_user_info_cb, entry);
			pidgin_blist_add_alert(mini_dialog);

			g_free(msg);
		}

		return -1;
	}
}

void authorization_init(void) {
	purple_signal_connect(
		purple_accounts_get_handle(),
		"account-authorization-requested",
		plugin,
		PURPLE_CALLBACK(request_authorization_cb),
		NULL
	);

	greylist = NULL;


}


void authorization_uninit(void) {
	g_list_foreach(greylist, (GFunc)g_free, NULL);
	g_list_free(greylist);
}


