/*
 *      NiceShaper - Dynamic Traffic Management
 *
 *      Copyright 2004 Mariusz Jedwabny <mariusz@jedwabny.net>
 *
 *      This file is subject to the terms and conditions of the GNU General Public
 *      License.  See the file COPYING in the main directory of this archive for
 *      more details.
 */

#include "niceshaper.h"

#include <sys/time.h>
#include <cstdio>

#include <vector>
#include <string>
#include <iostream>

#include "main.h"
#include "class.h"
#include "log.h"
#include "net.h"
#include "aux.h"
#include "sys.h"
#include "ifaces.h"
#include "tests.h"

using std::string;
using std::vector;

using namespace aux;

NiceShaper::NiceShaper()
{
    Reload = 50;
    CrossBar = 1;
    SectionHtbCeil = 0;
    SectionShape = 0;
    SectionHtbBurst = 0;
    SectionHtbCBurst = 0;
    IptMyParent = "";
    IptMyParentMode = "";
    IptMyChain = "ns_" + g_section_name;
}

NiceShaper::~NiceShaper()
{
    vector <string>::iterator fpvi;
    std::string buf, dev;

    for ( int n=NsNets.size()-1; n>=0; n-- ) { delete NsNets.back(); NsNets.pop_back(); }
    for ( int n=NsClasses.size()-1; n>=0; n-- ) { delete NsClasses.back(); NsClasses.pop_back(); }

    sys->ipt("iptables -t mangle -F " + IptMyChain + " 2> /dev/null > /dev/null");
    sys->ipt("iptables -t mangle -X " + IptMyChain + " 2> /dev/null > /dev/null");
}

int NiceShaper::init(vector < string > &fpv_conffile, vector < string > &fpv_classfile, vector < unsigned int > &protected_marks, vector <string> &ipt_rules)
{
    unsigned int first_classid = 4096;
    unsigned int classid = first_classid;
    unsigned int filterid = 0;
    string default_classtype = "standard-class";
    string configured_classtype = "";
    string buf;
    string option, param, value;
    vector <string> used_marks;
    vector <string> fpv_myclasses;
    vector <string>::iterator fpvi, fpvi_begin, fpvi_end, fpvi_tmp; 
    bool mydatablock;
    unsigned int configured_class_default_mark;
    unsigned int auto_class_default_mark;
    bool is_protected;  
    bool default_param_should_be_inserted;
    bool is_mode_configured = false;
    unsigned int max_htb_burst = 0;
    unsigned int max_htb_cburst = 0;
    NsClass *iterclass, *tpltcclass;    

    /* reading section config */
    if ( fpv_section_i( fpvi_begin, fpvi_end, fpv_conffile, g_section_name) == -1 ) {
        return -1;
    }

    /*fpvi = fpvi_begin;
    while ( fpvi <= fpvi_end ) {
        std::cout << "cf:" << *fpvi << std::endl;
        fpvi++;
    }*/

    // section directives
    fpvi = fpvi_begin;  
    while ( fpvi <= fpvi_end ) 
    {
        option = awk( *fpvi, 1 );
        param = awk( *fpvi, 2 );
        value = awk( *fpvi, 3 );
        if (option == "match")
        {
            NsNets.push_back(new NsNet(IptMyChain));
            if (NsNets.back()->init(*fpvi) == -1 ) return -1;
        }        
        else if ( option == "section" ) {
            if ( param == "speed" ) {
                SectionHtbCeil = unit_convert( value, BITS );
                if (( SectionHtbCeil > MAX_RATE ) || ( SectionHtbCeil < MIN_RATE )) log.error(806, *fpvi);
            }
            else if ( param == "shape" ) {
                SectionShape = unit_convert( value, BITS );
                if (( SectionShape > MAX_RATE ) || (SectionShape < MIN_RATE)) log.error(806, *fpvi);
            }
            else if ( param == "htb-burst" ) {
                SectionHtbBurst = unit_convert (value, BYTES);
            }
            else if ( param == "htb-cburst" ) {
                SectionHtbCBurst = unit_convert (value, BYTES);
            }
            else { log.error( 11, *fpvi ); }
        } 
        else if ( option == "reload" ) 
        {  
            //if ( param == "no" ) {
            //  Reload = 0;
            //}
            //else {
            Reload = (unsigned int)(str_to_double(param)*10);
            if (( Reload == 0 ) || ( Reload > 6000 )) {
                log.error( 13, *fpvi );
                return -1;
            }
            //}
        }
        else if ( option == "mode" )
        {
            if ( param == "download" ) { if (IptMyParent == "") IptMyParent = "POSTROUTING"; }
            else if ( param == "upload" ) { if (IptMyParent == "") IptMyParent = "PREROUTING"; }
            else { log.error( 14, *fpvi ); return -1; }

            if (IptMyParentMode == "") IptMyParentMode = " -A ";

            is_mode_configured = true;
        }
        else if (option == "iptables") 
        {
            if (param == "hook") {
                if ((value != "PREROUTING") && (value != "POSTROUTING")) { log.error( 701, *fpvi ); return -1; }
                IptMyParent = value;
            }
            else if (param == "hook-mode") {
                if (value == "append") IptMyParentMode = " -A ";
                else if (value == "insert") IptMyParentMode = " -I "; 
                else { log.error( 702, *fpvi ); }
            }
            else if (param == "target") { 
                //
            }           
            else { log.error( 11, *fpvi ); }
        }
        else if ( option == "debug" ) 
        {
            if ( param == "iptables" ) sys->setDebugIpt(true);
            else if ( param == "iproute" ) sys->setDebugTc(true);
        }

        if (log.getFatalError()) return -1;

        fpvi++;
    }

    if (!SectionHtbCeil) log.error(19, ""); 
    if (!SectionShape) log.error(20, ""); 
    if (!is_mode_configured) log.error(21, ""); 
    if (NsNets.empty()) log.error(22, "");
    if (SectionShape > SectionHtbCeil ) log.error(40, "");

    /* Initialize template object with default values */
    tpltcclass = new NsClass(IptMyChain, IptMyParent);

    // Class defaults directives
    fpvi = fpvi_begin;  
    while ( fpvi <= fpvi_end ) 
    {
        option = awk( *fpvi, 1 );
        param = awk( *fpvi, 2 );
        value = awk( *fpvi, 3 );
        if ((option == "low") || (option == "ceil") || (option == "rate") || (option == "strict") || (option == "hold")) {
            tpltcclass->store(*fpvi);
        }     
        else if (option == "iptables") 
        {
            if (param == "target") { 
                if ((value == "accept") || (value == "return") || (value == "drop")) tpltcclass->store(*fpvi);
                else { log.error( 11, *fpvi ); }
            }   
            else if ((param == "hook") || (param == "hook-mode")) {
                // Nothing
            }
            else { log.error( 11, *fpvi ); }
        }
        else if (option == "htb") {
            if ((param == "prio") || (param == "scheduler")) tpltcclass->store(*fpvi);
            else if ((param == "burst") || (param == "cburst")) tpltcclass->store(*fpvi);
            else log.error( 11, *fpvi ); 
        }    
        else if ( option == "sfq" ) {
            if ( param == "perturb" ) tpltcclass->store(*fpvi);
            else log.error( 11, *fpvi ); 
        }
        else if ( option == "esfq" ) {
            if ( param == "hash" ) {
                if (( value == "classic" ) || ( value == "dst" ) || ( value == "src" )) tpltcclass->store(*fpvi);
                else log.error( 11, *fpvi );
            }   
            else if ( param == "perturb" ) tpltcclass->store(*fpvi);
            else
                log.error( 11, *fpvi );
        }
        else if ( option == "imq" ) {
            if ( param == "autoredirect" ) {
                if ((value == "yes") || (value == "no")) tpltcclass->store(*fpvi);
                else if (value == "true") {
                    tpltcclass->store("imq autoredirect yes");
                    log.warning( 10, *fpvi);
                }
                else if ((value == "false") || (value == "none")) {
                    tpltcclass->store("imq autoredirect no");
                    log.warning( 9, *fpvi);
                }
                else { 
                    log.error( 11, *fpvi ); 
                }  
            }
            else { log.error( 11, *fpvi ); }      
        }
        else if ((option == "alter") || (option == "quota")) {
            tpltcclass->store(*fpvi);
        }
        else if (option == "type") {    
            if ((param == "standard-class") || (param == "do-not-shape") || (param == "virtual") || (param == "wrapper"))
                 default_classtype = param;
            else log.error( 11, *fpvi );
        }

        if (log.getFatalError()) return -1;

        fpvi++;
    }

    if (log.getFatalError()) return -1;

    /*fpvi = fpv_classfile.begin();
    while ( fpvi != fpv_classfile.end()) { 
        std::cout << "cl:" << *fpvi << std::endl;
        fpvi++;
    }*/

    mydatablock = false;
    fpvi=fpv_classfile.begin();
    while ( fpvi != fpv_classfile.end()) 
    {
        option = awk( *fpvi, 1 );
        if ( option == "class" )
        {
            if (mydatablock) {
                if (!configured_classtype.empty()) fpv_myclasses.push_back("type " + configured_classtype);
                else fpv_myclasses.push_back("type " + default_classtype);
                configured_classtype = "";
            }
            classid++;  
            configured_class_default_mark=0;
            auto_class_default_mark=0;
            if (awk(*fpvi, 2) == g_section_name) 
            {
                if (awk(*fpvi, 4).size() > MAX_CLASS_NAME_SIZE) { log.error(55, *fpvi); return -1; }
                mydatablock = true;

                fpv_myclasses.push_back( *fpvi );
                fpvi_tmp = fpvi;
                do {
                    if ( awk( *fpvi_tmp, 1 ) == "set-mark" ) configured_class_default_mark=read_fw_mark(awk( *fpvi_tmp, 2 )); 
                    fpvi_tmp++;         
                } while (( fpvi_tmp != fpv_classfile.end()) && ( awk( *fpvi_tmp, 1 ) != "class" ));
            } 
            else 
            {
                mydatablock = false;
            }
        }   
        else if ( option == "match" ) 
        {
            filterid++;     
            do {
                is_protected = false;
                for ( unsigned int n = protected_marks.size() ; n ; n-- ) 
                {
                    if ( filterid == protected_marks[n-1] ) { is_protected = true; filterid++; }
                }
            } while ( is_protected );

            if ( !mydatablock ) { fpvi++; continue; }

            if ( !auto_class_default_mark ) auto_class_default_mark=filterid;

            buf = *fpvi + " filterid " + int_to_str(filterid);

            if (configured_class_default_mark)
            {
                if (value_of_param (*fpvi, "set-mark").size()) 
                {
                    if (!is_in_vector (used_marks, value_of_param (*fpvi, "set-mark"))) {
                        used_marks.push_back (value_of_param (*fpvi, "set-mark"));
                        buf += " use-for-fw yes ";
                    }
                }
                else 
                {
                    buf += " set-mark " + int_to_str (configured_class_default_mark) + " ";
                    if (!is_in_vector (used_marks, int_to_str (configured_class_default_mark)))
                    {
                        used_marks.push_back (int_to_str (configured_class_default_mark));    
                        buf += " use-for-fw yes ";
                    }
                }
            }
            else if (value_of_param (*fpvi, "set-mark").size())
            {
                if (!is_in_vector (used_marks, value_of_param (*fpvi, "set-mark"))) {
                    used_marks.push_back (value_of_param (*fpvi, "set-mark"));
                    buf += " use-for-fw yes ";
                }
            }
            else if (value_of_param( *fpvi, "mark").size())
            {
                buf += " set-mark " + value_of_param (*fpvi, "mark");
                if (!is_in_vector (used_marks, value_of_param (*fpvi, "mark"))) {
                    used_marks.push_back (value_of_param (*fpvi, "mark"));
                    buf += " use-for-fw yes ";
                }   
            }
            else 
            {
                buf += " set-mark " + int_to_str(auto_class_default_mark) + " ";
                if (!is_in_vector (used_marks, int_to_str (auto_class_default_mark))) {
                    used_marks.push_back (int_to_str (auto_class_default_mark));
                    buf += " use-for-fw yes ";
                }
            }

            fpv_myclasses.push_back(buf);

            fpvi_tmp=fpvi;
            fpvi_tmp++;

            if ( fpvi_tmp == fpv_classfile.end()) default_param_should_be_inserted = true;
            else if ( awk(*fpvi_tmp, 1) != "match") default_param_should_be_inserted = true;
            else default_param_should_be_inserted = false;

            if ( !default_param_should_be_inserted ) { fpvi++; continue; }

            fpv_myclasses.push_back( "classid " + int_to_str(classid));
        }
        else if ( mydatablock )
        {
            if ( option == "type" ) configured_classtype = awk( *fpvi, 2 );
            else fpv_myclasses.push_back( *fpvi );
        }
        fpvi++;
    }

    if ( !fpv_myclasses.size() ) { log.error( 23, "" ); return -1; }

    if (mydatablock) {
        if (!configured_classtype.empty()) fpv_myclasses.push_back("type " + configured_classtype);
        else fpv_myclasses.push_back("type " + default_classtype);
    }
    configured_classtype = "";

    /*fpvi = fpv_myclasses.begin();
    while ( fpvi != fpv_myclasses.end()) { 
        std::cout << "mc:" << *fpvi << std::endl;
        fpvi++;
    }*/

    /* Initialize NsClass objects, using template object and extra parameters */     
    fpvi=fpv_myclasses.begin();
    do {
        option = awk (*fpvi, 1);
        if (option == "class") {
            NsClasses.push_back (new NsClass(*tpltcclass));
            // Completing section interfaces list
            std::string dev = trim_dev(awk(*fpvi, 3));
            if (!ifaces->isValidSysDev(dev)) { log.error (16, *fpvi); return -1; }
            if (!is_in_vector(SectionIfaces, dev)) SectionIfaces.push_back(dev);
        }
        iterclass=NsClasses.back ();
        if (iterclass->store(*fpvi) == -1) return -1;
        fpvi++; 
    } while (fpvi != fpv_myclasses.end());

    delete tpltcclass;  

    gettimeofday (&Ltt, NULL);
    buf = " -N " + IptMyChain;

    if (!g_fallback_to_ipt) ipt_rules.push_back (buf);
    else sys->ipt ("iptables -t mangle " + buf);

    for (unsigned int i=0; i < NsClasses.size(); i++) {
        iterclass=NsClasses[i];
        if (iterclass->validateParams(SectionShape) == -1) return -1;
        if (iterclass->prepareIptRules(g_fallback_to_ipt, ipt_rules) == -1) return -1;
        if (iterclass->prepareTcClass(SectionShape, NsClasses.size()) == -1) return -1;
        if (g_fallback_to_ipt && !(i%10)) { 
            log.setDoNotPutNewLineChar (true);
            log.onTerminal (".");
        }
        if (iterclass->htbBurst() > max_htb_burst) max_htb_burst = iterclass->htbBurst();
        if (iterclass->htbCBurst() > max_htb_cburst) max_htb_cburst = iterclass->htbCBurst();
    }
    log.onTerminal ("");

    if (SectionHtbBurst && max_htb_burst && (SectionHtbBurst < max_htb_burst)) { log.error (802); return -1; }
    else if (!SectionHtbBurst && max_htb_burst) SectionHtbBurst = max_htb_burst;

    if (SectionHtbCBurst && max_htb_cburst && (SectionHtbCBurst < max_htb_cburst)) { log.error (802); return -1; }
    else if (!SectionHtbCBurst && max_htb_cburst) SectionHtbCBurst = max_htb_cburst;

    for (unsigned int i = 0; i < NsNets.size(); i++) { 
        if (NsNets[i]->iptInit(g_fallback_to_ipt, IptMyParentMode, IptMyParent, ipt_rules) == -1) return -1; 
    }

    // Initialize common HTB classes
    fpvi = SectionIfaces.begin();
    while (fpvi != SectionIfaces.end())
    {
        std::string dev = *fpvi;

        if (dev.find("imq") != string::npos) sys->tc("ip link set " + dev + " up");
                
        if (!ifaces->isValidSysDev(dev)) { log.error (16, dev); return -1; }

        if (!g_fallback_to_tc) {
            sys->rtnlOpen();
            sys->tcClass(TC_ADD, ifaces->index(dev), 0, g_section_id, SectionHtbCeil, SectionHtbCeil, 5, compute_quantum(SectionHtbCeil), SectionHtbBurst, SectionHtbCBurst);
            sys->rtnlClose();
        }
        else {
            buf = " rate " + int_to_str(SectionHtbCeil) + " ceil " + int_to_str(SectionHtbCeil);
            if (SectionHtbBurst) buf += (" burst " + int_to_str(SectionHtbBurst));
            if (SectionHtbCBurst) buf += (" cburst " + int_to_str(SectionHtbCBurst));
            sys->tc("tc class add dev " + dev + " parent 1: classid 1:" + int_to_hex(g_section_id) + " htb " + buf + " quantum " + int_to_str(compute_quantum(SectionHtbCeil)));
        }
        fpvi++;
    }

    return 0;
}

int NiceShaper::checkIptables ()
{
    vector <string> iptables_summary;      
    struct timeval now;    
    double cycle_time;
    char buf[MAX_LONG_BUF_SIZE];    
    FILE *fp;  
    NsClass * iterclass;    
   
    gettimeofday(&now, NULL);
    cycle_time =  (double) (1000000 * (now.tv_sec - Ltt.tv_sec) + now.tv_usec - Ltt.tv_usec) / 1000000;
    gettimeofday(&Ltt, NULL);
    fp = popen(("iptables -t mangle -L -Z " + IptMyChain + " -vnx").c_str(), "r");
    for ( unsigned int n=1; n<=2; n++ ) {
        if ( fgets(buf, MAX_LONG_BUF_SIZE, fp) == NULL ) {
            log.error(12); 
            pclose(fp);
            return -1;
        }
    }

    while (fgets( buf, MAX_LONG_BUF_SIZE, fp)) {
        iptables_summary.push_back(string(buf));
    }
    pclose(fp);

    iptables_summary.pop_back();

    WorkingClasses.clear();
    ChangeableClasses.clear();

    SectionRate = 0; 
    SectionChangeableRate = 0; 
    SectionPotentialRate = 0; 
    Working = 0; 

    for (int i=NsClasses.size()-1; i>=0 ; i--) {
        iterclass=NsClasses[i];
        iterclass->check(iptables_summary, now, cycle_time);
        if (iterclass->work() && iterclass->sumInSection())
        {
            WorkingClasses.push_back(iterclass);
            SectionRate+=iterclass->rate();
            SectionPotentialRate+=iterclass->htbCeil();
            Working++;
            if ((iterclass->rate() > iterclass->nsLow()) && (iterclass->rate() <= iterclass->nsCeil()))
            {
                ChangeableClasses.push_back(iterclass);
                SectionChangeableRate += (iterclass->rate() - iterclass->nsLow());
            }
        }
    }

    SectionMeasureRate = SectionRate;

    if ( iptables_summary.size()) { log.error ( 12, "" ); return -1; }

    return 0;
}

int NiceShaper::judge ()
{
    unsigned int old_section_potential_rate = 0;
    unsigned int cycle_counter = 0;
    unsigned int sumofw = 0, sumofc = 0;
    unsigned int topbarbits;
    unsigned int alfa = 100;
    bool overload; 
    bool break_conditions;
    bool low_instead_softlimit = false;
    register int i = 0;
    NsClass *iterclass;   

    if (checkIptables() == -1) return -1;

    if (WorkingClasses.empty()) return 0; 

    for (i=WorkingClasses.size()-1; i>=0 ; i--) {
        iterclass=WorkingClasses[i];
        sumofw+=(iterclass->htbCeil()-iterclass->nsLow());
    }

    for (i=ChangeableClasses.size()-1 ; i>=0 ; i--) {
        iterclass=ChangeableClasses[i]; 
        sumofc+=(iterclass->htbCeil()-iterclass->nsLow());
    }
    
    if (SectionRate > SectionShape) {
        //lp->max -= lp->max / ( lp->massive / working ) * 1024; // NiceShaper 0.5 
        if (!sumofw) return 0;
        topbarbits = 1000*(unsigned int)(((double)SectionShape*CrossBar)/(SectionHtbCeil/Working));
        CrossBar -= (double)topbarbits/sumofw;
        if ( CrossBar < 0 ) CrossBar = 0;
        overload = true;
    }
    else {
        //lp->max += ( limit - utilize ) / working * 8; // NiceShaper 0.5
        topbarbits = (SectionShape-SectionRate)/Working; 
        if (sumofw) CrossBar += (double)topbarbits/sumofw;
        if ( CrossBar > 1 ) CrossBar = 1;       
        overload = false;       
    }

    for ( i=WorkingClasses.size()-1; i>=0; i-- ) {
        iterclass=WorkingClasses[i];    
        iterclass->setHardLimit( iterclass->nsLow() + (unsigned int)((double)( iterclass->nsCeil()-iterclass->nsLow()) * CrossBar));
        iterclass->computeSoftLimit(); 
        if ( iterclass->htbCeil() > iterclass->hardLimit()) {
            sumofw -= ( iterclass->htbCeil() - iterclass->hardLimit());
            if ( iterclass->rate() > iterclass->nsLow() ) { 
                sumofc -= ( iterclass->htbCeil() - iterclass->hardLimit());
            }
            iterclass->setHtbCeil( iterclass->hardLimit());
            if ( iterclass->rate() > iterclass->htbCeil() ) {
                SectionRate -= (iterclass->rate() - iterclass->htbCeil());
                iterclass->setRate( iterclass->htbCeil()); 
            }
        }
    }

    if ( ChangeableClasses.empty()) alfa = 100;
    else if ((alfa = 1000/ChangeableClasses.size()) > 100 ) alfa = 100;

    if ( ChangeableClasses.empty())
    {
        if ( !overload ) {
            for ( int i=WorkingClasses.size()-1; i>=0; i-- ) 
            {
                iterclass=WorkingClasses[i];
                if (sumofw) iterclass->incHtbCeil ((unsigned int)(( SectionShape - SectionRate ) * ((double)(iterclass->htbCeil()-iterclass->nsLow())/sumofw)));
                if ( iterclass->htbCeil() == iterclass->nsLow()) iterclass->incHtbCeil( alfa );
                if ( iterclass->htbCeil() > iterclass->nsCeil() ) iterclass->setHtbCeil( iterclass->nsCeil()); 

            }
            applyChanges();
        }

        return 0;
    }

    do {
        if ( cycle_counter ) {
            SectionRate = 0;
            SectionChangeableRate = 0;
            sumofc=0; sumofw=0;
            for ( int i=WorkingClasses.size()-1; i>=0; i-- ) {
                iterclass=WorkingClasses[i]; 
                if ( iterclass->rate() > iterclass->htbCeil()) 
                    iterclass->setRate( iterclass->htbCeil());
                if ((iterclass->rate() > iterclass->nsLow()) && (iterclass->rate() <= iterclass->nsCeil()))
                {
                    SectionChangeableRate += (iterclass->rate() - iterclass->nsLow());
                    sumofc += (iterclass->htbCeil()-iterclass->nsLow());
                }
                SectionRate += iterclass->rate();
                sumofw += (iterclass->htbCeil()-iterclass->nsLow());

            }
            if ( SectionRate < SectionShape )   break; 
            if ( !sumofc || !sumofw ) break;
        }

        old_section_potential_rate=SectionPotentialRate;
        SectionPotentialRate=0; 
        for ( i=WorkingClasses.size()-1; i>=0; i-- ) {
            iterclass=WorkingClasses[i];    

            if ( overload ) {
                if (( iterclass->rate() > iterclass->softLimit()) || ( low_instead_softlimit && ( iterclass->rate() > iterclass->nsLow()))) {
                    iterclass->decHtbCeil((unsigned int)(( SectionRate - SectionShape ) * ((double)( iterclass->htbCeil()-iterclass->nsLow())/sumofc))); 
                    iterclass->decHtbCeil( alfa );
                }
            }
            else {
                iterclass->incHtbCeil ((unsigned int)(( SectionShape - SectionRate ) * ((double)(iterclass->htbCeil()-iterclass->nsLow())/sumofw)));
                if ( iterclass->htbCeil() == iterclass->nsLow()) iterclass->incHtbCeil( alfa );
            }

            if ( iterclass->htbCeil() > iterclass->nsCeil() ) iterclass->setHtbCeil( iterclass->nsCeil()); 
            if ( iterclass->htbCeil() < iterclass->nsLow() ) iterclass->setHtbCeil( iterclass->nsLow()); 
            SectionPotentialRate+=iterclass->htbCeil();
        }

        cycle_counter++;

        if ( cycle_counter == 100 ) break_conditions = true;    
        else if (( overload ) && ( SectionPotentialRate == old_section_potential_rate ) && ( SectionChangeableRate > 1000 )) { low_instead_softlimit = true; break_conditions = false; } 
        else if (( overload ) && ( SectionPotentialRate < old_section_potential_rate )) { low_instead_softlimit = false; break_conditions = false; }
        else break_conditions = true;   
    } while (!break_conditions);

    applyChanges();

    if (log.getFatalError()) return -1;    
    else return 0;
}

int NiceShaper::applyChanges()
{
    for ( int i=WorkingClasses.size()-1; i>=0 ; i-- ) {
        WorkingClasses[i]->applyChanges(Working);
    }

    return 0;
}

vector <string> NiceShaper::stats()
{
    string buf, sum;
    unsigned int tcclasssize=NsClasses.size();
    vector <string> stats_table;

    stats_table.push_back(g_section_name + " ceil last-ceil last-utilize");

    if ( g_stats_sum_position != SS_FALSE )
    {
        sum = "sum(classes:" + int_to_str(Working) + ") " + int_to_str(SectionShape) + " " + int_to_str(SectionShape) + " " + int_to_str(SectionMeasureRate);
    }   

    if ( g_stats_sum_position == SS_TOP ) 
    {
        stats_table.push_back(sum);
    }

    if ( g_stats_classes != SC_FALSE )
    {
        for ( unsigned int i=0; i<tcclasssize; i++ ) {
            buf = NsClasses[i]->stats();
            if (buf.size()) stats_table.push_back(buf);
        }
    }

    if ( g_stats_sum_position == SS_BOTTOM ) 
    {
        stats_table.push_back(sum);
    }

    return stats_table;
}

std::vector <std::string> NiceShaper::dumpQuotaCounters ()
{
    unsigned int tcclasssize=NsClasses.size();
    vector <string> counters_table;
    std::string counters;

    for (unsigned int i=0; i<tcclasssize; i++) {
        counters = NsClasses[i]->dumpQuotaCounters();
        if (!counters.empty()) counters_table.push_back(counters);
    }

    return counters_table;
}

int NiceShaper::setQuotaCounters (std::vector <std::string> &counters_table)
{
    unsigned int tcclasssize=NsClasses.size();
    unsigned int counter_day, counter_week, counter_month;
    std::vector <std::string>::iterator fpvi, fpvi_end;
    std::string class_name = "";

    if (counters_table.empty()) return 0;
    
    fpvi = counters_table.begin();
    fpvi_end = counters_table.end();
    while (fpvi < fpvi_end) {
        class_name = awk(*fpvi, 1); 
        for (unsigned int i=0; i<tcclasssize; i++) {    
            if (class_name == NsClasses[i]->name()) { 
                counter_day = str_to_uint(awk (*fpvi, 2));
                counter_week = str_to_uint(awk (*fpvi, 3));
                counter_month = str_to_uint(awk (*fpvi, 4));
                NsClasses[i]->setQuotaCounters (counter_day, counter_week, counter_month); 
                break; 
            }
        }
        fpvi++;
    }

    return 0;
}



