/*
 * Copyright 2008 Vincent Sanders <vince@simtec.co.uk>
 * Copyright 2011 Ole Loots <ole@monochrom.net>
 *
 * This file is part of NetSurf, http://www.netsurf-browser.org/
 *
 * NetSurf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * NetSurf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifdef WITH_INTERNAL_FONT_DRIVER

#include <unistd.h>

#include "utils/utf8.h"
#include "utils/log.h"
#include "desktop/mouse.h"

#include "atari/plot/plot.h"
#include "atari/plot/fontplot.h"
#include "atari/plot/font_internal.h"


extern unsigned long atari_plot_flags;
extern int atari_plot_vdi_handle;

static int dtor( FONT_PLOTTER self );
static int str_width( FONT_PLOTTER self,const plot_font_style_t *fstyle, 	const char * str, size_t length, int * width  );
static int str_split( FONT_PLOTTER self, const plot_font_style_t *fstyle,const char *string,
					  size_t length,int x, size_t *char_offset, int *actual_x );
static int pixel_pos( FONT_PLOTTER self, const plot_font_style_t *fstyle,const char *string,
						size_t length,int x, size_t *char_offset, int *actual_x );
static int text( FONT_PLOTTER self,  int x, int y, const char *text, size_t length, const plot_font_style_t *fstyle );

static bool init = false;
static int vdih;
static struct bitmap * fontbmp;
extern struct s_vdi_sysinfo vdi_sysinfo;

const struct fb_font_desc font_regular;
const struct fb_font_desc font_italic;
const struct fb_font_desc font_bold;
const struct fb_font_desc font_italic_bold;

static MFDB tmp;
static int tmp_mfdb_size;

#define FONTDATAMAX 4096

static const struct fb_font_desc*
fb_get_font(const plot_font_style_t *fstyle)
{
	if (fstyle->weight >= 700) {
		if ((fstyle->flags & FONTF_ITALIC) ||
				(fstyle->flags & FONTF_OBLIQUE)) {
			return &font_italic_bold;
		} else {
			return &font_bold;
		}
	} else {
		if ((fstyle->flags & FONTF_ITALIC) ||
				(fstyle->flags & FONTF_OBLIQUE)) {
			return &font_italic;
		} else {
			return &font_regular;
		}
	}
}

static nserror utf8_to_font_encoding(const struct fb_font_desc* font,
				       const char *string,
				       size_t len,
				       char **result)
{
	return utf8_to_enc(string, font->encoding, len, result);

}

int ctor_font_plotter_internal( FONT_PLOTTER self )
{
	self->dtor = dtor;
	self->str_width = str_width;
	self->str_split = str_split;
	self->pixel_pos = pixel_pos;
	self->text = text;
	LOG(("%s: %s\n", (char*)__FILE__, __FUNCTION__));
	if( !init ) {
		vdih = self->vdi_handle;
		fontbmp = bitmap_create(48, 48, 0);
		fontbmp->opaque = false;
	}
	init = true;
	return( 1 );
}

static int dtor( FONT_PLOTTER self )
{
	if( tmp.fd_addr != NULL ){
		free( tmp.fd_addr );
	}
	return( 1 );
}

static int str_width( FONT_PLOTTER self,const plot_font_style_t *fstyle, const char * str,
					  size_t length, int * width  )
{
   const struct fb_font_desc* fb_font = fb_get_font(fstyle);
   *width = fb_font->width * utf8_bounded_length(str, length);
	return( 1 );
}

static int str_split( FONT_PLOTTER self, const plot_font_style_t * fstyle, const char *string,
					  size_t length,int x, size_t *char_offset, int *actual_x )
{
	const struct fb_font_desc* fb_font = fb_get_font(fstyle);
	int c_off = *char_offset = x / fb_font->width;
	if (*char_offset > length) {
		 *char_offset = length;
	} else {
		while (*char_offset > 0) {
			if (string[*char_offset] == ' ')
				break;
			(*char_offset)--;
		}
                if (*char_offset == 0) {
                        *char_offset = c_off;
                        while (*char_offset < length &&
                                        string[*char_offset] != ' ') {
                                (*char_offset)++;
                        }
                }
	}
	*actual_x = *char_offset * fb_font->width;
	return( 1  );
}

static int pixel_pos( FONT_PLOTTER self, const plot_font_style_t * fstyle,const char *string,
						size_t length,int x, size_t *char_offset, int *actual_x )
{
	const struct fb_font_desc* fb_font = fb_get_font(fstyle);
    *char_offset = x / fb_font->width;
    if (*char_offset > length)
    	*char_offset = length;
    *actual_x = *char_offset * fb_font->width;
	return( 1 );
}

static void draw_glyph1(FONT_PLOTTER self, GRECT *inloc, uint8_t *chrp, int pitch, uint32_t colour )
{
	size_t bmpstride;
	GRECT clip;
	uint32_t * fontdata;
	int xloop,yloop;
	int stride = pitch / 8;
	uint32_t * linebuf;
	GRECT loc = *inloc;

	fontbmp = bitmap_realloc( loc.g_w, loc.g_h, fontbmp->bpp, loc.g_w * fontbmp->bpp, BITMAP_GROW, fontbmp );
	bmpstride = bitmap_get_rowstride(fontbmp);
	for( yloop = 0; yloop < loc.g_h; yloop++) {
		uint32_t pixmask = 1 ;
		linebuf = (uint32_t *)(fontbmp->pixdata + (bmpstride * yloop));
		fontdata = (uint32_t*)(chrp + (stride*yloop));
		for(xloop = loc.g_w-1; xloop>=0; xloop--){
			linebuf[xloop] = ((pixmask & *fontdata) == 0) ? 0 : colour;
			pixmask = (pixmask << 1);
		}
	}
	plot_blit_bitmap(fontbmp, loc.g_x, loc.g_y, 0, BITMAPF_MONOGLYPH );
}

static int text( FONT_PLOTTER self,  int x, int y, const char *text, size_t length,
				 const plot_font_style_t *fstyle )
{
    const struct fb_font_desc* fb_font = fb_get_font(fstyle);
    const uint32_t *chrp;
    char *buffer = NULL;
    int chr;
    int blen;
    GRECT loc;
	uint32_t c;

    utf8_to_font_encoding(fb_font, text, length, &buffer);
    if (buffer == NULL)
        return 1;

        /* y is given to the fonts baseline we need it to the fonts top */
	y-=((fb_font->height * 75)/100);

	/* needed? */
    y+=1;  /* the coord is the bottom-left of the pixels offset by 1 to make
               *   it work since fb coords are the top-left of pixels
               */
    blen = strlen(buffer);
	if ( blen < 1 ) {
		return( 1 );
	}

	if(atari_plot_flags & PLOT_FLAG_OFFSCREEN ){
		/* 	when the plotter is an offscreen plotter the call to
			bitmap() isn't that expensive. Draw an 8 bit bitmap into the
			offscreen buffer.
		*/
		c =  fstyle->foreground;
		/* in -> BGR */
		/* out -> ARGB */
		c = (ABGR_TO_RGB(c) | 0xFF);
    	loc.g_y = y;
		loc.g_x = x;
    	loc.g_w = fb_font->width;
    	loc.g_h = fb_font->height;
    	for (chr = 0; chr < blen; chr++) {
        	loc.g_x = x;
        	chrp = fb_font->data + ((unsigned char)buffer[chr] * fb_font->height);
        	draw_glyph1(self, &loc, (uint8_t *)chrp, 32, c);
        	x+=fb_font->width;
    	}
	} else {
		/* render the whole string into an monochrom mfdb */
		/* and plot that to reduce overhead */
		loc.g_x = x;
    	loc.g_y = y;
    	loc.g_w = blen * fb_font->width;
		assert( loc.g_w > 0 );
    	loc.g_h = fb_font->height;
		int stride = MFDB_STRIDE( loc.g_w );
		if( tmp.fd_addr == NULL || tmp_mfdb_size < MFDB_SIZE( 1, stride, loc.g_h) ){
			tmp_mfdb_size = init_mfdb( 1, loc.g_w, loc.g_h+1,  MFDB_FLAG_STAND | MFDB_FLAG_ZEROMEM, &tmp );
		} else {
			void * buf = tmp.fd_addr;
			int size = init_mfdb( 1, loc.g_w, loc.g_h+1,  MFDB_FLAG_STAND | MFDB_FLAG_NOALLOC, &tmp );
			tmp.fd_addr = buf;
			memset( tmp.fd_addr, 0, size );
		}
		short ypos;
		int rowsize = tmp.fd_wdwidth << 1;
		char * d;
		uint32_t * pp;
		for (chr = 0; chr < blen; chr++) {
        	pp = (uint32_t*)fb_font->data + ((unsigned char)buffer[chr] * fb_font->height);
			d = ((uint8_t*)tmp.fd_addr) + chr;
			for( ypos=0; ypos<loc.g_h; ypos++){
				*d = (unsigned char)*pp++;
				d += rowsize;
			}
    	}
#ifdef WITH_8BPP_SUPPORT
    	if( app.nplanes > 8 ){
#endif
			//unsigned short out[4];
			RGB1000 out;
			//rgb_to_vdi1000( (unsigned char*)&fstyle->foreground, (unsigned short*)&out );
			out.blue = 0;
			out.green = 1000;
			out.red = 0;
			vs_color(atari_plot_vdi_handle, OFFSET_CUSTOM_COLOR, (short *)&out);
			vq_color(atari_plot_vdi_handle, OFFSET_CUSTOM_COLOR, 1, (short *)&out);
			//printf("r:%d,g:%d,b:%d", out.red, out.green, out.blue);
			//vsl_color(atari_plot_vdi_handle, OFFSET_CUSTOM_COLOR);
			//vsf_color(atari_plot_vdi_handle, OFFSET_CUSTOM_COLOR);
			plot_blit_mfdb(&loc, &tmp, OFFSET_CUSTOM_COLOR, PLOT_FLAG_TRANS);
#ifdef WITH_8BPP_SUPPORT
		} else {
			unsigned char c = RGB_TO_VDI(fstyle->foreground);
			plot_blit_mfdb(&loc, &tmp, c, PLOT_FLAG_TRANS );
		}
#endif
	}

    free(buffer);
	return( 1 );
}

/* ------------------*/
/* Fontdata 		 */
/* ------------------*/

static const uint32_t fontdata_bold[FONTDATAMAX] = {
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xE4, 0xAC, 0xA4, 0xA4, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEC, 0xA2, 0xA4, 0xA8, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEC, 0xA2, 0xAC, 0xA2, 0xEC, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEA, 0xAA, 0xAE, 0xA2, 0xE2, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xA8, 0xAE, 0xA2, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xE6, 0xA8, 0xAE, 0xAA, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xA2, 0xA4, 0xA4, 0xE4, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xAA, 0xAE, 0xAA, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xAA, 0xAE, 0xA2, 0xE2, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xE4, 0xAA, 0xAE, 0xAA, 0xEA, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEC, 0xAA, 0xAC, 0xAA, 0xEC, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xE4, 0xAA, 0xA8, 0xAA, 0xE4, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEC, 0xAA, 0xAA, 0xAA, 0xEC, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xA8, 0xAC, 0xA8, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xA8, 0xAC, 0xA8, 0xE8, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xCA, 0x4A, 0x4A, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x44, 0xCC, 0x44, 0x44, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4C, 0xC2, 0x44, 0x48, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4C, 0xC2, 0x4C, 0x42, 0xEC, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4A, 0xCA, 0x4E, 0x42, 0xE2, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xC8, 0x4E, 0x42, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x46, 0xC8, 0x4E, 0x4A, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xC2, 0x44, 0x44, 0xE4, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xCA, 0x4E, 0x4A, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xCA, 0x4E, 0x42, 0xE2, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x44, 0xCA, 0x4E, 0x4A, 0xEA, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4C, 0xCA, 0x4C, 0x4A, 0xEC, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x44, 0xCA, 0x48, 0x4A, 0xE4, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4C, 0xCA, 0x4A, 0x4A, 0xEC, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xC8, 0x4C, 0x48, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xC8, 0x4C, 0x48, 0xE8, 0x00, 0xFE,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C,
	0x1C, 0x00, 0x1C, 0x1C, 0x1C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x6C, 0x6C, 0xFE, 0xFE, 0xFE, 0x6C, 0x6C,
	0xFE, 0xFE, 0xFE, 0x6C, 0x6C, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x7C, 0xFE, 0xF6, 0xF0, 0xF8, 0x7C,
	0x3E, 0x1E, 0xDE, 0xFE, 0x7C, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xE0, 0xE6, 0xEE, 0x1C,
	0x38, 0x70, 0xEE, 0xCE, 0x0E, 0x00, 0x00, 0x00,
	0x00, 0x70, 0xF8, 0xF8, 0xF8, 0xF8, 0x76, 0xFE,
	0xFE, 0xEC, 0xEE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x06, 0x0C, 0x1C, 0x38, 0x38, 0x70, 0x70, 0x70,
	0x70, 0x70, 0x38, 0x38, 0x1C, 0x0C, 0x06, 0x00,
	0x60, 0x30, 0x38, 0x1C, 0x1C, 0x0E, 0x0E, 0x0E,
	0x0E, 0x0E, 0x1C, 0x1C, 0x38, 0x30, 0x60, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x54, 0xFE, 0x7C, 0x38,
	0x7C, 0xFE, 0x54, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x38, 0xFE,
	0xFE, 0x38, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x78, 0x70, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE,
	0xFE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x0E, 0x1E, 0x3C,
	0x78, 0xF0, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x7C, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0x7C, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x38, 0x78, 0x78, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0x0E, 0x0E, 0x7E, 0xFC,
	0xE0, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0x0E, 0x0E, 0x3C, 0x3C,
	0x0E, 0x0E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x0C, 0x1C, 0x3C, 0x7C, 0x6C, 0xEC,
	0xFE, 0xFE, 0x1C, 0x1C, 0x1C, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0xFE, 0xE0, 0xE0, 0xFC, 0xFE, 0x0E,
	0x0E, 0x0E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFC, 0xE0, 0xE0, 0xFC, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0xFE, 0x0E, 0x0E, 0x1C, 0x1C, 0x1C,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0xEE, 0xFE, 0x7C, 0xFE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0xEE, 0xEE, 0xFE, 0x7E,
	0x0E, 0x0E, 0x0E, 0x7C, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x38, 0x00,
	0x00, 0x00, 0x38, 0x38, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x38, 0x00,
	0x00, 0x18, 0x38, 0x38, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x06, 0x0E, 0x1C, 0x38, 0x70,
	0x70, 0x38, 0x1C, 0x0E, 0x06, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0xFE, 0x00,
	0x00, 0xFE, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x60, 0x70, 0x38, 0x1C, 0x0E,
	0x0E, 0x1C, 0x38, 0x70, 0x60, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0x0E, 0x0E, 0x3E, 0x7C,
	0x70, 0x00, 0x70, 0x70, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0xEE, 0xEE, 0xFE, 0xFE,
	0xEC, 0xE0, 0xE0, 0xFC, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0xEE, 0xEE, 0xEE, 0xFE,
	0xFE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0xFC, 0xFE, 0xEE, 0xEE, 0xEE, 0xFC, 0xFC,
	0xEE, 0xEE, 0xEE, 0xFE, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0xEE, 0xE0, 0xE0, 0xE0,
	0xE0, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0xF8, 0xFC, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFC, 0xF8, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0xFE, 0xE0, 0xE0, 0xE0, 0xF8, 0xF8,
	0xE0, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0xFE, 0xE0, 0xE0, 0xE0, 0xF8, 0xF8,
	0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0xEE, 0xE0, 0xE0, 0xFE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xFE, 0xFE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0xFE, 0x38, 0x38, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E,
	0x0E, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0xE0, 0xE6, 0xEE, 0xFE, 0xFC, 0xF8, 0xF0,
	0xF8, 0xFC, 0xFE, 0xEE, 0xE6, 0x00, 0x00, 0x00,
	0x00, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0,
	0xE0, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0xC6, 0xEE, 0xFE, 0xFE, 0xFE, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0xCE, 0xEE, 0xEE, 0xFE, 0xFE, 0xFE, 0xFE,
	0xFE, 0xFE, 0xEE, 0xEE, 0xE6, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0xFC, 0xFE, 0xEE, 0xEE, 0xEE, 0xEE, 0xFE,
	0xFC, 0xE0, 0xE0, 0xE0, 0xE0, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xFE, 0xFE, 0xFE, 0x7E, 0x06, 0x00, 0x00,
	0x00, 0xFC, 0xFE, 0xEE, 0xEE, 0xEE, 0xFE, 0xFC,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0xE0, 0xE0, 0xFC, 0x7E,
	0x0E, 0x0E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0xFE, 0x38, 0x38, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0x6C, 0x7C, 0x38, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0xFE, 0xFE, 0xFE, 0xEE, 0x44, 0x00, 0x00, 0x00,
	0x00, 0xEE, 0xEE, 0xEE, 0xEE, 0x7C, 0x38, 0x38,
	0x7C, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0x7C, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0xFE, 0x0E, 0x0E, 0x1C, 0x38, 0x38,
	0x70, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x3E, 0x38, 0x38, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0x3E, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x40, 0xE0, 0xF0, 0x78,
	0x3C, 0x1E, 0x0E, 0x04, 0x00, 0x00, 0x00, 0x00,
	0x00, 0xFC, 0xFC, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C,
	0x1C, 0x1C, 0x1C, 0xFC, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x7C, 0xEE, 0xC6, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00,
	0x00, 0x00, 0x00, 0x30, 0x38, 0x18, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x7E, 0x0E,
	0x7E, 0xFE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0xE0, 0xE0, 0xE0, 0xE0, 0xFC, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xFE, 0xEE,
	0xE0, 0xE0, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x0E, 0x0E, 0x0E, 0x7E, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xFE, 0xEE,
	0xFE, 0xFE, 0xE0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7E, 0x70, 0x70, 0x70, 0xF8, 0xF8,
	0x70, 0x70, 0x70, 0x70, 0x70, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x0E, 0x7E, 0x7C,
	0x00, 0x00, 0xE0, 0xE0, 0xE0, 0xFC, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x38, 0x00, 0x78, 0x78, 0x38,
	0x38, 0x38, 0x38, 0x7C, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1C, 0x1C, 0x00, 0x3C, 0x3C, 0x1C,
	0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x7C, 0x78,
	0x00, 0xE0, 0xE0, 0xE0, 0xE6, 0xEE, 0xFE, 0xFC,
	0xF8, 0xFC, 0xFE, 0xEE, 0xE6, 0x00, 0x00, 0x00,
	0x00, 0x78, 0x78, 0x38, 0x38, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xEC, 0xFE, 0xFE,
	0xFE, 0xFE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xFC, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xFC, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0xFC, 0xE0, 0xE0, 0xE0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x0E, 0x0F, 0x0F,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xDE, 0xFE, 0xE0,
	0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0xFE, 0xE0,
	0xFC, 0x7E, 0x0E, 0xFE, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x30, 0x70, 0x70, 0xFC, 0xFC, 0x70,
	0x70, 0x70, 0x70, 0x7E, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0x7C, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xEE, 0xEE, 0xEE,
	0xFE, 0xFE, 0xFE, 0xEE, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xEE, 0xEE, 0xEE,
	0x7C, 0x7C, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xFE, 0x7E, 0x0E, 0x0E, 0xFE, 0xFC,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0xFE, 0x1E,
	0x3C, 0x78, 0xF0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3E, 0x38, 0x38, 0x38, 0xF0, 0xF0,
	0x38, 0x38, 0x38, 0x3E, 0x1E, 0x00, 0x00, 0x00,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x00,
	0x00, 0xF0, 0xF8, 0x38, 0x38, 0x38, 0x1E, 0x1E,
	0x38, 0x38, 0x38, 0xF8, 0xF0, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x66, 0xFE, 0xFE, 0xCC, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0x28, 0x4C, 0x48, 0x48, 0x00, 0xFE,
	0x00, 0x38, 0x7C, 0xEE, 0xE0, 0xFC, 0xE0, 0xF8,
	0xE0, 0xE0, 0xEE, 0x7C, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x38, 0x18, 0x30, 0x00,
	0x00, 0x3E, 0x7E, 0x70, 0x70, 0x70, 0xF8, 0xF8,
	0x70, 0x70, 0x70, 0x70, 0x70, 0xF0, 0xE0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xEE, 0xEE, 0x66, 0xCC, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xDB, 0xDB, 0xDB, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x38, 0x38, 0x38, 0xFE, 0xFE, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00,
	0x00, 0x38, 0x38, 0x38, 0xFE, 0xFE, 0x38, 0xFE,
	0xFE, 0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00,
	0x00, 0x7C, 0xEE, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xCE, 0x1C,
	0x38, 0x70, 0xE0, 0xDB, 0x1B, 0x00, 0x00, 0x00,
	0xEE, 0x7C, 0x00, 0x7C, 0xFE, 0xEE, 0xE0, 0xFC,
	0x7E, 0x0E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x0E, 0x1C,
	0x38, 0x70, 0x38, 0x1C, 0x0E, 0x06, 0x00, 0x00,
	0x00, 0x7F, 0xFF, 0xEE, 0xEE, 0xEE, 0xEF, 0xEF,
	0xEE, 0xEE, 0xEE, 0xFF, 0x7F, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0xEE, 0x7C, 0x00, 0xFE, 0xFE, 0x0E, 0x1E, 0x3C,
	0x78, 0xF0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x18, 0x30, 0x30, 0x30, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x30, 0x30, 0x30, 0x60, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x36, 0x6C, 0x6C, 0x6C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x36, 0x36, 0x36, 0x6C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x7E,
	0x7E, 0x7E, 0x7E, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x3C, 0x3C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFE, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x7B, 0xDE, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xF1, 0xFB, 0x5F, 0x55, 0x51, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0xEE, 0x7C, 0x38, 0x00, 0x7E, 0xFE, 0xE0,
	0xFC, 0x7E, 0x0E, 0xFE, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x70, 0x38,
	0x1C, 0x0E, 0x1C, 0x38, 0x70, 0x60, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x6C, 0xFE, 0xFA,
	0xDE, 0xDE, 0xFC, 0xFE, 0x6E, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0xEE, 0x7C, 0x38, 0x00, 0xFE, 0xFE, 0x1E,
	0x3C, 0x78, 0xF0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0xEE, 0x00, 0xEE, 0xEE, 0xEE, 0xEE, 0x7C,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x38, 0x00,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x10, 0x7C, 0xFE, 0xD6,
	0xD0, 0xD0, 0xD6, 0xFE, 0x7C, 0x10, 0x10, 0x00,
	0x00, 0x3C, 0x7E, 0x76, 0x70, 0xFC, 0xFC, 0xFC,
	0x70, 0x76, 0xFE, 0xFE, 0xDC, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xEE, 0xEE, 0x7C, 0xFE,
	0xFE, 0xFE, 0xFE, 0x7C, 0xEE, 0xEE, 0x00, 0x00,
	0x00, 0x00, 0xEE, 0xEE, 0xEE, 0x7C, 0x38, 0xFE,
	0x38, 0xFE, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x38, 0x38, 0x38, 0x00,
	0x00, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x3C, 0x7C, 0x7C, 0x70, 0x3C, 0x7E, 0x66,
	0x7E, 0x3C, 0x0E, 0x3E, 0x3E, 0x3C, 0x00, 0x00,
	0x00, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x3C, 0x7E, 0xC3, 0x99, 0xBD, 0xA5, 0xA1,
	0xA5, 0xBD, 0x99, 0xC3, 0x7E, 0x3C, 0x00, 0x00,
	0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x7C, 0x00, 0xFC,
	0xFC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E,
	0x7C, 0xF8, 0xF8, 0x7C, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7E, 0x7E, 0x7E, 0x06, 0x02, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x7E, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x3C, 0x7E, 0xC3, 0xB9, 0xBD, 0xA5, 0xB9,
	0xB9, 0xA5, 0xA5, 0xC3, 0x7E, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x78, 0xFC, 0xCC, 0xFC, 0x78, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x38, 0xFE, 0xFE, 0xFE, 0x38,
	0x38, 0x00, 0xFE, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xF0, 0xF8, 0x18, 0x78, 0xF0, 0xC0,
	0xF8, 0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xF0, 0xF8, 0x18, 0x30, 0x18, 0xF8,
	0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x70, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0xFC, 0xE0, 0xE0, 0xC0,
	0x00, 0x00, 0x00, 0x7F, 0xDB, 0xDB, 0xDB, 0x7B,
	0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x38,
	0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x1C, 0x0C, 0x1C, 0x18, 0x00, 0x00,
	0x00, 0x00, 0x60, 0xE0, 0x60, 0x60, 0x60, 0xF0,
	0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x70, 0xF8, 0x88, 0xF8, 0x70, 0x00, 0xF8,
	0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD8,
	0x6C, 0x36, 0x36, 0x6C, 0xD8, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x60, 0xE0, 0x62, 0x66, 0xFE, 0x1C,
	0x38, 0x76, 0xEE, 0xD6, 0x3E, 0x3E, 0x06, 0x00,
	0x00, 0x00, 0x60, 0xE0, 0x62, 0x66, 0xFE, 0x1C,
	0x38, 0x70, 0xFC, 0xC6, 0x0C, 0x18, 0x1E, 0x00,
	0x00, 0x00, 0xE0, 0x30, 0x62, 0x36, 0xEE, 0x1C,
	0x3A, 0x76, 0xEE, 0xD6, 0x3E, 0x3E, 0x06, 0x00,
	0x00, 0x00, 0x1C, 0x1C, 0x1C, 0x00, 0x1C, 0x1C,
	0x7C, 0xF8, 0xE0, 0xEE, 0xEE, 0xFE, 0x7C, 0x00,
	0x00, 0xE0, 0x70, 0x00, 0x10, 0x38, 0x7C, 0xEE,
	0xFE, 0xFE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1C, 0x00, 0x10, 0x38, 0x7C, 0xEE,
	0xFE, 0xFE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x6C, 0xC6, 0x10, 0x38, 0x7C, 0xEE,
	0xFE, 0xFE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x76, 0xDC, 0x00, 0x10, 0x38, 0x7C, 0xEE,
	0xFE, 0xFE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0xEE, 0xEE, 0x10, 0x38, 0x7C, 0xEE, 0xFE,
	0xFE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x6C, 0x38, 0x38, 0x7C, 0xEE, 0xFE,
	0xFE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x1F, 0x3F, 0x7F, 0xFE, 0xEE, 0xFF, 0xFF,
	0xFE, 0xEE, 0xEF, 0xEF, 0xEF, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0xE0, 0xE0, 0xE0, 0xE0,
	0xE0, 0xE0, 0xEE, 0xFE, 0x7C, 0x0E, 0x6E, 0x7C,
	0x00, 0x70, 0x38, 0x00, 0xFE, 0xFE, 0xE0, 0xE0,
	0xF8, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x38, 0x00, 0xFE, 0xFE, 0xE0, 0xE0,
	0xF8, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xEE, 0x00, 0xFE, 0xFE, 0xE0, 0xE0,
	0xF8, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0xEE, 0xEE, 0x00, 0xFE, 0xFE, 0xE0, 0xE0, 0xF8,
	0xF8, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x70, 0x38, 0x00, 0x38, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x38, 0x00, 0x38, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x6C, 0x38, 0x38, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0xEE, 0xEE, 0x00, 0x38, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xF8, 0xFC, 0xEE, 0xEE, 0xEE, 0xFE,
	0xEE, 0xEE, 0xEE, 0xFC, 0xF8, 0x00, 0x00, 0x00,
	0x00, 0x76, 0xDC, 0x00, 0xEE, 0xEE, 0xEE, 0xF6,
	0xFE, 0xDE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x70, 0x38, 0x00, 0x7C, 0xFE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x38, 0x00, 0x7C, 0xFE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xEE, 0x00, 0x7C, 0xFE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x76, 0xDC, 0x00, 0x7C, 0xFE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0xEE, 0xEE, 0x00, 0x7C, 0xFE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xEE, 0x7C,
	0x38, 0x38, 0x7C, 0xEE, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x7A, 0xFC, 0xEE, 0xEE, 0xEE, 0xFE, 0xFE,
	0xFE, 0xEE, 0xEE, 0xFE, 0x7C, 0x80, 0x00, 0x00,
	0x00, 0x70, 0x38, 0x00, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x38, 0x00, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xEE, 0x00, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0xEE, 0xEE, 0x00, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x38, 0x00, 0xEE, 0xEE, 0xEE, 0xEE,
	0x7C, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xE0, 0xE0, 0xFC, 0xFE, 0xEE, 0xEE,
	0xEE, 0xFE, 0xFC, 0xE0, 0xE0, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xEE, 0xEE, 0xEE, 0xFE, 0xFC,
	0xEE, 0xEE, 0xEE, 0xFE, 0xFC, 0xE0, 0xC0, 0x00,
	0x00, 0x00, 0x70, 0x38, 0x00, 0x7C, 0x7E, 0x0E,
	0x7E, 0xFE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1C, 0x38, 0x00, 0x7C, 0x7E, 0x0E,
	0x7E, 0xFE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x7C, 0xEE, 0x00, 0x7C, 0x7E, 0x0E,
	0x7E, 0xFE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x76, 0xDC, 0x00, 0x7C, 0x7E, 0x0E,
	0x7E, 0xFE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xEE, 0xEE, 0x00, 0x7C, 0x7E, 0x0E,
	0x7E, 0xFE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x6C, 0x38, 0x00, 0x7C, 0x7E, 0x0E,
	0x7E, 0xFE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x74, 0x7E, 0x7E, 0x1A,
	0x7E, 0xFE, 0xD8, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xFE,
	0xEE, 0xE0, 0xEE, 0xFE, 0x7C, 0x1C, 0x7E, 0x7C,
	0x00, 0x00, 0x70, 0x38, 0x00, 0x7C, 0xFE, 0xEE,
	0xFE, 0xFE, 0xE0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x70, 0x00, 0x7C, 0xFE, 0xEE,
	0xFE, 0xFE, 0xE0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x7C, 0xEE, 0x00, 0x7C, 0xFE, 0xEE,
	0xFE, 0xFE, 0xE0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xEE, 0xEE, 0x00, 0x7C, 0xFE, 0xEE,
	0xFE, 0xFE, 0xE0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x70, 0x38, 0x00, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1C, 0x38, 0x00, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x7C, 0xEE, 0x00, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xEE, 0xEE, 0x00, 0x38, 0x38, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0xEC, 0x7C, 0xF8, 0xFC, 0x7E, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x76, 0xFE, 0xDC, 0x00, 0xFC, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xE0, 0x70, 0x00, 0x7C, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x70, 0x00, 0x7C, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x7C, 0xEE, 0x00, 0x7C, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x76, 0xFE, 0xDC, 0x00, 0x7C, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xEE, 0xEE, 0x00, 0x7C, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x38, 0x38, 0x00, 0xFE,
	0xFE, 0x00, 0x38, 0x38, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x02, 0x7C, 0xFE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x80, 0x00, 0x00,
	0x00, 0x00, 0xE0, 0x70, 0x00, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x70, 0x00, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x7C, 0xEE, 0x00, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xEE, 0xEE, 0x00, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1C, 0x38, 0x00, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x0E, 0xFE, 0xFC,
	0x00, 0x00, 0xE0, 0xE0, 0xFC, 0xFE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xFE, 0xFC, 0xE0, 0xE0,
	0x00, 0x00, 0xEE, 0xEE, 0x00, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x0E, 0xFE, 0xFC,
};

const struct fb_font_desc font_bold = {
	.name = "NetSurf Bold",
	.width = 8,
	.height = 16,
	.encoding = "CP1252",
	.data = fontdata_bold,
};


static const uint32_t fontdata_italic[FONTDATAMAX] = {
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xE4, 0xAC, 0xA4, 0xA4, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEC, 0xA2, 0xA4, 0xA8, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEC, 0xA2, 0xAC, 0xA2, 0xEC, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEA, 0xAA, 0xAE, 0xA2, 0xE2, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xA8, 0xAE, 0xA2, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xE6, 0xA8, 0xAE, 0xAA, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xA2, 0xA4, 0xA4, 0xE4, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xAA, 0xAE, 0xAA, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xAA, 0xAE, 0xA2, 0xE2, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xE4, 0xAA, 0xAE, 0xAA, 0xEA, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEC, 0xAA, 0xAC, 0xAA, 0xEC, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xE4, 0xAA, 0xA8, 0xAA, 0xE4, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEC, 0xAA, 0xAA, 0xAA, 0xEC, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xA8, 0xAC, 0xA8, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xA8, 0xAC, 0xA8, 0xE8, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xCA, 0x4A, 0x4A, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x44, 0xCC, 0x44, 0x44, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4C, 0xC2, 0x44, 0x48, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4C, 0xC2, 0x4C, 0x42, 0xEC, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4A, 0xCA, 0x4E, 0x42, 0xE2, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xC8, 0x4E, 0x42, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x46, 0xC8, 0x4E, 0x4A, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xC2, 0x44, 0x44, 0xE4, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xCA, 0x4E, 0x4A, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xCA, 0x4E, 0x42, 0xE2, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x44, 0xCA, 0x4E, 0x4A, 0xEA, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4C, 0xCA, 0x4C, 0x4A, 0xEC, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x44, 0xCA, 0x48, 0x4A, 0xE4, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4C, 0xCA, 0x4A, 0x4A, 0xEC, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xC8, 0x4C, 0x48, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xC8, 0x4C, 0x48, 0xE8, 0x00, 0xFE,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x06, 0x06, 0x06, 0x06, 0x0C, 0x0C, 0x0C,
	0x0C, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1B, 0x1B, 0x36, 0x36, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x1B, 0x1B, 0x1B, 0x7F, 0x7F, 0x36, 0x36,
	0xFF, 0xFF, 0x6C, 0x6C, 0x6C, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x3E, 0x7F, 0x6B, 0x68, 0x78, 0x3C,
	0x1E, 0x16, 0xD6, 0xFE, 0x7C, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xCE, 0x1C,
	0x38, 0x70, 0xE6, 0xC6, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x3E, 0x36, 0x36, 0x3C, 0x39, 0x7F,
	0x6E, 0xCC, 0xCC, 0xFE, 0x7A, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x06, 0x06, 0x0C, 0x0C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x06, 0x0E, 0x1C, 0x38, 0x30, 0x60, 0x60, 0xC0,
	0xC0, 0xC0, 0xC0, 0xC0, 0xE0, 0x70, 0x30, 0x00,
	0x18, 0x1C, 0x0E, 0x06, 0x06, 0x06, 0x06, 0x06,
	0x0C, 0x0C, 0x18, 0x38, 0x70, 0xE0, 0xC0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x36, 0x36, 0x1C, 0x7F,
	0xFE, 0x38, 0x6C, 0x6C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7E,
	0xFC, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x38, 0x70, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0xFC,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x0E, 0x1C,
	0x38, 0x70, 0xE0, 0xC0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x33, 0x67, 0x6E, 0x76,
	0xE6, 0xCC, 0xCC, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x06, 0x0E, 0x1E, 0x3C, 0x0C, 0x0C, 0x18,
	0x18, 0x18, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x03, 0x07, 0x1E, 0x7C,
	0x60, 0xC0, 0xC0, 0xFC, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x03, 0x06, 0x1E, 0x1E,
	0x06, 0x06, 0xCC, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x01, 0x03, 0x07, 0x0F, 0x1E, 0x36, 0x7F,
	0x7F, 0x0C, 0x0C, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x1F, 0x1F, 0x30, 0x30, 0x7C, 0x7E, 0x06,
	0x06, 0x06, 0xCE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x0F, 0x1F, 0x38, 0x30, 0x60, 0x7C, 0x7E,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x3F, 0x7F, 0x63, 0x03, 0x06, 0x06, 0x0C,
	0x0C, 0x18, 0x18, 0x30, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x33, 0x33, 0x1E, 0x7C,
	0x66, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x63, 0x63, 0x7E, 0x3E,
	0x06, 0x0C, 0x1C, 0xF8, 0xF0, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x0C, 0x00,
	0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x0C, 0x00,
	0x00, 0x00, 0x18, 0x38, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x06, 0x0E, 0x1C, 0x38, 0x70, 0x70,
	0x70, 0x38, 0x1C, 0x0C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x3E, 0x7C, 0x00, 0x00,
	0x7C, 0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x18, 0x18, 0x0C, 0x0C, 0x06, 0x0E,
	0x1C, 0x38, 0x70, 0x60, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x33, 0x03, 0x07, 0x0E,
	0x1C, 0x18, 0x00, 0x30, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x63, 0x6F, 0x6B, 0x6B,
	0xCF, 0xCE, 0xC0, 0xFC, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x33, 0x66, 0x7E, 0x7E,
	0x66, 0xCC, 0xCC, 0xCC, 0xCC, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x3F, 0x33, 0x63, 0x63, 0x7E, 0x7C,
	0x66, 0xC6, 0xC6, 0xFE, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x30, 0x60, 0x60, 0x60,
	0x60, 0xC0, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x3F, 0x33, 0x63, 0x63, 0x63, 0x63,
	0xC3, 0xC6, 0xCE, 0xFC, 0xF8, 0x00, 0x00, 0x00,
	0x00, 0x3F, 0x3F, 0x30, 0x30, 0x60, 0x7C, 0x7C,
	0x60, 0xC0, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x3F, 0x3F, 0x30, 0x30, 0x60, 0x7C, 0x7C,
	0x60, 0xC0, 0xC0, 0xC0, 0xC0, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x30, 0x60, 0x60, 0x6E,
	0x6E, 0xC6, 0xCE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x33, 0x33, 0x33, 0x63, 0x66, 0x7E, 0x7E,
	0xC6, 0xC6, 0xCC, 0xCC, 0xCC, 0x00, 0x00, 0x00,
	0x00, 0x3F, 0x3F, 0x0C, 0x0C, 0x18, 0x18, 0x18,
	0x18, 0x30, 0x30, 0xFC, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x03, 0x03, 0x03, 0x03, 0x06, 0x06, 0x06,
	0x06, 0x0C, 0xCC, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x30, 0x30, 0x33, 0x36, 0x7C, 0x78, 0x70,
	0x78, 0xF8, 0xDC, 0xCE, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x30, 0x30, 0x30, 0x30, 0x60, 0x60, 0x60,
	0x60, 0xC0, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x33, 0x33, 0x3F, 0x3F, 0x7F, 0x6B, 0x66,
	0x66, 0xC6, 0xCC, 0xCC, 0xCC, 0x00, 0x00, 0x00,
	0x00, 0x33, 0x33, 0x3B, 0x3B, 0x7B, 0x7E, 0x6E,
	0x6E, 0xCE, 0xCC, 0xCC, 0xCC, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x33, 0x63, 0x66, 0x66,
	0xC6, 0xCC, 0xCC, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x3F, 0x33, 0x33, 0x67, 0x7E, 0x7C,
	0x60, 0xC0, 0xC0, 0xC0, 0xC0, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x33, 0x63, 0x66, 0x66,
	0xC6, 0xF6, 0xDC, 0xF8, 0x7C, 0x0C, 0x00, 0x00,
	0x00, 0x3E, 0x3F, 0x33, 0x33, 0x63, 0x7E, 0x7C,
	0x6E, 0xC6, 0xC6, 0xCC, 0xCC, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x60, 0x60, 0x7C, 0x3E,
	0x06, 0x06, 0xCE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x7F, 0x7F, 0x18, 0x18, 0x30, 0x30, 0x30,
	0x30, 0x60, 0x60, 0x60, 0x60, 0x00, 0x00, 0x00,
	0x00, 0x33, 0x33, 0x33, 0x33, 0x63, 0x66, 0x66,
	0x66, 0xC6, 0xCC, 0xFC, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x33, 0x33, 0x33, 0x33, 0x63, 0x63, 0x66,
	0x66, 0x6C, 0x38, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x33, 0x33, 0x33, 0x33, 0x63, 0x63, 0x6B,
	0x6B, 0xFE, 0xFE, 0xEE, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x63, 0x63, 0x63, 0x77, 0x3E, 0x1C, 0x38,
	0x7C, 0xEE, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x66, 0x66, 0x66, 0x66, 0x3C, 0x3C, 0x18,
	0x18, 0x30, 0x30, 0x60, 0x60, 0x00, 0x00, 0x00,
	0x00, 0x7F, 0x7F, 0x03, 0x07, 0x0E, 0x0C, 0x18,
	0x38, 0x70, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x0F, 0x0F, 0x0C, 0x0C, 0x18, 0x18, 0x18,
	0x18, 0x30, 0x30, 0x3E, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xE0, 0x70,
	0x38, 0x1C, 0x0E, 0x06, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x1F, 0x1F, 0x03, 0x03, 0x06, 0x06, 0x06,
	0x06, 0x0C, 0x0C, 0x7C, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x0E, 0x1F, 0x3B, 0x63, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0xFE, 0x00,
	0x00, 0x00, 0x00, 0x0C, 0x0E, 0x0C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x3E, 0x06,
	0x7E, 0xFC, 0xCC, 0xFC, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x18, 0x30, 0x30, 0x3E, 0x7F, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x7E, 0x66,
	0xC0, 0xC0, 0xCC, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x03, 0x03, 0x03, 0x06, 0x3E, 0x7E, 0x66,
	0xC6, 0xCC, 0xCC, 0xFC, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x7E, 0x66,
	0xFE, 0xFC, 0xC0, 0xFC, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x07, 0x0F, 0x0C, 0x18, 0x7E, 0x7E, 0x30,
	0x30, 0x30, 0x60, 0x60, 0x60, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3F, 0x7F, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7E, 0x0C, 0xFC, 0xF8,
	0x00, 0x00, 0x30, 0x30, 0x30, 0x6E, 0x7F, 0x73,
	0x63, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x06, 0x06, 0x00, 0x1C, 0x3C, 0x0C,
	0x0C, 0x18, 0x18, 0x7C, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x03, 0x03, 0x00, 0x0E, 0x1E, 0x06,
	0x06, 0x0C, 0x0C, 0x0C, 0x0C, 0x18, 0xF8, 0xF0,
	0x00, 0x30, 0x30, 0x30, 0x33, 0x67, 0x6E, 0x7C,
	0x7C, 0xDC, 0xCE, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x0E, 0x06, 0x06, 0x0C, 0x0C, 0x0C,
	0x0C, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0x7F, 0x7F,
	0x6B, 0xD6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x6C, 0x7E, 0x76,
	0x66, 0xCC, 0xCC, 0xCC, 0xCC, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x7E, 0x66,
	0x66, 0xCC, 0xCC, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x3F, 0x33,
	0x63, 0x66, 0x66, 0xFE, 0xFC, 0xC0, 0xC0, 0xC0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3F, 0x7F, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7C, 0x0C, 0x0E, 0x0E,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x6F, 0x7F, 0x70,
	0x60, 0xC0, 0xC0, 0xC0, 0xC0, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3F, 0x7F, 0x60,
	0x7C, 0x3E, 0x06, 0xFE, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x06, 0x06, 0x0C, 0x0C, 0x3F, 0x3F, 0x18,
	0x18, 0x30, 0x30, 0x3E, 0x1C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x33, 0x33, 0x66,
	0x66, 0x66, 0xCC, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66,
	0xCC, 0xCC, 0xF8, 0x70, 0x20, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0x63, 0x63,
	0x6B, 0xDE, 0xFE, 0xFE, 0x6C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0x63, 0x77,
	0x3E, 0x7C, 0xEE, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0x63, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7E, 0x0C, 0xFC, 0xF8,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7F, 0x7F, 0x07,
	0x1E, 0x78, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x07, 0x0E, 0x0C, 0x0C, 0x18, 0x70, 0x70,
	0x18, 0x18, 0x30, 0x38, 0x1C, 0x00, 0x00, 0x00,
	0x06, 0x06, 0x06, 0x06, 0x0C, 0x0C, 0x0C, 0x0C,
	0x18, 0x18, 0x18, 0x18, 0x30, 0x30, 0x30, 0x00,
	0x00, 0x38, 0x1C, 0x0C, 0x18, 0x18, 0x0E, 0x0E,
	0x18, 0x30, 0x30, 0x70, 0xE0, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1D, 0x3F, 0x37, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0x28, 0x4C, 0x48, 0x48, 0x00, 0xFE,
	0x00, 0x06, 0x0F, 0x1B, 0x18, 0x7E, 0x30, 0x7C,
	0x60, 0x60, 0x6C, 0x3C, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x10, 0x20, 0x00,
	0x00, 0x07, 0x0F, 0x0C, 0x18, 0x7E, 0x7E, 0x30,
	0x30, 0x30, 0x60, 0x60, 0x60, 0xC0, 0x80, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x66, 0x66, 0x44, 0x88, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xDB, 0xDB, 0x00, 0x00, 0x00,
	0x00, 0x06, 0x06, 0x06, 0x1F, 0x3F, 0x0C, 0x0C,
	0x0C, 0x18, 0x18, 0x18, 0x18, 0x30, 0x00, 0x00,
	0x00, 0x06, 0x06, 0x06, 0x1F, 0x3F, 0x0C, 0x3F,
	0x3F, 0x18, 0x18, 0x18, 0x18, 0x30, 0x00, 0x00,
	0x00, 0x3E, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x61, 0x63, 0x06,
	0x0C, 0x30, 0x60, 0xDB, 0x9B, 0x00, 0x00, 0x00,
	0x63, 0x3E, 0x00, 0x1E, 0x3F, 0x33, 0x60, 0x7C,
	0x3E, 0x06, 0xCE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x0E,
	0x1C, 0x30, 0x38, 0x1C, 0x0C, 0x00, 0x00, 0x00,
	0x00, 0x1F, 0x3F, 0x33, 0x33, 0x66, 0x67, 0x67,
	0x66, 0xCC, 0xCC, 0xFF, 0x7F, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x63, 0x3E, 0x00, 0x7F, 0x7F, 0x03, 0x07, 0x0E,
	0x18, 0x70, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x04, 0x08, 0x10, 0x18, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x0C, 0x04, 0x08, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x09, 0x12, 0x1B, 0x36, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1B, 0x1B, 0x09, 0x12, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1E, 0x3F,
	0x3F, 0x7E, 0x7E, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x1E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x7F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x19, 0x26, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xFB, 0xFF, 0x55, 0xA2, 0xA2, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x63, 0x36, 0x1C, 0x00, 0x3F, 0x7F, 0x60,
	0x7C, 0x3E, 0x06, 0xFE, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x1C,
	0x0E, 0x0C, 0x1C, 0x38, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x7F, 0x6D,
	0x6F, 0xDE, 0xD8, 0xFE, 0x6E, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x63, 0x36, 0x1C, 0x00, 0x7F, 0x7F, 0x07,
	0x1E, 0x78, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x33, 0x00, 0x66, 0x66, 0x66, 0x3C, 0x3C,
	0x18, 0x30, 0x30, 0x30, 0x60, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x0C, 0x00,
	0x0C, 0x18, 0x18, 0x18, 0x18, 0x30, 0x30, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x04, 0x3E, 0x7F, 0x6B,
	0x68, 0xD0, 0xD6, 0xFE, 0x7C, 0x20, 0x20, 0x00,
	0x00, 0x0F, 0x1F, 0x19, 0x18, 0x30, 0x7C, 0x7C,
	0x30, 0x60, 0xE6, 0xFE, 0xDC, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x19, 0x33, 0x1E, 0x3F,
	0x33, 0x66, 0x7E, 0x3C, 0x66, 0xCC, 0x00, 0x00,
	0x00, 0x00, 0x19, 0x19, 0x19, 0x1E, 0x0C, 0x3F,
	0x0C, 0x7E, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x06, 0x06, 0x0C, 0x0C, 0x00,
	0x00, 0x18, 0x18, 0x30, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x07, 0x0E, 0x18, 0x18, 0x3C, 0x7E, 0x66,
	0x7E, 0x3C, 0x18, 0x18, 0x70, 0xE0, 0x00, 0x00,
	0x00, 0x00, 0x33, 0x66, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1F, 0x21, 0x21, 0x4D, 0x51, 0x51,
	0x91, 0xA1, 0x9A, 0x82, 0x7C, 0x38, 0x00, 0x00,
	0x00, 0x1C, 0x02, 0x1E, 0x22, 0x3C, 0x00, 0x7C,
	0x7C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1B,
	0x36, 0xD8, 0xD8, 0x6C, 0x36, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1E, 0x3E, 0x06, 0x0C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x3F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1F, 0x21, 0x49, 0x55, 0x55, 0x59,
	0x99, 0xA5, 0xA5, 0x82, 0x7E, 0x78, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1F, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x3F, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x06, 0x06, 0x3F, 0x3F, 0x0C,
	0x0C, 0x00, 0x7E, 0x7E, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x3C, 0x3E, 0x06, 0x3C, 0x78, 0x60,
	0x7C, 0x7C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x3C, 0x3E, 0x06, 0x18, 0x0C, 0x7C,
	0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x06, 0x0C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x33, 0x33,
	0x33, 0x66, 0x66, 0x7E, 0x7C, 0xC0, 0xC0, 0x80,
	0x00, 0x00, 0x00, 0x3F, 0x6F, 0xDB, 0xDB, 0x7B,
	0x1B, 0x36, 0x36, 0x36, 0x36, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C,
	0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x1C, 0x0C, 0x1C, 0x30, 0x00, 0x00,
	0x00, 0x00, 0x18, 0x38, 0x18, 0x30, 0x30, 0x78,
	0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x0C, 0x1E, 0x22, 0x3C, 0x18, 0x00, 0x3C,
	0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6C,
	0x36, 0x36, 0x36, 0x6C, 0xD8, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x30, 0x10, 0x23, 0x76, 0x0C,
	0x18, 0x64, 0xCC, 0x94, 0x3E, 0x08, 0x08, 0x00,
	0x00, 0x00, 0x10, 0x30, 0x10, 0x23, 0x76, 0x0C,
	0x18, 0x60, 0xDC, 0x82, 0x1C, 0x20, 0x3C, 0x00,
	0x00, 0x00, 0x38, 0x04, 0x18, 0x0B, 0x76, 0x0C,
	0x18, 0x64, 0xCC, 0x96, 0x3E, 0x08, 0x08, 0x00,
	0x00, 0x00, 0x00, 0x06, 0x06, 0x00, 0x0C, 0x0C,
	0x3C, 0x70, 0xC0, 0xC6, 0xC6, 0xFC, 0x78, 0x00,
	0x00, 0x18, 0x0C, 0x00, 0x04, 0x1C, 0x36, 0x63,
	0x7F, 0xFE, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x03, 0x06, 0x00, 0x04, 0x1C, 0x36, 0x63,
	0x7F, 0xFE, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1B, 0x11, 0x04, 0x1C, 0x36, 0x63,
	0x7F, 0xFE, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x1D, 0x37, 0x00, 0x04, 0x1C, 0x36, 0x63,
	0x7F, 0xFE, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x1B, 0x1B, 0x04, 0x0E, 0x36, 0x63, 0x7F,
	0x7F, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x11, 0x0E, 0x0E, 0x36, 0x63, 0x7F,
	0x7F, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x07, 0x0F, 0x1F, 0x3B, 0x66, 0x7F, 0x7F,
	0x66, 0xCC, 0xCC, 0xCF, 0xCF, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x33, 0x30, 0x60, 0x60, 0x60,
	0x60, 0xC0, 0xC6, 0xFE, 0x7C, 0x1C, 0xCC, 0x78,
	0x00, 0x0C, 0x06, 0x00, 0x3F, 0x7F, 0x60, 0x78,
	0x78, 0xC0, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x03, 0x06, 0x00, 0x3F, 0x7F, 0x60, 0x78,
	0x78, 0xC0, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1B, 0x00, 0x3F, 0x7F, 0x60, 0x78,
	0x78, 0xC0, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x1B, 0x1B, 0x00, 0x3F, 0x3F, 0x60, 0x60, 0x78,
	0x78, 0xC0, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x0C, 0x06, 0x00, 0x1F, 0x1F, 0x0C, 0x0C,
	0x0C, 0x18, 0x18, 0x7C, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x03, 0x06, 0x00, 0x1F, 0x1F, 0x0C, 0x0C,
	0x0C, 0x18, 0x18, 0x7C, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x06, 0x09, 0x1F, 0x1F, 0x0C, 0x0C, 0x0C,
	0x0C, 0x18, 0x18, 0x7C, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x1B, 0x1B, 0x00, 0x1F, 0x1F, 0x0C, 0x0C,
	0x0C, 0x18, 0x18, 0x7C, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1E, 0x1F, 0x1B, 0x33, 0x33, 0x7B,
	0x33, 0x66, 0x6E, 0x7C, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x1D, 0x37, 0x00, 0x31, 0x63, 0x73, 0x7B,
	0x7F, 0xDE, 0xCE, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x0C, 0x06, 0x00, 0x3E, 0x7F, 0x63, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x03, 0x06, 0x00, 0x3E, 0x7F, 0x63, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1B, 0x00, 0x3E, 0x7F, 0x63, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x1D, 0x37, 0x00, 0x3E, 0x7F, 0x63, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x1B, 0x1B, 0x00, 0x3E, 0x7F, 0x63, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0x77, 0x3E,
	0x1C, 0x38, 0x7C, 0xEE, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x1D, 0x3E, 0x37, 0x6B, 0x6B, 0x6B, 0xD6,
	0xD6, 0xD6, 0xEE, 0x7C, 0xB8, 0x00, 0x00, 0x00,
	0x00, 0x0C, 0x06, 0x00, 0x33, 0x33, 0x63, 0x66,
	0x66, 0xC6, 0xCE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x03, 0x06, 0x00, 0x33, 0x33, 0x63, 0x66,
	0x66, 0xC6, 0xCE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1B, 0x00, 0x33, 0x33, 0x63, 0x66,
	0x66, 0xC6, 0xCE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x1B, 0x1B, 0x00, 0x33, 0x33, 0x63, 0x63, 0x63,
	0x66, 0xC6, 0xCE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x06, 0x0C, 0x00, 0x33, 0x33, 0x66, 0x6E,
	0x7C, 0x38, 0x30, 0x60, 0x60, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x18, 0x18, 0x1F, 0x3F, 0x33, 0x33,
	0x3F, 0x7C, 0x60, 0x60, 0x60, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1F, 0x1B, 0x1B, 0x33, 0x3F, 0x3E,
	0x33, 0x66, 0x66, 0x7E, 0x7C, 0xC0, 0x80, 0x00,
	0x00, 0x00, 0x0C, 0x06, 0x00, 0x3F, 0x7F, 0x63,
	0x63, 0xCE, 0xCE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x06, 0x0C, 0x00, 0x3F, 0x7F, 0x63,
	0x63, 0xC6, 0xCE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x0E, 0x1B, 0x00, 0x3F, 0x7F, 0x63,
	0x63, 0xC6, 0xCE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1D, 0x37, 0x00, 0x3F, 0x7F, 0x63,
	0x63, 0xC6, 0xCE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1B, 0x1B, 0x00, 0x3F, 0x7F, 0x63,
	0x63, 0xC6, 0xCE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1B, 0x0E, 0x00, 0x3E, 0x7F, 0x63,
	0x63, 0xC6, 0xCE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3A, 0x3F, 0x0D,
	0x1F, 0x7E, 0xD8, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x7F,
	0x63, 0xC0, 0xC6, 0xFE, 0x7C, 0x30, 0xF8, 0xF0,
	0x00, 0x00, 0x0C, 0x06, 0x00, 0x3E, 0x7F, 0x63,
	0x7F, 0xFE, 0xC0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x06, 0x0C, 0x00, 0x3E, 0x7F, 0x63,
	0x7F, 0xFE, 0xC0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x0E, 0x1B, 0x00, 0x3E, 0x7F, 0x63,
	0x7F, 0xFE, 0xC0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1B, 0x1B, 0x00, 0x3E, 0x7F, 0x63,
	0x7F, 0xFE, 0xC0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x0C, 0x06, 0x00, 0x0C, 0x0C, 0x0C,
	0x0C, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x03, 0x06, 0x00, 0x0C, 0x0C, 0x0C,
	0x0C, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x06, 0x0F, 0x19, 0x00, 0x0C, 0x0C, 0x0C,
	0x0C, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1B, 0x1B, 0x00, 0x0C, 0x0C, 0x0C,
	0x0C, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x1B, 0x1F, 0x3E, 0x37, 0x3E, 0x7F, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x1D, 0x3F, 0x37, 0x00, 0x7E, 0x7F, 0x63,
	0x63, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x18, 0x0C, 0x00, 0x3E, 0x7F, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x06, 0x0C, 0x00, 0x3E, 0x7F, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x0E, 0x1B, 0x00, 0x3E, 0x7F, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x1D, 0x3F, 0x37, 0x00, 0x3E, 0x7F, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1B, 0x1B, 0x00, 0x3E, 0x7F, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x0C, 0x0C, 0x00, 0x3F,
	0x7E, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x7F, 0x67,
	0x6B, 0xD6, 0xE6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x18, 0x0C, 0x00, 0x63, 0x63, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x06, 0x0C, 0x00, 0x63, 0x63, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x0E, 0x1B, 0x00, 0x63, 0x63, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1B, 0x1B, 0x00, 0x63, 0x63, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x03, 0x06, 0x00, 0x63, 0x63, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7E, 0x0C, 0xF8, 0xF0,
	0x00, 0x00, 0x18, 0x18, 0x18, 0x3E, 0x3F, 0x33,
	0x33, 0x66, 0x66, 0x7E, 0x7C, 0xC0, 0xC0, 0xC0,
	0x00, 0x00, 0x00, 0x1B, 0x00, 0x63, 0x63, 0x63,
	0x63, 0xC6, 0xC6, 0xFE, 0x7E, 0x0C, 0xF8, 0xF0,
};

const struct fb_font_desc font_italic = {
	.name = "NetSurf Italic",
	.width = 8,
	.height = 16,
	.encoding = "CP1252",
	.data = fontdata_italic,
};


static const uint32_t fontdata_italic_bold[FONTDATAMAX] = {
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xE4, 0xAC, 0xA4, 0xA4, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEC, 0xA2, 0xA4, 0xA8, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEC, 0xA2, 0xAC, 0xA2, 0xEC, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEA, 0xAA, 0xAE, 0xA2, 0xE2, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xA8, 0xAE, 0xA2, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xE6, 0xA8, 0xAE, 0xAA, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xA2, 0xA4, 0xA4, 0xE4, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xAA, 0xAE, 0xAA, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xAA, 0xAE, 0xA2, 0xE2, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xE4, 0xAA, 0xAE, 0xAA, 0xEA, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEC, 0xAA, 0xAC, 0xAA, 0xEC, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xE4, 0xAA, 0xA8, 0xAA, 0xE4, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEC, 0xAA, 0xAA, 0xAA, 0xEC, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xA8, 0xAC, 0xA8, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0xA8, 0xAC, 0xA8, 0xE8, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xCA, 0x4A, 0x4A, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x44, 0xCC, 0x44, 0x44, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4C, 0xC2, 0x44, 0x48, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4C, 0xC2, 0x4C, 0x42, 0xEC, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4A, 0xCA, 0x4E, 0x42, 0xE2, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xC8, 0x4E, 0x42, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x46, 0xC8, 0x4E, 0x4A, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xC2, 0x44, 0x44, 0xE4, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xCA, 0x4E, 0x4A, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xCA, 0x4E, 0x42, 0xE2, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x44, 0xCA, 0x4E, 0x4A, 0xEA, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4C, 0xCA, 0x4C, 0x4A, 0xEC, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x44, 0xCA, 0x48, 0x4A, 0xE4, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4C, 0xCA, 0x4A, 0x4A, 0xEC, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xC8, 0x4C, 0x48, 0xEE, 0x00, 0xFE,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0x4E, 0xC8, 0x4C, 0x48, 0xE8, 0x00, 0xFE,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x0E, 0x0E, 0x0E, 0x1C, 0x1C, 0x1C,
	0x1C, 0x00, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x77, 0x77, 0x77, 0x77, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x36, 0x36, 0x7F, 0x7F, 0x7F, 0x36, 0x6C,
	0xFE, 0xFE, 0xFE, 0x6C, 0x6C, 0x00, 0x00, 0x00,
	0x00, 0x08, 0x3E, 0x7F, 0x7B, 0x78, 0x7C, 0x7C,
	0x3E, 0x1E, 0xDE, 0xFE, 0x7C, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x70, 0x73, 0x77, 0x0E,
	0x3C, 0x70, 0xEE, 0xCE, 0x0E, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x3E, 0x7E, 0x7C, 0x7C, 0x3B, 0xFE,
	0xFE, 0xEC, 0xEE, 0xFE, 0x74, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1C, 0x1C, 0x1C, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x03, 0x06, 0x0E, 0x1C, 0x1C, 0x38, 0x38, 0x70,
	0x70, 0x70, 0x38, 0x38, 0x1C, 0x0C, 0x06, 0x00,
	0x60, 0x30, 0x38, 0x1C, 0x1C, 0x0E, 0x0E, 0x0E,
	0x1C, 0x1C, 0x38, 0x38, 0x70, 0x60, 0xC0, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x2A, 0x7F, 0x3E, 0x18,
	0x7C, 0xFE, 0x54, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x1C, 0x7E,
	0x7E, 0x38, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x78, 0x70, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE,
	0xFE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x0F, 0x3C,
	0x78, 0xF0, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x3E, 0x77, 0x77, 0x77, 0x77, 0xEE,
	0xEE, 0xEE, 0xEE, 0x7C, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x0C, 0x1C, 0x3C, 0x3C, 0x1C, 0x1C, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x07, 0x07, 0x3E, 0x7C,
	0xE0, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x07, 0x07, 0x1E, 0x3C,
	0x0E, 0x0E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x02, 0x06, 0x0E, 0x1E, 0x3E, 0x76, 0xEC,
	0xFE, 0xFE, 0x1C, 0x1C, 0x1C, 0x00, 0x00, 0x00,
	0x00, 0x7F, 0x7F, 0x70, 0x70, 0x7C, 0x7E, 0x0E,
	0x0E, 0x0E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7E, 0x70, 0x70, 0x7C, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x7F, 0x7F, 0x07, 0x07, 0x0E, 0x0E, 0x1C,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x77, 0x7F, 0x3E, 0x7E,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x77, 0x77, 0x7F, 0x3E,
	0x0E, 0x0E, 0x0E, 0x7C, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x1C, 0x00,
	0x00, 0x00, 0x38, 0x38, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x1C, 0x00,
	0x00, 0x18, 0x38, 0x38, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x03, 0x07, 0x0E, 0x1C, 0x70,
	0x70, 0x38, 0x1C, 0x0E, 0x06, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7F, 0x7F, 0x00,
	0x00, 0xFE, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x30, 0x38, 0x1C, 0x0E, 0x0E,
	0x0E, 0x1C, 0x38, 0x70, 0x60, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x07, 0x07, 0x1E, 0x7C,
	0x70, 0x00, 0x70, 0x70, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x77, 0x77, 0x7F, 0xFE,
	0xEC, 0xE0, 0xE0, 0xFC, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x77, 0x77, 0x76, 0xFE,
	0xFE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x7E, 0x7F, 0x77, 0x77, 0x77, 0x7E, 0xFC,
	0xEE, 0xEE, 0xEE, 0xFE, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x77, 0x70, 0x70, 0xE0,
	0xE0, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0x7E, 0x77, 0x77, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFC, 0xF8, 0x00, 0x00, 0x00,
	0x00, 0x7F, 0x7F, 0x70, 0x70, 0x70, 0x7C, 0xF8,
	0xE0, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x7F, 0x7F, 0x70, 0x70, 0x70, 0x7C, 0xF8,
	0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x77, 0x70, 0x70, 0xFE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x77, 0x77, 0x77, 0x77, 0x77, 0x7E, 0xFE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x7F, 0x7F, 0x1C, 0x1C, 0x1C, 0x1C, 0x38,
	0x38, 0x38, 0x38, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x07, 0x06, 0x0E,
	0x0E, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x70, 0x70, 0x72, 0x77, 0x7F, 0x7E, 0xF8,
	0xF8, 0xFC, 0xFE, 0xEE, 0xE6, 0x00, 0x00, 0x00,
	0x00, 0x70, 0x70, 0x70, 0x70, 0x70, 0x70, 0xE0,
	0xE0, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x63, 0x77, 0x7F, 0x7F, 0x7F, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x67, 0x77, 0x77, 0x7F, 0x7F, 0x7E, 0xFE,
	0xFE, 0xFE, 0xEE, 0xEE, 0xE6, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x77, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x7E, 0x7F, 0x77, 0x77, 0x77, 0x77, 0xFE,
	0xFC, 0xE0, 0xE0, 0xE0, 0xE0, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x77, 0x77, 0x76, 0xEE,
	0xEE, 0xFE, 0xFE, 0xFE, 0x7E, 0x06, 0x00, 0x00,
	0x00, 0x7E, 0x7F, 0x77, 0x77, 0x77, 0x7E, 0xFC,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x70, 0x70, 0x7C, 0x3E,
	0x0E, 0x0E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x7F, 0x7F, 0x1C, 0x1C, 0x1C, 0x1C, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x77, 0x77, 0x77, 0x77, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x77, 0x77, 0x77, 0x77, 0x77, 0x76, 0xEE,
	0x6C, 0x7C, 0x38, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x77, 0x77, 0x77, 0x77, 0x77, 0x76, 0xEE,
	0xFE, 0xFE, 0xFE, 0xEE, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x77, 0x77, 0x77, 0x77, 0x3E, 0x1C, 0x38,
	0x7C, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x77, 0x77, 0x77, 0x77, 0x77, 0x3E, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x7F, 0x7F, 0x07, 0x07, 0x0E, 0x1C, 0x38,
	0x70, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x1F, 0x1F, 0x1C, 0x1C, 0x1C, 0x1C, 0x38,
	0x38, 0x38, 0x38, 0x3E, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x20, 0x70, 0x78, 0x38,
	0x1C, 0x1E, 0x0E, 0x04, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x7E, 0x7E, 0x0E, 0x0E, 0x0E, 0x0E, 0x1C,
	0x1C, 0x1C, 0x1C, 0xFC, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x08, 0x1C, 0x3E, 0x77, 0x63, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x1C, 0x0C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x3E, 0x0E,
	0x3E, 0x7E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x38, 0x38, 0x70, 0x7C, 0x7E, 0x6E,
	0xEE, 0xEE, 0xEE, 0xFC, 0xF8, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x7E, 0xEE,
	0xE0, 0xE0, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x3F, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x7E, 0xEE,
	0xFE, 0xFC, 0xE0, 0xFC, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x1F, 0x3F, 0x38, 0x38, 0x70, 0x7C, 0x7C,
	0x70, 0xE0, 0xE0, 0xE0, 0xE0, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x1C, 0xFC, 0xF8,
	0x00, 0x00, 0x70, 0x70, 0x70, 0x7C, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x0E, 0x0E, 0x00, 0x1E, 0x3E, 0x1C,
	0x1C, 0x38, 0x38, 0x7C, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x0E, 0x0E, 0x00, 0x1E, 0x3E, 0x0E,
	0x1C, 0x1C, 0x1C, 0x1C, 0x38, 0x38, 0xF8, 0xF0,
	0x00, 0x38, 0x38, 0x38, 0x3A, 0x7F, 0x7E, 0x7C,
	0x70, 0xF8, 0xFC, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x3C, 0x3C, 0x1C, 0x1C, 0x38, 0x38, 0x38,
	0x38, 0x70, 0x70, 0x70, 0x70, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x74, 0x7E, 0xFE,
	0xFE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFC, 0xF8, 0xE0, 0xE0, 0xE0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFC, 0x7C, 0x1C, 0x1E, 0x1E,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0x7E, 0x70,
	0x70, 0xE0, 0xE0, 0xE0, 0xE0, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1F, 0x3E, 0x70,
	0x7C, 0x3E, 0x0E, 0x7C, 0xF8, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x38, 0x38, 0x70, 0x7C, 0x7C, 0x70,
	0xE0, 0xE0, 0xE0, 0xFC, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0x7C, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x77, 0x76, 0xEE,
	0xFE, 0xFE, 0xFE, 0xEE, 0x44, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x77, 0x76, 0x7E,
	0x38, 0x7C, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xFE, 0x7E, 0x1C, 0x1C, 0xFC, 0xF8,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7F, 0x7E, 0x1E,
	0x3C, 0x78, 0xF0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x0F, 0x1F, 0x1C, 0x1C, 0x1C, 0x78, 0xF0,
	0x38, 0x38, 0x38, 0x3E, 0x1E, 0x00, 0x00, 0x00,
	0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x00,
	0x00, 0x78, 0x7C, 0x1C, 0x1C, 0x1C, 0x0F, 0x1E,
	0x38, 0x38, 0x38, 0xF8, 0xF0, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x33, 0x7F, 0x7F, 0x66, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x7F, 0x00, 0x77, 0x55, 0x55, 0x55, 0x77,
	0x00, 0xEE, 0x28, 0x4C, 0x48, 0x48, 0x00, 0xFE,
	0x00, 0x06, 0x1F, 0x3B, 0x70, 0xFE, 0x70, 0xF8,
	0xE0, 0xE0, 0xEE, 0xFC, 0x70, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x38, 0x30, 0x60, 0x00,
	0x00, 0x1F, 0x3F, 0x38, 0x38, 0x70, 0x7C, 0x7C,
	0x70, 0xE0, 0xE0, 0xE0, 0xE0, 0xC0, 0xC0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xEE, 0xEE, 0xCC, 0x98, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xDB, 0xDB, 0xDB, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x1C, 0x1C, 0x1C, 0x7F, 0x7F, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00,
	0x00, 0x1C, 0x1C, 0x1C, 0x7F, 0x7F, 0x1C, 0xFE,
	0xFE, 0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00,
	0x00, 0x3E, 0x77, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0x67, 0x1C,
	0x38, 0x70, 0xE0, 0xDB, 0x1B, 0x00, 0x00, 0x00,
	0x77, 0x3E, 0x00, 0x3E, 0x7F, 0x77, 0x70, 0x7C,
	0x3E, 0x0E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x07, 0x1C,
	0x38, 0x70, 0x38, 0x1C, 0x0E, 0x06, 0x00, 0x00,
	0x00, 0x3F, 0x7F, 0x77, 0x77, 0x77, 0x77, 0xEF,
	0xEE, 0xEE, 0xEE, 0xFF, 0x7F, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x77, 0x3E, 0x00, 0x7F, 0x7F, 0x07, 0x0E, 0x1C,
	0x38, 0x70, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x0C, 0x18, 0x18, 0x18, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x18, 0x18, 0x18, 0x30, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1B, 0x36, 0x36, 0x36, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1B, 0x1B, 0x1B, 0x36, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1E, 0x7E,
	0x7E, 0x7E, 0x7E, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x3C, 0x3C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFE, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x3B, 0x6E, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x79, 0x7F, 0x2F, 0x29, 0x29, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x77, 0x3E, 0x1C, 0x00, 0x1F, 0x3E, 0x70,
	0x7C, 0x3E, 0x0E, 0x7C, 0xF8, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x38, 0x38,
	0x1C, 0x0E, 0x1C, 0x38, 0x70, 0x60, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x7F, 0xFA,
	0xDE, 0xDE, 0xFC, 0xFE, 0x6E, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x77, 0x3E, 0x1C, 0x00, 0x7F, 0x7E, 0x1E,
	0x3C, 0x78, 0xF0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x3B, 0x00, 0x77, 0x77, 0x77, 0x77, 0x3E,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x1C, 0x00,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x3E, 0x7F, 0xD6,
	0xD0, 0xD0, 0xD6, 0xFE, 0x7C, 0x10, 0x10, 0x00,
	0x00, 0x1E, 0x3F, 0x3B, 0x38, 0x7E, 0x7E, 0xFC,
	0x70, 0x76, 0xFE, 0xFE, 0xDC, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x77, 0x77, 0x3E, 0xFE,
	0xFE, 0xFE, 0xFE, 0x7C, 0xEE, 0xEE, 0x00, 0x00,
	0x00, 0x00, 0x77, 0x77, 0x77, 0x3E, 0x1C, 0xFE,
	0x38, 0xFE, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x1C, 0x1C, 0x1C, 0x00,
	0x00, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3E, 0x3E, 0x38, 0x1E, 0x3F, 0x66,
	0x7E, 0x3C, 0x0E, 0x3E, 0x3E, 0x3C, 0x00, 0x00,
	0x00, 0x77, 0x77, 0x77, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x61, 0x4D, 0x5D, 0x51, 0xA1,
	0xA5, 0xBD, 0x99, 0xC3, 0x7E, 0x3C, 0x00, 0x00,
	0x00, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00, 0xFC,
	0xFC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E,
	0x7C, 0xF8, 0xF8, 0x7C, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x3F, 0x3F, 0x3F, 0x03, 0x01, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x7E, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3F, 0x61, 0x59, 0x5D, 0x55, 0xB9,
	0xB9, 0xA5, 0xA5, 0xC3, 0x7E, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7F, 0x7F, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x3C, 0x7E, 0x66, 0x7E, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1C, 0x1C, 0x7F, 0x7F, 0x7F, 0x38,
	0x38, 0x00, 0xFE, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x78, 0x7C, 0x0C, 0x3C, 0x78, 0xC0,
	0xF8, 0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x78, 0x7C, 0x0C, 0x18, 0x0C, 0xF8,
	0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x77, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0xFC, 0xE0, 0xE0, 0xC0,
	0x00, 0x00, 0x00, 0x3F, 0x6D, 0x6D, 0x6D, 0x7A,
	0x36, 0x36, 0x36, 0x36, 0x36, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x38,
	0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x1C, 0x0C, 0x1C, 0x18, 0x00, 0x00,
	0x00, 0x00, 0x30, 0x70, 0x30, 0x30, 0x30, 0xF0,
	0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x7C, 0x44, 0x7C, 0x38, 0x00, 0xF8,
	0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD8,
	0x6C, 0x36, 0x36, 0x6C, 0xD8, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x30, 0x70, 0x31, 0x33, 0x7F, 0x1C,
	0x38, 0x76, 0xEE, 0xD6, 0x3E, 0x3E, 0x06, 0x00,
	0x00, 0x00, 0x30, 0x70, 0x31, 0x33, 0x7F, 0x1C,
	0x38, 0x70, 0xFC, 0xC6, 0x0C, 0x18, 0x1E, 0x00,
	0x00, 0x00, 0x70, 0x18, 0x31, 0x1B, 0x77, 0x1C,
	0x3A, 0x76, 0xEE, 0xD6, 0x3E, 0x3E, 0x06, 0x00,
	0x00, 0x00, 0x0E, 0x0E, 0x0E, 0x00, 0x0E, 0x1C,
	0x7C, 0xF8, 0xE0, 0xEE, 0xEE, 0xFE, 0x7C, 0x00,
	0x00, 0x70, 0x38, 0x00, 0x08, 0x1C, 0x3E, 0xEE,
	0xFE, 0xFE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x07, 0x0E, 0x00, 0x08, 0x1C, 0x3E, 0xEE,
	0xFE, 0xFE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x36, 0x63, 0x08, 0x1C, 0x3E, 0xEE,
	0xFE, 0xFE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x3B, 0x6E, 0x00, 0x08, 0x1C, 0x3E, 0xEE,
	0xFE, 0xFE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x77, 0x77, 0x08, 0x1C, 0x3E, 0x76, 0xFE,
	0xFE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x36, 0x1C, 0x1C, 0x3E, 0x76, 0xFE,
	0xFE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x0F, 0x1F, 0x3F, 0x7F, 0x77, 0x7F, 0xFF,
	0xFE, 0xEE, 0xEF, 0xEF, 0xEF, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x70, 0x70, 0x70, 0xE0,
	0xE0, 0xE0, 0xEE, 0xFE, 0x7C, 0x0E, 0x6E, 0x7C,
	0x00, 0x38, 0x1C, 0x00, 0x7F, 0x7F, 0x70, 0xE0,
	0xF8, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1C, 0x00, 0x7F, 0x7F, 0x70, 0xE0,
	0xF8, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x77, 0x00, 0x7F, 0x7F, 0x70, 0xE0,
	0xF8, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x77, 0x77, 0x00, 0x7F, 0x7F, 0x70, 0x70, 0xF8,
	0xF8, 0xE0, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x1C, 0x00, 0x1C, 0x1C, 0x1C, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1C, 0x00, 0x1C, 0x1C, 0x1C, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x36, 0x1C, 0x1C, 0x1C, 0x1C, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x77, 0x77, 0x00, 0x1C, 0x1C, 0x1C, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x7E, 0x77, 0x77, 0x76, 0xFE,
	0xEE, 0xEE, 0xEE, 0xFC, 0xF8, 0x00, 0x00, 0x00,
	0x00, 0x3B, 0x6E, 0x00, 0x77, 0x77, 0x76, 0xF6,
	0xFE, 0xDE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x1C, 0x00, 0x3E, 0x7F, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1C, 0x00, 0x3E, 0x7F, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x77, 0x00, 0x3E, 0x7F, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x3B, 0x6E, 0x00, 0x3E, 0x7F, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x77, 0x77, 0x00, 0x3E, 0x7F, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0x76, 0x7C,
	0x38, 0x38, 0x7C, 0xEE, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x3D, 0x7E, 0x77, 0x77, 0x77, 0x7E, 0xFE,
	0xFE, 0xEE, 0xEE, 0xFE, 0x7C, 0x80, 0x00, 0x00,
	0x00, 0x38, 0x1C, 0x00, 0x77, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1C, 0x00, 0x77, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x77, 0x00, 0x77, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x77, 0x77, 0x00, 0x77, 0x77, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1C, 0x00, 0x77, 0x77, 0x76, 0xEE,
	0x7C, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x70, 0x70, 0x7E, 0x7F, 0x76, 0xEE,
	0xEE, 0xFE, 0xFC, 0xE0, 0xE0, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x7F, 0x77, 0x77, 0x77, 0x7E, 0xFC,
	0xEE, 0xEE, 0xEE, 0xFE, 0xFC, 0xE0, 0xC0, 0x00,
	0x00, 0x00, 0x70, 0x38, 0x00, 0x3C, 0x3E, 0x0E,
	0x3E, 0x7E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1C, 0x38, 0x00, 0x3C, 0x3E, 0x0E,
	0x3E, 0x7E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x7C, 0xEE, 0x00, 0x3C, 0x3E, 0x0E,
	0x3E, 0x7E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x76, 0xDC, 0x00, 0x3C, 0x3E, 0x0E,
	0x3E, 0x7E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xEE, 0xEE, 0x00, 0x3C, 0x3E, 0x0E,
	0x3E, 0x7E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x6C, 0x38, 0x00, 0x3C, 0x3E, 0x0E,
	0x3E, 0x7E, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x3A, 0x3F, 0x3F, 0x1A,
	0x7E, 0xFE, 0xD8, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0xFE,
	0xEE, 0xE0, 0xEE, 0xFE, 0x7C, 0x1C, 0x7E, 0x7C,
	0x00, 0x00, 0x38, 0x1C, 0x00, 0x3E, 0x7F, 0xEE,
	0xFE, 0xFE, 0xE0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1C, 0x38, 0x00, 0x3E, 0x7F, 0xEE,
	0xFE, 0xFE, 0xE0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x3E, 0x77, 0x00, 0x3E, 0x7F, 0xEE,
	0xFE, 0xFE, 0xE0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x77, 0x77, 0x00, 0x3E, 0x7F, 0xEE,
	0xFE, 0xFE, 0xE0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x38, 0x1C, 0x00, 0x1C, 0x1C, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x0E, 0x1C, 0x00, 0x1C, 0x1C, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x3E, 0x77, 0x00, 0x1C, 0x1C, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x77, 0x77, 0x00, 0x1C, 0x1C, 0x38,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x76, 0x3E, 0x7C, 0x7E, 0x3F, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x3B, 0x7F, 0x6E, 0x00, 0x7E, 0x7F, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x70, 0x38, 0x00, 0x3C, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1C, 0x38, 0x00, 0x3C, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x3E, 0x77, 0x00, 0x3C, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x3B, 0x7F, 0x6E, 0x00, 0x3C, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x77, 0x77, 0x00, 0x3C, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x1C, 0x1C, 0x00, 0xFE,
	0xFE, 0x00, 0x38, 0x38, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x01, 0x3E, 0x7E, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7C, 0x80, 0x00, 0x00,
	0x00, 0x00, 0x70, 0x38, 0x00, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1C, 0x38, 0x00, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x1C, 0x3E, 0x77, 0x00, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x77, 0x77, 0x00, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x0E, 0x1C, 0x00, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x0E, 0xFE, 0xFC,
	0x00, 0x00, 0x70, 0x70, 0x7E, 0x7F, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xFE, 0xFC, 0xE0, 0xE0,
	0x00, 0x00, 0x77, 0x77, 0x00, 0x77, 0x76, 0xEE,
	0xEE, 0xEE, 0xEE, 0xFE, 0x7E, 0x0E, 0xFE, 0xFC,
};

const struct fb_font_desc font_italic_bold = {
	.name = "NetSurf Italic Bold",
	.width = 8,
	.height = 16,
	.encoding = "CP1252",
	.data = fontdata_italic_bold,
};


static const uint32_t fontdata_regular[FONTDATAMAX] = {
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xE4, 0xAC, 0xA4, 0xA4, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEC, 0xA2, 0xA4, 0xA8, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEC, 0xA2, 0xAC, 0xA2, 0xEC, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEA, 0xAA, 0xAE, 0xA2, 0xE2, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xA8, 0xAE, 0xA2, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xE6, 0xA8, 0xAE, 0xAA, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xA2, 0xA4, 0xA4, 0xE4, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xAA, 0xAE, 0xAA, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xAA, 0xAE, 0xA2, 0xE2, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xE4, 0xAA, 0xAE, 0xAA, 0xEA, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEC, 0xAA, 0xAC, 0xAA, 0xEC, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xE4, 0xAA, 0xA8, 0xAA, 0xE4, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEC, 0xAA, 0xAA, 0xAA, 0xEC, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xA8, 0xAC, 0xA8, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0xA8, 0xAC, 0xA8, 0xE8, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xCA, 0x4A, 0x4A, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x44, 0xCC, 0x44, 0x44, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4C, 0xC2, 0x44, 0x48, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4C, 0xC2, 0x4C, 0x42, 0xEC, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4A, 0xCA, 0x4E, 0x42, 0xE2, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xC8, 0x4E, 0x42, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x46, 0xC8, 0x4E, 0x4A, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xC2, 0x44, 0x44, 0xE4, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xCA, 0x4E, 0x4A, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xCA, 0x4E, 0x42, 0xE2, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x44, 0xCA, 0x4E, 0x4A, 0xEA, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4C, 0xCA, 0x4C, 0x4A, 0xEC, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x44, 0xCA, 0x48, 0x4A, 0xE4, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4C, 0xCA, 0x4A, 0x4A, 0xEC, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xC8, 0x4C, 0x48, 0xEE, 0x00, 0xFE,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0x4E, 0xC8, 0x4C, 0x48, 0xE8, 0x00, 0xFE,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x6C, 0x6C, 0x6C, 0x6C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x6C, 0x6C, 0x6C, 0xFE, 0xFE, 0x6C, 0x6C,
	0xFE, 0xFE, 0x6C, 0x6C, 0x6C, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x7C, 0xFE, 0xD6, 0xD0, 0xF0, 0x7C,
	0x1E, 0x16, 0xD6, 0xFE, 0x7C, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xCE, 0x1C,
	0x38, 0x70, 0xE6, 0xC6, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x70, 0xF8, 0xD8, 0xD8, 0xF8, 0x72, 0xFE,
	0xDE, 0xCC, 0xCC, 0xFE, 0x7A, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x06, 0x0E, 0x1C, 0x18, 0x38, 0x30, 0x30, 0x30,
	0x30, 0x38, 0x18, 0x1C, 0x0E, 0x06, 0x00, 0x00,
	0x60, 0x70, 0x38, 0x18, 0x1C, 0x0C, 0x0C, 0x0C,
	0x0C, 0x1C, 0x18, 0x38, 0x70, 0x60, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x6C, 0x6C, 0x38, 0xFE,
	0xFE, 0x38, 0x6C, 0x6C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7E, 0x7E,
	0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x38, 0x30, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0x7E,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x0E, 0x1C,
	0x38, 0x70, 0xE0, 0xC0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0xCE, 0xCE, 0xD6, 0xD6,
	0xE6, 0xE6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x38, 0x78, 0x78, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0x06, 0x06, 0x3E, 0x7C,
	0xE0, 0xC0, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0x06, 0x06, 0x3C, 0x3C,
	0x06, 0x06, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x0C, 0x1C, 0x3C, 0x7C, 0xEC, 0xFE,
	0xFE, 0x0C, 0x0C, 0x0C, 0x0C, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0xFE, 0xC0, 0xC0, 0xC0, 0xFC, 0xFE,
	0x06, 0x06, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x3C, 0x7C, 0xE0, 0xC0, 0xC0, 0xFC, 0xFE,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0xFE, 0xC6, 0x0E, 0x0C, 0x1C, 0x18,
	0x38, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0xC6, 0xC6, 0x7C, 0x7C,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0xC6, 0xC6, 0xFE, 0x7E,
	0x06, 0x06, 0x0E, 0x7C, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x00, 0x18, 0x18, 0x38, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x06, 0x0E, 0x1C, 0x38, 0x70, 0x70,
	0x38, 0x1C, 0x0E, 0x06, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x7E, 0x7E, 0x00, 0x00,
	0x7E, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x60, 0x70, 0x38, 0x1C, 0x0E, 0x0E,
	0x1C, 0x38, 0x70, 0x60, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0xC6, 0x06, 0x1E, 0x3C,
	0x30, 0x30, 0x00, 0x30, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0xCE, 0xDE, 0xD6, 0xD6,
	0xDE, 0xCC, 0xC0, 0xFC, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0xC6, 0xC6, 0xFE, 0xFE,
	0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0xFC, 0xFE, 0xC6, 0xC6, 0xC6, 0xFC, 0xFC,
	0xC6, 0xC6, 0xC6, 0xFE, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0xC0, 0xC0, 0xC0, 0xC0,
	0xC0, 0xC0, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0xF8, 0xFC, 0xCE, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xCE, 0xFC, 0xF8, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0xFE, 0xC0, 0xC0, 0xC0, 0xF8, 0xF8,
	0xC0, 0xC0, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0xFE, 0xC0, 0xC0, 0xC0, 0xF8, 0xF8,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0xC0, 0xC0, 0xDE, 0xDE,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xFE, 0xFE,
	0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x7E, 0x7E, 0x18, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x7E, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
	0x0C, 0x0C, 0xCC, 0xFC, 0x78, 0x00, 0x00, 0x00,
	0x00, 0xC0, 0xC6, 0xCE, 0xDC, 0xF8, 0xF0, 0xE0,
	0xF0, 0xF8, 0xDC, 0xCE, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0,
	0xC0, 0xC0, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0xC6, 0xC6, 0xEE, 0xFE, 0xFE, 0xFE, 0xD6,
	0xD6, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0xC6, 0xC6, 0xE6, 0xE6, 0xF6, 0xF6, 0xDE,
	0xDE, 0xCE, 0xCE, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0xFC, 0xFE, 0xC6, 0xC6, 0xC6, 0xFE, 0xFC,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xD6, 0xDE, 0xFE, 0x7C, 0x06, 0x00, 0x00,
	0x00, 0xFC, 0xFE, 0xC6, 0xC6, 0xC6, 0xFC, 0xFC,
	0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0xC0, 0xC0, 0xFC, 0x7E,
	0x06, 0x06, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x7E, 0x7E, 0x18, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xEE,
	0x6C, 0x7C, 0x38, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xD6,
	0xD6, 0xFE, 0xFE, 0xEE, 0x44, 0x00, 0x00, 0x00,
	0x00, 0xC6, 0xC6, 0xC6, 0xEE, 0x7C, 0x38, 0x38,
	0x7C, 0xEE, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xEE, 0x7C, 0x38,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0xFE, 0x0E, 0x0C, 0x1C, 0x18, 0x38,
	0x30, 0x70, 0x60, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x3E, 0x3E, 0x30, 0x30, 0x30, 0x30, 0x30,
	0x30, 0x30, 0x30, 0x3E, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xE0, 0x70,
	0x38, 0x1C, 0x0E, 0x06, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0x7C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C,
	0x0C, 0x0C, 0x0C, 0x7C, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x7C, 0xEE, 0xC6, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00,
	0x00, 0x00, 0x30, 0x30, 0x18, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x7E, 0x06,
	0x7E, 0xFE, 0xC6, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0xC0, 0xC0, 0xC0, 0xC0, 0xFC, 0xFE, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xFE, 0xC6,
	0xC0, 0xC0, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x06, 0x06, 0x06, 0x06, 0x7E, 0xFE, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xFE, 0xC6,
	0xFE, 0xFE, 0xC0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x1E, 0x3E, 0x30, 0x30, 0x30, 0x78, 0x78,
	0x30, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0xFE, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7E, 0x06, 0x7E, 0x7C,
	0x00, 0x00, 0xC0, 0xC0, 0xC0, 0xDC, 0xFE, 0xE6,
	0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x18, 0x18, 0x00, 0x38, 0x38, 0x18,
	0x18, 0x18, 0x18, 0x3C, 0x3C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x0C, 0x0C, 0x00, 0x1C, 0x1C, 0x0C,
	0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x7C, 0x78,
	0x00, 0xC0, 0xC0, 0xC0, 0xC0, 0xC6, 0xCE, 0xDC,
	0xF8, 0xF8, 0xDC, 0xCE, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xEC, 0xFE, 0xFE,
	0xD6, 0xD6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xDC, 0xFE, 0xE6,
	0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xFE, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xFC, 0xFE, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0xFC, 0xC0, 0xC0, 0xC0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0xFE, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7E, 0x06, 0x07, 0x07,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xDE, 0xFE, 0xE0,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0xFE, 0xC0,
	0xFC, 0x7C, 0x06, 0xFE, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x30, 0x30, 0x30, 0x7C, 0xFC, 0x30,
	0x30, 0x30, 0x30, 0x3E, 0x1E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xCE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xC6, 0xC6,
	0xC6, 0xEE, 0x7C, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xC6, 0xC6,
	0xD6, 0xD6, 0xFE, 0xFE, 0x6C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xC6, 0xEE,
	0x7C, 0x7C, 0xEE, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7E, 0x06, 0xFE, 0xFC,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0xFE, 0x0E,
	0x3C, 0x78, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x0E, 0x1E, 0x18, 0x18, 0x18, 0x70, 0x70,
	0x18, 0x18, 0x18, 0x1E, 0x0E, 0x00, 0x00, 0x00,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x00, 0x70, 0x78, 0x18, 0x18, 0x18, 0x0E, 0x0E,
	0x18, 0x18, 0x18, 0x78, 0x70, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x76, 0xFE, 0xDC, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0xFE, 0x00, 0xEE, 0xAA, 0xAA, 0xAA, 0xEE,
	0x00, 0xEE, 0x28, 0x4C, 0x48, 0x48, 0x00, 0xFE,
	0x00, 0x18, 0x3C, 0x66, 0x60, 0xFC, 0x60, 0xF8,
	0x60, 0x60, 0x66, 0x3C, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x08, 0x10, 0x00,
	0x00, 0x1E, 0x3E, 0x30, 0x30, 0x30, 0x78, 0x78,
	0x30, 0x30, 0x30, 0x30, 0x30, 0xF0, 0x60, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x66, 0x66, 0x22, 0x44, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xDB, 0xDB, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x18, 0x18, 0x7E, 0x7E, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00,
	0x00, 0x18, 0x18, 0x18, 0x7E, 0x7E, 0x18, 0x7E,
	0x7E, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00,
	0x00, 0x7C, 0xC6, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC2, 0xC6, 0x0C,
	0x18, 0x30, 0x60, 0xDB, 0x9B, 0x00, 0x00, 0x00,
	0xC6, 0x7C, 0x00, 0x7C, 0xFE, 0xC6, 0xC0, 0xFC,
	0x7E, 0x06, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x1C,
	0x38, 0x30, 0x38, 0x1C, 0x0C, 0x00, 0x00, 0x00,
	0x00, 0x7F, 0xFF, 0xCC, 0xCC, 0xCC, 0xCF, 0xCF,
	0xCC, 0xCC, 0xCC, 0xFF, 0x7F, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0xC6, 0x7C, 0x00, 0xFE, 0xFE, 0x06, 0x0C, 0x18,
	0x30, 0x60, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x20, 0x30, 0x30, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x30, 0x30, 0x10, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x24, 0x48, 0x6C, 0x6C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x36, 0x36, 0x12, 0x24, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x7E,
	0x7E, 0x7E, 0x7E, 0x3C, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x3C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x32, 0x4C, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xF1, 0xFB, 0x5F, 0x55, 0x51, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0xC6, 0x6C, 0x38, 0x00, 0x7E, 0xFE, 0xC0,
	0xFC, 0x7C, 0x06, 0xFE, 0xFC, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x38,
	0x1C, 0x0C, 0x1C, 0x38, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x6C, 0xFE, 0xDA,
	0xDE, 0xDE, 0xD8, 0xFE, 0x6E, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x38, 0x6C, 0x54, 0xF6, 0xEE,
	0x6C, 0x7C, 0x28, 0x38, 0x10, 0x00, 0x00, 0x00,
	0x00, 0xC6, 0x6C, 0x38, 0x00, 0xFE, 0xFE, 0x0E,
	0x3C, 0x78, 0xE0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x6C, 0x00, 0xC6, 0xC6, 0xC6, 0xEE, 0x7C,
	0x38, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x10, 0x7C, 0xFE, 0xD6,
	0xD0, 0xD0, 0xD6, 0xFE, 0x7C, 0x10, 0x10, 0x00,
	0x00, 0x3C, 0x7E, 0x66, 0x60, 0x60, 0xF8, 0xF8,
	0x60, 0x60, 0xE6, 0xFE, 0xDC, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x3C, 0x7E,
	0x66, 0x66, 0x7E, 0x3C, 0x66, 0x66, 0x00, 0x00,
	0x00, 0x00, 0x66, 0x66, 0x66, 0x3C, 0x18, 0x7E,
	0x18, 0x7E, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x3C, 0x7C, 0x60, 0x60, 0x3C, 0x7E, 0x66,
	0x7E, 0x3C, 0x06, 0x06, 0x3E, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x3C, 0x7E, 0xC3, 0x99, 0xBD, 0xA5, 0xA1,
	0xA5, 0xBD, 0x99, 0xC3, 0x7E, 0x3C, 0x00, 0x00,
	0x00, 0x70, 0x08, 0x78, 0x88, 0x78, 0x00, 0xF8,
	0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x36,
	0x6C, 0xD8, 0xD8, 0x6C, 0x36, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7E, 0x7E, 0x06, 0x02, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x7E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x3C, 0x7E, 0xC3, 0xB9, 0xBD, 0xA5, 0xB9,
	0xB9, 0xA5, 0xA5, 0xC3, 0x7E, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7E, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x78, 0xFC, 0xCC, 0xFC, 0x78, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x7E, 0x7E, 0x18,
	0x18, 0x00, 0x7E, 0x7E, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xF0, 0xF8, 0x18, 0x78, 0xF0, 0xC0,
	0xF8, 0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xF0, 0xF8, 0x18, 0x30, 0x18, 0xF8,
	0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x30, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66,
	0x66, 0x66, 0x66, 0x7E, 0x7C, 0x60, 0x60, 0xC0,
	0x00, 0x00, 0x00, 0x7F, 0xDB, 0xDB, 0xDB, 0x7B,
	0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
	0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x1C, 0x0C, 0x1C, 0x18, 0x00, 0x00,
	0x00, 0x00, 0x60, 0xE0, 0x60, 0x60, 0x60, 0xF0,
	0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x70, 0xF8, 0x88, 0xF8, 0x70, 0x00, 0xF8,
	0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD8,
	0x6C, 0x36, 0x36, 0x6C, 0xD8, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x40, 0xC0, 0x42, 0x46, 0xEC, 0x18,
	0x30, 0x64, 0xCC, 0x94, 0x3E, 0x04, 0x04, 0x00,
	0x00, 0x00, 0x40, 0xC0, 0x42, 0x46, 0xEC, 0x18,
	0x30, 0x60, 0xDC, 0x82, 0x0C, 0x10, 0x1E, 0x00,
	0x00, 0x00, 0xE0, 0x10, 0x62, 0x16, 0xEC, 0x18,
	0x30, 0x64, 0xCC, 0x96, 0x3E, 0x04, 0x04, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18,
	0x78, 0xF0, 0xC0, 0xC6, 0xC6, 0xFE, 0x7C, 0x00,
	0x00, 0x60, 0x30, 0x00, 0x10, 0x38, 0x6C, 0xC6,
	0xFE, 0xFE, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x0C, 0x18, 0x00, 0x10, 0x38, 0x6C, 0xC6,
	0xFE, 0xFE, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x6C, 0x44, 0x10, 0x38, 0x6C, 0xC6,
	0xFE, 0xFE, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x76, 0xDC, 0x00, 0x10, 0x38, 0x6C, 0xC6,
	0xFE, 0xFE, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x6C, 0x6C, 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xFE,
	0xFE, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x44, 0x38, 0x38, 0x6C, 0xC6, 0xFE,
	0xFE, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x1F, 0x3F, 0x7C, 0xEC, 0xCC, 0xFE, 0xFE,
	0xCC, 0xCC, 0xCC, 0xCF, 0xCF, 0x00, 0x00, 0x00,
	0x00, 0x7C, 0xFE, 0xC6, 0xC0, 0xC0, 0xC0, 0xC0,
	0xC0, 0xC0, 0xC6, 0xFE, 0x7C, 0x0E, 0x66, 0x3C,
	0x00, 0x30, 0x18, 0x00, 0xFE, 0xFE, 0xC0, 0xF0,
	0xF0, 0xC0, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x0C, 0x18, 0x00, 0xFE, 0xFE, 0xC0, 0xF0,
	0xF0, 0xC0, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x6C, 0x00, 0xFE, 0xFE, 0xC0, 0xF0,
	0xF0, 0xC0, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x6C, 0x6C, 0x00, 0xFE, 0xFE, 0xC0, 0xC0, 0xF0,
	0xF0, 0xC0, 0xC0, 0xFE, 0xFE, 0x00, 0x00, 0x00,
	0x00, 0x30, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x0C, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x24, 0x18, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x66, 0x66, 0x00, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x78, 0x7C, 0x6E, 0x66, 0x66, 0xF6,
	0x66, 0x66, 0x6E, 0x7C, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x76, 0xDC, 0x00, 0xC6, 0xC6, 0xE6, 0xF6,
	0xFE, 0xDE, 0xCE, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x30, 0x18, 0x00, 0x7C, 0xFE, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x0C, 0x18, 0x00, 0x7C, 0xFE, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x6C, 0x00, 0x7C, 0xFE, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x76, 0xDC, 0x00, 0x7C, 0xFE, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x6C, 0x00, 0x7C, 0xFE, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xEE, 0x7C,
	0x38, 0x38, 0x7C, 0xEE, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x7E, 0xFE, 0xC6, 0xC6, 0xCE, 0xCE, 0xD6,
	0xD6, 0xE6, 0xE6, 0xFE, 0x7C, 0x80, 0x00, 0x00,
	0x00, 0x30, 0x18, 0x00, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x0C, 0x18, 0x00, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x6C, 0x00, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x6C, 0x6C, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x0C, 0x18, 0x00, 0x66, 0x66, 0x66, 0x7E,
	0x3C, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x60, 0x60, 0x7C, 0x7E, 0x66, 0x66,
	0x7E, 0x7C, 0x60, 0x60, 0x60, 0x00, 0x00, 0x00,
	0x00, 0x3C, 0x7E, 0x66, 0x66, 0x66, 0x7E, 0x7C,
	0x66, 0x66, 0x66, 0x7E, 0x7C, 0xE0, 0xC0, 0x00,
	0x00, 0x00, 0x30, 0x18, 0x00, 0x7E, 0xFE, 0xC6,
	0xC6, 0xCE, 0xCE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x18, 0x30, 0x00, 0x7E, 0xFE, 0xC6,
	0xC6, 0xC6, 0xCE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x6C, 0x00, 0x7E, 0xFE, 0xC6,
	0xC6, 0xC6, 0xCE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x76, 0xDC, 0x00, 0x7E, 0xFE, 0xC6,
	0xC6, 0xC6, 0xCE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x6C, 0x6C, 0x00, 0x7E, 0xFE, 0xC6,
	0xC6, 0xC6, 0xCE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x38, 0x6C, 0x38, 0x00, 0x7E, 0xFE, 0xC6,
	0xC6, 0xC6, 0xCE, 0xFE, 0x76, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x74, 0x7E, 0x1A,
	0x3E, 0x7E, 0xD8, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xFE,
	0xC6, 0xC0, 0xC6, 0xFE, 0x7C, 0x18, 0x7C, 0x78,
	0x00, 0x00, 0x30, 0x18, 0x00, 0x7C, 0xFE, 0xC6,
	0xFE, 0xFE, 0xC0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x18, 0x30, 0x00, 0x7C, 0xFE, 0xC6,
	0xFE, 0xFE, 0xC0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x6C, 0x00, 0x7C, 0xFE, 0xC6,
	0xFE, 0xFE, 0xC0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x6C, 0x6C, 0x00, 0x7C, 0xFE, 0xC6,
	0xFE, 0xFE, 0xC0, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x30, 0x18, 0x00, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x0C, 0x18, 0x00, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x3C, 0x66, 0x00, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x66, 0x66, 0x00, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x6C, 0x7C, 0xF8, 0xDC, 0x7C, 0xFE, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x76, 0xFE, 0xDC, 0x00, 0xFC, 0xFE, 0xC6,
	0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x60, 0x30, 0x00, 0x7C, 0xFE, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x18, 0x30, 0x00, 0x7C, 0xFE, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x6C, 0x00, 0x7C, 0xFE, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x76, 0xFE, 0xDC, 0x00, 0x7C, 0xFE, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x6C, 0x6C, 0x00, 0x7C, 0xFE, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7C, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x7E,
	0x7E, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x02, 0x7C, 0xFE, 0xCE,
	0xD6, 0xD6, 0xE6, 0xFE, 0x7C, 0x80, 0x00, 0x00,
	0x00, 0x00, 0x60, 0x30, 0x00, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x18, 0x30, 0x00, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x6C, 0x00, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x6C, 0x6C, 0x00, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x0C, 0x18, 0x00, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7E, 0x06, 0xFC, 0xF8,
	0x00, 0x00, 0x60, 0x60, 0x60, 0x7C, 0x7E, 0x66,
	0x66, 0x66, 0x66, 0x7E, 0x7C, 0x60, 0x60, 0x60,
	0x00, 0x00, 0x6C, 0x6C, 0x00, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xFE, 0x7E, 0x06, 0xFE, 0xFC,
};

const struct fb_font_desc font_regular = {
	.name = "NetSurf Regular",
	.width = 8,
	.height = 16,
	.encoding = "CP1252",
	.data = fontdata_regular,
};
#endif
