/*  ROSA Media Player
    Julia Mineeva, Evgeniy Augin. Copyright (c) 2011 ROSA  <support@rosalab.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "ffmpegprocess.h"

#include <QRegExp>
#include <QStringList>
#include <QApplication>
#include <QTime>

#include "global.h"
#include "preferences.h"

using namespace Global;

FfmpegProcess::FfmpegProcess( QObject* parent )
    : MyProcess( parent )
{
    connect( this, SIGNAL( lineAvailable( QByteArray ) ),
             this, SLOT( parseLine( QByteArray ) ) );

    connect( this, SIGNAL( finished( int, QProcess::ExitStatus ) ),
             this, SLOT( processFinished( int, QProcess::ExitStatus ) ) );

    connect( this, SIGNAL( error( QProcess::ProcessError ) ),
             this, SLOT( gotError( QProcess::ProcessError) ) );
}

FfmpegProcess::~FfmpegProcess()
{
}

bool FfmpegProcess::start()
{
    MyProcess::start();
    return waitForStarted();
}

void FfmpegProcess::writeToStdin( const QString& text )
{
    if ( isRunning() )
    {
        //qDebug("FfmpegProcess::writeToStdin");
        write( text.toLocal8Bit() + "\n");
    }
    else
    {
        qWarning("FfmpegProcess::writeToStdin: process not running");
    }
}

static QRegExp rx_cur_sec("size=\\s+(.*)\\s+time=(.*)\\s+bitrate");
static QRegExp rx_cut_ok( "^video: *([0-9]+)kB\\s" );
static QRegExp rx_libmp3lame( "\\s*D?E?V?S?D?T?\\slibmp3lame\\s+" );

/*
output example:
size=    7876kB time=00:03:21.62 bitrate= 320.0kbits/s
video:0kB audio:7876kB global headers:0kB muxing overhead 0.002480%
*/
void FfmpegProcess::parseLine( QByteArray ba )
{
    QString line = QString::fromLocal8Bit( ba );

    // Parse other things
    qDebug("FfmpegProcess::parseLine: '%s'", line.toUtf8().data() );

    if ( rx_cur_sec.indexIn( line ) != -1 )
    {
        QString str = rx_cur_sec.cap( 2 );
        QTime t;
        int sec = t.secsTo ( QTime::fromString( str, "hh:mm:ss.z" ) );
        qDebug("FfmpegProcess::parseLine:   sec: %s (%d)", str.toLocal8Bit().data(), sec );
        emit receivedCurSec( sec );
    }

    if ( rx_cut_ok.indexIn( line ) != -1 )
    {
        qDebug( "FfmpegProcess::parseLine:   receivedCutFinished" );
        emit receivedCutFinished();
    }
    else if ( rx_libmp3lame.indexIn( line ) != (-1) )
    {
        qDebug( "FfmpegProcess::parseLine:   libmp3lameAvailable" );
        emit libmp3lameAvailable();
    }
    else
    {
        emit lineAvailable( line );
    }
}

// Called when the process is finished
void FfmpegProcess::processFinished( int exitCode, QProcess::ExitStatus exitStatus )
{
    qDebug("FfmpegProcess::processFinished: exitCode: %d, status: %d", exitCode, (int) exitStatus);
    // Send this signal before the endoffile one, otherwise
    // the playlist will start to play next file before all
    // objects are notified that the process has exited.
    emit processExited();
}

void FfmpegProcess::gotError(QProcess::ProcessError error)
{
    qDebug("FfmpegProcess::gotError: %d", (int) error);
    emit receivedError( error );
}

