/***************************************************************************
 *   Copyright (C) 2005 by Thierry CHARLES   *
 *   thierry@les-charles.net   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "povinifile.h"

#include <wx/regex.h>

#include "lib/lib_logging.h"

void TIniProfile::clear()
{
    this->entriesMap.erase(this->entriesMap.begin(), this->entriesMap.end());
}

void TIniProfile::addEntry(const wxString & sKey, const wxString & sValue)
{
    this->entriesMap[sKey].Add(sValue);
}

const wxString TIniProfile::getEntry(const wxString & sKey, uint iIdx) const
{
    const wxArrayString & entries = this->getEntries(sKey);
    if(entries.GetCount() <= iIdx)
        return wxString();
    return entries.Item(iIdx);
}

const wxArrayString TIniProfile::getEntries(const wxString & sKey) const
{
    TEntriesMap::const_iterator it = this->entriesMap.find(sKey);
    if(it == this->entriesMap.end())
        return wxArrayString();
    return (*it).second;
}

bool TIniProfile::isEmpty() const
{
    return this->entriesMap.size() > 0;
}

void TIniProfile::toStream(std::ostream & stream) const
{
    for(TEntriesMap::const_iterator it = this->entriesMap.begin() ; it != this->entriesMap.end() ; it++)
    {
        for (uint i = 0 ; i < (*it).second.GetCount() ; i++)
        {
            stream << WX2ISO( (*it).first ) << "=" << WX2ISO( (*it).second.Item(i) ) << std::endl;
        }
    }
}




wxString TIniProfile::fromStream(std::istream & stream)
{
    this->clear();
    char szBuffer[65536];
    static const wxRegEx profile_title(wxT("(\\[[a-zA-Z0-9\300-\377_\\-]+\\])"));

    do
    {
        stream.getline(szBuffer,65536);
        wxString sBuf = ISO2WX(szBuffer);
        sBuf.Trim(true).Trim(false);

        if(sBuf.size())
        {
            if(profile_title.Matches(sBuf))
            {
                // changement de profil
                return sBuf;
            }

            int iSepPos = sBuf.Find('=');
            if(iSepPos > 0)
            {
                wxString sKey = sBuf.Left(iSepPos);
                wxString sValue = sBuf.Mid(iSepPos+1);

                this->addEntry(sKey, sValue);
            }
            else
            {
                LOG_MESSAGE(wxString::Format(wxT("Unrecognized line : %s"), sBuf.c_str()), logging::_WARN_);
            }
        }
    }while(!stream.eof());

    return wxString();
}








wxArrayString TIniFile::getProfilesList() const
{
    wxArrayString profiles;
    for(TIniProfileMap::const_iterator it = this->iniProfiles.begin() ; it != this->iniProfiles.end() ; it++)
    {
        profiles.Add((*it).first);
    }
    return profiles;
}

void TIniFile::toStream(std::ostream & stream) const
{
    this->defaultProfile.toStream(stream);

    for(TIniProfileMap::const_iterator it = this->iniProfiles.begin() ; it != this->iniProfiles.end() ; it++)
    {
        stream << "\n\n[" << WX2ISO((*it).first) << "]\n";
        (*it).second.toStream(stream);
    }
}


void TIniFile::fromStream(std::istream & stream)
{
    wxString sRet;
    wxString sLabel;
    TIniProfile profile;
    int iLabelEnd;

    sRet = this->defaultProfile.fromStream(stream);
    while (!stream.eof())
    {
        iLabelEnd = sRet.Find(']');
        sLabel = sRet.Mid(1,iLabelEnd-1);
        sRet = profile.fromStream(stream);
        this->iniProfiles[sLabel] = profile;
    }
}

void TIniFile::clear()
{
    this->defaultProfile.clear();
    this->iniProfiles.erase(this->iniProfiles.begin(), this->iniProfiles.end());
}