/**
* Project: VSXu Engine: Realtime modular visual programming engine.
*
* This file is part of Vovoid VSXu Engine.
*
* @author Jonatan Wallmander, Robert Wenzel, Vovoid Media Technologies AB Copyright (C) 2003-2013
* @see The GNU Lesser General Public License (LGPL)
*
* VSXu Engine is free software; you can redistribute it and/or modify
* it under the terms of the GNU Lesser General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
* or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
* for more details.
*
* You should have received a copy of the GNU Lesser General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#ifndef VSX_MANAGER_H
#define VSX_MANAGER_H

#include <vsx_string.h>
#include <vsx_platform.h>
#include <GL/glew.h>
#include <string>
#include <vector>

#if PLATFORM_FAMILY == PLATFORM_FAMILY_UNIX
  #define VSX_MANAGER_DLLIMPORT
#else
  #ifdef VSXU_STATIC
    #define VSX_MANAGER_DLLIMPORT
  #else
    #if defined(VSX_AUDOVISUAL_DLL)
      #define VSX_MANAGER_DLLIMPORT __declspec (dllexport)
    #else
      #define VSX_MANAGER_DLLIMPORT __declspec (dllimport)
    #endif
  #endif
#endif

class vsx_manager_abs
{
public:
  // init manager with base path to where the effects (.vsx files) can be found
  // i.e. if base_path is /usr/share/vsxu/   then the engine will look in
  // /usr/share/vsxu/_visuals (linux)
  //
  // sound types can be "pulseaudio", "media_player", "fmod"
  virtual void init(const char* base_path, const char* sound_type) = 0;
  
  
  // this function takes a path where VSXu visuals can be found and 
  // adds the visuals after the visuals in the default directory in 
  // the index of visuals
  virtual void add_visual_path(const char* new_visual_path) = 0;

  // before you render first time, you need to start
  virtual void start() = 0;
  virtual void render() = 0;
  // if you are going to destroy the GL Context - as is the case
  // when destroying a window - when undocking a window or going fullscreen perhaps
  // you need top stop() the engine so all visuals can unload their OpenGL handles and
  // get ready to be started again.
  virtual void stop() = 0;

  // flipping through different visuals
  virtual void toggle_randomizer() = 0;
  virtual void set_randomizer(bool status) = 0;
  virtual bool get_randomizer_status() = 0;
  virtual void pick_visual (int selection) = 0;
  virtual void pick_random_visual() = 0;
  virtual void next_visual() = 0;
  virtual void prev_visual() = 0;
  // if not empty string vsxu is asynchronously loading a new visualization before
  // fading to it, this method can be used to display to the user "loading visual xxxxxx..."
  virtual std::string visual_loading() = 0;

  // dump the list of all loadable visuals
  virtual std::vector<std::string> get_visual_filenames() = 0;

  // **************************************************************************
  // VISUAL METADATA
  // Provides metadata for information about current running visualization
  virtual std::string get_meta_visual_filename() = 0;
  virtual std::string get_meta_visual_name() = 0;
  virtual std::string get_meta_visual_creator() = 0;
  virtual std::string get_meta_visual_company() = 0;

  // **************************************************************************
  // EFFECT / FX LEVEL
  // Amplification/FX level (more = flashier, less = less busy)
  virtual float get_fx_level() = 0;
  virtual void inc_fx_level() = 0;
  virtual void dec_fx_level() = 0;

  // **************************************************************************
  // TIME SPEED MANIPULATION
  // time speed (more = faster movements, less = slow motion)
  virtual float get_speed() = 0;
  virtual void inc_speed() = 0;
  virtual void dec_speed() = 0;

  // **************************************************************************
  // MISC OPTIONS

  // set_option_preload_all
  //   should all states be loaded on initial frame? default: false
  //   This will make vsxu stall on startup while loading all presets.
  virtual void set_option_preload_all(bool value) = 0;

  // **************************************************************************
  // SOUND INJECTION
  //
  // Send sound data to vsx_engine for visualization.
  // NOTE! Arrays MUST be 512 bytes long.
  virtual void set_sound_freq(float* data) = 0;
  virtual void set_sound_wave(float* data) = 0;


  // arbitrary engine information (statistics etc)
  // returns information about currently playing effect
  virtual int get_engine_num_modules() = 0;
};

#endif

extern "C" {
VSX_MANAGER_DLLIMPORT vsx_manager_abs* manager_factory();
VSX_MANAGER_DLLIMPORT void manager_destroy(vsx_manager_abs* manager);
}
