/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#ifndef INDII_KRIG_NESTEDMATRIX_HPP
#define INDII_KRIG_NESTEDMATRIX_HPP

#include "boost/numeric/ublas/matrix.hpp"
#include "boost/numeric/ublas/operation.hpp"

namespace indii {
/**
 * Matrix surrounded by +infs on all sides. First element of each row is
 * guaranteed to be 16-byte aligned.
 */
class nested_matrix {
public:
  /**
   * Matrix type.
   */
  typedef boost::numeric::ublas::matrix<float> matrix_type;

  /**
   * Constructor.
   *
   * @param rows Number of rows.
   * @param cols Number of columns.
   */
  nested_matrix(const int rows = 0, const int cols = 0);

  /**
   * Assignment operator.
   */
  nested_matrix& operator=(const nested_matrix& o);

  /**
   * Get element.
   */
  matrix_type::value_type& operator()(const int i, const int j);

  /**
   * Get element.
   */
  const matrix_type::value_type& operator()(const int i, const int j)
      const;

  /**
   * Number of rows.
   */
  int size1() const;

  /**
   * Number of columns.
   */
  int size2() const;

  /**
   * Resize matrix.
   */
  void resize(const int rows, const int cols, const bool preserve = false);

  /**
   * Clear matrix (set to all zero).
   */
  void clear();
  /**
   * Set to all +inf.
   */
  void inf();

  /**
   * Get matrix.
   */
  template<class M1>
  void get(M1& X) const;

  /**
   * Set matrix.
   */
  template<class M1>
  void set(const M1& X);

private:
  /**
   * Underlying matrix.
   */
  matrix_type A;

  /**
   * Offset to first element in each row.
   */
  int offset;

  /**
   * Number of rows.
   */
  int rows;

  /**
   * Number of columns.
   */
  int cols;

  /**
   * Extra number of rows at each of top and bottom.
   */
  static const int EXTRA_ROWS = 1;

  /**
   * Extra number of extra columns on left and right.
   */
  static const int EXTRA_COLS = 8;
};
}

inline indii::nested_matrix::nested_matrix(const int rows, const int cols) :
    A(rows + 2*EXTRA_ROWS, (cols + 2*EXTRA_COLS + 3)/4*4), offset(EXTRA_COLS),
    rows(rows), cols(cols) {
  /* extra offset to get 16-byte aligned rows */
  size_t ptr = reinterpret_cast<size_t>(&A(0,0));
  offset += ((ptr + 15)/16*16 - ptr)/sizeof(matrix_type::value_type);
}

inline indii::nested_matrix& indii::nested_matrix::operator=(const nested_matrix& o) {
  A = o.A;
  return *this;
}

inline indii::nested_matrix::matrix_type::value_type& indii::nested_matrix::operator()(
    const int i, const int j) {
  return A(i + EXTRA_ROWS, j + offset);
}

inline const indii::nested_matrix::matrix_type::value_type& indii::nested_matrix::operator()(
    const int i, const int j) const {
  return A(i + EXTRA_ROWS, j + offset);
}

inline int indii::nested_matrix::size1() const {
  return rows;
}

inline int indii::nested_matrix::size2() const {
  return cols;
}

inline void indii::nested_matrix::resize(const int rows, const int cols,
    const bool preserve) {
  A.resize(rows + 2*EXTRA_ROWS, (cols + 2*EXTRA_COLS + 3)/4*4, preserve);
  size_t ptr = reinterpret_cast<size_t>(&A(0,0));
  offset = EXTRA_COLS + ((ptr + 15)/16*16 - ptr)/sizeof(matrix_type::value_type);
  this->rows = rows;
  this->cols = cols;
}

inline void indii::nested_matrix::clear() {
  A.clear();
}

inline void indii::nested_matrix::inf() {
  A = boost::numeric::ublas::scalar_matrix<float>(A.size1(), A.size2(), -logf(0.0));
}

template<class M1>
inline void indii::nested_matrix::get(M1& X) const {
  X = subrange(A, EXTRA_ROWS, EXTRA_ROWS + rows, offset, offset + cols);
}

template<class M1>
inline void indii::nested_matrix::set(const M1& X) {
  subrange(A, EXTRA_ROWS, EXTRA_ROWS + rows, offset, offset + cols) = X;
}

#endif
