/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#include "SimpleKriger.hpp"

//#include <sys/time.h>

indii::SimpleKriger::SimpleKriger(ImageResource* res) : res(res),
    len(0.2f),
    outputStd(0.125f),
    s2(0.01f),
    edge(0.0f),
    L(res->getHeight(), res->getWidth()),
    pm(0) {
  updateLightness();
}

void indii::SimpleKriger::addControl(const int x, const int y, const float z) {
  namespace ublas = boost::numeric::ublas;

  const int N = getNumControls();

  v.resize(N + 1, 2, true);
  this->z.resize(N + 1, true);
  C.resize(N + 1, N + 1, true);
  LU.resize(N + 1, N + 1, false);
  pm.resize(N + 1); // don't add false, will be interpreted as size not preserve
  w.resize(N + 1, false);

  /* augment control points */
  v(N,0) = x;
  v(N,1) = y;

  /* augment outputs */
  this->z(N) = bound(-1.0f, z, 1.0f);
  
  /* augment distance map */
  D.push_back(nested_matrix());
  D.back().resize(res->getHeight(), res->getWidth(), false);
  updateDistance(N);
  updateSpread();

  /* augment covariance map */
  updateCovariance(N);

  /* augment weights */
  updateWeights();

  /* post-conditions */
  assert ((int)C.size1() == getNumControls() && (int)C.size2() == getNumControls());
  assert ((int)LU.size1() == getNumControls() && (int)LU.size2() == getNumControls());
  assert ((int)pm.size() == getNumControls());
  assert ((int)v.size1() == getNumControls());
  assert ((int)this->z.size() == getNumControls());
  assert ((int)w.size() == getNumControls());
}

void indii::SimpleKriger::removeControl(const int i) {
  /* pre-condition */
  assert (i >= 0 && i < getNumControls());

  namespace ublas = boost::numeric::ublas;

  const int N = getNumControls();

  /* remove control point */
  ublas::subrange(v, i, N - 1, 0, v.size2()) = ublas::subrange(v, i + 1, N, 0, v.size2());
  v.resize(N - 1, v.size2(), true);

  /* remove output */
  ublas::subrange(z, i, N - 1) = ublas::subrange(z, i + 1, N);
  z.resize(N - 1, true);

  /* remove distance map */
  std::list<nested_matrix>::iterator iter = D.begin();
  std::advance(iter, i);
  D.erase(iter);

  /* update the rest */
  C.resize(N - 1, N - 1, false);
  LU.resize(N - 1, N - 1, false);
  pm.resize(N - 1); // don't add false, interpreted as size not preserve
  w.resize(N - 1, false);

  updateCovariance();
  updateWeights();

  /* post-conditions */
  assert ((int)C.size1() == getNumControls() && (int)C.size2() == getNumControls());
  assert ((int)LU.size1() == getNumControls() && (int)LU.size2() == getNumControls());
  assert ((int)pm.size() == getNumControls());
  assert ((int)v.size1() == getNumControls());
  assert ((int)z.size() == getNumControls());
  assert ((int)w.size() == getNumControls());
}

void indii::SimpleKriger::updateLightness() {
  /* construct lightness channel */
  Image* img = res->get();
  L.clear();

  #pragma omp parallel
  {
    int x, y;
    float r, g, b;

    #pragma omp for
    for (y = 0; y < img->getHeight(); ++y) {
      for (x = 0; x < img->getWidth(); ++x) {
        r = img->r(y, x);
        g = img->g(y, x);
        b = img->b(y, x);

        L(y,x) = cs.rgb2l(r, g, b);
      }
    }
  }
}

void indii::SimpleKriger::updateDistance(const int i) {
  /* pre-condition */
  assert (i >= 0 && i < getNumControls());

  std::list<nested_matrix>::iterator iter = D.begin();
  std::advance(iter, i);
  float k = 0.01*edge;
  distance.map(v(i,0), v(i,1), L, *iter, k);
}

void indii::SimpleKriger::updateDistance() {
  //#pragma omp parallel for
  for (int i = 0; i < getNumControls(); ++i) {
    updateDistance(i);
  }
}

void indii::SimpleKriger::updateSpread() {
  //timeval start, end;
  //gettimeofday(&start, NULL);

  if (getNumControls() > 0) {
    int numRows = std::min(L.size1(), std::max(131072/getNumControls()/L.size2(), 1)); // number of rows from each distance map
    int skipRows = L.size1()/numRows; // number of rows to skip each time
    int n = getNumControls()*numRows*L.size2(); // total number of pixels

    d.resize(n, false);
    std::list<nested_matrix>::iterator iter;
    int i, j, k = 0, row;
    for (iter = D.begin(); iter != D.end(); iter++) {
      for (i = 0; i < numRows; ++i) {
        row = (i + 1)*skipRows - 1;
        for (j = 0; j < L.size2(); ++j) {
          d[k] = (*iter)(row, j);
          ++k;
        }
      }
    }
    assert (k == n);
    std::sort(d.begin(), d.end());

    //gettimeofday(&end, NULL);
    //long startusec = start.tv_sec*1e6 + start.tv_usec;
    //long endusec = end.tv_sec*1e6 + end.tv_usec;
    //std::cerr << "spread: " << endusec - startusec << std::endl;
  }
}

void indii::SimpleKriger::updateCovariance(const int i) {
  /* pre-condition */
  assert (i < getNumControls());

  const float scaledLen = getScaledLength();
  const float l2 = scaledLen*scaledLen;
  const float z = -0.5/l2;
  const float o2 = outputStd*outputStd;

  int j;
  float d, s;

  /* covariance between control points */
  std::list<nested_matrix>::iterator iter = D.begin();
  std::advance(iter, i);
  for (j = 0; j <= i; ++j) {
    d = (*iter)(v(j,1), v(j,0));
    s = s2*expf(z*d*d);
    if (i == j) {
      C(i,j) = s + o2;
    } else {
      C(i,j) = s;
      C(j,i) = s;
    }
  }

  /* LU factorisation of covariance */
  for (int j = 0; j < (int)pm.size(); ++j) {
    pm(j) = j;
  }
  LU = C;
  boost::numeric::ublas::lu_factorize(LU, pm);
}

void indii::SimpleKriger::updateCovariance() {
  const float scaledLen = getScaledLength();
  const float l2 = scaledLen*scaledLen;
  const float z = -0.5f/l2;
  const float o2 = outputStd*outputStd;

  const int N = getNumControls();
  int i, j;
  float d, s;

  /* covariance between control points */
  std::list<nested_matrix>::iterator iter = D.begin();
  for (i = 0; i < N; ++i, ++iter) {
    for (j = 0; j <= i; ++j) {
      d = (*iter)(v(j,1), v(j,0));
      s = s2*expf(z*d*d);
      if (i == j) {
        C(i,j) = s + o2;
      } else {
        C(i,j) = s;
        C(j,i) = s;
      }
    }
  }

  /* LU factorisation of covariance */
  for (int j = 0; j < (int)pm.size(); ++j) {
    pm(j) = j;
  }
  LU = C;
  boost::numeric::ublas::lu_factorize(LU, pm);
}

void indii::SimpleKriger::updateWeights() {
  w = z;
  try {
    boost::numeric::ublas::lu_substitute(LU, pm, w);
  } catch (int e) {
    //
  }
}
