/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#ifndef INDII_KRIG_EDGEDISTANCEMAP_HPP
#define INDII_KRIG_EDGEDISTANCEMAP_HPP

#include "nested_matrix.hpp"

#include <cmath>
#include <cassert>
#include <iostream>
#include <xmmintrin.h>

namespace indii {
/**
 * Edge distance map from control point.
 */
class EdgeDistanceMap {
public:
  /**
   * Constructor.
   */
  EdgeDistanceMap();

  /**
   * Compute map.
   *
   * @param x Control point x-ordinate.
   * @param y Control point y-ordinate.
   * @param L Lightness matrix.
   * @param[out] D Distance matrix.
   * @param k Edge threshold.
   */
  void map(const int x, const int y, nested_matrix& L, nested_matrix& D,
      const float k = 0.0f);

private:
  void forward(const int x0, const int y0, const int x1, const int y1,
      const float k, nested_matrix& L, nested_matrix& D);

  void backward(const int x0, const int y0, const int x1, const int y1,
      const float k, nested_matrix& L, nested_matrix& D);

  void forwardSSE(const int x0, const int y0, const int x1, const int y1,
      const float k, nested_matrix& L, nested_matrix& D, int* progress);

  void backwardSSE(const int x0, const int y0, const int x1, const int y1,
      const float k, nested_matrix& L, nested_matrix& D, int* progress);
  
  /**
   * Threshold given edge setting.
   */
  __m128 threshold(const __m128 x, const __m128 k);
  
  /**
   * Threshold given edge setting.
   */
  float threshold(const float x, const float k);
  
  /**
   * Use SSE implementation?
   */
  bool useSSE;
};
}

inline __m128 indii::EdgeDistanceMap::threshold(const __m128 x, const __m128 k) {
  __m128 mask = _mm_cmpge_ps(x, k);
  return _mm_and_ps(x, mask);
}

inline float indii::EdgeDistanceMap::threshold(const float x, const float k) {
  return (x < k) ? 0.0f : x;
}

#endif
