/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#ifndef INDII_ILLUM_MODEL_ILLUMMODEL_HPP
#define INDII_ILLUM_MODEL_ILLUMMODEL_HPP

#include "IllumObservable.hpp"
#include "../../krig/SimpleKriger.hpp"
#include "../../model/Model.hpp"
#include "../../image/ImageResource.hpp"
#include "../../image/ImageManipulation.hpp"

#include "wx/colour.h"

#include <vector>

namespace indii {

  class IllumObserver;
  
/**
 * Kriging model.
 */
class IllumModel : public Model, public IllumObservable {
  friend class IllumController;
public:
  /**
   * Constructor.
   *
   * @param res Image resource.
   */
  IllumModel(ImageResource* res);

  /**
   * Destructor.
   */
  virtual ~IllumModel();

  /**
   * Get length scale.
   */
  float getLength() const;

  /**
   * Set length scale.
   */
  void setLength(const float len);

  /**
   * Get output standard deviation.
   */
  float getOutputStd() const;

  /**
   * Set output standard deviation.
   */
  void setOutputStd(const float outputStd);

  /**
   * Get blacks threshold.
   */
  float getBlacks() const;

  /**
   * Set blacks threshold.
   */
  void setBlacks(const float blacks);

  /**
   * Get whites threshold.
   */
  float getWhites() const;

  /**
   * Set whites threshold.
   */
  void setWhites(const float whites);

  /**
   * Get overall brightness.
   */
  float getBrightness() const;

  /**
   * Set overall brightness.
   */
  void setBrightness(const float brightness);

  /**
   * Get edge sensitivity.
   */
  float getEdge() const;
  
  /**
   * Set edge softness.
   */
  void setEdge(const float edge);

  /**
   * Get number of control points.
   *
   * @return Number of control points.
   */
  int getNumControls();
  
  /**
   * Get control point.
   *
   * @param i Index of control point.
   * @param[out] x X-coordinate.
   * @param[out] y Y-coordinate.
   * @param[out] output Output.
   */
  void getControl(const int i, int* x, int* y, float* output);
  
  /**
   * Add control point.
   *
   * @param x X-coordinate.
   * @param y Y-coordinate.
   * @param output Output.
   */
  void addControl(const int x, const int y, const float output = 0.0f);
  
  virtual void calcFg(const wxRect& rect, const float scale, Image& o);
  virtual void calcAn(const wxRect& rect, const float scale, wxDC& dc);
  virtual void setDefaults();
  virtual void setForDialog();

private:
  /**
   * Kriger.
   */
  SimpleKriger kriger;

  /**
   * Blacks threshold.
   */
  float blacks;

  /**
   * Whites threshold.
   */
  float whites;

  /**
   * Overall brightness.
   */
  float brightness;
  
  /**
   * Edge sensitivity.
   */
  float edge;
};
}

inline float indii::IllumModel::getLength() const {
  return kriger.getLength();
}

inline void indii::IllumModel::setLength(const float len) {
  kriger.setLength(len);
  notifyLength();
  notifyAll();
}

inline float indii::IllumModel::getOutputStd() const {
  return kriger.getOutputStd();
}

inline void indii::IllumModel::setOutputStd(const float outputStd) {
  kriger.setOutputStd(outputStd);
  notifyOutputStd();
  notifyAll();
}

inline float indii::IllumModel::getBlacks() const {
  return blacks;
}

inline void indii::IllumModel::setBlacks(const float blacks) {
  /* pre-condition */
  assert (blacks >= 0.0f && blacks <= 1.0f);

  this->blacks = blacks;
  notifyBlacks();
  notifyAll();
}

inline float indii::IllumModel::getWhites() const {
  return whites;
}

inline void indii::IllumModel::setWhites(const float whites) {
  /* pre-condition */
  assert (whites >= 0.0f && whites <= 1.0f);

  this->whites = whites;
  notifyWhites();
  notifyAll();
}

inline float indii::IllumModel::getBrightness() const {
  return brightness;
}

inline void indii::IllumModel::setBrightness(const float brightness) {
  /* pre-condition */
  assert (brightness >= -1.0f && brightness <= 1.0f);

  this->brightness = brightness;
  notifyBrightness();
  notifyAll();
}

inline float indii::IllumModel::getEdge() const {
  return kriger.getEdge();
}

inline void indii::IllumModel::setEdge(const float edge) {
  kriger.setEdge(edge);
  notifyEdge();
  notifyAll();
}

inline int indii::IllumModel::getNumControls() {
  return kriger.getNumControls();
}

inline void indii::IllumModel::getControl(const int i, int* x, int* y,
    float* output) {
  kriger.getControl(i, x, y);
  *output = kriger.getOutput(i);
}

inline void indii::IllumModel::addControl(const int x, const int y,
    const float output) {
  kriger.addControl(x, y, output);
}

#endif
