/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#include "IllumModel.hpp"

#include "IllumObserver.hpp"
#include "../../image/ColourSpace.hpp"
#include "../../gui/misc.hpp"

#include "wx/graphics.h"

#include <cassert>

namespace ublas = boost::numeric::ublas;

using namespace indii;

IllumModel::IllumModel(ImageResource* res) :
    Model(res), kriger(res) {
  setDefaults();
}

IllumModel::~IllumModel() {
  //
}

void IllumModel::setDefaults() {
  setBlacks(0.02f);
  setWhites(0.98f);
  setBrightness(0.0f);
  setChannelMix(0.299f, 0.587f, 0.114f);
  kriger.setLength(0.2f);
  kriger.setOutputStd(0.125f);
  kriger.setEdge(0.0f);
  notifyLength();
  notifyOutputStd();
  notifyBlacks();
  notifyWhites();
  notifyBrightness();
  notifyEdge();
  notifyAll();
}

void IllumModel::setForDialog() {
  notifyAll();
}

void IllumModel::calcFg(const wxRect& rect, const float scale, Image& o) {
  /* pre-conditions */
  assert(o.getWidth() >= rect.width);
  assert(o.getHeight() >= rect.height);

  Image& img = *res->get(scale);
  ublas::vector<float> l(rect.height * rect.width);
  kriger.query(rect, scale, l);

#pragma omp parallel
  {
    ColourSpace::rgb_t rgb(3);
    ColourSpace::hsl_t hsl(3);

    int x, y;
    float a;

#pragma omp for
    for (y = 0; y < rect.height; ++y) {
      for (x = 0; x < rect.width; ++x) {
        rgb(0) = img.r(rect.y + y, rect.x + x);
        rgb(1) = img.g(rect.y + y, rect.x + x);
        rgb(2) = img.b(rect.y + y, rect.x + x);
        cs.rgb2hsl(rgb, hsl);

        a = l(y * rect.width + x) + brightness;

        if (hsl(2) < blacks) { // blacks
          //hsl(1) *= hsl(2)/blacks;
          hsl(2) *= hsl(2) / blacks;
        }

        if (a >= 1.0f) {
          a = 1.0f;
          hsl(2) = 1.0f;
        } else if (a <= -1.0f) {
          a = -1.0f;
          hsl(2) = 0.0f;
        } else if (a >= 0.0) {
          hsl(2) /= 1.0f - a;
        } else {
          hsl(2) *= 1.0f + a;
        }
        hsl(2) = bound(0.0f, hsl(2), 1.0f);

        if (a > whites) {
          hsl(1) *= 1.0f - (a - whites) / (1.0f - whites);
        }
        hsl(1) = bound(0.0f, hsl(1), 1.0f);

        cs.hsl2rgb(hsl, rgb);
        o.r(y, x) = rgb(0);
        o.g(y, x) = rgb(1);
        o.b(y, x) = rgb(2);
      }
    }
  }
}

void IllumModel::calcAn(const wxRect& rect, const float scale, wxDC& dc) {
  int i, x, y;

  wxCoord x1, y1, x2, y2;
  wxCoord w, h, descent, externalLeading;

  dc.SetPen(*wxWHITE);
  dc.SetBrush(wxColour(0, 0, 0, 180));
  dc.SetTextForeground(*wxWHITE);
  dc.SetTextBackground(wxColour(0, 0, 0, 180));
  dc.SetFont(*wxNORMAL_FONT);

  for (i = 0; i < (int)kriger.getNumControls(); ++i) {
    kriger.getControl(i, &x, &y);
    x1 = x / scale;
    y1 = y / scale;

    if (x1 >= rect.x && x1 < rect.x + rect.width && y1 >= rect.y
        && y1 < rect.y + rect.height) {
      wxString label = wxString::Format(_("%d"), i + 1);
      dc.GetTextExtent(label, &w, &h, &descent, &externalLeading);

      x2 = intround(x1 - 0.5f * w);
      y2 = intround(y1 - 0.5f * h);

      dc.DrawCircle(x1, y1, 12);
      dc.DrawText(label, x2, y2);
    }
  }
}
