/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#include "IllumController.hpp"

using namespace indii;

IllumController::IllumController(IllumModel* model) : model(model),
    control(-1), lastDouble(false) {
  //
}

void IllumController::down(const int x0, const int y0, const int x, const int y,
    wxMouseEvent& evt) {
  /* pre-condition */
  assert (control == -1);

  int i, x1, y1;
  float d, zoom = std::min(1.0f, x/static_cast<float>(x0));
  bool handled = false;

  /* loop backwards, as control points a drawn in a forward loop, so later
   * ones overlay earlier ones and should get preference for this */
  for (i = (int)model->kriger.getNumControls() - 1; i >= 0; --i) {
    model->kriger.getControl(i, &x1, &y1);
    d = sqrtf(powf(x1 - x0, 2) + powf(y1 - y0, 2));
    if (d < 12.0/zoom) {
      break;
    }
  }

  if (i >= 0) {
    control = i;
    handled = true;
  }
  lastDouble = false;

  if (!handled) {
    evt.Skip();
  }
}

void IllumController::doubleDown(const int x0, const int y0, const int x,
    const int y, wxMouseEvent& evt) {
  /* pre-condition */
  assert (control == -1);

  int i, x1, y1;
  float d, zoom = std::min(1.0f, x/static_cast<float>(x0));

  /* loop backwards, as control points a drawn in a forward loop, so later
   * ones overlay earlier ones and should get preference for this */
  for (i = model->kriger.getNumControls() - 1; i >= 0; --i) {
    model->kriger.getControl(i, &x1, &y1);
    d = sqrtf(powf(x1 - x0, 2) + powf(y1 - y0, 2));
    if (d < 12.0/zoom) {
      control = i;
      break;
    }
  }

  if (i < 0 && x0 >= 0 && x0 < (int)model->res->getWidth() &&
      y0 >= 0 && y0 < (int)model->res->getHeight()) {
    model->kriger.addControl(x0, y0, model->kriger.query(x0, y0));
    control = model->kriger.getNumControls() - 1;
    model->notifyAll();
  }
  lastDouble = true;
}

void IllumController::up(const int x0, const int y0, const int x, const int y,
    wxMouseEvent& evt) {
  if (control != -1 && !lastDouble && overlay && (
      x0 < 0 || x0 >= (int)model->res->getWidth() ||
      y0 < 0 || y0 >= (int)model->res->getHeight())) {
    /* delete point */
    model->kriger.removeControl(control);
  }

  control = -1;
  lastDouble = false;
  model->notifyAll();
}

void IllumController::drag(const int x0, const int y0, const int x,
    const int y, const int dx0, const int dy0, const int dx, const int dy,
    wxMouseEvent& evt) {
  if (control != -1) {
    if (lastDouble) {
      model->kriger.setOutput(control, model->kriger.getOutput(control) - dy/128.0f);
      model->notifyAll();
    } else if (overlay) {
      int x1, y1;

      //model->kriger.getControl(control, &x1, &y1);
      //x1 += dx0;
      //y1 += dy0;;
      //x1 = std::max(0, std::min(x1, (int)model->res->getWidth() - 1));
      //y1 = std::max(0, std::min(y1, (int)model->res->getHeight() - 1));

      x1 = std::max(0, std::min(x0, (int)model->res->getWidth() - 1));
      y1 = std::max(0, std::min(y0, (int)model->res->getHeight() - 1));

      model->kriger.setControl(control, x1, y1);
      model->notifyAll();
    }
  } else {
    evt.Skip();
  }
}
