/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#include "IllumWindow.hpp"

#include "IllumDocument.hpp"
#include "../../gui/Window.hpp"
#include "../../reg/Components.hpp"
#include "../../image/FileImageResource.hpp"
#include "../../../gnome/gnome.hpp"

using namespace indii;

IllumWindow::IllumWindow(wxWindow* parent, const Mode mode,
    WindowObserver* obs) : Window(parent, mode, obs) {
  init();
  enable();
  if (mode == STANDALONE) {
    infoBar->showStartMessage();
  }
}

IllumWindow::IllumWindow(wxWindow* parent, ImageResource* res,
    IllumModel* model, const Mode mode, WindowObserver* obs) :
    Window(parent, mode, obs) {
  if (mode == STANDALONE) {
    IllumDocument* doc = new IllumDocument(notebook, res, model);
    init(doc);
  } else {
    this->doc = new IllumDocument(main, res, model);
    init(this->doc);
  }
  setModel(model);
  enable();
  infoBar->showInfoMessage(_("Double-click and hold, then drag up or down to adjust the brightness of the image"), MSG_MOUSE_INTERFACE);
}

IllumWindow::IllumWindow(wxWindow* parent, const wxString& path,
    const Mode mode, WindowObserver* obs) :
    Window(parent, mode, obs) {
  Document* doc = open(path);
  init(doc);
  setModel(doc->getModel());
  enable();
  infoBar->showInfoMessage(_("Double-click and hold, then drag up or down to adjust the brightness of the image"), MSG_MOUSE_INTERFACE);
}

Document* IllumWindow::open(const wxString& file) {
  infoBar->showInfoMessage(_("Double-click and hold, then drag up or down to adjust the brightness of the image"), MSG_MOUSE_INTERFACE);
  if (mode == STANDALONE) {
    return new IllumDocument(notebook, file);
  } else {
    return new IllumDocument(main, file);
  }
}

void IllumWindow::layoutDefault() {
  wxAuiPaneInfo& globalPaneInfo = manager->GetPane(globalPane);
  globalPane->Fit();
  globalPaneInfo.FloatingSize(globalPane->GetMinSize().GetWidth(), globalPane->GetMinSize().GetHeight() + EXTRA_PANE_HEIGHT);
  globalPaneInfo.Position(1);
  globalPaneInfo.Right();
  globalPaneInfo.Dock();
  globalPaneInfo.Show();

  wxAuiPaneInfo& mixerPaneInfo = manager->GetPane(mixerPane);
  mixerPane->Fit();
  mixerPaneInfo.FloatingSize(mixerPane->GetMinSize().GetWidth(), mixerPane->GetMinSize().GetHeight() + EXTRA_PANE_HEIGHT);
  mixerPaneInfo.Position(2);
  mixerPaneInfo.Right();
  mixerPaneInfo.Dock();
  mixerPaneInfo.Show();

  manager->Update();
}

void IllumWindow::initPanes() {
  int width, height;
  wxAuiPaneInfo paneInfo;
  paneInfo.DefaultPane();
  paneInfo.CloseButton(true);
  //paneInfo.PinButton(true);
  paneInfo.PaneBorder(true);
  paneInfo.CaptionVisible(true);

  /* global pane */
  globalPane = new IllumGlobalPane(main);
  width = globalPane->GetMinSize().GetWidth();
  height = globalPane->GetMinSize().GetHeight();

  #ifndef WX_28
  paneInfo.Name(_("Light"));
  #else
  paneInfo.Name(wxString("Light", wxConvUTF8));
  #endif
  paneInfo.MinSize(wxSize(width, height));
  paneInfo.BestSize(wxSize(width, height));
  paneInfo.FloatingSize(width + EXTRA_PANE_WIDTH, height + EXTRA_PANE_HEIGHT);
  paneInfo.Caption(_("Light settings"));
  paneInfo.Position(1);
  paneInfo.Right();
  manager->AddPane(globalPane, paneInfo);

  /* control point pane */
  mixerPane = new ChannelMixerPane(main);
  width = mixerPane->GetMinSize().GetWidth();
  height = mixerPane->GetMinSize().GetHeight();

  #ifndef WX_28
  paneInfo.Name(_("Mixer"));
  #else
  paneInfo.Name(wxString("Mixer", wxConvUTF8));
  #endif
  paneInfo.MinSize(wxSize(width, height));
  paneInfo.BestSize(wxSize(width, height));
  paneInfo.FloatingSize(width + EXTRA_PANE_WIDTH, height + EXTRA_PANE_HEIGHT);
  paneInfo.Caption(_("Channel mixer"));
  paneInfo.Position(2);
  paneInfo.Right();
  manager->AddPane(mixerPane, paneInfo);

  /* add panes to window menu */
  addPane(_("Light settings pane"));
  addPane(_("Channel mixer pane"));
}

void IllumWindow::enable(const bool enable) {
  Window::enable(enable);
  if (!enable) {
    globalPane->setModel(NULL);
    mixerPane->setModel(NULL);
  }
}

void IllumWindow::setModel(Model* model) {
  IllumModel* cmodel = dynamic_cast<IllumModel*>(model);

  globalPane->setModel(cmodel);
  mixerPane->setModel(cmodel);
}

bool IllumWindow::isShown(const int i) {
  /* pre-condition */
  assert (i < 2);

  switch (i) {
  case 0:
    return manager->GetPane(globalPane).IsShown();
  case 1:
    return manager->GetPane(mixerPane).IsShown();
  default:
    return false;
  }
}

void IllumWindow::show(const int i, const bool show) {
  /* pre-condition */
  assert (i < 2);

  switch (i) {
  case 0:
    manager->GetPane(globalPane).Show(show);
    break;
  case 1:
    manager->GetPane(mixerPane).Show(show);
    break;
  }
  manager->Update();
}

BEGIN_EVENT_TABLE(IllumWindow, Window)
END_EVENT_TABLE()
