/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#include "IllumGlobalPane.hpp"

#include "../../gui/Help.hpp"

#include "wx/wx.h"
#include "wx/gbsizer.h"

#include <cmath>

#define SLIDER_RES 4096.0f

using namespace indii;

IllumGlobalPane::IllumGlobalPane(wxWindow* parent) :
    Pane(parent), model(NULL) {
  static const int PADDING = 4;

  wxGridBagSizer* sizer = new wxGridBagSizer();

  slideLength = new wxSlider(this, ID_SLIDE_LENGTH, 1, 1, SLIDER_RES,
      wxDefaultPosition, wxSize(200,-1));
  //slideEdge = new wxSlider(this, ID_SLIDE_EDGE, 0, 0, SLIDER_RES,
  //    wxDefaultPosition, wxSize(120,-1));
  slideOutputStd = new wxSlider(this, ID_SLIDE_OUTPUT_STD, 1, 1, SLIDER_RES,
      wxDefaultPosition, wxSize(200,-1));
  slideBlacks = new wxSlider(this, ID_SLIDE_BLACKS, 0, 0, SLIDER_RES,
      wxDefaultPosition, wxSize(200,-1));
  slideWhites = new wxSlider(this, ID_SLIDE_WHITES, 0, 0, SLIDER_RES,
      wxDefaultPosition, wxSize(200,-1));
  slideBrightness = new wxSlider(this, ID_SLIDE_BRIGHTNESS, 0, -SLIDER_RES, SLIDER_RES,
      wxDefaultPosition, wxSize(200,-1));

  wxStaticText* labelLength = new wxStaticText(this, -1, _("Spread"));
  //wxStaticText* labelEdge = new wxStaticText(this, -1, _("Tolerance"));
  wxStaticText* labelOutputStd = new wxStaticText(this, -1, _("Smoothness"));
  wxStaticText* labelBlacks = new wxStaticText(this, -1, _("Blacks"));
  wxStaticText* labelWhites = new wxStaticText(this, -1, _("Whites"));
  wxStaticText* labelBrightness = new wxStaticText(this, -1, _("Brightness"));

  sizer->Add(labelLength, wxGBPosition(0,0), wxDefaultSpan,
      wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(slideLength, wxGBPosition(1,0), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxLEFT|wxRIGHT, PADDING);
  //sizer->Add(labelEdge, wxGBPosition(2,0), wxDefaultSpan,
  //    wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  //sizer->Add(slideEdge, wxGBPosition(3,0), wxDefaultSpan,
  //    wxALIGN_LEFT|wxEXPAND|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(labelOutputStd, wxGBPosition(2,0), wxDefaultSpan,
      wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(slideOutputStd, wxGBPosition(3,0), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(labelBlacks, wxGBPosition(4,0), wxDefaultSpan,
      wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(slideBlacks, wxGBPosition(5,0), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(labelWhites, wxGBPosition(6,0), wxDefaultSpan,
      wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(slideWhites, wxGBPosition(7,0), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(labelBrightness, wxGBPosition(8,0), wxDefaultSpan,
      wxALIGN_LEFT|wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(slideBrightness, wxGBPosition(9,0), wxDefaultSpan,
      wxALIGN_LEFT|wxEXPAND|wxLEFT|wxRIGHT, PADDING);

  sizer->SetFlexibleDirection(wxHORIZONTAL);
  sizer->AddGrowableCol(0);
  SetSizerAndFit(sizer);
  setModel(NULL);
}

void IllumGlobalPane::setModel(IllumModel* model) {
  this->model = model;
  ignore();

  Enable(model != NULL);

  /* must explicitly disable all controls, not just panel, on Mac */
  slideLength->Enable(model != NULL);
  //slideEdge->Enable(model != NULL);
  slideOutputStd->Enable(model != NULL);
  slideBlacks->Enable(model != NULL);
  slideWhites->Enable(model != NULL);
  slideBrightness->Enable(model != NULL);

  if (model != NULL) {
    watch(model);
    notifyLength();
    //notifyEdge();
    notifyOutputStd();
    notifyBlacks();
    notifyWhites();
    notifyBrightness();
  }
}

void IllumGlobalPane::notifyLength() {
  /* pre-condition */
  assert (model != NULL);

  int val = intround(SLIDER_RES*model->getLength());
  if (slideLength->GetValue() != val) {
    slideLength->SetValue(val);
  }
}

void IllumGlobalPane::notifyEdge() {
  /* pre-condition */
  assert (model != NULL);
  
  //int val = intround(SLIDER_RES*model->getEdge());
  //if (slideEdge->GetValue() != val) {
  //  slideEdge->SetValue(val);
  //}
}

void IllumGlobalPane::notifyOutputStd() {
  /* pre-condition */
  assert (model != NULL);

  int val = intround(SLIDER_RES*model->getOutputStd());
  if (slideOutputStd->GetValue() != val) {
    slideOutputStd->SetValue(val);
  }
}

void IllumGlobalPane::notifyBlacks() {
  /* pre-condition */
  assert (model != NULL);

  int val = intround(SLIDER_RES*model->getBlacks());
  if (slideBlacks->GetValue() != val) {
    slideBlacks->SetValue(val);
  }
}

void IllumGlobalPane::notifyWhites() {
  /* pre-condition */
  assert (model != NULL);

  int val = intround(SLIDER_RES*model->getWhites());
  if (slideWhites->GetValue() != val) {
    slideWhites->SetValue(val);
  }
}

void IllumGlobalPane::notifyBrightness() {
  /* pre-condition */
  assert (model != NULL);

  int val = intround(SLIDER_RES*model->getBrightness());
  if (slideBrightness->GetValue() != val) {
    slideBrightness->SetValue(val);
  }
}

void IllumGlobalPane::OnSlideLength(wxScrollEvent& evt) {
  if (model != NULL) {
    model->setLength(slideLength->GetValue()/SLIDER_RES);
  }
}

void IllumGlobalPane::OnSlideEdge(wxScrollEvent& evt) {
  //if (model != NULL) {
  //  model->setEdge(slideEdge->GetValue()/SLIDER_RES);
  //}
}

void IllumGlobalPane::OnSlideOutputStd(wxScrollEvent& evt) {
  if (model != NULL) {
    model->setOutputStd(slideOutputStd->GetValue()/SLIDER_RES);
  }
}

void IllumGlobalPane::OnSlideBlacks(wxScrollEvent& evt) {
  if (model != NULL) {
    model->setBlacks(slideBlacks->GetValue()/SLIDER_RES);
  }
}

void IllumGlobalPane::OnSlideWhites(wxScrollEvent& evt) {
  if (model != NULL) {
    model->setWhites(slideWhites->GetValue()/SLIDER_RES);
  }
}

void IllumGlobalPane::OnSlideBrightness(wxScrollEvent& evt) {
  if (model != NULL) {
    model->setBrightness(slideBrightness->GetValue()/SLIDER_RES);
  }
}

void IllumGlobalPane::OnHelp(wxHelpEvent& evt) {
  Help::show(530);
}

BEGIN_EVENT_TABLE(IllumGlobalPane, Pane)
EVT_COMMAND_SCROLL(ID_SLIDE_LENGTH, IllumGlobalPane::OnSlideLength)

/* not responsive enough for thumbtrack, so disable that */
//EVT_COMMAND_SCROLL_TOP(ID_SLIDE_EDGE, IllumGlobalPane::OnSlideEdge)
//EVT_COMMAND_SCROLL_BOTTOM(ID_SLIDE_EDGE, IllumGlobalPane::OnSlideEdge)
//EVT_COMMAND_SCROLL_LINEUP(ID_SLIDE_EDGE, IllumGlobalPane::OnSlideEdge)
//EVT_COMMAND_SCROLL_LINEDOWN(ID_SLIDE_EDGE, IllumGlobalPane::OnSlideEdge)
//EVT_COMMAND_SCROLL_PAGEUP(ID_SLIDE_EDGE, IllumGlobalPane::OnSlideEdge)
//EVT_COMMAND_SCROLL_PAGEDOWN(ID_SLIDE_EDGE, IllumGlobalPane::OnSlideEdge)
//EVT_COMMAND_SCROLL_THUMBRELEASE(ID_SLIDE_EDGE, IllumGlobalPane::OnSlideEdge)
//EVT_COMMAND_SCROLL_THUMBTRACK(ID_SLIDE_EDGE, IllumGlobalPane::OnSlideEdge) // too slow

EVT_COMMAND_SCROLL(ID_SLIDE_OUTPUT_STD, IllumGlobalPane::OnSlideOutputStd)
EVT_COMMAND_SCROLL(ID_SLIDE_BLACKS, IllumGlobalPane::OnSlideBlacks)
EVT_COMMAND_SCROLL(ID_SLIDE_WHITES, IllumGlobalPane::OnSlideWhites)
EVT_COMMAND_SCROLL(ID_SLIDE_BRIGHTNESS, IllumGlobalPane::OnSlideBrightness)
EVT_HELP(wxID_ANY, IllumGlobalPane::OnHelp)
END_EVENT_TABLE()
