// snd_dma.c -- main control for any streaming sound output device

#include "quakedef.h"

void S_Play(void);
void S_PlayVol(void);
void S_SoundList(void);
void S_Update_();
void S_StopAllSounds(void);

// =======================================================================
// Internal sound data & structures
// =======================================================================

channel_t channels[MAX_CHANNELS];
int total_channels;

volatile dma_t *shm = 0;

vec3_t listener_origin;
vec3_t listener_forward;
vec3_t listener_right;
vec3_t listener_up;
vec_t sound_nominal_clip_dist = 1000.0;

int soundtime;			// sample PAIRS
int paintedtime;		// sample PAIRS


#define	MAX_SFX		512
sfx_t *known_sfx;		// hunk allocated [MAX_SFX]
int num_sfx;

sfx_t *ambient_sfx[NUM_AMBIENTS];

int desired_speed = 11025;
int desired_bits = 16;

int sound_started = 0;

cvar_t bgmvolume = { "bgmvolume", "1", true };
cvar_t volume = { "volume", "0.7", true };

cvar_t nosound = { "nosound", "0" };
cvar_t precache = { "precache", "1" };
cvar_t loadas8bit = { "loadas8bit", "0" };
cvar_t bgmbuffer = { "bgmbuffer", "4096" };
cvar_t ambient_level = { "ambient_level", "0.3" };
cvar_t ambient_fade = { "ambient_fade", "100" };
cvar_t snd_noextraupdate = { "snd_noextraupdate", "0" };
cvar_t snd_show = { "snd_show", "0" };


// ====================================================================
// User-setable variables
// ====================================================================


//
// Fake dma is a synchronous faking of the DMA progress used for
// isolating performance in the renderer.  The fakedma_updates is
// number of times S_Update() is called per second.
//

#ifdef __FreeBSD__
qboolean fakedma = true;
#else
qboolean fakedma = false;
#endif
int fakedma_updates = 15;

void S_SoundInfo_f(void)
{
    if (!sound_started || !shm) {
	Con_Printf("sound system not started\n");
	return;
    }

    Con_Printf("%5d stereo\n", shm->channels - 1);
    Con_Printf("%5d samples\n", shm->samples);
    Con_Printf("%5d samplepos\n", shm->samplepos);
    Con_Printf("%5d samplebits\n", shm->samplebits);
    Con_Printf("%5d submission_chunk\n", shm->submission_chunk);
    Con_Printf("%5d speed\n", shm->speed);
    Con_Printf("0x%x dma buffer\n", shm->buffer);
    Con_Printf("%5d total_channels\n", total_channels);
}


void S_NoiseTrack_f(void)
{
    S_StartNoiseTrack(Cmd_Argv(1));
}


/*
================
S_Init
================
*/
void S_Init(void)
{
    int rc;

    Con_Printf("\nSound Initialization\n");

    if (COM_CheckParm("-nosound"))
	return;

    if (COM_CheckParm("-simsound"))
	fakedma = true;

    Cmd_AddCommand("play", S_Play);
    Cmd_AddCommand("playvol", S_PlayVol);
    Cmd_AddCommand("stopsound", S_StopAllSounds);
    Cmd_AddCommand("soundlist", S_SoundList);
    Cmd_AddCommand("soundinfo", S_SoundInfo_f);
    Cmd_AddCommand("noisetrack", S_NoiseTrack_f);

    Cvar_RegisterVariable(&nosound);
    Cvar_RegisterVariable(&volume);
    Cvar_RegisterVariable(&precache);
    Cvar_RegisterVariable(&loadas8bit);
    Cvar_RegisterVariable(&bgmvolume);
    Cvar_RegisterVariable(&bgmbuffer);
    Cvar_RegisterVariable(&ambient_level);
    Cvar_RegisterVariable(&ambient_fade);
    Cvar_RegisterVariable(&snd_noextraupdate);
    Cvar_RegisterVariable(&snd_show);

    if (host_parms.memsize < 0x800000) {
	Cvar_Set("loadas8bit", "1");
	Con_Printf("loading all sounds as 8bit\n");
    }

    if (!fakedma) {
	rc = SNDDMA_Init();
	if (!rc) {
	    Con_Printf("S_Init: SNDDMA_Init failed.\n");
	    return;
	}
    }

    SND_InitScaletable();

    known_sfx = Hunk_AllocName(MAX_SFX * sizeof(sfx_t), "sfx_t");
    num_sfx = 0;

// create a piece of DMA memory

    if (fakedma) {
	shm = (void *) Hunk_AllocName(sizeof(*shm), "shm");
	shm->splitbuffer = 0;
	shm->samplebits = 16;
	shm->speed = 22050;
	shm->channels = 2;
	shm->samples = 32768;
	shm->samplepos = 0;
	shm->soundalive = true;
	shm->gamealive = true;
	shm->submission_chunk = 1;
	shm->buffer = Hunk_AllocName(1 << 16, "shmbuf");
    }

    Con_Printf("Sound sampling rate: %i\n", shm->speed);

// provides a tick sound until washed clean

    shm->buffer[4] = shm->buffer[5] = 0x7f;

    sound_started = 1;

    ambient_sfx[AMBIENT_WATER] = S_PrecacheSound("ambience/water1.wav");
    ambient_sfx[AMBIENT_SKY] = S_PrecacheSound("ambience/wind2.wav");

    S_StopAllSounds();
}

// =======================================================================
// Shutdown sound engine
// =======================================================================

void S_Shutdown(void)
{
    if (!sound_started)
	return;

    if (shm)
	shm->gamealive = 0;

    if (!fakedma)
	SNDDMA_Shutdown();

    shm = 0;
    sound_started = 0;
}


// =======================================================================
// Load a sound
// =======================================================================

/*
==================
S_FindName

==================
*/
sfx_t *S_FindName(char *name)
{
    int i;
    sfx_t *sfx;

    if (!name)
	Sys_Error("S_FindName: NULL\n");

    if (Q_strlen(name) >= MAX_QPATH)
	Sys_Error("Sound name too long: %s", name);

// see if already loaded
    for (i = 0; i < num_sfx; i++)
	if (!Q_strcmp(known_sfx[i].name, name)) {
	    return &known_sfx[i];
	}

    if (num_sfx == MAX_SFX)
	Sys_Error("S_FindName: out of sfx_t");

    sfx = &known_sfx[i];
    strcpy(sfx->name, name);

    num_sfx++;

    return sfx;
}


/*
==================
S_TouchSound

==================
*/
void S_TouchSound(char *name)
{
    sfx_t *sfx;

    if (!sound_started)
	return;

    sfx = S_FindName(name);
    Cache_Check(&sfx->cache);
}

/*
==================
S_PrecacheSound

==================
*/
sfx_t *S_PrecacheSound(char *name)
{
    sfx_t *sfx;

    if (!sound_started || nosound.value)
	return NULL;

    sfx = S_FindName(name);

// cache it in
    if (precache.value)
	S_LoadSound(sfx);

    return sfx;
}


//=============================================================================

/*
=================
SND_PickChannel
=================
*/
channel_t *SND_PickChannel(int entnum, int entchannel)
{
    int ch_idx;
    int first_to_die;
    int life_left;

// Check for replacement sound, or find the best one to replace
    first_to_die = -1;
    life_left = 0x7fffffff;
    for (ch_idx = NUM_AMBIENTS;
	 ch_idx < NUM_AMBIENTS + MAX_DYNAMIC_CHANNELS; ch_idx++) {
	if (entchannel != 0	// channel 0 never overrides
	    && channels[ch_idx].entnum == entnum && (channels[ch_idx].entchannel == entchannel || entchannel == -1)) {	// allways override sound from same entity
	    first_to_die = ch_idx;
	    break;
	}
	// don't let monster sounds override player sounds
	if (channels[ch_idx].entnum == cl.viewentity
	    && entnum != cl.viewentity && channels[ch_idx].sfx)
	    continue;

	if (channels[ch_idx].end - paintedtime < life_left) {
	    life_left = channels[ch_idx].end - paintedtime;
	    first_to_die = ch_idx;
	}
    }

    if (first_to_die == -1)
	return NULL;

    if (channels[first_to_die].sfx)
	channels[first_to_die].sfx = NULL;

    return &channels[first_to_die];
}

/*
=================
SND_Spatialize
=================
*/
void SND_Spatialize(channel_t * ch)
{
    vec_t dot;
    vec_t ldist, rdist, dist;
    vec_t scale;
    vec3_t source_vec;
    sfx_t *snd;

// anything coming from the view entity will allways be full volume
    if (ch->entnum == cl.viewentity) {
	ch->leftvol = ch->master_vol;
	ch->rightvol = ch->master_vol;
	return;
    }
// calculate stereo seperation and distance attenuation

    snd = ch->sfx;
    VectorSubtract(ch->origin, listener_origin, source_vec);

    dist = VectorNormalize(source_vec) * ch->dist_mult;

    dot = DotProduct(listener_right, source_vec);
    rdist = dist - .2 - .3 * dot;
    ldist = dist - .2 + .3 * dot;

    if (ldist < 0)
	ldist = 0;

    if (rdist < 0)
	rdist = 0;

    if (shm->channels == 1) {
	if (rdist > ldist)
	    ldist = rdist;
	else
	    rdist = ldist;
    }
// add in distance effect
    scale = 1.0 - rdist;
    ch->rightvol = (int) (ch->master_vol * scale);
    if (ch->rightvol < 0)
	ch->rightvol = 0;

    scale = 1.0 - ldist;
    ch->leftvol = (int) (ch->master_vol * scale);
    if (ch->leftvol < 0)
	ch->leftvol = 0;
}


// =======================================================================
// Start a sound effect
// =======================================================================

void S_StartSound(int entnum, int entchannel, sfx_t * sfx, vec3_t origin,
		  float fvol, float attenuation)
{
    channel_t *target_chan, *check;
    sfxcache_t *sc;
    int vol;
    int ch_idx;
    int skip;

    if (!sound_started)
	return;

    if (!sfx)
	return;

    if (nosound.value)
	return;

    vol = fvol * 255;

// pick a channel to play on
    target_chan = SND_PickChannel(entnum, entchannel);
    if (!target_chan)
	return;

// spatialize
    memset(target_chan, 0, sizeof(*target_chan));
    VectorCopy(origin, target_chan->origin);
    target_chan->dist_mult = attenuation / sound_nominal_clip_dist;
    target_chan->master_vol = vol;
    target_chan->entnum = entnum;
    target_chan->entchannel = entchannel;
    SND_Spatialize(target_chan);

    if (!target_chan->leftvol && !target_chan->rightvol)
	return;			// not audible at all

// new channel
    sc = S_LoadSound(sfx);
    if (!sc) {
	target_chan->sfx = NULL;
	return;			// couldn't load the sound's data
    }

    target_chan->sfx = sfx;
    target_chan->pos = 0.0;
    target_chan->end = paintedtime + sc->length;

// if an identical sound has also been started this frame, offset the pos
// a bit to keep it from just making the first one louder
    check = &channels[NUM_AMBIENTS];
    for (ch_idx = NUM_AMBIENTS;
	 ch_idx < NUM_AMBIENTS + MAX_DYNAMIC_CHANNELS; ch_idx++, check++) {
	if (check == target_chan)
	    continue;
	if (check->sfx == sfx && !check->pos) {
	    skip = rand() % (int) (0.1 * shm->speed);
	    if (skip >= target_chan->end)
		skip = target_chan->end - 1;
	    target_chan->pos += skip;
	    target_chan->end -= skip;
	    break;
	}

    }
}

void S_StopSound(int entnum, int entchannel)
{
    int i;

    for (i = 0; i < MAX_DYNAMIC_CHANNELS; i++) {
	if (channels[i].entnum == entnum
	    && channels[i].entchannel == entchannel) {
	    channels[i].end = 0;
	    channels[i].sfx = NULL;
	    return;
	}
    }
}

void S_StopAllSounds(void)
{
    int i;

    if (!sound_started)
	return;

    total_channels = MAX_DYNAMIC_CHANNELS + NUM_AMBIENTS;	// no statics

    for (i = 0; i < MAX_CHANNELS; i++)
	if (channels[i].sfx)
	    channels[i].sfx = NULL;

    Q_memset(channels, 0, MAX_CHANNELS * sizeof(channel_t));

    S_ClearBuffer();
}

void S_ClearBuffer(void)
{
    int clear;

    if (!sound_started || !shm || !shm->buffer)
	return;

    if (shm->samplebits == 8)
	clear = 0x80;
    else
	clear = 0;

    Q_memset(shm->buffer, clear, shm->samples * shm->samplebits / 8);
}


/*
=================
S_StaticSound
=================
*/
void S_StaticSound(sfx_t * sfx, vec3_t origin, float vol,
		   float attenuation)
{
    channel_t *ss;
    sfxcache_t *sc;

    if (!sfx)
	return;

    if (total_channels == MAX_CHANNELS) {
	Con_Printf("total_channels == MAX_CHANNELS\n");
	return;
    }

    ss = &channels[total_channels];
    total_channels++;

    sc = S_LoadSound(sfx);
    if (!sc)
	return;

    if (sc->loopstart == -1) {
	Con_Printf("Sound %s not looped\n", sfx->name);
	return;
    }

    ss->sfx = sfx;
    VectorCopy(origin, ss->origin);
    ss->master_vol = vol;
    ss->dist_mult = (attenuation / 64) / sound_nominal_clip_dist;
    ss->end = paintedtime + sc->length;

    SND_Spatialize(ss);
}


//=============================================================================

/*
===================
S_UpdateAmbientSounds
===================
*/
void S_UpdateAmbientSounds(void)
{
    mleaf_t *l;
    float vol;
    int ambient_channel;
    channel_t *chan;

// calc ambient sound levels
    if (!cl.worldmodel)
	return;

    l = Mod_PointInLeaf(listener_origin, cl.worldmodel);
    if (!l || !ambient_level.value) {
	for (ambient_channel = 0; ambient_channel < NUM_AMBIENTS;
	     ambient_channel++)
	    channels[ambient_channel].sfx = NULL;
	return;
    }

    for (ambient_channel = 0; ambient_channel < NUM_AMBIENTS;
	 ambient_channel++) {
	chan = &channels[ambient_channel];
	chan->sfx = ambient_sfx[ambient_channel];

	vol =
	    ambient_level.value * l->ambient_sound_level[ambient_channel];
	if (vol < 8)
	    vol = 0;

	// don't adjust volume too fast
	if (chan->master_vol < vol) {
	    chan->master_vol += host_frametime * ambient_fade.value;
	    if (chan->master_vol > vol)
		chan->master_vol = vol;
	} else if (chan->master_vol > vol) {
	    chan->master_vol -= host_frametime * ambient_fade.value;
	    if (chan->master_vol < vol)
		chan->master_vol = vol;
	}

	chan->leftvol = chan->rightvol = chan->master_vol;
    }
}


/*
============
S_Update

Called once each time through the main loop
============
*/
void S_Update(vec3_t origin, vec3_t forward, vec3_t right, vec3_t up)
{
    int i, j;
    int total;
    channel_t *ch;
    channel_t *combine;

    VectorCopy(origin, listener_origin);
    VectorCopy(forward, listener_forward);
    VectorCopy(right, listener_right);
    VectorCopy(up, listener_up);

// update general area ambient sound sources
    S_UpdateAmbientSounds();

    combine = NULL;

// update spatialization for static and dynamic sounds  
    ch = channels + NUM_AMBIENTS;
    for (i = NUM_AMBIENTS; i < total_channels; i++, ch++) {
	if (!ch->sfx)
	    continue;
	SND_Spatialize(ch);	// respatialize channel
	if (!ch->leftvol && !ch->rightvol)
	    continue;

	// try to combine static sounds with a previous channel of the same
	// sound effect so we don't mix five torches every frame

	if (i >= MAX_DYNAMIC_CHANNELS + NUM_AMBIENTS) {
	    // see if it can just use the last one
	    if (combine && combine->sfx == ch->sfx) {
		combine->leftvol += ch->leftvol;
		combine->rightvol += ch->rightvol;
		ch->leftvol = ch->rightvol = 0;
		continue;
	    }
	    // search for one
	    combine = channels + MAX_DYNAMIC_CHANNELS + NUM_AMBIENTS;
	    for (j = MAX_DYNAMIC_CHANNELS + NUM_AMBIENTS; j < i;
		 j++, combine++)
		if (combine->sfx == ch->sfx)
		    break;

	    if (j == total_channels) {
		combine = NULL;
	    } else {
		if (combine != ch) {
		    combine->leftvol += ch->leftvol;
		    combine->rightvol += ch->rightvol;
		    ch->leftvol = ch->rightvol = 0;
		}
		continue;
	    }
	}


    }

//
// debugging output
//
    if (snd_show.value) {
	total = 0;
	ch = channels;
	for (i = 0; i < total_channels; i++, ch++)
	    if (ch->sfx && (ch->leftvol || ch->rightvol)) {
		Con_Printf("%3i %3i %s\n", ch->leftvol, ch->rightvol,
			   ch->sfx->name);
		total++;
	    }

	Con_Printf("----(%i)----\n", total);
    }
// mix some sound
    S_Update_();
}

void GetSoundtime(void)
{
    int samplepos;
    static int buffers;
    static int oldsamplepos;
    int fullsamples;

    fullsamples = shm->samples / shm->channels;

// it is possible to miscount buffers if it has wrapped twice between
// calls to S_Update.  Oh well.
    samplepos = SNDDMA_GetDMAPos();
    if (samplepos < oldsamplepos) {
	buffers++;		// buffer wrapped

	if (paintedtime > 0x40000000) {	// time to chop things off to avoid 32 bit limits
	    Con_Printf("paintedtime chopped off\n");
	    buffers = 0;
	    paintedtime = fullsamples;
	    S_StopAllSounds();
	}
    }
    oldsamplepos = samplepos;

    soundtime = buffers * fullsamples + samplepos / shm->channels;
}

void S_ExtraUpdate(void)
{
    if (snd_noextraupdate.value)
	return;			// don't pollute timings
    S_Update_();
}



void S_Update_(void)
{
    unsigned endtime;
    int samps;

    if (!sound_started)
	return;

// Updates DMA time
    GetSoundtime();

// check to make sure that we haven't overshot
    if (paintedtime < soundtime) {
//              Con_Printf ("S_Update_ : overflow\n");
	paintedtime = soundtime;
    }
// mix to 0.1 seconds ahead of current position
    endtime = soundtime + 0.1 * shm->speed;
    samps = shm->samples >> (shm->channels - 1);
    if (endtime - soundtime > samps)
	endtime = soundtime + samps;

    S_PaintChannels(endtime);

    if (cls.signon == SIGNONS && cl.paused)
	Q_memset(shm->buffer, (shm->samplebits == 8) * 0x80,
		 shm->samples * shm->samplebits / 8);
}

/*
===============================================================================

console functions

===============================================================================
*/

void S_Play(void)
{
    static int hash = 345;
    int i;
    char name[256];
    sfx_t *sfx;

    i = 1;
    while (i < Cmd_Argc()) {
	if (!Q_strrchr(Cmd_Argv(i), '.')) {
	    Q_strcpy(name, Cmd_Argv(i));
	    Q_strcat(name, ".wav");
	} else
	    Q_strcpy(name, Cmd_Argv(i));
	sfx = S_PrecacheSound(name);
	S_StartSound(hash++, 0, sfx, listener_origin, 1.0, 1.0);
	i++;
    }
}

void S_PlayVol(void)
{
    static int hash = 543;
    int i;
    float vol;
    char name[256];
    sfx_t *sfx;

    i = 1;
    while (i < Cmd_Argc()) {
	if (!Q_strrchr(Cmd_Argv(i), '.')) {
	    Q_strcpy(name, Cmd_Argv(i));
	    Q_strcat(name, ".wav");
	} else
	    Q_strcpy(name, Cmd_Argv(i));
	sfx = S_PrecacheSound(name);
	vol = Q_atof(Cmd_Argv(i + 1));
	S_StartSound(hash++, 0, sfx, listener_origin, vol, 1.0);
	i += 2;
    }
}

void S_SoundList(void)
{
    int i;
    sfx_t *sfx;
    sfxcache_t *sc;
    int size, total;

    total = 0;
    for (sfx = known_sfx, i = 0; i < num_sfx; i++, sfx++) {
	sc = Cache_Check(&sfx->cache);
	if (!sc)
	    continue;
	size = sc->length * sc->width * (sc->stereo + 1);
	total += size;
	if (sc->loopstart >= 0)
	    Con_Printf("L");
	else
	    Con_Printf(" ");
	Con_Printf("(%2db) %6i : %s\n", sc->width * 8, size, sfx->name);
    }
    Con_Printf("Total resident: %i\n", total);
}


void S_LocalSound(char *sound)
{
    sfx_t *sfx;

    if (nosound.value)
	return;
    if (!sound_started)
	return;

    sfx = S_PrecacheSound(sound);
    if (!sfx) {
	Con_Printf("S_LocalSound: can't cache %s\n", sound);
	return;
    }
    S_StartSound(cl.viewentity, -1, sfx, vec3_origin, 1, 1);
}
